// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2021-2022, NVIDIA CORPORATION. All rights reserved
 *
 * The driver handles Error's from Control Backbone(CBB) generated due to
 * illegal accesses. When an error is reported from a NOC within CBB,
 * the driver checks ErrVld status of all three Error Logger's of that NOC.
 * It then prints debug information about failed transaction using ErrLog
 * registers of error logger which has ErrVld set. Currently, SLV, DEC,
 * TMO, SEC, UNS are the codes which are supported by CBB.
 */

#include <linux/clk.h>
#include <linux/cpufeature.h>
#include <linux/debugfs.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/platform_device.h>
#include <linux/device.h>
#include <linux/io.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <soc/tegra/fuse.h>
#include <soc/tegra/tegra-cbb.h>

#define ERRLOGGER_0_ID_COREID_0     0x00000000
#define ERRLOGGER_0_ID_REVISIONID_0 0x00000004
#define ERRLOGGER_0_FAULTEN_0       0x00000008
#define ERRLOGGER_0_ERRVLD_0        0x0000000c
#define ERRLOGGER_0_ERRCLR_0        0x00000010
#define ERRLOGGER_0_ERRLOG0_0       0x00000014
#define ERRLOGGER_0_ERRLOG1_0       0x00000018
#define ERRLOGGER_0_RSVD_00_0       0x0000001c
#define ERRLOGGER_0_ERRLOG3_0       0x00000020
#define ERRLOGGER_0_ERRLOG4_0       0x00000024
#define ERRLOGGER_0_ERRLOG5_0       0x00000028
#define ERRLOGGER_0_STALLEN_0       0x00000038

#define ERRLOGGER_1_ID_COREID_0     0x00000080
#define ERRLOGGER_1_ID_REVISIONID_0 0x00000084
#define ERRLOGGER_1_FAULTEN_0       0x00000088
#define ERRLOGGER_1_ERRVLD_0        0x0000008c
#define ERRLOGGER_1_ERRCLR_0        0x00000090
#define ERRLOGGER_1_ERRLOG0_0       0x00000094
#define ERRLOGGER_1_ERRLOG1_0       0x00000098
#define ERRLOGGER_1_RSVD_00_0       0x0000009c
#define ERRLOGGER_1_ERRLOG3_0       0x000000a0
#define ERRLOGGER_1_ERRLOG4_0       0x000000a4
#define ERRLOGGER_1_ERRLOG5_0       0x000000a8
#define ERRLOGGER_1_STALLEN_0       0x000000b8

#define ERRLOGGER_2_ID_COREID_0     0x00000100
#define ERRLOGGER_2_ID_REVISIONID_0 0x00000104
#define ERRLOGGER_2_FAULTEN_0       0x00000108
#define ERRLOGGER_2_ERRVLD_0        0x0000010c
#define ERRLOGGER_2_ERRCLR_0        0x00000110
#define ERRLOGGER_2_ERRLOG0_0       0x00000114
#define ERRLOGGER_2_ERRLOG1_0       0x00000118
#define ERRLOGGER_2_RSVD_00_0       0x0000011c
#define ERRLOGGER_2_ERRLOG3_0       0x00000120
#define ERRLOGGER_2_ERRLOG4_0       0x00000124
#define ERRLOGGER_2_ERRLOG5_0       0x00000128
#define ERRLOGGER_2_STALLEN_0       0x00000138

#define CBB_NOC_INITFLOW GENMASK(23, 20)
#define CBB_NOC_TARGFLOW GENMASK(19, 16)
#define CBB_NOC_TARG_SUBRANGE GENMASK(15, 9)
#define CBB_NOC_SEQID GENMASK(8, 0)

#define BPMP_NOC_INITFLOW GENMASK(20, 18)
#define BPMP_NOC_TARGFLOW GENMASK(17, 13)
#define BPMP_NOC_TARG_SUBRANGE GENMASK(12, 9)
#define BPMP_NOC_SEQID GENMASK(8, 0)

#define AON_NOC_INITFLOW GENMASK(22, 21)
#define AON_NOC_TARGFLOW GENMASK(20, 15)
#define AON_NOC_TARG_SUBRANGE GENMASK(14, 9)
#define AON_NOC_SEQID GENMASK(8, 0)

#define SCE_NOC_INITFLOW GENMASK(21, 19)
#define SCE_NOC_TARGFLOW GENMASK(18, 14)
#define SCE_NOC_TARG_SUBRANGE GENMASK(13, 9)
#define SCE_NOC_SEQID GENMASK(8, 0)

#define CBB_NOC_AXCACHE GENMASK(3, 0)
#define CBB_NOC_NON_MOD GENMASK(4, 4)
#define CBB_NOC_AXPROT GENMASK(7, 5)
#define CBB_NOC_FALCONSEC GENMASK(9, 8)
#define CBB_NOC_GRPSEC GENMASK(16, 10)
#define CBB_NOC_VQC GENMASK(18, 17)
#define CBB_NOC_MSTR_ID GENMASK(22, 19)
#define CBB_NOC_AXI_ID GENMASK(30, 23)

#define CLUSTER_NOC_AXCACHE GENMASK(3, 0)
#define CLUSTER_NOC_AXPROT GENMASK(6, 4)
#define CLUSTER_NOC_FALCONSEC GENMASK(8, 7)
#define CLUSTER_NOC_GRPSEC GENMASK(15, 9)
#define CLUSTER_NOC_VQC GENMASK(17, 16)
#define CLUSTER_NOC_MSTR_ID GENMASK(21, 18)

#define CBB_ERR_OPC GENMASK(4, 1)
#define CBB_ERR_ERRCODE GENMASK(10, 8)
#define CBB_ERR_LEN1 GENMASK(27, 16)

#define DMAAPB_X_RAW_INTERRUPT_STATUS 0x2ec

struct tegra194_cbb_packet_header {
	bool lock;   // [0]
	u8 opc;      // [4:1]
	u8 errcode;  // [10:8]= RD, RDW, RDL, RDX, WR, WRW, WRC, PRE, URG
	u16 len1;    // [27:16]
	bool format; // [31]  = 1 -> FlexNoC versions 2.7 & above
};

struct tegra194_cbb_aperture {
	u8 initflow;
	u8 targflow;
	u8 targ_subrange;
	u8 init_mapping;
	u32 init_localaddress;
	u8 targ_mapping;
	u32 targ_localaddress;
	u16 seqid;
};

struct tegra194_cbb_userbits {
	u8 axcache;
	u8 non_mod;
	u8 axprot;
	u8 falconsec;
	u8 grpsec;
	u8 vqc;
	u8 mstr_id;
	u8 axi_id;
};

struct tegra194_cbb_noc_data {
	const char *name;
	bool erd_mask_inband_err;
	const char * const *master_id;
	unsigned int max_aperture;
	const struct tegra194_cbb_aperture *noc_aperture;
	const char * const *routeid_initflow;
	const char * const *routeid_targflow;
	void (*parse_routeid)(struct tegra194_cbb_aperture *info, u64 routeid);
	void (*parse_userbits)(struct tegra194_cbb_userbits *usrbits, u32 elog_5);
};

struct tegra194_axi2apb_bridge {
	struct resource res;
	void __iomem *base;
};

struct tegra194_cbb {
	struct tegra_cbb base;

	const struct tegra194_cbb_noc_data *noc;
	struct resource *res;

	void __iomem *regs;
	unsigned int num_intr;
	unsigned int sec_irq;
	unsigned int nonsec_irq;
	u32 errlog0;
	u32 errlog1;
	u32 errlog2;
	u32 errlog3;
	u32 errlog4;
	u32 errlog5;

	struct tegra194_axi2apb_bridge *bridges;
	unsigned int num_bridges;
};

static inline struct tegra194_cbb *to_tegra194_cbb(struct tegra_cbb *cbb)
{
	return container_of(cbb, struct tegra194_cbb, base);
}

static LIST_HEAD(cbb_list);
static DEFINE_SPINLOCK(cbb_lock);

static const char * const tegra194_cbb_trantype[] = {
	"RD  - Read, Incrementing",
	"RDW - Read, Wrap",			/* Not Supported */
	"RDX - Exclusive Read",			/* Not Supported */
	"RDL - Linked Read",			/* Not Supported */
	"WR  - Write, Incrementing",
	"WRW - Write, Wrap",			/* Not Supported */
	"WRC - Exclusive Write",		/* Not Supported */
	"PRE - Preamble Sequence for Fixed Accesses"
};

static const char * const tegra194_axi2apb_error[] = {
	"SFIFONE - Status FIFO Not Empty interrupt",
	"SFIFOF - Status FIFO Full interrupt",
	"TIM - Timer(Timeout) interrupt",
	"SLV - SLVERR interrupt",
	"NULL",
	"ERBF - Early response buffer Full interrupt",
	"NULL",
	"RDFIFOF - Read Response FIFO Full interrupt",
	"WRFIFOF - Write Response FIFO Full interrupt",
	"CH0DFIFOF - Ch0 Data FIFO Full interrupt",
	"CH1DFIFOF - Ch1 Data FIFO Full interrupt",
	"CH2DFIFOF - Ch2 Data FIFO Full interrupt",
	"UAT - Unsupported alignment type error",
	"UBS - Unsupported burst size error",
	"UBE - Unsupported Byte Enable error",
	"UBT - Unsupported burst type error",
	"BFS - Block Firewall security error",
	"ARFS - Address Range Firewall security error",
	"CH0RFIFOF - Ch0 Request FIFO Full interrupt",
	"CH1RFIFOF - Ch1 Request FIFO Full interrupt",
	"CH2RFIFOF - Ch2 Request FIFO Full interrupt"
};

static const char * const tegra194_master_id[] = {
	[0x0] = "CCPLEX",
	[0x1] = "CCPLEX_DPMU",
	[0x2] = "BPMP",
	[0x3] = "AON",
	[0x4] = "SCE",
	[0x5] = "GPCDMA_PERIPHERAL",
	[0x6] = "TSECA",
	[0x7] = "TSECB",
	[0x8] = "JTAGM_DFT",
	[0x9] = "CORESIGHT_AXIAP",
	[0xa] = "APE",
	[0xb] = "PEATR",
	[0xc] = "NVDEC",
	[0xd] = "RCE",
	[0xe] = "NVDEC1"
};

static const struct tegra_cbb_error tegra194_cbb_errors[] = {
	{
		.code = "SLV",
		.source = "Target",
		.desc = "Target error detected by CBB slave"
	}, {
		.code = "DEC",
		.source = "Initiator NIU",
		.desc = "Address decode error"
	}, {
		.code = "UNS",
		.source = "Target NIU",
		.desc = "Unsupported request. Not a valid transaction"
	}, {
		.code = "DISC", /* Not Supported by CBB */
		.source = "Power Disconnect",
		.desc = "Disconnected target or domain"
	}, {
		.code = "SEC",
		.source = "Initiator NIU or Firewall",
		.desc = "Security violation. Firewall error"
	}, {
		.code = "HIDE", /* Not Supported by CBB */
		.source = "Firewall",
		.desc = "Hidden security violation, reported as OK to initiator"
	}, {
		.code = "TMO",
		.source = "Target NIU",
		.desc = "Target time-out error"
	}, {
		.code = "RSV",
		.source = "None",
		.desc = "Reserved"
	}
};

/*
 * CBB NOC aperture lookup table as per file "cbb_central_noc_Structure.info".
 */
static const char * const tegra194_cbbcentralnoc_routeid_initflow[] = {
	[0x0] = "aon_p2ps/I/aon",
	[0x1] = "ape_p2ps/I/ape_p2ps",
	[0x2] = "bpmp_p2ps/I/bpmp_p2ps",
	[0x3] = "ccroc_p2ps/I/ccroc_p2ps",
	[0x4] = "csite_p2ps/I/0",
	[0x5] = "gpcdma_mmio_p2ps/I/0",
	[0x6] = "jtag_p2ps/I/0",
	[0x7] = "nvdec1_p2ps/I/0",
	[0x8] = "nvdec_p2ps/I/0",
	[0x9] = "rce_p2ps/I/rce_p2ps",
	[0xa] = "sce_p2ps/I/sce_p2ps",
	[0xb] = "tseca_p2ps/I/0",
	[0xc] = "tsecb_p2ps/I/0",
	[0xd] = "RESERVED",
	[0xe] = "RESERVED",
	[0xf] = "RESERVED"
};

static const char * const tegra194_cbbcentralnoc_routeid_targflow[] = {
	[0x0] = "SVC/T/intreg",
	[0x1] = "axis_satellite_axi2apb_p2pm/T/axis_satellite_axi2apb_p2pm",
	[0x2] = "axis_satellite_grout/T/axis_satellite_grout",
	[0x3] = "cbb_firewall/T/cbb_firewall",
	[0x4] = "gpu_p2pm/T/gpu_p2pm",
	[0x5] = "host1x_p2pm/T/host1x_p2pm",
	[0x6] = "sapb_3_p2pm/T/sapb_3_p2pm",
	[0x7] = "smmu0_p2pm/T/smmu0_p2pm",
	[0x8] = "smmu1_p2pm/T/smmu1_p2pm",
	[0x9] = "smmu2_p2pm/T/smmu2_p2pm",
	[0xa] = "stm_p2pm/T/stm_p2pm",
	[0xb] = "RESERVED",
	[0xc] = "RESERVED",
	[0xd] = "RESERVED",
	[0xe] = "RESERVED",
	[0xf] = "RESERVED"
};

/*
 * Fields of CBB NOC lookup table:
 * Init flow, Targ flow, Targ subrange, Init mapping, Init localAddress,
 *                                              Targ mapping, Targ localAddress
 * ----------------------------------------------------------------------------
 */
static const struct tegra194_cbb_aperture tegra194_cbbcentralnoc_apert_lookup[] = {
	{ 0x0, 0x0, 0x00, 0x0, 0x02300000,  0, 0x00000000 },
	{ 0x0, 0x1, 0x00, 0x0, 0x02003000,  0, 0x02003000 },
	{ 0x0, 0x1, 0x01, 0x0, 0x02006000,  2, 0x02006000 },
	{ 0x0, 0x1, 0x02, 0x0, 0x02016000,  3, 0x02016000 },
	{ 0x0, 0x1, 0x03, 0x0, 0x0201d000,  4, 0x0201d000 },
	{ 0x0, 0x1, 0x04, 0x0, 0x0202b000,  6, 0x0202b000 },
	{ 0x0, 0x1, 0x05, 0x0, 0x02434000, 20, 0x02434000 },
	{ 0x0, 0x1, 0x06, 0x0, 0x02436000, 21, 0x02436000 },
	{ 0x0, 0x1, 0x07, 0x0, 0x02438000, 22, 0x02438000 },
	{ 0x0, 0x1, 0x08, 0x0, 0x02445000, 24, 0x02445000 },
	{ 0x0, 0x1, 0x09, 0x0, 0x02446000, 25, 0x02446000 },
	{ 0x0, 0x1, 0x0a, 0x0, 0x02004000,  1, 0x02004000 },
	{ 0x0, 0x1, 0x0b, 0x0, 0x0201e000,  5, 0x0201e000 },
	{ 0x0, 0x1, 0x0c, 0x0, 0x0202c000,  7, 0x0202c000 },
	{ 0x0, 0x1, 0x0d, 0x0, 0x02204000,  8, 0x02204000 },
	{ 0x0, 0x1, 0x0e, 0x0, 0x02214000,  9, 0x02214000 },
	{ 0x0, 0x1, 0x0f, 0x0, 0x02224000, 10, 0x02224000 },
	{ 0x0, 0x1, 0x10, 0x0, 0x02234000, 11, 0x02234000 },
	{ 0x0, 0x1, 0x11, 0x0, 0x02244000, 12, 0x02244000 },
	{ 0x0, 0x1, 0x12, 0x0, 0x02254000, 13, 0x02254000 },
	{ 0x0, 0x1, 0x13, 0x0, 0x02264000, 14, 0x02264000 },
	{ 0x0, 0x1, 0x14, 0x0, 0x02274000, 15, 0x02274000 },
	{ 0x0, 0x1, 0x15, 0x0, 0x02284000, 16, 0x02284000 },
	{ 0x0, 0x1, 0x16, 0x0, 0x0243a000, 23, 0x0243a000 },
	{ 0x0, 0x1, 0x17, 0x0, 0x02370000, 17, 0x02370000 },
	{ 0x0, 0x1, 0x18, 0x0, 0x023d0000, 18, 0x023d0000 },
	{ 0x0, 0x1, 0x19, 0x0, 0x023e0000, 19, 0x023e0000 },
	{ 0x0, 0x1, 0x1a, 0x0, 0x02450000, 26, 0x02450000 },
	{ 0x0, 0x1, 0x1b, 0x0, 0x02460000, 27, 0x02460000 },
	{ 0x0, 0x1, 0x1c, 0x0, 0x02490000, 28, 0x02490000 },
	{ 0x0, 0x1, 0x1d, 0x0, 0x03130000, 31, 0x03130000 },
	{ 0x0, 0x1, 0x1e, 0x0, 0x03160000, 32, 0x03160000 },
	{ 0x0, 0x1, 0x1f, 0x0, 0x03270000, 33, 0x03270000 },
	{ 0x0, 0x1, 0x20, 0x0, 0x032e0000, 35, 0x032e0000 },
	{ 0x0, 0x1, 0x21, 0x0, 0x03300000, 36, 0x03300000 },
	{ 0x0, 0x1, 0x22, 0x0, 0x13090000, 40, 0x13090000 },
	{ 0x0, 0x1, 0x23, 0x0, 0x20120000, 43, 0x20120000 },
	{ 0x0, 0x1, 0x24, 0x0, 0x20170000, 44, 0x20170000 },
	{ 0x0, 0x1, 0x25, 0x0, 0x20190000, 45, 0x20190000 },
	{ 0x0, 0x1, 0x26, 0x0, 0x201b0000, 46, 0x201b0000 },
	{ 0x0, 0x1, 0x27, 0x0, 0x20250000, 47, 0x20250000 },
	{ 0x0, 0x1, 0x28, 0x0, 0x20260000, 48, 0x20260000 },
	{ 0x0, 0x1, 0x29, 0x0, 0x20420000, 49, 0x20420000 },
	{ 0x0, 0x1, 0x2a, 0x0, 0x20460000, 50, 0x20460000 },
	{ 0x0, 0x1, 0x2b, 0x0, 0x204f0000, 51, 0x204f0000 },
	{ 0x0, 0x1, 0x2c, 0x0, 0x20520000, 52, 0x20520000 },
	{ 0x0, 0x1, 0x2d, 0x0, 0x20580000, 53, 0x20580000 },
	{ 0x0, 0x1, 0x2e, 0x0, 0x205a0000, 54, 0x205a0000 },
	{ 0x0, 0x1, 0x2f, 0x0, 0x205c0000, 55, 0x205c0000 },
	{ 0x0, 0x1, 0x30, 0x0, 0x20690000, 56, 0x20690000 },
	{ 0x0, 0x1, 0x31, 0x0, 0x20770000, 57, 0x20770000 },
	{ 0x0, 0x1, 0x32, 0x0, 0x20790000, 58, 0x20790000 },
	{ 0x0, 0x1, 0x33, 0x0, 0x20880000, 59, 0x20880000 },
	{ 0x0, 0x1, 0x34, 0x0, 0x20990000, 62, 0x20990000 },
	{ 0x0, 0x1, 0x35, 0x0, 0x20e10000, 65, 0x20e10000 },
	{ 0x0, 0x1, 0x36, 0x0, 0x20e70000, 66, 0x20e70000 },
	{ 0x0, 0x1, 0x37, 0x0, 0x20e80000, 67, 0x20e80000 },
	{ 0x0, 0x1, 0x38, 0x0, 0x20f30000, 68, 0x20f30000 },
	{ 0x0, 0x1, 0x39, 0x0, 0x20f50000, 69, 0x20f50000 },
	{ 0x0, 0x1, 0x3a, 0x0, 0x20fc0000, 70, 0x20fc0000 },
	{ 0x0, 0x1, 0x3b, 0x0, 0x21110000, 72, 0x21110000 },
	{ 0x0, 0x1, 0x3c, 0x0, 0x21270000, 73, 0x21270000 },
	{ 0x0, 0x1, 0x3d, 0x0, 0x21290000, 74, 0x21290000 },
	{ 0x0, 0x1, 0x3e, 0x0, 0x21840000, 75, 0x21840000 },
	{ 0x0, 0x1, 0x3f, 0x0, 0x21880000, 76, 0x21880000 },
	{ 0x0, 0x1, 0x40, 0x0, 0x218d0000, 77, 0x218d0000 },
	{ 0x0, 0x1, 0x41, 0x0, 0x21950000, 78, 0x21950000 },
	{ 0x0, 0x1, 0x42, 0x0, 0x21960000, 79, 0x21960000 },
	{ 0x0, 0x1, 0x43, 0x0, 0x21a10000, 80, 0x21a10000 },
	{ 0x0, 0x1, 0x44, 0x0, 0x024a0000, 29, 0x024a0000 },
	{ 0x0, 0x1, 0x45, 0x0, 0x024c0000, 30, 0x024c0000 },
	{ 0x0, 0x1, 0x46, 0x0, 0x032c0000, 34, 0x032c0000 },
	{ 0x0, 0x1, 0x47, 0x0, 0x03400000, 37, 0x03400000 },
	{ 0x0, 0x1, 0x48, 0x0, 0x130a0000, 41, 0x130a0000 },
	{ 0x0, 0x1, 0x49, 0x0, 0x130c0000, 42, 0x130c0000 },
	{ 0x0, 0x1, 0x4a, 0x0, 0x208a0000, 60, 0x208a0000 },
	{ 0x0, 0x1, 0x4b, 0x0, 0x208c0000, 61, 0x208c0000 },
	{ 0x0, 0x1, 0x4c, 0x0, 0x209a0000, 63, 0x209a0000 },
	{ 0x0, 0x1, 0x4d, 0x0, 0x21a40000, 81, 0x21a40000 },
	{ 0x0, 0x1, 0x4e, 0x0, 0x03440000, 38, 0x03440000 },
	{ 0x0, 0x1, 0x4f, 0x0, 0x20d00000, 64, 0x20d00000 },
	{ 0x0, 0x1, 0x50, 0x0, 0x21000000, 71, 0x21000000 },
	{ 0x0, 0x1, 0x51, 0x0, 0x0b000000, 39, 0x0b000000 },
	{ 0x0, 0x2, 0x00, 0x0, 0x00000000,  0, 0x00000000 },
	{ 0x0, 0x3, 0x00, 0x0, 0x02340000,  0, 0x00000000 },
	{ 0x0, 0x4, 0x00, 0x0, 0x17000000,  0, 0x17000000 },
	{ 0x0, 0x4, 0x01, 0x0, 0x18000000,  1, 0x18000000 },
	{ 0x0, 0x5, 0x00, 0x0, 0x13e80000,  1, 0x13e80000 },
	{ 0x0, 0x5, 0x01, 0x0, 0x15810000, 12, 0x15810000 },
	{ 0x0, 0x5, 0x02, 0x0, 0x15840000, 14, 0x15840000 },
	{ 0x0, 0x5, 0x03, 0x0, 0x15a40000, 17, 0x15a40000 },
	{ 0x0, 0x5, 0x04, 0x0, 0x13f00000,  3, 0x13f00000 },
	{ 0x0, 0x5, 0x05, 0x0, 0x15820000, 13, 0x15820000 },
	{ 0x0, 0x5, 0x06, 0x0, 0x13ec0000,  2, 0x13ec0000 },
	{ 0x0, 0x5, 0x07, 0x0, 0x15200000,  6, 0x15200000 },
	{ 0x0, 0x5, 0x08, 0x0, 0x15340000,  7, 0x15340000 },
	{ 0x0, 0x5, 0x09, 0x0, 0x15380000,  8, 0x15380000 },
	{ 0x0, 0x5, 0x0a, 0x0, 0x15500000, 10, 0x15500000 },
	{ 0x0, 0x5, 0x0b, 0x0, 0x155c0000, 11, 0x155c0000 },
	{ 0x0, 0x5, 0x0c, 0x0, 0x15a00000, 16, 0x15a00000 },
	{ 0x0, 0x5, 0x0d, 0x0, 0x13e00000,  0, 0x13e00000 },
	{ 0x0, 0x5, 0x0e, 0x0, 0x15100000,  5, 0x15100000 },
	{ 0x0, 0x5, 0x0f, 0x0, 0x15480000,  9, 0x15480000 },
	{ 0x0, 0x5, 0x10, 0x0, 0x15880000, 15, 0x15880000 },
	{ 0x0, 0x5, 0x11, 0x0, 0x15a80000, 18, 0x15a80000 },
	{ 0x0, 0x5, 0x12, 0x0, 0x15b00000, 19, 0x15b00000 },
	{ 0x0, 0x5, 0x13, 0x0, 0x14800000,  4, 0x14800000 },
	{ 0x0, 0x5, 0x14, 0x0, 0x15c00000, 20, 0x15c00000 },
	{ 0x0, 0x5, 0x15, 0x0, 0x16000000, 21, 0x16000000 },
	{ 0x0, 0x6, 0x00, 0x0, 0x02000000,  4, 0x02000000 },
	{ 0x0, 0x6, 0x01, 0x0, 0x02007000,  5, 0x02007000 },
	{ 0x0, 0x6, 0x02, 0x0, 0x02008000,  6, 0x02008000 },
	{ 0x0, 0x6, 0x03, 0x0, 0x02013000,  7, 0x02013000 },
	{ 0x0, 0x6, 0x04, 0x0, 0x0201c000,  8, 0x0201c000 },
	{ 0x0, 0x6, 0x05, 0x0, 0x02020000,  9, 0x02020000 },
	{ 0x0, 0x6, 0x06, 0x0, 0x0202a000, 10, 0x0202a000 },
	{ 0x0, 0x6, 0x07, 0x0, 0x0202e000, 11, 0x0202e000 },
	{ 0x0, 0x6, 0x08, 0x0, 0x06400000, 33, 0x06400000 },
	{ 0x0, 0x6, 0x09, 0x0, 0x02038000, 12, 0x02038000 },
	{ 0x0, 0x6, 0x0a, 0x0, 0x00100000,  0, 0x00100000 },
	{ 0x0, 0x6, 0x0b, 0x0, 0x023b0000, 13, 0x023b0000 },
	{ 0x0, 0x6, 0x0c, 0x0, 0x02800000, 16, 0x02800000 },
	{ 0x0, 0x6, 0x0d, 0x0, 0x030e0000, 22, 0x030e0000 },
	{ 0x0, 0x6, 0x0e, 0x0, 0x03800000, 23, 0x03800000 },
	{ 0x0, 0x6, 0x0f, 0x0, 0x03980000, 25, 0x03980000 },
	{ 0x0, 0x6, 0x10, 0x0, 0x03a60000, 26, 0x03a60000 },
	{ 0x0, 0x6, 0x11, 0x0, 0x03d80000, 31, 0x03d80000 },
	{ 0x0, 0x6, 0x12, 0x0, 0x20000000, 36, 0x20000000 },
	{ 0x0, 0x6, 0x13, 0x0, 0x20050000, 38, 0x20050000 },
	{ 0x0, 0x6, 0x14, 0x0, 0x201e0000, 40, 0x201e0000 },
	{ 0x0, 0x6, 0x15, 0x0, 0x20280000, 42, 0x20280000 },
	{ 0x0, 0x6, 0x16, 0x0, 0x202c0000, 43, 0x202c0000 },
	{ 0x0, 0x6, 0x17, 0x0, 0x20390000, 44, 0x20390000 },
	{ 0x0, 0x6, 0x18, 0x0, 0x20430000, 45, 0x20430000 },
	{ 0x0, 0x6, 0x19, 0x0, 0x20440000, 46, 0x20440000 },
	{ 0x0, 0x6, 0x1a, 0x0, 0x204e0000, 47, 0x204e0000 },
	{ 0x0, 0x6, 0x1b, 0x0, 0x20550000, 48, 0x20550000 },
	{ 0x0, 0x6, 0x1c, 0x0, 0x20570000, 49, 0x20570000 },
	{ 0x0, 0x6, 0x1d, 0x0, 0x20590000, 50, 0x20590000 },
	{ 0x0, 0x6, 0x1e, 0x0, 0x20730000, 52, 0x20730000 },
	{ 0x0, 0x6, 0x1f, 0x0, 0x209f0000, 54, 0x209f0000 },
	{ 0x0, 0x6, 0x20, 0x0, 0x20e20000, 55, 0x20e20000 },
	{ 0x0, 0x6, 0x21, 0x0, 0x20ed0000, 56, 0x20ed0000 },
	{ 0x0, 0x6, 0x22, 0x0, 0x20fd0000, 57, 0x20fd0000 },
	{ 0x0, 0x6, 0x23, 0x0, 0x21120000, 59, 0x21120000 },
	{ 0x0, 0x6, 0x24, 0x0, 0x211a0000, 60, 0x211a0000 },
	{ 0x0, 0x6, 0x25, 0x0, 0x21850000, 61, 0x21850000 },
	{ 0x0, 0x6, 0x26, 0x0, 0x21860000, 62, 0x21860000 },
	{ 0x0, 0x6, 0x27, 0x0, 0x21890000, 63, 0x21890000 },
	{ 0x0, 0x6, 0x28, 0x0, 0x21970000, 64, 0x21970000 },
	{ 0x0, 0x6, 0x29, 0x0, 0x21990000, 65, 0x21990000 },
	{ 0x0, 0x6, 0x2a, 0x0, 0x21a00000, 66, 0x21a00000 },
	{ 0x0, 0x6, 0x2b, 0x0, 0x21a90000, 68, 0x21a90000 },
	{ 0x0, 0x6, 0x2c, 0x0, 0x21ac0000, 70, 0x21ac0000 },
	{ 0x0, 0x6, 0x2d, 0x0, 0x01f80000,  3, 0x01f80000 },
	{ 0x0, 0x6, 0x2e, 0x0, 0x024e0000, 14, 0x024e0000 },
	{ 0x0, 0x6, 0x2f, 0x0, 0x030c0000, 21, 0x030c0000 },
	{ 0x0, 0x6, 0x30, 0x0, 0x03820000, 24, 0x03820000 },
	{ 0x0, 0x6, 0x31, 0x0, 0x03aa0000, 27, 0x03aa0000 },
	{ 0x0, 0x6, 0x32, 0x0, 0x03c80000, 29, 0x03c80000 },
	{ 0x0, 0x6, 0x33, 0x0, 0x130e0000, 34, 0x130e0000 },
	{ 0x0, 0x6, 0x34, 0x0, 0x20020000, 37, 0x20020000 },
	{ 0x0, 0x6, 0x35, 0x0, 0x20060000, 39, 0x20060000 },
	{ 0x0, 0x6, 0x36, 0x0, 0x20200000, 41, 0x20200000 },
	{ 0x0, 0x6, 0x37, 0x0, 0x206a0000, 51, 0x206a0000 },
	{ 0x0, 0x6, 0x38, 0x0, 0x20740000, 53, 0x20740000 },
	{ 0x0, 0x6, 0x39, 0x0, 0x20fe0000, 58, 0x20fe0000 },
	{ 0x0, 0x6, 0x3a, 0x0, 0x21a20000, 67, 0x21a20000 },
	{ 0x0, 0x6, 0x3b, 0x0, 0x21aa0000, 69, 0x21aa0000 },
	{ 0x0, 0x6, 0x3c, 0x0, 0x02b80000, 17, 0x02b80000 },
	{ 0x0, 0x6, 0x3d, 0x0, 0x03080000, 20, 0x03080000 },
	{ 0x0, 0x6, 0x3e, 0x0, 0x13100000, 35, 0x13100000 },
	{ 0x0, 0x6, 0x3f, 0x0, 0x01f00000,  2, 0x01f00000 },
	{ 0x0, 0x6, 0x40, 0x0, 0x03000000, 19, 0x03000000 },
	{ 0x0, 0x6, 0x41, 0x0, 0x03c00000, 28, 0x03c00000 },
	{ 0x0, 0x6, 0x42, 0x0, 0x03d00000, 30, 0x03d00000 },
	{ 0x0, 0x6, 0x43, 0x0, 0x01700000,  1, 0x01700000 },
	{ 0x0, 0x6, 0x44, 0x0, 0x02c00000, 18, 0x02c00000 },
	{ 0x0, 0x6, 0x45, 0x0, 0x02600000, 15, 0x02600000 },
	{ 0x0, 0x6, 0x46, 0x0, 0x06000000, 32, 0x06000000 },
	{ 0x0, 0x6, 0x47, 0x0, 0x24000000, 71, 0x24000000 },
	{ 0x0, 0x7, 0x00, 0x0, 0x12000000,  0, 0x12000000 },
	{ 0x0, 0x8, 0x00, 0x0, 0x11000000,  0, 0x11000000 },
	{ 0x0, 0x9, 0x00, 0x0, 0x10000000,  0, 0x10000000 },
	{ 0x0, 0xa, 0x00, 0x0, 0x22000000,  0, 0x22000000 }
};

/*
 * BPMP NOC aperture lookup table as per file "BPMP_NOC_Structure.info".
 */
static const char * const tegra194_bpmpnoc_routeid_initflow[] = {
	[0x0] = "cbb_i/I/0",
	[0x1] = "cpu_m_i/I/0",
	[0x2] = "cpu_p_i/I/0",
	[0x3] = "cvc_i/I/0",
	[0x4] = "dma_m_i/I/0",
	[0x5] = "dma_p_i/I/0",
	[0x6] = "RESERVED",
	[0x7] = "RESERVED"
};

static const char * const tegra194_bpmpnoc_routeid_targflow[] = {
	[0x00] = "multiport0_t/T/actmon",
	[0x01] = "multiport0_t/T/ast_0",
	[0x02] = "multiport0_t/T/ast_1",
	[0x03] = "multiport0_t/T/atcm_cfg",
	[0x04] = "multiport0_t/T/car",
	[0x05] = "multiport0_t/T/central_pwr_mgr",
	[0x06] = "multiport0_t/T/central_vtg_ctlr",
	[0x07] = "multiport0_t/T/cfg",
	[0x08] = "multiport0_t/T/dma",
	[0x09] = "multiport0_t/T/err_collator",
	[0x0a] = "multiport0_t/T/err_collator_car",
	[0x0b] = "multiport0_t/T/fpga_misc",
	[0x0c] = "multiport0_t/T/fpga_uart",
	[0x0d] = "multiport0_t/T/gte",
	[0x0e] = "multiport0_t/T/hsp",
	[0x0f] = "multiport0_t/T/misc",
	[0x10] = "multiport0_t/T/pm",
	[0x11] = "multiport0_t/T/simon0",
	[0x12] = "multiport0_t/T/simon1",
	[0x13] = "multiport0_t/T/simon2",
	[0x14] = "multiport0_t/T/simon3",
	[0x15] = "multiport0_t/T/simon4",
	[0x16] = "multiport0_t/T/soc_therm",
	[0x17] = "multiport0_t/T/tke",
	[0x18] = "multiport0_t/T/vic_0",
	[0x19] = "multiport0_t/T/vic_1",
	[0x1a] = "ast0_t/T/0",
	[0x1b] = "ast1_t/T/0",
	[0x1c] = "bpmp_noc_firewall/T/0",
	[0x1d] = "cbb_t/T/0",
	[0x1e] = "cpu_t/T/0",
	[0x1f] = "svc_t/T/0"
};

/*
 * Fields of BPMP NOC lookup table:
 * Init flow, Targ flow, Targ subrange, Init mapping, Init localAddress,
 *                                              Targ mapping, Targ localAddress
 * ----------------------------------------------------------------------------
 */
static const struct tegra194_cbb_aperture tegra194_bpmpnoc_apert_lookup[] = {
	{ 0x0, 0x1c, 0x0, 0x0, 0x0d640000, 0,  0x00000000 },
	{ 0x0, 0x1e, 0x0, 0x0, 0x0d400000, 0,  0x0d400000 },
	{ 0x0, 0x00, 0x0, 0x0, 0x0d230000, 0,  0x00000000 },
	{ 0x0, 0x01, 0x0, 0x0, 0x0d040000, 0,  0x00000000 },
	{ 0x0, 0x02, 0x0, 0x0, 0x0d050000, 0,  0x00000000 },
	{ 0x0, 0x03, 0x0, 0x0, 0x0d000000, 0,  0x00000000 },
	{ 0x0, 0x04, 0x0, 0x0, 0x20ae0000, 3,  0x000e0000 },
	{ 0x0, 0x04, 0x1, 0x0, 0x20ac0000, 2,  0x000c0000 },
	{ 0x0, 0x04, 0x2, 0x0, 0x20a80000, 1,  0x00080000 },
	{ 0x0, 0x04, 0x3, 0x0, 0x20a00000, 0,  0x00000000 },
	{ 0x0, 0x05, 0x0, 0x0, 0x0d2a0000, 0,  0x00000000 },
	{ 0x0, 0x06, 0x0, 0x0, 0x0d290000, 0,  0x00000000 },
	{ 0x0, 0x07, 0x0, 0x0, 0x0d2c0000, 0,  0x00000000 },
	{ 0x0, 0x08, 0x0, 0x0, 0x0d0e0000, 4,  0x00080000 },
	{ 0x0, 0x08, 0x1, 0x0, 0x0d060000, 0,  0x00000000 },
	{ 0x0, 0x08, 0x2, 0x0, 0x0d080000, 1,  0x00020000 },
	{ 0x0, 0x08, 0x3, 0x0, 0x0d0a0000, 2,  0x00040000 },
	{ 0x0, 0x08, 0x4, 0x0, 0x0d0c0000, 3,  0x00060000 },
	{ 0x0, 0x09, 0x0, 0x0, 0x0d650000, 0,  0x00000000 },
	{ 0x0, 0x0a, 0x0, 0x0, 0x20af0000, 0,  0x00000000 },
	{ 0x0, 0x0b, 0x0, 0x0, 0x0d3e0000, 0,  0x00000000 },
	{ 0x0, 0x0c, 0x0, 0x0, 0x0d3d0000, 0,  0x00000000 },
	{ 0x0, 0x0d, 0x0, 0x0, 0x0d1e0000, 0,  0x00000000 },
	{ 0x0, 0x0e, 0x0, 0x0, 0x0d150000, 0,  0x00000000 },
	{ 0x0, 0x0e, 0x1, 0x0, 0x0d160000, 1,  0x00010000 },
	{ 0x0, 0x0e, 0x2, 0x0, 0x0d170000, 2,  0x00020000 },
	{ 0x0, 0x0e, 0x3, 0x0, 0x0d180000, 3,  0x00030000 },
	{ 0x0, 0x0e, 0x4, 0x0, 0x0d190000, 4,  0x00040000 },
	{ 0x0, 0x0e, 0x5, 0x0, 0x0d1a0000, 5,  0x00050000 },
	{ 0x0, 0x0e, 0x6, 0x0, 0x0d1b0000, 6,  0x00060000 },
	{ 0x0, 0x0e, 0x7, 0x0, 0x0d1c0000, 7,  0x00070000 },
	{ 0x0, 0x0e, 0x8, 0x0, 0x0d1d0000, 8,  0x00080000 },
	{ 0x0, 0x0f, 0x0, 0x0, 0x0d660000, 0,  0x00000000 },
	{ 0x0, 0x10, 0x0, 0x0, 0x0d1f0000, 0,  0x00000000 },
	{ 0x0, 0x10, 0x1, 0x0, 0x0d200000, 1,  0x00010000 },
	{ 0x0, 0x10, 0x2, 0x0, 0x0d210000, 2,  0x00020000 },
	{ 0x0, 0x10, 0x3, 0x0, 0x0d220000, 3,  0x00030000 },
	{ 0x0, 0x11, 0x0, 0x0, 0x0d240000, 0,  0x00000000 },
	{ 0x0, 0x12, 0x0, 0x0, 0x0d250000, 0,  0x00000000 },
	{ 0x0, 0x13, 0x0, 0x0, 0x0d260000, 0,  0x00000000 },
	{ 0x0, 0x14, 0x0, 0x0, 0x0d270000, 0,  0x00000000 },
	{ 0x0, 0x15, 0x0, 0x0, 0x0d2b0000, 0,  0x00000000 },
	{ 0x0, 0x16, 0x0, 0x0, 0x0d280000, 0,  0x00000000 },
	{ 0x0, 0x17, 0x0, 0x0, 0x0d0f0000, 0,  0x00000000 },
	{ 0x0, 0x17, 0x1, 0x0, 0x0d100000, 1,  0x00010000 },
	{ 0x0, 0x17, 0x2, 0x0, 0x0d110000, 2,  0x00020000 },
	{ 0x0, 0x17, 0x3, 0x0, 0x0d120000, 3,  0x00030000 },
	{ 0x0, 0x17, 0x4, 0x0, 0x0d130000, 4,  0x00040000 },
	{ 0x0, 0x17, 0x5, 0x0, 0x0d140000, 5,  0x00050000 },
	{ 0x0, 0x18, 0x0, 0x0, 0x0d020000, 0,  0x00000000 },
	{ 0x0, 0x19, 0x0, 0x0, 0x0d030000, 0,  0x00000000 },
	{ 0x0, 0x1f, 0x0, 0x0, 0x0d600000, 0,  0x00000000 },
	{ 0x0, 0x1f, 0x1, 0x0, 0x00000000, 0,  0x00000000 },
	{ 0x1, 0x1a, 0x0, 0x0, 0x40000000, 0,  0x40000000 },
	{ 0x1, 0x1a, 0x1, 0x1, 0x80000000, 1,  0x80000000 },
	{ 0x1, 0x1a, 0x2, 0x0, 0x00000000, 0,  0x00000000 },
	{ 0x2, 0x1c, 0x0, 0x0, 0x0d640000, 0,  0x00000000 },
	{ 0x2, 0x1d, 0x0, 0x0, 0x20b00000, 8,  0x20b00000 },
	{ 0x2, 0x1d, 0x1, 0x0, 0x20800000, 7,  0x20800000 },
	{ 0x2, 0x1d, 0x2, 0x0, 0x20c00000, 9,  0x20c00000 },
	{ 0x2, 0x1d, 0x3, 0x0, 0x0d800000, 3,  0x0d800000 },
	{ 0x2, 0x1d, 0x4, 0x0, 0x20000000, 6,  0x20000000 },
	{ 0x2, 0x1d, 0x5, 0x0, 0x0c000000, 2,  0x0c000000 },
	{ 0x2, 0x1d, 0x6, 0x0, 0x21000000, 10, 0x21000000 },
	{ 0x2, 0x1d, 0x7, 0x0, 0x0e000000, 4,  0x0e000000 },
	{ 0x2, 0x1d, 0x8, 0x0, 0x22000000, 11, 0x22000000 },
	{ 0x2, 0x1d, 0x9, 0x0, 0x08000000, 1,  0x08000000 },
	{ 0x2, 0x1d, 0xa, 0x0, 0x24000000, 12, 0x24000000 },
	{ 0x2, 0x1d, 0xb, 0x0, 0x00000000, 0,  0x00000000 },
	{ 0x2, 0x1d, 0xc, 0x0, 0x28000000, 13, 0x28000000 },
	{ 0x2, 0x1d, 0xd, 0x0, 0x10000000, 5,  0x10000000 },
	{ 0x2, 0x1d, 0xe, 0x0, 0x30000000, 14, 0x30000000 },
	{ 0x2, 0x00, 0x0, 0x0, 0x0d230000, 0,  0x00000000 },
	{ 0x2, 0x01, 0x0, 0x0, 0x0d040000, 0,  0x00000000 },
	{ 0x2, 0x02, 0x0, 0x0, 0x0d050000, 0,  0x00000000 },
	{ 0x2, 0x03, 0x0, 0x0, 0x0d000000, 0,  0x00000000 },
	{ 0x2, 0x04, 0x0, 0x0, 0x20ae0000, 3,  0x000e0000 },
	{ 0x2, 0x04, 0x1, 0x0, 0x20ac0000, 2,  0x000c0000 },
	{ 0x2, 0x04, 0x2, 0x0, 0x20a80000, 1,  0x00080000 },
	{ 0x2, 0x04, 0x3, 0x0, 0x20a00000, 0,  0x00000000 },
	{ 0x2, 0x05, 0x0, 0x0, 0x0d2a0000, 0,  0x00000000 },
	{ 0x2, 0x06, 0x0, 0x0, 0x0d290000, 0,  0x00000000 },
	{ 0x2, 0x07, 0x0, 0x0, 0x0d2c0000, 0,  0x00000000 },
	{ 0x2, 0x08, 0x0, 0x0, 0x0d0e0000, 4,  0x00080000 },
	{ 0x2, 0x08, 0x1, 0x0, 0x0d060000, 0,  0x00000000 },
	{ 0x2, 0x08, 0x2, 0x0, 0x0d080000, 1,  0x00020000 },
	{ 0x2, 0x08, 0x3, 0x0, 0x0d0a0000, 2,  0x00040000 },
	{ 0x2, 0x08, 0x4, 0x0, 0x0d0c0000, 3,  0x00060000 },
	{ 0x2, 0x09, 0x0, 0x0, 0x0d650000, 0,  0x00000000 },
	{ 0x2, 0x0a, 0x0, 0x0, 0x20af0000, 0,  0x00000000 },
	{ 0x2, 0x0b, 0x0, 0x0, 0x0d3e0000, 0,  0x00000000 },
	{ 0x2, 0x0c, 0x0, 0x0, 0x0d3d0000, 0,  0x00000000 },
	{ 0x2, 0x0d, 0x0, 0x0, 0x0d1e0000, 0,  0x00000000 },
	{ 0x2, 0x0e, 0x0, 0x0, 0x0d150000, 0,  0x00000000 },
	{ 0x2, 0x0e, 0x1, 0x0, 0x0d160000, 1,  0x00010000 },
	{ 0x2, 0x0e, 0x2, 0x0, 0x0d170000, 2,  0x00020000 },
	{ 0x2, 0x0e, 0x3, 0x0, 0x0d180000, 3,  0x00030000 },
	{ 0x2, 0x0e, 0x4, 0x0, 0x0d190000, 4,  0x00040000 },
	{ 0x2, 0x0e, 0x5, 0x0, 0x0d1a0000, 5,  0x00050000 },
	{ 0x2, 0x0e, 0x6, 0x0, 0x0d1b0000, 6,  0x00060000 },
	{ 0x2, 0x0e, 0x7, 0x0, 0x0d1c0000, 7,  0x00070000 },
	{ 0x2, 0x0e, 0x8, 0x0, 0x0d1d0000, 8,  0x00080000 },
	{ 0x2, 0x0f, 0x0, 0x0, 0x0d660000, 0,  0x00000000 },
	{ 0x2, 0x10, 0x0, 0x0, 0x0d1f0000, 0,  0x00000000 },
	{ 0x2, 0x10, 0x1, 0x0, 0x0d200000, 1,  0x00010000 },
	{ 0x2, 0x10, 0x2, 0x0, 0x0d210000, 2,  0x00020000 },
	{ 0x2, 0x10, 0x3, 0x0, 0x0d220000, 3,  0x00030000 },
	{ 0x2, 0x11, 0x0, 0x0, 0x0d240000, 0,  0x00000000 },
	{ 0x2, 0x12, 0x0, 0x0, 0x0d250000, 0,  0x00000000 },
	{ 0x2, 0x13, 0x0, 0x0, 0x0d260000, 0,  0x00000000 },
	{ 0x2, 0x14, 0x0, 0x0, 0x0d270000, 0,  0x00000000 },
	{ 0x2, 0x15, 0x0, 0x0, 0x0d2b0000, 0,  0x00000000 },
	{ 0x2, 0x16, 0x0, 0x0, 0x0d280000, 0,  0x00000000 },
	{ 0x2, 0x17, 0x0, 0x0, 0x0d0f0000, 0,  0x00000000 },
	{ 0x2, 0x17, 0x1, 0x0, 0x0d100000, 1,  0x00010000 },
	{ 0x2, 0x17, 0x2, 0x0, 0x0d110000, 2,  0x00020000 },
	{ 0x2, 0x17, 0x3, 0x0, 0x0d120000, 3,  0x00030000 },
	{ 0x2, 0x17, 0x4, 0x0, 0x0d130000, 4,  0x00040000 },
	{ 0x2, 0x17, 0x5, 0x0, 0x0d140000, 5,  0x00050000 },
	{ 0x2, 0x18, 0x0, 0x0, 0x0d020000, 0,  0x00000000 },
	{ 0x2, 0x19, 0x0, 0x0, 0x0d030000, 0,  0x00000000 },
	{ 0x2, 0x1f, 0x0, 0x0, 0x0d600000, 0,  0x00000000 },
	{ 0x2, 0x1f, 0x1, 0x0, 0x00000000, 0,  0x00000000 },
	{ 0x3, 0x1b, 0x0, 0x0, 0x40000000, 0,  0x40000000 },
	{ 0x3, 0x1b, 0x1, 0x1, 0x80000000, 1,  0x80000000 },
	{ 0x3, 0x1c, 0x0, 0x2, 0x0d640000, 0,  0x00000000 },
	{ 0x3, 0x1d, 0x0, 0x2, 0x20b00000, 8,  0x20b00000 },
	{ 0x3, 0x1d, 0x1, 0x2, 0x20800000, 7,  0x20800000 },
	{ 0x3, 0x1d, 0x2, 0x2, 0x20c00000, 9,  0x20c00000 },
	{ 0x3, 0x1d, 0x3, 0x2, 0x0d800000, 3,  0x0d800000 },
	{ 0x3, 0x1d, 0x4, 0x2, 0x20000000, 6,  0x20000000 },
	{ 0x3, 0x1d, 0x5, 0x2, 0x0c000000, 2,  0x0c000000 },
	{ 0x3, 0x1d, 0x6, 0x2, 0x21000000, 10, 0x21000000 },
	{ 0x3, 0x1d, 0x7, 0x2, 0x0e000000, 4,  0x0e000000 },
	{ 0x3, 0x1d, 0x8, 0x2, 0x22000000, 11, 0x22000000 },
	{ 0x3, 0x1d, 0x9, 0x2, 0x08000000, 1,  0x08000000 },
	{ 0x3, 0x1d, 0xa, 0x2, 0x24000000, 12, 0x24000000 },
	{ 0x3, 0x1d, 0xb, 0x2, 0x00000000, 0,  0x00000000 },
	{ 0x3, 0x1d, 0xc, 0x2, 0x28000000, 13, 0x28000000 },
	{ 0x3, 0x1d, 0xd, 0x2, 0x10000000, 5,  0x10000000 },
	{ 0x3, 0x1d, 0xe, 0x2, 0x30000000, 14, 0x30000000 },
	{ 0x3, 0x1e, 0x0, 0x2, 0x0d400000, 0,  0x0d400000 },
	{ 0x3, 0x00, 0x0, 0x2, 0x0d230000, 0,  0x00000000 },
	{ 0x3, 0x01, 0x0, 0x2, 0x0d040000, 0,  0x00000000 },
	{ 0x3, 0x02, 0x0, 0x2, 0x0d050000, 0,  0x00000000 },
	{ 0x3, 0x03, 0x0, 0x2, 0x0d000000, 0,  0x00000000 },
	{ 0x3, 0x04, 0x0, 0x2, 0x20ae0000, 3,  0x000e0000 },
	{ 0x3, 0x04, 0x1, 0x2, 0x20ac0000, 2,  0x000c0000 },
	{ 0x3, 0x04, 0x2, 0x2, 0x20a80000, 1,  0x00080000 },
	{ 0x3, 0x04, 0x3, 0x2, 0x20a00000, 0,  0x00000000 },
	{ 0x3, 0x05, 0x0, 0x2, 0x0d2a0000, 0,  0x00000000 },
	{ 0x3, 0x06, 0x0, 0x2, 0x0d290000, 0,  0x00000000 },
	{ 0x3, 0x07, 0x0, 0x2, 0x0d2c0000, 0,  0x00000000 },
	{ 0x3, 0x08, 0x0, 0x2, 0x0d0e0000, 4,  0x00080000 },
	{ 0x3, 0x08, 0x1, 0x2, 0x0d060000, 0,  0x00000000 },
	{ 0x3, 0x08, 0x2, 0x2, 0x0d080000, 1,  0x00020000 },
	{ 0x3, 0x08, 0x3, 0x2, 0x0d0a0000, 2,  0x00040000 },
	{ 0x3, 0x08, 0x4, 0x2, 0x0d0c0000, 3,  0x00060000 },
	{ 0x3, 0x09, 0x0, 0x2, 0x0d650000, 0,  0x00000000 },
	{ 0x3, 0x0a, 0x0, 0x2, 0x20af0000, 0,  0x00000000 },
	{ 0x3, 0x0b, 0x0, 0x2, 0x0d3e0000, 0,  0x00000000 },
	{ 0x3, 0x0c, 0x0, 0x2, 0x0d3d0000, 0,  0x00000000 },
	{ 0x3, 0x0d, 0x0, 0x2, 0x0d1e0000, 0,  0x00000000 },
	{ 0x3, 0x0e, 0x0, 0x2, 0x0d150000, 0,  0x00000000 },
	{ 0x3, 0x0e, 0x1, 0x2, 0x0d160000, 1,  0x00010000 },
	{ 0x3, 0x0e, 0x2, 0x2, 0x0d170000, 2,  0x00020000 },
	{ 0x3, 0x0e, 0x3, 0x2, 0x0d180000, 3,  0x00030000 },
	{ 0x3, 0x0e, 0x4, 0x2, 0x0d190000, 4,  0x00040000 },
	{ 0x3, 0x0e, 0x5, 0x2, 0x0d1a0000, 5,  0x00050000 },
	{ 0x3, 0x0e, 0x6, 0x2, 0x0d1b0000, 6,  0x00060000 },
	{ 0x3, 0x0e, 0x7, 0x2, 0x0d1c0000, 7,  0x00070000 },
	{ 0x3, 0x0e, 0x8, 0x2, 0x0d1d0000, 8,  0x00080000 },
	{ 0x3, 0x0f, 0x0, 0x2, 0x0d660000, 0,  0x00000000 },
	{ 0x3, 0x10, 0x0, 0x2, 0x0d1f0000, 0,  0x00000000 },
	{ 0x3, 0x10, 0x1, 0x2, 0x0d200000, 1,  0x00010000 },
	{ 0x3, 0x10, 0x2, 0x2, 0x0d210000, 2,  0x00020000 },
	{ 0x3, 0x10, 0x3, 0x2, 0x0d220000, 3,  0x00030000 },
	{ 0x3, 0x11, 0x0, 0x2, 0x0d240000, 0,  0x00000000 },
	{ 0x3, 0x12, 0x0, 0x2, 0x0d250000, 0,  0x00000000 },
	{ 0x3, 0x13, 0x0, 0x2, 0x0d260000, 0,  0x00000000 },
	{ 0x3, 0x14, 0x0, 0x2, 0x0d270000, 0,  0x00000000 },
	{ 0x3, 0x15, 0x0, 0x2, 0x0d2b0000, 0,  0x00000000 },
	{ 0x3, 0x16, 0x0, 0x2, 0x0d280000, 0,  0x00000000 },
	{ 0x3, 0x17, 0x0, 0x2, 0x0d0f0000, 0,  0x00000000 },
	{ 0x3, 0x17, 0x1, 0x2, 0x0d100000, 1,  0x00010000 },
	{ 0x3, 0x17, 0x2, 0x2, 0x0d110000, 2,  0x00020000 },
	{ 0x3, 0x17, 0x3, 0x2, 0x0d120000, 3,  0x00030000 },
	{ 0x3, 0x17, 0x4, 0x2, 0x0d130000, 4,  0x00040000 },
	{ 0x3, 0x17, 0x5, 0x2, 0x0d140000, 5,  0x00050000 },
	{ 0x3, 0x18, 0x0, 0x2, 0x0d020000, 0,  0x00000000 },
	{ 0x3, 0x19, 0x0, 0x2, 0x0d030000, 0,  0x00000000 },
	{ 0x3, 0x1f, 0x0, 0x2, 0x0d600000, 0,  0x00000000 },
	{ 0x3, 0x1f, 0x1, 0x0, 0x00000000, 0,  0x00000000 },
	{ 0x4, 0x1b, 0x0, 0x0, 0x40000000, 0,  0x40000000 },
	{ 0x4, 0x1b, 0x1, 0x1, 0x80000000, 1,  0x80000000 },
	{ 0x4, 0x1e, 0x0, 0x2, 0x0d400000, 0,  0x0d400000 },
	{ 0x4, 0x1e, 0x1, 0x0, 0x00000000, 0,  0x00000000 },
	{ 0x5, 0x1c, 0x0, 0x0, 0x0d640000, 0,  0x00000000 },
	{ 0x5, 0x1d, 0x0, 0x0, 0x20b00000, 8,  0x20b00000 },
	{ 0x5, 0x1d, 0x1, 0x0, 0x20800000, 7,  0x20800000 },
	{ 0x5, 0x1d, 0x2, 0x0, 0x20c00000, 9,  0x20c00000 },
	{ 0x5, 0x1d, 0x3, 0x0, 0x0d800000, 3,  0x0d800000 },
	{ 0x5, 0x1d, 0x4, 0x0, 0x20000000, 6,  0x20000000 },
	{ 0x5, 0x1d, 0x5, 0x0, 0x0c000000, 2,  0x0c000000 },
	{ 0x5, 0x1d, 0x6, 0x0, 0x21000000, 10, 0x21000000 },
	{ 0x5, 0x1d, 0x7, 0x0, 0x0e000000, 4,  0x0e000000 },
	{ 0x5, 0x1d, 0x8, 0x0, 0x22000000, 11, 0x22000000 },
	{ 0x5, 0x1d, 0x9, 0x0, 0x08000000, 1,  0x08000000 },
	{ 0x5, 0x1d, 0xa, 0x0, 0x24000000, 12, 0x24000000 },
	{ 0x5, 0x1d, 0xb, 0x0, 0x00000000, 0,  0x00000000 },
	{ 0x5, 0x1d, 0xc, 0x0, 0x28000000, 13, 0x28000000 },
	{ 0x5, 0x1d, 0xd, 0x0, 0x10000000, 5,  0x10000000 },
	{ 0x5, 0x1d, 0xe, 0x0, 0x30000000, 14, 0x30000000 },
	{ 0x5, 0x00, 0x0, 0x0, 0x0d230000, 0,  0x00000000 },
	{ 0x5, 0x01, 0x0, 0x0, 0x0d040000, 0,  0x00000000 },
	{ 0x5, 0x02, 0x0, 0x0, 0x0d050000, 0,  0x00000000 },
	{ 0x5, 0x03, 0x0, 0x0, 0x0d000000, 0,  0x00000000 },
	{ 0x5, 0x04, 0x0, 0x0, 0x20ae0000, 3,  0x000e0000 },
	{ 0x5, 0x04, 0x1, 0x0, 0x20ac0000, 2,  0x000c0000 },
	{ 0x5, 0x04, 0x2, 0x0, 0x20a80000, 1,  0x00080000 },
	{ 0x5, 0x04, 0x3, 0x0, 0x20a00000, 0,  0x00000000 },
	{ 0x5, 0x05, 0x0, 0x0, 0x0d2a0000, 0,  0x00000000 },
	{ 0x5, 0x06, 0x0, 0x0, 0x0d290000, 0,  0x00000000 },
	{ 0x5, 0x07, 0x0, 0x0, 0x0d2c0000, 0,  0x00000000 },
	{ 0x5, 0x08, 0x0, 0x0, 0x0d0e0000, 4,  0x00080000 },
	{ 0x5, 0x08, 0x1, 0x0, 0x0d060000, 0,  0x00000000 },
	{ 0x5, 0x08, 0x2, 0x0, 0x0d080000, 1,  0x00020000 },
	{ 0x5, 0x08, 0x3, 0x0, 0x0d0a0000, 2,  0x00040000 },
	{ 0x5, 0x08, 0x4, 0x0, 0x0d0c0000, 3,  0x00060000 },
	{ 0x5, 0x09, 0x0, 0x0, 0x0d650000, 0,  0x00000000 },
	{ 0x5, 0x0a, 0x0, 0x0, 0x20af0000, 0,  0x00000000 },
	{ 0x5, 0x0b, 0x0, 0x0, 0x0d3e0000, 0,  0x00000000 },
	{ 0x5, 0x0c, 0x0, 0x0, 0x0d3d0000, 0,  0x00000000 },
	{ 0x5, 0x0d, 0x0, 0x0, 0x0d1e0000, 0,  0x00000000 },
	{ 0x5, 0x0e, 0x0, 0x0, 0x0d150000, 0,  0x00000000 },
	{ 0x5, 0x0e, 0x1, 0x0, 0x0d160000, 1,  0x00010000 },
	{ 0x5, 0x0e, 0x2, 0x0, 0x0d170000, 2,  0x00020000 },
	{ 0x5, 0x0e, 0x3, 0x0, 0x0d180000, 3,  0x00030000 },
	{ 0x5, 0x0e, 0x4, 0x0, 0x0d190000, 4,  0x00040000 },
	{ 0x5, 0x0e, 0x5, 0x0, 0x0d1a0000, 5,  0x00050000 },
	{ 0x5, 0x0e, 0x6, 0x0, 0x0d1b0000, 6,  0x00060000 },
	{ 0x5, 0x0e, 0x7, 0x0, 0x0d1c0000, 7,  0x00070000 },
	{ 0x5, 0x0e, 0x8, 0x0, 0x0d1d0000, 8,  0x00080000 },
	{ 0x5, 0x0f, 0x0, 0x0, 0x0d660000, 0,  0x00000000 },
	{ 0x5, 0x10, 0x0, 0x0, 0x0d1f0000, 0,  0x00000000 },
	{ 0x5, 0x10, 0x1, 0x0, 0x0d200000, 1,  0x00010000 },
	{ 0x5, 0x10, 0x2, 0x0, 0x0d210000, 2,  0x00020000 },
	{ 0x5, 0x10, 0x3, 0x0, 0x0d220000, 3,  0x00030000 },
	{ 0x5, 0x11, 0x0, 0x0, 0x0d240000, 0,  0x00000000 },
	{ 0x5, 0x12, 0x0, 0x0, 0x0d250000, 0,  0x00000000 },
	{ 0x5, 0x13, 0x0, 0x0, 0x0d260000, 0,  0x00000000 },
	{ 0x5, 0x14, 0x0, 0x0, 0x0d270000, 0,  0x00000000 },
	{ 0x5, 0x15, 0x0, 0x0, 0x0d2b0000, 0,  0x00000000 },
	{ 0x5, 0x16, 0x0, 0x0, 0x0d280000, 0,  0x00000000 },
	{ 0x5, 0x17, 0x0, 0x0, 0x0d0f0000, 0,  0x00000000 },
	{ 0x5, 0x17, 0x1, 0x0, 0x0d100000, 1,  0x00010000 },
	{ 0x5, 0x17, 0x2, 0x0, 0x0d110000, 2,  0x00020000 },
	{ 0x5, 0x17, 0x3, 0x0, 0x0d120000, 3,  0x00030000 },
	{ 0x5, 0x17, 0x4, 0x0, 0x0d130000, 4,  0x00040000 },
	{ 0x5, 0x17, 0x5, 0x0, 0x0d140000, 5,  0x00050000 },
	{ 0x5, 0x18, 0x0, 0x0, 0x0d020000, 0,  0x00000000 },
	{ 0x5, 0x19, 0x0, 0x0, 0x0d030000, 0,  0x00000000 },
	{ 0x5, 0x1f, 0x0, 0x0, 0x0d600000, 0,  0x00000000 },
	{ 0x5, 0x1f, 0x1, 0x0, 0x00000000, 0,  0x00000000 }
};

/*
 * AON NOC aperture lookup table as per file "AON_NOC_Structure.info".
 */
static const char * const tegra194_aonnoc_routeid_initflow[] = {
	[0x0] = "cbb_i/I/0",
	[0x1] = "cpu_p_i/I/0",
	[0x2] = "dma_m_i/I/0",
	[0x3] = "dma_p_i/I/0"
};

static const char * const tegra194_aonnoc_routeid_targflow[] = {
	[0x00] = "multiport1_t/T/aon_misc",
	[0x01] = "multiport1_t/T/avic0",
	[0x02] = "multiport1_t/T/avic1",
	[0x03] = "multiport1_t/T/can1",
	[0x04] = "multiport1_t/T/can2",
	[0x05] = "multiport1_t/T/dma",
	[0x06] = "multiport1_t/T/dmic",
	[0x07] = "multiport1_t/T/err_collator",
	[0x08] = "multiport1_t/T/fpga_misc",
	[0x09] = "multiport1_t/T/gte",
	[0x0a] = "multiport1_t/T/hsp",
	[0x0b] = "multiport1_t/T/i2c2",
	[0x0c] = "multiport1_t/T/i2c8",
	[0x0d] = "multiport1_t/T/pwm",
	[0x0e] = "multiport1_t/T/spi2",
	[0x0f] = "multiport1_t/T/tke",
	[0x10] = "multiport1_t/T/uartg",
	[0x11] = "RESERVED",
	[0x12] = "RESERVED",
	[0x13] = "RESERVED",
	[0x14] = "RESERVED",
	[0x15] = "RESERVED",
	[0x16] = "RESERVED",
	[0x17] = "RESERVED",
	[0x18] = "RESERVED",
	[0x19] = "RESERVED",
	[0x1a] = "RESERVED",
	[0x1b] = "RESERVED",
	[0x1c] = "RESERVED",
	[0x1d] = "RESERVED",
	[0x1e] = "RESERVED",
	[0x1f] = "RESERVED",
	[0x20] = "multiport0_t/T/aovc",
	[0x21] = "multiport0_t/T/atcm",
	[0x22] = "multiport0_t/T/cast",
	[0x23] = "multiport0_t/T/dast",
	[0x24] = "multiport0_t/T/err_collator_car",
	[0x25] = "multiport0_t/T/gpio",
	[0x26] = "multiport0_t/T/i2c10",
	[0x27] = "multiport0_t/T/mss",
	[0x28] = "multiport0_t/T/padctl_a12",
	[0x29] = "multiport0_t/T/padctl_a14",
	[0x2a] = "multiport0_t/T/padctl_a15",
	[0x2b] = "multiport0_t/T/rtc",
	[0x2c] = "multiport0_t/T/tsc",
	[0x2d] = "RESERVED",
	[0x2e] = "RESERVED",
	[0x2f] = "RESERVED",
	[0x30] = "multiport2_t/T/aon_vref_ro",
	[0x31] = "multiport2_t/T/aopm",
	[0x32] = "multiport2_t/T/car",
	[0x33] = "multiport2_t/T/pmc",
	[0x34] = "ast1_t/T/0",
	[0x35] = "cbb_t/T/0",
	[0x36] = "cpu_t/T/0",
	[0x37] = "firewall_t/T/0",
	[0x38] = "svc_t/T/0",
	[0x39] = "uartc/T/uartc",
	[0x3a] = "RESERVED",
	[0x3b] = "RESERVED",
	[0x3c] = "RESERVED",
	[0x3d] = "RESERVED",
	[0x3e] = "RESERVED",
	[0x3f] = "RESERVED"
};

/*
 * Fields of AON NOC lookup table:
 * Init flow, Targ flow, Targ subrange, Init mapping, Init localAddress,
 *                                              Targ mapping, Targ localAddress
 * ----------------------------------------------------------------------------
 */
static const struct tegra194_cbb_aperture tegra194_aonnoc_aperture_lookup[] = {
	{ 0x0, 0x37, 0x00, 0, 0x0c640000, 0,  0x00000000 },
	{ 0x0, 0x20, 0x00, 0, 0x0c3b0000, 0,  0x00000000 },
	{ 0x0, 0x21, 0x00, 0, 0x0c000000, 0,  0x00000000 },
	{ 0x0, 0x22, 0x00, 0, 0x0c040000, 0,  0x00000000 },
	{ 0x0, 0x23, 0x00, 0, 0x0c050000, 0,  0x00000000 },
	{ 0x0, 0x24, 0x00, 0, 0x20cf0000, 0,  0x00000000 },
	{ 0x0, 0x25, 0x00, 0, 0x0c2f0000, 0,  0x00000000 },
	{ 0x0, 0x26, 0x00, 0, 0x0c230000, 0,  0x00000000 },
	{ 0x0, 0x27, 0x00, 0, 0x0c350000, 0,  0x00000000 },
	{ 0x0, 0x28, 0x00, 0, 0x0c301000, 0,  0x00000000 },
	{ 0x0, 0x29, 0x00, 0, 0x0c302000, 0,  0x00000000 },
	{ 0x0, 0x2a, 0x00, 0, 0x0c303000, 0,  0x00000000 },
	{ 0x0, 0x2b, 0x00, 0, 0x0c2a0000, 0,  0x00000000 },
	{ 0x0, 0x2c, 0x00, 0, 0x0c2b0000, 0,  0x00000000 },
	{ 0x0, 0x2c, 0x01, 0, 0x0c2c0000, 1,  0x00010000 },
	{ 0x0, 0x2c, 0x02, 0, 0x0c2d0000, 2,  0x00020000 },
	{ 0x0, 0x2c, 0x03, 0, 0x0c2e0000, 3,  0x00030000 },
	{ 0x0, 0x00, 0x00, 0, 0x0c660000, 0,  0x00000000 },
	{ 0x0, 0x01, 0x00, 0, 0x0c020000, 0,  0x00000000 },
	{ 0x0, 0x02, 0x00, 0, 0x0c030000, 0,  0x00000000 },
	{ 0x0, 0x03, 0x00, 0, 0x0c310000, 0,  0x00000000 },
	{ 0x0, 0x04, 0x00, 0, 0x0c320000, 0,  0x00000000 },
	{ 0x0, 0x05, 0x00, 0, 0x0c0a0000, 2,  0x00040000 },
	{ 0x0, 0x05, 0x01, 0, 0x0c0b0000, 3,  0x00050000 },
	{ 0x0, 0x05, 0x02, 0, 0x0c0e0000, 5,  0x00080000 },
	{ 0x0, 0x05, 0x03, 0, 0x0c060000, 0,  0x00000000 },
	{ 0x0, 0x05, 0x04, 0, 0x0c080000, 1,  0x00020000 },
	{ 0x0, 0x05, 0x05, 0, 0x0c0c0000, 4,  0x00060000 },
	{ 0x0, 0x06, 0x00, 0, 0x0c330000, 0,  0x00000000 },
	{ 0x0, 0x07, 0x00, 0, 0x0c650000, 0,  0x00000000 },
	{ 0x0, 0x08, 0x00, 0, 0x0c3e0000, 0,  0x00000000 },
	{ 0x0, 0x09, 0x00, 0, 0x0c1e0000, 0,  0x00000000 },
	{ 0x0, 0x0a, 0x00, 0, 0x0c150000, 0,  0x00000000 },
	{ 0x0, 0x0a, 0x01, 0, 0x0c160000, 1,  0x00010000 },
	{ 0x0, 0x0a, 0x02, 0, 0x0c170000, 2,  0x00020000 },
	{ 0x0, 0x0a, 0x03, 0, 0x0c180000, 3,  0x00030000 },
	{ 0x0, 0x0a, 0x04, 0, 0x0c190000, 4,  0x00040000 },
	{ 0x0, 0x0a, 0x05, 0, 0x0c1a0000, 5,  0x00050000 },
	{ 0x0, 0x0a, 0x06, 0, 0x0c1b0000, 6,  0x00060000 },
	{ 0x0, 0x0a, 0x07, 0, 0x0c1c0000, 7,  0x00070000 },
	{ 0x0, 0x0a, 0x08, 0, 0x0c1d0000, 8,  0x00080000 },
	{ 0x0, 0x0b, 0x00, 0, 0x0c240000, 0,  0x00000000 },
	{ 0x0, 0x0c, 0x00, 0, 0x0c250000, 0,  0x00000000 },
	{ 0x0, 0x0d, 0x00, 0, 0x0c340000, 0,  0x00000000 },
	{ 0x0, 0x0e, 0x00, 0, 0x0c260000, 0,  0x00000000 },
	{ 0x0, 0x0f, 0x00, 0, 0x0c0f0000, 0,  0x00000000 },
	{ 0x0, 0x0f, 0x01, 0, 0x0c100000, 1,  0x00010000 },
	{ 0x0, 0x0f, 0x02, 0, 0x0c110000, 2,  0x00020000 },
	{ 0x0, 0x0f, 0x03, 0, 0x0c120000, 3,  0x00030000 },
	{ 0x0, 0x0f, 0x04, 0, 0x0c130000, 4,  0x00040000 },
	{ 0x0, 0x0f, 0x05, 0, 0x0c140000, 5,  0x00050000 },
	{ 0x0, 0x10, 0x00, 0, 0x0c290000, 0,  0x00000000 },
	{ 0x0, 0x30, 0x00, 0, 0x20ce0000, 0,  0x00000000 },
	{ 0x0, 0x31, 0x00, 0, 0x0c1f0000, 0,  0x00000000 },
	{ 0x0, 0x31, 0x01, 0, 0x0c200000, 1,  0x00010000 },
	{ 0x0, 0x31, 0x02, 0, 0x0c210000, 2,  0x00020000 },
	{ 0x0, 0x31, 0x03, 0, 0x0c220000, 3,  0x00030000 },
	{ 0x0, 0x32, 0x00, 0, 0x20cc0000, 3,  0x001c0000 },
	{ 0x0, 0x32, 0x01, 0, 0x20c80000, 2,  0x00180000 },
	{ 0x0, 0x32, 0x02, 0, 0x20c00000, 1,  0x00100000 },
	{ 0x0, 0x32, 0x03, 0, 0x20b00000, 0,  0x00000000 },
	{ 0x0, 0x33, 0x00, 0, 0x0c360000, 0,  0x00000000 },
	{ 0x0, 0x33, 0x01, 0, 0x0c370000, 1,  0x00010000 },
	{ 0x0, 0x33, 0x02, 0, 0x0c3a0000, 3,  0x00040000 },
	{ 0x0, 0x33, 0x03, 0, 0x0c380000, 2,  0x00020000 },
	{ 0x0, 0x38, 0x00, 0, 0x0c600000, 0,  0x00000000 },
	{ 0x0, 0x38, 0x01, 0, 0x00000000, 0,  0x00000000 },
	{ 0x0, 0x39, 0x00, 0, 0x0c280000, 0,  0x00000000 },
	{ 0x1, 0x35, 0x00, 0, 0x00000000, 0,  0x00000000 },
	{ 0x1, 0x35, 0x01, 0, 0x00100000, 1,  0x00100000 },
	{ 0x1, 0x35, 0x02, 0, 0x05a00000, 11, 0x05a00000 },
	{ 0x1, 0x35, 0x03, 0, 0x05b00000, 32, 0x05b00000 },
	{ 0x1, 0x35, 0x04, 0, 0x05c00000, 33, 0x05c00000 },
	{ 0x1, 0x35, 0x05, 0, 0x05d00000, 12, 0x05d00000 },
	{ 0x1, 0x35, 0x06, 0, 0x20000000, 19, 0x20000000 },
	{ 0x1, 0x35, 0x07, 0, 0x20100000, 20, 0x20100000 },
	{ 0x1, 0x35, 0x08, 0, 0x20a00000, 24, 0x20a00000 },
	{ 0x1, 0x35, 0x09, 0, 0x20d00000, 25, 0x20d00000 },
	{ 0x1, 0x35, 0x0a, 0, 0x00200000, 2,  0x00200000 },
	{ 0x1, 0x35, 0x0b, 0, 0x05800000, 10, 0x05800000 },
	{ 0x1, 0x35, 0x0c, 0, 0x05e00000, 13, 0x05e00000 },
	{ 0x1, 0x35, 0x0d, 0, 0x20200000, 21, 0x20200000 },
	{ 0x1, 0x35, 0x0e, 0, 0x20800000, 23, 0x20800000 },
	{ 0x1, 0x35, 0x0f, 0, 0x20e00000, 26, 0x20e00000 },
	{ 0x1, 0x35, 0x10, 0, 0x00400000, 3,  0x00400000 },
	{ 0x1, 0x35, 0x11, 0, 0x20400000, 22, 0x20400000 },
	{ 0x1, 0x35, 0x12, 0, 0x00800000, 4,  0x00800000 },
	{ 0x1, 0x35, 0x13, 0, 0x05000000, 9,  0x05000000 },
	{ 0x1, 0x35, 0x14, 0, 0x0c800000, 34, 0x0c800000 },
	{ 0x1, 0x35, 0x15, 0, 0x01000000, 5,  0x01000000 },
	{ 0x1, 0x35, 0x16, 0, 0x03000000, 7,  0x03000000 },
	{ 0x1, 0x35, 0x17, 0, 0x04000000, 8,  0x04000000 },
	{ 0x1, 0x35, 0x18, 0, 0x0d000000, 16, 0x0d000000 },
	{ 0x1, 0x35, 0x19, 0, 0x21000000, 27, 0x21000000 },
	{ 0x1, 0x35, 0x1a, 0, 0x02000000, 6,  0x02000000 },
	{ 0x1, 0x35, 0x1b, 0, 0x06000000, 14, 0x06000000 },
	{ 0x1, 0x35, 0x1c, 0, 0x0e000000, 17, 0x0e000000 },
	{ 0x1, 0x35, 0x1d, 0, 0x22000000, 28, 0x22000000 },
	{ 0x1, 0x35, 0x1e, 0, 0x08000000, 15, 0x08000000 },
	{ 0x1, 0x35, 0x1f, 0, 0x24000000, 29, 0x24000000 },
	{ 0x1, 0x35, 0x20, 0, 0x28000000, 30, 0x28000000 },
	{ 0x1, 0x35, 0x21, 0, 0x10000000, 18, 0x10000000 },
	{ 0x1, 0x35, 0x22, 0, 0x30000000, 31, 0x30000000 },
	{ 0x1, 0x37, 0x00, 0, 0x0c640000, 0,  0x00000000 },
	{ 0x1, 0x20, 0x00, 0, 0x0c3b0000, 0,  0x00000000 },
	{ 0x1, 0x21, 0x00, 0, 0x0c000000, 0,  0x00000000 },
	{ 0x1, 0x22, 0x00, 0, 0x0c040000, 0,  0x00000000 },
	{ 0x1, 0x23, 0x00, 0, 0x0c050000, 0,  0x00000000 },
	{ 0x1, 0x24, 0x00, 0, 0x20cf0000, 0,  0x00000000 },
	{ 0x1, 0x25, 0x00, 0, 0x0c2f0000, 0,  0x00000000 },
	{ 0x1, 0x26, 0x00, 0, 0x0c230000, 0,  0x00000000 },
	{ 0x1, 0x27, 0x00, 0, 0x0c350000, 0,  0x00000000 },
	{ 0x1, 0x28, 0x00, 0, 0x0c301000, 0,  0x00000000 },
	{ 0x1, 0x29, 0x00, 0, 0x0c302000, 0,  0x00000000 },
	{ 0x1, 0x2a, 0x00, 0, 0x0c303000, 0,  0x00000000 },
	{ 0x1, 0x2b, 0x00, 0, 0x0c2a0000, 0,  0x00000000 },
	{ 0x1, 0x2c, 0x00, 0, 0x0c2b0000, 0,  0x00000000 },
	{ 0x1, 0x2c, 0x01, 0, 0x0c2c0000, 1,  0x00010000 },
	{ 0x1, 0x2c, 0x02, 0, 0x0c2d0000, 2,  0x00020000 },
	{ 0x1, 0x2c, 0x03, 0, 0x0c2e0000, 3,  0x00030000 },
	{ 0x1, 0x00, 0x00, 0, 0x0c660000, 0,  0x00000000 },
	{ 0x1, 0x01, 0x00, 0, 0x0c020000, 0,  0x00000000 },
	{ 0x1, 0x02, 0x00, 0, 0x0c030000, 0,  0x00000000 },
	{ 0x1, 0x03, 0x00, 0, 0x0c310000, 0,  0x00000000 },
	{ 0x1, 0x04, 0x00, 0, 0x0c320000, 0,  0x00000000 },
	{ 0x1, 0x05, 0x00, 0, 0x0c0a0000, 2,  0x00040000 },
	{ 0x1, 0x05, 0x01, 0, 0x0c0b0000, 3,  0x00050000 },
	{ 0x1, 0x05, 0x02, 0, 0x0c0e0000, 5,  0x00080000 },
	{ 0x1, 0x05, 0x03, 0, 0x0c060000, 0,  0x00000000 },
	{ 0x1, 0x05, 0x04, 0, 0x0c080000, 1,  0x00020000 },
	{ 0x1, 0x05, 0x05, 0, 0x0c0c0000, 4,  0x00060000 },
	{ 0x1, 0x06, 0x00, 0, 0x0c330000, 0,  0x00000000 },
	{ 0x1, 0x07, 0x00, 0, 0x0c650000, 0,  0x00000000 },
	{ 0x1, 0x08, 0x00, 0, 0x0c3e0000, 0,  0x00000000 },
	{ 0x1, 0x09, 0x00, 0, 0x0c1e0000, 0,  0x00000000 },
	{ 0x1, 0x0a, 0x00, 0, 0x0c150000, 0,  0x00000000 },
	{ 0x1, 0x0a, 0x01, 0, 0x0c160000, 1,  0x00010000 },
	{ 0x1, 0x0a, 0x02, 0, 0x0c170000, 2,  0x00020000 },
	{ 0x1, 0x0a, 0x03, 0, 0x0c180000, 3,  0x00030000 },
	{ 0x1, 0x0a, 0x04, 0, 0x0c190000, 4,  0x00040000 },
	{ 0x1, 0x0a, 0x05, 0, 0x0c1a0000, 5,  0x00050000 },
	{ 0x1, 0x0a, 0x06, 0, 0x0c1b0000, 6,  0x00060000 },
	{ 0x1, 0x0a, 0x07, 0, 0x0c1c0000, 7,  0x00070000 },
	{ 0x1, 0x0a, 0x08, 0, 0x0c1d0000, 8,  0x00080000 },
	{ 0x1, 0x0b, 0x00, 0, 0x0c240000, 0,  0x00000000 },
	{ 0x1, 0x0c, 0x00, 0, 0x0c250000, 0,  0x00000000 },
	{ 0x1, 0x0d, 0x00, 0, 0x0c340000, 0,  0x00000000 },
	{ 0x1, 0x0e, 0x00, 0, 0x0c260000, 0,  0x00000000 },
	{ 0x1, 0x0f, 0x00, 0, 0x0c0f0000, 0,  0x00000000 },
	{ 0x1, 0x0f, 0x01, 0, 0x0c100000, 1,  0x00010000 },
	{ 0x1, 0x0f, 0x02, 0, 0x0c110000, 2,  0x00020000 },
	{ 0x1, 0x0f, 0x03, 0, 0x0c120000, 3,  0x00030000 },
	{ 0x1, 0x0f, 0x04, 0, 0x0c130000, 4,  0x00040000 },
	{ 0x1, 0x0f, 0x05, 0, 0x0c140000, 5,  0x00050000 },
	{ 0x1, 0x10, 0x00, 0, 0x0c290000, 0,  0x00000000 },
	{ 0x1, 0x30, 0x00, 0, 0x20ce0000, 0,  0x00000000 },
	{ 0x1, 0x31, 0x00, 0, 0x0c1f0000, 0,  0x00000000 },
	{ 0x1, 0x31, 0x01, 0, 0x0c200000, 1,  0x00010000 },
	{ 0x1, 0x31, 0x02, 0, 0x0c210000, 2,  0x00020000 },
	{ 0x1, 0x31, 0x03, 0, 0x0c220000, 3,  0x00030000 },
	{ 0x1, 0x32, 0x00, 0, 0x20cc0000, 3,  0x001c0000 },
	{ 0x1, 0x32, 0x01, 0, 0x20c80000, 2,  0x00180000 },
	{ 0x1, 0x32, 0x02, 0, 0x20c00000, 1,  0x00100000 },
	{ 0x1, 0x32, 0x03, 0, 0x20b00000, 0,  0x00000000 },
	{ 0x1, 0x33, 0x00, 0, 0x0c360000, 0,  0x00000000 },
	{ 0x1, 0x33, 0x01, 0, 0x0c370000, 1,  0x00010000 },
	{ 0x1, 0x33, 0x02, 0, 0x0c3a0000, 3,  0x00040000 },
	{ 0x1, 0x33, 0x03, 0, 0x0c380000, 2,  0x00020000 },
	{ 0x1, 0x38, 0x00, 0, 0x0c600000, 0,  0x00000000 },
	{ 0x1, 0x38, 0x01, 0, 0x00000000, 0,  0x00000000 },
	{ 0x1, 0x39, 0x00, 0, 0x0c280000, 0,  0x00000000 },
	{ 0x2, 0x34, 0x00, 0, 0x40000000, 0,  0x40000000 },
	{ 0x2, 0x34, 0x01, 0, 0x80000000, 1,  0x80000000 },
	{ 0x2, 0x36, 0x00, 0, 0x0c400000, 0,  0x0c400000 },
	{ 0x2, 0x36, 0x01, 0, 0x00000000, 0,  0x00000000 },
	{ 0x3, 0x35, 0x00, 0, 0x00000000, 0,  0x00000000 },
	{ 0x3, 0x35, 0x01, 0, 0x00100000, 1,  0x00100000 },
	{ 0x3, 0x35, 0x02, 0, 0x05a00000, 11, 0x05a00000 },
	{ 0x3, 0x35, 0x03, 0, 0x05b00000, 32, 0x05b00000 },
	{ 0x3, 0x35, 0x04, 0, 0x05c00000, 33, 0x05c00000 },
	{ 0x3, 0x35, 0x05, 0, 0x05d00000, 12, 0x05d00000 },
	{ 0x3, 0x35, 0x06, 0, 0x20000000, 19, 0x20000000 },
	{ 0x3, 0x35, 0x07, 0, 0x20100000, 20, 0x20100000 },
	{ 0x3, 0x35, 0x08, 0, 0x20a00000, 24, 0x20a00000 },
	{ 0x3, 0x35, 0x09, 0, 0x20d00000, 25, 0x20d00000 },
	{ 0x3, 0x35, 0x0a, 0, 0x00200000, 2,  0x00200000 },
	{ 0x3, 0x35, 0x0b, 0, 0x05800000, 10, 0x05800000 },
	{ 0x3, 0x35, 0x0c, 0, 0x05e00000, 13, 0x05e00000 },
	{ 0x3, 0x35, 0x0d, 0, 0x20200000, 21, 0x20200000 },
	{ 0x3, 0x35, 0x0e, 0, 0x20800000, 23, 0x20800000 },
	{ 0x3, 0x35, 0x0f, 0, 0x20e00000, 26, 0x20e00000 },
	{ 0x3, 0x35, 0x10, 0, 0x00400000, 3,  0x00400000 },
	{ 0x3, 0x35, 0x11, 0, 0x20400000, 22, 0x20400000 },
	{ 0x3, 0x35, 0x12, 0, 0x00800000, 4,  0x00800000 },
	{ 0x3, 0x35, 0x13, 0, 0x50000000, 9,  0x05000000 },
	{ 0x3, 0x35, 0x14, 0, 0xc0800000, 34, 0x0c800000 },
	{ 0x3, 0x35, 0x15, 0, 0x10000000, 5,  0x01000000 },
	{ 0x3, 0x35, 0x16, 0, 0x30000000, 7,  0x03000000 },
	{ 0x3, 0x35, 0x17, 0, 0x04000000, 8,  0x04000000 },
	{ 0x3, 0x35, 0x18, 0, 0x0d000000, 16, 0x0d000000 },
	{ 0x3, 0x35, 0x19, 0, 0x21000000, 27, 0x21000000 },
	{ 0x3, 0x35, 0x1a, 0, 0x02000000, 6,  0x02000000 },
	{ 0x3, 0x35, 0x1b, 0, 0x06000000, 14, 0x06000000 },
	{ 0x3, 0x35, 0x1c, 0, 0x0e000000, 17, 0x0e000000 },
	{ 0x3, 0x35, 0x1d, 0, 0x22000000, 28, 0x22000000 },
	{ 0x3, 0x35, 0x1e, 0, 0x08000000, 15, 0x08000000 },
	{ 0x3, 0x35, 0x1f, 0, 0x24000000, 29, 0x24000000 },
	{ 0x3, 0x35, 0x20, 0, 0x28000000, 30, 0x28000000 },
	{ 0x3, 0x35, 0x21, 0, 0x10000000, 18, 0x10000000 },
	{ 0x3, 0x35, 0x22, 0, 0x30000000, 31, 0x30000000 },
	{ 0x3, 0x37, 0x00, 0, 0x0c640000, 0,  0x00000000 },
	{ 0x3, 0x20, 0x00, 0, 0x0c3b0000, 0,  0x00000000 },
	{ 0x3, 0x21, 0x00, 0, 0x0c000000, 0,  0x00000000 },
	{ 0x3, 0x22, 0x00, 0, 0x0c040000, 0,  0x00000000 },
	{ 0x3, 0x23, 0x00, 0, 0x0c050000, 0,  0x00000000 },
	{ 0x3, 0x24, 0x00, 0, 0x20cf0000, 0,  0x00000000 },
	{ 0x3, 0x25, 0x00, 0, 0x0c2f0000, 0,  0x00000000 },
	{ 0x3, 0x26, 0x00, 0, 0x0c230000, 0,  0x00000000 },
	{ 0x3, 0x27, 0x00, 0, 0x0c350000, 0,  0x00000000 },
	{ 0x3, 0x28, 0x00, 0, 0x0c301000, 0,  0x00000000 },
	{ 0x3, 0x29, 0x00, 0, 0x0c302000, 0,  0x00000000 },
	{ 0x3, 0x2a, 0x00, 0, 0x0c303000, 0,  0x00000000 },
	{ 0x3, 0x2b, 0x00, 0, 0x0c2a0000, 0,  0x00000000 },
	{ 0x3, 0x2c, 0x00, 0, 0x0c2b0000, 0,  0x00000000 },
	{ 0x3, 0x2c, 0x01, 0, 0x0c2c0000, 1,  0x00010000 },
	{ 0x3, 0x2c, 0x02, 0, 0x0c2d0000, 2,  0x00020000 },
	{ 0x3, 0x2c, 0x03, 0, 0x0c2e0000, 3,  0x00030000 },
	{ 0x3, 0x00, 0x00, 0, 0x0c660000, 0,  0x00000000 },
	{ 0x3, 0x01, 0x00, 0, 0x0c020000, 0,  0x00000000 },
	{ 0x3, 0x02, 0x00, 0, 0x0c030000, 0,  0x00000000 },
	{ 0x3, 0x03, 0x00, 0, 0x0c310000, 0,  0x00000000 },
	{ 0x3, 0x04, 0x00, 0, 0x0c320000, 0,  0x00000000 },
	{ 0x3, 0x05, 0x00, 0, 0x0c0a0000, 2,  0x00040000 },
	{ 0x3, 0x05, 0x01, 0, 0x0c0b0000, 3,  0x00050000 },
	{ 0x3, 0x05, 0x02, 0, 0x0c0e0000, 5,  0x00080000 },
	{ 0x3, 0x05, 0x03, 0, 0x0c060000, 0,  0x00000000 },
	{ 0x3, 0x05, 0x04, 0, 0x0c080000, 1,  0x00020000 },
	{ 0x3, 0x05, 0x05, 0, 0x0c0c0000, 4,  0x00060000 },
	{ 0x3, 0x06, 0x00, 0, 0x0c330000, 0,  0x00000000 },
	{ 0x3, 0x07, 0x00, 0, 0x0c650000, 0,  0x00000000 },
	{ 0x3, 0x08, 0x00, 0, 0x0c3e0000, 0,  0x00000000 },
	{ 0x3, 0x09, 0x00, 0, 0x0c1e0000, 0,  0x00000000 },
	{ 0x3, 0x0a, 0x00, 0, 0x0c150000, 0,  0x00000000 },
	{ 0x3, 0x0a, 0x01, 0, 0x0c160000, 1,  0x00010000 },
	{ 0x3, 0x0a, 0x02, 0, 0x0c170000, 2,  0x00020000 },
	{ 0x3, 0x0a, 0x03, 0, 0x0c180000, 3,  0x00030000 },
	{ 0x3, 0x0a, 0x04, 0, 0x0c190000, 4,  0x00040000 },
	{ 0x3, 0x0a, 0x05, 0, 0x0c1a0000, 5,  0x00050000 },
	{ 0x3, 0x0a, 0x06, 0, 0x0c1b0000, 6,  0x00060000 },
	{ 0x3, 0x0a, 0x07, 0, 0x0c1c0000, 7,  0x00070000 },
	{ 0x3, 0x0a, 0x08, 0, 0x0c1d0000, 8,  0x00080000 },
	{ 0x3, 0x0b, 0x00, 0, 0x0c240000, 0,  0x00000000 },
	{ 0x3, 0x0c, 0x00, 0, 0x0c250000, 0,  0x00000000 },
	{ 0x3, 0x0d, 0x00, 0, 0x0c340000, 0,  0x00000000 },
	{ 0x3, 0x0e, 0x00, 0, 0x0c260000, 0,  0x00000000 },
	{ 0x3, 0x0f, 0x00, 0, 0x0c0f0000, 0,  0x00000000 },
	{ 0x3, 0x0f, 0x01, 0, 0x0c100000, 1,  0x00010000 },
	{ 0x3, 0x0f, 0x02, 0, 0x0c110000, 2,  0x00020000 },
	{ 0x3, 0x0f, 0x03, 0, 0x0c120000, 3,  0x00030000 },
	{ 0x3, 0x0f, 0x04, 0, 0x0c130000, 4,  0x00040000 },
	{ 0x3, 0x0f, 0x05, 0, 0x0c140000, 5,  0x00050000 },
	{ 0x3, 0x10, 0x00, 0, 0x0c290000, 0,  0x00000000 },
	{ 0x3, 0x30, 0x00, 0, 0x20ce0000, 0,  0x00000000 },
	{ 0x3, 0x31, 0x00, 0, 0x0c1f0000, 0,  0x00000000 },
	{ 0x3, 0x31, 0x01, 0, 0x0c200000, 1,  0x00010000 },
	{ 0x3, 0x31, 0x02, 0, 0x0c210000, 2,  0x00020000 },
	{ 0x3, 0x31, 0x03, 0, 0x0c220000, 3,  0x00030000 },
	{ 0x3, 0x32, 0x00, 0, 0x20cc0000, 3,  0x001c0000 },
	{ 0x3, 0x32, 0x01, 0, 0x20c80000, 2,  0x00180000 },
	{ 0x3, 0x32, 0x02, 0, 0x20c00000, 1,  0x00100000 },
	{ 0x3, 0x32, 0x03, 0, 0x20b00000, 0,  0x00000000 },
	{ 0x3, 0x33, 0x00, 0, 0x0c360000, 0,  0x00000000 },
	{ 0x3, 0x33, 0x01, 0, 0x0c370000, 1,  0x00010000 },
	{ 0x3, 0x33, 0x02, 0, 0x0c3a0000, 3,  0x00040000 },
	{ 0x3, 0x33, 0x03, 0, 0x0c380000, 2,  0x00020000 },
	{ 0x3, 0x38, 0x00, 0, 0x0c600000, 0,  0x00000000 },
	{ 0x3, 0x38, 0x01, 0, 0x00000000, 0,  0x00000000 },
	{ 0x3, 0x39, 0x00, 0, 0x0c280000, 0,  0x00000000 }
};

/*
 * SCE/RCE NOC aperture lookup table as per file "AON_NOC_Structure.info".
 */
static const char * const tegra194_scenoc_routeid_initflow[] = {
	[0x0] = "cbb_i/I/0",
	[0x1] = "cpu_m_i/I/0",
	[0x2] = "cpu_p_i/I/0",
	[0x3] = "dma_m_i/I/0",
	[0x4] = "dma_p_i/I/0",
	[0x5] = "RESERVED",
	[0x6] = "RESERVED",
	[0x7] = "RESERVED"
};

static const char * const tegra194_scenoc_routeid_targflow[] = {
	[0x00] = "multiport0_t/T/atcm_cfg",
	[0x01] = "multiport0_t/T/car",
	[0x02] = "multiport0_t/T/cast",
	[0x03] = "multiport0_t/T/cfg",
	[0x04] = "multiport0_t/T/dast",
	[0x05] = "multiport0_t/T/dma",
	[0x06] = "multiport0_t/T/err_collator",
	[0x07] = "multiport0_t/T/err_collator_car",
	[0x08] = "multiport0_t/T/fpga_misc",
	[0x09] = "multiport0_t/T/fpga_uart",
	[0x0a] = "multiport0_t/T/gte",
	[0x0b] = "multiport0_t/T/hsp",
	[0x0c] = "multiport0_t/T/misc",
	[0x0d] = "multiport0_t/T/pm",
	[0x0e] = "multiport0_t/T/tke",
	[0x0f] = "RESERVED",
	[0x10] = "multiport1_t/T/hsm",
	[0x11] = "multiport1_t/T/vic0",
	[0x12] = "multiport1_t/T/vic1",
	[0x13] = "ast0_t/T/0",
	[0x14] = "ast1_t/T/0",
	[0x15] = "cbb_t/T/0",
	[0x16] = "cpu_t/T/0",
	[0x17] = "sce_noc_firewall/T/0",
	[0x18] = "svc_t/T/0",
	[0x19] = "RESERVED",
	[0x1a] = "RESERVED",
	[0x1b] = "RESERVED",
	[0x1c] = "RESERVED",
	[0x1d] = "RESERVED",
	[0x1e] = "RESERVED",
	[0x1f] = "RESERVED"
};

/*
 * Fields of SCE/RCE NOC lookup table:
 * Init flow, Targ flow, Targ subrange, Init mapping, Init localAddress,
 *                                              Targ mapping, Targ localAddress
 * ----------------------------------------------------------------------------
 */
static const struct tegra194_cbb_aperture tegra194_scenoc_apert_lookup[] = {
	{ 0x0, 0x16, 0x0,  0, 0x0b400000, 0,  0x0b400000 },
	{ 0x0, 0x16, 0x1,  0, 0x0bc00000, 1,  0x0bc00000 },
	{ 0x0, 0x0,  0x0,  0, 0x0b000000, 0,  0x00000000 },
	{ 0x0, 0x0,  0x1,  0, 0x0b800000, 1,  0x00000000 },
	{ 0x0, 0x1,  0x0,  0, 0x20de0000, 3,  0x000e0000 },
	{ 0x0, 0x1,  0x1,  0, 0x210e0000, 7,  0x000e0000 },
	{ 0x0, 0x1,  0x2,  0, 0x20dc0000, 2,  0x000c0000 },
	{ 0x0, 0x1,  0x3,  0, 0x210c0000, 6,  0x000c0000 },
	{ 0x0, 0x1,  0x4,  0, 0x20d80000, 1,  0x00080000 },
	{ 0x0, 0x1,  0x5,  0, 0x21080000, 5,  0x00080000 },
	{ 0x0, 0x1,  0x6,  0, 0x20d00000, 0,  0x00000000 },
	{ 0x0, 0x1,  0x7,  0, 0x21000000, 4,  0x00000000 },
	{ 0x0, 0x2,  0x0,  0, 0x0b040000, 0,  0x00000000 },
	{ 0x0, 0x2,  0x1,  0, 0x0b840000, 1,  0x00000000 },
	{ 0x0, 0x3,  0x0,  0, 0x0b230000, 0,  0x00000000 },
	{ 0x0, 0x3,  0x1,  0, 0x0ba30000, 1,  0x00000000 },
	{ 0x0, 0x4,  0x0,  0, 0x0b050000, 0,  0x00000000 },
	{ 0x0, 0x4,  0x1,  0, 0x0b850000, 1,  0x00000000 },
	{ 0x0, 0x5,  0x0,  0, 0x0b060000, 0,  0x00000000 },
	{ 0x0, 0x5,  0x1,  0, 0x0b070000, 1,  0x00010000 },
	{ 0x0, 0x5,  0x2,  0, 0x0b080000, 2,  0x00020000 },
	{ 0x0, 0x5,  0x3,  0, 0x0b090000, 3,  0x00030000 },
	{ 0x0, 0x5,  0x4,  0, 0x0b0a0000, 4,  0x00040000 },
	{ 0x0, 0x5,  0x5,  0, 0x0b0b0000, 5,  0x00050000 },
	{ 0x0, 0x5,  0x6,  0, 0x0b0c0000, 6,  0x00060000 },
	{ 0x0, 0x5,  0x7,  0, 0x0b0d0000, 7,  0x00070000 },
	{ 0x0, 0x5,  0x8,  0, 0x0b0e0000, 8,  0x00080000 },
	{ 0x0, 0x5,  0x9,  0, 0x0b860000, 9,  0x00000000 },
	{ 0x0, 0x5,  0xa,  0, 0x0b870000, 10, 0x00010000 },
	{ 0x0, 0x5,  0xb,  0, 0x0b880000, 11, 0x00020000 },
	{ 0x0, 0x5,  0xc,  0, 0x0b890000, 12, 0x00030000 },
	{ 0x0, 0x5,  0xd,  0, 0x0b8a0000, 13, 0x00040000 },
	{ 0x0, 0x5,  0xe,  0, 0x0b8b0000, 14, 0x00050000 },
	{ 0x0, 0x5,  0xf,  0, 0x0b8c0000, 15, 0x00060000 },
	{ 0x0, 0x5,  0x10, 0, 0x0b8d0000, 16, 0x00070000 },
	{ 0x0, 0x5,  0x11, 0, 0x0b8e0000, 17, 0x00080000 },
	{ 0x0, 0x6,  0x0,  0, 0x0b650000, 0,  0x00000000 },
	{ 0x0, 0x6,  0x1,  0, 0x0be50000, 1,  0x00000000 },
	{ 0x0, 0x7,  0x0,  0, 0x20df0000, 0,  0x00000000 },
	{ 0x0, 0x7,  0x1,  0, 0x210f0000, 1,  0x00000000 },
	{ 0x0, 0x8,  0x0,  0, 0x0b3e0000, 0,  0x00000000 },
	{ 0x0, 0x8,  0x1,  0, 0x0bbe0000, 1,  0x00000000 },
	{ 0x0, 0x9,  0x0,  0, 0x0b3d0000, 0,  0x00000000 },
	{ 0x0, 0x9,  0x1,  0, 0x0bbd0000, 1,  0x00000000 },
	{ 0x0, 0xa,  0x0,  0, 0x0b1e0000, 0,  0x00000000 },
	{ 0x0, 0xa,  0x1,  0, 0x0b9e0000, 1,  0x00000000 },
	{ 0x0, 0xb,  0x0,  0, 0x0b150000, 0,  0x00000000 },
	{ 0x0, 0xb,  0x1,  0, 0x0b160000, 1,  0x00010000 },
	{ 0x0, 0xb,  0x2,  0, 0x0b170000, 2,  0x00020000 },
	{ 0x0, 0xb,  0x3,  0, 0x0b180000, 3,  0x00030000 },
	{ 0x0, 0xb,  0x4,  0, 0x0b190000, 4,  0x00040000 },
	{ 0x0, 0xb,  0x5,  0, 0x0b1a0000, 5,  0x00050000 },
	{ 0x0, 0xb,  0x6,  0, 0x0b1b0000, 6,  0x00060000 },
	{ 0x0, 0xb,  0x7,  0, 0x0b1c0000, 7,  0x00070000 },
	{ 0x0, 0xb,  0x8,  0, 0x0b1d0000, 8,  0x00080000 },
	{ 0x0, 0xb,  0x9,  0, 0x0b950000, 9,  0x00000000 },
	{ 0x0, 0xb,  0xa,  0, 0x0b960000, 10, 0x00010000 },
	{ 0x0, 0xb,  0xb,  0, 0x0b970000, 11, 0x00020000 },
	{ 0x0, 0xb,  0xc,  0, 0x0b980000, 12, 0x00030000 },
	{ 0x0, 0xb,  0xd,  0, 0x0b990000, 13, 0x00040000 },
	{ 0x0, 0xb,  0xe,  0, 0x0b9a0000, 14, 0x00050000 },
	{ 0x0, 0xb,  0xf,  0, 0x0b9b0000, 15, 0x00060000 },
	{ 0x0, 0xb,  0x10, 0, 0x0b9c0000, 16, 0x00070000 },
	{ 0x0, 0xb,  0x11, 0, 0x0b9d0000, 17, 0x00080000 },
	{ 0x0, 0xc,  0x0,  0, 0x0b660000, 0,  0x00000000 },
	{ 0x0, 0xc,  0x1,  0, 0x0be60000, 1,  0x00000000 },
	{ 0x0, 0xd,  0x0,  0, 0x0b1f0000, 0,  0x00000000 },
	{ 0x0, 0xd,  0x1,  0, 0x0b200000, 1,  0x00010000 },
	{ 0x0, 0xd,  0x2,  0, 0x0b210000, 2,  0x00020000 },
	{ 0x0, 0xd,  0x3,  0, 0x0b220000, 3,  0x00030000 },
	{ 0x0, 0xd,  0x4,  0, 0x0b9f0000, 4,  0x00000000 },
	{ 0x0, 0xd,  0x5,  0, 0x0ba00000, 5,  0x00010000 },
	{ 0x0, 0xd,  0x6,  0, 0x0ba10000, 6,  0x00020000 },
	{ 0x0, 0xd,  0x7,  0, 0x0ba20000, 7,  0x00030000 },
	{ 0x0, 0xe,  0x0,  0, 0x0b0f0000, 0,  0x00000000 },
	{ 0x0, 0xe,  0x1,  0, 0x0b100000, 1,  0x00010000 },
	{ 0x0, 0xe,  0x2,  0, 0x0b110000, 2,  0x00020000 },
	{ 0x0, 0xe,  0x3,  0, 0x0b120000, 3,  0x00030000 },
	{ 0x0, 0xe,  0x4,  0, 0x0b130000, 4,  0x00040000 },
	{ 0x0, 0xe,  0x5,  0, 0x0b140000, 5,  0x00050000 },
	{ 0x0, 0xe,  0x6,  0, 0x0b8f0000, 6,  0x00000000 },
	{ 0x0, 0xe,  0x7,  0, 0x0b900000, 7,  0x00010000 },
	{ 0x0, 0xe,  0x8,  0, 0x0b910000, 8,  0x00020000 },
	{ 0x0, 0xe,  0x9,  0, 0x0b920000, 9,  0x00030000 },
	{ 0x0, 0xe,  0xa,  0, 0x0b930000, 10, 0x00040000 },
	{ 0x0, 0xe,  0xb,  0, 0x0b940000, 11, 0x00050000 },
	{ 0x0, 0x10, 0x0,  0, 0x0b240000, 0,  0x00000000 },
	{ 0x0, 0x10, 0x1,  0, 0x0ba40000, 1,  0x00000000 },
	{ 0x0, 0x11, 0x0,  0, 0x0b020000, 0,  0x00000000 },
	{ 0x0, 0x11, 0x1,  0, 0x0b820000, 1,  0x00000000 },
	{ 0x0, 0x12, 0x0,  0, 0x0b030000, 0,  0x00000000 },
	{ 0x0, 0x12, 0x1,  0, 0x0b830000, 1,  0x00000000 },
	{ 0x0, 0x17, 0x0,  0, 0x0b640000, 0,  0x00000000 },
	{ 0x0, 0x17, 0x1,  0, 0x0be40000, 1,  0x00000000 },
	{ 0x0, 0x18, 0x0,  0, 0x0b600000, 0,  0x00000000 },
	{ 0x0, 0x18, 0x1,  0, 0x0be00000, 1,  0x00000000 },
	{ 0x0, 0x18, 0x2,  0, 0x00000000, 0,  0x00000000 },
	{ 0x0, 0x18, 0x3,  0, 0x00000000, 0,  0x00000000 },
	{ 0x1, 0x13, 0x0,  0, 0x40000000, 0,  0x40000000 },
	{ 0x1, 0x13, 0x1,  1, 0x80000000, 1,  0x80000000 },
	{ 0x1, 0x13, 0x2,  0, 0x00000000, 0,  0x00000000 },
	{ 0x2, 0x15, 0x0,  0, 0x20c00000, 8,  0x20c00000 },
	{ 0x2, 0x15, 0x1,  0, 0x21100000, 22, 0x21100000 },
	{ 0x2, 0x15, 0x2,  0, 0x20e00000, 9,  0x20e00000 },
	{ 0x2, 0x15, 0x3,  0, 0x21200000, 23, 0x21200000 },
	{ 0x2, 0x15, 0x4,  0, 0x20800000, 7,  0x20800000 },
	{ 0x2, 0x15, 0x5,  0, 0x21400000, 24, 0x21400000 },
	{ 0x2, 0x15, 0x6,  0, 0x0b000000, 18, 0x0b000000 },
	{ 0x2, 0x15, 0x7,  0, 0x0b800000, 3,  0x0b800000 },
	{ 0x2, 0x15, 0x8,  0, 0x20000000, 6,  0x20000000 },
	{ 0x2, 0x15, 0x9,  0, 0x21800000, 25, 0x21800000 },
	{ 0x2, 0x15, 0xa,  0, 0x0a000000, 2,  0x0a000000 },
	{ 0x2, 0x15, 0xb,  0, 0x0a000000, 17, 0x0a000000 },
	{ 0x2, 0x15, 0xc,  0, 0x20000000, 21, 0x20000000 },
	{ 0x2, 0x15, 0xd,  0, 0x21000000, 10, 0x21000000 },
	{ 0x2, 0x15, 0xe,  0, 0x08000000, 1,  0x08000000 },
	{ 0x2, 0x15, 0xf,  0, 0x08000000, 16, 0x08000000 },
	{ 0x2, 0x15, 0x10, 0, 0x22000000, 11, 0x22000000 },
	{ 0x2, 0x15, 0x11, 0, 0x22000000, 26, 0x22000000 },
	{ 0x2, 0x15, 0x12, 0, 0x0c000000, 4,  0x0c000000 },
	{ 0x2, 0x15, 0x13, 0, 0x0c000000, 19, 0x0c000000 },
	{ 0x2, 0x15, 0x14, 0, 0x24000000, 12, 0x24000000 },
	{ 0x2, 0x15, 0x15, 0, 0x24000000, 27, 0x24000000 },
	{ 0x2, 0x15, 0x16, 0, 0x00000000, 0,  0x00000000 },
	{ 0x2, 0x15, 0x17, 0, 0x00000000, 15, 0x00000000 },
	{ 0x2, 0x15, 0x18, 0, 0x28000000, 13, 0x28000000 },
	{ 0x2, 0x15, 0x19, 0, 0x28000000, 28, 0x28000000 },
	{ 0x2, 0x15, 0x1a, 0, 0x10000000, 5,  0x10000000 },
	{ 0x2, 0x15, 0x1b, 0, 0x10000000, 20, 0x10000000 },
	{ 0x2, 0x15, 0x1c, 0, 0x30000000, 14, 0x30000000 },
	{ 0x2, 0x15, 0x1d, 0, 0x30000000, 29, 0x30000000 },
	{ 0x2, 0x0,  0x0,  0, 0x0b000000, 0,  0x00000000 },
	{ 0x2, 0x0,  0x1,  0, 0x0b800000, 1,  0x00000000 },
	{ 0x2, 0x1,  0x0,  0, 0x20de0000, 3,  0x000e0000 },
	{ 0x2, 0x1,  0x1,  0, 0x210e0000, 7,  0x000e0000 },
	{ 0x2, 0x1,  0x2,  0, 0x20dc0000, 2,  0x000c0000 },
	{ 0x2, 0x1,  0x3,  0, 0x210c0000, 6,  0x000c0000 },
	{ 0x2, 0x1,  0x4,  0, 0x20d80000, 1,  0x00080000 },
	{ 0x2, 0x1,  0x5,  0, 0x21080000, 5,  0x00080000 },
	{ 0x2, 0x1,  0x6,  0, 0x20d00000, 0,  0x00000000 },
	{ 0x2, 0x1,  0x7,  0, 0x21000000, 4,  0x00000000 },
	{ 0x2, 0x2,  0x0,  0, 0x0b040000, 0,  0x00000000 },
	{ 0x2, 0x2,  0x1,  0, 0x0b840000, 1,  0x00000000 },
	{ 0x2, 0x3,  0x0,  0, 0x0b230000, 0,  0x00000000 },
	{ 0x2, 0x3,  0x1,  0, 0x0ba30000, 1,  0x00000000 },
	{ 0x2, 0x4,  0x0,  0, 0x0b050000, 0,  0x00000000 },
	{ 0x2, 0x4,  0x1,  0, 0x0b850000, 1,  0x00000000 },
	{ 0x2, 0x5,  0x0,  0, 0x0b060000, 0,  0x00000000 },
	{ 0x2, 0x5,  0x1,  0, 0x0b070000, 1,  0x00010000 },
	{ 0x2, 0x5,  0x2,  0, 0x0b080000, 2,  0x00020000 },
	{ 0x2, 0x5,  0x3,  0, 0x0b090000, 3,  0x00030000 },
	{ 0x2, 0x5,  0x4,  0, 0x0b0a0000, 4,  0x00040000 },
	{ 0x2, 0x5,  0x5,  0, 0x0b0b0000, 5,  0x00050000 },
	{ 0x2, 0x5,  0x6,  0, 0x0b0c0000, 6,  0x00060000 },
	{ 0x2, 0x5,  0x7,  0, 0x0b0d0000, 7,  0x00070000 },
	{ 0x2, 0x5,  0x8,  0, 0x0b0e0000, 8,  0x00080000 },
	{ 0x2, 0x5,  0x9,  0, 0x0b860000, 9,  0x00000000 },
	{ 0x2, 0x5,  0xa,  0, 0x0b870000, 10, 0x00010000 },
	{ 0x2, 0x5,  0xb,  0, 0x0b880000, 11, 0x00020000 },
	{ 0x2, 0x5,  0xc,  0, 0x0b890000, 12, 0x00030000 },
	{ 0x2, 0x5,  0xd,  0, 0x0b8a0000, 13, 0x00040000 },
	{ 0x2, 0x5,  0xe,  0, 0x0b8b0000, 14, 0x00050000 },
	{ 0x2, 0x5,  0xf,  0, 0x0b8c0000, 15, 0x00060000 },
	{ 0x2, 0x5,  0x10, 0, 0x0b8d0000, 16, 0x00070000 },
	{ 0x2, 0x5,  0x11, 0, 0x0b8e0000, 17, 0x00080000 },
	{ 0x2, 0x6,  0x0,  0, 0x0b650000, 0,  0x00000000 },
	{ 0x2, 0x6,  0x1,  0, 0x0be50000, 1,  0x00000000 },
	{ 0x2, 0x7,  0x0,  0, 0x20df0000, 0,  0x00000000 },
	{ 0x2, 0x7,  0x1,  0, 0x210f0000, 1,  0x00000000 },
	{ 0x2, 0x8,  0x0,  0, 0x0b3e0000, 0,  0x00000000 },
	{ 0x2, 0x8,  0x1,  0, 0x0bbe0000, 1,  0x00000000 },
	{ 0x2, 0x9,  0x0,  0, 0x0b3d0000, 0,  0x00000000 },
	{ 0x2, 0x9,  0x1,  0, 0x0bbd0000, 1,  0x00000000 },
	{ 0x2, 0xa,  0x0,  0, 0x0b1e0000, 0,  0x00000000 },
	{ 0x2, 0xa,  0x1,  0, 0x0b9e0000, 1,  0x00000000 },
	{ 0x2, 0xb,  0x0,  0, 0x0b150000, 0,  0x00000000 },
	{ 0x2, 0xb,  0x1,  0, 0x0b160000, 1,  0x00010000 },
	{ 0x2, 0xb,  0x2,  0, 0x0b170000, 2,  0x00020000 },
	{ 0x2, 0xb,  0x3,  0, 0x0b180000, 3,  0x00030000 },
	{ 0x2, 0xb,  0x4,  0, 0x0b190000, 4,  0x00040000 },
	{ 0x2, 0xb,  0x5,  0, 0x0b1a0000, 5,  0x00050000 },
	{ 0x2, 0xb,  0x6,  0, 0x0b1b0000, 6,  0x00060000 },
	{ 0x2, 0xb,  0x7,  0, 0x0b1c0000, 7,  0x00070000 },
	{ 0x2, 0xb,  0x8,  0, 0x0b1d0000, 8,  0x00080000 },
	{ 0x2, 0xb,  0x9,  0, 0x0b950000, 9,  0x00000000 },
	{ 0x2, 0xb,  0xa,  0, 0x0b960000, 10, 0x00010000 },
	{ 0x2, 0xb,  0xb,  0, 0x0b970000, 11, 0x00020000 },
	{ 0x2, 0xb,  0xc,  0, 0x0b980000, 12, 0x00030000 },
	{ 0x2, 0xb,  0xd,  0, 0x0b990000, 13, 0x00040000 },
	{ 0x2, 0xb,  0xe,  0, 0x0b9a0000, 14, 0x00050000 },
	{ 0x2, 0xb,  0xf,  0, 0x0b9b0000, 15, 0x00060000 },
	{ 0x2, 0xb,  0x10, 0, 0x0b9c0000, 16, 0x00070000 },
	{ 0x2, 0xb,  0x11, 0, 0x0b9d0000, 17, 0x00080000 },
	{ 0x2, 0xc,  0x0,  0, 0x0b660000, 0,  0x00000000 },
	{ 0x2, 0xc,  0x1,  0, 0x0be60000, 1,  0x00000000 },
	{ 0x2, 0xd,  0x0,  0, 0x0b1f0000, 0,  0x00000000 },
	{ 0x2, 0xd,  0x1,  0, 0x0b200000, 1,  0x00010000 },
	{ 0x2, 0xd,  0x2,  0, 0x0b210000, 2,  0x00020000 },
	{ 0x2, 0xd,  0x3,  0, 0x0b220000, 3,  0x00030000 },
	{ 0x2, 0xd,  0x4,  0, 0x0b9f0000, 4,  0x00000000 },
	{ 0x2, 0xd,  0x5,  0, 0x0ba00000, 5,  0x00010000 },
	{ 0x2, 0xd,  0x6,  0, 0x0ba10000, 6,  0x00020000 },
	{ 0x2, 0xd,  0x7,  0, 0x0ba20000, 7,  0x00030000 },
	{ 0x2, 0xe,  0x0,  0, 0x0b0f0000, 0,  0x00000000 },
	{ 0x2, 0xe,  0x1,  0, 0x0b100000, 1,  0x00010000 },
	{ 0x2, 0xe,  0x2,  0, 0x0b110000, 2,  0x00020000 },
	{ 0x2, 0xe,  0x3,  0, 0x0b120000, 3,  0x00030000 },
	{ 0x2, 0xe,  0x4,  0, 0x0b130000, 4,  0x00040000 },
	{ 0x2, 0xe,  0x5,  0, 0x0b140000, 5,  0x00050000 },
	{ 0x2, 0xe,  0x6,  0, 0x0b8f0000, 6,  0x00000000 },
	{ 0x2, 0xe,  0x7,  0, 0x0b900000, 7,  0x00010000 },
	{ 0x2, 0xe,  0x8,  0, 0x0b910000, 8,  0x00020000 },
	{ 0x2, 0xe,  0x9,  0, 0x0b920000, 9,  0x00030000 },
	{ 0x2, 0xe,  0xa,  0, 0x0b930000, 10, 0x00040000 },
	{ 0x2, 0xe,  0xb,  0, 0x0b940000, 11, 0x00050000 },
	{ 0x2, 0x10, 0x0,  0, 0x0b240000, 0,  0x00000000 },
	{ 0x2, 0x10, 0x1,  0, 0x0ba40000, 1,  0x00000000 },
	{ 0x2, 0x11, 0x0,  0, 0x0b020000, 0,  0x00000000 },
	{ 0x2, 0x11, 0x1,  0, 0x0b820000, 1,  0x00000000 },
	{ 0x2, 0x12, 0x0,  0, 0x0b030000, 0,  0x00000000 },
	{ 0x2, 0x12, 0x1,  0, 0x0b830000, 1,  0x00000000 },
	{ 0x2, 0x17, 0x0,  0, 0x0b640000, 0,  0x00000000 },
	{ 0x2, 0x17, 0x1,  0, 0x0be40000, 1,  0x00000000 },
	{ 0x2, 0x18, 0x0,  0, 0x0b600000, 0,  0x00000000 },
	{ 0x2, 0x18, 0x1,  0, 0x0be00000, 1,  0x00000000 },
	{ 0x2, 0x18, 0x2,  0, 0x00000000, 0,  0x00000000 },
	{ 0x2, 0x18, 0x3,  0, 0x00000000, 0,  0x00000000 },
	{ 0x3, 0x14, 0x0,  0, 0x40000000, 0,  0x40000000 },
	{ 0x3, 0x14, 0x1,  1, 0x80000000, 1,  0x80000000 },
	{ 0x3, 0x16, 0x0,  2, 0x0b400000, 0,  0x0b400000 },
	{ 0x3, 0x16, 0x1,  2, 0x0bc00000, 1,  0x0bc00000 },
	{ 0x3, 0x16, 0x2,  0, 0x00000000, 0,  0x00000000 },
	{ 0x3, 0x16, 0x3,  0, 0x00000000, 0,  0x00000000 },
	{ 0x4, 0x15, 0x0,  0, 0x20c00000, 8,  0x20c00000 },
	{ 0x4, 0x15, 0x1,  0, 0x21100000, 22, 0x21100000 },
	{ 0x4, 0x15, 0x2,  0, 0x20e00000, 9,  0x20e00000 },
	{ 0x4, 0x15, 0x3,  0, 0x21200000, 23, 0x21200000 },
	{ 0x4, 0x15, 0x4,  0, 0x20800000, 7,  0x20800000 },
	{ 0x4, 0x15, 0x5,  0, 0x21400000, 24, 0x21400000 },
	{ 0x4, 0x15, 0x6,  0, 0x0b000000, 18, 0x0b000000 },
	{ 0x4, 0x15, 0x7,  0, 0x0b800000, 3,  0x0b800000 },
	{ 0x4, 0x15, 0x8,  0, 0x20000000, 6,  0x20000000 },
	{ 0x4, 0x15, 0x9,  0, 0x21800000, 25, 0x21800000 },
	{ 0x4, 0x15, 0xa,  0, 0x0a000000, 2,  0x0a000000 },
	{ 0x4, 0x15, 0xb,  0, 0x0a000000, 17, 0x0a000000 },
	{ 0x4, 0x15, 0xc,  0, 0x20000000, 21, 0x20000000 },
	{ 0x4, 0x15, 0xd,  0, 0x21000000, 10, 0x21000000 },
	{ 0x4, 0x15, 0xe,  0, 0x08000000, 1,  0x08000000 },
	{ 0x4, 0x15, 0xf,  0, 0x08000000, 16, 0x08000000 },
	{ 0x4, 0x15, 0x10, 0, 0x22000000, 11, 0x22000000 },
	{ 0x4, 0x15, 0x11, 0, 0x22000000, 26, 0x22000000 },
	{ 0x4, 0x15, 0x12, 0, 0x0c000000, 4,  0x0c000000 },
	{ 0x4, 0x15, 0x13, 0, 0x0c000000, 19, 0x0c000000 },
	{ 0x4, 0x15, 0x14, 0, 0x24000000, 12, 0x24000000 },
	{ 0x4, 0x15, 0x15, 0, 0x24000000, 27, 0x24000000 },
	{ 0x4, 0x15, 0x16, 0, 0x00000000, 0,  0x00000000 },
	{ 0x4, 0x15, 0x17, 0, 0x00000000, 15, 0x00000000 },
	{ 0x4, 0x15, 0x18, 0, 0x28000000, 13, 0x28000000 },
	{ 0x4, 0x15, 0x19, 0, 0x28000000, 28, 0x28000000 },
	{ 0x4, 0x15, 0x1a, 0, 0x10000000, 5,  0x10000000 },
	{ 0x4, 0x15, 0x1b, 0, 0x10000000, 20, 0x10000000 },
	{ 0x4, 0x15, 0x1c, 0, 0x30000000, 14, 0x30000000 },
	{ 0x4, 0x15, 0x1d, 0, 0x30000000, 29, 0x30000000 },
	{ 0x4, 0x0,  0x0,  0, 0x0b000000, 0,  0x00000000 },
	{ 0x4, 0x0,  0x1,  0, 0x0b800000, 1,  0x00000000 },
	{ 0x4, 0x1,  0x0,  0, 0x20de0000, 3,  0x000e0000 },
	{ 0x4, 0x1,  0x1,  0, 0x210e0000, 7,  0x000e0000 },
	{ 0x4, 0x1,  0x2,  0, 0x20dc0000, 2,  0x000c0000 },
	{ 0x4, 0x1,  0x3,  0, 0x210c0000, 6,  0x000c0000 },
	{ 0x4, 0x1,  0x4,  0, 0x20d80000, 1,  0x00080000 },
	{ 0x4, 0x1,  0x5,  0, 0x21080000, 5,  0x00080000 },
	{ 0x4, 0x1,  0x6,  0, 0x20d00000, 0,  0x00000000 },
	{ 0x4, 0x1,  0x7,  0, 0x21000000, 4,  0x00000000 },
	{ 0x4, 0x2,  0x0,  0, 0x0b040000, 0,  0x00000000 },
	{ 0x4, 0x2,  0x1,  0, 0x0b840000, 1,  0x00000000 },
	{ 0x4, 0x3,  0x0,  0, 0x0b230000, 0,  0x00000000 },
	{ 0x4, 0x3,  0x1,  0, 0x0ba30000, 1,  0x00000000 },
	{ 0x4, 0x4,  0x0,  0, 0x0b050000, 0,  0x00000000 },
	{ 0x4, 0x4,  0x1,  0, 0x0b850000, 1,  0x00000000 },
	{ 0x4, 0x5,  0x0,  0, 0x0b060000, 0,  0x00000000 },
	{ 0x4, 0x5,  0x1,  0, 0x0b070000, 1,  0x00010000 },
	{ 0x4, 0x5,  0x2,  0, 0x0b080000, 2,  0x00020000 },
	{ 0x4, 0x5,  0x3,  0, 0x0b090000, 3,  0x00030000 },
	{ 0x4, 0x5,  0x4,  0, 0x0b0a0000, 4,  0x00040000 },
	{ 0x4, 0x5,  0x5,  0, 0x0b0b0000, 5,  0x00050000 },
	{ 0x4, 0x5,  0x6,  0, 0x0b0c0000, 6,  0x00060000 },
	{ 0x4, 0x5,  0x7,  0, 0x0b0d0000, 7,  0x00070000 },
	{ 0x4, 0x5,  0x8,  0, 0x0b0e0000, 8,  0x00080000 },
	{ 0x4, 0x5,  0x9,  0, 0x0b860000, 9,  0x00000000 },
	{ 0x4, 0x5,  0xa,  0, 0x0b870000, 10, 0x00010000 },
	{ 0x4, 0x5,  0xb,  0, 0x0b880000, 11, 0x00020000 },
	{ 0x4, 0x5,  0xc,  0, 0x0b890000, 12, 0x00030000 },
	{ 0x4, 0x5,  0xd,  0, 0x0b8a0000, 13, 0x00040000 },
	{ 0x4, 0x5,  0xe,  0, 0x0b8b0000, 14, 0x00050000 },
	{ 0x4, 0x5,  0xf,  0, 0x0b8c0000, 15, 0x00060000 },
	{ 0x4, 0x5,  0x10, 0, 0x0b8d0000, 16, 0x00070000 },
	{ 0x4, 0x5,  0x11, 0, 0x0b8e0000, 17, 0x00080000 },
	{ 0x4, 0x6,  0x0,  0, 0x0b650000, 0,  0x00000000 },
	{ 0x4, 0x6,  0x1,  0, 0x0be50000, 1,  0x00000000 },
	{ 0x4, 0x7,  0x0,  0, 0x20df0000, 0,  0x00000000 },
	{ 0x4, 0x7,  0x1,  0, 0x210f0000, 1,  0x00000000 },
	{ 0x4, 0x8,  0x0,  0, 0x0b3e0000, 0,  0x00000000 },
	{ 0x4, 0x8,  0x1,  0, 0x0bbe0000, 1,  0x00000000 },
	{ 0x4, 0x9,  0x0,  0, 0x0b3d0000, 0,  0x00000000 },
	{ 0x4, 0x9,  0x1,  0, 0x0bbd0000, 1,  0x00000000 },
	{ 0x4, 0xa,  0x0,  0, 0x0b1e0000, 0,  0x00000000 },
	{ 0x4, 0xa,  0x1,  0, 0x0b9e0000, 1,  0x00000000 },
	{ 0x4, 0xb,  0x0,  0, 0x0b150000, 0,  0x00000000 },
	{ 0x4, 0xb,  0x1,  0, 0x0b160000, 1,  0x00010000 },
	{ 0x4, 0xb,  0x2,  0, 0x0b170000, 2,  0x00020000 },
	{ 0x4, 0xb,  0x3,  0, 0x0b180000, 3,  0x00030000 },
	{ 0x4, 0xb,  0x4,  0, 0x0b190000, 4,  0x00040000 },
	{ 0x4, 0xb,  0x5,  0, 0x0b1a0000, 5,  0x00050000 },
	{ 0x4, 0xb,  0x6,  0, 0x0b1b0000, 6,  0x00060000 },
	{ 0x4, 0xb,  0x7,  0, 0x0b1c0000, 7,  0x00070000 },
	{ 0x4, 0xb,  0x8,  0, 0x0b1d0000, 8,  0x00080000 },
	{ 0x4, 0xb,  0x9,  0, 0x0b950000, 9,  0x00000000 },
	{ 0x4, 0xb,  0xa,  0, 0x0b960000, 10, 0x00010000 },
	{ 0x4, 0xb,  0xb,  0, 0x0b970000, 11, 0x00020000 },
	{ 0x4, 0xb,  0xc,  0, 0x0b980000, 12, 0x00030000 },
	{ 0x4, 0xb,  0xd,  0, 0x0b990000, 13, 0x00040000 },
	{ 0x4, 0xb,  0xe,  0, 0x0b9a0000, 14, 0x00050000 },
	{ 0x4, 0xb,  0xf,  0, 0x0b9b0000, 15, 0x00060000 },
	{ 0x4, 0xb,  0x10, 0, 0x0b9c0000, 16, 0x00070000 },
	{ 0x4, 0xb,  0x11, 0, 0x0b9d0000, 17, 0x00080000 },
	{ 0x4, 0xc,  0x0,  0, 0x0b660000, 0,  0x00000000 },
	{ 0x4, 0xc,  0x1,  0, 0x0be60000, 1,  0x00000000 },
	{ 0x4, 0xd,  0x0,  0, 0x0b1f0000, 0,  0x00000000 },
	{ 0x4, 0xd,  0x1,  0, 0x0b200000, 1,  0x00010000 },
	{ 0x4, 0xd,  0x2,  0, 0x0b210000, 2,  0x00020000 },
	{ 0x4, 0xd,  0x3,  0, 0x0b220000, 3,  0x00030000 },
	{ 0x4, 0xd,  0x4,  0, 0x0b9f0000, 4,  0x00000000 },
	{ 0x4, 0xd,  0x5,  0, 0x0ba00000, 5,  0x00010000 },
	{ 0x4, 0xd,  0x6,  0, 0x0ba10000, 6,  0x00020000 },
	{ 0x4, 0xd,  0x7,  0, 0x0ba20000, 7,  0x00030000 },
	{ 0x4, 0xe,  0x0,  0, 0x0b0f0000, 0,  0x00000000 },
	{ 0x4, 0xe,  0x1,  0, 0x0b100000, 1,  0x00010000 },
	{ 0x4, 0xe,  0x2,  0, 0x0b110000, 2,  0x00020000 },
	{ 0x4, 0xe,  0x3,  0, 0x0b120000, 3,  0x00030000 },
	{ 0x4, 0xe,  0x4,  0, 0x0b130000, 4,  0x00040000 },
	{ 0x4, 0xe,  0x5,  0, 0x0b140000, 5,  0x00050000 },
	{ 0x4, 0xe,  0x6,  0, 0x0b8f0000, 6,  0x00000000 },
	{ 0x4, 0xe,  0x7,  0, 0x0b900000, 7,  0x00010000 },
	{ 0x4, 0xe,  0x8,  0, 0x0b910000, 8,  0x00020000 },
	{ 0x4, 0xe,  0x9,  0, 0x0b920000, 9,  0x00030000 },
	{ 0x4, 0xe,  0xa,  0, 0x0b930000, 10, 0x00040000 },
	{ 0x4, 0xe,  0xb,  0, 0x0b940000, 11, 0x00050000 },
	{ 0x4, 0x10, 0x0,  0, 0x0b240000, 0,  0x00000000 },
	{ 0x4, 0x10, 0x1,  0, 0x0ba40000, 1,  0x00000000 },
	{ 0x4, 0x11, 0x0,  0, 0x0b020000, 0,  0x00000000 },
	{ 0x4, 0x11, 0x1,  0, 0x0b820000, 1,  0x00000000 },
	{ 0x4, 0x12, 0x0,  0, 0x0b030000, 0,  0x00000000 },
	{ 0x4, 0x12, 0x1,  0, 0x0b830000, 1,  0x00000000 },
	{ 0x4, 0x17, 0x0,  0, 0x0b640000, 0,  0x00000000 },
	{ 0x4, 0x17, 0x1,  0, 0x0be40000, 1,  0x00000000 },
	{ 0x4, 0x18, 0x0,  0, 0x0b600000, 0,  0x00000000 },
	{ 0x4, 0x18, 0x1,  0, 0x0be00000, 1,  0x00000000 },
	{ 0x4, 0x18, 0x2,  0, 0x00000000, 0,  0x00000000 },
	{ 0x4, 0x18, 0x3,  0, 0x00000000, 0,  0x00000000 }
};

static void cbbcentralnoc_parse_routeid(struct tegra194_cbb_aperture *info, u64 routeid)
{
	info->initflow = FIELD_GET(CBB_NOC_INITFLOW, routeid);
	info->targflow = FIELD_GET(CBB_NOC_TARGFLOW, routeid);
	info->targ_subrange = FIELD_GET(CBB_NOC_TARG_SUBRANGE, routeid);
	info->seqid = FIELD_GET(CBB_NOC_SEQID, routeid);
}

static void bpmpnoc_parse_routeid(struct tegra194_cbb_aperture *info, u64 routeid)
{
	info->initflow = FIELD_GET(BPMP_NOC_INITFLOW, routeid);
	info->targflow = FIELD_GET(BPMP_NOC_TARGFLOW, routeid);
	info->targ_subrange = FIELD_GET(BPMP_NOC_TARG_SUBRANGE, routeid);
	info->seqid = FIELD_GET(BPMP_NOC_SEQID, routeid);
}

static void aonnoc_parse_routeid(struct tegra194_cbb_aperture *info, u64 routeid)
{
	info->initflow = FIELD_GET(AON_NOC_INITFLOW, routeid);
	info->targflow = FIELD_GET(AON_NOC_TARGFLOW, routeid);
	info->targ_subrange = FIELD_GET(AON_NOC_TARG_SUBRANGE, routeid);
	info->seqid = FIELD_GET(AON_NOC_SEQID, routeid);
}

static void scenoc_parse_routeid(struct tegra194_cbb_aperture *info, u64 routeid)
{
	info->initflow = FIELD_GET(SCE_NOC_INITFLOW, routeid);
	info->targflow = FIELD_GET(SCE_NOC_TARGFLOW, routeid);
	info->targ_subrange = FIELD_GET(SCE_NOC_TARG_SUBRANGE, routeid);
	info->seqid = FIELD_GET(SCE_NOC_SEQID, routeid);
}

static void cbbcentralnoc_parse_userbits(struct tegra194_cbb_userbits *usrbits, u32 elog_5)
{
	usrbits->axcache = FIELD_GET(CBB_NOC_AXCACHE, elog_5);
	usrbits->non_mod = FIELD_GET(CBB_NOC_NON_MOD, elog_5);
	usrbits->axprot = FIELD_GET(CBB_NOC_AXPROT, elog_5);
	usrbits->falconsec = FIELD_GET(CBB_NOC_FALCONSEC, elog_5);
	usrbits->grpsec = FIELD_GET(CBB_NOC_GRPSEC, elog_5);
	usrbits->vqc = FIELD_GET(CBB_NOC_VQC, elog_5);
	usrbits->mstr_id = FIELD_GET(CBB_NOC_MSTR_ID, elog_5) - 1;
	usrbits->axi_id = FIELD_GET(CBB_NOC_AXI_ID, elog_5);
}

static void clusternoc_parse_userbits(struct tegra194_cbb_userbits *usrbits, u32 elog_5)
{
	usrbits->axcache = FIELD_GET(CLUSTER_NOC_AXCACHE, elog_5);
	usrbits->axprot = FIELD_GET(CLUSTER_NOC_AXCACHE, elog_5);
	usrbits->falconsec = FIELD_GET(CLUSTER_NOC_FALCONSEC, elog_5);
	usrbits->grpsec = FIELD_GET(CLUSTER_NOC_GRPSEC, elog_5);
	usrbits->vqc = FIELD_GET(CLUSTER_NOC_VQC, elog_5);
	usrbits->mstr_id = FIELD_GET(CLUSTER_NOC_MSTR_ID, elog_5) - 1;
}

static void tegra194_cbb_fault_enable(struct tegra_cbb *cbb)
{
	struct tegra194_cbb *priv = to_tegra194_cbb(cbb);

	writel(1, priv->regs + ERRLOGGER_0_FAULTEN_0);
	writel(1, priv->regs + ERRLOGGER_1_FAULTEN_0);
	writel(1, priv->regs + ERRLOGGER_2_FAULTEN_0);
}

static void tegra194_cbb_stall_enable(struct tegra_cbb *cbb)
{
	struct tegra194_cbb *priv = to_tegra194_cbb(cbb);

	writel(1, priv->regs + ERRLOGGER_0_STALLEN_0);
	writel(1, priv->regs + ERRLOGGER_1_STALLEN_0);
	writel(1, priv->regs + ERRLOGGER_2_STALLEN_0);
}

static void tegra194_cbb_error_clear(struct tegra_cbb *cbb)
{
	struct tegra194_cbb *priv = to_tegra194_cbb(cbb);

	writel(1, priv->regs + ERRLOGGER_0_ERRCLR_0);
	writel(1, priv->regs + ERRLOGGER_1_ERRCLR_0);
	writel(1, priv->regs + ERRLOGGER_2_ERRCLR_0);
	dsb(sy);
}

static u32 tegra194_cbb_get_status(struct tegra_cbb *cbb)
{
	struct tegra194_cbb *priv = to_tegra194_cbb(cbb);
	u32 value;

	value = readl(priv->regs + ERRLOGGER_0_ERRVLD_0);
	value |= (readl(priv->regs + ERRLOGGER_1_ERRVLD_0) << 1);
	value |= (readl(priv->regs + ERRLOGGER_2_ERRVLD_0) << 2);

	dsb(sy);
	return value;
}

static u32 tegra194_axi2apb_status(void __iomem *addr)
{
	u32 value;

	value = readl(addr + DMAAPB_X_RAW_INTERRUPT_STATUS);
	writel(0xffffffff, addr + DMAAPB_X_RAW_INTERRUPT_STATUS);

	return value;
}

static bool tegra194_axi2apb_fatal(struct seq_file *file, unsigned int bridge, u32 status)
{
	bool is_fatal = true;
	size_t i;

	for (i = 0; i < ARRAY_SIZE(tegra194_axi2apb_error); i++) {
		if (status & BIT(i)) {
			tegra_cbb_print_err(file, "\t  AXI2APB_%d bridge error: %s\n",
					    bridge + 1, tegra194_axi2apb_error[i]);
			if (strstr(tegra194_axi2apb_error[i], "Firewall"))
				is_fatal = false;
		}
	}

	return is_fatal;
}

/*
 * Fetch InitlocalAddress from NOC Aperture lookup table
 * using Targflow, Targsubrange
 */
static u32 get_init_localaddress(const struct tegra194_cbb_aperture *info,
				 const struct tegra194_cbb_aperture *aper, unsigned int max)
{
	unsigned int t_f = 0, t_sr = 0;
	u32 addr = 0;

	for (t_f = 0; t_f < max; t_f++) {
		if (aper[t_f].targflow == info->targflow) {
			t_sr = t_f;

			do {
				if (aper[t_sr].targ_subrange == info->targ_subrange) {
					addr = aper[t_sr].init_localaddress;
					return addr;
				}

				if (t_sr >= max)
					return 0;

				t_sr++;
			} while (aper[t_sr].targflow == aper[t_sr - 1].targflow);

			t_f = t_sr;
		}
	}

	return addr;
}

static void print_errlog5(struct seq_file *file, struct tegra194_cbb *cbb)
{
	struct tegra194_cbb_userbits userbits;

	cbb->noc->parse_userbits(&userbits, cbb->errlog5);

	if (!strcmp(cbb->noc->name, "cbb-noc")) {
		tegra_cbb_print_err(file, "\t  Non-Modify\t\t: %#x\n", userbits.non_mod);
		tegra_cbb_print_err(file, "\t  AXI ID\t\t: %#x\n", userbits.axi_id);
	}

	tegra_cbb_print_err(file, "\t  Master ID\t\t: %s\n",
			    cbb->noc->master_id[userbits.mstr_id]);
	tegra_cbb_print_err(file, "\t  Security Group(GRPSEC): %#x\n", userbits.grpsec);
	tegra_cbb_print_cache(file, userbits.axcache);
	tegra_cbb_print_prot(file, userbits.axprot);
	tegra_cbb_print_err(file, "\t  FALCONSEC\t\t: %#x\n", userbits.falconsec);
	tegra_cbb_print_err(file, "\t  Virtual Queuing Channel(VQC): %#x\n", userbits.vqc);
}

/*
 *  Fetch Base Address/InitlocalAddress from NOC aperture lookup table using TargFlow &
 *  Targ_subRange extracted from RouteId. Perform address reconstruction as below:
 *
 *    Address = Base Address + (ErrLog3 + ErrLog4)
 */
static void
print_errlog3_4(struct seq_file *file, u32 errlog3, u32 errlog4,
		const struct tegra194_cbb_aperture *info,
		const struct tegra194_cbb_aperture *aperture, unsigned int max)
{
	u64 addr = (u64)errlog4 << 32 | errlog3;

	/*
	 * If errlog4[7] = "1", then it's a joker entry. Joker entries are a rare phenomenon and
	 * such addresses are not reliable. Debugging should be done using only the RouteId
	 * information.
	 */
	if (errlog4 & 0x80)
		tegra_cbb_print_err(file, "\t debug using RouteId alone as below address is a "
					  "joker entry and not reliable");

	addr += get_init_localaddress(info, aperture, max);

	tegra_cbb_print_err(file, "\t  Address accessed\t: %#llx\n", addr);
}

/*
 *  Get RouteId from ErrLog1+ErrLog2 registers and fetch values of
 *  InitFlow, TargFlow, Targ_subRange and SeqId values from RouteId
 */
static void
print_errlog1_2(struct seq_file *file, struct tegra194_cbb *cbb,
		struct tegra194_cbb_aperture *info)
{
	u64 routeid = (u64)cbb->errlog2 << 32 | cbb->errlog1;
	u32 seqid = 0;

	tegra_cbb_print_err(file, "\t  RouteId\t\t: %#llx\n", routeid);

	cbb->noc->parse_routeid(info, routeid);

	tegra_cbb_print_err(file, "\t  InitFlow\t\t: %s\n",
			    cbb->noc->routeid_initflow[info->initflow]);

	tegra_cbb_print_err(file, "\t  Targflow\t\t: %s\n",
			    cbb->noc->routeid_targflow[info->targflow]);

	tegra_cbb_print_err(file, "\t  TargSubRange\t\t: %d\n", info->targ_subrange);
	tegra_cbb_print_err(file, "\t  SeqId\t\t\t: %d\n", seqid);
}

/*
 * Print transcation type, error code and description from ErrLog0 for all
 * errors. For NOC slave errors, all relevant error info is printed using
 * ErrLog0 only. But additional information is printed for errors from
 * APB slaves because for them:
 *  - All errors are logged as SLV(slave) errors due to APB having only single
 *    bit pslverr to report all errors.
 *  - Exact cause is printed by reading DMAAPB_X_RAW_INTERRUPT_STATUS register.
 *  - The driver prints information showing AXI2APB bridge and exact error
 *    only if there is error in any AXI2APB slave.
 *  - There is still no way to disambiguate a DEC error from SLV error type.
 */
static bool print_errlog0(struct seq_file *file, struct tegra194_cbb *cbb)
{
	struct tegra194_cbb_packet_header hdr;
	bool is_fatal = true;

	hdr.lock = cbb->errlog0 & 0x1;
	hdr.opc = FIELD_GET(CBB_ERR_OPC, cbb->errlog0);
	hdr.errcode = FIELD_GET(CBB_ERR_ERRCODE, cbb->errlog0);
	hdr.len1 = FIELD_GET(CBB_ERR_LEN1, cbb->errlog0);
	hdr.format = (cbb->errlog0 >> 31);

	tegra_cbb_print_err(file, "\t  Transaction Type\t: %s\n",
			    tegra194_cbb_trantype[hdr.opc]);
	tegra_cbb_print_err(file, "\t  Error Code\t\t: %s\n",
			    tegra194_cbb_errors[hdr.errcode].code);
	tegra_cbb_print_err(file, "\t  Error Source\t\t: %s\n",
			    tegra194_cbb_errors[hdr.errcode].source);
	tegra_cbb_print_err(file, "\t  Error Description\t: %s\n",
			    tegra194_cbb_errors[hdr.errcode].desc);

	/*
	 * Do not crash system for errors which are only notifications to indicate a transaction
	 * was not allowed to be attempted.
	 */
	if (!strcmp(tegra194_cbb_errors[hdr.errcode].code, "SEC") ||
	    !strcmp(tegra194_cbb_errors[hdr.errcode].code, "DEC") ||
	    !strcmp(tegra194_cbb_errors[hdr.errcode].code, "UNS") ||
	    !strcmp(tegra194_cbb_errors[hdr.errcode].code, "DISC")) {
		is_fatal = false;
	} else if (!strcmp(tegra194_cbb_errors[hdr.errcode].code, "SLV") &&
		   cbb->num_bridges > 0) {
		unsigned int i;
		u32 status;

		/* For all SLV errors, read DMAAPB_X_RAW_INTERRUPT_STATUS
		 * register to get error status for all AXI2APB bridges.
		 * Print bridge details if a bit is set in a bridge's
		 * status register due to error in a APB slave connected
		 * to that bridge. For other NOC slaves, none of the status
		 * register will be set.
		 */

		for (i = 0; i < cbb->num_bridges; i++) {
			status = tegra194_axi2apb_status(cbb->bridges[i].base);

			if (status)
				is_fatal = tegra194_axi2apb_fatal(file, i, status);
		}
	}

	tegra_cbb_print_err(file, "\t  Packet header Lock\t: %d\n", hdr.lock);
	tegra_cbb_print_err(file, "\t  Packet header Len1\t: %d\n", hdr.len1);

	if (hdr.format)
		tegra_cbb_print_err(file, "\t  NOC protocol version\t: %s\n",
				    "version >= 2.7");
	else
		tegra_cbb_print_err(file, "\t  NOC protocol version\t: %s\n",
				    "version < 2.7");

	return is_fatal;
}

/*
 * Print debug information about failed transaction using
 * ErrLog registers of error loggger having ErrVld set
 */
static bool print_errloggerX_info(struct seq_file *file, struct tegra194_cbb *cbb,
				  int errloggerX)
{
	struct tegra194_cbb_aperture info = { 0, };
	bool is_fatal = true;

	tegra_cbb_print_err(file, "\tError Logger\t\t: %d\n", errloggerX);

	if (errloggerX == 0) {
		cbb->errlog0 = readl(cbb->regs + ERRLOGGER_0_ERRLOG0_0);
		cbb->errlog1 = readl(cbb->regs + ERRLOGGER_0_ERRLOG1_0);
		cbb->errlog2 = readl(cbb->regs + ERRLOGGER_0_RSVD_00_0);
		cbb->errlog3 = readl(cbb->regs + ERRLOGGER_0_ERRLOG3_0);
		cbb->errlog4 = readl(cbb->regs + ERRLOGGER_0_ERRLOG4_0);
		cbb->errlog5 = readl(cbb->regs + ERRLOGGER_0_ERRLOG5_0);
	} else if (errloggerX == 1) {
		cbb->errlog0 = readl(cbb->regs + ERRLOGGER_1_ERRLOG0_0);
		cbb->errlog1 = readl(cbb->regs + ERRLOGGER_1_ERRLOG1_0);
		cbb->errlog2 = readl(cbb->regs + ERRLOGGER_1_RSVD_00_0);
		cbb->errlog3 = readl(cbb->regs + ERRLOGGER_1_ERRLOG3_0);
		cbb->errlog4 = readl(cbb->regs + ERRLOGGER_1_ERRLOG4_0);
		cbb->errlog5 = readl(cbb->regs + ERRLOGGER_1_ERRLOG5_0);
	} else if (errloggerX == 2) {
		cbb->errlog0 = readl(cbb->regs + ERRLOGGER_2_ERRLOG0_0);
		cbb->errlog1 = readl(cbb->regs + ERRLOGGER_2_ERRLOG1_0);
		cbb->errlog2 = readl(cbb->regs + ERRLOGGER_2_RSVD_00_0);
		cbb->errlog3 = readl(cbb->regs + ERRLOGGER_2_ERRLOG3_0);
		cbb->errlog4 = readl(cbb->regs + ERRLOGGER_2_ERRLOG4_0);
		cbb->errlog5 = readl(cbb->regs + ERRLOGGER_2_ERRLOG5_0);
	}

	tegra_cbb_print_err(file, "\tErrLog0\t\t\t: %#x\n", cbb->errlog0);
	is_fatal = print_errlog0(file, cbb);

	tegra_cbb_print_err(file, "\tErrLog1\t\t\t: %#x\n", cbb->errlog1);
	tegra_cbb_print_err(file, "\tErrLog2\t\t\t: %#x\n", cbb->errlog2);
	print_errlog1_2(file, cbb, &info);

	tegra_cbb_print_err(file, "\tErrLog3\t\t\t: %#x\n", cbb->errlog3);
	tegra_cbb_print_err(file, "\tErrLog4\t\t\t: %#x\n", cbb->errlog4);
	print_errlog3_4(file, cbb->errlog3, cbb->errlog4, &info, cbb->noc->noc_aperture,
			cbb->noc->max_aperture);

	tegra_cbb_print_err(file, "\tErrLog5\t\t\t: %#x\n", cbb->errlog5);

	if (cbb->errlog5)
		print_errlog5(file, cbb);

	return is_fatal;
}

static bool print_errlog(struct seq_file *file, struct tegra194_cbb *cbb, u32 errvld)
{
	bool is_fatal = true;

	pr_crit("**************************************\n");
	pr_crit("CPU:%d, Error:%s\n", smp_processor_id(), cbb->noc->name);

	if (errvld & 0x1)
		is_fatal = print_errloggerX_info(file, cbb, 0);
	else if (errvld & 0x2)
		is_fatal = print_errloggerX_info(file, cbb, 1);
	else if (errvld & 0x4)
		is_fatal = print_errloggerX_info(file, cbb, 2);

	tegra_cbb_error_clear(&cbb->base);
	tegra_cbb_print_err(file, "\t**************************************\n");
	return is_fatal;
}

#ifdef CONFIG_DEBUG_FS
static DEFINE_MUTEX(cbb_err_mutex);

static int tegra194_cbb_debugfs_show(struct tegra_cbb *cbb, struct seq_file *file, void *data)
{
	struct tegra_cbb *noc;

	mutex_lock(&cbb_err_mutex);

	list_for_each_entry(noc, &cbb_list, node) {
		struct tegra194_cbb *priv = to_tegra194_cbb(noc);
		u32 status;

		status = tegra_cbb_get_status(noc);
		if (status)
			print_errlog(file, priv, status);
	}

	mutex_unlock(&cbb_err_mutex);

	return 0;
}
#endif

/*
 * Handler for CBB errors from different initiators
 */
static irqreturn_t tegra194_cbb_err_isr(int irq, void *data)
{
	bool is_inband_err = false, is_fatal = false;
	//struct tegra194_cbb *cbb = data;
	struct tegra_cbb *noc;
	unsigned long flags;
	u8 mstr_id = 0;

	spin_lock_irqsave(&cbb_lock, flags);

	/* XXX only process interrupts for "cbb" instead of iterating over all NOCs? */
	list_for_each_entry(noc, &cbb_list, node) {
		struct tegra194_cbb *priv = to_tegra194_cbb(noc);
		u32 status = 0;

		status = tegra_cbb_get_status(noc);

		if (status && ((irq == priv->sec_irq) || (irq == priv->nonsec_irq))) {
			tegra_cbb_print_err(NULL, "CPU:%d, Error: %s@%llx, irq=%d\n",
					    smp_processor_id(), priv->noc->name, priv->res->start,
					    irq);

			is_fatal = print_errlog(NULL, priv, status);

			/*
			 * If illegal request is from CCPLEX(0x1) initiator
			 * and error is fatal then call BUG() to crash system.
			 */
			if (priv->noc->erd_mask_inband_err) {
				mstr_id =  FIELD_GET(CBB_NOC_MSTR_ID, priv->errlog5);
				if (mstr_id == 0x1)
					is_inband_err = 1;
			}
		}
	}

	spin_unlock_irqrestore(&cbb_lock, flags);

	if (is_inband_err) {
		if (is_fatal)
			BUG();
		else
			WARN(true, "Warning due to CBB Error\n");
	}

	return IRQ_HANDLED;
}

/*
 * Register handler for CBB_NONSECURE & CBB_SECURE interrupts
 * for reporting CBB errors
 */
static int tegra194_cbb_interrupt_enable(struct tegra_cbb *cbb)
{
	struct tegra194_cbb *priv = to_tegra194_cbb(cbb);
	struct device *dev = cbb->dev;
	int err;

	if (priv->sec_irq) {
		err = devm_request_irq(dev, priv->sec_irq, tegra194_cbb_err_isr, 0, dev_name(dev),
				       priv);
		if (err) {
			dev_err(dev, "failed to register interrupt %u: %d\n", priv->sec_irq, err);
			return err;
		}
	}

	if (priv->nonsec_irq) {
		err = devm_request_irq(dev, priv->nonsec_irq, tegra194_cbb_err_isr, 0,
				       dev_name(dev), priv);
		if (err) {
			dev_err(dev, "failed to register interrupt %u: %d\n", priv->nonsec_irq,
				err);
			return err;
		}
	}

	return 0;
}

static void tegra194_cbb_error_enable(struct tegra_cbb *cbb)
{
	/*
	 * Set “StallEn=1” to enable queuing of error packets till
	 * first is served & cleared
	 */
	tegra_cbb_stall_enable(cbb);

	/* set “FaultEn=1” to enable error reporting signal “Fault” */
	tegra_cbb_fault_enable(cbb);
}

static const struct tegra_cbb_ops tegra194_cbb_ops = {
	.get_status = tegra194_cbb_get_status,
	.error_clear = tegra194_cbb_error_clear,
	.fault_enable = tegra194_cbb_fault_enable,
	.stall_enable = tegra194_cbb_stall_enable,
	.error_enable = tegra194_cbb_error_enable,
	.interrupt_enable = tegra194_cbb_interrupt_enable,
#ifdef CONFIG_DEBUG_FS
	.debugfs_show = tegra194_cbb_debugfs_show,
#endif
};

static struct tegra194_cbb_noc_data tegra194_cbb_central_noc_data = {
	.name = "cbb-noc",
	.erd_mask_inband_err = true,
	.master_id = tegra194_master_id,
	.noc_aperture = tegra194_cbbcentralnoc_apert_lookup,
	.max_aperture = ARRAY_SIZE(tegra194_cbbcentralnoc_apert_lookup),
	.routeid_initflow = tegra194_cbbcentralnoc_routeid_initflow,
	.routeid_targflow = tegra194_cbbcentralnoc_routeid_targflow,
	.parse_routeid = cbbcentralnoc_parse_routeid,
	.parse_userbits = cbbcentralnoc_parse_userbits
};

static struct tegra194_cbb_noc_data tegra194_aon_noc_data = {
	.name = "aon-noc",
	.erd_mask_inband_err = false,
	.master_id = tegra194_master_id,
	.noc_aperture = tegra194_aonnoc_aperture_lookup,
	.max_aperture = ARRAY_SIZE(tegra194_aonnoc_aperture_lookup),
	.routeid_initflow = tegra194_aonnoc_routeid_initflow,
	.routeid_targflow = tegra194_aonnoc_routeid_targflow,
	.parse_routeid = aonnoc_parse_routeid,
	.parse_userbits = clusternoc_parse_userbits
};

static struct tegra194_cbb_noc_data tegra194_bpmp_noc_data = {
	.name = "bpmp-noc",
	.erd_mask_inband_err = false,
	.master_id = tegra194_master_id,
	.noc_aperture = tegra194_bpmpnoc_apert_lookup,
	.max_aperture = ARRAY_SIZE(tegra194_bpmpnoc_apert_lookup),
	.routeid_initflow = tegra194_bpmpnoc_routeid_initflow,
	.routeid_targflow = tegra194_bpmpnoc_routeid_targflow,
	.parse_routeid = bpmpnoc_parse_routeid,
	.parse_userbits = clusternoc_parse_userbits
};

static struct tegra194_cbb_noc_data tegra194_rce_noc_data = {
	.name = "rce-noc",
	.erd_mask_inband_err = false,
	.master_id = tegra194_master_id,
	.noc_aperture = tegra194_scenoc_apert_lookup,
	.max_aperture = ARRAY_SIZE(tegra194_scenoc_apert_lookup),
	.routeid_initflow = tegra194_scenoc_routeid_initflow,
	.routeid_targflow = tegra194_scenoc_routeid_targflow,
	.parse_routeid = scenoc_parse_routeid,
	.parse_userbits = clusternoc_parse_userbits
};

static struct tegra194_cbb_noc_data tegra194_sce_noc_data = {
	.name = "sce-noc",
	.erd_mask_inband_err = false,
	.master_id = tegra194_master_id,
	.noc_aperture = tegra194_scenoc_apert_lookup,
	.max_aperture = ARRAY_SIZE(tegra194_scenoc_apert_lookup),
	.routeid_initflow = tegra194_scenoc_routeid_initflow,
	.routeid_targflow = tegra194_scenoc_routeid_targflow,
	.parse_routeid = scenoc_parse_routeid,
	.parse_userbits = clusternoc_parse_userbits
};

static const struct of_device_id tegra194_cbb_match[] = {
	{ .compatible = "nvidia,tegra194-cbb-noc", .data = &tegra194_cbb_central_noc_data },
	{ .compatible = "nvidia,tegra194-aon-noc", .data = &tegra194_aon_noc_data },
	{ .compatible = "nvidia,tegra194-bpmp-noc", .data = &tegra194_bpmp_noc_data },
	{ .compatible = "nvidia,tegra194-rce-noc", .data = &tegra194_rce_noc_data },
	{ .compatible = "nvidia,tegra194-sce-noc", .data = &tegra194_sce_noc_data },
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, tegra194_cbb_match);

static int tegra194_cbb_get_bridges(struct tegra194_cbb *cbb, struct device_node *np)
{
	struct tegra_cbb *entry;
	unsigned long flags;
	unsigned int i;
	int err;

	spin_lock_irqsave(&cbb_lock, flags);

	list_for_each_entry(entry, &cbb_list, node) {
		struct tegra194_cbb *priv = to_tegra194_cbb(entry);

		if (priv->bridges) {
			cbb->num_bridges = priv->num_bridges;
			cbb->bridges = priv->bridges;
			break;
		}
	}

	spin_unlock_irqrestore(&cbb_lock, flags);

	if (!cbb->bridges) {
		cbb->num_bridges = of_address_count(np);

		cbb->bridges = devm_kcalloc(cbb->base.dev, cbb->num_bridges,
					    sizeof(*cbb->bridges), GFP_KERNEL);
		if (!cbb->bridges)
			return -ENOMEM;

		for (i = 0; i < cbb->num_bridges; i++) {
			err = of_address_to_resource(np, i, &cbb->bridges[i].res);
			if (err < 0)
				return err;

			cbb->bridges[i].base = devm_ioremap_resource(cbb->base.dev,
								     &cbb->bridges[i].res);
			if (IS_ERR(cbb->bridges[i].base))
				return PTR_ERR(cbb->bridges[i].base);
		}
	}

	if (cbb->num_bridges > 0) {
		dev_dbg(cbb->base.dev, "AXI2APB bridge info present:\n");

		for (i = 0; i < cbb->num_bridges; i++)
			dev_dbg(cbb->base.dev, "  %u: %pR\n", i, &cbb->bridges[i].res);
	}

	return 0;
}

static int tegra194_cbb_probe(struct platform_device *pdev)
{
	const struct tegra194_cbb_noc_data *noc;
	struct tegra194_cbb *cbb;
	struct device_node *np;
	unsigned long flags;
	int err;

	noc = of_device_get_match_data(&pdev->dev);

	if (noc->erd_mask_inband_err) {
		/*
		 * Set Error Response Disable(ERD) bit to mask SError/inband
		 * error and only trigger interrupts for illegal access from
		 * CCPLEX initiator.
		 */
		err = tegra194_miscreg_mask_serror();
		if (err) {
			dev_err(&pdev->dev, "couldn't mask inband errors\n");
			return err;
		}
	}

	cbb = devm_kzalloc(&pdev->dev, sizeof(*cbb), GFP_KERNEL);
	if (!cbb)
		return -ENOMEM;

	INIT_LIST_HEAD(&cbb->base.node);
	cbb->base.ops = &tegra194_cbb_ops;
	cbb->base.dev = &pdev->dev;
	cbb->noc = noc;

	cbb->regs = devm_platform_get_and_ioremap_resource(pdev, 0, &cbb->res);
	if (IS_ERR(cbb->regs))
		return PTR_ERR(cbb->regs);

	err = tegra_cbb_get_irq(pdev, &cbb->nonsec_irq, &cbb->sec_irq);
	if (err)
		return err;

	np = of_parse_phandle(pdev->dev.of_node, "nvidia,axi2apb", 0);
	if (np) {
		err = tegra194_cbb_get_bridges(cbb, np);
		of_node_put(np);
		if (err < 0)
			return err;
	}

	platform_set_drvdata(pdev, cbb);

	spin_lock_irqsave(&cbb_lock, flags);
	list_add(&cbb->base.node, &cbb_list);
	spin_unlock_irqrestore(&cbb_lock, flags);

	return tegra_cbb_register(&cbb->base);
}

static void tegra194_cbb_remove(struct platform_device *pdev)
{
	struct tegra194_cbb *cbb = platform_get_drvdata(pdev);
	struct tegra_cbb *noc, *tmp;
	unsigned long flags;

	spin_lock_irqsave(&cbb_lock, flags);

	list_for_each_entry_safe(noc, tmp, &cbb_list, node) {
		struct tegra194_cbb *priv = to_tegra194_cbb(noc);

		if (cbb->res->start == priv->res->start) {
			list_del(&noc->node);
			break;
		}
	}

	spin_unlock_irqrestore(&cbb_lock, flags);
}

static int __maybe_unused tegra194_cbb_resume_noirq(struct device *dev)
{
	struct tegra194_cbb *cbb = dev_get_drvdata(dev);

	tegra194_cbb_error_enable(&cbb->base);
	dsb(sy);

	dev_dbg(dev, "%s resumed\n", cbb->noc->name);
	return 0;
}

static const struct dev_pm_ops tegra194_cbb_pm = {
	SET_NOIRQ_SYSTEM_SLEEP_PM_OPS(NULL, tegra194_cbb_resume_noirq)
};

static struct platform_driver tegra194_cbb_driver = {
	.probe = tegra194_cbb_probe,
	.remove_new = tegra194_cbb_remove,
	.driver = {
		.name = "tegra194-cbb",
		.of_match_table = of_match_ptr(tegra194_cbb_match),
		.pm = &tegra194_cbb_pm,
	},
};

static int __init tegra194_cbb_init(void)
{
	return platform_driver_register(&tegra194_cbb_driver);
}
pure_initcall(tegra194_cbb_init);

static void __exit tegra194_cbb_exit(void)
{
	platform_driver_unregister(&tegra194_cbb_driver);
}
module_exit(tegra194_cbb_exit);

MODULE_AUTHOR("Sumit Gupta <sumitg@nvidia.com>");
MODULE_DESCRIPTION("Control Backbone error handling driver for Tegra194");
