/*
 * Copyright (c) 2007, Dennis M. Sosnoski All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 * 
 * Redistributions of source code must retain the above copyright notice, this list of conditions and the following
 * disclaimer. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 * following disclaimer in the documentation and/or other materials provided with the distribution. Neither the name of
 * JiBX nor the names of its contributors may be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.jibx.ws.wsdl;

import java.util.List;

import org.jibx.binding.generator.CustomBase;
import org.jibx.binding.util.StringArray;
import org.jibx.runtime.IUnmarshallingContext;

/**
 * Method parameter or return collection value customization information.
 * 
 * @author Dennis M. Sosnoski
 */
public class CollectionValueCustom extends ValueCustom
{
    /** Enumeration of allowed attribute names */
    public static final StringArray s_allowedAttributes =
        new StringArray(new String[] { "item-element-name", "item-type" }, ValueCustom.s_allowedAttributes);
    
    // value customization information
    private String m_itemType;
    
    private String m_itemName;
    
    /**
     * Constructor.
     * 
     * @param parent
     * @param name
     */
    protected CollectionValueCustom(NestingBase parent, String name) {
        super(parent, name);
    }
    
    /**
     * Make sure all attributes are defined.
     * 
     * @param uctx unmarshalling context
     */
    private void preSet(IUnmarshallingContext uctx) {
        validateAttributes(uctx, s_allowedAttributes);
    }
    
    /**
     * Get item type for collection. This method should only be used after the
     * {@link #complete(String, String, List, Boolean)} method is called.
     * 
     * @return collection item type, <code>null</code> if not a collection
     */
    public String getItemType() {
        return m_itemType;
    }
    
    /**
     * Get name for elements representing items in collection. This method should only be used after the
     * {@link #complete(String, String, List, Boolean)} method is called.
     * 
     * @return collection item type, <code>null</code> if not a collection
     */
    public String getItemElementName() {
        return m_itemName;
    }
    
    /**
     * Complete customization information based on supplied type. If the type information has not previously been set,
     * this will set it. It will also derive the appropriate XML name, if not previously set.
     * 
     * @param type
     * @param itype
     * @param docs default documentation node list (<code>null</code> if none)
     * @param req required member flag (<code>null</code> if unknown)
     */
    /* package */void complete(String type, String itype, List docs, Boolean req) {
        if (m_itemType == null) {
            m_itemType = itype;
        }
        complete(type, docs, req);
        if (m_itemName == null && getElementName() != null) {
            m_itemName = deriveItemName(getElementName(), m_itemType, CustomBase.CAMEL_CASE_NAMES);
        }
    }
}