/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "doc.hxx"
#include "ndtxt.hxx"
#include "MarkManager.hxx"
#include "docary.hxx"
#include "switerator.hxx"
#include "fmtfld.hxx"
#include "docufld.hxx"
#include "txatbase.hxx"
#include "fmtautofmt.hxx"
#include "charfmt.hxx"
#include "paratr.hxx"
#include <svl/itemiter.hxx>
#include <svl/intitem.hxx>

#include <libxml/encoding.h>
#include <libxml/xmlwriter.h>
#include <boost/optional.hpp>

namespace
{

// Small helper class to ensure that we write to nodes.xml if nothing
// has been explicitly specified.
// Always use at the beginning of dumpAsXml().
// Also, there are some functions to save typing.
class WriterHelper
{
public:
    WriterHelper( xmlTextWriterPtr );
    ~WriterHelper();
    operator xmlTextWriterPtr();
    void startElement( const char* element );
    void endElement();
    void writeFormatAttribute( const char* attribute, const char* format, ... )
#ifdef LIBXML_ATTR_FORMAT
        LIBXML_ATTR_FORMAT(3,4)
#endif
        ;
private:
    xmlTextWriterPtr writer;
    bool owns;
};

WriterHelper::WriterHelper( xmlTextWriterPtr w )
    : writer( w )
    , owns( false )
{
    if( writer == NULL )
    {
        writer = xmlNewTextWriterFilename( "nodes.xml", 0 );
        xmlTextWriterStartDocument( writer, NULL, NULL, NULL );
        owns = true;
    }
}

WriterHelper::~WriterHelper()
{
    if( owns )
    {
        xmlTextWriterEndDocument( writer );
        xmlFreeTextWriter( writer );
    }
}

WriterHelper::operator xmlTextWriterPtr()
{
    return writer;
}

void WriterHelper::startElement( const char* element )
{
    xmlTextWriterStartElement( writer, BAD_CAST( element ));
}

void WriterHelper::endElement()
{
    xmlTextWriterEndElement( writer );
}

void WriterHelper::writeFormatAttribute( const char* attribute, const char* format, ... )
{
    va_list va;
    va_start( va, format );
    xmlTextWriterWriteVFormatAttribute( writer, BAD_CAST( attribute ), format, va );
    va_end( va );
}

// Hack: somehow conversion from "..." to va_list does
// bomb on two string litterals in the format.
static const char* TMP_FORMAT = "%" SAL_PRIuUINTPTR;
static const char* TMP_FORMAT_I32 = "%" SAL_PRIdINT32;

}

void SwDoc::dumpAsXml( xmlTextWriterPtr w )
{
    WriterHelper writer( w );
    writer.startElement( "doc" );
    writer.writeFormatAttribute( "ptr", "%p", this );
    m_pNodes->dumpAsXml( writer );
    mpMarkManager->dumpAsXml( writer );
    mpFldTypes->dumpAsXml( writer );
    mpTxtFmtCollTbl->dumpAsXml( writer );
    mpCharFmtTbl->dumpAsXml( writer );
    mpNumRuleTbl->dumpAsXml( writer );
    writer.endElement();
}

namespace sw {
namespace mark {
void MarkManager::dumpAsXml( xmlTextWriterPtr w )
{
    WriterHelper writer(w);
    writer.startElement("markManager");
    writer.startElement("fieldmarks");
    for (const_iterator_t it = m_vFieldmarks.begin(); it != m_vFieldmarks.end(); ++it)
    {
        pMark_t pMark = *it;
        writer.startElement("fieldmark");
        writer.writeFormatAttribute("startNode", TMP_FORMAT, pMark->GetMarkStart().nNode.GetIndex());
        writer.writeFormatAttribute("startOffset", TMP_FORMAT_I32, pMark->GetMarkStart().nContent.GetIndex());
        writer.writeFormatAttribute("endNode", TMP_FORMAT, pMark->GetMarkEnd().nNode.GetIndex());
        writer.writeFormatAttribute("endOffset", TMP_FORMAT_I32, pMark->GetMarkEnd().nContent.GetIndex());
        OString txt8 = OUStringToOString(pMark->GetName(), RTL_TEXTENCODING_UTF8);
        writer.writeFormatAttribute("name", "%s", BAD_CAST( txt8.getStr()));
        writer.endElement();
    }
    writer.endElement();
    writer.endElement();
}
} // namespace mark
} // namespace sw

void SwFldTypes::dumpAsXml( xmlTextWriterPtr w )
{
    WriterHelper writer(w);
    writer.startElement("swfldtypes");
    sal_uInt16 nCount = size();
    for (sal_uInt16 nType = 0; nType < nCount; ++nType)
    {
        const SwFieldType *pCurType = (*this)[nType];
        SwIterator<SwFmtFld, SwFieldType> aIter(*pCurType);
        for (const SwFmtFld* pCurFldFmt = aIter.First(); pCurFldFmt; pCurFldFmt = aIter.Next())
        {
            writer.startElement("swfmtfld");
            writer.writeFormatAttribute("ptr", "%p", pCurFldFmt);
            writer.writeFormatAttribute("pTxtAttr", "%p", pCurFldFmt->GetTxtFld());
            const char* name = "???";
            switch(pCurFldFmt->GetField()->GetTyp()->Which())
            {
                case RES_PAGENUMBERFLD: name = "swpagenumberfield"; break;
                case RES_POSTITFLD: name = "swpostitfield"; break;
                case RES_DATETIMEFLD: name = "swdatetimefield"; break;
                default:
                    SAL_INFO("sw.core", "unhandled field type " << pCurFldFmt->GetField()->GetTyp()->Which());
                    break;
            }
            writer.startElement(name);
            writer.writeFormatAttribute("ptr", "%p", pCurFldFmt->GetField());
            if (pCurFldFmt->GetField()->GetTyp()->Which() == RES_POSTITFLD)
            {
                const SwPostItField* pField = dynamic_cast<const SwPostItField*>(pCurFldFmt->GetField());
                OString txt8 = OUStringToOString(pField->GetName(), RTL_TEXTENCODING_UTF8);
                writer.writeFormatAttribute("name", "%s", BAD_CAST( txt8.getStr()));
            }
            writer.endElement();
            writer.endElement();
        }
    }
    writer.endElement();
}

void SwNodes::dumpAsXml( xmlTextWriterPtr w )
{
    WriterHelper writer( w );
    writer.startElement( "swnodes" );
    writer.writeFormatAttribute( "ptr", "%p", this );
    for( unsigned int i = 0; i < Count(); ++i )
    {
        ( *this )[ i ]->dumpAsXml( writer );
    }
    writer.endElement();
}

void SwNode::dumpAsXml( xmlTextWriterPtr w )
{
    WriterHelper writer( w );
    const char* name = "???";
    switch( GetNodeType())
    {
        case ND_ENDNODE:
            name = "end";
            break;
        case ND_STARTNODE:
        case ND_TEXTNODE:
            abort(); // overriden
        case ND_TABLENODE:
            name = "table";
            break;
        case ND_GRFNODE:
            name = "grf";
            break;
        case ND_OLENODE:
            name = "ole";
            break;
    }
    writer.startElement( name );
    writer.writeFormatAttribute( "ptr", "%p", this );
    writer.writeFormatAttribute( "index", TMP_FORMAT, GetIndex() );
    writer.endElement();
    if( GetNodeType() == ND_ENDNODE )
        writer.endElement(); // end start node
}

void SwStartNode::dumpAsXml( xmlTextWriterPtr w )
{
    WriterHelper writer( w );
    const char* name = "???";
    switch( GetNodeType() )
    {
        case ND_TABLENODE:
            name = "table";
            break;
        case ND_SECTIONNODE:
            name = "section";
            break;
        default:
            switch( GetStartNodeType())
            {
                case SwNormalStartNode:
                    name = "start";
                    break;
                case SwTableBoxStartNode:
                    name = "tablebox";
                    break;
                case SwFlyStartNode:
                    name = "fly";
                    break;
                case SwFootnoteStartNode:
                    name = "footnote";
                    break;
                case SwHeaderStartNode:
                    name = "header";
                    break;
                case SwFooterStartNode:
                    name = "footer";
                    break;
            }
            break;
    }
    writer.startElement( name );
    writer.writeFormatAttribute( "ptr", "%p", this );
    writer.writeFormatAttribute( "index", TMP_FORMAT, GetIndex() );
    // writer.endElement(); - it is a start node, so don't end, will make xml better nested
}

void SwCharFmts::dumpAsXml(xmlTextWriterPtr w)
{
    WriterHelper writer(w);
    if (size())
    {
        writer.startElement("swcharfmts");
        for (size_t i = 0; i < size(); ++i)
        {
            SwCharFmt* pFmt = static_cast<SwCharFmt*>(GetFmt(i));
            writer.startElement("swcharfmt");
            OString aName = OUStringToOString(pFmt->GetName(), RTL_TEXTENCODING_UTF8);
            writer.writeFormatAttribute("name", "%s", BAD_CAST(aName.getStr()));
            writer.endElement();
        }
        writer.endElement();
    }
}

void lcl_dumpSfxItemSet(WriterHelper& writer, const SfxItemSet* pSet)
{
    SfxItemIter aIter(*pSet);
    const SfxPoolItem* pItem = aIter.FirstItem();
    while (pItem)
    {
        writer.startElement("item");
        writer.writeFormatAttribute("whichId", TMP_FORMAT, pItem->Which());
        const char* pWhich = 0;
        boost::optional<OString> oValue;
        switch (pItem->Which())
        {
            case RES_CHRATR_POSTURE: pWhich = "character posture"; break;
            case RES_CHRATR_WEIGHT: pWhich = "character weight"; break;
            case RES_CHRATR_CJK_POSTURE: pWhich = "character cjk posture"; break;
            case RES_CHRATR_CJK_WEIGHT: pWhich = "character cjk weight"; break;
            case RES_CHRATR_CTL_POSTURE: pWhich = "character ctl posture"; break;
            case RES_CHRATR_CTL_WEIGHT: pWhich = "character ctl weight"; break;
            case RES_CHRATR_RSID: pWhich = "character rsid"; break;
            case RES_PARATR_OUTLINELEVEL: pWhich = "paragraph outline level"; oValue = OString::number(static_cast<const SfxUInt16Item*>(pItem)->GetValue()); break;
            case RES_PARATR_NUMRULE: pWhich = "paragraph numbering rule"; oValue = OUStringToOString(static_cast<const SwNumRuleItem*>(pItem)->GetValue(), RTL_TEXTENCODING_UTF8); break;
        }
        if (pWhich)
            writer.writeFormatAttribute("which", "%s", BAD_CAST(pWhich));
        if (oValue)
            writer.writeFormatAttribute("value", "%s", BAD_CAST(oValue->getStr()));
        pItem = aIter.NextItem();
        writer.endElement();
    }
}

void SwTxtFmtColls::dumpAsXml(xmlTextWriterPtr w)
{
    WriterHelper writer(w);
    if (size())
    {
        writer.startElement("swtxtfmtcolls");
        for (size_t i = 0; i < size(); ++i)
        {
            SwTxtFmtColl* pColl = static_cast<SwTxtFmtColl*>(GetFmt(i));
            writer.startElement("swtxtfmtcoll");
            OString aName = OUStringToOString(pColl->GetName(), RTL_TEXTENCODING_UTF8);
            writer.writeFormatAttribute("name", "%s", BAD_CAST(aName.getStr()));

            lcl_dumpSfxItemSet(writer, &pColl->GetAttrSet());
            writer.endElement();
        }
        writer.endElement();
    }
}

void SwNumRule::dumpAsXml(xmlTextWriterPtr w)
{
     WriterHelper writer(w);
     writer.startElement("swnumrule");
     OString aName = OUStringToOString(GetName(), RTL_TEXTENCODING_UTF8);
     writer.writeFormatAttribute("name", "%s", BAD_CAST(aName.getStr()));
     writer.writeFormatAttribute("isautorule", TMP_FORMAT, IsAutoRule());
     if (GetPoolFmtId() != USHRT_MAX)
         writer.writeFormatAttribute("poolfmtid", TMP_FORMAT, GetPoolFmtId());
     writer.endElement();
}

void SwNumRuleTbl::dumpAsXml(xmlTextWriterPtr w)
{
    if (!empty())
    {
        WriterHelper writer(w);
        writer.startElement("swnumruletbl");
        for (size_t i = 0; i < size(); ++i)
            operator[](i)->dumpAsXml(w);
        writer.endElement();
    }
}

void SwTxtNode::dumpAsXml( xmlTextWriterPtr w )
{
    WriterHelper writer( w );
    writer.startElement( "text" );
    writer.writeFormatAttribute( "ptr", "%p", this );
    writer.writeFormatAttribute( "index", TMP_FORMAT, GetIndex() );
    OUString txt = GetTxt();
    for( int i = 0; i < 32; ++i )
        txt = txt.replace( i, '*' );
    OString txt8 = OUStringToOString( txt, RTL_TEXTENCODING_UTF8 );
    xmlTextWriterWriteString( writer, BAD_CAST( txt8.getStr()));

    if (GetFmtColl())
    {
        SwTxtFmtColl* pColl = static_cast<SwTxtFmtColl*>(GetFmtColl());
        writer.startElement("swtxtfmtcoll");
        OString aName = OUStringToOString(pColl->GetName(), RTL_TEXTENCODING_UTF8);
        writer.writeFormatAttribute("name", "%s", BAD_CAST(aName.getStr()));
        writer.endElement();
    }

    if (HasSwAttrSet())
    {
        writer.startElement("attrset");
        const SwAttrSet& rAttrSet = GetSwAttrSet();
        lcl_dumpSfxItemSet(writer, &rAttrSet);
        writer.endElement();
    }

    if (HasHints())
    {
        writer.startElement("hints");
        SwpHints& rHints = GetSwpHints();
        for (sal_uInt16 i = 0; i < rHints.Count(); ++i)
        {
            writer.startElement("hint");
            SwTxtAttr* pHint = rHints.GetTextHint(i);

            if (pHint->GetStart())
                writer.writeFormatAttribute("start", TMP_FORMAT, *pHint->GetStart());
            if (pHint->GetEnd())
                writer.writeFormatAttribute("end", TMP_FORMAT, *pHint->GetEnd());

            const char* pWhich = "???";
            switch (pHint->Which())
            {
                case RES_TXTATR_AUTOFMT:
                    pWhich = "autofmt";
                    break;
                default:
                    break;
            }
            writer.writeFormatAttribute("which", "%s", BAD_CAST(pWhich));

            if (pHint->Which() == RES_TXTATR_AUTOFMT)
            {
                boost::shared_ptr<SfxItemSet> const pSet(pHint->GetAutoFmt().GetStyleHandle());
                writer.startElement("autofmt");
                lcl_dumpSfxItemSet(writer, pSet.get());
                writer.endElement();
            }

            writer.endElement();
        }
        writer.endElement();
    }
    if (GetNumRule())
        GetNumRule()->dumpAsXml(w);

    writer.endElement();
}

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
