# Copyright 2012-2016 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Model base class with creation/update timestamps."""

__all__ = [
    'now',
    'TimestampedModel',
    ]


from django.db import connection
from django.db.models import (
    DateTimeField,
    Model,
)
from maasserver import DefaultMeta


def now():
    """Current database time (as per start of current transaction)."""
    cursor = connection.cursor()
    cursor.execute("select now()")
    return cursor.fetchone()[0]


# Having 'object' here should not be required, but it is a workaround for the
# bug in PyCharm described here:
#     https://youtrack.jetbrains.com/issue/PY-12566
class TimestampedModel(Model, object):
    """Abstract base model with creation/update timestamps.

    Timestamps are taken from the database transaction clock.

    :ivar created: Object's creation time.
    :ivar updated: Time of object's latest update.
    """

    class Meta(DefaultMeta):
        abstract = True

    created = DateTimeField(editable=False)
    updated = DateTimeField(editable=False)

    def save(self, *args, **kwargs):
        """Set `created` and `updated` before saving.

        If the record is new (its ``id`` is `None`) then `created` is set to
        the current time if it has not already been set. Then `updated` is set
        to the same as `created`.

        If the record already exists, `updated` is set to the current time.
        """
        if self.id is None:
            # New record; set created if not set.
            if self.created is None:
                self.created = now()
            # Set updated to same as created.
            self.updated = self.created
        else:
            # Existing record; set updated always.
            self.updated = now()

        return super(TimestampedModel, self).save(*args, **kwargs)
