#    Copyright 2010 OpenStack Foundation
#    Copyright 2012 University Of Minho
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import binascii
import collections
import contextlib
import copy
import datetime
import errno
import glob
import io
import os
import random
import re
import shutil
import signal
import sys
import testtools
import threading
import time
import unittest
from unittest import mock

from castellan import key_manager
import ddt
import eventlet
from eventlet import greenthread
import fixtures
from lxml import etree
from os_brick import encryptors
from os_brick import exception as brick_exception
from os_brick.initiator import connector
import os_resource_classes as orc
import os_traits as ot
import os_vif
from oslo_concurrency import lockutils
from oslo_concurrency import processutils
from oslo_config import cfg
from oslo_serialization import jsonutils
from oslo_service import loopingcall
from oslo_utils import fileutils
from oslo_utils import fixture as utils_fixture
from oslo_utils.fixture import uuidsentinel as uuids
from oslo_utils import strutils
from oslo_utils import units
from oslo_utils import uuidutils
from oslo_utils import versionutils

from nova.api.metadata import base as instance_metadata
from nova.compute import manager
from nova.compute import power_state
from nova.compute import provider_tree
from nova.compute import task_states
from nova.compute import utils as compute_utils
from nova.compute import vm_states
import nova.conf
from nova import context
from nova.db import constants as db_const
from nova.db.main import api as db
from nova import exception
from nova.network import model as network_model
from nova import objects
from nova.objects import block_device as block_device_obj
from nova.objects import fields
from nova.objects import migrate_data as migrate_data_obj
from nova.objects import virtual_interface as obj_vif
from nova.pci import utils as pci_utils
import nova.privsep.fs
import nova.privsep.libvirt
from nova.storage import rbd_utils
from nova import test
from nova.tests import fixtures as nova_fixtures
from nova.tests.fixtures import libvirt as fakelibvirt
from nova.tests.fixtures import libvirt_data as fake_libvirt_data
from nova.tests.unit import fake_block_device
from nova.tests.unit import fake_diagnostics
from nova.tests.unit import fake_flavor
from nova.tests.unit import fake_instance
from nova.tests.unit import fake_network
import nova.tests.unit.image.fake as fake_image
from nova.tests.unit import matchers
from nova.tests.unit.objects import test_diagnostics
from nova.tests.unit.objects import test_pci_device
from nova.tests.unit.objects import test_vcpu_model
from nova.tests.unit import utils as test_utils
from nova import utils
from nova import version
from nova.virt import block_device as driver_block_device
from nova.virt import driver
from nova.virt import fake
from nova.virt import hardware
from nova.virt.image import model as imgmodel
from nova.virt.libvirt import blockinfo
from nova.virt.libvirt import config as vconfig
from nova.virt.libvirt import designer
from nova.virt.libvirt import driver as libvirt_driver
from nova.virt.libvirt import event as libvirtevent
from nova.virt.libvirt import guest as libvirt_guest
from nova.virt.libvirt import host
from nova.virt.libvirt.host import SEV_KERNEL_PARAM_FILE
from nova.virt.libvirt import imagebackend
from nova.virt.libvirt import imagecache
from nova.virt.libvirt import migration as libvirt_migrate
from nova.virt.libvirt.storage import dmcrypt
from nova.virt.libvirt.storage import lvm
from nova.virt.libvirt import utils as libvirt_utils
from nova.virt.libvirt import vif as libvirt_vif
from nova.virt.libvirt.volume import volume as volume_drivers

CONF = nova.conf.CONF

_fake_network_info = fake_network.fake_get_instance_nw_info

# TODO(sean-k-mooney): move the rest of the static data to fake_libvirt_data
# and use it directly instead of creating local references to the common data
_fake_NodeDevXml = fake_libvirt_data._fake_NodeDevXml
_fake_NodeDevXml_parents = fake_libvirt_data._fake_NodeDevXml_parents
_fake_NodeDevXml_children = fake_libvirt_data._fake_NodeDevXml_children

_fake_cpu_info = {
    "arch": "test_arch",
    "model": "test_model",
    "vendor": "test_vendor",
    "topology": {
        "sockets": 1,
        "cores": 8,
        "threads": 16
    },
    "features": ["feature1", "feature2"]
}

_fake_cpu_info_aarch64 = {
    "arch": fields.Architecture.AARCH64,
    "model": "test_model",
    "vendor": "test_vendor",
    "topology": {
        "sockets": 1,
        "cores": 8,
        "threads": 16
    },
    "features": ["feature1", "feature2"]
}

eph_default_ext = utils.get_hash_str(nova.privsep.fs._DEFAULT_FILE_SYSTEM)[:7]

_fake_qemu64_cpu_feature = """
<cpu mode='custom' match='exact'>
  <model fallback='forbid'>qemu64</model>
  <feature policy='require' name='svm'/>
  <feature policy='require' name='lm'/>
  <feature policy='require' name='nx'/>
  <feature policy='require' name='syscall'/>
  <feature policy='require' name='cx16'/>
  <feature policy='require' name='pni'/>
  <feature policy='require' name='sse2'/>
  <feature policy='require' name='sse'/>
  <feature policy='require' name='fxsr'/>
  <feature policy='require' name='mmx'/>
  <feature policy='require' name='clflush'/>
  <feature policy='require' name='pse36'/>
  <feature policy='require' name='pat'/>
  <feature policy='require' name='cmov'/>
  <feature policy='require' name='mca'/>
  <feature policy='require' name='pge'/>
  <feature policy='require' name='mtrr'/>
  <feature policy='require' name='sep'/>
  <feature policy='require' name='apic'/>
  <feature policy='require' name='cx8'/>
  <feature policy='require' name='mce'/>
  <feature policy='require' name='pae'/>
  <feature policy='require' name='msr'/>
  <feature policy='require' name='tsc'/>
  <feature policy='require' name='pse'/>
  <feature policy='require' name='de'/>
  <feature policy='require' name='fpu'/>
</cpu>
"""

_fake_sandy_bridge_cpu_feature = """<cpu mode='custom' match='exact'>
  <model fallback='forbid'>SandyBridge</model>
  <feature policy='require' name='aes'/>
  <feature policy='require' name='apic'/>
  <feature policy='require' name='avx'/>
  <feature policy='require' name='clflush'/>
  <feature policy='require' name='cmov'/>
  <feature policy='require' name='cx16'/>
  <feature policy='require' name='cx8'/>
  <feature policy='require' name='de'/>
  <feature policy='require' name='fpu'/>
  <feature policy='require' name='fxsr'/>
  <feature policy='require' name='lahf_lm'/>
  <feature policy='require' name='lm'/>
  <feature policy='require' name='mca'/>
  <feature policy='require' name='mce'/>
  <feature policy='require' name='mmx'/>
  <feature policy='require' name='msr'/>
  <feature policy='require' name='mtrr'/>
  <feature policy='require' name='nx'/>
  <feature policy='require' name='pae'/>
  <feature policy='require' name='pat'/>
  <feature policy='require' name='pclmuldq'/>
  <feature policy='require' name='pge'/>
  <feature policy='require' name='pni'/>
  <feature policy='require' name='popcnt'/>
  <feature policy='require' name='pse'/>
  <feature policy='require' name='pse36'/>
  <feature policy='require' name='rdtscp'/>
  <feature policy='require' name='sep'/>
  <feature policy='require' name='sse'/>
  <feature policy='require' name='sse2'/>
  <feature policy='require' name='sse4.1'/>
  <feature policy='require' name='sse4.2'/>
  <feature policy='require' name='ssse3'/>
  <feature policy='require' name='syscall'/>
  <feature policy='require' name='tsc'/>
  <feature policy='require' name='tsc-deadline'/>
  <feature policy='require' name='x2apic'/>
  <feature policy='require' name='xsave'/>
</cpu>
"""

_fake_broadwell_cpu_feature = """
<cpu mode='custom' match='exact'>
  <model fallback='forbid'>Broadwell-noTSX</model>
  <vendor>Intel</vendor>
  <feature policy='require' name='smap'/>
  <feature policy='require' name='adx'/>
  <feature policy='require' name='rdseed'/>
  <feature policy='require' name='invpcid'/>
  <feature policy='require' name='erms'/>
  <feature policy='require' name='bmi2'/>
  <feature policy='require' name='smep'/>
  <feature policy='require' name='avx2'/>
  <feature policy='require' name='bmi1'/>
  <feature policy='require' name='fsgsbase'/>
  <feature policy='require' name='3dnowprefetch'/>
  <feature policy='require' name='lahf_lm'/>
  <feature policy='require' name='lm'/>
  <feature policy='require' name='rdtscp'/>
  <feature policy='require' name='nx'/>
  <feature policy='require' name='syscall'/>
  <feature policy='require' name='avx'/>
  <feature policy='require' name='xsave'/>
  <feature policy='require' name='aes'/>
  <feature policy='require' name='tsc-deadline'/>
  <feature policy='require' name='popcnt'/>
  <feature policy='require' name='movbe'/>
  <feature policy='require' name='x2apic'/>
  <feature policy='require' name='sse4.2'/>
  <feature policy='require' name='sse4.1'/>
  <feature policy='require' name='pcid'/>
  <feature policy='require' name='cx16'/>
  <feature policy='require' name='fma'/>
  <feature policy='require' name='ssse3'/>
  <feature policy='require' name='pclmuldq'/>
  <feature policy='require' name='pni'/>
  <feature policy='require' name='sse2'/>
  <feature policy='require' name='sse'/>
  <feature policy='require' name='fxsr'/>
  <feature policy='require' name='mmx'/>
  <feature policy='require' name='clflush'/>
  <feature policy='require' name='pse36'/>
  <feature policy='require' name='pat'/>
  <feature policy='require' name='cmov'/>
  <feature policy='require' name='mca'/>
  <feature policy='require' name='pge'/>
  <feature policy='require' name='mtrr'/>
  <feature policy='require' name='sep'/>
  <feature policy='require' name='apic'/>
  <feature policy='require' name='cx8'/>
  <feature policy='require' name='mce'/>
  <feature policy='require' name='pae'/>
  <feature policy='require' name='msr'/>
  <feature policy='require' name='tsc'/>
  <feature policy='require' name='pse'/>
  <feature policy='require' name='de'/>
  <feature policy='require' name='fpu'/>
</cpu>
"""


def eph_name(size):
    return ('ephemeral_%(size)s_%(ext)s' %
            {'size': size, 'ext': eph_default_ext})


def fake_disk_info_byname(instance, type='qcow2'):
    """Return instance_disk_info corresponding accurately to the properties of
    the given Instance object. The info is returned as an OrderedDict of
    name->disk_info for each disk.

    :param instance: The instance we're generating fake disk_info for.
    :param type: libvirt's disk type.
    :return: disk_info
    :rtype: collections.OrderedDict
    """
    instance_dir = os.path.join(CONF.instances_path, instance.uuid)

    def instance_path(name):
        return os.path.join(instance_dir, name)

    disk_info = collections.OrderedDict()

    # root disk
    if (instance.image_ref is not None and
            instance.image_ref != uuids.fake_volume_backed_image_ref):
        cache_name = imagecache.get_cache_fname(instance.image_ref)
        disk_info['disk'] = {
            'type': type,
            'path': instance_path('disk'),
            'virt_disk_size': instance.flavor.root_gb * units.Gi,
            'backing_file': cache_name,
            'disk_size': instance.flavor.root_gb * units.Gi,
            'over_committed_disk_size': 0}

    swap_mb = instance.flavor.swap
    if swap_mb > 0:
        disk_info['disk.swap'] = {
            'type': type,
            'path': instance_path('disk.swap'),
            'virt_disk_size': swap_mb * units.Mi,
            'backing_file': 'swap_%s' % swap_mb,
            'disk_size': swap_mb * units.Mi,
            'over_committed_disk_size': 0}

    eph_gb = instance.flavor.ephemeral_gb
    if eph_gb > 0:
        disk_info['disk.local'] = {
            'type': type,
            'path': instance_path('disk.local'),
            'virt_disk_size': eph_gb * units.Gi,
            'backing_file': eph_name(eph_gb),
            'disk_size': eph_gb * units.Gi,
            'over_committed_disk_size': 0}

    if instance.config_drive:
        disk_info['disk.config'] = {
            'type': 'raw',
            'path': instance_path('disk.config'),
            'virt_disk_size': 1024,
            'backing_file': '',
            'disk_size': 1024,
            'over_committed_disk_size': 0}

    return disk_info


def fake_diagnostics_object(with_cpus=False, with_disks=False, with_nic=False):
    diag_dict = {'config_drive': False,
                 'driver': 'libvirt',
                 'hypervisor': 'kvm',
                 'hypervisor_os': 'linux',
                 'memory_details': {'maximum': 2048, 'used': 1234},
                 'state': 'running',
                 'uptime': 10}

    if with_cpus:
        diag_dict['cpu_details'] = []
        for id, t in enumerate([15340000000, 1640000000,
                                3040000000, 1420000000]):
            diag_dict['cpu_details'].append({'id': id, 'time': t})

    if with_disks:
        diag_dict['disk_details'] = []
        for i in range(2):
            diag_dict['disk_details'].append(
                {'read_bytes': 688640,
                 'read_requests': 169,
                 'write_bytes': 0,
                 'write_requests': 0,
                 'errors_count': 1})

    if with_nic:
        diag_dict['nic_details'] = [
            {'mac_address': '52:54:00:a4:38:38',
             'rx_drop': 0,
             'rx_errors': 0,
             'rx_octets': 4408,
             'rx_packets': 82,
             'tx_drop': 0,
             'tx_errors': 0,
             'tx_octets': 0,
             'tx_packets': 0}]

    return fake_diagnostics.fake_diagnostics_obj(**diag_dict)


def fake_disk_info_json(instance, type='qcow2'):
    """Return fake instance_disk_info corresponding accurately to the
    properties of the given Instance object.

    :param instance: The instance we're generating fake disk_info for.
    :param type: libvirt's disk type.
    :return: JSON representation of instance_disk_info for all disks.
    :rtype: str
    """
    disk_info = fake_disk_info_byname(instance, type)
    return jsonutils.dumps(disk_info.values())


def get_injection_info(network_info=None, admin_pass=None, files=None):
    return libvirt_driver.InjectionInfo(
        network_info=network_info, admin_pass=admin_pass, files=files)


def _concurrency(signal, wait, done, target, is_block_dev=False):
    signal.send()
    wait.wait()
    done.send()


class FakeVirtDomain(object):

    def __init__(self, fake_xml=None, uuidstr=None, id=None, name=None,
                 info=None):
        if uuidstr is None:
            uuidstr = uuids.fake
        self.uuidstr = uuidstr
        self.id = id
        self.domname = name
        self._info = info or (
            [power_state.RUNNING, 2048 * units.Mi,
             1234 * units.Mi, None, None])
        if fake_xml:
            self._fake_dom_xml = fake_xml
        else:
            self._fake_dom_xml = """
                <domain type='kvm'>
                    <name>testinstance1</name>
                    <devices>
                        <disk type='file'>
                            <source file='filename'/>
                        </disk>
                    </devices>
                </domain>
            """

    def name(self):
        if self.domname is None:
            return "fake-domain %s" % self
        else:
            return self.domname

    def ID(self):
        return self.id

    def info(self):
        return self._info

    def create(self):
        pass

    def managedSave(self, *args):
        pass

    def createWithFlags(self, launch_flags):
        pass

    def XMLDesc(self, flags):
        return self._fake_dom_xml

    def UUIDString(self):
        return self.uuidstr

    def attachDeviceFlags(self, xml, flags):
        pass

    def attachDevice(self, xml):
        pass

    def detachDeviceFlags(self, xml, flags):
        pass

    def snapshotCreateXML(self, xml, flags):
        pass

    def blockCommit(self, disk, base, top, bandwidth=0, flags=0):
        pass

    def blockRebase(self, disk, base, bandwidth=0, flags=0):
        pass

    def blockJobInfo(self, path, flags):
        pass

    def blockJobAbort(self, path, flags):
        pass

    def resume(self):
        pass

    def destroy(self):
        pass

    def fsFreeze(self, disks=None, flags=0):
        pass

    def fsThaw(self, disks=None, flags=0):
        pass

    def isActive(self):
        return True

    def isPersistent(self):
        return True

    def undefine(self):
        return True

    def setMetadata(self, metadata_type, metadata, key, uri, flags=0):
        pass


class CacheConcurrencyTestCase(test.NoDBTestCase):
    def setUp(self):
        super(CacheConcurrencyTestCase, self).setUp()

        self.flags(instances_path=self.useFixture(fixtures.TempDir()).path)

        # utils.synchronized() will create the lock_path for us if it
        # doesn't already exist. It will also delete it when it's done,
        # which can cause race conditions with the multiple threads we
        # use for tests. So, create the path here so utils.synchronized()
        # won't delete it out from under one of the threads.
        self.lock_path = os.path.join(CONF.instances_path, 'locks')
        fileutils.ensure_tree(self.lock_path)

        def fake_exists(fname):
            basedir = os.path.join(CONF.instances_path,
                                   CONF.image_cache.subdirectory_name)
            if fname == basedir or fname == self.lock_path:
                return True
            return False

        self.stub_out('os.path.exists', fake_exists)
        self.stub_out('oslo_concurrency.processutils.execute',
                      lambda *a, **kw: None)
        self.stub_out('nova.virt.disk.api.extend',
                      lambda image, size, use_cow=False: None)

    def _fake_instance(self, uuid):
        return objects.Instance(id=1, uuid=uuid)

    def test_same_fname_concurrency(self):
        # Ensures that the same fname cache runs at a sequentially.
        uuid = uuids.fake

        backend = imagebackend.Backend(False)
        wait1 = eventlet.event.Event()
        done1 = eventlet.event.Event()
        sig1 = eventlet.event.Event()
        thr1 = eventlet.spawn(backend.by_name(self._fake_instance(uuid),
                                              'name').cache,
                _concurrency, 'fname', None,
                signal=sig1, wait=wait1, done=done1)
        eventlet.sleep(0)
        # Thread 1 should run before thread 2.
        sig1.wait()

        wait2 = eventlet.event.Event()
        done2 = eventlet.event.Event()
        sig2 = eventlet.event.Event()
        thr2 = eventlet.spawn(backend.by_name(self._fake_instance(uuid),
                                              'name').cache,
                _concurrency, 'fname', None,
                signal=sig2, wait=wait2, done=done2)

        wait2.send()
        eventlet.sleep(0)
        try:
            self.assertFalse(done2.ready())
        finally:
            wait1.send()
        done1.wait()
        eventlet.sleep(0)
        self.assertTrue(done2.ready())
        # Wait on greenthreads to assert they didn't raise exceptions
        # during execution
        thr1.wait()
        thr2.wait()

    def test_different_fname_concurrency(self):
        # Ensures that two different fname caches are concurrent.
        uuid = uuids.fake

        backend = imagebackend.Backend(False)
        wait1 = eventlet.event.Event()
        done1 = eventlet.event.Event()
        sig1 = eventlet.event.Event()
        thr1 = eventlet.spawn(backend.by_name(self._fake_instance(uuid),
                                              'name').cache,
                _concurrency, 'fname2', None,
                signal=sig1, wait=wait1, done=done1)
        eventlet.sleep(0)
        # Thread 1 should run before thread 2.
        sig1.wait()

        wait2 = eventlet.event.Event()
        done2 = eventlet.event.Event()
        sig2 = eventlet.event.Event()
        thr2 = eventlet.spawn(backend.by_name(self._fake_instance(uuid),
                                              'name').cache,
                _concurrency, 'fname1', None,
                signal=sig2, wait=wait2, done=done2)
        eventlet.sleep(0)
        # Wait for thread 2 to start.
        sig2.wait()

        wait2.send()
        tries = 0
        while not done2.ready() and tries < 10:
            eventlet.sleep(0)
            tries += 1
        try:
            self.assertTrue(done2.ready())
        finally:
            wait1.send()
            eventlet.sleep(0)
        # Wait on greenthreads to assert they didn't raise exceptions
        # during execution
        thr1.wait()
        thr2.wait()


class FakeInvalidVolumeDriver(object):
    def __init__(self, *args, **kwargs):
        raise brick_exception.InvalidConnectorProtocol('oops!')


class FakeConfigGuestDisk(object):
    def __init__(self, *args, **kwargs):
        self.source_type = None
        self.driver_cache = None


class FakeConfigGuest(object):
    def __init__(self, *args, **kwargs):
        self.driver_cache = None


class FakeNodeDevice(object):
    def __init__(self, fakexml):
        self.xml = fakexml

    def XMLDesc(self, flags):
        return self.xml


def _create_test_instance():
    flavor = objects.Flavor(memory_mb=2048,
                            swap=0,
                            vcpu_weight=None,
                            root_gb=10,
                            id=2,
                            name=u'm1.small',
                            ephemeral_gb=20,
                            rxtx_factor=1.0,
                            flavorid=u'1',
                            vcpus=2,
                            extra_specs={})
    return {
        'id': 1,
        'uuid': uuids.instance,
        'memory_kb': '1024000',
        'basepath': '/some/path',
        'bridge_name': 'br100',
        'display_name': "Acme webserver",
        'vcpus': 2,
        'project_id': 'fake',
        'bridge': 'br101',
        'image_ref': '155d900f-4e14-4e4c-a73d-069cbf4541e6',
        'root_gb': 10,
        'ephemeral_gb': 20,
        'system_metadata': {
            'image_disk_format': 'raw'
        },
        'instance_type_id': flavor.id,
        'flavor': flavor,
        'new_flavor': None,
        'old_flavor': None,
        'pci_devices': objects.PciDeviceList(),
        'numa_topology': None,
        'config_drive': None,
        'vm_mode': None,
        'vm_state': None,
        'kernel_id': None,
        'ramdisk_id': None,
        'os_type': 'linux',
        'user_id': '838a72b0-0d54-4827-8fd6-fb1227633ceb',
        'ephemeral_key_uuid': None,
        'vcpu_model': None,
        'host': 'fake-host',
        'node': 'fake-node',
        'task_state': None,
        'vm_state': None,
        'trusted_certs': None,
        'resources': None,
        'migration_context': None,
        'info_cache': None,
        'cleaned': False,
    }


@ddt.ddt
class LibvirtConnTestCase(test.NoDBTestCase,
                          test_diagnostics.DiagnosticsComparisonMixin):

    REQUIRES_LOCKING = True

    _EPHEMERAL_20_DEFAULT = eph_name(20)

    def setUp(self):
        super(LibvirtConnTestCase, self).setUp()
        self.user_id = 'fake'
        self.project_id = 'fake'
        self.context = context.get_admin_context()
        temp_dir = self.useFixture(fixtures.TempDir()).path
        self.flags(instances_path=temp_dir)
        self.flags(snapshots_directory=temp_dir, group='libvirt')
        self.flags(sysinfo_serial="hardware", group="libvirt")

        # normally loaded during nova-compute startup
        os_vif.initialize()

        self.stub_out('nova.virt.disk.api.extend',
                      lambda image, size, use_cow=False: None)

        self.stub_out('nova.virt.libvirt.imagebackend.Image.'
                      'resolve_driver_format',
                      imagebackend.Image._get_driver_format)

        self.libvirt = self.useFixture(nova_fixtures.LibvirtFixture())
        self.cgroups = self.useFixture(nova_fixtures.CGroupsFixture())

        # ensure tests perform the same on all host architectures; this is
        # already done by the fakelibvirt fixture but we want to change the
        # architecture in some tests
        self.mock_uname = self.libvirt.mock_uname
        self.mock_uname.return_value = fakelibvirt.os_uname(
            'Linux', '', '5.4.0-0-generic', '', fields.Architecture.X86_64)

        self.test_instance = _create_test_instance()
        network_info = objects.InstanceInfoCache(
            network_info=_fake_network_info(self))
        self.test_instance['info_cache'] = network_info
        self.test_image_meta = {
            "disk_format": "raw",
        }
        self.image_service = self.useFixture(nova_fixtures.GlanceFixture(self))
        self.device_xml_tmpl = """
        <domain type='kvm'>
          <devices>
            <disk type='block' device='disk'>
              <driver name='qemu' type='raw' cache='none'/>
              <source dev='{device_path}'/>
              <target bus='virtio' dev='vdb'/>
              <serial>58a84f6d-3f0c-4e19-a0af-eb657b790657</serial>
              <address type='pci' domain='0x0' bus='0x0' slot='0x04' \
              function='0x0'/>
            </disk>
          </devices>
        </domain>
        """

    def relpath(self, path):
        return os.path.relpath(path, CONF.instances_path)

    def tearDown(self):
        super(LibvirtConnTestCase, self).tearDown()

    def test_driver_capabilities(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertTrue(drvr.capabilities['has_imagecache'],
                        'Driver capabilities for \'has_imagecache\' '
                        'is invalid')
        self.assertTrue(drvr.capabilities['supports_evacuate'],
                        'Driver capabilities for \'supports_evacuate\' '
                        'is invalid')
        self.assertFalse(drvr.capabilities['supports_migrate_to_same_host'],
                         'Driver capabilities for '
                         '\'supports_migrate_to_same_host\' is invalid')
        self.assertTrue(drvr.capabilities['supports_attach_interface'],
                        'Driver capabilities for '
                        '\'supports_attach_interface\' '
                        'is invalid')
        self.assertTrue(drvr.capabilities['supports_extend_volume'],
                        'Driver capabilities for '
                        '\'supports_extend_volume\' '
                        'is invalid')
        self.assertTrue(drvr.capabilities['supports_trusted_certs'],
                        'Driver capabilities for '
                        '\'supports_trusted_certs\' '
                        'is invalid')
        self.assertTrue(drvr.capabilities['supports_image_type_qcow2'],
                        'Driver capabilities for '
                        '\'supports_image_type_qcow2\' '
                        'is invalid')
        self.assertFalse(drvr.capabilities['supports_image_type_ploop'],
                         'Driver capabilities for '
                         '\'supports_image_type_ploop\' '
                         'is invalid')
        self.assertFalse(
            drvr.capabilities['supports_vtpm'],
            "Driver capabilities for 'supports_vtpm' is invalid",
        )
        self.assertTrue(
            drvr.capabilities['supports_socket_pci_numa_affinity'],
            "Driver capabilities for 'supports_socket_pci_numa_affinity' "
            "is invalid",
        )
        self.assertFalse(
            drvr.capabilities['supports_ephemeral_encryption'],
            "Driver capabilities for 'supports_ephemeral_encryption' "
            "is invalid",
        )
        self.assertFalse(
            drvr.capabilities['supports_ephemeral_encryption_luks'],
            "Driver capabilities for 'supports_ephemeral_encryption_luks' "
            " is invalid",
        )

    def test_driver_capabilities_qcow2_with_rbd(self):
        self.flags(images_type='rbd', group='libvirt')
        self.flags(force_raw_images=False)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertFalse(drvr.capabilities['supports_image_type_qcow2'],
                         'Driver capabilities for '
                         '\'supports_image_type_qcow2\' '
                         'is invalid when \'images_type=rbd\'')

        self.flags(images_type='rbd', group='libvirt')
        self.flags(force_raw_images=True)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertTrue(drvr.capabilities['supports_image_type_qcow2'])

    def test_driver_capabilities_qcow2_with_lvm(self):
        self.flags(images_type='lvm', group='libvirt')
        self.flags(force_raw_images=False)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertFalse(drvr.capabilities['supports_image_type_qcow2'],
                         'Driver capabilities for '
                         '\'supports_image_type_qcow2\' '
                         'is invalid when \'images_type=lvm\'')

        self.flags(images_type='lvm', group='libvirt')
        self.flags(force_raw_images=True)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertTrue(drvr.capabilities['supports_image_type_qcow2'])

    def test_driver_capabilities_ploop_with_virtuozzo(self):
        self.flags(virt_type='kvm', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertFalse(drvr.capabilities['supports_image_type_ploop'],
                         'Driver capabilities for '
                         '\'supports_image_type_ploop\' '
                         'is invalid when virt_type=kvm')

        self.flags(virt_type='parallels', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertTrue(drvr.capabilities['supports_image_type_ploop'])

    def test_driver_capabilities_vtpm(self):
        self.flags(swtpm_enabled=True, group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertTrue(
            drvr.capabilities['supports_vtpm'],
            "Driver capabilities for 'supports_vtpm' is invalid when "
            "'swtpm_enabled=True'"
        )

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    @mock.patch.object(
        host.Host, 'supports_secure_boot', new_callable=mock.PropertyMock)
    def test_driver_capabilities_secure_boot(self, mock_supports):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr.init_host("dummyhost")
        self.assertTrue(
            drvr.capabilities['supports_secure_boot'],
            "Driver capabilities for 'supports_secure_boot' is invalid when "
            "host should support this feature"
        )
        mock_supports.assert_called_once_with()

    @mock.patch.object(
        libvirt_driver.LibvirtDriver,
        '_register_all_undefined_instance_details',
        new=mock.Mock())
    @mock.patch.object(
        host.Host, 'supports_remote_managed_ports',
        new_callable=mock.PropertyMock)
    def test_driver_capabilities_remote_managed_ports(self, mock_supports):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr.init_host("dummyhost")
        self.assertTrue(
            drvr.capabilities['supports_remote_managed_ports'],
            "Driver capabilities for 'supports_remote_managed_ports' "
            "is invalid when host should support this feature"
        )
        mock_supports.assert_called_once_with()

    def test_driver_raises_on_non_linux_platform(self):
        with utils.temporary_mutation(sys, platform='darwin'):
            self.assertRaises(
                exception.InternalError, libvirt_driver.LibvirtDriver,
                fake.FakeVirtAPI(), False)

    def create_fake_libvirt_mock(self, **kwargs):
        """Defining mocks for LibvirtDriver(libvirt is not used)."""

        # A fake libvirt.virConnect
        class FakeLibvirtDriver(object):
            def defineXML(self, xml):
                return FakeVirtDomain()

        # Creating mocks
        fake = FakeLibvirtDriver()
        # Customizing above fake if necessary
        for key, val in kwargs.items():
            fake.__setattr__(key, val)

        self.stub_out('nova.virt.libvirt.driver.LibvirtDriver._conn', fake)
        self.stub_out('nova.virt.libvirt.host.Host.get_connection',
                      lambda x: fake)

    def fake_lookup(self, instance_name):
        return FakeVirtDomain()

    def fake_execute(self, *args, **kwargs):
        open(args[-1], "a").close()

    def _create_service(self, **kwargs):
        service_ref = {'host': kwargs.get('host', 'dummy'),
                       'disabled': kwargs.get('disabled', False),
                       'binary': 'nova-compute',
                       'topic': 'compute',
                       'report_count': 0}

        return objects.Service(**service_ref)

    def _get_pause_flag(self, drvr, network_info, power_on=True,
                          vifs_already_plugged=False):
        timeout = CONF.vif_plugging_timeout

        events = []
        if (
            CONF.libvirt.virt_type in ('kvm', 'qemu') and
            not vifs_already_plugged and power_on and timeout
        ):
            events = drvr._get_neutron_events(network_info)

        return bool(events)

    def test_public_api_signatures(self):
        baseinst = driver.ComputeDriver(None)
        inst = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertPublicAPISignatures(baseinst, inst)

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_cpu_traits')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_storage_bus_traits')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_video_model_traits')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_vif_model_traits')
    @mock.patch.object(host.Host, "has_min_version")
    def test_static_traits(
        self, mock_version, mock_vif_traits, mock_video_traits,
        mock_storage_traits, mock_cpu_traits,
    ):
        """Ensure driver capabilities are correctly retrieved and cached."""

        # we don't mock out calls to os_traits intentionally, so we need to
        # return valid traits here
        mock_cpu_traits.return_value = {'HW_CPU_HYPERTHREADING': True}
        mock_storage_traits.return_value = {'COMPUTE_STORAGE_BUS_VIRTIO': True}
        mock_video_traits.return_value = {'COMPUTE_GRAPHICS_MODEL_VGA': True}
        mock_vif_traits.return_value = {'COMPUTE_NET_VIF_MODEL_VIRTIO': True}

        # for support COMPUTE_VIOMMU_MODEL_VIRTIO
        mock_version.return_value = True

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        expected = {
            'COMPUTE_GRAPHICS_MODEL_VGA': True,
            'COMPUTE_NET_VIF_MODEL_VIRTIO': True,
            'COMPUTE_NET_VIRTIO_PACKED': True,
            'COMPUTE_SECURITY_TPM_1_2': False,
            'COMPUTE_SECURITY_TPM_2_0': False,
            'COMPUTE_STORAGE_BUS_VIRTIO': True,
            'COMPUTE_VIOMMU_MODEL_AUTO': True,
            'COMPUTE_VIOMMU_MODEL_INTEL': True,
            'COMPUTE_VIOMMU_MODEL_SMMUV3': True,
            'COMPUTE_VIOMMU_MODEL_VIRTIO': True,
            'HW_CPU_HYPERTHREADING': True
        }

        static_traits = drvr.static_traits

        # check that results are as expected and the individual helper
        # functions were called once each
        self.assertEqual(expected, static_traits)
        for mock_traits in (
            mock_vif_traits, mock_video_traits, mock_storage_traits,
            mock_cpu_traits,
        ):
            mock_traits.assert_called_once_with()
            mock_traits.reset_mock()

        static_traits = drvr.static_traits

        # now check that the results are still as expected but the helpers
        # weren't called since the value was cached
        self.assertEqual(expected, static_traits)
        for mock_traits in (
            mock_vif_traits, mock_video_traits, mock_storage_traits,
            mock_cpu_traits,
        ):
            mock_traits.assert_not_called()

    @mock.patch.object(libvirt_driver.LOG, 'debug')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_cpu_traits')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_storage_bus_traits')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_video_model_traits')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_vif_model_traits')
    def test_static_traits_invalid_trait(
        self, mock_vif_traits, mock_video_traits, mock_storage_traits,
        mock_cpu_traits, mock_log,
    ):
        """Ensure driver capabilities are correctly retrieved and cached."""
        mock_cpu_traits.return_value = {'foo': True}
        mock_storage_traits.return_value = {'bar': True}
        mock_video_traits.return_value = {'baz': True}
        mock_vif_traits.return_value = {'COMPUTE_NET_VIF_MODEL_VIRTIO': True}

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        expected = {
            'COMPUTE_NET_VIF_MODEL_VIRTIO': True,
            'COMPUTE_NET_VIRTIO_PACKED': True,
            'COMPUTE_SECURITY_TPM_1_2': False,
            'COMPUTE_SECURITY_TPM_2_0': False,
            'COMPUTE_VIOMMU_MODEL_AUTO': True,
            'COMPUTE_VIOMMU_MODEL_INTEL': True,
            'COMPUTE_VIOMMU_MODEL_SMMUV3': True,
            'COMPUTE_VIOMMU_MODEL_VIRTIO': False
        }

        static_traits = drvr.static_traits

        self.assertEqual(expected, static_traits)
        mock_log.assert_has_calls([
            mock.call("Trait '%s' is not valid; ignoring.", "foo"),
            mock.call("Trait '%s' is not valid; ignoring.", "bar"),
            mock.call("Trait '%s' is not valid; ignoring.", "baz"),
        ],
        any_order=True)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    @mock.patch.object(host.Host, "has_min_version")
    def test_min_version_start_ok(self, mock_version):
        mock_version.return_value = True
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.init_host("dummyhost")

    @mock.patch.object(host.Host, "has_min_version")
    def test_min_version_start_abort(self, mock_version):
        mock_version.return_value = False
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertRaises(exception.NovaException,
                          drvr.init_host,
                          "dummyhost")

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    @mock.patch.object(fakelibvirt.Connection, 'getLibVersion',
                       return_value=versionutils.convert_version_to_int(
                            libvirt_driver.NEXT_MIN_LIBVIRT_VERSION) - 1)
    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test_next_min_version_deprecation_warning(self, mock_warning,
                                                  mock_get_libversion):
        # Skip test if there's no currently planned new min version
        if (versionutils.convert_version_to_int(
                libvirt_driver.NEXT_MIN_LIBVIRT_VERSION) ==
            versionutils.convert_version_to_int(
                libvirt_driver.MIN_LIBVIRT_VERSION)):
            self.skipTest("NEXT_MIN_LIBVIRT_VERSION == MIN_LIBVIRT_VERSION")

        # Test that a warning is logged if the libvirt version is less than
        # the next required minimum version.
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.init_host("dummyhost")
        # assert that the next min version is in a warning message
        expected_arg = {'version': versionutils.convert_version_to_str(
            versionutils.convert_version_to_int(
                libvirt_driver.NEXT_MIN_LIBVIRT_VERSION))}
        version_arg_found = False
        for call in mock_warning.call_args_list:
            if call[0][1] == expected_arg:
                version_arg_found = True
                break
        self.assertTrue(version_arg_found)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    @mock.patch.object(fakelibvirt.Connection, 'getVersion',
                       return_value=versionutils.convert_version_to_int(
                            libvirt_driver.NEXT_MIN_QEMU_VERSION) - 1)
    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test_next_min_qemu_version_deprecation_warning(self, mock_warning,
                                                       mock_get_libversion):
        # Skip test if there's no currently planned new min version
        if (versionutils.convert_version_to_int(
                libvirt_driver.NEXT_MIN_QEMU_VERSION) ==
            versionutils.convert_version_to_int(
                libvirt_driver.MIN_QEMU_VERSION)):
            self.skipTest("NEXT_MIN_QEMU_VERSION == MIN_QEMU_VERSION")

        # Test that a warning is logged if the libvirt version is less than
        # the next required minimum version.
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.init_host("dummyhost")
        # assert that the next min version is in a warning message
        expected_arg = {'version': versionutils.convert_version_to_str(
            versionutils.convert_version_to_int(
                libvirt_driver.NEXT_MIN_QEMU_VERSION))}
        version_arg_found = False
        for call in mock_warning.call_args_list:
            if call[0][1] == expected_arg:
                version_arg_found = True
                break
        self.assertTrue(version_arg_found)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    @mock.patch.object(fakelibvirt.Connection, 'getLibVersion',
                       return_value=versionutils.convert_version_to_int(
                            libvirt_driver.NEXT_MIN_LIBVIRT_VERSION))
    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test_next_min_version_ok(self, mock_warning, mock_get_libversion):
        # Skip test if there's no currently planned new min version

        if (versionutils.convert_version_to_int(
                libvirt_driver.NEXT_MIN_LIBVIRT_VERSION) ==
            versionutils.convert_version_to_int(
                libvirt_driver.MIN_LIBVIRT_VERSION)):
            self.skipTest("NEXT_MIN_LIBVIRT_VERSION == MIN_LIBVIRT_VERSION")

        # Test that a warning is not logged if the libvirt version is greater
        # than or equal to NEXT_MIN_LIBVIRT_VERSION.
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.init_host("dummyhost")
        # assert that the next min version is in a warning message
        expected_arg = {'version': versionutils.convert_version_to_str(
            versionutils.convert_version_to_int(
                libvirt_driver.NEXT_MIN_LIBVIRT_VERSION))}
        version_arg_found = False
        for call in mock_warning.call_args_list:
            if call[0][1] == expected_arg:
                version_arg_found = True
                break
        self.assertFalse(version_arg_found)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    @mock.patch.object(fakelibvirt.Connection, 'getVersion',
                       return_value=versionutils.convert_version_to_int(
                            libvirt_driver.NEXT_MIN_QEMU_VERSION))
    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test_next_min_qemu_version_ok(self, mock_warning, mock_get_libversion):
        # Skip test if there's no currently planned new min version

        if (versionutils.convert_version_to_int(
                libvirt_driver.NEXT_MIN_QEMU_VERSION) ==
            versionutils.convert_version_to_int(
                libvirt_driver.MIN_QEMU_VERSION)):
            self.skipTest("NEXT_MIN_QEMU_VERSION == MIN_QEMU_VERSION")

        # Test that a warning is not logged if the libvirt version is greater
        # than or equal to NEXT_MIN_QEMU_VERSION.
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.init_host("dummyhost")
        # assert that the next min version is in a warning message
        expected_arg = {'version': versionutils.convert_version_to_str(
            versionutils.convert_version_to_int(
                libvirt_driver.NEXT_MIN_QEMU_VERSION))}
        version_arg_found = False
        for call in mock_warning.call_args_list:
            if call[0][1] == expected_arg:
                version_arg_found = True
                break
        self.assertFalse(version_arg_found)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    def test_min_version_ppc_ok(self):
        self.mock_uname.return_value = fakelibvirt.os_uname(
            'Linux', '', '5.4.0-0-generic', '', fields.Architecture.PPC64)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.init_host("dummyhost")

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    def test_min_version_s390_ok(self):
        self.mock_uname.return_value = fakelibvirt.os_uname(
            'Linux', '', '5.4.0-0-generic', '', fields.Architecture.S390X)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.init_host("dummyhost")

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    def test_file_backed_memory_support_called(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        with mock.patch.object(drvr,
                '_check_file_backed_memory_support') as mock_check_fb_support:
            drvr.init_host("dummyhost")
            self.assertTrue(mock_check_fb_support.called)

    def test_min_version_file_backed_ok(self):
        self.flags(file_backed_memory=1024, group='libvirt')
        self.flags(ram_allocation_ratio=1.0)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._check_file_backed_memory_support()

    def test_min_version_file_backed_bad_ram_allocation_ratio(self):
        self.flags(file_backed_memory=1024, group="libvirt")
        self.flags(ram_allocation_ratio=1.5)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertRaises(exception.InternalError,
                          drvr._check_file_backed_memory_support)

    def test__check_file_backed_memory_support__total_lt_reserved(self):
        """Ensure an error is raised if total memory < reserved.

        Placement won't allow $resource.total < $resource.reserved, so we need
        to catch this early.
        """
        self.flags(file_backed_memory=1024, group='libvirt')
        self.flags(ram_allocation_ratio=1.0, reserved_host_memory_mb=4096)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertRaises(
            exception.InternalError, drvr._check_file_backed_memory_support,
        )

    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test__check_file_backed_memory_support__has_reserved(self, mock_log):
        """Ensure a warning is issued if memory is reserved.

        It doesn't make sense to "reserve" memory when file-backed memory is in
        use. We should report things so as to avoid confusion.
        """
        self.flags(file_backed_memory=8192, group='libvirt')
        self.flags(ram_allocation_ratio=1.0)
        # we don't need to configure '[DEFAULT] reserved_host_memory_mb' since
        # it defaults to 512 (MB)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._check_file_backed_memory_support()
        mock_log.assert_called_once()
        self.assertIn(
            "Reserving memory via '[DEFAULT] reserved_host_memory_mb' is not "
            "compatible",
            str(mock_log.call_args[0]),
        )

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    def test__prepare_cpu_flag(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        # The `+` means the guest "require[s]" (i.e. enable in libvirt
        # parlance) the said CPU feature; and `-` will disable it
        feat1 = drvr._prepare_cpu_flag('+md-clear')
        feat2 = drvr._prepare_cpu_flag('pdpe1gb')
        feat3 = drvr._prepare_cpu_flag('-ssbd')

        self.assertIsInstance(feat1, vconfig.LibvirtConfigGuestCPUFeature)
        self.assertIsInstance(feat2, vconfig.LibvirtConfigGuestCPUFeature)
        self.assertIsInstance(feat3, vconfig.LibvirtConfigGuestCPUFeature)

        cpu = vconfig.LibvirtConfigGuestCPU()
        cpu.add_feature(feat1)
        cpu.add_feature(feat2)
        cpu.add_feature(feat3)

        # Verify that the resulting guest XML records both enabled
        # _and_ disabled CPU features
        expected_xml = '''
              <cpu match='exact'>
                <feature policy='require' name='md-clear'/>
                <feature policy='require' name='pdpe1gb'/>
                <feature policy='disable' name='ssbd'/>
              </cpu>'''
        self.assertXmlEqual(expected_xml, cpu.to_xml())

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    def test__check_cpu_compatibility_start_ok(self):
        self.flags(cpu_mode="custom",
                   cpu_models=["Penryn"],
                   group="libvirt")
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.init_host("dummyhost")

    def test__check_cpu_compatibility_none_models(self):
        self.flags(cpu_mode="custom",
                   cpu_models=[],
                   group="libvirt")
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertRaises(exception.Invalid, drvr.init_host, "dummyhost")

    def test__check_cpu_compatibility_none_mode(self):
        self.flags(cpu_mode="none",
                   cpu_models=["Penryn"],
                   group="libvirt")
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertRaises(exception.Invalid, drvr.init_host, "dummyhost")

    @mock.patch(
        'nova.virt.libvirt.host.libvirt.Connection.compareHypervisorCPU')
    def test__check_cpu_compatibility_advance_model(self, mocked_compare):
        mocked_compare.side_effect = (2, 0)
        self.flags(cpu_mode="custom",
                   cpu_models=["qemu64", "Broadwell-noTSX"],
                   group="libvirt")
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertRaises(exception.InvalidCPUInfo,
                          drvr.init_host, "dummyhost")

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    @mock.patch('nova.virt.libvirt.host.libvirt.Connection.compareCPU')
    def test__check_cpu_compatibility_skip_compare_at_init(
            self, mocked_compare
    ):
        self.flags(group='workarounds', skip_cpu_compare_at_startup=True)
        self.flags(cpu_mode="custom",
                   cpu_models=["Icelake-Server-noTSX"],
                   cpu_model_extra_flags = ["-mpx"],
                   group="libvirt")
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.init_host("dummyhost")
        mocked_compare.assert_not_called()

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    def test__check_cpu_compatibility_with_flag(self):
        self.flags(cpu_mode="custom",
                   cpu_models=["Penryn"],
                   cpu_model_extra_flags = ["aes"],
                   group="libvirt")
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.init_host("dummyhost")

    @mock.patch(
        'nova.virt.libvirt.host.libvirt.Connection.compareHypervisorCPU')
    def test__check_cpu_compatibility_advance_flag(self, mocked_compare):
        mocked_compare.side_effect = (-1, 0)
        self.flags(cpu_mode="custom",
                   cpu_models=["qemu64"],
                   cpu_model_extra_flags = ["avx", "avx2"],
                   group="libvirt")
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertRaises(exception.InvalidCPUInfo,
                          drvr.init_host, "dummyhost")

    @mock.patch(
        'nova.virt.libvirt.host.libvirt.Connection.compareHypervisorCPU')
    def test__check_cpu_compatibility_wrong_flag(self, mocked_compare):
        # here, and in the surrounding similar tests, the non-zero error
        # code in the compareCPU() side effect indicates error
        mocked_compare.side_effect = (-1, 0)
        self.flags(cpu_mode="custom",
                   cpu_models=["Broadwell-noTSX"],
                   cpu_model_extra_flags = ["a v x"],
                   group="libvirt")
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertRaises(exception.InvalidCPUInfo,
                          drvr.init_host, "dummyhost")

    @mock.patch(
        'nova.virt.libvirt.host.libvirt.Connection.compareHypervisorCPU')
    def test__check_cpu_compatibility_enabled_and_disabled_flags(
            self, mocked_compare
    ):
        mocked_compare.side_effect = (-1, 0)
        self.flags(
            cpu_mode="custom",
            cpu_models=["Cascadelake-Server"],
            cpu_model_extra_flags = ["-hle", "-rtm", "+ssbd", "mttr"],
            group="libvirt")
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertRaises(exception.InvalidCPUInfo,
                          drvr.init_host, "dummyhost")

    def test__check_cpu_compatibility_invalid_virt_type(self):
        """Test getting CPU traits when using a virt_type that doesn't support
        the feature, only kvm and qemu supports reporting CPU traits.
        """
        self.flags(cpu_mode='custom',
                   cpu_models=['IvyBridge'],
                   virt_type='lxc',
                   group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertRaises(exception.Invalid, drvr.init_host, "dummyhost")

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    def test__check_cpu_compatibility_aarch64_qemu_custom_start_OK(self):
        """Test getting CPU traits when using a virt_type that doesn't support
        the feature, only kvm and qemu supports reporting CPU traits.
        """
        self.flags(cpu_mode='custom',
                   cpu_models=['max'],
                   virt_type='qemu',
                   group='libvirt')
        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.AARCH64
        with mock.patch.object(host.Host, "get_capabilities",
                               return_value=caps):
            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
            drvr.init_host("dummyhost")

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    def test__check_vtpm_support_non_qemu(self):
        """Test checking for vTPM support when we're not using QEMU or KVM."""
        self.flags(swtpm_enabled=True, virt_type='lxc', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        exc = self.assertRaises(exception.InvalidConfiguration,
                                drvr.init_host, 'dummyhost')
        self.assertIn("vTPM support requires '[libvirt] virt_type' of 'qemu' "
                      "or 'kvm'; found 'lxc'.", str(exc))

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    @mock.patch('shutil.which')
    def test__check_vtpm_support_missing_exe(self, mock_which, mock_version):
        """Test checking for vTPM support when the swtpm binaries are
        missing.
        """
        self.flags(swtpm_enabled=True, virt_type='kvm', group='libvirt')
        mock_which.return_value = False

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        exc = self.assertRaises(exception.InvalidConfiguration,
                                drvr.init_host, "dummyhost")
        self.assertIn(
            "vTPM support is configured but the 'swtpm' and 'swtpm_setup' "
            "binaries could not be found on PATH.",
            str(exc),
        )

        mock_which.assert_has_calls(
            [mock.call('swtpm_setup'), mock.call('swtpm')],
        )

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    @mock.patch('shutil.which')
    @mock.patch('pwd.getpwnam')
    def test__check_vtpm_support_invalid_user(
        self, mock_getpwnam, mock_which, mock_version,
    ):
        """Test checking for vTPM support when the configured user is
        invalid.
        """
        self.flags(
            swtpm_user='lionel', swtpm_enabled=True, virt_type='kvm',
            group='libvirt')
        mock_which.return_value = True
        mock_getpwnam.side_effect = KeyError

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        exc = self.assertRaises(
            exception.InvalidConfiguration,
            drvr.init_host, "dummyhost")

        self.assertIn(
            "The user configured in '[libvirt] swtpm_user' does not exist "
            "on this host; expected 'lionel'.",
            str(exc),
        )
        mock_getpwnam.assert_called_with('lionel')

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    @mock.patch('shutil.which')
    @mock.patch('pwd.getpwnam')
    @mock.patch('grp.getgrnam')
    def test__check_vtpm_support_invalid_group(
        self, mock_getgrnam, mock_getpwnam, mock_which, mock_version,
    ):
        """Test checking for vTPM support when the configured group is
        invalid.
        """
        self.flags(
            swtpm_group='admins', swtpm_enabled=True, virt_type='kvm',
            group='libvirt')
        mock_which.return_value = True
        mock_getgrnam.side_effect = KeyError

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        exc = self.assertRaises(
            exception.InvalidConfiguration,
            drvr.init_host, "dummyhost")

        self.assertIn(
            "The group configured in '[libvirt] swtpm_group' does not exist "
            "on this host; expected 'admins'.",
            str(exc),
        )
        mock_getgrnam.assert_called_with('admins')

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    @mock.patch('shutil.which')
    @mock.patch('pwd.getpwnam')
    @mock.patch('grp.getgrnam')
    def test__check_vtpm_support(
        self, mock_getgrnam, mock_getpwnam, mock_which
    ):
        """Test checking for vTPM support when everything is configured
        correctly.
        """
        self.flags(swtpm_enabled=True, virt_type='kvm', group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.init_host('dummyhost')

        mock_which.assert_has_calls(
            [mock.call('swtpm_setup'), mock.call().__bool__()],
        )

    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test_check_cpu_set_configuration__no_configuration(self, mock_log):
        """Test that configuring no CPU option results no errors or logs.
        """
        self.flags(vcpu_pin_set=None, reserved_host_cpus=None)
        self.flags(cpu_shared_set=None, cpu_dedicated_set=None,
                   group='compute')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._check_cpu_set_configuration()

        mock_log.assert_not_called()

    def test_check_cpu_set_configuration__cpu_shared_set_cpu_dedicated_set(
            self):
        """Test that configuring 'cpu_shared_set' and 'cpu_dedicated_set' such
        that they overlap (are not disjoint) results in an error stating that
        this is not allowed.
        """
        self.flags(vcpu_pin_set=None, reserved_host_cpus=None)
        self.flags(cpu_shared_set='0-3', cpu_dedicated_set='3-5',
                   group='compute')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        self.assertRaises(exception.InvalidConfiguration,
                          drvr._check_cpu_set_configuration)

    def test_check_cpu_set_configuration__reserved_host_cpus_cpu_shared_set(
            self):
        """Test that configuring 'reserved_host_cpus' with one of the new
        options, in this case '[compute] cpu_shared_set', results in an error
        stating that this is not allowed.
        """
        self.flags(vcpu_pin_set=None, reserved_host_cpus=1)
        self.flags(cpu_shared_set='1-10', cpu_dedicated_set=None,
                   group='compute')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        ex = self.assertRaises(exception.InvalidConfiguration,
                               drvr._check_cpu_set_configuration)
        self.assertIn("The 'reserved_host_cpus' config option cannot be "
                      "defined alongside ", str(ex))

    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test_check_cpu_set_configuration__vcpu_pin_set(self, mock_log):
        """Test that configuring only 'vcpu_pin_set' results in a warning that
        the option is being used for VCPU inventory but this is deprecated
        behavior.
        """
        self.flags(vcpu_pin_set='0-3', reserved_host_cpus=None)
        self.flags(cpu_shared_set=None, cpu_dedicated_set=None,
                   group='compute')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._check_cpu_set_configuration()

        mock_log.assert_called_once()
        self.assertIn("When defined, 'vcpu_pin_set' will be used to calculate "
                      "'VCPU' inventory and schedule instances that have "
                      "'VCPU' allocations.",
                      str(mock_log.call_args[0]))

    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test_check_cpu_set_configuration__vcpu_pin_set_cpu_shared_set(
            self, mock_log):
        """Test that configuring both 'vcpu_pin_set' and 'cpu_shared_set'
        results in a warning that 'cpu_shared_set' is being ignored for
        calculating VCPU inventory.
        """
        self.flags(vcpu_pin_set='0-3', reserved_host_cpus=None)
        self.flags(cpu_shared_set='4-5', cpu_dedicated_set=None,
                   group='compute')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._check_cpu_set_configuration()

        mock_log.assert_called_once()
        self.assertIn("The '[compute] cpu_shared_set' and 'vcpu_pin_set' "
                      "config options have both been defined.",
                      str(mock_log.call_args[0]))

    def test_check_cpu_set_configuration__vcpu_pin_set_cpu_dedicated_set(
            self):
        """Test that configuring both 'vcpu_pin_set' and 'cpu_dedicated_set'
        results in an error stating that the two options cannot co-exist.
        """
        self.flags(vcpu_pin_set='0-3', reserved_host_cpus=None)
        self.flags(cpu_shared_set=None, cpu_dedicated_set='4-5',
                   group='compute')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        ex = self.assertRaises(exception.InvalidConfiguration,
                               drvr._check_cpu_set_configuration)
        self.assertIn("The 'vcpu_pin_set' config option has been deprecated "
                      "and cannot be defined alongside '[compute] "
                      "cpu_dedicated_set'.", str(ex))

    def _do_test_parse_migration_flags(self, lm_expected=None,
                                       bm_expected=None):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._parse_migration_flags()

        if lm_expected is not None:
            self.assertEqual(lm_expected, drvr._live_migration_flags)
        if bm_expected is not None:
            self.assertEqual(bm_expected, drvr._block_migration_flags)

    def test_parse_live_migration_flags_default(self):
        self._do_test_parse_migration_flags(
            lm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_LIVE))

    def test_parse_live_migration_flags(self):
        self._do_test_parse_migration_flags(
            lm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_LIVE))

    def test_parse_block_migration_flags_default(self):
        self._do_test_parse_migration_flags(
            bm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_LIVE |
                         libvirt_driver.libvirt.VIR_MIGRATE_NON_SHARED_INC))

    def test_parse_block_migration_flags(self):
        self._do_test_parse_migration_flags(
            bm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_LIVE |
                         libvirt_driver.libvirt.VIR_MIGRATE_NON_SHARED_INC))

    def test_live_migration_tunnelled_true(self):
        self.flags(live_migration_tunnelled=True, group='libvirt')
        self._do_test_parse_migration_flags(
            lm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_LIVE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_TUNNELLED),
            bm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_LIVE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_NON_SHARED_INC |
                         libvirt_driver.libvirt.VIR_MIGRATE_TUNNELLED))

    def test_live_migration_with_native_tls(self):
        self.flags(live_migration_with_native_tls=True, group='libvirt')
        self._do_test_parse_migration_flags(
            lm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_LIVE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_TLS),
            bm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_LIVE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_NON_SHARED_INC |
                         libvirt_driver.libvirt.VIR_MIGRATE_TLS))

    def test_live_migration_permit_postcopy_true(self):
        self.flags(live_migration_permit_post_copy=True, group='libvirt')
        self._do_test_parse_migration_flags(
            lm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_LIVE |
                         libvirt_driver.libvirt.VIR_MIGRATE_POSTCOPY),
            bm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_LIVE |
                         libvirt_driver.libvirt.VIR_MIGRATE_NON_SHARED_INC |
                         libvirt_driver.libvirt.VIR_MIGRATE_POSTCOPY))

    def test_live_migration_permit_auto_converge_true(self):
        self.flags(live_migration_permit_auto_converge=True, group='libvirt')
        self._do_test_parse_migration_flags(
            lm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_LIVE |
                         libvirt_driver.libvirt.VIR_MIGRATE_AUTO_CONVERGE),
            bm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_LIVE |
                         libvirt_driver.libvirt.VIR_MIGRATE_NON_SHARED_INC |
                         libvirt_driver.libvirt.VIR_MIGRATE_AUTO_CONVERGE))

    def test_live_migration_permit_auto_converge_and_post_copy_true(self):
        self.flags(live_migration_permit_auto_converge=True, group='libvirt')
        self.flags(live_migration_permit_post_copy=True, group='libvirt')
        self._do_test_parse_migration_flags(
            lm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_LIVE |
                         libvirt_driver.libvirt.VIR_MIGRATE_POSTCOPY),
            bm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_LIVE |
                         libvirt_driver.libvirt.VIR_MIGRATE_NON_SHARED_INC |
                         libvirt_driver.libvirt.VIR_MIGRATE_POSTCOPY))

    def test_live_migration_permit_postcopy_false(self):
        self._do_test_parse_migration_flags(
            lm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_LIVE),
            bm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_LIVE |
                         libvirt_driver.libvirt.VIR_MIGRATE_NON_SHARED_INC))

    def test_live_migration_permit_autoconverge_false(self):
        self._do_test_parse_migration_flags(
            lm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_LIVE),
            bm_expected=(libvirt_driver.libvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                         libvirt_driver.libvirt.VIR_MIGRATE_PERSIST_DEST |
                         libvirt_driver.libvirt.VIR_MIGRATE_PEER2PEER |
                         libvirt_driver.libvirt.VIR_MIGRATE_LIVE |
                         libvirt_driver.libvirt.VIR_MIGRATE_NON_SHARED_INC))

    @mock.patch('nova.utils.get_image_from_system_metadata')
    @mock.patch.object(host.Host,
                       'has_min_version', return_value=True)
    @mock.patch('nova.virt.libvirt.host.Host.get_guest')
    def test_set_admin_password(self, mock_get_guest, ver, mock_image):
        self.flags(virt_type='kvm', group='libvirt')
        instance = objects.Instance(**self.test_instance)
        mock_image.return_value = {"properties": {
            "hw_qemu_guest_agent": "yes"}}
        mock_guest = mock.Mock(spec=libvirt_guest.Guest)
        mock_get_guest.return_value = mock_guest

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.set_admin_password(instance, "123")

        mock_guest.set_user_password.assert_called_once_with("root", "123")

    @mock.patch('nova.objects.Instance.save')
    @mock.patch('oslo_serialization.base64.encode_as_text')
    @mock.patch('nova.api.metadata.password.convert_password')
    @mock.patch('nova.crypto.ssh_encrypt_text')
    @mock.patch('nova.utils.get_image_from_system_metadata')
    @mock.patch.object(host.Host,
                       'has_min_version', return_value=True)
    @mock.patch('nova.virt.libvirt.host.Host.get_guest')
    def test_set_admin_password_saves_sysmeta(self, mock_get_guest,
                                              ver, mock_image, mock_encrypt,
                                              mock_convert, mock_encode,
                                              mock_save):
        self.flags(virt_type='kvm', group='libvirt')
        instance = objects.Instance(**self.test_instance)
        # Password will only be saved in sysmeta if the key_data is present
        instance.key_data = 'ssh-rsa ABCFEFG'
        mock_image.return_value = {"properties": {
            "hw_qemu_guest_agent": "yes"}}
        mock_guest = mock.Mock(spec=libvirt_guest.Guest)
        mock_get_guest.return_value = mock_guest
        mock_convert.return_value = {'password_0': 'converted-password'}

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.set_admin_password(instance, "123")

        mock_guest.set_user_password.assert_called_once_with("root", "123")
        mock_encrypt.assert_called_once_with(instance.key_data, '123')
        mock_encode.assert_called_once_with(mock_encrypt.return_value)
        mock_convert.assert_called_once_with(None, mock_encode.return_value)
        self.assertEqual('converted-password',
                         instance.system_metadata['password_0'])
        mock_save.assert_called_once_with()

    @mock.patch('nova.virt.libvirt.host.Host.get_guest')
    def test_set_admin_password_parallels(self, mock_get_guest):
        self.flags(virt_type='parallels', group='libvirt')
        instance = objects.Instance(**self.test_instance)
        mock_guest = mock.Mock(spec=libvirt_guest.Guest)
        mock_get_guest.return_value = mock_guest

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.set_admin_password(instance, "123")

        mock_guest.set_user_password.assert_called_once_with("root", "123")

    @mock.patch('nova.virt.libvirt.host.Host.get_guest')
    def test_qemu_announce_self(self, mock_get_guest):
        # Enable the workaround, configure to call announce_self 3 times
        self.flags(enable_qemu_monitor_announce_self=True, group='workarounds')

        mock_guest = mock.Mock(spec=libvirt_guest.Guest)
        mock_get_guest.return_value = mock_guest

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._qemu_monitor_announce_self(mock_guest)

        # Ensure that 3 calls are made as defined by option
        # enable_qemu_monitor_announce_self_retries default of 3
        mock_guest.announce_self.assert_any_call()
        self.assertEqual(3, mock_guest.announce_self.call_count)

    @mock.patch('nova.utils.get_image_from_system_metadata')
    @mock.patch.object(host.Host,
                       'has_min_version', return_value=True)
    @mock.patch('nova.virt.libvirt.host.Host.get_guest')
    def test_set_admin_password_windows(self, mock_get_guest, ver, mock_image):
        self.flags(virt_type='kvm', group='libvirt')
        instance = objects.Instance(**self.test_instance)
        instance.os_type = "windows"
        mock_image.return_value = {"properties": {
            "hw_qemu_guest_agent": "yes"}}
        mock_guest = mock.Mock(spec=libvirt_guest.Guest)
        mock_get_guest.return_value = mock_guest

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.set_admin_password(instance, "123")

        mock_guest.set_user_password.assert_called_once_with(
            "Administrator", "123")

    @mock.patch('nova.utils.get_image_from_system_metadata')
    @mock.patch.object(host.Host,
                       'has_min_version', return_value=True)
    @mock.patch('nova.virt.libvirt.host.Host.get_guest')
    def test_set_admin_password_image(self, mock_get_guest, ver, mock_image):
        self.flags(virt_type='kvm', group='libvirt')
        instance = objects.Instance(**self.test_instance)
        mock_image.return_value = {"properties": {
            "hw_qemu_guest_agent": "yes",
            "os_admin_user": "foo"
        }}
        mock_guest = mock.Mock(spec=libvirt_guest.Guest)
        mock_get_guest.return_value = mock_guest

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.set_admin_password(instance, "123")

        mock_guest.set_user_password.assert_called_once_with("foo", "123")

    @mock.patch('nova.utils.get_image_from_system_metadata')
    @mock.patch.object(host.Host,
                       'has_min_version', return_value=True)
    def test_set_admin_password_bad_hyp(self, mock_svc, mock_image):
        self.flags(virt_type='lxc', group='libvirt')
        instance = objects.Instance(**self.test_instance)
        mock_image.return_value = {"properties": {
            "hw_qemu_guest_agent": "yes"}}
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertRaises(exception.SetAdminPasswdNotSupported,
                          drvr.set_admin_password, instance, "123")

    @mock.patch.object(host.Host,
                       'has_min_version', return_value=True)
    def test_set_admin_password_guest_agent_not_running(self, mock_svc):
        self.flags(virt_type='kvm', group='libvirt')
        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertRaises(exception.QemuGuestAgentNotEnabled,
                          drvr.set_admin_password, instance, "123")

    @mock.patch('nova.utils.get_image_from_system_metadata')
    @mock.patch.object(host.Host,
                       'has_min_version', return_value=True)
    @mock.patch('nova.virt.libvirt.host.Host.get_guest')
    def test_set_admin_password_error(self, mock_get_guest, ver, mock_image):
        self.flags(virt_type='kvm', group='libvirt')
        instance = objects.Instance(**self.test_instance)
        mock_image.return_value = {"properties": {
            "hw_qemu_guest_agent": "yes"}}
        mock_guest = mock.Mock(spec=libvirt_guest.Guest)
        mock_guest.set_user_password.side_effect = (
            fakelibvirt.libvirtError("error"))
        mock_get_guest.return_value = mock_guest

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        with mock.patch.object(
                drvr, '_save_instance_password_if_sshkey_present') as save_p:
            self.assertRaises(exception.NovaException,
                              drvr.set_admin_password, instance, "123")
            save_p.assert_not_called()

    @mock.patch('nova.utils.get_image_from_system_metadata')
    @mock.patch.object(host.Host,
                       'has_min_version', return_value=True)
    @mock.patch('nova.virt.libvirt.host.Host.get_guest')
    def test_set_admin_password_error_with_unicode(
            self, mock_get_guest, ver, mock_image):
        self.flags(virt_type='kvm', group='libvirt')
        instance = objects.Instance(**self.test_instance)
        mock_image.return_value = {"properties": {
            "hw_qemu_guest_agent": "yes"}}
        mock_guest = mock.Mock(spec=libvirt_guest.Guest)
        mock_guest.set_user_password.side_effect = (
                fakelibvirt.libvirtError(
                    b"failed: \xe9\x94\x99\xe8\xaf\xaf\xe3\x80\x82"))
        mock_get_guest.return_value = mock_guest

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertRaises(exception.NovaException,
                          drvr.set_admin_password, instance, "123")

    @mock.patch('nova.utils.get_image_from_system_metadata')
    @mock.patch.object(host.Host,
                       'has_min_version', return_value=True)
    @mock.patch('nova.virt.libvirt.host.Host.get_guest')
    def test_set_admin_password_not_implemented(
            self, mock_get_guest, ver, mock_image):
        self.flags(virt_type='kvm', group='libvirt')
        instance = objects.Instance(**self.test_instance)
        mock_image.return_value = {"properties": {
            "hw_qemu_guest_agent": "yes"}}
        mock_guest = mock.Mock(spec=libvirt_guest.Guest)
        not_implemented = fakelibvirt.make_libvirtError(
                fakelibvirt.libvirtError,
                "Guest agent disappeared while executing command",
                error_code=fakelibvirt.VIR_ERR_AGENT_UNRESPONSIVE)
        mock_guest.set_user_password.side_effect = not_implemented
        mock_get_guest.return_value = mock_guest

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertRaises(NotImplementedError,
                          drvr.set_admin_password, instance, "123")

    @mock.patch(
        'nova.virt.libvirt.driver.LibvirtDriver._handle_conn_event',
        new=mock.Mock())
    @mock.patch.object(objects.Service, 'save')
    @mock.patch.object(objects.Service, 'get_by_compute_host')
    def test_set_host_enabled_with_disable(self, mock_svc, mock_save):
        # Tests disabling an enabled host.
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        svc = self._create_service(host='fake-mini')
        mock_svc.return_value = svc
        with mock.patch.object(
                drvr, '_update_compute_provider_status') as ucps:
            drvr._set_host_enabled(False)
            ucps.assert_called_once_with(
                test.MatchType(context.RequestContext), svc)
        self.assertTrue(svc.disabled)
        mock_save.assert_called_once_with()

    @mock.patch(
        'nova.virt.libvirt.driver.LibvirtDriver._handle_conn_event',
        new=mock.Mock())
    @mock.patch.object(objects.Service, 'save')
    @mock.patch.object(objects.Service, 'get_by_compute_host')
    def test_set_host_enabled_with_enable(self, mock_svc, mock_save):
        # Tests enabling a disabled host.
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        svc = self._create_service(disabled=True, host='fake-mini')
        mock_svc.return_value = svc
        with mock.patch.object(
                drvr, '_update_compute_provider_status') as ucps:
            drvr._set_host_enabled(True)
            ucps.assert_not_called()
        # since disabled_reason is not set and not prefixed with "AUTO:",
        # service should not be enabled.
        mock_save.assert_not_called()
        self.assertTrue(svc.disabled)

    @mock.patch(
        'nova.virt.libvirt.driver.LibvirtDriver._handle_conn_event',
        new=mock.Mock())
    @mock.patch.object(objects.Service, 'save')
    @mock.patch.object(objects.Service, 'get_by_compute_host')
    def test_set_host_enabled_with_enable_state_enabled(self, mock_svc,
                                                        mock_save):
        # Tests enabling an enabled host.
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        svc = self._create_service(disabled=False, host='fake-mini')
        mock_svc.return_value = svc
        with mock.patch.object(
                drvr, '_update_compute_provider_status') as ucps:
            drvr._set_host_enabled(True)
            ucps.assert_not_called()
        self.assertFalse(svc.disabled)
        mock_save.assert_not_called()

    @mock.patch(
        'nova.virt.libvirt.driver.LibvirtDriver._handle_conn_event',
        new=mock.Mock())
    @mock.patch.object(objects.Service, 'save')
    @mock.patch.object(objects.Service, 'get_by_compute_host')
    def test_set_host_enabled_with_disable_state_disabled(self, mock_svc,
                                                          mock_save):
        # Tests disabling a disabled host.
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        svc = self._create_service(disabled=True, host='fake-mini')
        mock_svc.return_value = svc
        with mock.patch.object(
                drvr, '_update_compute_provider_status') as ucps:
            drvr._set_host_enabled(False)
            ucps.assert_not_called()
        mock_save.assert_not_called()
        self.assertTrue(svc.disabled)

    def test_set_host_enabled_swallows_exceptions(self):
        # Tests that set_host_enabled will swallow exceptions coming from the
        # db_api code so they don't break anything calling it, e.g. the
        # _get_new_connection method.
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        with mock.patch.object(db, 'service_get_by_compute_host') as db_mock:
            # Make db.service_get_by_compute_host raise NovaException; this
            # is more robust than just raising ComputeHostNotFound.
            db_mock.side_effect = exception.NovaException
            drvr._set_host_enabled(False)

    def test_update_compute_provider_status(self):
        """Tests happy path of calling _update_compute_provider_status"""
        virtapi = mock.Mock()
        drvr = libvirt_driver.LibvirtDriver(virtapi, read_only=True)
        ctxt = context.get_admin_context()
        service = self._create_service()
        service.compute_node = objects.ComputeNode(uuid=uuids.rp_uuid)
        drvr._update_compute_provider_status(ctxt, service)
        virtapi.update_compute_provider_status.assert_called_once_with(
            ctxt, uuids.rp_uuid, enabled=not service.disabled)

    def test_update_compute_provider_status_swallows_exceptions(self):
        """Tests error path handling in _update_compute_provider_status"""
        # First we'll make Service.compute_node loading raise an exception
        # by not setting the field and we cannot lazy-load it from an orphaned
        # Service object.
        virtapi = mock.Mock()
        drvr = libvirt_driver.LibvirtDriver(virtapi, read_only=True)
        ctxt = context.get_admin_context()
        service = self._create_service(host='fake-host', disabled=True)
        drvr._update_compute_provider_status(ctxt, service)
        virtapi.update_compute_provider_status.assert_not_called()
        self.assertIn('An error occurred while updating compute node resource '
                      'provider status to "disabled" for provider: fake-host',
                      self.stdlog.logger.output)

        # Now fix Service.compute_node loading but make the VirtAPI call fail.
        service.compute_node = objects.ComputeNode(uuid=uuids.rp_uuid)
        service.disabled = False  # make sure the log message logic works
        error = exception.TraitRetrievalFailed(error='oops')
        virtapi.update_compute_provider_status.side_effect = error
        drvr._update_compute_provider_status(ctxt, service)
        virtapi.update_compute_provider_status.assert_called_once_with(
            ctxt, uuids.rp_uuid, enabled=True)
        log_output = self.stdlog.logger.output
        self.assertIn('An error occurred while updating compute node resource '
                      'provider status to "enabled" for provider: %s' %
                      uuids.rp_uuid, log_output)
        # The error should have been logged as well.
        self.assertIn(str(error), log_output)

    @mock.patch('nova.context.get_admin_context')
    @mock.patch('nova.compute.utils.notify_about_libvirt_connect_error')
    def test_versioned_notification(self, mock_notify, mock_get):
        mock_get.return_value = self.context

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        fake_error = fakelibvirt.make_libvirtError(
            fakelibvirt.libvirtError, "Failed to connect to host",
            error_code=fakelibvirt.VIR_ERR_INTERNAL_ERROR)

        with mock.patch('nova.virt.libvirt.host.Host._get_connection',
                        side_effect=fake_error):
            self.assertRaises(exception.HypervisorUnavailable,
                              drvr._host.get_connection)
        mock_get.assert_called_once_with()
        mock_notify.assert_called_once_with(self.context, ip=CONF.my_ip,
                                            exception=fake_error)

    @mock.patch.object(host.Host, "has_min_version", return_value=False)
    def test_device_metadata(self, mock_version):
        xml = """
        <domain>
          <name>dummy</name>
          <uuid>32dfcb37-5af1-552b-357c-be8c3aa38310</uuid>
            <memory>1048576</memory>
              <vcpu>1</vcpu>
            <os>
                <type arch='x86_64' machine='pc-i440fx-2.4'>hvm</type>
            </os>
          <devices>
            <disk type='block' device='disk'>
              <driver name='qemu' type='qcow2'/>
              <source dev='/dev/mapper/generic'/>
              <target dev='sda' bus='scsi'/>
             <address type='drive' controller='0' bus='0' target='0' unit='0'/>
            </disk>
            <disk type='block' device='disk'>
              <driver name='qemu' type='qcow2'/>
              <source dev='/dev/mapper/generic-1'/>
              <target dev='hda' bus='ide'/>
             <address type='drive' controller='0' bus='1' target='0' unit='0'/>
            </disk>
            <disk type='block' device='disk'>
              <driver name='qemu' type='qcow2'/>
              <source dev='/dev/mapper/generic-2'/>
              <target dev='hdb' bus='ide'/>
             <address type='drive' controller='0' bus='1' target='1' unit='1'/>
            </disk>
            <disk type='block' device='disk'>
              <driver name='qemu' type='qcow2'/>
              <source dev='/dev/mapper/aa1'/>
              <target dev='sdb' bus='usb'/>
            </disk>
            <disk type='block' device='disk'>
              <driver name='qemu' type='qcow2'/>
              <source dev='/var/lib/libvirt/images/centos'/>
              <backingStore/>
              <target dev='vda' bus='virtio'/>
              <boot order='1'/>
              <alias name='virtio-disk0'/>
              <address type='pci' domain='0x0000' bus='0x00' slot='0x09'
              function='0x0'/>
            </disk>
            <disk type='file' device='disk'>
                <driver name='qemu' type='qcow2' cache='none'/>
                <source file='/var/lib/libvirt/images/generic.qcow2'/>
                <target dev='vdb' bus='virtio'/>
                <address type='virtio-mmio'/>
            </disk>
            <disk type='file' device='disk'>
                <driver name='qemu' type='qcow2'/>
                <source file='/var/lib/libvirt/images/test.qcow2'/>
                <backingStore/>
                <target dev='vdc' bus='virtio'/>
                <alias name='virtio-disk1'/>
                <address type='ccw' cssid='0xfe' ssid='0x0' devno='0x0000'/>
            </disk>
            <interface type='network'>
              <mac address='52:54:00:f6:35:8f'/>
              <source network='default'/>
              <model type='virtio'/>
              <address type='pci' domain='0x0000' bus='0x00' slot='0x03'
              function='0x0'/>
            </interface>
            <interface type='network'>
              <mac address='51:5a:2c:a4:5e:1b'/>
              <source network='default'/>
              <model type='virtio'/>
              <address type='pci' domain='0x0000' bus='0x00' slot='0x04'
              function='0x1'/>
            </interface>
            <interface type='network'>
              <mac address='fa:16:3e:d1:28:e4'/>
              <source network='default'/>
              <model type='virtio'/>
              <address type='virtio-mmio'/>
            </interface>
            <interface type='network'>
                <mac address='52:54:00:14:6f:50'/>
                <source network='default' bridge='virbr0'/>
                <target dev='vnet0'/>
                <model type='virtio'/>
                <alias name='net0'/>
                <address type='ccw' cssid='0xfe' ssid='0x0' devno='0x0001'/>
            </interface>
            <hostdev mode="subsystem" type="pci" managed="yes">
                <source>
                    <address bus="0x06" domain="0x0000" function="0x1"
                    slot="0x00"/>
                </source>
            </hostdev>
          </devices>
        </domain>"""

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        dom = fakelibvirt.Domain(drvr._get_connection(), xml, False)
        guest = libvirt_guest.Guest(dom)

        instance_ref = objects.Instance(**self.test_instance)
        bdms = block_device_obj.block_device_make_list_from_dicts(
            self.context, [
                fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 1,
                     'source_type': 'volume', 'destination_type': 'volume',
                     'device_name': '/dev/sda', 'tag': "db",
                     'volume_id': uuids.volume_1}),
                fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 2,
                     'source_type': 'volume', 'destination_type': 'volume',
                     'device_name': '/dev/hda', 'tag': "nfvfunc1",
                     'volume_id': uuids.volume_2}),
                fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 3,
                     'source_type': 'volume', 'destination_type': 'volume',
                     'device_name': '/dev/sdb', 'tag': "nfvfunc2",
                     'volume_id': uuids.volume_3}),
                fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 4,
                     'source_type': 'volume', 'destination_type': 'volume',
                     'device_name': '/dev/hdb',
                     'volume_id': uuids.volume_4}),
                fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 5,
                     'source_type': 'volume', 'destination_type': 'volume',
                     'device_name': '/dev/vda', 'tag': "nfvfunc3",
                     'volume_id': uuids.volume_5}),
                fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 6,
                     'source_type': 'volume', 'destination_type': 'volume',
                     'device_name': '/dev/vdb', 'tag': "nfvfunc4",
                     'volume_id': uuids.volume_6}),
                fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 7,
                     'source_type': 'volume', 'destination_type': 'volume',
                     'device_name': '/dev/vdc', 'tag': "nfvfunc5",
                     'volume_id': uuids.volume_7}),
            ]
        )
        vif = obj_vif.VirtualInterface(context=self.context)
        vif.address = '52:54:00:f6:35:8f'
        vif.network_id = 123
        vif.instance_uuid = '32dfcb37-5af1-552b-357c-be8c3aa38310'
        vif.uuid = '12ec4b21-ef22-6c21-534b-ba3e3ab3a311'
        vif.tag = 'mytag1'

        vif1 = obj_vif.VirtualInterface(context=self.context)
        vif1.address = '51:5a:2c:a4:5e:1b'
        vif1.network_id = 123
        vif1.instance_uuid = '32dfcb37-5af1-552b-357c-be8c3aa38310'
        vif1.uuid = 'abec4b21-ef22-6c21-534b-ba3e3ab3a312'

        vif2 = obj_vif.VirtualInterface(context=self.context)
        vif2.address = 'fa:16:3e:d1:28:e4'
        vif2.network_id = 123
        vif2.instance_uuid = '32dfcb37-5af1-552b-357c-be8c3aa38310'
        vif2.uuid = '645686e4-7086-4eab-8c2f-c41f017a1b16'
        vif2.tag = 'mytag2'

        vif3 = obj_vif.VirtualInterface(context=self.context)
        vif3.address = '52:54:00:14:6f:50'
        vif3.network_id = 123
        vif3.instance_uuid = '32dfcb37-5af1-552b-357c-be8c3aa38310'
        vif3.uuid = '99cc3604-782d-4a32-a27c-bc33ac56ce86'
        vif3.tag = 'mytag3'

        vif4 = obj_vif.VirtualInterface(context=self.context)
        vif4.address = 'da:d1:f2:91:95:c1'
        vif4.tag = 'pf_tag'

        vifs = [vif, vif1, vif2, vif3, vif4]

        network_info = _fake_network_info(self)
        network_info[0]['vnic_type'] = network_model.VNIC_TYPE_DIRECT_PHYSICAL
        network_info[0]['address'] = "51:5a:2c:a4:5e:1b"
        network_info[0]['details'] = dict(vlan='2145')
        network_info[0]['profile'] = dict(trusted='true')
        instance_ref.info_cache = objects.InstanceInfoCache(
            network_info=network_info)

        pci_utils.get_mac_by_pci_address.side_effect = None
        pci_utils.get_mac_by_pci_address.return_value = 'da:d1:f2:91:95:c1'
        with test.nested(
            mock.patch('nova.objects.VirtualInterfaceList'
                       '.get_by_instance_uuid', return_value=vifs),
            mock.patch('nova.objects.BlockDeviceMappingList'
                       '.get_by_instance_uuid', return_value=bdms),
            mock.patch('nova.virt.libvirt.host.Host.get_guest',
                       return_value=guest),
            mock.patch.object(nova.virt.libvirt.guest.Guest, 'get_xml_desc',
                              return_value=xml),
        ):
            metadata_obj = drvr._build_device_metadata(self.context,
                                                       instance_ref)
            metadata = metadata_obj.devices
            self.assertEqual(11, len(metadata))
            self.assertIsInstance(metadata[0],
                                  objects.DiskMetadata)
            self.assertIsInstance(metadata[0].bus,
                                  objects.SCSIDeviceBus)
            self.assertEqual(['db'], metadata[0].tags)
            self.assertEqual(uuids.volume_1, metadata[0].serial)
            self.assertFalse(metadata[0].bus.obj_attr_is_set('address'))
            self.assertEqual(['nfvfunc1'], metadata[1].tags)
            self.assertEqual(uuids.volume_2, metadata[1].serial)
            self.assertIsInstance(metadata[1],
                                  objects.DiskMetadata)
            self.assertIsInstance(metadata[1].bus,
                                  objects.IDEDeviceBus)
            self.assertEqual(['nfvfunc1'], metadata[1].tags)
            self.assertFalse(metadata[1].bus.obj_attr_is_set('address'))
            self.assertIsInstance(metadata[2],
                                  objects.DiskMetadata)
            self.assertIsInstance(metadata[2].bus,
                                  objects.USBDeviceBus)
            self.assertEqual(['nfvfunc2'], metadata[2].tags)
            self.assertEqual(uuids.volume_3, metadata[2].serial)
            self.assertFalse(metadata[2].bus.obj_attr_is_set('address'))
            self.assertIsInstance(metadata[3],
                                  objects.DiskMetadata)
            self.assertIsInstance(metadata[3].bus,
                                  objects.PCIDeviceBus)
            self.assertEqual(['nfvfunc3'], metadata[3].tags)
            # NOTE(artom) We're not checking volume 4 because it's not tagged
            # and only tagged devices appear in the metadata
            self.assertEqual(uuids.volume_5, metadata[3].serial)
            self.assertEqual('0000:00:09.0', metadata[3].bus.address)
            self.assertIsInstance(metadata[4],
                                  objects.DiskMetadata)
            self.assertEqual(['nfvfunc4'], metadata[4].tags)
            self.assertEqual(uuids.volume_6, metadata[4].serial)
            self.assertIsInstance(metadata[5],
                                  objects.DiskMetadata)
            self.assertEqual(['nfvfunc5'], metadata[5].tags)
            self.assertEqual(uuids.volume_7, metadata[5].serial)
            self.assertIsInstance(metadata[6],
                                  objects.NetworkInterfaceMetadata)
            self.assertIsInstance(metadata[6].bus,
                                  objects.PCIDeviceBus)
            self.assertEqual(['mytag1'], metadata[6].tags)
            self.assertEqual('0000:00:03.0', metadata[6].bus.address)
            self.assertFalse(metadata[6].vf_trusted)

            # Make sure that interface with vlan is exposed to the metadata
            self.assertIsInstance(metadata[7],
                                  objects.NetworkInterfaceMetadata)
            self.assertEqual('51:5a:2c:a4:5e:1b', metadata[7].mac)
            self.assertEqual(2145, metadata[7].vlan)
            self.assertTrue(metadata[7].vf_trusted)
            self.assertIsInstance(metadata[8],
                                  objects.NetworkInterfaceMetadata)
            self.assertEqual(['mytag2'], metadata[8].tags)
            self.assertFalse(metadata[8].vf_trusted)
            self.assertIsInstance(metadata[9],
                                  objects.NetworkInterfaceMetadata)
            self.assertEqual(['mytag3'], metadata[9].tags)
            self.assertFalse(metadata[9].vf_trusted)
            self.assertIsInstance(metadata[10],
                                  objects.NetworkInterfaceMetadata)
            self.assertEqual(['pf_tag'], metadata[10].tags)
            self.assertEqual('da:d1:f2:91:95:c1', metadata[10].mac)
            self.assertEqual('0000:06:00.1', metadata[10].bus.address)

    @mock.patch.object(host.Host, 'get_connection')
    @mock.patch.object(nova.virt.libvirt.guest.Guest, 'get_xml_desc')
    def test_detach_pci_devices(self, mocked_get_xml_desc, mock_conn):

        fake_domXML1_with_pci = (
            """<domain> <devices>
            <disk type='file' device='disk'>
            <driver name='qemu' type='qcow2' cache='none'/>
            <source file='xxx'/>
            <target dev='vda' bus='virtio'/>
            <alias name='virtio-disk0'/>
            <address type='pci' domain='0x0000' bus='0x00'
            slot='0x04' function='0x0'/>
            </disk>
            <hostdev mode="subsystem" type="pci" managed="yes">
            <source>
            <address function="0x1" slot="0x10" domain="0x0001"
             bus="0x04"/>
            </source>
            </hostdev></devices></domain>""")

        fake_domXML1_without_pci = (
            """<domain> <devices>
            <disk type='file' device='disk'>
            <driver name='qemu' type='qcow2' cache='none'/>
            <source file='xxx'/>
            <target dev='vda' bus='virtio'/>
            <alias name='virtio-disk0'/>
            <address type='pci' domain='0x0001' bus='0x00'
            slot='0x04' function='0x0'/>
            </disk></devices></domain>""")

        pci_device_info = {'compute_node_id': 1,
                           'instance_uuid': 'uuid',
                           'address': '0001:04:10.1'}
        pci_device = objects.PciDevice(**pci_device_info)
        pci_devices = [pci_device]
        mocked_get_xml_desc.return_value = fake_domXML1_without_pci

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        dom = fakelibvirt.Domain(
            drvr._get_connection(), fake_domXML1_with_pci, False)
        guest = libvirt_guest.Guest(dom)
        drvr._detach_pci_devices(guest, pci_devices)

    @mock.patch.object(host.Host, 'get_connection')
    @mock.patch.object(nova.virt.libvirt.guest.Guest, 'get_xml_desc')
    def test_detach_pci_devices_timeout(self, mocked_get_xml_desc, mock_conn):

        fake_domXML1_with_pci = (
            """<domain> <devices>
            <disk type='file' device='disk'>
            <driver name='qemu' type='qcow2' cache='none'/>
            <source file='xxx'/>
            <target dev='vda' bus='virtio'/>
            <alias name='virtio-disk0'/>
            <address type='pci' domain='0x0000' bus='0x00'
            slot='0x04' function='0x0'/>
            </disk>
            <hostdev mode="subsystem" type="pci" managed="yes">
            <source>
            <address function="0x1" slot="0x10" domain="0x0001"
             bus="0x04"/>
            </source>
            </hostdev></devices></domain>""")

        pci_device_info = {'compute_node_id': 1,
                           'instance_uuid': 'uuid',
                           'address': '0001:04:10.1'}
        pci_device = objects.PciDevice(**pci_device_info)
        pci_devices = [pci_device]
        mocked_get_xml_desc.return_value = fake_domXML1_with_pci

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        dom = fakelibvirt.Domain(
            drvr._get_connection(), fake_domXML1_with_pci, False)
        guest = libvirt_guest.Guest(dom)
        self.assertRaises(exception.PciDeviceDetachFailed,
                          drvr._detach_pci_devices, guest, pci_devices)

    @mock.patch.object(connector, 'get_connector_properties')
    def test_get_connector(self, fake_get_connector):
        initiator = 'fake.initiator.iqn'
        ip = 'fakeip'
        host = 'fakehost'
        wwpns = ['100010604b019419']
        wwnns = ['200010604b019419']
        self.flags(my_ip=ip)
        self.flags(host=host)

        expected = {
            'ip': ip,
            'initiator': initiator,
            'host': host,
            'wwpns': wwpns,
            'wwnns': wwnns
        }
        volume = {
            'id': 'fake'
        }

        # TODO(walter-boring) add the fake in os-brick
        fake_get_connector.return_value = expected
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        result = drvr.get_volume_connector(volume)
        self.assertThat(expected, matchers.DictMatches(result))

    @mock.patch.object(connector, 'get_connector_properties')
    def test_get_connector_storage_ip(self, fake_get_connector):
        ip = '100.100.100.100'
        storage_ip = '101.101.101.101'
        self.flags(my_block_storage_ip=storage_ip, my_ip=ip)
        volume = {
            'id': 'fake'
        }
        expected = {
            'ip': storage_ip
        }
        # TODO(walter-boring) add the fake in os-brick
        fake_get_connector.return_value = expected
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        result = drvr.get_volume_connector(volume)
        self.assertEqual(storage_ip, result['ip'])

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    def test_lifecycle_event_registration(self):
        calls = []

        def fake_registerErrorHandler(*args, **kwargs):
            calls.append('fake_registerErrorHandler')

        def fake_get_host_capabilities(**args):
            cpu = vconfig.LibvirtConfigGuestCPU()
            cpu.arch = fields.Architecture.ARMV7

            caps = vconfig.LibvirtConfigCaps()
            caps.host = vconfig.LibvirtConfigCapsHost()
            caps.host.cpu = cpu
            calls.append('fake_get_host_capabilities')
            return caps

        @mock.patch.object(fakelibvirt, 'registerErrorHandler',
                           side_effect=fake_registerErrorHandler)
        @mock.patch.object(host.Host, "get_capabilities",
                            side_effect=fake_get_host_capabilities)
        def test_init_host(get_host_capabilities, register_error_handler):
            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
            drvr.init_host("test_host")

        test_init_host()
        # NOTE(dkliban): Will fail if get_host_capabilities is called before
        # registerErrorHandler
        self.assertEqual('fake_registerErrorHandler', calls[0])
        self.assertEqual('fake_get_host_capabilities', calls[1])

    def test_sanitize_log_to_xml(self):
        # setup fake data
        data = {'auth_password': 'scrubme'}
        bdm = [{'connection_info': {'data': data}}]
        bdi = {'block_device_mapping': bdm}

        # Tests that the parameters to the _get_guest_xml method
        # are sanitized for passwords when logged.
        def fake_debug(*args, **kwargs):
            if 'auth_password' in args[0]:
                self.assertNotIn('scrubme', args[0])

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        conf = mock.Mock()
        with test.nested(
            mock.patch.object(libvirt_driver.LOG, 'debug',
                              side_effect=fake_debug),
            mock.patch.object(drvr, '_get_guest_config', return_value=conf)
        ) as (
            debug_mock, conf_mock
        ):
            drvr._get_guest_xml(self.context, self.test_instance,
                                network_info={}, disk_info={},
                                image_meta={}, block_device_info=bdi)
            # we don't care what the log message is, we just want to make sure
            # our stub method is called which asserts the password is scrubbed
            self.assertTrue(debug_mock.called)

    def test_get_guest_config_meta_with_no_port(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        meta = drvr._get_guest_config_meta(
                   objects.Instance(**self.test_instance),
                   _fake_network_info(self, num_networks=0))

        self.assertEqual(len(meta.ports.ports), 0)

    def test_get_guest_config_meta_with_multiple_ports(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        meta = drvr._get_guest_config_meta(
                   objects.Instance(**self.test_instance),
                   _fake_network_info(self, num_networks=2))

        self.assertEqual(len(meta.ports.ports), 2)

        # first port
        self.assertEqual(meta.ports.ports[0].uuid, getattr(uuids, 'vif1'))
        self.assertEqual(len(meta.ports.ports[0].ips), 2)
        self.assertEqual(meta.ports.ports[0].ips[0].address, '192.168.1.100')
        self.assertEqual(meta.ports.ports[0].ips[0].ip_type, 'fixed')
        self.assertEqual(meta.ports.ports[0].ips[0].ip_version, 4)
        self.assertEqual(meta.ports.ports[0].ips[1].address,
                         '2001:db8:0:1:dcad:beff:feef:1')
        self.assertEqual(meta.ports.ports[0].ips[1].ip_type, 'fixed')
        self.assertEqual(meta.ports.ports[0].ips[1].ip_version, 6)

        # second port
        self.assertEqual(meta.ports.ports[1].uuid, getattr(uuids, 'vif2'))
        self.assertEqual(len(meta.ports.ports[0].ips), 2)
        self.assertEqual(meta.ports.ports[1].ips[0].address, '192.168.2.100')
        self.assertEqual(meta.ports.ports[1].ips[0].ip_type, 'fixed')
        self.assertEqual(meta.ports.ports[1].ips[0].ip_version, 4)
        self.assertEqual(meta.ports.ports[1].ips[1].address,
                         '2001:db8:0:2:dcad:beff:feef:1')
        self.assertEqual(meta.ports.ports[1].ips[1].ip_type, 'fixed')
        self.assertEqual(meta.ports.ports[1].ips[1].ip_version, 6)

    @mock.patch.object(time, "time")
    def test_get_guest_config(self, time_mock):
        """Generate a "standard" guest with minimal configuration.

        This uses i440fx by default since that's our default machine type and
        x86 is our default architecture (in our test env, anyway).
        """
        time_mock.return_value = 1234567.89

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        test_instance = copy.deepcopy(self.test_instance)
        test_instance["display_name"] = "purple tomatoes"
        test_instance['system_metadata']['owner_project_name'] = 'sweetshop'
        test_instance['system_metadata']['owner_user_name'] = 'cupcake'
        ctxt = context.RequestContext(
            project_id=123,
            project_name="aubergine",
            user_id=456,
            user_name="pie",
        )
        flavor = objects.Flavor(
            name='m1.small',
            memory_mb=6,
            vcpus=28,
            root_gb=496,
            ephemeral_gb=8128,
            swap=33550336,
            extra_specs={},
        )
        instance_ref = objects.Instance(**test_instance)
        instance_ref.flavor = flavor
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(
            CONF.libvirt.virt_type,
            instance_ref,
            image_meta,
        )

        cfg = drvr._get_guest_config(
            instance_ref,
            _fake_network_info(self),
            image_meta, disk_info,
            context=ctxt,
        )

        self.assertEqual(cfg.uuid, instance_ref["uuid"])
        self.assertEqual(cfg.memory, 6 * units.Ki)
        self.assertEqual(cfg.vcpus, 28)
        self.assertEqual(cfg.os_type, fields.VMMode.HVM)
        self.assertEqual(cfg.os_boot_dev, ["hd"])
        self.assertIsNone(cfg.os_root)

        self.assertEqual(3, len(cfg.features))
        for idx, device_type in enumerate([
            vconfig.LibvirtConfigGuestFeatureACPI,
            vconfig.LibvirtConfigGuestFeatureAPIC,
            vconfig.LibvirtConfigGuestFeatureVMCoreInfo,
        ]):
            self.assertIsInstance(cfg.features[idx], device_type)

        self.assertEqual(len(cfg.devices), 11)
        for idx, device_type in enumerate([
            vconfig.LibvirtConfigGuestDisk,
            vconfig.LibvirtConfigGuestDisk,
            vconfig.LibvirtConfigGuestDisk,
            vconfig.LibvirtConfigGuestInterface,
            vconfig.LibvirtConfigGuestSerial,
            vconfig.LibvirtConfigGuestGraphics,
            vconfig.LibvirtConfigGuestVideo,
            vconfig.LibvirtConfigGuestInput,
            vconfig.LibvirtConfigGuestRng,
            vconfig.LibvirtConfigGuestUSBHostController,
            vconfig.LibvirtConfigMemoryBalloon,
        ]):
            self.assertIsInstance(cfg.devices[idx], device_type)

        self.assertEqual(len(cfg.metadata), 1)
        self.assertIsInstance(
            cfg.metadata[0], vconfig.LibvirtConfigGuestMetaNovaInstance)
        self.assertEqual(
            version.version_string_with_package(), cfg.metadata[0].package)
        self.assertEqual("purple tomatoes", cfg.metadata[0].name)
        self.assertEqual(1234567.89, cfg.metadata[0].creationTime)
        self.assertEqual("image", cfg.metadata[0].roottype)
        self.assertEqual(
            str(instance_ref["image_ref"]), cfg.metadata[0].rootid)

        self.assertIsInstance(
            cfg.metadata[0].owner, vconfig.LibvirtConfigGuestMetaNovaOwner)
        self.assertEqual(
            "838a72b0-0d54-4827-8fd6-fb1227633ceb",
            cfg.metadata[0].owner.userid)
        self.assertEqual("cupcake", cfg.metadata[0].owner.username)
        self.assertEqual("fake", cfg.metadata[0].owner.projectid)
        self.assertEqual("sweetshop", cfg.metadata[0].owner.projectname)
        self.assertIsInstance(
            cfg.metadata[0].flavor, vconfig.LibvirtConfigGuestMetaNovaFlavor)
        self.assertEqual("m1.small", cfg.metadata[0].flavor.name)
        self.assertEqual(6, cfg.metadata[0].flavor.memory)
        self.assertEqual(28, cfg.metadata[0].flavor.vcpus)
        self.assertEqual(496, cfg.metadata[0].flavor.disk)
        self.assertEqual(8128, cfg.metadata[0].flavor.ephemeral)
        self.assertEqual(33550336, cfg.metadata[0].flavor.swap)

        num_ports = 0
        for device in cfg.devices:
            try:
                if (
                    device.root_name == 'controller' and
                    device.model == 'pcie-root-port'
                ):
                    num_ports += 1
            except AttributeError:
                pass

        # i440fx is not pcie machine so there should be no pcie ports
        self.assertEqual(0, num_ports)

    @mock.patch.object(time, "time")
    def test_get_guest_config_no_pcie_ports(self, time_mock):
        """Generate a "standard" guest with minimal configuration.

        This uses i440fx by default since that's our default machine type and
        x86 is our default architecture (in our test env, anyway).
        """
        time_mock.return_value = 1234567.89

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        test_instance = copy.deepcopy(self.test_instance)
        test_instance["display_name"] = "purple tomatoes"
        test_instance['system_metadata']['owner_project_name'] = 'sweetshop'
        test_instance['system_metadata']['owner_user_name'] = 'cupcake'
        ctxt = context.RequestContext(
            project_id=123,
            project_name="aubergine",
            user_id=456,
            user_name="pie",
        )
        flavor = objects.Flavor(
            name='m1.small',
            memory_mb=6,
            vcpus=28,
            root_gb=496,
            ephemeral_gb=8128,
            swap=33550336,
            extra_specs={},
        )
        instance_ref = objects.Instance(**test_instance)
        instance_ref.flavor = flavor
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(
            CONF.libvirt.virt_type,
            instance_ref,
            image_meta,
        )

        cfg = drvr._get_guest_config(
            instance_ref,
            _fake_network_info(self),
            image_meta, disk_info,
            context=ctxt,
        )

        num_ports = 0
        for device in cfg.devices:
            try:
                if (
                    device.root_name == 'controller' and
                    device.model == 'pcie-root-port'
                ):
                    num_ports += 1
            except AttributeError:
                pass

        # i440fx is not pcie machine so there should be no pcie ports
        self.assertEqual(0, num_ports)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_q35(self):
        """Generate a "q35" guest with minimal configuration.

        This configures an explicit machine type (q35) but defaults to x86
        since this is our default architecture (in our test env, anyway).
        """
        self.flags(virt_type="kvm", group='libvirt')

        TEST_AMOUNT_OF_PCIE_SLOTS = 8
        CONF.set_override(
            "num_pcie_ports", TEST_AMOUNT_OF_PCIE_SLOTS,
            group='libvirt',
        )

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_machine_type": "q35"},
        })

        disk_info = blockinfo.get_disk_info(
            CONF.libvirt.virt_type,
            instance_ref,
            image_meta,
        )

        cfg = drvr._get_guest_config(
            instance_ref,
            _fake_network_info(self),
            image_meta,
            disk_info,
        )

        self.assertEqual(3, len(cfg.features))
        for idx, device_type in enumerate([
            vconfig.LibvirtConfigGuestFeatureACPI,
            vconfig.LibvirtConfigGuestFeatureAPIC,
            vconfig.LibvirtConfigGuestFeatureVMCoreInfo,
        ]):
            self.assertIsInstance(cfg.features[idx], device_type)

        self.assertEqual(len(cfg.devices), 19)
        for idx, device_type in enumerate([
            vconfig.LibvirtConfigGuestDisk,
            vconfig.LibvirtConfigGuestDisk,
            vconfig.LibvirtConfigGuestInterface,
            vconfig.LibvirtConfigGuestSerial,
            vconfig.LibvirtConfigGuestGraphics,
            vconfig.LibvirtConfigGuestVideo,
            vconfig.LibvirtConfigGuestInput,
            vconfig.LibvirtConfigGuestRng,
            vconfig.LibvirtConfigGuestPCIeRootController,
            vconfig.LibvirtConfigGuestPCIeRootPortController,
            vconfig.LibvirtConfigGuestPCIeRootPortController,
            vconfig.LibvirtConfigGuestPCIeRootPortController,
            vconfig.LibvirtConfigGuestPCIeRootPortController,
            vconfig.LibvirtConfigGuestPCIeRootPortController,
            vconfig.LibvirtConfigGuestPCIeRootPortController,
            vconfig.LibvirtConfigGuestPCIeRootPortController,
            vconfig.LibvirtConfigGuestPCIeRootPortController,
            vconfig.LibvirtConfigGuestUSBHostController,
            vconfig.LibvirtConfigMemoryBalloon,
        ]):
            self.assertIsInstance(cfg.devices[idx], device_type)

        num_ports = 0
        for device in cfg.devices:
            try:
                if (
                    device.root_name == 'controller' and
                    device.model == 'pcie-root-port'
                ):
                    num_ports += 1
            except AttributeError:
                pass

        self.assertEqual(TEST_AMOUNT_OF_PCIE_SLOTS, num_ports)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    @mock.patch('nova.virt.libvirt.utils.get_default_machine_type',
                new=mock.Mock(return_value='config-machine_type'))
    def test_get_guest_config_records_machine_type_in_instance(self):
        # Assert that the config derived machine type is used when it
        # isn't present in the image_meta of an instance.
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({})
        disk_info = blockinfo.get_disk_info(
            CONF.libvirt.virt_type,
            instance,
            image_meta
        )

        cfg = drvr._get_guest_config(
            instance,
            _fake_network_info(self),
            image_meta,
            disk_info
        )

        self.assertEqual(
            'config-machine_type',
            instance.system_metadata.get('image_hw_machine_type'),
        )
        self.assertEqual(
            'config-machine_type',
            cfg.os_mach_type,
        )

    def test_get_guest_config_missing_ownership_info(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        test_instance = copy.deepcopy(self.test_instance)

        ctxt = context.RequestContext(project_id=123,
                                      project_name="aubergine",
                                      user_id=456,
                                      user_name="pie")

        flavor = objects.Flavor(name='m1.small',
                                memory_mb=6,
                                vcpus=28,
                                root_gb=496,
                                ephemeral_gb=8128,
                                swap=33550336,
                                extra_specs={})
        instance_ref = objects.Instance(**test_instance)
        instance_ref.flavor = flavor
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref,
                                     _fake_network_info(self),
                                     image_meta, disk_info,
                                     context=ctxt)
        self.assertEqual("N/A",
                         cfg.metadata[0].owner.username)
        self.assertEqual("N/A",
                         cfg.metadata[0].owner.projectname)

    def test_get_guest_config_lxc(self):
        self.flags(virt_type='lxc', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        cfg = drvr._get_guest_config(instance_ref,
                                     _fake_network_info(self),
                                     image_meta, {'mapping': {}})
        self.assertEqual(instance_ref["uuid"], cfg.uuid)
        self.assertEqual(instance_ref.flavor.memory_mb * units.Ki, cfg.memory)
        self.assertEqual(instance_ref.flavor.vcpus, cfg.vcpus)
        self.assertEqual(fields.VMMode.EXE, cfg.os_type)
        self.assertEqual("/sbin/init", cfg.os_init_path)
        self.assertEqual("console=tty0 console=ttyS0 console=hvc0",
                         cfg.os_cmdline)
        self.assertEqual("OpenStack Nova", cfg.os_init_env['product_name'])
        self.assertIsNone(cfg.os_root)
        self.assertEqual(4, len(cfg.devices))
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestFilesys)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestInterface)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestConsole)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestUSBHostController)

    def test_get_guest_config_lxc_with_id_maps(self):
        self.flags(virt_type='lxc', group='libvirt')
        self.flags(uid_maps=['0:1000:100'], group='libvirt')
        self.flags(gid_maps=['0:1000:100'], group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        cfg = drvr._get_guest_config(instance_ref,
                                     _fake_network_info(self),
                                     image_meta, {'mapping': {}})
        self.assertEqual(instance_ref["uuid"], cfg.uuid)
        self.assertEqual(instance_ref.flavor.memory_mb * units.Ki, cfg.memory)
        self.assertEqual(instance_ref.vcpus, cfg.vcpus)
        self.assertEqual(fields.VMMode.EXE, cfg.os_type)
        self.assertEqual("/sbin/init", cfg.os_init_path)
        self.assertEqual("console=tty0 console=ttyS0 console=hvc0",
                         cfg.os_cmdline)
        self.assertIsNone(cfg.os_root)
        self.assertEqual(4, len(cfg.devices))
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestFilesys)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestInterface)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestConsole)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertEqual(len(cfg.idmaps), 2)
        self.assertIsInstance(cfg.idmaps[0],
                              vconfig.LibvirtConfigGuestUIDMap)
        self.assertIsInstance(cfg.idmaps[1],
                              vconfig.LibvirtConfigGuestGIDMap)

    def test_post_claim_migrate_data(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance = objects.Instance(**self.test_instance)
        md = objects.LibvirtLiveMigrateData()
        claim = mock.Mock(autospec=True)
        claimed_numa_topology = objects.InstanceNUMATopology()
        claim.claimed_numa_topology = claimed_numa_topology
        claim.flavor = instance.flavor
        numa_info = objects.LibvirtLiveMigrateNUMAInfo()
        with test.nested(
            mock.patch.object(drvr, '_get_live_migrate_numa_info',
                              return_value=numa_info),
            mock.patch('nova.objects.Instance.image_meta',
                       new_callable=mock.PropertyMock,
                       return_value='fake-image-meta')
        ) as (mock_get_lm_numa_info, mock_image_meta):
            claim.image_meta = instance.image_meta
            post_claim_md = drvr.post_claim_migrate_data(
                self.context, instance, md, claim)
            self.assertEqual(post_claim_md.dst_numa_info, numa_info)
            mock_get_lm_numa_info.assert_called_with(
                claimed_numa_topology, instance.flavor, 'fake-image-meta')

    @mock.patch.object(hardware, 'get_vcpu_pin_set', new=mock.Mock())
    def test_get_live_migrate_numa_info(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        vcpupin1 = vconfig.LibvirtConfigGuestCPUTuneVCPUPin()
        vcpupin1.id = 0
        vcpupin1.cpuset = set([0, 1])
        vcpupin2 = vconfig.LibvirtConfigGuestCPUTuneVCPUPin()
        vcpupin2.id = 1
        vcpupin2.cpuset = set([2, 3])
        emulatorpin = vconfig.LibvirtConfigGuestCPUTuneEmulatorPin()
        emulatorpin.cpuset = set([4, 5])
        guest_cpu_tune = vconfig.LibvirtConfigGuestCPUTune()
        guest_cpu_tune.vcpupin = [vcpupin1, vcpupin2]
        guest_cpu_tune.emulatorpin = emulatorpin
        guest_cpu_tune.vcpusched = [
            vconfig.LibvirtConfigGuestCPUTuneVCPUSched()]
        guest_cpu_tune.vcpusched[0].vcpus = set([6, 7])
        guest_cpu_tune.vcpusched[0].priority = 8

        memnode1 = vconfig.LibvirtConfigGuestNUMATuneMemNode()
        memnode1.cellid = 2
        memnode1.nodeset = [6, 7]
        memnode2 = vconfig.LibvirtConfigGuestNUMATuneMemNode()
        memnode2.cellid = 3
        memnode2.nodeset = [8, 9]
        guest_numa_tune = vconfig.LibvirtConfigGuestNUMATune()
        guest_numa_tune.memnodes = [memnode1, memnode2]

        expected_numa_info = objects.LibvirtLiveMigrateNUMAInfo(
            cpu_pins={'0': set([0, 1]), '1': set([2, 3])},
            cell_pins={'2': set([6, 7]), '3': set([8, 9])},
            emulator_pins=set([4, 5]),
            sched_vcpus=set([7, 6]),
            sched_priority=8)

        # NOTE(artom) This is a
        # (cpu_set, guest_cpu_tune, guest_cpu_numa, guest_numa_tune)
        # tuple. See _get_guest_numa_config() docstring for full documentation.
        # _get_live_migrate_numa_info() only cares about guest_cpu_tune for CPU
        # pinning and emulator thread pinning, and guest_numa_tune for cell
        # pinning; so only include those 2 in the tuple.
        guest_numa_config = (None, guest_cpu_tune, None, guest_numa_tune)

        with mock.patch.object(drvr, '_get_guest_numa_config',
                               return_value=guest_numa_config):
            self.assertEqual(
                expected_numa_info.obj_to_primitive(),
                drvr._get_live_migrate_numa_info(
                    'fake-instance-numa-topology',
                    'fake-flavor', 'fake-image-meta').obj_to_primitive())

    @mock.patch.object(hardware, 'get_vcpu_pin_set')
    def test_get_live_migrate_numa_info_empty(self, _):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        guest_numa_config = (None, None, None, None)
        with mock.patch.object(drvr, '_get_guest_numa_config',
                               return_value=guest_numa_config):
            self.assertEqual(
                objects.LibvirtLiveMigrateNUMAInfo().obj_to_primitive(),
                drvr._get_live_migrate_numa_info(
                    'fake-instance-numa-topology',
                    'fake-flavor', 'fake-image-meta').obj_to_primitive())

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_numa_host_instance_fits(self):
        self.flags(cpu_shared_set=None, cpu_dedicated_set=None,
                   group='compute')
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        flavor = objects.Flavor(memory_mb=1, vcpus=2, root_gb=496,
                                ephemeral_gb=8128, swap=33550336, name='fake',
                                extra_specs={})
        instance_ref.flavor = flavor

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology()

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        with test.nested(
                mock.patch.object(host.Host, 'has_min_version',
                                  return_value=True),
                mock.patch.object(host.Host, "get_capabilities",
                                  return_value=caps),
                mock.patch.object(host.Host, 'get_online_cpus',
                                  return_value=set([0, 1])),
                ):
            cfg = drvr._get_guest_config(instance_ref, [],
                                         image_meta, disk_info)
            self.assertIsNone(cfg.cpuset)
            self.assertIsNone(cfg.cputune)
            self.assertIsNone(cfg.cpu.numa)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    @mock.patch('nova.privsep.utils.supports_direct_io',
                new=mock.Mock(return_value=True))
    def test_get_guest_config_numa_host_instance_no_fit(self):
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        flavor = objects.Flavor(memory_mb=4096, vcpus=4, root_gb=496,
                                ephemeral_gb=8128, swap=33550336, name='fake',
                                extra_specs={})
        instance_ref.flavor = flavor

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology()

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        with test.nested(
                mock.patch.object(host.Host, "get_capabilities",
                                  return_value=caps),
                mock.patch.object(random, 'choice'),
                mock.patch.object(drvr, '_has_numa_support',
                                  return_value=False)
            ) as (_, choice_mock, _):
            cfg = drvr._get_guest_config(instance_ref, [],
                                         image_meta, disk_info)
            self.assertFalse(choice_mock.called)
            self.assertIsNone(cfg.cpuset)
            self.assertIsNone(cfg.cputune)
            self.assertIsNone(cfg.cpu.numa)

    def _test_get_guest_memory_backing_config(
            self, host_topology, inst_topology, numatune):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        flavor = objects.Flavor(memory_mb=4096, vcpus=4, root_gb=496,
                                ephemeral_gb=8128, swap=33550336, name='fake',
                                extra_specs={})
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        with mock.patch.object(
                drvr, "_get_host_numa_topology",
                return_value=host_topology):
            return drvr._get_guest_memory_backing_config(
                inst_topology, numatune, flavor, image_meta)

    @mock.patch.object(host.Host,
                       'has_min_version', return_value=True)
    def test_get_guest_memory_backing_config_large_success(self, mock_version):
        host_topology = objects.NUMATopology(cells=[
            objects.NUMACell(
                id=3,
                cpuset=set([1]),
                pcpuset=set(),
                siblings=[set([1])],
                memory=1024,
                mempages=[
                    objects.NUMAPagesTopology(size_kb=4, total=2000, used=0),
                    objects.NUMAPagesTopology(size_kb=2048, total=512, used=0),
                    objects.NUMAPagesTopology(size_kb=1048576, total=0,
                                              used=0),
                ])])
        inst_topology = objects.InstanceNUMATopology(cells=[
            objects.InstanceNUMACell(
                id=3, cpuset=set([0, 1]), pcpuset=set(), memory=1024,
                pagesize=2048),
        ])

        numa_tune = vconfig.LibvirtConfigGuestNUMATune()
        numa_tune.memnodes = [vconfig.LibvirtConfigGuestNUMATuneMemNode()]
        numa_tune.memnodes[0].cellid = 0
        numa_tune.memnodes[0].nodeset = [3]

        result = self._test_get_guest_memory_backing_config(
            host_topology, inst_topology, numa_tune)
        self.assertEqual(1, len(result.hugepages))
        self.assertEqual(2048, result.hugepages[0].size_kb)
        self.assertEqual([0], result.hugepages[0].nodeset)

    @mock.patch.object(host.Host,
                       'has_min_version', return_value=True)
    def test_get_guest_memory_backing_config_smallest(self, mock_version):
        host_topology = objects.NUMATopology(cells=[
            objects.NUMACell(
                id=3,
                cpuset=set([1]),
                pcpuset=set(),
                siblings=[set([1])],
                memory=1024,
                mempages=[
                    objects.NUMAPagesTopology(size_kb=4, total=2000, used=0),
                    objects.NUMAPagesTopology(size_kb=2048, total=512, used=0),
                    objects.NUMAPagesTopology(size_kb=1048576, total=0,
                                              used=0),
                ])])
        inst_topology = objects.InstanceNUMATopology(cells=[
            objects.InstanceNUMACell(
                id=3, cpuset=set([0, 1]), pcpuset=set(), memory=1024,
                pagesize=4),
        ])

        numa_tune = vconfig.LibvirtConfigGuestNUMATune()
        numa_tune.memnodes = [vconfig.LibvirtConfigGuestNUMATuneMemNode()]
        numa_tune.memnodes[0].cellid = 0
        numa_tune.memnodes[0].nodeset = [3]

        result = self._test_get_guest_memory_backing_config(
            host_topology, inst_topology, numa_tune)
        self.assertIsNone(result)

    def test_get_guest_memory_backing_config_realtime(self):
        extra_specs = {
            "hw:cpu_realtime": "yes",
            "hw:cpu_policy": "dedicated"
        }
        flavor = objects.Flavor(name='m1.small',
                                memory_mb=6,
                                vcpus=28,
                                root_gb=496,
                                ephemeral_gb=8128,
                                swap=33550336,
                                extra_specs=extra_specs)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        membacking = drvr._get_guest_memory_backing_config(
            None, None, flavor, image_meta)
        self.assertTrue(membacking.locked)
        self.assertFalse(membacking.sharedpages)

    def test_get_guest_memory_backing_config_locked_flavor(self):
        extra_specs = {
            "hw:locked_memory": "True",
            "hw:mem_page_size": 1000,
        }
        flavor = objects.Flavor(
            name='m1.small', memory_mb=6, vcpus=28, root_gb=496,
            ephemeral_gb=8128, swap=33550336, extra_specs=extra_specs)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        membacking = drvr._get_guest_memory_backing_config(
            None, None, flavor, image_meta)
        self.assertTrue(membacking.locked)

    def test_get_guest_memory_backing_config_locked_image_meta(self):
        extra_specs = {}
        flavor = objects.Flavor(
            name='m1.small',
            memory_mb=6,
            vcpus=28,
            root_gb=496,
            ephemeral_gb=8128,
            swap=33550336,
            extra_specs=extra_specs)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {
                "hw_locked_memory": "True",
                "hw_mem_page_size": 1000,
            }})
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        membacking = drvr._get_guest_memory_backing_config(
            None, None, flavor, image_meta)
        self.assertTrue(membacking.locked)

    def test_get_guest_memory_backing_config_realtime_invalid_share(self):
        """Test behavior when there is no pool of shared CPUS on which to place
        the emulator threads, isolating them from the instance CPU processes.
        """
        extra_specs = {
            "hw:cpu_realtime": "yes",
            "hw:cpu_policy": "dedicated",
            "hw:emulator_threads_policy": "share",
        }
        flavor = objects.Flavor(
            name='m1.small',
            memory_mb=6,
            vcpus=28,
            root_gb=496,
            ephemeral_gb=8128,
            swap=33550336,
            extra_specs=extra_specs)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        # this should fail because there is nowhere to place the emulator
        # threads
        self.assertRaises(
            exception.RealtimeMaskNotFoundOrInvalid,
            drvr._get_guest_memory_backing_config,
            None, None, flavor, image_meta,
        )

    def _test_sev_enabled(self, expected=None, host_sev_enabled=False,
                          enc_extra_spec=None, enc_image_prop=None,
                          hw_machine_type=None, hw_firmware_type=None):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._host._supports_amd_sev = host_sev_enabled

        extra_specs = {}
        if enc_extra_spec is not None:
            extra_specs['hw:mem_encryption'] = enc_extra_spec
        flavor = objects.Flavor(name='m1.fake')
        flavor.extra_specs = extra_specs

        image_props = {}
        image_props['hw_architecture'] = fields.Architecture.X86_64
        if hw_machine_type is not None:
            image_props['hw_machine_type'] = hw_machine_type
        if hw_firmware_type is not None:
            image_props['hw_firmware_type'] = hw_firmware_type
        if enc_image_prop is not None:
            image_props['hw_mem_encryption'] = enc_image_prop

        image_meta = fake_image.fake_image_obj(
            {'id': '150d530b-1c57-4367-b754-1f1b5237923d'},
            {}, image_props)

        enabled = drvr._sev_enabled(flavor, image_meta)

        if expected is None:
            self.fail("_test_sev_enabled called without an expected "
                      "return value. Maybe you expected an exception?")

        self.assertEqual(expected, enabled)

    def test_sev_enabled_no_host_support(self):
        self._test_sev_enabled(False)

    def test_sev_enabled_host_support_no_flavor_image(self):
        self._test_sev_enabled(False, host_sev_enabled=True)

    def test_sev_enabled_no_host_support_flavor_requested(self):
        self._test_sev_enabled(False, enc_extra_spec=True)

    def test_sev_enabled_no_host_support_image_requested(self):
        self._test_sev_enabled(False, enc_image_prop=True)

    def test_sev_enabled_host_support_flavor_requested(self):
        self._test_sev_enabled(True, host_sev_enabled=True,
                               enc_extra_spec=True, hw_firmware_type='uefi',
                               hw_machine_type='q35')

    def test_sev_enabled_host_support_image_requested(self):
        self._test_sev_enabled(True, host_sev_enabled=True,
                               enc_image_prop=True, hw_firmware_type='uefi',
                               hw_machine_type='q35')

    # The cases where the flavor and image requests contradict each other
    # are already covered by test_hardware.MemEncryptionConflictTestCase
    # so we don't need to test them in great detail here.
    def test_sev_enabled_host_extra_spec_image_conflict(self):
        exc = self.assertRaises(exception.FlavorImageConflict,
                                self._test_sev_enabled,
                                host_sev_enabled=True, enc_extra_spec=False,
                                enc_image_prop=True)
        self.assertEqual(
            "Flavor m1.fake has hw:mem_encryption extra spec explicitly set "
            "to False, conflicting with image fake_image which has "
            "hw_mem_encryption property explicitly set to True", str(exc))

    def test_sev_enabled_host_extra_spec_no_uefi(self):
        exc = self.assertRaises(exception.FlavorImageConflict,
                                self._test_sev_enabled,
                                host_sev_enabled=True, enc_extra_spec=True)
        self.assertEqual(
            "Memory encryption requested by hw:mem_encryption extra spec in "
            "m1.fake flavor but image fake_image doesn't have "
            "'hw_firmware_type' property set to 'uefi' or volume-backed "
            "instance was requested", str(exc))

    def test_sev_enabled_host_extra_spec_no_machine_type(self):
        exc = self.assertRaises(exception.InvalidMachineType,
                                self._test_sev_enabled,
                                host_sev_enabled=True, enc_extra_spec=True,
                                hw_firmware_type='uefi')
        self.assertEqual(
            "Machine type 'pc' is not compatible with image fake_image "
            "(150d530b-1c57-4367-b754-1f1b5237923d): q35 type is required "
            "for SEV to work", str(exc))

    def test_sev_enabled_host_extra_spec_pc(self):
        exc = self.assertRaises(exception.InvalidMachineType,
                                self._test_sev_enabled,
                                host_sev_enabled=True, enc_extra_spec=True,
                                hw_firmware_type='uefi', hw_machine_type='pc')
        self.assertEqual(
            "Machine type 'pc' is not compatible with image fake_image "
            "(150d530b-1c57-4367-b754-1f1b5237923d): q35 type is required "
            "for SEV to work", str(exc))

    def _setup_fake_domain_caps(self, fake_domain_caps):
        sev_feature = vconfig.LibvirtConfigDomainCapsFeatureSev()
        sev_feature.cbitpos = 47
        sev_feature.reduced_phys_bits = 1
        domain_caps = vconfig.LibvirtConfigDomainCaps()
        domain_caps._features = vconfig.LibvirtConfigDomainCapsFeatures()
        domain_caps._features.features = [sev_feature]
        domain_caps._os = vconfig.LibvirtConfigDomainCapsOS()
        domain_caps._os.loader_paths = ['foo']

        fake_domain_caps.return_value = collections.defaultdict(
            dict, {'x86_64': {'q35': domain_caps}})

    @mock.patch.object(host.Host, 'get_domain_capabilities')
    def test_find_sev_feature_missing_arch(self, fake_domain_caps):
        self._setup_fake_domain_caps(fake_domain_caps)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertIsNone(drvr._find_sev_feature('arm1', 'q35'))

    @mock.patch.object(host.Host, 'get_domain_capabilities')
    def test_find_sev_feature_missing_mach_type(self, fake_domain_caps):
        self._setup_fake_domain_caps(fake_domain_caps)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertIsNone(drvr._find_sev_feature('x86_64', 'g3beige'))

    @mock.patch.object(host.Host, 'get_domain_capabilities')
    def test_find_sev_feature(self, fake_domain_caps):
        self._setup_fake_domain_caps(fake_domain_caps)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        feature = drvr._find_sev_feature('x86_64', 'q35')
        self.assertIsInstance(feature,
                              vconfig.LibvirtConfigDomainCapsFeatureSev)
        self.assertEqual(47, feature.cbitpos)
        self.assertEqual(1, feature.reduced_phys_bits)

    def _setup_sev_guest(self, extra_image_properties=None):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._host._supports_uefi = True
        drvr._host._supports_amd_sev = True

        ctxt = context.RequestContext(project_id=123,
                                      project_name="aubergine",
                                      user_id=456,
                                      user_name="pie")

        extra_specs = {
            "hw:mem_encryption": True,
        }
        flavor = objects.Flavor(name='m1.small',
                                memory_mb=6,
                                vcpus=28,
                                root_gb=496,
                                ephemeral_gb=8128,
                                swap=33550336,
                                extra_specs=extra_specs)

        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor = flavor
        image_meta_properties = {
            'hw_firmware_type': 'uefi',
            'hw_machine_type': 'q35'}
        if extra_image_properties:
            image_meta_properties.update(extra_image_properties)
        image_meta = objects.ImageMeta.from_dict({
            'id': 'd9c6aeee-8258-4bdb-bca4-39940461b182',
            'name': 'fakeimage',
            'disk_format': 'raw',
            'properties': image_meta_properties})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        return drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info,
                                      context=ctxt)

    def test_get_guest_config_sev_no_feature(self):
        self.assertRaises(exception.MissingDomainCapabilityFeatureException,
                          self._setup_sev_guest)

    @mock.patch.object(host.Host, 'get_domain_capabilities')
    @mock.patch.object(designer, 'set_driver_iommu_for_all_devices')
    def test_get_guest_config_sev(self, mock_designer, fake_domain_caps):
        self._setup_fake_domain_caps(fake_domain_caps)
        cfg = self._setup_sev_guest()

        # SEV-related tag should be set
        self.assertIsInstance(cfg.launch_security,
                              vconfig.LibvirtConfigGuestSEVLaunchSecurity)
        self.assertIsInstance(cfg.membacking,
                              vconfig.LibvirtConfigGuestMemoryBacking)
        self.assertTrue(cfg.membacking.locked)

        mock_designer.assert_called_once_with(cfg)

    def test_get_guest_memory_backing_config_file_backed(self):
        self.flags(file_backed_memory=1024, group="libvirt")

        result = self._test_get_guest_memory_backing_config(
            None, None, None
        )
        self.assertTrue(result.sharedaccess)
        self.assertTrue(result.filesource)
        self.assertTrue(result.allocateimmediate)

    def test_get_guest_memory_backing_config_file_backed_hugepages(self):
        self.flags(file_backed_memory=1024, group="libvirt")
        host_topology = objects.NUMATopology(cells=[
            objects.NUMACell(
                id=3,
                cpuset=set([1]),
                pcpuset=set(),
                siblings=[set([1])],
                memory=1024,
                mempages=[
                    objects.NUMAPagesTopology(size_kb=4, total=2000, used=0),
                    objects.NUMAPagesTopology(size_kb=2048, total=512, used=0),
                    objects.NUMAPagesTopology(size_kb=1048576, total=0,
                                              used=0),
                ])])
        inst_topology = objects.InstanceNUMATopology(cells=[
            objects.InstanceNUMACell(
                id=3, cpuset=set([0, 1]), pcpuset=set(), memory=1024,
                pagesize=2048),
        ])

        numa_tune = vconfig.LibvirtConfigGuestNUMATune()
        numa_tune.memnodes = [vconfig.LibvirtConfigGuestNUMATuneMemNode()]
        numa_tune.memnodes[0].cellid = 0
        numa_tune.memnodes[0].nodeset = [3]

        self.assertRaises(exception.MemoryPagesUnsupported,
                          self._test_get_guest_memory_backing_config,
                          host_topology, inst_topology, numa_tune)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_numa_host_instance_pci_no_numa_info(self):
        self.flags(cpu_shared_set='3', cpu_dedicated_set=None,
                   group='compute')

        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        flavor = objects.Flavor(memory_mb=1, vcpus=2, root_gb=496,
                                ephemeral_gb=8128, swap=33550336, name='fake',
                                extra_specs={})
        instance_ref.flavor = flavor

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology()

        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        pci_device_info = dict(test_pci_device.fake_db_dev)
        pci_device_info.update(compute_node_id=1,
                               label='fake',
                               status=fields.PciDeviceStatus.AVAILABLE,
                               address='0000:00:00.1',
                               instance_uuid=None,
                               request_id=uuids.pci_req1,
                               extra_info={},
                               numa_node=None)
        pci_device = objects.PciDevice(**pci_device_info)
        instance_ref.pci_devices = objects.PciDeviceList(objects=[pci_device])
        pci_req = objects.InstancePCIRequest(
            request_id=uuids.pci_req1, alias_name='pci-alias-1')
        instance_ref.pci_requests = objects.InstancePCIRequests(
            requests=[pci_req])

        with test.nested(
                mock.patch.object(host.Host, 'has_min_version',
                                  return_value=True),
                mock.patch.object(host.Host, "get_capabilities",
                                  return_value=caps),
                mock.patch.object(host.Host, 'get_online_cpus',
                                  return_value=set([3]))
        ):
            cfg = conn._get_guest_config(instance_ref, [],
                                         image_meta, disk_info)
            self.assertEqual(set([3]), cfg.cpuset)
            self.assertIsNone(cfg.cputune)
            self.assertIsNone(cfg.cpu.numa)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    @mock.patch('nova.privsep.utils.supports_direct_io',
                new=mock.Mock(return_value=True))
    def test_get_guest_config_numa_host_instance_2pci_no_fit(self):
        self.flags(cpu_shared_set='3', cpu_dedicated_set=None,
                   group='compute')
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        flavor = objects.Flavor(memory_mb=4096, vcpus=4, root_gb=496,
                                ephemeral_gb=8128, swap=33550336, name='fake',
                                extra_specs={})
        instance_ref.flavor = flavor

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology()

        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        pci_device_info = dict(test_pci_device.fake_db_dev)
        pci_device_info.update(compute_node_id=1,
                               label='fake',
                               status=fields.PciDeviceStatus.AVAILABLE,
                               address='0000:00:00.1',
                               instance_uuid=None,
                               request_id=uuids.pci_req1,
                               extra_info={},
                               numa_node=1)
        pci_device = objects.PciDevice(**pci_device_info)
        pci_device_info.update(numa_node=0, address='0000:00:00.2')
        pci_device2 = objects.PciDevice(**pci_device_info)
        instance_ref.pci_devices = objects.PciDeviceList(
            objects=[pci_device, pci_device2]
        )
        instance_ref.pci_requests = objects.InstancePCIRequests(
            requests=[
                objects.InstancePCIRequest(
                    request_id=uuids.pci_req1, alias_name="pci-alias-1"
                )
            ]
        )
        with test.nested(
                mock.patch.object(
                    host.Host, "get_capabilities", return_value=caps),
                mock.patch.object(host.Host, 'get_online_cpus',
                                  return_value=set([3])),
                mock.patch.object(random, 'choice'),
                mock.patch.object(conn, '_has_numa_support',
                                  return_value=False)
            ) as (_, _, choice_mock, _):
            cfg = conn._get_guest_config(instance_ref, [],
                                         image_meta, disk_info)
            self.assertFalse(choice_mock.called)
            self.assertEqual(set([3]), cfg.cpuset)
            self.assertIsNone(cfg.cputune)
            self.assertIsNone(cfg.cpu.numa)

    @mock.patch.object(fakelibvirt.Connection, 'getType')
    @mock.patch.object(fakelibvirt.Connection, 'getVersion')
    @mock.patch.object(fakelibvirt.Connection, 'getLibVersion')
    @mock.patch.object(host.Host, 'get_capabilities')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_set_host_enabled')
    def _test_get_guest_config_numa_unsupported(self, fake_lib_version,
                                                fake_version, fake_type,
                                                fake_arch, exception_class,
                                                pagesize, mock_host,
                                                mock_caps, mock_lib_version,
                                                mock_version, mock_type):
        instance_topology = objects.InstanceNUMATopology(cells=[
            objects.InstanceNUMACell(
                id=0, cpuset=set([0]), pcpuset=set(),
                memory=1024, pagesize=pagesize),
        ])
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.numa_topology = instance_topology
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        flavor = objects.Flavor(memory_mb=1, vcpus=2, root_gb=496,
                                ephemeral_gb=8128, swap=33550336, name='fake',
                                extra_specs={})
        instance_ref.flavor = flavor

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fake_arch
        caps.host.topology = fakelibvirt.NUMATopology()

        mock_type.return_value = fake_type
        mock_version.return_value = fake_version
        mock_lib_version.return_value = fake_lib_version
        mock_caps.return_value = caps

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        self.assertRaises(exception_class,
                          drvr._get_guest_config,
                          instance_ref, [],
                          image_meta, disk_info)

    def test_get_guest_config_numa_other_arch_qemu(self):
        self.flags(virt_type='kvm', group='libvirt')

        self._test_get_guest_config_numa_unsupported(
            versionutils.convert_version_to_int(
                libvirt_driver.MIN_LIBVIRT_VERSION),
            versionutils.convert_version_to_int(
                libvirt_driver.MIN_QEMU_VERSION),
            host.HV_DRIVER_QEMU,
            fields.Architecture.S390,
            exception.NUMATopologyUnsupported,
            None)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_numa_host_instance_fit_w_cpu_pinset(self):
        self.flags(cpu_shared_set='2-3', cpu_dedicated_set=None,
                   group='compute')

        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        flavor = objects.Flavor(memory_mb=1024, vcpus=2, root_gb=496,
                                ephemeral_gb=8128, swap=33550336, name='fake',
                                extra_specs={})
        instance_ref.flavor = flavor

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology(kb_mem=4194304)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        with test.nested(
                mock.patch.object(host.Host, 'has_min_version',
                                  return_value=True),
                mock.patch.object(host.Host, 'get_capabilities',
                                  return_value=caps),
                mock.patch.object(host.Host, 'get_online_cpus',
                                  return_value=set([2, 3])),
                ):
            cfg = drvr._get_guest_config(instance_ref, [],
                                         image_meta, disk_info)
            # NOTE(ndipanov): we make sure that pin_set was taken into account
            # when choosing viable cells
            self.assertEqual(set([2, 3]), cfg.cpuset)
            self.assertIsNone(cfg.cputune)
            self.assertIsNone(cfg.cpu.numa)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_non_numa_host_instance_topo(self):
        instance_topology = objects.InstanceNUMATopology(cells=[
            objects.InstanceNUMACell(
                id=0, cpuset=set([0]), pcpuset=set(), memory=1024),
            objects.InstanceNUMACell(
                id=1, cpuset=set([2]), pcpuset=set(), memory=1024),
        ])
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.numa_topology = instance_topology
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        flavor = objects.Flavor(memory_mb=2048, vcpus=2, root_gb=496,
                                ephemeral_gb=8128, swap=33550336, name='fake',
                                extra_specs={})
        instance_ref.flavor = flavor

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = None

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        with test.nested(
                mock.patch.object(
                    objects.InstanceNUMATopology, "get_by_instance_uuid",
                    return_value=instance_topology),
                mock.patch.object(host.Host, 'has_min_version',
                                  return_value=True),
                mock.patch.object(host.Host, "get_capabilities",
                                  return_value=caps)):
            cfg = drvr._get_guest_config(instance_ref, [],
                                         image_meta, disk_info)
            self.assertIsNone(cfg.cpuset)
            self.assertIsNone(cfg.cputune)
            self.assertIsNone(cfg.numatune)
            self.assertIsNotNone(cfg.cpu.numa)
            for instance_cell, numa_cfg_cell in zip(
                    instance_topology.cells, cfg.cpu.numa.cells):
                self.assertEqual(instance_cell.id, numa_cfg_cell.id)
                self.assertEqual(instance_cell.total_cpus, numa_cfg_cell.cpus)
                self.assertEqual(instance_cell.memory * units.Ki,
                                 numa_cfg_cell.memory)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_numa_host_instance_topo(self):
        self.flags(cpu_shared_set='0-5', cpu_dedicated_set=None,
                   group='compute')

        instance_topology = objects.InstanceNUMATopology(cells=[
            objects.InstanceNUMACell(
                id=1, cpuset=set([0, 1]), pcpuset=set(), memory=1024,
                pagesize=None),
            objects.InstanceNUMACell(
                id=2, cpuset=set([2, 3]), pcpuset=set(), memory=1024,
                pagesize=None),
        ])
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.numa_topology = instance_topology
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        flavor = objects.Flavor(memory_mb=2048, vcpus=4, root_gb=496,
                                ephemeral_gb=8128, swap=33550336, name='fake',
                                extra_specs={})
        instance_ref.flavor = flavor

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology()

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        with test.nested(
                mock.patch.object(
                    objects.InstanceNUMATopology, "get_by_instance_uuid",
                    return_value=instance_topology),
                mock.patch.object(host.Host, 'has_min_version',
                                  return_value=True),
                mock.patch.object(host.Host, "get_capabilities",
                                  return_value=caps),
                mock.patch.object(host.Host, 'get_online_cpus',
                                  return_value=set(range(8))),
                ):
            cfg = drvr._get_guest_config(instance_ref, [],
                                         image_meta, disk_info)
            self.assertIsNone(cfg.cpuset)
            # Test that the pinning is correct and limited to allowed only
            self.assertEqual(0, cfg.cputune.vcpupin[0].id)
            self.assertEqual(set([2, 3]), cfg.cputune.vcpupin[0].cpuset)
            self.assertEqual(1, cfg.cputune.vcpupin[1].id)
            self.assertEqual(set([2, 3]), cfg.cputune.vcpupin[1].cpuset)
            self.assertEqual(2, cfg.cputune.vcpupin[2].id)
            self.assertEqual(set([4, 5]), cfg.cputune.vcpupin[2].cpuset)
            self.assertEqual(3, cfg.cputune.vcpupin[3].id)
            self.assertEqual(set([4, 5]), cfg.cputune.vcpupin[3].cpuset)
            self.assertIsNotNone(cfg.cpu.numa)

            self.assertIsInstance(cfg.cputune.emulatorpin,
                                  vconfig.LibvirtConfigGuestCPUTuneEmulatorPin)
            self.assertEqual(set([2, 3, 4, 5]), cfg.cputune.emulatorpin.cpuset)

            for instance_cell, numa_cfg_cell, index in zip(
                    instance_topology.cells,
                    cfg.cpu.numa.cells,
                    range(len(instance_topology.cells))):
                self.assertEqual(index, numa_cfg_cell.id)
                self.assertEqual(instance_cell.total_cpus, numa_cfg_cell.cpus)
                self.assertEqual(instance_cell.memory * units.Ki,
                                 numa_cfg_cell.memory)

            allnodes = [cell.id for cell in instance_topology.cells]
            self.assertEqual(allnodes, cfg.numatune.memory.nodeset)
            self.assertEqual("strict", cfg.numatune.memory.mode)

            for instance_cell, memnode, index in zip(
                    instance_topology.cells,
                    cfg.numatune.memnodes,
                    range(len(instance_topology.cells))):
                self.assertEqual(index, memnode.cellid)
                self.assertEqual([instance_cell.id], memnode.nodeset)
                self.assertEqual("strict", memnode.mode)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_numa_host_instance_topo_reordered(self):
        instance_topology = objects.InstanceNUMATopology(cells=[
            objects.InstanceNUMACell(
                id=3, cpuset=set([0, 1]), pcpuset=set(), memory=1024),
            objects.InstanceNUMACell(
                id=0, cpuset=set([2, 3]), pcpuset=set(), memory=1024),
        ])
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.numa_topology = instance_topology
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        flavor = objects.Flavor(memory_mb=2048, vcpus=4, root_gb=496,
                                ephemeral_gb=8128, swap=33550336, name='fake',
                                extra_specs={})
        instance_ref.flavor = flavor

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology()

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        with test.nested(
                mock.patch.object(
                    objects.InstanceNUMATopology, "get_by_instance_uuid",
                    return_value=instance_topology),
                mock.patch.object(host.Host, 'has_min_version',
                                  return_value=True),
                mock.patch.object(host.Host, "get_capabilities",
                                  return_value=caps),
                mock.patch.object(host.Host, 'get_online_cpus',
                                  return_value=set(range(8))),
                ):
            cfg = drvr._get_guest_config(instance_ref, [],
                                         image_meta, disk_info)
            self.assertIsNone(cfg.cpuset)
            # Test that the pinning is correct and limited to allowed only
            self.assertEqual(0, cfg.cputune.vcpupin[0].id)
            self.assertEqual(set([6, 7]), cfg.cputune.vcpupin[0].cpuset)
            self.assertEqual(1, cfg.cputune.vcpupin[1].id)
            self.assertEqual(set([6, 7]), cfg.cputune.vcpupin[1].cpuset)
            self.assertEqual(2, cfg.cputune.vcpupin[2].id)
            self.assertEqual(set([0, 1]), cfg.cputune.vcpupin[2].cpuset)
            self.assertEqual(3, cfg.cputune.vcpupin[3].id)
            self.assertEqual(set([0, 1]), cfg.cputune.vcpupin[3].cpuset)
            self.assertIsNotNone(cfg.cpu.numa)

            self.assertIsInstance(cfg.cputune.emulatorpin,
                                  vconfig.LibvirtConfigGuestCPUTuneEmulatorPin)
            self.assertEqual(set([0, 1, 6, 7]), cfg.cputune.emulatorpin.cpuset)

            for index, (instance_cell, numa_cfg_cell) in enumerate(zip(
                    instance_topology.cells,
                    cfg.cpu.numa.cells)):
                self.assertEqual(index, numa_cfg_cell.id)
                self.assertEqual(instance_cell.total_cpus, numa_cfg_cell.cpus)
                self.assertEqual(instance_cell.memory * units.Ki,
                                 numa_cfg_cell.memory)
                self.assertIsNone(numa_cfg_cell.memAccess)

            allnodes = set([cell.id for cell in instance_topology.cells])
            self.assertEqual(allnodes, set(cfg.numatune.memory.nodeset))
            self.assertEqual("strict", cfg.numatune.memory.mode)

            for index, (instance_cell, memnode) in enumerate(zip(
                    instance_topology.cells,
                    cfg.numatune.memnodes)):
                self.assertEqual(index, memnode.cellid)
                self.assertEqual([instance_cell.id], memnode.nodeset)
                self.assertEqual("strict", memnode.mode)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_numa_host_instance_topo_cpu_pinning(self):
        instance_topology = objects.InstanceNUMATopology(cells=[
            objects.InstanceNUMACell(
                id=1, cpuset=set(), pcpuset=set([0, 1]), memory=1024,
                cpu_pinning={0: 24, 1: 25}),
            objects.InstanceNUMACell(
                id=0, cpuset=set(), pcpuset=set([2, 3]), memory=1024,
                cpu_pinning={2: 0, 3: 1}),
        ])
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.numa_topology = instance_topology
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        flavor = objects.Flavor(memory_mb=2048, vcpus=2, root_gb=496,
                                ephemeral_gb=8128, swap=33550336, name='fake',
                                extra_specs={})
        instance_ref.flavor = flavor

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology(
            cpu_nodes=4, cpu_sockets=1, cpu_cores=4, cpu_threads=2)

        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        with test.nested(
                mock.patch.object(
                    objects.InstanceNUMATopology, "get_by_instance_uuid",
                    return_value=instance_topology),
                mock.patch.object(host.Host, 'has_min_version',
                                  return_value=True),
                mock.patch.object(host.Host, "get_capabilities",
                                  return_value=caps),
                mock.patch.object(host.Host, 'get_online_cpus',
                                  return_value=set(range(32))),
                ):
            cfg = conn._get_guest_config(instance_ref, [],
                                         image_meta, disk_info)
            self.assertIsNone(cfg.cpuset)
            # Test that the pinning is correct and limited to allowed only
            self.assertEqual(0, cfg.cputune.vcpupin[0].id)
            self.assertEqual(set([24]), cfg.cputune.vcpupin[0].cpuset)
            self.assertEqual(1, cfg.cputune.vcpupin[1].id)
            self.assertEqual(set([25]), cfg.cputune.vcpupin[1].cpuset)
            self.assertEqual(2, cfg.cputune.vcpupin[2].id)
            self.assertEqual(set([0]), cfg.cputune.vcpupin[2].cpuset)
            self.assertEqual(3, cfg.cputune.vcpupin[3].id)
            self.assertEqual(set([1]), cfg.cputune.vcpupin[3].cpuset)
            self.assertIsNotNone(cfg.cpu.numa)

            # Emulator must be pinned to union of cfg.cputune.vcpupin[*].cpuset
            self.assertIsInstance(cfg.cputune.emulatorpin,
                                  vconfig.LibvirtConfigGuestCPUTuneEmulatorPin)
            self.assertEqual(set([0, 1, 24, 25]),
                             cfg.cputune.emulatorpin.cpuset)

            for i, (instance_cell, numa_cfg_cell) in enumerate(zip(
                    instance_topology.cells, cfg.cpu.numa.cells)):
                self.assertEqual(i, numa_cfg_cell.id)
                self.assertEqual(instance_cell.total_cpus, numa_cfg_cell.cpus)
                self.assertEqual(instance_cell.memory * units.Ki,
                                 numa_cfg_cell.memory)
                self.assertIsNone(numa_cfg_cell.memAccess)

            allnodes = set([cell.id for cell in instance_topology.cells])
            self.assertEqual(allnodes, set(cfg.numatune.memory.nodeset))
            self.assertEqual("strict", cfg.numatune.memory.mode)

            for i, (instance_cell, memnode) in enumerate(zip(
                    instance_topology.cells, cfg.numatune.memnodes)):
                self.assertEqual(i, memnode.cellid)
                self.assertEqual([instance_cell.id], memnode.nodeset)
                self.assertEqual("strict", memnode.mode)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_numa_host_instance_cpu_mixed(self):
        """Test to create mixed instance libvirt configuration which has a
        default emulator thread policy and verify the NUMA topology related
        settings.
        """
        self.flags(cpu_shared_set='2-5,8-29',
                   cpu_dedicated_set='6,7,30,31',
                   group='compute')

        instance_topology = objects.InstanceNUMATopology(cells=[
            objects.InstanceNUMACell(
                id=3, cpuset=set([0, 1]), pcpuset=set([2, 3]), memory=1024,
                cpu_pinning={2: 30, 3: 31}
            ),
            objects.InstanceNUMACell(
                id=0, cpuset=set([4, 5, 6]), pcpuset=set([7]), memory=1024,
                cpu_pinning={7: 6}
            ),
        ])
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.numa_topology = instance_topology
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        flavor = objects.Flavor(memory_mb=2048, vcpus=8, root_gb=496,
                                ephemeral_gb=8128, swap=33550336, name='fake',
                                extra_specs={})
        instance_ref.flavor = flavor

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology(
            cpu_nodes=4, cpu_sockets=1, cpu_cores=4, cpu_threads=2)
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        with test.nested(
            mock.patch.object(
                objects.InstanceNUMATopology, "get_by_instance_uuid",
                return_value=instance_topology),
            mock.patch.object(host.Host, 'has_min_version', return_value=True),
            mock.patch.object(host.Host, "get_capabilities",
                              return_value=caps),
            mock.patch.object(host.Host, 'get_online_cpus',
                              return_value=set(range(32)))
        ):
            cfg = conn._get_guest_config(instance_ref, [],
                                         image_meta, disk_info)
            self.assertIsNone(cfg.cpuset)

            # NOTE(huaqiang): Within a mixed instance, it is expected that the
            # pinned and unpinned CPUs, which belong to the same instance NUMA
            # cell, are scheduled on a same host NUMA cell, the instance pinned
            # CPU is 1:1 scheduled to a dedicated host CPU, each unpinned CPU
            # floats over the shared CPU list of the same host NUMA cell.
            # The host NUMA cell's dedicated CPU list and shared CPU list are
            # calculated from a combination of '[compute]cpu_dedicated_set',
            # '[compute]cpu_shared_set', the host NUMA topology and the online
            # CPUs.
            #
            # The first instance NUMA cell is fit into the fourth host NUMA
            # cell due to having the same 'id' 3. Instance CPU 0 and 1 are
            # unpinned CPUs, check each of them floats on the host NUMA cell's
            # sharing CPUs, which are CPU 24-29.
            self.assertEqual(0, cfg.cputune.vcpupin[0].id)
            self.assertEqual(set(range(24, 30)), cfg.cputune.vcpupin[0].cpuset)
            self.assertEqual(1, cfg.cputune.vcpupin[1].id)
            self.assertEqual(set(range(24, 30)), cfg.cputune.vcpupin[1].cpuset)
            # Check each of the instance NUMA cell's pinned CPUs is pinned to a
            # dedicated CPU from the fourth host NUMA cell.
            self.assertEqual(2, cfg.cputune.vcpupin[2].id)
            self.assertEqual(set([30]), cfg.cputune.vcpupin[2].cpuset)
            self.assertEqual(3, cfg.cputune.vcpupin[3].id)
            self.assertEqual(set([31]), cfg.cputune.vcpupin[3].cpuset)

            # Instance CPU 4-7 belong to the second instance NUMA cell, which
            # is fit into host NUMA cell 0. CPU 4-6 are unpinned CPUs, each of
            # them floats on the host NUMA cell's sharing CPU set, CPU 2-5.
            self.assertEqual(4, cfg.cputune.vcpupin[4].id)
            self.assertEqual(set(range(2, 6)), cfg.cputune.vcpupin[4].cpuset)
            self.assertEqual(5, cfg.cputune.vcpupin[5].id)
            self.assertEqual(set(range(2, 6)), cfg.cputune.vcpupin[5].cpuset)
            self.assertEqual(6, cfg.cputune.vcpupin[6].id)
            self.assertEqual(set(range(2, 6)), cfg.cputune.vcpupin[6].cpuset)
            # Instance CPU 7 is pinned to the host NUMA cell's dedicated CPU 6.
            self.assertEqual(set([6]), cfg.cputune.vcpupin[7].cpuset)
            self.assertIsNotNone(cfg.cpu.numa)

            # Check emulator thread is pinned to union of
            # cfg.cputune.vcpupin[*].cpuset
            self.assertIsInstance(cfg.cputune.emulatorpin,
                                  vconfig.LibvirtConfigGuestCPUTuneEmulatorPin)
            self.assertEqual(
                set([2, 3, 4, 5, 6, 24, 25, 26, 27, 28, 29, 30, 31]),
                cfg.cputune.emulatorpin.cpuset)

            for i, (instance_cell, numa_cfg_cell) in enumerate(
                zip(instance_topology.cells, cfg.cpu.numa.cells)
            ):
                self.assertEqual(i, numa_cfg_cell.id)
                self.assertEqual(instance_cell.total_cpus, numa_cfg_cell.cpus)
                self.assertEqual(instance_cell.memory * units.Ki,
                                 numa_cfg_cell.memory)
                self.assertIsNone(numa_cfg_cell.memAccess)

            allnodes = set([cell.id for cell in instance_topology.cells])
            self.assertEqual(allnodes, set(cfg.numatune.memory.nodeset))
            self.assertEqual("strict", cfg.numatune.memory.mode)

            for i, (instance_cell, memnode) in enumerate(
                zip(instance_topology.cells, cfg.numatune.memnodes)
            ):
                self.assertEqual(i, memnode.cellid)
                self.assertEqual([instance_cell.id], memnode.nodeset)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_numa_host_instance_cpu_mixed_isolated_emu(
            self):
        """Test to create mixed instance libvirt configuration which has an
        ISOLATED emulator thread policy and verify the NUMA topology related
        settings.
        """
        self.flags(cpu_shared_set='2-5,8-29',
                   cpu_dedicated_set='6,7,30,31',
                   group='compute')
        instance_topology = objects.InstanceNUMATopology(
            emulator_threads_policy=fields.CPUEmulatorThreadsPolicy.ISOLATE,
            cells=[objects.InstanceNUMACell(
                id=0, cpuset=set([0, 1]), pcpuset=set([2]), memory=1024,
                cpu_pinning={2: 6},
                cpuset_reserved=set([7]))])
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.numa_topology = instance_topology
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        flavor = objects.Flavor(memory_mb=2048, vcpus=8, root_gb=496,
                                ephemeral_gb=8128, swap=33550336, name='fake',
                                extra_specs={})
        instance_ref.flavor = flavor

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology(
            cpu_nodes=4, cpu_sockets=1, cpu_cores=4, cpu_threads=2)

        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        with test.nested(
            mock.patch.object(
                objects.InstanceNUMATopology, "get_by_instance_uuid",
                return_value=instance_topology),
            mock.patch.object(host.Host, 'has_min_version',
                              return_value=True),
            mock.patch.object(host.Host, "get_capabilities",
                              return_value=caps),
            mock.patch.object(host.Host, 'get_online_cpus',
                              return_value=set(range(32))),
        ):
            cfg = conn._get_guest_config(instance_ref, [],
                                         image_meta, disk_info)
            self.assertIsNone(cfg.cpuset)
            # NOTE(huaqiang): The instance NUMA cell is fit into the first host
            # NUMA cell, which is matched by the 'id' fields of two objects.
            # CPU 2-5 are the first host NUMA cell's floating CPU set.
            # Check any instance unpinned CPU is floating on this CPU set.
            self.assertEqual(0, cfg.cputune.vcpupin[0].id)
            self.assertEqual(set([2, 3, 4, 5]), cfg.cputune.vcpupin[0].cpuset)
            self.assertEqual(1, cfg.cputune.vcpupin[1].id)
            self.assertEqual(set([2, 3, 4, 5]), cfg.cputune.vcpupin[1].cpuset)
            # Check instance CPU 2, a pinned CPU, is pinned to a dedicated CPU
            # of host's first NUMA cell.
            self.assertEqual(2, cfg.cputune.vcpupin[2].id)
            self.assertEqual(set([6]), cfg.cputune.vcpupin[2].cpuset)
            self.assertIsNotNone(cfg.cpu.numa)

            # With an ISOLATE policy, emulator thread will be pinned to the
            # reserved host CPU.
            self.assertIsInstance(cfg.cputune.emulatorpin,
                                  vconfig.LibvirtConfigGuestCPUTuneEmulatorPin)
            self.assertEqual(set([7]), cfg.cputune.emulatorpin.cpuset)

            for i, (instance_cell, numa_cfg_cell) in enumerate(
                zip(instance_topology.cells, cfg.cpu.numa.cells)
            ):
                self.assertEqual(i, numa_cfg_cell.id)
                self.assertEqual(instance_cell.total_cpus, numa_cfg_cell.cpus)
                self.assertEqual(instance_cell.memory * units.Ki,
                                 numa_cfg_cell.memory)
                self.assertIsNone(numa_cfg_cell.memAccess)

            allnodes = set([cell.id for cell in instance_topology.cells])
            self.assertEqual(allnodes, set(cfg.numatune.memory.nodeset))
            self.assertEqual("strict", cfg.numatune.memory.mode)

            for i, (instance_cell, memnode) in enumerate(
                zip(instance_topology.cells, cfg.numatune.memnodes)
            ):
                self.assertEqual(i, memnode.cellid)
                self.assertEqual([instance_cell.id], memnode.nodeset)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_numa_host_instance_cpu_mixed_realtime(self):
        """Test of creating mixed instance libvirt configuration. which is
        created through 'hw:cpu_realtime_mask' and 'hw:cpu_realtime' extra
        specs, verifying the NUMA topology and real-time related settings.
        """
        self.flags(cpu_shared_set='2-5,8-29',
                   cpu_dedicated_set='6,7,30,31',
                   group='compute')

        instance_topology = objects.InstanceNUMATopology(
            cells=[
                objects.InstanceNUMACell(
                    id=0, cpuset=set([2]), pcpuset=set([0, 1]),
                    cpu_pinning={0: 6, 1: 7},
                    memory=1024, pagesize=2048),
                objects.InstanceNUMACell(
                    id=3, cpuset=set([3]), pcpuset=set([4, 5]),
                    cpu_pinning={4: 30, 5: 31},
                    memory=1024, pagesize=2048)])
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.numa_topology = instance_topology
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        # NOTE(huaqiang): libvirt driver takes the real-time CPU list from the
        # flavor extra spec 'hw:cpu_realtime_mask'. For a mixed instance with
        # real-time CPUs, the dedicated CPU and the real-time CPU are the
        # same CPU set, this is checked in API layer.
        flavor = objects.Flavor(
            vcpus=6, memory_mb=2048, root_gb=496,
            ephemeral_gb=8128, swap=33550336, name='fake',
            extra_specs={
                "hw:numa_nodes": "2",
                "hw:cpu_realtime": "yes",
                "hw:cpu_policy": "mixed",
                "hw:cpu_realtime_mask": "^2-3"
            })
        instance_ref.flavor = flavor

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology(
            cpu_nodes=4, cpu_sockets=1, cpu_cores=4, cpu_threads=2)
        for i, cell in enumerate(caps.host.topology.cells):
            cell.mempages = fakelibvirt.create_mempages(
                [(4, 1024 * i), (2048, i)])

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        with test.nested(
            mock.patch.object(
                objects.InstanceNUMATopology, "get_by_instance_uuid",
                return_value=instance_topology),
            mock.patch.object(host.Host, 'has_min_version', return_value=True),
            mock.patch.object(host.Host, "get_capabilities",
                              return_value=caps),
            mock.patch.object(host.Host, 'get_online_cpus',
                              return_value=set(range(32))),
        ):
            cfg = drvr._get_guest_config(instance_ref, [],
                                         image_meta, disk_info)

            for instance_cell, numa_cfg_cell, index in zip(
                instance_topology.cells,
                cfg.cpu.numa.cells,
                range(len(instance_topology.cells))
            ):
                self.assertEqual(index, numa_cfg_cell.id)
                self.assertEqual(instance_cell.total_cpus, numa_cfg_cell.cpus)
                self.assertEqual(instance_cell.memory * units.Ki,
                                 numa_cfg_cell.memory)
                self.assertEqual("shared", numa_cfg_cell.memAccess)

            allnodes = [cell.id for cell in instance_topology.cells]
            self.assertEqual(allnodes, cfg.numatune.memory.nodeset)
            self.assertEqual("strict", cfg.numatune.memory.mode)

            for instance_cell, memnode, index in zip(
                instance_topology.cells,
                cfg.numatune.memnodes,
                range(len(instance_topology.cells))
            ):
                self.assertEqual(index, memnode.cellid)
                self.assertEqual([instance_cell.id], memnode.nodeset)
                self.assertEqual("strict", memnode.mode)

            # NOTE(huaqiang): Instance first NUMA cell is fit to the first host
            # NUMA cell. In this host NUMA cell, CPU 2-5 are the sharing CPU
            # set, CPU 6, 7 are dedicated CPUs.
            #
            # Check instance CPU 0, 1 are 1:1 pinned on host NUMA cell's
            # dedicated CPUs.
            self.assertEqual(set([6]), cfg.cputune.vcpupin[0].cpuset)
            self.assertEqual(set([7]), cfg.cputune.vcpupin[1].cpuset)
            # Check CPU 2, an unpinned CPU, is floating on this host NUMA
            # cell's sharing CPU set.
            self.assertEqual(set([2, 3, 4, 5]), cfg.cputune.vcpupin[2].cpuset)

            # The second instance NUMA cell is fit to the fourth host NUMA
            # cell due to a same 'id'. Host CPU 24-29 are sharing CPU set, host
            # CPU 30, 31 are dedicated CPU to be pinning.
            #
            # Check CPU 3 is floating on the sharing CPU set.
            self.assertEqual(set([24, 25, 26, 27, 28, 29]),
                             cfg.cputune.vcpupin[3].cpuset)
            # Check CPU 4, 5 are pinned on host dedicated CPUs.
            self.assertEqual(set([30]), cfg.cputune.vcpupin[4].cpuset)
            self.assertEqual(set([31]), cfg.cputune.vcpupin[5].cpuset)

            # Check the real-time host CPUs are excluded from the host CPU
            # list the emulator is floating on.
            self.assertEqual(set([2, 3, 4, 5, 24, 25, 26, 27, 28, 29]),
                             cfg.cputune.emulatorpin.cpuset)

            # Check the real-time scheduler is set, and all real-time CPUs are
            # in the vcpusched[0].vcpus list. In nova, the real-time scheduler
            # is always set to 'fifo', and there is always only one element in
            # cfg.cputune.vcpusched.
            self.assertEqual(1, len(cfg.cputune.vcpusched))
            self.assertEqual("fifo", cfg.cputune.vcpusched[0].scheduler)
            self.assertEqual(set([0, 1, 4, 5]), cfg.cputune.vcpusched[0].vcpus)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_numa_host_mempages_shared(self):
        self.flags(cpu_shared_set='2-5', cpu_dedicated_set=None,
                   group='compute')

        instance_topology = objects.InstanceNUMATopology(cells=[
            objects.InstanceNUMACell(
                id=1, cpuset=set([0, 1]), pcpuset=set(), memory=1024,
                pagesize=2048),
            objects.InstanceNUMACell(
                id=2, cpuset=set([2, 3]), pcpuset=set(), memory=1024,
                pagesize=2048),
        ])
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.numa_topology = instance_topology
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        flavor = objects.Flavor(memory_mb=2048, vcpus=4, root_gb=496,
                                ephemeral_gb=8128, swap=33550336, name='fake',
                                extra_specs={})
        instance_ref.flavor = flavor

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology()
        for i, cell in enumerate(caps.host.topology.cells):
            cell.mempages = fakelibvirt.create_mempages(
                [(4, 1024 * i), (2048, i)])

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        with test.nested(
                mock.patch.object(
                    objects.InstanceNUMATopology, "get_by_instance_uuid",
                    return_value=instance_topology),
                mock.patch.object(host.Host, 'has_min_version',
                                  return_value=True),
                mock.patch.object(host.Host, "get_capabilities",
                                  return_value=caps),
                mock.patch.object(host.Host, 'get_online_cpus',
                                  return_value=set([2, 3, 4, 5])),
                ):
            cfg = drvr._get_guest_config(instance_ref, [],
                                         image_meta, disk_info)

            for instance_cell, numa_cfg_cell, index in zip(
                    instance_topology.cells,
                    cfg.cpu.numa.cells,
                    range(len(instance_topology.cells))):
                self.assertEqual(index, numa_cfg_cell.id)
                self.assertEqual(instance_cell.total_cpus, numa_cfg_cell.cpus)
                self.assertEqual(instance_cell.memory * units.Ki,
                                 numa_cfg_cell.memory)
                self.assertEqual("shared", numa_cfg_cell.memAccess)

            allnodes = [cell.id for cell in instance_topology.cells]
            self.assertEqual(allnodes, cfg.numatune.memory.nodeset)
            self.assertEqual("strict", cfg.numatune.memory.mode)

            for instance_cell, memnode, index in zip(
                    instance_topology.cells,
                    cfg.numatune.memnodes,
                    range(len(instance_topology.cells))):
                self.assertEqual(index, memnode.cellid)
                self.assertEqual([instance_cell.id], memnode.nodeset)
                self.assertEqual("strict", memnode.mode)

            self.assertEqual(0, len(cfg.cputune.vcpusched))
            self.assertEqual(set([2, 3, 4, 5]), cfg.cputune.emulatorpin.cpuset)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_numa_host_instance_cpu_pinning_realtime(
            self):
        self.flags(cpu_shared_set=None, cpu_dedicated_set='4-7',
                   group='compute')

        instance_topology = objects.InstanceNUMATopology(cells=[
            objects.InstanceNUMACell(
                id=2, cpuset=set(), pcpuset=set([0, 1]),
                cpu_policy=fields.CPUAllocationPolicy.DEDICATED,
                cpu_pinning={0: 4, 1: 5},
                memory=1024, pagesize=2048),
            objects.InstanceNUMACell(
                id=3, cpuset=set(), pcpuset=set([2, 3]),
                cpu_policy=fields.CPUAllocationPolicy.DEDICATED,
                cpu_pinning={2: 6, 3: 7},
                memory=1024, pagesize=2048),
        ])
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.numa_topology = instance_topology
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        flavor = objects.Flavor(memory_mb=2048, vcpus=4, root_gb=496,
                                ephemeral_gb=8128, swap=33550336, name='fake',
                                extra_specs={
                                    "hw:numa_nodes": "2",
                                    "hw:cpu_realtime": "yes",
                                    "hw:cpu_policy": "dedicated",
                                    "hw:cpu_realtime_mask": "^0-1"
                                })
        instance_ref.flavor = flavor

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology()
        for i, cell in enumerate(caps.host.topology.cells):
            cell.mempages = fakelibvirt.create_mempages(
                [(4, 1024 * i), (2048, i)])

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        with test.nested(
                mock.patch.object(
                    objects.InstanceNUMATopology, "get_by_instance_uuid",
                    return_value=instance_topology),
                mock.patch.object(host.Host, 'has_min_version',
                                  return_value=True),
                mock.patch.object(host.Host, "get_capabilities",
                                  return_value=caps),
                mock.patch.object(host.Host, 'get_online_cpus',
                                  return_value=set(range(8))),
                ):
            cfg = drvr._get_guest_config(instance_ref, [],
                                         image_meta, disk_info)

            for instance_cell, numa_cfg_cell, index in zip(
                    instance_topology.cells,
                    cfg.cpu.numa.cells,
                    range(len(instance_topology.cells))):
                self.assertEqual(index, numa_cfg_cell.id)
                self.assertEqual(instance_cell.total_cpus, numa_cfg_cell.cpus)
                self.assertEqual(instance_cell.memory * units.Ki,
                                 numa_cfg_cell.memory)
                self.assertEqual("shared", numa_cfg_cell.memAccess)

            allnodes = [cell.id for cell in instance_topology.cells]
            self.assertEqual(allnodes, cfg.numatune.memory.nodeset)
            self.assertEqual("strict", cfg.numatune.memory.mode)

            for instance_cell, memnode, index in zip(
                    instance_topology.cells,
                    cfg.numatune.memnodes,
                    range(len(instance_topology.cells))):
                self.assertEqual(index, memnode.cellid)
                self.assertEqual([instance_cell.id], memnode.nodeset)
                self.assertEqual("strict", memnode.mode)

            self.assertEqual(1, len(cfg.cputune.vcpusched))
            self.assertEqual("fifo", cfg.cputune.vcpusched[0].scheduler)

            self.assertEqual(set([4]), cfg.cputune.vcpupin[0].cpuset)
            self.assertEqual(set([5]), cfg.cputune.vcpupin[1].cpuset)
            self.assertEqual(set([6]), cfg.cputune.vcpupin[2].cpuset)
            self.assertEqual(set([7]), cfg.cputune.vcpupin[3].cpuset)

            # We ensure that emulator threads are pinned on host CPUs
            # 4-5 which are "normal" vCPUs
            self.assertEqual(set([4, 5]), cfg.cputune.emulatorpin.cpuset)

            # We ensure that the vCPUs RT are 2-3 set to the host CPUs
            # which are 6, 7
            self.assertEqual(set([2, 3]), cfg.cputune.vcpusched[0].vcpus)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_numa_host_instance_isolated_emulthreads(self):
        self.flags(cpu_shared_set=None, cpu_dedicated_set='4-8',
                   group='compute')

        instance_topology = objects.InstanceNUMATopology(
            emulator_threads_policy=(
                fields.CPUEmulatorThreadsPolicy.ISOLATE),
            cells=[
                objects.InstanceNUMACell(
                    id=0, cpuset=set(), pcpuset=set([0, 1]),
                    memory=1024, pagesize=2048,
                    cpu_policy=fields.CPUAllocationPolicy.DEDICATED,
                    cpu_pinning={0: 4, 1: 5},
                    cpuset_reserved=set([6])),
                objects.InstanceNUMACell(
                    id=1, cpuset=set(), pcpuset=set([2, 3]),
                    memory=1024, pagesize=2048,
                    cpu_policy=fields.CPUAllocationPolicy.DEDICATED,
                    cpu_pinning={2: 7, 3: 8}),
            ])

        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.numa_topology = instance_topology
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = "x86_64"
        caps.host.topology = fakelibvirt.NUMATopology()

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref, image_meta)

        with test.nested(
                mock.patch.object(
                    objects.InstanceNUMATopology, "get_by_instance_uuid",
                    return_value=instance_topology),
                mock.patch.object(host.Host, 'has_min_version',
                                  return_value=True),
                mock.patch.object(host.Host, "get_capabilities",
                                  return_value=caps),
                mock.patch.object(host.Host, 'get_online_cpus',
                                  return_value=set(range(10))),
                ):
            cfg = drvr._get_guest_config(instance_ref, [],
                                         image_meta, disk_info)

            self.assertEqual(set([6]), cfg.cputune.emulatorpin.cpuset)
            self.assertEqual(set([4]), cfg.cputune.vcpupin[0].cpuset)
            self.assertEqual(set([5]), cfg.cputune.vcpupin[1].cpuset)
            self.assertEqual(set([7]), cfg.cputune.vcpupin[2].cpuset)
            self.assertEqual(set([8]), cfg.cputune.vcpupin[3].cpuset)

    def test_get_guest_config_numa_host_instance_shared_emulthreads_err(
            self):
        self.flags(cpu_shared_set='48-50', cpu_dedicated_set='4-8',
                   group='compute')

        instance_topology = objects.InstanceNUMATopology(
            emulator_threads_policy=(
                fields.CPUEmulatorThreadsPolicy.SHARE),
            cells=[
                objects.InstanceNUMACell(
                    id=0, cpuset=set(), pcpuset=set([0, 1]),
                    memory=1024, pagesize=2048,
                    cpu_policy=fields.CPUAllocationPolicy.DEDICATED,
                    cpu_pinning={0: 4, 1: 5},
                    cpuset_reserved=set([6])),
                objects.InstanceNUMACell(
                    id=1, cpuset=set(), pcpuset=set([2, 3]),
                    memory=1024, pagesize=2048,
                    cpu_policy=fields.CPUAllocationPolicy.DEDICATED,
                    cpu_pinning={2: 7, 3: 8}),
            ])

        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.numa_topology = instance_topology
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = "x86_64"
        caps.host.topology = fakelibvirt.NUMATopology()

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref, image_meta)

        with test.nested(
                mock.patch.object(
                    objects.InstanceNUMATopology, "get_by_instance_uuid",
                    return_value=instance_topology),
                mock.patch.object(host.Host, 'has_min_version',
                                  return_value=True),
                mock.patch.object(host.Host, "get_capabilities",
                                  return_value=caps),
                mock.patch.object(host.Host, 'get_online_cpus',
                                  return_value=set(range(10))),
                ):
            # pCPUs [48-50] are not online
            self.assertRaises(exception.Invalid, drvr._get_guest_config,
                              instance_ref, [], image_meta, disk_info)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_numa_host_instance_shared_emulator_threads(self):
        self.flags(cpu_shared_set='0,1', cpu_dedicated_set='2-7',
                   group='compute')
        instance_topology = objects.InstanceNUMATopology(
            emulator_threads_policy=(
                fields.CPUEmulatorThreadsPolicy.SHARE),
            cells=[
                objects.InstanceNUMACell(
                    id=0, cpuset=set(), pcpuset=set([0, 1]),
                    memory=1024, pagesize=2048,
                    cpu_policy=fields.CPUAllocationPolicy.DEDICATED,
                    cpu_pinning={0: 2, 1: 3},
                    cpuset_reserved=set([6])),
                objects.InstanceNUMACell(
                    id=1, cpuset=set(), pcpuset=set([2, 3]),
                    memory=1024, pagesize=2048,
                    cpu_policy=fields.CPUAllocationPolicy.DEDICATED,
                    cpu_pinning={2: 4, 3: 5}),
            ])

        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.numa_topology = instance_topology
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = "x86_64"
        caps.host.topology = fakelibvirt.NUMATopology()

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref, image_meta)

        with test.nested(
                mock.patch.object(
                    objects.InstanceNUMATopology, "get_by_instance_uuid",
                    return_value=instance_topology),
                mock.patch.object(host.Host, 'has_min_version',
                                  return_value=True),
                mock.patch.object(host.Host, "get_capabilities",
                                  return_value=caps),
                mock.patch.object(host.Host, 'get_online_cpus',
                                  return_value=set(range(8))),
                ):
            cfg = drvr._get_guest_config(instance_ref, [],
                                         image_meta, disk_info)

            # emulator threads should be mapped to cores from 'cpu_shared_set'
            self.assertEqual(set([0, 1]), cfg.cputune.emulatorpin.cpuset)
            self.assertEqual(set([2]), cfg.cputune.vcpupin[0].cpuset)
            self.assertEqual(set([3]), cfg.cputune.vcpupin[1].cpuset)
            self.assertEqual(set([4]), cfg.cputune.vcpupin[2].cpuset)
            self.assertEqual(set([5]), cfg.cputune.vcpupin[3].cpuset)

    def test_get_cpu_numa_config_from_instance(self):
        topology = objects.InstanceNUMATopology(cells=[
            objects.InstanceNUMACell(
                id=0, cpuset=set([1, 2]), pcpuset=set(), memory=128),
            objects.InstanceNUMACell(
                id=1, cpuset=set([3, 4]), pcpuset=set(), memory=128),
        ])
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        conf = drvr._get_cpu_numa_config_from_instance(topology, True)

        self.assertIsInstance(conf, vconfig.LibvirtConfigGuestCPUNUMA)
        self.assertEqual(0, conf.cells[0].id)
        self.assertEqual(set([1, 2]), conf.cells[0].cpus)
        self.assertEqual(131072, conf.cells[0].memory)
        self.assertEqual("shared", conf.cells[0].memAccess)
        self.assertEqual(1, conf.cells[1].id)
        self.assertEqual(set([3, 4]), conf.cells[1].cpus)
        self.assertEqual(131072, conf.cells[1].memory)
        self.assertEqual("shared", conf.cells[1].memAccess)

    def test_get_cpu_numa_config_from_instance_none(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        conf = drvr._get_cpu_numa_config_from_instance(None, False)
        self.assertIsNone(conf)

    @mock.patch.object(libvirt_driver.LibvirtDriver, "_has_numa_support",
                       return_value=True)
    def test_get_memnode_numa_config_from_instance(self, mock_numa):
        instance_topology = objects.InstanceNUMATopology(cells=[
            objects.InstanceNUMACell(
                id=0, cpuset=set([1, 2]), pcpuset=set(), memory=128),
            objects.InstanceNUMACell(
                id=1, cpuset=set([3, 4]), pcpuset=set(), memory=128),
            objects.InstanceNUMACell(
                id=16, cpuset=set([5, 6]), pcpuset=set(), memory=128),
        ])

        host_topology = objects.NUMATopology(cells=[
            objects.NUMACell(
                id=0,
                cpuset=set([1, 2]),
                pcpuset=set(),
                siblings=[set([1]), set([2])],
                memory=1024,
                mempages=[]),
            objects.NUMACell(
                id=1,
                cpuset=set([3, 4]),
                pcpuset=set(),
                siblings=[set([3]), set([4])],
                memory=1024,
                mempages=[]),
            objects.NUMACell(
                id=16,
                cpuset=set([5, 6]),
                pcpuset=set(),
                siblings=[set([5]), set([6])],
                memory=1024,
                mempages=[])])

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        with test.nested(
                mock.patch.object(drvr, "_get_host_numa_topology",
                                  return_value=host_topology),
                mock.patch.object(hardware, "get_cpu_shared_set",
                                  return_value=[1, 2, 3, 4, 5, 6])):
            guest_numa_config = drvr._get_guest_numa_config(
                instance_topology, flavor={}, image_meta={})
            self.assertEqual(2, guest_numa_config.numatune.memnodes[2].cellid)
            self.assertEqual([16],
                guest_numa_config.numatune.memnodes[2].nodeset)
            self.assertEqual(set([5, 6]),
                guest_numa_config.numaconfig.cells[2].cpus)

    @mock.patch.object(libvirt_driver.LibvirtDriver, "_has_numa_support",
                       return_value=True)
    @mock.patch.object(host.Host, "get_capabilities")
    def test_does_not_want_hugepages(self, mock_caps, mock_numa):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_topology = objects.InstanceNUMATopology(cells=[
            objects.InstanceNUMACell(
                id=1, cpuset=set([0, 1]), pcpuset=set(),
                memory=1024, pagesize=4),
            objects.InstanceNUMACell(
                id=2, cpuset=set([2, 3]), pcpuset=set(),
                memory=1024, pagesize=4),
        ])

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology()

        mock_caps.return_value = caps

        host_topology = drvr._get_host_numa_topology()

        self.assertFalse(drvr._wants_hugepages(None, None))
        self.assertFalse(drvr._wants_hugepages(host_topology, None))
        self.assertFalse(drvr._wants_hugepages(None, instance_topology))
        self.assertFalse(drvr._wants_hugepages(host_topology,
                                               instance_topology))

    @mock.patch.object(libvirt_driver.LibvirtDriver, "_has_numa_support",
                       return_value=True)
    @mock.patch.object(host.Host, "get_capabilities")
    def test_does_want_hugepages(self, mock_caps, mock_numa):
        for arch in [fields.Architecture.I686,
                     fields.Architecture.X86_64,
                     fields.Architecture.AARCH64,
                     fields.Architecture.PPC64LE,
                     fields.Architecture.PPC64]:
            self._test_does_want_hugepages(mock_caps, mock_numa, arch)

    def _test_does_want_hugepages(self, mock_caps, mock_numa, architecture):
        self.flags(reserved_huge_pages=[
            {'node': 0, 'size': 2048, 'count': 128},
            {'node': 1, 'size': 2048, 'count': 1},
            {'node': 3, 'size': 2048, 'count': 64}])
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_topology = objects.InstanceNUMATopology(
            cells=[
                objects.InstanceNUMACell(
                    id=1, cpuset=set([0, 1]), pcpuset=set(),
                    memory=1024, pagesize=2048),
                objects.InstanceNUMACell(
                    id=2, cpuset=set([2, 3]), pcpuset=set(),
                    memory=1024, pagesize=2048),
            ])

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = architecture
        caps.host.topology = fakelibvirt.NUMATopology()
        for i, cell in enumerate(caps.host.topology.cells):
            cell.mempages = fakelibvirt.create_mempages(
                [(4, 1024 * i), (2048, i)])

        mock_caps.return_value = caps

        host_topology = drvr._get_host_numa_topology()
        self.assertEqual(128, host_topology.cells[0].mempages[1].reserved)
        self.assertEqual(1, host_topology.cells[1].mempages[1].reserved)
        self.assertEqual(0, host_topology.cells[2].mempages[1].reserved)
        self.assertEqual(64, host_topology.cells[3].mempages[1].reserved)

        self.assertTrue(drvr._wants_hugepages(host_topology,
                                              instance_topology))

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_clock(self):
        self.flags(virt_type='kvm', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        hpet_map = {
            fields.Architecture.X86_64: True,
            fields.Architecture.I686: True,
            fields.Architecture.PPC: False,
            fields.Architecture.PPC64: False,
            fields.Architecture.ARMV7: False,
            fields.Architecture.AARCH64: False,
            }

        for guestarch, expect_hpet in hpet_map.items():
            self.mock_uname.return_value = fakelibvirt.os_uname(
                'Linux', '', '5.4.0-0-generic', '', guestarch)

            cfg = drvr._get_guest_config(
                instance_ref, [], image_meta, disk_info)
            self.assertIsInstance(
                cfg.clock, vconfig.LibvirtConfigGuestClock)
            self.assertEqual(cfg.clock.offset, "utc")
            self.assertIsInstance(
                cfg.clock.timers[0], vconfig.LibvirtConfigGuestTimer)
            self.assertIsInstance(
                cfg.clock.timers[1], vconfig.LibvirtConfigGuestTimer)
            self.assertEqual(cfg.clock.timers[0].name, "pit")
            self.assertEqual(cfg.clock.timers[0].tickpolicy, "delay")
            self.assertEqual(cfg.clock.timers[1].name, "rtc")
            self.assertEqual(cfg.clock.timers[1].tickpolicy, "catchup")
            if expect_hpet:
                self.assertEqual(3, len(cfg.clock.timers))
                self.assertIsInstance(
                    cfg.clock.timers[2], vconfig.LibvirtConfigGuestTimer)
                self.assertEqual('hpet', cfg.clock.timers[2].name)
                self.assertFalse(cfg.clock.timers[2].present)
            else:
                self.assertEqual(2, len(cfg.clock.timers))

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_clock_hpet_false(self):
        self.flags(virt_type='kvm', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_time_hpet": "false"}})
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        hpet_map = {
            fields.Architecture.X86_64: True,
            fields.Architecture.I686: True,
            fields.Architecture.PPC: False,
            fields.Architecture.PPC64: False,
            fields.Architecture.ARMV7: False,
            fields.Architecture.AARCH64: False,
            }

        for guestarch, expect_hpet in hpet_map.items():
            self.mock_uname.return_value = fakelibvirt.os_uname(
                'Linux', '', '5.4.0-0-generic', '', guestarch)

            cfg = drvr._get_guest_config(
                instance_ref, [], image_meta, disk_info)
            self.assertIsInstance(
                cfg.clock, vconfig.LibvirtConfigGuestClock)
            self.assertEqual(cfg.clock.offset, "utc")
            self.assertIsInstance(
                cfg.clock.timers[0], vconfig.LibvirtConfigGuestTimer)
            self.assertIsInstance(
                cfg.clock.timers[1], vconfig.LibvirtConfigGuestTimer)
            self.assertEqual(cfg.clock.timers[0].name, "pit")
            self.assertEqual(cfg.clock.timers[0].tickpolicy, "delay")
            self.assertEqual(cfg.clock.timers[1].name, "rtc")
            self.assertEqual(cfg.clock.timers[1].tickpolicy, "catchup")
            if expect_hpet:
                self.assertEqual(3, len(cfg.clock.timers))
                self.assertIsInstance(
                    cfg.clock.timers[2], vconfig.LibvirtConfigGuestTimer)
                self.assertEqual('hpet', cfg.clock.timers[2].name)
                self.assertFalse(cfg.clock.timers[2].present)
            else:
                self.assertEqual(2, len(cfg.clock.timers))

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_clock_hpet_true(self):
        self.flags(virt_type='kvm', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "id": uuids.image_id,
            "disk_format": "raw",
            "properties": {"hw_time_hpet": "true"}})
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        hpet_map = {
            fields.Architecture.X86_64: True,
            fields.Architecture.I686: True,
            fields.Architecture.PPC: False,
            fields.Architecture.PPC64: False,
            fields.Architecture.ARMV7: False,
            fields.Architecture.AARCH64: False,
            }

        for guestarch, expect_hpet in hpet_map.items():
            self.mock_uname.return_value = fakelibvirt.os_uname(
                'Linux', '', '5.4.0-0-generic', '', guestarch)

            cfg = drvr._get_guest_config(
                instance_ref, [], image_meta, disk_info)
            self.assertIsInstance(
                cfg.clock, vconfig.LibvirtConfigGuestClock)
            self.assertEqual(cfg.clock.offset, "utc")
            self.assertIsInstance(
                cfg.clock.timers[0], vconfig.LibvirtConfigGuestTimer)
            self.assertIsInstance(
                cfg.clock.timers[1], vconfig.LibvirtConfigGuestTimer)
            self.assertEqual(cfg.clock.timers[0].name, "pit")
            self.assertEqual(cfg.clock.timers[0].tickpolicy, "delay")
            self.assertEqual(cfg.clock.timers[1].name, "rtc")
            self.assertEqual(cfg.clock.timers[1].tickpolicy, "catchup")
            if expect_hpet:
                self.assertEqual(3, len(cfg.clock.timers))
                self.assertIsInstance(
                    cfg.clock.timers[2], vconfig.LibvirtConfigGuestTimer)
                self.assertEqual('hpet', cfg.clock.timers[2].name)
                self.assertTrue(cfg.clock.timers[2].present)
            else:
                self.assertEqual(2, len(cfg.clock.timers))

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_clock_hpet_invalid(self):
        self.flags(virt_type='kvm', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_time_hpet": "blah"}})
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        hpet_map = {
            fields.Architecture.X86_64: True,
            fields.Architecture.I686: True,
            fields.Architecture.PPC: False,
            fields.Architecture.PPC64: False,
            fields.Architecture.ARMV7: False,
            fields.Architecture.AARCH64: False,
            }

        for guestarch, expect_hpet in hpet_map.items():
            self.mock_uname.return_value = fakelibvirt.os_uname(
                'Linux', '', '5.4.0-0-generic', '', guestarch)

            cfg = drvr._get_guest_config(
                instance_ref, [], image_meta, disk_info)
            self.assertIsInstance(
                cfg.clock, vconfig.LibvirtConfigGuestClock)
            self.assertEqual(cfg.clock.offset, "utc")
            self.assertIsInstance(
                cfg.clock.timers[0], vconfig.LibvirtConfigGuestTimer)
            self.assertIsInstance(
                cfg.clock.timers[1], vconfig.LibvirtConfigGuestTimer)
            self.assertEqual(cfg.clock.timers[0].name, "pit")
            self.assertEqual(cfg.clock.timers[0].tickpolicy, "delay")
            self.assertEqual(cfg.clock.timers[1].name, "rtc")
            self.assertEqual(cfg.clock.timers[1].tickpolicy, "catchup")
            if expect_hpet:
                self.assertEqual(3, len(cfg.clock.timers))
                self.assertIsInstance(
                    cfg.clock.timers[2], vconfig.LibvirtConfigGuestTimer)
                self.assertEqual('hpet', cfg.clock.timers[2].name)
                # a non-boolean value of hw_time_hpet should be treated as
                # False
                self.assertFalse(cfg.clock.timers[2].present)
            else:
                self.assertEqual(2, len(cfg.clock.timers))

    def test_get_guest_config_windows_timer(self):
        self.mock_uname.return_value = fakelibvirt.os_uname(
            'Linux', '', '5.4.0-0-generic', '', fields.Architecture.I686)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref['os_type'] = 'windows'
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref,
                                     _fake_network_info(self),
                                     image_meta, disk_info)

        self.assertIsInstance(cfg.clock,
                              vconfig.LibvirtConfigGuestClock)
        self.assertEqual(cfg.clock.offset, "localtime")

        self.assertEqual(4, len(cfg.clock.timers), cfg.clock.timers)
        self.assertEqual("pit", cfg.clock.timers[0].name)
        self.assertEqual("rtc", cfg.clock.timers[1].name)
        self.assertEqual("hpet", cfg.clock.timers[2].name)
        self.assertFalse(cfg.clock.timers[2].present)
        self.assertEqual("hypervclock", cfg.clock.timers[3].name)
        self.assertTrue(cfg.clock.timers[3].present)

        self.assertEqual(4, len(cfg.features))
        self.assertIsInstance(cfg.features[0],
                              vconfig.LibvirtConfigGuestFeatureACPI)
        self.assertIsInstance(cfg.features[1],
                              vconfig.LibvirtConfigGuestFeatureAPIC)
        self.assertIsInstance(cfg.features[2],
                              vconfig.LibvirtConfigGuestFeatureHyperV)
        self.assertIsInstance(
            cfg.features[3], vconfig.LibvirtConfigGuestFeatureVMCoreInfo)

    @mock.patch.object(host.Host, 'has_min_version',
                       new=mock.Mock(return_value=True))
    def _test_get_guest_config_windows_hyperv(
            self, flavor=None, image_meta=None, hvid_hidden=False):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref['os_type'] = 'windows'
        if flavor is not None:
            instance_ref.flavor = flavor
        if image_meta is None:
            image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref,
                                     _fake_network_info(self),
                                     image_meta, disk_info)

        self.assertIsInstance(cfg.clock,
                              vconfig.LibvirtConfigGuestClock)
        self.assertEqual(cfg.clock.offset, "localtime")

        num_features = 5 if hvid_hidden else 4
        self.assertEqual(num_features, len(cfg.features))
        self.assertIsInstance(cfg.features[0],
                              vconfig.LibvirtConfigGuestFeatureACPI)
        self.assertIsInstance(cfg.features[1],
                              vconfig.LibvirtConfigGuestFeatureAPIC)
        self.assertIsInstance(cfg.features[2],
                              vconfig.LibvirtConfigGuestFeatureHyperV)
        self.assertIsInstance(
            cfg.features[3], vconfig.LibvirtConfigGuestFeatureVMCoreInfo)
        if hvid_hidden:
            self.assertIsInstance(
                cfg.features[4], vconfig.LibvirtConfigGuestFeatureKvmHidden)

        self.assertTrue(cfg.features[2].relaxed)
        self.assertTrue(cfg.features[2].spinlocks)
        self.assertEqual(8191, cfg.features[2].spinlock_retries)
        self.assertTrue(cfg.features[2].vapic)
        self.assertEqual(hvid_hidden, cfg.features[2].vendorid_spoof)

    @mock.patch.object(host.Host, 'has_min_version',
                       new=mock.Mock(return_value=True))
    def test_get_guest_config_apic_workaround(self):
        self.flags(virt_type='qemu', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(
            CONF.libvirt.virt_type, instance_ref, image_meta)

        cfg = drvr._get_guest_config(
            instance_ref, _fake_network_info(self), image_meta, disk_info)

        self.assertEqual(3, len(cfg.features))
        self.assertIsInstance(
            cfg.features[0], vconfig.LibvirtConfigGuestFeatureACPI)
        self.assertIsInstance(
            cfg.features[1], vconfig.LibvirtConfigGuestFeatureAPIC)
        self.assertIsInstance(
            cfg.features[2], vconfig.LibvirtConfigGuestFeatureVMCoreInfo)

        self.flags(libvirt_disable_apic=True, group='workarounds')
        cfg = drvr._get_guest_config(
            instance_ref, _fake_network_info(self), image_meta, disk_info)

        self.assertEqual(2, len(cfg.features))
        self.assertIsInstance(
            cfg.features[0], vconfig.LibvirtConfigGuestFeatureACPI)
        self.assertIsInstance(
            cfg.features[1], vconfig.LibvirtConfigGuestFeatureVMCoreInfo)

    def test_get_guest_config_windows_hyperv_feature2(self):
        self._test_get_guest_config_windows_hyperv()

    def test_get_guest_config_windows_hyperv_all_hide_flv(self):
        # Similar to test_get_guest_config_windows_hyperv_feature2
        #   but also test hiding the HyperV signature with the flavor
        #   extra_spec "hw:hide_hypervisor_id"
        flavor_hide_id = fake_flavor.fake_flavor_obj(self.context,
            extra_specs={"hw:hide_hypervisor_id": "true"},
            expected_attrs={"extra_specs"})
        # this works for kvm (the default, tested below) and qemu
        self.flags(virt_type='qemu', group='libvirt')

        self._test_get_guest_config_windows_hyperv(
            flavor=flavor_hide_id, hvid_hidden=True)

    def test_get_guest_config_windows_hyperv_all_hide_flv_old(self):
        # Similar to test_get_guest_config_windows_hyperv_feature2
        #   but also test hiding the HyperV signature with the flavor
        #   extra_spec "hide_hypervisor_id"
        flavor_hide_id = fake_flavor.fake_flavor_obj(self.context,
            extra_specs={"hide_hypervisor_id": "true"},
            expected_attrs={"extra_specs"})
        # this works for kvm (the default, tested below) and qemu
        self.flags(virt_type='qemu', group='libvirt')

        self._test_get_guest_config_windows_hyperv(
            flavor=flavor_hide_id, hvid_hidden=True)

    def test_get_guest_config_windows_hyperv_all_hide_img(self):
        # Similar to test_get_guest_config_windows_hyperv_feature2
        #   but also test hiding the HyperV signature with the image
        #   property "img_hide_hypervisor_id"
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"img_hide_hypervisor_id": "true"}})

        self._test_get_guest_config_windows_hyperv(
            image_meta=image_meta, hvid_hidden=True)

    def test_get_guest_config_windows_hyperv_all_hide_flv_img(self):
        # Similar to test_get_guest_config_windows_hyperv_feature2
        #   but also test hiding the HyperV signature with both the flavor
        #   extra_spec "hw:hide_hypervisor_id" and the image property
        #   "img_hide_hypervisor_id"
        flavor_hide_id = fake_flavor.fake_flavor_obj(self.context,
            extra_specs={"hw:hide_hypervisor_id": "true"},
            expected_attrs={"extra_specs"})
        self.flags(virt_type='qemu', group='libvirt')

        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"img_hide_hypervisor_id": "true"}})

        self._test_get_guest_config_windows_hyperv(
            flavor=flavor_hide_id, image_meta=image_meta, hvid_hidden=True)

    def test_get_guest_config_with_two_nics(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref,
                                     _fake_network_info(self, 2),
                                     image_meta, disk_info)
        self.assertEqual(3, len(cfg.features))
        self.assertIsInstance(cfg.features[0],
                              vconfig.LibvirtConfigGuestFeatureACPI)
        self.assertIsInstance(cfg.features[1],
                              vconfig.LibvirtConfigGuestFeatureAPIC)
        self.assertIsInstance(
            cfg.features[2], vconfig.LibvirtConfigGuestFeatureVMCoreInfo)
        self.assertEqual(cfg.memory, instance_ref.flavor.memory_mb * units.Ki)
        self.assertEqual(cfg.vcpus, instance_ref.flavor.vcpus)
        self.assertEqual(cfg.os_type, fields.VMMode.HVM)
        self.assertEqual(cfg.os_boot_dev, ["hd"])
        self.assertIsNone(cfg.os_root)
        self.assertEqual(len(cfg.devices), 11)
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestInterface)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestInterface)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigGuestInput)
        self.assertIsInstance(cfg.devices[8],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[9],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[10],
                              vconfig.LibvirtConfigMemoryBalloon)

    def test_get_guest_config_with_uefi(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_firmware_type": "uefi"}})
        instance_ref = objects.Instance(**self.test_instance)

        disk_info = blockinfo.get_disk_info(
            CONF.libvirt.virt_type, instance_ref, image_meta)
        cfg = drvr._get_guest_config(
            instance_ref, [], image_meta, disk_info)
        # these paths are derived from the FakeLibvirtFixture
        self.assertEqual('/usr/share/OVMF/OVMF_CODE.fd', cfg.os_loader)
        self.assertEqual('/usr/share/OVMF/OVMF_VARS.fd', cfg.os_nvram_template)

    def test_get_guest_config_with_secure_boot_and_smm_required(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        # uefi only used with secure boot
        drvr._host._supports_uefi = True
        # smm only used with secure boot
        drvr._host._supports_secure_boot = True

        # NOTE(imranh2): Current way of gathering firmwares is inflexible
        # nova/tests/fixtures/libvirt.py FakeLoaders has requires-smm
        # defined. do the following to make sure we get this programtically
        # in the future we should test firmwares that both do and don't
        # require smm but the current way firmware is selected doesn't
        # make it possible to do so.
        loader, nvram_template, requires_smm = drvr._host.get_loader(
            'x86_64', 'q35', True)

        image_meta = objects.ImageMeta.from_dict({
            'disk_format': 'raw',
            # secure boot requires UEFI
            'properties': {
                'hw_firmware_type': 'uefi',
                'hw_machine_type': 'q35',
                'os_secure_boot': 'required',
            },
        })
        instance_ref = objects.Instance(**self.test_instance)

        disk_info = blockinfo.get_disk_info(
            CONF.libvirt.virt_type, instance_ref, image_meta)

        cfg = drvr._get_guest_config(
            instance_ref, [], image_meta, disk_info)
        # if we require it make sure it's there
        if requires_smm:
            self.assertTrue(any(isinstance(feature,
              vconfig.LibvirtConfigGuestFeatureSMM)
                for feature in cfg.features))

    @ddt.data(True, False)
    def test_get_guest_config_with_secure_boot_required(
        self, host_has_support,
    ):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._host._supports_uefi = True
        drvr._host._supports_secure_boot = host_has_support

        image_meta = objects.ImageMeta.from_dict({
            'disk_format': 'raw',
            # secure boot requires UEFI
            'properties': {
                'hw_firmware_type': 'uefi',
                'hw_machine_type': 'q35',
                'os_secure_boot': 'required',
            },
        })
        instance_ref = objects.Instance(**self.test_instance)

        disk_info = blockinfo.get_disk_info(
            CONF.libvirt.virt_type, instance_ref, image_meta)
        if host_has_support:
            # if the host supports it, we should get the feature
            cfg = drvr._get_guest_config(
                instance_ref, [], image_meta, disk_info)
            # these paths are derived from the FakeLibvirtFixture
            self.assertEqual(
                '/usr/share/OVMF/OVMF_CODE.secboot.fd', cfg.os_loader)
            self.assertEqual(
                '/usr/share/OVMF/OVMF_VARS.secboot.fd', cfg.os_nvram_template)
            self.assertTrue(cfg.os_loader_secure)
        else:
            # if not, we should see an exception
            self.assertRaises(
                exception.SecureBootNotSupported,
                drvr._get_guest_config,
                instance_ref, [], image_meta, disk_info)

    @ddt.data(True, False)
    def test_get_guest_config_with_secure_boot_optional(
        self, host_has_support,
    ):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._host._supports_uefi = True
        drvr._host._supports_secure_boot = host_has_support

        image_meta = objects.ImageMeta.from_dict({
            'disk_format': 'raw',
            # secure boot requires UEFI
            'properties': {
                'hw_firmware_type': 'uefi',
                'hw_machine_type': 'q35',
                'os_secure_boot': 'optional',
            },
        })
        instance_ref = objects.Instance(**self.test_instance)

        disk_info = blockinfo.get_disk_info(
            CONF.libvirt.virt_type, instance_ref, image_meta)
        cfg = drvr._get_guest_config(
            instance_ref, [], image_meta, disk_info)
        if host_has_support:
            # if the host supports it we should get the feature
            self.assertEqual(
                '/usr/share/OVMF/OVMF_CODE.secboot.fd', cfg.os_loader)
            self.assertEqual(
                '/usr/share/OVMF/OVMF_VARS.secboot.fd', cfg.os_nvram_template)
            self.assertTrue(cfg.os_loader_secure)
        else:
            # if not, silently ignore
            self.assertEqual(
                '/usr/share/OVMF/OVMF_CODE.fd', cfg.os_loader)
            self.assertEqual(
                '/usr/share/OVMF/OVMF_VARS.fd', cfg.os_nvram_template)
            self.assertFalse(cfg.os_loader_secure)

    def test_check_uefi_support_aarch64(self):
        self.mock_uname.return_value = fakelibvirt.os_uname(
            'Linux', '', '5.4.0-0-generic', '', fields.Architecture.AARCH64)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        self.assertTrue(drvr._check_uefi_support(image_meta))

    def test_get_guest_config_with_block_device(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        conn_info = {'driver_volume_type': 'fake', 'data': {}}
        bdms = block_device_obj.block_device_make_list_from_dicts(
            self.context, [
                fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 1,
                     'source_type': 'volume', 'destination_type': 'volume',
                     'device_name': '/dev/vdc'}),
                fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 2,
                     'source_type': 'volume', 'destination_type': 'volume',
                     'device_name': '/dev/vdd'}),
            ]
        )
        info = {'block_device_mapping': driver_block_device.convert_volumes(
            bdms
        )}
        info['block_device_mapping'][0]['connection_info'] = conn_info
        info['block_device_mapping'][1]['connection_info'] = conn_info

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta,
                                            info)
        with mock.patch.object(
                driver_block_device.DriverVolumeBlockDevice, 'save'
        ) as mock_save:
            cfg = drvr._get_guest_config(instance_ref, [],
                                         image_meta, disk_info,
                                         None, info)
            self.assertIsInstance(cfg.devices[2],
                                  vconfig.LibvirtConfigGuestDisk)
            self.assertEqual(cfg.devices[2].target_dev, 'vdc')
            self.assertIsInstance(cfg.devices[3],
                                  vconfig.LibvirtConfigGuestDisk)
            self.assertEqual(cfg.devices[3].target_dev, 'vdd')
            mock_save.assert_called_with()

    def test_get_guest_config_lxc_with_attached_volume(self):
        self.flags(virt_type='lxc', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        conn_info = {'driver_volume_type': 'fake', 'data': {}}
        bdms = block_device_obj.block_device_make_list_from_dicts(
            self.context, [
              fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 1,
                     'source_type': 'volume', 'destination_type': 'volume',
                     'boot_index': 0}),
              fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 2,
                     'source_type': 'volume', 'destination_type': 'volume',
                    }),
              fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 3,
                     'source_type': 'volume', 'destination_type': 'volume',
                    }),
           ]
        )
        info = {'block_device_mapping': driver_block_device.convert_volumes(
            bdms
        )}

        info['block_device_mapping'][0]['connection_info'] = conn_info
        info['block_device_mapping'][1]['connection_info'] = conn_info
        info['block_device_mapping'][2]['connection_info'] = conn_info
        info['block_device_mapping'][0]['mount_device'] = '/dev/vda'
        info['block_device_mapping'][1]['mount_device'] = '/dev/vdc'
        info['block_device_mapping'][2]['mount_device'] = '/dev/vdd'
        with mock.patch.object(
                driver_block_device.DriverVolumeBlockDevice, 'save'
        ) as mock_save:
            disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                                instance_ref,
                                                image_meta,
                                                info)
            cfg = drvr._get_guest_config(instance_ref, [],
                                         image_meta, disk_info,
                                         None, info)
            self.assertIsInstance(cfg.devices[1],
                                  vconfig.LibvirtConfigGuestDisk)
            self.assertEqual(cfg.devices[1].target_dev, 'vdc')
            self.assertIsInstance(cfg.devices[2],
                                  vconfig.LibvirtConfigGuestDisk)
            self.assertEqual(cfg.devices[2].target_dev, 'vdd')
            mock_save.assert_called_with()

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_with_configdrive(self):
        # It's necessary to check if the architecture is power, because
        # power doesn't have support to ide, and so libvirt translate
        # all ide calls to scsi

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        # make configdrive.required_by() return True
        instance_ref['config_drive'] = True

        # Pick the first drive letter on the bus that is available
        # as the config drive. Delete the last device hardcode as
        # the config drive here.

        disk_map = {
            fields.Architecture.X86_64: 'hda',
            fields.Architecture.I686: 'hda',
            fields.Architecture.PPC: 'sda',
            fields.Architecture.PPC64: 'sda',
            fields.Architecture.AARCH64: 'sda',
        }
        for guestarch, disk in disk_map.items():
            self.mock_uname.return_value = fakelibvirt.os_uname(
                'Linux', '', '5.4.0-0-generic', '', guestarch)

            disk_info = blockinfo.get_disk_info(
                CONF.libvirt.virt_type, instance_ref, image_meta)
            cfg = drvr._get_guest_config(
                instance_ref, [], image_meta, disk_info)

            self.assertIsInstance(
                cfg.devices[2], vconfig.LibvirtConfigGuestDisk)
            self.assertEqual(cfg.devices[2].target_dev, disk)

    def test_get_guest_config_default_with_virtio_scsi_bus(self):
        self._test_get_guest_config_with_virtio_scsi_bus()

    @mock.patch.object(rbd_utils.RBDDriver, '_check_for_import_failure',
                       new=mock.Mock())
    @mock.patch.object(rbd_utils.RBDDriver, 'get_mon_addrs')
    @mock.patch.object(rbd_utils, 'rbd', new=mock.Mock())
    def test_get_guest_config_rbd_with_virtio_scsi_bus(
            self, mock_get_mon_addrs):
        self.flags(images_type='rbd', group='libvirt')
        mock_get_mon_addrs.return_value = ("host", 9876)
        self._test_get_guest_config_with_virtio_scsi_bus()

    def _test_get_guest_config_with_virtio_scsi_bus(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_scsi_model": "virtio-scsi",
                           "hw_disk_bus": "scsi"}})
        instance_ref = objects.Instance(**self.test_instance)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta,
                                            [])
        cfg = drvr._get_guest_config(instance_ref, [], image_meta, disk_info)
        self.assertIsInstance(cfg.devices[0],
                         vconfig.LibvirtConfigGuestDisk)
        self.assertEqual(0, cfg.devices[0].device_addr.unit)
        self.assertIsInstance(cfg.devices[1],
                         vconfig.LibvirtConfigGuestDisk)
        self.assertEqual(1, cfg.devices[1].device_addr.unit)
        self.assertIsInstance(cfg.devices[2],
                         vconfig.LibvirtConfigGuestController)
        self.assertEqual(cfg.devices[2].model, 'virtio-scsi')

    def test_get_guest_config_with_virtio_scsi_bus_bdm(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_scsi_model": "virtio-scsi",
                           "hw_disk_bus": "scsi"}})
        instance_ref = objects.Instance(**self.test_instance)
        conn_info = {'driver_volume_type': 'fake', 'data': {}}
        bdms = block_device_obj.block_device_make_list_from_dicts(
            self.context, [
                fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 1,
                     'source_type': 'volume', 'destination_type': 'volume',
                     'device_name': '/dev/sdc', 'disk_bus': 'scsi'}),
                fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 2,
                     'source_type': 'volume', 'destination_type': 'volume',
                     'device_name': '/dev/sdd', 'disk_bus': 'scsi'}),
                ]
        )
        bd_info = {
            'block_device_mapping': driver_block_device.convert_volumes(bdms)}
        bd_info['block_device_mapping'][0]['connection_info'] = conn_info
        bd_info['block_device_mapping'][1]['connection_info'] = conn_info

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta,
                                            bd_info)
        with mock.patch.object(
                driver_block_device.DriverVolumeBlockDevice, 'save'
        ) as mock_save:
            cfg = drvr._get_guest_config(instance_ref, [], image_meta,
                    disk_info, [], bd_info)
            self.assertIsInstance(cfg.devices[2],
                             vconfig.LibvirtConfigGuestDisk)
            self.assertEqual(cfg.devices[2].target_dev, 'sdc')
            self.assertEqual(cfg.devices[2].target_bus, 'scsi')
            self.assertEqual(2, cfg.devices[2].device_addr.unit)
            self.assertIsInstance(cfg.devices[3],
                             vconfig.LibvirtConfigGuestDisk)
            self.assertEqual(cfg.devices[3].target_dev, 'sdd')
            self.assertEqual(cfg.devices[3].target_bus, 'scsi')
            self.assertEqual(3, cfg.devices[3].device_addr.unit)
            self.assertIsInstance(cfg.devices[4],
                             vconfig.LibvirtConfigGuestController)
            self.assertEqual(cfg.devices[4].model, 'virtio-scsi')
            mock_save.assert_called_with()

    def test_get_guest_config_one_scsi_volume_with_configdrive(self):
        """Tests that the unit attribute is only incremented for block devices
        that have a scsi bus. Unit numbering should begin at 0 since we are not
        booting from volume.
        """
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_scsi_model": "virtio-scsi",
                           "hw_disk_bus": "scsi",
                           "hw_architecture": fields.Architecture.X86_64}})
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.config_drive = 'True'
        conn_info = {'driver_volume_type': 'fake'}
        bdms = block_device_obj.block_device_make_list_from_dicts(
            self.context, [
                fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 1,
                     'source_type': 'volume', 'destination_type': 'volume',
                     'device_name': '/dev/sdc', 'disk_bus': 'scsi'}),
                ]
        )
        bd_info = {
            'block_device_mapping': driver_block_device.convert_volumes(bdms)}
        bd_info['block_device_mapping'][0]['connection_info'] = conn_info

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta,
                                            bd_info)
        with mock.patch.object(
                driver_block_device.DriverVolumeBlockDevice, 'save'):
            cfg = drvr._get_guest_config(instance_ref, [], image_meta,
                    disk_info, [], bd_info)

            # The device order is determined by the order that devices are
            # appended in _get_guest_storage_config in the driver.

            # The first device will be the instance's local disk (since we're
            # not booting from volume). It should begin unit numbering at 0.
            self.assertIsInstance(cfg.devices[0],
                                  vconfig.LibvirtConfigGuestDisk)
            self.assertIn('disk', cfg.devices[0].source_path)
            self.assertEqual('sda', cfg.devices[0].target_dev)
            self.assertEqual('scsi', cfg.devices[0].target_bus)
            self.assertEqual(0, cfg.devices[0].device_addr.unit)

            # The second device will be the ephemeral disk
            # (the flavor in self.test_instance has ephemeral_gb > 0).
            # It should have the next unit number of 1.
            self.assertIsInstance(cfg.devices[1],
                                  vconfig.LibvirtConfigGuestDisk)
            self.assertIn('disk.local', cfg.devices[1].source_path)
            self.assertEqual('sdb', cfg.devices[1].target_dev)
            self.assertEqual('scsi', cfg.devices[1].target_bus)
            self.assertEqual(1, cfg.devices[1].device_addr.unit)

            # This is the config drive. It should not have unit number set.
            self.assertIsInstance(cfg.devices[2],
                                  vconfig.LibvirtConfigGuestDisk)
            self.assertIn('disk.config', cfg.devices[2].source_path)
            self.assertEqual('hda', cfg.devices[2].target_dev)
            self.assertEqual('ide', cfg.devices[2].target_bus)
            self.assertIsNone(cfg.devices[2].device_addr)

            # And this is the attached volume.
            self.assertIsInstance(cfg.devices[3],
                                  vconfig.LibvirtConfigGuestDisk)
            self.assertEqual('sdc', cfg.devices[3].target_dev)
            self.assertEqual('scsi', cfg.devices[3].target_bus)
            self.assertEqual(2, cfg.devices[3].device_addr.unit)

    def test_get_guest_config_boot_from_volume_with_configdrive(self):
        """Tests that the unit attribute is only incremented for block devices
        that have a scsi bus and that the bootable volume in a boot-from-volume
        scenario always has the unit set to 0.
        """
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_scsi_model": "virtio-scsi",
                           "hw_disk_bus": "scsi",
                           "hw_architecture": fields.Architecture.X86_64}})
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.config_drive = 'True'
        conn_info = {'driver_volume_type': 'fake'}
        bdms = block_device_obj.block_device_make_list_from_dicts(
            self.context, [
                # This is the boot volume (boot_index = 0).
                fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 1,
                     'source_type': 'volume', 'destination_type': 'volume',
                     'device_name': '/dev/sda', 'boot_index': 0}),
                # This is just another attached volume.
                fake_block_device.FakeDbBlockDeviceDict(
                    {'id': 2,
                     'source_type': 'volume', 'destination_type': 'volume',
                     'device_name': '/dev/sdc', 'disk_bus': 'scsi'}),
                ]
        )
        bd_info = {
            'block_device_mapping': driver_block_device.convert_volumes(bdms)}
        bd_info['block_device_mapping'][0]['connection_info'] = conn_info
        bd_info['block_device_mapping'][1]['connection_info'] = conn_info

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta,
                                            bd_info)
        with mock.patch.object(
                driver_block_device.DriverVolumeBlockDevice, 'save'):
            cfg = drvr._get_guest_config(instance_ref, [], image_meta,
                    disk_info, [], bd_info)

            # The device order is determined by the order that devices are
            # appended in _get_guest_storage_config in the driver.

            # The first device will be the ephemeral disk
            # (the flavor in self.test_instance has ephemeral_gb > 0).
            # It should begin unit numbering at 1 because 0 is reserved for the
            # boot volume for boot-from-volume.
            self.assertIsInstance(cfg.devices[0],
                                  vconfig.LibvirtConfigGuestDisk)
            self.assertIn('disk.local', cfg.devices[0].source_path)
            self.assertEqual('sdb', cfg.devices[0].target_dev)
            self.assertEqual('scsi', cfg.devices[0].target_bus)
            self.assertEqual(1, cfg.devices[0].device_addr.unit)

            # The second device will be the config drive. It should not have a
            # unit number set.
            self.assertIsInstance(cfg.devices[1],
                                  vconfig.LibvirtConfigGuestDisk)
            self.assertIn('disk.config', cfg.devices[1].source_path)
            self.assertEqual('hda', cfg.devices[1].target_dev)
            self.assertEqual('ide', cfg.devices[1].target_bus)
            self.assertIsNone(cfg.devices[1].device_addr)

            # The third device will be the boot volume. It should have a
            # unit number of 0.
            self.assertIsInstance(cfg.devices[2],
                                  vconfig.LibvirtConfigGuestDisk)
            self.assertEqual('sda', cfg.devices[2].target_dev)
            self.assertEqual('scsi', cfg.devices[2].target_bus)
            self.assertEqual(0, cfg.devices[2].device_addr.unit)

            # The fourth device will be the other attached volume.
            self.assertIsInstance(cfg.devices[3],
                                  vconfig.LibvirtConfigGuestDisk)
            self.assertEqual('sdc', cfg.devices[3].target_dev)
            self.assertEqual('scsi', cfg.devices[3].target_bus)
            self.assertEqual(2, cfg.devices[3].device_addr.unit)

    def _get_guest_config_with_graphics(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        return cfg

    def test_get_guest_config_with_vnc(self):
        self.flags(enabled=True, server_listen='10.0.0.1', group='vnc')
        self.flags(virt_type='kvm', group='libvirt')
        self.flags(pointer_model='ps2mouse')
        self.flags(enabled=False, group='spice')

        cfg = self._get_guest_config_with_graphics()

        self.assertEqual(len(cfg.devices), 8)
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual(cfg.devices[3].type, 'vnc')
        self.assertEqual(cfg.devices[3].listen, '10.0.0.1')
        self.assertIsNone(cfg.devices[3].keymap)
        self.assertIsNone(cfg.devices[3].image_compression)
        self.assertIsNone(cfg.devices[3].jpeg_compression)
        self.assertIsNone(cfg.devices[3].zlib_compression)
        self.assertIsNone(cfg.devices[3].playback_compression)
        self.assertIsNone(cfg.devices[3].streaming_mode)

    def test_get_guest_config_with_vnc_and_tablet(self):
        self.flags(enabled=True, group='vnc')
        self.flags(virt_type='kvm', group='libvirt')
        self.flags(enabled=False, group='spice')
        self.flags(pointer_model='usbtablet')

        cfg = self._get_guest_config_with_graphics()

        self.assertEqual(len(cfg.devices), 9)
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestInput)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[8],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual(cfg.devices[3].type, 'vnc')
        self.assertIsNone(cfg.devices[3].image_compression)
        self.assertIsNone(cfg.devices[3].jpeg_compression)
        self.assertIsNone(cfg.devices[3].zlib_compression)
        self.assertIsNone(cfg.devices[3].playback_compression)
        self.assertIsNone(cfg.devices[3].streaming_mode)
        self.assertEqual(cfg.devices[5].type, 'tablet')

    def test_get_guest_config_with_spice_and_tablet(self):
        self.flags(enabled=False, group='vnc')
        self.flags(virt_type='kvm', group='libvirt')
        self.flags(enabled=True,
                   agent_enabled=False,
                   server_listen='10.0.0.1',
                   group='spice')
        self.flags(pointer_model='usbtablet')

        cfg = self._get_guest_config_with_graphics()

        self.assertEqual(len(cfg.devices), 9)
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestInput)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[8],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual(cfg.devices[3].type, 'spice')
        self.assertEqual(cfg.devices[3].listen, '10.0.0.1')
        self.assertIsNone(cfg.devices[3].keymap)
        self.assertIsNone(cfg.devices[3].image_compression)
        self.assertIsNone(cfg.devices[3].jpeg_compression)
        self.assertIsNone(cfg.devices[3].zlib_compression)
        self.assertIsNone(cfg.devices[3].playback_compression)
        self.assertIsNone(cfg.devices[3].streaming_mode)
        self.assertEqual(cfg.devices[5].type, 'tablet')

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_with_spice_and_agent(self):
        self.flags(enabled=False, group='vnc')
        self.flags(virt_type='kvm', group='libvirt')
        self.flags(enabled=True, agent_enabled=True, group='spice')
        self.flags(pointer_model='usbtablet')

        video_map = {
            fields.Architecture.X86_64: 'qxl',
            fields.Architecture.I686: 'qxl',
            fields.Architecture.PPC: 'vga',
            fields.Architecture.PPC64: 'vga',
            fields.Architecture.PPC64LE: 'vga',
            fields.Architecture.AARCH64: 'virtio',
        }
        for guestarch, video_type in video_map.items():
            self.mock_uname.return_value = fakelibvirt.os_uname(
                'Linux', '', '5.4.0-0-generic', '', guestarch)

            cfg = self._get_guest_config_with_graphics()

            if guestarch != fields.Architecture.AARCH64:
                self.assertEqual(len(cfg.devices), 9)
            else:
                # AArch64 adds a keyboard by default
                self.assertEqual(len(cfg.devices), 10)

            devices = iter(cfg.devices)

            self.assertIsInstance(
                next(devices), vconfig.LibvirtConfigGuestDisk)
            self.assertIsInstance(
                next(devices), vconfig.LibvirtConfigGuestDisk)
            self.assertIsInstance(
                next(devices), vconfig.LibvirtConfigGuestSerial)
            self.assertIsInstance(
                next(devices), vconfig.LibvirtConfigGuestChannel)
            self.assertIsInstance(
                next(devices), vconfig.LibvirtConfigGuestGraphics)
            self.assertIsInstance(
                next(devices), vconfig.LibvirtConfigGuestVideo)

            if guestarch == fields.Architecture.AARCH64:
                # AArch64 adds a keyboard by default
                self.assertIsInstance(
                    next(devices), vconfig.LibvirtConfigGuestInput)

            self.assertIsInstance(
                next(devices), vconfig.LibvirtConfigGuestRng)
            self.assertIsInstance(
                next(devices), vconfig.LibvirtConfigGuestUSBHostController)
            self.assertIsInstance(
                next(devices), vconfig.LibvirtConfigMemoryBalloon)

            self.assertEqual(cfg.devices[3].target_name, "com.redhat.spice.0")
            self.assertEqual(cfg.devices[3].type, 'spicevmc')
            self.assertEqual(cfg.devices[4].type, "spice")
            self.assertIsNone(cfg.devices[4].image_compression)
            self.assertIsNone(cfg.devices[4].jpeg_compression)
            self.assertIsNone(cfg.devices[4].zlib_compression)
            self.assertIsNone(cfg.devices[4].playback_compression)
            self.assertIsNone(cfg.devices[4].streaming_mode)
            self.assertEqual(cfg.devices[5].type, video_type)

    def test_get_guest_config_with_spice_compression(self):
        self.flags(enabled=False, group='vnc')
        self.flags(virt_type='kvm', group='libvirt')
        self.flags(enabled=True,
                   agent_enabled=False,
                   image_compression='auto_lz',
                   jpeg_compression='never',
                   zlib_compression='always',
                   playback_compression=False,
                   streaming_mode='all',
                   server_listen='10.0.0.1',
                   group='spice')
        self.flags(pointer_model='usbtablet')

        cfg = self._get_guest_config_with_graphics()

        self.assertEqual(len(cfg.devices), 9)
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestInput)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[8],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual(cfg.devices[3].type, 'spice')
        self.assertEqual(cfg.devices[3].listen, '10.0.0.1')
        self.assertEqual(cfg.devices[3].image_compression, 'auto_lz')
        self.assertEqual(cfg.devices[3].jpeg_compression, 'never')
        self.assertEqual(cfg.devices[3].zlib_compression, 'always')
        self.assertFalse(cfg.devices[3].playback_compression)
        self.assertEqual(cfg.devices[3].streaming_mode, 'all')

    @mock.patch.object(host.Host, 'get_guest')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_get_serial_ports_from_guest')
    @mock.patch('nova.console.serial.acquire_port')
    @mock.patch('nova.virt.hardware.get_number_of_serial_ports',
                return_value=1)
    def test_create_serial_console_devices_based_on_arch(
        self, mock_get_port_number, mock_acquire_port, mock_ports, mock_guest,
    ):
        self.flags(enabled=True, group='serial_console')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        expected = {
          fields.Architecture.X86_64: vconfig.LibvirtConfigGuestSerial,
          fields.Architecture.S390: vconfig.LibvirtConfigGuestConsole,
          fields.Architecture.S390X: vconfig.LibvirtConfigGuestConsole}

        for guest_arch, device_type in expected.items():
            self.mock_uname.return_value = fakelibvirt.os_uname(
                'Linux', '', '5.4.0-0-generic', '', guest_arch)

            guest = vconfig.LibvirtConfigGuest()
            drvr._create_consoles(
                guest_cfg=guest,
                instance=instance,
                flavor={},
                image_meta=image_meta)
            self.assertEqual(1, len(guest.devices))
            console_device = guest.devices[0]
            self.assertIsInstance(console_device, device_type)
            self.assertEqual("tcp", console_device.type)

    @mock.patch.object(host.Host, 'get_guest')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_get_serial_ports_from_guest')
    @mock.patch('nova.virt.hardware.get_number_of_serial_ports',
                return_value=4)
    def test_create_serial_console_devices_with_limit_exceeded_based_on_arch(
        self, mock_get_port_number, mock_ports, mock_guest,
    ):
        self.flags(enabled=True, group='serial_console')
        self.flags(virt_type="qemu", group='libvirt')
        flavor = 'fake_flavor'
        image_meta = objects.ImageMeta.from_dict({})
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        guest = vconfig.LibvirtConfigGuest()
        instance = objects.Instance(**self.test_instance)

        self.mock_uname.return_value = fakelibvirt.os_uname(
            'Linux', '', '5.4.0-0-generic', '', fields.Architecture.X86_64)
        self.assertRaises(
            exception.SerialPortNumberLimitExceeded,
            drvr._create_consoles,
            guest, instance, flavor, image_meta)
        self.mock_uname.assert_called_once()
        mock_get_port_number.assert_called_with(flavor, image_meta)

        self.mock_uname.reset_mock()
        mock_get_port_number.reset_mock()

        self.mock_uname.return_value = fakelibvirt.os_uname(
            'Linux', '', '5.4.0-0-generic', '', fields.Architecture.S390)
        drvr._create_consoles(guest, instance, flavor, image_meta)
        self.mock_uname.assert_called_once()
        mock_get_port_number.assert_called_with(flavor, image_meta)

        self.mock_uname.reset_mock()
        mock_get_port_number.reset_mock()

        self.mock_uname.return_value = fakelibvirt.os_uname(
            'Linux', '', '5.4.0-0-generic', '', fields.Architecture.S390X)
        drvr._create_consoles(guest, instance, flavor, image_meta)
        self.mock_uname.assert_called_once()
        mock_get_port_number.assert_called_with(flavor, image_meta)

    @mock.patch('nova.console.serial.acquire_port')
    def test_get_guest_config_serial_console(self, acquire_port):
        self.flags(enabled=True, group='serial_console')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        acquire_port.return_value = 11111

        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        self.assertEqual(9, len(cfg.devices))
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestInput)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[8],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual("tcp", cfg.devices[2].type)
        self.assertEqual(11111, cfg.devices[2].listen_port)

    def test_get_guest_config_serial_console_through_flavor(self):
        self.flags(enabled=True, group='serial_console')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor.extra_specs = {'hw:serial_port_count': 3}
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        self.assertEqual(11, len(cfg.devices))
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigGuestInput)
        self.assertIsInstance(cfg.devices[8],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[9],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[10],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual("tcp", cfg.devices[2].type)
        self.assertEqual("tcp", cfg.devices[3].type)
        self.assertEqual("tcp", cfg.devices[4].type)

    def test_get_guest_config_serial_console_invalid_flavor(self):
        self.flags(enabled=True, group='serial_console')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor.extra_specs = {'hw:serial_port_count': "a"}
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        self.assertRaises(
            exception.ImageSerialPortNumberInvalid,
            drvr._get_guest_config, instance_ref, [],
            image_meta, disk_info)

    def test_get_guest_config_serial_console_image_and_flavor(self):
        self.flags(enabled=True, group='serial_console')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_serial_port_count": "3"}})
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor.extra_specs = {'hw:serial_port_count': 4}

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref, [], image_meta,
                                     disk_info)
        self.assertEqual(11, len(cfg.devices), cfg.devices)
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigGuestInput)
        self.assertIsInstance(cfg.devices[8],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[9],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[10],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual("tcp", cfg.devices[2].type)
        self.assertEqual("tcp", cfg.devices[3].type)
        self.assertEqual("tcp", cfg.devices[4].type)

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    @mock.patch('nova.console.serial.acquire_port')
    @mock.patch('nova.virt.hardware.get_number_of_serial_ports',
                return_value=1)
    def test_guest_config_char_device_logd(
        self, mock_get_number_serial_ports, mock_acquire_port,
        mock_host_has_min_version,
    ):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        def _test_consoles(arch_to_mock, serial_enabled,
                           expected_device_type, expected_device_cls,
                           virt_type='qemu'):
            self.mock_uname.return_value = fakelibvirt.os_uname(
                'Linux', '', '5.4.0-0-generic', '', arch_to_mock)
            self.flags(virt_type=virt_type, group='libvirt')
            self.flags(enabled=serial_enabled, group='serial_console')
            guest_cfg = vconfig.LibvirtConfigGuest()
            instance = objects.Instance(**self.test_instance)
            image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

            drvr._create_consoles(
                guest_cfg,
                instance=instance,
                flavor=None,
                image_meta=image_meta)

            self.assertEqual(1, len(guest_cfg.devices))
            device = guest_cfg.devices[0]
            self.assertEqual(expected_device_type, device.type)
            self.assertIsInstance(device, expected_device_cls)
            self.assertIsInstance(device.log,
                                  vconfig.LibvirtConfigGuestCharDeviceLog)
            self.assertEqual("off", device.log.append)
            self.assertIsNotNone(device.log.file)
            self.assertTrue(device.log.file.endswith("console.log"))

        _test_consoles(fields.Architecture.X86_64, False,
                       "pty", vconfig.LibvirtConfigGuestSerial)
        _test_consoles(fields.Architecture.S390, True,
                       "tcp", vconfig.LibvirtConfigGuestConsole)

    @mock.patch('nova.console.serial.acquire_port')
    def test_get_guest_config_serial_console_through_port_rng_exhausted(
            self, acquire_port):
        self.flags(enabled=True, group='serial_console')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        acquire_port.side_effect = exception.SocketPortRangeExhaustedException(
            '127.0.0.1')
        self.assertRaises(
            exception.SocketPortRangeExhaustedException,
            drvr._get_guest_config, instance_ref, [],
            image_meta, disk_info)

    @mock.patch('nova.console.serial.release_port')
    @mock.patch.object(libvirt_driver.LibvirtDriver, 'get_info')
    @mock.patch.object(host.Host, 'get_guest')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_get_serial_ports_from_guest')
    def test_serial_console_release_port(
            self, mock_get_serial_ports_from_guest, mock_get_guest,
            mock_get_info, mock_release_port):
        self.flags(enabled="True", group='serial_console')

        guest = libvirt_guest.Guest(FakeVirtDomain())
        guest.power_off = mock.Mock()
        mock_get_info.return_value = hardware.InstanceInfo(
            state=power_state.SHUTDOWN)
        mock_get_guest.return_value = guest
        mock_get_serial_ports_from_guest.return_value = iter([
            ('127.0.0.1', 10000), ('127.0.0.1', 10001)])

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._destroy(objects.Instance(**self.test_instance))
        mock_release_port.assert_has_calls(
            [mock.call(host='127.0.0.1', port=10000),
             mock.call(host='127.0.0.1', port=10001)])

    @mock.patch.object(libvirt_guest.Guest, "get_xml_desc")
    def test_get_serial_ports_from_guest(self, mock_get_xml_desc):
        i = self._test_get_serial_ports_from_guest(None,
                                                   mock_get_xml_desc)
        self.assertEqual([
            ('127.0.0.1', 100),
            ('127.0.0.1', 101),
            ('127.0.0.2', 100),
            ('127.0.0.2', 101)], list(i))

    @mock.patch.object(libvirt_guest.Guest, "get_xml_desc")
    def test_get_serial_ports_from_guest_bind_only(self, mock_get_xml_desc):
        i = self._test_get_serial_ports_from_guest('bind',
                                                   mock_get_xml_desc)
        self.assertEqual([
            ('127.0.0.1', 101),
            ('127.0.0.2', 100)], list(i))

    @mock.patch.object(libvirt_guest.Guest, "get_xml_desc")
    def test_get_serial_ports_from_guest_connect_only(self,
                                                      mock_get_xml_desc):
        i = self._test_get_serial_ports_from_guest('connect',
                                                   mock_get_xml_desc)
        self.assertEqual([
            ('127.0.0.1', 100),
            ('127.0.0.2', 101)], list(i))

    @mock.patch.object(libvirt_guest.Guest, "get_xml_desc")
    def test_get_serial_ports_from_guest_on_s390(self, mock_get_xml_desc):
        i = self._test_get_serial_ports_from_guest(None,
                                                   mock_get_xml_desc,
                                                   'console')
        self.assertEqual([
            ('127.0.0.1', 100),
            ('127.0.0.1', 101),
            ('127.0.0.2', 100),
            ('127.0.0.2', 101)], list(i))

    def _test_get_serial_ports_from_guest(self, mode, mock_get_xml_desc,
                                          dev_name='serial'):
        xml = """
        <domain type='kvm'>
          <devices>
            <%(dev_name)s type="tcp">
              <source host="127.0.0.1" service="100" mode="connect"/>
            </%(dev_name)s>
            <%(dev_name)s type="tcp">
              <source host="127.0.0.1" service="101" mode="bind"/>
            </%(dev_name)s>
            <%(dev_name)s type="tcp">
              <source host="127.0.0.2" service="100" mode="bind"/>
            </%(dev_name)s>
            <%(dev_name)s type="tcp">
              <source host="127.0.0.2" service="101" mode="connect"/>
            </%(dev_name)s>
          </devices>
        </domain>""" % {'dev_name': dev_name}

        mock_get_xml_desc.return_value = xml

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        guest = libvirt_guest.Guest(FakeVirtDomain())
        return drvr._get_serial_ports_from_guest(guest, mode=mode)

    def test_get_scsi_controller_next_unit_from_guest(self):
        xml = """
    <domain type='kvm'>
      <devices>
        <disk type='file' device='disk'>
          <target dev='sda' bus='scsi'/>
          <address type='drive' controller='0' bus='0' target='0' unit='0'/>
        </disk>
        <disk type='file' device='disk'>
          <target dev='vda' bus='virtio'/>
        </disk>
        <disk type='file' device='cdrom'>
          <target dev='sdc' bus='scsi'/>
          <address type='drive' controller='0' bus='0' target='0' unit='1'/>
        </disk>
      </devices>
    </domain>
            """
        self._test_get_scsi_controller_next_unit_from_guest(xml, 2)

    def test_get_scsi_controller_next_unit_from_guest_no_scsi(self):
        xml = """
    <domain type='kvm'>
      <devices>
        <disk type='file' device='disk'>
          <target dev='vda' bus='virtio'/>
        </disk>
        <disk type='file' device='disk'>
          <target dev='vdb' bus='virtio'/>
        </disk>
      </devices>
    </domain>
            """
        self._test_get_scsi_controller_next_unit_from_guest(xml, 0)

    @mock.patch.object(libvirt_guest.Guest, "get_xml_desc")
    def _test_get_scsi_controller_next_unit_from_guest(self, xml,
                                                       expect_num,
                                                       mock_get_xml_desc):
        mock_get_xml_desc.return_value = xml

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        guest = libvirt_guest.Guest(FakeVirtDomain())
        i = drvr._get_scsi_controller_next_unit(guest)
        self.assertEqual(expect_num, i)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_with_type_kvm_on_s390(self):
        self.flags(enabled=False, group='vnc')
        self.flags(virt_type='kvm', group='libvirt')
        self.mock_uname.return_value = fakelibvirt.os_uname(
            'Linux', '', '5.4.0-0-generic', '', fields.Architecture.S390X)

        instance_ref = objects.Instance(**self.test_instance)

        cfg = self._get_guest_config_via_fake_api(instance_ref)

        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        terminal_device = cfg.devices[2]
        self.assertIsInstance(terminal_device,
                              vconfig.LibvirtConfigGuestConsole)
        self.assertEqual("sclp", terminal_device.target_type)
        self.assertEqual("pty", terminal_device.type)
        self.assertEqual("s390-ccw-virtio", cfg.os_mach_type)

    def _get_guest_config_via_fake_api(self, instance):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance,
                                            image_meta)
        return drvr._get_guest_config(instance, [],
                                      image_meta, disk_info)

        self.flags(enabled=True, group='vnc')
        self.flags(virt_type='kvm', group='libvirt')
        self.flags(enabled=True, agent_enabled=True, group='spice')
        self.flags(pointer_model='usbtablet')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        self.assertEqual(len(cfg.devices), 11)
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestChannel)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigGuestInput)
        self.assertIsInstance(cfg.devices[8],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[9],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[10],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual(cfg.devices[3].target_name, "com.redhat.spice.0")
        self.assertEqual(cfg.devices[3].type, 'spicevmc')
        self.assertEqual(cfg.devices[4].type, 'vnc')
        self.assertEqual(cfg.devices[5].type, 'spice')
        self.assertEqual(cfg.devices[7].type, 'tablet')

    def test_get_guest_config_with_watchdog_action_image_meta(self):
        self.flags(virt_type='kvm', group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_watchdog_action": "none"}})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref, [], image_meta, disk_info)
        self.assertEqual(len(cfg.devices), 10)
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestInput)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[8],
                              vconfig.LibvirtConfigGuestWatchdog)
        self.assertIsInstance(cfg.devices[9],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual("none", cfg.devices[8].action)

    def _test_guest_add_pointer_device(self, image_meta):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        guest = vconfig.LibvirtConfigGuest()
        guest.os_type = fields.VMMode.HVM
        image_meta = objects.ImageMeta.from_dict(image_meta)
        return drvr._guest_add_pointer_device(guest, image_meta)

    def test_guest_add_pointer_device(self):
        """Check behavior with host-only configuration.

        Mismatches between the host-level configuration and other attributes
        should never result in an exception but rather no device additions.
        """
        self.flags(enabled=True, agent_enabled=False, group='spice')

        image_meta = {'properties': {}}

        self.flags(pointer_model=None)
        dev = self._test_guest_add_pointer_device(image_meta)
        self.assertIsNone(dev)

        self.flags(pointer_model='ps2mouse')
        dev = self._test_guest_add_pointer_device(image_meta)
        self.assertIsNone(dev)

        self.flags(pointer_model='usbtablet')
        dev = self._test_guest_add_pointer_device(image_meta)
        self.assertIsNotNone(dev)

        image_meta = {'properties': {'hw_pointer_model': 'usbtablet'}}

        self.flags(pointer_model=None)
        dev = self._test_guest_add_pointer_device(image_meta)
        self.assertIsNotNone(dev)

        self.flags(pointer_model='ps2mouse')
        dev = self._test_guest_add_pointer_device(image_meta)
        self.assertIsNotNone(dev)

        self.flags(pointer_model='usbtablet')
        dev = self._test_guest_add_pointer_device(image_meta)
        self.assertIsNotNone(dev)

        image_meta = {'properties': {'hw_input_bus': 'usb'}}

        dev = self._test_guest_add_pointer_device(image_meta)
        self.assertEqual('tablet', dev.type)
        self.assertEqual('usb', dev.bus)

        image_meta = {'properties': {'hw_input_bus': 'virtio'}}

        dev = self._test_guest_add_pointer_device(image_meta)
        self.assertEqual('tablet', dev.type)
        self.assertEqual('virtio', dev.bus)

    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test_guest_add_pointer_device__fail_with_spice_agent(self, mock_warn):
        self.flags(enabled=False, group='vnc')
        self.flags(enabled=True, agent_enabled=True, group='spice')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        guest = vconfig.LibvirtConfigGuest()
        guest.os_type = fields.VMMode.HVM
        image_meta = objects.ImageMeta.from_dict(
            {'properties': {'hw_pointer_model': 'usbtablet'}})
        dev = drvr._guest_add_pointer_device(guest, image_meta)

        self.assertIsNone(dev)
        mock_warn.assert_called_once()
        self.assertIn(
            'USB tablet requested for guests but the SPICE agent is enabled',
            str(mock_warn.call_args[0]),
        )

    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test_guest_add_pointer_device__fail_with_non_hvm(self, mock_warn):
        self.flags(enabled=True, agent_enabled=False, group='spice')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        guest = vconfig.LibvirtConfigGuest()
        guest.os_type = fields.VMMode.EXE
        image_meta = objects.ImageMeta.from_dict(
            {'properties': {'hw_pointer_model': 'usbtablet'}})
        dev = drvr._guest_add_pointer_device(guest, image_meta)

        self.assertIsNone(dev)
        mock_warn.assert_called_once()
        self.assertIn(
            'USB tablet requested for guests on non-HVM host',
            str(mock_warn.call_args[0]),
        )

    def _test_guest_add_keyboard_device(self, image_meta, arch=None):
        self.mock_uname.return_value = fakelibvirt.os_uname(
            'Linux', '', '5.4.0-0-generic', '',
            arch or fields.Architecture.X86_64)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        guest = vconfig.LibvirtConfigGuest()
        guest.os_type = fields.VMMode.HVM
        image_meta = objects.ImageMeta.from_dict({'properties': image_meta})
        return drvr._guest_add_keyboard_device(guest, image_meta)

    def test_guest_add_keyboard_device(self):
        props = {}

        dev = self._test_guest_add_keyboard_device(props)
        self.assertIsNone(dev)

        props = {'hw_input_bus': 'usb'}

        dev = self._test_guest_add_keyboard_device(props)
        self.assertEqual('usb', dev.bus)
        self.assertEqual('keyboard', dev.type)

        props = {'hw_input_bus': 'virtio'}

        dev = self._test_guest_add_keyboard_device(props)
        self.assertEqual('virtio', dev.bus)
        self.assertEqual('keyboard', dev.type)

        props = {'hw_architecture': fields.Architecture.AARCH64}

        dev = self._test_guest_add_keyboard_device(props)
        self.assertEqual('usb', dev.bus)
        self.assertEqual('keyboard', dev.type)

        props = {}

        dev = self._test_guest_add_keyboard_device(
            props, arch=fields.Architecture.AARCH64)
        self.assertEqual('usb', dev.bus)
        self.assertEqual('keyboard', dev.type)

    def test_get_guest_config_with_watchdog_action_flavor(self):
        self.flags(virt_type='kvm', group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor.extra_specs = {"hw:watchdog_action": 'none'}
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)

        self.assertEqual(10, len(cfg.devices))
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[5],
                            vconfig.LibvirtConfigGuestInput)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[8],
                              vconfig.LibvirtConfigGuestWatchdog)
        self.assertIsInstance(cfg.devices[9],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual("none", cfg.devices[8].action)

    def test_get_guest_config_with_watchdog_overrides_flavor(self):
        self.flags(virt_type='kvm', group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor.extra_specs = {'hw:watchdog_action': 'none'}
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_watchdog_action": "pause"}})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)

        self.assertEqual(10, len(cfg.devices))
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestInput)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[8],
                              vconfig.LibvirtConfigGuestWatchdog)
        self.assertIsInstance(cfg.devices[9],
                              vconfig.LibvirtConfigMemoryBalloon)
        self.assertEqual("pause", cfg.devices[8].action)

    def test_get_guest_config_with_video_driver_image_meta(self):
        self.flags(virt_type='kvm', group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_video_model": "vmvga"}})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref, [], image_meta, disk_info)
        self.assertEqual(len(cfg.devices), 9)
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestInput)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[8],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual(cfg.devices[3].type, "vnc")
        self.assertEqual(cfg.devices[4].type, "vmvga")

    def test_get_guest_config_with_qga_through_image_meta(self):
        self.flags(virt_type='kvm', group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_qemu_guest_agent": "yes"}})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref, [], image_meta, disk_info)
        self.assertEqual(10, len(cfg.devices))
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestInput)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestChannel)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[8],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[9],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual(cfg.devices[3].type, "vnc")
        self.assertEqual(cfg.devices[5].type, "tablet")
        self.assertEqual(cfg.devices[6].type, "unix")
        self.assertEqual(cfg.devices[6].target_name, "org.qemu.guest_agent.0")

    @mock.patch.object(host.Host, 'get_domain_capabilities')
    @mock.patch.object(designer, 'set_driver_iommu_for_all_devices')
    def test_get_guest_config_with_qga_through_image_meta_with_sev(
        self, mock_designer, fake_domain_caps,
    ):
        self._setup_fake_domain_caps(fake_domain_caps)
        extra_properties = {"hw_qemu_guest_agent": "yes"}
        cfg = self._setup_sev_guest(extra_properties)

        self.assertIsInstance(cfg.devices[8],
                              vconfig.LibvirtConfigGuestController)
        self.assertIsInstance(cfg.devices[9],
                              vconfig.LibvirtConfigGuestChannel)

        self.assertEqual(cfg.devices[8].type, "virtio-serial")
        self.assertTrue(cfg.devices[8].uses_virtio)
        self.assertEqual(cfg.devices[9].type, "unix")
        self.assertEqual(cfg.devices[9].target_name, "org.qemu.guest_agent.0")

    def test_get_guest_config_with_vtpm(self):
        self.flags(virt_type='kvm', group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance = objects.Instance(**self.test_instance)
        instance.system_metadata['vtpm_secret_uuid'] = uuids.vtpm
        image_meta = objects.ImageMeta.from_dict({
            'disk_format': 'raw',
            'properties': {
                'hw_tpm_version': '2.0',
                'hw_tpm_model': 'tpm-crb',
            },
        })

        disk_info = blockinfo.get_disk_info(
            CONF.libvirt.virt_type, instance, image_meta)
        cfg = drvr._get_guest_config(instance, [], image_meta, disk_info)
        self.assertEqual(10, len(cfg.devices))
        self.assertIsInstance(
            cfg.devices[0], vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(
            cfg.devices[1], vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(
            cfg.devices[2], vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(
            cfg.devices[3], vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(
            cfg.devices[4], vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(
            cfg.devices[5], vconfig.LibvirtConfigGuestInput)
        self.assertIsInstance(
            cfg.devices[6], vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(
            cfg.devices[7], vconfig.LibvirtConfigGuestVTPM)
        self.assertIsInstance(
            cfg.devices[8], vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(
            cfg.devices[9], vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual(cfg.devices[3].type, 'vnc')
        self.assertEqual(cfg.devices[5].type, 'tablet')
        self.assertEqual(cfg.devices[7].version, '2.0')
        self.assertEqual(cfg.devices[7].model, 'tpm-crb')
        self.assertEqual(cfg.devices[7].secret_uuid, uuids.vtpm)

    def test_get_guest_config_with_video_driver_vram(self):
        self.flags(enabled=False, group='vnc')
        self.flags(virt_type='kvm', group='libvirt')
        self.flags(enabled=True,
                   agent_enabled=True,
                   group='spice')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor.extra_specs = {'hw_video:ram_max_mb': "100"}
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_video_model": "qxl",
                           "hw_video_ram": "64"}})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        self.assertEqual(9, len(cfg.devices))
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestChannel)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[8],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual(cfg.devices[4].type, "spice")
        self.assertEqual(cfg.devices[5].type, "qxl")
        self.assertEqual(cfg.devices[5].vram, 65536)

    def _test_add_video_driver(self, model):
        self.flags(virt_type='kvm', group='libvirt')
        # we could have used VNC here also we just need to enable
        # one of the graphic consoles libvirt supports or else
        # the call to _guest_add_video_device will not work.
        self.flags(enabled=True, agent_enabled=True, group='spice')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        guest = vconfig.LibvirtConfigGuest()
        flavor = objects.Flavor(
            extra_specs={'hw_video:ram_max_mb': '512'})
        image_meta = objects.ImageMeta.from_dict({
            'properties': {
                'hw_video_model': model,
                'hw_video_ram': 8,
            },
        })

        self.assertTrue(drvr._guest_add_video_device(guest))
        video = drvr._add_video_driver(guest, image_meta, flavor)
        self.assertEqual(model, video.type)
        self.assertEqual(8192, video.vram)  # should be in bytes

    def test__add_video_driver(self):
        self._test_add_video_driver('qxl')

    def test__video_model_supported(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        # assert that all known vif models pass
        for model in nova.objects.fields.VideoModel.ALL:
            self.assertTrue(drvr._video_model_supported(model))
        # then assert that fake models fail
        self.assertFalse(drvr._video_model_supported("fake"))

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_video_model_supported')
    def test__add_video_driver_gop(self, _supports_gop_video):
        _supports_gop_video.return_value = True
        self._test_add_video_driver('gop')
        _supports_gop_video.return_value = False
        self.assertRaises(exception.InvalidVideoMode,
                          self._test_add_video_driver, 'gop')

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_video_model_supported')
    def test__add_video_driver_none(self, _supports_none_video):
        _supports_none_video.return_value = True
        self._test_add_video_driver('none')
        _supports_none_video.return_value = False
        self.assertRaises(exception.InvalidVideoMode,
                          self._test_add_video_driver, 'none')

    @mock.patch('nova.virt.disk.api.teardown_container')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.get_info')
    @mock.patch('nova.virt.disk.api.setup_container')
    @mock.patch('oslo_utils.fileutils.ensure_tree')
    @mock.patch('nova.virt.libvirt.utils.get_instance_path')
    def test_create_guest_with_network__unmount_fs_if_error_during_lxc_create(
        self, mock_get_inst_path, mock_ensure_tree, mock_setup_container,
        mock_get_info, mock_teardown,
    ):
        """If we hit an error during a `_create_guest` call to `libvirt+lxc`
        we need to ensure the guest FS is unmounted from the host so that any
        future `lvremove` calls will work.
        """
        self.flags(virt_type='lxc', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        mock_instance = mock.MagicMock()
        mock_get_inst_path.return_value = '/tmp/'
        mock_image_backend = mock.MagicMock()
        drvr.image_backend = mock_image_backend
        mock_image = mock.MagicMock()
        mock_image.path = '/tmp/test.img'
        drvr.image_backend.by_name.return_value = mock_image
        mock_setup_container.return_value = '/dev/nbd0'
        mock_get_info.side_effect = exception.InstanceNotFound(
                                                   instance_id='foo')
        drvr._conn.defineXML = mock.Mock()
        drvr._conn.defineXML.side_effect = ValueError('somethingbad')
        with test.nested(
              mock.patch.object(drvr, '_is_booted_from_volume',
                                return_value=False),
              mock.patch.object(drvr, 'plug_vifs'),
              mock.patch.object(drvr, '_cleanup')):
            self.assertRaises(ValueError,
                              drvr._create_guest_with_network,
                              self.context,
                              'xml',
                              mock_instance, None, None)

            mock_teardown.assert_called_with(container_dir='/tmp/rootfs')

    def test_video_driver_flavor_limit_not_set(self):
        self.flags(virt_type='kvm', group='libvirt')
        self.flags(enabled=True,
                   agent_enabled=True,
                   group='spice')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_video_model": "qxl",
                           "hw_video_ram": "64"}})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        with mock.patch.object(objects.Instance, 'save'):
            self.assertRaises(exception.RequestedVRamTooHigh,
                              drvr._get_guest_config,
                              instance_ref,
                              [],
                              image_meta,
                              disk_info)

    def test_video_driver_ram_above_flavor_limit(self):
        self.flags(virt_type='kvm', group='libvirt')
        self.flags(enabled=True,
                   agent_enabled=True,
                   group='spice')

        instance_ref = objects.Instance(**self.test_instance)
        flavor = instance_ref.get_flavor()
        flavor.extra_specs = {'hw_video:ram_max_mb': "50"}
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_video_model": "qxl",
                           "hw_video_ram": "64"}})
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        with mock.patch.object(objects.Instance, 'save'):
            self.assertRaises(exception.RequestedVRamTooHigh,
                              drvr._get_guest_config,
                              instance_ref,
                              [],
                              image_meta,
                              disk_info)

    def test_get_guest_config_without_qga_through_image_meta(self):
        self.flags(virt_type='kvm', group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_qemu_guest_agent": "no"}})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref, [], image_meta, disk_info)
        self.assertEqual(len(cfg.devices), 9)
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestInput)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[8],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual(cfg.devices[3].type, 'vnc')
        self.assertEqual(cfg.devices[5].type, 'tablet')

    def test_get_guest_config_with_rng_device(self):
        self.flags(virt_type='kvm', group='libvirt')
        self.flags(pointer_model='ps2mouse')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw"})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        self.assertEqual(len(cfg.devices), 8)
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual(cfg.devices[5].model, 'random')
        self.assertEqual(cfg.devices[5].backend, '/dev/urandom')
        self.assertIsNone(cfg.devices[5].rate_bytes)
        self.assertIsNone(cfg.devices[5].rate_period)

    def test_get_guest_config_with_rng_not_allowed(self):
        self.flags(virt_type='kvm', group='libvirt')
        self.flags(pointer_model='ps2mouse')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor.extra_specs = {'hw_rng:allowed': 'False'}
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw"})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        self.assertEqual(len(cfg.devices), 7)
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigMemoryBalloon)

    def test_get_guest_config_with_rng_limits(self):
        self.flags(virt_type='kvm', group='libvirt')
        self.flags(pointer_model='ps2mouse')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor.extra_specs = {'hw_rng:rate_bytes': '1024',
                                           'hw_rng:rate_period': '2'}
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw"})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        self.assertEqual(len(cfg.devices), 8)
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual(cfg.devices[5].model, 'random')
        self.assertEqual(cfg.devices[5].backend, '/dev/urandom')
        self.assertEqual(cfg.devices[5].rate_bytes, 1024)
        self.assertEqual(cfg.devices[5].rate_period, 2)

    @test.patch_exists(SEV_KERNEL_PARAM_FILE, result=False, other=True)
    def test_get_guest_config_with_rng_backend(self):
        self.flags(virt_type='kvm',
                   rng_dev_path='/dev/hw_rng',
                   group='libvirt')
        self.flags(pointer_model='ps2mouse')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw"})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        self.assertEqual(len(cfg.devices), 8)
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[1],
                            vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestSerial)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestRng)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestUSBHostController)
        self.assertIsInstance(cfg.devices[7],
                              vconfig.LibvirtConfigMemoryBalloon)

        self.assertEqual(cfg.devices[5].model, 'random')
        self.assertEqual(cfg.devices[5].backend, '/dev/hw_rng')
        self.assertIsNone(cfg.devices[5].rate_bytes)
        self.assertIsNone(cfg.devices[5].rate_period)

    @mock.patch('nova.virt.libvirt.driver.os.path.exists')
    def test_get_guest_config_with_rng_dev_not_present(self, mock_path):
        self.flags(
            virt_type='kvm', rng_dev_path='/dev/hw_rng', group='libvirt')
        mock_path.return_value = False

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw"})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        self.assertRaises(exception.RngDeviceNotExist,
                          drvr._get_guest_config,
                          instance_ref,
                          [],
                          image_meta, disk_info)

    def test_get_guest_config_with_cpu_quota(self):
        self.flags(virt_type='kvm', group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor.extra_specs = {'quota:cpu_shares': '10000',
                                           'quota:cpu_period': '20000'}
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)

        self.assertEqual(10000, cfg.cputune.shares)
        self.assertEqual(20000, cfg.cputune.period)

    def test_get_guest_config_with_hiding_hypervisor_id(self):
        self.flags(virt_type='kvm', group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        instance_ref = objects.Instance(**self.test_instance)

        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"img_hide_hypervisor_id": "true"}})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref,
                                     [],
                                     image_meta,
                                     disk_info)

        self.assertTrue(
            any(isinstance(feature, vconfig.LibvirtConfigGuestFeatureKvmHidden)
                           for feature in cfg.features))

    def test_get_guest_config_with_hiding_hypervisor_id_flavor_extra_specs(
            self):
        # Input to the test: flavor extra_specs
        flavor_hide_id = fake_flavor.fake_flavor_obj(self.context,
            extra_specs={"hw:hide_hypervisor_id": "true"},
            expected_attrs={"extra_specs"})

        self.flags(virt_type='kvm', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor = flavor_hide_id
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw"})
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref,
                                     [],
                                     image_meta,
                                     disk_info)

        self.assertTrue(
            any(isinstance(feature, vconfig.LibvirtConfigGuestFeatureKvmHidden)
                           for feature in cfg.features))

    def test_get_guest_config_with_hiding_hypervisor_id_img_and_flavor(
            self):
        # Input to the test: image metadata (true) and flavor
        #     extra_specs (true)
        flavor_hide_id = fake_flavor.fake_flavor_obj(self.context,
            extra_specs={"hw:hide_hypervisor_id": "true"},
            expected_attrs={"extra_specs"})
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"img_hide_hypervisor_id": "true"}})

        self.flags(virt_type='kvm', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor = flavor_hide_id
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref,
                                     [],
                                     image_meta,
                                     disk_info)

        self.assertTrue(
            any(isinstance(feature, vconfig.LibvirtConfigGuestFeatureKvmHidden)
                           for feature in cfg.features))

    def test_get_guest_config_with_hiding_hypervisor_id_img_or_flavor(
            self):
        # Input to the test: image metadata (false) and flavor
        #     extra_specs (true)
        flavor_hide_id = fake_flavor.fake_flavor_obj(self.context,
            extra_specs={"hw:hide_hypervisor_id": "true"},
            expected_attrs={"extra_specs"})
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"img_hide_hypervisor_id": "false"}})

        self.flags(virt_type='kvm', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor = flavor_hide_id
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref,
                                     [],
                                     image_meta,
                                     disk_info)

        self.assertTrue(
            any(isinstance(feature, vconfig.LibvirtConfigGuestFeatureKvmHidden)
                           for feature in cfg.features))

        # Input to the test: image metadata (true) and flavor
        #     extra_specs (false)
        flavor_hide_id = fake_flavor.fake_flavor_obj(self.context,
            extra_specs={"hw:hide_hypervisor_id": "false"},
            expected_attrs={"extra_specs"})
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"img_hide_hypervisor_id": "true"}})

        instance_ref.flavor = flavor_hide_id

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref,
                                     [],
                                     image_meta,
                                     disk_info)

        self.assertTrue(
            any(isinstance(feature, vconfig.LibvirtConfigGuestFeatureKvmHidden)
                           for feature in cfg.features))

    def test_get_guest_config_without_hiding_hypervisor_id(self):
        self.flags(virt_type='kvm', group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        instance_ref = objects.Instance(**self.test_instance)

        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"img_hide_hypervisor_id": "false"}})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref,
                                     [],
                                     image_meta,
                                     disk_info)

        self.assertFalse(
            any(isinstance(feature, vconfig.LibvirtConfigGuestFeatureKvmHidden)
                           for feature in cfg.features))

    def test_get_guest_config_without_hiding_hypervisor_id_flavor_extra_specs(
            self):
        flavor_hide_id = fake_flavor.fake_flavor_obj(self.context,
            extra_specs={"hw:hide_hypervisor_id": "false"},
            expected_attrs={"extra_specs"})

        self.flags(virt_type='qemu', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor = flavor_hide_id
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw"})
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref,
                                     [],
                                     image_meta,
                                     disk_info)

        self.assertFalse(
            any(isinstance(feature, vconfig.LibvirtConfigGuestFeatureKvmHidden)
                           for feature in cfg.features))

    def test_get_guest_config_with_pmu(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        for virt_type in ('qemu', 'kvm'):
            self.flags(virt_type=virt_type, group='libvirt')

            for state in ("true", "false"):
                # assert that values set in flavor are reflected in
                # the xml generated.
                flavor = fake_flavor.fake_flavor_obj(self.context,
                    extra_specs={"hw:pmu": state},
                    expected_attrs={"extra_specs"})
                image_meta = objects.ImageMeta.from_dict({
                    "disk_format": "raw"})

                instance_ref.flavor = flavor
                disk_info = blockinfo.get_disk_info(
                    CONF.libvirt.virt_type, instance_ref, image_meta)
                cfg = drvr._get_guest_config(
                    instance_ref, [], image_meta, disk_info)
                self.assertTrue(any(isinstance(
                    feature, vconfig.LibvirtConfigGuestFeaturePMU) and
                    feature.state == strutils.bool_from_string(state)
                    for feature in cfg.features))

                # assert that values set in image are reflected in
                # the xml generated.
                instance_ref.flavor.extra_specs.pop('hw:pmu')
                image_meta = objects.ImageMeta.from_dict({
                    "disk_format": "raw",
                    "properties": {"hw_pmu": state}})

                disk_info = blockinfo.get_disk_info(
                    CONF.libvirt.virt_type, instance_ref, image_meta)
                cfg = drvr._get_guest_config(
                    instance_ref, [], image_meta, disk_info)
                self.assertTrue(any(isinstance(
                    feature, vconfig.LibvirtConfigGuestFeaturePMU) and
                    feature.state == strutils.bool_from_string(state)
                    for feature in cfg.features))

    def test_get_guest_config_with_pmu_unset(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        for virt_type in ('qemu', 'kvm'):
            self.flags(virt_type=virt_type, group='libvirt')

            # assert that if not set in image or flavor no pmu feature object
            # is created.
            flavor = fake_flavor.fake_flavor_obj(
                self.context, extra_specs={}, expected_attrs={"extra_specs"})
            image_meta = objects.ImageMeta.from_dict({"disk_format": "raw"})

            instance_ref.flavor = flavor
            disk_info = blockinfo.get_disk_info(
                CONF.libvirt.virt_type, instance_ref, image_meta)
            cfg = drvr._get_guest_config(
                instance_ref, [], image_meta, disk_info)
            self.assertFalse(
                any(isinstance(feature, vconfig.LibvirtConfigGuestFeaturePMU)
                    for feature in cfg.features))

    def test_get_guest_config_with_pmu_non_qemu_kvm(self):
        # assert that for virt_types other then qemu and kvm
        # the xml the element is not generated.
        self.flags(virt_type='lxc', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)

        for state in ("true", "false"):
            flavor = fake_flavor.fake_flavor_obj(self.context,
                extra_specs={"hw:pmu": state},
                expected_attrs={"extra_specs"})
            image_meta = objects.ImageMeta.from_dict({
                "disk_format": "raw"})

            instance_ref.flavor = flavor
            disk_info = blockinfo.get_disk_info(
                CONF.libvirt.virt_type, instance_ref, image_meta)
            cfg = drvr._get_guest_config(
                instance_ref, [], image_meta, disk_info)
            self.assertFalse(
                any(isinstance(feature, vconfig.LibvirtConfigGuestFeaturePMU)
                for feature in cfg.features))

            # assert that values set in image are also ignored.
            instance_ref.flavor.extra_specs.pop('hw:pmu')
            image_meta = objects.ImageMeta.from_dict({
                "disk_format": "raw",
                "properties": {"hw_pmu": state}})

            disk_info = blockinfo.get_disk_info(
                CONF.libvirt.virt_type, instance_ref, image_meta)
            cfg = drvr._get_guest_config(
                instance_ref, [], image_meta, disk_info)
            self.assertFalse(
                any(isinstance(feature, vconfig.LibvirtConfigGuestFeaturePMU)
                    for feature in cfg.features))

    def _test_get_guest_config_disk_cachemodes(self, images_type):
        # Verify that the configured cachemodes are propagated to the device
        # configurations.
        if images_type == 'flat':
            cachemode = 'file=directsync'
        elif images_type == 'lvm':
            cachemode = 'block=writethrough'
        elif images_type == 'rbd':
            cachemode = 'network=writeback'
        self.flags(disk_cachemodes=[cachemode], group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        for d in cfg.devices:
            if isinstance(d, vconfig.LibvirtConfigGuestDisk):
                expected = cachemode.split('=')
                self.assertEqual(expected[0], d.source_type)
                self.assertEqual(expected[1], d.driver_cache)

    def test_get_guest_config_disk_cachemodes_file(self):
        self.flags(images_type='flat', group='libvirt')
        self._test_get_guest_config_disk_cachemodes('flat')

    def test_get_guest_config_disk_cachemodes_block(self):
        self.flags(images_type='lvm', group='libvirt')
        self.flags(images_volume_group='vols', group='libvirt')
        self._test_get_guest_config_disk_cachemodes('lvm')

    @mock.patch.object(rbd_utils, 'rbd')
    @mock.patch.object(rbd_utils, 'rados')
    @mock.patch.object(rbd_utils.RBDDriver, 'get_mon_addrs',
                       return_value=(mock.Mock(), mock.Mock()))
    def test_get_guest_config_disk_cachemodes_network(
            self, mock_get_mon_addrs, mock_rados, mock_rbd):
        self.flags(images_type='rbd', group='libvirt')
        self._test_get_guest_config_disk_cachemodes('rbd')

    def test_get_guest_config_with_bogus_cpu_quota(self):
        self.flags(virt_type='kvm', group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor.extra_specs = {'quota:cpu_shares': 'fishfood',
                                           'quota:cpu_period': '20000'}
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        self.assertRaises(ValueError,
                          drvr._get_guest_config,
                          instance_ref, [], image_meta, disk_info)

    def test_get_update_guest_cputune(self):
        # No CPU controller on the host
        self.cgroups.version = 0

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor.extra_specs = {'quota:cpu_shares': '10000',
                                           'quota:cpu_period': '20000'}
        self.assertRaises(
            exception.UnsupportedHostCPUControlPolicy,
            drvr._update_guest_cputune, {}, instance_ref.flavor)

    def _create_fake_service_compute(self):
        service_info = {
            'id': 1729,
            'host': 'fake',
            'report_count': 0
        }
        service_ref = objects.Service(**service_info)

        compute_info = {
            'id': 1729,
            'vcpus': 2,
            'memory_mb': 1024,
            'local_gb': 2048,
            'vcpus_used': 0,
            'memory_mb_used': 0,
            'local_gb_used': 0,
            'free_ram_mb': 1024,
            'free_disk_gb': 2048,
            'hypervisor_type': 'xen',
            'hypervisor_version': 1,
            'running_vms': 0,
            'cpu_info': '',
            'current_workload': 0,
            'service_id': service_ref['id'],
            'host': service_ref['host']
        }
        compute_ref = objects.ComputeNode(**compute_info)
        return (service_ref, compute_ref)

    def test_get_guest_config_with_pci_passthrough_kvm(self):
        self.flags(virt_type='kvm', group='libvirt')
        service_ref, compute_ref = self._create_fake_service_compute()

        instance = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        pci_device_info = dict(test_pci_device.fake_db_dev)
        pci_device_info.update(compute_node_id=1,
                               label='fake',
                               status=fields.PciDeviceStatus.ALLOCATED,
                               address='0000:00:00.1',
                               compute_id=compute_ref.id,
                               instance_uuid=instance.uuid,
                               request_id=uuids.pci_req1,
                               extra_info={})
        pci_device = objects.PciDevice(**pci_device_info)
        pci_list = objects.PciDeviceList()
        pci_list.objects.append(pci_device)
        instance.pci_devices = pci_list
        instance.pci_requests = objects.InstancePCIRequests(
            requests=[
                objects.InstancePCIRequest(
                    request_id=uuids.pci_req1, alias_name="pci-alias"
                )
            ]
        )

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance,
                                            image_meta)
        cfg = drvr._get_guest_config(instance, [],
                                     image_meta, disk_info)

        had_pci = 0
        # care only about the PCI devices
        for dev in cfg.devices:
            if type(dev) is vconfig.LibvirtConfigGuestHostdevPCI:
                had_pci += 1
                self.assertEqual(dev.type, 'pci')
                self.assertEqual(dev.managed, 'yes')
                self.assertEqual(dev.mode, 'subsystem')

                self.assertEqual(dev.domain, "0000")
                self.assertEqual(dev.bus, "00")
                self.assertEqual(dev.slot, "00")
                self.assertEqual(dev.function, "1")
        self.assertEqual(had_pci, 1)

    def test_get_guest_config_os_command_line_through_image_meta(self):
        self.flags(virt_type="kvm",
                   cpu_mode='none',
                   group='libvirt')

        self.test_instance['kernel_id'] = "fake_kernel_id"

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"os_command_line":
                           "fake_os_command_line"}})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref,
                                     _fake_network_info(self),
                                     image_meta, disk_info)
        self.assertEqual(cfg.os_cmdline, "fake_os_command_line")

    def test_get_guest_config_os_command_line_without_kernel_id(self):
        self.flags(virt_type="kvm",
                cpu_mode='none',
                group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"os_command_line":
                           "fake_os_command_line"}})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref,
                                     _fake_network_info(self),
                                     image_meta, disk_info)
        self.assertIsNone(cfg.os_cmdline)

    def test_get_guest_config_os_command_empty(self):
        self.flags(virt_type="kvm",
                   cpu_mode='none',
                   group='libvirt')

        self.test_instance['kernel_id'] = "fake_kernel_id"

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"os_command_line": ""}})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        # the instance has 'root=/dev/vda console=tty0 console=ttyS0
        # console=hvc0' set by default, so testing an empty string and None
        # value in the os_command_line image property must pass
        cfg = drvr._get_guest_config(instance_ref,
                                     _fake_network_info(self),
                                     image_meta, disk_info)
        self.assertNotEqual(cfg.os_cmdline, "")

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       "_get_guest_storage_config")
    @mock.patch.object(libvirt_driver.LibvirtDriver, "_has_numa_support")
    def test_get_guest_config_armv7(self, mock_numa, mock_storage):
        self.flags(virt_type="kvm", group="libvirt")
        self.mock_uname.return_value = fakelibvirt.os_uname(
            'Linux', '', '5.4.0-0-generic', '', fields.Architecture.ARMV7)

        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        cfg = drvr._get_guest_config(instance_ref,
                                     _fake_network_info(self),
                                     image_meta, disk_info)
        self.assertEqual(cfg.os_mach_type, "virt")

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       "_get_guest_storage_config")
    @mock.patch.object(libvirt_driver.LibvirtDriver, "_has_numa_support")
    @test.patch_exists(SEV_KERNEL_PARAM_FILE, result=False, other=True)
    def test_get_guest_config_aarch64(self, mock_numa, mock_storage):
        TEST_AMOUNT_OF_PCIE_SLOTS = 8
        CONF.set_override("num_pcie_ports", TEST_AMOUNT_OF_PCIE_SLOTS,
                group='libvirt')

        self.flags(virt_type="kvm", group="libvirt")
        self.mock_uname.return_value = fakelibvirt.os_uname(
            'Linux', '', '5.4.0-0-generic', '', fields.Architecture.AARCH64)

        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        cfg = drvr._get_guest_config(instance_ref,
                                     _fake_network_info(self),
                                     image_meta, disk_info)
        self.assertEqual(cfg.os_mach_type, "virt")

        num_ports = 0
        for device in cfg.devices:
            try:
                if (device.root_name == 'controller' and
                        device.model == 'pcie-root-port'):
                    num_ports += 1
            except AttributeError:
                pass

        self.assertEqual(TEST_AMOUNT_OF_PCIE_SLOTS, num_ports)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       "_get_guest_storage_config")
    @mock.patch.object(libvirt_driver.LibvirtDriver, "_has_numa_support")
    @test.patch_exists(SEV_KERNEL_PARAM_FILE, result=False, other=True)
    def test_get_guest_config_aarch64_with_graphics(
        self, mock_numa, mock_storage,
    ):
        self.mock_uname.return_value = fakelibvirt.os_uname(
            'Linux', '', '5.4.0-0-generic', '', fields.Architecture.AARCH64)
        self.flags(enabled=True, server_listen='10.0.0.1', group='vnc')
        self.flags(virt_type='kvm', group='libvirt')
        self.flags(enabled=False, group='spice')

        cfg = self._get_guest_config_with_graphics()

        self.assertEqual(cfg.os_mach_type, "virt")

        usbhost_exists = False
        keyboard_exists = False
        for device in cfg.devices:
            if device.root_name == 'controller' and device.type == 'usb':
                usbhost_exists = True
            if device.root_name == 'input' and device.type == 'keyboard':
                keyboard_exists = True
        self.assertTrue(usbhost_exists)
        self.assertTrue(keyboard_exists)

    def _get_guest_config_machine_type_through_image_meta(self, mach_type):
        self.flags(virt_type="kvm", group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_machine_type": mach_type}})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        return drvr._get_guest_config(
            instance_ref, _fake_network_info(self), image_meta, disk_info)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_machine_type_through_image_meta(self):
        cfg = self._get_guest_config_machine_type_through_image_meta(
            "fake_machine_type")
        self.assertEqual(cfg.os_mach_type, "fake_machine_type")

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_machine_type_through_image_meta_sev(
            self):
        fake_q35 = "fake-q35-2.11"
        cfg = self._get_guest_config_machine_type_through_image_meta(fake_q35)
        self.assertEqual(cfg.os_mach_type, fake_q35)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_machine_type_from_config(self):
        self.flags(virt_type='kvm', group='libvirt')
        self.flags(hw_machine_type=['x86_64=fake_machine_type'],
                group='libvirt')

        def fake_getCapabilities():
            return """
            <capabilities>
                <host>
                    <uuid>cef19ce0-0ca2-11df-855d-b19fbce37686</uuid>
                    <cpu>
                      <arch>x86_64</arch>
                      <model>Penryn</model>
                      <vendor>Intel</vendor>
                      <topology sockets='1' cores='2' threads='1'/>
                      <feature name='xtpr'/>
                    </cpu>
                </host>
            </capabilities>
            """

        def fake_baselineCPU(cpu, flag):
            return """<cpu mode='custom' match='exact'>
                        <model fallback='allow'>Penryn</model>
                        <vendor>Intel</vendor>
                        <feature policy='require' name='xtpr'/>
                      </cpu>
                   """

        def fake_getCPUModelNames(arch):
            return [
                '486',
                'pentium',
                'pentium2',
                'pentium3',
                'pentiumpro',
                'coreduo',
                'n270',
                'core2duo',
                'qemu32',
                'kvm32',
                'cpu64-rhel5',
                'cpu64-rhel6',
                'qemu64',
                'kvm64',
                'Conroe',
                'Penryn',
                'Nehalem',
                'Nehalem-IBRS',
                'Westmere',
                'Westmere-IBRS',
                'SandyBridge',
                'SandyBridge-IBRS',
                'IvyBridge',
                'IvyBridge-IBRS',
                'Haswell-noTSX',
                'Haswell-noTSX-IBRS',
                'Haswell',
                'Haswell-IBRS',
                'Broadwell-noTSX',
                'Broadwell-noTSX-IBRS',
                'Broadwell',
                'Broadwell-IBRS',
                'Skylake-Client',
                'Skylake-Client-IBRS',
                'Skylake-Server',
                'Skylake-Server-IBRS',
                'Cascadelake-Server',
                'Icelake-Client',
                'Icelake-Server',
                'athlon',
                'phenom',
                'Opteron_G1',
                'Opteron_G2',
                'Opteron_G3',
                'Opteron_G4',
                'Opteron_G5',
                'EPYC',
                'EPYC-IBPB']

        def fake_getCPUMap():
            return (2, [True, True], 2)

        # Make sure the host arch is mocked as x86_64
        self.create_fake_libvirt_mock(getCapabilities=fake_getCapabilities,
                                      baselineCPU=fake_baselineCPU,
                                      getCPUModelNames=fake_getCPUModelNames,
                                      getVersion=lambda: 1005001,
                                      getCPUMap=fake_getCPUMap)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        image_meta.properties = objects.ImageMetaProps.from_dict(
            {'hw_machine_type': 'fake_machine_type'})

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref,
                                     _fake_network_info(self),
                                     image_meta, disk_info)
        self.assertEqual(cfg.os_mach_type, "fake_machine_type")

    def _test_get_guest_config_ppc64(self, device_index):
        """Test for nova.virt.libvirt.driver.LibvirtDriver._get_guest_config.
        """
        self.flags(virt_type='kvm', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        expected = (fields.Architecture.PPC64, fields.Architecture.PPC)
        for guestarch in expected:
            self.mock_uname.return_value = fakelibvirt.os_uname(
                'Linux', '', '5.4.0-0-generic', '', guestarch)

            cfg = drvr._get_guest_config(
                instance_ref, [], image_meta, disk_info)
            self.assertIsInstance(
                cfg.devices[device_index], vconfig.LibvirtConfigGuestVideo)
            self.assertEqual(cfg.devices[device_index].type, 'vga')

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_ppc64_through_image_meta_vnc_enabled(self):
        self.flags(enabled=True, group='vnc')
        self._test_get_guest_config_ppc64(4)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_config_ppc64_through_image_meta_spice_enabled(
        self):
        self.flags(enabled=True,
                   agent_enabled=True,
                   group='spice')
        self._test_get_guest_config_ppc64(6)

    def _test_get_guest_config_bootmenu(self, image_meta, extra_specs):
        self.flags(virt_type='kvm', group='libvirt')
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor.extra_specs = extra_specs
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref, image_meta)
        conf = conn._get_guest_config(instance_ref, [], image_meta, disk_info)
        self.assertTrue(conf.os_bootmenu)

    def test_get_guest_config_bootmenu_via_image_meta(self):
        image_meta = objects.ImageMeta.from_dict(
            {"disk_format": "raw",
             "properties": {"hw_boot_menu": "True"}})
        self._test_get_guest_config_bootmenu(image_meta, {})

    def test_get_guest_config_bootmenu_via_extra_specs(self):
        image_meta = objects.ImageMeta.from_dict(
            self.test_image_meta)
        self._test_get_guest_config_bootmenu(image_meta,
                                             {'hw:boot_menu': 'True'})

    def test_get_guest_cpu_config_none(self):
        self.flags(cpu_mode="none", group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        self.assertIsInstance(conf.cpu,
                              vconfig.LibvirtConfigGuestCPU)
        self.assertIsNone(conf.cpu.mode)
        self.assertIsNone(conf.cpu.model)
        self.assertEqual(conf.cpu.sockets, instance_ref.flavor.vcpus)
        self.assertEqual(conf.cpu.cores, 1)
        self.assertEqual(conf.cpu.threads, 1)

    def test_get_guest_cpu_config_default_kvm(self):
        self.flags(virt_type="kvm",
                   cpu_mode='none',
                   group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        self.assertIsInstance(conf.cpu,
                              vconfig.LibvirtConfigGuestCPU)
        self.assertIsNone(conf.cpu.mode)
        self.assertIsNone(conf.cpu.model)
        self.assertEqual(conf.cpu.sockets, instance_ref.flavor.vcpus)
        self.assertEqual(conf.cpu.cores, 1)
        self.assertEqual(conf.cpu.threads, 1)

    def test_get_guest_cpu_config_default_lxc(self):
        self.flags(virt_type="lxc",
                   cpu_mode='none',
                   group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        self.assertIsNone(conf.cpu)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_cpu_config_automatic(self):
        expected = {
            fields.Architecture.X86_64: "host-model",
            fields.Architecture.I686: "host-model",
            fields.Architecture.PPC: "host-model",
            fields.Architecture.PPC64: "host-model",
            fields.Architecture.ARMV7: "host-model",
            fields.Architecture.AARCH64: "host-passthrough",
        }
        for guestarch, expect_mode in expected.items():
            self.mock_uname.return_value = fakelibvirt.os_uname(
                'Linux', '', '5.4.0-0-generic', '', guestarch)

            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
            instance_ref = objects.Instance(**self.test_instance)
            image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

            disk_info = blockinfo.get_disk_info(
                CONF.libvirt.virt_type, instance_ref, image_meta)
            conf = drvr._get_guest_config(
                instance_ref, _fake_network_info(self), image_meta, disk_info)
            self.assertIsInstance(conf.cpu, vconfig.LibvirtConfigGuestCPU)
            self.assertEqual(conf.cpu.mode, expect_mode)

    def test_get_guest_cpu_config_manual(self):
        for mode in ('host-passthrough', 'host-model'):
            self.flags(cpu_mode=mode, group='libvirt')

            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
            instance_ref = objects.Instance(**self.test_instance)
            image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

            disk_info = blockinfo.get_disk_info(
                CONF.libvirt.virt_type, instance_ref, image_meta)
            conf = drvr._get_guest_config(
                instance_ref, _fake_network_info(self), image_meta, disk_info)
            self.assertIsInstance(conf.cpu, vconfig.LibvirtConfigGuestCPU)
            self.assertEqual(conf.cpu.mode, mode)
            self.assertIsNone(conf.cpu.model)
            self.assertEqual(conf.cpu.sockets, instance_ref.flavor.vcpus)
            self.assertEqual(conf.cpu.cores, 1)
            self.assertEqual(conf.cpu.threads, 1)

    def test_get_guest_cpu_config_custom(self):
        self.flags(cpu_mode="custom", cpu_models=["Penryn"], group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        self.assertIsInstance(conf.cpu,
                              vconfig.LibvirtConfigGuestCPU)
        self.assertEqual(conf.cpu.mode, "custom")
        self.assertEqual(conf.cpu.model, "Penryn")
        self.assertEqual(conf.cpu.sockets, instance_ref.flavor.vcpus)
        self.assertEqual(conf.cpu.cores, 1)
        self.assertEqual(conf.cpu.threads, 1)

    def test_get_guest_cpu_config_qemu_custom_aarch64(self):
        self.flags(cpu_mode='custom', group='libvirt', cpu_models=['max'])
        self.mock_uname.return_value = fakelibvirt.os_uname(
            'Linux', '', '5.4.0-0-generic', '', fields.Architecture.AARCH64)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(
            CONF.libvirt.virt_type, instance_ref, image_meta)
        conf = drvr._get_guest_config(
            instance_ref, _fake_network_info(self), image_meta, disk_info)
        self.assertIsInstance(conf.cpu, vconfig.LibvirtConfigGuestCPU)
        self.assertEqual(conf.cpu.mode, 'custom')

    def test_get_x86_64_hw_emulated_architecture_aarch64(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            'properties': {
                    'hw_architecture': 'x86_64',
                    'hw_emulation_architecture': 'aarch64',
                    'hw_machine_type': 'virt',
                    'hw_firmware_type': 'uefi',
                    }})

        self.assertEqual(drvr._check_emulation_arch(image_meta),
                        'aarch64')

    def test_get_x86_64_hw_emulated_architecture_ppc64(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            'properties': {
                    'hw_architecture': 'x86_64',
                    'hw_emulation_architecture': 'ppc64le',
                    'hw_machine_type': 'pseries',
                    }})

        self.assertEqual(drvr._check_emulation_arch(image_meta),
                        'ppc64le')

    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test_get_guest_cpu_config_custom_with_extra_flags(self,
            mock_warn):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        self.flags(cpu_mode="custom",
                   cpu_models=["IvyBridge"],
                   cpu_model_extra_flags="pcid",
                   group='libvirt')
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        self.assertIsInstance(conf.cpu,
                              vconfig.LibvirtConfigGuestCPU)
        self.assertEqual(conf.cpu.mode, "custom")
        self.assertEqual(conf.cpu.model, "IvyBridge")
        self.assertIn(conf.cpu.features.pop().name, "pcid")
        self.assertEqual(conf.cpu.sockets, instance_ref.flavor.vcpus)
        self.assertEqual(conf.cpu.cores, 1)
        self.assertEqual(conf.cpu.threads, 1)
        self.assertFalse(mock_warn.called)

    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test_get_guest_cpu_config_custom_with_extra_flags_upper_case(self,
            mock_warn):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        self.flags(cpu_mode="custom",
                   cpu_models=["IvyBridge"],
                   cpu_model_extra_flags="PCID",
                   group='libvirt')
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        self.assertIsInstance(conf.cpu,
                              vconfig.LibvirtConfigGuestCPU)
        self.assertEqual("custom", conf.cpu.mode)
        self.assertEqual("IvyBridge", conf.cpu.model)
        # At this point the upper case CPU flag is normalized to lower
        # case, so assert for that
        self.assertEqual("pcid", conf.cpu.features.pop().name)
        self.assertEqual(instance_ref.flavor.vcpus, conf.cpu.sockets)
        self.assertEqual(1, conf.cpu.cores)
        self.assertEqual(1, conf.cpu.threads)
        mock_warn.assert_not_called()

    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test_get_guest_cpu_config_custom_with_multiple_extra_flags(self,
            mock_warn):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        self.flags(cpu_mode="custom",
                   cpu_models=["IvyBridge"],
                   cpu_model_extra_flags=['pcid', 'vmx'],
                   group='libvirt')
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        features = [feature.name for feature in conf.cpu.features]
        self.assertIsInstance(conf.cpu,
                              vconfig.LibvirtConfigGuestCPU)
        self.assertEqual(conf.cpu.mode, "custom")
        self.assertEqual(conf.cpu.model, "IvyBridge")
        self.assertIn("pcid", features)
        self.assertIn("vmx", features)
        self.assertEqual(conf.cpu.sockets, instance_ref.flavor.vcpus)
        self.assertEqual(conf.cpu.cores, 1)
        self.assertEqual(conf.cpu.threads, 1)
        mock_warn.assert_not_called()

    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test_get_guest_cpu_config_custom_flags_enabled_and_disabled(
            self, mock_warn
    ):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        self.flags(
            cpu_mode="custom",
            cpu_models=["Cascadelake-Server"],
            cpu_model_extra_flags=['-hle', '-rtm', '+ssbd', 'mtrr'],
            group='libvirt')
        disk_info = blockinfo.get_disk_info(
            CONF.libvirt.virt_type,
            instance_ref,
            image_meta)
        conf = drvr._get_guest_config(
            instance_ref,
            _fake_network_info(self),
            image_meta, disk_info)
        features = [(feature.name, feature.policy)
                    for feature in conf.cpu.features]
        self.assertIsInstance(
            conf.cpu,
            vconfig.LibvirtConfigGuestCPU)
        self.assertEqual(conf.cpu.mode, "custom")
        self.assertEqual(conf.cpu.model, "Cascadelake-Server")
        self.assertIn(("ssbd", 'require'), features)
        self.assertIn(("mtrr", 'require'), features)
        self.assertIn(("hle", 'disable'), features)
        self.assertIn(("rtm", 'disable'), features)
        self.assertEqual(conf.cpu.sockets, instance_ref.flavor.vcpus)
        self.assertEqual(conf.cpu.cores, 1)
        self.assertEqual(conf.cpu.threads, 1)
        mock_warn.assert_not_called()

    def test_get_guest_cpu_config_custom_upper_cpu_model(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        self.flags(cpu_mode="custom",
                   cpu_models=["PENRYN", "IVYBRIDGE"],
                   group="libvirt")
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        self.assertEqual(conf.cpu.mode, "custom")
        self.assertEqual(conf.cpu.model, "Penryn")
        self.assertEqual(conf.cpu.sockets, instance_ref.flavor.vcpus)
        self.assertEqual(conf.cpu.cores, 1)
        self.assertEqual(conf.cpu.threads, 1)

    def test_get_guest_cpu_config_custom_without_traits(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        self.flags(cpu_mode="custom",
                   cpu_models=["SandyBridge", "IvyBridge"],
                   group="libvirt")
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        self.assertEqual(conf.cpu.mode, "custom")
        self.assertEqual(conf.cpu.model, "SandyBridge")
        self.assertEqual(conf.cpu.sockets, instance_ref.flavor.vcpus)
        self.assertEqual(conf.cpu.cores, 1)
        self.assertEqual(conf.cpu.threads, 1)

    @mock.patch('nova.virt.libvirt.host.libvirt.Connection.baselineCPU')
    def test_get_guest_cpu_config_custom_with_traits(self, mocked_baseline):
        mocked_baseline.side_effect = ('', _fake_qemu64_cpu_feature,
                                       _fake_broadwell_cpu_feature)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        extra_specs = {
            "trait:HW_CPU_X86_AVX": "required",
            "trait:HW_CPU_X86_AVX2": "required"
        }
        test_instance = _create_test_instance()
        test_instance["flavor"]["extra_specs"] = extra_specs
        instance_ref = objects.Instance(**test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        self.flags(cpu_mode="custom",
                   cpu_models=["qemu64", "Broadwell-noTSX"],
                   group="libvirt")
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        self.assertEqual(conf.cpu.mode, "custom")
        self.assertEqual(conf.cpu.model, "Broadwell-noTSX")
        self.assertEqual(conf.cpu.sockets, instance_ref.flavor.vcpus)
        self.assertEqual(conf.cpu.cores, 1)
        self.assertEqual(conf.cpu.threads, 1)

    @mock.patch('nova.virt.libvirt.host.libvirt.Connection.baselineCPU')
    def test_get_guest_cpu_config_custom_with_traits_multi_models(self,
            mocked_baseline):
        mocked_baseline.side_effect = ('', _fake_qemu64_cpu_feature,
                                       _fake_broadwell_cpu_feature)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        extra_specs = {
            "trait:HW_CPU_X86_SSE41": "required",
            "trait:HW_CPU_X86_SSE42": "required"
        }
        test_instance = _create_test_instance()
        test_instance["flavor"]["extra_specs"] = extra_specs
        instance_ref = objects.Instance(**test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        self.flags(cpu_mode="custom",
                   cpu_models=["qemu64", "SandyBridge", "Broadwell-noTSX"],
                   group="libvirt")
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        self.assertEqual(conf.cpu.mode, "custom")
        self.assertEqual(conf.cpu.model, "SandyBridge")
        self.assertEqual(conf.cpu.sockets, instance_ref.flavor.vcpus)
        self.assertEqual(conf.cpu.cores, 1)
        self.assertEqual(conf.cpu.threads, 1)

    @mock.patch('nova.virt.libvirt.host.libvirt.Connection.baselineCPU')
    def test_get_guest_cpu_config_custom_with_traits_none_model(self,
            mocked_baseline):
        mocked_baseline.side_effect = ('', _fake_qemu64_cpu_feature,
                                       _fake_sandy_bridge_cpu_feature)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        extra_specs = {
            "trait:HW_CPU_X86_AVX": "required",
            "trait:HW_CPU_X86_AVX2": "required"
        }
        test_instance = _create_test_instance()
        test_instance["flavor"]["extra_specs"] = extra_specs
        instance_ref = objects.Instance(**test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        self.flags(cpu_mode="custom",
                   cpu_models=["qemu64", "SandyBridge"],
                   group="libvirt")
        self.assertRaises(exception.InvalidCPUInfo,
                          drvr._get_guest_config,
                          instance_ref,
                          _fake_network_info(self),
                          image_meta,
                          disk_info)

    @mock.patch('nova.virt.libvirt.host.libvirt.Connection.baselineCPU')
    def test_get_guest_cpu_config_custom_with_progressive_model(self,
            mocked_baseline):
        """Test progressive models

        If require two flags: flag1 and flag2, and there are three sorted
        CPU models: [model1, model2, model3], model1 only has flag1, model2
        only has flag2, model3 both have flag1 and flag2.

        Test that the driver will select model3 but not model2.
        """
        # Assume that qemu64 have flag avx2 for the test.
        fake_qemu64_cpu_feature_with_avx2 = _fake_qemu64_cpu_feature.split(
            '\n')
        fake_qemu64_cpu_feature_with_avx2.insert(
            -2, "  <feature policy='require' name='avx2'/>")
        fake_qemu64_cpu_feature_with_avx2 = "\n".join(
            fake_qemu64_cpu_feature_with_avx2)

        mocked_baseline.side_effect = ('', fake_qemu64_cpu_feature_with_avx2,
                                       _fake_sandy_bridge_cpu_feature,
                                       _fake_broadwell_cpu_feature)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        extra_specs = {
            "trait:HW_CPU_X86_AVX": "required",
            "trait:HW_CPU_X86_AVX2": "required"
        }
        test_instance = _create_test_instance()
        test_instance["flavor"]["extra_specs"] = extra_specs
        instance_ref = objects.Instance(**test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        self.flags(cpu_mode="custom",
                   cpu_models=["qemu64", "SandyBridge", "Broadwell-noTSX"],
                   group="libvirt")
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        self.assertEqual(conf.cpu.mode, "custom")
        self.assertEqual(conf.cpu.model, "Broadwell-noTSX")
        self.assertEqual(conf.cpu.sockets, instance_ref.flavor.vcpus)
        self.assertEqual(conf.cpu.cores, 1)
        self.assertEqual(conf.cpu.threads, 1)

    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test_get_guest_cpu_config_host_model_with_extra_flags(self,
            mock_warn):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        self.flags(cpu_mode="host-model",
                   cpu_model_extra_flags="pdpe1gb",
                   group='libvirt')
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        features = [feature.name for feature in conf.cpu.features]
        self.assertIsInstance(conf.cpu,
                              vconfig.LibvirtConfigGuestCPU)
        self.assertEqual(conf.cpu.mode, "host-model")
        self.assertIn("pdpe1gb", features)
        self.assertEqual(conf.cpu.sockets, instance_ref.flavor.vcpus)
        self.assertEqual(conf.cpu.cores, 1)
        self.assertEqual(conf.cpu.threads, 1)
        # For 'host-model', it is now valid to use 'extra_flags';
        # assert that no warning is thrown
        mock_warn.assert_not_called()

    def test_get_guest_cpu_config_host_passthrough_with_extra_flags(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        self.flags(cpu_mode="host-passthrough",
                   cpu_model_extra_flags="invtsc",
                   group='libvirt')
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        with mock.patch.object(libvirt_driver.LOG, 'warning') as mock_warn:
            conf = drvr._get_guest_config(instance_ref,
                                          _fake_network_info(self),
                                          image_meta, disk_info)
            # We have lifted the restriction for 'host-passthrough' as well;
            # so here too, assert that no warning is thrown
            mock_warn.assert_not_called()

        features = [feature.name for feature in conf.cpu.features]
        self.assertIsInstance(conf.cpu,
                              vconfig.LibvirtConfigGuestCPU)
        self.assertEqual(conf.cpu.mode, "host-passthrough")
        self.assertIn("invtsc", features)
        self.assertEqual(conf.cpu.sockets, instance_ref.flavor.vcpus)
        self.assertEqual(conf.cpu.cores, 1)
        self.assertEqual(conf.cpu.threads, 1)

    def test_get_guest_cpu_config_maxphysaddr_passthrough_flavor_espec(self):
        self.flags(cpu_mode="none", group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor.extra_specs = {
            'hw:maxphysaddr_mode': 'passthrough'}
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        self.assertIsInstance(conf.cpu.maxphysaddr,
                              vconfig.LibvirtConfigGuestCPUMaxPhysAddr)
        self.assertEqual(conf.cpu.maxphysaddr.mode, 'passthrough')
        self.assertIsNone(conf.cpu.maxphysaddr.bits)

    def test_get_guest_cpu_config_maxphysaddr_emulate_flavor_espec(self):
        self.flags(cpu_mode="none", group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor.extra_specs = {
                'hw:maxphysaddr_mode': 'emulate',
                'hw:maxphysaddr_bits': 42}
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        self.assertIsInstance(conf.cpu.maxphysaddr,
                              vconfig.LibvirtConfigGuestCPUMaxPhysAddr)
        self.assertEqual(conf.cpu.maxphysaddr.mode, 'emulate')
        self.assertEqual(conf.cpu.maxphysaddr.bits, 42)

    def test_get_guest_cpu_config_maxphysaddr_passthrough_image_meta(self):
        self.flags(cpu_mode="none", group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_maxphysaddr_mode": "passthrough"},
        })

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        self.assertIsInstance(conf.cpu.maxphysaddr,
                              vconfig.LibvirtConfigGuestCPUMaxPhysAddr)
        self.assertEqual(conf.cpu.maxphysaddr.mode, 'passthrough')
        self.assertIsNone(conf.cpu.maxphysaddr.bits)

    def test_get_guest_cpu_config_maxphysaddr_emulate_image_meta(self):
        self.flags(cpu_mode="none", group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_maxphysaddr_mode": "emulate",
                           "hw_maxphysaddr_bits": 42},
        })

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        self.assertIsInstance(conf.cpu.maxphysaddr,
                              vconfig.LibvirtConfigGuestCPUMaxPhysAddr)
        self.assertEqual(conf.cpu.maxphysaddr.mode, 'emulate')
        self.assertEqual(conf.cpu.maxphysaddr.bits, 42)

    def test_get_guest_cpu_topology(self):
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.flavor.vcpus = 8
        instance_ref.flavor.extra_specs = {'hw:cpu_max_sockets': '4'}
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        conf = drvr._get_guest_config(instance_ref,
                                      _fake_network_info(self),
                                      image_meta, disk_info)
        self.assertIsInstance(conf.cpu,
                              vconfig.LibvirtConfigGuestCPU)
        self.assertEqual(conf.cpu.mode, "host-model")
        self.assertEqual(conf.cpu.sockets, 4)
        self.assertEqual(conf.cpu.cores, 2)
        self.assertEqual(conf.cpu.threads, 1)

    def test_get_guest_iommu_not_enabled(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        test_instance = _create_test_instance()
        instance_ref = objects.Instance(**test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        for device in cfg.devices:
            self.assertNotEqual('iommu', device.root_name)

    def test_get_guest_iommu_config_model(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_machine_type": "q35"},
        })

        extra_specs = {
            "hw:viommu_model": 'intel',
        }
        test_instance = _create_test_instance()
        test_instance["flavor"]["extra_specs"] = extra_specs
        instance_ref = objects.Instance(**test_instance)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        count = 0
        for device in cfg.devices:
            if device.root_name == 'iommu':
                count += 1
                self.assertIsInstance(device,
                                      vconfig.LibvirtConfigGuestIOMMU)
                self.assertEqual('intel', device.model)
                self.assertTrue(hasattr(device, "aw_bits"))
                self.assertTrue(device.interrupt_remapping)
                self.assertTrue(device.caching_mode)
                self.assertTrue(device.eim)
                self.assertTrue(device.iotlb)

        self.assertEqual(1, count)
        self.assertEqual('q35', cfg.os_mach_type)

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    def test_get_guest_iommu_config_model_auto(self, has_min_version):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_machine_type": "q35"},
        })

        extra_specs = {
            "hw:viommu_model": 'auto',
        }
        test_instance = _create_test_instance()
        test_instance["flavor"]["extra_specs"] = extra_specs
        instance_ref = objects.Instance(**test_instance)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        count = 0
        for device in cfg.devices:
            if device.root_name == 'iommu':
                count += 1
                self.assertIsInstance(device,
                                      vconfig.LibvirtConfigGuestIOMMU)
                self.assertEqual('virtio', device.model)
                self.assertEqual(48, device.aw_bits)
                self.assertTrue(device.interrupt_remapping)
                self.assertTrue(device.caching_mode)
                self.assertTrue(device.eim)
                self.assertTrue(device.iotlb)

        self.assertEqual(1, count)
        self.assertEqual('q35', cfg.os_mach_type)

    @mock.patch.object(host.Host, 'has_min_version', return_value=False)
    def test_get_guest_iommu_config_model_auto_intel(self, has_min_version):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_machine_type": "q35"},
        })

        extra_specs = {
            "hw:viommu_model": 'auto',
        }
        test_instance = _create_test_instance()
        test_instance["flavor"]["extra_specs"] = extra_specs
        instance_ref = objects.Instance(**test_instance)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        count = 0
        for device in cfg.devices:
            if device.root_name == 'iommu':
                count += 1
                self.assertIsInstance(device,
                                      vconfig.LibvirtConfigGuestIOMMU)
                self.assertEqual('intel', device.model)
                self.assertTrue(device.interrupt_remapping)
                self.assertTrue(device.caching_mode)
                self.assertTrue(device.eim)
                self.assertTrue(device.iotlb)

        self.assertEqual(1, count)
        self.assertEqual('q35', cfg.os_mach_type)

    @mock.patch.object(host.Host, 'has_min_version', return_value=False)
    def test_get_guest_iommu_config_model_auto_aarch64(self, has_min_version):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_viommu_model": 'auto',
                           "hw_architecture": fields.Architecture.AARCH64,
                           "hw_machine_type": "virt"},
        })
        extra_specs = {
            "hw:viommu_model": 'auto',
        }
        test_instance = _create_test_instance()
        test_instance["flavor"]["extra_specs"] = extra_specs
        instance_ref = objects.Instance(**test_instance)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        count = 0
        for device in cfg.devices:
            if device.root_name == 'iommu':
                count += 1
                self.assertIsInstance(device,
                                      vconfig.LibvirtConfigGuestIOMMU)
                self.assertEqual('smmuv3', device.model)
                self.assertTrue(hasattr(device, "aw_bits"))
                self.assertTrue(device.interrupt_remapping)
                self.assertTrue(device.caching_mode)
                self.assertFalse(device.eim)
                self.assertTrue(device.iotlb)
        self.assertEqual(1, count)

    def test_get_guest_iommu_config_not_support_machine_type(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_machine_type": "pc-i440fx-2.11"},
        })
        extra_specs = {
            "hw:viommu_model": 'auto',
        }
        test_instance = _create_test_instance()
        test_instance["flavor"]["extra_specs"] = extra_specs
        instance_ref = objects.Instance(**test_instance)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        self.assertRaises(
            exception.InvalidVIOMMUMachineType, drvr._get_guest_config,
            instance_ref, [], image_meta, disk_info
        )

    def test_get_guest_iommu_config_not_support_architecture(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "raw",
            "properties": {"hw_architecture": fields.Architecture.PPC64LE,
                           "hw_machine_type": "pc-i440fx-2.11"},
        })
        extra_specs = {
            "hw:viommu_model": 'auto',
        }
        test_instance = _create_test_instance()
        test_instance["flavor"]["extra_specs"] = extra_specs
        instance_ref = objects.Instance(**test_instance)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        self.assertRaises(
            exception.InvalidVIOMMUArchitecture, drvr._get_guest_config,
            instance_ref, [], image_meta, disk_info
        )

    def test_get_guest_memory_balloon_config_by_default(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        for device in cfg.devices:
            if device.root_name == 'memballoon':
                self.assertIsInstance(device,
                                      vconfig.LibvirtConfigMemoryBalloon)
                self.assertEqual('virtio', device.model)
                self.assertEqual(10, device.period)

    def test_get_guest_memory_balloon_config_disable(self):
        self.flags(mem_stats_period_seconds=0, group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        no_exist = True
        for device in cfg.devices:
            if device.root_name == 'memballoon':
                no_exist = False
                break
        self.assertTrue(no_exist)

    def test_get_guest_memory_balloon_config_period_value(self):
        self.flags(mem_stats_period_seconds=21, group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        for device in cfg.devices:
            if device.root_name == 'memballoon':
                self.assertIsInstance(device,
                                      vconfig.LibvirtConfigMemoryBalloon)
                self.assertEqual('virtio', device.model)
                self.assertEqual(21, device.period)

    def test_get_guest_memory_balloon_config_qemu(self):
        self.flags(virt_type='qemu', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        for device in cfg.devices:
            if device.root_name == 'memballoon':
                self.assertIsInstance(device,
                                      vconfig.LibvirtConfigMemoryBalloon)
                self.assertEqual('virtio', device.model)
                self.assertEqual(10, device.period)

    def test_get_guest_memory_balloon_config_lxc(self):
        self.flags(virt_type='lxc', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        cfg = drvr._get_guest_config(instance_ref, [],
                                     image_meta, disk_info)
        no_exist = True
        for device in cfg.devices:
            if device.root_name == 'memballoon':
                no_exist = False
                break
        self.assertTrue(no_exist)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_get_guest_usb_controller(self):
        self.flags(enabled=True, group='vnc')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(
            CONF.libvirt.virt_type, instance_ref, image_meta)

        # we don't have any input manually configured and our disks are using
        # the sata bus, but having VNC enabled means a USB tablet will be added
        # by default
        cfg = drvr._get_guest_config(instance_ref, [], image_meta, disk_info)
        for device in cfg.devices:
            if isinstance(device, vconfig.LibvirtConfigGuestUSBHostController):
                # the model property should be unset, meaning auto-configured
                self.assertIsNone(device.model)
                break
        else:
            self.fail('Did not find a USB host controller')

        # disabling VNC means we should no longer have a video device present,
        # meaning no USB controller is necessary now
        self.flags(enabled=False, group='vnc')

        cfg = drvr._get_guest_config(instance_ref, [], image_meta, disk_info)
        for device in cfg.devices:
            if isinstance(device, vconfig.LibvirtConfigGuestUSBHostController):
                self.assertEqual('none', device.model)
                break
        else:
            self.fail('Did not find a USB host controller')

        # but adding a USB-based disk device should necessitate a USB
        # controller once again
        image_meta = objects.ImageMeta.from_dict(
            {'properties': {'hw_disk_bus': 'usb'}})
        disk_info = blockinfo.get_disk_info(
            CONF.libvirt.virt_type, instance_ref, image_meta)

        cfg = drvr._get_guest_config(instance_ref, [], image_meta, disk_info)
        for device in cfg.devices:
            if isinstance(device, vconfig.LibvirtConfigGuestUSBHostController):
                # the model property should be unset, meaning auto-configured
                self.assertIsNone(device.model)
                break
        else:
            self.fail('Did not find a USB host controller')

        # while PPC64 should always get a USB controller, regardless of config
        self.mock_uname.return_value = fakelibvirt.os_uname(
            'Linux', '', '5.4.0-0-generic', '', fields.Architecture.PPC64)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(
            CONF.libvirt.virt_type, instance_ref, image_meta)

        cfg = drvr._get_guest_config(instance_ref, [], image_meta, disk_info)
        for device in cfg.devices:
            if isinstance(device, vconfig.LibvirtConfigGuestUSBHostController):
                # the model property should be unset, meaning auto-configured
                self.assertIsNone(device.model)
                break
        else:
            self.fail('Did not find a USB host controller')

    @mock.patch.object(libvirt_driver, 'LOG')
    @mock.patch.object(
        fakelibvirt, 'VIR_PERF_PARAM_CPU_CLOCK', 'cpu_clock', create=True)
    @mock.patch.object(fakelibvirt, 'VIR_PERF_PARAM_CMT', 'cmt', create=True)
    def test_get_supported_perf_events(self, mock_log):
        self.flags(
            enabled_perf_events=['cpu_clock', 'foo', 'cmt'], group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        events = drvr._get_supported_perf_events()

        self.assertEqual(['cpu_clock'], events)
        self.assertEqual(2, len(mock_log.warning.mock_calls))

    def test_xml_and_uri_no_ramdisk_no_kernel(self):
        instance_data = dict(self.test_instance)
        self._check_xml_and_uri(instance_data,
                                expect_kernel=False, expect_ramdisk=False)

    def test_xml_and_uri_no_ramdisk(self):
        instance_data = dict(self.test_instance)
        instance_data['kernel_id'] = 'aki-deadbeef'
        self._check_xml_and_uri(instance_data,
                                expect_kernel=True, expect_ramdisk=False)

    def test_xml_and_uri_no_kernel(self):
        instance_data = dict(self.test_instance)
        instance_data['ramdisk_id'] = 'ari-deadbeef'
        self._check_xml_and_uri(instance_data,
                                expect_kernel=False, expect_ramdisk=False)

    def test_xml_and_uri(self):
        instance_data = dict(self.test_instance)
        instance_data['ramdisk_id'] = 'ari-deadbeef'
        instance_data['kernel_id'] = 'aki-deadbeef'
        self._check_xml_and_uri(instance_data,
                                expect_kernel=True, expect_ramdisk=True)

    def test_xml_and_uri_rescue(self):
        instance_data = dict(self.test_instance)
        instance_data['ramdisk_id'] = 'ari-deadbeef'
        instance_data['kernel_id'] = 'aki-deadbeef'
        self._check_xml_and_uri(instance_data, expect_kernel=True,
                                expect_ramdisk=True, rescue=instance_data)

    def test_xml_and_uri_rescue_no_kernel_no_ramdisk(self):
        instance_data = dict(self.test_instance)
        self._check_xml_and_uri(instance_data, expect_kernel=False,
                                expect_ramdisk=False, rescue=instance_data)

    def test_xml_and_uri_rescue_no_kernel(self):
        instance_data = dict(self.test_instance)
        instance_data['ramdisk_id'] = 'aki-deadbeef'
        self._check_xml_and_uri(instance_data, expect_kernel=False,
                                expect_ramdisk=True, rescue=instance_data)

    def test_xml_and_uri_rescue_no_ramdisk(self):
        instance_data = dict(self.test_instance)
        instance_data['kernel_id'] = 'aki-deadbeef'
        self._check_xml_and_uri(instance_data, expect_kernel=True,
                                expect_ramdisk=False, rescue=instance_data)

    def test_xml_uuid(self):
        self._check_xml_and_uuid(self.test_image_meta)

    def test_lxc_container_and_uri(self):
        instance_data = dict(self.test_instance)
        self._check_xml_and_container(instance_data)

    def test_xml_disk_prefix(self):
        instance_data = dict(self.test_instance)
        self._check_xml_and_disk_prefix(instance_data, None)

    def test_xml_user_specified_disk_prefix(self):
        instance_data = dict(self.test_instance)
        self._check_xml_and_disk_prefix(instance_data, 'sd')

    def test_xml_disk_driver(self):
        instance_data = dict(self.test_instance)
        self._check_xml_and_disk_driver(instance_data)

    def test_xml_disk_bus_virtio(self):
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        self._check_xml_and_disk_bus(image_meta,
                                     None,
                                     (("disk", "virtio", "vda"),))

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_xml_disk_bus_ide(self):
        # It's necessary to check if the architecture is power, because
        # power doesn't have support to ide, and so libvirt translate
        # all ide calls to scsi

        expected = {
            fields.Architecture.X86_64: ("cdrom", "ide", "hda"),
            fields.Architecture.I686: ("cdrom", "ide", "hda"),
            fields.Architecture.PPC: ("cdrom", "scsi", "sda"),
            fields.Architecture.PPC64: ("cdrom", "scsi", "sda"),
            fields.Architecture.PPC64LE: ("cdrom", "scsi", "sda"),
            fields.Architecture.AARCH64: ("cdrom", "scsi", "sda"),
        }

        for guestarch, expected_disk in expected.items():
            self.mock_uname.return_value = fakelibvirt.os_uname(
                'Linux', '', '5.4.0-0-generic', '', guestarch)

            image_meta = objects.ImageMeta.from_dict({"disk_format": "iso"})
            self._check_xml_and_disk_bus(image_meta, None, (expected_disk,))

    def test_xml_disk_bus_sata(self):
        # NOTE(sean-k-mooney): here we assert that when
        # root_device_name is set in the block_device_info
        # and hw_disk_bus is set in the image properties,
        # we use the property value
        expected = ("disk", "sata", "vda")

        image_meta = objects.ImageMeta.from_dict({"properties": {
            "hw_disk_bus": "sata"}})
        block_device_info = {'root_device_name': "vda"}
        self._check_xml_and_disk_bus(image_meta,
                                     block_device_info,
                                     (expected,))

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_xml_disk_bus_ide_and_virtio(self):
        expected = {
            fields.Architecture.X86_64: ("cdrom", "ide", "hda"),
            fields.Architecture.I686: ("cdrom", "ide", "hda"),
            fields.Architecture.PPC: ("cdrom", "scsi", "sda"),
            fields.Architecture.PPC64: ("cdrom", "scsi", "sda"),
            fields.Architecture.PPC64LE: ("cdrom", "scsi", "sda"),
            fields.Architecture.AARCH64: ("cdrom", "scsi", "sda"),
        }

        swap = {'device_name': '/dev/vdc', 'swap_size': 1}
        ephemerals = [
            {
                'device_type': 'disk',
                'disk_bus': 'virtio',
                'device_name': '/dev/vdb',
                'size': 1,
            },
        ]
        block_device_info = {'swap': swap, 'ephemerals': ephemerals}
        image_meta = objects.ImageMeta.from_dict({
            "disk_format": "iso"})

        for guestarch, expected_disk in expected.items():
            self.mock_uname.return_value = fakelibvirt.os_uname(
                'Linux', '', '5.4.0-0-generic', '', guestarch)

            self._check_xml_and_disk_bus(
                image_meta,
                block_device_info,
                (
                    expected_disk,
                    ("disk", "virtio", "vdb"),
                    ("disk", "virtio", "vdc"),
                ),
            )

    @mock.patch.object(host.Host, 'get_guest')
    def test_instance_exists(self, mock_get_guest):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertTrue(drvr.instance_exists(None))

        mock_get_guest.side_effect = exception.InstanceNotFound(
            instance_id='something')
        self.assertFalse(drvr.instance_exists(None))

        mock_get_guest.side_effect = exception.InternalError(err='something')
        self.assertFalse(drvr.instance_exists(None))

    @mock.patch.object(host.Host, "list_instance_domains")
    def test_list_instances(self, mock_list):
        vm1 = FakeVirtDomain(id=3, name="instance00000001")
        vm2 = FakeVirtDomain(id=17, name="instance00000002")
        vm3 = FakeVirtDomain(name="instance00000003")
        vm4 = FakeVirtDomain(name="instance00000004")

        mock_list.return_value = [vm1, vm2, vm3, vm4]

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        names = drvr.list_instances()
        self.assertEqual(names[0], vm1.name())
        self.assertEqual(names[1], vm2.name())
        self.assertEqual(names[2], vm3.name())
        self.assertEqual(names[3], vm4.name())
        mock_list.assert_called_with(only_running=False)

    @mock.patch.object(host.Host, "list_instance_domains")
    def test_list_instance_uuids(self, mock_list):
        vm1 = FakeVirtDomain(id=3, name="instance00000001")
        vm2 = FakeVirtDomain(id=17, name="instance00000002")
        vm3 = FakeVirtDomain(name="instance00000003")
        vm4 = FakeVirtDomain(name="instance00000004")

        mock_list.return_value = [vm1, vm2, vm3, vm4]
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        uuids = drvr.list_instance_uuids()
        self.assertEqual(len(uuids), 4)
        self.assertEqual(uuids[0], vm1.UUIDString())
        self.assertEqual(uuids[1], vm2.UUIDString())
        self.assertEqual(uuids[2], vm3.UUIDString())
        self.assertEqual(uuids[3], vm4.UUIDString())
        mock_list.assert_called_with(only_running=False)

    @mock.patch('nova.virt.libvirt.host.Host.get_online_cpus',
                return_value=set([0, 1, 2, 3]))
    def test_get_pcpu_available(self, get_online_cpus):
        """Test what happens when the '[compute] cpu_dedicated_set' config
        option is set.
        """
        self.flags(vcpu_pin_set=None)
        self.flags(cpu_dedicated_set='2-3', cpu_shared_set=None,
                   group='compute')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        pcpus = drvr._get_pcpu_available()
        self.assertEqual(set([2, 3]), pcpus)

    @mock.patch('nova.virt.libvirt.host.Host.get_online_cpus',
                return_value=set([0, 1, 2, 3]))
    def test_get_pcpu_available__cpu_dedicated_set_unset(
            self, get_online_cpus):
        """Test what happens when the '[compute] cpu_dedicated_set' config
        option is not set.
        """
        self.flags(vcpu_pin_set=None)
        self.flags(cpu_dedicated_set=None, cpu_shared_set=None,
                   group='compute')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        pcpus = drvr._get_pcpu_available()
        self.assertEqual(set([]), pcpus)

    @mock.patch('nova.virt.libvirt.host.Host.get_online_cpus',
                return_value=set([4, 5]))
    def test_get_pcpu_available__cpu_dedicated_set_invalid(self,
            get_online_cpus):
        """Test what happens when the '[compute] cpu_dedicated_set' config
        option is set but it's invalid.
        """
        self.flags(vcpu_pin_set=None)
        self.flags(cpu_dedicated_set='4-6', cpu_shared_set=None,
                   group='compute')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertRaises(exception.Invalid, drvr._get_pcpu_available)

    @mock.patch('nova.virt.libvirt.host.Host.get_available_cpus',
                return_value=set([0, 1, 2, 3]))
    def test_get_pcpu_available_for_power_mgmt(self, get_available_cpus):
        """Test what happens when the '[compute] cpu_dedicated_set' config
        option is set and power management is defined.
        """
        self.flags(vcpu_pin_set=None)
        self.flags(cpu_dedicated_set='2-3', cpu_shared_set=None,
                   group='compute')
        self.flags(cpu_power_management=True, group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        pcpus = drvr._get_pcpu_available()
        self.assertEqual(set([2, 3]), pcpus)

    @mock.patch('nova.virt.libvirt.host.Host.get_available_cpus',
                return_value=set([4, 5]))
    def test_get_pcpu_available__cpu_dedicated_set_invalid_for_pm(self,
            get_available_cpus):
        """Test what happens when the '[compute] cpu_dedicated_set' config
        option is set but it's invalid with power management set.
        """
        self.flags(vcpu_pin_set=None)
        self.flags(cpu_dedicated_set='4-6', cpu_shared_set=None,
                   group='compute')
        self.flags(cpu_power_management=True, group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertRaises(exception.Invalid, drvr._get_pcpu_available)

    @mock.patch('nova.virt.libvirt.host.Host.get_online_cpus',
                return_value=set([0, 1, 2, 3]))
    def test_get_vcpu_available(self, get_online_cpus):
        """Test what happens when neither the '[compute] cpu_shared_set' or
        legacy 'vcpu_pin_set' config options are set.
        """
        self.flags(vcpu_pin_set=None)
        self.flags(cpu_shared_set=None, cpu_dedicated_set=None,
                   group='compute')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        vcpus = drvr._get_vcpu_available()
        self.assertEqual(set([0, 1, 2, 3]), vcpus)

    @mock.patch('nova.virt.libvirt.host.Host.get_online_cpus',
                return_value=set([4, 5, 6, 7]))
    def test_get_vcpu_available__with_cpu_shared_set(self, get_online_cpus):
        """Test what happens when the '[compute] cpu_shared_set' config option
        is set.
        """
        self.flags(vcpu_pin_set=None)
        self.flags(cpu_shared_set='4-5', cpu_dedicated_set=None,
                   group='compute')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        vcpus = drvr._get_vcpu_available()
        self.assertEqual(set([4, 5]), vcpus)

    @mock.patch('nova.virt.libvirt.host.Host.get_online_cpus',
                return_value=set([4, 5, 6, 7]))
    def test_get_vcpu_available__with_vcpu_pin_set(self, get_online_cpus):
        """Test what happens when the legacy 'vcpu_pin_set' config option is
        set.
        """
        self.flags(vcpu_pin_set='4-5')
        self.flags(cpu_shared_set=None, cpu_dedicated_set=None,
                   group='compute')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        vcpus = drvr._get_vcpu_available()
        self.assertEqual(set([4, 5]), vcpus)

    @mock.patch('nova.virt.libvirt.host.Host.get_online_cpus',
                return_value=set([4, 5, 6, 7]))
    def test_get_vcpu_available__with_cpu_dedicated_set(self, get_online_cpus):
        """Test what happens when the '[compute] cpu_dedicated_set' config
        option is set.
        """
        self.flags(vcpu_pin_set=None)
        self.flags(cpu_shared_set=None, cpu_dedicated_set='4-5',
                   group='compute')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        vcpus = drvr._get_vcpu_available()
        self.assertEqual(set([]), vcpus)

    @mock.patch('nova.virt.libvirt.host.Host.get_online_cpus',
                return_value=set([4, 5]))
    def test_get_vcpu_available__cpu_shared_set_invalid(self, get_online_cpus):
        """Test what happens when the '[compute] cpu_shared_set' config option
        is set but it's invalid.
        """
        self.flags(vcpu_pin_set=None)
        self.flags(cpu_shared_set='4-6', cpu_dedicated_set=None,
                   group='compute')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertRaises(exception.Invalid, drvr._get_vcpu_available)

    @mock.patch('nova.virt.libvirt.host.Host.get_online_cpus',
                return_value=set([4, 5]))
    def test_get_vcpu_available__vcpu_pin_set_invalid(self, get_online_cpus):
        """Test what happens when the legacy 'vcpu_pin_set' config option is
        set but it's invalid.
        """
        self.flags(vcpu_pin_set='4-6')
        self.flags(cpu_shared_set=None, cpu_dedicated_set=None,
                   group='compute')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertRaises(exception.Invalid, drvr._get_vcpu_available)

    @mock.patch.object(host.Host, "has_min_version", return_value=True)
    def test_quiesce(self, mock_has_min_version):
        self.create_fake_libvirt_mock(lookupByUUIDString=self.fake_lookup)
        with mock.patch.object(FakeVirtDomain, "fsFreeze") as mock_fsfreeze:
            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
            instance = objects.Instance(**self.test_instance)
            image_meta = objects.ImageMeta.from_dict(
                {"properties": {"hw_qemu_guest_agent": "yes"}})
            self.assertIsNone(drvr.quiesce(self.context, instance, image_meta))
            mock_fsfreeze.assert_called_once_with()

    @mock.patch.object(host.Host, "has_min_version", return_value=True)
    def test_unquiesce(self, mock_has_min_version):
        self.create_fake_libvirt_mock(getLibVersion=lambda: 1002005,
                                      lookupByUUIDString=self.fake_lookup)
        with mock.patch.object(FakeVirtDomain, "fsThaw") as mock_fsthaw:
            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
            instance = objects.Instance(**self.test_instance)
            image_meta = objects.ImageMeta.from_dict(
                {"properties": {"hw_qemu_guest_agent": "yes"}})
            self.assertIsNone(drvr.unquiesce(self.context, instance,
                                             image_meta))
            mock_fsthaw.assert_called_once_with()

    def test_set_quiesced_agent_connection_fails(self):
        # This is require to mock guest host
        self.create_fake_libvirt_mock(lookupByUUIDString=self.fake_lookup)

        with mock.patch.object(FakeVirtDomain, "fsFreeze") as mock_fsfreeze:
            error = fakelibvirt.make_libvirtError(
                fakelibvirt.libvirtError,
                "QEMU guest agent is not connected",
                error_code=fakelibvirt.VIR_ERR_AGENT_UNRESPONSIVE)

            mock_fsfreeze.side_effect = error
            mock_fsfreeze.error_code = error.get_error_code()

            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
            instance = objects.Instance(**self.test_instance)
            image_meta = objects.ImageMeta.from_dict(
                {"properties": {"hw_qemu_guest_agent": "yes", }})
            self.assertRaises(exception.InstanceQuiesceFailed,
                drvr._set_quiesced, self.context, instance, image_meta, True)

    def test_create_snapshot_metadata(self):
        base = objects.ImageMeta.from_dict(
            {'disk_format': 'raw'})
        instance_data = {'kernel_id': 'kernel',
                    'project_id': 'prj_id',
                    'ramdisk_id': 'ram_id',
                    'os_type': None}
        instance = objects.Instance(**instance_data)
        img_fmt = 'raw'
        snp_name = 'snapshot_name'
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        ret = drvr._create_snapshot_metadata(base, instance, img_fmt, snp_name)
        expected = {'status': 'active',
                    'name': snp_name,
                    'properties': {
                                   'kernel_id': instance['kernel_id'],
                                   'image_location': 'snapshot',
                                   'image_state': 'available',
                                   'owner_id': instance['project_id'],
                                   'ramdisk_id': instance['ramdisk_id'],
                                   },
                    'disk_format': img_fmt,
                    'container_format': 'bare',
                    }
        self.assertEqual(ret, expected)

        # simulate an instance with os_type field defined
        # disk format equals to ami
        # container format not equals to bare
        instance['os_type'] = 'linux'
        base = objects.ImageMeta.from_dict(
            {'disk_format': 'ami',
             'container_format': 'test_container'})
        expected['properties']['os_type'] = instance['os_type']
        expected['disk_format'] = base.disk_format
        expected['container_format'] = base.container_format
        ret = drvr._create_snapshot_metadata(base, instance, img_fmt, snp_name)
        self.assertEqual(ret, expected)

    def test_get_volume_driver(self):
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        connection_info = {'driver_volume_type': 'fake',
                           'data': {'device_path': '/fake',
                                    'access_mode': 'rw'}}
        driver = conn._get_volume_driver(connection_info)
        result = isinstance(driver, volume_drivers.LibvirtFakeVolumeDriver)
        self.assertTrue(result)

    def test_get_volume_driver_unknown(self):
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        connection_info = {'driver_volume_type': 'unknown',
                           'data': {'device_path': '/fake',
                                    'access_mode': 'rw'}}
        self.assertRaises(
            exception.VolumeDriverNotFound,
            conn._get_volume_driver,
            connection_info
        )

    def _fake_libvirt_config_guest_disk(self):
        fake_config = vconfig.LibvirtConfigGuestDisk()
        fake_config.source_type = "network"
        fake_config.source_device = "fake-type"
        fake_config.driver_name = "qemu"
        fake_config.driver_format = "raw"
        fake_config.driver_cache = "none"
        fake_config.source_protocol = "fake"
        fake_config.source_name = "fake"
        fake_config.target_bus = "fake-bus"
        fake_config.target_dev = "vdb"

        return fake_config

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_sev_enabled',
                       new=mock.Mock(return_value=False))
    @mock.patch.object(volume_drivers.LibvirtFakeVolumeDriver, 'get_config')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_set_cache_mode')
    def test_get_volume_config(self, mock_set_cache_mode, mock_get_config):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {
            'driver_volume_type': 'fake',
            'data': {
                'device_path': '/fake',
                'access_mode': 'rw'
            }
        }
        generated_config = self._fake_libvirt_config_guest_disk()
        mock_get_config.return_value = copy.deepcopy(generated_config)

        returned_config = drvr._get_volume_config(
            instance,
            connection_info,
            mock.sentinel.disk_info)

        mock_get_config.assert_called_once_with(
            connection_info,
            mock.sentinel.disk_info)
        mock_set_cache_mode.assert_called_once_with(returned_config)
        self.assertEqual(generated_config.to_xml(), returned_config.to_xml())

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_sev_enabled',
                       new=mock.Mock(return_value=True))
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_set_cache_mode',
                       new=mock.Mock())
    @mock.patch.object(volume_drivers.LibvirtFakeVolumeDriver, 'get_config')
    def test_get_volume_config_sev(self, mock_get_config):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {
            'driver_volume_type': 'fake',
            'data': {
                'device_path': '/fake',
                'access_mode': 'rw'
            }
        }
        generated_config = self._fake_libvirt_config_guest_disk()
        generated_config.target_bus = 'virtio'
        mock_get_config.return_value = copy.deepcopy(generated_config)

        returned_config = drvr._get_volume_config(
            instance,
            connection_info,
            mock.sentinel.disk_info)

        # Assert that driver_iommu is enabled for this virtio volume
        self.assertTrue(returned_config.driver_iommu)

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_volume_driver')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_attach_encryptor')
    def test_connect_volume_encryption_success(
            self, mock_attach_encryptor, mock_get_volume_driver):

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        mock_volume_driver = mock.MagicMock(
            spec=volume_drivers.LibvirtBaseVolumeDriver)
        mock_get_volume_driver.return_value = mock_volume_driver

        connection_info = {'driver_volume_type': 'fake',
                           'data': {'device_path': '/fake',
                                    'access_mode': 'rw',
                                    'volume_id': uuids.volume_id}}
        encryption = {'provider': encryptors.LUKS,
                      'encryption_key_id': uuids.encryption_key_id}
        instance = mock.sentinel.instance

        drvr._connect_volume(self.context, connection_info, instance,
                             encryption=encryption)

        mock_get_volume_driver.assert_called_once_with(connection_info)
        mock_volume_driver.connect_volume.assert_called_once_with(
            connection_info, instance)
        mock_attach_encryptor.assert_called_once_with(
            self.context, connection_info, encryption)
        mock_volume_driver.disconnect_volume.assert_not_called()

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_volume_driver')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_attach_encryptor')
    def test_connect_volume_encryption_fail(
            self, mock_attach_encryptor, mock_get_volume_driver):

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        mock_volume_driver = mock.MagicMock(
            spec=volume_drivers.LibvirtBaseVolumeDriver)
        mock_get_volume_driver.return_value = mock_volume_driver

        connection_info = {'driver_volume_type': 'fake',
                           'data': {'device_path': '/fake',
                                    'access_mode': 'rw',
                                    'volume_id': uuids.volume_id}}
        encryption = {'provider': encryptors.LUKS,
                      'encryption_key_id': uuids.encryption_key_id}
        instance = mock.sentinel.instance
        mock_attach_encryptor.side_effect = processutils.ProcessExecutionError

        self.assertRaises(processutils.ProcessExecutionError,
                          drvr._connect_volume,
                          self.context, connection_info, instance,
                          encryption=encryption)

        mock_get_volume_driver.assert_called_once_with(connection_info)
        mock_volume_driver.connect_volume.assert_called_once_with(
            connection_info, instance)
        mock_attach_encryptor.assert_called_once_with(
            self.context, connection_info, encryption)
        mock_volume_driver.disconnect_volume.assert_called_once_with(
            connection_info, instance)

    @mock.patch.object(key_manager, 'API')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_volume_encryption')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_allow_native_luksv1')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_volume_encryptor')
    @mock.patch('nova.virt.libvirt.host.Host')
    @mock.patch('os_brick.encryptors.luks.is_luks')
    def test_connect_volume_luks(self, mock_is_volume_luks, mock_host,
            mock_get_volume_encryptor, mock_allow_native_luksv1,
            mock_get_volume_encryption, mock_get_key_mgr):

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        connection_info = {'driver_volume_type': 'fake',
                           'data': {'device_path': '/fake',
                                    'access_mode': 'rw',
                                    'volume_id': uuids.volume_id}}
        encryption = {'provider': encryptors.LUKS,
                      'encryption_key_id': uuids.encryption_key_id}
        instance = mock.sentinel.instance

        # Mock out find_secret so we don't skip ahead
        drvr._host.find_secret.return_value = None

        # Mock out the encryptors
        mock_encryptor = mock.Mock()
        mock_get_volume_encryptor.return_value = mock_encryptor
        mock_is_volume_luks.return_value = True

        # Mock out the key manager
        key = u'3734363537333734'
        key_encoded = binascii.unhexlify(key)
        mock_key = mock.Mock()
        mock_key_mgr = mock.Mock()
        mock_get_key_mgr.return_value = mock_key_mgr
        mock_key_mgr.get.return_value = mock_key
        mock_key.get_encoded.return_value = key_encoded

        # assert that the secret is created for the encrypted volume during
        # _connect_volume when _allow_native_luksv1 is True
        mock_get_volume_encryption.return_value = encryption
        mock_allow_native_luksv1.return_value = True

        drvr._connect_volume(self.context, connection_info, instance,
                             encryption=encryption)
        drvr._host.create_secret.assert_called_once_with('volume',
                    uuids.volume_id, password=key)
        mock_encryptor.attach_volume.assert_not_called()

        # assert that the encryptor is used if is_luks is False
        drvr._host.create_secret.reset_mock()
        mock_get_volume_encryption.reset_mock()
        mock_allow_native_luksv1.return_value = False

        drvr._connect_volume(self.context, connection_info, instance,
                             encryption=encryption)
        drvr._host.create_secret.assert_not_called()
        mock_encryptor.attach_volume.assert_called_once_with(self.context,
                                                             **encryption)

        # assert that we format the volume if it is not already formatted
        mock_allow_native_luksv1.return_value = True
        mock_is_volume_luks.return_value = False

        drvr._connect_volume(self.context, connection_info, instance,
                             encryption=encryption)
        mock_encryptor._format_volume.assert_called_once_with(key,
                                                              **encryption)

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_volume_encryptor')
    def test_disconnect_volume_luks(self, mock_get_volume_encryptor):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._host = mock.Mock()
        drvr._host.find_secret.return_value = mock.Mock()
        connection_info = {'driver_volume_type': 'fake',
                           'data': {'device_path': '/fake',
                                    'access_mode': 'rw',
                                    'volume_id': uuids.volume_id}}
        encryption = {'provider': encryptors.LUKS,
                      'encryption_key_id': uuids.encryption_key_id}
        instance = mock.sentinel.instance

        # Mock out the encryptors
        mock_encryptor = mock.Mock()
        mock_get_volume_encryptor.return_value = mock_encryptor

        # assert that a secret is deleted if found
        drvr._disconnect_volume(self.context, connection_info, instance)
        drvr._host.delete_secret.assert_called_once_with('volume',
                                                         uuids.volume_id)
        mock_encryptor.detach_volume.assert_not_called()

        # assert that no attempt to remove the secret is made when
        # destroy_secrets=False
        drvr._host.find_secret.reset_mock()
        drvr._host.delete_secret.reset_mock()
        drvr._disconnect_volume(
            self.context,
            connection_info,
            instance,
            encryption=encryption,
            destroy_secrets=False
        )
        drvr._host.delete_secret.assert_not_called()

        # assert that the encryptor is used if no secret is found
        drvr._host.find_secret.reset_mock()
        drvr._host.delete_secret.reset_mock()
        drvr._host.find_secret.return_value = None

        drvr._disconnect_volume(self.context, connection_info, instance,
                                encryption=encryption)
        drvr._host.delete_secret.assert_not_called()
        mock_encryptor.detach_volume.assert_called_once_with(**encryption)

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_detach_encryptor')
    @mock.patch('nova.objects.InstanceList.get_uuids_by_host')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_driver')
    @mock.patch('nova.volume.cinder.API.get')
    def test_disconnect_multiattach_single_connection(
            self, mock_volume_get, mock_get_volume_driver,
            mock_get_instances, mock_detach_encryptor):

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        mock_volume_driver = mock.MagicMock(
            spec=volume_drivers.LibvirtBaseVolumeDriver)
        mock_get_volume_driver.return_value = mock_volume_driver

        attachments = (
            [('70ab645f-6ffc-406a-b3d2-5007a0c01b82',
              {'mountpoint': u'/dev/vdb',
               'attachment_id': u'9402c249-99df-4f72-89e7-fd611493ee5d'}),
             ('00803490-f768-4049-aa7d-151f54e6311e',
              {'mountpoint': u'/dev/vdb',
               'attachment_id': u'd6128a7b-19c8-4a3e-8036-011396df95ac'})])

        mock_volume_get.return_value = {
            'attachments': collections.OrderedDict(attachments),
            'multiattach': True,
            'id': 'd30559cf-f092-4693-8589-0d0a1e7d9b1f',
        }

        fake_connection_info = {
            'multiattach': True,
            'volume_id': 'd30559cf-f092-4693-8589-0d0a1e7d9b1f'}
        fake_instance_1 = fake_instance.fake_instance_obj(
            self.context,
            host='fake-host-1')

        mock_get_instances.return_value = (
            ['00803490-f768-4049-aa7d-151f54e6311e'])
        drvr._disconnect_volume(
            self.context, fake_connection_info, fake_instance_1)
        mock_volume_driver.disconnect_volume.assert_called_once_with(
            fake_connection_info, fake_instance_1, force=False)

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_detach_encryptor')
    @mock.patch('nova.objects.InstanceList.get_uuids_by_host')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_driver')
    @mock.patch('nova.volume.cinder.API.get')
    def test_disconnect_multiattach_multi_connection(
            self, mock_volume_get, mock_get_volume_driver,
            mock_get_instances, mock_detach_encryptor):

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        mock_volume_driver = mock.MagicMock(
            spec=volume_drivers.LibvirtBaseVolumeDriver)
        mock_get_volume_driver.return_value = mock_volume_driver

        attachments = (
            [('70ab645f-6ffc-406a-b3d2-5007a0c01b82',
              {'mountpoint': u'/dev/vdb',
               'attachment_id': u'9402c249-99df-4f72-89e7-fd611493ee5d'}),
             ('00803490-f768-4049-aa7d-151f54e6311e',
              {'mountpoint': u'/dev/vdb',
               'attachment_id': u'd6128a7b-19c8-4a3e-8036-011396df95ac'})])

        mock_volume_get.return_value = {
            'attachments': collections.OrderedDict(attachments),
            'multiattach': True,
            'id': 'd30559cf-f092-4693-8589-0d0a1e7d9b1f',
        }

        fake_connection_info = {
            'multiattach': True,
            'volume_id': 'd30559cf-f092-4693-8589-0d0a1e7d9b1f'}
        fake_instance_1 = fake_instance.fake_instance_obj(
            self.context,
            host='fake-host-1')

        mock_get_instances.return_value = (
            ['00803490-f768-4049-aa7d-151f54e6311e',
             '70ab645f-6ffc-406a-b3d2-5007a0c01b82'])
        drvr._disconnect_volume(
            self.context, fake_connection_info, fake_instance_1)
        mock_volume_driver.disconnect_volume.assert_not_called()

    def test_attach_invalid_volume_type(self):
        self.create_fake_libvirt_mock()
        libvirt_driver.LibvirtDriver._conn.lookupByUUIDString \
            = self.fake_lookup
        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertRaises(exception.VolumeDriverNotFound,
                          drvr.attach_volume, None,
                          {"driver_volume_type": "badtype"},
                          instance,
                          "/dev/sda")

    def test_attach_blockio_invalid_hypervisor(self):
        self.flags(virt_type='lxc', group='libvirt')
        self.create_fake_libvirt_mock()
        libvirt_driver.LibvirtDriver._conn.lookupByUUIDString \
            = self.fake_lookup
        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertRaises(exception.InvalidHypervisorType,
                          drvr.attach_volume, None,
                          {"driver_volume_type": "fake",
                           "data": {"logical_block_size": "4096",
                                    "physical_block_size": "4096"}
                          },
                          instance,
                          "/dev/sda")

    def _test_check_discard(self, mock_log, driver_discard=None,
                            bus=None, should_log=False):
        mock_config = mock.Mock()
        mock_config.driver_discard = driver_discard
        mock_config.target_bus = bus
        mock_instance = mock.Mock()
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._check_discard_for_attach_volume(mock_config, mock_instance)
        self.assertEqual(should_log, mock_log.called)

    @mock.patch('nova.virt.libvirt.driver.LOG.debug')
    def test_check_discard_for_attach_volume_no_unmap(self, mock_log):
        self._test_check_discard(mock_log, driver_discard=None,
                                 bus='scsi', should_log=False)

    @mock.patch('nova.virt.libvirt.driver.LOG.debug')
    def test_check_discard_for_attach_volume_blk_controller(self, mock_log):
        self._test_check_discard(mock_log, driver_discard='unmap',
                                 bus='virtio', should_log=True)

    @mock.patch('nova.virt.libvirt.driver.LOG.debug')
    def test_check_discard_for_attach_volume_valid_controller(self, mock_log):
        self._test_check_discard(mock_log, driver_discard='unmap',
                                 bus='scsi', should_log=False)

    @mock.patch('nova.virt.libvirt.driver.LOG.debug')
    def test_check_discard_for_attach_volume_blk_controller_no_unmap(self,
                                                                     mock_log):
        self._test_check_discard(mock_log, driver_discard=None,
                                 bus='virtio', should_log=False)

    @mock.patch('nova.virt.libvirt.blockinfo.get_info_from_bdm')
    def test_attach_volume_with_exception(self, mock_get_info):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {"driver_volume_type": "fake",
                           "data": {"device_path": "/fake",
                                    "access_mode": "rw"}}
        bdm = {'device_name': 'vdb',
               'disk_bus': 'fake-bus',
               'device_type': 'fake-type'}
        disk_info = {'bus': bdm['disk_bus'], 'type': bdm['device_type'],
                     'dev': 'vdb'}
        libvirt_exc = fakelibvirt.make_libvirtError(fakelibvirt.libvirtError,
            "Target vdb already exists', device is busy",
            error_code=fakelibvirt.VIR_ERR_INTERNAL_ERROR)

        with test.nested(
            mock.patch.object(drvr._host, 'get_guest'),
            mock.patch('nova.virt.libvirt.driver.LOG'),
            mock.patch.object(drvr, '_connect_volume'),
            mock.patch.object(drvr, '_disconnect_volume'),
            mock.patch.object(drvr, '_get_volume_config'),
            mock.patch.object(drvr, '_check_discard_for_attach_volume'),
            mock.patch.object(drvr, '_build_device_metadata'),
        ) as (mock_get_guest, mock_log, mock_connect_volume,
              mock_disconnect_volume, mock_get_volume_config,
              mock_check_discard, mock_build_metadata):

            mock_conf = mock.MagicMock()
            mock_guest = mock.MagicMock()
            mock_guest.attach_device.side_effect = libvirt_exc
            mock_get_volume_config.return_value = mock_conf
            mock_get_guest.return_value = mock_guest
            mock_get_info.return_value = disk_info
            mock_build_metadata.return_value = objects.InstanceDeviceMetadata()

            self.assertRaises(fakelibvirt.libvirtError, drvr.attach_volume,
                self.context, connection_info, instance, "/dev/vdb",
                disk_bus=bdm['disk_bus'], device_type=bdm['device_type'])
            mock_log.exception.assert_called_once_with(u'Failed to attach '
                'volume at mountpoint: %s', '/dev/vdb', instance=instance)
            mock_disconnect_volume.assert_called_once()

    @mock.patch('nova.utils.get_image_from_system_metadata')
    @mock.patch('nova.virt.libvirt.blockinfo.get_info_from_bdm')
    @mock.patch('nova.virt.libvirt.host.Host._get_domain')
    def test_attach_volume_with_vir_domain_affect_live_flag(self,
            mock_get_domain, mock_get_info, get_image):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        image_meta = {}
        get_image.return_value = image_meta
        mock_dom = mock.MagicMock()
        mock_get_domain.return_value = mock_dom

        connection_info = {"driver_volume_type": "fake",
                           "data": {"device_path": "/fake",
                                    "access_mode": "rw"}}
        bdm = {'device_name': 'vdb',
               'disk_bus': 'fake-bus',
               'device_type': 'fake-type'}
        disk_info = {'bus': bdm['disk_bus'], 'type': bdm['device_type'],
                     'dev': 'vdb'}
        mock_get_info.return_value = disk_info
        mock_conf = mock.MagicMock()
        flags = (fakelibvirt.VIR_DOMAIN_AFFECT_CONFIG |
                 fakelibvirt.VIR_DOMAIN_AFFECT_LIVE)

        with test.nested(
            mock.patch.object(drvr, '_connect_volume'),
            mock.patch.object(drvr, '_get_volume_config',
                              return_value=mock_conf),
            mock.patch.object(drvr, '_check_discard_for_attach_volume'),
            mock.patch.object(drvr, '_build_device_metadata'),
            mock.patch.object(objects.Instance, 'save')
        ) as (mock_connect_volume, mock_get_volume_config, mock_check_discard,
              mock_build_metadata, mock_save):
            for state in (power_state.RUNNING, power_state.PAUSED):
                mock_dom.info.return_value = [state, 512, 512, 2, 1234, 5678]
                mock_build_metadata.return_value = \
                    objects.InstanceDeviceMetadata()

                drvr.attach_volume(self.context, connection_info, instance,
                                   "/dev/vdb", disk_bus=bdm['disk_bus'],
                                   device_type=bdm['device_type'])

                mock_get_domain.assert_called_with(instance)
                mock_get_info.assert_called_with(
                    instance,
                    CONF.libvirt.virt_type,
                    test.MatchType(objects.ImageMeta),
                    bdm)
                mock_connect_volume.assert_called_with(
                    self.context, connection_info, instance, encryption=None)
                mock_get_volume_config.assert_called_with(
                    instance, connection_info, disk_info)
                mock_dom.attachDeviceFlags.assert_called_with(
                    mock_conf.to_xml(), flags=flags)
                mock_check_discard.assert_called_with(mock_conf, instance)
                mock_build_metadata.assert_called_with(self.context, instance)
                mock_save.assert_called_with()

    @mock.patch('threading.Event', new=mock.Mock())
    @mock.patch('nova.virt.libvirt.host.Host._get_domain')
    def test_detach_volume_with_vir_domain_affect_live_flag(self,
            mock_get_domain, use_alias=True):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        volume_id = uuids.volume
        alias_xml = '<alias name="%s"/>' % vconfig.make_libvirt_device_alias(
            volume_id)
        mock_xml_with_disk = """<domain>
  <devices>
    <disk type='file'>
      %(alias)s
      <source file='/path/to/fake-volume'/>
      <target dev='vdc' bus='virtio'/>
    </disk>
  </devices>
</domain>""" % {'alias': use_alias and alias_xml or ''}
        mock_xml_without_disk = """<domain>
  <devices>
  </devices>
</domain>"""
        mock_dom = mock.MagicMock()

        # Second time don't return anything about disk vdc so it looks removed
        return_list = [
            mock_xml_with_disk,  # persistent domain
            mock_xml_with_disk,  # live domain
            mock_xml_without_disk,  # persistent gone
            mock_xml_without_disk,  # persistent gone (no-alias retry)
            mock_xml_without_disk,  # live gone
            mock_xml_without_disk,  # live gone (no-alias retry)
        ]
        if not use_alias:
            # If we're not using the alias to detach we will end up with two
            # extra checks of the XML with the alias (which will fail to find
            # anything) before we fall back to by-path and succeed.
            return_list = [
                mock_xml_with_disk,  # persistent check for missing alias
                mock_xml_with_disk,   # live check for missing alias
            ] + return_list

        # Doubling the size of return list because we test with two guest power
        # states
        mock_dom.XMLDesc.side_effect = return_list + return_list

        connection_info = {"driver_volume_type": "fake",
                           "data": {"device_path": "/fake",
                                    "volume_id": volume_id,
                                    "access_mode": "rw"}}

        with mock.patch.object(drvr, '_disconnect_volume') as \
                mock_disconnect_volume:
            for state in (power_state.RUNNING, power_state.PAUSED):
                mock_dom.reset_mock()
                mock_dom.info.return_value = [state, 512, 512, 2, 1234, 5678]
                mock_get_domain.return_value = mock_dom
                drvr.detach_volume(
                    self.context, connection_info, instance, '/dev/vdc')

                mock_get_domain.assert_called_with(instance)
                xml = """<disk type="file" device="disk">
                            %(alias)s
                            <source file="/path/to/fake-volume"/>
                            <target bus="virtio" dev="vdc"/>
                        </disk>""" % {'alias': use_alias and alias_xml or ''}
                # we expect two separate detach calls
                self.assertEqual(2, mock_dom.detachDeviceFlags.call_count)
                # one for the persistent domain
                call = mock_dom.detachDeviceFlags.mock_calls[0]
                self.assertXmlEqual(xml, call.args[0])
                self.assertEqual(
                    {"flags": fakelibvirt.VIR_DOMAIN_AFFECT_CONFIG},
                    call.kwargs)
                # and one for the live domain
                call = mock_dom.detachDeviceFlags.mock_calls[1]
                self.assertXmlEqual(xml, call.args[0])
                self.assertEqual(
                    {"flags": fakelibvirt.VIR_DOMAIN_AFFECT_LIVE},
                    call.kwargs)

                mock_disconnect_volume.assert_called_with(
                    self.context, connection_info, instance, encryption=None)

    def test_detach_volume_with_vir_domain_affect_live_flag_legacy(self):
        self.test_detach_volume_with_vir_domain_affect_live_flag(
            use_alias=False)

    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._disconnect_volume')
    @mock.patch('nova.virt.libvirt.host.Host._get_domain')
    def test_detach_volume_disk_not_found(self, mock_get_domain,
                                          mock_disconnect_volume):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        mock_xml_without_disk = """<domain>
  <devices>
  </devices>
</domain>"""
        mock_dom = mock.MagicMock(return_value=mock_xml_without_disk)

        connection_info = {"driver_volume_type": "fake",
                           "data": {"device_path": "/fake",
                                    "access_mode": "rw"}}

        mock_dom.info.return_value = [power_state.RUNNING, 512, 512, 2, 1234,
                                      5678]
        mock_get_domain.return_value = mock_dom

        drvr.detach_volume(
            self.context, connection_info, instance, '/dev/vdc')

        mock_get_domain.assert_called_once_with(instance)
        mock_disconnect_volume.assert_called_once_with(
            self.context, connection_info, instance, encryption=None)

    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_encryptor')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._disconnect_volume')
    @mock.patch('nova.virt.libvirt.host.Host._get_domain')
    def test_detach_volume_disk_not_found_encryption(self, mock_get_domain,
                                                     mock_disconnect_volume,
                                                     mock_get_encryptor):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        mock_xml_without_disk = """<domain>
  <devices>
  </devices>
</domain>"""
        mock_dom = mock.MagicMock(return_value=mock_xml_without_disk)
        encryption = mock.MagicMock()

        connection_info = {"driver_volume_type": "fake",
                           "data": {"device_path": "/fake",
                                    "access_mode": "rw"}}

        mock_dom.info.return_value = [power_state.RUNNING, 512, 512, 2, 1234,
                                      5678]
        mock_get_domain.return_value = mock_dom

        drvr.detach_volume(self.context, connection_info, instance,
                           '/dev/vdc', encryption)

        mock_disconnect_volume.assert_called_once_with(
            self.context, connection_info, instance, encryption=encryption)

    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.'
                '_detach_with_retry')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_driver')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_encryptor')
    @mock.patch('nova.virt.libvirt.host.Host.get_guest')
    def test_detach_volume_order_with_encryptors(self, mock_get_guest,
        mock_get_encryptor, mock_get_volume_driver, mock_detach_with_retry
    ):

        mock_volume_driver = mock.MagicMock(
            spec=volume_drivers.LibvirtBaseVolumeDriver)
        mock_get_volume_driver.return_value = mock_volume_driver
        mock_guest = mock.MagicMock(spec=libvirt_guest.Guest)
        mock_guest.get_power_state.return_value = power_state.RUNNING
        mock_get_guest.return_value = mock_guest
        mock_encryptor = mock.MagicMock(
                spec=encryptors.nop.NoOpEncryptor)
        mock_get_encryptor.return_value = mock_encryptor

        mock_order = mock.Mock()
        mock_order.attach_mock(mock_volume_driver.disconnect_volume,
                'disconnect_volume')
        mock_order.attach_mock(mock_detach_with_retry,
                'detach_volume')
        mock_order.attach_mock(mock_encryptor.detach_volume,
                'detach_encryptor')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {"driver_volume_type": "fake",
                           "data": {"device_path": "/fake",
                                    "access_mode": "rw"}}
        encryption = {"provider": "NoOpEncryptor"}
        drvr.detach_volume(
            self.context, connection_info, instance, '/dev/vdc',
            encryption=encryption)

        mock_order.assert_has_calls([
            mock.call.detach_volume(
                mock_guest,
                instance.uuid,
                # it is functools.partial(driver._get_guest_disk_device, 'vdc')
                # see assert below
                mock.ANY,
                device_name='vdc',
            ),
            mock.call.detach_encryptor(**encryption),
            mock.call.disconnect_volume(
                connection_info,
                instance,
                force=False,
            )
        ])
        get_device_conf_func = mock_detach_with_retry.mock_calls[0][1][2]
        self.assertEqual(drvr._get_guest_disk_device,
                         get_device_conf_func.func)
        self.assertEqual((mock_get_guest.return_value, 'vdc'),
                         get_device_conf_func.args)

    def test_extend_volume(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {
            'driver_volume_type': 'fake',
            'data': {
                'volume_id': uuids.volume_id,
                'device_path': '/fake',
                'access_mode': 'rw'
            }
        }

        new_size = 20 * units.Gi

        guest = mock.Mock(spec=libvirt_guest.Guest)
        # block_device
        block_device = mock.Mock(spec=libvirt_guest.BlockDevice,
                                 _disk=mock.sentinel.disk)

        block_device.resize = mock.Mock()
        guest.get_block_device = mock.Mock(return_value=block_device)
        drvr._host.get_guest = mock.Mock(return_value=guest)
        drvr._extend_volume = mock.Mock(return_value=new_size)

        for state in (power_state.RUNNING, power_state.PAUSED):
            guest.get_power_state = mock.Mock(return_value=state)
            drvr.extend_volume(
                self.context, connection_info, instance, new_size)
            drvr._extend_volume.assert_called_with(connection_info,
                                                   instance,
                                                   new_size)
            guest.get_block_device.assert_called_with('/fake')
            block_device.resize.assert_called_with(new_size)

    def test_extend_volume_with_volume_driver_without_support(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {
            'driver_volume_type': 'fake',
            'serial': uuids.volume_id,
            'data': {
                'volume_id': uuids.volume_id,
                'device_path': '/fake',
                'access_mode': 'rw'
            }
        }

        with mock.patch.object(drvr, '_extend_volume',
                               side_effect=NotImplementedError()):
            self.assertRaises(exception.ExtendVolumeNotSupported,
                              drvr.extend_volume, self.context,
                              connection_info, instance, 0)

    def test_extend_volume_with_new_size_none(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {
            'driver_volume_type': 'fake',
            'serial': uuids.volume_id,
            'data': {
                'volume_id': uuids.volume_id,
                'device_path': '/fake',
                'access_mode': 'rw'
            }
        }
        with mock.patch.object(drvr, '_extend_volume', return_value=None):
            self.assertRaises(
                exception.VolumeExtendFailed,
                drvr.extend_volume,
                self.context,
                connection_info,
                instance,
                0
            )

    def test_extend_volume_disk_not_found(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {
            'driver_volume_type': 'fake',
            'serial': uuids.volume_id,
            'data': {
                'volume_id': uuids.volume_id,
                'device_path': '/fake',
                'access_mode': 'rw'
            }
        }
        new_size = 20 * units.Gi

        xml_no_disk = "<domain><devices></devices></domain>"
        dom = fakelibvirt.Domain(drvr._get_connection(), xml_no_disk, False)
        guest = libvirt_guest.Guest(dom)
        guest.get_power_state = mock.Mock(return_value=power_state.RUNNING)
        drvr._host.get_guest = mock.Mock(return_value=guest)
        drvr._extend_volume = mock.Mock(return_value=new_size)
        drvr.extend_volume(self.context, connection_info, instance, new_size)

    def test_extend_volume_with_instance_not_found(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {
            'driver_volume_type': 'fake',
            'serial': uuids.volume_id,
            'data': {
                'volume_id': uuids.volume_id,
                'device_path': '/fake',
                'access_mode': 'rw'
            }
        }

        with test.nested(
            mock.patch.object(host.Host, '_get_domain',
                              side_effect=exception.InstanceNotFound(
                                  instance_id=instance.uuid)),
            mock.patch.object(drvr, '_extend_volume')
        ) as (_get_domain, _extend_volume):
            self.assertRaises(exception.InstanceNotFound,
                              drvr.extend_volume, self.context,
                              connection_info, instance, 0)

    def test_extend_volume_with_libvirt_error(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {
            'driver_volume_type': 'fake',
            'serial': uuids.volume_id,
            'data': {
                'volume_id': uuids.volume_id,
                'device_path': '/fake',
                'access_mode': 'rw'
            }
        }

        new_size = 20 * units.Gi

        guest = mock.Mock(spec=libvirt_guest.Guest)
        guest.get_power_state = mock.Mock(return_value=power_state.RUNNING)
        # block_device
        block_device = mock.Mock(spec=libvirt_guest.BlockDevice,
                                 _disk=mock.sentinel.disk)
        block_device.resize = mock.Mock(
            side_effect=fakelibvirt.libvirtError('ERR'))
        guest.get_block_device = mock.Mock(return_value=block_device)
        drvr._host.get_guest = mock.Mock(return_value=guest)
        drvr._extend_volume = mock.Mock(return_value=new_size)

        self.assertRaises(fakelibvirt.libvirtError,
                          drvr.extend_volume, self.context,
                          connection_info, instance, new_size)

    def test_extend_volume_with_no_device_path_attribute(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {
            'serial': uuids.volume_id,
            'driver_volume_type': 'fake',
            'data': {
                'cluster_name': 'fake',
                'auth_enabled': False,
                'volume_id': uuids.volume_id,
                'access_mode': 'rw'
            }
        }
        new_size = 20 * units.Gi

        guest = mock.Mock(spec=libvirt_guest.Guest)
        # block_device
        block_device = mock.Mock(spec=libvirt_guest.BlockDevice,
                                 _disk=mock.sentinel.disk)

        block_device.resize = mock.Mock()
        disk = mock.Mock(
            spec=vconfig.LibvirtConfigGuestDisk,
            serial=uuids.volume_id,
            target_dev='vdb')
        guest.get_block_device = mock.Mock(return_value=block_device)
        guest.get_all_disks = mock.Mock(return_value=[disk])
        drvr._host.get_guest = mock.Mock(return_value=guest)
        drvr._extend_volume = mock.Mock(return_value=new_size)

        for state in (power_state.RUNNING, power_state.PAUSED):
            guest.get_power_state = mock.Mock(return_value=state)
            drvr.extend_volume(self.context, connection_info, instance,
                               new_size)
            drvr._extend_volume.assert_called_with(connection_info, instance,
                                                   new_size)
            guest.get_block_device.assert_called_with('vdb')
            block_device.resize.assert_called_with(new_size)

    def test_extend_volume_no_disk_found_by_serial(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {
            'serial': uuids.volume_id,
            'driver_volume_type': 'fake',
            'data': {
                'cluster_name': 'fake',
                'auth_enabled': False,
                'volume_id': uuids.volume_id,
                'access_mode': 'rw'
            }
        }
        new_size = 20 * units.Gi

        guest = mock.Mock(spec=libvirt_guest.Guest)
        # block_device
        block_device = mock.Mock(spec=libvirt_guest.BlockDevice,
                                 _disk=mock.sentinel.disk)
        block_device.resize = mock.Mock()
        disk = mock.Mock(
            spec=vconfig.LibvirtConfigGuestDisk,
            serial=uuids.different_volume_id,
            target_dev='vdb')
        guest.get_block_device = mock.Mock(return_value=block_device)
        guest.get_all_disks = mock.Mock(return_value=[disk])
        drvr._host.get_guest = mock.Mock(return_value=guest)
        drvr._extend_volume = mock.Mock(return_value=new_size)
        guest.get_power_state = mock.Mock(return_value=power_state.RUNNING)

        self.assertRaises(
            exception.VolumeNotFound,
            drvr.extend_volume,
            self.context,
            connection_info,
            instance,
            new_size
        )

    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    def test_extend_volume_luksv1_unknown_path(self,
                                               mock_get_encryption_metadata):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {
            'serial': uuids.volume_id,
            'driver_volume_type': 'fake',
            'data': {
                'access_mode': 'rw'
            }
        }

        disk_1 = mock.Mock(spec=vconfig.LibvirtConfigGuestDisk,
                           serial=uuids.volume_id,
                           target_dev=mock.sentinel.disk_1_target_dev)
        block_device = mock.Mock(spec=libvirt_guest.BlockDevice,
                                 _disk=mock.sentinel.disk)
        guest = mock.Mock(spec=libvirt_guest.Guest)
        guest.get_block_device.return_value = block_device
        guest.get_power_state.return_value = power_state.RUNNING
        guest.get_all_disks.return_value = [disk_1]

        # The requested_size is provided to extend_volume in bytes.
        new_size = 20 * units.Gi

        drvr._host.get_guest = mock.Mock(return_value=guest)
        drvr._extend_volume = mock.Mock(return_value=new_size)

        mock_get_encryption_metadata.return_value = {
            'provider': 'luks',
            'control_location': 'front-end'}

        # Assert that DiskNotFound is raised
        self.assertRaises(exception.DiskNotFound, drvr.extend_volume,
                          self.context, connection_info, instance, new_size)

        # Assert that resize is not called
        block_device.resize.assert_not_called()

    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    @mock.patch('nova.virt.images.privileged_qemu_img_info')
    def test_extend_volume_luksv1_DiskNotFound(self, mock_qemu_img_info,
                                               mock_get_encryption_metadata):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {
            'serial': uuids.volume_id,
            'driver_volume_type': 'fake',
            'data': {
                'device_path': mock.sentinel.device_path,
                'access_mode': 'rw'
            }
        }

        block_device = mock.Mock(spec=libvirt_guest.BlockDevice,
                                 _disk=mock.sentinel.disk)
        guest = mock.Mock(spec=libvirt_guest.Guest)
        guest.get_block_device.return_value = block_device
        guest.get_power_state.return_value = power_state.RUNNING

        # The requested_size is provided to extend_volume in bytes.
        new_size = 20 * units.Gi

        drvr._host.get_guest = mock.Mock(return_value=guest)
        drvr._extend_volume = mock.Mock(return_value=new_size)

        mock_qemu_img_info.side_effect = exception.DiskNotFound(
            location=mock.sentinel.device_path)
        mock_get_encryption_metadata.return_value = {
            'provider': 'luks',
            'control_location': 'front-end'}

        # Assert that DiskNotFound is raised
        self.assertRaises(exception.DiskNotFound, drvr.extend_volume,
                          self.context, connection_info, instance, new_size)

        # Assert that resize is not called
        block_device.resize.assert_not_called()

    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    @mock.patch('nova.virt.images.privileged_qemu_img_info')
    def test_extend_volume_luksv1_block(self, mock_qemu_img_info,
                                        mock_get_encryption_metadata):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {
            'serial': uuids.volume_id,
            'driver_volume_type': 'fake',
            'data': {
                'device_path': mock.sentinel.device_path,
                'access_mode': 'rw'
            }
        }

        block_device = mock.Mock(spec=libvirt_guest.BlockDevice,
                                 _disk=mock.sentinel.disk)
        guest = mock.Mock(spec=libvirt_guest.Guest)
        guest.get_block_device.return_value = block_device
        guest.get_power_state.return_value = power_state.RUNNING

        # The requested_size is provided to extend_volume in bytes.
        new_size = 20 * units.Gi
        # The LUKSv1 payload offset as reported by qemu-img info in bytes.
        payload_offset = 2048 * units.Ki
        # The new size is provided to Libvirt virDomainBlockResize
        new_size_minus_offset = new_size - payload_offset

        drvr._host.get_guest = mock.Mock(return_value=guest)
        drvr._extend_volume = mock.Mock(return_value=new_size)

        mock_qemu_img_info.return_value = mock.Mock(
            format_specific={'data': {'payload-offset': payload_offset}})
        mock_get_encryption_metadata.return_value = {
            'provider': 'luks',
            'control_location': 'front-end'}

        # Extend the volume to new_size
        drvr.extend_volume(self.context, connection_info, instance, new_size)

        # Assert that the expected calls are made prior to the device resize.
        drvr._host.get_guest.assert_called_once_with(instance)
        guest.get_power_state.assert_called_once_with(drvr._host)
        guest.get_block_device(mock.sentinel.device_path)

        mock_get_encryption_metadata.assert_called_once_with(
            self.context, drvr._volume_api, uuids.volume_id, connection_info)
        mock_qemu_img_info.assert_called_once_with(mock.sentinel.device_path)

        # Assert that the Libvirt call to resize the device within the instance
        # is called with the LUKSv1 payload offset taken into account.
        block_device.resize.assert_called_once_with(new_size_minus_offset)

    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    @mock.patch('nova.virt.images.privileged_qemu_img_info')
    def test_extend_volume_luksv1_rbd(self, mock_qemu_img_info,
                                        mock_get_encryption_metadata):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {
            'serial': uuids.volume_id,
            'driver_volume_type': 'rbd',
            'data': {
                'name': 'pool/volume',
                'auth_enabled': 'true',
                'auth_username': 'username',
                'access_mode': 'rw'
            }
        }
        disk_1 = mock.Mock(spec=vconfig.LibvirtConfigGuestDisk,
                           serial=uuids.volume_id,
                           target_dev=mock.sentinel.disk_1_target_dev)
        disk_2 = mock.Mock(spec=vconfig.LibvirtConfigGuestDisk,
                           serial=uuids.extra_volume_id,
                           target_dev=mock.sentinel.disk_2_target_dev)
        block_device = mock.Mock(spec=libvirt_guest.BlockDevice,
                                 _disk=mock.sentinel.disk_1_target_dev)
        guest = mock.Mock(spec=libvirt_guest.Guest)
        guest.get_block_device.return_value = block_device
        guest.get_power_state.return_value = power_state.RUNNING
        guest.get_all_disks.return_value = [disk_1, disk_2]

        # The requested_size is provided to extend_volume in bytes.
        new_size = 20 * units.Gi
        # The LUKSv1 payload offset as reported by qemu-img info in bytes.
        payload_offset = 2048 * units.Ki
        # The new size is provided to Libvirt virDomainBlockResize
        new_size_minus_offset = new_size - payload_offset

        drvr._host.get_guest = mock.Mock(return_value=guest)
        drvr._extend_volume = mock.Mock(return_value=new_size)

        mock_qemu_img_info.return_value = mock.Mock(
            format_specific={'data': {'payload-offset': payload_offset}})
        mock_get_encryption_metadata.return_value = {
            'provider': 'luks',
            'control_location': 'front-end'}

        # Extend the volume to new_size
        drvr.extend_volume(self.context, connection_info, instance, new_size)

        # Assert that the expected calls are made prior to the device resize.
        drvr._host.get_guest.assert_called_once_with(instance)
        guest.get_power_state.assert_called_once_with(drvr._host)
        guest.get_block_device(mock.sentinel.disk_1_target_dev)

        mock_get_encryption_metadata.assert_called_once_with(
            self.context, drvr._volume_api, uuids.volume_id, connection_info)
        mock_qemu_img_info.assert_called_once_with(
            'rbd:pool/volume:id=username')

        # Assert that the Libvirt call to resize the device within the instance
        # is called with the LUKSv1 payload offset taken into account.
        block_device.resize.assert_called_once_with(new_size_minus_offset)

    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_encryptor')
    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    def test_extend_volume_os_brick_block(self, mock_get_encryption_metadata,
                                          mock_get_encryptor):
        """Test extend volume that uses an os-brick encryptor."""
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        connection_info = {
            'serial': uuids.volume_id,
            'driver_volume_type': 'fake',
            'data': {
                'device_path': mock.sentinel.device_path,
                'access_mode': 'rw'
            }
        }

        block_device = mock.Mock(spec=libvirt_guest.BlockDevice,
                                 _disk=mock.sentinel.disk)
        guest = mock.Mock(spec=libvirt_guest.Guest)
        guest.get_block_device.return_value = block_device
        guest.get_power_state.return_value = power_state.RUNNING

        # The requested_size is provided to extend_volume in bytes.
        new_size = 20 * units.Gi
        # Decrypted volume size reported by os-brick will be smaller
        new_size_minus_offset = new_size - (16384 * units.Ki)

        mock_brick_extend = mock_get_encryptor.return_value.extend_volume
        mock_brick_extend.return_value = new_size_minus_offset

        drvr._host.get_guest = mock.Mock(return_value=guest)
        drvr._extend_volume = mock.Mock(return_value=new_size)

        encryption = {'provider': 'luks2', 'control_location': 'front-end'}
        mock_get_encryption_metadata.return_value = encryption

        # Extend the volume to new_size
        drvr.extend_volume(self.context, connection_info, instance, new_size)

        # Assert that the expected calls are made prior to the device resize.
        drvr._host.get_guest.assert_called_once_with(instance)
        guest.get_power_state.assert_called_once_with(drvr._host)
        guest.get_block_device(mock.sentinel.device_path)

        # Assert calls to the os-brick encryptor extend
        mock_get_encryptor.assert_called_once_with(connection_info, encryption)
        mock_brick_extend.assert_called_once_with(self.context, **encryption)

        mock_get_encryption_metadata.assert_called_once_with(
            self.context, drvr._volume_api, uuids.volume_id, connection_info)

        # Assert that the Libvirt call to resize the device within the instance
        # is called with the size reported by os-brick
        block_device.resize.assert_called_once_with(new_size_minus_offset)

    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_encryptor')
    def test_use_encryptor_connection_info_incomplete(self,
            mock_get_encryptor, mock_get_metadata):
        """Assert no attach attempt is made given incomplete connection_info.
        """
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        connection_info = {'data': {}}

        drvr._attach_encryptor(self.context, connection_info, None)

        mock_get_metadata.assert_not_called()
        mock_get_encryptor.assert_not_called()

    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_encryptor')
    def test_attach_encryptor_unencrypted_volume_meta_missing(self,
            mock_get_encryptor, mock_get_metadata):
        """Assert that if not provided encryption metadata is fetched even
        if the volume is ultimately unencrypted and no attempt to attach
        is made.
        """
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        encryption = {}
        connection_info = {'data': {'volume_id': uuids.volume_id}}
        mock_get_metadata.return_value = encryption

        drvr._attach_encryptor(self.context, connection_info, None)

        mock_get_metadata.assert_called_once_with(self.context,
                drvr._volume_api, uuids.volume_id, connection_info)
        mock_get_encryptor.assert_not_called()

    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_encryptor')
    def test_attach_encryptor_unencrypted_volume_meta_provided(self,
            mock_get_encryptor, mock_get_metadata):
        """Assert that if an empty encryption metadata dict is provided that
        there is no additional attempt to lookup the metadata or attach the
        encryptor.
        """
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        encryption = {}
        connection_info = {'data': {'volume_id': uuids.volume_id}}

        drvr._attach_encryptor(self.context, connection_info, encryption)

        mock_get_metadata.assert_not_called()
        mock_get_encryptor.assert_not_called()

    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_encryptor')
    def test_attach_encryptor_encrypted_volume_meta_missing(self,
            mock_get_encryptor, mock_get_metadata):
        """Assert that if missing the encryption metadata of an encrypted
        volume is fetched and then used to attach the encryptor for the volume.
        """
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        mock_encryptor = mock.MagicMock()
        mock_get_encryptor.return_value = mock_encryptor
        encryption = {'provider': encryptors.PLAIN,
                      'control_location': 'front-end'}
        mock_get_metadata.return_value = encryption
        connection_info = {'data': {'volume_id': uuids.volume_id}}

        drvr._attach_encryptor(self.context, connection_info, None)

        mock_get_metadata.assert_called_once_with(self.context,
                drvr._volume_api, uuids.volume_id, connection_info)
        mock_get_encryptor.assert_called_once_with(connection_info,
                                                   encryption)
        mock_encryptor.attach_volume.assert_called_once_with(self.context,
                                                             **encryption)

    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_encryptor')
    def test_attach_encryptor_encrypted_volume_meta_provided(self,
            mock_get_encryptor, mock_get_metadata):
        """Assert that when provided there are no further attempts to fetch the
        encryption metadata for the volume and that the provided metadata is
        then used to attach the volume.
        """
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        mock_encryptor = mock.MagicMock()
        mock_get_encryptor.return_value = mock_encryptor
        encryption = {'provider': encryptors.PLAIN,
                      'control_location': 'front-end'}
        connection_info = {'data': {'volume_id': uuids.volume_id}}

        drvr._attach_encryptor(self.context, connection_info, encryption)

        mock_get_metadata.assert_not_called()
        mock_get_encryptor.assert_called_once_with(connection_info,
                                                   encryption)
        mock_encryptor.attach_volume.assert_called_once_with(self.context,
                                                             **encryption)

    @mock.patch.object(key_manager, 'API')
    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_encryptor')
    def test_attach_encryptor_encrypted_native_luks_serial(self,
            mock_get_encryptor, mock_get_metadata, mock_get_key_mgr):
        """Uses native luks encryption with a provider encryptor and the
        connection_info has a serial but not volume_id in the 'data'
        sub-dict.
        """
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        mock_encryptor = mock.MagicMock()
        mock_get_encryptor.return_value = mock_encryptor
        encryption = {'provider': 'luks', 'control_location': 'front-end',
                      'encryption_key_id': uuids.encryption_key_id}
        connection_info = {'serial': uuids.serial, 'data': {}}
        # Mock out the key manager
        key = u'3734363537333734'
        key_encoded = binascii.unhexlify(key)
        mock_key = mock.Mock()
        mock_key_mgr = mock.Mock()
        mock_get_key_mgr.return_value = mock_key_mgr
        mock_key_mgr.get.return_value = mock_key
        mock_key.get_encoded.return_value = key_encoded

        with mock.patch.object(drvr, '_allow_native_luksv1',
                               return_value=True):
            with mock.patch.object(drvr._host, 'create_secret') as crt_scrt:
                drvr._attach_encryptor(self.context, connection_info,
                                       encryption)

        mock_get_metadata.assert_not_called()
        mock_get_encryptor.assert_not_called()
        crt_scrt.assert_called_once_with(
            'volume', uuids.serial, password=key)

    @mock.patch.object(key_manager, 'API')
    def test_attach_encryptor_secret_exists(self, mock_key_manager_api):
        connection_info = {'data': {'volume_id': uuids.volume_id}}
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        with test.nested(
            mock.patch.object(drvr, '_get_volume_encryption'),
            mock.patch.object(drvr._host, 'find_secret')
        ) as (mock_get_volume_encryption, mock_find_secret):
            drvr._attach_encryptor(self.context, connection_info, None)

        # Assert we called find_secret and nothing else
        mock_find_secret.assert_called_once_with('volume', uuids.volume_id)
        mock_get_volume_encryption.assert_not_called()
        mock_key_manager_api.assert_not_called()

    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_encryptor')
    def test_detach_encryptor_connection_info_incomplete(self,
            mock_get_encryptor, mock_get_metadata):
        """Assert no detach attempt is made given incomplete connection_info.
        """
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        connection_info = {'data': {}}

        drvr._detach_encryptor(self.context, connection_info, None)

        mock_get_metadata.assert_not_called()
        mock_get_encryptor.assert_not_called()

    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_encryptor')
    def test_detach_encryptor_unencrypted_volume_meta_missing(self,
            mock_get_encryptor, mock_get_metadata):
        """Assert that if not provided encryption metadata is fetched even
        if the volume is ultimately unencrypted and no attempt to detach
        is made.
        """
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        encryption = {}
        connection_info = {'data': {'volume_id': uuids.volume_id}}
        mock_get_metadata.return_value = encryption

        drvr._detach_encryptor(self.context, connection_info, None)

        mock_get_metadata.assert_called_once_with(self.context,
                drvr._volume_api, uuids.volume_id, connection_info)
        mock_get_encryptor.assert_not_called()

    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_encryptor')
    def test_detach_encryptor_unencrypted_volume_meta_provided(self,
            mock_get_encryptor, mock_get_metadata):
        """Assert that if an empty encryption metadata dict is provided that
        there is no additional attempt to lookup the metadata or detach the
        encryptor.
        """
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        encryption = {}
        connection_info = {'data': {'volume_id': uuids.volume_id}}

        drvr._detach_encryptor(self.context, connection_info, encryption)

        mock_get_metadata.assert_not_called()
        mock_get_encryptor.assert_not_called()

    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_encryptor')
    def test_detach_encryptor_encrypted_volume_meta_missing(
        self, mock_get_encryptor, mock_get_metadata
    ):
        """Assert that if missing the encryption metadata of an encrypted
        volume is fetched and then used to detach the encryptor for the volume.
        """
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        mock_encryptor = mock.MagicMock()
        mock_get_encryptor.return_value = mock_encryptor
        encryption = {'provider': 'luks', 'control_location': 'front-end'}
        mock_get_metadata.return_value = encryption
        connection_info = {
            'data': {
                'device_path': '/dev/foo',
                'volume_id': uuids.volume_id
            }
        }

        drvr._detach_encryptor(self.context, connection_info, None)

        mock_get_metadata.assert_called_once_with(self.context,
                drvr._volume_api, uuids.volume_id, connection_info)
        mock_get_encryptor.assert_called_once_with(connection_info,
                                                   encryption)
        mock_encryptor.detach_volume.assert_called_once_with(**encryption)

    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_encryptor')
    def test_detach_encryptor_encrypted_volume_meta_provided(
        self,
        mock_get_encryptor,
        mock_get_metadata
    ):
        """Assert that when provided there are no further attempts to fetch the
        encryption metadata for the volume and that the provided metadata is
        then used to detach the volume.
        """
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        mock_encryptor = mock.MagicMock()
        mock_get_encryptor.return_value = mock_encryptor
        encryption = {'provider': 'luks', 'control_location': 'front-end'}
        connection_info = {
            'data': {
                'device_path': '/dev/foo',
                'volume_id': uuids.volume_id
            }
        }

        drvr._detach_encryptor(self.context, connection_info, encryption)

        mock_get_metadata.assert_not_called()
        mock_get_encryptor.assert_called_once_with(connection_info,
                                                   encryption)
        mock_encryptor.detach_volume.assert_called_once_with(**encryption)

    @mock.patch('nova.virt.libvirt.host.Host.find_secret')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_encryptor')
    def test_detach_encryptor_native_luks_device_path_secret_missing(
        self, mock_get_encryptor, mock_find_secret
    ):
        """Assert that the encryptor is not built when the associated volume
           secret is missing and device_path is also missing from the
           connection_info.
        """
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        encryption = {'provider': 'luks', 'control_location': 'front-end',
                      'encryption_key_id': uuids.encryption_key_id}
        connection_info = {'data': {'volume_id': uuids.volume_id}}
        mock_find_secret.return_value = False

        drvr._detach_encryptor(self.context, connection_info, encryption)

        mock_find_secret.assert_called_once_with('volume', uuids.volume_id)
        mock_get_encryptor.assert_not_called()

    @mock.patch('nova.virt.libvirt.host.Host.delete_secret')
    @mock.patch('nova.virt.libvirt.host.Host.find_secret', new=mock.Mock())
    def test_detach_encryptor_skip_secret_removal(self, mock_delete_secret):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._detach_encryptor(
            self.context,
            {
                'data': {
                    'volume_id': uuids.volume_id
                }
            },
            None,
            destroy_secrets=False
        )
        # Assert that no attempt is made to delete the volume secert
        mock_delete_secret.assert_not_called()

        drvr._detach_encryptor(
            self.context,
            {
                'data': {
                    'volume_id': uuids.volume_id
                }
            },
            None,
            destroy_secrets=True
        )
        # Assert that volume secert is deleted
        mock_delete_secret.assert_called_once_with('volume', uuids.volume_id)

    def test_allow_native_luksv1(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertFalse(drvr._allow_native_luksv1({}))
        self.assertFalse(drvr._allow_native_luksv1({
            'provider': 'nova.volume.encryptors.cryptsetup.CryptSetupEncryptor'
        }))
        self.assertFalse(drvr._allow_native_luksv1({
            'provider': 'CryptSetupEncryptor'}))
        self.assertFalse(drvr._allow_native_luksv1({
            'provider': encryptors.PLAIN}))
        self.assertTrue(drvr._allow_native_luksv1({
            'provider': 'nova.volume.encryptors.luks.LuksEncryptor'}))
        self.assertTrue(drvr._allow_native_luksv1({
            'provider': 'LuksEncryptor'}))
        self.assertTrue(drvr._allow_native_luksv1({
            'provider': encryptors.LUKS}))

    def test_multi_nic(self):
        network_info = _fake_network_info(self, 2)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        xml = drvr._get_guest_xml(self.context, instance_ref,
                                  network_info, disk_info,
                                  image_meta)
        tree = etree.fromstring(xml)
        interfaces = tree.findall("./devices/interface")
        self.assertEqual(len(interfaces), 2)
        self.assertEqual(interfaces[0].get('type'), 'bridge')

    def _check_xml_and_container(self, instance):
        instance_ref = objects.Instance(**instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        self.flags(virt_type='lxc', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        self.assertEqual(drvr._uri(), 'lxc:///')

        network_info = _fake_network_info(self)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        xml = drvr._get_guest_xml(self.context, instance_ref,
                                  network_info, disk_info,
                                  image_meta)
        tree = etree.fromstring(xml)

        check = [
            (lambda t: t.find('.').get('type'), 'lxc'),
            (lambda t: t.find('./os/type').text, 'exe'),
            (lambda t: t.find("./os/initenv[@name='product_name']").text,
                    'OpenStack Nova'),
            (lambda t: t.find('./devices/filesystem/target').get('dir'), '/')]

        for i, (check, expected_result) in enumerate(check):
            self.assertEqual(check(tree),
                             expected_result,
                             '%s failed common check %d' % (xml, i))

        target = tree.find('./devices/filesystem/source').get('dir')
        self.assertGreater(len(target), 0)

    def _check_xml_and_disk_prefix(self, instance, prefix):
        instance_ref = objects.Instance(**instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        def _get_prefix(p, default):
            if p:
                return p + 'a'
            return default

        type_disk_map = {
            'qemu': [
                (lambda t: t.find('.').get('type'), 'qemu'),
                (lambda t: t.find('./devices/disk/target').get('dev'),
                 _get_prefix(prefix, 'vda'))],
            'kvm': [
                (lambda t: t.find('.').get('type'), 'kvm'),
                (lambda t: t.find('./devices/disk/target').get('dev'),
                 _get_prefix(prefix, 'vda'))],
            }

        for (virt_type, checks) in type_disk_map.items():
            self.flags(virt_type=virt_type, group='libvirt')
            if prefix:
                self.flags(disk_prefix=prefix, group='libvirt')
            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

            network_info = _fake_network_info(self)
            disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                                instance_ref,
                                                image_meta)

            xml = drvr._get_guest_xml(self.context, instance_ref,
                                      network_info, disk_info,
                                      image_meta)
            tree = etree.fromstring(xml)

            for i, (check, expected_result) in enumerate(checks):
                self.assertEqual(check(tree),
                                 expected_result,
                                 '%s != %s failed check %d' %
                                 (check(tree), expected_result, i))

    def _check_xml_and_disk_driver(self, image_meta):
        os_open = os.open
        directio_supported = True

        def os_open_stub(path, flags, *args, **kwargs):
            if flags & os.O_DIRECT:
                if not directio_supported:
                    raise OSError(errno.EINVAL,
                                  '%s: %s' % (os.strerror(errno.EINVAL), path))
                flags &= ~os.O_DIRECT
            return os_open(path, flags, *args, **kwargs)

        self.stub_out('os.open', os_open_stub)

        def connection_supports_direct_io_stub(dirpath):
            return directio_supported

        self.stub_out('nova.privsep.utils.supports_direct_io',
                      connection_supports_direct_io_stub)

        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        network_info = _fake_network_info(self)

        drv = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        xml = drv._get_guest_xml(self.context, instance_ref,
                                 network_info, disk_info, image_meta)
        tree = etree.fromstring(xml)
        disks = tree.findall('./devices/disk/driver')
        for guest_disk in disks:
            self.assertEqual(guest_disk.get("cache"), "none")

        directio_supported = False

        # The O_DIRECT availability is cached on first use in
        # LibvirtDriver, hence we re-create it here
        drv = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        xml = drv._get_guest_xml(self.context, instance_ref,
                                 network_info, disk_info, image_meta)
        tree = etree.fromstring(xml)
        disks = tree.findall('./devices/disk/driver')
        for guest_disk in disks:
            self.assertEqual(guest_disk.get("cache"), "writeback")

    def _check_xml_and_disk_bus(self, image_meta,
                                block_device_info, wantConfig):
        instance_ref = objects.Instance(**self.test_instance)
        network_info = _fake_network_info(self)

        drv = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta,
                                            block_device_info)

        xml = drv._get_guest_xml(self.context, instance_ref,
                                 network_info, disk_info, image_meta,
                                 block_device_info=block_device_info)
        tree = etree.fromstring(xml)

        got_disks = tree.findall('./devices/disk')
        got_disk_targets = tree.findall('./devices/disk/target')
        for i in range(len(wantConfig)):
            want_device_type = wantConfig[i][0]
            want_device_bus = wantConfig[i][1]
            want_device_dev = wantConfig[i][2]

            got_device_type = got_disks[i].get('device')
            got_device_bus = got_disk_targets[i].get('bus')
            got_device_dev = got_disk_targets[i].get('dev')

            self.assertEqual(got_device_type, want_device_type)
            self.assertEqual(got_device_bus, want_device_bus)
            self.assertEqual(got_device_dev, want_device_dev)

    def _check_xml_and_uuid(self, image_meta):
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        network_info = _fake_network_info(self)

        drv = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)
        xml = drv._get_guest_xml(self.context, instance_ref,
                                 network_info, disk_info, image_meta)
        tree = etree.fromstring(xml)
        self.assertEqual(tree.find('./uuid').text,
                         instance_ref['uuid'])

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       "_get_host_sysinfo_serial_hardware",)
    def _check_xml_and_uri(self, instance, mock_serial,
                           expect_ramdisk=False, expect_kernel=False,
                           rescue=None):
        mock_serial.return_value = "cef19ce0-0ca2-11df-855d-b19fbce37686"
        instance_ref = objects.Instance(**instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        type_uri_map = {'qemu': ('qemu:///system',
                             [(lambda t: t.find('.').get('type'), 'qemu'),
                              (lambda t: t.find('./os/type').text,
                               fields.VMMode.HVM),
                              (lambda t: t.find('./devices/emulator'), None)]),
                        'kvm': ('qemu:///system',
                             [(lambda t: t.find('.').get('type'), 'kvm'),
                              (lambda t: t.find('./os/type').text,
                               fields.VMMode.HVM),
                              (lambda t: t.find('./devices/emulator'), None)]),
        }

        hypervisors_to_check = ['qemu', 'kvm']

        for hypervisor_type in hypervisors_to_check:
            check_list = type_uri_map[hypervisor_type][1]

            if rescue:
                suffix = '.rescue'
            else:
                suffix = ''
            if expect_kernel:
                check = (lambda t: self.relpath(t.find('./os/kernel').text).
                         split('/')[1], 'kernel' + suffix)
            else:
                check = (lambda t: t.find('./os/kernel'), None)
            check_list.append(check)

            if expect_kernel:
                check = (lambda t: "no_timer_check" in t.find('./os/cmdline').
                         text, hypervisor_type == "qemu")
                check_list.append(check)
            # Hypervisors that only support vm_mode.HVM should not produce
            # configuration that results in kernel arguments
            if not expect_kernel and (hypervisor_type in
                                      ['qemu', 'kvm']):
                check = (lambda t: t.find('./os/root'), None)
                check_list.append(check)
                check = (lambda t: t.find('./os/cmdline'), None)
                check_list.append(check)

            if expect_ramdisk:
                check = (lambda t: self.relpath(t.find('./os/initrd').text).
                         split('/')[1], 'ramdisk' + suffix)
            else:
                check = (lambda t: t.find('./os/initrd'), None)
            check_list.append(check)

            if hypervisor_type in ['qemu', 'kvm']:
                xpath = "./sysinfo/system/entry"
                check = (lambda t: t.findall(xpath)[0].get("name"),
                         "manufacturer")
                check_list.append(check)
                check = (lambda t: t.findall(xpath)[0].text,
                         version.vendor_string())
                check_list.append(check)

                check = (lambda t: t.findall(xpath)[1].get("name"),
                         "product")
                check_list.append(check)
                check = (lambda t: t.findall(xpath)[1].text,
                         version.product_string())
                check_list.append(check)

                check = (lambda t: t.findall(xpath)[2].get("name"),
                         "version")
                check_list.append(check)
                # NOTE(sirp): empty strings don't roundtrip in lxml (they are
                # converted to None), so we need an `or ''` to correct for that
                check = (lambda t: t.findall(xpath)[2].text or '',
                         version.version_string_with_package())
                check_list.append(check)

                check = (lambda t: t.findall(xpath)[3].get("name"),
                         "serial")
                check_list.append(check)
                check = (lambda t: t.findall(xpath)[3].text,
                         "cef19ce0-0ca2-11df-855d-b19fbce37686")
                check_list.append(check)

                check = (lambda t: t.findall(xpath)[4].get("name"),
                         "uuid")
                check_list.append(check)
                check = (lambda t: t.findall(xpath)[4].text,
                         instance['uuid'])
                check_list.append(check)

            if hypervisor_type in ['qemu', 'kvm']:
                check = (lambda t: t.findall('./devices/serial')[0].get(
                        'type'), 'pty')
                check_list.append(check)
            else:
                check = (lambda t: t.find('./devices/console').get(
                        'type'), 'pty')
                check_list.append(check)

        common_checks = [
            (lambda t: t.find('.').tag, 'domain'),
            (lambda t: t.find('./memory').text, '2097152')]
        if rescue:
            common_checks += [
                (lambda t: self.relpath(t.findall('./devices/disk/source')[0].
                    get('file')).split('/')[1], 'disk.rescue'),
                (lambda t: self.relpath(t.findall('./devices/disk/source')[1].
                    get('file')).split('/')[1], 'disk')]
        else:
            common_checks += [(lambda t: self.relpath(t.findall(
                './devices/disk/source')[0].get('file')).split('/')[1],
                               'disk')]
            common_checks += [(lambda t: self.relpath(t.findall(
                './devices/disk/source')[1].get('file')).split('/')[1],
                               'disk.local')]

        for virt_type in hypervisors_to_check:
            expected_uri = type_uri_map[virt_type][0]
            checks = type_uri_map[virt_type][1]
            self.flags(virt_type=virt_type, group='libvirt')

            with mock.patch('nova.virt.libvirt.driver.libvirt') as old_virt:
                del old_virt.VIR_CONNECT_BASELINE_CPU_EXPAND_FEATURES

                drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

                self.assertEqual(drvr._uri(), expected_uri)

                network_info = _fake_network_info(self)
                disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                                    instance_ref,
                                                    image_meta,
                                                    rescue=rescue)

                xml = drvr._get_guest_xml(self.context, instance_ref,
                                          network_info, disk_info,
                                          image_meta,
                                          rescue=rescue)
                tree = etree.fromstring(xml)
                for i, (check, expected_result) in enumerate(checks):
                    self.assertEqual(check(tree),
                                     expected_result,
                                     '%s != %s failed check %d' %
                                     (check(tree), expected_result, i))

                for i, (check, expected_result) in enumerate(common_checks):
                    self.assertEqual(check(tree),
                                     expected_result,
                                     '%s != %s failed common check %d' %
                                     (check(tree), expected_result, i))

        # This test is supposed to make sure we don't
        # override a specifically set uri
        #
        # Deliberately not just assigning this string to CONF.connection_uri
        # and checking against that later on. This way we make sure the
        # implementation doesn't fiddle around with the CONF.
        testuri = 'something completely different'
        self.flags(connection_uri=testuri, group='libvirt')
        for virt_type in type_uri_map:
            self.flags(virt_type=virt_type, group='libvirt')
            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
            self.assertEqual(drvr._uri(), testuri)

    @mock.patch(
        'nova.network.neutron.API.has_port_binding_extension',
        new=mock.Mock(return_value=False))
    @mock.patch.object(libvirt_driver.LibvirtDriver,
        '_create_shared_storage_test_file')
    @mock.patch.object(fakelibvirt.Connection, 'compareHypervisorCPU')
    def test_check_can_live_migrate_dest_all_pass_with_block_migration(
        self, mock_cpu, mock_test_file,
    ):
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.vcpu_model = test_vcpu_model.fake_vcpumodel
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        compute_info = {'disk_available_least': 400,
                        'cpu_info': 'asdf',
                        }
        filename = "file"

        # _check_cpu_match
        mock_cpu.return_value = 1

        # mounted_on_same_shared_storage
        mock_test_file.return_value = filename

        # No need for the src_compute_info
        return_value = drvr.check_can_live_migrate_destination(self.context,
                instance_ref, None, compute_info, True)
        return_value.is_volume_backed = False
        self.assertEqual({'filename': 'file',
                          'image_type': 'default',
                          'disk_available_mb': 409600,
                          'disk_over_commit': False,
                          'block_migration': True,
                          'is_volume_backed': False,
                          'dst_wants_file_backed_memory': False,
                          'graphics_listen_addr_spice': '127.0.0.1',
                          'graphics_listen_addr_vnc': '127.0.0.1',
                          'serial_listen_addr': None},
                         return_value.obj_to_primitive()['nova_object.data'])

    @mock.patch(
        'nova.network.neutron.API.has_port_binding_extension',
        new=mock.Mock(return_value=False))
    @mock.patch.object(libvirt_driver.LibvirtDriver,
        '_create_shared_storage_test_file')
    @mock.patch.object(fakelibvirt.Connection, 'compareHypervisorCPU')
    def test_check_can_live_migrate_dest_all_pass_with_over_commit(
        self, mock_cpu, mock_test_file,
    ):
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.vcpu_model = test_vcpu_model.fake_vcpumodel
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        compute_info = {'disk_available_least': -1000,
                        'free_disk_gb': 50,
                        'cpu_info': 'asdf',
                        }
        filename = "file"

        # _check_cpu_match
        mock_cpu.return_value = 1

        # mounted_on_same_shared_storage
        mock_test_file.return_value = filename

        # No need for the src_compute_info
        return_value = drvr.check_can_live_migrate_destination(self.context,
                instance_ref, None, compute_info, True, True)
        return_value.is_volume_backed = False
        self.assertEqual({'filename': 'file',
                          'image_type': 'default',
                          'disk_available_mb': 51200,
                          'disk_over_commit': True,
                          'block_migration': True,
                          'is_volume_backed': False,
                          'dst_wants_file_backed_memory': False,
                          'graphics_listen_addr_spice': '127.0.0.1',
                          'graphics_listen_addr_vnc': '127.0.0.1',
                          'serial_listen_addr': None},
                         return_value.obj_to_primitive()['nova_object.data'])

    @mock.patch(
        'nova.network.neutron.API.has_port_binding_extension',
        new=mock.Mock(return_value=False))
    @mock.patch.object(libvirt_driver.LibvirtDriver,
        '_create_shared_storage_test_file')
    @mock.patch.object(fakelibvirt.Connection, 'compareHypervisorCPU')
    def test_check_can_live_migrate_dest_all_pass_no_block_migration(
        self, mock_cpu, mock_test_file,
    ):
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.vcpu_model = test_vcpu_model.fake_vcpumodel
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        compute_info = {'disk_available_least': 400,
                        'cpu_info': 'asdf',
                        }
        filename = "file"

        # _check_cpu_match
        mock_cpu.return_value = 1
        # mounted_on_same_shared_storage
        mock_test_file.return_value = filename
        # No need for the src_compute_info
        return_value = drvr.check_can_live_migrate_destination(self.context,
                instance_ref, None, compute_info, False)
        return_value.is_volume_backed = False
        self.assertEqual({'filename': 'file',
                          'image_type': 'default',
                          'block_migration': False,
                          'disk_over_commit': False,
                          'disk_available_mb': 409600,
                          'is_volume_backed': False,
                          'dst_wants_file_backed_memory': False,
                          'graphics_listen_addr_spice': '127.0.0.1',
                          'graphics_listen_addr_vnc': '127.0.0.1',
                          'serial_listen_addr': None},
                         return_value.obj_to_primitive()['nova_object.data'])

    @mock.patch(
        'nova.network.neutron.API.has_port_binding_extension',
        new=mock.Mock(return_value=False))
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_create_shared_storage_test_file',
                       return_value='fake')
    @mock.patch.object(fakelibvirt.Connection, 'compareHypervisorCPU')
    def test_check_can_live_migrate_dest_fills_listen_addrs(
        self, mock_cpu, mock_test_file,
    ):
        # Tests that check_can_live_migrate_destination returns the listen
        # addresses required by check_can_live_migrate_source.
        self.flags(server_listen='192.0.2.12', group='vnc')
        self.flags(server_listen='198.51.100.34', group='spice')
        self.flags(proxyclient_address='203.0.113.56', group='serial_console')
        self.flags(enabled=True, group='serial_console')
        mock_cpu.return_value = 1

        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.vcpu_model = test_vcpu_model.fake_vcpumodel
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        compute_info = {'cpu_info': 'asdf', 'disk_available_least': 1}
        result = drvr.check_can_live_migrate_destination(
            self.context, instance_ref, compute_info, compute_info)

        self.assertEqual('192.0.2.12',
                         str(result.graphics_listen_addr_vnc))
        self.assertEqual('198.51.100.34',
                         str(result.graphics_listen_addr_spice))
        self.assertEqual('203.0.113.56',
                         str(result.serial_listen_addr))

    @mock.patch(
        'nova.network.neutron.API.has_port_binding_extension',
        new=mock.Mock(return_value=False))
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_create_shared_storage_test_file',
                       return_value='fake')
    @mock.patch.object(fakelibvirt.Connection, 'compareHypervisorCPU',
                       return_value=1)
    def test_check_can_live_migrate_dest_ensure_serial_adds_not_set(
        self, mock_cpu, mock_test_file,
    ):
        self.flags(proxyclient_address='127.0.0.1', group='serial_console')
        self.flags(enabled=False, group='serial_console')
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.vcpu_model = test_vcpu_model.fake_vcpumodel
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        compute_info = {'cpu_info': 'asdf', 'disk_available_least': 1}
        result = drvr.check_can_live_migrate_destination(
            self.context, instance_ref, compute_info, compute_info)
        self.assertIsNone(result.serial_listen_addr)

    @mock.patch(
        'nova.network.neutron.API.has_port_binding_extension',
        new=mock.Mock(return_value=False))
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_create_shared_storage_test_file',
                       return_value='fake')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_compare_cpu')
    def test_check_can_live_migrate_guest_cpu_none_model(
        self, mock_cpu, mock_test_file,
    ):
        # Tests that when instance.vcpu_model.model is None, the host cpu
        # model is used for live migration.
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.vcpu_model = test_vcpu_model.fake_vcpumodel
        instance_ref.vcpu_model.model = None
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        compute_info = {'cpu_info': 'asdf', 'disk_available_least': 1}
        result = drvr.check_can_live_migrate_destination(
            self.context, instance_ref, compute_info, compute_info)
        result.is_volume_backed = False
        mock_cpu.assert_called_once_with(None, 'asdf', instance_ref)
        self.assertEqual({'filename': 'fake',
                          'image_type': CONF.libvirt.images_type,
                          'block_migration': False,
                          'disk_over_commit': False,
                          'disk_available_mb': 1024,
                          'is_volume_backed': False,
                          'dst_wants_file_backed_memory': False,
                          'graphics_listen_addr_spice': '127.0.0.1',
                          'graphics_listen_addr_vnc': '127.0.0.1',
                          'serial_listen_addr': None},
                         result.obj_to_primitive()['nova_object.data'])

    @mock.patch(
        'nova.network.neutron.API.has_port_binding_extension',
        new=mock.Mock(return_value=False))
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_create_shared_storage_test_file',
                       return_value='fake')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_compare_cpu')
    def test_check_can_live_migrate_guest_cpu_none_model_skip_compare(
            self, mock_cpu, mock_test_file):
        self.flags(group='workarounds', skip_cpu_compare_on_dest=True)
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.vcpu_model = test_vcpu_model.fake_vcpumodel
        instance_ref.vcpu_model.model = None
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        compute_info = {'cpu_info': 'asdf', 'disk_available_least': 1}
        drvr.check_can_live_migrate_destination(
            self.context, instance_ref, compute_info, compute_info)
        mock_cpu.assert_not_called()

    @mock.patch(
        'nova.network.neutron.API.has_port_binding_extension',
        new=mock.Mock(return_value=False))
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_create_shared_storage_test_file',
                       return_value='fake')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_compare_cpu')
    def test_check_can_live_migrate_dest_numa_lm(
        self, mock_cpu, mock_test_file,
    ):
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.numa_topology = objects.InstanceNUMATopology(
            cells=[objects.InstanceNUMACell()])
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        compute_info = {'cpu_info': 'asdf', 'disk_available_least': 1}
        result = drvr.check_can_live_migrate_destination(
            self.context, instance_ref, compute_info, compute_info)
        self.assertTrue(result.dst_supports_numa_live_migration)

    @mock.patch(
        'nova.network.neutron.API.has_port_binding_extension',
        new=mock.Mock(return_value=False))
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_create_shared_storage_test_file',
                       return_value='fake')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_compare_cpu')
    def test_check_can_live_migrate_dest_numa_lm_no_instance_numa(
        self, mock_cpu, mock_test_file,
    ):
        instance_ref = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        compute_info = {'cpu_info': 'asdf', 'disk_available_least': 1}
        result = drvr.check_can_live_migrate_destination(
            self.context, instance_ref, compute_info, compute_info)
        self.assertNotIn('dst_supports_numa_live_migration', result)

    @mock.patch(
        'nova.network.neutron.API.has_port_binding_extension',
        new=mock.Mock(return_value=False))
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_host_can_support_mdev_live_migration')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_create_shared_storage_test_file',
                       return_value='fake')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_compare_cpu')
    def test_check_can_live_migrate_dest_mdev_lm(
        self, mock_cpu, mock_test_file, mock_can_sup_mdev_lm,
    ):
        mock_can_sup_mdev_lm.return_value = True
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.numa_topology = objects.InstanceNUMATopology(
            cells=[objects.InstanceNUMACell()])
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        compute_info = {'cpu_info': 'asdf', 'disk_available_least': 1}
        result = drvr.check_can_live_migrate_destination(
            self.context, instance_ref, compute_info, compute_info)
        self.assertTrue(result.dst_supports_mdev_live_migration)

    @mock.patch(
        'nova.network.neutron.API.has_port_binding_extension',
        new=mock.Mock(return_value=False))
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_host_can_support_mdev_live_migration')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_create_shared_storage_test_file',
                       return_value='fake')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_compare_cpu')
    def test_check_can_live_migrate_dest_mdev_lm_no_host_support(
        self, mock_cpu, mock_test_file, mock_can_sup_mdev_lm,
    ):
        mock_can_sup_mdev_lm.return_value = False
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.numa_topology = objects.InstanceNUMATopology(
            cells=[objects.InstanceNUMACell()])
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        compute_info = {'cpu_info': 'asdf', 'disk_available_least': 1}
        result = drvr.check_can_live_migrate_destination(
            self.context, instance_ref, compute_info, compute_info)
        self.assertNotIn('dst_supports_mdev_live_migration', result)

    @mock.patch(
        'nova.network.neutron.API.has_port_binding_extension',
        new=mock.Mock(return_value=False))
    @mock.patch.object(libvirt_driver.LibvirtDriver,
        '_create_shared_storage_test_file')
    @mock.patch.object(fakelibvirt.Connection, 'compareHypervisorCPU')
    def test_check_can_live_migrate_dest_no_instance_cpu_info(
        self, mock_cpu, mock_test_file,
    ):
        instance_ref = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        compute_info = {'cpu_info': jsonutils.dumps({
            "vendor": "AMD",
            "arch": fields.Architecture.I686,
            "features": ["sse3"],
            "model": "Opteron_G3",
            "topology": {"cores": 2, "threads": 1, "sockets": 4}
        }), 'disk_available_least': 1}
        filename = "file"

        # _check_cpu_match
        mock_cpu.return_value = 1
        # mounted_on_same_shared_storage
        mock_test_file.return_value = filename

        return_value = drvr.check_can_live_migrate_destination(self.context,
                instance_ref, compute_info, compute_info, False)
        # NOTE(danms): Compute manager would have set this, so set it here
        return_value.is_volume_backed = False
        self.assertEqual({'filename': 'file',
                          'image_type': 'default',
                          'block_migration': False,
                          'disk_over_commit': False,
                          'disk_available_mb': 1024,
                          'is_volume_backed': False,
                          'dst_wants_file_backed_memory': False,
                          'graphics_listen_addr_spice': '127.0.0.1',
                          'graphics_listen_addr_vnc': '127.0.0.1',
                          'serial_listen_addr': None},
                         return_value.obj_to_primitive()['nova_object.data'])

    @mock.patch(
        'nova.network.neutron.API.has_port_binding_extension',
        new=mock.Mock(return_value=False))
    @mock.patch.object(libvirt_driver.LibvirtDriver,
        '_create_shared_storage_test_file')
    @mock.patch.object(fakelibvirt.Connection, 'compareHypervisorCPU')
    def test_check_can_live_migrate_dest_file_backed(
        self, mock_cpu, mock_test_file,
    ):
        self.flags(file_backed_memory=1024, group='libvirt')

        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.vcpu_model = test_vcpu_model.fake_vcpumodel

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        compute_info = {'disk_available_least': 400,
                        'cpu_info': 'asdf',
                        }

        filename = "file"

        # _check_cpu_match
        mock_cpu.return_value = 1
        # mounted_on_same_shared_storage
        mock_test_file.return_value = filename
        # No need for the src_compute_info
        return_value = drvr.check_can_live_migrate_destination(self.context,
                instance_ref, None, compute_info, False)

        self.assertTrue(return_value.dst_wants_file_backed_memory)

    @mock.patch.object(fakelibvirt.Connection, 'compareHypervisorCPU')
    def test_check_can_live_migrate_dest_incompatible_cpu_raises(
            self, mock_cpu):
        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.vcpu_model = test_vcpu_model.fake_vcpumodel
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        compute_info = {'cpu_info': 'asdf', 'disk_available_least': 1}

        mock_cpu.side_effect = exception.InvalidCPUInfo(reason='foo')
        self.assertRaises(exception.MigrationPreCheckError,
                          drvr.check_can_live_migrate_destination,
                          self.context, instance_ref,
                          compute_info, compute_info, False)

    @mock.patch(
        'nova.network.neutron.API.has_port_binding_extension',
        new=mock.Mock(return_value=True))
    @mock.patch.object(
        libvirt_driver.LibvirtDriver,
        '_create_shared_storage_test_file',
        return_value='fake')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_compare_cpu')
    def test_check_can_live_migrate_dest_create_dummy_vif(
        self, mock_cpu, mock_test_file,
    ):
        instance_ref = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        compute_info = {'cpu_info': 'asdf', 'disk_available_least': 1}
        result = drvr.check_can_live_migrate_destination(
            self.context, instance_ref, compute_info, compute_info)
        self.assertIn('vifs', result)
        self.assertIsInstance(result.vifs, list)

        for vif in result.vifs:
            self.assertTrue(vif.supports_os_vif_delegation)

    def test_check_source_migrate_data_at_dest_no_src_mdevs(self):
        """Noop test in case no mdevs from the source."""
        instance_ref = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        dest_check_data = objects.LibvirtLiveMigrateData(
            filename="file",
            block_migration=True,
            disk_over_commit=False,
            disk_available_mb=1024)
        allocations = {uuids.rp1: {'resources': {orc.VGPU: 1}}}
        result = drvr.check_source_migrate_data_at_dest(
            self.context, instance_ref, dest_check_data, None, None,
            allocations)
        self.assertEqual(dest_check_data, result)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_get_supported_vgpu_types')
    def test_check_source_migrate_data_at_dest_types_not_supported(self,
                                                                   mock_get):
        """Raises an exception if the destination doesn't support the source
        mdev types.
        """
        mock_get.return_value = ['type1']
        instance_ref = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        dest_check_data = objects.LibvirtLiveMigrateData(
            filename="file",
            block_migration=True,
            disk_over_commit=False,
            disk_available_mb=1024,
            source_mdev_types={uuids.mdev1: 'wrongtype'})
        allocations = {uuids.rp1: {'resources': {orc.VGPU: 1}}}
        self.assertRaises(exception.MigrationPreCheckError,
                          drvr.check_source_migrate_data_at_dest,
                          self.context, instance_ref, dest_check_data,
                          None, None, allocations)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_get_mdev_types_from_uuids')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_allocate_mdevs')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_get_supported_vgpu_types')
    def test_check_source_migrate_data_at_dest_types_missing_mdevs(self,
                                                                   mock_get,
                                                                   mock_alloc,
                                                                   mock_gmtfu):
        """Raises an exception if the destination doesn't allocate all the
        necessary mediated devices.
        """
        mock_get.return_value = ['type1']
        mock_alloc.return_value = [uuids.dst_mdev1]
        mock_gmtfu.return_value = {uuids.dst_mdev1: 'type1'}
        instance_ref = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertEqual({}, drvr.instance_claimed_mdevs)
        dest_check_data = objects.LibvirtLiveMigrateData(
            filename="file",
            block_migration=True,
            disk_over_commit=False,
            disk_available_mb=1024,
            src_supports_mdev_live_migration=True,
            source_mdev_types={uuids.src_mdev1: 'type1',
                               uuids.src_mdev2: 'type1'})
        allocations = {uuids.rp1: {'resources': {orc.VGPU: 2}}}
        self.assertRaises(exception.MigrationPreCheckError,
                          drvr.check_source_migrate_data_at_dest,
                          self.context, instance_ref, dest_check_data,
                          None, None, allocations)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_get_mdev_types_from_uuids')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_allocate_mdevs')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_get_supported_vgpu_types')
    def test_check_source_migrate_data_at_dest_types_claim_mdevs(self,
                                                                 mock_get,
                                                                 mock_alloc,
                                                                 mock_gmtfu):
        mock_get.return_value = ['type1']
        mock_alloc.return_value = [uuids.dst_mdev1]
        mock_gmtfu.return_value = {uuids.dst_mdev1: 'type1'}
        instance_ref = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertEqual({}, drvr.instance_claimed_mdevs)
        dest_check_data = objects.LibvirtLiveMigrateData(
            filename="file",
            block_migration=True,
            disk_over_commit=False,
            disk_available_mb=1024,
            src_supports_mdev_live_migration=True,
            source_mdev_types={uuids.src_mdev1: 'type1'})
        allocations = {uuids.rp1: {'resources': {orc.VGPU: 1}}}
        drvr.check_source_migrate_data_at_dest(
            self.context, instance_ref, dest_check_data, None, None,
            allocations)
        mock_alloc.assert_called_once_with(allocations)
        mock_gmtfu.assert_called_once_with([uuids.dst_mdev1])
        self.assertEqual({uuids.src_mdev1: uuids.dst_mdev1},
                          dest_check_data.target_mdevs)
        self.assertEqual({instance_ref.uuid: [uuids.dst_mdev1]},
                         drvr.instance_claimed_mdevs)

    @mock.patch.object(fakelibvirt.Connection, 'getLibVersion')
    @mock.patch.object(fakelibvirt.Connection, 'getVersion')
    def _test_host_can_support_mdev_lm(self, mock_getversion,
                                      mock_getlibversion,
                                      old_libvirt, old_qemu, expected):
        min_libvirt_ver = versionutils.convert_version_to_int(
            libvirt_driver.MIN_MDEV_LIVEMIG_LIBVIRT_VERSION)
        min_qemu_ver = versionutils.convert_version_to_int(
            libvirt_driver.MIN_MDEV_LIVEMIG_QEMU_VERSION)

        mock_getversion.return_value = (min_qemu_ver - 1 if old_qemu
                                        else min_qemu_ver)
        mock_getlibversion.return_value = (min_libvirt_ver - 1 if old_libvirt
                                        else min_libvirt_ver)
        driver = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        self.assertEqual(expected,
                         driver._host_can_support_mdev_live_migration())

    def test_host_can_support_mdev_lm(self):
        self._test_host_can_support_mdev_lm(old_libvirt=False, old_qemu=False,
                                            expected=True)

    def test_host_can_support_mdev_lm_old_libvirt(self):
        self._test_host_can_support_mdev_lm(old_libvirt=True, old_qemu=False,
                                            expected=False)

    def test_host_can_support_mdev_lm_old_qemu(self):
        self._test_host_can_support_mdev_lm(old_libvirt=False, old_qemu=True,
                                            expected=False)

    def test_host_can_support_mdev_lm_both_old(self):
        self._test_host_can_support_mdev_lm(old_libvirt=True, old_qemu=True,
                                            expected=False)

    @mock.patch.object(host.Host, 'compare_hypervisor_cpu')
    @mock.patch.object(nova.virt.libvirt, 'config')
    def test_compare_cpu_compatible_host_cpu(self, mock_vconfig, mock_compare):
        instance = objects.Instance(**self.test_instance)
        mock_compare.return_value = 5
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        ret = conn._compare_cpu(None, jsonutils.dumps(_fake_cpu_info),
                instance)
        self.assertIsNone(ret)

    @mock.patch.object(host.Host, 'compare_hypervisor_cpu')
    @mock.patch.object(nova.virt.libvirt, 'config')
    def test_compare_cpu_handles_not_supported_error_gracefully(self,
                                                                mock_vconfig,
                                                                mock_compare):
        instance = objects.Instance(**self.test_instance)
        not_supported_exc = fakelibvirt.make_libvirtError(
                fakelibvirt.libvirtError,
                'this function is not supported by the connection driver:'
                ' virCompareCPU',
                error_code=fakelibvirt.VIR_ERR_NO_SUPPORT)
        mock_compare.side_effect = not_supported_exc
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        ret = conn._compare_cpu(None, jsonutils.dumps(_fake_cpu_info),
                instance)
        self.assertIsNone(ret)

    @mock.patch.object(host.Host, 'get_capabilities')
    @mock.patch.object(host.Host, 'compare_hypervisor_cpu')
    def test_compare_cpu_host_aarch64(self,
                                      mock_compare,
                                      mock_caps):
        instance = objects.Instance(**self.test_instance)
        mock_compare.return_value = 6
        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.AARCH64
        caps.host.topology = fakelibvirt.NUMATopology()

        mock_caps.return_value = caps
        self.mock_uname.return_value = fakelibvirt.os_uname(
            'Linux', '', '5.4.0-0-generic', '', fields.Architecture.AARCH64)
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        ret = conn._compare_cpu(None,
                                jsonutils.dumps(_fake_cpu_info_aarch64),
                                instance)
        mock_compare.assert_called_once_with(caps.host.cpu.to_xml())
        self.assertIsNone(ret)

    @mock.patch.object(host.Host, 'compare_hypervisor_cpu')
    @mock.patch.object(nova.virt.libvirt.LibvirtDriver,
                       '_vcpu_model_to_cpu_config')
    def test_compare_cpu_compatible_guest_cpu(self, mock_vcpu_to_cpu,
                                              mock_compare):
        instance = objects.Instance(**self.test_instance)
        mock_compare.return_value = 6
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        ret = conn._compare_cpu(jsonutils.dumps(_fake_cpu_info), None,
                instance)
        self.assertIsNone(ret)

    def test_compare_cpu_virt_type_qemu(self):
        instance = objects.Instance(**self.test_instance)
        self.flags(virt_type='qemu', group='libvirt')
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        ret = conn._compare_cpu(None, None, instance)
        self.assertIsNone(ret)

    @mock.patch.object(host.Host, 'compare_hypervisor_cpu')
    @mock.patch.object(nova.virt.libvirt, 'config')
    def test_compare_cpu_invalid_cpuinfo_raises(self, mock_vconfig,
                                                mock_compare):
        instance = objects.Instance(**self.test_instance)
        mock_compare.return_value = 0
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertRaises(exception.InvalidCPUInfo,
                          conn._compare_cpu, None,
                          jsonutils.dumps(_fake_cpu_info),
                          instance)

    @mock.patch.object(host.Host, 'compare_hypervisor_cpu')
    @mock.patch.object(nova.virt.libvirt, 'config')
    def test_compare_cpu_incompatible_cpu_raises(self, mock_vconfig,
                                                 mock_compare):
        instance = objects.Instance(**self.test_instance)
        mock_compare.side_effect = fakelibvirt.libvirtError('cpu')
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertRaises(exception.InvalidCPUInfo,
                          conn._compare_cpu, None,
                          jsonutils.dumps(_fake_cpu_info),
                          instance)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_cleanup_shared_storage_test_file')
    def test_check_can_live_migrate_dest_cleanup_works_correctly(
            self, mock_clean):
        objects.Instance(**self.test_instance)
        dest_check_data = objects.LibvirtLiveMigrateData(
            filename="file",
            block_migration=True,
            disk_over_commit=False,
            disk_available_mb=1024)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        drvr.cleanup_live_migration_destination_check(self.context,
                                                      dest_check_data)
        mock_clean.assert_called_once_with('file')

    @mock.patch('os.path.exists', return_value=True)
    @mock.patch('os.utime')
    def test_check_shared_storage_test_file_exists(self, mock_utime,
                                                   mock_path_exists):
        tmpfile_path = os.path.join(CONF.instances_path, 'tmp123')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertTrue(drvr._check_shared_storage_test_file(
            'tmp123', mock.sentinel.instance))
        mock_utime.assert_called_once_with(CONF.instances_path, None)
        mock_path_exists.assert_called_once_with(tmpfile_path)

    @mock.patch('os.path.exists', return_value=False)
    @mock.patch('os.utime')
    def test_check_shared_storage_test_file_does_not_exist(self, mock_utime,
                                                   mock_path_exists):
        tmpfile_path = os.path.join(CONF.instances_path, 'tmp123')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertFalse(drvr._check_shared_storage_test_file(
            'tmp123', mock.sentinel.instance))
        mock_utime.assert_called_once_with(CONF.instances_path, None)
        mock_path_exists.assert_called_once_with(tmpfile_path)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_get_mdev_types_from_uuids')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_assert_source_can_live_migrate_mdevs')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_get_all_assigned_mediated_devices')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_check_shared_storage_test_file')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_is_shared_block_storage')
    def _test_can_live_migrate_source(self, mock_is_shared, mock_check_shared,
                                      mock_get_all_assigned_mdevs,
                                      mock_assert_src_can_lv_mdevs,
                                      mock_get_types,
                                      block_migration=False,
                                      is_shared_block_storage=False,
                                      is_shared_instance_path=False,
                                      disk_available_mb=1024,
                                      exception=None,
                                      numa_lm=True,
                                      instance_mdevs=False,
                                      assert_can_lv_mdevs=True):
        instance = objects.Instance(**self.test_instance)
        if numa_lm:
            instance.numa_topology = objects.InstanceNUMATopology(cells=[
                objects.InstanceNUMACell()])
        dest_check_data = objects.LibvirtLiveMigrateData(
            filename='file',
            image_type='default',
            block_migration=block_migration,
            disk_over_commit=False,
            disk_available_mb=disk_available_mb)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        mock_is_shared.return_value = is_shared_block_storage
        mock_check_shared.return_value = is_shared_instance_path

        if not instance_mdevs:
            mock_get_all_assigned_mdevs.return_value = {}
        else:
            mock_get_all_assigned_mdevs.return_value = instance_mdevs
            mock_get_types.return_value = {uuids.mdev1: 'fake_type'}
            if not assert_can_lv_mdevs:
                mock_assert_src_can_lv_mdevs.side_effect = exception(
                    reason='kaboom')

        if exception:
            self.assertRaises(exception, drvr.check_can_live_migrate_source,
                              self.context, instance, dest_check_data)
        else:
            ret = drvr.check_can_live_migrate_source(self.context, instance,
                                                     dest_check_data)
            if numa_lm:
                self.assertTrue(ret.src_supports_numa_live_migration)
            else:
                self.assertNotIn('src_supports_numa_live_migration', ret)

        mock_is_shared.assert_called_once_with(instance, dest_check_data, None)
        mock_check_shared.assert_called_once_with('file', instance)

        if exception:
            return (instance, dest_check_data)

        if block_migration:
            self.assertIsInstance(ret, objects.LibvirtLiveMigrateData)
            self.assertIn('is_shared_block_storage', ret)
            self.assertFalse(ret.is_shared_block_storage)
            self.assertIn('is_shared_instance_path', ret)
            self.assertFalse(ret.is_shared_instance_path)

        if is_shared_block_storage:
            self.assertTrue(ret.is_shared_block_storage)

        if is_shared_instance_path:
            self.assertTrue(ret.is_shared_instance_path)

        if instance_mdevs:
            mock_get_types.assert_called_once_with(instance_mdevs.keys())
            self.assertEqual({uuids.mdev1: 'fake_type'}, ret.source_mdev_types)
            if assert_can_lv_mdevs:
                mock_assert_src_can_lv_mdevs.assert_called_once_with(
                    instance, dest_check_data)
        return (instance, dest_check_data)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_assert_dest_node_has_enough_disk')
    def test_check_can_live_migrate_source_block_migration(
            self, mock_assert_dest):
        instance, dest_check_data = self._test_can_live_migrate_source(
            block_migration=True)
        mock_assert_dest.assert_called_once_with(
            self.context, instance, dest_check_data.disk_available_mb,
            False, None)

    def test_check_can_live_migrate_source_numa_lm(self):
        self._test_can_live_migrate_source(is_shared_block_storage=True,
                                           numa_lm=True)
        self._test_can_live_migrate_source(is_shared_block_storage=True,
                                           numa_lm=False)

    def test_check_can_live_migrate_source_shared_block_storage(self):
        self._test_can_live_migrate_source(is_shared_block_storage=True)

    def test_check_can_live_migrate_source_shared_instance_path(self):
        self._test_can_live_migrate_source(is_shared_instance_path=True)

    def test_check_can_live_migrate_source_non_shared_fails(self):
        self._test_can_live_migrate_source(
            exception=exception.InvalidSharedStorage)

    def test_check_can_live_migrate_source_shared_block_migration_fails(self):
        self._test_can_live_migrate_source(
            block_migration=True, is_shared_block_storage=True,
            exception=exception.InvalidLocalStorage)

    def test_check_can_live_migrate_shared_path_block_migration_fails(self):
        self._test_can_live_migrate_source(
            block_migration=True, is_shared_instance_path=True,
            exception=exception.InvalidLocalStorage)

    def test_check_can_live_migrate_non_shared_non_block_migration_fails(self):
        self._test_can_live_migrate_source(
            exception=exception.InvalidSharedStorage)

    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.'
                '_get_instance_disk_info')
    def test_check_can_live_migrate_source_with_dest_not_enough_disk(
            self, mock_get_bdi):
        mock_get_bdi.return_value = [{"virt_disk_size": 2}]
        instance, _ = self._test_can_live_migrate_source(
            block_migration=True, disk_available_mb=0,
            exception=exception.MigrationError)
        mock_get_bdi.assert_called_once_with(instance, None)

    def test_check_can_live_migrate_source_mdev_lm(self):
        instance_mdevs = {uuids.mdev1: uuids.inst1}
        self._test_can_live_migrate_source(is_shared_block_storage=True,
                                           instance_mdevs=instance_mdevs)

    def test_check_can_live_migrate_source_mdev_lm_fails(self):
        instance_mdevs = {uuids.mdev1: uuids.inst1}
        self._test_can_live_migrate_source(
            is_shared_block_storage=True,
            instance_mdevs=instance_mdevs,
            assert_can_lv_mdevs=False,
            exception=exception.MigrationPreCheckError)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_host_can_support_mdev_live_migration')
    def test_assert_source_can_lv_mdevs_fails_due_to_src(self, mock_host):
        mock_host.return_value = False
        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        dest_check_data = objects.LibvirtLiveMigrateData(
            dst_supports_mdev_live_migration=True)
        self.assertRaises(exception.MigrationPreCheckError,
                          drvr._assert_source_can_live_migrate_mdevs,
                          instance, dest_check_data)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_host_can_support_mdev_live_migration')
    def test_assert_source_can_lv_mdevs_fails_due_to_dest(self, mock_host):
        mock_host.return_value = True
        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        dest_check_data = objects.LibvirtLiveMigrateData()
        self.assertRaises(exception.MigrationPreCheckError,
                          drvr._assert_source_can_live_migrate_mdevs,
                          instance, dest_check_data)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_host_can_support_mdev_live_migration')
    def test_assert_source_can_lv_mdevs_works(self, mock_host):
        mock_host.return_value = True
        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        dest_check_data = objects.LibvirtLiveMigrateData(
            dst_supports_mdev_live_migration=True)
        drvr._assert_source_can_live_migrate_mdevs(instance, dest_check_data)
        mock_host.assert_called_once_with()

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.'
                '_assert_dest_node_has_enough_disk')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.'
                '_is_shared_block_storage', return_value=False)
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.'
                '_check_shared_storage_test_file', return_value=False)
    def test_check_can_live_migrate_source_bm_with_bdm_tunnelled_error(
            self, mock_check, mock_shared_block, mock_enough,
            mock_min_version):

        self.flags(live_migration_tunnelled=True,
                   group='libvirt')
        bdi = {'block_device_mapping': ['bdm']}
        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        dest_check_data = objects.LibvirtLiveMigrateData(
            filename='file',
            image_type='default',
            block_migration=True,
            disk_over_commit=False,
            disk_available_mb=100)
        drvr._parse_migration_flags()
        self.assertRaises(exception.MigrationPreCheckError,
                          drvr.check_can_live_migrate_source,
                          self.context, instance, dest_check_data,
                          block_device_info=bdi)

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.'
                '_assert_dest_node_has_enough_disk')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.'
                '_is_shared_block_storage')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.'
                '_check_shared_storage_test_file')
    def _test_check_can_live_migrate_source_block_migration_none(
            self, block_migrate, is_shared_instance_path, is_share_block,
            mock_check, mock_shared_block, mock_enough, mock_version):

        mock_check.return_value = is_shared_instance_path
        mock_shared_block.return_value = is_share_block
        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        dest_check_data = objects.LibvirtLiveMigrateData(
            filename='file',
            image_type='default',
            disk_over_commit=False,
            disk_available_mb=100)
        dest_check_data_ret = drvr.check_can_live_migrate_source(
                          self.context, instance, dest_check_data)
        self.assertEqual(block_migrate, dest_check_data_ret.block_migration)

    def test_check_can_live_migrate_source_block_migration_none_shared1(self):
        self._test_check_can_live_migrate_source_block_migration_none(
            False,
            True,
            False)

    def test_check_can_live_migrate_source_block_migration_none_shared2(self):
        self._test_check_can_live_migrate_source_block_migration_none(
            False,
            False,
            True)

    def test_check_can_live_migrate_source_block_migration_none_no_share(self):
        self._test_check_can_live_migrate_source_block_migration_none(
            True,
            False,
            False)

    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.'
                '_assert_dest_node_has_enough_disk')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.'
                '_is_shared_block_storage')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.'
                '_check_shared_storage_test_file')
    def test_check_can_live_migration_source_disk_over_commit_none(self,
            mock_check, mock_shared_block, mock_disk_check):

        mock_check.return_value = False
        mock_shared_block.return_value = False
        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        dest_check_data = objects.LibvirtLiveMigrateData(
            filename='file',
            image_type='default',
            disk_available_mb=100)

        drvr.check_can_live_migrate_source(
            self.context, instance, dest_check_data)

        self.assertFalse(mock_disk_check.called)

    def _is_shared_block_storage_test_create_mocks(self, disks):
        # Test data
        instance_xml = ("<domain type='kvm'><name>instance-0000000a</name>"
                        "<devices>{}</devices></domain>")
        disks_xml = ''
        for dsk in disks:
            if dsk['type'] != 'network':
                disks_xml = ''.join([disks_xml,
                                "<disk type='{type}'>"
                                "<driver name='qemu' type='{driver}'/>"
                                "<source {source}='{source_path}'/>"
                                "<target dev='{target_dev}' bus='virtio'/>"
                                "</disk>".format(**dsk)])
            else:
                disks_xml = ''.join([disks_xml,
                                "<disk type='{type}'>"
                                "<driver name='qemu' type='{driver}'/>"
                                "<source protocol='{source_proto}'"
                                "name='{source_image}' >"
                                "<host name='hostname' port='7000'/>"
                                "<config file='/path/to/file'/>"
                                "</source>"
                                "<target dev='{target_dev}'"
                                "bus='ide'/>".format(**dsk)])

        # Preparing mocks
        mock_virDomain = mock.Mock(fakelibvirt.virDomain)
        mock_virDomain.XMLDesc = mock.Mock()
        mock_virDomain.XMLDesc.return_value = (instance_xml.format(disks_xml))

        mock_lookup = mock.Mock()

        def mock_lookup_side_effect(name):
            return mock_virDomain
        mock_lookup.side_effect = mock_lookup_side_effect

        mock_qemu_img_info = mock.Mock()
        mock_qemu_img_info.return_value = mock.Mock(disk_size=10737418240,
                                                    virtual_size=10737418240)
        mock_stat = mock.Mock()
        mock_stat.return_value = mock.Mock(st_blocks=20971520)
        mock_get_size = mock.Mock()
        mock_get_size.return_value = 10737418240

        return (mock_stat, mock_get_size, mock_qemu_img_info, mock_lookup)

    def test_is_shared_block_storage_rbd(self):
        self.flags(images_type='rbd', group='libvirt')
        bdi = {'block_device_mapping': []}
        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        mock_get_instance_disk_info = mock.Mock()
        data = objects.LibvirtLiveMigrateData(image_type='rbd')
        with mock.patch.object(drvr, '_get_instance_disk_info',
                               mock_get_instance_disk_info):
            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
            self.assertTrue(drvr._is_shared_block_storage(instance, data,
                                  block_device_info=bdi))
        self.assertEqual(0, mock_get_instance_disk_info.call_count)

    def test_is_shared_block_storage_lvm(self):
        self.flags(images_type='lvm', group='libvirt')
        bdi = {'block_device_mapping': []}
        instance = objects.Instance(**self.test_instance)
        mock_get_instance_disk_info = mock.Mock()
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        data = objects.LibvirtLiveMigrateData(image_type='lvm',
                                              is_volume_backed=False,
                                              is_shared_instance_path=False)
        with mock.patch.object(drvr, '_get_instance_disk_info',
                               mock_get_instance_disk_info):
            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
            self.assertFalse(drvr._is_shared_block_storage(
                                    instance, data,
                                    block_device_info=bdi))
        self.assertEqual(0, mock_get_instance_disk_info.call_count)

    def test_is_shared_block_storage_qcow2(self):
        self.flags(images_type='qcow2', group='libvirt')
        bdi = {'block_device_mapping': []}
        instance = objects.Instance(**self.test_instance)
        mock_get_instance_disk_info = mock.Mock()
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        data = objects.LibvirtLiveMigrateData(image_type='qcow2',
                                              is_volume_backed=False,
                                              is_shared_instance_path=False)
        with mock.patch.object(drvr, '_get_instance_disk_info',
                               mock_get_instance_disk_info):
            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
            self.assertFalse(drvr._is_shared_block_storage(
                                    instance, data,
                                    block_device_info=bdi))
        self.assertEqual(0, mock_get_instance_disk_info.call_count)

    def test_is_shared_block_storage_rbd_only_source(self):
        self.flags(images_type='rbd', group='libvirt')
        bdi = {'block_device_mapping': []}
        instance = objects.Instance(**self.test_instance)
        mock_get_instance_disk_info = mock.Mock()
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        data = objects.LibvirtLiveMigrateData(is_shared_instance_path=False,
                                              is_volume_backed=False)
        with mock.patch.object(drvr, '_get_instance_disk_info',
                               mock_get_instance_disk_info):
            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
            self.assertFalse(drvr._is_shared_block_storage(
                                  instance, data,
                                  block_device_info=bdi))
        self.assertEqual(0, mock_get_instance_disk_info.call_count)

    def test_is_shared_block_storage_rbd_only_dest(self):
        bdi = {'block_device_mapping': []}
        instance = objects.Instance(**self.test_instance)
        mock_get_instance_disk_info = mock.Mock()
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        data = objects.LibvirtLiveMigrateData(image_type='rbd',
                                              is_volume_backed=False,
                                              is_shared_instance_path=False)
        with mock.patch.object(drvr, '_get_instance_disk_info',
                               mock_get_instance_disk_info):
            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
            self.assertFalse(drvr._is_shared_block_storage(
                                    instance, data,
                                    block_device_info=bdi))
        self.assertEqual(0, mock_get_instance_disk_info.call_count)

    def test_is_shared_block_storage_volume_backed(self):
        disks = [{'type': 'block',
                 'driver': 'raw',
                 'source': 'dev',
                 'source_path': '/dev/disk',
                 'target_dev': 'vda'}]
        bdi = {'block_device_mapping': [
                  {'connection_info': 'info', 'mount_device': '/dev/vda'}]}
        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        (mock_stat, mock_get_size, mock_qemu_img_info, mock_lookup) =\
            self._is_shared_block_storage_test_create_mocks(disks)
        data = objects.LibvirtLiveMigrateData(is_volume_backed=True,
                                              is_shared_instance_path=False)
        with mock.patch.object(host.Host, '_get_domain', mock_lookup):
            self.assertTrue(drvr._is_shared_block_storage(instance, data,
                                  block_device_info = bdi))
        mock_lookup.assert_called_once_with(instance)

    def test_is_shared_block_storage_volume_backed_with_disk(self):
        disks = [{'type': 'block',
                 'driver': 'raw',
                 'source': 'dev',
                 'source_path': '/dev/disk',
                 'target_dev': 'vda'},
                {'type': 'file',
                 'driver': 'raw',
                 'source': 'file',
                 'source_path': '/instance/disk.local',
                 'target_dev': 'vdb'}]
        bdi = {'block_device_mapping': [
                  {'connection_info': 'info', 'mount_device': '/dev/vda'}]}
        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        (mock_stat, mock_get_size, mock_qemu_img_info, mock_lookup) =\
            self._is_shared_block_storage_test_create_mocks(disks)
        data = objects.LibvirtLiveMigrateData(is_volume_backed=True,
                                              is_shared_instance_path=False)
        with test.nested(
                mock.patch('os.stat', mock_stat),
                mock.patch('os.path.getsize', mock_get_size),
                mock.patch.object(libvirt_driver.disk_api,
                                  'get_disk_info', mock_qemu_img_info),
                mock.patch.object(host.Host, '_get_domain', mock_lookup)):
            self.assertFalse(drvr._is_shared_block_storage(
                                    instance, data,
                                    block_device_info = bdi))
        mock_stat.assert_called_once_with('/instance/disk.local')
        mock_get_size.assert_called_once_with('/instance/disk.local')
        mock_lookup.assert_called_once_with(instance)

    def test_is_shared_block_storage_nfs(self):
        bdi = {'block_device_mapping': []}
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        mock_image_backend = mock.MagicMock()
        drvr.image_backend = mock_image_backend
        mock_backend = mock.MagicMock()
        mock_image_backend.backend.return_value = mock_backend
        mock_backend.is_file_in_instance_path.return_value = True
        mock_get_instance_disk_info = mock.Mock()
        data = objects.LibvirtLiveMigrateData(
            is_shared_instance_path=True,
            image_type='foo')
        with mock.patch.object(drvr, '_get_instance_disk_info',
                               mock_get_instance_disk_info):
            self.assertTrue(drvr._is_shared_block_storage(
                'instance', data, block_device_info=bdi))
        self.assertEqual(0, mock_get_instance_disk_info.call_count)

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    @mock.patch.object(fakelibvirt.virDomain, "migrateToURI3")
    @mock.patch.object(fakelibvirt.virDomain, "XMLDesc")
    def test_live_migration_update_graphics_xml(self, mock_xml,
                                                      mock_migrateToURI3,
                                                      mock_min_version):
        self.compute = manager.ComputeManager()
        instance_ref = objects.Instance(**self.test_instance)
        target_connection = '127.0.0.2'

        xml_tmpl = ("<domain type='kvm'>"
                    "<devices>"
                    "<graphics type='vnc' listen='{vnc}'>"
                    "<listen address='{vnc}'/>"
                    "</graphics>"
                    "<graphics type='spice' listen='{spice}'>"
                    "<listen address='{spice}'/>"
                    "</graphics>"
                    "</devices>"
                    "</domain>")

        initial_xml = xml_tmpl.format(vnc='1.2.3.4',
                                      spice='5.6.7.8')

        target_xml = xml_tmpl.format(vnc='10.0.0.1',
                                     spice='10.0.0.2')
        target_xml = etree.tostring(etree.fromstring(target_xml),
                                    encoding='unicode')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        # Preparing mocks
        mock_xml.return_value = initial_xml
        mock_migrateToURI3.side_effect = fakelibvirt.libvirtError("ERR")

        disk_paths = ['vda', 'vdb']
        _bandwidth = CONF.libvirt.live_migration_bandwidth
        params = {
            'migrate_uri': 'tcp://127.0.0.2',
            'migrate_disks': disk_paths,
            'bandwidth': _bandwidth,
            'destination_xml': target_xml,
            'persistent_xml': target_xml,
        }

        # start test
        migrate_data = objects.LibvirtLiveMigrateData(
            graphics_listen_addr_vnc='10.0.0.1',
            graphics_listen_addr_spice='10.0.0.2',
            serial_listen_addr='127.0.0.1',
            serial_listen_ports=[1234],
            target_connect_addr=target_connection,
            bdms=[],
            block_migration=False)
        dom = fakelibvirt.virDomain
        guest = libvirt_guest.Guest(dom)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertRaises(fakelibvirt.libvirtError,
                          drvr._live_migration_operation,
                          self.context, instance_ref, target_connection,
                          False, migrate_data, guest, disk_paths)
        mock_xml.assert_called_once_with(
                flags=fakelibvirt.VIR_DOMAIN_XML_MIGRATABLE)
        mock_migrateToURI3.assert_called_once_with(
                drvr._live_migration_uri(target_connection),
                params=params, flags=0)

    def test_live_migration_parallels_no_new_xml(self):
        self.flags(virt_type='parallels', group='libvirt')
        self.flags(enabled=False, group='vnc')
        target_connection = None
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        instance_dict = dict(self.test_instance)
        instance_dict.update({'host': 'fake',
                              'power_state': power_state.RUNNING,
                              'vm_state': vm_states.ACTIVE})
        instance = objects.Instance(**instance_dict)

        params = {
            'bandwidth': CONF.libvirt.live_migration_bandwidth,
        }
        migrate_data = objects.LibvirtLiveMigrateData(
            target_connect_addr=target_connection,
            block_migration=False,
            serial_listen_addr='127.0.0.1',
            serial_listen_ports=[1234])
        dom_mock = mock.MagicMock()
        guest = libvirt_guest.Guest(dom_mock)
        drvr._live_migration_operation(self.context, instance,
                                       target_connection, False,
                                       migrate_data, guest, None)
        dom_mock.migrateToURI3.assert_called_once_with(
            drvr._live_migration_uri(target_connection),
            params=params, flags=0)

    @mock.patch.object(utils, 'spawn')
    @mock.patch.object(host.Host, 'get_guest')
    @mock.patch.object(fakelibvirt.Connection, '_mark_running')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_live_migration_monitor')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_live_migration_copy_disk_paths')
    def test_live_migration_parallels_no_migrate_disks(self,
                                                       mock_copy_disk_paths,
                                                       mock_monitor,
                                                       mock_running,
                                                       mock_guest,
                                                       mock_thread):
        self.flags(virt_type='parallels', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        instance_dict = dict(self.test_instance)
        instance_dict.update({'host': 'fake',
                              'power_state': power_state.RUNNING,
                              'vm_state': vm_states.ACTIVE})
        instance = objects.Instance(**instance_dict)
        migrate_data = objects.LibvirtLiveMigrateData(
            block_migration=True)
        dom = fakelibvirt.Domain(drvr._get_connection(), '<domain/>', True)
        guest = libvirt_guest.Guest(dom)
        mock_guest.return_value = guest
        drvr._live_migration(self.context, instance, 'dest',
                             lambda: None, lambda: None, True,
                             migrate_data)
        self.assertFalse(mock_copy_disk_paths.called)
        mock_thread.assert_called_once_with(
            drvr._live_migration_operation,
            self.context, instance, 'dest', True,
            migrate_data, guest, [])

    @mock.patch.object(fakelibvirt.virDomain, "migrateToURI3")
    @mock.patch.object(nova.virt.libvirt.migration,
                       'get_updated_guest_xml', return_value='')
    @mock.patch.object(fakelibvirt.virDomain, "XMLDesc")
    def test_live_migration_update_volume_xml(self, mock_xml,
                                    mock_updated_guest_xml,
                                    mock_migrateToURI3):
        self.compute = manager.ComputeManager()
        instance_ref = objects.Instance(**self.test_instance)
        target_connection = '127.0.0.2'

        target_xml = self.device_xml_tmpl.format(
            device_path='/dev/disk/by-path/'
            'ip-1.2.3.4:3260-iqn.'
            'cde.67890.opst-lun-Z')

        # Prepare mocks
        mock_xml.return_value = target_xml

        disk_paths = ['vda', 'vdb']
        params = {
            'migrate_disks': disk_paths,
            'migrate_uri': 'tcp://127.0.0.2',
            'bandwidth': CONF.libvirt.live_migration_bandwidth,
            'destination_xml': target_xml,
            'persistent_xml': target_xml,
        }

        # Start test
        connection_info = {
            u'driver_volume_type': u'iscsi',
            u'serial': u'58a84f6d-3f0c-4e19-a0af-eb657b790657',
            u'data': {
                u'access_mode': u'rw', u'target_discovered': False,
                u'target_iqn': u'ip-1.2.3.4:3260-iqn.cde.67890.opst-lun-Z',
                u'volume_id': u'58a84f6d-3f0c-4e19-a0af-eb657b790657',
                'device_path':
                u'/dev/disk/by-path/ip-1.2.3.4:3260-iqn.cde.67890.opst-lun-Z',
            },
        }
        bdm = objects.LibvirtLiveMigrateBDMInfo(
            serial='58a84f6d-3f0c-4e19-a0af-eb657b790657',
            bus='virtio', type='disk', dev='vdb',
            connection_info=connection_info)
        migrate_data = objects.LibvirtLiveMigrateData(
            serial_listen_addr='',
            target_connect_addr=target_connection,
            bdms=[bdm],
            block_migration=False)

        dom = fakelibvirt.virDomain
        guest = libvirt_guest.Guest(dom)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        mock_updated_guest_xml.return_value = target_xml
        drvr._live_migration_operation(
                         self.context, instance_ref, target_connection,
                         False, migrate_data, guest, disk_paths)
        mock_migrateToURI3.assert_called_once_with(
                drvr._live_migration_uri(target_connection),
                params=params, flags=0)
        mock_updated_guest_xml.assert_called_once_with(
                instance_ref,
                guest,
                migrate_data,
                mock.ANY,
                get_vif_config=None,
                new_resources=None
        )

    def test_live_migration_update_vifs_xml(self):
        """Tests that when migrate_data.vifs is populated, the destination
        guest xml is updated with the migrate_data.vifs configuration.
        """
        instance = objects.Instance(**self.test_instance)

        source_vif_normal = network_model.VIF(
            id=uuids.port_id, type=network_model.VIF_TYPE_OVS,
            vnic_type=network_model.VNIC_TYPE_NORMAL, details={'foo': 'bar'},
            profile={'binding:host_id': 'fake-source-host'})

        vif = objects.VIFMigrateData(port_id=uuids.port_id,
                                     source_vif=source_vif_normal)
        migrate_data = objects.LibvirtLiveMigrateData(
            serial_listen_addr='',
            target_connect_addr=None,
            bdms=[],
            block_migration=False,
            vifs=[vif])

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        guest = libvirt_guest.Guest(mock.MagicMock())
        fake_xml = '<domain type="qemu"/>'

        def fake_get_updated_guest_xml(
            instance,
            guest,
            migrate_data,
            get_volume_config,
            get_vif_config=None,
            new_resources=None
        ):
            self.assertIsNotNone(get_vif_config)
            return fake_xml

        @mock.patch.object(drvr, "detach_interface")
        @mock.patch('nova.virt.libvirt.migration.get_updated_guest_xml',
                    side_effect=fake_get_updated_guest_xml)
        @mock.patch.object(drvr._host, 'has_min_version', return_value=True)
        @mock.patch.object(guest, 'migrate')
        def _test_normal(migrate, has_min_version,
                         get_updated_guest_xml, detach):
            drvr._live_migration_operation(
                self.context, instance, 'dest.host', False,
                migrate_data, guest, [])
            get_updated_guest_xml.assert_called_once()
            migrate.assert_called()
            detach.assert_not_called()

        _test_normal()

        source_vif_direct = network_model.VIF(
            id=uuids.port_id, type=network_model.VIF_TYPE_OVS,
            vnic_type=network_model.VNIC_TYPE_DIRECT, details={'foo': 'bar'},
            profile={'binding:host_id': 'fake-source-host'})

        vif_direct = objects.VIFMigrateData(port_id=uuids.port_id,
                                            source_vif=source_vif_direct)
        migrate_data = objects.LibvirtLiveMigrateData(
            serial_listen_addr='', target_connect_addr=None,
            bdms=[], block_migration=False, vifs=[vif_direct])

        @mock.patch.object(drvr, "detach_interface")
        @mock.patch('nova.virt.libvirt.migration.get_updated_guest_xml',
                    side_effect=fake_get_updated_guest_xml)
        @mock.patch.object(drvr._host, 'has_min_version', return_value=True)
        @mock.patch.object(guest, 'migrate')
        def _test_direct(migrate, has_min_version,
                         get_updated_guest_xml, detach):
            drvr._live_migration_operation(
                self.context, instance, 'dest.host', False,
                migrate_data, guest, [])
            get_updated_guest_xml.assert_called_once()
            migrate.assert_called()
            detach.assert_called()

        _test_direct()

        migrate_data = objects.LibvirtLiveMigrateData(
            serial_listen_addr='', target_connect_addr=None,
            bdms=[], block_migration=False, vifs=[vif, vif_direct])

        @mock.patch.object(drvr, "detach_interface")
        @mock.patch('nova.virt.libvirt.migration.get_updated_guest_xml',
                    side_effect=fake_get_updated_guest_xml)
        @mock.patch.object(drvr._host, 'has_min_version', return_value=True)
        @mock.patch.object(guest, 'migrate')
        def _test_mix(migrate, has_min_version, get_updated_guest_xml, detach):
            drvr._live_migration_operation(
                self.context, instance, 'dest.host', False,
                migrate_data, guest, [])
            get_updated_guest_xml.assert_called_once()
            migrate.assert_called()
            detach.assert_called_once()

        _test_mix()

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    @mock.patch.object(fakelibvirt.virDomain, "migrateToURI3")
    @mock.patch.object(nova.virt.libvirt.migration,
                       'get_updated_guest_xml', return_value='')
    @mock.patch.object(fakelibvirt.virDomain, "XMLDesc")
    def test_live_migration_with_valid_target_connect_addr(self, mock_xml,
                                                      mock_updated_guest_xml,
                                                      mock_migrateToURI3,
                                                      mock_min_version):
        self.compute = manager.ComputeManager()
        instance_ref = objects.Instance(**self.test_instance)
        target_connection = '127.0.0.2'

        target_xml = self.device_xml_tmpl.format(
            device_path='/dev/disk/by-path/'
            'ip-1.2.3.4:3260-iqn.'
            'cde.67890.opst-lun-Z')

        # Prepare mocks
        mock_xml.return_value = target_xml

        disk_paths = ['vda', 'vdb']
        params = {
            'migrate_disks': disk_paths,
            'migrate_uri': 'tcp://127.0.0.2',
            'bandwidth': CONF.libvirt.live_migration_bandwidth,
            'destination_xml': target_xml,
            'persistent_xml': target_xml,
        }

        # start test
        connection_info = {
            u'driver_volume_type': u'iscsi',
            u'serial': u'58a84f6d-3f0c-4e19-a0af-eb657b790657',
            u'data': {
                u'access_mode': u'rw', u'target_discovered': False,
                u'target_iqn': u'ip-1.2.3.4:3260-iqn.cde.67890.opst-lun-Z',
                u'volume_id': u'58a84f6d-3f0c-4e19-a0af-eb657b790657',
                'device_path':
                u'/dev/disk/by-path/ip-1.2.3.4:3260-iqn.cde.67890.opst-lun-Z',
            },
        }
        bdm = objects.LibvirtLiveMigrateBDMInfo(
            serial='58a84f6d-3f0c-4e19-a0af-eb657b790657',
            bus='virtio', type='disk', dev='vdb',
            connection_info=connection_info)
        migrate_data = objects.LibvirtLiveMigrateData(
            serial_listen_addr='',
            target_connect_addr=target_connection,
            bdms=[bdm],
            block_migration=False)
        dom = fakelibvirt.virDomain
        guest = libvirt_guest.Guest(dom)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        mock_updated_guest_xml.return_value = target_xml
        drvr._live_migration_operation(self.context, instance_ref,
                                       target_connection, False, migrate_data,
                                       guest, disk_paths)
        mock_migrateToURI3.assert_called_once_with(
                drvr._live_migration_uri(target_connection),
                params=params, flags=0)

    def test_update_volume_xml(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        initial_xml = self.device_xml_tmpl.format(
            device_path='/dev/disk/by-path/'
            'ip-1.2.3.4:3260-iqn.'
            'abc.12345.opst-lun-X')
        target_xml = self.device_xml_tmpl.format(
            device_path='/dev/disk/by-path/'
            'ip-1.2.3.4:3260-iqn.'
            'cde.67890.opst-lun-Z')
        target_xml = etree.tostring(etree.fromstring(target_xml),
                                    encoding='unicode')
        serial = "58a84f6d-3f0c-4e19-a0af-eb657b790657"

        bdmi = objects.LibvirtLiveMigrateBDMInfo(serial=serial,
                                                 bus='virtio',
                                                 type='disk',
                                                 dev='vdb')
        bdmi.connection_info = {u'driver_volume_type': u'iscsi',
           'serial': u'58a84f6d-3f0c-4e19-a0af-eb657b790657',
            u'data': {u'access_mode': u'rw', u'target_discovered': False,
            u'target_iqn': u'ip-1.2.3.4:3260-iqn.cde.67890.opst-lun-Z',
            u'volume_id': u'58a84f6d-3f0c-4e19-a0af-eb657b790657',
           'device_path':
              u'/dev/disk/by-path/ip-1.2.3.4:3260-iqn.cde.67890.opst-lun-Z'}}

        conf = vconfig.LibvirtConfigGuestDisk()
        conf.source_device = bdmi.type
        conf.driver_name = "qemu"
        conf.driver_format = "raw"
        conf.driver_cache = "none"
        conf.target_dev = bdmi.dev
        conf.target_bus = bdmi.bus
        conf.serial = bdmi.connection_info.get('serial')
        conf.source_type = "block"
        conf.source_path = bdmi.connection_info['data'].get('device_path')

        instance = objects.Instance(**self.test_instance)
        guest = libvirt_guest.Guest(mock.MagicMock())

        with test.nested(
            mock.patch.object(
                drvr, '_get_volume_config', return_value=conf
            ),
            mock.patch.object(
                guest, 'get_xml_desc', return_value=initial_xml
            ),
            mock.patch.object(
                drvr, '_sev_enabled', new=mock.Mock(return_value=False)
            )
        ):
            config = libvirt_migrate.get_updated_guest_xml(
                instance,
                guest,
                objects.LibvirtLiveMigrateData(
                    bdms=[bdmi],
                    serial_listen_addr='127.0.0.1',
                    serial_listen_ports=[1234]
                ),
                drvr._get_volume_config
            )
            parser = etree.XMLParser(remove_blank_text=True)
            config = etree.fromstring(config, parser)
            target_xml = etree.fromstring(target_xml, parser)
            self.assertXmlEqual(
                etree.tostring(target_xml, encoding='unicode'),
                etree.tostring(config, encoding='unicode'))

    def test_live_migration_uri(self):
        addresses = ('127.0.0.1', '127.0.0.1:4444', '[::1]:4444',
                    '[0:0:0:0:0:0:0:1]:4444', u'127.0.0.1', u'destination',
        )

        hypervisor_uri_map = (
            ('kvm', 'qemu+tcp://%s/system'),
            ('qemu', 'qemu+tcp://%s/system'),
            ('parallels', 'parallels+tcp://%s/system'),
            # anything else will return None
            ('lxc', None),
        )

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        for dest in addresses:
            for hyperv, uri in hypervisor_uri_map:
                self.flags(virt_type=hyperv, group='libvirt')
                if uri is not None:
                    uri = uri % dest
                    self.assertEqual(uri, drvr._live_migration_uri(dest))
                else:
                    self.assertRaises(exception.LiveMigrationURINotAvailable,
                                      drvr._live_migration_uri,
                                      dest)

    def test_live_migration_uri_ipv6(self):
        addresses = ('::1', '0:0:0:0:0:0:0:1', u'::1')

        hypervisor_uri_map = (
            ('kvm', 'qemu+tcp://[%s]/system'),
            ('qemu', 'qemu+tcp://[%s]/system'),
            ('parallels', 'parallels+tcp://[%s]/system'),
            # anything else will return None
            ('lxc', None),
        )

        for dest in addresses:
            for hyperv, uri in hypervisor_uri_map:
                self.flags(virt_type=hyperv, group='libvirt')
                drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
                if uri is not None:
                    uri = uri % dest
                    self.assertEqual(uri, drvr._live_migration_uri(dest))
                else:
                    self.assertRaises(exception.LiveMigrationURINotAvailable,
                                      drvr._live_migration_uri,
                                      dest)

    def test_live_migration_uri_forced(self):
        dest = 'destination'
        self.flags(virt_type='kvm', group='libvirt')

        forced_uri = 'foo://%s/bar'
        self.flags(live_migration_uri=forced_uri, group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertEqual(forced_uri % dest, drvr._live_migration_uri(dest))

    def test_live_migration_scheme(self):
        self.flags(live_migration_scheme='ssh', group='libvirt')
        dest = 'destination'
        uri = 'qemu+ssh://%s/system'
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertEqual(uri % dest, drvr._live_migration_uri(dest))

    def test_live_migration_scheme_does_not_override_uri(self):
        forced_uri = 'qemu+ssh://%s/system'
        self.flags(live_migration_uri=forced_uri, group='libvirt')
        self.flags(live_migration_scheme='tcp', group='libvirt')
        dest = 'destination'
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertEqual(forced_uri % dest, drvr._live_migration_uri(dest))

    def test_migrate_uri(self):
        hypervisor_uri_map = (
            ('kvm', 'tcp://%s'),
            ('qemu', 'tcp://%s'),
        )
        dest = 'destination'
        for hyperv, uri in hypervisor_uri_map:
            self.flags(virt_type=hyperv, group='libvirt')
            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
            if uri is not None:
                uri = uri % dest
            self.assertEqual(uri, drvr._migrate_uri(dest))

    def test_migrate_uri_forced_live_migration_uri(self):
        dest = 'destination'
        self.flags(virt_type='kvm', group='libvirt')

        forced_uri = 'qemu+tcp://user:pass@%s/system'
        self.flags(live_migration_uri=forced_uri, group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertEqual('tcp://%s' % dest, drvr._migrate_uri(dest))

    def test_migrate_uri_forced_live_migration_inboud_addr(self):
        self.flags(virt_type='kvm', group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        addresses = ('127.0.0.1', '127.0.0.1:4444', '[::1]:4444',
                    '[0:0:0:0:0:0:0:1]:4444', u'127.0.0.1', u'destination',
        )

        for dest in addresses:
            uri = 'tcp://%s'
            result = drvr._migrate_uri(dest)
            self.assertEqual(uri % dest, result)
            self.assertIsInstance(result, str)

    def test_migrate_uri_forced_live_migration_inboud_addr_ipv6(self):
        self.flags(virt_type='kvm', group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        addresses = ('::1', '0:0:0:0:0:0:0:1', u'::1')

        for dest in addresses:
            uri = 'tcp://[%s]'
            result = drvr._migrate_uri(dest)
            self.assertEqual(uri % dest, result)
            self.assertIsInstance(result, str)

    def test_update_volume_xml_no_serial(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        xml_tmpl = """
        <domain type='kvm'>
          <devices>
            <disk type='block' device='disk'>
              <driver name='qemu' type='raw' cache='none'/>
              <source dev='{device_path}'/>
              <target bus='virtio' dev='vdb'/>
              <serial></serial>
              <address type='pci' domain='0x0' bus='0x0' slot='0x04' \
              function='0x0'/>
            </disk>
          </devices>
        </domain>
        """

        initial_xml = xml_tmpl.format(device_path='/dev/disk/by-path/'
                                      'ip-1.2.3.4:3260-iqn.'
                                      'abc.12345.opst-lun-X')
        target_xml = xml_tmpl.format(device_path='/dev/disk/by-path/'
                                     'ip-1.2.3.4:3260-iqn.'
                                     'abc.12345.opst-lun-X')
        target_xml = etree.tostring(etree.fromstring(target_xml),
                                    encoding='unicode')
        serial = "58a84f6d-3f0c-4e19-a0af-eb657b790657"
        connection_info = {
            u'driver_volume_type': u'iscsi',
            'serial': u'58a84f6d-3f0c-4e19-a0af-eb657b790657',
            u'data': {
                u'access_mode': u'rw', u'target_discovered': False,
                u'target_iqn': u'ip-1.2.3.4:3260-iqn.cde.67890.opst-lun-Z',
                u'volume_id': u'58a84f6d-3f0c-4e19-a0af-eb657b790657',
                u'device_path':
                u'/dev/disk/by-path/ip-1.2.3.4:3260-iqn.cde.67890.opst-lun-Z',
            },
        }
        bdmi = objects.LibvirtLiveMigrateBDMInfo(serial=serial,
                                                 bus='virtio',
                                                 dev='vdb',
                                                 type='disk')
        bdmi.connection_info = connection_info

        conf = vconfig.LibvirtConfigGuestDisk()
        conf.source_device = bdmi.type
        conf.driver_name = "qemu"
        conf.driver_format = "raw"
        conf.driver_cache = "none"
        conf.target_dev = bdmi.dev
        conf.target_bus = bdmi.bus
        conf.serial = bdmi.connection_info.get('serial')
        conf.source_type = "block"
        conf.source_path = bdmi.connection_info['data'].get('device_path')

        instance = objects.Instance(**self.test_instance)
        guest = libvirt_guest.Guest(mock.MagicMock())

        with test.nested(
            mock.patch.object(
                drvr, '_get_volume_config', return_value=conf
            ),
            mock.patch.object(
                guest, 'get_xml_desc', return_value=initial_xml
            ),
            mock.patch.object(
                drvr, '_sev_enabled', new=mock.Mock(return_value=False)
            )
        ):
            config = libvirt_migrate.get_updated_guest_xml(
                instance,
                guest,
                objects.LibvirtLiveMigrateData(
                    bdms=[bdmi],
                    serial_listen_addr = '127.0.0.1',
                    serial_listen_ports = [1234]
                ),
                drvr._get_volume_config
            )
            self.assertEqual(target_xml, config)

    def test_update_volume_xml_no_connection_info(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        initial_xml = self.device_xml_tmpl.format(
            device_path='/dev/disk/by-path/'
            'ip-1.2.3.4:3260-iqn.'
            'abc.12345.opst-lun-X')
        target_xml = self.device_xml_tmpl.format(
            device_path='/dev/disk/by-path/'
            'ip-1.2.3.4:3260-iqn.'
            'abc.12345.opst-lun-X')
        target_xml = etree.tostring(etree.fromstring(target_xml),
                                    encoding='unicode')
        serial = "58a84f6d-3f0c-4e19-a0af-eb657b790657"
        bdmi = objects.LibvirtLiveMigrateBDMInfo(serial=serial,
                                                 dev='vdb',
                                                 type='disk',
                                                 bus='scsi',
                                                 format='qcow')
        bdmi.connection_info = {}
        conf = vconfig.LibvirtConfigGuestDisk()
        instance = objects.Instance(**self.test_instance)
        guest = libvirt_guest.Guest(mock.MagicMock())

        with test.nested(
            mock.patch.object(
                drvr, '_get_volume_config', return_value=conf
            ),
            mock.patch.object(
                guest, 'get_xml_desc', return_value=initial_xml
            ),
            mock.patch.object(
                drvr, '_sev_enabled', new=mock.Mock(return_value=False)
            )
        ):
            config = libvirt_migrate.get_updated_guest_xml(
                instance,
                guest,
                objects.LibvirtLiveMigrateData(
                    bdms=[bdmi],
                    serial_listen_addr='127.0.0.1',
                    serial_listen_ports=[1234]
                ),
                drvr._get_volume_config
            )
            self.assertEqual(target_xml, config)

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_get_serial_ports_from_guest')
    @mock.patch.object(fakelibvirt.virDomain, "migrateToURI3")
    @mock.patch.object(fakelibvirt.virDomain, "XMLDesc")
    def test_live_migration_update_serial_console_xml(self, mock_xml,
                                                      mock_migrateToURI3,
                                                      mock_get,
                                                      mock_min_version):
        self.compute = manager.ComputeManager()
        instance_ref = objects.Instance(**self.test_instance)
        target_connection = '127.0.0.2'

        xml_tmpl = ("<domain type='kvm'>"
                    "<devices>"
                    "<console type='tcp'>"
                    "<source mode='bind' host='{addr}' service='{port}'/>"
                    "<target type='serial' port='0'/>"
                    "</console>"
                    "</devices>"
                    "</domain>")

        initial_xml = xml_tmpl.format(addr='9.0.0.1', port='10100')

        target_xml = xml_tmpl.format(addr='9.0.0.12', port='10200')
        target_xml = etree.tostring(etree.fromstring(target_xml),
                                    encoding='unicode')

        # Preparing mocks
        mock_xml.return_value = initial_xml
        mock_migrateToURI3.side_effect = fakelibvirt.libvirtError("ERR")

        disk_paths = ['vda', 'vdb']
        params = {
            'migrate_uri': 'tcp://127.0.0.2',
            'migrate_disks': ['vda', 'vdb'],
            'bandwidth': CONF.libvirt.live_migration_bandwidth,
            'destination_xml': target_xml,
            'persistent_xml': target_xml,
        }

        # start test
        migrate_data = objects.LibvirtLiveMigrateData(
            graphics_listen_addr_vnc='10.0.0.1',
            graphics_listen_addr_spice='10.0.0.2',
            serial_listen_addr='9.0.0.12',
            target_connect_addr=target_connection,
            bdms=[],
            block_migration=False,
            serial_listen_ports=[10200])
        dom = fakelibvirt.virDomain
        guest = libvirt_guest.Guest(dom)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertRaises(fakelibvirt.libvirtError,
                          drvr._live_migration_operation,
                          self.context, instance_ref, target_connection,
                          False, migrate_data, guest, disk_paths)
        mock_xml.assert_called_once_with(
                flags=fakelibvirt.VIR_DOMAIN_XML_MIGRATABLE)
        mock_migrateToURI3.assert_called_once_with(
                drvr._live_migration_uri(target_connection),
                params=params, flags=0)

    def test_live_migration_fails_without_serial_console_address(self):
        self.compute = manager.ComputeManager()
        self.flags(enabled=True, group='serial_console')
        self.flags(proxyclient_address='', group='serial_console')
        instance_dict = dict(self.test_instance)
        instance_dict.update({'host': 'fake',
                              'power_state': power_state.RUNNING,
                              'vm_state': vm_states.ACTIVE})
        instance_ref = objects.Instance(**instance_dict)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        # Preparing mocks
        dom = fakelibvirt.virDomain
        guest = libvirt_guest.Guest(dom)

        # start test
        migrate_data = objects.LibvirtLiveMigrateData(
            serial_listen_addr='',
            target_connect_addr=None,
            bdms=[],
            block_migration=False)
        self.assertRaises(exception.MigrationError,
                          drvr._live_migration_operation,
                          self.context, instance_ref, 'dest',
                          False, migrate_data, guest, [])

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    @mock.patch.object(fakelibvirt.virDomain, "migrateToURI3")
    @mock.patch('nova.virt.libvirt.migration.get_updated_guest_xml',
                return_value='')
    @mock.patch('nova.virt.libvirt.guest.Guest.get_xml_desc',
                return_value='<xml></xml>')
    def test_live_migration_uses_migrateToURI3(
            self, mock_old_xml, mock_new_xml, mock_migrateToURI3,
            mock_min_version):

        target_connection = '127.0.0.2'
        # Preparing mocks
        disk_paths = ['vda', 'vdb']
        params = {
            'migrate_uri': 'tcp://127.0.0.2',
            'migrate_disks': ['vda', 'vdb'],
            'bandwidth': CONF.libvirt.live_migration_bandwidth,
        }
        mock_migrateToURI3.side_effect = fakelibvirt.libvirtError("ERR")

        # Start test
        migrate_data = objects.LibvirtLiveMigrateData(
            graphics_listen_addr_vnc='0.0.0.0',
            graphics_listen_addr_spice='0.0.0.0',
            serial_listen_addr='127.0.0.1',
            serial_listen_ports=[1234],
            target_connect_addr=target_connection,
            bdms=[],
            block_migration=False)

        dom = fakelibvirt.virDomain
        guest = libvirt_guest.Guest(dom)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        instance = objects.Instance(**self.test_instance)
        self.assertRaises(fakelibvirt.libvirtError,
                          drvr._live_migration_operation,
                          self.context, instance, target_connection,
                          False, migrate_data, guest, disk_paths)
        mock_migrateToURI3.assert_called_once_with(
            drvr._live_migration_uri(target_connection),
            params=params, flags=0)

    @mock.patch.object(fakelibvirt.virDomain, "migrateToURI3")
    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    @mock.patch('nova.virt.libvirt.guest.Guest.get_xml_desc',
                return_value='<xml/>')
    def _test_live_migration_block_migration_flags(self,
            device_names, expected_flags,
            mock_old_xml, mock_min_version, mock_migrateToURI3):

        target_connection = '127.0.0.2'
        migrate_data = objects.LibvirtLiveMigrateData(
            graphics_listen_addr_vnc='0.0.0.0',
            graphics_listen_addr_spice='0.0.0.0',
            serial_listen_addr='127.0.0.1',
            serial_listen_ports=[1234],
            target_connect_addr=target_connection,
            bdms=[],
            block_migration=True)

        dom = fakelibvirt.virDomain
        guest = libvirt_guest.Guest(dom)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._parse_migration_flags()

        instance = objects.Instance(**self.test_instance)
        drvr._live_migration_operation(self.context, instance,
                                       target_connection,
                                       True, migrate_data, guest,
                                       device_names)

        params = {
            'migrate_uri': 'tcp://127.0.0.2',
            'migrate_disks': device_names,
            'bandwidth': CONF.libvirt.live_migration_bandwidth,
            'destination_xml': '<xml/>',
            'persistent_xml': '<xml/>',
        }
        if not params['migrate_disks']:
            del params['migrate_disks']

        mock_migrateToURI3.assert_called_once_with(
            drvr._live_migration_uri(target_connection), params=params,
            flags=expected_flags)

    def test_live_migration_block_migration_with_devices(self):
        device_names = ['vda']
        expected_flags = (fakelibvirt.VIR_MIGRATE_NON_SHARED_INC |
                          fakelibvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                          fakelibvirt.VIR_MIGRATE_PERSIST_DEST |
                          fakelibvirt.VIR_MIGRATE_PEER2PEER |
                          fakelibvirt.VIR_MIGRATE_LIVE)

        self._test_live_migration_block_migration_flags(device_names,
                                                        expected_flags)

    def test_live_migration_block_migration_all_filtered(self):
        device_names = []
        expected_flags = (fakelibvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                          fakelibvirt.VIR_MIGRATE_PERSIST_DEST |
                          fakelibvirt.VIR_MIGRATE_PEER2PEER |
                          fakelibvirt.VIR_MIGRATE_LIVE)

        self._test_live_migration_block_migration_flags(device_names,
                                                        expected_flags)

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    @mock.patch.object(fakelibvirt.virDomain, "migrateToURI3")
    @mock.patch('nova.virt.libvirt.migration.get_updated_guest_xml',
                return_value='')
    @mock.patch('nova.virt.libvirt.guest.Guest.get_xml_desc', return_value='')
    def test_block_live_migration_tunnelled_migrateToURI3(
            self, mock_old_xml, mock_new_xml,
            mock_migrateToURI3, mock_min_version):
        self.flags(live_migration_tunnelled=True, group='libvirt')

        target_connection = None
        device_names = ['disk1', 'disk2']

        # Preparing mocks
        # Since we are passing the VIR_MIGRATE_TUNNELLED flag, the
        # 'parms' dict will not (as expected) contain 'migrate_disks'
        params = {
            'bandwidth': CONF.libvirt.live_migration_bandwidth
        }
        # Start test
        migrate_data = objects.LibvirtLiveMigrateData(
            graphics_listen_addr_vnc='0.0.0.0',
            graphics_listen_addr_spice='0.0.0.0',
            serial_listen_addr='127.0.0.1',
            serial_listen_ports=[1234],
            target_connect_addr=target_connection,
            bdms=[],
            block_migration=True)

        dom = fakelibvirt.virDomain
        guest = libvirt_guest.Guest(dom)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._parse_migration_flags()
        instance = objects.Instance(**self.test_instance)
        drvr._live_migration_operation(self.context, instance,
                                       target_connection, True, migrate_data,
                                       guest, device_names)

        expected_flags = (fakelibvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                          fakelibvirt.VIR_MIGRATE_PERSIST_DEST |
                          fakelibvirt.VIR_MIGRATE_TUNNELLED |
                          fakelibvirt.VIR_MIGRATE_PEER2PEER |
                          fakelibvirt.VIR_MIGRATE_LIVE)
        mock_migrateToURI3.assert_called_once_with(
            drvr._live_migration_uri(target_connection),
            params=params, flags=expected_flags)

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    @mock.patch.object(fakelibvirt.virDomain, "migrateToURI3")
    @mock.patch('nova.virt.libvirt.migration.get_updated_guest_xml',
                return_value='')
    def test_live_migration_paused_instance_postcopy(self, mock_new_xml,
                                                     mock_migrateToURI3,
                                                     mock_min_version):
        disk_paths = []
        params = {'bandwidth': CONF.libvirt.live_migration_bandwidth}
        migrate_data = objects.LibvirtLiveMigrateData(block_migration=False,
                                                      serial_listen_addr=False)
        dom = fakelibvirt.virDomain
        guest = libvirt_guest.Guest(dom)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._parse_migration_flags()
        instance = objects.Instance(**self.test_instance)
        instance.vm_state = vm_states.PAUSED

        drvr._live_migration_operation(self.context, instance, 'dest',
                                       True, migrate_data, guest,
                                       disk_paths)

        # Verify VIR_MIGRATE_POSTCOPY flag was not set
        self.assertEqual(drvr._live_migration_flags, 27)
        mock_migrateToURI3.assert_called_once_with(
            drvr._live_migration_uri('dest'), params=params, flags=27)

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    @mock.patch.object(fakelibvirt.virDomain, "migrateToURI3")
    @mock.patch('nova.virt.libvirt.migration.get_updated_guest_xml',
                return_value='')
    @mock.patch('nova.virt.libvirt.guest.Guest.get_xml_desc', return_value='')
    def test_block_live_migration_native_tls(
            self, mock_old_xml, mock_new_xml,
            mock_migrateToURI3, mock_min_version):
        self.flags(live_migration_with_native_tls=True, group='libvirt')
        target_connection = None
        disk_paths = ['vda', 'vdb']

        params = {
            'bandwidth': CONF.libvirt.live_migration_bandwidth,
            'migrate_disks': disk_paths
        }

        # Start test
        migrate_data = objects.LibvirtLiveMigrateData(
            graphics_listen_addr_vnc='0.0.0.0',
            graphics_listen_addr_spice='0.0.0.0',
            serial_listen_addr='127.0.0.1',
            serial_listen_ports=[1234],
            target_connect_addr=target_connection,
            bdms=[],
            block_migration=True)

        dom = fakelibvirt.virDomain
        guest = libvirt_guest.Guest(dom)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._parse_migration_flags()
        instance = objects.Instance(**self.test_instance)
        drvr._live_migration_operation(self.context, instance,
                                       target_connection, True, migrate_data,
                                       guest, disk_paths)

        expected_flags = (fakelibvirt.VIR_MIGRATE_UNDEFINE_SOURCE |
                          fakelibvirt.VIR_MIGRATE_PERSIST_DEST |
                          fakelibvirt.VIR_MIGRATE_PEER2PEER |
                          fakelibvirt.VIR_MIGRATE_NON_SHARED_INC |
                          fakelibvirt.VIR_MIGRATE_TLS |
                          fakelibvirt.VIR_MIGRATE_LIVE)
        mock_migrateToURI3.assert_called_once_with(
            drvr._live_migration_uri(target_connection),
            params=params, flags=expected_flags)

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    @mock.patch.object(fakelibvirt.virDomain, "migrateToURI3")
    @mock.patch('nova.virt.libvirt.guest.Guest.get_xml_desc',
                return_value='<xml/>')
    def test_live_migration_raises_exception(self, mock_xml,
                                             mock_migrateToURI3,
                                             mock_min_version):
        # Prepare data
        self.compute = manager.ComputeManager()
        instance_ref = objects.Instance(**self.test_instance)
        target_connection = '127.0.0.2'

        disk_paths = ['vda', 'vdb']
        params = {
            'migrate_uri': 'tcp://127.0.0.2',
            'migrate_disks': disk_paths,
            'bandwidth': CONF.libvirt.live_migration_bandwidth,
            'destination_xml': '<xml/>',
            'persistent_xml': '<xml/>',
        }

        # Prepare mocks
        mock_migrateToURI3.side_effect = fakelibvirt.libvirtError("ERR")

        # Start test
        migrate_data = objects.LibvirtLiveMigrateData(
            graphics_listen_addr_vnc='10.0.0.1',
            graphics_listen_addr_spice='10.0.0.2',
            serial_listen_addr='127.0.0.1',
            serial_listen_ports=[1234],
            target_connect_addr=target_connection,
            bdms=[],
            block_migration=False)
        dom = fakelibvirt.virDomain
        guest = libvirt_guest.Guest(dom)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertRaises(fakelibvirt.libvirtError,
                          drvr._live_migration_operation,
                          self.context, instance_ref, target_connection,
                          False, migrate_data, guest, disk_paths)
        mock_migrateToURI3.assert_called_once_with(
                drvr._live_migration_uri(target_connection),
                params=params, flags=0)

    @mock.patch('shutil.rmtree')
    @mock.patch('os.path.exists', return_value=True)
    @mock.patch('nova.virt.libvirt.utils.get_instance_path_at_destination')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.destroy')
    def test_rollback_live_migration_at_dest_not_shared(self, mock_destroy,
                                                        mock_get_instance_path,
                                                        mock_exist,
                                                        mock_shutil
                                                        ):
        # destroy method may raise InstanceTerminationFailure or
        # InstancePowerOffFailure, here use their base class Invalid.
        mock_destroy.side_effect = exception.Invalid(reason='just test')
        fake_instance_path = os.path.join(cfg.CONF.instances_path,
                                          '/fake_instance_uuid')
        mock_get_instance_path.return_value = fake_instance_path
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        instance = objects.Instance(id=1, uuid=uuids.instance)
        migrate_data = objects.LibvirtLiveMigrateData(
            is_shared_instance_path=False,
            instance_relative_path=False)
        self.assertRaises(exception.Invalid,
                          drvr.rollback_live_migration_at_destination,
                          "context", instance, [], None, True, migrate_data)
        mock_exist.assert_called_once_with(fake_instance_path)
        mock_shutil.assert_called_once_with(fake_instance_path)

    @mock.patch('shutil.rmtree')
    @mock.patch('os.path.exists')
    @mock.patch('nova.virt.libvirt.utils.get_instance_path_at_destination')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.destroy')
    def test_rollback_live_migration_at_dest_shared(self, mock_destroy,
                                                    mock_get_instance_path,
                                                    mock_exist,
                                                    mock_shutil
                                                    ):

        def fake_destroy(ctxt, instance, network_info,
                         block_device_info=None, destroy_disks=True):
            # This is just here to test the signature. Seems there should
            # be a better way to do this with mock and autospec.
            pass

        mock_destroy.side_effect = fake_destroy
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertEqual({}, drvr.instance_claimed_mdevs)

        instance = objects.Instance(id=1, uuid=uuids.instance)
        migrate_data = objects.LibvirtLiveMigrateData(
            is_shared_instance_path=True,
            instance_relative_path=False)
        drvr.instance_claimed_mdevs = {instance.uuid: [uuids.mdev1],
                                       uuids.other_inst: [uuids.mdev2]}
        drvr.rollback_live_migration_at_destination("context", instance, [],
                                                    None, True, migrate_data)
        mock_destroy.assert_called_once_with("context", instance, [],
                                             None, True)
        self.assertFalse(mock_get_instance_path.called)
        self.assertFalse(mock_exist.called)
        self.assertFalse(mock_shutil.called)
        # Assert we delete existing claimed mdevs for the instance
        self.assertEqual({uuids.other_inst: [uuids.mdev2]},
                         drvr.instance_claimed_mdevs)

    @mock.patch.object(fakelibvirt.Domain, "XMLDesc")
    def test_live_migration_copy_disk_paths_tunnelled(self, mock_xml):
        self.flags(live_migration_tunnelled=True, group='libvirt')
        xml = """
        <domain>
          <name>dummy</name>
          <uuid>d4e13113-918e-42fe-9fc9-861693ffd432</uuid>
          <devices>
            <disk type="file">
               <source file="/var/lib/nova/instance/123/disk.root"/>
               <target dev="vda"/>
            </disk>
            <disk type="file">
               <source file="/var/lib/nova/instance/123/disk.shared"/>
               <target dev="vdb"/>
               <shareable/>
            </disk>
            <disk type="file">
               <source file="/var/lib/nova/instance/123/disk.config"/>
               <target dev="vdc"/>
               <readonly/>
            </disk>
            <disk type="block">
               <source dev="/dev/mapper/somevol"/>
               <target dev="vdd"/>
            </disk>
            <disk type="network">
               <source protocol="https" name="url_path">
                 <host name="hostname" port="443"/>
               </source>
            </disk>
          </devices>
        </domain>"""
        mock_xml.return_value = xml

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._parse_migration_flags()
        dom = fakelibvirt.Domain(drvr._get_connection(), xml, False)
        guest = libvirt_guest.Guest(dom)

        paths = drvr._live_migration_copy_disk_paths(None, None, guest)
        self.assertEqual((["/var/lib/nova/instance/123/disk.root",
                          "/dev/mapper/somevol"], ['vda', 'vdd']), paths)

    @mock.patch.object(host.Host, "get_connection")
    @mock.patch.object(host.Host, "has_min_version", return_value=True)
    @mock.patch('nova.virt.driver.get_block_device_info')
    @mock.patch('nova.objects.BlockDeviceMappingList.get_by_instance_uuid')
    @mock.patch.object(fakelibvirt.Domain, "XMLDesc")
    def test_live_migration_copy_disk_paths_selective_block_migration(
            self, mock_xml, mock_get_instance,
            mock_block_device_info, mock_version, mock_conn):
        xml = """
        <domain>
          <name>dummy</name>
          <uuid>d4e13113-918e-42fe-9fc9-861693ffd432</uuid>
          <devices>
            <disk type="file">
               <source file="/var/lib/nova/instance/123/disk.root"/>
               <target dev="vda"/>
            </disk>
            <disk type="file">
               <source file="/var/lib/nova/instance/123/disk.shared"/>
               <target dev="vdb"/>
            </disk>
            <disk type="file">
               <source file="/var/lib/nova/instance/123/disk.config"/>
               <target dev="vdc"/>
            </disk>
            <disk type="block">
               <source dev="/dev/mapper/somevol"/>
               <target dev="vdd"/>
            </disk>
            <disk type="network">
               <source protocol="https" name="url_path">
                 <host name="hostname" port="443"/>
               </source>
            </disk>
          </devices>
        </domain>"""
        mock_xml.return_value = xml
        instance = objects.Instance(**self.test_instance)
        instance.root_device_name = '/dev/vda'
        block_device_info = {
            'swap': {
                'disk_bus': u'virtio',
                'swap_size': 10,
                'device_name': u'/dev/vdc'
            },
            'root_device_name': u'/dev/vda',
            'ephemerals': [{
                'guest_format': u'ext3',
                'device_name': u'/dev/vdb',
                'disk_bus': u'virtio',
                'device_type': u'disk',
                'size': 1
            }],
            'block_device_mapping': [{
                'guest_format': None,
                'boot_index': None,
                'mount_device': u'/dev/vdd',
                'connection_info': {
                    u'driver_volume_type': u'iscsi',
                    'serial': u'147df29f-aec2-4851-b3fe-f68dad151834',
                    u'data': {
                        u'access_mode': u'rw',
                        u'target_discovered': False,
                        u'encrypted': False,
                        u'qos_specs': None,
                        u'target_iqn': u'iqn.2010-10.org.openstack:'
                                       u'volume-147df29f-aec2-4851-b3fe-'
                                       u'f68dad151834',
                        u'target_portal': u'10.102.44.141:3260', u'volume_id':
                            u'147df29f-aec2-4851-b3fe-f68dad151834',
                        u'target_lun': 1,
                        u'auth_password': u'cXELT66FngwzTwpf',
                        u'auth_username': u'QbQQjj445uWgeQkFKcVw',
                        u'auth_method': u'CHAP'
                    }
                },
                'disk_bus': None,
                'device_type': None,
                'delete_on_termination': False
            }]
        }
        mock_block_device_info.return_value = block_device_info
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        dom = fakelibvirt.Domain(drvr._get_connection(), xml, False)
        guest = libvirt_guest.Guest(dom)
        return_value = drvr._live_migration_copy_disk_paths(self.context,
                                                            instance,
                                                            guest)
        expected = (['/var/lib/nova/instance/123/disk.root',
                     '/var/lib/nova/instance/123/disk.shared',
                     '/var/lib/nova/instance/123/disk.config'],
                    ['vda', 'vdb', 'vdc'])
        self.assertEqual(expected, return_value)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       "_live_migration_copy_disk_paths")
    def test_live_migration_data_gb_plain(self, mock_paths):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)

        data_gb = drvr._live_migration_data_gb(instance, [])
        self.assertEqual(2, data_gb)
        self.assertEqual(0, mock_paths.call_count)

    def test_live_migration_data_gb_block(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)

        def fake_stat(path):
            class StatResult(object):
                def __init__(self, size):
                    self._size = size

                @property
                def st_size(self):
                    return self._size

            if path == "/var/lib/nova/instance/123/disk.root":
                return StatResult(10 * units.Gi)
            elif path == "/dev/mapper/somevol":
                return StatResult(1.5 * units.Gi)
            else:
                raise Exception("Should not be reached")

        disk_paths = ["/var/lib/nova/instance/123/disk.root",
                      "/dev/mapper/somevol"]
        with mock.patch.object(os, "stat") as mock_stat:
            mock_stat.side_effect = fake_stat
            data_gb = drvr._live_migration_data_gb(instance, disk_paths)
            # Expecting 2 GB for RAM, plus 10 GB for disk.root
            # and 1.5 GB rounded to 2 GB for somevol, so 14 GB
            self.assertEqual(14, data_gb)

    EXPECT_SUCCESS = 1
    EXPECT_FAILURE = 2
    EXPECT_ABORT = 3

    @mock.patch.object(libvirt_guest.Guest, "migrate_start_postcopy")
    @mock.patch.object(time, "time")
    @mock.patch.object(time, "sleep",
                       side_effect=lambda x: eventlet.sleep(0))
    @mock.patch.object(host.Host, "get_connection")
    @mock.patch.object(libvirt_guest.Guest, "get_job_info")
    @mock.patch.object(objects.Instance, "save")
    @mock.patch.object(objects.Migration, "save")
    @mock.patch.object(fakelibvirt.Connection, "_mark_running")
    @mock.patch.object(fakelibvirt.virDomain, "abortJob")
    @mock.patch.object(libvirt_guest.Guest, "pause")
    def _test_live_migration_monitoring(self,
                                        job_info_records,
                                        time_records,
                                        expect_result,
                                        mock_pause,
                                        mock_abort,
                                        mock_running,
                                        mock_save,
                                        mock_mig_save,
                                        mock_job_info,
                                        mock_conn,
                                        mock_sleep,
                                        mock_time,
                                        mock_postcopy_switch,
                                        current_mig_status=None,
                                        expected_mig_status=None,
                                        scheduled_action=None,
                                        scheduled_action_executed=False,
                                        block_migration=False,
                                        expected_switch=False):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        drvr.active_migrations[instance.uuid] = collections.deque()
        dom = fakelibvirt.Domain(drvr._get_connection(), "<domain/>", True)
        guest = libvirt_guest.Guest(dom)
        finish_event = eventlet.event.Event()

        def fake_job_info():
            while True:
                self.assertGreater(len(job_info_records), 0)
                rec = job_info_records.pop(0)

                if type(rec) is str:
                    if rec == "thread-finish":
                        finish_event.send()
                    elif rec == "domain-stop":
                        dom.destroy()
                    elif rec == "force_complete":
                        drvr.active_migrations[instance.uuid].append(
                            "force-complete")
                else:
                    if len(time_records) > 0:
                        time_records.pop(0)
                    return rec
            return rec

        def fake_time():
            if len(time_records) > 0:
                return time_records[0]
            else:
                return int(
                    datetime.datetime(2001, 1, 20, 20, 1, 0)
                    .strftime('%s'))

        mock_job_info.side_effect = fake_job_info
        mock_time.side_effect = fake_time

        dest = mock.sentinel.migrate_dest
        migration = objects.Migration(context=self.context, id=1)
        migrate_data = objects.LibvirtLiveMigrateData(
            migration=migration, block_migration=block_migration)

        if current_mig_status:
            migrate_data.migration.status = current_mig_status
        else:
            migrate_data.migration.status = "unset"
        migrate_data.migration.save()

        fake_post_method = mock.MagicMock()
        fake_recover_method = mock.MagicMock()
        drvr._live_migration_monitor(self.context, instance,
                                     guest, dest,
                                     fake_post_method,
                                     fake_recover_method,
                                     False,
                                     migrate_data,
                                     finish_event,
                                     [])
        if scheduled_action_executed:
            if scheduled_action == 'pause':
                self.assertTrue(mock_pause.called)
            if scheduled_action == 'postcopy_switch':
                self.assertTrue(mock_postcopy_switch.called)
        else:
            if scheduled_action == 'pause':
                self.assertFalse(mock_pause.called)
            if scheduled_action == 'postcopy_switch':
                self.assertFalse(mock_postcopy_switch.called)
        mock_mig_save.assert_called_with()

        if expect_result == self.EXPECT_SUCCESS:
            self.assertFalse(fake_recover_method.called,
                             'Recover method called when success expected')
            self.assertFalse(mock_abort.called,
                             'abortJob not called when success expected')
            if expected_switch:
                self.assertTrue(mock_postcopy_switch.called)
            fake_post_method.assert_called_once_with(
                self.context, instance, dest, False, migrate_data)
        else:
            if expect_result == self.EXPECT_ABORT:
                self.assertTrue(mock_abort.called,
                                'abortJob called when abort expected')
            else:
                self.assertFalse(mock_abort.called,
                                 'abortJob not called when failure expected')
            self.assertFalse(fake_post_method.called,
                             'Post method called when success not expected')
            if expected_mig_status:
                fake_recover_method.assert_called_once_with(
                    self.context, instance, dest, migrate_data,
                    migration_status=expected_mig_status)
            else:
                fake_recover_method.assert_called_once_with(
                    self.context, instance, dest, migrate_data)
        self.assertNotIn(instance.uuid, drvr.active_migrations)

    def test_live_migration_monitor_success(self):
        # A normal sequence where see all the normal job states
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_COMPLETED),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                                             self.EXPECT_SUCCESS)

    def test_live_migration_handle_pause_normal(self):
        # A normal sequence where see all the normal job states, and pause
        # scheduled in between VIR_DOMAIN_JOB_UNBOUNDED
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "force_complete",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_COMPLETED),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                                             self.EXPECT_SUCCESS,
                                             current_mig_status="running",
                                             scheduled_action="pause",
                                             scheduled_action_executed=True)

    def test_live_migration_handle_pause_on_start(self):
        # A normal sequence where see all the normal job states, and pause
        # scheduled in case of job type VIR_DOMAIN_JOB_NONE and finish_event is
        # not ready yet
        domain_info_records = [
            "force_complete",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_COMPLETED),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                                             self.EXPECT_SUCCESS,
                                             current_mig_status="preparing",
                                             scheduled_action="pause",
                                             scheduled_action_executed=True)

    def test_live_migration_handle_pause_on_finish(self):
        # A normal sequence where see all the normal job states, and pause
        # scheduled in case of job type VIR_DOMAIN_JOB_NONE and finish_event is
        # ready
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            "force_complete",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_COMPLETED),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                                             self.EXPECT_SUCCESS,
                                             current_mig_status="completed",
                                             scheduled_action="pause",
                                             scheduled_action_executed=False)

    def test_live_migration_handle_pause_on_cancel(self):
        # A normal sequence where see all the normal job states, and pause
        # scheduled in case of job type VIR_DOMAIN_JOB_CANCELLED
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            "force_complete",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_CANCELLED),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                                             self.EXPECT_FAILURE,
                                             current_mig_status="cancelled",
                                             expected_mig_status='cancelled',
                                             scheduled_action="pause",
                                             scheduled_action_executed=False)

    def test_live_migration_handle_pause_on_failure(self):
        # A normal sequence where see all the normal job states, and pause
        # scheduled in case of job type VIR_DOMAIN_JOB_FAILED
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            "force_complete",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_FAILED),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                                             self.EXPECT_FAILURE,
                                             scheduled_action="pause",
                                             scheduled_action_executed=False)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       "_is_post_copy_enabled")
    def test_live_migration_handle_postcopy_normal(self,
            mock_postcopy_enabled):
        # A normal sequence where see all the normal job states, and postcopy
        # switch scheduled in between VIR_DOMAIN_JOB_UNBOUNDED
        mock_postcopy_enabled.return_value = True
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "force_complete",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_COMPLETED),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                self.EXPECT_SUCCESS,
                current_mig_status="running",
                scheduled_action="postcopy_switch",
                scheduled_action_executed=True)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       "_is_post_copy_enabled")
    def test_live_migration_handle_postcopy_on_start(self,
            mock_postcopy_enabled):
        # A normal sequence where see all the normal job states, and postcopy
        # switch scheduled in case of job type VIR_DOMAIN_JOB_NONE and
        # finish_event is not ready yet
        mock_postcopy_enabled.return_value = True
        domain_info_records = [
            "force_complete",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_COMPLETED),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                self.EXPECT_SUCCESS,
                current_mig_status="preparing",
                scheduled_action="postcopy_switch",
                scheduled_action_executed=True)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       "_is_post_copy_enabled")
    def test_live_migration_handle_postcopy_on_finish(self,
            mock_postcopy_enabled):
        # A normal sequence where see all the normal job states, and postcopy
        # switch scheduled in case of job type VIR_DOMAIN_JOB_NONE and
        # finish_event is ready
        mock_postcopy_enabled.return_value = True
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            "force_complete",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_COMPLETED),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                self.EXPECT_SUCCESS,
                current_mig_status="completed",
                scheduled_action="postcopy_switch",
                scheduled_action_executed=False)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       "_is_post_copy_enabled")
    def test_live_migration_handle_postcopy_on_cancel(self,
            mock_postcopy_enabled):
        # A normal sequence where see all the normal job states, and postcopy
        # scheduled in case of job type VIR_DOMAIN_JOB_CANCELLED
        mock_postcopy_enabled.return_value = True
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            "force_complete",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_CANCELLED),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                self.EXPECT_FAILURE,
                current_mig_status="cancelled",
                expected_mig_status='cancelled',
                scheduled_action="postcopy_switch",
                scheduled_action_executed=False)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       "_is_post_copy_enabled")
    def test_live_migration_handle_pause_on_postcopy(self,
            mock_postcopy_enabled):
        # A normal sequence where see all the normal job states, and pause
        # scheduled after migration switched to postcopy
        mock_postcopy_enabled.return_value = True
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "force_complete",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_COMPLETED),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                self.EXPECT_SUCCESS,
                current_mig_status="running (post-copy)",
                scheduled_action="pause",
                scheduled_action_executed=False)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       "_is_post_copy_enabled")
    def test_live_migration_handle_postcopy_on_postcopy(self,
            mock_postcopy_enabled):
        # A normal sequence where see all the normal job states, and pause
        # scheduled after migration switched to postcopy
        mock_postcopy_enabled.return_value = True
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "force_complete",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_COMPLETED),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                self.EXPECT_SUCCESS,
                current_mig_status="running (post-copy)",
                scheduled_action="postcopy_switch",
                scheduled_action_executed=False)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       "_is_post_copy_enabled")
    def test_live_migration_handle_postcopy_on_failure(self,
            mock_postcopy_enabled):
        # A normal sequence where see all the normal job states, and postcopy
        # scheduled in case of job type VIR_DOMAIN_JOB_FAILED
        mock_postcopy_enabled.return_value = True
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            "force_complete",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_FAILED),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                self.EXPECT_FAILURE,
                scheduled_action="postcopy_switch",
                scheduled_action_executed=False)

    def test_live_migration_monitor_success_race(self):
        # A normalish sequence but we're too slow to see the
        # completed job state
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                                             self.EXPECT_SUCCESS)

    def test_live_migration_monitor_failed(self):
        # A failed sequence where we see all the expected events
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_FAILED),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                                             self.EXPECT_FAILURE)

    def test_live_migration_monitor_failed_race(self):
        # A failed sequence where we are too slow to see the
        # failed event
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                                             self.EXPECT_FAILURE)

    def test_live_migration_monitor_cancelled(self):
        # A cancelled sequence where we see all the events
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_CANCELLED),
        ]

        self._test_live_migration_monitoring(domain_info_records, [],
                                             self.EXPECT_FAILURE,
                                             expected_mig_status='cancelled')

    @mock.patch.object(fakelibvirt.virDomain, "migrateSetMaxDowntime")
    @mock.patch("nova.virt.libvirt.migration.downtime_steps")
    def test_live_migration_monitor_downtime(self, mock_downtime_steps,
                                             mock_set_downtime):
        self.flags(live_migration_completion_timeout=1000000,
                   group='libvirt')
        # We've setup 4 fake downtime steps - first value is the
        # time delay, second is the downtime value
        downtime_steps = [
            (90, 10),
            (180, 50),
            (270, 200),
            (500, 300),
        ]
        mock_downtime_steps.return_value = downtime_steps

        # Each one of these fake times is used for time.time()
        # when a new domain_info_records entry is consumed.
        # Times are chosen so that only the first 3 downtime
        # steps are needed.
        fake_times = [0, 1, 30, 95, 150, 200, 300]

        # A normal sequence where see all the normal job states
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_COMPLETED),
        ]

        self._test_live_migration_monitoring(domain_info_records,
                                             fake_times, self.EXPECT_SUCCESS)

        mock_set_downtime.assert_has_calls([mock.call(10),
                                            mock.call(50),
                                            mock.call(200)])

    def test_live_migration_monitor_completion(self):
        self.flags(live_migration_completion_timeout=100,
                   group='libvirt')
        # Each one of these fake times is used for time.time()
        # when a new domain_info_records entry is consumed.
        fake_times = [0, 40, 80, 120, 160, 200, 240, 280, 320]

        # A normal sequence where see all the normal job states
        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_CANCELLED),
        ]

        self._test_live_migration_monitoring(domain_info_records,
                                             fake_times, self.EXPECT_ABORT,
                                             expected_mig_status='cancelled')

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       "_is_post_copy_enabled")
    def test_live_migration_monitor_force_complete_postcopy(self,
            mock_postcopy_enabled):
        self.flags(live_migration_completion_timeout=40,
                   live_migration_timeout_action='force_complete',
                   group='libvirt')
        mock_postcopy_enabled.return_value = True

        # Each one of these fake times is used for time.time()
        # when a new domain_info_records entry is consumed.
        fake_times = [0, 40, 80, 120, 160, 200, 240, 280, 320]

        domain_info_records = [
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_NONE),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_UNBOUNDED),
            "thread-finish",
            "domain-stop",
            libvirt_guest.JobInfo(
                type=fakelibvirt.VIR_DOMAIN_JOB_COMPLETED),
        ]

        self._test_live_migration_monitoring(domain_info_records, fake_times,
                                             self.EXPECT_SUCCESS,
                                             expected_switch=True)

    @mock.patch.object(host.Host, "get_connection")
    @mock.patch.object(utils, "spawn")
    @mock.patch.object(libvirt_driver.LibvirtDriver, "_live_migration_monitor")
    @mock.patch.object(host.Host, "get_guest")
    @mock.patch.object(fakelibvirt.Connection, "_mark_running")
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       "_live_migration_copy_disk_paths")
    @mock.patch.object(libvirt_driver.LibvirtDriver, "live_migration_abort")
    def _test_live_migration_main(self, mock_abort, mock_copy_disk_path,
                                  mock_running, mock_guest, mock_monitor,
                                  mock_thread, mock_conn,
                                  mon_side_effect=None):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)

        dom = fakelibvirt.Domain(drvr._get_connection(),
                                 "<domain><name>demo</name></domain>", True)
        guest = libvirt_guest.Guest(dom)
        migrate_data = objects.LibvirtLiveMigrateData(block_migration=True)
        disks_to_copy = (['/some/path/one', '/test/path/two'],
                         ['vda', 'vdb'])
        mock_copy_disk_path.return_value = disks_to_copy

        mock_guest.return_value = guest
        mock_monitor.side_effect = mon_side_effect

        def fake_post():
            pass

        def fake_recover():
            pass

        if mon_side_effect:
            self.assertRaises(mon_side_effect, drvr._live_migration,
                              self.context, instance, "fakehost", fake_post,
                              fake_recover, True, migrate_data)
            mock_abort.assert_called_once_with(instance)
        else:
            drvr._live_migration(self.context, instance, "fakehost", fake_post,
                                 fake_recover, True, migrate_data)

        mock_copy_disk_path.assert_called_once_with(self.context, instance,
                                                    guest)

        class AnyEventletEvent(object):
            def __eq__(self, other):
                return type(other) is eventlet.event.Event

        mock_thread.assert_called_once_with(
            drvr._live_migration_operation,
            self.context, instance, "fakehost", True,
            migrate_data, guest, disks_to_copy[1])
        mock_monitor.assert_called_once_with(
            self.context, instance, guest, "fakehost",
            fake_post, fake_recover, True,
            migrate_data, AnyEventletEvent(), disks_to_copy[0])

    def test_live_migration_main(self):
        self._test_live_migration_main()

    def test_live_migration_main_monitoring_failed(self):
        self._test_live_migration_main(mon_side_effect=Exception)

    @mock.patch.object(host.Host, "get_connection", new=mock.Mock())
    @mock.patch.object(utils, "spawn", new=mock.Mock())
    @mock.patch.object(host.Host, "get_guest")
    @mock.patch.object(
        libvirt_driver.LibvirtDriver, "_live_migration_copy_disk_paths")
    def _test_live_migration_monitor_job_stats_exception(
        self, exc, mock_copy_disk_paths, mock_get_guest, expect_success=True
    ):
        # Verify behavior when various exceptions are raised inside of
        # Guest.get_job_info() during live migration monitoring.
        mock_domain = mock.Mock(fakelibvirt.virDomain)
        guest = libvirt_guest.Guest(mock_domain)
        mock_get_guest.return_value = guest

        # First, raise the exception from jobStats(), then return "completed"
        # to make sure we exit the monitoring loop.
        guest._domain.jobStats.side_effect = [
            exc,
            {'type': fakelibvirt.VIR_DOMAIN_JOB_COMPLETED},
        ]

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        post_method = mock.Mock()
        migrate_data = mock.Mock()
        disks_to_copy = (['/some/path/one', '/test/path/two'],
                         ['vda', 'vdb'])
        mock_copy_disk_paths.return_value = disks_to_copy

        func = drvr._live_migration
        args = (self.context, instance, mock.sentinel.dest, post_method,
                mock.sentinel.recover_method, mock.sentinel.block_migration,
                migrate_data)

        if expect_success:
            func(*args)
            post_method.assert_called_once_with(
                self.context, instance, mock.sentinel.dest,
                mock.sentinel.block_migration, migrate_data
            )
        else:
            actual_exc = self.assertRaises(
                fakelibvirt.libvirtError, func, *args)
            self.assertEqual(exc, actual_exc)

    def test_live_migration_monitor_job_stats_no_domain(self):
        exp = fakelibvirt.make_libvirtError(
            fakelibvirt.libvirtError, 'no domain',
            error_code=fakelibvirt.VIR_ERR_NO_DOMAIN
        )
        self._test_live_migration_monitor_job_stats_exception(
            exp, expect_success=True)

    def test_live_migration_monitor_job_stats_op_invalid(self):
        exp = fakelibvirt.make_libvirtError(
            fakelibvirt.libvirtError, 'operation invalid',
            error_code=fakelibvirt.VIR_ERR_OPERATION_INVALID
        )
        self._test_live_migration_monitor_job_stats_exception(
            exp, expect_success=True)

    def test_live_migration_monitor_job_stats_no_ram_info_set(self):
        exp = fakelibvirt.make_libvirtError(
            fakelibvirt.libvirtError, 'internal error',
            error_message='migration was active, but no RAM info was set',
            error_code=fakelibvirt.VIR_ERR_INTERNAL_ERROR
        )
        self._test_live_migration_monitor_job_stats_exception(
            exp, expect_success=True)

    def test_live_migration_monitor_job_stats_internal_error(self):
        exp = fakelibvirt.make_libvirtError(
            fakelibvirt.libvirtError,
            'some other internal error',
            error_code=fakelibvirt.VIR_ERR_INTERNAL_ERROR
        )
        self._test_live_migration_monitor_job_stats_exception(
            exp, expect_success=False)

    @mock.patch('os.path.exists', return_value=False)
    @mock.patch('nova.virt.libvirt.utils.create_image')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_fetch_instance_kernel_ramdisk')
    def _do_test_create_images_and_backing(self, disk_type, mock_fetch,
                                           mock_create, mock_exists):
        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        disk_info = {'path': 'foo', 'type': disk_type,
                     'disk_size': 1 * 1024 ** 3,
                     'virt_disk_size': 20 * 1024 ** 3,
                     'backing_file': None}

        drvr._create_images_and_backing(self.context, instance,
                                        "/fake/instance/dir", [disk_info])

        mock_fetch.assert_called_once_with(self.context, instance,
                                           fallback_from_host=None)
        mock_create.assert_called_once_with(
             '/fake/instance/dir/foo',
             disk_info['type'],
             disk_info['virt_disk_size'],
        )
        mock_exists.assert_called_once_with('/fake/instance/dir/foo')

    def test_create_images_and_backing_qcow2(self):
        self._do_test_create_images_and_backing('qcow2')

    def test_create_images_and_backing_raw(self):
        self._do_test_create_images_and_backing('raw')

    def test_create_images_and_backing_images_not_exist_no_fallback(self):
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.test_instance.update({'user_id': 'fake-user',
                                   'os_type': None,
                                   'project_id': 'fake-project'})
        instance = objects.Instance(**self.test_instance)

        backing_file = imagecache.get_cache_fname(instance.image_ref)
        disk_info = [
            {u'backing_file': backing_file,
             u'disk_size': 10747904,
             u'path': u'disk_path',
             u'type': u'qcow2',
             u'virt_disk_size': 25165824}]

        with mock.patch.object(libvirt_driver.libvirt_utils, 'fetch_image',
                               side_effect=exception.ImageNotFound(
                                   image_id="fake_id")):
            self.assertRaises(exception.ImageNotFound,
                              conn._create_images_and_backing,
                              self.context, instance,
                              "/fake/instance/dir", disk_info)

    @mock.patch('nova.virt.libvirt.utils.create_image')
    @mock.patch('nova.privsep.path.utime')
    def test_create_images_and_backing_images_not_exist_fallback(
            self, mock_utime, mock_create_cow_image):
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        base_dir = os.path.join(CONF.instances_path,
                                CONF.image_cache.subdirectory_name)
        trusted_certs = objects.TrustedCerts(
            ids=['0b5d2c72-12cc-4ba6-a8d7-3ff5cc1d8cb8',
                 '674736e3-f25c-405c-8362-bbf991e0ce0a'])
        self.test_instance.update({'user_id': 'fake-user',
                                   'os_type': None,
                                   'trusted_certs': trusted_certs,
                                   'kernel_id': uuids.kernel_id,
                                   'ramdisk_id': uuids.ramdisk_id,
                                   'project_id': 'fake-project'})
        instance = objects.Instance(**self.test_instance)

        backing_file = imagecache.get_cache_fname(instance.image_ref)
        backfile_path = os.path.join(base_dir, backing_file)
        disk_size = 10747904
        virt_disk_size = 25165824
        disk_info = [
            {u'backing_file': backing_file,
             u'disk_size': disk_size,
             u'path': u'disk_path',
             u'type': u'qcow2',
             u'virt_disk_size': virt_disk_size}]

        def fake_copy_image(src, dest, **kwargs):
            # backing file should be present and have a smaller size
            # than instance root disk in order to assert resize_image()
            if dest == backfile_path:
                # dest is created under TempDir() fixture,
                # it will go away after test cleanup
                with open(dest, 'a'):
                    pass
        with test.nested(
            mock.patch.object(libvirt_driver.libvirt_utils, 'copy_image',
                              side_effect=fake_copy_image),
            mock.patch.object(libvirt_driver.libvirt_utils, 'fetch_image',
                              side_effect=exception.ImageNotFound(
                                  image_id=uuids.fake_id)),
            mock.patch.object(imagebackend.Qcow2, 'resize_image'),
            mock.patch.object(imagebackend.Image, 'get_disk_size',
                              return_value=disk_size),
        ) as (copy_image_mock, fetch_image_mock, resize_image_mock,
              get_disk_size_mock):
            conn._create_images_and_backing(self.context, instance,
                                            "/fake/instance/dir", disk_info,
                                            fallback_from_host="fake_host")
            kernel_path = os.path.join(CONF.instances_path,
                                       self.test_instance['uuid'],
                                       'kernel')
            ramdisk_path = os.path.join(CONF.instances_path,
                                        self.test_instance['uuid'],
                                        'ramdisk')
            copy_image_mock.assert_has_calls([
                mock.call(dest=backfile_path, src=backfile_path,
                          host='fake_host', receive=True),
                mock.call(dest=kernel_path, src=kernel_path,
                          host='fake_host', receive=True),
                mock.call(dest=ramdisk_path, src=ramdisk_path,
                          host='fake_host', receive=True)
            ])
            fetch_image_mock.assert_has_calls([
                mock.call(context=self.context,
                          target=backfile_path,
                          image_id=self.test_instance['image_ref'],
                          trusted_certs=trusted_certs),
                mock.call(self.context, kernel_path, instance.kernel_id,
                          trusted_certs),
                mock.call(self.context, ramdisk_path, instance.ramdisk_id,
                          trusted_certs)
            ])
            resize_image_mock.assert_called_once_with(virt_disk_size)

        mock_utime.assert_called()
        mock_create_cow_image.assert_called_once_with(
            '/fake/instance/dir/disk_path',
            'qcow2',
            virt_disk_size,
            backing_file=backfile_path,
        )

    @mock.patch('nova.virt.libvirt.imagebackend.Image.exists',
        new=mock.Mock(return_value=True))
    def test_create_images_backing_images_and_fallback_not_exist(self):
        self.flags(images_type='raw', group='libvirt')
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        base_dir = os.path.join(CONF.instances_path,
                                CONF.image_cache.subdirectory_name)
        self.test_instance.update({
            'user_id': 'fake-user',
            'os_type': None,
            'kernel_id': uuids.kernel_id,
            'ramdisk_id': uuids.ramdisk_id,
            'project_id': 'fake-project'
        })
        instance = objects.Instance(**self.test_instance)

        backing_file = imagecache.get_cache_fname(instance.image_ref)
        backfile_path = os.path.join(base_dir, backing_file)
        disk_size = 10747904
        virt_disk_size = 25165824
        disk_info = [{
            'backing_file': backing_file,
            'disk_size': disk_size,
            'path': 'disk_path',
            'type': 'raw',
            'virt_disk_size': virt_disk_size
        }]

        with test.nested(
            mock.patch.object(libvirt_driver.libvirt_utils, 'copy_image'),
            mock.patch.object(libvirt_driver.libvirt_utils, 'fetch_image',
                side_effect=exception.ImageNotFound(
                    image_id=uuids.fake_id)),
            mock.patch.object(imagebackend.Flat, 'resize_image'),
        ) as (copy_image_mock, fetch_image_mock, resize_image_mock):
            conn._create_images_and_backing(
                self.context, instance, "/fake/instance/dir", disk_info,
                fallback_from_host="fake_host")
            kernel_path = os.path.join(CONF.instances_path,
                self.test_instance['uuid'], 'kernel')
            ramdisk_path = os.path.join(CONF.instances_path,
                self.test_instance['uuid'], 'ramdisk')
            copy_image_mock.assert_has_calls([
                mock.call(dest=kernel_path, src=kernel_path,
                          host='fake_host', receive=True),
                mock.call(dest=ramdisk_path, src=ramdisk_path,
                          host='fake_host', receive=True)
            ])
            fetch_image_mock.assert_has_calls([
                mock.call(context=self.context,
                          target=backfile_path,
                          image_id=self.test_instance['image_ref'],
                          trusted_certs=None),
                mock.call(self.context, kernel_path, instance.kernel_id,
                          None),
                mock.call(self.context, ramdisk_path, instance.ramdisk_id,
                          None)
            ])
            resize_image_mock.assert_called_once_with(virt_disk_size)

    @mock.patch('nova.virt.libvirt.utils.create_image',
                new=mock.NonCallableMock())
    @mock.patch.object(libvirt_driver.libvirt_utils, 'fetch_image')
    def test_create_images_and_backing_images_exist(
            self, mock_fetch_image):
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.test_instance.update({'user_id': 'fake-user',
                                   'os_type': None,
                                   'kernel_id': 'fake_kernel_id',
                                   'ramdisk_id': 'fake_ramdisk_id',
                                   'project_id': 'fake-project'})
        instance = objects.Instance(**self.test_instance)

        disk_info = [
            {u'backing_file': imagecache.get_cache_fname(instance.image_ref),
             u'disk_size': 10747904,
             u'path': u'disk_path',
             u'type': u'qcow2',
             u'virt_disk_size': 25165824}]

        with test.nested(
                mock.patch.object(imagebackend.Image, 'get_disk_size',
                                  return_value=0),
                mock.patch.object(os.path, 'exists', return_value=True)
        ):
            conn._create_images_and_backing(self.context, instance,
                                            '/fake/instance/dir', disk_info)
        self.assertFalse(mock_fetch_image.called)

    @mock.patch('nova.privsep.path.utime')
    @mock.patch('nova.virt.libvirt.utils.create_image')
    def test_create_images_and_backing_ephemeral_gets_created(
            self, mock_create_cow_image, mock_utime):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        base_dir = os.path.join(CONF.instances_path,
                                CONF.image_cache.subdirectory_name)
        instance = objects.Instance(**self.test_instance)
        disk_info_byname = fake_disk_info_byname(instance)

        disk_info_byname['disk.local']['backing_file'] = 'ephemeral_foo'
        disk_info_byname['disk.local']['virt_disk_size'] = 1 * units.Gi

        disk_info = disk_info_byname.values()

        with test.nested(
            mock.patch.object(libvirt_driver.libvirt_utils, 'fetch_image'),
            mock.patch.object(drvr, '_create_ephemeral'),
            mock.patch.object(imagebackend.Image, 'verify_base_size'),
            mock.patch.object(imagebackend.Image, 'get_disk_size')
        ) as (fetch_image_mock, create_ephemeral_mock, verify_base_size_mock,
              disk_size_mock):
            disk_size_mock.return_value = 0
            drvr._create_images_and_backing(self.context, instance,
                                            CONF.instances_path, disk_info)
            self.assertEqual(len(create_ephemeral_mock.call_args_list), 1)

            root_backing, ephemeral_backing = [
                os.path.join(base_dir, name)
                for name in (disk_info_byname['disk']['backing_file'],
                             'ephemeral_foo')
            ]

            create_ephemeral_mock.assert_called_once_with(
                ephemeral_size=1, fs_label='ephemeral_foo',
                os_type='linux', target=ephemeral_backing)

            fetch_image_mock.assert_called_once_with(
                context=self.context, image_id=instance.image_ref,
                target=root_backing, trusted_certs=instance.trusted_certs)

            verify_base_size_mock.assert_has_calls([
                mock.call(root_backing, instance.flavor.root_gb * units.Gi),
                mock.call(ephemeral_backing, 1 * units.Gi)
            ])

            mock_utime.assert_has_calls([
                mock.call(root_backing),
                mock.call(ephemeral_backing)])

            # TODO(efried): Should these be disk_info[path]??
            mock_create_cow_image.assert_has_calls([
                mock.call(
                    CONF.instances_path + '/disk',
                    'qcow2',
                    disk_info_byname['disk']['virt_disk_size'],
                    backing_file=root_backing,
                ),
                mock.call(
                    CONF.instances_path + '/disk.local',
                    'qcow2',
                    disk_info_byname['disk.local']['virt_disk_size'],
                    backing_file=ephemeral_backing,
                ),
            ])

    def test_create_images_and_backing_disk_info_none(self):
        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        backend = self.useFixture(nova_fixtures.LibvirtImageBackendFixture())

        drvr._create_images_and_backing(self.context, instance,
                                        "/fake/instance/dir", None)

        # Assert that we did nothing
        self.assertEqual({}, backend.created_disks)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_fetch_instance_kernel_ramdisk')
    def test_create_images_and_backing_parallels(self, mock_fetch):
        self.flags(virt_type='parallels', group='libvirt')
        instance = objects.Instance(**self.test_instance)
        instance.vm_mode = fields.VMMode.EXE
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        drvr._create_images_and_backing(self.context, instance,
                                        '/fake/instance/dir', None)
        self.assertFalse(mock_fetch.called)

    def _migrate_data_to_dict(self, migrate_data):
        primitive = migrate_data.obj_to_primitive()['nova_object.data']
        primitive['bdms'] = [bdm['nova_object.data'] for bdm in
                             primitive['bdms']]
        return primitive

    def _generate_target_ret(self, target_connect_addr=None):
        return self._migrate_data_to_dict(objects.LibvirtLiveMigrateData(
            block_migration=False,
            instance_relative_path='foo',
            is_shared_block_storage=False,
            is_shared_instance_path=False,
            serial_listen_ports=[],
            supported_perf_events=[],
            graphics_listen_addr_spice='127.0.0.1',
            graphics_listen_addr_vnc='127.0.0.1',
            serial_listen_addr='127.0.0.1',
            target_connect_addr=target_connect_addr,
            bdms=[
                objects.LibvirtLiveMigrateBDMInfo(
                    serial='12345', bus='scsi', dev='sda', type='disk',
                    boot_index=None, format=None,
                    connection_info_json=jsonutils.dumps({
                        'serial': '12345',
                        'data': {
                            'device_path': '/dev/disk/by-path/ip-1.2.3.4:3260'
                                           '-iqn.abc.12345.opst-lun-X'}})),
                objects.LibvirtLiveMigrateBDMInfo(
                    serial='67890', bus='scsi', dev='sdb', type='disk',
                    boot_index=None, format=None,
                    connection_info_json=jsonutils.dumps({
                        'serial': '67890',
                        'data': {
                            'device_path': '/dev/disk/by-path/ip-1.2.3.4:3260'
                                           '-iqn.cde.67890.opst-lun-Z'}}))]))

    def test_pre_live_migration_works_correctly_mocked(self):
        self._test_pre_live_migration_works_correctly_mocked()

    def test_pre_live_migration_with_transport_ip(self):
        self.flags(live_migration_inbound_addr='127.0.0.2',
                   group='libvirt')
        target_ret = self._generate_target_ret('127.0.0.2')
        self._test_pre_live_migration_works_correctly_mocked(
            target_ret=target_ret)

    @mock.patch.object(libvirt_driver.LibvirtDriver, 'plug_vifs')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_connect_volume')
    @mock.patch('nova.virt.libvirt.utils.file_open',
                side_effect=[io.BytesIO(b''), io.BytesIO(b'')])
    def _test_pre_live_migration_works_correctly_mocked(
            self, mock_file_open, mock_connect, mock_plug,
            target_ret=None):
        # Creating testdata
        c = context.get_admin_context()
        instance = objects.Instance(root_device_name='/dev/vda',
                                    **self.test_instance)
        bdms = objects.BlockDeviceMappingList(objects=[
            fake_block_device.fake_bdm_object(c, {
                'connection_info': jsonutils.dumps({
                    'serial': '12345',
                    'data': {
                        'device_path': '/dev/disk/by-path/ip-1.2.3.4:3260'
                                       '-iqn.abc.12345.opst-lun-X'
                    }
                }),
                'device_name': '/dev/sda',
                'volume_id': uuids.volume1,
                'source_type': 'volume',
                'destination_type': 'volume'
            }),
            fake_block_device.fake_bdm_object(c, {
                'connection_info': jsonutils.dumps({
                    'serial': '67890',
                    'data': {
                        'device_path': '/dev/disk/by-path/ip-1.2.3.4:3260'
                                       '-iqn.cde.67890.opst-lun-Z'
                    }
                }),
                'device_name': '/dev/sdb',
                'volume_id': uuids.volume2,
                'source_type': 'volume',
                'destination_type': 'volume'
            })
        ])
        # We go through get_block_device_info to simulate what the
        # ComputeManager sends to the driver (make sure we're using the
        # correct type of BDM objects since there are many of them and
        # they are super confusing).
        block_device_info = driver.get_block_device_info(instance, bdms)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        class FakeNetworkInfo(object):
            def fixed_ips(self):
                return ["test_ip_addr"]

        self.stub_out('nova.virt.libvirt.driver.LibvirtDriver.'
                      '_create_images_and_backing',
                      lambda *args, **kwargs: None)

        nw_info = FakeNetworkInfo()

        expected_connect_calls = []
        for v in block_device_info['block_device_mapping']:
            expected_connect_calls.append(
                mock.call(c, v['connection_info'], instance))

        migrate_data = migrate_data_obj.LibvirtLiveMigrateData(
            block_migration=False,
            instance_relative_path='foo',
            is_shared_block_storage=False,
            is_shared_instance_path=False,
            graphics_listen_addr_vnc='127.0.0.1',
            graphics_listen_addr_spice='127.0.0.1',
            serial_listen_addr='127.0.0.1',
        )

        if not target_ret:
            target_ret = self._generate_target_ret()
        result = drvr.pre_live_migration(
            c, instance, block_device_info, nw_info, None,
            migrate_data=migrate_data)
        result_dict = self._migrate_data_to_dict(result)
        # connection_info_json is a string, needs special handling
        for bdms in zip(target_ret.pop('bdms'), result_dict.pop('bdms')):
            conninfo0 = jsonutils.loads(bdms[0].pop('connection_info_json'))
            conninfo1 = jsonutils.loads(bdms[1].pop('connection_info_json'))
            self.assertEqual(conninfo0, conninfo1)
            self.assertThat(bdms[0], matchers.DictMatches(bdms[1]))
        self.assertThat(target_ret, matchers.DictMatches(result_dict))
        mock_connect.assert_has_calls(expected_connect_calls)
        self.assertEqual(len(expected_connect_calls), mock_connect.call_count)
        mock_plug.assert_called_once_with(test.MatchType(objects.Instance),
                                          nw_info)

    @mock.patch.object(os, 'mkdir')
    @mock.patch('nova.virt.libvirt.utils.get_instance_path_at_destination')
    @mock.patch('nova.virt.libvirt.driver.remotefs.'
                'RemoteFilesystem.copy_file')
    @mock.patch('nova.virt.driver.block_device_info_get_mapping')
    @mock.patch('nova.virt.configdrive.required_by', return_value=True)
    def test_pre_live_migration_block_with_config_drive_success(
            self, mock_required_by, block_device_info_get_mapping,
            mock_copy_file, mock_get_instance_path, mock_mkdir):
        self.flags(config_drive_format='iso9660')
        vol = {'block_device_mapping': [
                  {'connection_info': 'dummy', 'mount_device': '/dev/sda'},
                  {'connection_info': 'dummy', 'mount_device': '/dev/sdb'}]}
        fake_instance_path = os.path.join(cfg.CONF.instances_path,
                                          '/fake_instance_uuid')
        mock_get_instance_path.return_value = fake_instance_path

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        instance = objects.Instance(**self.test_instance)
        migrate_data = objects.LibvirtLiveMigrateData()
        migrate_data.is_shared_instance_path = False
        migrate_data.is_shared_block_storage = False
        migrate_data.block_migration = True
        migrate_data.instance_relative_path = 'foo'
        src = "%s:%s/disk.config" % (instance.host, fake_instance_path)

        result = drvr.pre_live_migration(
            self.context, instance, vol, [], None, migrate_data)

        block_device_info_get_mapping.assert_called_once_with(
            {'block_device_mapping': [
                {'connection_info': 'dummy', 'mount_device': '/dev/sda'},
                {'connection_info': 'dummy', 'mount_device': '/dev/sdb'}
            ]}
        )
        mock_copy_file.assert_called_once_with(src, fake_instance_path)

        migrate_data.graphics_listen_addrs_vnc = '127.0.0.1'
        migrate_data.graphics_listen_addrs_spice = '127.0.0.1'
        migrate_data.serial_listen_addr = '127.0.0.1'
        self.assertEqual(migrate_data, result)

    @mock.patch('nova.virt.driver.block_device_info_get_mapping',
                return_value=())
    @mock.patch('nova.virt.libvirt.utils.file_open',
                side_effect=[io.BytesIO(b''), io.BytesIO(b'')])
    def test_pre_live_migration_block_with_config_drive_mocked_with_vfat(
            self, mock_open, mock_block_device_info_get_mapping):
        self.flags(config_drive_format='vfat')
        # Creating testdata
        vol = {'block_device_mapping': [
            {'connection_info': 'dummy', 'mount_device': '/dev/sda'},
            {'connection_info': 'dummy', 'mount_device': '/dev/sdb'}]}
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        instance = objects.Instance(**self.test_instance)
        instance.config_drive = 'True'

        migrate_data = migrate_data_obj.LibvirtLiveMigrateData(
            is_shared_instance_path=False,
            is_shared_block_storage=False,
            block_migration=False,
            instance_relative_path='foo',
        )
        res_data = drvr.pre_live_migration(
            self.context, instance, vol, [], None, migrate_data)
        mock_block_device_info_get_mapping.assert_called_once_with(
            {'block_device_mapping': [
                {'connection_info': 'dummy', 'mount_device': '/dev/sda'},
                {'connection_info': 'dummy', 'mount_device': '/dev/sdb'}
            ]}
        )
        self.assertEqual({'block_migration': False,
                          'instance_relative_path': 'foo',
                          'is_shared_block_storage': False,
                          'is_shared_instance_path': False,
                          'serial_listen_ports': [],
                          'supported_perf_events': [],
                          'target_connect_addr': None,
                          'bdms': []},
                         res_data.obj_to_primitive()['nova_object.data'])

    def _test_pre_live_migration_volume_backed(self, encrypted_volumes=False):
        inst_ref = objects.Instance(root_device_name='/dev/vda',
                                    **self.test_instance)
        bdms = objects.BlockDeviceMappingList(objects=[
                fake_block_device.fake_bdm_object(self.context, {
                    'connection_info': jsonutils.dumps({
                        'serial': uuids.vol1,
                        'data': {
                            'device_path': '/dev/disk/path/lun-X'
                        }
                    }),
                    'device_name': '/dev/sda',
                    'volume_id': uuids.vol1,
                    'source_type': 'volume',
                    'destination_type': 'volume'
                }),
                fake_block_device.fake_bdm_object(self.context, {
                    'connection_info': jsonutils.dumps({
                        'serial': uuids.vol2,
                        'data': {
                            'device_path': '/dev/disk/path/lun-Z'
                        }
                    }),
                    'device_name': '/dev/sdb',
                    'volume_id': uuids.vol2,
                    'source_type': 'volume',
                    'destination_type': 'volume'
                })
            ])
        # We go through get_block_device_info to simulate what the
        # ComputeManager sends to the driver (make sure we're using the
        # correct type of BDM objects since there are many of them and
        # they are super confusing).
        block_device_info = driver.get_block_device_info(inst_ref, bdms)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        with test.nested(
            mock.patch.object(drvr._host, 'find_secret'),
            mock.patch.object(drvr, '_connect_volume'),
            mock.patch.object(drvr, 'plug_vifs'),
        ) as (mock_find_secret, mock_connect_volume,
              mock_plug_vifs):

            mock_find_secret.return_value = None
            if encrypted_volumes:
                secret_vol1 = mock.Mock()
                secret_vol1.UUIDString.return_value = uuids.secret_vol1
                secret_vol2 = mock.Mock()
                secret_vol2.UUIDString.return_value = uuids.secret_vol2
                mock_find_secret.side_effect = [secret_vol1, secret_vol2]

            migrate_data = migrate_data_obj.LibvirtLiveMigrateData(
                is_shared_instance_path=True,
                is_shared_block_storage=False,
                is_volume_backed=True,
                block_migration=False,
                instance_relative_path=inst_ref['name'],
                disk_over_commit=False,
                disk_available_mb=123,
                image_type='qcow2',
                filename='foo',
            )

            expected_migrate_data = migrate_data_obj.LibvirtLiveMigrateData(
                is_shared_instance_path=True,
                is_shared_block_storage=False,
                is_volume_backed=True,
                block_migration=False,
                instance_relative_path=inst_ref['name'],
                disk_over_commit=False,
                disk_available_mb=123,
                image_type='qcow2',
                filename='foo',
                serial_listen_ports=[],
                supported_perf_events=[],
                target_connect_addr=None,
            )

            bdmi_vol1 = migrate_data_obj.LibvirtLiveMigrateBDMInfo()
            bdmi_vol1.boot_index = None
            bdmi_vol1.format = None
            bdmi_vol1.serial = uuids.vol1
            bdmi_vol1.connection_info = {
                u'data': {'device_path': u'/dev/disk/path/lun-X'},
                u'serial': uuids.vol1}
            bdmi_vol1.bus = 'scsi'
            bdmi_vol1.dev = 'sda'
            bdmi_vol1.type = 'disk'

            bdmi_vol2 = migrate_data_obj.LibvirtLiveMigrateBDMInfo()
            bdmi_vol2.boot_index = None
            bdmi_vol2.format = None
            bdmi_vol2.serial = uuids.vol2
            bdmi_vol2.connection_info = {
                u'data': {'device_path': u'/dev/disk/path/lun-Z'},
                u'serial': uuids.vol2}
            bdmi_vol2.bus = 'scsi'
            bdmi_vol2.dev = 'sdb'
            bdmi_vol2.type = 'disk'

            if encrypted_volumes:
                bdmi_vol1.encryption_secret_uuid = uuids.secret_vol1
                bdmi_vol2.encryption_secret_uuid = uuids.secret_vol2

            expected_migrate_data.bdms = [bdmi_vol1, bdmi_vol2]

            returned_migrate_data = drvr.pre_live_migration(
                self.context, inst_ref, block_device_info, [], None,
                migrate_data)

            expected_connect_volume_calls = []
            for bdm in block_device_info['block_device_mapping']:
                expected_call = mock.call(self.context, bdm['connection_info'],
                                          inst_ref)
                expected_connect_volume_calls.append(expected_call)
            mock_connect_volume.assert_has_calls(expected_connect_volume_calls)

            if encrypted_volumes:
                mock_find_secret.assert_has_calls(
                    [mock.call('volume', uuids.vol1),
                     mock.call('volume', uuids.vol2)])

            # FIXME(lyarwood): This is taken from test_os_vif_util.py and as
            # noted there should be removed if the ComparableVersionedObject
            # mix-in is ever used for these objects.
            expected_migrate_data.obj_reset_changes(recursive=True)
            returned_migrate_data.obj_reset_changes(recursive=True)
            expected = expected_migrate_data.obj_to_primitive()
            returned = returned_migrate_data.obj_to_primitive()
            # We have to manually deserialize the connection_info_json so
            # that the equality comparison uses a dict rather than a string
            # with a random hashseed sort order on the keys.
            for migrate_data in (expected, returned):
                for bdm_data in migrate_data['nova_object.data']['bdms']:
                    bdm = bdm_data['nova_object.data']
                    bdm['connection_info_json'] = (
                        jsonutils.loads(bdm['connection_info_json']))
            self.assertEqual(expected, returned)

    def test_pre_live_migration_volume_backed(self):
        self._test_pre_live_migration_volume_backed()

    def test_pre_live_migration_volume_backed_encrypted(self):
        self._test_pre_live_migration_volume_backed(encrypted_volumes=True)

    @mock.patch.object(eventlet.greenthread, 'sleep',
                       side_effect=eventlet.sleep(0))
    @mock.patch.object(libvirt_driver.LibvirtDriver, 'plug_vifs',
                       side_effect=processutils.ProcessExecutionError)
    def test_pre_live_migration_plug_vifs_retry_fails(self, mock_plug,
                                                      mock_sleep):
        self.flags(live_migration_retry_count=3)
        instance = objects.Instance(**self.test_instance)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        disk_info_json = jsonutils.dumps({})
        migrate_data = migrate_data_obj.LibvirtLiveMigrateData(
            is_shared_block_storage=True,
            is_shared_instance_path=True,
            block_migration=False,
        )
        self.assertRaises(processutils.ProcessExecutionError,
                          drvr.pre_live_migration,
                          self.context, instance, block_device_info=None,
                          network_info=[], disk_info=disk_info_json,
                          migrate_data=migrate_data)
        # Called 3 times because of live_migration_retry_count is 3
        mock_plug.assert_has_calls([mock.call(instance, [])] * 3)
        self.assertEqual(3, mock_plug.call_count)
        # Called 'live_migration_retry_count - 1' times
        mock_sleep.assert_has_calls([mock.call(1)] * 2)
        self.assertEqual(2, mock_sleep.call_count)

    @mock.patch.object(eventlet.greenthread, 'sleep',
                       side_effect=eventlet.sleep(0))
    @mock.patch.object(libvirt_driver.LibvirtDriver, 'plug_vifs')
    def test_pre_live_migration_plug_vifs_retry_works(self, mock_plug,
                                                      mock_sleep):
        self.flags(live_migration_retry_count=3)
        instance = objects.Instance(**self.test_instance)

        mock_plug.side_effect = [processutils.ProcessExecutionError(),
                                 processutils.ProcessExecutionError(), None]

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        disk_info_json = jsonutils.dumps({})
        migrate_data = migrate_data_obj.LibvirtLiveMigrateData(
            is_shared_block_storage=True,
            is_shared_instance_path=True,
            block_migration=False,
        )
        drvr.pre_live_migration(self.context, instance, block_device_info=None,
                                network_info=[], disk_info=disk_info_json,
                                migrate_data=migrate_data)
        # Called 3 times
        mock_plug.assert_has_calls([mock.call(instance, [])] * 3)
        self.assertEqual(3, mock_plug.call_count)
        # Called 2 times because the third 'plug_vifs' call is successful.
        mock_sleep.assert_has_calls([mock.call(1)] * 2)
        self.assertEqual(2, mock_sleep.call_count)

    def test_pre_live_migration_plug_vifs_with_dest_port_bindings(self):
        """Tests that we use the LibvirtLiveMigrateData.vifs destination host
        port binding details when plugging VIFs during pre_live_migration.
        """
        source_vif = network_model.VIF(
            id=uuids.port_id, type=network_model.VIF_TYPE_OVS,
            vnic_type=network_model.VNIC_TYPE_NORMAL, details={'foo': 'bar'},
            profile={'binding:host_id': 'fake-source-host'})
        migrate_vifs = [objects.VIFMigrateData(
            port_id=uuids.port_id, vnic_type=network_model.VNIC_TYPE_NORMAL,
            vif_type=network_model.VIF_TYPE_OVS, vif_details={'bar': 'baz'},
            profile={'binding:host_id': 'fake-dest-host'},
            host='fake-dest-host', source_vif=source_vif)]
        migrate_data = migrate_data_obj.LibvirtLiveMigrateData(
            vifs=migrate_vifs)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance()
        network_info = mock.NonCallableMock()
        with mock.patch.object(drvr, 'plug_vifs') as plug_vifs:
            drvr._pre_live_migration_plug_vifs(
                instance, network_info, migrate_data)
        expected_network_info = network_model.NetworkInfo([
            migrate_vifs[0].get_dest_vif()])
        plug_vifs.assert_called_once_with(instance, expected_network_info)

    def test_pre_live_migration_image_not_created_with_shared_storage(self):
        migrate_data_set = [{'is_shared_block_storage': False,
                             'is_shared_instance_path': True,
                             'is_volume_backed': False,
                             'filename': 'foo',
                             'instance_relative_path': 'bar',
                             'disk_over_commit': False,
                             'disk_available_mb': 123,
                             'image_type': 'qcow2',
                             'block_migration': False},
                            {'is_shared_block_storage': True,
                             'is_shared_instance_path': True,
                             'is_volume_backed': False,
                             'filename': 'foo',
                             'instance_relative_path': 'bar',
                             'disk_over_commit': False,
                             'disk_available_mb': 123,
                             'image_type': 'qcow2',
                             'block_migration': False},
                            {'is_shared_block_storage': False,
                             'is_shared_instance_path': True,
                             'is_volume_backed': False,
                             'filename': 'foo',
                             'instance_relative_path': 'bar',
                             'disk_over_commit': False,
                             'disk_available_mb': 123,
                             'image_type': 'qcow2',
                             'block_migration': True}]

        def _to_obj(d):
            return migrate_data_obj.LibvirtLiveMigrateData(**d)
        migrate_data_set = map(_to_obj, migrate_data_set)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        # creating mocks
        with test.nested(
            mock.patch.object(drvr,
                              '_create_images_and_backing'),
            mock.patch.object(drvr, 'plug_vifs'),
        ) as (
            create_image_mock,
            plug_mock,
        ):
            disk_info_json = jsonutils.dumps({})
            for migrate_data in migrate_data_set:
                res = drvr.pre_live_migration(self.context, instance,
                                              block_device_info=None,
                                              network_info=[],
                                              disk_info=disk_info_json,
                                              migrate_data=migrate_data)
                self.assertFalse(create_image_mock.called)
                self.assertIsInstance(res,
                                      objects.LibvirtLiveMigrateData)

    @mock.patch('nova.virt.libvirt.utils.file_open',
                side_effect=[io.BytesIO(b''), io.BytesIO(b'')])
    def test_pre_live_migration_with_not_shared_instance_path(
            self, mock_file_open):
        migrate_data = migrate_data_obj.LibvirtLiveMigrateData(
            is_shared_block_storage=False,
            is_shared_instance_path=False,
            block_migration=False,
            instance_relative_path='foo',
        )

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)

        def check_instance_dir(context, instance,
                               instance_dir, disk_info,
                               fallback_from_host=False):
            self.assertTrue(instance_dir)
        # creating mocks
        with test.nested(
            mock.patch.object(drvr,
                              '_create_images_and_backing',
                              side_effect=check_instance_dir),
            mock.patch.object(drvr, 'plug_vifs'),
        ) as (
            create_image_mock,
            plug_mock,
        ):
            disk_info_json = jsonutils.dumps({})
            res = drvr.pre_live_migration(self.context, instance,
                                          block_device_info=None,
                                          network_info=[],
                                          disk_info=disk_info_json,
                                          migrate_data=migrate_data)
            create_image_mock.assert_has_calls(
                [mock.call(self.context, instance, mock.ANY, {},
                           fallback_from_host=instance.host)])
            self.assertIsInstance(res, objects.LibvirtLiveMigrateData)

    def test_pre_live_migration_recreate_disk_info(self):

        migrate_data = migrate_data_obj.LibvirtLiveMigrateData(
            is_shared_block_storage=False,
            is_shared_instance_path=False,
            block_migration=True,
            instance_relative_path='/some/path/',
        )
        disk_info = [{'disk_size': 5368709120, 'type': 'raw',
                      'virt_disk_size': 5368709120,
                      'path': '/some/path/disk',
                      'backing_file': '', 'over_committed_disk_size': 0},
                     {'disk_size': 1073741824, 'type': 'raw',
                      'virt_disk_size': 1073741824,
                      'path': '/some/path/disk.eph0',
                      'backing_file': '', 'over_committed_disk_size': 0}]
        image_disk_info = {'/some/path/disk': 'raw',
                           '/some/path/disk.eph0': 'raw'}

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        instance_path = os.path.dirname(disk_info[0]['path'])
        disk_info_path = os.path.join(instance_path, 'disk.info')

        with test.nested(
            mock.patch('builtins.open', new_callable=mock.mock_open),
            mock.patch.object(os, 'mkdir'),
            mock.patch.object(drvr, '_create_images_and_backing')
        ) as (
            mock_open, mkdir, create_images_and_backing
        ):
            drvr.pre_live_migration(self.context, instance,
                                    block_device_info=None,
                                    network_info=[],
                                    disk_info=jsonutils.dumps(disk_info),
                                    migrate_data=migrate_data)

            self.assertIn(mock.call(disk_info_path, 'w'), mock_open.mock_calls)
            mock_open.return_value.write.assert_called_with(
                jsonutils.dumps(image_disk_info))

    @mock.patch('nova.virt.libvirt.utils.file_open',
                side_effect=[io.BytesIO(b''), io.BytesIO(b'')])
    def test_pre_live_migration_with_perf_events(self, mock_file_open):

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._supported_perf_events = ['cmt']

        migrate_data = migrate_data_obj.LibvirtLiveMigrateData(
            is_shared_block_storage=False,
            is_shared_instance_path=False,
            block_migration=False,
            instance_relative_path='foo',
        )

        instance = objects.Instance(**self.test_instance)

        res = drvr.pre_live_migration(self.context, instance,
                                      block_device_info=None,
                                      network_info=[],
                                      disk_info=None,
                                      migrate_data=migrate_data)
        self.assertEqual(['cmt'], res.supported_perf_events)

    @mock.patch('os.stat')
    @mock.patch('os.path.getsize')
    @mock.patch('nova.virt.disk.api.get_disk_info')
    @mock.patch('nova.virt.libvirt.utils.get_disk_backing_file',
                return_value='file')
    def test_get_instance_disk_info_works_correctly(
            self, mock_get_disk_backing_file, mock_qemu_img_info,
            mock_get_size, mock_stat):
        # Test data
        instance = objects.Instance(**self.test_instance)
        dummyxml = ("<domain type='kvm'><name>instance-0000000a</name>"
                    "<devices>"
                    "<disk type='file'><driver name='qemu' type='raw'/>"
                    "<source file='/test/disk'/>"
                    "<target dev='vda' bus='virtio'/></disk>"
                    "<disk type='file'><driver name='qemu' type='qcow2'/>"
                    "<source file='/test/disk.local'/>"
                    "<target dev='vdb' bus='virtio'/></disk>"
                    "</devices></domain>")

        # Preparing mocks
        vdmock = mock.Mock(autospec=fakelibvirt.virDomain)
        vdmock.XMLDesc.return_value = dummyxml

        mock_qemu_img_info.return_value = mock.Mock(disk_size=3328599655,
                                                    virtual_size=21474836480)
        mock_stat.return_value = mock.Mock(st_blocks=20971520)
        mock_get_size.return_value = 10737418240

        def fake_lookup(_uuid):
            if _uuid == instance.uuid:
                return vdmock
        self.create_fake_libvirt_mock(lookupByUUIDString=fake_lookup)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        info = drvr.get_instance_disk_info(instance)
        info = jsonutils.loads(info)
        self.assertEqual(info[0]['type'], 'raw')
        self.assertEqual(info[0]['path'], '/test/disk')
        self.assertEqual(info[0]['disk_size'], 10737418240)
        self.assertEqual(info[0]['virt_disk_size'], 10737418240)
        self.assertEqual(info[0]['backing_file'], "")
        self.assertEqual(info[0]['over_committed_disk_size'], 0)
        self.assertEqual(info[1]['type'], 'qcow2')
        self.assertEqual(info[1]['path'], '/test/disk.local')
        self.assertEqual(info[1]['disk_size'], 3328599655)
        self.assertEqual(info[1]['virt_disk_size'], 21474836480)
        self.assertEqual(info[1]['backing_file'], "file")
        self.assertEqual(info[1]['over_committed_disk_size'], 18146236825)

        vdmock.XMLDesc.assert_called_once_with(0)
        mock_qemu_img_info.assert_called_once_with('/test/disk.local')
        mock_stat.assert_called_once_with('/test/disk')
        mock_get_size.assert_called_once_with('/test/disk')
        mock_get_disk_backing_file.assert_called()

    def test_post_live_migration(self):
        vol1_conn_info = {'data': {'test_data': mock.sentinel.vol1},
                          'serial': 'fake_serial1'}
        vol2_conn_info = {'data': {'test_data': mock.sentinel.vol2},
                          'serial': 'fake_serial2'}

        bdi = {'block_device_mapping': [
                  {'attachment_id': None,
                   'connection_info': vol1_conn_info,
                    'mount_device': '/dev/sda',
                   },
                   {'attachment_id': None,
                    'connection_info': vol2_conn_info,
                    'mount_device': '/dev/sdb', }]}

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        inst_ref = {'id': 'foo'}
        cntx = context.get_admin_context()

        # Set up the mock expectations
        @mock.patch.object(driver, 'block_device_info_get_mapping',
                           return_value=bdi['block_device_mapping'])
        @mock.patch.object(drvr, '_disconnect_volume')
        def _test(_disconnect_volume, block_device_info_get_mapping):
            drvr.post_live_migration(cntx, inst_ref, bdi)

            block_device_info_get_mapping.assert_called_once_with(bdi)
            _disconnect_volume.assert_has_calls([
                mock.call(cntx, vol1_conn_info, inst_ref),
                mock.call(cntx, vol2_conn_info, inst_ref)])

        _test()

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_disconnect_volume')
    @mock.patch.object(driver, 'block_device_info_get_mapping')
    def test_post_live_migration_exception_swallowed(self, mock_get_bdm,
                                                     mock_disconnect_volume):
        vol_1_conn_info = {'data': {'volume_id': uuids.vol_1_id}}
        vol_2_conn_info = {'data': {'volume_id': uuids.vol_2_id}}
        mock_get_bdm.return_value = [{'connection_info': vol_1_conn_info},
                                     {'connection_info': vol_2_conn_info}]

        # Raise an exception with the first call to disconnect_volume
        mock_disconnect_volume.side_effect = [test.TestingException, None]

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr.post_live_migration(mock.sentinel.ctxt, mock.sentinel.instance,
                                 mock.sentinel.bdi)

        # Assert disconnect_volume is called twice despite the exception
        mock_disconnect_volume.assert_has_calls([
            mock.call(mock.sentinel.ctxt, vol_1_conn_info,
                      mock.sentinel.instance),
            mock.call(mock.sentinel.ctxt, vol_2_conn_info,
                      mock.sentinel.instance)])

        # Assert that we log the failure to disconnect the first volume
        self.assertIn("Ignoring exception while attempting to disconnect "
                      "volume %s" % uuids.vol_1_id, self.stdlog.logger.output)

    @mock.patch('os.stat')
    @mock.patch('os.path.getsize')
    @mock.patch('nova.virt.disk.api.get_disk_info')
    @mock.patch('nova.virt.libvirt.utils.get_disk_backing_file',
                return_value='file')
    def test_get_instance_disk_info_excludes_volumes(
            self, mock_get_disk_backing_file, mock_qemu_img_info,
            mock_get_size, mock_stat):
        # Test data
        instance = objects.Instance(**self.test_instance)
        dummyxml = ("<domain type='kvm'><name>instance-0000000a</name>"
                    "<devices>"
                    "<disk type='file'><driver name='qemu' type='raw'/>"
                    "<source file='/test/disk'/>"
                    "<target dev='vda' bus='virtio'/></disk>"
                    "<disk type='file'><driver name='qemu' type='qcow2'/>"
                    "<source file='/test/disk.local'/>"
                    "<target dev='vdb' bus='virtio'/></disk>"
                    "<disk type='file'><driver name='qemu' type='qcow2'/>"
                    "<source file='/fake/path/to/volume1'/>"
                    "<target dev='vdc' bus='virtio'/></disk>"
                    "<disk type='file'><driver name='qemu' type='qcow2'/>"
                    "<source file='/fake/path/to/volume2'/>"
                    "<target dev='vdd' bus='virtio'/></disk>"
                    "</devices></domain>")

        # Preparing mocks
        vdmock = mock.Mock(autospec=fakelibvirt.virDomain)
        vdmock.XMLDesc.return_value = dummyxml

        mock_qemu_img_info.return_value = mock.Mock(disk_size=3328599655,
                                                    virtual_size=21474836480)
        mock_stat.return_value = mock.Mock(st_blocks=20971520)
        mock_get_size.return_value = 10737418240

        def fake_lookup(_uuid):
            if _uuid == instance.uuid:
                return vdmock
        self.create_fake_libvirt_mock(lookupByUUIDString=fake_lookup)

        conn_info = {'driver_volume_type': 'fake'}
        info = {'block_device_mapping': [
                  {'connection_info': conn_info, 'mount_device': '/dev/vdc'},
                  {'connection_info': conn_info, 'mount_device': '/dev/vdd'}]}
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        info = drvr.get_instance_disk_info(instance,
                                           block_device_info=info)
        info = jsonutils.loads(info)
        self.assertEqual(info[0]['type'], 'raw')
        self.assertEqual(info[0]['path'], '/test/disk')
        self.assertEqual(info[0]['disk_size'], 10737418240)
        self.assertEqual(info[0]['backing_file'], "")
        self.assertEqual(info[0]['over_committed_disk_size'], 0)
        self.assertEqual(info[1]['type'], 'qcow2')
        self.assertEqual(info[1]['path'], '/test/disk.local')
        self.assertEqual(info[1]['virt_disk_size'], 21474836480)
        self.assertEqual(info[1]['backing_file'], "file")
        self.assertEqual(info[1]['over_committed_disk_size'], 18146236825)

        vdmock.XMLDesc.assert_called_once_with(0)
        mock_qemu_img_info.assert_called_once_with('/test/disk.local')
        mock_stat.assert_called_once_with('/test/disk')
        mock_get_size.assert_called_once_with('/test/disk')
        mock_get_disk_backing_file.assert_called()

    @mock.patch('os.stat')
    @mock.patch('os.path.getsize')
    def test_get_instance_disk_info_no_bdinfo_passed(self, mock_get_size,
                                                     mock_stat):
        # NOTE(ndipanov): _get_disk_overcommitted_size_total calls this method
        # without access to Nova's block device information. We want to make
        # sure that we guess volumes mostly correctly in that case as well
        instance = objects.Instance(**self.test_instance)
        dummyxml = ("<domain type='kvm'><name>instance-0000000a</name>"
                    "<devices>"
                    "<disk type='file'><driver name='qemu' type='raw'/>"
                    "<source file='/test/disk'/>"
                    "<target dev='vda' bus='virtio'/></disk>"
                    "<disk type='block'><driver name='qemu' type='raw'/>"
                    "<source file='/fake/path/to/volume1'/>"
                    "<target dev='vdb' bus='virtio'/></disk>"
                    "</devices></domain>")
        path = '/test/disk'
        size = 10737418240

        # Preparing mocks
        vdmock = mock.Mock(autospec=fakelibvirt.virDomain)
        vdmock.XMLDesc.return_value = dummyxml

        mock_stat.return_value = mock.Mock(st_blocks=20971520)
        mock_get_size.return_value = 10737418240

        def fake_lookup(_uuid):
            if _uuid == instance.uuid:
                return vdmock
        self.create_fake_libvirt_mock(lookupByUUIDString=fake_lookup)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        info = drvr.get_instance_disk_info(instance)

        info = jsonutils.loads(info)
        self.assertEqual(1, len(info))
        self.assertEqual(info[0]['type'], 'raw')
        self.assertEqual(info[0]['path'], path)
        self.assertEqual(info[0]['disk_size'], size)
        self.assertEqual(info[0]['backing_file'], "")
        self.assertEqual(info[0]['over_committed_disk_size'], 0)

        vdmock.XMLDesc.assert_called_once_with(0)
        mock_stat.assert_called_once_with(path)
        mock_get_size.assert_called_once_with(path)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_undefined_instance_details',
                       new=mock.Mock())
    def test_spawn_with_network_info(self, power_on=True):
        def fake_getLibVersion():
            return fakelibvirt.FAKE_LIBVIRT_VERSION

        def fake_getCapabilities():
            return """
            <capabilities>
                <host>
                    <uuid>cef19ce0-0ca2-11df-855d-b19fbce37686</uuid>
                    <cpu>
                      <arch>x86_64</arch>
                      <model>Penryn</model>
                      <vendor>Intel</vendor>
                      <topology sockets='1' cores='2' threads='1'/>
                      <feature name='xtpr'/>
                    </cpu>
                </host>
            </capabilities>
            """

        def fake_baselineCPU(cpu, flag):
            return """<cpu mode='custom' match='exact'>
                        <model fallback='allow'>Penryn</model>
                        <vendor>Intel</vendor>
                        <feature policy='require' name='xtpr'/>
                      </cpu>
                   """

        def fake_getCPUModelNames(arch):
            return [
                '486',
                'pentium',
                'pentium2',
                'pentium3',
                'pentiumpro',
                'coreduo',
                'n270',
                'core2duo',
                'qemu32',
                'kvm32',
                'cpu64-rhel5',
                'cpu64-rhel6',
                'qemu64',
                'kvm64',
                'Conroe',
                'Penryn',
                'Nehalem',
                'Nehalem-IBRS',
                'Westmere',
                'Westmere-IBRS',
                'SandyBridge',
                'SandyBridge-IBRS',
                'IvyBridge',
                'IvyBridge-IBRS',
                'Haswell-noTSX',
                'Haswell-noTSX-IBRS',
                'Haswell',
                'Haswell-IBRS',
                'Broadwell-noTSX',
                'Broadwell-noTSX-IBRS',
                'Broadwell',
                'Broadwell-IBRS',
                'Skylake-Client',
                'Skylake-Client-IBRS',
                'Skylake-Server',
                'Skylake-Server-IBRS',
                'Cascadelake-Server',
                'Icelake-Client',
                'Icelake-Server',
                'athlon',
                'phenom',
                'Opteron_G1',
                'Opteron_G2',
                'Opteron_G3',
                'Opteron_G4',
                'Opteron_G5',
                'EPYC',
                'EPYC-IBPB']

        def fake_getCPUMap():
            return (2, [True, True], 2)

        # _fake_network_info must be called before create_fake_libvirt_mock(),
        # as _fake_network_info calls importutils.import_class() and
        # create_fake_libvirt_mock() mocks importutils.import_class().
        network_info = _fake_network_info(self)
        self.create_fake_libvirt_mock(getLibVersion=fake_getLibVersion,
                                      getCapabilities=fake_getCapabilities,
                                      getCPUModelNames=fake_getCPUModelNames,
                                      getVersion=lambda: 1005001,
                                      baselineCPU=fake_baselineCPU,
                                      getCPUMap=fake_getCPUMap)

        instance = objects.Instance(**self.test_instance)
        instance.image_ref = uuids.image_ref
        instance.config_drive = ''
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        self.useFixture(nova_fixtures.LibvirtImageBackendFixture())
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        with test.nested(
            utils.tempdir(),
            mock.patch('nova.virt.libvirt.driver.libvirt'),
            mock.patch.object(drvr, '_build_device_metadata'),
            mock.patch.object(drvr, 'get_info'),
        ) as (
            tmpdir,
            mock_orig_libvirt,
            mock_build_device_metadata,
            mock_get_info,
        ):
            self.flags(instances_path=tmpdir)

            if power_on:
                hw_running = hardware.InstanceInfo(state=power_state.RUNNING)
                mock_get_info.return_value = hw_running
            else:
                # Avoid a test timeout since _wait_for_boot is threaded.
                mock_get_info.side_effect = Exception('do not call get_info')
            mock_build_device_metadata.return_value = None

            del mock_orig_libvirt.VIR_CONNECT_BASELINE_CPU_EXPAND_FEATURES

            drvr.spawn(self.context, instance, image_meta, [], 'herp', {},
                       network_info=network_info, power_on=power_on)

            if power_on:
                mock_get_info.assert_called_once_with(instance)
            else:
                mock_get_info.assert_not_called()
            mock_build_device_metadata.assert_called_once_with(self.context,
                                                               instance)

    @mock.patch.object(host.Host, "_check_machine_type", new=mock.Mock())
    def test_spawn_power_on_false(self):
        self.test_spawn_with_network_info(power_on=False)

    @mock.patch('nova.virt.libvirt.blockinfo.get_disk_info')
    def _test_spawn_accels(self, accel_info, mock_get_disk_info):
        mock_get_disk_info.return_value = {'mapping': None}
        self.stub_out('nova.virt.libvirt.driver.LibvirtDriver.'
                      '_create_image', lambda *a, **kw: (False, False))
        self.stub_out('nova.virt.libvirt.driver.LibvirtDriver.'
                      '_ensure_console_log_for_instance',
                      lambda *a, **kw: None)
        self.stub_out('nova.virt.libvirt.driver.LibvirtDriver.'
                      '_allocate_mdevs', lambda *a, **kw: None)
        self.stub_out('nova.virt.libvirt.driver.LibvirtDriver.'
                      '_create_guest_with_network', lambda *a, **kw: None)

        instance = objects.Instance(**self.test_instance)
        instance.image_ref = uuids.image_ref
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        drvr.spawn(self.context, instance, image_meta, [], None, {},
                   accel_info=accel_info, power_on=False)
        return instance

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_undefined_instance_details',
                       new=mock.Mock())
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_guest_xml')
    def test_spawn_accels_no_accel_info(self, mock_get_guest_xml):
        # accel_info should be passed to get_guest_xml even if it is []
        accel_info = []
        instance = self._test_spawn_accels(accel_info)
        mock_get_guest_xml.assert_called_once_with(
            self.context, instance, mock.ANY, mock.ANY, mock.ANY,
            block_device_info=None, mdevs=mock.ANY,
            accel_info=[])

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_undefined_instance_details',
                       new=mock.Mock())
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_guest_xml')
    def test_spawn_accels_with_accel_info(self, mock_get_guest_xml):
        # accel_info should be passed to get_guest_xml if it is not []
        accel_info = nova_fixtures.CyborgFixture.bound_arq_list
        instance = self._test_spawn_accels(accel_info)
        mock_get_guest_xml.assert_called_once_with(
            self.context, instance, mock.ANY, mock.ANY, mock.ANY,
            block_device_info=None, mdevs=mock.ANY,
            accel_info=accel_info)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_undefined_instance_details',
                       new=mock.Mock())
    # Methods called directly by spawn()
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_guest_xml')
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_create_guest_with_network')
    @mock.patch.object(libvirt_driver.LibvirtDriver, 'get_info')
    # Methods called by _create_configdrive via post_xml_callback
    @mock.patch('nova.virt.configdrive.ConfigDriveBuilder._make_iso9660')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_build_device_metadata')
    @mock.patch.object(instance_metadata, 'InstanceMetadata')
    def test_spawn_with_config_drive(self, mock_instance_metadata,
                                     mock_build_device_metadata,
                                     mock_mkisofs, mock_get_info,
                                     mock_create_guest_with_network,
                                     mock_get_guest_xml):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        instance.config_drive = 'True'
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        instance_info = hardware.InstanceInfo(state=power_state.RUNNING)

        mock_build_device_metadata.return_value = None

        def fake_create_guest_with_network(
            context, xml, instance, network_info, block_device_info,
            power_on=True, vifs_already_plugged=False, post_xml_callback=None,
            external_events=None, cleanup_instance_dir=False,
            cleanup_instance_disks=False,
        ):
            # The config disk should be created by this callback, so we need
            # to execute it.
            post_xml_callback()

        backend = self.useFixture(
            nova_fixtures.LibvirtImageBackendFixture(exists=lambda _: False))

        mock_get_info.return_value = instance_info
        mock_create_guest_with_network.side_effect = \
            fake_create_guest_with_network

        drvr.spawn(self.context, instance, image_meta, [], None, {})

        # We should have imported 'disk.config'
        config_disk = backend.disks['disk.config']
        config_disk.import_file.assert_called_once_with(instance, mock.ANY,
                                                        'disk.config')

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_undefined_instance_details',
                       new=mock.Mock())
    def test_spawn_without_image_meta(self):
        instance_ref = self.test_instance
        instance_ref['image_ref'] = uuids.image_ref
        instance = objects.Instance(**instance_ref)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        backend = self.useFixture(nova_fixtures.LibvirtImageBackendFixture())

        self.stub_out('nova.virt.libvirt.driver.LibvirtDriver._get_guest_xml',
                      lambda *a, **kw: None)
        self.stub_out('nova.virt.libvirt.driver.LibvirtDriver.'
                      '_create_guest_with_network',
                      lambda *a, **kw: None)
        self.stub_out('nova.virt.libvirt.driver.LibvirtDriver.get_info',
                      lambda self, instance: hardware.InstanceInfo(
                          state=power_state.RUNNING))

        drvr.spawn(self.context, instance, image_meta, [], None, {})

        # We should have created a root disk and an ephemeral disk
        self.assertEqual(['disk', 'disk.local'],
                         sorted(backend.created_disks.keys()))

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_undefined_instance_details',
                       new=mock.Mock())
    def _test_spawn_disks(self, image_ref, block_device_info):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        # Volume-backed instance created without image
        instance = objects.Instance(**self.test_instance)
        instance.image_ref = image_ref
        instance.root_device_name = '/dev/vda'
        instance.uuid = uuids.instance_uuid

        backend = self.useFixture(nova_fixtures.LibvirtImageBackendFixture())

        with test.nested(
                mock.patch.object(drvr, '_get_guest_xml'),
                mock.patch.object(drvr, '_create_guest_with_network'),
                mock.patch.object(drvr, 'get_info')
        ) as (
                mock_get_guest_xml, mock_create_guest_with_network,
                mock_get_info
        ):
            hw_running = hardware.InstanceInfo(state=power_state.RUNNING)
            mock_get_info.return_value = hw_running

            drvr.spawn(self.context, instance,
                       image_meta, [], None, {},
                       block_device_info=block_device_info)

        # Return a sorted list of created disks
        return sorted(backend.created_disks.keys())

    def test_spawn_from_volume_no_image_ref(self):
        block_device_info = {'root_device_name': '/dev/vda',
                             'block_device_mapping': [
                                {'mount_device': 'vda',
                                 'boot_index': 0}]}

        disks_created = self._test_spawn_disks(None, block_device_info)

        # We should have created the ephemeral disk, and nothing else
        self.assertEqual(['disk.local'], disks_created)

    def test_spawn_from_volume_with_image_ref(self):
        block_device_info = {'root_device_name': '/dev/vda',
                             'block_device_mapping': [
                                 {'mount_device': 'vda',
                                  'boot_index': 0}]}

        disks_created = self._test_spawn_disks(uuids.image_ref,
                                               block_device_info)

        # We should have created the ephemeral disk, and nothing else
        self.assertEqual(['disk.local'], disks_created)

    def test_spawn_from_image(self):
        disks_created = self._test_spawn_disks(uuids.image_ref, None)

        # We should have created the root and ephemeral disks
        self.assertEqual(['disk', 'disk.local'], disks_created)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_undefined_instance_details',
                       new=mock.Mock())
    def test_spawn_lxc_from_volume(self):
        self.flags(virt_type="lxc",
                   group='libvirt')

        def check_setup_container(image, container_dir=None):
            self.assertIsInstance(image, imgmodel.LocalBlockImage)
            self.assertEqual(image.path, '/dev/path/to/dev')
            return '/dev/nbd1'

        bdm = {
                  'guest_format': None,
                  'boot_index': 0,
                  'mount_device': '/dev/sda',
                  'connection_info': {
                      'driver_volume_type': 'iscsi',
                      'serial': 'afc1',
                      'data': {
                          'access_mode': 'rw',
                          'target_discovered': False,
                          'encrypted': False,
                          'qos_specs': None,
                          'target_iqn': 'iqn: volume-afc1',
                          'target_portal': 'ip: 3260',
                          'volume_id': 'afc1',
                          'target_lun': 1,
                          'auth_password': 'uj',
                          'auth_username': '47',
                          'auth_method': 'CHAP'
                      }
                  },
                  'disk_bus': 'scsi',
                  'device_type': 'disk',
                  'delete_on_termination': False
              }

        def _connect_volume_side_effect(ctxt, connection_info, instance):
            bdm['connection_info']['data']['device_path'] = '/dev/path/to/dev'

        def _get(key, opt=None):
            return bdm.get(key, opt)

        def getitem(key):
            return bdm[key]

        def setitem(key, val):
            bdm[key] = val

        bdm_mock = mock.MagicMock()
        bdm_mock.__getitem__.side_effect = getitem
        bdm_mock.__setitem__.side_effect = setitem
        bdm_mock.get = _get

        disk_mock = mock.MagicMock()
        disk_mock.source_path = '/dev/path/to/dev'

        block_device_info = {'block_device_mapping': [bdm_mock],
                             'root_device_name': '/dev/sda'}

        # Volume-backed instance created without image
        instance_ref = self.test_instance
        instance_ref['image_ref'] = ''
        instance_ref['root_device_name'] = '/dev/sda'
        instance_ref['ephemeral_gb'] = 0
        instance_ref['uuid'] = uuids.fake
        inst_obj = objects.Instance(**instance_ref)
        image_meta = objects.ImageMeta.from_dict({})

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        with test.nested(
            mock.patch.object(drvr, 'plug_vifs'),
            mock.patch.object(drvr, '_create_guest'),
            mock.patch.object(drvr, '_connect_volume',
                              side_effect=_connect_volume_side_effect),
            mock.patch.object(drvr, '_get_volume_config',
                                     return_value=disk_mock),
            mock.patch.object(drvr, 'get_info',
                              return_value=hardware.InstanceInfo(
                              state=power_state.RUNNING)),
            mock.patch('nova.virt.disk.api.setup_container',
                       side_effect=check_setup_container),
            mock.patch('nova.virt.disk.api.teardown_container'),
            mock.patch.object(objects.Instance, 'save')):

            drvr.spawn(self.context, inst_obj, image_meta, [], None, {},
                       network_info=[],
                       block_device_info=block_device_info)
            self.assertEqual('/dev/nbd1',
                             inst_obj.system_metadata.get(
                             'rootfs_device_name'))

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_undefined_instance_details',
                       new=mock.Mock())
    def test_spawn_with_pci_devices(self):
        class FakeLibvirtPciDevice(object):
            def dettach(self):
                return None

            def reset(self):
                return None

        def fake_node_device_lookup_by_name(address):
            pattern = "pci_%(hex)s{4}_%(hex)s{2}_%(hex)s{2}_%(oct)s{1}" % {
                'hex': r'[\da-f]', 'oct': '[0-8]'}
            pattern = re.compile(pattern)
            if pattern.match(address) is None:
                raise fakelibvirt.libvirtError()
            return FakeLibvirtPciDevice()

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        self.stub_out('nova.virt.libvirt.driver.LibvirtDriver._get_guest_xml',
                      lambda *args, **kwargs: None)
        self.stub_out('nova.virt.libvirt.driver.LibvirtDriver.'
                      '_create_guest_with_network',
                      lambda *args, **kwargs: None)
        self.stub_out('nova.virt.libvirt.driver.LibvirtDriver.get_info',
                      lambda self, instance: hardware.InstanceInfo(
                          state=power_state.RUNNING))

        mock_connection = mock.MagicMock(
            nodeDeviceLookupByName=fake_node_device_lookup_by_name)

        instance_ref = self.test_instance
        instance_ref['image_ref'] = uuids.image_ref
        instance = objects.Instance(**instance_ref)
        instance['pci_devices'] = objects.PciDeviceList(
            objects=[objects.PciDevice(address='0000:00:00.0')])

        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        self.useFixture(nova_fixtures.LibvirtImageBackendFixture())

        with mock.patch.object(drvr, '_get_connection',
                               return_value=mock_connection):
            drvr.spawn(self.context, instance, image_meta, [], None, {})

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_undefined_instance_details',
                       new=mock.Mock())
    @mock.patch('nova.crypto.ensure_vtpm_secret')
    @mock.patch.object(hardware, 'get_vtpm_constraint')
    @mock.patch(
        'nova.virt.libvirt.driver.LibvirtDriver._create_guest_with_network')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.get_info')
    @mock.patch(
        'nova.virt.libvirt.driver.LibvirtDriver._get_guest_xml',
        new=mock.Mock())
    def test_spawn_with_vtpm(
        self, mock_get_info, mock_create_guest, mock_get_vtpm,
        mock_ensure_vtpm,
    ):
        """Ensure spawning with vTPM requested results in pre-config of
        instance.
        """
        self.flags(swtpm_enabled=True, group='libvirt')
        self.useFixture(nova_fixtures.LibvirtImageBackendFixture())

        mock_get_info.return_value = hardware.InstanceInfo(
            state=power_state.RUNNING)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance_ref = self.test_instance
        instance_ref['image_ref'] = uuids.image_ref
        instance = objects.Instance(**instance_ref)
        instance.system_metadata = {}
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        drvr.spawn(self.context, instance, image_meta, [], None, {})

        mock_get_vtpm.assert_called_once_with(instance.flavor, image_meta)
        mock_ensure_vtpm.assert_called_once_with(self.context, instance)

    def _test_create_image_plain(self, os_type='', filename='', mkfs=False):
        gotFiles = []

        instance_ref = self.test_instance
        instance_ref['image_ref'] = uuids.image_ref
        instance = objects.Instance(**instance_ref)
        instance['os_type'] = os_type

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.stub_out('nova.virt.libvirt.driver.LibvirtDriver._get_guest_xml',
                      lambda *args, **kwargs: None)
        self.stub_out('nova.virt.libvirt.driver.LibvirtDriver.'
                      '_create_guest_with_network',
                      lambda *args, **kwargs: None)
        self.stub_out('nova.virt.libvirt.driver.LibvirtDriver.get_info',
                      lambda self, instance: hardware.InstanceInfo(
                          state=power_state.RUNNING))
        if mkfs:
            self.stub_out(
                'nova.privsep.fs._MKFS_COMMAND',
                {os_type: 'mkfs.ext4 --label %(fs_label)s %(target)s'})

        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance,
                                            image_meta)

        self.useFixture(
            nova_fixtures.LibvirtImageBackendFixture(got_files=gotFiles))

        drvr._create_image(self.context, instance, disk_info['mapping'])
        drvr._get_guest_xml(self.context, instance, None,
                            disk_info, image_meta)

        wantFiles = [
            {'filename': imagecache.get_cache_fname(uuids.image_ref),
             'size': 10 * units.Gi},
            {'filename': filename,
             'size': 20 * units.Gi},
            ]
        self.assertEqual(wantFiles, gotFiles)

    def test_create_image_plain_os_type_blank(self):
        self._test_create_image_plain(os_type='',
                                      filename=self._EPHEMERAL_20_DEFAULT,
                                      mkfs=False)

    def test_create_image_plain_os_type_none(self):
        self._test_create_image_plain(os_type=None,
                                      filename=self._EPHEMERAL_20_DEFAULT,
                                      mkfs=False)

    def test_create_image_plain_os_type_set_no_fs(self):
        self._test_create_image_plain(os_type='test',
                                      filename=self._EPHEMERAL_20_DEFAULT,
                                      mkfs=False)

    def test_create_image_plain_os_type_set_with_fs(self):
        ephemeral_file_name = ('ephemeral_20_%s' % utils.get_hash_str(
            'mkfs.ext4 --label %(fs_label)s %(target)s')[:7])

        self._test_create_image_plain(os_type='test',
                                      filename=ephemeral_file_name,
                                      mkfs=True)

    def test_create_image_initrd(self):
        kernel_id = uuids.kernel_id
        ramdisk_id = uuids.ramdisk_id

        kernel_fname = imagecache.get_cache_fname(kernel_id)
        ramdisk_fname = imagecache.get_cache_fname(ramdisk_id)

        filename = self._EPHEMERAL_20_DEFAULT

        gotFiles = []

        instance_ref = self.test_instance
        instance_ref['image_ref'] = uuids.instance_id
        instance_ref['kernel_id'] = uuids.kernel_id
        instance_ref['ramdisk_id'] = uuids.ramdisk_id
        instance_ref['os_type'] = 'test'
        instance = objects.Instance(**instance_ref)

        driver = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        backend = self.useFixture(
            nova_fixtures.LibvirtImageBackendFixture(got_files=gotFiles))

        with test.nested(
            mock.patch.object(driver, '_get_guest_xml'),
            mock.patch.object(driver, '_create_guest_with_network'),
            mock.patch.object(driver, 'get_info',
              return_value=[hardware.InstanceInfo(state=power_state.RUNNING)])
        ):
            image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
            disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                                instance,
                                                image_meta)
            driver._create_image(self.context, instance, disk_info['mapping'])

        # Assert that kernel and ramdisk were fetched with fetch_raw_image
        # and no size
        for name, disk in backend.disks.items():
            cache = disk.cache
            if name in ('kernel', 'ramdisk'):
                cache.assert_called_once_with(
                    context=self.context, filename=mock.ANY, image_id=mock.ANY,
                    fetch_func=backend.mock_fetch_raw_image)

        wantFiles = [
            {'filename': kernel_fname,
             'size': None},
            {'filename': ramdisk_fname,
             'size': None},
            {'filename': imagecache.get_cache_fname(uuids.instance_id),
             'size': 10 * units.Gi},
            {'filename': filename,
             'size': 20 * units.Gi},
            ]
        self.assertEqual(wantFiles, gotFiles)

    def test_injection_info_is_sanitized(self):
        info = get_injection_info(
            network_info=mock.sentinel.network_info,
            files=mock.sentinel.files,
            admin_pass='verybadpass')
        self.assertNotIn('verybadpass', str(info))
        self.assertNotIn('verybadpass', repr(info))

    @mock.patch(
        'nova.virt.libvirt.driver.LibvirtDriver._build_device_metadata')
    @mock.patch('nova.api.metadata.base.InstanceMetadata')
    @mock.patch('nova.virt.configdrive.ConfigDriveBuilder.make_drive')
    def test_create_configdrive(self, mock_make_drive,
                                mock_instance_metadata,
                                mock_build_device_metadata):
        instance = objects.Instance(**self.test_instance)
        instance.config_drive = 'True'

        backend = self.useFixture(
            nova_fixtures.LibvirtImageBackendFixture(exists=lambda x: False))

        mock_build_device_metadata.return_value = None
        injection_info = get_injection_info(
            network_info=mock.sentinel.network_info,
            admin_pass=mock.sentinel.admin_pass,
            files=mock.sentinel.files
        )
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._create_configdrive(self.context, instance, injection_info)

        expected_config_drive_path = os.path.join(
            CONF.instances_path, instance.uuid, 'disk.config')
        mock_make_drive.assert_called_once_with(expected_config_drive_path)
        mock_instance_metadata.assert_called_once_with(instance,
            network_info=mock.sentinel.network_info,
            content=mock.sentinel.files,
            extra_md={'admin_pass': mock.sentinel.admin_pass})

        backend.disks['disk.config'].import_file.assert_called_once_with(
            instance, mock.ANY, 'disk.config')

    @ddt.unpack
    @ddt.data({'expected': 200, 'flavor_size': 200},
              {'expected': 100, 'flavor_size': 200, 'bdi_size': 100},
              {'expected': 200, 'flavor_size': 200, 'bdi_size': 100,
               'legacy': True})
    def test_create_image_with_swap(self, expected,
                                    flavor_size=None, bdi_size=None,
                                    legacy=False):
        # Test the precedence of swap disk size specified in both the bdm and
        # the flavor.
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance_ref = self.test_instance
        instance_ref['image_ref'] = ''
        instance = objects.Instance(**instance_ref)

        if flavor_size is not None:
            instance.flavor.swap = flavor_size

        bdi = {'block_device_mapping': [{'boot_index': 0}]}
        if bdi_size is not None:
            bdi['swap'] = {'swap_size': bdi_size, 'device_name': '/dev/vdb'}

        create_image_kwargs = {}
        if legacy:
            create_image_kwargs['ignore_bdi_for_swap'] = True

        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance, image_meta,
                                            block_device_info=bdi)

        backend = self.useFixture(nova_fixtures.LibvirtImageBackendFixture())
        drvr._create_image(self.context, instance, disk_info['mapping'],
                           block_device_info=bdi, **create_image_kwargs)

        backend.mock_create_swap.assert_called_once_with(
            target='swap_%i' % expected, swap_mb=expected,
            context=self.context)
        backend.disks['disk.swap'].cache.assert_called_once_with(
            fetch_func=mock.ANY, filename='swap_%i' % expected,
            size=expected * units.Mi, context=self.context, swap_mb=expected)

    @mock.patch.object(nova.virt.libvirt.imagebackend.Image, 'cache')
    def test_create_vz_container_with_swap(self, mock_cache):
        self.flags(virt_type='parallels', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        instance_ref = copy.deepcopy(self.test_instance)
        instance_ref['vm_mode'] = fields.VMMode.EXE
        instance_ref['flavor'].swap = 1024
        instance = objects.Instance(**instance_ref)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance, image_meta)
        self.assertRaises(exception.Invalid,
                          drvr._create_image,
                          self.context, instance, disk_info['mapping'])

    @mock.patch.object(nova.virt.libvirt.imagebackend.Image, 'cache',
                       side_effect=exception.ImageNotFound(image_id='fake-id'))
    def test_create_image_not_exist_no_fallback(self, mock_cache):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance,
                                            image_meta)
        self.assertRaises(exception.ImageNotFound,
                          drvr._create_image,
                          self.context, instance, disk_info['mapping'])

    @mock.patch.object(nova.virt.libvirt.imagebackend.Image, 'cache')
    def test_create_image_not_exist_fallback(self, mock_cache):

        def side_effect(fetch_func, filename, size=None, *args, **kwargs):
            def second_call(fetch_func, filename, size=None, *args, **kwargs):
                # call copy_from_host ourselves because we mocked image.cache()
                fetch_func('fake-target')
                # further calls have no side effect
                mock_cache.side_effect = None
            mock_cache.side_effect = second_call
            # raise an error only the first call
            raise exception.ImageNotFound(image_id='fake-id')

        mock_cache.side_effect = side_effect
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance,
                                            image_meta)

        with mock.patch('nova.virt.libvirt.utils.copy_image') as mock_copy:
            drvr._create_image(self.context, instance, disk_info['mapping'],
                               fallback_from_host='fake-source-host')
            mock_copy.assert_called_once_with(src='fake-target',
                                              dest='fake-target',
                                              host='fake-source-host',
                                              receive=True)

    @mock.patch('nova.privsep.fs.get_file_extension_for_os_type')
    def test_create_image_with_ephemerals(self, mock_get_ext):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance_ref = self.test_instance
        instance_ref['image_ref'] = ''
        instance = objects.Instance(**instance_ref)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        bdi = {'ephemerals': [{'size': 100}],
               'block_device_mapping': [{'boot_index': 0}]}
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance, image_meta,
                                            block_device_info=bdi)
        mock_get_ext.return_value = mock.sentinel.file_ext
        backend = self.useFixture(nova_fixtures.LibvirtImageBackendFixture())

        drvr._create_image(self.context, instance, disk_info['mapping'],
                           block_device_info=bdi)

        filename = 'ephemeral_100_%s' % mock.sentinel.file_ext
        backend.mock_create_ephemeral.assert_called_once_with(
            target=filename, ephemeral_size=100, fs_label='ephemeral0',
            is_block_dev=mock.sentinel.is_block_dev, os_type='linux',
            specified_fs=None, context=self.context, vm_mode=None)
        backend.disks['disk.eph0'].cache.assert_called_once_with(
            fetch_func=mock.ANY, context=self.context,
            filename=filename, size=100 * units.Gi, ephemeral_size=mock.ANY,
            specified_fs=None)

    def test_create_image_resize_snap_backend(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        instance.task_state = task_states.RESIZE_FINISH
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance,
                                            image_meta)

        backend = self.useFixture(nova_fixtures.LibvirtImageBackendFixture())

        drvr._create_image(self.context, instance, disk_info['mapping'])

        # Assert we called create_snap on the root disk
        backend.disks['disk'].create_snap.assert_called_once_with(
            libvirt_utils.RESIZE_SNAPSHOT_NAME)

    @mock.patch('nova.privsep.fs.mkfs')
    def test_create_ephemeral_specified_fs(self, fake_mkfs):
        self.flags(default_ephemeral_format='ext3')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._create_ephemeral('/dev/something', 20, 'myVol', 'linux',
                               is_block_dev=True, specified_fs='ext4')
        fake_mkfs.assert_has_calls([mock.call('ext4', '/dev/something',
                                              'myVol')])

    @mock.patch('nova.privsep.path.utime')
    @mock.patch('nova.virt.libvirt.utils.fetch_image')
    @mock.patch('nova.virt.libvirt.utils.create_image')
    def test_create_ephemeral_specified_fs_not_valid(
            self, mock_create_cow_image, mock_fetch_image, mock_utime):
        CONF.set_override('default_ephemeral_format', 'ext4')
        ephemerals = [{'device_type': 'disk',
                       'disk_bus': 'virtio',
                       'device_name': '/dev/vdb',
                       'guest_format': 'dummy',
                       'size': 1}]
        block_device_info = {
                'ephemerals': ephemerals}
        instance_ref = self.test_instance
        instance_ref['image_ref'] = uuids.image_ref
        instance = objects.Instance(**instance_ref)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        image_meta = objects.ImageMeta.from_dict({'disk_format': 'raw'})
        disk_info = blockinfo.get_disk_info(
            CONF.libvirt.virt_type, instance, image_meta,
            block_device_info=block_device_info)

        with test.nested(
            mock.patch('oslo_concurrency.processutils.execute'),
            mock.patch.object(drvr, 'get_info'),
            mock.patch.object(drvr, '_create_guest_with_network'),
            mock.patch.object(imagebackend.Image, 'verify_base_size'),
            mock.patch.object(imagebackend.Image, 'get_disk_size')
        ) as (execute_mock, get_info_mock,
              create_mock, verify_base_size_mock, disk_size_mock):
            disk_size_mock.return_value = 0
            self.assertRaises(exception.InvalidBDMFormat, drvr._create_image,
                              context, instance, disk_info['mapping'],
                              block_device_info=block_device_info)

    @mock.patch('nova.privsep.fs.mkfs')
    def test_create_ephemeral_default(self, fake_mkfs):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._create_ephemeral('/dev/something', 20, 'myVol', 'linux',
                               is_block_dev=True)
        fake_mkfs.assert_has_calls([mock.call('ext4', '/dev/something',
                                              'myVol')])

    @mock.patch('nova.privsep.fs.mkfs')
    def test_create_ephemeral_with_conf(self, fake_mkfs):
        CONF.set_override('default_ephemeral_format', 'ext4')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._create_ephemeral('/dev/something', 20, 'myVol', 'linux',
                               is_block_dev=True)
        fake_mkfs.assert_has_calls([mock.call('ext4', '/dev/something',
                                              'myVol')])

    @mock.patch('nova.privsep.fs.configurable_mkfs')
    def test_create_ephemeral_with_arbitrary(self, fake_mkfs):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.stub_out('nova.privsep.fs._MKFS_COMMAND',
                      {'linux': 'mkfs.ext4 --label %(fs_label)s %(target)s'})
        drvr._create_ephemeral('/dev/something', 20, 'myVol', 'linux',
                               is_block_dev=True)
        fake_mkfs.assert_has_calls([mock.call('linux', 'myVol',
                                              '/dev/something', True, None,
                                              None)])

    @mock.patch('nova.privsep.fs.configurable_mkfs')
    def test_create_ephemeral_with_ext3(self, fake_mkfs):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.stub_out('nova.privsep.fs._MKFS_COMMAND',
                      {'linux': 'mkfs.ext3 --label %(fs_label)s %(target)s'})
        drvr._create_ephemeral('/dev/something', 20, 'myVol', 'linux',
                               is_block_dev=True)
        fake_mkfs.assert_has_calls([mock.call('linux', 'myVol',
                                              '/dev/something', True, None,
                                              None)])

    @mock.patch('nova.virt.libvirt.utils.create_ploop_image')
    def test_create_ephemeral_parallels(self, mock_create_ploop):
        self.flags(virt_type='parallels', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._create_ephemeral('/dev/something', 20, 'myVol', 'linux',
                               is_block_dev=False,
                               specified_fs='fs_format',
                               vm_mode=fields.VMMode.EXE)
        mock_create_ploop.assert_called_once_with('expanded',
                                                  '/dev/something',
                                                  '20G', 'fs_format')

    @mock.patch('nova.privsep.fs.unprivileged_mkfs')
    @mock.patch('nova.virt.libvirt.utils.create_image')
    def test_create_swap_default(self, mock_create_image, mock_mkfs):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._create_swap('/dev/something', 1)
        mock_mkfs.assert_has_calls([mock.call('swap', '/dev/something')])

    def test_ensure_console_log_for_instance_pass(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        with test.nested(
                mock.patch.object(drvr, '_get_console_log_path'),
                mock.patch('nova.virt.libvirt.utils.file_open')
            ) as (mock_path, mock_open):
            drvr._ensure_console_log_for_instance(mock.ANY)
            mock_path.assert_called_once()
            mock_open.assert_called_once()

    def test_ensure_console_log_for_instance_pass_w_permissions(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        with test.nested(
                mock.patch.object(drvr, '_get_console_log_path'),
                mock.patch('nova.virt.libvirt.utils.file_open',
                                  side_effect=IOError(errno.EACCES, 'exc'))
            ) as (mock_path, mock_open):
            drvr._ensure_console_log_for_instance(mock.ANY)
            mock_path.assert_called_once()
            mock_open.assert_called_once()

    def test_ensure_console_log_for_instance_fail(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        with test.nested(
                mock.patch.object(drvr, '_get_console_log_path'),
                mock.patch('nova.virt.libvirt.utils.file_open',
                                  side_effect=IOError(errno.EREMOTE, 'exc'))
            ) as (mock_path, mock_open):
            self.assertRaises(
                IOError,
                drvr._ensure_console_log_for_instance,
                mock.ANY)

    @mock.patch('nova.privsep.path.last_bytes',
                return_value=(b'67890', 0))
    def test_get_console_output_file(self, mock_last_bytes):
        with utils.tempdir() as tmpdir:
            self.flags(instances_path=tmpdir)

            instance_ref = self.test_instance
            instance_ref['image_ref'] = 123456
            instance = objects.Instance(**instance_ref)

            console_dir = (os.path.join(tmpdir, instance['name']))
            console_log = '%s/console.log' % (console_dir)
            fake_dom_xml = """
                <domain type='kvm'>
                    <devices>
                        <disk type='file'>
                            <source file='filename'/>
                        </disk>
                        <console type='file'>
                            <source path='%s'/>
                            <target port='0'/>
                        </console>
                    </devices>
                </domain>
            """ % console_log

            def fake_lookup(id):
                return FakeVirtDomain(fake_dom_xml)

            self.create_fake_libvirt_mock()
            libvirt_driver.LibvirtDriver._conn.lookupByUUIDString = fake_lookup

            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

            try:
                prev_max = libvirt_driver.MAX_CONSOLE_BYTES
                libvirt_driver.MAX_CONSOLE_BYTES = 5
                with mock.patch('os.path.exists', return_value=True):
                    output = drvr.get_console_output(self.context, instance)
            finally:
                libvirt_driver.MAX_CONSOLE_BYTES = prev_max

            self.assertEqual(b'67890', output)

    # this test resembles test_get_console_output_file() except
    # that the instance was created with a tcp-based serial console
    # which results in a different XML
    @mock.patch('nova.privsep.path.last_bytes',
                return_value=(b'67891', 0))
    def test_get_console_output_tcp(self, mock_last_bytes):
        with utils.tempdir() as tmpdir:
            self.flags(instances_path=tmpdir)
            # flags to enable the serial console are not necessary
            # as we use a fake dom

            instance_ref = self.test_instance
            instance_ref['image_ref'] = 123456
            instance = objects.Instance(**instance_ref)

            console_dir = (os.path.join(tmpdir, instance['name']))
            console_log = '%s/console.log' % (console_dir)
            fake_dom_xml = """
                <domain type='kvm'>
                    <devices>
                        <disk type='file'>
                            <source file='filename'/>
                        </disk>
                        <console type='tcp'>
                            <log file='%s'/>
                        </console>
                    </devices>
                </domain>
            """ % console_log

            def fake_lookup(id):
                return FakeVirtDomain(fake_dom_xml)

            self.create_fake_libvirt_mock()
            libvirt_driver.LibvirtDriver._conn.lookupByUUIDString = fake_lookup

            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

            try:
                prev_max = libvirt_driver.MAX_CONSOLE_BYTES
                libvirt_driver.MAX_CONSOLE_BYTES = 5
                with mock.patch('os.path.exists', return_value=True):
                    output = drvr.get_console_output(self.context, instance)
            finally:
                libvirt_driver.MAX_CONSOLE_BYTES = prev_max

            self.assertEqual(b'67891', output)

    def test_get_console_output_file_missing(self):
        with utils.tempdir() as tmpdir:
            self.flags(instances_path=tmpdir)

            instance_ref = self.test_instance
            instance_ref['image_ref'] = 123456
            instance = objects.Instance(**instance_ref)

            console_log = os.path.join(tmpdir, instance['name'],
                                       'non-existent.log')
            fake_dom_xml = """
                <domain type='kvm'>
                    <devices>
                        <disk type='file'>
                            <source file='filename'/>
                        </disk>
                        <console type='file'>
                            <source path='%s'/>
                            <target port='0'/>
                        </console>
                    </devices>
                </domain>
            """ % console_log

            def fake_lookup(id):
                return FakeVirtDomain(fake_dom_xml)

            self.create_fake_libvirt_mock()
            libvirt_driver.LibvirtDriver._conn.lookupByUUIDString = fake_lookup

            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

            with mock.patch('os.path.exists', return_value=False):
                output = drvr.get_console_output(self.context, instance)

            self.assertEqual('', output)

    @mock.patch('os.path.exists', return_value=True)
    @mock.patch('nova.privsep.path.last_bytes',
                return_value=(b'67890', 0))
    @mock.patch('nova.privsep.path.writefile')
    @mock.patch('nova.privsep.libvirt.readpty')
    def test_get_console_output_pty(self, mocked_readfile, mocked_writefile,
                                    mocked_last_bytes, mocked_path_exists):
        with utils.tempdir() as tmpdir:
            self.flags(instances_path=tmpdir)

            instance_ref = self.test_instance
            instance_ref['image_ref'] = 123456
            instance = objects.Instance(**instance_ref)

            console_dir = (os.path.join(tmpdir, instance['name']))
            pty_file = '%s/fake_pty' % (console_dir)
            fake_dom_xml = """
                <domain type='kvm'>
                    <devices>
                        <disk type='file'>
                            <source file='filename'/>
                        </disk>
                        <console type='pty'>
                            <source path='%s'/>
                            <target port='0'/>
                        </console>
                    </devices>
                </domain>
            """ % pty_file

            def fake_lookup(id):
                return FakeVirtDomain(fake_dom_xml)

            mocked_readfile.return_value = 'foo'

            self.create_fake_libvirt_mock()
            libvirt_driver.LibvirtDriver._conn.lookupByUUIDString = fake_lookup

            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

            try:
                prev_max = libvirt_driver.MAX_CONSOLE_BYTES
                libvirt_driver.MAX_CONSOLE_BYTES = 5
                output = drvr.get_console_output(self.context, instance)
            finally:
                libvirt_driver.MAX_CONSOLE_BYTES = prev_max

            self.assertEqual(b'67890', output)

    def test_get_console_output_pty_not_available(self):
        instance = objects.Instance(**self.test_instance)
        fake_dom_xml = """
            <domain type='kvm'>
                <devices>
                    <disk type='file'>
                        <source file='filename'/>
                    </disk>
                    <console type='pty'>
                        <target port='0'/>
                    </console>
                </devices>
            </domain>
        """

        def fake_lookup(id):
            return FakeVirtDomain(fake_dom_xml)

        self.create_fake_libvirt_mock()
        libvirt_driver.LibvirtDriver._conn.lookupByUUIDString = fake_lookup
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertRaises(exception.ConsoleNotAvailable,
                          drvr.get_console_output, self.context, instance)

    @mock.patch('nova.virt.libvirt.host.Host._get_domain')
    @mock.patch.object(libvirt_guest.Guest, "get_xml_desc")
    def test_get_console_output_not_available(self, mock_get_xml, get_domain):
        xml = """
        <domain type='kvm'>
            <devices>
                <disk type='file'>
                    <source file='filename'/>
                </disk>
                <console type='foo'>
                    <source path='srcpath'/>
                    <target port='0'/>
                </console>
            </devices>
        </domain>
        """
        mock_get_xml.return_value = xml
        get_domain.return_value = mock.MagicMock()
        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertRaises(exception.ConsoleNotAvailable,
                          drvr.get_console_output, self.context, instance)

    @mock.patch('nova.virt.libvirt.host.Host._get_domain')
    @mock.patch.object(libvirt_guest.Guest, 'get_xml_desc')
    def test_get_console_output_logrotate(self, mock_get_xml, get_domain):
        self.useFixture(nova_fixtures.PrivsepFixture())

        fake_files = {}
        fake_files['console.log'] = b'uvwxyz'
        fake_files['console.log.0'] = b'klmnopqrst'
        fake_files['console.log.1'] = b'abcdefghij'

        def mock_path_exists(path):
            return os.path.basename(path) in fake_files

        def fake_open(path, mode):
            if path.endswith('console.log'):
                return io.BytesIO(b'uvwxyz')
            if path.endswith('console.log.0'):
                return io.BytesIO(b'klmnopqrst')
            if path.endswith('console.log.1'):
                return io.BytesIO(b'abcdefghij')
            raise Exception('No such file in testing')

        xml = """
        <domain type='kvm'>
            <devices>
                <disk type='file'>
                    <source file='filename'/>
                </disk>
                <console type='file'>
                    <source path='console.log'/>
                    <target port='0'/>
                </console>
            </devices>
        </domain>
        """
        mock_get_xml.return_value = xml
        get_domain.return_value = mock.MagicMock()

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance = objects.Instance(**self.test_instance)

        def _get_logd_output(bytes_to_read):
            with utils.tempdir() as tmp_dir:
                self.flags(instances_path=tmp_dir)
                log_data = ""
                try:
                    prev_max = libvirt_driver.MAX_CONSOLE_BYTES
                    libvirt_driver.MAX_CONSOLE_BYTES = bytes_to_read
                    with test.nested(
                            mock.patch('os.path.exists',
                                       side_effect=mock_path_exists),
                            mock.patch('builtins.open', fake_open)):
                        log_data = drvr.get_console_output(self.context,
                                                           instance)
                finally:
                    libvirt_driver.MAX_CONSOLE_BYTES = prev_max
                return log_data

        # span across only 1 file (with remaining bytes)
        self.assertEqual(b'wxyz', _get_logd_output(4))
        # span across only 1 file (exact bytes)
        self.assertEqual(b'uvwxyz', _get_logd_output(6))
        # span across 2 files (with remaining bytes)
        self.assertEqual(b'opqrstuvwxyz', _get_logd_output(12))
        # span across all files (exact bytes)
        self.assertEqual(b'abcdefghijklmnopqrstuvwxyz', _get_logd_output(26))
        # span across all files with more bytes than available
        self.assertEqual(b'abcdefghijklmnopqrstuvwxyz', _get_logd_output(30))
        # files are not available
        fake_files = {}
        self.assertEqual('', _get_logd_output(30))

    def test_get_host_ip_addr(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        ip = drvr.get_host_ip_addr()
        self.assertEqual(ip, CONF.my_ip)

    def test_get_host_ip_addr_defaults_to_my_ip(self):
        CONF.set_default("my_ip", "10.0.0.3")
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        ip = drvr.get_host_ip_addr()
        self.assertEqual(ip, "10.0.0.3")

    def test_get_host_ip_addr_override_via_migration_inbound_addr(self):
        CONF.set_default("my_ip", "10.0.0.3")
        CONF.set_default(
            "migration_inbound_addr", "my-migration-hostname", group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        ip = drvr.get_host_ip_addr()
        self.assertEqual(ip, "my-migration-hostname")

    def test_get_host_ip_addr_override_via_migration_inbound_addr_template(
        self
    ):
        CONF.set_default("my_ip", "10.0.0.3")
        CONF.set_default("migration_inbound_addr", "%s", group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        ip = drvr.get_host_ip_addr()
        self.assertEqual(ip, "compute1")

    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test_check_my_ip(self, mock_log):

        self.libvirt.mock_get_machine_ips.return_value = [
            '8.8.8.8', '75.75.75.75']
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._check_my_ip()
        mock_log.assert_called_once_with(u'my_ip address (%(my_ip)s) was '
                                         u'not found on any of the '
                                         u'interfaces: %(ifaces)s',
                                         {'ifaces': '8.8.8.8, 75.75.75.75',
                                          'my_ip': mock.ANY})

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    def test_init_host_checks_ip(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        with mock.patch.object(drvr, '_check_my_ip') as mock_check:
            drvr.init_host('fake-host')
            mock_check.assert_called_once_with()

    def test_conn_event_handler(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        service_mock = mock.MagicMock()
        service_mock.disabled.return_value = False
        drvr._host._init_events.return_value = None
        with test.nested(
            mock.patch.object(drvr._host, "_connect",
                              side_effect=fakelibvirt.make_libvirtError(
                                  fakelibvirt.libvirtError,
                                  "Failed to connect to host",
                                  error_code=
                                  fakelibvirt.VIR_ERR_INTERNAL_ERROR)),
            mock.patch.object(objects.Service, "get_by_compute_host",
                              return_value=service_mock)):

            # verify that the driver registers for the close callback
            # and re-connects after receiving the callback
            self.assertRaises(exception.HypervisorUnavailable,
                              drvr.init_host,
                              "wibble")
            self.assertTrue(service_mock.disabled)

    def test_command_with_broken_connection(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        service_mock = mock.MagicMock()
        service_mock.disabled.return_value = False
        drvr._host._init_events.return_value = None
        with test.nested(
            mock.patch.object(drvr._host, "_connect",
                              side_effect=fakelibvirt.make_libvirtError(
                                  fakelibvirt.libvirtError,
                                  "Failed to connect to host",
                                  error_code=
                                  fakelibvirt.VIR_ERR_INTERNAL_ERROR)),
            mock.patch.object(host.Host, "has_min_version",
                              return_value=True),
            mock.patch.object(drvr, "_do_quality_warnings",
                              return_value=None),
            mock.patch.object(objects.Service, "get_by_compute_host",
                              return_value=service_mock),
            mock.patch.object(host.Host, "get_capabilities")):

            self.assertRaises(exception.HypervisorUnavailable,
                              drvr.init_host, ("wibble",))
            self.assertTrue(service_mock.disabled)

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    def test_service_resume_after_broken_connection(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        service_mock = mock.MagicMock()
        service_mock.disabled.return_value = True
        drvr._host._init_events.return_value = None
        with test.nested(
            mock.patch.object(drvr._host, "_connect",
                              return_value=mock.MagicMock()),
            mock.patch.object(host.Host, "has_min_version",
                              return_value=True),
            mock.patch.object(drvr, "_do_quality_warnings",
                              return_value=None),
            mock.patch.object(objects.Service, "get_by_compute_host",
                              return_value=service_mock),
            mock.patch.object(host.Host, "get_capabilities")):

            drvr.init_host("wibble")
            drvr.get_num_instances()
            drvr._host._dispatch_conn_event()
            self.assertFalse(service_mock.disabled)
            self.assertIsNone(service_mock.disabled_reason)

    @mock.patch.object(libvirt_driver.AsyncDeviceEventsHandler,
                       'cleanup_waiters')
    @mock.patch.object(libvirt_driver.LibvirtDriver, 'delete_instance_files')
    @mock.patch.object(host.Host, '_get_domain',
                       side_effect=exception.InstanceNotFound(
                           instance_id=uuids.instance))
    @mock.patch.object(objects.Instance, 'save')
    def test_immediate_delete(self, mock_save, mock_get, mock_delete,
        mock_cleanup_async_detach
    ):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        instance = objects.Instance(self.context, **self.test_instance)
        drvr.destroy(self.context, instance, {})
        mock_save.assert_called_once_with()
        mock_get.assert_has_calls([mock.call(instance)] * 3)
        self.assertEqual(3, mock_get.call_count)
        mock_delete.assert_called_once_with(instance)
        mock_cleanup_async_detach.assert_called_once_with(instance.uuid)

    @mock.patch.object(objects.Instance, 'get_by_uuid')
    @mock.patch.object(objects.Instance, 'obj_load_attr', autospec=True)
    @mock.patch.object(objects.Instance, 'save', autospec=True)
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_destroy')
    @mock.patch.object(libvirt_driver.LibvirtDriver, 'delete_instance_files')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_disconnect_volume')
    @mock.patch.object(driver, 'block_device_info_get_mapping')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_undefine_domain')
    def _test_destroy_removes_disk(self, mock_undefine_domain, mock_mapping,
                                   mock_disconnect_volume,
                                   mock_delete_instance_files, mock_destroy,
                                   mock_inst_save, mock_inst_obj_load_attr,
                                   mock_get_by_uuid, volume_fail=False):
        instance = objects.Instance(self.context, **self.test_instance)
        vol = {'block_device_mapping': [
              {'connection_info': 'dummy', 'mount_device': '/dev/sdb'}]}

        mock_mapping.return_value = vol['block_device_mapping']
        mock_delete_instance_files.return_value = True
        mock_get_by_uuid.return_value = instance
        if volume_fail:
            mock_disconnect_volume.return_value = (
                exception.VolumeNotFound('vol'))

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr.destroy(self.context, instance, [], vol)

    def test_destroy_removes_disk(self):
        self._test_destroy_removes_disk(volume_fail=False)

    def test_destroy_removes_disk_volume_fails(self):
        self._test_destroy_removes_disk(volume_fail=True)

    @mock.patch.object(libvirt_driver.LibvirtDriver, 'unplug_vifs')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_destroy')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_undefine_domain')
    def test_destroy_not_removes_disk(self, mock_undefine_domain, mock_destroy,
                                      mock_unplug_vifs):
        instance = fake_instance.fake_instance_obj(
            None, name='instancename', id=1,
            uuid='875a8070-d0b9-4949-8b31-104d125c9a64',
            expected_attrs=['resources'])

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr.destroy(self.context, instance, [], None, False)

    @mock.patch.object(libvirt_driver.LibvirtDriver, 'cleanup')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_teardown_container')
    @mock.patch.object(host.Host, '_get_domain')
    def test_destroy_lxc_calls_teardown_container(self, mock_get_domain,
                                                  mock_teardown_container,
                                                  mock_cleanup):
        self.flags(virt_type='lxc', group='libvirt')
        fake_domain = FakeVirtDomain()

        def destroy_side_effect(*args, **kwargs):
            fake_domain._info[0] = power_state.SHUTDOWN

        with mock.patch.object(fake_domain, 'destroy',
               side_effect=destroy_side_effect) as mock_domain_destroy:
            mock_get_domain.return_value = fake_domain
            instance = objects.Instance(**self.test_instance)

            drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
            network_info = []
            drvr.destroy(self.context, instance, network_info, None, False)

            mock_get_domain.assert_has_calls([mock.call(instance),
                                              mock.call(instance)])
            mock_domain_destroy.assert_called_once_with()
            mock_teardown_container.assert_called_once_with(instance)
            mock_cleanup.assert_called_once_with(self.context, instance,
                                                 network_info, None, False,
                                                 destroy_secrets=True)

    @mock.patch.object(libvirt_driver.LibvirtDriver, 'cleanup')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_teardown_container')
    @mock.patch.object(host.Host, '_get_domain')
    def test_destroy_lxc_calls_teardown_container_when_no_domain(self,
            mock_get_domain, mock_teardown_container, mock_cleanup):
        self.flags(virt_type='lxc', group='libvirt')
        instance = objects.Instance(**self.test_instance)
        inf_exception = exception.InstanceNotFound(instance_id=instance.uuid)
        mock_get_domain.side_effect = inf_exception

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        network_info = []
        drvr.destroy(self.context, instance, network_info, None, False)

        mock_get_domain.assert_has_calls([mock.call(instance),
                                          mock.call(instance)])
        mock_teardown_container.assert_called_once_with(instance)
        mock_cleanup.assert_called_once_with(self.context, instance,
                                             network_info, None, False,
                                             destroy_secrets=True)

    @mock.patch.object(host.Host, 'get_guest')
    def test_reboot_different_ids(self, mock_get):
        class FakeLoopingCall(object):
            def start(self, *a, **k):
                return self

            def wait(self):
                return None

        self.flags(wait_soft_reboot_seconds=1, group='libvirt')
        info_tuple = ('fake', 'fake', 'fake', 'also_fake')

        # Mock domain
        mock_domain = mock.create_autospec(fakelibvirt.virDomain)
        mock_domain.info.side_effect = [
            (libvirt_guest.VIR_DOMAIN_RUNNING,) + info_tuple,
            (libvirt_guest.VIR_DOMAIN_CRASHED,) + info_tuple]
        mock_domain.ID.side_effect = ['some_fake_id', 'some_fake_id',
                                      'some_other_fake_id',
                                      'some_other_fake_id']
        fake_guest = libvirt_guest.Guest(mock_domain)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)

        mock_get.return_value = fake_guest
        self.stub_out('oslo_service.loopingcall.FixedIntervalLoopingCall',
                      lambda *a, **k: FakeLoopingCall())

        drvr.reboot(None, instance, [], 'SOFT')

        mock_domain.info.assert_has_calls([mock.call()] * 2)
        self.assertEqual(2, mock_domain.info.call_count)
        mock_domain.ID.assert_has_calls([mock.call()] * 4)
        self.assertEqual(4, mock_domain.ID.call_count)
        mock_domain.shutdown.assert_called_once_with()
        mock_get.assert_has_calls([mock.call(instance)] * 2, any_order=True)
        self.assertEqual(2, mock_get.call_count)

    @mock.patch.object(loopingcall, 'FixedIntervalLoopingCall')
    @mock.patch.object(greenthread, 'sleep')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_hard_reboot')
    @mock.patch.object(host.Host, '_get_domain')
    def test_reboot_same_ids(self, mock_get_domain, mock_hard_reboot,
                             mock_sleep, mock_loopingcall):
        class FakeLoopingCall(object):
            def start(self, *a, **k):
                return self

            def wait(self):
                return None

        self.flags(wait_soft_reboot_seconds=1, group='libvirt')
        info_tuple = ('fake', 'fake', 'fake', 'also_fake')
        self.reboot_hard_reboot_called = False

        # Mock domain
        mock_domain = mock.Mock(fakelibvirt.virDomain)
        return_values = [(libvirt_guest.VIR_DOMAIN_RUNNING,) + info_tuple,
                         (libvirt_guest.VIR_DOMAIN_CRASHED,) + info_tuple]
        mock_domain.info.side_effect = return_values
        mock_domain.ID.return_value = 'some_fake_id'
        mock_domain.shutdown.side_effect = mock.Mock()

        def fake_hard_reboot(*args, **kwargs):
            self.reboot_hard_reboot_called = True

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        mock_get_domain.return_value = mock_domain
        mock_hard_reboot.side_effect = fake_hard_reboot
        mock_loopingcall.return_value = FakeLoopingCall()
        drvr.reboot(None, instance, [], 'SOFT')
        self.assertTrue(self.reboot_hard_reboot_called)

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_hard_reboot')
    @mock.patch.object(host.Host, '_get_domain')
    def test_soft_reboot_libvirt_exception(self, mock_get_domain,
                                           mock_hard_reboot):
        # Tests that a hard reboot is performed when a soft reboot results
        # in raising a libvirtError.
        info_tuple = ('fake', 'fake', 'fake', 'also_fake')
        # setup mocks
        mock_virDomain = mock.Mock(fakelibvirt.virDomain)
        mock_virDomain.info.return_value = (
            (libvirt_guest.VIR_DOMAIN_RUNNING,) + info_tuple)
        mock_virDomain.ID.return_value = 'some_fake_id'
        mock_virDomain.shutdown.side_effect = fakelibvirt.libvirtError('Err')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        context = None
        instance = objects.Instance(**self.test_instance)
        network_info = []
        mock_get_domain.return_value = mock_virDomain

        drvr.reboot(context, instance, network_info, 'SOFT')

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_hard_reboot')
    @mock.patch.object(host.Host, '_get_domain')
    def _test_resume_state_on_host_boot_with_state(self, state,
                                                   mock_get_domain,
                                                   mock_hard_reboot):
        mock_virDomain = mock.Mock(fakelibvirt.virDomain)
        mock_virDomain.info.return_value = ([state, None, None, None, None])

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        mock_get_domain.return_value = mock_virDomain
        instance = objects.Instance(**self.test_instance)
        network_info = _fake_network_info(self)

        drvr.resume_state_on_host_boot(self.context, instance, network_info,
                                       block_device_info=None)

        ignored_states = (power_state.RUNNING,
                          power_state.SUSPENDED,
                          power_state.NOSTATE,
                          power_state.PAUSED)
        self.assertEqual(mock_hard_reboot.called, state not in ignored_states)

    def test_resume_state_on_host_boot_with_running_state(self):
        self._test_resume_state_on_host_boot_with_state(power_state.RUNNING)

    def test_resume_state_on_host_boot_with_suspended_state(self):
        self._test_resume_state_on_host_boot_with_state(power_state.SUSPENDED)

    def test_resume_state_on_host_boot_with_paused_state(self):
        self._test_resume_state_on_host_boot_with_state(power_state.PAUSED)

    def test_resume_state_on_host_boot_with_nostate(self):
        self._test_resume_state_on_host_boot_with_state(power_state.NOSTATE)

    def test_resume_state_on_host_boot_with_shutdown_state(self):
        self._test_resume_state_on_host_boot_with_state(power_state.RUNNING)

    def test_resume_state_on_host_boot_with_crashed_state(self):
        self._test_resume_state_on_host_boot_with_state(power_state.CRASHED)

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_hard_reboot')
    @mock.patch.object(host.Host, '_get_domain')
    def test_resume_state_on_host_boot_with_instance_not_found_on_driver(
            self, mock_get_domain, mock_hard_reboot):
        instance = objects.Instance(**self.test_instance)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        mock_get_domain.side_effect = exception.InstanceNotFound(
            instance_id='fake')
        drvr.resume_state_on_host_boot(self.context, instance, network_info=[],
                                       block_device_info=None)

        mock_hard_reboot.assert_called_once_with(self.context,
                                                 instance, [], None)

    @mock.patch('nova.virt.libvirt.LibvirtDriver.get_info')
    @mock.patch('nova.virt.libvirt.LibvirtDriver._create_guest_with_network')
    @mock.patch('nova.virt.libvirt.LibvirtDriver._get_guest_xml')
    @mock.patch('nova.virt.libvirt.LibvirtDriver.'
                '_get_instance_disk_info_from_config')
    @mock.patch('nova.virt.libvirt.LibvirtDriver.destroy')
    @mock.patch('nova.virt.libvirt.LibvirtDriver.'
                '_get_all_assigned_mediated_devices')
    def test_hard_reboot(self, mock_get_mdev, mock_destroy, mock_get_disk_info,
                         mock_get_guest_xml, mock_create_guest_with_network,
                         mock_get_info):
        self.context.auth_token = True  # any non-None value will suffice
        instance = objects.Instance(**self.test_instance)
        network_info = _fake_network_info(self)
        block_device_info = None

        dummyxml = ("<domain type='kvm'><name>instance-0000000a</name>"
                    "<devices>"
                    "<disk type='file'><driver name='qemu' type='raw'/>"
                    "<source file='/test/disk'/>"
                    "<target dev='vda' bus='virtio'/></disk>"
                    "<disk type='file'><driver name='qemu' type='qcow2'/>"
                    "<source file='/test/disk.local'/>"
                    "<target dev='vdb' bus='virtio'/></disk>"
                    "</devices></domain>")

        mock_get_mdev.return_value = {uuids.mdev1: uuids.inst1}
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        return_values = [hardware.InstanceInfo(state=power_state.SHUTDOWN),
                         hardware.InstanceInfo(state=power_state.RUNNING)]
        mock_get_info.side_effect = return_values

        mock_get_guest_xml.return_value = dummyxml
        mock_get_disk_info.return_value = \
            fake_disk_info_byname(instance).values()

        backend = self.useFixture(nova_fixtures.LibvirtImageBackendFixture())

        accel_info = [{'k1': 'v1', 'k2': 'v2'}]
        with mock.patch('os.path.exists', return_value=True):
            drvr._hard_reboot(self.context, instance, network_info,
                              block_device_info, accel_info=accel_info)

        disks = backend.disks

        # NOTE(mdbooth): _create_images_and_backing() passes a full path in
        # 'disk_name' when creating a disk. This is wrong, but happens to
        # work due to handling by each individual backend. This will be
        # fixed in a subsequent commit.
        #
        # We translate all the full paths into disk names here to make the
        # test readable
        disks = {os.path.basename(name): value
                 for name, value in disks.items()}

        # We should have called cache() on the root and ephemeral disks
        for name in ('disk', 'disk.local'):
            self.assertTrue(disks[name].cache.called)

        mock_get_mdev.assert_called_once_with(instance)
        mock_destroy.assert_called_once_with(self.context, instance,
                network_info, destroy_disks=False,
                block_device_info=block_device_info,
                destroy_secrets=False)

        mock_get_guest_xml.assert_called_once_with(self.context, instance,
            network_info, mock.ANY, mock.ANY,
            block_device_info=block_device_info, mdevs=[uuids.mdev1],
            accel_info=accel_info)
        mock_create_guest_with_network.assert_called_once_with(self.context,
            dummyxml, instance, network_info, block_device_info,
            vifs_already_plugged=True, external_events=[])

    @mock.patch('oslo_utils.fileutils.ensure_tree', new=mock.Mock())
    @mock.patch('nova.virt.libvirt.LibvirtDriver.get_info')
    @mock.patch('nova.virt.libvirt.LibvirtDriver._create_guest_with_network')
    @mock.patch('nova.virt.libvirt.LibvirtDriver._get_guest_xml')
    @mock.patch('nova.virt.libvirt.LibvirtDriver.destroy', new=mock.Mock())
    @mock.patch(
        'nova.virt.libvirt.LibvirtDriver._get_all_assigned_mediated_devices',
        new=mock.Mock(return_value={}))
    def test_hard_reboot_wait_for_plug(
        self, mock_get_guest_xml, mock_create_guest_with_network, mock_get_info
    ):
        self.flags(
            group="workarounds",
            wait_for_vif_plugged_event_during_hard_reboot=["normal"])
        self.context.auth_token = None
        instance = objects.Instance(**self.test_instance)
        network_info = _fake_network_info(self, num_networks=4)
        network_info[0]["vnic_type"] = "normal"
        network_info[1]["vnic_type"] = "direct"
        network_info[2]["vnic_type"] = "normal"
        network_info[3]["vnic_type"] = "direct-physical"
        block_device_info = None
        return_values = [hardware.InstanceInfo(state=power_state.SHUTDOWN),
                         hardware.InstanceInfo(state=power_state.RUNNING)]
        mock_get_info.side_effect = return_values
        mock_get_guest_xml.return_value = mock.sentinel.xml

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._hard_reboot(
                self.context, instance, network_info, block_device_info)

        mock_create_guest_with_network.assert_called_once_with(
            self.context, mock.sentinel.xml, instance, network_info,
            block_device_info,
            vifs_already_plugged=False,
            external_events=[
                ('network-vif-plugged', uuids.vif1),
                ('network-vif-plugged', uuids.vif3),
            ]
        )

    @mock.patch('oslo_utils.fileutils.ensure_tree')
    @mock.patch('oslo_service.loopingcall.FixedIntervalLoopingCall')
    @mock.patch('nova.virt.libvirt.LibvirtDriver._create_guest_with_network')
    @mock.patch('nova.virt.libvirt.LibvirtDriver._create_images_and_backing')
    @mock.patch('nova.virt.libvirt.LibvirtDriver.'
                '_get_instance_disk_info_from_config')
    @mock.patch('nova.virt.libvirt.utils.get_instance_path')
    @mock.patch('nova.virt.libvirt.LibvirtDriver._get_guest_config')
    @mock.patch('nova.virt.libvirt.blockinfo.get_disk_info')
    @mock.patch('nova.virt.libvirt.LibvirtDriver._destroy')
    @mock.patch('nova.virt.libvirt.LibvirtDriver.'
                '_get_all_assigned_mediated_devices')
    @mock.patch('builtins.open', new=mock.mock_open())
    def test_hard_reboot_does_not_call_glance_show(self,
            mock_get_mdev, mock_destroy, mock_get_disk_info,
            mock_get_guest_config, mock_get_instance_path,
            mock_get_instance_disk_info, mock_create_images_and_backing,
            mock_create_domand_and_network,
            mock_looping_call, mock_ensure_tree):
        """For a hard reboot, we shouldn't need an additional call to glance
        to get the image metadata.

        This is important for automatically spinning up instances on a
        host-reboot, since we won't have a user request context that'll allow
        the Glance request to go through. We have to rely on the cached image
        metadata, instead.

        https://bugs.launchpad.net/nova/+bug/1339386
        """
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        instance = objects.Instance(**self.test_instance)

        mock_get_mdev.return_value = {}

        network_info = mock.MagicMock()
        block_device_info = mock.MagicMock()
        mock_get_disk_info.return_value = {}
        mock_get_guest_config.return_value = mock.MagicMock()
        mock_get_instance_path.return_value = '/foo'
        mock_looping_call.return_value = mock.MagicMock()
        drvr._image_api = mock.MagicMock()

        drvr._hard_reboot(self.context, instance, network_info,
                          block_device_info)

        self.assertFalse(drvr._image_api.get.called)
        mock_ensure_tree.assert_called_once_with('/foo')

    def test_suspend(self):
        guest = libvirt_guest.Guest(FakeVirtDomain(id=1))
        dom = guest._domain

        instance = objects.Instance(**self.test_instance)
        instance.ephemeral_key_uuid = None

        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        @mock.patch.object(dmcrypt, 'delete_volume')
        @mock.patch.object(conn, '_get_instance_disk_info_from_config',
                           return_value=[])
        @mock.patch.object(conn, '_detach_mediated_devices')
        @mock.patch.object(conn, '_detach_direct_passthrough_ports')
        @mock.patch.object(conn, '_detach_pci_devices')
        @mock.patch.object(conn._host, 'get_guest', return_value=guest)
        def suspend(mock_get_guest,
                    mock_detach_pci_devices,
                    mock_detach_direct_passthrough_ports,
                    mock_detach_mediated_devices,
                    mock_get_instance_disk_info,
                    mock_delete_volume):
            mock_managedSave = mock.Mock()
            dom.managedSave = mock_managedSave

            conn.suspend(self.context, instance)

            mock_managedSave.assert_called_once_with(0)
            self.assertFalse(mock_get_instance_disk_info.called)
            mock_delete_volume.assert_has_calls([mock.call(disk['path'])
                for disk in mock_get_instance_disk_info.return_value], False)

        suspend()

    @mock.patch.object(time, 'sleep')
    @mock.patch.object(host.Host, '_get_domain')
    def _test_clean_shutdown(self, mock_get_domain,
                             mock_sleep, seconds_to_shutdown,
                             timeout, retry_interval,
                             shutdown_attempts, succeeds):
        info_tuple = ('fake', 'fake', 'fake', 'also_fake')
        shutdown_count = []

        # Mock domain
        mock_domain = mock.Mock(fakelibvirt.virDomain)
        return_infos = [(libvirt_guest.VIR_DOMAIN_RUNNING,) + info_tuple]
        return_shutdowns = [shutdown_count.append("shutdown")]
        retry_countdown = retry_interval
        for x in range(min(seconds_to_shutdown, timeout)):
            return_infos.append(
                (libvirt_guest.VIR_DOMAIN_RUNNING,) + info_tuple)
            if retry_countdown == 0:
                return_shutdowns.append(shutdown_count.append("shutdown"))
                retry_countdown = retry_interval
            else:
                retry_countdown -= 1

        if seconds_to_shutdown < timeout:
            return_infos.append(
                (libvirt_guest.VIR_DOMAIN_SHUTDOWN,) + info_tuple)

        mock_domain.info.side_effect = return_infos
        mock_domain.shutdown.side_effect = return_shutdowns

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        mock_get_domain.return_value = mock_domain
        result = drvr._clean_shutdown(instance, timeout, retry_interval)

        self.assertEqual(succeeds, result)
        self.assertEqual(shutdown_attempts, len(shutdown_count))

    def test_clean_shutdown_first_time(self):
        self._test_clean_shutdown(seconds_to_shutdown=2,
                                  timeout=5,
                                  retry_interval=3,
                                  shutdown_attempts=1,
                                  succeeds=True)

    def test_clean_shutdown_with_retry(self):
        self._test_clean_shutdown(seconds_to_shutdown=4,
                                  timeout=5,
                                  retry_interval=3,
                                  shutdown_attempts=2,
                                  succeeds=True)

    def test_clean_shutdown_failure(self):
        self._test_clean_shutdown(seconds_to_shutdown=6,
                                  timeout=5,
                                  retry_interval=3,
                                  shutdown_attempts=2,
                                  succeeds=False)

    def test_clean_shutdown_no_wait(self):
        self._test_clean_shutdown(seconds_to_shutdown=6,
                                  timeout=0,
                                  retry_interval=3,
                                  shutdown_attempts=1,
                                  succeeds=False)

    @mock.patch.object(FakeVirtDomain, 'attachDeviceFlags')
    @mock.patch.object(FakeVirtDomain, 'ID', return_value=1)
    @mock.patch.object(utils, 'get_image_from_system_metadata',
                       return_value={})
    def test_attach_direct_passthrough_ports(self,
            mock_get_image_metadata, mock_ID, mock_attachDevice):
        instance = objects.Instance(**self.test_instance)

        network_info = _fake_network_info(self)
        network_info[0]['vnic_type'] = network_model.VNIC_TYPE_DIRECT
        guest = libvirt_guest.Guest(FakeVirtDomain())
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        drvr._attach_direct_passthrough_ports(
            self.context, instance, guest, network_info)
        mock_get_image_metadata.assert_called_once_with(
            instance.system_metadata)
        self.assertTrue(mock_attachDevice.called)

    @mock.patch.object(FakeVirtDomain, 'attachDeviceFlags')
    @mock.patch.object(FakeVirtDomain, 'ID', return_value=1)
    @mock.patch.object(utils, 'get_image_from_system_metadata',
                       return_value={})
    def test_attach_direct_physical_passthrough_ports(self,
            mock_get_image_metadata, mock_ID, mock_attachDevice):
        instance = objects.Instance(**self.test_instance)

        network_info = _fake_network_info(self)
        network_info[0]['vnic_type'] = network_model.VNIC_TYPE_DIRECT_PHYSICAL
        guest = libvirt_guest.Guest(FakeVirtDomain())
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        drvr._attach_direct_passthrough_ports(
            self.context, instance, guest, network_info)
        mock_get_image_metadata.assert_called_once_with(
            instance.system_metadata)
        self.assertTrue(mock_attachDevice.called)

    @mock.patch.object(FakeVirtDomain, 'attachDeviceFlags')
    @mock.patch.object(FakeVirtDomain, 'ID', return_value=1)
    @mock.patch.object(utils, 'get_image_from_system_metadata',
                       return_value={})
    def test_attach_direct_passthrough_ports_with_info_cache(self,
            mock_get_image_metadata, mock_ID, mock_attachDevice):
        instance = objects.Instance(**self.test_instance)

        network_info = _fake_network_info(self)
        network_info[0]['vnic_type'] = network_model.VNIC_TYPE_DIRECT
        instance.info_cache = objects.InstanceInfoCache(
            network_info=network_info)
        guest = libvirt_guest.Guest(FakeVirtDomain())
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        drvr._attach_direct_passthrough_ports(
            self.context, instance, guest, None)
        mock_get_image_metadata.assert_called_once_with(
            instance.system_metadata)
        self.assertTrue(mock_attachDevice.called)

    @mock.patch.object(
        host.Host, 'has_min_version', new=mock.Mock(return_value=True)
    )
    def _test_detach_direct_passthrough_ports(
            self, vif_type, detach_device=True,
            vnic_type=network_model.VNIC_TYPE_DIRECT):
        instance = objects.Instance(**self.test_instance)

        expeted_pci_slot = "0000:00:00.0"
        network_info = _fake_network_info(self)
        network_info[0]['vnic_type'] = vnic_type
        # some more adjustments for the fake network_info so that
        # the correct get_config function will be executed (vif's
        # get_config_hw_veb - which is according to the real SRIOV vif)
        # and most importantly the pci_slot which is translated to
        # cfg.source_dev, then to PciDevice.address and sent to
        # _detach_pci_devices
        network_info[0]['profile'] = dict(pci_slot=expeted_pci_slot)
        network_info[0]['type'] = vif_type
        network_info[0]['details'] = dict(vlan="2145")
        instance.info_cache = objects.InstanceInfoCache(
            network_info=network_info)
        # fill the pci_devices of the instance so that
        # instance.get_instance_pci_devs will not return an empty list
        # which will eventually fail the assertion for detachDeviceFlags
        expected_pci_device_obj = (
            objects.PciDevice(
                address=expeted_pci_slot, request_id=None, compute_node_id=42
            )
        )
        instance.pci_devices = objects.PciDeviceList()
        instance.pci_devices.objects = [expected_pci_device_obj]

        domain = FakeVirtDomain(id=24601, name='Jean Valjean')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        guest = libvirt_guest.Guest(domain)

        with mock.patch.object(
            drvr, '_detach_pci_devices'
        ) as mock_detach_pci, mock.patch.object(
            drvr, 'detach_interface'
        ) as mock_detach_interface:
            drvr._detach_direct_passthrough_ports(
                self.context, instance, guest)
            if detach_device:
                mock_detach_pci.assert_called_once_with(
                    guest, [expected_pci_device_obj])
            else:
                mock_detach_interface.assert_called_once()

    def test_detach_direct_passthrough_ports_ovs_hw_offload(self):
        # Note: test detach_direct_passthrough_ports method for vif with config
        # LibvirtConfigGuestInterface
        self._test_detach_direct_passthrough_ports("ovs", detach_device=False)

    def test_detach_direct_passthrough_ports_sriov_nic_agent(self):
        # Note: test detach_direct_passthrough_ports method for vif with config
        # LibvirtConfigGuestInterface
        self._test_detach_direct_passthrough_ports(
            "hw_veb", detach_device=False
        )

    def test_detach_direct_physical_passthrough_ports_sriov_nic_agent(self):
        self._test_detach_direct_passthrough_ports(
            "hostdev_physical",
            vnic_type=network_model.VNIC_TYPE_DIRECT_PHYSICAL
        )

    def test_detach_direct_passthrough_ports_infiniband(self):
        # Note: test detach_direct_passthrough_ports method for vif with config
        # LibvirtConfigGuestHostdevPCI
        self._test_detach_direct_passthrough_ports("ib_hostdev")

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    @mock.patch.object(FakeVirtDomain, 'detachDeviceFlags')
    def test_detach_duplicate_mac_direct_passthrough_ports(
        self, mock_detachDeviceFlags, mock_has_min_version):
        instance = objects.Instance(**self.test_instance)

        network_info = _fake_network_info(self, 2)

        direct_physical = network_model.VNIC_TYPE_DIRECT_PHYSICAL
        for network_info_inst in network_info:
            network_info_inst['vnic_type'] = direct_physical
            network_info_inst['type'] = "hostdev_physical"
            network_info_inst['details'] = dict(vlan="2145")
            network_info_inst['address'] = "fa:16:3e:96:2a:48"

        network_info[0]['profile'] = dict(pci_slot="0000:00:00.0")
        network_info[1]['profile'] = dict(pci_slot="0000:00:00.1")

        instance.info_cache = objects.InstanceInfoCache(
            network_info=network_info)
        # fill the pci_devices of the instance so that
        # instance.get_instance_pci_devs will not return an empty list
        # which will eventually fail the assertion for detachDeviceFlags
        instance.pci_devices = objects.PciDeviceList()
        instance.pci_devices.objects = [
            objects.PciDevice(address='0000:00:00.0', request_id=None),
            objects.PciDevice(address='0000:00:00.1', request_id=None)
        ]

        domain = FakeVirtDomain()
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        guest = libvirt_guest.Guest(domain)

        drvr._detach_direct_passthrough_ports(self.context, instance, guest)

        expected_xml = [
            ('<hostdev mode="subsystem" type="pci" managed="yes">\n'
             '  <source>\n'
             '    <address bus="0x00" domain="0x0000" '
             'function="0x0" slot="0x00"/>\n'
             '  </source>\n'
             '</hostdev>\n'),
            ('<hostdev mode="subsystem" type="pci" managed="yes">\n'
             '  <source>\n'
             '    <address bus="0x00" domain="0x0000" '
             'function="0x1" slot="0x00"/>\n'
             '  </source>\n'
             '</hostdev>\n')
        ]

        self.assertEqual(2, mock_detachDeviceFlags.call_count)
        for index, call in enumerate(mock_detachDeviceFlags.mock_calls):
            self.assertXmlEqual(expected_xml[index], call.args[0])
            self.assertEqual({"flags": 1}, call.kwargs)

    def test_resume(self):
        dummyxml = ("<domain type='kvm'><name>instance-0000000a</name>"
                    "<devices>"
                    "<disk type='file'><driver name='qemu' type='raw'/>"
                    "<source file='/test/disk'/>"
                    "<target dev='vda' bus='virtio'/></disk>"
                    "<disk type='file'><driver name='qemu' type='qcow2'/>"
                    "<source file='/test/disk.local'/>"
                    "<target dev='vdb' bus='virtio'/></disk>"
                    "</devices></domain>")
        instance = objects.Instance(**self.test_instance)
        network_info = _fake_network_info(self)
        block_device_info = None
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        guest = libvirt_guest.Guest('fake_dom')
        with test.nested(
            mock.patch.object(drvr, '_get_existing_domain_xml',
                              return_value=dummyxml),
            mock.patch.object(drvr, '_create_guest_with_network',
                              return_value=guest),
            mock.patch.object(drvr, '_attach_pci_devices'),
            mock.patch('nova.objects.Instance.get_pci_devices',
                return_value='fake_pci_devs'),
            mock.patch.object(utils, 'get_image_from_system_metadata'),
            mock.patch.object(guest, 'sync_guest_time'),
            mock.patch.object(drvr, '_wait_for_running',
                              side_effect=loopingcall.LoopingCallDone()),
            mock.patch.object(drvr,
                              '_get_mdevs_from_guest_config',
                              return_value='fake_mdevs'),
            mock.patch.object(drvr, '_attach_mediated_devices'),
        ) as (_get_existing_domain_xml, _create_guest_with_network,
              _attach_pci_devices, get_instance_pci_devs, get_image_metadata,
              mock_sync_time, mock_wait,
              _get_mdevs_from_guest_config,
              _attach_mediated_devices):
            get_image_metadata.return_value = {'bar': 234}

            drvr.resume(self.context, instance, network_info,
                        block_device_info)
            _get_existing_domain_xml.assert_has_calls([mock.call(instance,
                                            network_info, block_device_info)])
            _create_guest_with_network.assert_has_calls([
                mock.call(
                    self.context, dummyxml, instance, network_info,
                    block_device_info, vifs_already_plugged=True,
                )])
            self.assertTrue(mock_sync_time.called)
            _attach_pci_devices.assert_has_calls([mock.call(guest,
                                                 'fake_pci_devs')])
            _attach_mediated_devices.assert_has_calls(
                [mock.call(guest, 'fake_mdevs')]
            )

    @mock.patch.object(host.Host, '_get_domain')
    @mock.patch.object(libvirt_driver.LibvirtDriver, 'get_info')
    @mock.patch.object(libvirt_driver.LibvirtDriver, 'delete_instance_files')
    @mock.patch.object(objects.Instance, 'save')
    def test_destroy_undefines(self, mock_save, mock_delete_instance_files,
                               mock_get_info, mock_get_domain):
        dom_mock = mock.MagicMock()
        dom_mock.undefineFlags.return_value = 1

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        mock_get_domain.return_value = dom_mock
        mock_get_info.return_value = hardware.InstanceInfo(
            state=power_state.SHUTDOWN, internal_id=-1)
        mock_delete_instance_files.return_value = None

        instance = objects.Instance(self.context, **self.test_instance)
        drvr.destroy(self.context, instance, [])
        mock_save.assert_called_once_with()

    @mock.patch.object(rbd_utils.RBDDriver, '_destroy_volume')
    @mock.patch.object(rbd_utils.RBDDriver, '_disconnect_from_rados')
    @mock.patch.object(rbd_utils.RBDDriver, '_connect_to_rados')
    @mock.patch.object(rbd_utils, 'rbd')
    @mock.patch.object(rbd_utils, 'rados')
    def test_cleanup_rbd(self, mock_rados, mock_rbd, mock_connect,
                         mock_disconnect, mock_destroy_volume):
        mock_connect.return_value = mock.MagicMock(), mock.MagicMock()
        instance = objects.Instance(**self.test_instance)
        all_volumes = [uuids.other_instance + '_disk',
                       uuids.other_instance + '_disk.swap',
                       instance.uuid + '_disk',
                       instance.uuid + '_disk.swap']

        mock_rbd.RBD.return_value.list.return_value = all_volumes
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._cleanup_rbd(instance)
        calls = [mock.call(mock.ANY, instance.uuid + '_disk'),
                 mock.call(mock.ANY, instance.uuid + '_disk.swap')]
        mock_destroy_volume.assert_has_calls(calls)
        self.assertEqual(2, mock_destroy_volume.call_count)

    @mock.patch.object(rbd_utils.RBDDriver, '_destroy_volume')
    @mock.patch.object(rbd_utils.RBDDriver, '_disconnect_from_rados')
    @mock.patch.object(rbd_utils.RBDDriver, '_connect_to_rados')
    @mock.patch.object(rbd_utils, 'rbd')
    @mock.patch.object(rbd_utils, 'rados')
    def test_cleanup_rbd_resize_reverting(self, mock_rados, mock_rbd,
                                          mock_connect, mock_disconnect,
                                          mock_destroy_volume):
        mock_connect.return_value = mock.MagicMock(), mock.MagicMock()
        instance = objects.Instance(**self.test_instance)
        instance.task_state = task_states.RESIZE_REVERTING
        all_volumes = [uuids.other_instance + '_disk',
                       uuids.other_instance + '_disk.local',
                       instance.uuid + '_disk',
                       instance.uuid + '_disk.local']
        mock_rbd.RBD.return_value.list.return_value = all_volumes
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._cleanup_rbd(instance)
        mock_destroy_volume.assert_called_once_with(
            mock.ANY, instance.uuid + '_disk.local')

    @mock.patch.object(objects.Instance, 'save')
    def test_destroy_undefines_no_undefine_flags(self, mock_save):
        mock_domain = mock.Mock(fakelibvirt.virDomain)
        mock_domain.undefineFlags.side_effect = fakelibvirt.libvirtError('Err')
        mock_domain.ID.return_value = 123

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._host._get_domain = mock.Mock(return_value=mock_domain)
        drvr.delete_instance_files = mock.Mock(return_value=None)
        drvr.get_info = mock.Mock(return_value=
            hardware.InstanceInfo(state=power_state.SHUTDOWN, internal_id=-1)
        )

        instance = objects.Instance(self.context, **self.test_instance)
        drvr.destroy(self.context, instance, [])

        self.assertEqual(2, mock_domain.ID.call_count)
        mock_domain.destroy.assert_called_once_with()
        mock_domain.undefineFlags.assert_called_once_with(1)
        mock_domain.undefine.assert_called_once_with()
        mock_save.assert_called_once_with()

    @mock.patch.object(objects.Instance, 'save')
    def test_destroy_undefines_no_attribute_with_managed_save(self, mock_save):
        mock_domain = mock.Mock(fakelibvirt.virDomain)
        mock_domain.undefineFlags.side_effect = AttributeError()
        mock_domain.ID.return_value = 123

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._host._get_domain = mock.Mock(return_value=mock_domain)
        drvr.delete_instance_files = mock.Mock(return_value=None)
        drvr.get_info = mock.Mock(return_value=
            hardware.InstanceInfo(state=power_state.SHUTDOWN, internal_id=-1)
        )

        instance = objects.Instance(self.context, **self.test_instance)
        drvr.destroy(self.context, instance, [])

        self.assertEqual(1, mock_domain.ID.call_count)
        mock_domain.destroy.assert_called_once_with()
        mock_domain.undefineFlags.assert_called_once_with(1)
        mock_domain.hasManagedSaveImage.assert_has_calls([mock.call(0)])
        mock_domain.managedSaveRemove.assert_called_once_with(0)
        mock_domain.undefine.assert_called_once_with()
        mock_save.assert_called_once_with()

    @mock.patch.object(objects.Instance, 'save')
    def test_destroy_undefines_no_attribute_no_managed_save(self, mock_save):
        mock_domain = mock.Mock(fakelibvirt.virDomain)
        mock_domain.undefineFlags.side_effect = AttributeError()
        mock_domain.hasManagedSaveImage.side_effect = AttributeError()
        mock_domain.ID.return_value = 123

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._host._get_domain = mock.Mock(return_value=mock_domain)
        drvr.delete_instance_files = mock.Mock(return_value=None)
        drvr.get_info = mock.Mock(return_value=
            hardware.InstanceInfo(state=power_state.SHUTDOWN, internal_id=-1)
        )

        instance = objects.Instance(self.context, **self.test_instance)
        drvr.destroy(self.context, instance, [])

        self.assertEqual(1, mock_domain.ID.call_count)
        mock_domain.destroy.assert_called_once_with()
        mock_domain.undefineFlags.assert_called_once_with(1)
        mock_domain.hasManagedSaveImage.assert_has_calls([mock.call(0)])
        mock_domain.undefine.assert_called_once_with()
        mock_save.assert_called_once_with()

    @mock.patch('nova.utils.get_image_from_system_metadata')
    @mock.patch.object(objects.Instance, 'save')
    def test_destroy_removes_nvram_host_support_uefi(self,
                                                     mock_save,
                                                     mock_image):
        mock_domain = mock.Mock(fakelibvirt.virDomain)
        mock_domain.ID.return_value = 123

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._host._get_domain = mock.Mock(return_value=mock_domain)
        drvr.delete_instance_files = mock.Mock(return_value=None)
        drvr.get_info = mock.Mock(return_value=hardware.InstanceInfo(
            state=power_state.SHUTDOWN, internal_id=-1))

        instance = objects.Instance(self.context, **self.test_instance)
        mock_image.return_value = {"properties": {
            "hw_firmware_type": "bios"}}
        drvr.destroy(self.context, instance, [])

        self.assertEqual(1, mock_domain.ID.call_count)
        mock_domain.destroy.assert_called_once_with()
        # NVRAM flag should not called only host support uefi
        mock_domain.undefineFlags.assert_called_once_with(
            fakelibvirt.VIR_DOMAIN_UNDEFINE_MANAGED_SAVE
        )
        mock_domain.undefine.assert_not_called()
        mock_save.assert_called_once_with()

    @mock.patch('nova.utils.get_image_from_system_metadata')
    @mock.patch.object(objects.Instance, 'save')
    def test_destroy_removes_nvram_host_and_guest_support_uefi(self,
                                                               mock_save,
                                                               mock_image):
        mock_domain = mock.Mock(fakelibvirt.virDomain)
        mock_domain.ID.return_value = 123

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr._host._get_domain = mock.Mock(return_value=mock_domain)
        drvr.delete_instance_files = mock.Mock(return_value=None)
        drvr.get_info = mock.Mock(return_value=hardware.InstanceInfo(
            state=power_state.SHUTDOWN, internal_id=-1))

        instance = objects.Instance(self.context, **self.test_instance)
        mock_image.return_value = {"properties": {
            "hw_firmware_type": "uefi"}}
        drvr.destroy(self.context, instance, [])

        self.assertEqual(1, mock_domain.ID.call_count)
        mock_domain.destroy.assert_called_once_with()
        # undefineFlags should now be called with 5 as uefi supported
        # by both host and guest
        mock_domain.undefineFlags.assert_has_calls([mock.call(
            fakelibvirt.VIR_DOMAIN_UNDEFINE_MANAGED_SAVE |
            fakelibvirt.VIR_DOMAIN_UNDEFINE_NVRAM
        )])
        mock_domain.undefine.assert_not_called()
        mock_save.assert_called_once_with()

    def test_destroy_timed_out(self):
        mock_virdomain = mock.Mock(autospec=fakelibvirt.virDomain)
        mock_virdomain.destroy.side_effect = fakelibvirt.libvirtError(
            'timed out')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.stub_out('nova.virt.libvirt.host.Host._get_domain',
                      lambda self, instance: mock_virdomain)
        self.stub_out('nova.tests.fixtures.libvirt.libvirtError.'
                      'get_error_code',
                      lambda self: fakelibvirt.VIR_ERR_OPERATION_TIMEOUT)
        instance = objects.Instance(**self.test_instance)
        self.assertRaises(exception.InstancePowerOffFailure,
                drvr.destroy, self.context, instance, [])

        mock_virdomain.ID.assert_called_once_with()
        mock_virdomain.destroy.assert_called_once_with()

    def test_private_destroy_not_found(self):
        ex = fakelibvirt.make_libvirtError(
                fakelibvirt.libvirtError,
                "No such domain",
                error_code=fakelibvirt.VIR_ERR_NO_DOMAIN)
        mock_virdomain = mock.Mock(autospec=fakelibvirt.virDomain)
        mock_virdomain.destroy.side_effect = ex
        mock_virdomain.info.side_effect = ex

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.stub_out('nova.virt.libvirt.host.Host._get_domain',
                      lambda self, instance: mock_virdomain)
        instance = objects.Instance(**self.test_instance)
        # NOTE(vish): verifies destroy doesn't raise if the instance disappears
        drvr._destroy(instance)

        mock_virdomain.ID.assert_called_once_with()
        mock_virdomain.destroy.assert_called_once_with()
        mock_virdomain.info.assert_called_once_with()
        mock_virdomain.UUIDString.assert_called_once_with()

    def test_private_destroy_lxc_processes_refused_to_die(self):
        self.flags(virt_type='lxc', group='libvirt')
        ex = fakelibvirt.make_libvirtError(
                fakelibvirt.libvirtError, "",
                error_message="internal error: Some processes refused to die",
                error_code=fakelibvirt.VIR_ERR_INTERNAL_ERROR)

        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        with mock.patch.object(conn._host, '_get_domain') as mock_get_domain, \
             mock.patch.object(conn, 'get_info') as mock_get_info:
            mock_domain = mock.MagicMock()
            mock_domain.ID.return_value = 1
            mock_get_domain.return_value = mock_domain
            mock_domain.destroy.side_effect = ex

            mock_info = mock.MagicMock()
            mock_info.internal_id = 1
            mock_info.state = power_state.SHUTDOWN
            mock_get_info.return_value = mock_info

            instance = objects.Instance(**self.test_instance)
            conn._destroy(instance)

    def test_private_destroy_processes_refused_to_die_still_raises(self):
        ex = fakelibvirt.make_libvirtError(
                fakelibvirt.libvirtError, "",
                error_message="internal error: Some processes refused to die",
                error_code=fakelibvirt.VIR_ERR_INTERNAL_ERROR)

        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        with mock.patch.object(conn._host, '_get_domain') as mock_get_domain:
            mock_domain = mock.MagicMock()
            mock_domain.ID.return_value = 1
            mock_get_domain.return_value = mock_domain
            mock_domain.destroy.side_effect = ex

            instance = objects.Instance(**self.test_instance)
            self.assertRaises(fakelibvirt.libvirtError, conn._destroy,
                              instance)

    @mock.patch.object(libvirt_driver.LOG, 'warning')
    def test_VIR_ERR_SYSTEM_ERROR_during_destroy(self, mock_warning):
        ex = fakelibvirt.make_libvirtError(
                fakelibvirt.libvirtError,
                ("Failed to terminate process 26425 with SIGKILL: "
                                     "Device or resource busy"),
                error_code=fakelibvirt.VIR_ERR_SYSTEM_ERROR,
                int1=errno.EBUSY)

        mock_guest = mock.Mock(libvirt_guest.Guest, id=1)
        mock_guest.poweroff = mock.Mock(side_effect=ex)

        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        with mock.patch.object(drvr._host, 'get_guest',
                               return_value=mock_guest):
            raised = self.assertRaises(fakelibvirt.libvirtError,
                                       drvr._destroy,
                                       instance)
            self.assertEqual(fakelibvirt.VIR_ERR_SYSTEM_ERROR,
                             raised.get_error_code())

        mock_warning.assert_called_once()
        mock_guest.poweroff.assert_called_once()

    @mock.patch.object(fakelibvirt.libvirtError, 'get_error_code')
    @mock.patch.object(host.Host, '_get_domain',
                       side_effect=exception.InstanceNotFound(
                           instance_id=uuids.instance))
    def test_undefine_domain_with_not_found_instance(self, mock_get_domain,
                                                     mock_get_error):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)

        # NOTE(wenjianhn): verifies undefine doesn't raise if the
        # instance disappears
        drvr._undefine_domain(instance)

        mock_get_domain.assert_called_once_with(instance)
        mock_get_error.assert_not_called()

    @mock.patch.object(host.Host, "get_guest")
    def test_undefine_domain_handles_libvirt_errors(self, mock_get):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        fake_guest = mock.Mock()
        mock_get.return_value = fake_guest

        unexpected = fakelibvirt.make_libvirtError(
                fakelibvirt.libvirtError, "Random", error_code=1)
        fake_guest.delete_configuration.side_effect = unexpected

        # ensure raise unexpected error code
        self.assertRaises(type(unexpected), drvr._undefine_domain, instance)

        ignored = fakelibvirt.make_libvirtError(
                fakelibvirt.libvirtError, "No such domain",
                error_code=fakelibvirt.VIR_ERR_NO_DOMAIN)
        fake_guest.delete_configuration.side_effect = ignored

        # ensure no raise for no such domain
        drvr._undefine_domain(instance)

    @mock.patch.object(host.Host, "list_instance_domains")
    @mock.patch.object(objects.BlockDeviceMappingList, "bdms_by_instance_uuid")
    @mock.patch.object(objects.InstanceList, "get_by_filters")
    def test_disk_over_committed_size_total(self, mock_get, mock_bdms,
                                            mock_list):
        # Ensure destroy calls managedSaveRemove for saved instance.
        class DiagFakeDomain(object):
            def __init__(self, name):
                self._name = name
                self._uuid = uuids.fake

            def ID(self):
                return 1

            def name(self):
                return self._name

            def UUIDString(self):
                return self._uuid

            def XMLDesc(self, flags):
                return "<domain><name>%s</name></domain>" % self._name

        instance_domains = [
            DiagFakeDomain("instance0000001"),
            DiagFakeDomain("instance0000002")]
        mock_list.return_value = instance_domains

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        fake_disks = {'instance0000001':
                      [{'type': 'qcow2', 'path': '/somepath/disk1',
                        'virt_disk_size': '10737418240',
                        'backing_file': '/somepath/disk1',
                        'disk_size': '83886080',
                        'over_committed_disk_size': '10653532160'}],
                      'instance0000002':
                      [{'type': 'raw', 'path': '/somepath/disk2',
                        'virt_disk_size': '10737418240',
                        'backing_file': '',
                        'disk_size': '5350000000',
                        'over_committed_disk_size': '5387418240'}]}

        def get_info(cfg, block_device_info):
            return fake_disks.get(cfg.name)

        instance_uuids = [dom.UUIDString() for dom in instance_domains]
        instances = [objects.Instance(
            uuid=instance_uuids[0],
            root_device_name='/dev/vda'),
            objects.Instance(
            uuid=instance_uuids[1],
            root_device_name='/dev/vdb')
        ]
        mock_get.return_value = instances

        with mock.patch.object(
                drvr, "_get_instance_disk_info_from_config") as mock_info:
            mock_info.side_effect = get_info

            result = drvr._get_disk_over_committed_size_total()
            self.assertEqual(result, 16040950400)
            mock_list.assert_called_once_with(only_running=False)
            self.assertEqual(2, mock_info.call_count)

        filters = {'uuid': instance_uuids}
        mock_get.assert_called_once_with(mock.ANY, filters, use_slave=True)
        mock_bdms.assert_called_with(mock.ANY, instance_uuids)

    @mock.patch.object(host.Host, "list_instance_domains")
    @mock.patch.object(objects.BlockDeviceMappingList, "bdms_by_instance_uuid")
    @mock.patch.object(objects.InstanceList, "get_by_filters")
    def test_disk_over_committed_size_total_eperm(self, mock_get, mock_bdms,
                                                  mock_list):
        # Ensure destroy calls managedSaveRemove for saved instance.
        class DiagFakeDomain(object):
            def __init__(self, name):
                self._name = name
                self._uuid = uuidutils.generate_uuid()

            def ID(self):
                return 1

            def name(self):
                return self._name

            def UUIDString(self):
                return self._uuid

            def XMLDesc(self, flags):
                return "<domain><name>%s</name></domain>" % self._name

        instance_domains = [
            DiagFakeDomain("instance0000001"),
            DiagFakeDomain("instance0000002"),
            DiagFakeDomain("instance0000003"),
            DiagFakeDomain("instance0000004"),
            DiagFakeDomain("instance0000005")]
        mock_list.return_value = instance_domains

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        fake_disks = {'instance0000001':
                      [{'type': 'qcow2', 'path': '/somepath/disk1',
                        'virt_disk_size': '10737418240',
                        'backing_file': '/somepath/disk1',
                        'disk_size': '83886080',
                        'over_committed_disk_size': '10653532160'}],
                      'instance0000002':
                      [{'type': 'raw', 'path': '/somepath/disk2',
                        'virt_disk_size': '0',
                        'backing_file': '/somepath/disk2',
                        'disk_size': '10737418240',
                        'over_committed_disk_size': '21474836480'}],
                      'instance0000003':
                      [{'type': 'raw', 'path': '/somepath/disk3',
                        'virt_disk_size': '0',
                        'backing_file': '/somepath/disk3',
                        'disk_size': '21474836480',
                        'over_committed_disk_size': '32212254720'}],
                      'instance0000004':
                      [{'type': 'raw', 'path': '/somepath/disk4',
                        'virt_disk_size': '0',
                        'backing_file': '/somepath/disk4',
                        'disk_size': '32212254720',
                        'over_committed_disk_size': '42949672960'}]}

        def side_effect(cfg, block_device_info):
            if cfg.name == 'instance0000001':
                self.assertEqual('/dev/vda',
                                 block_device_info['root_device_name'])
                raise OSError(errno.ENOENT, 'No such file or directory')
            if cfg.name == 'instance0000002':
                self.assertEqual('/dev/vdb',
                                 block_device_info['root_device_name'])
                raise OSError(errno.ESTALE, 'Stale NFS file handle')
            if cfg.name == 'instance0000003':
                self.assertEqual('/dev/vdc',
                                 block_device_info['root_device_name'])
                raise OSError(errno.EACCES, 'Permission denied')
            if cfg.name == 'instance0000004':
                self.assertEqual('/dev/vdd',
                                 block_device_info['root_device_name'])
                return fake_disks.get(cfg.name)
        get_disk_info = mock.Mock()
        get_disk_info.side_effect = side_effect
        drvr._get_instance_disk_info_from_config = get_disk_info

        instance_uuids = [dom.UUIDString() for dom in instance_domains]
        instances = [objects.Instance(
            uuid=instance_uuids[0],
            root_device_name='/dev/vda'),
            objects.Instance(
            uuid=instance_uuids[1],
            root_device_name='/dev/vdb'),
            objects.Instance(
            uuid=instance_uuids[2],
            root_device_name='/dev/vdc'),
            objects.Instance(
            uuid=instance_uuids[3],
            root_device_name='/dev/vdd'),
        ]
        mock_get.return_value = instances

        # NOTE(danms): We need to have found bdms for our instances,
        # but we don't really need them to be complete as we just need
        # to make it to our side_effect above. Exclude the last domain
        # to simulate the case where we have an instance with no BDMs.
        mock_bdms.return_value = {uuid: [] for uuid in instance_uuids
                                  if uuid != instance_domains[-1].UUIDString()}

        result = drvr._get_disk_over_committed_size_total()
        self.assertEqual(42949672960, result)
        mock_list.assert_called_once_with(only_running=False)
        self.assertEqual(5, get_disk_info.call_count)
        filters = {'uuid': instance_uuids}
        mock_get.assert_called_once_with(mock.ANY, filters, use_slave=True)
        mock_bdms.assert_called_with(mock.ANY, instance_uuids)

    @mock.patch.object(host.Host, "list_instance_domains")
    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       "_get_instance_disk_info_from_config",
                       side_effect=exception.VolumeBDMPathNotFound(path='bar'))
    @mock.patch.object(objects.BlockDeviceMappingList, "bdms_by_instance_uuid")
    @mock.patch.object(objects.InstanceList, "get_by_filters")
    def test_disk_over_committed_size_total_bdm_not_found(self,
                                                          mock_get,
                                                          mock_bdms,
                                                          mock_get_disk_info,
                                                          mock_list_domains):
        mock_dom = mock.Mock()
        mock_dom.XMLDesc.return_value = "<domain/>"
        mock_list_domains.return_value = [mock_dom]
        # Tests that we handle VolumeBDMPathNotFound gracefully.
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertEqual(0, drvr._get_disk_over_committed_size_total())

    @mock.patch('nova.virt.libvirt.host.Host.list_instance_domains')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.'
                '_get_instance_disk_info_from_config',
                side_effect=exception.DiskNotFound(location='/opt/stack/foo'))
    @mock.patch('nova.objects.BlockDeviceMappingList.bdms_by_instance_uuid',
                return_value=objects.BlockDeviceMappingList())
    @mock.patch('nova.objects.InstanceList.get_by_filters',
                return_value=objects.InstanceList(objects=[
                    objects.Instance(uuid=uuids.instance,
                                     vm_state=vm_states.ACTIVE,
                                     task_state=task_states.DELETING)]))
    def test_disk_over_committed_size_total_disk_not_found_ignore_task_state(
            self, mock_get, mock_bdms, mock_get_disk_info, mock_list_domains):
        """Tests that we handle DiskNotFound gracefully for an instance that
        is undergoing a task_state transition.
        """
        mock_dom = mock.Mock()
        mock_dom.XMLDesc.return_value = "<domain/>"
        mock_dom.UUIDString.return_value = uuids.instance
        mock_list_domains.return_value = [mock_dom]
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertEqual(0, drvr._get_disk_over_committed_size_total())

    @mock.patch('nova.virt.libvirt.host.Host.list_instance_domains')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.'
                '_get_instance_disk_info_from_config',
                side_effect=exception.DiskNotFound(location='/opt/stack/foo'))
    @mock.patch('nova.objects.BlockDeviceMappingList.bdms_by_instance_uuid',
                return_value=objects.BlockDeviceMappingList())
    @mock.patch('nova.objects.InstanceList.get_by_filters',
                return_value=objects.InstanceList(objects=[
                    objects.Instance(uuid=uuids.instance,
                                     task_state=None,
                                     vm_state=vm_states.RESIZED)]))
    def test_disk_over_committed_size_total_disk_not_found_ignore_vmstate(
            self, mock_get, mock_bdms, mock_get_disk_info, mock_list_domains):
        """Tests that we handle DiskNotFound gracefully for an instance that
        is resized but resize is not confirmed yet.
        """
        mock_dom = mock.Mock()
        mock_dom.XMLDesc.return_value = "<domain/>"
        mock_dom.UUIDString.return_value = uuids.instance
        mock_list_domains.return_value = [mock_dom]
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertEqual(0, drvr._get_disk_over_committed_size_total())

    @mock.patch('nova.virt.libvirt.storage.lvm.get_volume_size')
    @mock.patch('nova.virt.disk.api.get_disk_size',
                new_callable=mock.NonCallableMock)
    def test_get_instance_disk_info_from_config_block_devices(self,
            mock_disk_api, mock_get_volume_size):
        """Test that for block devices the actual and virtual sizes are
        reported as the same and that the disk_api is not used.
        """
        c = context.get_admin_context()
        instance = objects.Instance(root_device_name='/dev/vda',
                                    **self.test_instance)
        bdms = objects.BlockDeviceMappingList(objects=[
            fake_block_device.fake_bdm_object(c, {
                'device_name': '/dev/mapper/vg-lv',
                'source_type': 'image',
                'destination_type': 'local'
            }),

        ])
        block_device_info = driver.get_block_device_info(instance, bdms)

        config = vconfig.LibvirtConfigGuest()
        disk_config = vconfig.LibvirtConfigGuestDisk()
        disk_config.source_type = "block"
        disk_config.source_path = mock.sentinel.volume_path
        config.devices.append(disk_config)

        mock_get_volume_size.return_value = mock.sentinel.volume_size

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        disk_info = drvr._get_instance_disk_info_from_config(config,
                                                             block_device_info)

        mock_get_volume_size.assert_called_once_with(mock.sentinel.volume_path)
        self.assertEqual(disk_info[0]['disk_size'],
                         disk_info[0]['virt_disk_size'])

    @mock.patch('os.stat')
    @mock.patch('os.path.getsize')
    def test_get_instance_disk_info_from_config_raw_files(self,
            mock_getsize, mock_stat):
        """Test that over_committed_disk_size are calculated also for raw
        images_type, since disk can be sparsely allocated if
        [compute]/preallocate_images option is not set to space.
        """
        config = vconfig.LibvirtConfigGuest()
        disk_config = vconfig.LibvirtConfigGuestDisk()
        disk_config.source_type = "file"
        disk_config.source_path = "fake"
        disk_config.driver_format = "raw"
        config.devices.append(disk_config)

        disk_virtual_size = 53687091200
        disk_actual_size = 3687091200
        disk_actual_size_blocks = disk_actual_size / 512
        expected_over_committed_disk_size = disk_virtual_size -\
                                                disk_actual_size

        mock_getsize.return_value = disk_virtual_size
        mock_stat.return_value = mock.Mock(st_blocks=disk_actual_size_blocks)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        disk_info = drvr._get_instance_disk_info_from_config(config, None)

        self.assertEqual(expected_over_committed_disk_size,
                         disk_info[0]['over_committed_disk_size'])

    @mock.patch('nova.virt.disk.api.get_disk_info')
    @mock.patch('nova.virt.libvirt.utils.get_disk_backing_file',
                return_value='file')
    def test_get_instance_disk_info_from_config_negative(self,
            mock_backing_file, mock_disk_info):
        """Test that over_committed_disk_size  is set to 0 when
        disk_actual_size is greater than disk_virtual_size
        """
        config = vconfig.LibvirtConfigGuest()

        disk_config = vconfig.LibvirtConfigGuestDisk()
        disk_config.source_type = "file"
        disk_config.source_path = "fake"
        disk_config.driver_format = 'qcow2'
        config.devices.append(disk_config)

        disk_virtual_size = 53687091200
        disk_actual_size = 54000000000
        expected_over_committed_disk_size = 0

        mock_disk_info.return_value = mock.Mock(disk_size=disk_actual_size,
                                                virtual_size=disk_virtual_size)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        disk_info = drvr._get_instance_disk_info_from_config(config, None)
        self.assertEqual(expected_over_committed_disk_size,
                         disk_info[0]['over_committed_disk_size'])

    def test_cpu_info(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        def get_host_capabilities_stub(self):
            cpu = vconfig.LibvirtConfigCPU()
            cpu.model = "Opteron_G4"
            cpu.vendor = "AMD"
            cpu.arch = fields.Architecture.X86_64

            cpu.cells = 1
            cpu.cores = 2
            cpu.threads = 1
            cpu.sockets = 4

            cpu.maxphysaddr = vconfig.LibvirtConfigCPUMaxPhysAddr()
            cpu.maxphysaddr.mode = "emulate"
            cpu.maxphysaddr.bits = 42

            cpu.add_feature(vconfig.LibvirtConfigCPUFeature("extapic"))
            cpu.add_feature(vconfig.LibvirtConfigCPUFeature("3dnow"))

            caps = vconfig.LibvirtConfigCaps()
            caps.host = vconfig.LibvirtConfigCapsHost()
            caps.host.cpu = cpu

            guest = vconfig.LibvirtConfigCapsGuest()
            guest.ostype = fields.VMMode.HVM
            guest.arch = fields.Architecture.X86_64
            guest.domains['kvm'] = vconfig.LibvirtConfigCapsGuestDomain()
            caps.guests.append(guest)

            guest = vconfig.LibvirtConfigCapsGuest()
            guest.ostype = fields.VMMode.HVM
            guest.arch = fields.Architecture.I686
            guest.domains['kvm'] = vconfig.LibvirtConfigCapsGuestDomain()
            caps.guests.append(guest)

            return caps

        self.stub_out('nova.virt.libvirt.host.Host.get_capabilities',
                      get_host_capabilities_stub)

        want = {"vendor": "AMD",
                "features": set(["extapic", "3dnow"]),
                "maxphysaddr": {"mode": "emulate", "bits": 42},
                "model": "Opteron_G4",
                "arch": fields.Architecture.X86_64,
                "topology": {"cells": 1, "cores": 2, "threads": 1,
                             "sockets": 4}}
        got = drvr._get_cpu_info()
        self.assertEqual(want, got)

    @mock.patch.object(host.Host, 'list_pci_devices',
                       return_value=['pci_0000_04_00_3', 'pci_0000_04_10_7',
                                     'pci_0000_04_11_7'])
    def test_get_pci_passthrough_devices(self, mock_list):
        pci_utils.get_ifname_by_pci_address.return_value = 'ens1'

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        devs = ['pci_0000_04_00_3', 'pci_0000_04_10_7', 'pci_0000_04_11_7']
        node_devs = {}
        for dev_name in devs:
            node_devs[dev_name] = (
                fakelibvirt.NodeDevice(
                    drvr._get_connection(), xml=_fake_NodeDevXml[dev_name]))
            for child in _fake_NodeDevXml_children[dev_name]:
                node_devs[child] = (
                    fakelibvirt.NodeDevice(
                        drvr._get_connection(),
                        xml=_fake_NodeDevXml[child]))

        self.useFixture(fixtures.MockPatchObject(
            drvr._host, 'list_all_devices',
            return_value=node_devs.values()))

        self.stub_out(
            'nova.virt.libvirt.host.Host.device_lookup_by_name',
            lambda self, name: node_devs.get(name))

        actjson = drvr._get_pci_passthrough_devices()
        mock_list.assert_not_called()
        expectvfs = [
            {
                "dev_id": "pci_0000_04_00_3",
                "address": "0000:04:00.3",
                "product_id": '1521',
                "vendor_id": '8086',
                "dev_type": fields.PciDeviceType.SRIOV_PF,
                "phys_function": None,
                "numa_node": None,
                # value defined in the LibvirtFixture
                "mac_address": "52:54:00:1e:59:c6",
            },
            {
                "dev_id": "pci_0000_04_10_7",
                "domain": 0,
                "address": "0000:04:10.7",
                "product_id": '1520',
                "vendor_id": '8086',
                "numa_node": None,
                "dev_type": fields.PciDeviceType.SRIOV_VF,
                "phys_function": [('0x0000', '0x04', '0x00', '0x3')],
                "parent_addr": "0000:04:00.3",
                "parent_ifname": "ens1",
            },
            {
                "dev_id": "pci_0000_04_11_7",
                "domain": 0,
                "address": "0000:04:11.7",
                "product_id": '1520',
                "vendor_id": '8086',
                "numa_node": 0,
                "dev_type": fields.PciDeviceType.SRIOV_VF,
                "phys_function": [('0x0000', '0x04', '0x00', '0x3')],
                "parent_addr": "0000:04:00.3",
                "parent_ifname": "ens1",
            }
        ]

        actualvfs = jsonutils.loads(actjson)
        for dev in range(len(actualvfs)):
            for key in actualvfs[dev].keys():
                if key not in ['phys_function', 'virt_functions',
                               'label', 'capabilities']:
                    self.assertEqual(expectvfs[dev][key], actualvfs[dev][key])

        # The first call for every VF is to determine parent_ifname and
        # the second call to determine the MAC address.
        pci_utils.get_ifname_by_pci_address.assert_has_calls([
            mock.call('0000:04:10.7', pf_interface=True),
            mock.call('0000:04:11.7', pf_interface=True),
        ])

    @mock.patch.object(host.Host, 'has_min_version',
                       new=mock.Mock(return_value=True))
    def _test_get_host_numa_topology(self):
        nodes = 1
        sockets = 4
        cores = 1
        threads = 2
        total_cores = nodes * sockets * cores * threads

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology(
            cpu_nodes=nodes, cpu_sockets=sockets, cpu_cores=cores,
            cpu_threads=threads)
        for i, cell in enumerate(caps.host.topology.cells):
            cell.mempages = fakelibvirt.create_mempages(
                [(4, 1024 * i), (2048, i)])

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        with test.nested(
                mock.patch.object(host.Host, 'get_capabilities',
                                  return_value=caps),
                mock.patch.object(host.Host, 'get_online_cpus',
                                  return_value=set(range(total_cores))),
                ):
            got_topo = drvr._get_host_numa_topology()

            # there should be varying amounts of mempages for each cell
            self.assertEqual(4, got_topo.cells[0].mempages[0].size_kb)
            self.assertEqual(0, got_topo.cells[0].mempages[0].total)
            self.assertEqual(2048, got_topo.cells[0].mempages[1].size_kb)
            self.assertEqual(0, got_topo.cells[0].mempages[1].total)
            self.assertEqual(4, got_topo.cells[1].mempages[0].size_kb)
            self.assertEqual(1024, got_topo.cells[1].mempages[0].total)
            self.assertEqual(2048, got_topo.cells[1].mempages[1].size_kb)
            self.assertEqual(1, got_topo.cells[1].mempages[1].total)

            # none of the topologies should have pinned CPUs yet
            self.assertEqual(set([]), got_topo.cells[0].pinned_cpus)
            self.assertEqual(set([]), got_topo.cells[1].pinned_cpus)
            self.assertEqual(set([]), got_topo.cells[2].pinned_cpus)
            self.assertEqual(set([]), got_topo.cells[3].pinned_cpus)

            # Each cell should be in its own socket
            self.assertEqual(0, got_topo.cells[0].socket)
            self.assertEqual(1, got_topo.cells[1].socket)
            self.assertEqual(2, got_topo.cells[2].socket)
            self.assertEqual(3, got_topo.cells[3].socket)

            # return to caller for further checks
            return got_topo

    def test_get_host_numa_topology(self):
        """Check that the host NUMA topology is generated correctly for a
        fairly complex configuration.
        """
        self.flags(cpu_shared_set='0-1', cpu_dedicated_set='2-6',
                   group='compute')
        self.flags(vcpu_pin_set=None)
        self.flags(physnets=['foo', 'bar', 'baz'], group='neutron')
        # we need to call the below again to ensure the updated 'physnets'
        # value is read and the new groups created
        nova.conf.neutron.register_dynamic_opts(CONF)
        self.flags(numa_nodes=[0, 2], group='neutron_tunnel')
        self.flags(numa_nodes=[1], group='neutron_physnet_foo')
        self.flags(numa_nodes=[3], group='neutron_physnet_bar')
        self.flags(numa_nodes=[1, 2, 3], group='neutron_physnet_baz')

        got_topo = self._test_get_host_numa_topology()

        # only cores 0 and 1 are configured as shared using the
        # 'cpu_shared_set' config option
        self.assertEqual(set([0, 1]), got_topo.cells[0].cpuset)
        self.assertEqual(set(), got_topo.cells[0].pcpuset)
        self.assertEqual(set(), got_topo.cells[1].cpuset)
        self.assertEqual(set([2, 3]), got_topo.cells[1].pcpuset)
        self.assertEqual(set(), got_topo.cells[2].cpuset)
        self.assertEqual(set([4, 5]), got_topo.cells[2].pcpuset)
        self.assertEqual(set(), got_topo.cells[3].cpuset)
        self.assertEqual(set([6]), got_topo.cells[3].pcpuset)

        # all cells except the last one should have siblings
        self.assertEqual([set([0, 1])], got_topo.cells[0].siblings)
        self.assertEqual([set([2, 3])], got_topo.cells[1].siblings)
        self.assertEqual([set([4, 5])], got_topo.cells[2].siblings)
        self.assertEqual([set([6])], got_topo.cells[3].siblings)

        self.assertEqual(set(),
                         got_topo.cells[0].network_metadata.physnets)
        self.assertEqual(set(['foo', 'baz']),
                         got_topo.cells[1].network_metadata.physnets)
        self.assertEqual(set(['baz']),
                         got_topo.cells[2].network_metadata.physnets)
        self.assertEqual(set(['bar', 'baz']),
                         got_topo.cells[3].network_metadata.physnets)

        self.assertTrue(got_topo.cells[0].network_metadata.tunneled)
        self.assertFalse(got_topo.cells[1].network_metadata.tunneled)
        self.assertTrue(got_topo.cells[2].network_metadata.tunneled)
        self.assertFalse(got_topo.cells[3].network_metadata.tunneled)

    def test_get_host_numa_topology__vcpu_pin_set_fallback(self):
        """Check that the host NUMA topology will fall back to using
        'vcpu_pin_set' if 'cpu_dedicated_set' is not defined.
        """
        self.flags(cpu_shared_set='0-1', cpu_dedicated_set=None,
                   group='compute')
        self.flags(vcpu_pin_set='2-6')

        got_topo = self._test_get_host_numa_topology()

        # cores 0 and 1 are configured as shared using the 'cpu_shared_set'
        # config option but because 'vcpu_pin_set' is configured this
        # configuration is ignored. All the cores listed in 'vcpu_pin_set' are
        # dual reported for upgrade reasons
        self.assertEqual(set(), got_topo.cells[0].cpuset)
        self.assertEqual(set(), got_topo.cells[0].pcpuset)
        self.assertEqual(set([2, 3]), got_topo.cells[1].cpuset)
        self.assertEqual(set([2, 3]), got_topo.cells[1].pcpuset)
        self.assertEqual(set([4, 5]), got_topo.cells[2].cpuset)
        self.assertEqual(set([4, 5]), got_topo.cells[2].pcpuset)
        self.assertEqual(set([6]), got_topo.cells[3].cpuset)
        self.assertEqual(set([6]), got_topo.cells[3].pcpuset)

        # all cells except the first and last one should have siblings
        self.assertEqual([], got_topo.cells[0].siblings)
        self.assertEqual([set([2, 3])], got_topo.cells[1].siblings)
        self.assertEqual([set([4, 5])], got_topo.cells[2].siblings)
        self.assertEqual([set([6])], got_topo.cells[3].siblings)

    def test_get_host_numa_topology__no_cpu_configuration(self):
        """Check that the host NUMA topology will fall back to using
        'vcpu_pin_set' if 'cpu_dedicated_set' is not defined.
        """
        self.flags(cpu_shared_set=None, cpu_dedicated_set=None,
                   group='compute')
        self.flags(vcpu_pin_set=None)

        got_topo = self._test_get_host_numa_topology()

        # there's no CPU configuration so every core is dual-reported for
        # upgrade reasons
        self.assertEqual(set([0, 1]), got_topo.cells[0].cpuset)
        self.assertEqual(set([0, 1]), got_topo.cells[0].pcpuset)
        self.assertEqual(set([2, 3]), got_topo.cells[1].cpuset)
        self.assertEqual(set([2, 3]), got_topo.cells[1].pcpuset)
        self.assertEqual(set([4, 5]), got_topo.cells[2].cpuset)
        self.assertEqual(set([4, 5]), got_topo.cells[2].pcpuset)
        self.assertEqual(set([6, 7]), got_topo.cells[3].cpuset)
        self.assertEqual(set([6, 7]), got_topo.cells[3].pcpuset)

        # all cells should have siblings
        self.assertEqual([set([0, 1])], got_topo.cells[0].siblings)
        self.assertEqual([set([2, 3])], got_topo.cells[1].siblings)
        self.assertEqual([set([4, 5])], got_topo.cells[2].siblings)
        self.assertEqual([set([6, 7])], got_topo.cells[3].siblings)

    def test_get_host_numa_topology__only_shared_cpus(self):
        """Check that the host NUMA topology does not use 'cpu_shared_set' if
        'cpu_dedicated_set' is not defined.
        """
        self.flags(cpu_shared_set='0-6', cpu_dedicated_set=None,
                   group='compute')
        self.flags(vcpu_pin_set=None)

        got_topo = self._test_get_host_numa_topology()

        # only cores 0 and 1 are configured as shared using the
        # 'cpu_shared_set' config option, but the rest are dual reported
        # for upgrade reasons
        self.assertEqual(set([0, 1]), got_topo.cells[0].cpuset)
        self.assertEqual(set(), got_topo.cells[0].pcpuset)
        self.assertEqual(set([2, 3]), got_topo.cells[1].cpuset)
        self.assertEqual(set([]), got_topo.cells[1].pcpuset)
        self.assertEqual(set([4, 5]), got_topo.cells[2].cpuset)
        self.assertEqual(set([]), got_topo.cells[2].pcpuset)
        self.assertEqual(set([6]), got_topo.cells[3].cpuset)
        self.assertEqual(set([]), got_topo.cells[3].pcpuset)

        # all cells except the lasat one should have siblings
        self.assertEqual([set([0, 1])], got_topo.cells[0].siblings)
        self.assertEqual([set([2, 3])], got_topo.cells[1].siblings)
        self.assertEqual([set([4, 5])], got_topo.cells[2].siblings)
        self.assertEqual([set([6])], got_topo.cells[3].siblings)

    def test_get_host_numa_topology_empty(self):
        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = None

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        with test.nested(
            mock.patch.object(host.Host, 'has_min_version', return_value=True),
            mock.patch.object(host.Host, "get_capabilities",
                              return_value=caps)
        ) as (has_min_version, get_caps):
            self.assertIsNone(drvr._get_host_numa_topology())
        self.assertEqual(2, get_caps.call_count)

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    def test_get_host_numa_topology_missing_network_metadata(self,
            mock_version):
        self.flags(cpu_shared_set='0-5', cpu_dedicated_set=None,
                   group='compute')
        self.flags(physnets=['bar'], group='neutron')
        # we need to call the below again to ensure the updated 'physnets'
        # value is read and the new groups created
        nova.conf.neutron.register_dynamic_opts(CONF)

        # we explicitly avoid registering a '[neutron_physnets_bar] numa_nodes'
        # option here

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology()

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        with test.nested(
                mock.patch.object(host.Host, "get_capabilities",
                                  return_value=caps),
                mock.patch.object(host.Host, 'get_online_cpus',
                                  return_value=set(range(6)))
                ):
            self.assertRaisesRegex(
                exception.InvalidNetworkNUMAAffinity,
                "Invalid NUMA network affinity configured: the physnet 'bar' "
                "was listed in '\\[neutron\\] physnets' but no corresponding "
                "'\\[neutron_physnet_bar\\] numa_nodes' option was defined.",
                drvr._get_host_numa_topology)

    @mock.patch.object(host.Host, 'has_min_version', return_value=True)
    def _test_get_host_numa_topology_invalid_network_affinity(self,
            group_name, mock_version):
        self.flags(cpu_shared_set='0-5', cpu_dedicated_set=None,
                   group='compute')
        self.flags(physnets=['foo', 'bar'], group='neutron')
        # we need to call the below again to ensure the updated 'physnets'
        # value is read and the new groups created
        nova.conf.neutron.register_dynamic_opts(CONF)

        # set defaults...
        for group_ in ['neutron_physnet_foo', 'neutron_physnet_bar',
                       'neutron_tunnel']:
            self.flags(numa_nodes=[0], group=group_)

        # but override them for the error case
        self.flags(numa_nodes=[4], group=group_name)

        caps = vconfig.LibvirtConfigCaps()
        caps.host = vconfig.LibvirtConfigCapsHost()
        caps.host.cpu = vconfig.LibvirtConfigCPU()
        caps.host.cpu.arch = fields.Architecture.X86_64
        caps.host.topology = fakelibvirt.NUMATopology()

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        with test.nested(
                mock.patch.object(host.Host, "get_capabilities",
                                  return_value=caps),
                mock.patch.object(host.Host, 'get_online_cpus',
                                  return_value=set(range(6)))
                ):
            self.assertRaisesRegex(
                exception.InvalidNetworkNUMAAffinity,
                r'node 4 for \w+ \w+ is not present',
                drvr._get_host_numa_topology)

    def test_get_host_numa_topology_invalid_physical_network_affinity(self):
        """Ensure errors are raised for non-existent NUMA nodes.

        If a physical network is affined to a non-existent NUMA node, an
        exception should be raised. Prove this to be the case.
        """
        self._test_get_host_numa_topology_invalid_network_affinity(
            'neutron_physnet_bar')

    def test_get_host_numa_topology_invalid_tunnel_network_affinity(self):
        """Ensure errors are raised for non-existent NUMA nodes.

        If a tunneled network is affined to a non-existent NUMA node, an
        exception should be raised. Prove this to be the case.
        """
        self._test_get_host_numa_topology_invalid_network_affinity(
            'neutron_tunnel')

    def test_diagnostic_vcpus_exception(self):
        xml = """
                <domain type='kvm'>
                    <devices>
                        <disk type='file'>
                            <source file='filename'/>
                            <target dev='vda' bus='virtio'/>
                        </disk>
                        <disk type='block'>
                            <source dev='/path/to/dev/1'/>
                            <target dev='vdb' bus='virtio'/>
                        </disk>
                        <interface type='network'>
                            <mac address='52:54:00:a4:38:38'/>
                            <source network='default'/>
                            <target dev='vnet0'/>
                        </interface>
                    </devices>
                </domain>
            """

        class DiagFakeDomain(FakeVirtDomain):

            def __init__(self):
                super(DiagFakeDomain, self).__init__(fake_xml=xml)

            def vcpus(self):
                raise fakelibvirt.libvirtError('vcpus missing')

            def blockStats(self, path):
                return (169, 688640, 0, 0, 1)

            def interfaceStats(self, path):
                return (4408, 82, 0, 0, 0, 0, 0, 0)

            def memoryStats(self):
                return {'actual': 220160, 'rss': 200164}

            def maxMemory(self):
                return 280160

        self.stub_out('nova.virt.libvirt.host.Host._get_domain',
                      lambda self, instance: DiagFakeDomain())

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        actual = drvr.get_diagnostics(instance)
        expect = {'vda_read': 688640,
                  'vda_read_req': 169,
                  'vda_write': 0,
                  'vda_write_req': 0,
                  'vda_errors': 1,
                  'vdb_read': 688640,
                  'vdb_read_req': 169,
                  'vdb_write': 0,
                  'vdb_write_req': 0,
                  'vdb_errors': 1,
                  'memory': 280160,
                  'memory-actual': 220160,
                  'memory-rss': 200164,
                  'vnet0_rx': 4408,
                  'vnet0_rx_drop': 0,
                  'vnet0_rx_errors': 0,
                  'vnet0_rx_packets': 82,
                  'vnet0_tx': 0,
                  'vnet0_tx_drop': 0,
                  'vnet0_tx_errors': 0,
                  'vnet0_tx_packets': 0,
                  }
        self.assertEqual(actual, expect)

        lt = datetime.datetime(2012, 11, 22, 12, 00, 00)
        diags_time = datetime.datetime(2012, 11, 22, 12, 00, 10)
        self.useFixture(utils_fixture.TimeFixture(diags_time))

        instance.launched_at = lt
        actual = drvr.get_instance_diagnostics(instance)
        expected = fake_diagnostics_object(with_disks=True, with_nic=True)
        self.assertDiagnosticsEqual(expected, actual)

    def test_diagnostic_blockstats_exception(self):
        xml = """
                <domain type='kvm'>
                    <devices>
                        <disk type='file'>
                            <source file='filename'/>
                            <target dev='vda' bus='virtio'/>
                        </disk>
                        <disk type='block'>
                            <source dev='/path/to/dev/1'/>
                            <target dev='vdb' bus='virtio'/>
                        </disk>
                        <interface type='network'>
                            <mac address='52:54:00:a4:38:38'/>
                            <source network='default'/>
                            <target dev='vnet0'/>
                        </interface>
                    </devices>
                </domain>
            """

        class DiagFakeDomain(FakeVirtDomain):

            def __init__(self):
                super(DiagFakeDomain, self).__init__(fake_xml=xml)

            def vcpus(self):
                return ([(0, 1, 15340000000, 0),
                         (1, 1, 1640000000, 0),
                         (2, 1, 3040000000, 0),
                         (3, 1, 1420000000, 0)],
                        [(True, False),
                         (True, False),
                         (True, False),
                         (True, False)])

            def blockStats(self, path):
                raise fakelibvirt.libvirtError('blockStats missing')

            def interfaceStats(self, path):
                return (4408, 82, 0, 0, 0, 0, 0, 0)

            def memoryStats(self):
                return {'actual': 220160, 'rss': 200164}

            def maxMemory(self):
                return 280160

        self.stub_out('nova.virt.libvirt.host.Host._get_domain',
                      lambda self, instance: DiagFakeDomain())

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        actual = drvr.get_diagnostics(instance)
        expect = {'cpu0_time': 15340000000,
                  'cpu1_time': 1640000000,
                  'cpu2_time': 3040000000,
                  'cpu3_time': 1420000000,
                  'memory': 280160,
                  'memory-actual': 220160,
                  'memory-rss': 200164,
                  'vnet0_rx': 4408,
                  'vnet0_rx_drop': 0,
                  'vnet0_rx_errors': 0,
                  'vnet0_rx_packets': 82,
                  'vnet0_tx': 0,
                  'vnet0_tx_drop': 0,
                  'vnet0_tx_errors': 0,
                  'vnet0_tx_packets': 0,
                  }
        self.assertEqual(actual, expect)

        lt = datetime.datetime(2012, 11, 22, 12, 00, 00)
        diags_time = datetime.datetime(2012, 11, 22, 12, 00, 10)
        self.useFixture(utils_fixture.TimeFixture(diags_time))

        instance.launched_at = lt
        actual = drvr.get_instance_diagnostics(instance)
        expected = fake_diagnostics_object(with_cpus=True, with_nic=True)
        self.assertDiagnosticsEqual(expected, actual)

    def test_diagnostic_interfacestats_exception(self):
        xml = """
                <domain type='kvm'>
                    <devices>
                        <disk type='file'>
                            <source file='filename'/>
                            <target dev='vda' bus='virtio'/>
                        </disk>
                        <disk type='block'>
                            <source dev='/path/to/dev/1'/>
                            <target dev='vdb' bus='virtio'/>
                        </disk>
                        <interface type='network'>
                            <mac address='52:54:00:a4:38:38'/>
                            <source network='default'/>
                            <target dev='vnet0'/>
                        </interface>
                    </devices>
                </domain>
            """

        class DiagFakeDomain(FakeVirtDomain):

            def __init__(self):
                super(DiagFakeDomain, self).__init__(fake_xml=xml)

            def vcpus(self):
                return ([(0, 1, 15340000000, 0),
                         (1, 1, 1640000000, 0),
                         (2, 1, 3040000000, 0),
                         (3, 1, 1420000000, 0)],
                        [(True, False),
                         (True, False),
                         (True, False),
                         (True, False)])

            def blockStats(self, path):
                return (169, 688640, 0, 0, 1)

            def interfaceStats(self, path):
                raise fakelibvirt.libvirtError('interfaceStat missing')

            def memoryStats(self):
                return {'actual': 220160, 'rss': 200164}

            def maxMemory(self):
                return 280160

        self.stub_out('nova.virt.libvirt.host.Host._get_domain',
                      lambda self, instance: DiagFakeDomain())

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        actual = drvr.get_diagnostics(instance)
        expect = {'cpu0_time': 15340000000,
                  'cpu1_time': 1640000000,
                  'cpu2_time': 3040000000,
                  'cpu3_time': 1420000000,
                  'vda_read': 688640,
                  'vda_read_req': 169,
                  'vda_write': 0,
                  'vda_write_req': 0,
                  'vda_errors': 1,
                  'vdb_read': 688640,
                  'vdb_read_req': 169,
                  'vdb_write': 0,
                  'vdb_write_req': 0,
                  'vdb_errors': 1,
                  'memory': 280160,
                  'memory-actual': 220160,
                  'memory-rss': 200164,
                  }
        self.assertEqual(actual, expect)

        lt = datetime.datetime(2012, 11, 22, 12, 00, 00)
        diags_time = datetime.datetime(2012, 11, 22, 12, 00, 10)
        self.useFixture(utils_fixture.TimeFixture(diags_time))

        instance.launched_at = lt
        actual = drvr.get_instance_diagnostics(instance)
        expected = fake_diagnostics_object(with_cpus=True, with_disks=True)
        self.assertDiagnosticsEqual(expected, actual)

    def test_diagnostic_memorystats_exception(self):
        xml = """
                <domain type='kvm'>
                    <devices>
                        <disk type='file'>
                            <source file='filename'/>
                            <target dev='vda' bus='virtio'/>
                        </disk>
                        <disk type='block'>
                            <source dev='/path/to/dev/1'/>
                            <target dev='vdb' bus='virtio'/>
                        </disk>
                        <interface type='network'>
                            <mac address='52:54:00:a4:38:38'/>
                            <source network='default'/>
                            <target dev='vnet0'/>
                        </interface>
                    </devices>
                </domain>
            """

        class DiagFakeDomain(FakeVirtDomain):

            def __init__(self):
                super(DiagFakeDomain, self).__init__(fake_xml=xml)

            def vcpus(self):
                return ([(0, 1, 15340000000, 0),
                         (1, 1, 1640000000, 0),
                         (2, 1, 3040000000, 0),
                         (3, 1, 1420000000, 0)],
                        [(True, False),
                         (True, False),
                         (True, False),
                         (True, False)])

            def blockStats(self, path):
                return (169, 688640, 0, 0, 1)

            def interfaceStats(self, path):
                return (4408, 82, 0, 0, 0, 0, 0, 0)

            def memoryStats(self):
                raise fakelibvirt.libvirtError('memoryStats missing')

            def maxMemory(self):
                return 280160

        self.stub_out('nova.virt.libvirt.host.Host._get_domain',
                      lambda self, instance: DiagFakeDomain())

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        actual = drvr.get_diagnostics(instance)
        expect = {'cpu0_time': 15340000000,
                  'cpu1_time': 1640000000,
                  'cpu2_time': 3040000000,
                  'cpu3_time': 1420000000,
                  'vda_read': 688640,
                  'vda_read_req': 169,
                  'vda_write': 0,
                  'vda_write_req': 0,
                  'vda_errors': 1,
                  'vdb_read': 688640,
                  'vdb_read_req': 169,
                  'vdb_write': 0,
                  'vdb_write_req': 0,
                  'vdb_errors': 1,
                  'memory': 280160,
                  'vnet0_rx': 4408,
                  'vnet0_rx_drop': 0,
                  'vnet0_rx_errors': 0,
                  'vnet0_rx_packets': 82,
                  'vnet0_tx': 0,
                  'vnet0_tx_drop': 0,
                  'vnet0_tx_errors': 0,
                  'vnet0_tx_packets': 0,
                  }
        self.assertEqual(actual, expect)

        lt = datetime.datetime(2012, 11, 22, 12, 00, 00)
        diags_time = datetime.datetime(2012, 11, 22, 12, 00, 10)
        self.useFixture(utils_fixture.TimeFixture(diags_time))

        instance.launched_at = lt
        actual = drvr.get_instance_diagnostics(instance)
        expected = fake_diagnostics_object(with_cpus=True, with_disks=True,
                                           with_nic=True)
        self.assertDiagnosticsEqual(expected, actual)

    def test_diagnostic_full(self):
        xml = """
                <domain type='kvm'>
                    <devices>
                        <disk type='file'>
                            <source file='filename'/>
                            <target dev='vda' bus='virtio'/>
                        </disk>
                        <disk type='block'>
                            <source dev='/path/to/dev/1'/>
                            <target dev='vdb' bus='virtio'/>
                        </disk>
                        <interface type='network'>
                            <mac address='52:54:00:a4:38:38'/>
                            <source network='default'/>
                            <target dev='vnet0'/>
                        </interface>
                    </devices>
                </domain>
            """

        class DiagFakeDomain(FakeVirtDomain):

            def __init__(self):
                super(DiagFakeDomain, self).__init__(fake_xml=xml)

            def vcpus(self):
                return ([(0, 1, 15340000000, 0),
                         (1, 1, 1640000000, 0),
                         (2, 1, 3040000000, 0),
                         (3, 1, 1420000000, 0)],
                        [(True, False),
                         (True, False),
                         (True, False),
                         (True, False)])

            def blockStats(self, path):
                return (169, 688640, 0, 0, 1)

            def interfaceStats(self, path):
                return (4408, 82, 0, 0, 0, 0, 0, 0)

            def memoryStats(self):
                return {'actual': 220160, 'rss': 200164}

            def maxMemory(self):
                return 280160

        self.stub_out('nova.virt.libvirt.host.Host._get_domain',
                      lambda self, instance: DiagFakeDomain())

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        actual = drvr.get_diagnostics(instance)
        expect = {'cpu0_time': 15340000000,
                  'cpu1_time': 1640000000,
                  'cpu2_time': 3040000000,
                  'cpu3_time': 1420000000,
                  'vda_read': 688640,
                  'vda_read_req': 169,
                  'vda_write': 0,
                  'vda_write_req': 0,
                  'vda_errors': 1,
                  'vdb_read': 688640,
                  'vdb_read_req': 169,
                  'vdb_write': 0,
                  'vdb_write_req': 0,
                  'vdb_errors': 1,
                  'memory': 280160,
                  'memory-actual': 220160,
                  'memory-rss': 200164,
                  'vnet0_rx': 4408,
                  'vnet0_rx_drop': 0,
                  'vnet0_rx_errors': 0,
                  'vnet0_rx_packets': 82,
                  'vnet0_tx': 0,
                  'vnet0_tx_drop': 0,
                  'vnet0_tx_errors': 0,
                  'vnet0_tx_packets': 0,
                  }
        self.assertEqual(actual, expect)

        lt = datetime.datetime(2012, 11, 22, 12, 00, 00)
        diags_time = datetime.datetime(2012, 11, 22, 12, 00, 10)
        self.useFixture(utils_fixture.TimeFixture(diags_time))

        instance.launched_at = lt
        actual = drvr.get_instance_diagnostics(instance)
        expected = fake_diagnostics_object(with_cpus=True, with_disks=True,
                                           with_nic=True)
        self.assertDiagnosticsEqual(expected, actual)

    @mock.patch.object(host.Host, '_get_domain')
    def test_diagnostic_full_with_multiple_interfaces(self, mock_get_domain):
        xml = """
                <domain type='kvm'>
                    <devices>
                        <disk type='file'>
                            <source file='filename'/>
                            <target dev='vda' bus='virtio'/>
                        </disk>
                        <disk type='block'>
                            <source dev='/path/to/dev/1'/>
                            <target dev='vdb' bus='virtio'/>
                        </disk>
                        <interface type='network'>
                            <mac address='52:54:00:a4:38:38'/>
                            <source network='default'/>
                            <target dev='vnet0'/>
                        </interface>
                        <interface type="bridge">
                            <mac address="53:55:00:a5:39:39"/>
                            <model type="virtio"/>
                            <target dev="br0"/>
                        </interface>
                        <interface type='hostdev' managed='yes'>
                            <mac address="54:56:00:a6:40:40"/>
                            <driver name='vfio'/>
                        </interface>
                    </devices>
                </domain>
            """

        class DiagFakeDomain(FakeVirtDomain):

            def __init__(self):
                super(DiagFakeDomain, self).__init__(fake_xml=xml)

            def vcpus(self):
                return ([(0, 1, 15340000000, 0),
                         (1, 1, 1640000000, 0),
                         (2, 1, 3040000000, 0),
                         (3, 1, 1420000000, 0)],
                        [(True, False),
                         (True, False),
                         (True, False),
                         (True, False)])

            def blockStats(self, path):
                return (169, 688640, 0, 0, 1)

            def interfaceStats(self, path):
                return (4408, 82, 0, 0, 0, 0, 0, 0)

            def memoryStats(self):
                return {'actual': 220160, 'rss': 200164}

            def maxMemory(self):
                return 280160

        def fake_get_domain(self):
            return DiagFakeDomain()

        mock_get_domain.side_effect = fake_get_domain

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        actual = drvr.get_diagnostics(instance)
        expect = {'cpu0_time': 15340000000,
                  'cpu1_time': 1640000000,
                  'cpu2_time': 3040000000,
                  'cpu3_time': 1420000000,
                  'vda_read': 688640,
                  'vda_read_req': 169,
                  'vda_write': 0,
                  'vda_write_req': 0,
                  'vda_errors': 1,
                  'vdb_read': 688640,
                  'vdb_read_req': 169,
                  'vdb_write': 0,
                  'vdb_write_req': 0,
                  'vdb_errors': 1,
                  'memory': 280160,
                  'memory-actual': 220160,
                  'memory-rss': 200164,
                  'vnet0_rx': 4408,
                  'vnet0_rx_drop': 0,
                  'vnet0_rx_errors': 0,
                  'vnet0_rx_packets': 82,
                  'vnet0_tx': 0,
                  'vnet0_tx_drop': 0,
                  'vnet0_tx_errors': 0,
                  'vnet0_tx_packets': 0,
                  'br0_rx': 4408,
                  'br0_rx_drop': 0,
                  'br0_rx_errors': 0,
                  'br0_rx_packets': 82,
                  'br0_tx': 0,
                  'br0_tx_drop': 0,
                  'br0_tx_errors': 0,
                  'br0_tx_packets': 0,
                  }
        self.assertEqual(actual, expect)

        lt = datetime.datetime(2012, 11, 22, 12, 00, 00)
        diags_time = datetime.datetime(2012, 11, 22, 12, 00, 10)
        self.useFixture(utils_fixture.TimeFixture(diags_time))

        instance.launched_at = lt
        actual = drvr.get_instance_diagnostics(instance)
        expected = fake_diagnostics_object(with_cpus=True, with_disks=True,
                                           with_nic=True)
        expected.add_nic(mac_address='53:55:00:a5:39:39',
                         rx_drop=0,
                         rx_errors=0,
                         rx_octets=4408,
                         rx_packets=82,
                         tx_drop=0,
                         tx_errors=0,
                         tx_octets=0,
                         tx_packets=0)

        expected.add_nic(mac_address='54:56:00:a6:40:40')

        self.assertDiagnosticsEqual(expected, actual)

    @mock.patch.object(host.Host, "list_instance_domains")
    def test_failing_vcpu_count(self, mock_list):
        """Domain can fail to return the vcpu description in case it's
        just starting up or shutting down. Make sure None is handled
        gracefully.
        """

        class DiagFakeDomain(object):
            def __init__(self, vcpus):
                self._vcpus = vcpus

            def vcpus(self):
                if self._vcpus is None:
                    raise fakelibvirt.libvirtError("fake-error")
                else:
                    return ([[1, 2, 3, 4]] * self._vcpus, [True] * self._vcpus)

            def ID(self):
                return 1

            def name(self):
                return "instance000001"

            def UUIDString(self):
                return "19479fee-07a5-49bb-9138-d3738280d63c"

        mock_list.return_value = [
            DiagFakeDomain(None), DiagFakeDomain(5)]

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        self.assertEqual(6, drvr._get_vcpu_used())
        mock_list.assert_called_with(only_running=True)

    def test_get_instance_capabilities(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        def get_host_capabilities_stub(self):
            caps = vconfig.LibvirtConfigCaps()

            guest = vconfig.LibvirtConfigCapsGuest()
            guest.ostype = 'hvm'
            guest.arch = fields.Architecture.X86_64
            guest.domains['kvm'] = vconfig.LibvirtConfigCapsGuestDomain()
            guest.domains['qemu'] = vconfig.LibvirtConfigCapsGuestDomain()
            caps.guests.append(guest)

            guest = vconfig.LibvirtConfigCapsGuest()
            guest.ostype = 'hvm'
            guest.arch = fields.Architecture.I686
            guest.domains['kvm'] = vconfig.LibvirtConfigCapsGuestDomain()
            caps.guests.append(guest)

            # Include one that is not known to nova to make sure it
            # does not trip us up.
            guest = vconfig.LibvirtConfigCapsGuest()
            guest.ostype = 'hvm'
            guest.arch = 'itanic'
            guest.domains['kvm'] = vconfig.LibvirtConfigCapsGuestDomain()
            caps.guests.append(guest)

            return caps

        self.stub_out('nova.virt.libvirt.host.Host.get_capabilities',
                      get_host_capabilities_stub)

        want = [(fields.Architecture.X86_64, 'kvm', 'hvm'),
                (fields.Architecture.X86_64, 'qemu', 'hvm'),
                (fields.Architecture.I686, 'kvm', 'hvm')]
        got = drvr._get_instance_capabilities()
        self.assertEqual(want, got)

    def test_set_cache_mode(self):
        self.flags(disk_cachemodes=['file=directsync'], group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        fake_conf = FakeConfigGuestDisk()

        fake_conf.source_type = 'file'
        drvr._set_cache_mode(fake_conf)
        self.assertEqual(fake_conf.driver_cache, 'directsync')

    def test_set_cache_mode_shareable(self):
        """Tests that when conf.shareable is True, the configuration is
        ignored and the driver_cache is forced to 'none'.
        """
        self.flags(disk_cachemodes=['block=writeback'], group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        fake_conf = FakeConfigGuestDisk()
        fake_conf.shareable = True
        fake_conf.source_type = 'block'
        drvr._set_cache_mode(fake_conf)
        self.assertEqual('none', fake_conf.driver_cache)

    def _make_fake_conf(self, cache=None):
        if cache:
            self.flags(disk_cachemodes=['block=' + cache], group='libvirt')
        else:
            self.flags(group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        fake_conf = FakeConfigGuestDisk()
        fake_conf.source_type = 'block'
        fake_conf.driver_io = 'native'
        drvr._set_cache_mode(fake_conf)
        return fake_conf

    def test_set_cache_mode_driver_io_writeback(self):
        """Tests that when conf.driver_io is 'native' and driver_cache is
        'writeback', then conf.driver_io is forced to 'threads'
        """
        fake_conf = self._make_fake_conf('writeback')
        self.assertEqual('writeback', fake_conf.driver_cache)
        self.assertEqual('threads', fake_conf.driver_io)

    def test_set_cache_mode_driver_io_writethrough(self):
        """Tests that when conf.driver_io is 'native' and driver_cache is
        'writethrough', then conf.driver_io is forced to 'threads'
        """
        fake_conf = self._make_fake_conf('writethrough')
        self.assertEqual('writethrough', fake_conf.driver_cache)
        self.assertEqual('threads', fake_conf.driver_io)

    def test_set_cache_mode_driver_io_unsafe(self):
        """Tests that when conf.driver_io is 'native' and driver_cache is
        'unsafe', then conf.driver_io is forced to 'threads'
        """
        fake_conf = self._make_fake_conf('unsafe')
        self.assertEqual('unsafe', fake_conf.driver_cache)
        self.assertEqual('threads', fake_conf.driver_io)

    def test_without_set_cache_mode_driver_io(self):
        """Tests that when conf.driver_io is 'native' and driver_cache is
        not set(this is default settings), then conf.driver_io is kept as
        'native'
        """
        fake_conf = self._make_fake_conf()
        self.assertIsNone(fake_conf.driver_cache)
        self.assertEqual('native', fake_conf.driver_io)

    def test_set_cache_mode_invalid_mode(self):
        self.flags(disk_cachemodes=['file=FAKE'], group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        fake_conf = FakeConfigGuestDisk()

        fake_conf.source_type = 'file'
        drvr._set_cache_mode(fake_conf)
        self.assertIsNone(fake_conf.driver_cache)

    def test_set_cache_mode_invalid_object(self):
        self.flags(disk_cachemodes=['file=directsync'], group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        fake_conf = FakeConfigGuest()

        fake_conf.driver_cache = 'fake'
        drvr._set_cache_mode(fake_conf)
        self.assertEqual(fake_conf.driver_cache, 'fake')

    @mock.patch('os.unlink')
    @mock.patch.object(os.path, 'exists')
    def _test_shared_storage_detection(self, is_same,
                                       mock_exists, mock_unlink):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr.get_host_ip_addr = mock.MagicMock(return_value='bar')
        mock_exists.return_value = is_same
        with test.nested(
            mock.patch.object(drvr._remotefs, 'create_file'),
            mock.patch.object(drvr._remotefs, 'remove_file')
        ) as (mock_rem_fs_create, mock_rem_fs_remove):
            result = drvr._is_path_shared_with('host', '/path')
        mock_rem_fs_create.assert_any_call('host', mock.ANY)
        create_args, create_kwargs = mock_rem_fs_create.call_args
        self.assertTrue(create_args[1].startswith('/path'))
        if is_same:
            mock_unlink.assert_called_once_with(mock.ANY)
        else:
            mock_rem_fs_remove.assert_called_with('host', mock.ANY)
            remove_args, remove_kwargs = mock_rem_fs_remove.call_args
            self.assertTrue(remove_args[1].startswith('/path'))
        return result

    def test_shared_storage_detection_same_host(self):
        self.assertTrue(self._test_shared_storage_detection(True))

    def test_shared_storage_detection_different_host(self):
        self.assertFalse(self._test_shared_storage_detection(False))

    @mock.patch.object(os, 'unlink')
    @mock.patch.object(os.path, 'exists')
    @mock.patch('oslo_concurrency.processutils.execute')
    @mock.patch.object(libvirt_driver.LibvirtDriver, 'get_host_ip_addr',
                       return_value='foo')
    def test_shared_storage_detection_easy(self, mock_get, mock_exec,
                                           mock_exists, mock_unlink):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertTrue(drvr._is_path_shared_with('foo', '/path'))
        mock_get.assert_called_once_with()
        mock_exec.assert_not_called()
        mock_exists.assert_not_called()
        mock_unlink.assert_not_called()

    @mock.patch.object(os, 'unlink')
    @mock.patch.object(os.path, 'exists')
    @mock.patch('oslo_concurrency.processutils.execute')
    def test_shared_storage_detection_same_host_migration_inbound_addr(
        self, mock_exec, mock_exists, mock_unlink
    ):
        CONF.set_default(
            "migration_inbound_addr", "source-hostname", group="libvirt")
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertTrue(drvr._is_path_shared_with('source-hostname', '/path'))
        mock_exec.assert_not_called()
        mock_exists.assert_not_called()
        mock_unlink.assert_not_called()

    @mock.patch.object(os.path, 'exists')
    def test_shared_storage_detection_migration_inbound_addr(
        self, mock_exists
    ):
        CONF.set_default(
            "migration_inbound_addr", "source-hostname", group="libvirt")
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        mock_exists.return_value = False
        with test.nested(
            mock.patch.object(drvr._remotefs, 'create_file'),
            mock.patch.object(drvr._remotefs, 'remove_file')
        ) as (mock_rem_fs_create, mock_rem_fs_remove):
            self.assertFalse(
                drvr._is_path_shared_with('dest-hostname', '/path'))

        mock_rem_fs_create.assert_any_call('dest-hostname', mock.ANY)
        create_args, create_kwargs = mock_rem_fs_create.call_args
        self.assertTrue(create_args[1].startswith('/path'))

        mock_rem_fs_remove.assert_called_with('dest-hostname', mock.ANY)
        remove_args, remove_kwargs = mock_rem_fs_remove.call_args
        self.assertTrue(remove_args[1].startswith('/path'))

    def test_store_pid_remove_pid(self):
        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        popen = mock.Mock(pid=3)
        drvr.job_tracker.add_job(instance, popen.pid)
        self.assertIn(3, drvr.job_tracker.jobs[instance.uuid])
        drvr.job_tracker.remove_job(instance, popen.pid)
        self.assertNotIn(instance.uuid, drvr.job_tracker.jobs)

    @mock.patch('nova.virt.libvirt.host.Host._get_domain')
    def test_get_domain_info_with_more_return(self, mock_get_domain):
        instance = objects.Instance(**self.test_instance)
        dom_mock = mock.MagicMock()
        dom_mock.info.return_value = [
            1, 2048, 737, 8, 12345, 888888
        ]
        dom_mock.ID.return_value = mock.sentinel.instance_id
        mock_get_domain.return_value = dom_mock
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        info = drvr.get_info(instance)
        self.assertEqual(1, info.state)
        self.assertEqual(mock.sentinel.instance_id, info.internal_id)
        dom_mock.info.assert_called_once_with()
        dom_mock.ID.assert_called_once_with()
        mock_get_domain.assert_called_once_with(instance)

    @mock.patch.object(hardware, 'get_vtpm_constraint')
    @mock.patch.object(libvirt_guest.Guest, 'create')
    def test_create_guest__with_callback(
        self, mock_guest_create, mock_get_vtpm,
    ):
        """Check that callback function is called if provided."""
        instance = objects.Instance(**self.test_instance)
        xml = '<xml>'
        callback = mock.Mock()

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._create_guest(
            self.context, xml, instance, post_xml_callback=callback)

        mock_get_vtpm.assert_not_called()
        mock_guest_create.assert_called_once_with(xml, drvr._host)
        mock_guest_create.return_value.launch.assert_called_once_with(
            pause=False)
        callback.assert_called_once()

    @mock.patch.object(hardware, 'get_vtpm_constraint')
    @mock.patch.object(libvirt_guest.Guest, 'create')
    def test_create_guest__no_launch(self, mock_guest_create, mock_get_vtpm):
        """Check that guest is not started unless requested."""
        instance = objects.Instance(**self.test_instance)
        xml = '<xml>'

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._create_guest(
            self.context, xml, instance, power_on=False, pause=False)

        mock_get_vtpm.assert_not_called()
        mock_guest_create.assert_called_once_with(xml, drvr._host)
        mock_guest_create.return_value.launch.assert_not_called()

    @mock.patch('nova.crypto.ensure_vtpm_secret')
    @mock.patch(
        'nova.objects.Instance.image_meta',
        new_callable=mock.PropertyMock)
    @mock.patch.object(hardware, 'get_vtpm_constraint')
    @mock.patch.object(libvirt_guest.Guest, 'create')
    def test_create_guest__with_vtpm_support_but_no_request(
        self, mock_guest_create, mock_get_vtpm, mock_image_meta, mock_secret,
    ):
        """Check that vTPM is not created unless requested by the guest."""
        self.flags(swtpm_enabled=True, group='libvirt')

        instance = objects.Instance(**self.test_instance)
        instance = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({})
        xml = '<xml>'

        mock_image_meta.return_value = image_meta
        mock_get_vtpm.return_value = None

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._create_guest(
            self.context, xml, instance)

        # we should have queried the instance's flavor, image for vTPM stuff...
        mock_get_vtpm.assert_called_once_with(instance.flavor, image_meta)
        mock_guest_create.assert_called_once_with(xml, drvr._host)
        mock_guest_create.return_value.launch.assert_called_once()
        # ...but we should not have created the secret because it wasn't needed
        mock_secret.assert_not_called()

    @mock.patch('nova.virt.libvirt.host.Host')
    @mock.patch('nova.crypto.ensure_vtpm_secret')
    @mock.patch(
        'nova.objects.Instance.image_meta',
        new_callable=mock.PropertyMock)
    @mock.patch.object(hardware, 'get_vtpm_constraint')
    @mock.patch.object(libvirt_guest.Guest, 'create')
    def test_create_guest__with_vtpm(
        self, mock_guest_create, mock_get_vtpm, mock_image_meta, mock_secret,
        mock_host,
    ):
        """Check that vTPM secret is created and cleaned up again after."""
        self.flags(swtpm_enabled=True, group='libvirt')

        instance = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({})
        xml = '<xml>'

        mock_image_meta.return_value = image_meta
        mock_secret.return_value = (uuids.fake_secret, 'passphrase')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._create_guest(self.context, xml, instance)

        # we should have queried the instance's flavor, image for vTPM stuff
        mock_get_vtpm.assert_called_once_with(instance.flavor, image_meta)
        mock_guest_create.assert_called_once_with(xml, drvr._host)
        mock_guest_create.return_value.launch.assert_called_once()

        # we should also have created the secret...
        drvr._host.create_secret.assert_called_once_with(
            'vtpm', instance.uuid, password='passphrase',
            uuid=uuids.fake_secret)
        # ...and undefined it after
        drvr._host.create_secret.return_value.undefine.assert_called_once()

    @mock.patch('nova.virt.libvirt.host.Host')
    @mock.patch('nova.crypto.ensure_vtpm_secret')
    @mock.patch(
        'nova.objects.Instance.image_meta',
        new_callable=mock.PropertyMock)
    @mock.patch.object(hardware, 'get_vtpm_constraint')
    @mock.patch.object(libvirt_guest.Guest, 'create')
    def test_create_guest__with_vtpm_error(
        self, mock_guest_create, mock_get_vtpm, mock_image_meta, mock_secret,
        mock_host,
    ):
        """Check that vTPM secret is always cleaned up even if there's an
        error.
        """
        self.flags(swtpm_enabled=True, group='libvirt')

        instance = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict({})
        xml = '<xml>'

        mock_guest_create.side_effect = ValueError('foo')
        mock_image_meta.return_value = image_meta
        mock_secret.return_value = (uuids.fake_secret, 'passphrase')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        self.assertRaises(
            ValueError, drvr._create_guest, self.context, xml, instance)

        # we should have queried the instance's flavor, image for vTPM stuff
        mock_get_vtpm.assert_called_once_with(instance.flavor, image_meta)
        mock_guest_create.assert_called_once_with(xml, drvr._host)

        # we should also have created the secret...
        drvr._host.create_secret.assert_called_once_with(
            'vtpm', instance.uuid, password='passphrase',
            uuid=uuids.fake_secret)
        # ...and undefined it after, despite the error
        drvr._host.create_secret.return_value.undefine.assert_called_once()

    @mock.patch('nova.virt.disk.api.clean_lxc_namespace')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.get_info')
    @mock.patch('nova.virt.disk.api.setup_container')
    @mock.patch('oslo_utils.fileutils.ensure_tree')
    @mock.patch('nova.virt.libvirt.utils.get_instance_path')
    def test_create_guest_with_network__lxc(
        self, mock_get_inst_path, mock_ensure_tree, mock_setup_container,
        mock_get_info, mock_clean,
    ):
        self.flags(virt_type='lxc', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        mock_instance = mock.MagicMock()
        inst_sys_meta = dict()
        mock_instance.system_metadata = inst_sys_meta
        mock_get_inst_path.return_value = '/tmp/'
        mock_image_backend = mock.MagicMock()
        drvr.image_backend = mock_image_backend
        mock_image = mock.MagicMock()
        mock_image.path = '/tmp/test.img'
        drvr.image_backend.by_name.return_value = mock_image
        mock_setup_container.return_value = '/dev/nbd0'
        mock_get_info.return_value = hardware.InstanceInfo(
            state=power_state.RUNNING)

        with test.nested(
                mock.patch.object(drvr, '_is_booted_from_volume',
                                  return_value=False),
                mock.patch.object(drvr, '_create_guest'),
                mock.patch.object(drvr, 'plug_vifs')):
            drvr._create_guest_with_network(self.context, 'xml',
                                            mock_instance, [], None)

        self.assertEqual('/dev/nbd0', inst_sys_meta['rootfs_device_name'])
        self.assertFalse(mock_instance.called)
        mock_get_inst_path.assert_has_calls([mock.call(mock_instance)])
        mock_ensure_tree.assert_has_calls([mock.call('/tmp/rootfs')])
        drvr.image_backend.by_name.assert_has_calls([mock.call(mock_instance,
                                                    'disk')])

        setup_container_call = mock.call(
            mock_image.get_model(),
            container_dir='/tmp/rootfs')
        mock_setup_container.assert_has_calls([setup_container_call])
        mock_get_info.assert_has_calls([mock.call(mock_instance)])
        mock_clean.assert_has_calls([mock.call(container_dir='/tmp/rootfs')])

    @mock.patch('nova.virt.disk.api.clean_lxc_namespace')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.get_info')
    @mock.patch('nova.virt.libvirt.utils.chown_for_id_maps')
    @mock.patch('nova.virt.disk.api.setup_container')
    @mock.patch('oslo_utils.fileutils.ensure_tree')
    @mock.patch('nova.virt.libvirt.utils.get_instance_path')
    def test_create_guest_with_network__lxc_id_maps(
        self, mock_get_inst_path, mock_ensure_tree, mock_setup_container,
        mock_chown, mock_get_info, mock_clean,
    ):
        self.flags(virt_type='lxc', uid_maps=["0:1000:100"],
                   gid_maps=["0:1000:100"], group='libvirt')

        def chown_side_effect(path, id_maps):
            self.assertEqual('/tmp/rootfs', path)
            self.assertIsInstance(id_maps[0], vconfig.LibvirtConfigGuestUIDMap)
            self.assertEqual(0, id_maps[0].start)
            self.assertEqual(1000, id_maps[0].target)
            self.assertEqual(100, id_maps[0].count)
            self.assertIsInstance(id_maps[1], vconfig.LibvirtConfigGuestGIDMap)
            self.assertEqual(0, id_maps[1].start)
            self.assertEqual(1000, id_maps[1].target)
            self.assertEqual(100, id_maps[1].count)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        mock_instance = mock.MagicMock()
        inst_sys_meta = dict()
        mock_instance.system_metadata = inst_sys_meta
        mock_get_inst_path.return_value = '/tmp/'
        mock_image_backend = mock.MagicMock()
        drvr.image_backend = mock_image_backend
        mock_image = mock.MagicMock()
        mock_image.path = '/tmp/test.img'
        drvr.image_backend.by_name.return_value = mock_image
        mock_setup_container.return_value = '/dev/nbd0'
        mock_chown.side_effect = chown_side_effect
        mock_get_info.return_value = hardware.InstanceInfo(
            state=power_state.RUNNING)

        with test.nested(
            mock.patch.object(drvr, '_is_booted_from_volume',
                              return_value=False),
            mock.patch.object(drvr, '_create_guest'),
            mock.patch.object(drvr, 'plug_vifs'),
        ) as (
            mock_is_booted_from_volume, mock_create_guest, mock_plug_vifs,
        ):
            drvr._create_guest_with_network(self.context, 'xml',
                                            mock_instance, [], None)

        self.assertEqual('/dev/nbd0', inst_sys_meta['rootfs_device_name'])
        self.assertFalse(mock_instance.called)
        mock_get_inst_path.assert_has_calls([mock.call(mock_instance)])
        mock_ensure_tree.assert_has_calls([mock.call('/tmp/rootfs')])
        drvr.image_backend.by_name.assert_has_calls([mock.call(mock_instance,
                                                    'disk')])

        setup_container_call = mock.call(
            mock_image.get_model(),
            container_dir='/tmp/rootfs')
        mock_setup_container.assert_has_calls([setup_container_call])
        mock_get_info.assert_has_calls([mock.call(mock_instance)])
        mock_clean.assert_has_calls([mock.call(container_dir='/tmp/rootfs')])

    @mock.patch('nova.virt.disk.api.teardown_container')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.get_info')
    @mock.patch('nova.virt.disk.api.setup_container')
    @mock.patch('oslo_utils.fileutils.ensure_tree')
    @mock.patch('nova.virt.libvirt.utils.get_instance_path')
    def test_create_guest_with_network__lxc_not_running(
        self, mock_get_inst_path, mock_ensure_tree, mock_setup_container,
        mock_get_info, mock_teardown,
    ):
        self.flags(virt_type='lxc', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        mock_instance = mock.MagicMock()
        inst_sys_meta = dict()
        mock_instance.system_metadata = inst_sys_meta
        mock_get_inst_path.return_value = '/tmp/'
        mock_image_backend = mock.MagicMock()
        drvr.image_backend = mock_image_backend
        mock_image = mock.MagicMock()
        mock_image.path = '/tmp/test.img'
        drvr.image_backend.by_name.return_value = mock_image
        mock_setup_container.return_value = '/dev/nbd0'
        mock_get_info.return_value = hardware.InstanceInfo(
            state=power_state.SHUTDOWN)

        with test.nested(
                mock.patch.object(drvr, '_is_booted_from_volume',
                                  return_value=False),
                mock.patch.object(drvr, '_create_guest'),
                mock.patch.object(drvr, 'plug_vifs')):
            drvr._create_guest_with_network(self.context, 'xml',
                                            mock_instance, [], None)

        self.assertEqual('/dev/nbd0', inst_sys_meta['rootfs_device_name'])
        self.assertFalse(mock_instance.called)
        mock_get_inst_path.assert_has_calls([mock.call(mock_instance)])
        mock_ensure_tree.assert_has_calls([mock.call('/tmp/rootfs')])
        drvr.image_backend.by_name.assert_has_calls([mock.call(mock_instance,
                                                    'disk')])

        setup_container_call = mock.call(
            mock_image.get_model(),
            container_dir='/tmp/rootfs')
        mock_setup_container.assert_has_calls([setup_container_call])
        mock_get_info.assert_has_calls([mock.call(mock_instance)])
        teardown_call = mock.call(container_dir='/tmp/rootfs')
        mock_teardown.assert_has_calls([teardown_call])

    def test_get_vnc_console(self):
        instance = objects.Instance(**self.test_instance)
        dummyxml = ("<domain type='kvm'><name>instance-0000000a</name>"
                    "<devices>"
                    "<graphics type='vnc' port='5900'/>"
                    "</devices></domain>")

        vdmock = mock.create_autospec(fakelibvirt.virDomain)
        vdmock.XMLDesc.return_value = dummyxml

        def fake_lookup(_uuid):
            if _uuid == instance['uuid']:
                return vdmock
        self.create_fake_libvirt_mock(lookupByUUIDString=fake_lookup)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        vnc_dict = drvr.get_vnc_console(self.context, instance)
        self.assertEqual(vnc_dict.port, '5900')
        vdmock.XMLDesc.assert_called_once_with(flags=0)

    def test_get_vnc_console_unavailable(self):
        instance = objects.Instance(**self.test_instance)
        dummyxml = ("<domain type='kvm'><name>instance-0000000a</name>"
                    "<devices></devices></domain>")

        vdmock = mock.create_autospec(fakelibvirt.virDomain)
        vdmock.XMLDesc.return_value = dummyxml

        def fake_lookup(_uuid):
            if _uuid == instance['uuid']:
                return vdmock
        self.create_fake_libvirt_mock(lookupByUUIDString=fake_lookup)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertRaises(exception.ConsoleTypeUnavailable,
                          drvr.get_vnc_console, self.context, instance)
        vdmock.XMLDesc.assert_called_once_with(flags=0)

    def test_get_spice_console(self):
        instance = objects.Instance(**self.test_instance)
        dummyxml = ("<domain type='kvm'><name>instance-0000000a</name>"
                    "<devices>"
                    "<graphics type='spice' port='5950'/>"
                    "</devices></domain>")

        vdmock = mock.create_autospec(fakelibvirt.virDomain)
        vdmock.XMLDesc.return_value = dummyxml

        def fake_lookup(_uuid):
            if _uuid == instance['uuid']:
                return vdmock
        self.create_fake_libvirt_mock(lookupByUUIDString=fake_lookup)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        spice_dict = drvr.get_spice_console(self.context, instance)
        self.assertEqual(spice_dict.port, '5950')
        vdmock.XMLDesc.assert_called_once_with(flags=0)

    def test_get_spice_console_unavailable(self):
        instance = objects.Instance(**self.test_instance)
        dummyxml = ("<domain type='kvm'><name>instance-0000000a</name>"
                    "<devices></devices></domain>")

        vdmock = mock.create_autospec(fakelibvirt.virDomain)
        vdmock.XMLDesc.return_value = dummyxml

        def fake_lookup(_uuid):
            if _uuid == instance['uuid']:
                return vdmock
        self.create_fake_libvirt_mock(lookupByUUIDString=fake_lookup)

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertRaises(exception.ConsoleTypeUnavailable,
                          drvr.get_spice_console, self.context, instance)
        vdmock.XMLDesc.assert_called_once_with(flags=0)

    def test_detach_volume_with_instance_not_found(self):
        # Test that detach_volume() method does not raise exception,
        # if the instance does not exist.

        instance = objects.Instance(**self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        with test.nested(
            mock.patch.object(host.Host, '_get_domain',
                              side_effect=exception.InstanceNotFound(
                                  instance_id=instance.uuid)),
            mock.patch.object(drvr, '_disconnect_volume')
        ) as (_get_domain, _disconnect_volume):
            connection_info = {'driver_volume_type': 'fake'}
            drvr.detach_volume(
                self.context, connection_info, instance, '/dev/sda')
            _get_domain.assert_called_once_with(instance)
            _disconnect_volume.assert_called_once_with(
                self.context, connection_info, instance, encryption=None)

    def _test_attach_detach_interface_get_config(self, method_name):
        """Tests that the get_config() method is properly called in
        attach_interface() and detach_interface().

        method_name: either \"attach_interface\" or \"detach_interface\"
                     depending on the method to test.
        """
        self.stub_out('nova.virt.libvirt.host.Host._get_domain',
                      lambda self, instance: FakeVirtDomain())

        instance = objects.Instance(**self.test_instance)
        instance.info_cache = None
        network_info = _fake_network_info(self)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)

        fake_image_meta = objects.ImageMeta.from_dict(
            {'id': instance['image_ref']})

        if method_name == "attach_interface":
            mock_build = self.useFixture(fixtures.MockPatchObject(
                drvr, '_build_device_metadata')).mock
            mock_build.return_value = objects.InstanceDeviceMetadata()
            mock_save = self.useFixture(fixtures.MockPatchObject(
                objects.Instance, 'save')).mock
            mock_get_network_info = self.useFixture(fixtures.MockPatchObject(
                objects.Instance, 'get_network_info')).mock

        expected = drvr.vif_driver.get_config(instance, network_info[0],
                                              fake_image_meta,
                                              instance.get_flavor(),
                                              CONF.libvirt.virt_type)
        mock_get_config = self.useFixture(fixtures.MockPatchObject(
            drvr.vif_driver, 'get_config')).mock
        mock_get_config.return_value = expected

        if method_name == "attach_interface":
            drvr.attach_interface(self.context, instance, fake_image_meta,
                                  network_info[0])
            mock_build.assert_called_once_with(self.context, instance)
            mock_save.assert_called_once_with()
            mock_get_network_info.assert_called_once_with()
        elif method_name == "detach_interface":
            drvr.detach_interface(self.context, instance, network_info[0])
        else:
            raise ValueError("Unhandled method %s" % method_name)

        mock_get_config.assert_called_once_with(
            instance, network_info[0], test.MatchType(objects.ImageMeta),
            test.MatchType(objects.Flavor), CONF.libvirt.virt_type)

    @mock.patch.object(lockutils, "external_lock")
    def test_attach_interface_get_config(self, mock_lock):
        """Tests that the get_config() method is properly called in
        attach_interface().
        """
        mock_lock.return_value = threading.Semaphore()

        self._test_attach_detach_interface_get_config("attach_interface")

    def test_detach_interface_get_config(self):
        """Tests that the get_config() method is properly called in
        detach_interface().
        """
        self._test_attach_detach_interface_get_config("detach_interface")

    @mock.patch.object(blockinfo, 'get_root_info')
    @mock.patch.object(blockinfo, 'get_disk_bus_for_device_type')
    def test_default_root_device_name(self, mock_get_disk, mock_get_root):
        instance = {'uuid': 'fake_instance'}
        image_meta = objects.ImageMeta.from_dict({'id': uuids.image_id})
        root_bdm = {'source_type': 'image',
                    'destination_type': 'volume',
                    'image_id': 'fake_id'}
        self.flags(virt_type='qemu', group='libvirt')

        mock_get_disk.side_effect = ['virtio', 'ide']
        mock_get_root.return_value = {'dev': 'vda'}

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertEqual(drvr.default_root_device_name(instance, image_meta,
                                                       root_bdm), '/dev/vda')
        self.assertEqual(2, mock_get_disk.call_count)
        mock_get_disk.assert_has_calls([
            mock.call(instance, 'qemu', image_meta, 'disk'),
            mock.call(instance, 'qemu', image_meta, 'cdrom')])
        mock_get_root.assert_called_once_with(instance, 'qemu', image_meta,
                                              root_bdm, 'virtio', 'ide')

    @mock.patch.object(objects.BlockDeviceMapping, "save")
    def test_default_device_names_for_instance(self, save_mock):
        instance = objects.Instance(**self.test_instance)
        instance.root_device_name = '/dev/vda'
        ephemerals = [objects.BlockDeviceMapping(
                        **fake_block_device.AnonFakeDbBlockDeviceDict(
                            {'device_name': 'vdb',
                             'source_type': 'blank',
                             'volume_size': 2,
                             'destination_type': 'local'}))]
        swap = [objects.BlockDeviceMapping(
                        **fake_block_device.AnonFakeDbBlockDeviceDict(
                            {'device_name': 'vdg',
                             'source_type': 'blank',
                             'volume_size': 512,
                             'guest_format': 'swap',
                             'destination_type': 'local'}))]
        block_device_mapping = [
            objects.BlockDeviceMapping(
                **fake_block_device.AnonFakeDbBlockDeviceDict(
                    {'source_type': 'volume',
                     'destination_type': 'volume',
                     'volume_id': 'fake-image-id',
                     'device_name': '/dev/vdxx',
                     'disk_bus': 'scsi',
                     'attachment_id': None}))]

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr.default_device_names_for_instance(instance,
                                               instance.root_device_name,
                                               ephemerals, swap,
                                               block_device_mapping)

        # Ephemeral device name was correct so no changes
        self.assertEqual('/dev/vdb', ephemerals[0].device_name)
        # Swap device name was incorrect so it was changed
        self.assertEqual('/dev/vdc', swap[0].device_name)
        # Volume device name was changed too, taking the bus into account
        self.assertEqual('/dev/sda', block_device_mapping[0].device_name)

        self.assertEqual(3, save_mock.call_count)

    def _test_get_device_name_for_instance(self, new_bdm, expected_dev):
        instance = objects.Instance(**self.test_instance)
        instance.root_device_name = '/dev/vda'
        instance.ephemeral_gb = 0
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        got_dev = drvr.get_device_name_for_instance(
            instance, [], new_bdm)
        self.assertEqual(expected_dev, got_dev)

    def test_get_device_name_for_instance_simple(self):
        new_bdm = objects.BlockDeviceMapping(
            context=context,
            source_type='volume', destination_type='volume',
            boot_index=-1, volume_id='fake-id',
            device_name=None, guest_format=None,
            disk_bus=None, device_type=None)
        self._test_get_device_name_for_instance(new_bdm, '/dev/vdb')

    def test_get_device_name_for_instance_suggested(self):
        new_bdm = objects.BlockDeviceMapping(
            context=context,
            source_type='volume', destination_type='volume',
            boot_index=-1, volume_id='fake-id',
            device_name='/dev/vdg', guest_format=None,
            disk_bus=None, device_type=None)
        self._test_get_device_name_for_instance(new_bdm, '/dev/vdb')

    def test_get_device_name_for_instance_bus(self):
        new_bdm = objects.BlockDeviceMapping(
            context=context,
            source_type='volume', destination_type='volume',
            boot_index=-1, volume_id='fake-id',
            device_name=None, guest_format=None,
            disk_bus='scsi', device_type=None)
        self._test_get_device_name_for_instance(new_bdm, '/dev/sda')

    def test_get_device_name_for_instance_device_type(self):
        new_bdm = objects.BlockDeviceMapping(
            context=context,
            source_type='volume', destination_type='volume',
            boot_index=-1, volume_id='fake-id',
            device_name=None, guest_format=None,
            disk_bus=None, device_type='floppy')
        self._test_get_device_name_for_instance(new_bdm, '/dev/fda')

    def test_is_supported_fs_format(self):
        supported_fs = [nova.privsep.fs.FS_FORMAT_EXT2,
                        nova.privsep.fs.FS_FORMAT_EXT3,
                        nova.privsep.fs.FS_FORMAT_EXT4,
                        nova.privsep.fs.FS_FORMAT_XFS]
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        for fs in supported_fs:
            self.assertTrue(drvr.is_supported_fs_format(fs))

        supported_fs = ['', 'dummy']
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        for fs in supported_fs:
            self.assertFalse(drvr.is_supported_fs_format(fs))

    @mock.patch("nova.objects.instance.Instance.image_meta",
                new_callable=mock.PropertyMock)
    @mock.patch("nova.virt.libvirt.driver.LibvirtDriver.attach_interface")
    @mock.patch('nova.virt.libvirt.guest.Guest.get_interfaces')
    @mock.patch('nova.virt.libvirt.host.Host.write_instance_config')
    @mock.patch('nova.virt.libvirt.host.Host.get_guest')
    def test_post_live_migration_at_destination(
            self, mock_get_guest, mock_write_instance_config,
            mock_get_interfaces, mock_attach, mock_image_meta):
        instance = objects.Instance(id=1, uuid=uuids.instance)
        dom = mock.MagicMock()
        guest = libvirt_guest.Guest(dom)

        mock_get_guest.return_value = guest

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        self.assertEqual({}, drvr.instance_claimed_mdevs)
        net_info = network_model.NetworkInfo()
        mock_get_interfaces.return_type = []
        mock_image_meta.return_value = mock.sentinel.image_meta
        drvr.instance_claimed_mdevs = {instance.uuid: [uuids.mdev1],
                                       uuids.other_inst: [uuids.mdev2]}
        drvr.post_live_migration_at_destination(mock.ANY, instance, net_info)
        # Assert that we don't try to write anything to the destination node
        # since the source live migrated with the VIR_MIGRATE_PERSIST_DEST flag
        mock_write_instance_config.assert_not_called()
        mock_attach.assert_not_called()
        # Assert we delete existing claimed mdevs for the instance
        self.assertEqual({uuids.other_inst: [uuids.mdev2]},
                         drvr.instance_claimed_mdevs)
        vif = network_model.VIF(id=uuids.port_id,
            vnic_type=network_model.VNIC_TYPE_NORMAL)
        vif_direct = network_model.VIF(id=uuids.port_id,
            vnic_type=network_model.VNIC_TYPE_DIRECT)

        net_info = network_model.NetworkInfo([vif, vif_direct])
        mock_get_interfaces.return_type = [vif]
        drvr.post_live_migration_at_destination(mock.ANY, instance, net_info)
        mock_attach.assert_called_once_with(mock.ANY, instance,
                                            mock.sentinel.image_meta,
                                            vif_direct)

    def test_create_guest_with_network__propagates_exceptions(self):
        self.flags(virt_type='lxc', group='libvirt')

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(id=1, uuid=uuids.instance,
                                    image_ref=uuids.image_ref)

        with test.nested(
              mock.patch.object(drvr, '_create_domain_setup_lxc'),
              mock.patch.object(drvr, '_create_domain_cleanup_lxc'),
              mock.patch.object(drvr, '_is_booted_from_volume',
                                return_value=False),
              mock.patch.object(drvr, 'plug_vifs'),
              mock.patch.object(drvr, '_create_guest',
                                side_effect=exception.NovaException),
              mock.patch.object(drvr, '_cleanup')):
            self.assertRaises(exception.NovaException,
                              drvr._create_guest_with_network,
                              self.context,
                              'xml',
                              instance, None, None)

    def test_create_guest_with_network__without_pause(self):
        self.flags(virt_type='lxc', group='libvirt')

        @contextlib.contextmanager
        def fake_lxc_disk_handler(*args, **kwargs):
            yield

        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)

        with test.nested(
              mock.patch.object(drvr, '_lxc_disk_handler',
                                side_effect=fake_lxc_disk_handler),
              mock.patch.object(drvr, 'plug_vifs'),
              mock.patch.object(drvr, '_create_guest'),
              mock.patch.object(drvr, 'cleanup')) as (
              _handler, cleanup, create, plug_vifs):
            domain = drvr._create_guest_with_network(self.context, 'xml',
                                                     instance, None, None)
            self.assertEqual(0, create.call_args_list[0][1]['pause'])
            self.assertEqual(0, domain.resume.call_count)

    def _test_create_guest_with_network__events(
        self, neutron_failure=None, power_on=True,
    ):
        generated_events = []

        def wait_timeout():
            event = mock.MagicMock()
            if neutron_failure == 'timeout':
                raise eventlet.timeout.Timeout()
            elif neutron_failure == 'error':
                event.status = 'failed'
            else:
                event.status = 'completed'
            return event

        def fake_prepare(instance, name, tag):
            m = mock.MagicMock()
            m.instance = instance
            m.event_name = '%s-%s' % (name, tag)
            m.wait.side_effect = wait_timeout
            generated_events.append(m)
            return m

        virtapi = manager.ComputeVirtAPI(mock.MagicMock())
        prepare = virtapi._compute.instance_events.prepare_for_instance_event
        prepare.side_effect = fake_prepare
        drvr = libvirt_driver.LibvirtDriver(virtapi, False)

        instance = objects.Instance(**self.test_instance)
        instance.vm_state = vm_states.BUILDING
        vifs = [
            network_model.VIF(id=uuids.vif_1, active=False),
            network_model.VIF(id=uuids.vif_2, active=False)
        ]

        @mock.patch.object(drvr, 'plug_vifs')
        @mock.patch.object(drvr, '_create_guest')
        @mock.patch.object(drvr, '_cleanup')
        def test_create(cleanup, create, plug_vifs):
            domain = drvr._create_guest_with_network(self.context, 'xml',
                                                     instance, vifs, None,
                                                     power_on=power_on)
            plug_vifs.assert_called_with(instance, vifs)

            pause = self._get_pause_flag(drvr, vifs, power_on=power_on)
            self.assertEqual(pause,
                             create.call_args_list[0][1]['pause'])
            if pause:
                domain.resume.assert_called_once_with()
            if neutron_failure and CONF.vif_plugging_is_fatal:
                cleanup.assert_called_once_with(self.context,
                                                instance, network_info=vifs,
                                                block_device_info=None)

        test_create()

        if CONF.vif_plugging_timeout and power_on:
            prepare.assert_has_calls([
                mock.call(instance, 'network-vif-plugged', uuids.vif_1),
                mock.call(instance, 'network-vif-plugged', uuids.vif_2)])
            for event in generated_events:
                if neutron_failure and generated_events.index(event) != 0:
                    self.assertEqual(0, event.call_count)
                elif (neutron_failure == 'error' and
                          not CONF.vif_plugging_is_fatal):
                    event.wait.assert_called_once_with()
        else:
            self.assertEqual(0, prepare.call_count)

    def test_create_guest_with_network__events_neutron(self):
        self._test_create_guest_with_network__events()

    def test_create_guest_with_network_events_neutron_power_off(self):
        # Tests that we don't wait for events if we don't start the instance.
        self._test_create_guest_with_network__events(power_on=False)

    def test_create_guest_with_network__events_nowait(self):
        self.flags(vif_plugging_timeout=0)
        self._test_create_guest_with_network__events()

    def test_create_guest_with_network__events_failed_nonfatal_timeout(self):
        self.flags(vif_plugging_is_fatal=False)
        self._test_create_guest_with_network__events(neutron_failure='timeout')

    def test_create_guest_with_network__events_failed_fatal_timeout(self):
        self.assertRaises(exception.VirtualInterfaceCreateException,
                          self._test_create_guest_with_network__events,
                          neutron_failure='timeout')

    def test_create_guest_with_network__events_failed_nonfatal_error(self):
        self.flags(vif_plugging_is_fatal=False)
        self._test_create_guest_with_network__events(neutron_failure='error')

    def test_create_guest_with_network__events_failed_fatal_error(self):
        self.assertRaises(exception.VirtualInterfaceCreateException,
                          self._test_create_guest_with_network__events,
                          neutron_failure='error')

    def test_create_guest_with_network__with_other_error(self):
        drvr = libvirt_driver.LibvirtDriver(mock.MagicMock(), False)

        @mock.patch.object(drvr, 'plug_vifs')
        @mock.patch.object(drvr, '_create_guest')
        @mock.patch.object(drvr, '_cleanup')
        def the_test(mock_cleanup, mock_create, mock_plug):
            instance = objects.Instance(**self.test_instance)
            mock_create.side_effect = test.TestingException
            self.assertRaises(
                test.TestingException, drvr._create_guest_with_network,
                self.context, 'xml', instance, [], None,
                cleanup_instance_dir=mock.sentinel.cleanup_instance_dir,
                cleanup_instance_disks=mock.sentinel.cleanup_instance_disks)
            mock_cleanup.assert_called_once_with(
                self.context, instance, [], None, destroy_vifs=True,
                cleanup_instance_dir=mock.sentinel.cleanup_instance_dir,
                cleanup_instance_disks=mock.sentinel.cleanup_instance_disks)
        the_test()

    @mock.patch('os_brick.encryptors.get_encryption_metadata')
    @mock.patch('nova.virt.libvirt.blockinfo.get_info_from_bdm')
    def test_create_guest_with_network__with_bdm(
        self, get_info_from_bdm, get_encryption_metadata,
    ):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        mock_dom = mock.MagicMock()
        mock_encryption_meta = mock.MagicMock()
        get_encryption_metadata.return_value = mock_encryption_meta

        fake_xml = """
            <domain>
                <name>instance-00000001</name>
                <memory>1048576</memory>
                <vcpu>1</vcpu>
                <devices>
                    <disk type='file' device='disk'>
                        <driver name='qemu' type='raw' cache='none'/>
                        <source file='/path/fake-volume1'/>
                        <target dev='vda' bus='virtio'/>
                    </disk>
                </devices>
            </domain>
        """
        fake_volume_id = "fake-volume-id"
        connection_info = {"driver_volume_type": "fake",
                           "data": {"access_mode": "rw",
                                    "volume_id": fake_volume_id}}

        def fake_getitem(*args, **kwargs):
            fake_bdm = {'connection_info': connection_info,
                        'mount_device': '/dev/vda'}
            return fake_bdm.get(args[0])

        mock_volume = mock.MagicMock()
        mock_volume.__getitem__.side_effect = fake_getitem
        block_device_info = {'block_device_mapping': [mock_volume]}
        network_info = [network_model.VIF(id='1'),
                        network_model.VIF(id='2', active=True)]

        with test.nested(
            mock.patch.object(drvr, 'plug_vifs'),
            mock.patch.object(drvr, '_create_guest'),
        ) as (plug_vifs, create_guest):
            create_guest.return_value = libvirt_guest.Guest(mock_dom)

            guest = drvr._create_guest_with_network(
                self.context, fake_xml, instance, network_info,
                block_device_info=block_device_info)

            plug_vifs.assert_called_once_with(instance, network_info)
            pause = self._get_pause_flag(drvr, network_info)
            create_guest.assert_called_once_with(
                self.context, fake_xml, instance, pause=pause, power_on=True,
                post_xml_callback=None)
            self.assertEqual(mock_dom, guest._domain)

    def test_get_guest_storage_config(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)

        test_instance = copy.deepcopy(self.test_instance)
        test_instance["default_swap_device"] = None
        instance = objects.Instance(**test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        flavor = instance.get_flavor()
        conn_info = {'driver_volume_type': 'fake', 'data': {}}
        bdm = objects.BlockDeviceMapping(
            self.context,
            **fake_block_device.FakeDbBlockDeviceDict({
                   'id': 1,
                   'source_type': 'volume',
                   'destination_type': 'volume',
                   'device_name': '/dev/vdc'}))
        bdi = {'block_device_mapping':
               driver_block_device.convert_volumes([bdm])}
        bdm = bdi['block_device_mapping'][0]
        bdm['connection_info'] = conn_info
        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance,
                                            image_meta,
                                            bdi)
        mock_conf = mock.MagicMock(source_path='fake')

        with test.nested(
            mock.patch.object(driver_block_device.DriverVolumeBlockDevice,
                              'save'),
            mock.patch.object(drvr, '_connect_volume'),
            mock.patch.object(drvr, '_get_volume_config',
                              return_value=mock_conf)
        ) as (volume_save, connect_volume, get_volume_config):
            devices = drvr._get_guest_storage_config(self.context, instance,
                image_meta, disk_info, False, bdi, flavor, "hvm")

            self.assertEqual(3, len(devices))
            self.assertEqual('/dev/vdb', instance.default_ephemeral_device)
            self.assertIsNone(instance.default_swap_device)
            connect_volume.assert_called_with(self.context,
                bdm['connection_info'], instance)
            get_volume_config.assert_called_with(
                instance,
                bdm['connection_info'],
                {'bus': 'virtio', 'type': 'disk', 'dev': 'vdc'})
            volume_save.assert_called_once_with()

    def test_get_neutron_events(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        network_info = [network_model.VIF(id='1'),
                        network_model.VIF(id='2', active=True)]
        events = drvr._get_neutron_events(network_info)
        self.assertEqual([('network-vif-plugged', '1')], events)

    def test_get_neutron_events_remote_managed(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        network_info = [
            network_model.VIF(
                id=uuids.vif_1,
                vnic_type=network_model.VNIC_TYPE_REMOTE_MANAGED),
            network_model.VIF(
                id=uuids.vif_2,
                vnic_type=network_model.VNIC_TYPE_REMOTE_MANAGED,
                active=True),
        ]
        events = drvr._get_neutron_events(network_info)
        # For VNIC_TYPE_REMOTE_MANAGED events are only bind-time currently.
        # Until this changes, they need to be filtered out to avoid waiting
        # for them unnecessarily.
        self.assertEqual([], events)

    def test_unplug_vifs_ignores_errors(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        with mock.patch.object(drvr, 'vif_driver') as vif_driver:
            vif_driver.unplug.side_effect = exception.InternalError('foo')
            drvr._unplug_vifs('inst', [1], ignore_errors=True)
            vif_driver.unplug.assert_called_once_with('inst', 1)

    def test_unplug_vifs_reports_errors(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        with mock.patch.object(drvr, 'vif_driver') as vif_driver:
            vif_driver.unplug.side_effect = exception.InternalError('foo')
            self.assertRaises(exception.InternalError,
                              drvr.unplug_vifs, 'inst', [1])
            vif_driver.unplug.assert_called_once_with('inst', 1)

    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._unplug_vifs')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._undefine_domain')
    def test_cleanup_pass_with_no_connection_info(self, undefine, unplug):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        drvr._disconnect_volume = mock.Mock()
        fake_inst = objects.Instance(**self.test_instance)
        fake_bdms = [{'connection_info': None}]
        with mock.patch('nova.virt.driver'
                        '.block_device_info_get_mapping',
                        return_value=fake_bdms):
            drvr.cleanup('ctxt', fake_inst, 'netinfo', destroy_disks=False)
        self.assertFalse(drvr._disconnect_volume.called)

    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._unplug_vifs')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._undefine_domain')
    def test_cleanup_pass_with_no_mount_device(self, undefine, unplug):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        drvr._disconnect_volume = mock.Mock()
        fake_inst = objects.Instance(**self.test_instance)
        fake_bdms = [{'connection_info': 'foo',
                     'mount_device': None}]
        with mock.patch('nova.virt.driver'
                        '.block_device_info_get_mapping',
                        return_value=fake_bdms):
            drvr.cleanup('ctxt', fake_inst, 'netinfo', destroy_disks=False)
        self.assertTrue(drvr._disconnect_volume.called)

    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._unplug_vifs')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._undefine_domain')
    def test_cleanup_wants_vif_errors_ignored(self, undefine, unplug):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        fake_inst = objects.Instance(**self.test_instance)
        with mock.patch.object(drvr._conn, 'lookupByUUIDString') as lookup:
            lookup.return_value = fake_inst
            # NOTE(danms): Make unplug cause us to bail early, since
            # we only care about how it was called
            unplug.side_effect = test.TestingException
            self.assertRaises(test.TestingException,
                              drvr.cleanup, 'ctxt', fake_inst, 'netinfo')
            unplug.assert_called_once_with(fake_inst, 'netinfo', True)

    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._undefine_domain')
    @mock.patch('nova.crypto.delete_vtpm_secret')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.delete_instance_files')
    @mock.patch('nova.virt.driver.block_device_info_get_mapping')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._unplug_vifs')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_vpmems',
                new=mock.Mock(return_value=None))
    def test_cleanup_pass(
        self, mock_unplug, mock_get_mapping, mock_delete_files,
        mock_delete_vtpm, mock_undefine,
    ):
        """Test with default parameters."""
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        fake_inst = objects.Instance(**self.test_instance)
        mock_get_mapping.return_value = []
        mock_delete_files.return_value = True

        with mock.patch.object(fake_inst, 'save'):
            drvr.cleanup('ctxt', fake_inst, 'netinfo')

        mock_unplug.assert_called_once_with(fake_inst, 'netinfo', True)
        mock_get_mapping.assert_called_once_with(None)
        mock_delete_files.assert_called_once_with(fake_inst)
        mock_delete_vtpm.assert_called_once_with('ctxt', fake_inst)
        mock_undefine.assert_called_once_with(fake_inst)

    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._undefine_domain')
    @mock.patch('nova.crypto.delete_vtpm_secret')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.delete_instance_files')
    @mock.patch('nova.virt.driver.block_device_info_get_mapping')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._unplug_vifs')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_vpmems',
                new=mock.Mock(return_value=None))
    def test_cleanup_instance_marked_deleted(
        self, mock_unplug, mock_get_mapping, mock_delete_files,
        mock_delete_vtpm, mock_undefine,
    ):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        fake_inst = objects.Instance(**self.test_instance)

        fake_inst.vm_state = vm_states.DELETED
        fake_inst.deleted = True

        with mock.patch.object(fake_inst, 'save') as instance_save:
            instance_save.side_effect = exception.InstanceNotFound(
                instance_id=uuids.instance)
            drvr.cleanup('ctxt', fake_inst, 'netinfo')
        mock_delete_vtpm.assert_called_once_with('ctxt', fake_inst)
        mock_undefine.assert_called_once_with(fake_inst)

    @mock.patch.object(libvirt_driver.LibvirtDriver, 'delete_instance_files',
                       return_value=True)
    @mock.patch.object(objects.Instance, 'save')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_undefine_domain')
    def test_cleanup_migrate_data_shared_block_storage(self,
                                                       _undefine_domain,
                                                       save,
                                                       delete_instance_files):
        # Tests the cleanup method when migrate_data has
        # is_shared_block_storage=True and destroy_disks=False.
        instance = objects.Instance(self.context, **self.test_instance)
        migrate_data = objects.LibvirtLiveMigrateData(
                is_shared_block_storage=True)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        drvr.cleanup(
            self.context, instance, network_info={}, destroy_disks=False,
            migrate_data=migrate_data, destroy_vifs=False)
        delete_instance_files.assert_called_once_with(instance)
        self.assertEqual(1, int(instance.system_metadata['clean_attempts']))
        self.assertTrue(instance.cleaned)
        save.assert_called_once_with()

    @mock.patch.object(libvirt_driver.LibvirtDriver, 'delete_instance_files',
                       return_value=True)
    @mock.patch.object(objects.Instance, 'save')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_undefine_domain')
    def test_cleanup_instance_dir_with_rbd_workaround(self,
            _undefine_domain, save, delete_instance_files):
        self.flags(images_type='rbd', group='libvirt')
        self.flags(ensure_libvirt_rbd_instance_dir_cleanup=True,
                   group='workarounds')
        instance = objects.Instance(self.context, **self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        # destroy_disks=False here as check_instance_shared_storage_local call
        # would return None when using the rbd imagebackend
        drvr.cleanup(self.context, instance, network_info={},
                     destroy_disks=False)
        delete_instance_files.assert_called_once_with(instance)
        self.assertEqual(1, int(instance.system_metadata['clean_attempts']))
        self.assertTrue(instance.cleaned)
        save.assert_called_once_with()

    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._disconnect_volume')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._undefine_domain',
                new=mock.Mock())
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_vpmems',
                new=mock.Mock(return_value=None))
    def test_cleanup_destroy_secrets(self, mock_disconnect_volume):
        block_device_info = {
            'block_device_mapping': [
                {
                    'connection_info': mock.sentinel.connection_info
                }
            ]
        }
        instance = objects.Instance(self.context, **self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())

        # Pass destroy_vifs=False and destroy_disks=False as we only care about
        # asserting the behaviour of destroy_secrets in this test.
        drvr.cleanup(
            self.context,
            instance,
            network_info={},
            block_device_info=block_device_info,
            destroy_vifs=False,
            destroy_disks=False,
            destroy_secrets=False
        )
        drvr.cleanup(
            self.context,
            instance,
            network_info={},
            block_device_info=block_device_info,
            destroy_vifs=False,
            destroy_disks=False,
        )

        # Assert that disconnect_volume is called with destroy_secrets=False
        # and destroy_secrets=True by default
        mock_disconnect_volume.assert_has_calls([
            mock.call(
                self.context,
                mock.sentinel.connection_info,
                instance,
                destroy_secrets=False,
                force=True
            ),
            mock.call(
                self.context,
                mock.sentinel.connection_info,
                instance,
                destroy_secrets=True,
                force=True
            )
        ])

    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_driver')
    @mock.patch(
        'nova.virt.libvirt.driver.LibvirtDriver._should_disconnect_target',
        new=mock.Mock(return_value=True))
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._detach_encryptor',
                new=mock.Mock())
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._undefine_domain',
                new=mock.Mock())
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_vpmems',
                new=mock.Mock(return_value=None))
    def test_cleanup_disconnect_volume(self, mock_vol_driver):
        """Verify that we call disconnect_volume() with force=True

        cleanup() is called by destroy() when an instance is being deleted and
        force=True should be passed down to os-brick's disconnect_volume()
        call, which will ensure removal of devices regardless of errors.

        We need to ensure that devices are removed when an instance is being
        deleted to avoid leaving leftover devices that could later be
        erroneously connected by external entities (example: multipathd) to
        instances that should not have access to the volumes.

        See https://bugs.launchpad.net/nova/+bug/2004555 for details.
        """
        connection_info = mock.MagicMock()
        block_device_info = {
            'block_device_mapping': [
                {
                    'connection_info': connection_info
                }
            ]
        }
        instance = objects.Instance(self.context, **self.test_instance)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())

        drvr.cleanup(
            self.context,
            instance,
            network_info={},
            block_device_info=block_device_info,
            destroy_vifs=False,
            destroy_disks=False,
        )
        mock_vol_driver.return_value.disconnect_volume.assert_called_once_with(
            connection_info, instance, force=True)

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_volume_encryption')
    @mock.patch.object(libvirt_driver.LibvirtDriver, '_allow_native_luksv1')
    def test_swap_volume_native_luks_blocked(self, mock_allow_native_luksv1,
                                             mock_get_encryption):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        mock_allow_native_luksv1.return_value = True

        # dest volume is encrypted
        mock_get_encryption.side_effect = [{}, {'provider': 'luks'}]
        self.assertRaises(NotImplementedError, drvr.swap_volume, self.context,
                            {}, {}, None, None, None)

        # src volume is encrypted
        mock_get_encryption.side_effect = [{'provider': 'luks'}, {}]
        self.assertRaises(NotImplementedError, drvr.swap_volume, self.context,
                            {}, {}, None, None, None)

        # both volumes are encrypted
        mock_get_encryption.side_effect = [{'provider': 'luks'},
                                           {'provider': 'luks'}]
        self.assertRaises(NotImplementedError, drvr.swap_volume, self.context,
                            {}, {}, None, None, None)

    @mock.patch('nova.virt.libvirt.host.Host.write_instance_config')
    def test_swap_volume_copy(self, mock_write_instance_config):
        """Assert the happy path of calling virDomainBlockCopy to swap"""
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())

        mock_guest = mock.MagicMock(spec=libvirt_guest.Guest)
        mock_dev = mock.MagicMock(spec=libvirt_guest.BlockDevice)
        mock_conf = mock.MagicMock(
            spec=vconfig.LibvirtConfigGuestDisk,
            source_type=mock.sentinel.source_type,
            target_dev=mock.sentinel.target_dev,
            source_path=None)

        mock_dev.is_job_complete.return_value = True
        mock_guest.get_block_device.return_value = mock_dev
        mock_guest.get_xml_desc.side_effect = [
            mock.sentinel.original_xml_desc,
            mock.sentinel.new_xml_desc]
        mock_guest.has_persistent_configuration.return_value = False
        mock_conf.to_xml.return_value = mock.sentinel.conf_xml

        resize_to = 1
        expected_resize_to = resize_to * units.Gi

        drvr._swap_volume(mock_guest, 'vdb', mock_conf, resize_to, None)

        # Assert that virDomainBlockCopy is called
        mock_dev.copy.assert_called_once_with(
            mock.sentinel.conf_xml, reuse_ext=True)

        # Assert that we abort once and then pivot
        mock_dev.abort_job.assert_has_calls([
            mock.call(), mock.call(pivot=True)])

        # Assert that virDomainBlockResize is called
        mock_dev.resize.assert_called_once_with(expected_resize_to)

        # Assert the new domain XML is written to disk on success
        mock_write_instance_config.assert_called_once_with(
            mock.sentinel.new_xml_desc)

    @mock.patch('nova.virt.libvirt.host.Host.write_instance_config')
    def test_swap_volume_copy_failure(self, mock_write_instance_config):
        """Assert that exception.VolumeRebaseFailed is raised on failure"""
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())

        mock_guest = mock.MagicMock(spec=libvirt_guest.Guest)
        mock_dev = mock.MagicMock(spec=libvirt_guest.BlockDevice)
        mock_conf = mock.MagicMock(
            spec=vconfig.LibvirtConfigGuestDisk,
            source_type=mock.sentinel.source_type,
            target_dev=mock.sentinel.target_dev,
            source_path=None)

        mock_dev.copy.side_effect = test.TestingException()
        mock_guest.get_block_device.return_value = mock_dev
        mock_guest.get_xml_desc.side_effect = [
            mock.sentinel.original_xml_desc,
            mock.sentinel.new_xml_desc]
        mock_guest.has_persistent_configuration.return_value = False
        mock_conf.to_xml.return_value = mock.sentinel.conf_xml

        # Assert that exception.VolumeRebaseFailed is raised
        self.assertRaises(exception.VolumeRebaseFailed, drvr._swap_volume,
                          mock_guest, 'vdb', mock_conf, 0, None)

        # Assert that virDomainBlockCopy is called
        mock_dev.copy.assert_called_once_with(
            mock.sentinel.conf_xml, reuse_ext=True)

        # Assert the new domain XML is written to disk on success
        mock_write_instance_config.assert_called_once_with(
            mock.sentinel.original_xml_desc)

    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._disconnect_volume')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._swap_volume')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_config')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._connect_volume')
    @mock.patch('nova.virt.libvirt.host.Host.get_guest')
    def test_swap_volume(self, get_guest, connect_volume, get_volume_config,
                         swap_volume, disconnect_volume):
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        instance = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        hw_firmware_type = image_meta.properties.get(
            'hw_firmware_type')
        old_connection_info = {'driver_volume_type': 'fake',
                               'serial': 'old-volume-id',
                               'data': {'device_path': '/fake-old-volume',
                                        'access_mode': 'rw'}}
        new_connection_info = {'driver_volume_type': 'fake',
                               'serial': 'new-volume-id',
                               'data': {'device_path': '/fake-new-volume',
                                        'access_mode': 'rw'}}
        mock_dom = mock.MagicMock()
        guest = libvirt_guest.Guest(mock_dom)
        mock_dom.XMLDesc.return_value = """<domain>
          <devices>
            <disk type='file'>
                <source file='/fake-old-volume'/>
                <target dev='vdb' bus='virtio'/>
            </disk>
          </devices>
        </domain>
        """
        mock_dom.name.return_value = 'inst'
        mock_dom.UUIDString.return_value = 'uuid'
        get_guest.return_value = guest
        conf = mock.MagicMock(source_path='/fake-new-volume')
        get_volume_config.return_value = conf

        conn.swap_volume(self.context, old_connection_info,
                         new_connection_info, instance, '/dev/vdb', 1)

        get_guest.assert_called_once_with(instance)
        connect_volume.assert_called_once_with(self.context,
                                               new_connection_info, instance)

        swap_volume.assert_called_once_with(guest, 'vdb',
                                            conf, 1, hw_firmware_type)
        disconnect_volume.assert_called_once_with(self.context,
                                                  old_connection_info,
                                                  instance)

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_volume_encryption')
    @mock.patch('nova.virt.libvirt.guest.BlockDevice.copy')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._disconnect_volume')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._connect_volume')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_config')
    @mock.patch('nova.virt.libvirt.guest.Guest.get_disk')
    @mock.patch('nova.virt.libvirt.host.Host.get_guest')
    @mock.patch('nova.virt.libvirt.host.Host.write_instance_config')
    def test_swap_volume_disconnect_new_volume_on_copy_error(self,
            write_config, get_guest, get_disk, get_volume_config,
            connect_volume, disconnect_volume, copy, get_volume_encryption):
        """Assert that disconnect_volume is called for the new volume if an
           error is encountered while rebasing
        """
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        instance = objects.Instance(**self.test_instance)
        guest = libvirt_guest.Guest(mock.MagicMock())
        get_guest.return_value = guest
        get_volume_encryption.return_value = {}
        exc = fakelibvirt.make_libvirtError(fakelibvirt.libvirtError,
              'internal error', error_code=fakelibvirt.VIR_ERR_INTERNAL_ERROR)
        copy.side_effect = exc

        self.assertRaises(exception.VolumeRebaseFailed, conn.swap_volume,
                          self.context, mock.sentinel.old_connection_info,
                          mock.sentinel.new_connection_info,
                          instance, '/dev/vdb', 0)
        connect_volume.assert_called_once_with(self.context,
                mock.sentinel.new_connection_info, instance)
        disconnect_volume.assert_called_once_with(self.context,
                mock.sentinel.new_connection_info, instance)

    @mock.patch.object(libvirt_driver.LibvirtDriver, '_get_volume_encryption')
    @mock.patch('nova.virt.libvirt.guest.BlockDevice.is_job_complete')
    @mock.patch('nova.virt.libvirt.guest.BlockDevice.abort_job')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._disconnect_volume')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._connect_volume')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver._get_volume_config')
    @mock.patch('nova.virt.libvirt.guest.Guest.get_disk')
    @mock.patch('nova.virt.libvirt.host.Host.get_guest')
    @mock.patch('nova.virt.libvirt.host.Host.write_instance_config')
    def test_swap_volume_disconnect_new_volume_on_pivot_error(self,
            write_config, get_guest, get_disk, get_volume_config,
            connect_volume, disconnect_volume, abort_job, is_job_complete,
            get_volume_encryption):
        """Assert that disconnect_volume is called for the new volume if an
           error is encountered while pivoting to the new volume
        """
        conn = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        instance = objects.Instance(**self.test_instance)
        guest = libvirt_guest.Guest(mock.MagicMock())
        get_guest.return_value = guest
        get_volume_encryption.return_value = {}
        exc = fakelibvirt.make_libvirtError(fakelibvirt.libvirtError,
              'internal error', error_code=fakelibvirt.VIR_ERR_INTERNAL_ERROR)
        is_job_complete.return_value = True
        abort_job.side_effect = [None, exc]

        self.assertRaises(exception.VolumeRebaseFailed, conn.swap_volume,
                          self.context, mock.sentinel.old_connection_info,
                          mock.sentinel.new_connection_info,
                          instance, '/dev/vdb', 0)
        connect_volume.assert_called_once_with(self.context,
                mock.sentinel.new_connection_info, instance)
        disconnect_volume.assert_called_once_with(self.context,
                mock.sentinel.new_connection_info, instance)

    @mock.patch.object(fileutils, 'delete_if_exists')
    @mock.patch('nova.virt.libvirt.guest.BlockDevice.is_job_complete')
    @mock.patch('nova.privsep.path.chown')
    def _test_live_snapshot(
            self, mock_chown, mock_is_job_complete, mock_delete,
            can_quiesce=False, require_quiesce=False):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI())
        mock_dom = mock.MagicMock()
        test_image_meta = self.test_image_meta.copy()
        if require_quiesce:
            test_image_meta = {'properties': {'os_require_quiesce': 'yes'}}

        with test.nested(
                mock.patch.object(drvr._conn, 'defineXML', create=True),
                mock.patch('nova.virt.libvirt.utils.get_disk_size'),
                mock.patch('nova.virt.libvirt.utils.get_disk_backing_file'),
                mock.patch('nova.virt.libvirt.utils.create_image'),
                mock.patch('nova.virt.libvirt.utils.extract_snapshot'),
                mock.patch.object(drvr, '_set_quiesced'),
                mock.patch.object(drvr, '_can_quiesce')
        ) as (mock_define, mock_size, mock_backing, mock_create_cow,
              mock_snapshot, mock_quiesce, mock_can_quiesce):

            xmldoc = "<domain/>"
            srcfile = "/first/path"
            dstfile = "/second/path"
            bckfile = "/other/path"
            dltfile = dstfile + ".delta"

            mock_dom.XMLDesc.return_value = xmldoc
            mock_dom.isPersistent.return_value = True
            mock_size.return_value = 1004009
            mock_backing.return_value = bckfile
            guest = libvirt_guest.Guest(mock_dom)

            if not can_quiesce:
                mock_can_quiesce.side_effect = (
                    exception.InstanceQuiesceNotSupported(
                        instance_id=self.test_instance['id'], reason='test'))

            image_meta = objects.ImageMeta.from_dict(test_image_meta)

            mock_is_job_complete.return_value = True

            drvr._live_snapshot(self.context, self.test_instance, guest,
                                srcfile, dstfile, "qcow2", "qcow2", image_meta)

            mock_dom.XMLDesc.assert_called_once_with(flags=(
                fakelibvirt.VIR_DOMAIN_XML_INACTIVE |
                fakelibvirt.VIR_DOMAIN_XML_SECURE))
            mock_dom.blockRebase.assert_called_once_with(
                srcfile, dltfile, 0, flags=(
                    fakelibvirt.VIR_DOMAIN_BLOCK_REBASE_COPY |
                    fakelibvirt.VIR_DOMAIN_BLOCK_REBASE_REUSE_EXT |
                    fakelibvirt.VIR_DOMAIN_BLOCK_REBASE_SHALLOW))

            mock_size.assert_called_once_with(srcfile, format="qcow2")
            mock_backing.assert_called_once_with(srcfile, basename=False,
                                                 format="qcow2")
            mock_create_cow.assert_called_once_with(
                dltfile, 'qcow2', 1004009, backing_file=bckfile)
            mock_chown.assert_called_once_with(dltfile, uid=os.getuid())
            mock_snapshot.assert_called_once_with(dltfile, "qcow2",
                                                  dstfile, "qcow2")
            mock_delete.assert_called_once_with(dltfile)
            mock_define.assert_called_once_with(xmldoc)
            mock_quiesce.assert_any_call(mock.ANY, self.test_instance,
                                         mock.ANY, True)

            if can_quiesce:
                mock_quiesce.assert_any_call(mock.ANY, self.test_instance,
                                             mock.ANY, False)

    def test_live_snapshot(self):
        self._test_live_snapshot()

    def test_live_snapshot_with_quiesce(self):
        self._test_live_snapshot(can_quiesce=True)

    def test_live_snapshot_with_require_quiesce(self):
        self._test_live_snapshot(can_quiesce=True, require_quiesce=True)

    def test_live_snapshot_with_require_quiesce_fails(self):
        self.assertRaises(exception.InstanceQuiesceNotSupported,
                          self._test_live_snapshot,
                          can_quiesce=False, require_quiesce=True)

    @mock.patch.object(libvirt_driver.LibvirtDriver, "_live_migration")
    def test_live_migration_hostname_valid(self, mock_lm):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr.live_migration(self.context, self.test_instance,
                            "host1.example.com",
                            lambda x: x,
                            lambda x: x)
        self.assertEqual(1, mock_lm.call_count)

    @mock.patch.object(libvirt_driver.LibvirtDriver, "_live_migration")
    @mock.patch('nova.virt.libvirt.utils.is_valid_hostname')
    def test_live_migration_hostname_invalid(self, mock_hostname, mock_lm):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        mock_hostname.return_value = False
        self.assertRaises(exception.InvalidHostname,
                          drvr.live_migration,
                          self.context, self.test_instance,
                          "foo/?com=/bin/sh",
                          lambda x: x,
                          lambda x: x)

    def test_live_migration_force_complete(self):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = fake_instance.fake_instance_obj(
            None, name='instancename', id=1,
            uuid='c83a75d4-4d53-4be5-9a40-04d9c0389ff8')
        drvr.active_migrations[instance.uuid] = collections.deque()
        drvr.live_migration_force_complete(instance)
        self.assertEqual(
            1, drvr.active_migrations[instance.uuid].count("force-complete"))

    @mock.patch.object(host.Host, "get_connection")
    @mock.patch.object(fakelibvirt.virDomain, "abortJob")
    def test_live_migration_abort(self, mock_abort, mock_conn):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        dom = fakelibvirt.Domain(drvr._get_connection(), "<domain/>", False)
        guest = libvirt_guest.Guest(dom)
        with mock.patch.object(nova.virt.libvirt.host.Host, 'get_guest',
                               return_value=guest):
            drvr.live_migration_abort(self.test_instance)
            self.assertTrue(mock_abort.called)

    @mock.patch('os.path.exists', return_value=True)
    @mock.patch('tempfile.mkstemp')
    @mock.patch('os.close', return_value=None)
    def test_check_instance_shared_storage_local_raw(self,
                                                 mock_close,
                                                 mock_mkstemp,
                                                 mock_exists):
        instance_uuid = uuids.fake
        self.flags(images_type='raw', group='libvirt')
        self.flags(instances_path='/tmp')
        mock_mkstemp.return_value = (-1,
                                     '/tmp/{0}/file'.format(instance_uuid))
        driver = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        temp_file = driver.check_instance_shared_storage_local(self.context,
                                                               instance)
        self.assertEqual('/tmp/{0}/file'.format(instance_uuid),
                         temp_file['filename'])

    def test_check_instance_shared_storage_local_rbd(self):
        self.flags(images_type='rbd', group='libvirt')
        driver = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        self.assertIsNone(driver.
                          check_instance_shared_storage_local(self.context,
                                                              instance))

    def test_version_to_string(self):
        string_ver = libvirt_utils.version_to_string((4, 33, 173))
        self.assertEqual("4.33.173", string_ver)

    def test_virtuozzo_min_version_fail(self):
        self.flags(virt_type='parallels', group='libvirt')
        driver = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        with mock.patch.object(driver._conn, 'getVersion') as mock_getver:
            mock_getver.return_value = \
                versionutils.convert_version_to_int(
                libvirt_driver.MIN_VIRTUOZZO_VERSION) - 1

            self.assertRaises(exception.NovaException,
                              driver.init_host, 'wibble')

    @mock.patch.object(libvirt_driver.LibvirtDriver,
                       '_register_all_undefined_instance_details',
                       new=mock.Mock())
    @mock.patch.object(fakelibvirt.Connection, 'getVersion',
                       return_value=versionutils.convert_version_to_int(
                           libvirt_driver.MIN_VIRTUOZZO_VERSION))
    def test_virtuozzo_min_version_ok(self, mock_get_virtuozzo_version):
        self.flags(virt_type='parallels', group='libvirt')
        driver = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        driver.init_host('wibble')

    def test_get_guest_config_parallels_vm(self):
        self.flags(virt_type='parallels', group='libvirt')
        self.flags(images_type='ploop', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta)

        cfg = drvr._get_guest_config(instance_ref,
                                    _fake_network_info(self),
                                    image_meta, disk_info)
        self.assertEqual("parallels", cfg.virt_type)
        self.assertEqual(instance_ref["uuid"], cfg.uuid)
        self.assertEqual(instance_ref.flavor.memory_mb * units.Ki, cfg.memory)
        self.assertEqual(instance_ref.flavor.vcpus, cfg.vcpus)
        self.assertEqual(fields.VMMode.HVM, cfg.os_type)
        self.assertIsNone(cfg.os_root)
        self.assertEqual(7, len(cfg.devices))
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertEqual(cfg.devices[0].driver_format, "ploop")
        self.assertIsInstance(cfg.devices[1],
                              vconfig.LibvirtConfigGuestDisk)
        self.assertIsInstance(cfg.devices[2],
                              vconfig.LibvirtConfigGuestInterface)
        self.assertIsInstance(cfg.devices[3],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[4],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[5],
                              vconfig.LibvirtConfigGuestInput)
        self.assertIsInstance(cfg.devices[6],
                              vconfig.LibvirtConfigGuestUSBHostController)

    def test_get_guest_config_parallels_ct_rescue(self):
        self._test_get_guest_config_parallels_ct(rescue=True)

    def test_get_guest_config_parallels_ct(self):
        self._test_get_guest_config_parallels_ct(rescue=False)

    def _test_get_guest_config_parallels_ct(self, rescue=False):
        self.flags(virt_type='parallels', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        ct_instance = self.test_instance.copy()
        ct_instance["vm_mode"] = fields.VMMode.EXE
        instance_ref = objects.Instance(**ct_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)

        if rescue:
            rescue_data = ct_instance
            disk_info = {'mapping': {'root': {'dev': 'hda'},
                                     'disk.rescue': {'dev': 'hda'}}}
        else:
            rescue_data = None
            disk_info = {'mapping': {'disk': {}}}

        cfg = drvr._get_guest_config(instance_ref,
                                    _fake_network_info(self),
                                    image_meta, disk_info, rescue_data)
        self.assertEqual("parallels", cfg.virt_type)
        self.assertEqual(instance_ref["uuid"], cfg.uuid)
        self.assertEqual(instance_ref.flavor.memory_mb * units.Ki, cfg.memory)
        self.assertEqual(instance_ref.flavor.vcpus, cfg.vcpus)
        self.assertEqual(fields.VMMode.EXE, cfg.os_type)
        self.assertEqual("/sbin/init", cfg.os_init_path)
        self.assertIsNone(cfg.os_root)
        if rescue:
            self.assertEqual(6, len(cfg.devices))
        else:
            self.assertEqual(5, len(cfg.devices))
        self.assertIsInstance(cfg.devices[0],
                              vconfig.LibvirtConfigGuestFilesys)

        device_index = 0
        fs = cfg.devices[device_index]
        self.assertEqual(fs.source_type, "file")
        self.assertEqual(fs.driver_type, "ploop")
        self.assertEqual(fs.target_dir, "/")

        if rescue:
            device_index = 1
            fs = cfg.devices[device_index]
            self.assertEqual(fs.source_type, "file")
            self.assertEqual(fs.driver_type, "ploop")
            self.assertEqual(fs.target_dir, "/mnt/rescue")

        self.assertIsInstance(cfg.devices[device_index + 1],
                              vconfig.LibvirtConfigGuestInterface)
        self.assertIsInstance(cfg.devices[device_index + 2],
                              vconfig.LibvirtConfigGuestGraphics)
        self.assertIsInstance(cfg.devices[device_index + 3],
                              vconfig.LibvirtConfigGuestVideo)
        self.assertIsInstance(cfg.devices[device_index + 4],
                              vconfig.LibvirtConfigGuestUSBHostController)

    def _test_get_guest_config_parallels_volume(self, vmmode, devices):
        self.flags(virt_type='parallels', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        ct_instance = self.test_instance.copy()
        ct_instance["vm_mode"] = vmmode
        instance_ref = objects.Instance(**ct_instance)

        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        conn_info = {'driver_volume_type': 'fake', 'data': {}}
        bdm = objects.BlockDeviceMapping(
            self.context,
            **fake_block_device.FakeDbBlockDeviceDict(
                {'id': 0,
                 'source_type': 'volume', 'destination_type': 'volume',
                 'device_name': '/dev/sda', 'boot_index': 0}))
        info = {'block_device_mapping': driver_block_device.convert_volumes(
                [bdm])}
        info['block_device_mapping'][0]['connection_info'] = conn_info

        disk_info = blockinfo.get_disk_info(CONF.libvirt.virt_type,
                                            instance_ref,
                                            image_meta,
                                            info)

        with mock.patch.object(
                        driver_block_device.DriverVolumeBlockDevice, 'save'
                                ) as mock_save:
            cfg = drvr._get_guest_config(instance_ref,
                                    _fake_network_info(self),
                                    image_meta, disk_info, None, info)
            mock_save.assert_called_once_with()

        self.assertEqual("parallels", cfg.virt_type)
        self.assertEqual(instance_ref["uuid"], cfg.uuid)
        self.assertEqual(instance_ref.flavor.memory_mb * units.Ki, cfg.memory)
        self.assertEqual(instance_ref.flavor.vcpus, cfg.vcpus)
        self.assertEqual(vmmode, cfg.os_type)
        self.assertIsNone(cfg.os_root)
        self.assertEqual(devices, len(cfg.devices))

        disk_found = False

        for dev in cfg.devices:
            result = isinstance(dev, vconfig.LibvirtConfigGuestFilesys)
            self.assertFalse(result)
            if (isinstance(dev, vconfig.LibvirtConfigGuestDisk) and
                (dev.source_path is None or
               'disk.local' not in dev.source_path)):
                self.assertEqual("disk", dev.source_device)
                self.assertEqual("sda", dev.target_dev)
                disk_found = True

        self.assertTrue(disk_found)

    def test_get_guest_config_parallels_volume(self):
        self._test_get_guest_config_parallels_volume(fields.VMMode.EXE, 5)
        self._test_get_guest_config_parallels_volume(fields.VMMode.HVM, 7)

    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.'
                '_guest_add_accel_pci_devices')
    def test_get_guest_config_accel_pci(self, mock_add_accel):
        # For an ARQ list with attach handle type 'PCI', the list should
        # be passed intact to _guest_add_accel_pci_devices.
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        instance.image_ref = uuids.image_ref
        instance.config_drive = ''
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = {'mapping': {}}

        accel_info = copy.deepcopy(nova_fixtures.CyborgFixture.bound_arq_list)
        for arq in accel_info:
            arq['attach_handle_type'] = 'PCI'
        drvr._get_guest_config(instance, network_info=[],
            image_meta=image_meta, disk_info=disk_info, accel_info=accel_info)
        mock_add_accel.assert_called_once_with(mock.ANY, accel_info)

    @mock.patch.object(libvirt_driver.LOG, 'info')
    @mock.patch('nova.virt.libvirt.driver.LibvirtDriver.'
                '_guest_add_accel_pci_devices')
    def test_get_guest_config_accel_nonpci(self, mock_add_accel, mock_log):
        # For an ARQ list with attach handle type != 'PCI',
        # _guest_add_accel_pci_devices should get [].
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance = objects.Instance(**self.test_instance)
        instance.image_ref = uuids.image_ref
        instance.config_drive = ''
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        disk_info = {'mapping': {}}

        # This list has ARQs with attach handle type 'TEST_PCI'.
        accel_info = nova_fixtures.CyborgFixture.bound_arq_list

        drvr._get_guest_config(instance, network_info=[],
            image_meta=image_meta, disk_info=disk_info, accel_info=accel_info)
        mock_add_accel.assert_called_once_with(mock.ANY, [])
        self.assertIn('Ignoring accelerator requests for instance',
                      str(mock_log.call_args[0]))

    def test_get_guest_disk_config_rbd_older_config_drive_fall_back(self):
        # New config drives are stored in rbd but existing instances have
        # config drives in the old location under the instances path.
        # Test that the driver falls back to 'flat' for config drive if it
        # doesn't exist in rbd.
        self.flags(images_type='rbd', group='libvirt')
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        drvr.image_backend = mock.Mock()
        mock_rbd_image = mock.Mock()
        mock_flat_image = mock.Mock()
        mock_flat_image.libvirt_info.return_value = mock.sentinel.diskconfig
        drvr.image_backend.by_name.side_effect = [mock_rbd_image,
                                                  mock_flat_image]
        mock_rbd_image.exists.return_value = False
        instance = objects.Instance()
        disk_mapping = {'disk.config': {'bus': 'ide',
                                        'dev': 'hda',
                                        'type': 'file'}}
        flavor = objects.Flavor(extra_specs={})

        diskconfig = drvr._get_guest_disk_config(
            instance, 'disk.config', disk_mapping, flavor,
            drvr._get_disk_config_image_type())

        self.assertEqual(2, drvr.image_backend.by_name.call_count)
        call1 = mock.call(instance, 'disk.config', 'rbd',
                disk_info_mapping=disk_mapping['disk.config'])
        call2 = mock.call(instance, 'disk.config', 'flat',
                disk_info_mapping=disk_mapping['disk.config'])
        drvr.image_backend.by_name.assert_has_calls([call1, call2])
        self.assertEqual(mock.sentinel.diskconfig, diskconfig)

    def _test_suspend_guest_for_snapshot(self, live_snapshot, state):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance_ref = objects.Instance(**self.test_instance)

        with mock.patch.object(drvr, "suspend") as mock_suspend:
            drvr._suspend_guest_for_snapshot(
                self.context, live_snapshot, state, instance_ref)

            return mock_suspend.called

    def test_suspend_guest_for_snapshot(self):
        # Ensure that suspend() is only called on RUNNING or PAUSED instances
        for test_power_state in power_state.STATE_MAP.keys():
            if test_power_state in (power_state.RUNNING, power_state.PAUSED):
                self.assertTrue(self._test_suspend_guest_for_snapshot(
                    False, test_power_state))
            else:
                self.assertFalse(self._test_suspend_guest_for_snapshot(
                    False, test_power_state))

    def test_suspend_guest_for_snapshot__lxc(self):
        self.flags(virt_type='lxc', group='libvirt')
        # Ensure that suspend() is never called with LXC
        for test_power_state in power_state.STATE_MAP.keys():
            self.assertFalse(self._test_suspend_guest_for_snapshot(
                False, test_power_state))

    def test_suspend_guest_for_snapshot__live_snapshots(self):
        # Ensure that suspend() is never called for live snapshots
        for test_power_state in power_state.STATE_MAP.keys():
            self.assertFalse(self._test_suspend_guest_for_snapshot(
                True, test_power_state))

    def _test_resume_guest_after_snapshot(self, live_snapshot, state):
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        instance_ref = objects.Instance(**self.test_instance)
        guest = mock.Mock()

        with test.nested(
            mock.patch.object(drvr, '_attach_pci_devices'),
            mock.patch.object(drvr, '_attach_direct_passthrough_ports'),
        ):
            drvr._resume_guest_after_snapshot(
                self.context, live_snapshot, state, instance_ref, guest)

            return guest.launch.called

    def test_resume_guest_after_snapshot(self):
        # Ensure that launch() is only called on RUNNING or PAUSED instances
        for test_power_state in power_state.STATE_MAP.keys():
            if test_power_state in (power_state.RUNNING, power_state.PAUSED):
                self.assertTrue(self._test_resume_guest_after_snapshot(
                    False, test_power_state))
            else:
                self.assertFalse(self._test_resume_guest_after_snapshot(
                    False, test_power_state))

    def test_resume_guest_after_snapshot__lxc(self):
        self.flags(virt_type='lxc', group='libvirt')
        # Ensure that launch() is never called with LXC
        for test_power_state in power_state.STATE_MAP.keys():
            self.assertFalse(self._test_resume_guest_after_snapshot(
                False, test_power_state))

    def test_resume_guest_after_snapshot__live_snapshots(self):
        # Ensure that launch() is never called for live snapshots
        for test_power_state in power_state.STATE_MAP.keys():
            self.assertFalse(self._test_resume_guest_after_snapshot(
                True, test_power_state))

    @mock.patch('os.walk')
    @mock.patch('os.path.exists')
    @mock.patch('os.path.getsize')
    @mock.patch('os.path.isdir')
    @mock.patch('nova.virt.images.qemu_img_info')
    @mock.patch.object(host.Host, '_get_domain')
    def test_get_instance_disk_info_parallels_ct(self, mock_get_domain,
                                                 mock_qemu_img_info,
                                                 mock_isdir,
                                                 mock_getsize,
                                                 mock_exists,
                                                 mock_walk):

        dummyxml = ("<domain type='parallels'><name>instance-0000000a</name>"
                    "<os><type>exe</type></os>"
                    "<devices>"
                    "<filesystem type='file'>"
                    "<driver format='ploop' type='ploop'/>"
                    "<source file='/test/disk'/>"
                    "<target dir='/'/></filesystem>"
                    "</devices></domain>")

        mock_qemu_img_info.return_value = mock.Mock(
            virtual_size=21474836480, image="/test/disk/root.hds",
            file_format="ploop", size=827327254, backing_file=None)

        self.flags(virt_type='parallels', group='libvirt')
        instance = objects.Instance(**self.test_instance)
        instance.vm_mode = fields.VMMode.EXE
        fake_dom = FakeVirtDomain(fake_xml=dummyxml)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), False)
        mock_get_domain.return_value = fake_dom
        mock_walk.return_value = [('/test/disk', [],
                                   ['DiskDescriptor.xml', 'root.hds'])]

        def getsize_sideeffect(*args, **kwargs):
            if args[0] == '/test/disk/DiskDescriptor.xml':
                return 790
            if args[0] == '/test/disk/root.hds':
                return 827326464

        mock_getsize.side_effect = getsize_sideeffect
        mock_exists.return_value = True
        mock_isdir.return_value = True

        info = drvr.get_instance_disk_info(instance)
        info = jsonutils.loads(info)
        self.assertEqual(info[0]['type'], 'ploop')
        self.assertEqual(info[0]['path'], '/test/disk')
        self.assertEqual(info[0]['disk_size'], 827327254)
        self.assertEqual(info[0]['over_committed_disk_size'], 20647509226)
        self.assertEqual(info[0]['virt_disk_size'], 21474836480)

    def test_get_guest_config_with_mdevs(self):
        mdevs = [uuids.mdev1]
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        instance_ref = objects.Instance(**self.test_instance)
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        cfg = drvr._get_guest_config(instance_ref,
                                     _fake_network_info(self),
                                     image_meta, {'mapping': {}},
                                     mdevs=mdevs)
        # Loop over all devices to make sure we have at least one mediated one.
        for device in cfg.devices:
            if isinstance(device, vconfig.LibvirtConfigGuestHostdevMDEV):
                # Make sure we use the provided UUID
                self.assertEqual(uuids.mdev1, device.uuid)
                break
        else:
            assert False, "Unable to find any mediated device for the guest."

    @mock.patch('nova.virt.hardware.get_vpmems')
    def test_get_guest_config_with_vpmems(self, mock_get_vpmems_label):
        vpmem_0 = objects.LibvirtVPMEMDevice(
            label='4GB', name='ns_0', devpath='/dev/dax0.0',
            size=4292870144, align=2097152)
        vpmem_1 = objects.LibvirtVPMEMDevice(
            label='16GB', name='ns_1', devpath='/dev/dax0.1',
            size=17177772032, align=2097152)
        resource_0 = objects.Resource(
            provider_uuid=uuids.rp,
            resource_class="CUSTOM_PMEM_NAMESPACE_4GB",
            identifier='ns_0', metadata=vpmem_0)
        resource_1 = objects.Resource(
            provider_uuid=uuids.rp,
            resource_class="CUSTOM_PMEM_NAMESPACE_16GB",
            identifier='ns_1', metadata=vpmem_1)
        resources = objects.ResourceList(objects=[resource_0, resource_1])

        instance_ref = objects.Instance(**self.test_instance)
        instance_ref.resources = resources
        image_meta = objects.ImageMeta.from_dict(self.test_image_meta)
        drvr = libvirt_driver.LibvirtDriver(fake.FakeVirtAPI(), True)
        drvr._vpmems_by_name = {"ns_0": vpmem_0, "ns_1": vpmem_1}

        mock_get_vpmems_label.return_value = ['4GB', '16GB']
        cfg = drvr._get_guest_config(instance_ref,
                                     _fake_network_info(self),
                                     image_meta, {'mapping': {}})
        vpmem_amount = 0
        for device in cfg.devices:
            if isinstance(device, vconfig.LibvirtConfigGuestVPMEM):
                s