/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */

namespace nss_test {

// RSA-PSS test vectors, pss-vect.txt, Example 1: A 1024-bit RSA Key Pair
// <ftp://ftp.rsasecurity.com/pub/pkcs/pkcs-1/pkcs-1v2-1-vec.zip>
const uint8_t kTestVector1Pkcs8[] = {
    0x30, 0x82, 0x02, 0x72, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
    0x02, 0x5c, 0x30, 0x82, 0x02, 0x58, 0x02, 0x01, 0x00, 0x02, 0x81, 0x80,
    0xa5, 0x6e, 0x4a, 0x0e, 0x70, 0x10, 0x17, 0x58, 0x9a, 0x51, 0x87, 0xdc,
    0x7e, 0xa8, 0x41, 0xd1, 0x56, 0xf2, 0xec, 0x0e, 0x36, 0xad, 0x52, 0xa4,
    0x4d, 0xfe, 0xb1, 0xe6, 0x1f, 0x7a, 0xd9, 0x91, 0xd8, 0xc5, 0x10, 0x56,
    0xff, 0xed, 0xb1, 0x62, 0xb4, 0xc0, 0xf2, 0x83, 0xa1, 0x2a, 0x88, 0xa3,
    0x94, 0xdf, 0xf5, 0x26, 0xab, 0x72, 0x91, 0xcb, 0xb3, 0x07, 0xce, 0xab,
    0xfc, 0xe0, 0xb1, 0xdf, 0xd5, 0xcd, 0x95, 0x08, 0x09, 0x6d, 0x5b, 0x2b,
    0x8b, 0x6d, 0xf5, 0xd6, 0x71, 0xef, 0x63, 0x77, 0xc0, 0x92, 0x1c, 0xb2,
    0x3c, 0x27, 0x0a, 0x70, 0xe2, 0x59, 0x8e, 0x6f, 0xf8, 0x9d, 0x19, 0xf1,
    0x05, 0xac, 0xc2, 0xd3, 0xf0, 0xcb, 0x35, 0xf2, 0x92, 0x80, 0xe1, 0x38,
    0x6b, 0x6f, 0x64, 0xc4, 0xef, 0x22, 0xe1, 0xe1, 0xf2, 0x0d, 0x0c, 0xe8,
    0xcf, 0xfb, 0x22, 0x49, 0xbd, 0x9a, 0x21, 0x37, 0x02, 0x03, 0x01, 0x00,
    0x01, 0x02, 0x81, 0x80, 0x33, 0xa5, 0x04, 0x2a, 0x90, 0xb2, 0x7d, 0x4f,
    0x54, 0x51, 0xca, 0x9b, 0xbb, 0xd0, 0xb4, 0x47, 0x71, 0xa1, 0x01, 0xaf,
    0x88, 0x43, 0x40, 0xae, 0xf9, 0x88, 0x5f, 0x2a, 0x4b, 0xbe, 0x92, 0xe8,
    0x94, 0xa7, 0x24, 0xac, 0x3c, 0x56, 0x8c, 0x8f, 0x97, 0x85, 0x3a, 0xd0,
    0x7c, 0x02, 0x66, 0xc8, 0xc6, 0xa3, 0xca, 0x09, 0x29, 0xf1, 0xe8, 0xf1,
    0x12, 0x31, 0x88, 0x44, 0x29, 0xfc, 0x4d, 0x9a, 0xe5, 0x5f, 0xee, 0x89,
    0x6a, 0x10, 0xce, 0x70, 0x7c, 0x3e, 0xd7, 0xe7, 0x34, 0xe4, 0x47, 0x27,
    0xa3, 0x95, 0x74, 0x50, 0x1a, 0x53, 0x26, 0x83, 0x10, 0x9c, 0x2a, 0xba,
    0xca, 0xba, 0x28, 0x3c, 0x31, 0xb4, 0xbd, 0x2f, 0x53, 0xc3, 0xee, 0x37,
    0xe3, 0x52, 0xce, 0xe3, 0x4f, 0x9e, 0x50, 0x3b, 0xd8, 0x0c, 0x06, 0x22,
    0xad, 0x79, 0xc6, 0xdc, 0xee, 0x88, 0x35, 0x47, 0xc6, 0xa3, 0xb3, 0x25,
    0x02, 0x40, 0xe7, 0xe8, 0x94, 0x27, 0x20, 0xa8, 0x77, 0x51, 0x72, 0x73,
    0xa3, 0x56, 0x05, 0x3e, 0xa2, 0xa1, 0xbc, 0x0c, 0x94, 0xaa, 0x72, 0xd5,
    0x5c, 0x6e, 0x86, 0x29, 0x6b, 0x2d, 0xfc, 0x96, 0x79, 0x48, 0xc0, 0xa7,
    0x2c, 0xbc, 0xcc, 0xa7, 0xea, 0xcb, 0x35, 0x70, 0x6e, 0x09, 0xa1, 0xdf,
    0x55, 0xa1, 0x53, 0x5b, 0xd9, 0xb3, 0xcc, 0x34, 0x16, 0x0b, 0x3b, 0x6d,
    0xcd, 0x3e, 0xda, 0x8e, 0x64, 0x43, 0x02, 0x40, 0xb6, 0x9d, 0xca, 0x1c,
    0xf7, 0xd4, 0xd7, 0xec, 0x81, 0xe7, 0x5b, 0x90, 0xfc, 0xca, 0x87, 0x4a,
    0xbc, 0xde, 0x12, 0x3f, 0xd2, 0x70, 0x01, 0x80, 0xaa, 0x90, 0x47, 0x9b,
    0x6e, 0x48, 0xde, 0x8d, 0x67, 0xed, 0x24, 0xf9, 0xf1, 0x9d, 0x85, 0xba,
    0x27, 0x58, 0x74, 0xf5, 0x42, 0xcd, 0x20, 0xdc, 0x72, 0x3e, 0x69, 0x63,
    0x36, 0x4a, 0x1f, 0x94, 0x25, 0x45, 0x2b, 0x26, 0x9a, 0x67, 0x99, 0xfd,
    0x02, 0x40, 0x28, 0xfa, 0x13, 0x93, 0x86, 0x55, 0xbe, 0x1f, 0x8a, 0x15,
    0x9c, 0xba, 0xca, 0x5a, 0x72, 0xea, 0x19, 0x0c, 0x30, 0x08, 0x9e, 0x19,
    0xcd, 0x27, 0x4a, 0x55, 0x6f, 0x36, 0xc4, 0xf6, 0xe1, 0x9f, 0x55, 0x4b,
    0x34, 0xc0, 0x77, 0x79, 0x04, 0x27, 0xbb, 0xdd, 0x8d, 0xd3, 0xed, 0xe2,
    0x44, 0x83, 0x28, 0xf3, 0x85, 0xd8, 0x1b, 0x30, 0xe8, 0xe4, 0x3b, 0x2f,
    0xff, 0xa0, 0x27, 0x86, 0x19, 0x79, 0x02, 0x40, 0x1a, 0x8b, 0x38, 0xf3,
    0x98, 0xfa, 0x71, 0x20, 0x49, 0x89, 0x8d, 0x7f, 0xb7, 0x9e, 0xe0, 0xa7,
    0x76, 0x68, 0x79, 0x12, 0x99, 0xcd, 0xfa, 0x09, 0xef, 0xc0, 0xe5, 0x07,
    0xac, 0xb2, 0x1e, 0xd7, 0x43, 0x01, 0xef, 0x5b, 0xfd, 0x48, 0xbe, 0x45,
    0x5e, 0xae, 0xb6, 0xe1, 0x67, 0x82, 0x55, 0x82, 0x75, 0x80, 0xa8, 0xe4,
    0xe8, 0xe1, 0x41, 0x51, 0xd1, 0x51, 0x0a, 0x82, 0xa3, 0xf2, 0xe7, 0x29,
    0x02, 0x40, 0x27, 0x15, 0x6a, 0xba, 0x41, 0x26, 0xd2, 0x4a, 0x81, 0xf3,
    0xa5, 0x28, 0xcb, 0xfb, 0x27, 0xf5, 0x68, 0x86, 0xf8, 0x40, 0xa9, 0xf6,
    0xe8, 0x6e, 0x17, 0xa4, 0x4b, 0x94, 0xfe, 0x93, 0x19, 0x58, 0x4b, 0x8e,
    0x22, 0xfd, 0xde, 0x1e, 0x5a, 0x2e, 0x3b, 0xd8, 0xaa, 0x5b, 0xa8, 0xd8,
    0x58, 0x41, 0x94, 0xeb, 0x21, 0x90, 0xac, 0xf8, 0x32, 0xb8, 0x47, 0xf1,
    0x3a, 0x3d, 0x24, 0xa7, 0x9f, 0x4d};
const uint8_t kTestVector1Spki[] = {
    0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
    0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30, 0x81,
    0x89, 0x02, 0x81, 0x81, 0x00, 0xa5, 0x6e, 0x4a, 0x0e, 0x70, 0x10, 0x17,
    0x58, 0x9a, 0x51, 0x87, 0xdc, 0x7e, 0xa8, 0x41, 0xd1, 0x56, 0xf2, 0xec,
    0x0e, 0x36, 0xad, 0x52, 0xa4, 0x4d, 0xfe, 0xb1, 0xe6, 0x1f, 0x7a, 0xd9,
    0x91, 0xd8, 0xc5, 0x10, 0x56, 0xff, 0xed, 0xb1, 0x62, 0xb4, 0xc0, 0xf2,
    0x83, 0xa1, 0x2a, 0x88, 0xa3, 0x94, 0xdf, 0xf5, 0x26, 0xab, 0x72, 0x91,
    0xcb, 0xb3, 0x07, 0xce, 0xab, 0xfc, 0xe0, 0xb1, 0xdf, 0xd5, 0xcd, 0x95,
    0x08, 0x09, 0x6d, 0x5b, 0x2b, 0x8b, 0x6d, 0xf5, 0xd6, 0x71, 0xef, 0x63,
    0x77, 0xc0, 0x92, 0x1c, 0xb2, 0x3c, 0x27, 0x0a, 0x70, 0xe2, 0x59, 0x8e,
    0x6f, 0xf8, 0x9d, 0x19, 0xf1, 0x05, 0xac, 0xc2, 0xd3, 0xf0, 0xcb, 0x35,
    0xf2, 0x92, 0x80, 0xe1, 0x38, 0x6b, 0x6f, 0x64, 0xc4, 0xef, 0x22, 0xe1,
    0xe1, 0xf2, 0x0d, 0x0c, 0xe8, 0xcf, 0xfb, 0x22, 0x49, 0xbd, 0x9a, 0x21,
    0x37, 0x02, 0x03, 0x01, 0x00, 0x01};
// RSA-PSS test vectors, pss-vect.txt, Example 1.1
const uint8_t kTestVector1Data[] = {
    0xcd, 0xc8, 0x7d, 0xa2, 0x23, 0xd7, 0x86, 0xdf, 0x3b, 0x45, 0xe0, 0xbb,
    0xbc, 0x72, 0x13, 0x26, 0xd1, 0xee, 0x2a, 0xf8, 0x06, 0xcc, 0x31, 0x54,
    0x75, 0xcc, 0x6f, 0x0d, 0x9c, 0x66, 0xe1, 0xb6, 0x23, 0x71, 0xd4, 0x5c,
    0xe2, 0x39, 0x2e, 0x1a, 0xc9, 0x28, 0x44, 0xc3, 0x10, 0x10, 0x2f, 0x15,
    0x6a, 0x0d, 0x8d, 0x52, 0xc1, 0xf4, 0xc4, 0x0b, 0xa3, 0xaa, 0x65, 0x09,
    0x57, 0x86, 0xcb, 0x76, 0x97, 0x57, 0xa6, 0x56, 0x3b, 0xa9, 0x58, 0xfe,
    0xd0, 0xbc, 0xc9, 0x84, 0xe8, 0xb5, 0x17, 0xa3, 0xd5, 0xf5, 0x15, 0xb2,
    0x3b, 0x8a, 0x41, 0xe7, 0x4a, 0xa8, 0x67, 0x69, 0x3f, 0x90, 0xdf, 0xb0,
    0x61, 0xa6, 0xe8, 0x6d, 0xfa, 0xae, 0xe6, 0x44, 0x72, 0xc0, 0x0e, 0x5f,
    0x20, 0x94, 0x57, 0x29, 0xcb, 0xeb, 0xe7, 0x7f, 0x06, 0xce, 0x78, 0xe0,
    0x8f, 0x40, 0x98, 0xfb, 0xa4, 0x1f, 0x9d, 0x61, 0x93, 0xc0, 0x31, 0x7e,
    0x8b, 0x60, 0xd4, 0xb6, 0x08, 0x4a, 0xcb, 0x42, 0xd2, 0x9e, 0x38, 0x08,
    0xa3, 0xbc, 0x37, 0x2d, 0x85, 0xe3, 0x31, 0x17, 0x0f, 0xcb, 0xf7, 0xcc,
    0x72, 0xd0, 0xb7, 0x1c, 0x29, 0x66, 0x48, 0xb3, 0xa4, 0xd1, 0x0f, 0x41,
    0x62, 0x95, 0xd0, 0x80, 0x7a, 0xa6, 0x25, 0xca, 0xb2, 0x74, 0x4f, 0xd9,
    0xea, 0x8f, 0xd2, 0x23, 0xc4, 0x25, 0x37, 0x02, 0x98, 0x28, 0xbd, 0x16,
    0xbe, 0x02, 0x54, 0x6f, 0x13, 0x0f, 0xd2, 0xe3, 0x3b, 0x93, 0x6d, 0x26,
    0x76, 0xe0, 0x8a, 0xed, 0x1b, 0x73, 0x31, 0x8b, 0x75, 0x0a, 0x01, 0x67,
    0xd0};
const uint8_t kTestVector1Sig[] = {
    0x90, 0x74, 0x30, 0x8f, 0xb5, 0x98, 0xe9, 0x70, 0x1b, 0x22, 0x94, 0x38,
    0x8e, 0x52, 0xf9, 0x71, 0xfa, 0xac, 0x2b, 0x60, 0xa5, 0x14, 0x5a, 0xf1,
    0x85, 0xdf, 0x52, 0x87, 0xb5, 0xed, 0x28, 0x87, 0xe5, 0x7c, 0xe7, 0xfd,
    0x44, 0xdc, 0x86, 0x34, 0xe4, 0x07, 0xc8, 0xe0, 0xe4, 0x36, 0x0b, 0xc2,
    0x26, 0xf3, 0xec, 0x22, 0x7f, 0x9d, 0x9e, 0x54, 0x63, 0x8e, 0x8d, 0x31,
    0xf5, 0x05, 0x12, 0x15, 0xdf, 0x6e, 0xbb, 0x9c, 0x2f, 0x95, 0x79, 0xaa,
    0x77, 0x59, 0x8a, 0x38, 0xf9, 0x14, 0xb5, 0xb9, 0xc1, 0xbd, 0x83, 0xc4,
    0xe2, 0xf9, 0xf3, 0x82, 0xa0, 0xd0, 0xaa, 0x35, 0x42, 0xff, 0xee, 0x65,
    0x98, 0x4a, 0x60, 0x1b, 0xc6, 0x9e, 0xb2, 0x8d, 0xeb, 0x27, 0xdc, 0xa1,
    0x2c, 0x82, 0xc2, 0xd4, 0xc3, 0xf6, 0x6c, 0xd5, 0x00, 0xf1, 0xff, 0x2b,
    0x99, 0x4d, 0x8a, 0x4e, 0x30, 0xcb, 0xb3, 0x3c};

// RSA-PSS test vectors, pss-vect.txt, Example 2: A 1025-bit RSA Key Pair
// <ftp://ftp.rsasecurity.com/pub/pkcs/pkcs-1/pkcs-1v2-1-vec.zip>
const uint8_t kTestVector2Pkcs8[] = {
    0x30, 0x82, 0x02, 0x75, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
    0x02, 0x5f, 0x30, 0x82, 0x02, 0x5b, 0x02, 0x01, 0x00, 0x02, 0x81, 0x81,
    0x01, 0xd4, 0x0c, 0x1b, 0xcf, 0x97, 0xa6, 0x8a, 0xe7, 0xcd, 0xbd, 0x8a,
    0x7b, 0xf3, 0xe3, 0x4f, 0xa1, 0x9d, 0xcc, 0xa4, 0xef, 0x75, 0xa4, 0x74,
    0x54, 0x37, 0x5f, 0x94, 0x51, 0x4d, 0x88, 0xfe, 0xd0, 0x06, 0xfb, 0x82,
    0x9f, 0x84, 0x19, 0xff, 0x87, 0xd6, 0x31, 0x5d, 0xa6, 0x8a, 0x1f, 0xf3,
    0xa0, 0x93, 0x8e, 0x9a, 0xbb, 0x34, 0x64, 0x01, 0x1c, 0x30, 0x3a, 0xd9,
    0x91, 0x99, 0xcf, 0x0c, 0x7c, 0x7a, 0x8b, 0x47, 0x7d, 0xce, 0x82, 0x9e,
    0x88, 0x44, 0xf6, 0x25, 0xb1, 0x15, 0xe5, 0xe9, 0xc4, 0xa5, 0x9c, 0xf8,
    0xf8, 0x11, 0x3b, 0x68, 0x34, 0x33, 0x6a, 0x2f, 0xd2, 0x68, 0x9b, 0x47,
    0x2c, 0xbb, 0x5e, 0x5c, 0xab, 0xe6, 0x74, 0x35, 0x0c, 0x59, 0xb6, 0xc1,
    0x7e, 0x17, 0x68, 0x74, 0xfb, 0x42, 0xf8, 0xfc, 0x3d, 0x17, 0x6a, 0x01,
    0x7e, 0xdc, 0x61, 0xfd, 0x32, 0x6c, 0x4b, 0x33, 0xc9, 0x02, 0x03, 0x01,
    0x00, 0x01, 0x02, 0x81, 0x80, 0x02, 0x7d, 0x14, 0x7e, 0x46, 0x73, 0x05,
    0x73, 0x77, 0xfd, 0x1e, 0xa2, 0x01, 0x56, 0x57, 0x72, 0x17, 0x6a, 0x7d,
    0xc3, 0x83, 0x58, 0xd3, 0x76, 0x04, 0x56, 0x85, 0xa2, 0xe7, 0x87, 0xc2,
    0x3c, 0x15, 0x57, 0x6b, 0xc1, 0x6b, 0x9f, 0x44, 0x44, 0x02, 0xd6, 0xbf,
    0xc5, 0xd9, 0x8a, 0x3e, 0x88, 0xea, 0x13, 0xef, 0x67, 0xc3, 0x53, 0xec,
    0xa0, 0xc0, 0xdd, 0xba, 0x92, 0x55, 0xbd, 0x7b, 0x8b, 0xb5, 0x0a, 0x64,
    0x4a, 0xfd, 0xfd, 0x1d, 0xd5, 0x16, 0x95, 0xb2, 0x52, 0xd2, 0x2e, 0x73,
    0x18, 0xd1, 0xb6, 0x68, 0x7a, 0x1c, 0x10, 0xff, 0x75, 0x54, 0x5f, 0x3d,
    0xb0, 0xfe, 0x60, 0x2d, 0x5f, 0x2b, 0x7f, 0x29, 0x4e, 0x36, 0x01, 0xea,
    0xb7, 0xb9, 0xd1, 0xce, 0xcd, 0x76, 0x7f, 0x64, 0x69, 0x2e, 0x3e, 0x53,
    0x6c, 0xa2, 0x84, 0x6c, 0xb0, 0xc2, 0xdd, 0x48, 0x6a, 0x39, 0xfa, 0x75,
    0xb1, 0x02, 0x41, 0x01, 0x66, 0x01, 0xe9, 0x26, 0xa0, 0xf8, 0xc9, 0xe2,
    0x6e, 0xca, 0xb7, 0x69, 0xea, 0x65, 0xa5, 0xe7, 0xc5, 0x2c, 0xc9, 0xe0,
    0x80, 0xef, 0x51, 0x94, 0x57, 0xc6, 0x44, 0xda, 0x68, 0x91, 0xc5, 0xa1,
    0x04, 0xd3, 0xea, 0x79, 0x55, 0x92, 0x9a, 0x22, 0xe7, 0xc6, 0x8a, 0x7a,
    0xf9, 0xfc, 0xad, 0x77, 0x7c, 0x3c, 0xcc, 0x2b, 0x9e, 0x3d, 0x36, 0x50,
    0xbc, 0xe4, 0x04, 0x39, 0x9b, 0x7e, 0x59, 0xd1, 0x02, 0x41, 0x01, 0x4e,
    0xaf, 0xa1, 0xd4, 0xd0, 0x18, 0x4d, 0xa7, 0xe3, 0x1f, 0x87, 0x7d, 0x12,
    0x81, 0xdd, 0xda, 0x62, 0x56, 0x64, 0x86, 0x9e, 0x83, 0x79, 0xe6, 0x7a,
    0xd3, 0xb7, 0x5e, 0xae, 0x74, 0xa5, 0x80, 0xe9, 0x82, 0x7a, 0xbd, 0x6e,
    0xb7, 0xa0, 0x02, 0xcb, 0x54, 0x11, 0xf5, 0x26, 0x67, 0x97, 0x76, 0x8f,
    0xb8, 0xe9, 0x5a, 0xe4, 0x0e, 0x3e, 0x8a, 0x01, 0xf3, 0x5f, 0xf8, 0x9e,
    0x56, 0xc0, 0x79, 0x02, 0x40, 0xe2, 0x47, 0xcc, 0xe5, 0x04, 0x93, 0x9b,
    0x8f, 0x0a, 0x36, 0x09, 0x0d, 0xe2, 0x00, 0x93, 0x87, 0x55, 0xe2, 0x44,
    0x4b, 0x29, 0x53, 0x9a, 0x7d, 0xa7, 0xa9, 0x02, 0xf6, 0x05, 0x68, 0x35,
    0xc0, 0xdb, 0x7b, 0x52, 0x55, 0x94, 0x97, 0xcf, 0xe2, 0xc6, 0x1a, 0x80,
    0x86, 0xd0, 0x21, 0x3c, 0x47, 0x2c, 0x78, 0x85, 0x18, 0x00, 0xb1, 0x71,
    0xf6, 0x40, 0x1d, 0xe2, 0xe9, 0xc2, 0x75, 0x6f, 0x31, 0x02, 0x40, 0xb1,
    0x2f, 0xba, 0x75, 0x78, 0x55, 0xe5, 0x86, 0xe4, 0x6f, 0x64, 0xc3, 0x8a,
    0x70, 0xc6, 0x8b, 0x3f, 0x54, 0x8d, 0x93, 0xd7, 0x87, 0xb3, 0x99, 0x99,
    0x9d, 0x4c, 0x8f, 0x0b, 0xbd, 0x25, 0x81, 0xc2, 0x1e, 0x19, 0xed, 0x00,
    0x18, 0xa6, 0xd5, 0xd3, 0xdf, 0x86, 0x42, 0x4b, 0x3a, 0xbc, 0xad, 0x40,
    0x19, 0x9d, 0x31, 0x49, 0x5b, 0x61, 0x30, 0x9f, 0x27, 0xc1, 0xbf, 0x55,
    0xd4, 0x87, 0xc1, 0x02, 0x40, 0x56, 0x4b, 0x1e, 0x1f, 0xa0, 0x03, 0xbd,
    0xa9, 0x1e, 0x89, 0x09, 0x04, 0x25, 0xaa, 0xc0, 0x5b, 0x91, 0xda, 0x9e,
    0xe2, 0x50, 0x61, 0xe7, 0x62, 0x8d, 0x5f, 0x51, 0x30, 0x4a, 0x84, 0x99,
    0x2f, 0xdc, 0x33, 0x76, 0x2b, 0xd3, 0x78, 0xa5, 0x9f, 0x03, 0x0a, 0x33,
    0x4d, 0x53, 0x2b, 0xd0, 0xda, 0xe8, 0xf2, 0x98, 0xea, 0x9e, 0xd8, 0x44,
    0x63, 0x6a, 0xd5, 0xfb, 0x8c, 0xbd, 0xc0, 0x3c, 0xad};
const uint8_t kTestVector2Spki[] = {
    0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
    0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30, 0x81,
    0x89, 0x02, 0x81, 0x81, 0x01, 0xd4, 0x0c, 0x1b, 0xcf, 0x97, 0xa6, 0x8a,
    0xe7, 0xcd, 0xbd, 0x8a, 0x7b, 0xf3, 0xe3, 0x4f, 0xa1, 0x9d, 0xcc, 0xa4,
    0xef, 0x75, 0xa4, 0x74, 0x54, 0x37, 0x5f, 0x94, 0x51, 0x4d, 0x88, 0xfe,
    0xd0, 0x06, 0xfb, 0x82, 0x9f, 0x84, 0x19, 0xff, 0x87, 0xd6, 0x31, 0x5d,
    0xa6, 0x8a, 0x1f, 0xf3, 0xa0, 0x93, 0x8e, 0x9a, 0xbb, 0x34, 0x64, 0x01,
    0x1c, 0x30, 0x3a, 0xd9, 0x91, 0x99, 0xcf, 0x0c, 0x7c, 0x7a, 0x8b, 0x47,
    0x7d, 0xce, 0x82, 0x9e, 0x88, 0x44, 0xf6, 0x25, 0xb1, 0x15, 0xe5, 0xe9,
    0xc4, 0xa5, 0x9c, 0xf8, 0xf8, 0x11, 0x3b, 0x68, 0x34, 0x33, 0x6a, 0x2f,
    0xd2, 0x68, 0x9b, 0x47, 0x2c, 0xbb, 0x5e, 0x5c, 0xab, 0xe6, 0x74, 0x35,
    0x0c, 0x59, 0xb6, 0xc1, 0x7e, 0x17, 0x68, 0x74, 0xfb, 0x42, 0xf8, 0xfc,
    0x3d, 0x17, 0x6a, 0x01, 0x7e, 0xdc, 0x61, 0xfd, 0x32, 0x6c, 0x4b, 0x33,
    0xc9, 0x02, 0x03, 0x01, 0x00, 0x01};
// RSA-PSS test vectors, pss-vect.txt, Example 2.1
const uint8_t kTestVector2Data[] = {
    0xda, 0xba, 0x03, 0x20, 0x66, 0x26, 0x3f, 0xae, 0xdb, 0x65, 0x98,
    0x48, 0x11, 0x52, 0x78, 0xa5, 0x2c, 0x44, 0xfa, 0xa3, 0xa7, 0x6f,
    0x37, 0x51, 0x5e, 0xd3, 0x36, 0x32, 0x10, 0x72, 0xc4, 0x0a, 0x9d,
    0x9b, 0x53, 0xbc, 0x05, 0x01, 0x40, 0x78, 0xad, 0xf5, 0x20, 0x87,
    0x51, 0x46, 0xaa, 0xe7, 0x0f, 0xf0, 0x60, 0x22, 0x6d, 0xcb, 0x7b,
    0x1f, 0x1f, 0xc2, 0x7e, 0x93, 0x60};
const uint8_t kTestVector2Sig[] = {
    0x01, 0x4c, 0x5b, 0xa5, 0x33, 0x83, 0x28, 0xcc, 0xc6, 0xe7, 0xa9, 0x0b,
    0xf1, 0xc0, 0xab, 0x3f, 0xd6, 0x06, 0xff, 0x47, 0x96, 0xd3, 0xc1, 0x2e,
    0x4b, 0x63, 0x9e, 0xd9, 0x13, 0x6a, 0x5f, 0xec, 0x6c, 0x16, 0xd8, 0x88,
    0x4b, 0xdd, 0x99, 0xcf, 0xdc, 0x52, 0x14, 0x56, 0xb0, 0x74, 0x2b, 0x73,
    0x68, 0x68, 0xcf, 0x90, 0xde, 0x09, 0x9a, 0xdb, 0x8d, 0x5f, 0xfd, 0x1d,
    0xef, 0xf3, 0x9b, 0xa4, 0x00, 0x7a, 0xb7, 0x46, 0xce, 0xfd, 0xb2, 0x2d,
    0x7d, 0xf0, 0xe2, 0x25, 0xf5, 0x46, 0x27, 0xdc, 0x65, 0x46, 0x61, 0x31,
    0x72, 0x1b, 0x90, 0xaf, 0x44, 0x53, 0x63, 0xa8, 0x35, 0x8b, 0x9f, 0x60,
    0x76, 0x42, 0xf7, 0x8f, 0xab, 0x0a, 0xb0, 0xf4, 0x3b, 0x71, 0x68, 0xd6,
    0x4b, 0xae, 0x70, 0xd8, 0x82, 0x78, 0x48, 0xd8, 0xef, 0x1e, 0x42, 0x1c,
    0x57, 0x54, 0xdd, 0xf4, 0x2c, 0x25, 0x89, 0xb5, 0xb3};

// RSA-PSS test vectors, pss-vect.txt, Example 3: A 1026-bit RSA Key Pair
// <ftp://ftp.rsasecurity.com/pub/pkcs/pkcs-1/pkcs-1v2-1-vec.zip>
const uint8_t kTestVector3Pkcs8[] = {
    0x30, 0x82, 0x02, 0x76, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
    0x02, 0x60, 0x30, 0x82, 0x02, 0x5c, 0x02, 0x01, 0x00, 0x02, 0x81, 0x81,
    0x02, 0xf2, 0x46, 0xef, 0x45, 0x1e, 0xd3, 0xee, 0xbb, 0x9a, 0x31, 0x02,
    0x00, 0xcc, 0x25, 0x85, 0x9c, 0x04, 0x8e, 0x4b, 0xe7, 0x98, 0x30, 0x29,
    0x91, 0x11, 0x2e, 0xb6, 0x8c, 0xe6, 0xdb, 0x67, 0x4e, 0x28, 0x0d, 0xa2,
    0x1f, 0xed, 0xed, 0x1a, 0xe7, 0x48, 0x80, 0xca, 0x52, 0x2b, 0x18, 0xdb,
    0x24, 0x93, 0x85, 0x01, 0x28, 0x27, 0xc5, 0x15, 0xf0, 0xe4, 0x66, 0xa1,
    0xff, 0xa6, 0x91, 0xd9, 0x81, 0x70, 0x57, 0x4e, 0x9d, 0x0e, 0xad, 0xb0,
    0x87, 0x58, 0x6c, 0xa4, 0x89, 0x33, 0xda, 0x3c, 0xc9, 0x53, 0xd9, 0x5b,
    0xd0, 0xed, 0x50, 0xde, 0x10, 0xdd, 0xcb, 0x67, 0x36, 0x10, 0x7d, 0x6c,
    0x83, 0x1c, 0x7f, 0x66, 0x3e, 0x83, 0x3c, 0xa4, 0xc0, 0x97, 0xe7, 0x00,
    0xce, 0x0f, 0xb9, 0x45, 0xf8, 0x8f, 0xb8, 0x5f, 0xe8, 0xe5, 0xa7, 0x73,
    0x17, 0x25, 0x65, 0xb9, 0x14, 0xa4, 0x71, 0xa4, 0x43, 0x02, 0x03, 0x01,
    0x00, 0x01, 0x02, 0x81, 0x80, 0x65, 0x14, 0x51, 0x73, 0x3b, 0x56, 0xde,
    0x5a, 0xc0, 0xa6, 0x89, 0xa4, 0xae, 0xb6, 0xe6, 0x89, 0x4a, 0x69, 0x01,
    0x4e, 0x07, 0x6c, 0x88, 0xdd, 0x7a, 0x66, 0x7e, 0xab, 0x32, 0x32, 0xbb,
    0xcc, 0xd2, 0xfc, 0x44, 0xba, 0x2f, 0xa9, 0xc3, 0x1d, 0xb4, 0x6f, 0x21,
    0xed, 0xd1, 0xfd, 0xb2, 0x3c, 0x5c, 0x12, 0x8a, 0x5d, 0xa5, 0xba, 0xb9,
    0x1e, 0x7f, 0x95, 0x2b, 0x67, 0x75, 0x9c, 0x7c, 0xff, 0x70, 0x54, 0x15,
    0xac, 0x9f, 0xa0, 0x90, 0x7c, 0x7c, 0xa6, 0x17, 0x8f, 0x66, 0x8f, 0xb9,
    0x48, 0xd8, 0x69, 0xda, 0x4c, 0xc3, 0xb7, 0x35, 0x6f, 0x40, 0x08, 0xdf,
    0xd5, 0x44, 0x9d, 0x32, 0xee, 0x02, 0xd9, 0xa4, 0x77, 0xeb, 0x69, 0xfc,
    0x29, 0x26, 0x6e, 0x5d, 0x90, 0x70, 0x51, 0x23, 0x75, 0xa5, 0x0f, 0xbb,
    0xcc, 0x27, 0xe2, 0x38, 0xad, 0x98, 0x42, 0x5f, 0x6e, 0xbb, 0xf8, 0x89,
    0x91, 0x02, 0x41, 0x01, 0xbd, 0x36, 0xe1, 0x8e, 0xce, 0x4b, 0x0f, 0xdb,
    0x2e, 0x9c, 0x9d, 0x54, 0x8b, 0xd1, 0xa7, 0xd6, 0xe2, 0xc2, 0x1c, 0x6f,
    0xdc, 0x35, 0x07, 0x4a, 0x1d, 0x05, 0xb1, 0xc6, 0xc8, 0xb3, 0xd5, 0x58,
    0xea, 0x26, 0x39, 0xc9, 0xa9, 0xa4, 0x21, 0x68, 0x01, 0x69, 0x31, 0x72,
    0x52, 0x55, 0x8b, 0xd1, 0x48, 0xad, 0x21, 0x5a, 0xac, 0x55, 0x0e, 0x2d,
    0xcf, 0x12, 0xa8, 0x2d, 0x0e, 0xbf, 0xe8, 0x53, 0x02, 0x41, 0x01, 0xb1,
    0xb6, 0x56, 0xad, 0x86, 0xd8, 0xe1, 0x9d, 0x5d, 0xc8, 0x62, 0x92, 0xb3,
    0xa1, 0x92, 0xfd, 0xf6, 0xe0, 0xdd, 0x37, 0x87, 0x7b, 0xad, 0x14, 0x82,
    0x2f, 0xa0, 0x01, 0x90, 0xca, 0xb2, 0x65, 0xf9, 0x0d, 0x3f, 0x02, 0x05,
    0x7b, 0x6f, 0x54, 0xd6, 0xec, 0xb1, 0x44, 0x91, 0xe5, 0xad, 0xea, 0xce,
    0xbc, 0x48, 0xbf, 0x0e, 0xbd, 0x2a, 0x2a, 0xd2, 0x6d, 0x40, 0x2e, 0x54,
    0xf6, 0x16, 0x51, 0x02, 0x40, 0x1f, 0x27, 0x79, 0xfd, 0x2e, 0x3e, 0x5e,
    0x6b, 0xae, 0x05, 0x53, 0x95, 0x18, 0xfb, 0xa0, 0xcd, 0x0e, 0xad, 0x1a,
    0xa4, 0x51, 0x3a, 0x7c, 0xba, 0x18, 0xf1, 0xcf, 0x10, 0xe3, 0xf6, 0x81,
    0x95, 0x69, 0x3d, 0x27, 0x8a, 0x0f, 0x0e, 0xe7, 0x2f, 0x89, 0xf9, 0xbc,
    0x76, 0x0d, 0x80, 0xe2, 0xf9, 0xd0, 0x26, 0x1d, 0x51, 0x65, 0x01, 0xc6,
    0xae, 0x39, 0xf1, 0x4a, 0x47, 0x6c, 0xe2, 0xcc, 0xf5, 0x02, 0x41, 0x01,
    0x1a, 0x0d, 0x36, 0x79, 0x4b, 0x04, 0xa8, 0x54, 0xaa, 0xb4, 0xb2, 0x46,
    0x2d, 0x43, 0x9a, 0x50, 0x46, 0xc9, 0x1d, 0x94, 0x0b, 0x2b, 0xc6, 0xf7,
    0x5b, 0x62, 0x95, 0x6f, 0xef, 0x35, 0xa2, 0xa6, 0xe6, 0x3c, 0x53, 0x09,
    0x81, 0x7f, 0x30, 0x7b, 0xbf, 0xf9, 0xd5, 0x9e, 0x7e, 0x33, 0x1b, 0xd3,
    0x63, 0xf6, 0xd6, 0x68, 0x49, 0xb1, 0x83, 0x46, 0xad, 0xea, 0x16, 0x9f,
    0x0a, 0xe9, 0xae, 0xc1, 0x02, 0x40, 0x0b, 0x30, 0xf0, 0xec, 0xf5, 0x58,
    0x75, 0x2f, 0xb3, 0xa6, 0xce, 0x4b, 0xa2, 0xb8, 0xc6, 0x75, 0xf6, 0x59,
    0xeb, 0xa6, 0xc3, 0x76, 0x58, 0x5a, 0x1b, 0x39, 0x71, 0x2d, 0x03, 0x8a,
    0xe3, 0xd2, 0xb4, 0x6f, 0xcb, 0x41, 0x8a, 0xe1, 0x5d, 0x09, 0x05, 0xda,
    0x64, 0x40, 0xe1, 0x51, 0x3a, 0x30, 0xb9, 0xb7, 0xd6, 0x66, 0x8f, 0xbc,
    0x5e, 0x88, 0xe5, 0xab, 0x7a, 0x17, 0x5e, 0x73, 0xba, 0x35};
const uint8_t kTestVector3Spki[] = {
    0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
    0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30, 0x81,
    0x89, 0x02, 0x81, 0x81, 0x02, 0xf2, 0x46, 0xef, 0x45, 0x1e, 0xd3, 0xee,
    0xbb, 0x9a, 0x31, 0x02, 0x00, 0xcc, 0x25, 0x85, 0x9c, 0x04, 0x8e, 0x4b,
    0xe7, 0x98, 0x30, 0x29, 0x91, 0x11, 0x2e, 0xb6, 0x8c, 0xe6, 0xdb, 0x67,
    0x4e, 0x28, 0x0d, 0xa2, 0x1f, 0xed, 0xed, 0x1a, 0xe7, 0x48, 0x80, 0xca,
    0x52, 0x2b, 0x18, 0xdb, 0x24, 0x93, 0x85, 0x01, 0x28, 0x27, 0xc5, 0x15,
    0xf0, 0xe4, 0x66, 0xa1, 0xff, 0xa6, 0x91, 0xd9, 0x81, 0x70, 0x57, 0x4e,
    0x9d, 0x0e, 0xad, 0xb0, 0x87, 0x58, 0x6c, 0xa4, 0x89, 0x33, 0xda, 0x3c,
    0xc9, 0x53, 0xd9, 0x5b, 0xd0, 0xed, 0x50, 0xde, 0x10, 0xdd, 0xcb, 0x67,
    0x36, 0x10, 0x7d, 0x6c, 0x83, 0x1c, 0x7f, 0x66, 0x3e, 0x83, 0x3c, 0xa4,
    0xc0, 0x97, 0xe7, 0x00, 0xce, 0x0f, 0xb9, 0x45, 0xf8, 0x8f, 0xb8, 0x5f,
    0xe8, 0xe5, 0xa7, 0x73, 0x17, 0x25, 0x65, 0xb9, 0x14, 0xa4, 0x71, 0xa4,
    0x43, 0x02, 0x03, 0x01, 0x00, 0x01};
// RSA-PSS test vectors, pss-vect.txt, Example 3.1
const uint8_t kTestVector3Data[] = {
    0x59, 0x4b, 0x37, 0x33, 0x3b, 0xbb, 0x2c, 0x84, 0x52, 0x4a,
    0x87, 0xc1, 0xa0, 0x1f, 0x75, 0xfc, 0xec, 0x0e, 0x32, 0x56,
    0xf1, 0x08, 0xe3, 0x8d, 0xca, 0x36, 0xd7, 0x0d, 0x00, 0x57};
const uint8_t kTestVector3Sig[] = {
    0x00, 0x88, 0xb1, 0x35, 0xfb, 0x17, 0x94, 0xb6, 0xb9, 0x6c, 0x4a, 0x3e,
    0x67, 0x81, 0x97, 0xf8, 0xca, 0xc5, 0x2b, 0x64, 0xb2, 0xfe, 0x90, 0x7d,
    0x6f, 0x27, 0xde, 0x76, 0x11, 0x24, 0x96, 0x4a, 0x99, 0xa0, 0x1a, 0x88,
    0x27, 0x40, 0xec, 0xfa, 0xed, 0x6c, 0x01, 0xa4, 0x74, 0x64, 0xbb, 0x05,
    0x18, 0x23, 0x13, 0xc0, 0x13, 0x38, 0xa8, 0xcd, 0x09, 0x72, 0x14, 0xcd,
    0x68, 0xca, 0x10, 0x3b, 0xd5, 0x7d, 0x3b, 0xc9, 0xe8, 0x16, 0x21, 0x3e,
    0x61, 0xd7, 0x84, 0xf1, 0x82, 0x46, 0x7a, 0xbf, 0x8a, 0x01, 0xcf, 0x25,
    0x3e, 0x99, 0xa1, 0x56, 0xea, 0xa8, 0xe3, 0xe1, 0xf9, 0x0e, 0x3c, 0x6e,
    0x4e, 0x3a, 0xa2, 0xd8, 0x3e, 0xd0, 0x34, 0x5b, 0x89, 0xfa, 0xfc, 0x9c,
    0x26, 0x07, 0x7c, 0x14, 0xb6, 0xac, 0x51, 0x45, 0x4f, 0xa2, 0x6e, 0x44,
    0x6e, 0x3a, 0x2f, 0x15, 0x3b, 0x2b, 0x16, 0x79, 0x7f};

// RSA-PSS test vectors, pss-vect.txt, Example 4: A 1027-bit RSA Key Pair
// <ftp://ftp.rsasecurity.com/pub/pkcs/pkcs-1/pkcs-1v2-1-vec.zip>
const uint8_t kTestVector4Pkcs8[] = {
    0x30, 0x82, 0x02, 0x78, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
    0x02, 0x62, 0x30, 0x82, 0x02, 0x5e, 0x02, 0x01, 0x00, 0x02, 0x81, 0x81,
    0x05, 0x4a, 0xdb, 0x78, 0x86, 0x44, 0x7e, 0xfe, 0x6f, 0x57, 0xe0, 0x36,
    0x8f, 0x06, 0xcf, 0x52, 0xb0, 0xa3, 0x37, 0x07, 0x60, 0xd1, 0x61, 0xce,
    0xf1, 0x26, 0xb9, 0x1b, 0xe7, 0xf8, 0x9c, 0x42, 0x1b, 0x62, 0xa6, 0xec,
    0x1d, 0xa3, 0xc3, 0x11, 0xd7, 0x5e, 0xd5, 0x0e, 0x0a, 0xb5, 0xff, 0xf3,
    0xfd, 0x33, 0x8a, 0xcc, 0x3a, 0xa8, 0xa4, 0xe7, 0x7e, 0xe2, 0x63, 0x69,
    0xac, 0xb8, 0x1b, 0xa9, 0x00, 0xfa, 0x83, 0xf5, 0x30, 0x0c, 0xf9, 0xbb,
    0x6c, 0x53, 0xad, 0x1d, 0xc8, 0xa1, 0x78, 0xb8, 0x15, 0xdb, 0x42, 0x35,
    0xa9, 0xa9, 0xda, 0x0c, 0x06, 0xde, 0x4e, 0x61, 0x5e, 0xa1, 0x27, 0x7c,
    0xe5, 0x59, 0xe9, 0xc1, 0x08, 0xde, 0x58, 0xc1, 0x4a, 0x81, 0xaa, 0x77,
    0xf5, 0xa6, 0xf8, 0xd1, 0x33, 0x54, 0x94, 0x49, 0x88, 0x48, 0xc8, 0xb9,
    0x59, 0x40, 0x74, 0x0b, 0xe7, 0xbf, 0x7c, 0x37, 0x05, 0x02, 0x03, 0x01,
    0x00, 0x01, 0x02, 0x81, 0x80, 0xfa, 0x04, 0x1f, 0x8c, 0xd9, 0x69, 0x7c,
    0xee, 0xd3, 0x8e, 0xc8, 0xca, 0xa2, 0x75, 0x52, 0x3b, 0x4d, 0xd7, 0x2b,
    0x09, 0xa3, 0x01, 0xd3, 0x54, 0x1d, 0x72, 0xf5, 0xd3, 0x1c, 0x05, 0xcb,
    0xce, 0x2d, 0x69, 0x83, 0xb3, 0x61, 0x83, 0xaf, 0x10, 0x69, 0x0b, 0xd4,
    0x6c, 0x46, 0x13, 0x1e, 0x35, 0x78, 0x94, 0x31, 0xa5, 0x56, 0x77, 0x1d,
    0xd0, 0x04, 0x9b, 0x57, 0x46, 0x1b, 0xf0, 0x60, 0xc1, 0xf6, 0x84, 0x72,
    0xe8, 0xa6, 0x7c, 0x25, 0xf3, 0x57, 0xe5, 0xb6, 0xb4, 0x73, 0x8f, 0xa5,
    0x41, 0xa7, 0x30, 0x34, 0x6b, 0x4a, 0x07, 0x64, 0x9a, 0x2d, 0xfa, 0x80,
    0x6a, 0x69, 0xc9, 0x75, 0xb6, 0xab, 0xa6, 0x46, 0x78, 0xac, 0xc7, 0xf5,
    0x91, 0x3e, 0x89, 0xc6, 0x22, 0xf2, 0xd8, 0xab, 0xb1, 0xe3, 0xe3, 0x25,
    0x54, 0xe3, 0x9d, 0xf9, 0x4b, 0xa6, 0x0c, 0x00, 0x2e, 0x38, 0x7d, 0x90,
    0x11, 0x02, 0x41, 0x02, 0x92, 0x32, 0x33, 0x6d, 0x28, 0x38, 0x94, 0x5d,
    0xba, 0x9d, 0xd7, 0x72, 0x3f, 0x4e, 0x62, 0x4a, 0x05, 0xf7, 0x37, 0x5b,
    0x92, 0x7a, 0x87, 0xab, 0xe6, 0xa8, 0x93, 0xa1, 0x65, 0x8f, 0xd4, 0x9f,
    0x47, 0xf6, 0xc7, 0xb0, 0xfa, 0x59, 0x6c, 0x65, 0xfa, 0x68, 0xa2, 0x3f,
    0x0a, 0xb4, 0x32, 0x96, 0x2d, 0x18, 0xd4, 0x34, 0x3b, 0xd6, 0xfd, 0x67,
    0x1a, 0x5e, 0xa8, 0xd1, 0x48, 0x41, 0x39, 0x95, 0x02, 0x41, 0x02, 0x0e,
    0xf5, 0xef, 0xe7, 0xc5, 0x39, 0x4a, 0xed, 0x22, 0x72, 0xf7, 0xe8, 0x1a,
    0x74, 0xf4, 0xc0, 0x2d, 0x14, 0x58, 0x94, 0xcb, 0x1b, 0x3c, 0xab, 0x23,
    0xa9, 0xa0, 0x71, 0x0a, 0x2a, 0xfc, 0x7e, 0x33, 0x29, 0xac, 0xbb, 0x74,
    0x3d, 0x01, 0xf6, 0x80, 0xc4, 0xd0, 0x2a, 0xfb, 0x4c, 0x8f, 0xde, 0x7e,
    0x20, 0x93, 0x08, 0x11, 0xbb, 0x2b, 0x99, 0x57, 0x88, 0xb5, 0xe8, 0x72,
    0xc2, 0x0b, 0xb1, 0x02, 0x41, 0x02, 0x6e, 0x7e, 0x28, 0x01, 0x0e, 0xcf,
    0x24, 0x12, 0xd9, 0x52, 0x3a, 0xd7, 0x04, 0x64, 0x7f, 0xb4, 0xfe, 0x9b,
    0x66, 0xb1, 0xa6, 0x81, 0x58, 0x1b, 0x0e, 0x15, 0x55, 0x3a, 0x89, 0xb1,
    0x54, 0x28, 0x28, 0x89, 0x8f, 0x27, 0x24, 0x3e, 0xba, 0xb4, 0x5f, 0xf5,
    0xe1, 0xac, 0xb9, 0xd4, 0xdf, 0x1b, 0x05, 0x1f, 0xbc, 0x62, 0x82, 0x4d,
    0xbc, 0x6f, 0x6c, 0x93, 0x26, 0x1a, 0x78, 0xb9, 0xa7, 0x59, 0x02, 0x41,
    0x01, 0x2d, 0xdc, 0xc8, 0x6e, 0xf6, 0x55, 0x99, 0x8c, 0x39, 0xdd, 0xae,
    0x11, 0x71, 0x86, 0x69, 0xe5, 0xe4, 0x6c, 0xf1, 0x49, 0x5b, 0x07, 0xe1,
    0x3b, 0x10, 0x14, 0xcd, 0x69, 0xb3, 0xaf, 0x68, 0x30, 0x4a, 0xd2, 0xa6,
    0xb6, 0x43, 0x21, 0xe7, 0x8b, 0xf3, 0xbb, 0xca, 0x9b, 0xb4, 0x94, 0xe9,
    0x1d, 0x45, 0x17, 0x17, 0xe2, 0xd9, 0x75, 0x64, 0xc6, 0x54, 0x94, 0x65,
    0xd0, 0x20, 0x5c, 0xf4, 0x21, 0x02, 0x41, 0x01, 0x06, 0x00, 0xc4, 0xc2,
    0x18, 0x47, 0x45, 0x9f, 0xe5, 0x76, 0x70, 0x3e, 0x2e, 0xbe, 0xca, 0xe8,
    0xa5, 0x09, 0x4e, 0xe6, 0x3f, 0x53, 0x6b, 0xf4, 0xac, 0x68, 0xd3, 0xc1,
    0x3e, 0x5e, 0x4f, 0x12, 0xac, 0x5c, 0xc1, 0x0a, 0xb6, 0xa2, 0xd0, 0x5a,
    0x19, 0x92, 0x14, 0xd1, 0x82, 0x47, 0x47, 0xd5, 0x51, 0x90, 0x96, 0x36,
    0xb7, 0x74, 0xc2, 0x2c, 0xac, 0x0b, 0x83, 0x75, 0x99, 0xab, 0xcc, 0x75};
const uint8_t kTestVector4Spki[] = {
    0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
    0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30, 0x81,
    0x89, 0x02, 0x81, 0x81, 0x05, 0x4a, 0xdb, 0x78, 0x86, 0x44, 0x7e, 0xfe,
    0x6f, 0x57, 0xe0, 0x36, 0x8f, 0x06, 0xcf, 0x52, 0xb0, 0xa3, 0x37, 0x07,
    0x60, 0xd1, 0x61, 0xce, 0xf1, 0x26, 0xb9, 0x1b, 0xe7, 0xf8, 0x9c, 0x42,
    0x1b, 0x62, 0xa6, 0xec, 0x1d, 0xa3, 0xc3, 0x11, 0xd7, 0x5e, 0xd5, 0x0e,
    0x0a, 0xb5, 0xff, 0xf3, 0xfd, 0x33, 0x8a, 0xcc, 0x3a, 0xa8, 0xa4, 0xe7,
    0x7e, 0xe2, 0x63, 0x69, 0xac, 0xb8, 0x1b, 0xa9, 0x00, 0xfa, 0x83, 0xf5,
    0x30, 0x0c, 0xf9, 0xbb, 0x6c, 0x53, 0xad, 0x1d, 0xc8, 0xa1, 0x78, 0xb8,
    0x15, 0xdb, 0x42, 0x35, 0xa9, 0xa9, 0xda, 0x0c, 0x06, 0xde, 0x4e, 0x61,
    0x5e, 0xa1, 0x27, 0x7c, 0xe5, 0x59, 0xe9, 0xc1, 0x08, 0xde, 0x58, 0xc1,
    0x4a, 0x81, 0xaa, 0x77, 0xf5, 0xa6, 0xf8, 0xd1, 0x33, 0x54, 0x94, 0x49,
    0x88, 0x48, 0xc8, 0xb9, 0x59, 0x40, 0x74, 0x0b, 0xe7, 0xbf, 0x7c, 0x37,
    0x05, 0x02, 0x03, 0x01, 0x00, 0x01};
// RSA-PSS test vectors, pss-vect.txt, Example 4.1
const uint8_t kTestVector4Data[] = {0x9f, 0xb0, 0x3b, 0x82,
                                    0x7c, 0x82, 0x17, 0xd9};
const uint8_t kTestVector4Sig[] = {
    0x03, 0x23, 0xd5, 0xb7, 0xbf, 0x20, 0xba, 0x45, 0x39, 0x28, 0x9a, 0xe4,
    0x52, 0xae, 0x42, 0x97, 0x08, 0x0f, 0xef, 0xf4, 0x51, 0x84, 0x23, 0xff,
    0x48, 0x11, 0xa8, 0x17, 0x83, 0x7e, 0x7d, 0x82, 0xf1, 0x83, 0x6c, 0xdf,
    0xab, 0x54, 0x51, 0x4f, 0xf0, 0x88, 0x7b, 0xdd, 0xee, 0xbf, 0x40, 0xbf,
    0x99, 0xb0, 0x47, 0xab, 0xc3, 0xec, 0xfa, 0x6a, 0x37, 0xa3, 0xef, 0x00,
    0xf4, 0xa0, 0xc4, 0xa8, 0x8a, 0xae, 0x09, 0x04, 0xb7, 0x45, 0xc8, 0x46,
    0xc4, 0x10, 0x7e, 0x87, 0x97, 0x72, 0x3e, 0x8a, 0xc8, 0x10, 0xd9, 0xe3,
    0xd9, 0x5d, 0xfa, 0x30, 0xff, 0x49, 0x66, 0xf4, 0xd7, 0x5d, 0x13, 0x76,
    0x8d, 0x20, 0x85, 0x7f, 0x2b, 0x14, 0x06, 0xf2, 0x64, 0xcf, 0xe7, 0x5e,
    0x27, 0xd7, 0x65, 0x2f, 0x4b, 0x5e, 0xd3, 0x57, 0x5f, 0x28, 0xa7, 0x02,
    0xf8, 0xc4, 0xed, 0x9c, 0xf9, 0xb2, 0xd4, 0x49, 0x48};

// RSA-PSS test vectors, pss-vect.txt, Example 5: A 1028-bit RSA Key Pair
// <ftp://ftp.rsasecurity.com/pub/pkcs/pkcs-1/pkcs-1v2-1-vec.zip>
const uint8_t kTestVector5Pkcs8[] = {
    0x30, 0x82, 0x02, 0x78, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
    0x02, 0x62, 0x30, 0x82, 0x02, 0x5e, 0x02, 0x01, 0x00, 0x02, 0x81, 0x81,
    0x0d, 0x10, 0xf6, 0x61, 0xf2, 0x99, 0x40, 0xf5, 0xed, 0x39, 0xaa, 0x26,
    0x09, 0x66, 0xde, 0xb4, 0x78, 0x43, 0x67, 0x9d, 0x2b, 0x6f, 0xb2, 0x5b,
    0x3d, 0xe3, 0x70, 0xf3, 0xac, 0x7c, 0x19, 0x91, 0x63, 0x91, 0xfd, 0x25,
    0xfb, 0x52, 0x7e, 0xbf, 0xa6, 0xa4, 0xb4, 0xdf, 0x45, 0xa1, 0x75, 0x9d,
    0x99, 0x6c, 0x4b, 0xb4, 0xeb, 0xd1, 0x88, 0x28, 0xc4, 0x4f, 0xc5, 0x2d,
    0x01, 0x91, 0x87, 0x17, 0x40, 0x52, 0x5f, 0x47, 0xa4, 0xb0, 0xcc, 0x8d,
    0xa3, 0x25, 0xed, 0x8a, 0xa6, 0x76, 0xb0, 0xd0, 0xf6, 0x26, 0xe0, 0xa7,
    0x7f, 0x07, 0x69, 0x21, 0x70, 0xac, 0xac, 0x80, 0x82, 0xf4, 0x2f, 0xaa,
    0x7d, 0xc7, 0xcd, 0x12, 0x3e, 0x73, 0x0e, 0x31, 0xa8, 0x79, 0x85, 0x20,
    0x4c, 0xab, 0xcb, 0xe6, 0x67, 0x0d, 0x43, 0xa2, 0xdd, 0x2b, 0x2d, 0xde,
    0xf5, 0xe0, 0x53, 0x92, 0xfc, 0x21, 0x3b, 0xc5, 0x07, 0x02, 0x03, 0x01,
    0x00, 0x01, 0x02, 0x81, 0x81, 0x03, 0xce, 0x08, 0xb1, 0x04, 0xff, 0xf3,
    0x96, 0xa9, 0x79, 0xbd, 0x3e, 0x4e, 0x46, 0x92, 0x5b, 0x63, 0x19, 0xdd,
    0xb6, 0x3a, 0xcb, 0xcf, 0xd8, 0x19, 0xf1, 0x7d, 0x16, 0xb8, 0x07, 0x7b,
    0x3a, 0x87, 0x10, 0x1f, 0xf3, 0x4b, 0x77, 0xfe, 0x48, 0xb8, 0xb2, 0x05,
    0xa9, 0x6e, 0x91, 0x51, 0xba, 0x8e, 0xce, 0xa6, 0x4d, 0x0c, 0xce, 0x7b,
    0x23, 0xc3, 0xe6, 0xa6, 0xb8, 0x30, 0x58, 0xbc, 0x49, 0xda, 0xe8, 0x16,
    0xae, 0x73, 0x6d, 0xb5, 0xa4, 0x70, 0x8e, 0x2a, 0xd4, 0x35, 0x23, 0x2b,
    0x56, 0x7f, 0x90, 0x96, 0xce, 0x59, 0xff, 0x28, 0x06, 0x1e, 0x79, 0xab,
    0x1c, 0x02, 0xd7, 0x17, 0xe6, 0xb2, 0x3c, 0xea, 0x6d, 0xb8, 0xeb, 0x51,
    0x92, 0xfa, 0x7c, 0x1e, 0xab, 0x22, 0x7d, 0xba, 0x74, 0x62, 0x1c, 0x45,
    0x60, 0x18, 0x96, 0xee, 0xf1, 0x37, 0x92, 0xc8, 0x44, 0x0b, 0xeb, 0x15,
    0xaa, 0xc1, 0x02, 0x41, 0x03, 0xf2, 0xf3, 0x31, 0xf4, 0x14, 0x2d, 0x4f,
    0x24, 0xb4, 0x3a, 0xa1, 0x02, 0x79, 0xa8, 0x96, 0x52, 0xd4, 0xe7, 0x53,
    0x72, 0x21, 0xa1, 0xa7, 0xb2, 0xa2, 0x5d, 0xeb, 0x55, 0x1e, 0x5d, 0xe9,
    0xac, 0x49, 0x74, 0x11, 0xc2, 0x27, 0xa9, 0x4e, 0x45, 0xf9, 0x1c, 0x2d,
    0x1c, 0x13, 0xcc, 0x04, 0x6c, 0xf4, 0xce, 0x14, 0xe3, 0x2d, 0x05, 0x87,
    0x34, 0x21, 0x0d, 0x44, 0xa8, 0x7e, 0xe1, 0xb7, 0x3f, 0x02, 0x41, 0x03,
    0x4f, 0x09, 0x0d, 0x73, 0xb5, 0x58, 0x03, 0x03, 0x0c, 0xf0, 0x36, 0x1a,
    0x5d, 0x80, 0x81, 0xbf, 0xb7, 0x9f, 0x85, 0x15, 0x23, 0xfe, 0xac, 0x0a,
    0x21, 0x24, 0xd0, 0x8d, 0x40, 0x13, 0xff, 0x08, 0x48, 0x77, 0x71, 0xa8,
    0x70, 0xd0, 0x47, 0x9d, 0xc0, 0x68, 0x6c, 0x62, 0xf7, 0x71, 0x8d, 0xfe,
    0xcf, 0x02, 0x4b, 0x17, 0xc9, 0x26, 0x76, 0x78, 0x05, 0x91, 0x71, 0x33,
    0x9c, 0xc0, 0x08, 0x39, 0x02, 0x41, 0x02, 0xaa, 0x66, 0x3a, 0xdb, 0xf5,
    0x1a, 0xb8, 0x87, 0xa0, 0x18, 0xcb, 0x42, 0x6e, 0x78, 0xbc, 0x2f, 0xe1,
    0x82, 0xdc, 0xb2, 0xf7, 0xbc, 0xb5, 0x04, 0x41, 0xd1, 0x7f, 0xdf, 0x0f,
    0x06, 0x79, 0x8b, 0x50, 0x71, 0xc6, 0xe2, 0xf5, 0xfe, 0xb4, 0xd5, 0x4a,
    0xd8, 0x18, 0x23, 0x11, 0xc1, 0xef, 0x62, 0xd4, 0xc4, 0x9f, 0x18, 0xd1,
    0xf5, 0x1f, 0x54, 0xb2, 0xd2, 0xcf, 0xfb, 0xa4, 0xda, 0x1b, 0xe5, 0x02,
    0x41, 0x02, 0xbb, 0xe7, 0x06, 0x07, 0x8b, 0x5c, 0x0b, 0x39, 0x15, 0x12,
    0xd4, 0x11, 0xdb, 0x1b, 0x19, 0x9b, 0x5a, 0x56, 0x64, 0xb8, 0x40, 0x42,
    0xea, 0xd3, 0x7f, 0xe9, 0x94, 0xae, 0x72, 0xb9, 0x53, 0x2d, 0xfb, 0xfb,
    0x3e, 0x9e, 0x69, 0x81, 0xa0, 0xfb, 0xb8, 0x06, 0x51, 0x31, 0x41, 0xb7,
    0xc2, 0x16, 0x3f, 0xe5, 0x6c, 0x39, 0x5e, 0x4b, 0xfa, 0xee, 0x57, 0xe3,
    0x83, 0x3f, 0x9b, 0x91, 0x8d, 0xf9, 0x02, 0x40, 0x02, 0x42, 0xb6, 0xcd,
    0x00, 0xd3, 0x0a, 0x76, 0x7a, 0xee, 0x9a, 0x89, 0x8e, 0xad, 0x45, 0x3c,
    0x8e, 0xae, 0xa6, 0x3d, 0x50, 0x0b, 0x7d, 0x1e, 0x00, 0x71, 0x3e, 0xda,
    0xe5, 0x1c, 0xe3, 0x6b, 0x23, 0xb6, 0x64, 0xdf, 0x26, 0xe6, 0x3e, 0x26,
    0x6e, 0xc8, 0xf7, 0x6e, 0x6e, 0x63, 0xed, 0x1b, 0xa4, 0x1e, 0xb0, 0x33,
    0xb1, 0x20, 0xf7, 0xea, 0x52, 0x12, 0xae, 0x21, 0xa9, 0x8f, 0xbc, 0x16};
const uint8_t kTestVector5Spki[] = {
    0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
    0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30, 0x81,
    0x89, 0x02, 0x81, 0x81, 0x0d, 0x10, 0xf6, 0x61, 0xf2, 0x99, 0x40, 0xf5,
    0xed, 0x39, 0xaa, 0x26, 0x09, 0x66, 0xde, 0xb4, 0x78, 0x43, 0x67, 0x9d,
    0x2b, 0x6f, 0xb2, 0x5b, 0x3d, 0xe3, 0x70, 0xf3, 0xac, 0x7c, 0x19, 0x91,
    0x63, 0x91, 0xfd, 0x25, 0xfb, 0x52, 0x7e, 0xbf, 0xa6, 0xa4, 0xb4, 0xdf,
    0x45, 0xa1, 0x75, 0x9d, 0x99, 0x6c, 0x4b, 0xb4, 0xeb, 0xd1, 0x88, 0x28,
    0xc4, 0x4f, 0xc5, 0x2d, 0x01, 0x91, 0x87, 0x17, 0x40, 0x52, 0x5f, 0x47,
    0xa4, 0xb0, 0xcc, 0x8d, 0xa3, 0x25, 0xed, 0x8a, 0xa6, 0x76, 0xb0, 0xd0,
    0xf6, 0x26, 0xe0, 0xa7, 0x7f, 0x07, 0x69, 0x21, 0x70, 0xac, 0xac, 0x80,
    0x82, 0xf4, 0x2f, 0xaa, 0x7d, 0xc7, 0xcd, 0x12, 0x3e, 0x73, 0x0e, 0x31,
    0xa8, 0x79, 0x85, 0x20, 0x4c, 0xab, 0xcb, 0xe6, 0x67, 0x0d, 0x43, 0xa2,
    0xdd, 0x2b, 0x2d, 0xde, 0xf5, 0xe0, 0x53, 0x92, 0xfc, 0x21, 0x3b, 0xc5,
    0x07, 0x02, 0x03, 0x01, 0x00, 0x01};
// RSA-PSS test vectors, pss-vect.txt, Example 5.1
const uint8_t kTestVector5Data[] = {
    0x30, 0xc7, 0xd5, 0x57, 0x45, 0x8b, 0x43, 0x6d, 0xec, 0xfd, 0xc1, 0x4d,
    0x06, 0xcb, 0x7b, 0x96, 0xb0, 0x67, 0x18, 0xc4, 0x8d, 0x7d, 0xe5, 0x74,
    0x82, 0xa8, 0x68, 0xae, 0x7f, 0x06, 0x58, 0x70, 0xa6, 0x21, 0x65, 0x06,
    0xd1, 0x1b, 0x77, 0x93, 0x23, 0xdf, 0xdf, 0x04, 0x6c, 0xf5, 0x77, 0x51,
    0x29, 0x13, 0x4b, 0x4d, 0x56, 0x89, 0xe4, 0xd9, 0xc0, 0xce, 0x1e, 0x12,
    0xd7, 0xd4, 0xb0, 0x6c, 0xb5, 0xfc, 0x58, 0x20, 0xde, 0xcf, 0xa4, 0x1b,
    0xaf, 0x59, 0xbf, 0x25, 0x7b, 0x32, 0xf0, 0x25, 0xb7, 0x67, 0x9b, 0x44,
    0x5b, 0x94, 0x99, 0xc9, 0x25, 0x55, 0x14, 0x58, 0x85, 0x99, 0x2f, 0x1b,
    0x76, 0xf8, 0x48, 0x91, 0xee, 0x4d, 0x3b, 0xe0, 0xf5, 0x15, 0x0f, 0xd5,
    0x90, 0x1e, 0x3a, 0x4c, 0x8e, 0xd4, 0x3f, 0xd3, 0x6b, 0x61, 0xd0, 0x22,
    0xe6, 0x5a, 0xd5, 0x00, 0x8d, 0xbf, 0x33, 0x29, 0x3c, 0x22, 0xbf, 0xbf,
    0xd0, 0x73, 0x21, 0xf0, 0xf1, 0xd5, 0xfa, 0x9f, 0xdf, 0x00, 0x14, 0xc2,
    0xfc, 0xb0, 0x35, 0x8a, 0xad, 0x0e, 0x35, 0x4b, 0x0d, 0x29};
const uint8_t kTestVector5Sig[] = {
    0x0b, 0xa3, 0x73, 0xf7, 0x6e, 0x09, 0x21, 0xb7, 0x0a, 0x8f, 0xbf, 0xe6,
    0x22, 0xf0, 0xbf, 0x77, 0xb2, 0x8a, 0x3d, 0xb9, 0x8e, 0x36, 0x10, 0x51,
    0xc3, 0xd7, 0xcb, 0x92, 0xad, 0x04, 0x52, 0x91, 0x5a, 0x4d, 0xe9, 0xc0,
    0x17, 0x22, 0xf6, 0x82, 0x3e, 0xeb, 0x6a, 0xdf, 0x7e, 0x0c, 0xa8, 0x29,
    0x0f, 0x5d, 0xe3, 0xe5, 0x49, 0x89, 0x0a, 0xc2, 0xa3, 0xc5, 0x95, 0x0a,
    0xb2, 0x17, 0xba, 0x58, 0x59, 0x08, 0x94, 0x95, 0x2d, 0xe9, 0x6f, 0x8d,
    0xf1, 0x11, 0xb2, 0x57, 0x52, 0x15, 0xda, 0x6c, 0x16, 0x15, 0x90, 0xc7,
    0x45, 0xbe, 0x61, 0x24, 0x76, 0xee, 0x57, 0x8e, 0xd3, 0x84, 0xab, 0x33,
    0xe3, 0xec, 0xe9, 0x74, 0x81, 0xa2, 0x52, 0xf5, 0xc7, 0x9a, 0x98, 0xb5,
    0x53, 0x2a, 0xe0, 0x0c, 0xdd, 0x62, 0xf2, 0xec, 0xc0, 0xcd, 0x1b, 0xae,
    0xfe, 0x80, 0xd8, 0x0b, 0x96, 0x21, 0x93, 0xec, 0x1d};

// RSA-PSS test vectors, pss-vect.txt, Example 6: A 1029-bit RSA Key Pair
// <ftp://ftp.rsasecurity.com/pub/pkcs/pkcs-1/pkcs-1v2-1-vec.zip>
const uint8_t kTestVector6Pkcs8[] = {
    0x30, 0x82, 0x02, 0x79, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
    0x02, 0x63, 0x30, 0x82, 0x02, 0x5f, 0x02, 0x01, 0x00, 0x02, 0x81, 0x81,
    0x16, 0x4c, 0xa3, 0x1c, 0xff, 0x60, 0x9f, 0x3a, 0x0e, 0x71, 0x01, 0xb0,
    0x39, 0xf2, 0xe4, 0xfe, 0x6d, 0xd3, 0x75, 0x19, 0xab, 0x98, 0x59, 0x8d,
    0x17, 0x9e, 0x17, 0x49, 0x96, 0x59, 0x80, 0x71, 0xf4, 0x7d, 0x3a, 0x04,
    0x55, 0x91, 0x58, 0xd7, 0xbe, 0x37, 0x3c, 0xf1, 0xaa, 0x53, 0xf0, 0xaa,
    0x6e, 0xf0, 0x90, 0x39, 0xe5, 0x67, 0x8c, 0x2a, 0x4c, 0x63, 0x90, 0x05,
    0x14, 0xc8, 0xc4, 0xf8, 0xaa, 0xed, 0x5d, 0xe1, 0x2a, 0x5f, 0x10, 0xb0,
    0x9c, 0x31, 0x1a, 0xf8, 0xc0, 0xff, 0xb5, 0xb7, 0xa2, 0x97, 0xf2, 0xef,
    0xc6, 0x3b, 0x8d, 0x6b, 0x05, 0x10, 0x93, 0x1f, 0x0b, 0x98, 0xe4, 0x8b,
    0xf5, 0xfc, 0x6e, 0xc4, 0xe7, 0xb8, 0xdb, 0x1f, 0xfa, 0xeb, 0x08, 0xc3,
    0x8e, 0x02, 0xad, 0xb8, 0xf0, 0x3a, 0x48, 0x22, 0x9c, 0x99, 0xe9, 0x69,
    0x43, 0x1f, 0x61, 0xcb, 0x8c, 0x4d, 0xc6, 0x98, 0xd1, 0x02, 0x03, 0x01,
    0x00, 0x01, 0x02, 0x81, 0x81, 0x03, 0xb6, 0x64, 0xee, 0x3b, 0x75, 0x66,
    0x72, 0x3f, 0xc6, 0xea, 0xf2, 0x8a, 0xbb, 0x43, 0x0a, 0x39, 0x80, 0xf1,
    0x12, 0x6c, 0x81, 0xde, 0x8a, 0xd7, 0x09, 0xea, 0xb3, 0x9a, 0xc9, 0xdc,
    0xd0, 0xb1, 0x55, 0x0b, 0x37, 0x29, 0xd8, 0x70, 0x68, 0xe9, 0x52, 0x00,
    0x9d, 0xf5, 0x44, 0x53, 0x4c, 0x1f, 0x50, 0x82, 0x9a, 0x78, 0xf4, 0x59,
    0x1e, 0xb8, 0xfd, 0x57, 0x14, 0x04, 0x26, 0xa6, 0xbb, 0x04, 0x05, 0xb6,
    0xa6, 0xf5, 0x1a, 0x57, 0xd9, 0x26, 0x7b, 0x7b, 0xbc, 0x65, 0x33, 0x91,
    0xa6, 0x99, 0xa2, 0xa9, 0x0d, 0xac, 0x8a, 0xe2, 0x26, 0xbc, 0xc6, 0x0f,
    0xa8, 0xcd, 0x93, 0x4c, 0x73, 0xc7, 0xb0, 0x3b, 0x1f, 0x6b, 0x81, 0x81,
    0x58, 0x63, 0x18, 0x38, 0xa8, 0x61, 0x2e, 0x6e, 0x6e, 0xa9, 0x2b, 0xe2,
    0x4f, 0x83, 0x24, 0xfa, 0xf5, 0xb1, 0xfd, 0x85, 0x87, 0x22, 0x52, 0x67,
    0xba, 0x6f, 0x02, 0x41, 0x04, 0xf0, 0x54, 0x8c, 0x96, 0x26, 0xab, 0x1e,
    0xbf, 0x12, 0x44, 0x93, 0x47, 0x41, 0xd9, 0x9a, 0x06, 0x22, 0x0e, 0xfa,
    0x2a, 0x58, 0x56, 0xaa, 0x0e, 0x75, 0x73, 0x0b, 0x2e, 0xc9, 0x6a, 0xdc,
    0x86, 0xbe, 0x89, 0x4f, 0xa2, 0x80, 0x3b, 0x53, 0xa5, 0xe8, 0x5d, 0x27,
    0x6a, 0xcb, 0xd2, 0x9a, 0xb8, 0x23, 0xf8, 0x0a, 0x73, 0x91, 0xbb, 0x54,
    0xa5, 0x05, 0x16, 0x72, 0xfb, 0x04, 0xee, 0xb5, 0x43, 0x02, 0x41, 0x04,
    0x83, 0xe0, 0xae, 0x47, 0x91, 0x55, 0x87, 0x74, 0x3f, 0xf3, 0x45, 0x36,
    0x2b, 0x55, 0x5d, 0x39, 0x62, 0xd9, 0x8b, 0xb6, 0xf1, 0x5f, 0x84, 0x8b,
    0x4c, 0x92, 0xb1, 0x77, 0x1c, 0xa8, 0xed, 0x10, 0x7d, 0x8d, 0x3e, 0xe6,
    0x5e, 0xc4, 0x45, 0x17, 0xdd, 0x0f, 0xaa, 0x48, 0x1a, 0x38, 0x7e, 0x90,
    0x2f, 0x7a, 0x2e, 0x74, 0x7c, 0x26, 0x9e, 0x7e, 0xa4, 0x44, 0x80, 0xbc,
    0x53, 0x8b, 0x8e, 0x5b, 0x02, 0x41, 0x03, 0xa8, 0xe8, 0xae, 0xa9, 0x92,
    0x0c, 0x1a, 0xa3, 0xb2, 0xf0, 0xd8, 0x46, 0xe4, 0xb8, 0x50, 0xd8, 0x1c,
    0xa3, 0x06, 0xa5, 0x1c, 0x83, 0x54, 0x4f, 0x94, 0x9f, 0x64, 0xf9, 0x0d,
    0xcf, 0x3f, 0x8e, 0x26, 0x61, 0xf0, 0x7e, 0x56, 0x12, 0x20, 0xa1, 0x80,
    0x38, 0x8f, 0xbe, 0x27, 0x3e, 0x70, 0xe2, 0xe5, 0xdc, 0xa8, 0x3a, 0x0e,
    0x13, 0x48, 0xdd, 0x64, 0x90, 0xc7, 0x31, 0xd6, 0xec, 0xe1, 0xab, 0x02,
    0x41, 0x01, 0x35, 0xbd, 0xcd, 0xb6, 0x0b, 0xf2, 0x19, 0x7c, 0x43, 0x6e,
    0xd3, 0x4b, 0x32, 0xcd, 0x8b, 0x4f, 0xc7, 0x77, 0x78, 0x83, 0x2b, 0xa7,
    0x67, 0x03, 0x55, 0x1f, 0xb2, 0x42, 0xb3, 0x01, 0x69, 0x95, 0x93, 0xaf,
    0x77, 0xfd, 0x8f, 0xc3, 0x94, 0xa8, 0x52, 0x6a, 0xd2, 0x3c, 0xc4, 0x1a,
    0x03, 0x80, 0x6b, 0xd8, 0x97, 0xfe, 0x4b, 0x0e, 0xa6, 0x46, 0x55, 0x8a,
    0xad, 0xdc, 0xc9, 0x9e, 0x8a, 0x25, 0x02, 0x41, 0x03, 0x04, 0xc0, 0x3d,
    0x9c, 0x73, 0x65, 0x03, 0xa9, 0x84, 0xab, 0xbd, 0x9b, 0xa2, 0x23, 0x01,
    0x40, 0x7c, 0x4a, 0x2a, 0xb1, 0xdd, 0x85, 0x76, 0x64, 0x81, 0xb6, 0x0d,
    0x45, 0x40, 0x11, 0x52, 0xe6, 0x92, 0xbe, 0x14, 0xf4, 0x12, 0x1d, 0x9a,
    0xa3, 0xfd, 0x6e, 0x0b, 0x4d, 0x1d, 0x3a, 0x97, 0x35, 0x38, 0xa3, 0x1d,
    0x42, 0xee, 0x6e, 0x1e, 0x5e, 0xf6, 0x20, 0x23, 0x1a, 0x2b, 0xba, 0xf3,
    0x5f};
const uint8_t kTestVector6Spki[] = {
    0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
    0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30, 0x81,
    0x89, 0x02, 0x81, 0x81, 0x16, 0x4c, 0xa3, 0x1c, 0xff, 0x60, 0x9f, 0x3a,
    0x0e, 0x71, 0x01, 0xb0, 0x39, 0xf2, 0xe4, 0xfe, 0x6d, 0xd3, 0x75, 0x19,
    0xab, 0x98, 0x59, 0x8d, 0x17, 0x9e, 0x17, 0x49, 0x96, 0x59, 0x80, 0x71,
    0xf4, 0x7d, 0x3a, 0x04, 0x55, 0x91, 0x58, 0xd7, 0xbe, 0x37, 0x3c, 0xf1,
    0xaa, 0x53, 0xf0, 0xaa, 0x6e, 0xf0, 0x90, 0x39, 0xe5, 0x67, 0x8c, 0x2a,
    0x4c, 0x63, 0x90, 0x05, 0x14, 0xc8, 0xc4, 0xf8, 0xaa, 0xed, 0x5d, 0xe1,
    0x2a, 0x5f, 0x10, 0xb0, 0x9c, 0x31, 0x1a, 0xf8, 0xc0, 0xff, 0xb5, 0xb7,
    0xa2, 0x97, 0xf2, 0xef, 0xc6, 0x3b, 0x8d, 0x6b, 0x05, 0x10, 0x93, 0x1f,
    0x0b, 0x98, 0xe4, 0x8b, 0xf5, 0xfc, 0x6e, 0xc4, 0xe7, 0xb8, 0xdb, 0x1f,
    0xfa, 0xeb, 0x08, 0xc3, 0x8e, 0x02, 0xad, 0xb8, 0xf0, 0x3a, 0x48, 0x22,
    0x9c, 0x99, 0xe9, 0x69, 0x43, 0x1f, 0x61, 0xcb, 0x8c, 0x4d, 0xc6, 0x98,
    0xd1, 0x02, 0x03, 0x01, 0x00, 0x01};
// RSA-PSS test vectors, pss-vect.txt, Example 6.1
const uint8_t kTestVector6Data[] = {
    0x0a, 0x20, 0xb7, 0x74, 0xad, 0xdc, 0x2f, 0xa5, 0x12, 0x45, 0xed,
    0x7c, 0xb9, 0xda, 0x60, 0x9e, 0x50, 0xca, 0xc6, 0x63, 0x6a, 0x52,
    0x54, 0x3f, 0x97, 0x45, 0x8e, 0xed, 0x73, 0x40, 0xf8, 0xd5, 0x3f,
    0xfc, 0x64, 0x91, 0x8f, 0x94, 0x90, 0x78, 0xee, 0x03, 0xef, 0x60,
    0xd4, 0x2b, 0x5f, 0xec, 0x24, 0x60, 0x50, 0xbd, 0x55, 0x05, 0xcd,
    0x8c, 0xb5, 0x97, 0xba, 0xd3, 0xc4, 0xe7, 0x13, 0xb0, 0xef, 0x30,
    0x64, 0x4e, 0x76, 0xad, 0xab, 0xb0, 0xde, 0x01, 0xa1, 0x56, 0x1e,
    0xfb, 0x25, 0x51, 0x58, 0xc7, 0x4f, 0xc8, 0x01, 0xe6, 0xe9, 0x19,
    0xe5, 0x81, 0xb4, 0x6f, 0x0f, 0x0d, 0xdd, 0x08, 0xe4, 0xf3, 0x4c,
    0x78, 0x10, 0xb5, 0xed, 0x83, 0x18, 0xf9, 0x1d, 0x7c, 0x8c};
const uint8_t kTestVector6Sig[] = {
    0x04, 0xc0, 0xcf, 0xac, 0xec, 0x04, 0xe5, 0xba, 0xdb, 0xec, 0xe1, 0x59,
    0xa5, 0xa1, 0x10, 0x3f, 0x69, 0xb3, 0xf3, 0x2b, 0xa5, 0x93, 0xcb, 0x4c,
    0xc4, 0xb1, 0xb7, 0xab, 0x45, 0x59, 0x16, 0xa9, 0x6a, 0x27, 0xcd, 0x26,
    0x78, 0xea, 0x0f, 0x46, 0xba, 0x37, 0xf7, 0xfc, 0x9c, 0x86, 0x32, 0x5f,
    0x29, 0x73, 0x3b, 0x38, 0x9f, 0x1d, 0x97, 0xf4, 0x3e, 0x72, 0x01, 0xc0,
    0xf3, 0x48, 0xfc, 0x45, 0xfe, 0x42, 0x89, 0x23, 0x35, 0x36, 0x2e, 0xee,
    0x01, 0x8b, 0x5b, 0x16, 0x1f, 0x2f, 0x93, 0x93, 0x03, 0x12, 0x25, 0xc7,
    0x13, 0x01, 0x2a, 0x57, 0x6b, 0xc8, 0x8e, 0x23, 0x05, 0x24, 0x89, 0x86,
    0x8d, 0x90, 0x10, 0xcb, 0xf0, 0x33, 0xec, 0xc5, 0x68, 0xe8, 0xbc, 0x15,
    0x2b, 0xdc, 0x59, 0xd5, 0x60, 0xe4, 0x12, 0x91, 0x91, 0x5d, 0x28, 0x56,
    0x52, 0x08, 0xe2, 0x2a, 0xee, 0xc9, 0xef, 0x85, 0xd1};

// RSA-PSS test vectors, pss-vect.txt, Example 7: A 1030-bit RSA Key Pair
// <ftp://ftp.rsasecurity.com/pub/pkcs/pkcs-1/pkcs-1v2-1-vec.zip>
const uint8_t kTestVector7Pkcs8[] = {
    0x30, 0x82, 0x02, 0x77, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
    0x02, 0x61, 0x30, 0x82, 0x02, 0x5d, 0x02, 0x01, 0x00, 0x02, 0x81, 0x81,
    0x37, 0xc9, 0xda, 0x4a, 0x66, 0xc8, 0xc4, 0x08, 0xb8, 0xda, 0x27, 0xd0,
    0xc9, 0xd7, 0x9f, 0x8c, 0xcb, 0x1e, 0xaf, 0xc1, 0xd2, 0xfe, 0x48, 0x74,
    0x6d, 0x94, 0x0b, 0x7c, 0x4e, 0xf5, 0xde, 0xe1, 0x8a, 0xd1, 0x26, 0x47,
    0xce, 0xfa, 0xa0, 0xc4, 0xb3, 0x18, 0x8b, 0x22, 0x1c, 0x51, 0x53, 0x86,
    0x75, 0x9b, 0x93, 0xf0, 0x20, 0x24, 0xb2, 0x5a, 0xb9, 0x24, 0x2f, 0x83,
    0x57, 0xd8, 0xf3, 0xfd, 0x49, 0x64, 0x0e, 0xe5, 0xe6, 0x43, 0xea, 0xf6,
    0xc6, 0x4d, 0xee, 0xfa, 0x70, 0x89, 0x72, 0x7c, 0x8f, 0xf0, 0x39, 0x93,
    0x33, 0x39, 0x15, 0xc6, 0xef, 0x21, 0xbf, 0x59, 0x75, 0xb6, 0xe5, 0x0d,
    0x11, 0x8b, 0x51, 0x00, 0x8e, 0xc3, 0x3e, 0x9f, 0x01, 0xa0, 0xa5, 0x45,
    0xa1, 0x0a, 0x83, 0x6a, 0x43, 0xdd, 0xbc, 0xa9, 0xd8, 0xb5, 0xc5, 0xd3,
    0x54, 0x80, 0x22, 0xd7, 0x06, 0x4e, 0xa2, 0x9a, 0xb3, 0x02, 0x03, 0x01,
    0x00, 0x01, 0x02, 0x81, 0x80, 0x3b, 0xed, 0x99, 0x90, 0x52, 0xd9, 0x57,
    0xbc, 0x06, 0xd6, 0x51, 0xee, 0xf6, 0xe3, 0xa9, 0x80, 0x94, 0xb1, 0x62,
    0x1b, 0xd3, 0x8b, 0x54, 0x49, 0xbd, 0x6c, 0x4a, 0xea, 0x3d, 0xe7, 0xe0,
    0x84, 0x67, 0x9a, 0x44, 0x84, 0xde, 0xd2, 0x5b, 0xe0, 0xf0, 0x82, 0x6c,
    0xf3, 0x37, 0x78, 0x25, 0x41, 0x4b, 0x14, 0xd4, 0xd6, 0x1d, 0xb1, 0x4d,
    0xe6, 0x26, 0xfb, 0xb8, 0x0e, 0x5f, 0x4f, 0xae, 0xc9, 0x56, 0xf9, 0xa0,
    0xa2, 0xd2, 0x4f, 0x99, 0x57, 0x63, 0x80, 0xf0, 0x84, 0xeb, 0x62, 0xe4,
    0x6a, 0x57, 0xd5, 0x54, 0x27, 0x8b, 0x53, 0x56, 0x26, 0x19, 0x3c, 0xe0,
    0x20, 0x60, 0x57, 0x5e, 0xb6, 0x6c, 0x57, 0x98, 0xd3, 0x6f, 0x6c, 0x5d,
    0x40, 0xfb, 0x00, 0xd8, 0x09, 0xb4, 0x2a, 0x73, 0x10, 0x2c, 0x1c, 0x74,
    0xee, 0x95, 0xbd, 0x71, 0x42, 0x0f, 0xff, 0xef, 0x63, 0x18, 0xb5, 0x2c,
    0x29, 0x02, 0x41, 0x07, 0xee, 0xfb, 0x42, 0x4b, 0x0e, 0x3a, 0x40, 0xe4,
    0x20, 0x8e, 0xe5, 0xaf, 0xb2, 0x80, 0xb2, 0x23, 0x17, 0x30, 0x81, 0x14,
    0xdd, 0xe0, 0xb4, 0xb6, 0x4f, 0x73, 0x01, 0x84, 0xec, 0x68, 0xda, 0x6c,
    0xe2, 0x86, 0x7a, 0x9f, 0x48, 0xed, 0x77, 0x26, 0xd5, 0xe2, 0x61, 0x4e,
    0xd0, 0x4a, 0x54, 0x10, 0x73, 0x6c, 0x8c, 0x71, 0x4e, 0xe7, 0x02, 0x47,
    0x42, 0x98, 0xc6, 0x29, 0x2a, 0xf0, 0x75, 0x35, 0x02, 0x41, 0x07, 0x08,
    0x30, 0xdb, 0xf9, 0x47, 0xea, 0xc0, 0x22, 0x8d, 0xe2, 0x63, 0x14, 0xb5,
    0x9b, 0x66, 0x99, 0x4c, 0xc6, 0x0e, 0x83, 0x60, 0xe7, 0x5d, 0x38, 0x76,
    0x29, 0x8f, 0x8f, 0x8a, 0x7d, 0x14, 0x1d, 0xa0, 0x64, 0xe5, 0xca, 0x02,
    0x6a, 0x97, 0x3e, 0x28, 0xf2, 0x54, 0x73, 0x8c, 0xee, 0x66, 0x9c, 0x72,
    0x1b, 0x03, 0x4c, 0xb5, 0xf8, 0xe2, 0x44, 0xda, 0xdd, 0x7c, 0xd1, 0xe1,
    0x59, 0xd5, 0x47, 0x02, 0x41, 0x05, 0x24, 0xd2, 0x0c, 0x3d, 0x95, 0xcf,
    0xf7, 0x5a, 0xf2, 0x31, 0x34, 0x83, 0x22, 0x7d, 0x87, 0x02, 0x71, 0x7a,
    0xa5, 0x76, 0xde, 0x15, 0x5f, 0x96, 0x05, 0x15, 0x50, 0x1a, 0xdb, 0x1d,
    0x70, 0xe1, 0xc0, 0x4d, 0xe9, 0x1b, 0x75, 0xb1, 0x61, 0xdb, 0xf0, 0x39,
    0x83, 0x56, 0x12, 0x7e, 0xde, 0xda, 0x7b, 0xbc, 0x19, 0xa3, 0x2d, 0xc1,
    0x62, 0x1c, 0xc9, 0xf5, 0x3c, 0x26, 0x5d, 0x0c, 0xe3, 0x31, 0x02, 0x41,
    0x05, 0xf9, 0x84, 0xa1, 0xf2, 0x3c, 0x93, 0x8d, 0x6a, 0x0e, 0x89, 0x72,
    0x4b, 0xcf, 0x3d, 0xd9, 0x3f, 0x99, 0x46, 0x92, 0x60, 0x37, 0xfe, 0x7c,
    0x6b, 0x13, 0xa2, 0x9e, 0x52, 0x84, 0x85, 0x5f, 0x89, 0x08, 0x95, 0x91,
    0xd4, 0x40, 0x97, 0x56, 0x27, 0xbf, 0x5c, 0x9e, 0x3a, 0x8b, 0x5c, 0xa7,
    0x9c, 0x77, 0x2a, 0xd2, 0x73, 0xe4, 0x0d, 0x32, 0x1a, 0xf4, 0xa6, 0xc9,
    0x7d, 0xfd, 0xed, 0x78, 0xd3, 0x02, 0x40, 0xdd, 0xd9, 0x18, 0xad, 0xad,
    0xa2, 0x9d, 0xca, 0xb9, 0x81, 0xff, 0x9a, 0xcb, 0xa4, 0x25, 0x70, 0x23,
    0xc0, 0x9a, 0x38, 0x01, 0xcc, 0xce, 0x09, 0x8c, 0xe2, 0x68, 0xf8, 0x55,
    0xd0, 0xdf, 0x57, 0x0c, 0xd6, 0xe7, 0xb9, 0xb1, 0x4b, 0xd9, 0xa5, 0xa9,
    0x25, 0x4c, 0xbc, 0x31, 0x5b, 0xe6, 0xf8, 0xba, 0x1e, 0x25, 0x46, 0xdd,
    0xd5, 0x69, 0xc5, 0xea, 0x19, 0xee, 0xd8, 0x35, 0x3b, 0xde, 0x5e};
const uint8_t kTestVector7Spki[] = {
    0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
    0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30, 0x81,
    0x89, 0x02, 0x81, 0x81, 0x37, 0xc9, 0xda, 0x4a, 0x66, 0xc8, 0xc4, 0x08,
    0xb8, 0xda, 0x27, 0xd0, 0xc9, 0xd7, 0x9f, 0x8c, 0xcb, 0x1e, 0xaf, 0xc1,
    0xd2, 0xfe, 0x48, 0x74, 0x6d, 0x94, 0x0b, 0x7c, 0x4e, 0xf5, 0xde, 0xe1,
    0x8a, 0xd1, 0x26, 0x47, 0xce, 0xfa, 0xa0, 0xc4, 0xb3, 0x18, 0x8b, 0x22,
    0x1c, 0x51, 0x53, 0x86, 0x75, 0x9b, 0x93, 0xf0, 0x20, 0x24, 0xb2, 0x5a,
    0xb9, 0x24, 0x2f, 0x83, 0x57, 0xd8, 0xf3, 0xfd, 0x49, 0x64, 0x0e, 0xe5,
    0xe6, 0x43, 0xea, 0xf6, 0xc6, 0x4d, 0xee, 0xfa, 0x70, 0x89, 0x72, 0x7c,
    0x8f, 0xf0, 0x39, 0x93, 0x33, 0x39, 0x15, 0xc6, 0xef, 0x21, 0xbf, 0x59,
    0x75, 0xb6, 0xe5, 0x0d, 0x11, 0x8b, 0x51, 0x00, 0x8e, 0xc3, 0x3e, 0x9f,
    0x01, 0xa0, 0xa5, 0x45, 0xa1, 0x0a, 0x83, 0x6a, 0x43, 0xdd, 0xbc, 0xa9,
    0xd8, 0xb5, 0xc5, 0xd3, 0x54, 0x80, 0x22, 0xd7, 0x06, 0x4e, 0xa2, 0x9a,
    0xb3, 0x02, 0x03, 0x01, 0x00, 0x01};
// RSA-PSS test vectors, pss-vect.txt, Example 7.1
const uint8_t kTestVector7Data[] = {
    0x9e, 0xad, 0x0e, 0x01, 0x94, 0x56, 0x40, 0x67, 0x4e, 0xb4, 0x1c, 0xad,
    0x43, 0x5e, 0x23, 0x74, 0xea, 0xef, 0xa8, 0xad, 0x71, 0x97, 0xd9, 0x79,
    0x13, 0xc4, 0x49, 0x57, 0xd8, 0xd8, 0x3f, 0x40, 0xd7, 0x6e, 0xe6, 0x0e,
    0x39, 0xbf, 0x9c, 0x0f, 0x9e, 0xaf, 0x30, 0x21, 0x42, 0x1a, 0x07, 0x4d,
    0x1a, 0xde, 0x96, 0x2c, 0x6e, 0x9d, 0x3d, 0xc3, 0xbb, 0x17, 0x4f, 0xe4,
    0xdf, 0xe6, 0x52, 0xb0, 0x91, 0x15, 0x49, 0x5b, 0x8f, 0xd2, 0x79, 0x41,
    0x74, 0x02, 0x0a, 0x06, 0x02, 0xb5, 0xca, 0x51, 0x84, 0x8c, 0xfc, 0x96,
    0xce, 0x5e, 0xb5, 0x7f, 0xc0, 0xa2, 0xad, 0xc1, 0xdd, 0xa3, 0x6a, 0x7c,
    0xc4, 0x52, 0x64, 0x1a, 0x14, 0x91, 0x1b, 0x37, 0xe4, 0x5b, 0xfa, 0x11,
    0xda, 0xa5, 0xc7, 0xec, 0xdb, 0x74, 0xf6, 0xd0, 0x10, 0x0d, 0x1d, 0x3e,
    0x39, 0xe7, 0x52, 0x80, 0x0e, 0x20, 0x33, 0x97, 0xde, 0x02, 0x33, 0x07,
    0x7b, 0x9a, 0x88, 0x85, 0x55, 0x37, 0xfa, 0xe9, 0x27, 0xf9, 0x24, 0x38,
    0x0d, 0x78, 0x0f, 0x98, 0xe1, 0x8d, 0xcf, 0xf3, 0x9c, 0x5e, 0xa7, 0x41,
    0xb1, 0x7d, 0x6f, 0xdd, 0x18, 0x85, 0xbc, 0x9d, 0x58, 0x14, 0x82, 0xd7,
    0x71, 0xce, 0xb5, 0x62, 0xd7, 0x8a, 0x8b, 0xf8, 0x8f, 0x0c, 0x75, 0xb1,
    0x13, 0x63, 0xe5, 0xe3, 0x6c, 0xd4, 0x79, 0xce, 0xb0, 0x54, 0x5f, 0x9d,
    0xa8, 0x42, 0x03, 0xe0, 0xe6, 0xe5, 0x08, 0x37, 0x5c, 0xc9, 0xe8, 0x44,
    0xb8, 0x8b, 0x7a, 0xc7, 0xa0, 0xa2, 0x01, 0xea, 0x0f, 0x1b, 0xee, 0x9a,
    0x2c, 0x57, 0x79, 0x20, 0xca, 0x02, 0xc0, 0x1b, 0x9d, 0x83, 0x20, 0xe9,
    0x74, 0xa5, 0x6f, 0x4e, 0xfb, 0x57, 0x63, 0xb9, 0x62, 0x55, 0xab, 0xbf,
    0x80, 0x37, 0xbf, 0x18, 0x02, 0xcf, 0x01, 0x8f, 0x56, 0x37, 0x94, 0x93,
    0xe5, 0x69, 0xa9};
const uint8_t kTestVector7Sig[] = {
    0x18, 0x7f, 0x39, 0x07, 0x23, 0xc8, 0x90, 0x25, 0x91, 0xf0, 0x15, 0x4b,
    0xae, 0x6d, 0x4e, 0xcb, 0xff, 0xe0, 0x67, 0xf0, 0xe8, 0xb7, 0x95, 0x47,
    0x6e, 0xa4, 0xf4, 0xd5, 0x1c, 0xcc, 0x81, 0x05, 0x20, 0xbb, 0x3c, 0xa9,
    0xbc, 0xa7, 0xd0, 0xb1, 0xf2, 0xea, 0x8a, 0x17, 0xd8, 0x73, 0xfa, 0x27,
    0x57, 0x0a, 0xcd, 0x64, 0x2e, 0x38, 0x08, 0x56, 0x1c, 0xb9, 0xe9, 0x75,
    0xcc, 0xfd, 0x80, 0xb2, 0x3d, 0xc5, 0x77, 0x1c, 0xdb, 0x33, 0x06, 0xa5,
    0xf2, 0x31, 0x59, 0xda, 0xcb, 0xd3, 0xaa, 0x2d, 0xb9, 0x3d, 0x46, 0xd7,
    0x66, 0xe0, 0x9e, 0xd1, 0x5d, 0x90, 0x0a, 0xd8, 0x97, 0xa8, 0xd2, 0x74,
    0xdc, 0x26, 0xb4, 0x7e, 0x99, 0x4a, 0x27, 0xe9, 0x7e, 0x22, 0x68, 0xa7,
    0x66, 0x53, 0x3a, 0xe4, 0xb5, 0xe4, 0x2a, 0x2f, 0xca, 0xf7, 0x55, 0xc1,
    0xc4, 0x79, 0x4b, 0x29, 0x4c, 0x60, 0x55, 0x58, 0x23};

// RSA-PSS test vectors, pss-vect.txt, Example 8: A 1031-bit RSA Key Pair
// <ftp://ftp.rsasecurity.com/pub/pkcs/pkcs-1/pkcs-1v2-1-vec.zip>
const uint8_t kTestVector8Pkcs8[] = {
    0x30, 0x82, 0x02, 0x78, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
    0x02, 0x62, 0x30, 0x82, 0x02, 0x5e, 0x02, 0x01, 0x00, 0x02, 0x81, 0x81,
    0x49, 0x53, 0x70, 0xa1, 0xfb, 0x18, 0x54, 0x3c, 0x16, 0xd3, 0x63, 0x1e,
    0x31, 0x63, 0x25, 0x5d, 0xf6, 0x2b, 0xe6, 0xee, 0xe8, 0x90, 0xd5, 0xf2,
    0x55, 0x09, 0xe4, 0xf7, 0x78, 0xa8, 0xea, 0x6f, 0xbb, 0xbc, 0xdf, 0x85,
    0xdf, 0xf6, 0x4e, 0x0d, 0x97, 0x20, 0x03, 0xab, 0x36, 0x81, 0xfb, 0xba,
    0x6d, 0xd4, 0x1f, 0xd5, 0x41, 0x82, 0x9b, 0x2e, 0x58, 0x2d, 0xe9, 0xf2,
    0xa4, 0xa4, 0xe0, 0xa2, 0xd0, 0x90, 0x0b, 0xef, 0x47, 0x53, 0xdb, 0x3c,
    0xee, 0x0e, 0xe0, 0x6c, 0x7d, 0xfa, 0xe8, 0xb1, 0xd5, 0x3b, 0x59, 0x53,
    0x21, 0x8f, 0x9c, 0xce, 0xea, 0x69, 0x5b, 0x08, 0x66, 0x8e, 0xde, 0xaa,
    0xdc, 0xed, 0x94, 0x63, 0xb1, 0xd7, 0x90, 0xd5, 0xeb, 0xf2, 0x7e, 0x91,
    0x15, 0xb4, 0x6c, 0xad, 0x4d, 0x9a, 0x2b, 0x8e, 0xfa, 0xb0, 0x56, 0x1b,
    0x08, 0x10, 0x34, 0x47, 0x39, 0xad, 0xa0, 0x73, 0x3f, 0x02, 0x03, 0x01,
    0x00, 0x01, 0x02, 0x81, 0x80, 0x6c, 0x66, 0xff, 0xe9, 0x89, 0x80, 0xc3,
    0x8f, 0xcd, 0xea, 0xb5, 0x15, 0x98, 0x98, 0x83, 0x61, 0x65, 0xf4, 0xb4,
    0xb8, 0x17, 0xc4, 0xf6, 0xa8, 0xd4, 0x86, 0xee, 0x4e, 0xa9, 0x13, 0x0f,
    0xe9, 0xb9, 0x09, 0x2b, 0xd1, 0x36, 0xd1, 0x84, 0xf9, 0x5f, 0x50, 0x4a,
    0x60, 0x7e, 0xac, 0x56, 0x58, 0x46, 0xd2, 0xfd, 0xd6, 0x59, 0x7a, 0x89,
    0x67, 0xc7, 0x39, 0x6e, 0xf9, 0x5a, 0x6e, 0xee, 0xbb, 0x45, 0x78, 0xa6,
    0x43, 0x96, 0x6d, 0xca, 0x4d, 0x8e, 0xe3, 0xde, 0x84, 0x2d, 0xe6, 0x32,
    0x79, 0xc6, 0x18, 0x15, 0x9c, 0x1a, 0xb5, 0x4a, 0x89, 0x43, 0x7b, 0x6a,
    0x61, 0x20, 0xe4, 0x93, 0x0a, 0xfb, 0x52, 0xa4, 0xba, 0x6c, 0xed, 0x8a,
    0x49, 0x47, 0xac, 0x64, 0xb3, 0x0a, 0x34, 0x97, 0xcb, 0xe7, 0x01, 0xc2,
    0xd6, 0x26, 0x6d, 0x51, 0x72, 0x19, 0xad, 0x0e, 0xc6, 0xd3, 0x47, 0xdb,
    0xe9, 0x02, 0x41, 0x08, 0xda, 0xd7, 0xf1, 0x13, 0x63, 0xfa, 0xa6, 0x23,
    0xd5, 0xd6, 0xd5, 0xe8, 0xa3, 0x19, 0x32, 0x8d, 0x82, 0x19, 0x0d, 0x71,
    0x27, 0xd2, 0x84, 0x6c, 0x43, 0x9b, 0x0a, 0xb7, 0x26, 0x19, 0xb0, 0xa4,
    0x3a, 0x95, 0x32, 0x0e, 0x4e, 0xc3, 0x4f, 0xc3, 0xa9, 0xce, 0xa8, 0x76,
    0x42, 0x23, 0x05, 0xbd, 0x76, 0xc5, 0xba, 0x7b, 0xe9, 0xe2, 0xf4, 0x10,
    0xc8, 0x06, 0x06, 0x45, 0xa1, 0xd2, 0x9e, 0xdb, 0x02, 0x41, 0x08, 0x47,
    0xe7, 0x32, 0x37, 0x6f, 0xc7, 0x90, 0x0f, 0x89, 0x8e, 0xa8, 0x2e, 0xb2,
    0xb0, 0xfc, 0x41, 0x85, 0x65, 0xfd, 0xae, 0x62, 0xf7, 0xd9, 0xec, 0x4c,
    0xe2, 0x21, 0x7b, 0x97, 0x99, 0x0d, 0xd2, 0x72, 0xdb, 0x15, 0x7f, 0x99,
    0xf6, 0x3c, 0x0d, 0xcb, 0xb9, 0xfb, 0xac, 0xdb, 0xd4, 0xc4, 0xda, 0xdb,
    0x6d, 0xf6, 0x77, 0x56, 0x35, 0x8c, 0xa4, 0x17, 0x48, 0x25, 0xb4, 0x8f,
    0x49, 0x70, 0x6d, 0x02, 0x41, 0x05, 0xc2, 0xa8, 0x3c, 0x12, 0x4b, 0x36,
    0x21, 0xa2, 0xaa, 0x57, 0xea, 0x2c, 0x3e, 0xfe, 0x03, 0x5e, 0xff, 0x45,
    0x60, 0xf3, 0x3d, 0xde, 0xbb, 0x7a, 0xda, 0xb8, 0x1f, 0xce, 0x69, 0xa0,
    0xc8, 0xc2, 0xed, 0xc1, 0x65, 0x20, 0xdd, 0xa8, 0x3d, 0x59, 0xa2, 0x3b,
    0xe8, 0x67, 0x96, 0x3a, 0xc6, 0x5f, 0x2c, 0xc7, 0x10, 0xbb, 0xcf, 0xb9,
    0x6e, 0xe1, 0x03, 0xde, 0xb7, 0x71, 0xd1, 0x05, 0xfd, 0x85, 0x02, 0x41,
    0x04, 0xca, 0xe8, 0xaa, 0x0d, 0x9f, 0xaa, 0x16, 0x5c, 0x87, 0xb6, 0x82,
    0xec, 0x14, 0x0b, 0x8e, 0xd3, 0xb5, 0x0b, 0x24, 0x59, 0x4b, 0x7a, 0x3b,
    0x2c, 0x22, 0x0b, 0x36, 0x69, 0xbb, 0x81, 0x9f, 0x98, 0x4f, 0x55, 0x31,
    0x0a, 0x1a, 0xe7, 0x82, 0x36, 0x51, 0xd4, 0xa0, 0x2e, 0x99, 0x44, 0x79,
    0x72, 0x59, 0x51, 0x39, 0x36, 0x34, 0x34, 0xe5, 0xe3, 0x0a, 0x7e, 0x7d,
    0x24, 0x15, 0x51, 0xe1, 0xb9, 0x02, 0x41, 0x07, 0xd3, 0xe4, 0x7b, 0xf6,
    0x86, 0x60, 0x0b, 0x11, 0xac, 0x28, 0x3c, 0xe8, 0x8d, 0xbb, 0x3f, 0x60,
    0x51, 0xe8, 0xef, 0xd0, 0x46, 0x80, 0xe4, 0x4c, 0x17, 0x1e, 0xf5, 0x31,
    0xb8, 0x0b, 0x2b, 0x7c, 0x39, 0xfc, 0x76, 0x63, 0x20, 0xe2, 0xcf, 0x15,
    0xd8, 0xd9, 0x98, 0x20, 0xe9, 0x6f, 0xf3, 0x0d, 0xc6, 0x96, 0x91, 0x83,
    0x9c, 0x4b, 0x40, 0xd7, 0xb0, 0x6e, 0x45, 0x30, 0x7d, 0xc9, 0x1f, 0x3f};
const uint8_t kTestVector8Spki[] = {
    0x30, 0x81, 0x9f, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
    0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8d, 0x00, 0x30, 0x81,
    0x89, 0x02, 0x81, 0x81, 0x49, 0x53, 0x70, 0xa1, 0xfb, 0x18, 0x54, 0x3c,
    0x16, 0xd3, 0x63, 0x1e, 0x31, 0x63, 0x25, 0x5d, 0xf6, 0x2b, 0xe6, 0xee,
    0xe8, 0x90, 0xd5, 0xf2, 0x55, 0x09, 0xe4, 0xf7, 0x78, 0xa8, 0xea, 0x6f,
    0xbb, 0xbc, 0xdf, 0x85, 0xdf, 0xf6, 0x4e, 0x0d, 0x97, 0x20, 0x03, 0xab,
    0x36, 0x81, 0xfb, 0xba, 0x6d, 0xd4, 0x1f, 0xd5, 0x41, 0x82, 0x9b, 0x2e,
    0x58, 0x2d, 0xe9, 0xf2, 0xa4, 0xa4, 0xe0, 0xa2, 0xd0, 0x90, 0x0b, 0xef,
    0x47, 0x53, 0xdb, 0x3c, 0xee, 0x0e, 0xe0, 0x6c, 0x7d, 0xfa, 0xe8, 0xb1,
    0xd5, 0x3b, 0x59, 0x53, 0x21, 0x8f, 0x9c, 0xce, 0xea, 0x69, 0x5b, 0x08,
    0x66, 0x8e, 0xde, 0xaa, 0xdc, 0xed, 0x94, 0x63, 0xb1, 0xd7, 0x90, 0xd5,
    0xeb, 0xf2, 0x7e, 0x91, 0x15, 0xb4, 0x6c, 0xad, 0x4d, 0x9a, 0x2b, 0x8e,
    0xfa, 0xb0, 0x56, 0x1b, 0x08, 0x10, 0x34, 0x47, 0x39, 0xad, 0xa0, 0x73,
    0x3f, 0x02, 0x03, 0x01, 0x00, 0x01};
// RSA-PSS test vectors, pss-vect.txt, Example 8.1
const uint8_t kTestVector8Data[] = {
    0x81, 0x33, 0x2f, 0x4b, 0xe6, 0x29, 0x48, 0x41, 0x5e, 0xa1, 0xd8, 0x99,
    0x79, 0x2e, 0xea, 0xcf, 0x6c, 0x6e, 0x1d, 0xb1, 0xda, 0x8b, 0xe1, 0x3b,
    0x5c, 0xea, 0x41, 0xdb, 0x2f, 0xed, 0x46, 0x70, 0x92, 0xe1, 0xff, 0x39,
    0x89, 0x14, 0xc7, 0x14, 0x25, 0x97, 0x75, 0xf5, 0x95, 0xf8, 0x54, 0x7f,
    0x73, 0x56, 0x92, 0xa5, 0x75, 0xe6, 0x92, 0x3a, 0xf7, 0x8f, 0x22, 0xc6,
    0x99, 0x7d, 0xdb, 0x90, 0xfb, 0x6f, 0x72, 0xd7, 0xbb, 0x0d, 0xd5, 0x74,
    0x4a, 0x31, 0xde, 0xcd, 0x3d, 0xc3, 0x68, 0x58, 0x49, 0x83, 0x6e, 0xd3,
    0x4a, 0xec, 0x59, 0x63, 0x04, 0xad, 0x11, 0x84, 0x3c, 0x4f, 0x88, 0x48,
    0x9f, 0x20, 0x97, 0x35, 0xf5, 0xfb, 0x7f, 0xda, 0xf7, 0xce, 0xc8, 0xad,
    0xdc, 0x58, 0x18, 0x16, 0x8f, 0x88, 0x0a, 0xcb, 0xf4, 0x90, 0xd5, 0x10,
    0x05, 0xb7, 0xa8, 0xe8, 0x4e, 0x43, 0xe5, 0x42, 0x87, 0x97, 0x75, 0x71,
    0xdd, 0x99, 0xee, 0xa4, 0xb1, 0x61, 0xeb, 0x2d, 0xf1, 0xf5, 0x10, 0x8f,
    0x12, 0xa4, 0x14, 0x2a, 0x83, 0x32, 0x2e, 0xdb, 0x05, 0xa7, 0x54, 0x87,
    0xa3, 0x43, 0x5c, 0x9a, 0x78, 0xce, 0x53, 0xed, 0x93, 0xbc, 0x55, 0x08,
    0x57, 0xd7, 0xa9, 0xfb};
const uint8_t kTestVector8Sig[] = {
    0x02, 0x62, 0xac, 0x25, 0x4b, 0xfa, 0x77, 0xf3, 0xc1, 0xac, 0xa2, 0x2c,
    0x51, 0x79, 0xf8, 0xf0, 0x40, 0x42, 0x2b, 0x3c, 0x5b, 0xaf, 0xd4, 0x0a,
    0x8f, 0x21, 0xcf, 0x0f, 0xa5, 0xa6, 0x67, 0xcc, 0xd5, 0x99, 0x3d, 0x42,
    0xdb, 0xaf, 0xb4, 0x09, 0xc5, 0x20, 0xe2, 0x5f, 0xce, 0x2b, 0x1e, 0xe1,
    0xe7, 0x16, 0x57, 0x7f, 0x1e, 0xfa, 0x17, 0xf3, 0xda, 0x28, 0x05, 0x2f,
    0x40, 0xf0, 0x41, 0x9b, 0x23, 0x10, 0x6d, 0x78, 0x45, 0xaa, 0xf0, 0x11,
    0x25, 0xb6, 0x98, 0xe7, 0xa4, 0xdf, 0xe9, 0x2d, 0x39, 0x67, 0xbb, 0x00,
    0xc4, 0xd0, 0xd3, 0x5b, 0xa3, 0x55, 0x2a, 0xb9, 0xa8, 0xb3, 0xee, 0xf0,
    0x7c, 0x7f, 0xec, 0xdb, 0xc5, 0x42, 0x4a, 0xc4, 0xdb, 0x1e, 0x20, 0xcb,
    0x37, 0xd0, 0xb2, 0x74, 0x47, 0x69, 0x94, 0x0e, 0xa9, 0x07, 0xe1, 0x7f,
    0xbb, 0xca, 0x67, 0x3b, 0x20, 0x52, 0x23, 0x80, 0xc5};

// RSA-PSS test vectors, pss-vect.txt, Example 9: A 1536-bit RSA Key Pair
// <ftp://ftp.rsasecurity.com/pub/pkcs/pkcs-1/pkcs-1v2-1-vec.zip>
const uint8_t kTestVector9Pkcs8[] = {
    0x30, 0x82, 0x03, 0x92, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
    0x03, 0x7c, 0x30, 0x82, 0x03, 0x78, 0x02, 0x01, 0x00, 0x02, 0x81, 0xc0,
    0xe6, 0xbd, 0x69, 0x2a, 0xc9, 0x66, 0x45, 0x79, 0x04, 0x03, 0xfd, 0xd0,
    0xf5, 0xbe, 0xb8, 0xb9, 0xbf, 0x92, 0xed, 0x10, 0x00, 0x7f, 0xc3, 0x65,
    0x04, 0x64, 0x19, 0xdd, 0x06, 0xc0, 0x5c, 0x5b, 0x5b, 0x2f, 0x48, 0xec,
    0xf9, 0x89, 0xe4, 0xce, 0x26, 0x91, 0x09, 0x97, 0x9c, 0xbb, 0x40, 0xb4,
    0xa0, 0xad, 0x24, 0xd2, 0x24, 0x83, 0xd1, 0xee, 0x31, 0x5a, 0xd4, 0xcc,
    0xb1, 0x53, 0x42, 0x68, 0x35, 0x26, 0x91, 0xc5, 0x24, 0xf6, 0xdd, 0x8e,
    0x6c, 0x29, 0xd2, 0x24, 0xcf, 0x24, 0x69, 0x73, 0xae, 0xc8, 0x6c, 0x5b,
    0xf6, 0xb1, 0x40, 0x1a, 0x85, 0x0d, 0x1b, 0x9a, 0xd1, 0xbb, 0x8c, 0xbc,
    0xec, 0x47, 0xb0, 0x6f, 0x0f, 0x8c, 0x7f, 0x45, 0xd3, 0xfc, 0x8f, 0x31,
    0x92, 0x99, 0xc5, 0x43, 0x3d, 0xdb, 0xc2, 0xb3, 0x05, 0x3b, 0x47, 0xde,
    0xd2, 0xec, 0xd4, 0xa4, 0xca, 0xef, 0xd6, 0x14, 0x83, 0x3d, 0xc8, 0xbb,
    0x62, 0x2f, 0x31, 0x7e, 0xd0, 0x76, 0xb8, 0x05, 0x7f, 0xe8, 0xde, 0x3f,
    0x84, 0x48, 0x0a, 0xd5, 0xe8, 0x3e, 0x4a, 0x61, 0x90, 0x4a, 0x4f, 0x24,
    0x8f, 0xb3, 0x97, 0x02, 0x73, 0x57, 0xe1, 0xd3, 0x0e, 0x46, 0x31, 0x39,
    0x81, 0x5c, 0x6f, 0xd4, 0xfd, 0x5a, 0xc5, 0xb8, 0x17, 0x2a, 0x45, 0x23,
    0x0e, 0xcb, 0x63, 0x18, 0xa0, 0x4f, 0x14, 0x55, 0xd8, 0x4e, 0x5a, 0x8b,
    0x02, 0x03, 0x01, 0x00, 0x01, 0x02, 0x81, 0xc0, 0x6a, 0x7f, 0xd8, 0x4f,
    0xb8, 0x5f, 0xad, 0x07, 0x3b, 0x34, 0x40, 0x6d, 0xb7, 0x4f, 0x8d, 0x61,
    0xa6, 0xab, 0xc1, 0x21, 0x96, 0xa9, 0x61, 0xdd, 0x79, 0x56, 0x5e, 0x9d,
    0xa6, 0xe5, 0x18, 0x7b, 0xce, 0x2d, 0x98, 0x02, 0x50, 0xf7, 0x35, 0x95,
    0x75, 0x35, 0x92, 0x70, 0xd9, 0x15, 0x90, 0xbb, 0x0e, 0x42, 0x7c, 0x71,
    0x46, 0x0b, 0x55, 0xd5, 0x14, 0x10, 0xb1, 0x91, 0xbc, 0xf3, 0x09, 0xfe,
    0xa1, 0x31, 0xa9, 0x2c, 0x8e, 0x70, 0x27, 0x38, 0xfa, 0x71, 0x9f, 0x1e,
    0x00, 0x41, 0xf5, 0x2e, 0x40, 0xe9, 0x1f, 0x22, 0x9f, 0x4d, 0x96, 0xa1,
    0xe6, 0xf1, 0x72, 0xe1, 0x55, 0x96, 0xb4, 0x51, 0x0a, 0x6d, 0xae, 0xc2,
    0x61, 0x05, 0xf2, 0xbe, 0xbc, 0x53, 0x31, 0x6b, 0x87, 0xbd, 0xf2, 0x13,
    0x11, 0x66, 0x60, 0x70, 0xe8, 0xdf, 0xee, 0x69, 0xd5, 0x2c, 0x71, 0xa9,
    0x76, 0xca, 0xae, 0x79, 0xc7, 0x2b, 0x68, 0xd2, 0x85, 0x80, 0xdc, 0x68,
    0x6d, 0x9f, 0x51, 0x29, 0xd2, 0x25, 0xf8, 0x2b, 0x3d, 0x61, 0x55, 0x13,
    0xa8, 0x82, 0xb3, 0xdb, 0x91, 0x41, 0x6b, 0x48, 0xce, 0x08, 0x88, 0x82,
    0x13, 0xe3, 0x7e, 0xeb, 0x9a, 0xf8, 0x00, 0xd8, 0x1c, 0xab, 0x32, 0x8c,
    0xe4, 0x20, 0x68, 0x99, 0x03, 0xc0, 0x0c, 0x7b, 0x5f, 0xd3, 0x1b, 0x75,
    0x50, 0x3a, 0x6d, 0x41, 0x96, 0x84, 0xd6, 0x29, 0x02, 0x60, 0xf8, 0xeb,
    0x97, 0xe9, 0x8d, 0xf1, 0x26, 0x64, 0xee, 0xfd, 0xb7, 0x61, 0x59, 0x6a,
    0x69, 0xdd, 0xcd, 0x0e, 0x76, 0xda, 0xec, 0xe6, 0xed, 0x4b, 0xf5, 0xa1,
    0xb5, 0x0a, 0xc0, 0x86, 0xf7, 0x92, 0x8a, 0x4d, 0x2f, 0x87, 0x26, 0xa7,
    0x7e, 0x51, 0x5b, 0x74, 0xda, 0x41, 0x98, 0x8f, 0x22, 0x0b, 0x1c, 0xc8,
    0x7a, 0xa1, 0xfc, 0x81, 0x0c, 0xe9, 0x9a, 0x82, 0xf2, 0xd1, 0xce, 0x82,
    0x1e, 0xdc, 0xed, 0x79, 0x4c, 0x69, 0x41, 0xf4, 0x2c, 0x7a, 0x1a, 0x0b,
    0x8c, 0x4d, 0x28, 0xc7, 0x5e, 0xc6, 0x0b, 0x65, 0x22, 0x79, 0xf6, 0x15,
    0x4a, 0x76, 0x2a, 0xed, 0x16, 0x5d, 0x47, 0xde, 0xe3, 0x67, 0x02, 0x60,
    0xed, 0x4d, 0x71, 0xd0, 0xa6, 0xe2, 0x4b, 0x93, 0xc2, 0xe5, 0xf6, 0xb4,
    0xbb, 0xe0, 0x5f, 0x5f, 0xb0, 0xaf, 0xa0, 0x42, 0xd2, 0x04, 0xfe, 0x33,
    0x78, 0xd3, 0x65, 0xc2, 0xf2, 0x88, 0xb6, 0xa8, 0xda, 0xd7, 0xef, 0xe4,
    0x5d, 0x15, 0x3e, 0xef, 0x40, 0xca, 0xcc, 0x7b, 0x81, 0xff, 0x93, 0x40,
    0x02, 0xd1, 0x08, 0x99, 0x4b, 0x94, 0xa5, 0xe4, 0x72, 0x8c, 0xd9, 0xc9,
    0x63, 0x37, 0x5a, 0xe4, 0x99, 0x65, 0xbd, 0xa5, 0x5c, 0xbf, 0x0e, 0xfe,
    0xd8, 0xd6, 0x55, 0x3b, 0x40, 0x27, 0xf2, 0xd8, 0x62, 0x08, 0xa6, 0xe6,
    0xb4, 0x89, 0xc1, 0x76, 0x12, 0x80, 0x92, 0xd6, 0x29, 0xe4, 0x9d, 0x3d,
    0x02, 0x60, 0x2b, 0xb6, 0x8b, 0xdd, 0xfb, 0x0c, 0x4f, 0x56, 0xc8, 0x55,
    0x8b, 0xff, 0xaf, 0x89, 0x2d, 0x80, 0x43, 0x03, 0x78, 0x41, 0xe7, 0xfa,
    0x81, 0xcf, 0xa6, 0x1a, 0x38, 0xc5, 0xe3, 0x9b, 0x90, 0x1c, 0x8e, 0xe7,
    0x11, 0x22, 0xa5, 0xda, 0x22, 0x27, 0xbd, 0x6c, 0xde, 0xeb, 0x48, 0x14,
    0x52, 0xc1, 0x2a, 0xd3, 0xd6, 0x1d, 0x5e, 0x4f, 0x77, 0x6a, 0x0a, 0xb5,
    0x56, 0x59, 0x1b, 0xef, 0xe3, 0xe5, 0x9e, 0x5a, 0x7f, 0xdd, 0xb8, 0x34,
    0x5e, 0x1f, 0x2f, 0x35, 0xb9, 0xf4, 0xce, 0xe5, 0x7c, 0x32, 0x41, 0x4c,
    0x08, 0x6a, 0xec, 0x99, 0x3e, 0x93, 0x53, 0xe4, 0x80, 0xd9, 0xee, 0xc6,
    0x28, 0x9f, 0x02, 0x60, 0x4f, 0xf8, 0x97, 0x70, 0x9f, 0xad, 0x07, 0x97,
    0x46, 0x49, 0x45, 0x78, 0xe7, 0x0f, 0xd8, 0x54, 0x61, 0x30, 0xee, 0xab,
    0x56, 0x27, 0xc4, 0x9b, 0x08, 0x0f, 0x05, 0xee, 0x4a, 0xd9, 0xf3, 0xe4,
    0xb7, 0xcb, 0xa9, 0xd6, 0xa5, 0xdf, 0xf1, 0x13, 0xa4, 0x1c, 0x34, 0x09,
    0x33, 0x68, 0x33, 0xf1, 0x90, 0x81, 0x6d, 0x8a, 0x6b, 0xc4, 0x2e, 0x9b,
    0xec, 0x56, 0xb7, 0x56, 0x7d, 0x0f, 0x3c, 0x9c, 0x69, 0x6d, 0xb6, 0x19,
    0xb2, 0x45, 0xd9, 0x01, 0xdd, 0x85, 0x6d, 0xb7, 0xc8, 0x09, 0x2e, 0x77,
    0xe9, 0xa1, 0xcc, 0xcd, 0x56, 0xee, 0x4d, 0xba, 0x42, 0xc5, 0xfd, 0xb6,
    0x1a, 0xec, 0x26, 0x69, 0x02, 0x60, 0x77, 0xb9, 0xd1, 0x13, 0x7b, 0x50,
    0x40, 0x4a, 0x98, 0x27, 0x29, 0x31, 0x6e, 0xfa, 0xfc, 0x7d, 0xfe, 0x66,
    0xd3, 0x4e, 0x5a, 0x18, 0x26, 0x00, 0xd5, 0xf3, 0x0a, 0x0a, 0x85, 0x12,
    0x05, 0x1c, 0x56, 0x0d, 0x08, 0x1d, 0x4d, 0x0a, 0x18, 0x35, 0xec, 0x3d,
    0x25, 0xa6, 0x0f, 0x4e, 0x4d, 0x6a, 0xa9, 0x48, 0xb2, 0xbf, 0x3d, 0xbb,
    0x5b, 0x12, 0x4c, 0xbb, 0xc3, 0x48, 0x92, 0x55, 0xa3, 0xa9, 0x48, 0x37,
    0x2f, 0x69, 0x78, 0x49, 0x67, 0x45, 0xf9, 0x43, 0xe1, 0xdb, 0x4f, 0x18,
    0x38, 0x2c, 0xea, 0xa5, 0x05, 0xdf, 0xc6, 0x57, 0x57, 0xbb, 0x3f, 0x85,
    0x7a, 0x58, 0xdc, 0xe5, 0x21, 0x56};
const uint8_t kTestVector9Spki[] = {
    0x30, 0x81, 0xdf, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
    0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0xcd, 0x00, 0x30, 0x81,
    0xc9, 0x02, 0x81, 0xc1, 0x00, 0xe6, 0xbd, 0x69, 0x2a, 0xc9, 0x66, 0x45,
    0x79, 0x04, 0x03, 0xfd, 0xd0, 0xf5, 0xbe, 0xb8, 0xb9, 0xbf, 0x92, 0xed,
    0x10, 0x00, 0x7f, 0xc3, 0x65, 0x04, 0x64, 0x19, 0xdd, 0x06, 0xc0, 0x5c,
    0x5b, 0x5b, 0x2f, 0x48, 0xec, 0xf9, 0x89, 0xe4, 0xce, 0x26, 0x91, 0x09,
    0x97, 0x9c, 0xbb, 0x40, 0xb4, 0xa0, 0xad, 0x24, 0xd2, 0x24, 0x83, 0xd1,
    0xee, 0x31, 0x5a, 0xd4, 0xcc, 0xb1, 0x53, 0x42, 0x68, 0x35, 0x26, 0x91,
    0xc5, 0x24, 0xf6, 0xdd, 0x8e, 0x6c, 0x29, 0xd2, 0x24, 0xcf, 0x24, 0x69,
    0x73, 0xae, 0xc8, 0x6c, 0x5b, 0xf6, 0xb1, 0x40, 0x1a, 0x85, 0x0d, 0x1b,
    0x9a, 0xd1, 0xbb, 0x8c, 0xbc, 0xec, 0x47, 0xb0, 0x6f, 0x0f, 0x8c, 0x7f,
    0x45, 0xd3, 0xfc, 0x8f, 0x31, 0x92, 0x99, 0xc5, 0x43, 0x3d, 0xdb, 0xc2,
    0xb3, 0x05, 0x3b, 0x47, 0xde, 0xd2, 0xec, 0xd4, 0xa4, 0xca, 0xef, 0xd6,
    0x14, 0x83, 0x3d, 0xc8, 0xbb, 0x62, 0x2f, 0x31, 0x7e, 0xd0, 0x76, 0xb8,
    0x05, 0x7f, 0xe8, 0xde, 0x3f, 0x84, 0x48, 0x0a, 0xd5, 0xe8, 0x3e, 0x4a,
    0x61, 0x90, 0x4a, 0x4f, 0x24, 0x8f, 0xb3, 0x97, 0x02, 0x73, 0x57, 0xe1,
    0xd3, 0x0e, 0x46, 0x31, 0x39, 0x81, 0x5c, 0x6f, 0xd4, 0xfd, 0x5a, 0xc5,
    0xb8, 0x17, 0x2a, 0x45, 0x23, 0x0e, 0xcb, 0x63, 0x18, 0xa0, 0x4f, 0x14,
    0x55, 0xd8, 0x4e, 0x5a, 0x8b, 0x02, 0x03, 0x01, 0x00, 0x01};
// RSA-PSS test vectors, pss-vect.txt, Example 9.1
const uint8_t kTestVector9Data[] = {
    0xa8, 0x8e, 0x26, 0x58, 0x55, 0xe9, 0xd7, 0xca, 0x36, 0xc6, 0x87, 0x95,
    0xf0, 0xb3, 0x1b, 0x59, 0x1c, 0xd6, 0x58, 0x7c, 0x71, 0xd0, 0x60, 0xa0,
    0xb3, 0xf7, 0xf3, 0xea, 0xef, 0x43, 0x79, 0x59, 0x22, 0x02, 0x8b, 0xc2,
    0xb6, 0xad, 0x46, 0x7c, 0xfc, 0x2d, 0x7f, 0x65, 0x9c, 0x53, 0x85, 0xaa,
    0x70, 0xba, 0x36, 0x72, 0xcd, 0xde, 0x4c, 0xfe, 0x49, 0x70, 0xcc, 0x79,
    0x04, 0x60, 0x1b, 0x27, 0x88, 0x72, 0xbf, 0x51, 0x32, 0x1c, 0x4a, 0x97,
    0x2f, 0x3c, 0x95, 0x57, 0x0f, 0x34, 0x45, 0xd4, 0xf5, 0x79, 0x80, 0xe0,
    0xf2, 0x0d, 0xf5, 0x48, 0x46, 0xe6, 0xa5, 0x2c, 0x66, 0x8f, 0x12, 0x88,
    0xc0, 0x3f, 0x95, 0x00, 0x6e, 0xa3, 0x2f, 0x56, 0x2d, 0x40, 0xd5, 0x2a,
    0xf9, 0xfe, 0xb3, 0x2f, 0x0f, 0xa0, 0x6d, 0xb6, 0x5b, 0x58, 0x8a, 0x23,
    0x7b, 0x34, 0xe5, 0x92, 0xd5, 0x5c, 0xf9, 0x79, 0xf9, 0x03, 0xa6, 0x42,
    0xef, 0x64, 0xd2, 0xed, 0x54, 0x2a, 0xa8, 0xc7, 0x7d, 0xc1, 0xdd, 0x76,
    0x2f, 0x45, 0xa5, 0x93, 0x03, 0xed, 0x75, 0xe5, 0x41, 0xca, 0x27, 0x1e,
    0x2b, 0x60, 0xca, 0x70, 0x9e, 0x44, 0xfa, 0x06, 0x61, 0x13, 0x1e, 0x8d,
    0x5d, 0x41, 0x63, 0xfd, 0x8d, 0x39, 0x85, 0x66, 0xce, 0x26, 0xde, 0x87,
    0x30, 0xe7, 0x2f, 0x9c, 0xca, 0x73, 0x76, 0x41, 0xc2, 0x44, 0x15, 0x94,
    0x20, 0x63, 0x70, 0x28, 0xdf, 0x0a, 0x18, 0x07, 0x9d, 0x62, 0x08, 0xea,
    0x8b, 0x47, 0x11, 0xa2, 0xc7, 0x50, 0xf5};
const uint8_t kTestVector9Sig[] = {
    0x58, 0x61, 0x07, 0x22, 0x6c, 0x3c, 0xe0, 0x13, 0xa7, 0xc8, 0xf0, 0x4d,
    0x1a, 0x6a, 0x29, 0x59, 0xbb, 0x4b, 0x8e, 0x20, 0x5b, 0xa4, 0x3a, 0x27,
    0xb5, 0x0f, 0x12, 0x41, 0x11, 0xbc, 0x35, 0xef, 0x58, 0x9b, 0x03, 0x9f,
    0x59, 0x32, 0x18, 0x7c, 0xb6, 0x96, 0xd7, 0xd9, 0xa3, 0x2c, 0x0c, 0x38,
    0x30, 0x0a, 0x5c, 0xdd, 0xa4, 0x83, 0x4b, 0x62, 0xd2, 0xeb, 0x24, 0x0a,
    0xf3, 0x3f, 0x79, 0xd1, 0x3d, 0xfb, 0xf0, 0x95, 0xbf, 0x59, 0x9e, 0x0d,
    0x96, 0x86, 0x94, 0x8c, 0x19, 0x64, 0x74, 0x7b, 0x67, 0xe8, 0x9c, 0x9a,
    0xba, 0x5c, 0xd8, 0x50, 0x16, 0x23, 0x6f, 0x56, 0x6c, 0xc5, 0x80, 0x2c,
    0xb1, 0x3e, 0xad, 0x51, 0xbc, 0x7c, 0xa6, 0xbe, 0xf3, 0xb9, 0x4d, 0xcb,
    0xdb, 0xb1, 0xd5, 0x70, 0x46, 0x97, 0x71, 0xdf, 0x0e, 0x00, 0xb1, 0xa8,
    0xa0, 0x67, 0x77, 0x47, 0x2d, 0x23, 0x16, 0x27, 0x9e, 0xda, 0xe8, 0x64,
    0x74, 0x66, 0x8d, 0x4e, 0x1e, 0xff, 0xf9, 0x5f, 0x1d, 0xe6, 0x1c, 0x60,
    0x20, 0xda, 0x32, 0xae, 0x92, 0xbb, 0xf1, 0x65, 0x20, 0xfe, 0xf3, 0xcf,
    0x4d, 0x88, 0xf6, 0x11, 0x21, 0xf2, 0x4b, 0xbd, 0x9f, 0xe9, 0x1b, 0x59,
    0xca, 0xf1, 0x23, 0x5b, 0x2a, 0x93, 0xff, 0x81, 0xfc, 0x40, 0x3a, 0xdd,
    0xf4, 0xeb, 0xde, 0xa8, 0x49, 0x34, 0xa9, 0xcd, 0xaf, 0x8e, 0x1a, 0x9e};

// RSA-PSS test vectors, pss-vect.txt, Example 10: A 2048-bit RSA Key Pair
// <ftp://ftp.rsasecurity.com/pub/pkcs/pkcs-1/pkcs-1v2-1-vec.zip>
const uint8_t kTestVector10Pkcs8[] = {
    0x30, 0x82, 0x04, 0xb9, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
    0x04, 0xa3, 0x30, 0x82, 0x04, 0x9f, 0x02, 0x01, 0x00, 0x02, 0x82, 0x01,
    0x00, 0xa5, 0xdd, 0x86, 0x7a, 0xc4, 0xcb, 0x02, 0xf9, 0x0b, 0x94, 0x57,
    0xd4, 0x8c, 0x14, 0xa7, 0x70, 0xef, 0x99, 0x1c, 0x56, 0xc3, 0x9c, 0x0e,
    0xc6, 0x5f, 0xd1, 0x1a, 0xfa, 0x89, 0x37, 0xce, 0xa5, 0x7b, 0x9b, 0xe7,
    0xac, 0x73, 0xb4, 0x5c, 0x00, 0x17, 0x61, 0x5b, 0x82, 0xd6, 0x22, 0xe3,
    0x18, 0x75, 0x3b, 0x60, 0x27, 0xc0, 0xfd, 0x15, 0x7b, 0xe1, 0x2f, 0x80,
    0x90, 0xfe, 0xe2, 0xa7, 0xad, 0xcd, 0x0e, 0xef, 0x75, 0x9f, 0x88, 0xba,
    0x49, 0x97, 0xc7, 0xa4, 0x2d, 0x58, 0xc9, 0xaa, 0x12, 0xcb, 0x99, 0xae,
    0x00, 0x1f, 0xe5, 0x21, 0xc1, 0x3b, 0xb5, 0x43, 0x14, 0x45, 0xa8, 0xd5,
    0xae, 0x4f, 0x5e, 0x4c, 0x7e, 0x94, 0x8a, 0xc2, 0x27, 0xd3, 0x60, 0x40,
    0x71, 0xf2, 0x0e, 0x57, 0x7e, 0x90, 0x5f, 0xbe, 0xb1, 0x5d, 0xfa, 0xf0,
    0x6d, 0x1d, 0xe5, 0xae, 0x62, 0x53, 0xd6, 0x3a, 0x6a, 0x21, 0x20, 0xb3,
    0x1a, 0x5d, 0xa5, 0xda, 0xbc, 0x95, 0x50, 0x60, 0x0e, 0x20, 0xf2, 0x7d,
    0x37, 0x39, 0xe2, 0x62, 0x79, 0x25, 0xfe, 0xa3, 0xcc, 0x50, 0x9f, 0x21,
    0xdf, 0xf0, 0x4e, 0x6e, 0xea, 0x45, 0x49, 0xc5, 0x40, 0xd6, 0x80, 0x9f,
    0xf9, 0x30, 0x7e, 0xed, 0xe9, 0x1f, 0xff, 0x58, 0x73, 0x3d, 0x83, 0x85,
    0xa2, 0x37, 0xd6, 0xd3, 0x70, 0x5a, 0x33, 0xe3, 0x91, 0x90, 0x09, 0x92,
    0x07, 0x0d, 0xf7, 0xad, 0xf1, 0x35, 0x7c, 0xf7, 0xe3, 0x70, 0x0c, 0xe3,
    0x66, 0x7d, 0xe8, 0x3f, 0x17, 0xb8, 0xdf, 0x17, 0x78, 0xdb, 0x38, 0x1d,
    0xce, 0x09, 0xcb, 0x4a, 0xd0, 0x58, 0xa5, 0x11, 0x00, 0x1a, 0x73, 0x81,
    0x98, 0xee, 0x27, 0xcf, 0x55, 0xa1, 0x3b, 0x75, 0x45, 0x39, 0x90, 0x65,
    0x82, 0xec, 0x8b, 0x17, 0x4b, 0xd5, 0x8d, 0x5d, 0x1f, 0x3d, 0x76, 0x7c,
    0x61, 0x37, 0x21, 0xae, 0x05, 0x02, 0x03, 0x01, 0x00, 0x01, 0x02, 0x82,
    0x01, 0x00, 0x2d, 0x2f, 0xf5, 0x67, 0xb3, 0xfe, 0x74, 0xe0, 0x61, 0x91,
    0xb7, 0xfd, 0xed, 0x6d, 0xe1, 0x12, 0x29, 0x0c, 0x67, 0x06, 0x92, 0x43,
    0x0d, 0x59, 0x69, 0x18, 0x40, 0x47, 0xda, 0x23, 0x4c, 0x96, 0x93, 0xde,
    0xed, 0x16, 0x73, 0xed, 0x42, 0x95, 0x39, 0xc9, 0x69, 0xd3, 0x72, 0xc0,
    0x4d, 0x6b, 0x47, 0xe0, 0xf5, 0xb8, 0xce, 0xe0, 0x84, 0x3e, 0x5c, 0x22,
    0x83, 0x5d, 0xbd, 0x3b, 0x05, 0xa0, 0x99, 0x79, 0x84, 0xae, 0x60, 0x58,
    0xb1, 0x1b, 0xc4, 0x90, 0x7c, 0xbf, 0x67, 0xed, 0x84, 0xfa, 0x9a, 0xe2,
    0x52, 0xdf, 0xb0, 0xd0, 0xcd, 0x49, 0xe6, 0x18, 0xe3, 0x5d, 0xfd, 0xfe,
    0x59, 0xbc, 0xa3, 0xdd, 0xd6, 0x6c, 0x33, 0xce, 0xbb, 0xc7, 0x7a, 0xd4,
    0x41, 0xaa, 0x69, 0x5e, 0x13, 0xe3, 0x24, 0xb5, 0x18, 0xf0, 0x1c, 0x60,
    0xf5, 0xa8, 0x5c, 0x99, 0x4a, 0xd1, 0x79, 0xf2, 0xa6, 0xb5, 0xfb, 0xe9,
    0x34, 0x02, 0xb1, 0x17, 0x67, 0xbe, 0x01, 0xbf, 0x07, 0x34, 0x44, 0xd6,
    0xba, 0x1d, 0xd2, 0xbc, 0xa5, 0xbd, 0x07, 0x4d, 0x4a, 0x5f, 0xae, 0x35,
    0x31, 0xad, 0x13, 0x03, 0xd8, 0x4b, 0x30, 0xd8, 0x97, 0x31, 0x8c, 0xbb,
    0xba, 0x04, 0xe0, 0x3c, 0x2e, 0x66, 0xde, 0x6d, 0x91, 0xf8, 0x2f, 0x96,
    0xea, 0x1d, 0x4b, 0xb5, 0x4a, 0x5a, 0xae, 0x10, 0x2d, 0x59, 0x46, 0x57,
    0xf5, 0xc9, 0x78, 0x95, 0x53, 0x51, 0x2b, 0x29, 0x6d, 0xea, 0x29, 0xd8,
    0x02, 0x31, 0x96, 0x35, 0x7e, 0x3e, 0x3a, 0x6e, 0x95, 0x8f, 0x39, 0xe3,
    0xc2, 0x34, 0x40, 0x38, 0xea, 0x60, 0x4b, 0x31, 0xed, 0xc6, 0xf0, 0xf7,
    0xff, 0x6e, 0x71, 0x81, 0xa5, 0x7c, 0x92, 0x82, 0x6a, 0x26, 0x8f, 0x86,
    0x76, 0x8e, 0x96, 0xf8, 0x78, 0x56, 0x2f, 0xc7, 0x1d, 0x85, 0xd6, 0x9e,
    0x44, 0x86, 0x12, 0xf7, 0x04, 0x8f, 0x02, 0x81, 0x80, 0xcf, 0xd5, 0x02,
    0x83, 0xfe, 0xee, 0xb9, 0x7f, 0x6f, 0x08, 0xd7, 0x3c, 0xbc, 0x7b, 0x38,
    0x36, 0xf8, 0x2b, 0xbc, 0xd4, 0x99, 0x47, 0x9f, 0x5e, 0x6f, 0x76, 0xfd,
    0xfc, 0xb8, 0xb3, 0x8c, 0x4f, 0x71, 0xdc, 0x9e, 0x88, 0xbd, 0x6a, 0x6f,
    0x76, 0x37, 0x1a, 0xfd, 0x65, 0xd2, 0xaf, 0x18, 0x62, 0xb3, 0x2a, 0xfb,
    0x34, 0xa9, 0x5f, 0x71, 0xb8, 0xb1, 0x32, 0x04, 0x3f, 0xfe, 0xbe, 0x3a,
    0x95, 0x2b, 0xaf, 0x75, 0x92, 0x44, 0x81, 0x48, 0xc0, 0x3f, 0x9c, 0x69,
    0xb1, 0xd6, 0x8e, 0x4c, 0xe5, 0xcf, 0x32, 0xc8, 0x6b, 0xaf, 0x46, 0xfe,
    0xd3, 0x01, 0xca, 0x1a, 0xb4, 0x03, 0x06, 0x9b, 0x32, 0xf4, 0x56, 0xb9,
    0x1f, 0x71, 0x89, 0x8a, 0xb0, 0x81, 0xcd, 0x8c, 0x42, 0x52, 0xef, 0x52,
    0x71, 0x91, 0x5c, 0x97, 0x94, 0xb8, 0xf2, 0x95, 0x85, 0x1d, 0xa7, 0x51,
    0x0f, 0x99, 0xcb, 0x73, 0xeb, 0x02, 0x81, 0x80, 0xcc, 0x4e, 0x90, 0xd2,
    0xa1, 0xb3, 0xa0, 0x65, 0xd3, 0xb2, 0xd1, 0xf5, 0xa8, 0xfc, 0xe3, 0x1b,
    0x54, 0x44, 0x75, 0x66, 0x4e, 0xab, 0x56, 0x1d, 0x29, 0x71, 0xb9, 0x9f,
    0xb7, 0xbe, 0xf8, 0x44, 0xe8, 0xec, 0x1f, 0x36, 0x0b, 0x8c, 0x2a, 0xc8,
    0x35, 0x96, 0x92, 0x97, 0x1e, 0xa6, 0xa3, 0x8f, 0x72, 0x3f, 0xcc, 0x21,
    0x1f, 0x5d, 0xbc, 0xb1, 0x77, 0xa0, 0xfd, 0xac, 0x51, 0x64, 0xa1, 0xd4,
    0xff, 0x7f, 0xbb, 0x4e, 0x82, 0x99, 0x86, 0x35, 0x3c, 0xb9, 0x83, 0x65,
    0x9a, 0x14, 0x8c, 0xdd, 0x42, 0x0c, 0x7d, 0x31, 0xba, 0x38, 0x22, 0xea,
    0x90, 0xa3, 0x2b, 0xe4, 0x6c, 0x03, 0x0e, 0x8c, 0x17, 0xe1, 0xfa, 0x0a,
    0xd3, 0x78, 0x59, 0xe0, 0x6b, 0x0a, 0xa6, 0xfa, 0x3b, 0x21, 0x6d, 0x9c,
    0xbe, 0x6c, 0x0e, 0x22, 0x33, 0x97, 0x69, 0xc0, 0xa6, 0x15, 0x91, 0x3e,
    0x5d, 0xa7, 0x19, 0xcf, 0x02, 0x81, 0x80, 0x1c, 0x2d, 0x1f, 0xc3, 0x2f,
    0x6b, 0xc4, 0x00, 0x4f, 0xd8, 0x5d, 0xfd, 0xe0, 0xfb, 0xbf, 0x9a, 0x4c,
    0x38, 0xf9, 0xc7, 0xc4, 0xe4, 0x1d, 0xea, 0x1a, 0xa8, 0x82, 0x34, 0xa2,
    0x01, 0xcd, 0x92, 0xf3, 0xb7, 0xda, 0x52, 0x65, 0x83, 0xa9, 0x8a, 0xd8,
    0x5b, 0xb3, 0x60, 0xfb, 0x98, 0x3b, 0x71, 0x1e, 0x23, 0x44, 0x9d, 0x56,
    0x1d, 0x17, 0x78, 0xd7, 0xa5, 0x15, 0x48, 0x6b, 0xcb, 0xf4, 0x7b, 0x46,
    0xc9, 0xe9, 0xe1, 0xa3, 0xa1, 0xf7, 0x70, 0x00, 0xef, 0xbe, 0xb0, 0x9a,
    0x8a, 0xfe, 0x47, 0xe5, 0xb8, 0x57, 0xcd, 0xa9, 0x9c, 0xb1, 0x6d, 0x7f,
    0xff, 0x9b, 0x71, 0x2e, 0x3b, 0xd6, 0x0c, 0xa9, 0x6d, 0x9c, 0x79, 0x73,
    0xd6, 0x16, 0xd4, 0x69, 0x34, 0xa9, 0xc0, 0x50, 0x28, 0x1c, 0x00, 0x43,
    0x99, 0xce, 0xff, 0x1d, 0xb7, 0xdd, 0xa7, 0x87, 0x66, 0xa8, 0xa9, 0xb9,
    0xcb, 0x08, 0x73, 0x02, 0x81, 0x80, 0xcb, 0x3b, 0x3c, 0x04, 0xca, 0xa5,
    0x8c, 0x60, 0xbe, 0x7d, 0x9b, 0x2d, 0xeb, 0xb3, 0xe3, 0x96, 0x43, 0xf4,
    0xf5, 0x73, 0x97, 0xbe, 0x08, 0x23, 0x6a, 0x1e, 0x9e, 0xaf, 0xaa, 0x70,
    0x65, 0x36, 0xe7, 0x1c, 0x3a, 0xcf, 0xe0, 0x1c, 0xc6, 0x51, 0xf2, 0x3c,
    0x9e, 0x05, 0x85, 0x8f, 0xee, 0x13, 0xbb, 0x6a, 0x8a, 0xfc, 0x47, 0xdf,
    0x4e, 0xdc, 0x9a, 0x4b, 0xa3, 0x0b, 0xce, 0xcb, 0x73, 0xd0, 0x15, 0x78,
    0x52, 0x32, 0x7e, 0xe7, 0x89, 0x01, 0x5c, 0x2e, 0x8d, 0xee, 0x7b, 0x9f,
    0x05, 0xa0, 0xf3, 0x1a, 0xc9, 0x4e, 0xb6, 0x17, 0x31, 0x64, 0x74, 0x0c,
    0x5c, 0x95, 0x14, 0x7c, 0xd5, 0xf3, 0xb5, 0xae, 0x2c, 0xb4, 0xa8, 0x37,
    0x87, 0xf0, 0x1d, 0x8a, 0xb3, 0x1f, 0x27, 0xc2, 0xd0, 0xee, 0xa2, 0xdd,
    0x8a, 0x11, 0xab, 0x90, 0x6a, 0xba, 0x20, 0x7c, 0x43, 0xc6, 0xee, 0x12,
    0x53, 0x31, 0x02, 0x81, 0x80, 0x12, 0xf6, 0xb2, 0xcf, 0x13, 0x74, 0xa7,
    0x36, 0xfa, 0xd0, 0x56, 0x16, 0x05, 0x0f, 0x96, 0xab, 0x4b, 0x61, 0xd1,
    0x17, 0x7c, 0x7f, 0x9d, 0x52, 0x5a, 0x29, 0xf3, 0xd1, 0x80, 0xe7, 0x76,
    0x67, 0xe9, 0x9d, 0x99, 0xab, 0xf0, 0x52, 0x5d, 0x07, 0x58, 0x66, 0x0f,
    0x37, 0x52, 0x65, 0x5b, 0x0f, 0x25, 0xb8, 0xdf, 0x84, 0x31, 0xd9, 0xa8,
    0xff, 0x77, 0xc1, 0x6c, 0x12, 0xa0, 0xa5, 0x12, 0x2a, 0x9f, 0x0b, 0xf7,
    0xcf, 0xd5, 0xa2, 0x66, 0xa3, 0x5c, 0x15, 0x9f, 0x99, 0x12, 0x08, 0xb9,
    0x03, 0x16, 0xff, 0x44, 0x4f, 0x3e, 0x0b, 0x6b, 0xd0, 0xe9, 0x3b, 0x8a,
    0x7a, 0x24, 0x48, 0xe9, 0x57, 0xe3, 0xdd, 0xa6, 0xcf, 0xcf, 0x22, 0x66,
    0xb1, 0x06, 0x01, 0x3a, 0xc4, 0x68, 0x08, 0xd3, 0xb3, 0x88, 0x7b, 0x3b,
    0x00, 0x34, 0x4b, 0xaa, 0xc9, 0x53, 0x0b, 0x4c, 0xe7, 0x08, 0xfc, 0x32,
    0xb6};
const uint8_t kTestVector10Spki[] = {
    0x30, 0x82, 0x01, 0x21, 0x30, 0x0d, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86,
    0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0e, 0x00,
    0x30, 0x82, 0x01, 0x09, 0x02, 0x82, 0x01, 0x00, 0xa5, 0xdd, 0x86, 0x7a,
    0xc4, 0xcb, 0x02, 0xf9, 0x0b, 0x94, 0x57, 0xd4, 0x8c, 0x14, 0xa7, 0x70,
    0xef, 0x99, 0x1c, 0x56, 0xc3, 0x9c, 0x0e, 0xc6, 0x5f, 0xd1, 0x1a, 0xfa,
    0x89, 0x37, 0xce, 0xa5, 0x7b, 0x9b, 0xe7, 0xac, 0x73, 0xb4, 0x5c, 0x00,
    0x17, 0x61, 0x5b, 0x82, 0xd6, 0x22, 0xe3, 0x18, 0x75, 0x3b, 0x60, 0x27,
    0xc0, 0xfd, 0x15, 0x7b, 0xe1, 0x2f, 0x80, 0x90, 0xfe, 0xe2, 0xa7, 0xad,
    0xcd, 0x0e, 0xef, 0x75, 0x9f, 0x88, 0xba, 0x49, 0x97, 0xc7, 0xa4, 0x2d,
    0x58, 0xc9, 0xaa, 0x12, 0xcb, 0x99, 0xae, 0x00, 0x1f, 0xe5, 0x21, 0xc1,
    0x3b, 0xb5, 0x43, 0x14, 0x45, 0xa8, 0xd5, 0xae, 0x4f, 0x5e, 0x4c, 0x7e,
    0x94, 0x8a, 0xc2, 0x27, 0xd3, 0x60, 0x40, 0x71, 0xf2, 0x0e, 0x57, 0x7e,
    0x90, 0x5f, 0xbe, 0xb1, 0x5d, 0xfa, 0xf0, 0x6d, 0x1d, 0xe5, 0xae, 0x62,
    0x53, 0xd6, 0x3a, 0x6a, 0x21, 0x20, 0xb3, 0x1a, 0x5d, 0xa5, 0xda, 0xbc,
    0x95, 0x50, 0x60, 0x0e, 0x20, 0xf2, 0x7d, 0x37, 0x39, 0xe2, 0x62, 0x79,
    0x25, 0xfe, 0xa3, 0xcc, 0x50, 0x9f, 0x21, 0xdf, 0xf0, 0x4e, 0x6e, 0xea,
    0x45, 0x49, 0xc5, 0x40, 0xd6, 0x80, 0x9f, 0xf9, 0x30, 0x7e, 0xed, 0xe9,
    0x1f, 0xff, 0x58, 0x73, 0x3d, 0x83, 0x85, 0xa2, 0x37, 0xd6, 0xd3, 0x70,
    0x5a, 0x33, 0xe3, 0x91, 0x90, 0x09, 0x92, 0x07, 0x0d, 0xf7, 0xad, 0xf1,
    0x35, 0x7c, 0xf7, 0xe3, 0x70, 0x0c, 0xe3, 0x66, 0x7d, 0xe8, 0x3f, 0x17,
    0xb8, 0xdf, 0x17, 0x78, 0xdb, 0x38, 0x1d, 0xce, 0x09, 0xcb, 0x4a, 0xd0,
    0x58, 0xa5, 0x11, 0x00, 0x1a, 0x73, 0x81, 0x98, 0xee, 0x27, 0xcf, 0x55,
    0xa1, 0x3b, 0x75, 0x45, 0x39, 0x90, 0x65, 0x82, 0xec, 0x8b, 0x17, 0x4b,
    0xd5, 0x8d, 0x5d, 0x1f, 0x3d, 0x76, 0x7c, 0x61, 0x37, 0x21, 0xae, 0x05,
    0x02, 0x03, 0x01, 0x00, 0x01};
// RSA-PSS test vectors, pss-vect.txt, Example 10.1
const uint8_t kTestVector10Data[] = {
    0x88, 0x31, 0x77, 0xe5, 0x12, 0x6b, 0x9b, 0xe2, 0xd9, 0xa9,
    0x68, 0x03, 0x27, 0xd5, 0x37, 0x0c, 0x6f, 0x26, 0x86, 0x1f,
    0x58, 0x20, 0xc4, 0x3d, 0xa6, 0x7a, 0x3a, 0xd6, 0x09};
const uint8_t kTestVector10Sig[] = {
    0x82, 0xc2, 0xb1, 0x60, 0x09, 0x3b, 0x8a, 0xa3, 0xc0, 0xf7, 0x52, 0x2b,
    0x19, 0xf8, 0x73, 0x54, 0x06, 0x6c, 0x77, 0x84, 0x7a, 0xbf, 0x2a, 0x9f,
    0xce, 0x54, 0x2d, 0x0e, 0x84, 0xe9, 0x20, 0xc5, 0xaf, 0xb4, 0x9f, 0xfd,
    0xfd, 0xac, 0xe1, 0x65, 0x60, 0xee, 0x94, 0xa1, 0x36, 0x96, 0x01, 0x14,
    0x8e, 0xba, 0xd7, 0xa0, 0xe1, 0x51, 0xcf, 0x16, 0x33, 0x17, 0x91, 0xa5,
    0x72, 0x7d, 0x05, 0xf2, 0x1e, 0x74, 0xe7, 0xeb, 0x81, 0x14, 0x40, 0x20,
    0x69, 0x35, 0xd7, 0x44, 0x76, 0x5a, 0x15, 0xe7, 0x9f, 0x01, 0x5c, 0xb6,
    0x6c, 0x53, 0x2c, 0x87, 0xa6, 0xa0, 0x59, 0x61, 0xc8, 0xbf, 0xad, 0x74,
    0x1a, 0x9a, 0x66, 0x57, 0x02, 0x28, 0x94, 0x39, 0x3e, 0x72, 0x23, 0x73,
    0x97, 0x96, 0xc0, 0x2a, 0x77, 0x45, 0x5d, 0x0f, 0x55, 0x5b, 0x0e, 0xc0,
    0x1d, 0xdf, 0x25, 0x9b, 0x62, 0x07, 0xfd, 0x0f, 0xd5, 0x76, 0x14, 0xce,
    0xf1, 0xa5, 0x57, 0x3b, 0xaa, 0xff, 0x4e, 0xc0, 0x00, 0x69, 0x95, 0x16,
    0x59, 0xb8, 0x5f, 0x24, 0x30, 0x0a, 0x25, 0x16, 0x0c, 0xa8, 0x52, 0x2d,
    0xc6, 0xe6, 0x72, 0x7e, 0x57, 0xd0, 0x19, 0xd7, 0xe6, 0x36, 0x29, 0xb8,
    0xfe, 0x5e, 0x89, 0xe2, 0x5c, 0xc1, 0x5b, 0xeb, 0x3a, 0x64, 0x75, 0x77,
    0x55, 0x92, 0x99, 0x28, 0x0b, 0x9b, 0x28, 0xf7, 0x9b, 0x04, 0x09, 0x00,
    0x0b, 0xe2, 0x5b, 0xbd, 0x96, 0x40, 0x8b, 0xa3, 0xb4, 0x3c, 0xc4, 0x86,
    0x18, 0x4d, 0xd1, 0xc8, 0xe6, 0x25, 0x53, 0xfa, 0x1a, 0xf4, 0x04, 0x0f,
    0x60, 0x66, 0x3d, 0xe7, 0xf5, 0xe4, 0x9c, 0x04, 0x38, 0x8e, 0x25, 0x7f,
    0x1c, 0xe8, 0x9c, 0x95, 0xda, 0xb4, 0x8a, 0x31, 0x5d, 0x9b, 0x66, 0xb1,
    0xb7, 0x62, 0x82, 0x33, 0x87, 0x6f, 0xf2, 0x38, 0x52, 0x30, 0xd0, 0x70,
    0xd0, 0x7e, 0x16, 0x66};

}  // namespace nss_test
