// vim:expandtab:shiftwidth=2:tabstop=2:
// Copyright (C) 2013 Canonical Ltd.

// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.

// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

#include "oxide_content_main_delegate.h"

#include <dlfcn.h>
#include <string>
#include <vector>

#include "base/base_paths.h"
#include "base/command_line.h"
#include "base/file_util.h"
#include "base/files/file_path.h"
#include "base/lazy_instance.h"
#include "base/logging.h"
#include "base/path_service.h"
#include "cc/base/switches.h"
#include "content/public/common/content_client.h"
#include "content/public/common/content_switches.h"
#include "content/public/renderer/content_renderer_client.h"
#include "ui/base/layout.h"
#include "ui/base/resource/resource_bundle.h"
#include "ui/gl/gl_implementation.h"
#include "ui/gl/gl_surface.h"
#include "ui/native_theme/native_theme_switches.h"

#include "shared/browser/oxide_browser_process_main.h"
#include "shared/common/oxide_constants.h"
#include "shared/common/oxide_content_client.h"
#include "shared/gl/oxide_shared_gl_context.h"
#include "shared/renderer/oxide_content_renderer_client.h"
#include "shared/sandbox_ipc/oxide_sandbox_ipc_process.h"

namespace oxide {

namespace {

base::LazyInstance<oxide::ContentRendererClient> g_content_renderer_client =
    LAZY_INSTANCE_INITIALIZER;

struct MainFunction {
  const char* name;
  int (*function)(const content::MainFunctionParams&);
};

}

ContentMainDelegate::ContentMainDelegate() {}

content::ContentBrowserClient*
ContentMainDelegate::CreateContentBrowserClient() {
  NOTREACHED() << "CreateContentBrowserClient() hasn't been implemented";
  return NULL;
}

content::ContentRendererClient*
ContentMainDelegate::CreateContentRendererClient() {
  return g_content_renderer_client.Pointer();
}

ContentMainDelegate::~ContentMainDelegate() {}

bool ContentMainDelegate::BasicStartupComplete(int* exit_code) {
  content::SetContentClient(CreateContentClient());

  CommandLine* command_line = CommandLine::ForCurrentProcess();
  std::string process_type =
      command_line->GetSwitchValueASCII(switches::kProcessType);

  if (process_type.empty()) {
    // We need to override FILE_EXE in the browser process to the path of the
    // renderer, as various bits of Chrome use this to find other resources
    base::FilePath subprocess_exe;
    const char* subprocess_path = getenv("OXIDE_SUBPROCESS_PATH");
    if (subprocess_path) {
      // Make sure that we have a properly formed absolute path
      // there are some load issues if not.
      subprocess_exe =
          base::MakeAbsoluteFilePath(base::FilePath(subprocess_path));
    } else {
      subprocess_exe = base::FilePath(FILE_PATH_LITERAL(OXIDE_SUBPROCESS_PATH));
      if (!subprocess_exe.IsAbsolute()) {
        Dl_info info;
        int rv = dladdr(reinterpret_cast<void *>(BrowserProcessMain::Exists),
                        &info);
        DCHECK_NE(rv, 0) << "Failed to determine module path";

        base::FilePath subprocess_rel(subprocess_exe);
        subprocess_exe = base::FilePath(info.dli_fname).DirName();

        std::vector<base::FilePath::StringType> components;
        subprocess_rel.GetComponents(&components);
        for (size_t i = 0; i < components.size(); ++i) {
          subprocess_exe = subprocess_exe.Append(components[i]);
        }
      }
    }

    PathService::Override(base::FILE_EXE, subprocess_exe);
    PathService::Override(base::FILE_MODULE, subprocess_exe);

    // Pick the correct subprocess path
    command_line->AppendSwitchASCII(switches::kBrowserSubprocessPath,
                                    subprocess_exe.value().c_str());

    // This is needed so that we can share GL resources with the embedder
    command_line->AppendSwitch(switches::kInProcessGPU);

    command_line->AppendSwitch(switches::kEnableGestureTapHighlight);

    int flags = BrowserProcessMain::instance()->flags();

    // We need both of this here to test compositing support. It's also needed
    // to work around a mesa race - see https://launchpad.net/bugs/1267893
    gfx::GLSurface::InitializeOneOff();

    SharedGLContext* shared_gl_context =
        BrowserProcessMain::instance()->shared_gl_context();
    if (!shared_gl_context ||
        shared_gl_context->GetImplementation() != gfx::GetGLImplementation()) {
      command_line->AppendSwitch(switches::kDisableAcceleratedCompositing);
      command_line->AppendSwitch(switches::kDisableForceCompositingMode);
      command_line->AppendSwitch(switches::kDisableThreadedCompositing);

      if (flags & BrowserProcessMain::ENABLE_VIEWPORT) {
        flags &= ~BrowserProcessMain::ENABLE_VIEWPORT;
        LOG(WARNING) <<
          "Disabling viewport mode and pinch gestures, which do not work "
          "correctly without compositing";
      }
    }

    if (flags & BrowserProcessMain::ENABLE_VIEWPORT) {
      command_line->AppendSwitch(switches::kEnableViewport);
      command_line->AppendSwitch(switches::kEnableViewportMeta);
      command_line->AppendSwitch(switches::kEnablePinch);
      if (getenv("OXIDE_ENABLE_PINCH_VIRTUAL_VIEWPORT")) {
        command_line->AppendSwitch(cc::switches::kEnablePinchVirtualViewport);
      }
    }
    if (flags & BrowserProcessMain::ENABLE_OVERLAY_SCROLLBARS) {
      command_line->AppendSwitch(switches::kEnableOverlayScrollbar);
    }

    const char* renderer_cmd_prefix = getenv("OXIDE_RENDERER_CMD_PREFIX");
    if (renderer_cmd_prefix) {
      command_line->AppendSwitchASCII(switches::kRendererCmdPrefix,
                                      renderer_cmd_prefix);
    }
    if (getenv("OXIDE_NO_SANDBOX")) {
      command_line->AppendSwitch(switches::kNoSandbox);
    }
    if (getenv("OXIDE_SINGLE_PROCESS")) {
      LOG(WARNING) <<
          "User scripts currently don't work correctly in single process "
          "mode. See https://launchpad.net/bugs/1283291";
      command_line->AppendSwitch(switches::kSingleProcess);
    }
  }

  return false;
}

void ContentMainDelegate::PreSandboxStartup() {
  ui::ResourceBundle::InitSharedInstanceLocaleOnly(std::string(), NULL);

  base::FilePath dir_exe;
  PathService::Get(base::DIR_EXE, &dir_exe);

  ui::ResourceBundle::GetSharedInstance().AddDataPackFromPath(
      dir_exe.Append(FILE_PATH_LITERAL("oxide.pak")),
      ui::SCALE_FACTOR_NONE);
  ui::ResourceBundle::GetSharedInstance().AddDataPackFromPath(
      dir_exe.Append(FILE_PATH_LITERAL("oxide_100_percent.pak")),
      ui::SCALE_FACTOR_100P);
  ui::ResourceBundle::GetSharedInstance().AddDataPackFromPath(
      dir_exe.Append(FILE_PATH_LITERAL("oxide_200_percent.pak")),
      ui::SCALE_FACTOR_200P);
}

int ContentMainDelegate::RunProcess(
    const std::string& process_type,
    const content::MainFunctionParams& main_function_params) {
  if (process_type.empty()) {
    if (!BrowserProcessMain::Exists()) {
      // We arrive here if some calls the renderer with no --process-type
      LOG(ERROR) <<
          "The Oxide renderer cannot be used to run a browser process";
      return 1;
    }

    return BrowserProcessMain::instance()->RunBrowserMain(
        main_function_params);
  }

  static const MainFunction kMainFunctions[] = {
    { kSandboxIPCProcess, SandboxIPCProcessMain }
  };

  for (size_t i = 0; i < arraysize(kMainFunctions); ++i) {
    if (process_type == kMainFunctions[i].name)
      return kMainFunctions[i].function(main_function_params);
  }

  return -1;
}

void ContentMainDelegate::ProcessExiting(const std::string& process_type) {
  if (process_type.empty() && BrowserProcessMain::Exists()) {
    BrowserProcessMain::instance()->ShutdownBrowserMain();
  }
}

} // namespace oxide
