/*
 * Copyright (C) 2012 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Pawel Stolowski <pawel.stolowski@canonical.com>
 * based on python code by David Calle <davidc@framli.eu>
 *
 */

namespace Unity.VideoLens
{
  public class Thumbnailer
  {
    private static const int MAX_THUMBNAILERS = 3;

    private string videos_folder;
    private string cache_directory;
    private Gee.LinkedList<GLib.Pid?> thumbnailer_pids;

    public Thumbnailer (string cache_dir)
    {
      cache_directory = cache_dir;
      videos_folder = GLib.Environment.get_user_special_dir (GLib.UserDirectory.VIDEOS);
      thumbnailer_pids = new Gee.LinkedList<GLib.Pid?> ();
    }

    public string get_icon (string video_file)
    {
      /* This method checks several locations for a video thumbnail.

        1) <filename>.jpg file in the same folder (not activated for now)
        2) Nautilus thumbnails
        3) Cached thumbnails generated by the scope

        If nothing has been found, it tries to generate a thumbnail with Totem,
        stores and uses it.
        If the generation fails or is slow, it fallbacks to the standard video
        icon.
      */
      var file = GLib.File.new_for_path (video_file);
      var video_path = file.get_path ();

      string? icon_path = null;

      // check if nautilus thumbnail is available
      try
      {
        var finfo = file.query_info (GLib.FileAttribute.THUMBNAIL_PATH + "," + GLib.FileAttribute.THUMBNAILING_FAILED, GLib.FileQueryInfoFlags.NONE);
        if (finfo.get_attribute_boolean (GLib.FileAttribute.THUMBNAILING_FAILED))
          return "video";
        else
          icon_path = finfo.get_attribute_as_string (GLib.FileAttribute.THUMBNAIL_PATH);
      }
      catch (GLib.Error e)
      {
        // silently ignore
      }

      if (icon_path == null || icon_path == "")
      {
        string check_path = cache_directory + "/" + video_path.replace (videos_folder, "").replace ("/", "_");

        // check for thumbnail generated by this scope
        if (Utils.is_regular_file (check_path))
          return check_path;

        create_thumbnail (video_path, check_path);

        // FIXME: this is not reliable, since we spawn thumbnailer async
        if (Utils.is_regular_file (check_path))
          return check_path;
      }

      if (icon_path == null || icon_path == "")
        icon_path = "video";

      return icon_path;
    }

    private void thumbnailer_process_watch (Pid pid, int status)
    {
      thumbnailer_pids.remove (pid);
    }

    public void create_thumbnail (string video_path, string thumbnail_path)
    {
      if (thumbnailer_pids.size < MAX_THUMBNAILERS)
      {
        GLib.Pid pid;
        string[] args = {"/usr/bin/totem-video-thumbnailer", video_path, thumbnail_path};
        try
        {
          GLib.Process.spawn_async (null, args, null, GLib.SpawnFlags.DO_NOT_REAP_CHILD, null, out pid);
          GLib.ChildWatch.add (pid, thumbnailer_process_watch);
          thumbnailer_pids.add (pid);
        }
        catch (Error e)
        {
          warning ("Failed to spawn thumbailer: %s", e.message);
        }
      }
    }
  }
}