/*
 * Copyright Samsung Electronics Co.,LTD.
 * Copyright (C) 2011 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/mman.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <math.h>
#include <sys/poll.h>

#include <cutils/log.h>

#include <utils/Log.h>

#include "ExynosJpegApi.h"

#define JPEG_ERROR_LOG(fmt,...)

#define NUM_JPEG_DEC_IN_PLANES (1)
#define NUM_JPEG_DEC_OUT_PLANES (1)

#define NUM_JPEG_DEC_IN_BUFS (1)
#define NUM_JPEG_DEC_OUT_BUFS (1)

#define MAX_JPG_WIDTH (8192)
#define MAX_JPG_HEIGHT (8192)

ExynosJpegDecoder::ExynosJpegDecoder()
{
    t_iJpegFd = -1;
    t_bFlagCreate = false;
}

ExynosJpegDecoder::~ExynosJpegDecoder()
{
    if (t_bFlagCreate == true) {
        this->destroy();
    }
}

int ExynosJpegDecoder::create(void)
{
    return ExynosJpegBase::create(MODE_DECODE);
}

int ExynosJpegDecoder::destroy(void)
{
    return ExynosJpegBase::destroy(NUM_JPEG_DEC_IN_BUFS, NUM_JPEG_DEC_OUT_BUFS);
}

int ExynosJpegDecoder::setJpegConfig(void *pConfig)
{
    return ExynosJpegBase::setJpegConfig(MODE_DECODE, pConfig);
}

int ExynosJpegDecoder::getInBuf(int *piBuf, int *piInputSize)
{
    return getBuf(t_bFlagCreateInBuf, &t_stJpegInbuf, piBuf, piInputSize, \
                        NUM_JPEG_DEC_IN_PLANES, NUM_JPEG_DEC_IN_PLANES);
}

int ExynosJpegDecoder::getOutBuf(int *piBuf, int *piOutputSize, int iSize)
{
    return getBuf(t_bFlagCreateOutBuf, &t_stJpegOutbuf, piBuf, piOutputSize, iSize, t_iPlaneNum);
}

int  ExynosJpegDecoder::setInBuf(int iBuf, int iSize)
{
    int iRet = ERROR_NONE;
    iRet = setBuf(&t_stJpegInbuf, &iBuf, &iSize, NUM_JPEG_DEC_IN_PLANES);

    if (iRet == ERROR_NONE) {
        t_bFlagCreateInBuf = true;
    }

    return iRet;
}

int  ExynosJpegDecoder::setOutBuf(int *piBuf, int *iSize)
{
    int iRet = ERROR_NONE;
    iRet = setBuf(&t_stJpegOutbuf, piBuf, iSize, t_iPlaneNum);

    if (iRet == ERROR_NONE) {
        t_bFlagCreateOutBuf = true;
    }

    return iRet;
}

int ExynosJpegDecoder::getSize(int *piW, int *piH)
{
    if (t_bFlagCreate == false) {
        return ERROR_JPEG_DEVICE_NOT_CREATE_YET;
    }

    int iRet = t_v4l2GetFmt(t_iJpegFd, V4L2_BUF_TYPE_VIDEO_CAPTURE_MPLANE, &t_stJpegConfig);
    if (iRet < 0) {
        JPEG_ERROR_LOG("[%s,%d]: get image size failed\n", __func__,iRet);
        return ERROR_GET_SIZE_FAIL;
    }

    *piW = t_stJpegConfig.width;
    *piH = t_stJpegConfig.height;

    return ERROR_NONE;
}

int ExynosJpegDecoder::setColorFormat(int iV4l2ColorFormat)
{
    return ExynosJpegBase::setColorFormat(MODE_DECODE, iV4l2ColorFormat);
}

int ExynosJpegDecoder::setJpegFormat(int iV4l2JpegFormat)
{
    return ExynosJpegBase::setJpegFormat(MODE_DECODE, iV4l2JpegFormat);
}

int ExynosJpegDecoder::updateConfig(void)
{
    return ExynosJpegBase::updateConfig(MODE_DECODE, \
        NUM_JPEG_DEC_IN_BUFS, NUM_JPEG_DEC_OUT_BUFS, \
        NUM_JPEG_DEC_IN_PLANES, NUM_JPEG_DEC_OUT_PLANES);
}

int ExynosJpegDecoder::setScaledSize(int iW, int iH)
{
    if (t_bFlagCreate == false) {
        return ERROR_JPEG_DEVICE_NOT_CREATE_YET;
    }

    if (iW < 0 || MAX_JPG_WIDTH < iW) {
        return ERROR_INVALID_IMAGE_SIZE;
    }

    if (iH < 0 || MAX_JPG_HEIGHT < iH) {
        return ERROR_INVALID_IMAGE_SIZE;
    }

    t_stJpegConfig.scaled_width = iW;
    t_stJpegConfig.scaled_height = iH;

    return ERROR_NONE;
}

int ExynosJpegDecoder::setJpegSize(int iJpegSize)
{
    if (t_bFlagCreate == false) {
        return ERROR_JPEG_DEVICE_NOT_CREATE_YET;
    }

    if (iJpegSize<=0) {
        return ERROR_JPEG_SIZE_TOO_SMALL;
    }

    t_stJpegConfig.sizeJpeg = iJpegSize;

    return ERROR_NONE;
}

int ExynosJpegDecoder::decode(void)
{
#ifdef WA_BLOCKING_ARTIFACT
    int ret = ExynosJpegBase::execute(NUM_JPEG_DEC_OUT_PLANES, t_iPlaneNum);

    if (ret == ERROR_NONE) {
        reduceBlockingArtifact((unsigned char *)t_stJpegOutbuf.addr[0],
                                    t_stJpegConfig.pix.dec_fmt.out_fmt,
                                    t_stJpegConfig.scaled_width,
                                    t_stJpegConfig.scaled_height);
    }

    return ret;
#else // WA_BLOCKING_ARTIFACT
    return ExynosJpegBase::execute(NUM_JPEG_DEC_OUT_PLANES, t_iPlaneNum);
#endif // WA_BLOCKING_ARTIFACT
}

#ifdef WA_BLOCKING_ARTIFACT

#define ABS(a)          (((a) < 0) ? (-(a)) : (a))
#define MIN(a, b)       (((a) < (b)) ? (a) : (b))
#define MAX(a, b)       (((a) > (b)) ? (a) : (b))
#define CLIP3(a, b, c)  (MIN(b, MAX(a, c)))
#define CLIP255(a)      (CLIP3(0, 255, (a)))

int Ytable[3][3][3][3][3][64] = {
{{{{{   2,  3,  2,  2,  2,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  3,  1,  2,  2,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   2,  3,  2,  2,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } },
{{{   2,  2,  1,  2,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0, -1, -1,  0, -1, -1,  0,  0, -1,  0, -1, },
{   3,  3,  1,  2,  2,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   2,  2,  1,  2,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1,  0, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1,  0, -1, -1, -1, -1, } },
{{   2,  3,  2,  2,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  3,  2,  2,  2,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  3,  2,  2,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } },
{{{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  3,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  3,  2,  2,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } } },
{{{{   2,  2,  1,  2,  2,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  2,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0, -1, },
{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  1,  2,  1,  1,  0,  0,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, },
{   2,  3,  2,  2,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } },
{{{   2,  2,  1,  1,  1,  1,  0,  0,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   1,  2,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0, -1,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  1,  1,  1,  1,  0,  0,  1,  2,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, },
{   2,  2,  1,  2,  1,  1,  0,  0,  2,  2,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1,  0, -1, -1, } } },
{{{   2,  2,  1,  2,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1,  0, -1,  0, -1,  0, -1, -1, -1, -1, -1, -1,  0, },
{   2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1, -1, -1, -1, -1, -1, },
{   1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1, -1, },
{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   2,  2,  1,  2,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1,  0, -1, -1, },
{   2,  2,  2,  2,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } } },
{{{{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  3,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1,  0,  0, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1, -1,  0,  0, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   3,  3,  2,  3,  2,  2,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  3,  2,  2,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } },
{{{   2,  3,  1,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1,  0,  0,  0, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   1,  2,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   2,  3,  2,  2,  2,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } },
{{{   3,  3,  2,  2,  2,  2,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  3,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   2,  2,  1,  2,  2,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   3,  3,  2,  2,  2,  2,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  3,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } } } },
{{{{{   2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  2,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   2,  3,  2,  2,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } },
{{{   2,  2,  1,  2,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  1,  1,  1,  1,  0,  1,  1,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  2,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   1,  2,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   2,  2,  2,  2,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } },
{{{   2,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } } },
{{{{   2,  2,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1,  0, -1, -1, -1, },
{   1,  1,  0,  0,  1,  1,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0, },
{   2,  2,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, } },
{{   0,  1,  0,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   1,  2,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1,  0,  0, -1, },
{   0,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  2,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1,  0, -1, -1,  0, -1,  0, -1,  0, -1, } } },
{{{   2,  2,  1,  1,  1,  1,  0,  1,  2,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1,  0,  0,  0, -1, -1, -1, -1, -1, -1,  0, -1, },
{   1,  2,  0,  1,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1,  0, },
{   1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, } },
{{   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0, -1, } } },
{{{   2,  1,  1,  2,  2,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1, -1,  0, -1, -1, -1,  0, -1, -1, -1, -1, -1, },
{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1,  0, -1, -1, -1, -1, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, } },
{{   1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, },
{   1,  1,  1,  1,  0,  1,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, } } } },
{{{{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1,  0, -1, -1, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } },
{{{   2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  1,  2,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  1,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } },
{{{   2,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   1,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   2,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } } } },
{{{{{   3,  3,  2,  2,  2,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  1,  2,  1,  1,  0,  0,  2,  2,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   3,  3,  2,  2,  2,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  1,  1,  1,  0,  2,  2,  1,  2,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } },
{{{   3,  3,  1,  2,  2,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1, -1, -1, -1,  0,  0,  0, -1, -1, -1, -1, -1, },
{   2,  3,  1,  2,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, } },
{{   2,  2,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1,  0, -1, -1, -1, -1, },
{   2,  2,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0, -1,  0, -1, -1, } },
{{   2,  3,  1,  2,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  2,  1,  2,  1,  1,  0,  0,  1,  2,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } },
{{{   3,  3,  2,  3,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  3,  2,  2,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, } },
{{   3,  3,  2,  3,  2,  2,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  3,  2,  2,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } } },
{{{{   3,  3,  1,  2,  2,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  0,  0,  2,  2,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0,  0, -1,  0, -1, -1, -1, -1, } },
{{   1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  3,  1,  2,  1,  1,  0,  0,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  0,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  0,  0,  2,  2,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } },
{{{   2,  2,  1,  2,  1,  1,  0,  0,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1, -1, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  1,  1,  1,  0,  0,  2,  2,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0,  0, -1, -1, -1, -1, -1, -1, } },
{{   0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, },
{   0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  1,  2,  1,  1,  0,  0,  1,  2,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   1,  2,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1,  0, -1,  0,  0, -1, },
{   2,  2,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1,  0, -1, -1, -1,  0, -1, -1, } } },
{{{   2,  3,  1,  2,  2,  1,  1,  0,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, },
{   1,  1,  1,  1,  0,  1,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, },
{   1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1, -1, -1, -1, -1, -1, } },
{{   2,  2,  1,  2,  1,  1,  1,  0,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   1,  2,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } } },
{{{{   3,  4,  2,  3,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  3,  2,  2,  1,  0,  2,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, } },
{{   2,  2,  1,  2,  2,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1,  0, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1,  0,  0, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1,  0,  0, -1,  0, -1, -1, -1, -1, } },
{{   3,  3,  2,  3,  2,  2,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  3,  2,  2,  1,  0,  2,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } },
{{{   3,  3,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  3,  1,  2,  2,  2,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  1,  2,  1,  1,  1,  0,  2,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   1,  2,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0, -1, -1, },
{   1,  2,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0, -1, -1, },
{   1,  2,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   3,  3,  2,  2,  2,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   1,  1,  1,  1,  0,  1,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  3,  2,  2,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } },
{{{   3,  3,  2,  3,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  3,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   2,  2,  2,  2,  2,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } },
{{   3,  3,  2,  3,  2,  2,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  2,  2,  1,  2,  1,  0,  2,  2,  1,  2,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, } } } } }};


int Utable[3][3][3][3][3][64] = {
    {{{{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1,  0,  0,  0,  0,  0, -1,  0, -1,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1,  0,  0,  0, -1,  0,  0,  0, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1, -1,  0,  0, -1,  0,  0,  0, -1, -1, } },
{{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1,  0,  0,  0, -1,  0,  0,  0, -1,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0, -1, -1,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1,  0,  0,  0, -1,  0,  0,  0,  0,  0, } } },
{{{   3,  3,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, },
{   3,  3,  1,  1,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, } },
{{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  3,  2,  2,  1,  1,  1,  1,  2,  2,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1,  0,  0,  0,  0,  0, },
{   2,  3,  2,  2,  2,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1,  0,  0,  0,  0,  0, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, } },
{{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } } },
{{{{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, },
{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, } },
{{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  1,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, },
{   3,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, } },
{{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0, },
{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0, },
{   1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1,  0, -1, -1,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  2,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  1,  1,  2,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1,  0, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  2,  2,  2,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } } },
{{{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  3,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  0,  1,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, },
{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  2,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  1,  1,  0,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   2,  3,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  3,  2,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   2,  3,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  2,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1, -1, -1, },
{   2,  1,  1,  1,  1,  1,  0,  0,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  1,  0,  1,  1,  0,  0,  1,  0,  1,  1,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, },
{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1,  0, },
{   2,  2,  1,  1,  1,  1,  0,  0,  2,  2,  1,  2,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   3,  3,  2,  2,  2,  1,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1,  0, },
{   2,  2,  1,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1, -1, -1, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1, -1, -1,  0, -1, -1, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1, -1, } },
{{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1,  0, -1, -1,  0,  0, -1, -1, },
{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0, },
{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1,  0,  0, -1,  0, } } } } },
{{{{{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0, -1, -1,  0, -1,  0,  0,  0,  0, -1, -1, -1,  0,  0, -1,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0, -1, -1,  0, -1, -1, -1,  0,  0, -1, -1,  0, -1,  0,  0,  0,  0, },
{   3,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  1,  1,  1,  1,  1,  2,  2,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1,  0,  0,  0, -1,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  2,  2,  1,  2,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1, -1, -1,  0, -1, -1,  0,  0, -1, -1,  0, },
{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1,  0, -1,  0, -1,  0,  0,  0,  0,  0, } } },
{{{   3,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, },
{   2,  3,  1,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  3,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0, } } },
{{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0, },
{   3,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } } },
{{{{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  1,  1,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  2,  1,  1,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  1,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  1,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0,  0,  0, },
{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1,  0, -1, -1, },
{   2,  2,  1,  1,  1,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1,  0,  0,  0, } },
{{   0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0, },
{   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0, } },
{{   1,  2,  1,  1,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  2,  1,  2,  2,  2,  1,  2,  1,  2,  2,  1,  1,  1,  0,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0, -1,  0, -1,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1,  0,  0,  0,  0, -1, } } },
{{{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, -1, -1,  1,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1, -1,  0,  0, },
{   2,  2,  1,  2,  2,  1,  1,  1,  2,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   2,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   2,  2,  2,  2,  1,  2,  2,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1,  0,  0, -1, },
{   2,  2,  1,  1,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  2,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1,  0,  0, -1, } } } },
{{{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  2,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1, -1,  0, -1, },
{   2,  3,  1,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  0,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  1,  0,  1,  1,  0,  0,  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, },
{   2,  2,  1,  1,  1,  1,  1,  0,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, } },
{{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  2,  1,  1,  1,  1,  0,  1,  0,  1,  0,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, -1, -1,  0,  0,  0,  0, -1, -1, -1, -1,  0,  0, -1,  0,  0,  0, } } },
{{{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, },
{   2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0, -1, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0, -1, -1, } },
{{   1,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1,  0, -1,  0,  0,  0, },
{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1,  0,  0, -1,  0, -1, -1,  0, -1, -1,  0,  0,  0, } } } } },
{{{{{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1,  0,  0,  0,  0, -1, -1, -1, -1, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0,  0,  0,  0, -1,  0, -1, -1, },
{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1,  0,  0,  0,  0, -1,  0, -1, -1, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0,  0, -1,  0, -1,  0, },
{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, } },
{{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, },
{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, } },
{{   2,  3,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0, -1, -1, -1,  0,  0,  0,  0,  0, },
{   2,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  0,  2,  2,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  1, -1,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1, -1,  0,  0,  0,  0, },
{   2,  3,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0, -1, -1,  0,  0, -1,  0,  0,  0, -1, -1,  0, -1,  0,  0,  0,  0, } } },
{{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  0,  1,  1,  1,  0,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, } },
{{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  2,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0, } } } },
{{{{   2,  3,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  3,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, } },
{{   1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, },
{   1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  1,  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, },
{   2,  3,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, },
{   2,  2,  1,  2,  1,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, } },
{{   1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0, },
{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  1,  0,  1,  0,  0,  1,  1,  0,  1,  0,  1,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0, },
{   1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  2,  1,  2,  1,  1,  1,  0,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  1,  1,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, } },
{{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  0,  1,  1,  1,  2,  1,  2,  1,  0,  2,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  1,  1,  1,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1,  0, -1, -1,  0,  0,  0, },
{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1,  0,  0, -1, } } } },
{{{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, } },
{{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  2,  2,  2,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  0,  1,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, },
{   3,  3,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  0,  1,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, },
{   2,  2,  1,  2,  1,  1,  0,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  3,  2,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1,  0, -1,  0, } } },
{{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1,  0,  0,  0,  0, -1, -1, -1, -1, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1,  0, -1,  0, -1, -1, },
{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1,  0, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1,  0, -1, -1, -1, -1, } },
{{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0,  0,  0, -1, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1,  0, -1, -1, } } } } }};


int Vtable[3][3][3][3][3][64] = {
    {{{{{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1,  0,  0,  0,  0,  0, -1,  0, -1,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1,  0,  0,  0, -1,  0,  0,  0, -1,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0, -1,  0, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0, -1, -1,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1,  0,  0,  0, -1,  0,  0,  0,  0,  0, } } },
{{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, },
{   3,  3,  1,  1,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, } },
{{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  3,  2,  2,  1,  1,  1,  1,  2,  2,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0, } } },
{{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  3,  2,  2,  2,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } } },
{{{{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, },
{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  0,  1,  1,  0,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  3,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0, },
{   2,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  1,  0,  1,  0,  1,  1,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  2,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0, },
{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0, },
{   1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  2,  1,  1,  2,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  0,  0,  1,  0,  1,  1, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1,  0, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } } },
{{{{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  3,  2,  2,  2,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  2,  2,  2,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   2,  3,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  2,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1,  0,  0, -1, },
{   2,  1,  0,  1,  1,  1,  0,  0,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  1,  1,  1,  0,  1,  1,  0,  0,  1,  1,  1,  0,  0,  1,  0,  0,  1,  0,  1,  1,  0,  0,  1,  0,  1,  1,  0,  1,  1,  0,  1,  0,  0,  1,  1,  0,  0,  0, },
{   3,  3,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, } },
{{   2,  2,  1,  2,  1,  1,  0,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, },
{   2,  2,  1,  1,  1,  1,  0,  0,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  1,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   3,  3,  2,  2,  1,  1,  1,  1,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1,  0,  0,  0, -1,  0, },
{   2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  3,  2,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0, -1,  0, } } },
{{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1,  0, -1, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1, -1, } },
{{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0,  0, -1, -1, },
{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1,  0,  0, -1,  0, } } } } },
{{{{{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0, -1, -1,  0, -1,  0,  0,  0,  0, -1, -1, -1,  0,  0, -1,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0, -1, -1,  0, -1, -1, -1,  0,  0, -1, -1,  0, -1,  0, -1,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  3,  2,  2,  1,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1,  0,  0,  0, -1,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  2,  2,  1,  2,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0, -1, -1,  0,  0, -1, -1, -1,  0, -1, -1,  0,  0, -1, -1,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0, -1,  0,  0,  0, -1,  0, -1,  0, -1,  0,  0,  0,  0,  0, } } },
{{{   3,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  3,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  2,  2,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1,  0,  0,  0,  0,  0,  0, } } },
{{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, } },
{{   2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } } },
{{{{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  1,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  1,  1,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  1,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1,  0,  0,  0, },
{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1,  0,  0,  0, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  1,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1,  0,  0,  0, } },
{{   0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0, },
{   0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0, } },
{{   1,  2,  1,  1,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  2,  1,  2,  2,  2,  1,  2,  1,  2,  2,  1,  1,  1,  0,  1,  0,  0,  0,  1,  1,  1,  1,  0,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1, -1,  0, -1, -1,  0, -1, -1,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  2,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1,  0,  0,  0,  0, -1, } } },
{{{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0, -1, -1,  1,  0,  0,  0,  0,  0, -1, -1,  0,  0, -1, -1,  0,  0, },
{   2,  2,  1,  2,  2,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  1,  1,  1, -1,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1,  0,  0, -1, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   2,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   2,  2,  2,  2,  2,  2,  2,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1,  0,  0, -1, },
{   1,  2,  1,  1,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  2,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1,  0, -1, -1, } } } },
{{{{   2,  3,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  2,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  0,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  1,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  1,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  2,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  1,  1,  1,  1,  0,  1,  1,  1,  0,  1,  1,  1,  0,  1,  1,  1,  1,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  1,  2,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, },
{   2,  2,  1,  1,  1,  1,  0,  1,  1,  2,  1,  2,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1,  0, -1,  0, -1,  0, },
{   1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  0,  1,  1,  1,  0,  0,  0,  1,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0, },
{   3,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  2,  1,  1,  1,  1,  0,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1, -1, -1,  0,  0,  0,  0, -1,  0, -1, -1,  0,  0, -1,  0,  0,  0, } } },
{{{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, },
{   2,  2,  1,  2,  2,  1,  1,  2,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0, -1, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0, } },
{{   1,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1, },
{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } },
{{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1,  0, -1,  0,  0,  0, },
{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1,  0,  0, -1,  0, -1, -1,  0, -1, -1,  0,  0,  0, } } } } },
{{{{{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0,  0,  0,  0, -1,  0, -1, -1, },
{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1,  0,  0,  0,  0, -1,  0, -1, -1, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  2,  1,  2,  2,  1,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1,  0, -1, -1,  0,  0, -1,  0, -1,  0, },
{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, } },
{{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, },
{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, } },
{{   2,  3,  2,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  1,  1,  1,  2,  2,  2,  2,  2,  1,  1,  0,  2,  2,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  1, -1,  0,  0,  0,  0,  0, -1, -1,  0, -1, -1, -1,  0,  0,  0,  0, },
{   2,  3,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  0, -1, -1,  0,  0, -1,  0,  0,  0,  0, -1,  0, -1,  0,  0,  0,  0, } } },
{{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, } },
{{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  2,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, } },
{{   2,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } } },
{{{{   2,  3,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  3,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, } },
{{   1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, },
{   1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  0,  1,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1,  0, },
{   2,  3,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0, } },
{{   1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  0,  1,  0,  0,  0,  1,  1,  1,  0,  1,  0,  0,  0,  1,  1,  1,  0,  1,  0,  0, },
{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0, },
{   1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0, } },
{{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   2,  2,  1,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  2,  2,  2,  2,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, },
{   2,  2,  1,  1,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, } },
{{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  1,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  1,  1,  0,  0,  0,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1,  0, -1, -1,  0,  0,  0, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, -1,  0,  0, -1, } } } },
{{{{   3,  3,  2,  2,  2,  2,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, } },
{{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  0,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  0,  1,  0,  0,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   2,  3,  2,  2,  2,  1,  1,  1,  2,  2,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  1,  2,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  1,  1,  0,  0,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, } } },
{{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  0,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, -1, },
{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  2,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0, -1,  0, -1, -1,  0, -1, },
{   2,  2,  1,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1, } },
{{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  2,  1,  1,  1,  2,  2,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  0,  1,  0,  1,  1,  1,  0,  0,  1,  0,  1,  0,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  1,  2,  1,  1,  0,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, } },
{{   3,  3,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  0,  1,  0,  1,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   3,  3,  2,  2,  1,  1,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1,  0, -1, -1, } } },
{{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0, -1,  0,  0,  0,  0, -1, -1, -1, -1, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1,  0, -1, -1, },
{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, },
{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  0,  1,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1, -1,  0,  0,  0,  0, -1, -1, -1, -1, } },
{{   2,  2,  2,  2,  2,  2,  1,  1,  2,  2,  1,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1,  0,  0,  0, -1, },
{   2,  2,  2,  2,  1,  1,  1,  1,  2,  2,  1,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, },
{   2,  2,  2,  2,  1,  2,  1,  1,  2,  2,  2,  2,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0, -1, -1, -1,  0, -1,  0, -1, -1, } } } } }};

char CLIPtable[6][256]={
    {  0,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,240,241,242,243,244,245,246,247,248,249,250,251,252,253,254},
    {  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255},
    {  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255,255},
    {  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255,255,255},
    {  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255,255,255,255},
    {  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99,100,101,102,103,104,105,106,107,108,109,110,111,112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255,255,255,255,255}};



inline void pixOperation(unsigned char *img_0, unsigned char *img_1, unsigned char *img_2, unsigned char *img_3, int idx[4][5], int img_offset_0, int img_offset_1)
{
    int ivalue[4];

    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][0]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][1]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][2]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][3]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][4]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][5]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][6]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][7]); *img_0 = ivalue[0]; img_0 += 2;

    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][0]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][1]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][2]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][3]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][4]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][5]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][6]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][7]); *img_1 = ivalue[1]; img_1 += 2;

    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][0]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][1]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][2]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][3]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][4]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][5]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][6]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][7]); *img_2 = ivalue[2]; img_2 += 4;

    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][0]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][1]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][2]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][3]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][4]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][5]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][6]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][7]); *img_3 = ivalue[3]; img_3 += 4;

    img_0 += img_offset_0; img_1 += img_offset_0;    img_2 += img_offset_1;    img_3 += img_offset_1;

    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][8] ); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][9] ); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][10]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][11]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][12]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][13]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][14]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][15]); *img_0 = ivalue[0]; img_0 += 2;

    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][8] ); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][9] ); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][10]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][11]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][12]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][13]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][14]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][15]); *img_1 = ivalue[1]; img_1 += 2;

    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][8] ); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][9] ); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][10]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][11]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][12]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][13]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][14]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][15]); *img_2 = ivalue[2]; img_2 += 4;

    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][8] ); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][9] ); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][10]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][11]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][12]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][13]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][14]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][15]); *img_3 = ivalue[3]; img_3 += 4;

    img_0 += img_offset_0; img_1 += img_offset_0;    img_2 += img_offset_1;    img_3 += img_offset_1;
/*
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][16]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][17]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][18]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][19]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][20]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][21]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][22]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][23]); *img_0 = ivalue[0]; img_0 += 2;

    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][16]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][17]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][18]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][19]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][20]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][21]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][22]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][23]); *img_1 = ivalue[1]; img_1 += 2;
*/
    img_0 += 16; img_1 += 16;

    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][16]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][17]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][18]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][19]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][20]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][21]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][22]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][23]); *img_2 = ivalue[2]; img_2 += 4;

    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][16]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][17]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][18]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][19]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][20]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][21]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][22]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][23]); *img_3 = ivalue[3]; img_3 += 4;

    img_0 += img_offset_0; img_1 += img_offset_0;    img_2 += img_offset_1;    img_3 += img_offset_1;

    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][24]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][25]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][26]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][27]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][28]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][29]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][30]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][31]); *img_0 = ivalue[0]; img_0 += 2;

    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][24]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][25]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][26]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][27]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][28]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][29]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][30]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][31]); *img_1 = ivalue[1]; img_1 += 2;

    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][24]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][25]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][26]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][27]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][28]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][29]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][30]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][31]); *img_2 = ivalue[2]; img_2 += 4;

    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][24]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][25]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][26]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][27]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][28]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][29]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][30]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][31]); *img_3 = ivalue[3]; img_3 += 4;

    img_0 += img_offset_0; img_1 += img_offset_0;    img_2 += img_offset_1;    img_3 += img_offset_1;
/*
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][32]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][33]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][34]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][35]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][36]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][37]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][38]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][39]); *img_0 = ivalue[0]; img_0 += 2;

    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][32]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][33]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][34]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][35]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][36]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][37]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][38]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][39]); *img_1 = ivalue[1]; img_1 += 2;
*/
    img_0 += 16; img_1 += 16;

    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][32]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][33]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][34]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][35]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][36]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][37]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][38]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][39]); *img_2 = ivalue[2]; img_2 += 4;

    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][32]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][33]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][34]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][35]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][36]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][37]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][38]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][39]); *img_3 = ivalue[3]; img_3 += 4;

    img_0 += img_offset_0; img_1 += img_offset_0;    img_2 += img_offset_1;    img_3 += img_offset_1;
/*
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][40]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][41]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][42]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][43]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][44]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][45]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][46]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][47]); *img_0 = ivalue[0]; img_0 += 2;

    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][40]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][41]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][42]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][43]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][44]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][45]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][46]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][47]); *img_1 = ivalue[1]; img_1 += 2;
*/
    img_0 += 16; img_1 += 16;

    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][40]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][41]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][42]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][43]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][44]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][45]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][46]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][47]); *img_2 = ivalue[2]; img_2 += 4;

    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][40]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][41]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][42]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][43]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][44]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][45]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][46]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][47]); *img_3 = ivalue[3]; img_3 += 4;

    img_0 += img_offset_0; img_1 += img_offset_0;    img_2 += img_offset_1;    img_3 += img_offset_1;

    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][48]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][49]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][50]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][51]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][52]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][53]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][54]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][55]); *img_0 = ivalue[0]; img_0 += 2;

    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][48]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][49]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][50]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][51]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][52]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][53]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][54]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][55]); *img_1 = ivalue[1]; img_1 += 2;

    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][48]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][49]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][50]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][51]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][52]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][53]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][54]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][55]); *img_2 = ivalue[2]; img_2 += 4;

    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][48]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][49]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][50]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][51]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][52]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][53]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][54]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][55]); *img_3 = ivalue[3]; img_3 += 4;

    img_0 += img_offset_0; img_1 += img_offset_0;    img_2 += img_offset_1;    img_3 += img_offset_1;

    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][56]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][57]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][58]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][59]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][60]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][61]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][62]); *img_0 = ivalue[0]; img_0 += 2;
    ivalue[0] = CLIP255(*img_0+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][63]); *img_0 = ivalue[0]; img_0 += 2;

    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][56]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][57]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][58]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][59]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][60]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][61]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][62]); *img_1 = ivalue[1]; img_1 += 2;
    ivalue[1] = CLIP255(*img_1+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][63]); *img_1 = ivalue[1]; img_1 += 2;

    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][56]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][57]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][58]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][59]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][60]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][61]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][62]); *img_2 = ivalue[2]; img_2 += 4;
    ivalue[2] = CLIP255(*img_2+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][63]); *img_2 = ivalue[2]; img_2 += 4;

    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][56]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][57]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][58]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][59]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][60]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][61]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][62]); *img_3 = ivalue[3]; img_3 += 4;
    ivalue[3] = CLIP255(*img_3+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][63]); *img_3 = ivalue[3]; img_3 += 4;

    img_0 += img_offset_0; img_1 += img_offset_0;    img_2 += img_offset_1;    img_3 += img_offset_1;

}

inline void fwHDT4x4(int *inblk4x4, int *outblk4x4)
{
 int tmpblk4x4[16];

 int add_0_8 = inblk4x4[0] + inblk4x4[8];
 int add_4_12 = inblk4x4[4] + inblk4x4[12];
 int add_2_10 = inblk4x4[2] + inblk4x4[10];
 int add_6_14 = inblk4x4[6] + inblk4x4[14];
 int add_1_9 = inblk4x4[1] + inblk4x4[9];
 int add_5_13 = inblk4x4[5] + inblk4x4[13];
 int add_3_11 = inblk4x4[3] + inblk4x4[11];
 int add_7_15 = inblk4x4[7] + inblk4x4[15];
 int add_tmp_4_6;
 int add_tmp_5_7;

 // First, Vertical Hadamard Transform
 tmpblk4x4[0]  = (add_0_8 + add_4_12);
 tmpblk4x4[2]  = (add_2_10 + add_6_14);
 tmpblk4x4[1]  = (add_1_9 + add_5_13);
 tmpblk4x4[3]  = (add_3_11 + add_7_15);

 tmpblk4x4[4]  = (add_0_8 - (add_4_12));
 tmpblk4x4[6]  = (add_2_10 - (add_6_14));
 tmpblk4x4[5]  = (add_1_9 - (add_5_13));
 tmpblk4x4[7]  = (add_3_11 - (add_7_15));

 tmpblk4x4[8]  = (inblk4x4[0] - inblk4x4[8] + inblk4x4[4] - inblk4x4[12]);
 tmpblk4x4[10] = (inblk4x4[2] - inblk4x4[10] + inblk4x4[6] - inblk4x4[14]);
 tmpblk4x4[9]  = (inblk4x4[1] - inblk4x4[9] + inblk4x4[5] - inblk4x4[13]);
 tmpblk4x4[11] = (inblk4x4[3] - inblk4x4[11] + inblk4x4[7] - inblk4x4[15]);


 add_tmp_4_6 = tmpblk4x4[4] + tmpblk4x4[6];
 add_tmp_5_7 = tmpblk4x4[5] + tmpblk4x4[7];

 // Second, Horizontal Hadamard Transform
 outblk4x4[0]  = (tmpblk4x4[0] + tmpblk4x4[2] - tmpblk4x4[1] - tmpblk4x4[3]);
 outblk4x4[1]  = (tmpblk4x4[0] - tmpblk4x4[2] + tmpblk4x4[1] - tmpblk4x4[3]);
 outblk4x4[2]  = (add_tmp_4_6 + add_tmp_5_7);
 outblk4x4[3]  = (add_tmp_4_6 - add_tmp_5_7);
 outblk4x4[4]  = (tmpblk4x4[8] + tmpblk4x4[10] + tmpblk4x4[9] + tmpblk4x4[11]);
}

void ExynosJpegDecoder::reduceBlockingArtifact(unsigned char *addr, int iColor, int width, int height)
{
    if (iColor != V4L2_PIX_FMT_YUYV) {
        return;
    }

    int i, j, x, y;

    int width2;

    int index_i, index_j;

    unsigned char *image = NULL;

    unsigned char *image_p1;
    unsigned char *image_p3;
    unsigned char *image_p16;

    unsigned char *image_ptr[4];
    unsigned char *img[4];
    unsigned char *block[4];

    int y_mul_width2;

    int block_offset_0;
    int block_offset_1;
    int img_offset_0;
    int img_offset_1;

    int f_0[16];
    int f_1[16];
    int f_2[16];
    int f_3[16];
    int F[4][5];
    int idx[4][5];

    int ivalue[4];

    width2 = width<<1;
    block_offset_0 = (width2<<1)-16;
    block_offset_1 = (width2<<1)-32;
    img_offset_0 = width2-16;
    img_offset_1 = width2-32;

    image_p1 = addr + 1;
    image_p3 = addr + 3;
    image_p16 = addr + 16;

    for(y=8;y<height;y+=8){

        y_mul_width2 = y*width2;

        image_ptr[0]     = addr+y_mul_width2;
        image_ptr[1]     = image_p16+y_mul_width2;
        image_ptr[2]     = image_p1+y_mul_width2;
        image_ptr[3]     = image_p3+y_mul_width2;

        for(x=0;x<width2;x+=32){
            img[0] = image_ptr[0]+x;
            img[1] = image_ptr[1]+x;
            img[2] = image_ptr[2]+x;
            img[3] = image_ptr[3]+x;

            block[0] = img[0];
            block[1] = img[1];
            block[2] = img[2];
            block[3] = img[3];

            //j=0
            f_0[0]  = *block[0]; block[0] += 4;  f_1[0]  = *block[1]; block[1] += 4;  f_2[0]  = *block[2]; block[2] += 8;  f_3[0]  = *block[3]; block[3] += 8; //j=0, i=0
            f_0[1]  = *block[0]; block[0] += 4;  f_1[1]  = *block[1]; block[1] += 4;  f_2[1]  = *block[2]; block[2] += 8;  f_3[1]  = *block[3]; block[3] += 8; //j=0, i=1
            f_0[2]  = *block[0]; block[0] += 4;  f_1[2]  = *block[1]; block[1] += 4;  f_2[2]  = *block[2]; block[2] += 8;  f_3[2]  = *block[3]; block[3] += 8; //j=0, i=2
            f_0[3]  = *block[0]; block[0] += 4;  f_1[3]  = *block[1]; block[1] += 4;  f_2[3]  = *block[2]; block[2] += 8;  f_3[3]  = *block[3]; block[3] += 8; //j=0, i=3
            block[0] += block_offset_0;    block[1] += block_offset_0;    block[2] += block_offset_1;    block[3] += block_offset_1;
            //j=1
            f_0[4]  = *block[0]; block[0] += 4;  f_1[4]  = *block[1]; block[1] += 4;  f_2[4]  = *block[2]; block[2] += 8;  f_3[4]  = *block[3]; block[3] += 8; //j=1, i=0
            f_0[5]  = *block[0]; block[0] += 4;  f_1[5]  = *block[1]; block[1] += 4;  f_2[5]  = *block[2]; block[2] += 8;  f_3[5]  = *block[3]; block[3] += 8; //j=1, i=1
            f_0[6]  = *block[0]; block[0] += 4;  f_1[6]  = *block[1]; block[1] += 4;  f_2[6]  = *block[2]; block[2] += 8;  f_3[6]  = *block[3]; block[3] += 8; //j=1, i=2
            f_0[7]  = *block[0]; block[0] += 4;  f_1[7]  = *block[1]; block[1] += 4;  f_2[7]  = *block[2]; block[2] += 8;  f_3[7]  = *block[3]; block[3] += 8; //j=1, i=3
            block[0] += block_offset_0;    block[1] += block_offset_0;    block[2] += block_offset_1;    block[3] += block_offset_1;
            //j=2
            f_0[8]  = *block[0]; block[0] += 4;  f_1[8]  = *block[1]; block[1] += 4;  f_2[8]  = *block[2]; block[2] += 8;  f_3[8]  = *block[3]; block[3] += 8; //j=2, i=0
            f_0[9]  = *block[0]; block[0] += 4;  f_1[9]  = *block[1]; block[1] += 4;  f_2[9]  = *block[2]; block[2] += 8;  f_3[9]  = *block[3]; block[3] += 8; //j=2, i=1
            f_0[10] = *block[0]; block[0] += 4;  f_1[10] = *block[1]; block[1] += 4;  f_2[10] = *block[2]; block[2] += 8;  f_3[10] = *block[3]; block[3] += 8; //j=2, i=2
            f_0[11] = *block[0]; block[0] += 4;  f_1[11] = *block[1]; block[1] += 4;  f_2[11] = *block[2]; block[2] += 8;  f_3[11] = *block[3]; block[3] += 8; //j=2, i=3
            block[0] += block_offset_0;    block[1] += block_offset_0;    block[2] += block_offset_1;    block[3] += block_offset_1;
            //j=3
            f_0[12] = *block[0]; block[0] += 4;  f_1[12] = *block[1]; block[1] += 4;  f_2[12] = *block[2]; block[2] += 8;  f_3[12] = *block[3]; block[3] += 8; //j=3, i=0
            f_0[13] = *block[0]; block[0] += 4;  f_1[13] = *block[1]; block[1] += 4;  f_2[13] = *block[2]; block[2] += 8;  f_3[13] = *block[3]; block[3] += 8; //j=3, i=1
            f_0[14] = *block[0]; block[0] += 4;  f_1[14] = *block[1]; block[1] += 4;  f_2[14] = *block[2]; block[2] += 8;  f_3[14] = *block[3]; block[3] += 8; //j=3, i=2
            f_0[15] = *block[0]; block[0] += 4;  f_1[15] = *block[1]; block[1] += 4;  f_2[15] = *block[2]; block[2] += 8;  f_3[15] = *block[3]; block[3] += 8; //j=3, i=3
            block[0] += block_offset_0;    block[1] += block_offset_0;    block[2] += block_offset_1;    block[3] += block_offset_1;

            //i=0
            fwHDT4x4(f_0, F[0]);
            idx[0][0] = (F[0][0]==0)?1:((F[0][0]<0)?0:2);
            idx[0][1] = (F[0][1]==0)?1:((F[0][1]<0)?0:2);
            idx[0][2] = (F[0][2]==0)?1:((F[0][2]<0)?0:2);
            idx[0][3] = (F[0][3]==0)?1:((F[0][3]<0)?0:2);
            idx[0][4] = (F[0][4]==0)?1:((F[0][4]<0)?0:2);

            //i=1
            fwHDT4x4(f_1, F[1]);
            idx[1][0] = (F[1][0]==0)?1:((F[1][0]<0)?0:2);
            idx[1][1] = (F[1][1]==0)?1:((F[1][1]<0)?0:2);
            idx[1][2] = (F[1][2]==0)?1:((F[1][2]<0)?0:2);
            idx[1][3] = (F[1][3]==0)?1:((F[1][3]<0)?0:2);
            idx[1][4] = (F[1][4]==0)?1:((F[1][4]<0)?0:2);

            //i=2
            fwHDT4x4(f_2, F[2]);
            idx[2][0] = (F[2][0]==0)?1:((F[2][0]<0)?0:2);
            idx[2][1] = (F[2][1]==0)?1:((F[2][1]<0)?0:2);
            idx[2][2] = (F[2][2]==0)?1:((F[2][2]<0)?0:2);
            idx[2][3] = (F[2][3]==0)?1:((F[2][3]<0)?0:2);
            idx[2][4] = (F[2][4]==0)?1:((F[2][4]<0)?0:2);

            //i=3
            fwHDT4x4(f_3, F[3]);
            idx[3][0] = (F[3][0]==0)?1:((F[3][0]<0)?0:2);
            idx[3][1] = (F[3][1]==0)?1:((F[3][1]<0)?0:2);
            idx[3][2] = (F[3][2]==0)?1:((F[3][2]<0)?0:2);
            idx[3][3] = (F[3][3]==0)?1:((F[3][3]<0)?0:2);
            idx[3][4] = (F[3][4]==0)?1:((F[3][4]<0)?0:2);

            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][0]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][1]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][2]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][3]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][4]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][5]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][6]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][7]); *img[0] = ivalue[0]; img[0] += 2;

            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][0]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][1]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][2]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][3]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][4]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][5]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][6]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][7]); *img[1] = ivalue[1]; img[1] += 2;

            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][0]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][1]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][2]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][3]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][4]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][5]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][6]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][7]); *img[2] = ivalue[2]; img[2] += 4;

            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][0]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][1]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][2]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][3]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][4]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][5]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][6]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][7]); *img[3] = ivalue[3]; img[3] += 4;

            img[0] += img_offset_0; img[1] += img_offset_0;    img[2] += img_offset_1;    img[3] += img_offset_1;

            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][8] ); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][9] ); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][10]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][11]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][12]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][13]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][14]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][15]); *img[0] = ivalue[0]; img[0] += 2;

            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][8] ); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][9] ); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][10]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][11]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][12]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][13]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][14]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][15]); *img[1] = ivalue[1]; img[1] += 2;

            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][8] ); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][9] ); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][10]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][11]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][12]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][13]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][14]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][15]); *img[2] = ivalue[2]; img[2] += 4;

            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][8] ); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][9] ); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][10]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][11]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][12]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][13]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][14]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][15]); *img[3] = ivalue[3]; img[3] += 4;

            img[0] += img_offset_0; img[1] += img_offset_0;    img[2] += img_offset_1;    img[3] += img_offset_1;

            img[0] += 16; img[1] += 16;

            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][16]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][17]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][18]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][19]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][20]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][21]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][22]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][23]); *img[2] = ivalue[2]; img[2] += 4;

            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][16]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][17]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][18]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][19]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][20]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][21]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][22]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][23]); *img[3] = ivalue[3]; img[3] += 4;

            img[0] += img_offset_0; img[1] += img_offset_0;    img[2] += img_offset_1;    img[3] += img_offset_1;

            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][24]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][25]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][26]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][27]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][28]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][29]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][30]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][31]); *img[0] = ivalue[0]; img[0] += 2;

            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][24]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][25]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][26]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][27]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][28]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][29]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][30]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][31]); *img[1] = ivalue[1]; img[1] += 2;
#if 1
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][24]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][25]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][26]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][27]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][28]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][29]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][30]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][31]); *img[2] = ivalue[2]; img[2] += 4;

            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][24]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][25]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][26]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][27]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][28]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][29]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][30]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][31]); *img[3] = ivalue[3]; img[3] += 4;
#else
            img[2] += 32;
            img[3] += 32;
#endif
            img[0] += img_offset_0; img[1] += img_offset_0;    img[2] += img_offset_1;    img[3] += img_offset_1;

            img[0] += 16; img[1] += 16;

            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][32]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][33]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][34]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][35]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][36]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][37]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][38]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][39]); *img[2] = ivalue[2]; img[2] += 4;

            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][32]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][33]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][34]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][35]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][36]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][37]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][38]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][39]); *img[3] = ivalue[3]; img[3] += 4;

            img[0] += img_offset_0; img[1] += img_offset_0;    img[2] += img_offset_1;    img[3] += img_offset_1;

            img[0] += 16; img[1] += 16;
#if 1
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][40]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][41]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][42]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][43]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][44]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][45]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][46]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][47]); *img[2] = ivalue[2]; img[2] += 4;

            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][40]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][41]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][42]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][43]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][44]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][45]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][46]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][47]); *img[3] = ivalue[3]; img[3] += 4;
#else
            img[2] += 32;
            img[3] += 32;
#endif
            img[0] += img_offset_0; img[1] += img_offset_0;    img[2] += img_offset_1;    img[3] += img_offset_1;

            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][48]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][49]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][50]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][51]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][52]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][53]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][54]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][55]); *img[0] = ivalue[0]; img[0] += 2;

            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][48]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][49]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][50]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][51]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][52]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][53]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][54]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][55]); *img[1] = ivalue[1]; img[1] += 2;

            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][48]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][49]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][50]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][51]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][52]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][53]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][54]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][55]); *img[2] = ivalue[2]; img[2] += 4;

            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][48]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][49]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][50]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][51]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][52]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][53]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][54]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][55]); *img[3] = ivalue[3]; img[3] += 4;

            img[0] += img_offset_0; img[1] += img_offset_0;    img[2] += img_offset_1;    img[3] += img_offset_1;

            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][56]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][57]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][58]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][59]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][60]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][61]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][62]); *img[0] = ivalue[0]; img[0] += 2;
            ivalue[0] = CLIP255(*img[0]+Ytable[idx[0][0]][idx[0][1]][idx[0][2]][idx[0][3]][idx[0][4]][63]); *img[0] = ivalue[0]; img[0] += 2;

            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][56]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][57]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][58]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][59]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][60]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][61]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][62]); *img[1] = ivalue[1]; img[1] += 2;
            ivalue[1] = CLIP255(*img[1]+Ytable[idx[1][0]][idx[1][1]][idx[1][2]][idx[1][3]][idx[1][4]][63]); *img[1] = ivalue[1]; img[1] += 2;

            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][56]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][57]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][58]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][59]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][60]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][61]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][62]); *img[2] = ivalue[2]; img[2] += 4;
            ivalue[2] = CLIP255(*img[2]+Utable[idx[2][0]][idx[2][1]][idx[2][2]][idx[2][3]][idx[2][4]][63]); *img[2] = ivalue[2]; img[2] += 4;

            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][56]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][57]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][58]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][59]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][60]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][61]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][62]); *img[3] = ivalue[3]; img[3] += 4;
            ivalue[3] = CLIP255(*img[3]+Vtable[idx[3][0]][idx[3][1]][idx[3][2]][idx[3][3]][idx[3][4]][63]); *img[3] = ivalue[3]; img[3] += 4;

            img[0] += img_offset_0; img[1] += img_offset_0;    img[2] += img_offset_1;    img[3] += img_offset_1;

        }//x
    }//y
}

#endif // WA_BLOCKING_ARTIFACT
