#!/usr/bin/python3
# encoding=utf-8
#
# Copyright © 2015-2016 Simon McVittie <smcv@debian.org>
#             2015-2016 Alexandre Detiste <alexandre@detiste.be>
#             2016 Stephen Kitt
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
# You can find the GPL license text on a Debian system under
# /usr/share/common-licenses/GPL-2.

import configparser
import logging
import os
import re

from .. import GameData
from ..build import (PackagingTask)
from ..util import (TemporaryUmask,
                    which,
                    mkdir_p)
from ..version import (FORMAT)

logger = logging.getLogger('game-data-packager.games.z_code')

class ZCodeGameData(GameData):
    def __init__(self, shortname, data):
        super(ZCodeGameData, self).__init__(shortname, data)
        for package in self.packages.values():
            package.z_file = None
            for install in package.install:
                if re.match('^.z[12345678]$', os.path.splitext(install)[1]):
                    assert package.z_file is None
                    package.z_file = install
            assert package.z_file

        if self.engine is None:
            # RPM format can't handle alternatives and will ignore this
            self.engine = 'gargoyle-free | zcode-interpreter'
        if self.genre is None:
            self.genre = 'Adventure'

    def construct_task(self, **kwargs):
        return ZCodeTask(self, **kwargs)

class ZCodeTask(PackagingTask):
    def fill_extra_files(self, package, destdir):
        super(ZCodeTask, self).fill_extra_files(package, destdir)

        install_to = self.packaging.substitute(package.install_to,
                package.name).lstrip('/')

        with TemporaryUmask(0o022):
            appdir = os.path.join(destdir, 'usr/share/applications')
            mkdir_p(appdir)
            desktop = configparser.RawConfigParser()
            desktop.optionxform = lambda option: option
            desktop['Desktop Entry'] = {}
            entry = desktop['Desktop Entry']
            entry['Type'] = 'Application'
            entry['Categories'] = 'Game;'
            entry['GenericName'] = self.game.genre + ' Game'
            entry['Name'] = package.longname or self.game.longname
            engine = 'zcode-interpreter'
            entry['Terminal'] = 'false'
            if FORMAT != 'deb':
                # keep engines sorted by relevance
                for try_engine, terminal in (('gargoyle', False),
                                             ('gargoyle-free', False),
                                             ('frotz', True),
                                             ('nfrotz', True),
                                             ('fizmo', True),
                                             ('fizmo-cursenw', True),
                                             ('fizmo-console', True),
                                             ('zjip', True)):
                    if which(try_engine):
                        engine = try_engine
                        if terminal:
                            entry['Terminal'] = 'true'
                        break
                else:
                    engine = 'gargoyle'
            entry['TryExec'] = engine
            arg = '/' + install_to + '/' + package.z_file
            entry['Exec'] = engine + ' ' + arg

            pixdir = os.path.join(destdir, 'usr/share/pixmaps')
            if os.path.exists(os.path.join(pixdir, '%s.png' % self.game.shortname)):
                entry['Icon'] = self.game.shortname
            else:
                entry['Icon'] = 'utilities-terminal'

            if package.aliases:
                entry['Keywords'] = ';'.join(package.aliases) + ';'

            with open(os.path.join(appdir, '%s.desktop' % package.name),
                      'w', encoding='utf-8') as output:
                 desktop.write(output, space_around_delimiters=False)

            self.packaging.override_lintian(destdir, package.name,
                    'desktop-command-not-in-package',
                    'usr/share/applications/%s.desktop %s'
                     % (package.name, engine))

            engine = which(engine)
            bindir = os.path.join(destdir, self.packaging.BINDIR)
            mkdir_p(bindir)
            pgm = package.name[0:len(package.name)-len('-data')]
            path = os.path.join(bindir, pgm)
            with open(path, 'w') as f:
                 f.write('#!/bin/sh\n')
                 f.write('test -x %s && exec %s $@ %s\n' %
                         (engine, engine, arg))
                 f.write('echo "You need to install some Z-Code interpreter '
                               'like Frotz or Gargoyle to play this game"\n')
                 os.chmod(path, 0o755)

GAME_DATA_SUBCLASS = ZCodeGameData
