// Copyright (c) 2005, Rodrigo Braz Monteiro
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   * Redistributions of source code must retain the above copyright notice,
//     this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above copyright notice,
//     this list of conditions and the following disclaimer in the documentation
//     and/or other materials provided with the distribution.
//   * Neither the name of the Aegisub Group nor the names of its contributors
//     may be used to endorse or promote products derived from this software
//     without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// Aegisub Project http://www.aegisub.org/
//
// $Id$

/// @file aegisublocale.cpp
/// @brief Enumerate available locales for picking translation on Windows
/// @ingroup utility
///

#include "config.h"

#include "aegisublocale.h"

#ifndef AGI_PRE
#include <algorithm>
#include <functional>
#include <locale.h>

#include <wx/dir.h>
#include <wx/filename.h>
#include <wx/intl.h>
#include <wx/stdpaths.h>
#include <wx/choicdlg.h> // Keep this last so wxUSE_CHOICEDLG is set.
#endif

#include "standard_paths.h"

#ifndef AEGISUB_CATALOG
#define AEGISUB_CATALOG "aegisub"
#endif

wxTranslations *AegisubLocale::GetTranslations() {
	wxTranslations *translations = wxTranslations::Get();
	if (!translations) {
		wxTranslations::Set(translations = new wxTranslations);
		wxFileTranslationsLoader::AddCatalogLookupPathPrefix(StandardPaths::DecodePath("?data/locale/"));
	}
	return translations;
}

void AegisubLocale::Init(wxString const& language) {
	wxTranslations *translations = GetTranslations();
	translations->SetLanguage(language);
	translations->AddCatalog(AEGISUB_CATALOG);
	translations->AddStdCatalog();

	setlocale(LC_NUMERIC, "C");
	setlocale(LC_CTYPE, "C");
	active_language = language;
}

wxString AegisubLocale::PickLanguage() {
	wxArrayString langs = GetTranslations()->GetAvailableTranslations(AEGISUB_CATALOG);
	langs.insert(langs.begin(), "en_US");

	// Check if user local language is available, if so, make it first
	const wxLanguageInfo *info = wxLocale::GetLanguageInfo(wxLocale::GetSystemLanguage());
	if (info) {
		wxArrayString::iterator it = std::find(langs.begin(), langs.end(), info->CanonicalName);
		if (it != langs.end())
			std::rotate(langs.begin(), it, it + 1);
	}

	// Nothing to pick
	if (langs.empty()) return "";

	// Only one language, so don't bother asking the user
	if (langs.size() == 1 && !active_language)
		return langs[0];

	// Generate names
	wxArrayString langNames;
	for (size_t i = 0; i < langs.size(); ++i) {
		const wxLanguageInfo *info = wxLocale::FindLanguageInfo(langs[i]);
		if (info)
			langNames.push_back(wxLocale::GetLanguageName(info->Language));
		else
			langNames.push_back(langs[i]);
	}

	long style = wxDEFAULT_DIALOG_STYLE | wxRESIZE_BORDER | wxOK | wxCENTRE;
	if (!active_language.empty())
		style |= wxCANCEL;

	wxSingleChoiceDialog dialog(NULL, "Please choose a language:", "Language", langNames,
#if wxCHECK_VERSION(2, 9, 4)
			(void **)0,
#else
			0,
#endif
			style);
	if (dialog.ShowModal() == wxID_OK) {
		int picked = dialog.GetSelection();
		if (langs[picked] != active_language)
			return langs[picked];
	}

	return "";
}
