/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * rapath - print derivable path information from argus data.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#include <argus_client.h>
#include <rapath.h>

#include <math.h>


void RaProcessSrvRecord (struct ArgusServiceRecord *);

int RaInitialized = 0;
int RaPrintSVGOutput = 0;

extern int RaHistoStart;
extern int RaHistoEnd;

void
ArgusClientInit ()
{
   struct ArgusModeStruct *mode = NULL;
 
   if (!(RaInitialized)) {
      RaWriteOut = 0;
      RaCumulativeMerge = 1;
      hfield = 17;
      pfield = 8;
 
      bzero ((char *) RaPrintAlgorithms, sizeof(RaPrintAlgorithms));
 
      if ((mode = ArgusModeList) != NULL) {
         while (mode) {
            if (!(strcasecmp (mode->mode, "svg")))
               RaPrintSVGOutput++;

            mode = mode->nxt;
         }
      }

      bzero ((char *) RaSortAlgorithms, sizeof(RaSortAlgorithms));
      RaSortAlgorithms[0] = RaSortAlgorithmTable[RASORTSRCTTL];
 
      if ((RaModelerQueue = RaNewQueue()) == NULL)
         exit(0);
   
      if ((RaHashTable.array = (struct RaHashTableHeader **) ArgusCalloc (RA_HASHTABLESIZE,
                                    sizeof (struct RaHashTableHeader *))) != NULL) {
         RaHashTable.size = RA_HASHTABLESIZE;
      }

      if (Hflag) {
         if (!(RaHistoTimeParse (Hstr)))
            usage();
      }

      if (nflag) {
         hfield = 15;
         pfield =  5;
      }
      RaInitialized++;
   }
}


void
ArgusClientTimeout ()
{
   RaProcessQueue (RaModelerQueue, ARGUS_STATUS);

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));

   if (ArgusMinuteUpdate++ == 60) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s HashHdrs %d Hash %d Queue %d\n",
                     print_time(&ArgusGlobalTime), RaAllocHashTableHeaders,
                     RaHashTable.count, RaModelerQueue->count);
      ArgusMinuteUpdate = 1;
   }

   if (ArgusHourlyUpdate++ == 3600) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));
      ArgusHourlyUpdate = 1;
   }
#endif
}



int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{

   if ((sig >= 0) && (!RaParseCompleting)) {
      RaParseCompleting++;
      RaProcessQueue (RaModelerQueue, ARGUS_STOP);
      fflush(stdout);
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "RaParseComplete: returning\n");
#endif
}


void
parse_arg (int argc, char**argv)
{ 
}

void
usage ()
{
   extern char version[];

   fprintf (stderr, "Rapath Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options]\n", ArgusProgramName);

   fprintf (stderr, "options: -D <level>       specify debug level\n");
   fprintf (stderr, "         -n               don't convert numbers to names.\n");
   fprintf (stderr, "         -r <filelist>    read argus data <filelist>. '-' denotes stdin.\n");
   fprintf (stderr, "         -S <host[:port]> specify remote argus <host> and optional port number.\n");
   fprintf (stderr, "         -t <timerange>   specify <timerange> for reading records.\n");
   fprintf (stderr, "                 format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                          timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                              mm/dd[/yy]\n");
   fprintf (stderr, "                                              -%%d{yMhdms}\n");
   fprintf (stderr, "         -T <secs>        attach to remote server for T seconds.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "         -U <user/auth>   specify <user/auth> authentication information.\n");
#endif
   exit(1);
}

void RaPrintArgusPath (struct ArgusRecordStore *);

struct ArgusServiceRecord ArgusThisSrv;

void
RaProcessRecord (struct ArgusRecord *argus)
{
   if (argus->ahdr.type & ARGUS_MAR)
      return;

   else {
      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:
            switch (argus->argus_far.flow.ip_flow.ip_p) {
               case IPPROTO_UDP:
                  if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_RTP_FLOWTAG)
                     if (!((argus->argus_far.src.count > 3) || (argus->argus_far.dst.count > 3)))
                        argus->argus_far.flow.ip_flow.tp_p = 0;
                  break;

            }
            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
         default:
            return;
      }
   }

   if (((argus->argus_far.status & ARGUS_ICMPUNREACH_MAPPED) ||
        (argus->argus_far.status & ARGUS_ICMPTIMXCED_MAPPED)) &&
                         (ArgusThisFarStatus & ARGUS_ICMP_DSR_STATUS)) {

     argus->argus_far.flow.ip_flow.ip_id = 0;

      switch (argus->argus_far.flow.ip_flow.ip_p) {
         case IPPROTO_UDP:
         case IPPROTO_TCP:
            argus->argus_far.flow.ip_flow.sport = 0xFFFF;
            argus->argus_far.flow.ip_flow.dport = 0xFFFF;
            break;

         case IPPROTO_ICMP:
            break;

         default:
            break;
      }

      ArgusThisSrv.argus = argus;
      RaProcessSrvRecord (&ArgusThisSrv);
   }
}


void RaUpdateArgusStorePath(struct ArgusRecord *, struct ArgusRecordStore *);

void
RaProcessSrvRecord (struct ArgusServiceRecord *srv)
{
   struct ArgusRecord *argus = srv->argus;

   struct ArgusRecordStore *store, *qstr;
   int RaTimeout = -1, i = RaHistoTimeSeries;

   RaThisActiveDuration = RaGetActiveDuration(argus);

   if (Hflag) {
      if (RaThisActiveDuration < RaHistoStart)
         return;

      for (i = 0; i < RaHistoTimeSeries; i++) {
         if (RaThisActiveDuration <= RaHistoTimeValues[i]) {
            RaThisActiveIndex = i;
            break;
         }
      }
      if (i == RaHistoTimeSeries)
         return;
      else
         i++;
   }

   RaThisActiveIndex = i;

   if ((store = RaFindArgusRecord(&RaHashTable, argus)) != NULL) {
      RaThisArgusStore = store;

      if (RaCheckTimeout (store, argus))
         RaSendArgusRecord (store);

      if (!(store->data[0])) {
         struct ArgusRecordData *data = NULL;

         if ((data = RaNewArgusData(argus)) != NULL) {
            data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
            data->status |= RA_MODIFIED;
            data->status |= srv->status & RA_SVCTEST;

            if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
               double sumtime;

               bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], (char *)&data->agr, sizeof(data->agr));
               data->act.n        = data->agr.act.n;
               sumtime            = data->agr.act.meanval * data->agr.act.n;
               data->act.sumtime  = sumtime;
               data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

               data->idle.n       = data->agr.idle.n;
               sumtime            = data->agr.idle.meanval * data->agr.idle.n;
               data->idle.sumtime = sumtime;
               data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
            }

            store->data[0] = data;
         } else
            ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));

      } else
         RaMergeArgusRecord (argus, store, RaThisActiveIndex);

      RaUpdateArgusStorePath(argus, store);

   } else {
      if ((argus->argus_far.status & ARGUS_ICMPTIMXCED_MAPPED)) {
         if ((store = RaNewArgusStore(argus)) != NULL) {
            if ((store->queue = RaNewQueue()) != NULL) {
               RaAddToQueue(RaModelerQueue, &store->qhdr);

               if ((qstr = RaNewArgusStore(argus)) != NULL) {
                  RaAddToQueue(store->queue, &qstr->qhdr);
                  if ((qstr->data[0] = RaNewArgusData(argus)) != NULL) {
                     qstr->data[0]->farhdrstatus = ArgusIndexRecord (qstr->data[0]->argus, qstr->data[0]->farhdrs);
                     qstr->status |= RA_MODIFIED;
                  }
               }
   
               RaThisArgusStore = store;

               if ((store->data[0] = RaNewArgusData(argus)) != NULL) {
                  int tmp = RaThisActiveIndex;
                  store->data[0]->farhdrstatus = ArgusIndexRecord (store->data[0]->argus, store->data[0]->farhdrs);
                  store->status |= RA_MODIFIED;

                  RaThisActiveIndex = 0;
                  if ((store->rahtblhdr = RaAddHashEntry (&RaHashTable, store)) == NULL)
                     ArgusLog (LOG_ERR, "RaProcessSrvRecord RaAddHashEntry %s\n", strerror(errno));
                  RaThisActiveIndex = tmp;
               }
               store->ArgusTimeout = RaTimeout;
               store->status |= RA_MODIFIED;
               store->status |= srv->status & RA_SVCTEST;

            } else
               ArgusLog (LOG_ERR, "RaProcessSrvRecord RaNewQueue %s\n", strerror(errno));
         } else
            ArgusLog (LOG_ERR, "RaProcessSrvRecord RaNewArgusStore %s\n", strerror(errno));
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessSrvRecord: done\n");
#endif
}

char *RaPrintTimeStats (struct ArgusRecord *, char *);

char *
RaPrintTimeStats (struct ArgusRecord *argus, char *str)
{
   char *retn = NULL;
   struct ArgusAGRStruct *agr;

   ArgusThisFarStatus = ArgusIndexRecord (argus, ArgusThisFarHdrs);

   if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL) {
      retn = str;
      sprintf (retn, "%7d +/- %-6d   max %7d  min %7d   n = %d", agr->act.meanval, agr->act.stdev,
                                                  agr->act.maxval, agr->act.minval, agr->count);
   }

   return (retn);
}

int RaCompareArgusStore (const void *, const void *);
void RaPackQueue (struct RaQueueStruct *);
void RaSortQueue (struct RaQueueStruct *);

void
RaPrintArgusPath (struct ArgusRecordStore *store)
{
   char buf[MAXSTRLEN], *ptr = buf, *str = NULL;
   struct ArgusRecordStore *obj = NULL;
   struct ArgusRecordData *data = NULL;
   struct ArgusRecord *argus = store->data[0]->argus;
   unsigned int srchost, dsthost, intnode, ttl;
   char srcAddrString[64], dstAddrString[64], intAddrString[64];
   char statsbuf[256], *stats = statsbuf;
   char date[128];
   int i, len, pad, margin;

   str = get_ip_string (store->data[0]->argus);
   printf ("%s\n", str);
   len = strlen(str);
   
   ArgusPrintDate (date, store->data[0]->argus);

   pad = len - (strlen(date) + (3 * hfield) + (cflag ? ((4 * 10) + 3) : 0) + (Iflag ? 7 : 0) + (gflag ? 9 : 0));
 
   if (pad < hfield) {
      pad += 2 * hfield;
      hfield = pad / 3;
      pad = hfield;
   }

   if (store->queue) {
      RaSortQueue(store->queue);

      for (i = 0; i < store->queue->count; i++) {
         struct ArgusICMPObject *icmp = NULL;

         if ((obj = (struct ArgusRecordStore *) store->queue->array[i]) == NULL)
            ArgusLog (LOG_ERR, "RaSortQueue array error");

         icmp = (struct ArgusICMPObject *) obj->data[0]->farhdrs[ARGUS_ICMP_DSR_INDEX];

         data = obj->data[0];
         argus = data->argus;

         if (argus && (obj->status & RA_MODIFIED)) {
            if (data->act.n > 0) {
               data->agr.act.meanval = data->act.sumtime/data->act.n;
               data->agr.act.stdev = sqrt (data->act.sumsqrd/data->act.n - pow (data->act.sumtime/data->act.n, 2.0));
            }
            if (data->idle.n > 0) {
               data->agr.idle.meanval = data->idle.sumtime/data->idle.n;
               data->agr.idle.stdev = sqrt (data->idle.sumsqrd/data->idle.n - pow (data->idle.sumtime/data->idle.n, 2.0));
            }
      
            data->agr.type   = ARGUS_AGR_DSR;
            data->agr.length = sizeof(data->agr);
            bcopy ((char *) argus, buf, argus->ahdr.length);
            bcopy ((char *)&data->agr,&buf[argus->ahdr.length], data->agr.length);
      
            argus = (struct ArgusRecord *) ptr;
            argus->ahdr.length += data->agr.length;
         }

         srchost = argus->argus_far.flow.ip_flow.ip_src;
         dsthost = argus->argus_far.flow.ip_flow.ip_dst;

         intnode = icmp->osrcaddr;
   
         ttl = argus->argus_far.attr_ip.sttl;
   
         sprintf(srcAddrString, "%s", ipaddr_string(&srchost));
         sprintf(dstAddrString, "%s", ipaddr_string(&dsthost));
         sprintf(intAddrString, "%s", ipaddr_string(&intnode));

         stats = RaPrintTimeStats (argus, statsbuf);

         if (idflag)
            printf ("                ");

         margin = pad + ((2 * hfield) - (strlen(srcAddrString) + strlen(dstAddrString)));

         if (dsthost != intnode) {
            if (nflag)
               printf ("Path  %15.15s  ->  %15.15s    INode:  %15.15s  ",
                               srcAddrString, dstAddrString, intAddrString);
            else
               printf ("Path  %s  ->  %s    INode:  %*.*s  ",
                               srcAddrString, dstAddrString,
                               margin, margin, intAddrString);
         } else {
            if (nflag)
               printf ("Path  %15.15s  ->  %15.15s    TNode:  %15.15s  ",
                               srcAddrString, dstAddrString, intAddrString);
            else
               printf ("Path  %s  ->  %s    TNode:  %*.*s  ",
                               srcAddrString, dstAddrString,
                               margin, margin, intAddrString);
         }
         printf ("  Dis: %2d  %s\n", ttl, stats);
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaPrintArgusPath (0x%x) returning\n", store);
#endif
}


void
RaUpdateArgusStorePath(struct ArgusRecord *argus, struct ArgusRecordStore *store)
{
   int found = 0;
   struct ArgusRecordStore *str;
   struct ArgusICMPObject *argicmp = (struct ArgusICMPObject *) ArgusThisFarHdrs[ARGUS_ICMP_DSR_INDEX];

   store->status |= RA_MODIFIED;
   store->qhdr.lasttime = ArgusGlobalTime;

   if (store->queue) {
      str = (struct ArgusRecordStore *) store->queue->start;
      do {
         struct ArgusICMPObject *stricmp = (struct ArgusICMPObject *) str->data[0]->farhdrs[ARGUS_ICMP_DSR_INDEX];

         if ((argus->argus_far.attr_ip.sttl == str->data[0]->argus->argus_far.attr_ip.sttl)) {
            if ((stricmp && argicmp) && (stricmp->osrcaddr == argicmp->osrcaddr)) {
               RaMergeArgusRecord(argus, str, 0);
               found++;
               break;
            }
         }
         str = (struct ArgusRecordStore *) str->qhdr.nxt;
      } while (str != (struct ArgusRecordStore *) store->queue->start);

      if (!(found) && ((argus->argus_far.status & ARGUS_ICMPTIMXCED_MAPPED)  ||
                      ((argus->argus_far.status & ARGUS_ICMPUNREACH_MAPPED)  &&
                       (store->queue->count > 0)))) {
         if (store->queue && ((str = RaNewArgusStore(argus)) != NULL)) {
            if ((str->data[0] = RaNewArgusData(argus)) != NULL) {
               RaAddToQueue (store->queue, &str->qhdr);
               str->data[0]->farhdrstatus = ArgusIndexRecord (str->data[0]->argus, str->data[0]->farhdrs);
               str->status |= RA_MODIFIED;
            }
         }
      }

      if (str != NULL) {
         if (!(str->data[RaThisActiveIndex])) {
            struct ArgusRecordData *data = NULL;

            if ((data = RaNewArgusData(argus)) != NULL) {
               data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
               data->status |= RA_MODIFIED;

               if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
                  double sumtime;

                  bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], (char *)&data->agr, sizeof(data->agr));
                  data->act.n        = data->agr.act.n;
                  sumtime            = data->agr.act.meanval * data->agr.act.n;
                  data->act.sumtime  = sumtime;
                  data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

                  data->idle.n       = data->agr.idle.n;
                  sumtime            = data->agr.idle.meanval * data->agr.idle.n;
                  data->idle.sumtime = sumtime;
                  data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
               }

               str->data[RaThisActiveIndex] = data;
            } else
               ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));

         } else
            RaMergeArgusRecord(argus, str, RaThisActiveIndex);
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaUpdateArgusStore: \n");
#endif
}

void RaPrintSVGHeader(void);
void RaPrintSVGFooter(void);
void RaPrintSVGFunctions(struct ArgusRecordStore *);

void
RaPrintSVGHeader()
{
   extern char version[];

   fprintf (stdout, "<html>\n");
   fprintf (stdout, "<head>\n");
   fprintf (stdout, "<!-- Generated using %s %s -->\n", ArgusProgramName, version);
   fprintf (stdout, "<script language=\"Javascript\">\n");
   fprintf (stdout, "<!--\n");

   fprintf (stdout, "   var ChartType = \"Histogram\"\n");
   fprintf (stdout, "   var TotalElements = %d\n", RaHistoTimeSeries);
   fprintf (stdout, "   var WidthStart = %f\n", RaHistoStart/1000000.0);
   fprintf (stdout, "   var WidthStop  = %f\n", RaHistoEnd/1000000.0);
   fprintf (stdout, "   var XAxisTickElements = 5\n\n");

   fflush(stdout);
}

void
RaPrintSVGFunctions(struct ArgusRecordStore *store)
{
   char buf[MAXSTRLEN], *ptr = buf;
   struct ArgusRecordStore *obj = NULL;
   struct ArgusRecordData *data = NULL;
   struct ArgusRecord *argus = store->data[0]->argus;
   struct ArgusAGRStruct *agr = NULL;
   unsigned int srchost, dsthost, intnode, ttl, i, x;
   char srcAddrString[64], dstAddrString[64], intAddrString[64];
   char sdate[128], ldate[128];
   int RaSeriesNumber = 0;

   bzero(sdate, sizeof(sdate));
   bzero(ldate, sizeof(ldate));

   ArgusPrintStartDate (sdate, store->data[0]->argus);
   ArgusPrintLastDate (ldate, store->data[0]->argus);

   if (store->queue) {
      RaSortQueue(store->queue);

      fprintf (stdout, "   var SeriesNumber = %d\n\n", store->queue->count);

      fprintf (stdout, "   function Run() {\n");
      fprintf (stdout, "      Populate ()\n");
      fprintf (stdout, "      Display ()\n");
      fprintf (stdout, "   }\n\n");

      fprintf (stdout, "   function Populate() {\n");

      for (x = 0; x < store->queue->count; x++) {
         struct ArgusICMPObject *icmp = NULL;

         if ((obj = (struct ArgusRecordStore *) store->queue->array[x]) == NULL)
            ArgusLog (LOG_ERR, "RaSortQueue array error");

         icmp = (struct ArgusICMPObject *) obj->data[0]->farhdrs[ARGUS_ICMP_DSR_INDEX];

         data = obj->data[0];
         argus = data->argus;

         if (argus && (obj->status & RA_MODIFIED)) {
            if (data->act.n > 0) {
               data->agr.act.meanval = data->act.sumtime/data->act.n;
               data->agr.act.stdev = sqrt (data->act.sumsqrd/data->act.n - pow (data->act.sumtime/data->act.n, 2.0));
            }
            if (data->idle.n > 0) {
               data->agr.idle.meanval = data->idle.sumtime/data->idle.n;
               data->agr.idle.stdev = sqrt (data->idle.sumsqrd/data->idle.n - pow (data->idle.sumtime/data->idle.n, 2.0));
            }
   
            data->agr.type   = ARGUS_AGR_DSR;
            data->agr.length = sizeof(data->agr);
            bcopy ((char *) argus, buf, argus->ahdr.length);
            bcopy ((char *)&data->agr,&buf[argus->ahdr.length], data->agr.length);
   
            argus = (struct ArgusRecord *) ptr;
            argus->ahdr.length += data->agr.length;
         }

         srchost = argus->argus_far.flow.ip_flow.ip_src;
         dsthost = argus->argus_far.flow.ip_flow.ip_dst;

         intnode = icmp->osrcaddr;

         ttl = argus->argus_far.attr_ip.sttl;

         sprintf(srcAddrString, "%s", ipaddr_string(&srchost));
         sprintf(dstAddrString, "%s", ipaddr_string(&dsthost));
         sprintf(intAddrString, "%s", ipaddr_string(&intnode));

         if (dsthost != intnode) {
            ArgusThisFarStatus = ArgusIndexRecord (argus, ArgusThisFarHdrs);
 
            agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX];

            fprintf (stdout, "      window.addPathHost(%2d, %f, %f, %f, %f, %d)\n", RaSeriesNumber, 
                                                agr->act.meanval/1000000.0, agr->act.stdev/1000000.0,
                                                agr->act.maxval/1000000.0, agr->act.minval/1000000.0, agr->count);
            fprintf (stdout, "      window.addSeriesAttribute(%2d, \"%s\")\n", RaSeriesNumber, intAddrString);

         } else {
            fprintf (stdout, "      window.addPathTerminus(%2d, %f, %f, %f, %f, %d)\n", RaSeriesNumber, 
                                                agr->act.meanval/1000000.0, agr->act.stdev/1000000.0,
                                                agr->act.maxval/1000000.0, agr->act.minval/1000000.0, agr->count);
            fprintf (stdout, "      window.addSeriesAttribute(%2d, \"%s\")\n", RaSeriesNumber, dstAddrString);
         }

         for (i = 1; i < RaHistoTimeSeries; i++) {
            if ((data = obj->data[i]) != NULL) {
               argus = data->argus;

               if (argus && (obj->status & RA_MODIFIED)) {
                  if (data->act.n > 0) {
                     data->agr.act.meanval = data->act.sumtime/data->act.n;
                     data->agr.act.stdev = sqrt (data->act.sumsqrd/data->act.n - pow (data->act.sumtime/data->act.n, 2.0));
                  }
                  if (data->idle.n > 0) {
                     data->agr.idle.meanval = data->idle.sumtime/data->idle.n;
                     data->agr.idle.stdev = sqrt (data->idle.sumsqrd/data->idle.n - pow (data->idle.sumtime/data->idle.n, 2.0));
                  }

                  data->agr.type   = ARGUS_AGR_DSR;
                  data->agr.length = sizeof(data->agr);
                  bcopy ((char *) argus, buf, argus->ahdr.length);
                  bcopy ((char *)&data->agr,&buf[argus->ahdr.length], data->agr.length);

                  argus = (struct ArgusRecord *) ptr;
                  argus->ahdr.length += data->agr.length;
               }

               ArgusThisFarStatus = ArgusIndexRecord (argus, ArgusThisFarHdrs);

               agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX];

               fprintf (stdout, "      window.addArgusValue(%2d, %3d, %f, %d)\n", RaSeriesNumber, i - 1,
                                                agr->act.meanval/1000000.0, agr->count);
            }
         }

         obj = (struct ArgusRecordStore *) obj->qhdr.nxt;
         RaSeriesNumber++;
      }

      fprintf (stdout, "   }\n\n");
      fprintf (stdout, "   function Display() {\n");
      fprintf (stdout, "      window.setTitle(\"Path Statistics\")\n");
      fprintf (stdout, "      window.setSubTitle1(\"%s -> %s\")\n",srcAddrString, dstAddrString);
      fprintf (stdout, "      window.setSubTitle2(\"%s- %s\")\n",sdate, ldate);
      fprintf (stdout, "      window.setXCaption(\"Duration\")\n");
      fprintf (stdout, "      window.setAxis(\"Transactions\")\n");
      fprintf (stdout, "      window.DrawPathTerminus()\n");
      fprintf (stdout, "      window.setCircles()\n");
      fprintf (stdout, "      window.DrawSeriesLine()\n");
      fprintf (stdout, "   }\n\n");
   }
}

void
RaPrintSVGFooter()
{

   fprintf (stdout, "//-->\n");
   fprintf (stdout, "</script>\n");
   fprintf (stdout, "<title>Current Histogram Demo</title></head>\n");
   fprintf (stdout, "<body bgcolor=\"#999999\" onload=\"Run()\">\n");
   fprintf (stdout, "   <embed name=\"histogram\" ");
   fprintf (stdout, "src=\"histogram_long_multi.svg\" ");
   fprintf (stdout, "wmode=\"transparent\" ");
   fprintf (stdout, "width=\"800\" height=\"190\" ");
   fprintf (stdout, "type=\"image/svg+xml\" ");
   fprintf (stdout, "pluginspage=\"http://www.adobe.com/svg/viewer/install/\">\n");
   fprintf (stdout, "   </embed>\n");
   fprintf (stdout, "</body>\n");
   fprintf (stdout, "</html>\n");
   fflush(stdout);
}


#include <stdio.h>
#include <errno.h>

#define RA_MAXQSCAN  25600
#define RA_MAXQSIZE  250000
 
void
RaProcessQueue(struct RaQueueStruct *queue, unsigned char status)
{
   struct ArgusRecordStore *obj = NULL;
   int cnt = 0;
 
   switch (status) {
      case ARGUS_STOP: {
         if (queue->count > 0) {
            if (RaPrintSVGOutput) {
               if (queue->count == 1)
                  RaPrintSVGHeader();
               else
                  ArgusLog (LOG_ERR, "svg mode and multiple paths: error\n");
            }

            while ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL) {
               if (RaPrintSVGOutput) {
                  RaPrintSVGFunctions(obj);
               } else
                  RaTimeoutArgusStore(obj);
            }

            if (RaPrintSVGOutput)
               RaPrintSVGFooter();
         }

         break;
      }

      default:
         while (queue->count > RA_MAXQSIZE) {
            obj = (struct ArgusRecordStore *) RaRemoveFromQueue(RaModelerQueue, RaModelerQueue->start->prv);
            RaTimeoutArgusStore(obj);
         }

         if ((cnt = ((queue->count > RA_MAXQSCAN) ? RA_MAXQSCAN : queue->count)) != 0) {
            while (cnt--) {
               if ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL) {
                  if (RaCheckTimeout(obj, NULL))
                     RaTimeoutArgusStore(obj);
                  else
                     RaAddToQueue(queue, &obj->qhdr);

               } else
                  cnt++;
            }
         }
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessQueue (0x%x, %d) returning\n", queue, status);
#endif
}


int ArgusSeriesNumber = 0;

int
RaSendArgusRecord(struct ArgusRecordStore *store)
{
   int retn = 0;

   RaPrintArgusPath(store);

   store->qhdr.logtime = ArgusGlobalTime;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaSendArgusRecord(0x%x) done.\n", store);
#endif
   return (retn);
}
