//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Fit/FitSessionWidget.cpp
//! @brief     Implements class FitSessionWidget
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Fit/FitSessionWidget.h"
#include "Base/Util/Assert.h"
#include "GUI/View/Fit/FitParameterWidget.h"
#include "GUI/View/Fit/FitSessionController.h"
#include "GUI/View/Fit/MinimizerSettingsWidget.h"
#include "GUI/View/Fit/RunFitControlWidget.h"
#include "GUI/View/Tool/mainwindow_constants.h"
#include <QSettings>
#include <QTabWidget>
#include <QVBoxLayout>

FitSessionWidget::FitSessionWidget(QWidget* parent)
    : QWidget(parent)
    , m_controlWidget(new RunFitControlWidget)
    , m_fitParametersWidget(new FitParameterWidget)
    , m_minimizerSettingsWidget(new MinimizerSettingsWidget)
    , m_sessionController(nullptr)
{
    auto* layout = new QVBoxLayout(this);
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setSpacing(0);
    m_tabWidget = new QTabWidget(this);
    m_tabWidget->addTab(m_fitParametersWidget, "Fit Parameters");
    m_tabWidget->addTab(m_minimizerSettingsWidget, "Minimizer");

    layout->addWidget(m_tabWidget);
    layout->addWidget(m_controlWidget);

    applySettings();
}

FitSessionWidget::~FitSessionWidget()
{
    saveSettings();
}

void FitSessionWidget::setJobItem(JobItem* jobItem)
{
    ASSERT(jobItem);
    m_fitParametersWidget->setJobOrRealItem(jobItem);
    m_minimizerSettingsWidget->setJobItem(jobItem);
    m_controlWidget->setJobOrRealItem(jobItem);
}

void FitSessionWidget::setModelTuningWidget(ParameterTuningWidget* tuningWidget)
{
    ASSERT(m_fitParametersWidget);
    ASSERT(tuningWidget);
    m_fitParametersWidget->setParameterTuningWidget(tuningWidget);
}

void FitSessionWidget::setSessionController(FitSessionController* sessionController)
{
    if (m_sessionController) {
        disconnect(m_sessionController, nullptr, this, nullptr);
        disconnect(m_controlWidget, nullptr, m_sessionController, nullptr);
    }

    m_sessionController = sessionController;

    if (m_sessionController) {
        connect(m_sessionController, &FitSessionController::fittingError, this,
                &FitSessionWidget::onFittingError);
        connect(m_sessionController, &QObject::destroyed,
                [this] { m_sessionController = nullptr; });
        connect(m_controlWidget, &RunFitControlWidget::startFittingPushed, m_sessionController,
                &FitSessionController::onStartFittingRequest);
        connect(m_controlWidget, &RunFitControlWidget::stopFittingPushed, m_sessionController,
                &FitSessionController::onStopFittingRequest);
    }
}

void FitSessionWidget::onFittingError(const QString& text)
{
    m_controlWidget->onFittingError(text);
}

void FitSessionWidget::applySettings()
{
    QSettings settings;
    if (settings.childGroups().contains(GUI::Constants::S_FIT_SESSION_WIDGET)) {
        settings.beginGroup(GUI::Constants::S_FIT_SESSION_WIDGET);
        m_tabWidget->setCurrentIndex(
            settings.value(GUI::Constants::S_FIT_SESSION_WIDGET_CURRENT_TAB).toInt());
        settings.endGroup();
    }
}

void FitSessionWidget::saveSettings()
{
    QSettings settings;
    settings.beginGroup(GUI::Constants::S_FIT_SESSION_WIDGET);
    settings.setValue(GUI::Constants::S_FIT_SESSION_WIDGET_CURRENT_TAB,
                      m_tabWidget->currentIndex());
    settings.endGroup();
    settings.sync();
}
