//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/SampleDesigner/ParticleLayoutForm.cpp
//! @brief     Implements class ParticleLayoutForm
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/SampleDesigner/ParticleLayoutForm.h"
#include "Base/Util/Assert.h"
#include "GUI/Model/Sample/InterferenceItems.h"
#include "GUI/Model/Sample/ItemWithParticles.h"
#include "GUI/Model/Sample/LayerItem.h"
#include "GUI/Model/Sample/ParticleLayoutItem.h"
#include "GUI/Support/Util/ActionFactory.h"
#include "GUI/View/Numeric/DoubleSpinBox.h"
#include "GUI/View/SampleDesigner/InterferenceForm.h"
#include "GUI/View/SampleDesigner/LayerEditorUtil.h"
#include "GUI/View/SampleDesigner/LayerForm.h"
#include "GUI/View/SampleDesigner/SampleEditorController.h"
#include "GUI/View/Tool/GroupBoxCollapser.h"
#include <QAction>
#include <QPushButton>

ParticleLayoutForm::ParticleLayoutForm(LayerForm* form, ParticleLayoutItem* pLayoutItem,
                                       SampleEditorController* ec)
    : QGroupBox(form)
    , m_layoutItem(pLayoutItem)
    , m_ec(ec)
{
    FormLayouter layouter(this, ec);
    layouter.setContentsMargins(30, 6, 0, 0);
    int rowOfTotalDensity = layouter.addValue(m_layoutItem->ownDensity());
    m_totalDensitySpinBox =
        layouter.widgetAt<DoubleSpinBox*>(rowOfTotalDensity, QFormLayout::FieldRole);
    ASSERT(m_totalDensitySpinBox);
    layouter.addRow(new InterferenceForm(this, pLayoutItem, ec));

    for (auto* particle : m_layoutItem->itemsWithParticles())
        layouter.addRow(
            LayerEditorUtil::createWidgetForItemWithParticles(this, particle, true, ec));

    auto* btn = LayerEditorUtil::createAddParticleButton(
        this,
        [=](FormFactorItemCatalog::Type type) { ec->addParticleLayoutItem(pLayoutItem, type); },
        [=](ItemWithParticlesCatalog::Type type) { ec->addParticleLayoutItem(pLayoutItem, type); });
    m_structureEditingWidgets << btn;
    layouter.addStructureEditingRow(btn);

    m_collapser = GroupBoxCollapser::installIntoGroupBox(this);
    m_collapser->setExpanded(pLayoutItem->isExpandLayout());
    connect(m_collapser, &GroupBoxCollapser::toggled, this,
            [pLayoutItem](bool b) { pLayoutItem->setExpandLayout(b); });

    // top right corner actions
    // show in real space
    {
        auto* showInRealspaceAction =
            ActionFactory::createShowInRealspaceAction(this, "particle layout", [ec, pLayoutItem] {
                ec->requestViewInRealspace(pLayoutItem);
            });
        m_collapser->addAction(showInRealspaceAction);
    }
    // duplicate
    {
        m_duplicateAction =
            ActionFactory::createDuplicateAction(this, "particle layout", [ec, pLayoutItem, form] {
                ec->duplicateLayoutItem(form, pLayoutItem);
            });
        m_collapser->addAction(m_duplicateAction);
    }
    // remove
    {
        m_removeAction =
            ActionFactory::createRemoveAction(this, "particle layout", [ec, pLayoutItem, form] {
                ec->removeLayoutItem(form, pLayoutItem);
            });
        m_collapser->addAction(m_removeAction);
    }

    m_layout = layouter.layout();

    updateDensityEnabling();
    updateTitle(form->layerItem());
}

void ParticleLayoutForm::enableStructureEditing(bool b)
{
    m_removeAction->setVisible(b);
    m_duplicateAction->setVisible(b);

    for (auto* w : m_structureEditingWidgets)
        w->setVisible(b);
}

ParticleLayoutItem* ParticleLayoutForm::layoutItem() const
{
    return m_layoutItem;
}

void ParticleLayoutForm::onParticleAdded(ItemWithParticles* p)
{
    int index = m_layoutItem->itemsWithParticles().indexOf(p);
    const int rowInLayout = m_layout->rowCount() - 1
                            - (m_layoutItem->itemsWithParticles().size() - 1) + index; // -1: btn

    m_layout->insertRow(rowInLayout,
                        LayerEditorUtil::createWidgetForItemWithParticles(this, p, true, m_ec));
}

void ParticleLayoutForm::onAboutToRemoveParticle(ItemWithParticles* item)
{
    int index = m_layoutItem->itemsWithParticles().indexOf(item);
    const int rowInLayout =
        m_layout->rowCount() - m_layoutItem->itemsWithParticles().size() - 1 + index; // -1: btn

    m_layout->removeRow(rowInLayout);
}

void ParticleLayoutForm::updateDensityEnabling()
{
    m_totalDensitySpinBox->setEnabled(!m_layoutItem->totalDensityIsDefinedByInterference());
}

void ParticleLayoutForm::updateDensityValue()
{
    if (m_layoutItem->totalDensityIsDefinedByInterference())
        m_layoutItem->setOwnDensity(m_layoutItem->totalDensityValue());

    m_totalDensitySpinBox->updateValue();
}

void ParticleLayoutForm::updateTitle(const LayerItem* layerItem)
{
    const auto layouts = layerItem->layoutItems();
    if (layouts.size() > 1)
        m_collapser->setTitle("Particle layout "
                              + QString::number(layouts.indexOf(m_layoutItem) + 1));
    else
        m_collapser->setTitle("Particle layout");
}
