/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.1

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Zonstat    zonmin          Zonal minimum
      Zonstat    zonmax          Zonal maximum
      Zonstat    zonrange        Zonal range
      Zonstat    zonsum          Zonal sum
      Zonstat    zonmean         Zonal mean
      Zonstat    zonavg          Zonal average
      Zonstat    zonstd          Zonal standard deviation
      Zonstat    zonstd1         Zonal standard deviation [Normalize by (n-1)]
      Zonstat    zonvar          Zonal variance
      Zonstat    zonvar1         Zonal variance [Normalize by (n-1)]
      Zonstat    zonpctl         Zonal percentiles
*/

#include <cdi.h>

#include "cdo_options.h"
#include "functs.h"
#include "process_int.h"
#include "param_conversion.h"
#include <mpim_grid.h>
#include "percentiles.h"

static void
addOperators(void)
{
  // clang-format off
  cdoOperatorAdd("zonmin",   func_min,   0, nullptr);
  cdoOperatorAdd("zonmax",   func_max,   0, nullptr);
  cdoOperatorAdd("zonrange", func_range, 0, nullptr);
  cdoOperatorAdd("zonsum",   func_sum,   0, nullptr);
  cdoOperatorAdd("zonmean",  func_mean,  0, nullptr);
  cdoOperatorAdd("zonavg",   func_avg,   0, nullptr);
  cdoOperatorAdd("zonvar",   func_var,   0, nullptr);
  cdoOperatorAdd("zonvar1",  func_var1,  0, nullptr);
  cdoOperatorAdd("zonstd",   func_std,   0, nullptr);
  cdoOperatorAdd("zonstd1",  func_std1,  0, nullptr);
  cdoOperatorAdd("zonpctl",  func_pctl,  0, nullptr);
  // clang-format on
}

void *
Zonstat(void *process)
{
  int gridID1 = -1, gridID2 = -1;
  int zongridID = -1;
  int index;
  int nrecs;
  int varID, levelID;

  cdoInitialize(process);

  addOperators();

  const auto operatorID = cdoOperatorID();
  const auto operfunc = cdoOperatorF1(operatorID);

  double pn = 0;
  if (operfunc == func_pctl)
    {
      operatorInputArg("percentile number");
      pn = parameter2double(cdoOperatorArgv(0));
      percentile_check_number(pn);
    }

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto ngrids = vlistNgrids(vlistID1);
  for (index = 0; index < ngrids; index++)
    {
      const auto gridID = vlistGrid(vlistID1, index);
      if (gridInqXsize(gridID) > 1 || gridInqType(gridID) == GRID_GAUSSIAN_REDUCED)
        {
          if (gridID1 == -1) gridID1 = gridID;
        }
      else
        {
          if (zongridID == -1) zongridID = gridID;
        }
    }

  int ndiffgrids = 0;
  for (index = 0; index < ngrids; index++)
    {
      if (zongridID != -1 && zongridID == vlistGrid(vlistID1, index)) continue;
      if (gridID1 != vlistGrid(vlistID1, index)) ndiffgrids++;
    }
  if (ndiffgrids) cdoAbort("Too many different grids!");

  if (gridID1 != -1)
    {
      const auto gridtype = gridInqType(gridID1);
      if (gridtype == GRID_LONLAT || gridtype == GRID_GAUSSIAN || gridtype == GRID_GAUSSIAN_REDUCED || gridtype == GRID_GENERIC)
        {
          if (zongridID != -1 && gridInqYsize(zongridID) == gridInqYsize(gridID1))
            gridID2 = zongridID;
          else
            gridID2 = gridToZonal(gridID1);
        }
      else
        {
          cdoAbort("Unsupported gridtype: %s", gridNamePtr(gridInqType(gridID1)));
        }
    }
  else
    {
      gridID2 = zongridID;
      cdoWarning("Input stream contains only zonal data!");
    }

  for (index = 0; index < ngrids; index++) vlistChangeGridIndex(vlistID2, index, gridID2);

  if (Options::cdoChunkType == CDI_UNDEFID)
    {
      const auto nvars = vlistNvars(vlistID2);
      for (varID = 0; varID < nvars; ++varID) vlistDefVarChunkType(vlistID2, varID, CDI_CHUNK_AUTO);
    }

  const auto streamID2 = cdoOpenWrite(1);
  cdoDefVlist(streamID2, vlistID2);

  gridID1 = vlistInqVarGrid(vlistID1, 0);
  const auto nlatmax = gridInqYsize(gridID1); /* max nlat ? */

  const auto gridsizemax = vlistGridsizeMax(vlistID1);

  Field field1, field2;
  field1.resize(gridsizemax);
  field2.resize(nlatmax);
  field2.grid = gridID2;

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoReadRecord(streamID1, field1.vec.data(), &field1.nmiss);

          field1.grid = vlistInqVarGrid(vlistID1, varID);
          field1.missval = vlistInqVarMissval(vlistID1, varID);
          field2.missval = vlistInqVarMissval(vlistID1, varID);

          if (zongridID != -1 && zongridID == field1.grid)
            {
              varrayCopy(nlatmax, field1.vec, field2.vec);
              field2.nmiss = field1.nmiss;
            }
          else
            {
              if (operfunc == func_pctl)
                zonpctl(field1, field2, pn);
              else
                zonfun(field1, field2, operfunc);
            }

          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, field2.vec.data(), field2.nmiss);
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
