/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2014 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */


/**
   \file cdw_config.c
   \brief Functions operating on configuration file and configuration variable(s)

   This file defines functions that:
   \li initialize and destroy configuration module
   \li read and write configuration file (with error handling)
   \li set default configuration that can be used when program can't read configuration from file
   \li validate configuration
*/

#define _BSD_SOURCE /* strdup() */
#define _GNU_SOURCE /* strcasestr() */

#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <errno.h>

#include <sys/stat.h>
#include <sys/types.h>


#include "cdw_config.h"
#include "cdw_config_window.h"
#include "gettext.h"
#include "cdw_string.h"
#include "cdw_fs.h"
#include "cdw_widgets.h"
#include "cdw_disc.h"
#include "cdw_debug.h"
#include "cdw_utils.h"
#include "cdw_ext_tools.h"
#include "cdw_mkudffs_options.h"
#include "cdw_erase_disc.h"
#include "cdw_drive.h"
#include "cdw_iso9660.h"

#include "config_cdw_undefine.h"

#include "config_cdw.h"

extern const char *cdw_log_file_name;


#define CDW_CONFIG_VOLUME_ITEMS_MAX 6 /* 74min CD, 80min CD, DVD, DVD+R DL, custom, auto */


cdw_id_clabel_t cdw_config_volume_size_items[CDW_CONFIG_VOLUME_ITEMS_MAX] = {
	/* 2TRANS: this is dropdown item label: 650MB CD */
	{ CDW_CONFIG_VOLUME_SIZE_CD74,               gettext_noop("74 min CD (650 MB)") },
	/* 2TRANS: this is dropdown item label: 700MB CD */
	{ CDW_CONFIG_VOLUME_SIZE_CD80,               gettext_noop("80 min CD (700 MB)") },
	/* 2TRANS: this is dropdown item label: DVD */
	{ CDW_CONFIG_VOLUME_SIZE_DVD_GENERIC,        gettext_noop("Generic DVD (4.7 GB)") },
	/* 2TRANS: this is dropdown item label: DVD+R DL */
	{ CDW_CONFIG_VOLUME_SIZE_DVD_RP_DL,          gettext_noop("DVD+R DL (8.5 GB)") },
	/* 2TRANS: this is dropdown item label: custom value of
	   ISO volume; user can enter arbitrary natural number */
	{ CDW_CONFIG_VOLUME_SIZE_CUSTOM,             gettext_noop("Custom value") },
	/* 2TRANS: this is dropdown item label:
	   "automatic" = automatic detection/resolution of size of ISO volume */
	{ CDW_CONFIG_VOLUME_SIZE_AUTO,               gettext_noop("Get sizes from disc") }
};

/** \brief Flag letting you know if options module works in abnormal mode
    Set to true if something went wrong during module setup -
    cdw will work without reading/writing configuration to disk file */
bool failsafe_mode = false;

/** \brief Project-wide variable holding current configuration of cdw */
cdw_config_t global_config;



static char *cdw_config_init_config_dir(const char *base_dir);
static cdw_rv_t cdw_config_read_from_file(const char *fullpath);
static void cdw_config_free_paths(void);


/** \brief Hardwired name of cdw configuration file */
static const char *old_config_file_name = ".cdw.conf";
static const char *config_file_name = "cdw.conf";
/** \brief Stores directory name where cdw config file is stored,
    also used as base path needed by some option values; if there are
    no errors during configuration of options module, this path is
    set to user home directory; since this path is initialized with
    value from cdw_fs.c module, ending slash is included */
static char *base_dir_fullpath = (char *) NULL;
/** \brief Full path to cdw configuration dir */
static char *config_dir_fullpath = (char *) NULL;
/** \brief Full path to cdw configuration file */
static char *config_file_fullpath = (char *) NULL;
/** \brief Full path to cdw configuration file in old localization ($HOME/.cdw.conf) */
static char *old_config_file_fullpath = (char *) NULL;



/* pair of low level functions that directly access configuration file;
   they are used by cdw_config_write_to_file() and
   cdw_config_read_from_file() respectively */
static void cdw_config_var_write_to_file(FILE *config_file, cdw_config_t *_config);
static cdw_rv_t cdw_config_var_read_from_file(FILE *config_file, cdw_config_t *_config);

static cdw_rv_t cdw_config_var_set_defaults(cdw_config_t *_config);
static cdw_rv_t cdw_config_module_set_paths(void);

static bool cdw_config_option_match(char *option_name, char *option_value, const char *searched_option_name, const char *option_values[], cdw_id_t *option, int init, int max);


static cdw_rv_t cdw_config_general_copy(cdw_config_general_t *dest, cdw_config_general_t *src);
static cdw_rv_t cdw_config_general_set_defaults(cdw_config_general_t *general);
static void     cdw_config_debug_general_print_options(cdw_config_general_t *general);


struct cdw_config_vst_t {
	cdw_id_t id;
	const char *old_label;
	const char *new_label;
	long int size;
};



static struct cdw_config_vst_t cdw_config_volume_size_table[] = {
	/*         id;                      old_label             new_label;               size                            */
	{ CDW_CONFIG_VOLUME_SIZE_CD74,        "650",                "cd74",                681984000  / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_CD80,        "703",                "cd80",                737280000  / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_GENERIC, "4482",               "dvd",                 4700372992 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_R,       "4489",               "dvd-r",               4707319808 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_RP,      "4482",               "dvd+r",               4700372992 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_RW,      "4489",               "dvd-rw",              4707319808 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_RWP,     "4482",               "dvd+rw",              4700372992 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_R_DL,    "dvd-r dl",           "dvd-r dl",            8543666176 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_RP_DL,   "dvd+r dl",           "dvd+r dl",            8547991552 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_CUSTOM,      "custom",             "custom",              0                              },
	{ CDW_CONFIG_VOLUME_SIZE_AUTO,        "auto",               "auto",                0                              },
	{ -1,                                 (char *) NULL,        (char *) NULL,         0                              } };


long int cdw_config_get_volume_size_mb_by_id(cdw_id_t id)
{
	struct cdw_config_vst_t *table = cdw_config_volume_size_table;
	for (int i = 0; table[i].id != -1; i++) {
		if (table[i].id == id) {
			return table[i].size;
		}
	}
	cdw_vdm ("ERROR: id %lld not found\n", id);
	return 0;
}





cdw_id_t cdw_config_volume_id_by_label(const char *label)
{
	struct cdw_config_vst_t *table = cdw_config_volume_size_table;
	for (int i = 0; table[i].id != -1; i++) {
		if (!strcasecmp(label, table[i].old_label) || !strcasecmp(label, table[i].new_label)) {
			return table[i].id;
		}
	}
	cdw_vdm ("WARNING: label \"%s\" not found\n", label);
	return -1;
}





/**
 * \brief Initialize configuration module
 *
 * Initialize configuration module - prepare some default option values and set paths
 * to base directory (should be user home directory) and to cdw configuration
 * file (will be in base directory).
 *
 * \return CDW_OK on success
 * \return CDW_ERROR on errors - the module wasn't configured at all
 * \return CDW_CANCEL if module cannot access cdw configuration file (module is working in failsafe mode)
 */
cdw_rv_t cdw_config_module_init(void)
{
	/* FIXME: problem with including cdw_task.h leads to using
	   magic number. */
	global_config.task_id = 0; //CDW_TASK_NONE;

	cdw_rv_t crv = cdw_config_module_set_paths();
	if (crv == CDW_NO || crv == CDW_OK) { /* paths are set correctly... */
		if (crv == CDW_NO) { /* ... but not to $HOME */
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Cannot find home directory and settings file. Will use configuration file in temporary location."),
					   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
		}

		failsafe_mode = false;
	} else { /* error situation; we can check what happened, but not in this release */
		failsafe_mode = true;
	}

	crv = cdw_config_var_init_fields(&global_config);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to initialize fields of config var\n");
		return CDW_ERROR;
	}

	crv = cdw_config_var_set_defaults(&global_config);
	if (crv != CDW_OK) {
		cdw_config_var_free_fields(&global_config);
		cdw_vdm ("ERROR: failed to set defaults for config variable\n");
		return CDW_ERROR;
	}

	if (failsafe_mode) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Config file error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Cannot read or write settings file. Will use temporary configuration."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		return CDW_CANCEL;
	}


	/* This overwrites default option values with values found in config
	   file, don't create config file if it doesn't exist already */

	char *path = config_file_fullpath;
	if (old_config_file_fullpath) {
		if (!access(old_config_file_fullpath, F_OK)) {
			/* can't log into log file, since it is not configured yet */

			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Information"),
					   /* 2TRANS: this is message in dialog window */
					   _("Migrating cdw configuration files to new location (into .cdw/ subdirectory in home dir)"),
					   CDW_BUTTONS_OK, CDW_COLORS_DIALOG);
			path = old_config_file_fullpath;
		}
	}

	crv = cdw_config_read_from_file(path);

	if (crv == CDW_ERROR || crv == CDW_CANCEL) {
		failsafe_mode = true;
		cdw_vdm ("ERROR: failed to read config from file, setting failsafe mode\n");
		return CDW_CANCEL;
	} else {
		return CDW_OK;
	}
}





/**
 * \brief Write current configuration to disk file
 *
 * Write content of config variable to disk file.
 *
 * This function only opens (for writing) config file, (in future: does
 * basic error checking) calls function doing actual writing, and then
 * closes config file.
 *
 * If configuration module works in failsafe mode, this function silently
 * skips writing to file. Caller is responsible for informing user about
 * failsafe_mode being set to true (and about consequences of this fact).
 *
 * \return CDW_OK on success
 * \return CDW_ERROR if function failed to open file for writing
 * \return CDW_CANCEL if failsafe mode is in effect and writing was skipped
 */
cdw_rv_t cdw_config_write_to_file(void)
{
	if (failsafe_mode) {
		/* this might be redundant if caller checks for
		   failsafe_mode, but just to be sure: silently skip writing */

		/* emergency mode, don't work with filesystem */
		return CDW_CANCEL;
	}

	cdw_assert (config_file_fullpath, "full path to config file is null\n");

	FILE *config_file = fopen(config_file_fullpath, "w");
	if (!config_file) {
		int e = errno;
		cdw_vdm ("ERROR: failed open config file \"%s\" for writing\n", config_file_fullpath);

		if (e == EACCES) { /* conf file has no write permissions */
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("An error occurred when saving configuration. Please check config file permissions. Any changes will be lost after closing cdw."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		} else {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Unknown error occurred when saving configuration. Any changes will be lost after closing cdw."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		}

		return CDW_ERROR;
	} else {
		cdw_config_var_write_to_file(config_file, &global_config);
		fclose(config_file);
		config_file = (FILE *) NULL;
		return CDW_OK;
	}
}





/**
 * \brief Read config file, put config values into config variable (high-level)
 *
 * Open config file, call function reading config file, close config
 * file. If config file does not exists, it is not created.
 *
 * If configuration module works in failsafe mode, this function silently skips
 * reading from file. Caller is responsible for informing user about
 * failsafe_mode being set to true (and about consequences of this fact).
 *
 * \return CDW_OK if configuration was read from file without problems
 * \return CDW_NO if there was no configuration file and now we are using default configuration
 * \return CDW_CANCEL if failsafe mode is in effect and reading was skipped (program is using default configuration now)
 * \return CDW_GEN_ERROR if function failed to open config file
 */
cdw_rv_t cdw_config_read_from_file(const char *fullpath)
{
	if (failsafe_mode) {
		cdw_vdm ("ERROR: failsafe mode is in effect, not reading config file\n");
		return CDW_CANCEL; /* emergency mode, don't work with filesystem */
	}

	cdw_assert (fullpath, "full path to config file is null\n");

	FILE *config_file = fopen(fullpath, "r");
	if (!config_file) {
		int e = errno;
		cdw_vdm ("WARNING: failed open config file \"%s\" for reading (errno = \"%s\")\n",
			 fullpath, strerror(e));

		if (e == ENOENT) { /* no such file */
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file problems"),
					   /* 2TRANS: this is message in dialog window */
					   _("Cannot open config file. Will use default configuration."),
					   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
			return CDW_NO;
		} else if (e == EACCES) { /* file permission problems */
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Cannot open config file. Please check file permissions."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			return CDW_ERROR;
		} else {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Unknown error occurred when reading configuration. Default values will be used."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			return CDW_ERROR;
		}
	} else { /* config file already exists and it is now open to read */
		cdw_rv_t r = cdw_config_var_read_from_file(config_file, &global_config);
		fclose(config_file);
		config_file = (FILE *) NULL;
		if (r == CDW_OK) {
			return CDW_OK; /* configuration read from existing configuration file */
		} else { /* problems with reading config file */
			cdw_vdm ("ERROR: failed to read from config file\n");
			return CDW_ERROR;
		}
	}
}





/**
   \brief Set path to cdw config file and to "base directory"

   Function uses call to cdw fs module to get path to home dir, which is
   used to initialize two paths used by config module:
   Base directory full path (char *base_dir_fullpath)
   Full config file path (char *config_file_fullpath)

   If path to user home directory can't be obtained, the function tries
   to get path to tmp directory. If this fails too, function returns
   CDW_ERROR and both paths are set to NULL.

   If function finishes successfully, dir paths in the two variables are
   the same, the difference is in file name in config_file_fullpath.
   Base dir path is ended with slash.

   On success, if function used user home directory, CDW_OK is returned.
   If function used tmp directory, CDW_NO is returned.

   \return CDW_OK if base_dir_fullpath is set to $HOME
   \return CDW_NO if base_dir_fullpath is set to tmp dir
   \return CDW_ERROR function can't get any path because of errors
*/
cdw_rv_t cdw_config_module_set_paths(void)
{
	cdw_assert (!base_dir_fullpath,
		    "ERROR: called this function when base dir is already initialized\n");
	cdw_assert (!config_file_fullpath,
		    "ERROR: called this function when base dir is already initialized\n");

	base_dir_fullpath = cdw_fs_get_home_or_tmp_dirpath();
	if (!base_dir_fullpath) {
		cdw_vdm ("ERROR: failed to create base dir fullpath\n");
		return CDW_ERROR;
	}

	/* at this point we have path to user home directory
	   or some tmp directory; now we set path to conf file */
	config_dir_fullpath = cdw_config_init_config_dir(base_dir_fullpath);
	if (!config_dir_fullpath) {
		cdw_config_free_paths();
		cdw_vdm ("ERROR: failed to create base dir fullpath\n");
		return CDW_ERROR;
	}

	config_file_fullpath = cdw_string_concat(config_dir_fullpath, config_file_name, (char *) NULL);
	if (!config_file_fullpath) {
		cdw_config_free_paths();
		cdw_vdm ("ERROR: failed to create config file fullpath\n");
		return CDW_ERROR;
	}

	old_config_file_fullpath = cdw_string_concat(base_dir_fullpath, old_config_file_name, (char *) NULL);
	if (!old_config_file_fullpath) {
		cdw_config_free_paths();
		cdw_vdm ("ERROR: failed to create config file fullpath\n");
		return CDW_ERROR;
	}

	cdw_vdm ("INFO: base directory:          \"%s\"\n", base_dir_fullpath);
	cdw_vdm ("INFO: configuration directory: \"%s\"\n", config_dir_fullpath);
	cdw_vdm ("INFO: configuration file:      \"%s\"\n", config_file_fullpath);

	if (cdw_string_starts_with_ci(base_dir_fullpath, "/tmp/")) {
		return CDW_NO; /* base dir is in tmp dir */
	} else {
		return CDW_OK; /* base dir is not in tmp dir */
	}
}





char *cdw_config_init_config_dir(const char *base_dir)
{
	char *dir = cdw_string_concat(base_dir, ".cdw/", (char *) NULL);
	if (!dir) {
		cdw_vdm ("ERROR: failed to concat config dir\n");
		return (char *) NULL;
	}

	int rv = cdw_fs_check_existing_path(dir, R_OK | W_OK, CDW_FS_DIR);
	if (rv == 0) {
		;
	} else if (rv == ENOENT) {
		int d = mkdir(dir, S_IRWXU);
		int e = errno;
		if (d != 0) {
			cdw_vdm ("ERROR: failed to create new dir \"%s\", reason: \"%s\"\n", dir, strerror(e));
			free(dir);
			dir = (char *) NULL;
		}
	} else {
		cdw_vdm ("ERROR: invalid dir path \"%s\"\n", dir);
		free(dir);
		dir = (char *) NULL;
	}

	return dir;
}





const char *cdw_config_get_config_dir(void)
{
	return config_dir_fullpath;
}





/**
   \brief Deallocate 'configuration' module resources

   Deallocate all 'configuration' module resources that were allocated during
   program run and were not freed before.
*/
void cdw_config_module_clean(void)
{
	if (config_file_fullpath) {
		/* some changes could be made outside of cdw configuration window,
		   this line ensures that the changes will be saved */
		cdw_config_write_to_file();

		/* part of migrating cdw config file to new location */
		if (old_config_file_fullpath) {
			if (!access(old_config_file_fullpath, F_OK)) {
				unlink(old_config_file_fullpath);
			}
		}
	} else {
		/* may happen if config module wasn't correctly
		   initialized, e.g. when there are some problems
		   at startup of application */
		cdw_vdm ("ERROR: path to config file not initialized\n");
	}

	cdw_config_free_paths();

	cdw_config_var_free_fields(&global_config);

	return;
}





/*
  Free all strings with paths used by cdw_config module.
*/
void cdw_config_free_paths(void)
{
	if (config_file_fullpath) {
		free(config_file_fullpath);
		config_file_fullpath = (char *) NULL;
	}

	if (old_config_file_fullpath) {
		free(old_config_file_fullpath);
		old_config_file_fullpath = (char *) NULL;
	}

	if (config_dir_fullpath) {
		free(config_dir_fullpath);
		config_dir_fullpath = (char *) NULL;
	}

	if (base_dir_fullpath) {
		free(base_dir_fullpath);
		base_dir_fullpath = (char *) NULL;
	}

	return;
}





/**
   \brief Copy values of options from one configuration variable to other

   Function does not validate \p src nor \p dest

   \param dest - config variable to be used as destination of copied values
   \param src - config variable to be used as source of copied values

   \return CDW_ERROR if function failed to copy one of fields
   \return CDW_OK on success
*/
cdw_rv_t cdw_config_var_copy(cdw_config_t *dest, cdw_config_t *src)
{
	dest->task_id = src->task_id;

	/* Write. */
	cdw_write_copy(&dest->burn, &src->burn);

	/* Erase. */
	cdw_erase_copy(&dest->erase, &src->erase);


	/* CONFIG_PAGE_ID_HW */
	strncpy(dest->custom_drive, src->custom_drive, OPTION_FIELD_LEN_MAX);
	dest->custom_drive[OPTION_FIELD_LEN_MAX] = '\0';
	strncpy(dest->selected_drive, src->selected_drive, OPTION_FIELD_LEN_MAX);
	dest->selected_drive[OPTION_FIELD_LEN_MAX] = '\0';
	strncpy(dest->scsi, src->scsi, OPTION_FIELD_LEN_MAX);
	dest->scsi[OPTION_FIELD_LEN_MAX] = '\0';


	/* CONFIG_PAGE_ID_AUDIO */
	cdw_rv_t crv = cdw_string_set(&(dest->audiodir), src->audiodir);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set audiodir from src = \"%s\"\n", src->audiodir);
		return CDW_ERROR;
	}


	/* ISO9660 filesystem. */
	crv = cdw_iso9660_copy(&(dest->iso9660), &(src->iso9660));
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to copy ISO9660 options\n");
		return CDW_ERROR;
	}


	/* UDF filesystem. */
	crv = cdw_udf_copy(&(dest->udf), &(src->udf));
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to copy UDF options\n");
		return CDW_ERROR;
	}


	/* General options, including options from CONFIG_PAGE_ID_OTHER page. */
	crv = cdw_config_general_copy(&dest->general, &src->general);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to copy general options\n");
		return CDW_ERROR;
	}

	return CDW_OK;
}





cdw_rv_t cdw_config_general_copy(cdw_config_general_t *dest, cdw_config_general_t *src)
{
	cdw_rv_t crv = cdw_string_set(&(dest->image_fullpath), src->image_fullpath);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set image_full_path from src = \"%s\"\n", src->image_fullpath);
		return CDW_ERROR;
	}

	crv = cdw_string_set(&(dest->log_fullpath), src->log_fullpath);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set log_full_path from src = \"%s\"\n", src->log_fullpath);
		return CDW_ERROR;
	}

	dest->show_log                    = src->show_log;
	dest->volume_size_id              = src->volume_size_id;
	dest->volume_size_custom_value    = src->volume_size_custom_value;
	dest->selected_follow_symlinks    = src->selected_follow_symlinks;
	dest->display_hidden_files        = src->display_hidden_files;     /* This one is in "general.", but is not displayed in Configuration window. */
	dest->support_dvd_rp_dl           = src->support_dvd_rp_dl;
	dest->show_dvd_rw_support_warning = src->show_dvd_rw_support_warning;

	return CDW_OK;
}





/**
   \brief Check if values in config variable are valid

   Check values stored in given configuration variable *AND* displayed
   in main configuration window. Important: the function validates
   only variables that are displayed in main configuration window of
   the application.

   Function can check variables from all pages of the configuration
   window: you should pass -1 via \page.

   Function can also check variables from a specific page of the
   configuration window only: you should pass one of values declared
   in cdw_config.h/'enum c_pages_order' via \page. ORing the IDs of
   pages doesn't work.

   If page, for which caller requested validation (or any page if \p
   page is set to -1) contains errors, then \p page will be set to ID
   of page that has errors, and \p field will be set to index of
   invalid field in that page. CDW_NO is returned.

   If there are no invalid fields, value of \p field and \p page is
   unspecified, and CDW_OK is returned.

   \param config - configuration variable to be checked
   \param page - number of page, from which parameters should be checked
   \param field - index of field with invalid value

   \return CDW_OK if all checked fields are valid
   \return CDW_NO if some field is invalid and the error must be fixed
*/
cdw_rv_t cdw_config_var_validate(cdw_config_t *config, cdw_id_t *page, int *field)
{
	/* IDs are defined in cdw_config.h  */

	/* CONFIG_PAGE_ID_HW */
	if (*page == -1 || *page == CONFIG_PAGE_ID_HW) {
		/* FIXME: use ID instead of string */
		if (!strcmp(config->selected_drive, "custom")) {
			/* user has selected "use custom drive path, and
			   the path cannot be empty; we only check if
			   the path is empty, we don't check if path is
			   in any way valid */

			if (!strlen(config->custom_drive)) {
				*field = f_custom_drive_i;
				*page = CONFIG_PAGE_ID_HW;
				return CDW_NO;
			}
		}

		cdw_rv_t crv = cdw_string_security_parser(config->custom_drive, (char *) NULL);
		if (crv == CDW_NO) {
			*field = f_custom_drive_i;
			*page = CONFIG_PAGE_ID_HW;
			return CDW_NO;
		}
		/* device path with ending slashes is not accepted by
		   other modules; info in config window asks to provide
		   device path without ending slashes, but let's
		   validate and fix path entered by user */
		size_t len = strlen(config->custom_drive);
		if (len > 1) {
			for (size_t i = len - 1; i > 0; i--) {
				if (config->custom_drive[i] == '/') {
					config->custom_drive[i] = '\0';
				} else {
					break;
				}
			}
		}

		if (strlen(config->scsi)) {
			crv = cdw_string_security_parser(config->scsi, (char *) NULL);
			if (crv == CDW_NO) {
				*field = f_scsi_i;
				*page = CONFIG_PAGE_ID_HW;
				return crv;
			}
		}

	}


	/* CONFIG_PAGE_ID_AUDIO */
	if (*page == -1 || *page == CONFIG_PAGE_ID_AUDIO) {
		cdw_rv_t crv = cdw_string_security_parser(config->audiodir, (char *) NULL);
		if (crv == CDW_NO) {
			*field = f_audiodir_i;
			*page = CONFIG_PAGE_ID_AUDIO;
			return CDW_NO;
		}
	}


	/* CONFIG_PAGE_ID_TOOLS */
	if (*page == -1 || *page == CONFIG_PAGE_ID_TOOLS) {
		; /* tool paths are read from "which" output, I won't validate this */
		; /* WARNING for the future: keep in mind that at some
		     occasions "tools" page is hidden, and it must not be
		     validated then, because in case of errors user will
		     be unable to visit the page to fix the errors */
	}


	/* CONFIG_PAGE_ID_OTHER */
	if (*page == -1 || *page == CONFIG_PAGE_ID_OTHER) {
		if (!strlen(config->general.log_fullpath)) {
			/* there has to be some log path specified */
			*field = f_log_fp_i;
			*page = CONFIG_PAGE_ID_OTHER;
			return CDW_NO;
		}
		cdw_rv_t crv = cdw_string_security_parser(config->general.log_fullpath, (char *) NULL);
		if (crv == CDW_NO) {
			*field = f_log_fp_i;
			*page = CONFIG_PAGE_ID_OTHER;
			return CDW_NO;
		}
		if (config->general.volume_size_custom_value < 0) {
			*field = f_cust_volume_size_i;
			*page = CONFIG_PAGE_ID_OTHER;
			return CDW_NO;
		}

		if (config->general.volume_size_id == CDW_CONFIG_VOLUME_SIZE_CUSTOM
		    && config->general.volume_size_custom_value == 0) {

			*field = f_cust_volume_size_i;
			*page = CONFIG_PAGE_ID_OTHER;
			return CDW_NO;
		}
	}

	return CDW_OK;
}





/**
   \brief Properly initialize fields of data structure of type cdw_config_t

   Set/initialize/allocate some fields of given variable of type
   cdw_config_t, so that it can be used safely by other C functions,
   i.e. there are no uninitialized pointers, and all non-NULL
   pointers point to some malloc()ed space.

   It is a good idea to call this function right after defining
   a cdw_config_t variable. You have to call this function before
   performing any operations on the variable.

   This function is different than cdw_config_var_set_defaults(),
   because cdw_config_var_set_defaults() is aware of cdw option values
   and sets higher-level values of fields that cdw_config_var_init_fields()
   only initialize with simple, low-level states.

   \p config must be valid, non-null pointer

   \param config - configuration variable with fields to initialize;

   \return CDW_OK on success
   \return CDW_ERROR on malloc() error
*/
cdw_rv_t cdw_config_var_init_fields(cdw_config_t *config)
{
	cdw_assert (config, "ERROR: passed NULL config variable\n");


	/* CONFIG_PAGE_ID_AUDIO */
	config->audiodir = (char *) NULL;


	/* CONFIG_PAGE_ID_HW */
	/* No pointers to initialize. */


	/* CONFIG_PAGE_ID_TOOLS */
	/* Paths to external tools are initialized by other module;
	   the paths aren't tightly related to options in this file */


	/* General options, including those presented in
	   CONFIG_PAGE_ID_OTHER page. */
	config->general.image_fullpath = (char *) NULL;
	config->general.log_fullpath = (char *) NULL;


	/* ISO9660 and UDF options. */
	cdw_iso9660_init(&(config->iso9660));
	cdw_udf_init(&(config->udf));


	/* Other. */
	cdw_write_init(&config->burn);

	return CDW_OK;
}





/**
   \brief Deallocate all resources used by given cdw_config_t variable

   Free all memory that is referenced by fields (pointers) in given
   cdw_config_t variable.

   This function can recognize unallocated resources
   and can handle them properly.

   \param _config - config variable with fields to free

   \return CDW_OK otherwise
*/
cdw_rv_t cdw_config_var_free_fields(cdw_config_t *config)
{
	cdw_assert (config, "passing null config pointer\n");


	/* CONFIG_PAGE_ID_AUDIO */
	if (config->audiodir) {
		free(config->audiodir);
		config->audiodir = (char *) NULL;
	}


	/* ISO9660 and UDF options. */
	cdw_iso9660_clean(&(config->iso9660));
	cdw_udf_clean(&(config->udf));


	/* General options, including those presented in
	   CONFIG_PAGE_ID_OTHER page. */
	if (config->general.image_fullpath) {
		free(config->general.image_fullpath);
		config->general.image_fullpath = (char *) NULL;
	}

	if (config->general.log_fullpath) {
		free(config->general.log_fullpath);
		config->general.log_fullpath = (char *) NULL;
	}


	/* Other options that aren't presented in Configuration window. */
	cdw_write_clean(&config->burn);

	return CDW_OK;
}





bool cdw_config_has_scsi_device(void)
{
	if (!strlen(global_config.scsi)) {
		return false;
	} else {
		return true;
	}
}




/* ***************************** */
/* ***** private functions ***** */
/* ***************************** */





/**
   \brief Set default values in given cdw configuration variable

   Set default values of fields in given configuration variable.
   This function can be called to make sure that some default values
   in configuration variable exist - just in case if config file does
   not exists or is broken or incomplete.

   This function depends on base_dir_fullpath to be set. If it is not set to some
   valid directory (which is indicated by failsafe_mode set to true), last
   attempt is made to set base dir to sane value: is is set to "/tmp".

   \param config - config variable in which to set default values

   \return CDW_ERROR on malloc() errors
   \return CDW_OK on success
 */
cdw_rv_t cdw_config_var_set_defaults(cdw_config_t *config)
{
	cdw_assert (base_dir_fullpath, "ERROR: base_dir_fullpath must not be NULL\n");


	/* CONFIG_PAGE_ID_AUDIO */
	cdw_rv_t crv = CDW_NO;
	if (failsafe_mode) {
		crv = cdw_string_set(&(config->audiodir), "/tmp/audio/");
	} else {
		char *tmp2 = cdw_string_concat(base_dir_fullpath, "audio/", (char *) NULL);
		if (!tmp2) {
			cdw_vdm ("ERROR: failed to concat string for audiodir\n");
			return CDW_ERROR;
		} else {
			crv = cdw_string_set(&(config->audiodir), tmp2);

			free(tmp2);
			tmp2 = (char *) NULL;
		}
	}
	if (!(config->audiodir) || crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set audiodir\n");
		return CDW_ERROR;
	}



	/* CONFIG_PAGE_ID_HW */
	strcpy(config->custom_drive, "");
	strcpy(config->selected_drive, "default");
	strcpy(config->scsi,""); /* some (most?) users will prefer to leave it empty */



	/* ISO9660 and UDF options/ */
	crv = cdw_iso9660_set_defaults(&(config->iso9660));
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set defaults for ISO9660 options\n");
		return CDW_ERROR;
	}

	crv = cdw_udf_set_defaults(&(config->udf));
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set defaults for UDF options\n");
		return CDW_ERROR;
	}



	/* General options, including those presented in
	   CONFIG_PAGE_ID_OTHER page. */
	crv = cdw_config_general_set_defaults(&config->general);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set defaults for general options\n");
		return CDW_ERROR;
	}


	/* Other options, not presented in configuration window. */
	cdw_erase_set_defaults(&config->erase);

	crv = cdw_write_set_defaults(&config->burn);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set defaults for write options\n");
		return CDW_ERROR;
	}

	return CDW_OK;
}





cdw_rv_t cdw_config_general_set_defaults(cdw_config_general_t *general)
{
	cdw_rv_t crv = cdw_string_set(&(general->image_fullpath), "/tmp/image.iso");
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set image_full_path\n");
		return CDW_ERROR;
	}

#ifndef NDEBUG
	size_t len = strlen(base_dir_fullpath);
	cdw_assert (base_dir_fullpath[len - 1] == '/',
		    "ERROR: base_dir_fullpath is not ended with slash: \"%s\"\n",
		    base_dir_fullpath);
#endif
	char *tmp = cdw_string_concat(config_dir_fullpath, cdw_log_file_name, (char *) NULL);
	if (!tmp) {
		cdw_vdm ("ERROR: failed to concat string for log_full_path\n");
		return CDW_ERROR;
	} else {
		crv = cdw_string_set(&(general->log_fullpath), tmp);
		free(tmp);
		tmp = (char *) NULL;

		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to set log_full_path to \"%s\"\n", tmp);
			return CDW_ERROR;
		}
	}

	general->show_log = true;
	general->volume_size_id = CDW_CONFIG_VOLUME_SIZE_CD74;
	if (general->volume_size_custom_value < 0) {
		general->volume_size_custom_value = 0;
	}
	general->selected_follow_symlinks = false;
	general->display_hidden_files = true;     /* Not included in Configuration window. */
	general->support_dvd_rp_dl = false;
	/* Show warning dialog. Dialog code will display a message
	   box, and will set this flag to "false", to not to annoy
	   user with repeated messages. */
	general->show_dvd_rw_support_warning = true;

	return CDW_OK;
}



extern const char *cdw_mkudffs_option_blocksize[];
extern const char *cdw_mkudffs_option_udfrev[];
extern const char *cdw_mkudffs_option_strategy[];
extern const char *cdw_mkudffs_option_spartable[];
extern const char *cdw_mkudffs_option_media_type[];
extern const char *cdw_mkudffs_option_space[];
extern const char *cdw_mkudffs_option_ad[];
extern const char *cdw_mkudffs_option_encoding[];





/**
   \brief Write values from given configuration variable to cdw config file

   Write values from given configuration variable config to hard disk
   configuration file config_file. This is low level function that does actual
   writing to file. The file should be opened for writing before calling this
   function. This function will not check for validity of file nor
   configuration variable.

   This is the only function that writes to configuration file. If you
   would like to change layout of config file - you can do it here.

   \param config_file - config file already opened for writing
   \param config - configuration variable with option values to be written to file
*/
void cdw_config_var_write_to_file(FILE *config_file, cdw_config_t *config)
{
	fprintf(config_file, "####################################\n");
	fprintf(config_file, "#   %s %s configuration file       \n", PACKAGE, VERSION);
	fprintf(config_file, "####################################\n");


	fprintf(config_file, "\n\n\n");
	fprintf(config_file, "### \"Writing\" options\n");
	fprintf(config_file, "pad=%d\n",       config->burn.cdrecord_pad ? 1 : 0);
	fprintf(config_file, "pad_size=%d\n",  config->burn.cdrecord_pad_size);
	fprintf(config_file, "eject=%d\n",     config->burn.eject ? 1 : 0);
	fprintf(config_file, "burnproof=%d\n", config->burn.cdrecord_burnproof ? 1 : 0);

	fprintf(config_file, "other_cdrecord_options=%s\n",     config->burn.cdrecord_other_options     ? config->burn.cdrecord_other_options : "");
	fprintf(config_file, "other_growisofs_options=%s\n",    config->burn.growisofs_other_options    ? config->burn.growisofs_other_options : "");
	fprintf(config_file, "other_xorriso_burn_options=%s\n", config->burn.xorriso_burn_other_options ? config->burn.xorriso_burn_other_options : "");


	fprintf(config_file, "\n\n\n");
	fprintf(config_file, "### Erase options\n");

	fprintf(config_file, "\n# These two entries refer to the same option. The second one is a new notation for options.\n");
	fprintf(config_file, "# fast / all\n");
	fprintf(config_file, "blank=%s\n",            config->erase.erase_mode == CDW_ERASE_MODE_FAST ? "fast" : "all");
	fprintf(config_file, "erase.erase_mode=%s\n", config->erase.erase_mode == CDW_ERASE_MODE_FAST ? "fast" : "all");

	fprintf(config_file, "\n");
	fprintf(config_file, "erase.eject=%d\n", config->erase.eject ? 1 : 0);


	fprintf(config_file, "\n\n");
	fprintf(config_file, "### \"Hardware\" options\n");

	fprintf(config_file, "# string indicating drive to be used by cdw\n");
	fprintf(config_file, "selected_drive=%s\n", config->selected_drive);
	fprintf(config_file, "# manually entered path to a drive\n");
	fprintf(config_file, "custom_drive=%s\n", config->custom_drive);
	fprintf(config_file, "# SCSI device descriptor, used by cdrecord\n");
	fprintf(config_file, "scsi=%s\n", config->scsi);



	fprintf(config_file, "\n\n\n");
	fprintf(config_file, "### \"Audio\" options\n");
	fprintf(config_file, "audiodir=%s\n", config->audiodir ? config->audiodir : "");



	/* "iso9660." notation for options in this section has been
	   introduced in cdw version 0.8.0. Remove the old notation
	   someday in the future. */
	fprintf(config_file, "\n\n");
	fprintf(config_file, "### ISO9660 file system options\n");

	fprintf(config_file, "\n# These two entries refer to the same option. The second one is a new notation for options.\n");
	fprintf(config_file, "joliet=%d\n",                            config->iso9660.joliet_information ? 1 : 0);
	fprintf(config_file, "iso9660.joliet_information=%d\n",        config->iso9660.joliet_information ? 1 : 0);

	fprintf(config_file, "\n# These two entries refer to the same option. The second one is a new notation for options.\n");
	fprintf(config_file, "joliet_long=%d\n",                       config->iso9660.joliet_long ? 1 : 0);
	fprintf(config_file, "iso9660.joliet_long=%d\n",               config->iso9660.joliet_long ? 1 : 0);

	fprintf(config_file, "\n# These two entries refer to the same option. The second one is a new notation for options.\n");
	fprintf(config_file, "rock_ridge=%lld\n",                      config->iso9660.rock_ridge);
	fprintf(config_file, "iso9660.rock_ridge=%lld\n",              config->iso9660.rock_ridge);

	fprintf(config_file, "\n# These two entries refer to the same option. The second one is a new notation for options.\n");
	fprintf(config_file, "follow_symlinks=%d\n",                   config->iso9660.follow_symlinks ? 1 : 0);
	fprintf(config_file, "iso9660.follow_symlinks=%d\n",           config->iso9660.follow_symlinks ? 1 : 0);

	fprintf(config_file, "\n");
	fprintf(config_file, "iso9660.pad=%d\n",                       config->iso9660.pad ? 1 : 0);

	fprintf(config_file, "\n# These two entries refer to the same option. The second one is a new notation for options.\n");
	fprintf(config_file, "# Allowed values for iso_level are 1, 2, 3, 4.\n");
	fprintf(config_file, "iso_level=%lld\n",                       config->iso9660.iso_level);
	fprintf(config_file, "iso9660.iso_level=%lld\n",               config->iso9660.iso_level);

	fprintf(config_file, "\n# These two entries refer to the same option. The second one is a new notation for options.\n");
	fprintf(config_file, "other_mkisofs_options=%s\n",             config->iso9660.mkisofs_other_options ? config->iso9660.mkisofs_other_options : "");
	fprintf(config_file, "iso9660.mkisofs_other_options=%s\n",     config->iso9660.mkisofs_other_options ? config->iso9660.mkisofs_other_options : "");

	fprintf(config_file, "\n# These two entries refer to the same option. The second one is a new notation for options.\n");
	fprintf(config_file, "other_xorriso_iso_options=%s\n",         config->iso9660.xorriso_iso_other_options ? config->iso9660.xorriso_iso_other_options : "");
	fprintf(config_file, "iso9660.xorriso_iso_other_options=%s\n", config->iso9660.xorriso_iso_other_options ? config->iso9660.xorriso_iso_other_options : "");

	fprintf(config_file, "\n# These two entries refer to the same option. The second one is a new notation for options.\n");
	fprintf(config_file, "# Well, in fact this field stores all options related to boot disc.\n");
	fprintf(config_file, "boot_image_path=%s\n",                   config->iso9660.boot_disc_options ? config->iso9660.boot_disc_options : "");
	fprintf(config_file, "iso9660.boot_disc_options=%s\n",         config->iso9660.boot_disc_options ? config->iso9660.boot_disc_options : "");



	fprintf(config_file, "\n\n");
	fprintf(config_file, "### UDF file system options ###\n");

	fprintf(config_file, "udf.mkudffs_blocksize=%s\n",  cdw_mkudffs_option_blocksize[config->udf.mkudffs_blocksize]);
	fprintf(config_file, "udf.mkudffs_udfrev=%s\n",     cdw_mkudffs_option_udfrev[config->udf.mkudffs_udfrev]);
	fprintf(config_file, "udf.mkudffs_strategy=%s\n",   cdw_mkudffs_option_strategy[config->udf.mkudffs_strategy]);
	fprintf(config_file, "udf.mkudffs_spartable=%s\n",  cdw_mkudffs_option_spartable[config->udf.mkudffs_spartable]);
	fprintf(config_file, "udf.mkudffs_media_type=%s\n", cdw_mkudffs_option_media_type[config->udf.mkudffs_media_type]);
	fprintf(config_file, "udf.mkudffs_space=%s\n",      cdw_mkudffs_option_space[config->udf.mkudffs_space]);
	fprintf(config_file, "udf.mkudffs_ad=%s\n",         cdw_mkudffs_option_ad[config->udf.mkudffs_ad]);
	fprintf(config_file, "udf.mkudffs_noefe=%d\n",      config->udf.mkudffs_noefe ? 1 : 0);
	fprintf(config_file, "udf.mkudffs_encoding=%s\n",   cdw_mkudffs_option_encoding[config->udf.mkudffs_encoding]);

	fprintf(config_file, "udf.mkudffs_other_options=%s\n", config->udf.mkudffs_other_options);

	fprintf(config_file, "udf.mkudffs_lvid=%s\n",  config->udf.mkudffs_lvid);
	fprintf(config_file, "udf.mkudffs_vid=%s\n",   config->udf.mkudffs_vid);
	fprintf(config_file, "udf.mkudffs_vsid=%s\n",  config->udf.mkudffs_vsid);
	fprintf(config_file, "udf.mkudffs_fsid=%s\n",  config->udf.mkudffs_fsid);

	fprintf(config_file, "udf.rsync_options=%s\n", config->udf.rsync_options);

	fprintf(config_file, "udf.mount_point=%s\n",   config->udf.mount_point);



	/* "general." notation for options in this section has been
	   introduced in cdw version 0.8.0. Remove the old notation
	   someday in the future. */
	fprintf(config_file, "\n\n");
	fprintf(config_file, "### General options ###\n");

	fprintf(config_file, "\n# Full path to both ISO9660 and UDF image file.\n");
	fprintf(config_file, "\n# These two entries refer to the same option. The second one is a new notation for options.\n");
	fprintf(config_file, "iso_image_full_path=%s\n",               config->general.image_fullpath ? config->general.image_fullpath : "");
	fprintf(config_file, "general.image_full_path=%s\n",           config->general.image_fullpath ? config->general.image_fullpath : "");

	fprintf(config_file, "\n# These two entries refer to the same option. The second one is a new notation for options.\n");
	fprintf(config_file, "logfile=%s\n",                           config->general.log_fullpath ? config->general.log_fullpath : "");
	fprintf(config_file, "general.log_full_path=%s\n",             config->general.log_fullpath ? config->general.log_fullpath : "");

	fprintf(config_file, "\n# These two entries refer to the same option. The second one is a new notation for options.\n");
	fprintf(config_file, "showlog=%d\n",                           config->general.show_log ? 1 : 0);
	fprintf(config_file, "general.show_log=%d\n",                  config->general.show_log ? 1 : 0);

	fprintf(config_file, "\n# These three entries refer to the same option. The third one is a new notation for options.\n");
	fprintf(config_file, "cdsize=%s\n",                            cdw_config_volume_size_table[config->general.volume_size_id].new_label);
	fprintf(config_file, "cdsize=%s\n",                            cdw_config_volume_size_table[config->general.volume_size_id].old_label);
	fprintf(config_file, "general.volume_size_id=%s\n",            cdw_config_volume_size_table[config->general.volume_size_id].new_label);

	fprintf(config_file, "\n# These two entries refer to the same option. The second one is a new notation for options.\n");
	fprintf(config_file, "user_cdsize=%ld\n",                      config->general.volume_size_custom_value);
	fprintf(config_file, "general.volume_size_custom_value=%ld\n", config->general.volume_size_custom_value);

	fprintf(config_file, "\n# Follow symbolic links when calculating size of selected files in main window.\n");
	fprintf(config_file, "general.selected_follow_symlinks=%d\n",  config->general.selected_follow_symlinks ? 1 : 0);

	fprintf(config_file, "\n# Display hidden files in browsers of native file system.\n");
	fprintf(config_file, "general.display_hidden_files=%d\n",      config->general.display_hidden_files ? 1 : 0);

	return;
}





/**
   \brief Read config file, store configuration in data structure (low-level code)

   Read config file line by line and store values found in that file into data
   structure. This is low level function that deals with disk file. It should
   be called by other function that will open file with correct permissions
   before passing it to this function.

   Data read from file is stored in \p config variable.

   This function calls cdw_string_security_parser() for every option
   value and additionally performs some basic validation of _some_ fields.
   When function finds some invalid value in config file, it discards it
   and doesn't modify existing value of field of \p config variable.

   \param config_file - config file opened for reading
   \param config - variable into which store values that were read

   \return CDW_OK on success
   \return CDW_ERROR on malloc() error
 */
cdw_rv_t cdw_config_var_read_from_file(FILE *config_file, cdw_config_t *config)
{
	cdw_rv_t crv = CDW_ERROR;

	cdw_option_t option = {
		.name = (char *) NULL,
		.value = (char *) NULL
	};

	char *line = (char *) NULL;

	while (1) {
		if (line) {
			free(line);
			line = (char *) NULL;
		}

		line = my_readline_10k(config_file);
		if (!line) {
			break;
		}
		cdw_string_rtrim(line);

		/* Deallocate strings from previous loop iteration. */
		cdw_config_option_free(&option);

		if (!cdw_config_split_options_line(&option, line)) {
			continue; /* empty or invalid line, or comment */
		}

		cdw_rv_t v = cdw_string_security_parser(option.value, (char *) NULL);
		if (v != CDW_OK) {
			cdw_vdm ("WARNING: option value \"%s=%s\" from config file rejected as insecure\n",
				 option.name, option.value);
			continue;
		}


		/* Writing. */
		if (!strcasecmp(option.name, "pad")) {
			cdw_string_get_bool_value(option.value, &(config->burn.cdrecord_pad));
			continue;
		}

		if (!strcasecmp(option.name, "pad_size")) {
			config->burn.cdrecord_pad_size = atoi(option.value);
			continue;
		}

		if (!strcasecmp(option.name, "eject")) {
			cdw_string_get_bool_value(option.value, &(config->burn.eject));
			continue;
		}

		if (!strcasecmp(option.name, "burnproof")) {
			cdw_string_get_bool_value(option.value, &(config->burn.cdrecord_burnproof));
			continue;
		}

		if (!strcasecmp(option.name, "other_cdrecord_options")) {
			crv = cdw_string_set(&(config->burn.cdrecord_other_options), option.value);
			if (crv != CDW_OK) {
				cdw_vdm ("ERROR: failed to set other_cdrecord_options from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}
			continue;
		}

		if (!strcasecmp(option.name, "other_growisofs_options")) {
			crv = cdw_string_set(&(config->burn.growisofs_other_options), option.value);
			if (crv != CDW_OK) {
				cdw_vdm ("ERROR: failed to set other_growisofs_options from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}
			continue;
		}
		if (!strcasecmp(option.name, "other_xorriso_burn_options")) {
			crv = cdw_string_set(&(config->burn.xorriso_burn_other_options), option.value);
			if (crv != CDW_OK) {
				cdw_vdm ("ERROR: failed to set other_xorriso_burn_options from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}
			continue;
		}



		/* Erase. */
		if (!strcasecmp(option.name, "blank")
		    || !strcasecmp(option.name, "erase.erase_mode")) {

			if (!strcasecmp(option.value, "fast")) {
				config->erase.erase_mode = CDW_ERASE_MODE_FAST;
			} else if (!strcasecmp(option.value, "all")) {
				config->erase.erase_mode = CDW_ERASE_MODE_ALL;
			} else {
				; /* there should be some default value in option.name already */
			}
			continue;
		}

		if (!strcasecmp(option.name, "erase.eject")) {
			cdw_string_get_bool_value(option.value, &(config->erase.eject));
			continue;
		}



		/* CONFIG_PAGE_ID_HW */
		if (!strcasecmp(option.name, "selected_drive")) {
			strncpy(config->selected_drive, option.value, OPTION_FIELD_LEN_MAX);
			config->selected_drive[OPTION_FIELD_LEN_MAX] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "custom_drive")) {
			if (!strlen(option.value)) {
				/* don't erase value that may already be set
				   after reading cdrw_device field from config file */
				;
			} else {
				strncpy(config->custom_drive, option.value, OPTION_FIELD_LEN_MAX);
				config->custom_drive[OPTION_FIELD_LEN_MAX] = '\0';
			}
			continue;
		}

		if (!strcasecmp(option.name, "scsi")) {
			strncpy(config->scsi, option.value, OPTION_FIELD_LEN_MAX);
			config->scsi[OPTION_FIELD_LEN_MAX] = '\0';
			continue;
		}



		/* CONFIG_PAGE_ID_AUDIO */
		if (!strcasecmp(option.name, "audiodir")) {
			crv = cdw_string_set(&(config->audiodir), option.value);
			if (crv != CDW_OK) {
				cdw_vdm ("ERROR: failed to set audiodir from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}
			continue;
		}



		/* ISO9660 filesystem. */
		if (!strcasecmp(option.name, "joliet")
		    || !strcasecmp(option.name, "iso9660.joliet_information")) {

			cdw_string_get_bool_value(option.value, &(config->iso9660.joliet_information));
			continue;
		}

		if (!strcasecmp(option.name, "joliet_long")
		    || !strcasecmp(option.name, "iso9660.joliet_long")) {

			cdw_string_get_bool_value(option.value, &(config->iso9660.joliet_long));
			continue;
		}

		if (!strcasecmp(option.name, "rock_ridge")
		    || !strcasecmp(option.name, "iso9660.rock_ridge")) {

			int i = atoi(option.value);
			if (i < 0 || i > 2) {
				; /* probably "trash" value, rejecting completely */
			} else {
				config->iso9660.rock_ridge = i;
			}
			continue;
		}

		if (!strcasecmp(option.name, "follow_symlinks")
		    || !strcasecmp(option.name, "iso9660.follow_symlinks")) {

			cdw_string_get_bool_value(option.value, &(config->iso9660.follow_symlinks));
			continue;
		}

		if (!strcasecmp(option.name, "iso9660.pad")) {
			cdw_string_get_bool_value(option.value, &(config->iso9660.pad));
			continue;
		}

		if (!strcasecmp(option.name, "iso_level")
		    || !strcasecmp(option.name, "iso9660.iso_level")) {

			int i = atoi(option.value);
			if (i < 1 || i > 4) {
				; /* probably "trash" value, rejecting completely */
			} else {
				config->iso9660.iso_level = i;
			}
			continue;
		}

		if (!strcasecmp(option.name, "other_mkisofs_options")
		    || !strcasecmp(option.name, "iso9660.mkisofs_other_options")) {

			crv = cdw_string_set(&(config->iso9660.mkisofs_other_options), option.value);
			if (crv != CDW_OK) {
				cdw_vdm ("ERROR: failed to set other_mkisofs_options from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}
			continue;
		}

		if (!strcasecmp(option.name, "other_xorriso_iso_options")
		    || !strcasecmp(option.name, "iso9660.xorriso_iso_other_options")) {

			crv = cdw_string_set(&(config->iso9660.xorriso_iso_other_options), option.value);
			if (crv != CDW_OK) {
				cdw_vdm ("ERROR: failed to set other_xorriso_iso_options from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}
			continue;
		}

		if (!strcasecmp(option.name, "boot_image_path")
		    || !strcasecmp(option.name, "iso9660.boot_disc_options")) {

			crv = cdw_string_set(&(config->iso9660.boot_disc_options), option.value);
			if (crv != CDW_OK) {
				cdw_vdm ("ERROR: failed to set boot_disc_options from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}
			continue;
		}




		/* UDF options */

		if (cdw_config_option_match(option.name, option.value, "udf.mkudffs_blocksize", cdw_mkudffs_option_blocksize, &(config->udf.mkudffs_blocksize), CDW_MKUDFFS_BLOCKSIZE_UNSPECIFIED, CDW_MKUDFFS_BLOCKSIZE_MAX)) {
			continue;
		}

		if (cdw_config_option_match(option.name, option.value, "udf.mkudffs_udfrev", cdw_mkudffs_option_udfrev, &(config->udf.mkudffs_udfrev), CDW_MKUDFFS_UDFREV_UNSPECIFIED, CDW_MKUDFFS_UDFREV_MAX)) {
			continue;
		}

		if (cdw_config_option_match(option.name, option.value, "udf.mkudffs_strategy", cdw_mkudffs_option_strategy, &(config->udf.mkudffs_strategy), CDW_MKUDFFS_STRATEGY_UNSPECIFIED, CDW_MKUDFFS_STRATEGY_MAX)) {
			continue;
		}

		if (cdw_config_option_match(option.name, option.value, "udf.mkudffs_spartable", cdw_mkudffs_option_spartable, &(config->udf.mkudffs_spartable), CDW_MKUDFFS_SPARTABLE_UNSPECIFIED, CDW_MKUDFFS_SPARTABLE_MAX)) {
			continue;
		}

		if (cdw_config_option_match(option.name, option.value, "udf.mkudffs_media_type", cdw_mkudffs_option_media_type, &(config->udf.mkudffs_media_type), CDW_MKUDFFS_MEDIA_TYPE_UNSPECIFIED, CDW_MKUDFFS_MEDIA_TYPE_MAX)) {
			continue;
		}

		if (cdw_config_option_match(option.name, option.value, "udf.mkudffs_space", cdw_mkudffs_option_space, &(config->udf.mkudffs_space), CDW_MKUDFFS_SPACE_UNSPECIFIED, CDW_MKUDFFS_SPACE_MAX)) {
			continue;
		}

		if (cdw_config_option_match(option.name, option.value, "udf.mkudffs_ad", cdw_mkudffs_option_ad, &(config->udf.mkudffs_ad), CDW_MKUDFFS_AD_UNSPECIFIED, CDW_MKUDFFS_AD_MAX)) {
			continue;
		}

		if (!strcasecmp(option.name, "udf.mkudffs_noefe")) {
			cdw_string_get_bool_value(option.value, &(config->udf.mkudffs_noefe));
			continue;
		}

		if (cdw_config_option_match(option.name, option.value, "udf.mkudffs_encoding", cdw_mkudffs_option_encoding, &(config->udf.mkudffs_encoding), CDW_MKUDFFS_ENCODING_UNSPECIFIED, CDW_MKUDFFS_ENCODING_MAX)) {
			continue;
		}


		if (!strcasecmp(option.name, "udf.mkudffs_other_options")) {
			strncpy(config->udf.mkudffs_other_options, option.value, CDW_MKUDFFS_OTHER_OPTIONS_LEN_MAX);
			config->udf.mkudffs_other_options[CDW_MKUDFFS_OTHER_OPTIONS_LEN_MAX] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "udf.mkudffs_lvid")) {
			strncpy(config->udf.mkudffs_lvid, option.value, CDW_UDF_LVID_LEN_MAX);
			config->udf.mkudffs_lvid[CDW_UDF_LVID_LEN_MAX] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "udf.mkudffs_vid")) {
			strncpy(config->udf.mkudffs_vid, option.value, CDW_UDF_VID_LEN_MAX);
			config->udf.mkudffs_vid[CDW_UDF_VID_LEN_MAX] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "udf.mkudffs_vsid")) {
			strncpy(config->udf.mkudffs_vsid, option.value, CDW_UDF_VSID_LEN_MAX);
			config->udf.mkudffs_vsid[CDW_UDF_VSID_LEN_MAX] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "udf.mkudffs_fsid")) {
			strncpy(config->udf.mkudffs_fsid, option.value, CDW_UDF_FSID_LEN_MAX);
			config->udf.mkudffs_fsid[CDW_UDF_FSID_LEN_MAX] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "udf.rsync_options")) {
			strncpy(config->udf.rsync_options, option.value, CDW_RSYNC_OPTIONS_LEN_MAX);
			config->udf.rsync_options[CDW_RSYNC_OPTIONS_LEN_MAX] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "udf.mount_point")) {
			crv = cdw_string_set(&(config->udf.mount_point), option.value);
			if (crv != CDW_OK) {
				cdw_vdm ("ERROR: failed to set udf.mount_point from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}
			continue;
		}



		/* General options (including options from CONFIG_PAGE_ID_OTHER page). */

		if (!strcasecmp(option.name, "iso_image_full_path")
		    || !strcasecmp(option.name, "general.image_full_path")) {

			crv = cdw_string_set(&(config->general.image_fullpath), option.value);
			if (crv != CDW_OK) {
				cdw_vdm ("ERROR: failed to set image_full_path from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}
			continue;
		}

		if (!strcasecmp(option.name, "logfile")
		    || !strcasecmp(option.name, "general.log_full_path")) {

			crv = cdw_string_set(&(config->general.log_fullpath), option.value);
			if (crv != CDW_OK) {
				cdw_vdm ("ERROR: failed to set log_full_path from option.value = \"%s\"\n", option.value);
				return CDW_ERROR;
			}
			continue;
		}

		if (!strcasecmp(option.name, "showlog")
		    || !strcasecmp(option.name, "general.show_log")) {

			cdw_string_get_bool_value(option.value, &(config->general.show_log));
			continue;
		}

		if (!strcasecmp(option.name, "cdsize")
		    || !strcasecmp(option.name, "general.volume_size_id")) {

			cdw_id_t id = cdw_config_volume_id_by_label(option.value);
			if (id != -1) {
				config->general.volume_size_id = id;
			}
			continue;
		}

		if (!strcasecmp(option.name, "user_cdsize")
		    || !strcasecmp(option.name, "general.volume_size_custom_value")) {

			config->general.volume_size_custom_value = strtol(option.value, (char **) NULL, 10);
			continue;
		}

		if (!strcasecmp(option.name, "general.selected_follow_symlinks")) {
			cdw_string_get_bool_value(option.value, &(config->general.selected_follow_symlinks));
			continue;
		}

		if (!strcasecmp(option.name, "general.display_hidden_files")) {
			cdw_string_get_bool_value(option.value, &(config->general.display_hidden_files));
			continue;
		}

	} /* while () */

	cdw_config_option_free(&option);
	cdw_string_delete(&line);

	return CDW_OK;
}





/*
  Check if \p option_name matches \p searched_option_name.  If it
  does, look up \p option_value in \p option_values and return index
  of matched option value through \p option.

  If \p option_name and \p searched_option_name match, initial/default
  value \p init will be assigned to \p option.

  \p max is a number of options values in \p option_values.

  \return true when both option name and option value matches have been made/found
  \return false otherwise
*/
bool cdw_config_option_match(char *option_name, char *option_value, const char *searched_option_name, const char *option_values[], cdw_id_t *option, int init, int max)
{
	if (!strcasecmp(option_name, searched_option_name)) {
		*option = init;
		for (cdw_id_t i = 0; i < max; i++) {
			if (!strcasecmp(option_value, option_values[i])) {
				*option = i;
				return true;
			}
		}
	}
	return false;
}





/**
   \brief Split line containing '=' char into 'name' and 'value' part

   \date Function's top-level comment reviewed on 2012-02-07
   \date Function's body reviewed on 2012-02-07

   The function takes one 'char *' string, recognizes where is first
   '=' char in it. What is on the left side of the char is treated
   as option name, and what is on the right side of the char is treated
   as option value. Both name and value of an option are duplicated
   into appropriate fields of \p option. Both strings are trimmed of
   whitespaces.

   The function modifies its \p line argument!

   The function recognizes '#' as comment char and erases everything from
   line starting from '#' char;

   \p line must be proper char * string ending with '\0'

   Both fields of \p option are set to NULL if:
   \li function's \p line argument is NULL;
   \li function's \p line argument is empty;
   \li function's \p line argument does not contain '=' char;
   \li function's \p line argument is a comment line.
   \li function's \p line argument has '=' char but don't have any option name

   Otherwise the fields of \p option are set with freshly allocated
   strings: name and value of option.

   Use cdw_config_option_free() on \p option to conveniently
   deallocate both strings.

   \param option - variable where new strings with name and value are placed
   \param line - line that you want to extract option from

   \return true on success (both fields of \p option are set)
   \return false otherwise
*/
bool cdw_config_split_options_line(cdw_option_t *option, char *line)
{
	if (!line) {
		cdw_vdm ("ERROR: input line is NULL\n");
		return false;
	}

	cdw_vdm ("INFO: input line = '%s'\n", line);

	option->name = (char *) NULL;
	option->value = (char *) NULL;


	/* this will make sure that comment, starting at any position
	   in line, will be erased from line */
	size_t n = strcspn(line, "#"); /* number of chars before start of comment */
	if (n != 0) {
		line[n] = '\0';
	}

	if (line[0] == '\0') {
		cdw_sdm ("INFO: empty line\n");
		return false;
	} else if (line[0] == '#') {
		cdw_sdm ("INFO: comment line: \"%s\"\n", line);
		return false;
	} else {
		; /* The line potentially is valid. */
	}

	char *eq = strstr(line, "=");
	if (!eq) {
		cdw_vdm ("WARNING: line is invalid (\"%s\")\n", line);
		return false;
	}
	*eq = '\0';

	option->name = cdw_string_trim(line);
	option->value = cdw_string_trim(eq + 1);
	cdw_vdm ("INFO: trimmed name: \"%s\"\n", option->name);
	cdw_vdm ("INFO: trimmed value: \"%s\"\n", option->value);

	if (!option->name || !option->value || !strcmp(option->name, "")) {
		if (option->name) {
			free(option->name);
			option->name = (char *) NULL;
		}
		if (option->value) {
			free(option->value);
			option->value = (char *) NULL;
		}
		cdw_vdm ("ERROR: some string was empty\n");
		return false;
	}

	return true;
}





/**
   \brief Deallocate strings from an option

   \date Function's top-level comment reviewed on 2012-02-07
   \date Function's body reviewed on 2012-02-07

   Function deallocates two strings from given \p option.
   It checks for NULL pointers prior to deallocation, and sets
   pointers to NULL after deallocation.

   Call this function only for \p option processed by
   cdw_config_split_options_line().

   \param option - variable which you want to process
*/
void cdw_config_option_free(cdw_option_t *option)
{
	cdw_assert (option, "ERROR: pointer is NULL\n");

	if (option->value) {
		free(option->value);
		option->value = (char *) NULL;
	}

	if (option->name) {
		free(option->name);
		option->name = (char *) NULL;
	}

	return;
}





/* simple getter - some files use global config variable just to get this field */
const char *cdw_config_get_custom_drive(void)
{
	return global_config.custom_drive;
}





const char *cdw_config_get_scsi_drive(void)
{
	return global_config.scsi;
}





/* get either custom value of ISO9660 volume size,
   or one of predefined volume size values */
long int cdw_config_get_current_volume_size_value_megabytes(void)
{
	cdw_assert (global_config.general.volume_size_id != CDW_CONFIG_VOLUME_SIZE_AUTO,
		    "ERROR: current volume size ID is \"auto\", so you should fetch volume size from disc\n");

	if (global_config.general.volume_size_id == CDW_CONFIG_VOLUME_SIZE_CUSTOM) {
		/* value entered by user in configuration window */
		cdw_vdm ("INFO: returning custom value %ld\n", global_config.general.volume_size_custom_value);
		return global_config.general.volume_size_custom_value;
	} else {
		/* one of predefined values, corresponding to total
		   capacity of selected disc type */
		return cdw_config_get_volume_size_mb_by_id(global_config.general.volume_size_id);
	}
}




void cdw_config_debug_print_config(cdw_config_t *config)
{
	cdw_vdm ("\n\nINFO: configuration:\n\n");
#if 1
	cdw_erase_debug_print_options(&config->erase);
#endif

#if 1
	cdw_write_debug_print_options(&config->burn);
#endif

#if 1
	cdw_vdm ("INFO: Hardware options:\n");
	cdw_vdm ("INFO:   custom drive = \"%s\"\n", config->custom_drive);
	cdw_vdm ("INFO: selected drive = \"%s\"\n", config->selected_drive);
	cdw_vdm ("INFO:           scsi = \"%s\"\n\n", config->scsi);
#if 0 /* unused, maybe will be enabled in future */
	cdw_vdm ("INFO:     mountpoint = \"%s\"\n", config->mountpoint);
	cdw_vdm ("INFO:          cdrom = \"%s\"\n", config->cdrom);
#endif
#endif


#if 1
	cdw_vdm ("INFO: Audio options:\n");
	cdw_vdm ("INFO: audiodir = \"%s\"\n\n", config->audiodir);
#endif


#if 1
	cdw_iso9660_debug_print_options(&config->iso9660);
#endif


#if 1
	cdw_udf_debug_print_options(&config->udf);
#endif


#if 1
	cdw_vdm ("INFO: External tools options:\n");
	cdw_ext_tools_debug_print_config();
#endif


#if 1
	cdw_config_debug_general_print_options(&config->general);
#endif

	return;
}





void cdw_config_debug_general_print_options(cdw_config_general_t *general)
{
	cdw_vdm ("INFO: General options:\n");
	cdw_vdm ("INFO:             image full path = \"%s\"\n", general->image_fullpath);
	cdw_vdm ("INFO:               log full path = \"%s\"\n", general->log_fullpath);
	cdw_vdm ("INFO:                    show log = %s\n",     general->show_log ? "true" : "false");
	cdw_vdm ("INFO:              volume size id = %s (%lld)\n", cdw_utils_id_label_table_get_label(cdw_config_volume_size_items, general->volume_size_id), general->volume_size_id);
	cdw_vdm ("INFO:    volume size custom value = %ld\n",    general->volume_size_custom_value);
	cdw_vdm ("INFO:    selected follow symlinks = %s\n",     general->selected_follow_symlinks ? "true" : "false");
	cdw_vdm ("INFO:        display hidden files = %s\n",     general->display_hidden_files ? "true" : "false");
	cdw_vdm ("INFO:           support DVD+RP DL = %s\n",     general->support_dvd_rp_dl ? "true" : "false");
	cdw_vdm ("INFO: show DVD-RW support warning = %s\n",     general->show_dvd_rw_support_warning ? "true" : "false");

	return;
}





bool cdw_config_support_dvd_rp_dl(void)
{
	return global_config.general.support_dvd_rp_dl;
}





bool cdw_config_follow_symlinks(void)
{
	return global_config.general.selected_follow_symlinks;
}





#ifdef CDW_UNIT_TEST_CODE


/* *********************** */
/* *** unit tests code *** */
/* *********************** */


static void test_cdw_config_split_options_line(void);


void cdw_config_run_tests(void)
{
	fprintf(stderr, "testing cdw_config.c\n");

	test_cdw_config_split_options_line();

	fprintf(stderr, "done\n\n");

	return;
}





void test_cdw_config_split_options_line(void)
{
	fprintf(stderr, "\ttesting cdw_config_split_options_line()... ");

	struct {
		const char *line;
		bool expected_return_value;
		bool expected_null_name;
		bool expected_null_value;
		const char *expected_name;
		const char *expected_value;

	} input_data[] = {
		/* correct line */
		{ "option name=value string",
		  true,
		  false, false,
		  "option name", "value string" },

		/* correct line */
		{ "\t option name \t\t = \t value string \t",
		  true,
		  false, false,
		  "option name", "value string" },

		/* correct line, but no value */
		{ "\t option name = \t \t",
		  true,
		  false, false,
		  "option name", "" },

		/* correct line, but no value (2) */
		{ "\t option name = #value string\t \t",
		  true,
		  false, false,
		  "option name", "" },

		/* incorrect line, no '=' char (1) */
		{ "\t option name \t # = value string\t \t",
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		/* incorrect line, no '=' char (2) */
		{ "\t option name \t  value string\t \t",
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		/* incorrect line, all content is comment */
		{ " \t #\t option name = value string\t \t",
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		/* incorrect line, empty line */
		{ "",
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		/* incorrect line, empty line (2) */
		{ " \t \t    \t\t\t\t     \t               \t#    ",
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		/* incorrect line, line = NULL */
		{ (char *) NULL,
		  false,
		  true, true,
		  (char *) NULL, (char *) NULL },

		{ (char *) NULL,
		  false,
		  false, true, /* <-- guard */
		  (char *) NULL, (char *) NULL }
	};

	cdw_option_t option;
	option.name = (char *) NULL;
	option.value = (char *) NULL;
	int i = 0;
	while (input_data[i].expected_null_name == input_data[i].expected_null_value) {
		char *line = (char *) NULL;
		if (input_data[i].line) {
			line = strdup(input_data[i].line);
		}

		bool rv = cdw_config_split_options_line(&option, line);
		cdw_assert (input_data[i].expected_return_value == rv, "ERROR: failed at function call#%d\n", i);

		if (input_data[i].expected_null_name) {
			cdw_assert (!option.name, "ERROR: name is not NULL (#%d)\n", i);
		} else {
			cdw_assert (option.name, "ERROR: name is NULL (#%d)\n", i);
		}

		if (input_data[i].expected_null_value) {
			cdw_assert (!option.value, "ERROR: value is not NULL (#%d)\n", i);
		} else {
			cdw_assert (option.value, "ERROR: value is NULL (#%d)\n", i);
		}

		if (!input_data[i].expected_null_name && input_data[i].expected_null_value) {
			cdw_assert (!strcmp(option.name, input_data[i].expected_name), "ERROR: unexpected name #%d: \"%s\"\n", i, option.name);
			cdw_assert (!strcmp(option.value, input_data[i].expected_value), "ERROR: unexpected value #%d: \"%s\"\n", i, option.value);
		}

		cdw_config_option_free(&option);
		if (line) {
			free(line);
			line = (char *) NULL;
		}

		i++;
	}


	fprintf(stderr, "OK\n");

	return;
}



#endif /* #ifdef CDW_UNIT_TEST_CODE */
