/****************************************************************************
 *    lib/c/Printer.cpp - This file is part of coala						*
 *																			*
 *    Copyright (C) 2009  Torsten Grote										*
 *																			*
 *    This program is free software; you can redistribute it and/or modify	*
 *    it under the terms of the GNU General Public License as published by	*
 *    the Free Software Foundation; either version 3 of the License, or		*
 *    (at your option) any later version.									*
 *																			*
 *    This program is distributed in the hope that it will be useful,		*
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of		*
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *    GNU General Public License for more details.							*
 *																			*
 *    You should have received a copy of the GNU General Public License		*
 *    along with this program; if not, see http://www.gnu.org/licenses		*
 ****************************************************************************/

#include "Printer.h"

using namespace C;

Printer::Printer(std::ostream* output_stream, int debug_level, bool fake_negation,
				 bool incremental, bool reverse, bool no_where_check, bool no_dir_enc, string language) {
	o				= output_stream;
	debug			= debug_level;
	no_direct_enc	= no_dir_enc;
	fake_neg		= fake_negation;
	inc				= incremental;
	rev				= reverse;
	no_w_chk		= no_where_check;
	lang			= language;
	
	neg = fake_neg ? "neg_" : "-";
	T = inc ? "t" : "T";
	
	section_ 		= 'b';
	printed_ltl_aux_ = false;

	if(!inc) {
		*o << "#domain time(T).\n";
		*o << "time(1..t).\n";
	}
}

Printer::~Printer() {

}

void Printer::setSection(char section) {
	if(inc) section_ = section;
}

void Printer::print() {
	if(no_direct_enc) {
		if(rev) printBackwardsEncoding();
		else printEncoding();
	}
	else {
		// hide everything except actions and fluents
		if(debug < 3) *o << "#hide.\n";
	}
	
	if(inc) {
		*o << "\n#base.\n" << b.str() << extra.str();
		*o << "\n#cumulative t.\n" << c.str();
		*o << "\n#volatile t.\n" << v.str();
		*o << "\n#base.\n";
	}
	else if(no_direct_enc || printed_ltl_aux_) {
		*o << b.str();
		*o << c.str();
		*o << v.str();
	}
	*o << extra.str();
}

void Printer::add(std::string text) {
	if(inc) {
		if(section_ == 'b')			b << text;
		else if(section_ == 'c')	c << text;
		else if(section_ == 'v')	v << text;
	}
	else {
		*o << text;
		// TODO remove flush
		// o->flush();
	}
}

void Printer::addExtra(std::string text) {
	extra << text;
}

void Printer::printEncoding() {
	b << "\n%\n% Encoding of Action Language C\n%\n";
	
	// #base.
	
	// % Artificial fluent "true" holds in initial state
	b << "holds(\"true\",0).\n";

	// % Guess initial state
	b << "1 { holds(F,0), holds(neg(F),0) } 1 :- fluent(F).\n";

	// % Propagate composite fluents
	b << "holds(G,0)      :- cfluent(G), holds(F,0) : fpart(G,F).\n";
	b << "holds(neg(G),0) :- fpart(G,F), not holds(F,0).\n";

	// % Check static laws in the initial state
	b << ":- caused(F,G,\"true\",\"true\"), holds(G,0), not holds(F,0).\n";

	// % Get nicer output
	c << "hol(F,0) :- fluent(F), holds(F,0).\n";

	// #cumulative t.
	
	// % Artificial fluent "true" holds in every state
	c << "holds(\"true\","+T+").\n";
	
	// % Artificial action "true" is executed in every step
	c << "occurs(\"true\","+T+"-1).\n";
	
	if(lang != "m") {
		// % Guess atomic actions to apply
		c << "1 { occurs(A,"+T+"-1), occurs(neg(A),"+T+"-1) } 1 :- action(A).\n";
	}
	
	// % Propagate composite fluents
	c << "holds(G,"+T+")      :- cfluent(G), holds(F,"+T+") : fpart(G,F).\n";
	c << "holds(neg(G),"+T+") :- fpart(G,F), not holds(F,"+T+").\n";
	
	// % Propagate composite actions
	c << "occurs(A,"+T+"-1)      :- caction(A), occurs(B,"+T+"-1) : apart(A,B).\n";
	c << "occurs(neg(A),"+T+"-1) :- caction(A), apart(A,B), not occurs(B,"+T+"-1).\n";
	
	// % Apply the causal rule that fire
	c << "holds(F,"+T+") :- caused(F,G,P,A), occurs(A,"+T+"-1), holds(P,"+T+"-1), not { holds(G,"+T+") } 0.\n";
	
	// % Check static laws in every state
	c << ":- caused(F,G,\"true\",\"true\"), holds(G,"+T+"), not holds(F,"+T+").\n";
		
	// % Successor state must be total and consistent
	c << ":- holds(F,"+T+"), holds(neg(F),"+T+").\n";
	c << ":- fluent(F), not holds(F,"+T+"), not holds(neg(F),"+T+").\n";
	
	// % Get nicer output
	c << "occ(A,"+T+"-1) :- action(A), occurs(A,"+T+"-1).\n";
	c << "hol(F,"+T+  ") :- fluent(F), holds(F,"+T +").\n";
	
	// hide everything except actions and fluents
	if(debug < 3) b << "#hide.\n";
	b << "#show occ(X1,T).\n";
	if(debug > 0) b << "#show hol(X1,T).\n";
}

void Printer::printBackwardsEncoding() {
	b << "\n%\n% Backwards Encoding of Action Language C\n%\n";
	
	// #base.
	
	// % Artificial fluent "true" holds in goal state
	b << "holds(\"true\",0).\n";
	
	// % Guess goal state
	b << "1 { holds(F,0), holds(neg(F),0) } 1 :- fluent(F).\n";
	
	// % Propagate composite fluents
	b << "holds(G,0)      :- cfluent(G), holds(F,0) : fpart(G,F).\n";
	b << "holds(neg(G),0) :- fpart(G,F), not holds(F,0).\n";
	
	// % shortcut for laws
	b << "static(F,G)      :- caused(F,G,\"true\",\"true\").\n";
	b << "dynamic(F,G,P,A) :- caused(F,G,P,A), not static(F,G).\n";
	
	// % Check static laws
	b << ":- static(F,G), holds(G,0), not holds(F,0).\n";
	
	// #cumulative t.
	
	// % Artificial action "true" is applied in every step
	c << "occurs(\"true\",-"+T+").\n";
	
	if(lang != "m") {
		// % Guess atomic actions to apply
		c << "1 { occurs(A,-"+T+"), occurs(neg(A),-"+T+") } 1 :- action(A).\n";
	}

	// % Propagate composite actions
	c << "occurs(A,-"+T+")      :- caction(A), occurs(B,-"+T+") : apart(A,B).\n";
	c << "occurs(neg(A),-"+T+") :- caction(A), apart(A,B), not occurs(B,-"+T+").\n";
	
	// % Guess predecessor state
	c << "1 { holds(F,-"+T+"), holds(neg(F),-"+T+") } 1 :- fluent(F).\n";
	
	// % Artificial fluent "true" holds in the predecessor state
	c << "holds(\"true\",-"+T+").\n";
	
	// % Propagate composite fluents
	c << "holds(G,-"+T+")      :- cfluent(G), holds(F,-"+T+") : fpart(G,F).\n";
	c << "holds(neg(G),-"+T+") :- fpart(G,F), not holds(F,-"+T+").\n";
	
	// % Check static laws
	c << ":- static(F,G), holds(G,-"+T+"), not holds(F,-"+T+").\n";
	
	// % Determine dynamic laws that fire
	c << "fire(F,G,P,A,-"+T+") :- dynamic(F,G,P,A), occurs(A,-"+T+"), holds(P,-"+T+"), holds(G,-"+T+"+1).\n";
	
	// % Effects of causal laws that fire must be consistent with successor state
	c << ":- fire(F,G,P,A,-"+T+"), not holds(F,-"+T+"+1).\n";
	
	// % For every successor state literal, at least one causal rule must fire
	c << ":- fluent(F), holds(    F ,-"+T+"+1), { fire(    F ,G,P,A,-"+T+") : dynamic(    F ,G,P,A), holds(H,-"+T+"+1) : static(    F ,H) } 0.\n";
	c << ":- fluent(F), holds(neg(F),-"+T+"+1), { fire(neg(F),G,P,A,-"+T+") : dynamic(neg(F),G,P,A), holds(H,-"+T+"+1) : static(neg(F),H) } 0.\n";
	
	// % Get nicer output
	c << "occ(A,-"+T+") :- action(A), occurs(A,-"+T+").\n";
	
	// hide everything except actions and fluents
	if(debug < 3) b << "#hide.\n";
	b << "#show occ(X1,T).\n";
	if(debug > 0) b << "#show holds(X1,T).\n";
}

void Printer::printLTLAuxilliaries() {
	if(printed_ltl_aux_) return;
// % el = loop_start
// % le = loop_exists
// % nl = after_loop_start
// % il = in_loop
	
	// % guess which state is equivalent to the last
	c << "0 { loop_start(T-1) : time(T) } 1.\n";
	if(debug > 2) c << "#show loop_start/1.\n";
	
	// % a loop exists if there's a start
	c << "loop_exists :- loop_start(T-1).\n";
	if(debug > 0) c << "#show loop_exists.\n";

	// % predicate for first state after loop start
	c << "after_loop_start(T) :- loop_start(T-1).\n";
	if(debug > 2) c << "#show after_loop_start/1.\n";

	// % states which are in the loop
	c << "in_loop(T) :- loop_start(T-1).\n";
	c << "in_loop(T) :- in_loop(T-1).\n";
	if(debug > 2) c << "#show in_loop/1.\n";
	
	// % idling only at the beginning
	v << ":- idle(T), not idle(T-1).\n";
	
	// % disallow loop by idling
	v << ":- loop_exists, idle(t-1).\n";

	// % compute only counter examples
	v << ":- not ltl_counter_example.\n";
	v << "#show ltl_counter_example.\n";

	if(no_direct_enc) {		
		// % define idling in a lparse compatible way
		c << "idle(F,T-1) :- holds(F,T-1), holds(F,T), fluent(F).\n";
		c << "idle(F,T-1) :- holds(neg(F),T-1), holds(neg(F),T), fluent(F).\n";
		c << "idle(T-1) :- idle(F,T-1) : fluent(F).\n";
		
		// % check that loop guess was correct
		v << ":- loop_start(T-1), holds(F,T-1), holds(neg(F),t).\n";
		v << ":- loop_start(T-1), holds(F,t), holds(neg(F),T-1).\n";
	}
	
	printed_ltl_aux_ = true;
}

void Printer::printIncLTLAuxilliaries() {
	if(printed_ltl_aux_) return;
// % el = loop_start
// % le = loop_exists
// % nl = after_loop_start
// % il = in_loop

	c << "time(t).\n";

	// % guess which state is equivalent to the last
	// % incremental version of  0 { loop_start(T-1) : time(T) } 1.
	c << "0 { loop_start(t-1) } 1 :- not in_loop(t-1).\n";
	if(debug > 2) c << "#show loop_start/1.\n";

	// % states which are in the loop
	c << "in_loop(t) :- loop_start(t-1).\n";
	c << "in_loop(t) :- in_loop(t-1).\n";
	if(debug > 2) c << "#show in_loop/1.\n";

	// % a loop exists if there's a start
	c << "loop_exists(t-1) :- loop_start(t-1).\n";
	c << "loop_exists(t-1) :- loop_exists(t-2).\n";
	if(debug > 0) c << "#show loop_exists/1.\n";

	// % predicate for first state after loop start
	c << "after_loop_start(t) :- loop_start(t-1).\n";
	if(debug > 2) c << "#show after_loop_start/1.\n";

	// % once there is a counter example there will be always one
	c << "ltl_counter_example(t) :- ltl_counter_example(t-1).\n";

	// % compute only counter examples
	v << ":- not ltl_counter_example(t).\n";
	v << "#show ltl_counter_example/1.\n";

	// % idling only at the beginning
	v << ":- idle(t), not idle(t-1).\n";

	// % disallow loop by idling
	v << ":- loop_exists(t), idle(t-1).\n";

	if(no_direct_enc) {
		// % idling if no fluents changed
		c << "idle(F,t-1) :- holds(F,t-1), holds(F,t), fluent(F).\n";
		c << "idle(F,t-1) :- holds(neg(F),t-1), holds(neg(F),t), fluent(F).\n";
		c << "idle(t-1) :- idle(F,t-1) : fluent(F).\n";

		// % check that loop guess was correct
		v << ":- loop_start(T-1), holds(F,T-1), holds(neg(F),t), time(T).\n";
		v << ":- loop_start(T-1), holds(F,t), holds(neg(F),T-1), time(T).\n";
	}

	printed_ltl_aux_ = true;
}

