/*
 * Copyright (c) 2003-2012
 * Distributed Systems Software.  All rights reserved.
 * See the file LICENSE for redistribution information.
 *
 * $Id: xml.h 2594 2012-10-19 17:28:49Z brachman $
 */

/*****************************************************************************
 * COPYRIGHT AND PERMISSION NOTICE
 * 
 * Copyright (c) 2001-2003 The Queen in Right of Canada
 * 
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation 
 * the rights to use, copy, modify, merge, publish, distribute, and/or sell
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, provided that the above copyright notice(s) and this
 * permission notice appear in all copies of the Software and that both the
 * above copyright notice(s) and this permission notice appear in supporting
 * documentation.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF THIRD PARTY RIGHTS.
 * IN NO EVENT SHALL THE COPYRIGHT HOLDER OR HOLDERS INCLUDED IN THIS NOTICE 
 * BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES,
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, 
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS 
 * SOFTWARE.
 * 
 * Except as contained in this notice, the name of a copyright holder shall not
 * be used in advertising or otherwise to promote the sale, use or other
 * dealings in this Software without prior written authorization of the
 * copyright holder.
 ***************************************************************************/

#ifndef _XML_H_
#define _XML_H_

#include <ctype.h>
#include <string.h>
#include <expat.h>

typedef struct Parse_xml_error {
  char *name;
  char *mesg;
  int line;
  int pos;
} Parse_xml_error;

typedef enum Parse_attr_type {
  ATTR_REQUIRED  = 0,
  ATTR_IMPLIED   = 1,
  ATTR_IGNORE    = 2,
  ATTR_IGNORE_NS = 3,
  ATTR_END       = 4
} Parse_attr_type;

typedef struct Parse_attr_ctl {
  int warn_on_invalid_syntax;
} Parse_attr_ctl;

/*
 * Specification for an XML attribute value validation callback.
 * The VALIDATE function is called, with a pointer to this structure,
 * and the name and value of the attribute.
 * VALIDATE returns NULL if VALUE is invalid and if ERRMSG is non-NULL,
 * it is set to point to an error message; otherwise, the returned value
 * is the one to be used (not necessarily the same as VALUE).
 * VALUES is an array of valid attribute values; if non-zero, NOCASE means
 * that upper and lower case are equivalent for VALUE.
 * ARG is a function-dependent argument for VALIDATE.
 */
typedef struct Parse_attr_validate Parse_attr_validate;
struct Parse_attr_validate {
  char *(*validate)(Parse_attr_validate *v, const char *name,
					const char *value, char **errmsg);
  char **values;
  int nocase;
  void *arg;
};

#define ATTR_BINARY	(&parse_xml_attr_validate_binary)
extern Parse_attr_validate parse_xml_attr_validate_binary;

typedef struct Parse_attr_tab {
  char *name;				/* The attribute name */
  char **value;				/* Set to the attribute value */
  Parse_attr_type type;		/* The type of attribute */
  Parse_attr_validate *validate;	/* Optional attribute value validation */
  Parse_attr_ctl *ctl;		/* Optional control info */
  int count;				/* Used internally */
} Parse_attr_tab;

typedef enum Parse_xml_result {
  PARSE_XML_ERROR    = -1,
  PARSE_XML_EMPTY    = 0,
  PARSE_XML_OK       = 1
} Parse_xml_result;

#ifdef __cplusplus
extern "C" {
#endif

extern void parse_xml_init(char *name, XML_Parser p);
extern void parse_xml_end(void);
extern Parse_xml_result parse_xml_push(void *state);
extern Parse_xml_result parse_xml_pop(void **state);
extern Parse_xml_result parse_xml_top(void **state);
extern int parse_xml_is_empty(void);
extern int parse_xml_is_not_empty(void);
extern void parse_xml_set_error(char *);
extern int parse_xml_is_error(Parse_xml_error *);
extern char *parse_xml_get_error(void);
extern void parse_xml_set_complete(void);
extern int parse_xml_is_complete(void);
extern int parse_xml_attr(Parse_attr_tab *tab, const char **attr,
						  char **errmsg);
extern char *parse_xml_attr_validator(Parse_attr_validate *v, const char *name,
									  const char *value, char **ermsg);
extern void parse_xml_default_handler(void *userData, const XML_Char *s,
									  int len);
extern void parse_xml_comment_handler(void *userData, const XML_Char *data);

extern int parse_xmlns_name(const char *expanded_name, char **uri,
							char **name);
extern void parse_xml_init_xmlns(char *ns);
extern char *parse_xml_xmlns(void);
extern char *xml_escape_cdata(char *str);
extern char *xml_unescape_cdata(char *str);
extern char *xml_attribute_value_encode(char *value, int quote_char);
extern char *xml_attribute_value_decode(char *enc_value);
extern char *xml_cdata_canonicalize(char *value);

#ifdef __cplusplus
}
#endif

#endif
