/*
 * Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either 
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public 
 * License along with this library.  If not, see <http://www.gnu.org/licenses/>. */

#include "JobPreview.h"

#include <QFileInfo>
#include <qendian.h>

#include <klocalizedstring.h>
#include <threadweaver/ThreadWeaver.h>

#include "libraw/libraw.h"

#include "LibRawInterface.h"
#include "MessagesModel.h"
#include "RawImageInfo.h"
#include "PostProcessor.h"
#include "PreviewInfo.h"
#include "ProcessingOptions.h"

struct JobPreview::Private {
  Private(const ProcessingOptions& _processingOptions) : processor(_processingOptions), processingOptions( _processingOptions )
  {
  }
  RawImageInfoSP rawImageInfo;
  PostProcessor processor;
  ProcessingOptions processingOptions;
  static ThreadWeaver::Weaver* weaver;
  int priority;
};

ThreadWeaver::Weaver* JobPreview::Private::weaver = 0;

JobPreview::JobPreview( RawImageInfoSP _rawImageInfo, int _priority ) : d(new Private(_rawImageInfo->processingOptions()))
{
  Q_ASSERT( _rawImageInfo );
  d->rawImageInfo = _rawImageInfo;
  d->processor.setConvertToSRGB( true );
  d->priority = _priority;
}

JobPreview::~JobPreview()
{
  delete d;
}

ThreadWeaver::Weaver* JobPreview::weaver()
{
  if(not Private::weaver)
  {
    Private::weaver = new ThreadWeaver::Weaver;
    Private::weaver->setMaximumNumberOfThreads( 1 );
  }
  return Private::weaver;
}

int JobPreview::priority() const
{
  return d->priority;
}

void JobPreview::run()
{
  d->rawImageInfo->setStatus( RawImageInfo::PREVIEWING );

  LibRaw raw;
  LibRawInterface::prepareRaw(raw, d->processingOptions);
  raw.imgdata.params.half_size = 1;

  QByteArray imageData;
  int width, height;
  if(not LibRawInterface::decodeRaw(raw, d->rawImageInfo, imageData, width, height, i18n("Preview")))
  {
    return;
  }
  
  emit( emitIntermediateData( (uchar*)imageData.data(), width, height ) );
  d->processor.apply16( imageData );
  PreviewInfo* pi = new PreviewInfo( d->rawImageInfo, width, height, imageData );
  d->rawImageInfo->setStatus( RawImageInfo::IDLING );
  emit( imageFinished( pi ) );
  MessagesModel::instance()->tellInfo(i18n("Preview"), i18n("Finished preview of %1.", d->rawImageInfo->fileInfo().absoluteFilePath()));
}

#include "JobPreview.moc"
