/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/*
 * SPDX-FileCopyrightText: 2002 Mikael Hallendal <micke@imendio.com>
 * SPDX-FileCopyrightText: 2008 Imendio AB
 * SPDX-FileCopyrightText: 2017, 2018 Sébastien Wilmet <swilmet@gnome.org>
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

#ifndef DH_LINK_H
#define DH_LINK_H

#include <glib-object.h>

G_BEGIN_DECLS

/**
 * DhLinkType:
 * @DH_LINK_TYPE_BOOK: The top-level page of a #DhBook.
 * @DH_LINK_TYPE_PAGE: A page.
 * @DH_LINK_TYPE_KEYWORD: Another kind of keyword.
 * @DH_LINK_TYPE_FUNCTION: A function keyword.
 * @DH_LINK_TYPE_STRUCT: A struct keyword.
 * @DH_LINK_TYPE_MACRO: A macro keyword.
 * @DH_LINK_TYPE_ENUM: An enum keyword.
 * @DH_LINK_TYPE_TYPEDEF: A typedef keyword.
 * @DH_LINK_TYPE_PROPERTY: A property keyword.
 * @DH_LINK_TYPE_SIGNAL: A signal keyword.
 *
 * The type of the content the link points to.
 */
typedef enum {
        DH_LINK_TYPE_BOOK,
        DH_LINK_TYPE_PAGE,
        DH_LINK_TYPE_KEYWORD,
        DH_LINK_TYPE_FUNCTION,
        DH_LINK_TYPE_STRUCT,
        DH_LINK_TYPE_MACRO,
        DH_LINK_TYPE_ENUM,
        DH_LINK_TYPE_TYPEDEF,
        DH_LINK_TYPE_PROPERTY,
        DH_LINK_TYPE_SIGNAL
} DhLinkType;

/**
 * DhLinkFlags:
 * @DH_LINK_FLAGS_NONE: No flags set.
 * @DH_LINK_FLAGS_DEPRECATED: The symbol that the link points to is deprecated.
 */
typedef enum {
        DH_LINK_FLAGS_NONE       = 0,
        DH_LINK_FLAGS_DEPRECATED = 1 << 0
} DhLinkFlags;

typedef struct _DhLink DhLink;

#define DH_TYPE_LINK (dh_link_get_type ())

GType        dh_link_get_type           (void);

DhLink *     dh_link_new_book           (const gchar   *base_path,
                                         const gchar   *book_id,
                                         const gchar   *book_title,
                                         const gchar   *relative_url);

DhLink *     dh_link_new                (DhLinkType     type,
                                         DhLink        *book_link,
                                         const gchar   *name,
                                         const gchar   *relative_url);

DhLink *     dh_link_ref                (DhLink        *link);

void         dh_link_unref              (DhLink        *link);

DhLinkType   dh_link_get_link_type      (DhLink        *link);

DhLinkFlags  dh_link_get_flags          (DhLink        *link);

void         dh_link_set_flags          (DhLink        *link,
                                         DhLinkFlags    flags);

const gchar *dh_link_get_name           (DhLink        *link);

gboolean     dh_link_match_relative_url (DhLink        *link,
                                         const gchar   *relative_url);

gboolean     dh_link_belongs_to_page    (DhLink        *link,
                                         const gchar   *page_id);

gchar *      dh_link_get_uri            (DhLink        *link);

const gchar *dh_link_get_book_title     (DhLink        *link);

const gchar *dh_link_get_book_id        (DhLink        *link);

gint         dh_link_compare            (gconstpointer  a,
                                         gconstpointer  b);

const gchar *dh_link_type_to_string     (DhLinkType     link_type);

G_END_DECLS

#endif /* DH_LINK_H */
