/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_case_cstr.hpp
 *
 * @brief [LEVEL: beta] Macros to construct Test_Suites. 
 *
 * $Id: test_case_cstr.hpp,v 1.26 2005/06/23 09:54:24 esdentem Exp $
 * 
 * @author Christian Schallhart
 *
 * This files contains the macros to construct a standard @ref
 * diagnostics::unittest::Test_Suite of name NAME. Such a Test_Suite
 * consists of
 *
 * @arg NAME/fct/normal Test_Suite which contains those Test_Cases
 * which test for the indented usage. A @ref
 * diagnostics::unittest::Test_Case is added to this suite with the
 * macro @ref TEST_NORMAL_CASE.
 *
 * @arg NAME/fct/abnormal Test_Suite which contains those Test_Cases
 * which test for invalid usage. A @ref
 * diagnostics::unittest::Test_Case is added to this suite with the
 * macro @ref TEST_ABNORMAL_CASE.
 *
 * @arg NAME/nonfct Test_Suite which contains non-functional test such
 * as performance evaluations. A @ref diagnostics::unittest::Test_Case
 * is added to this suite with the macro @ref TEST_NONFCT_CASE.
 *
 * @arg NAME/impl Test_Suite contains tests which are aimed at testing
 * whether the platform is suitable for the component, e.g., if big
 * endians are necessary for correctness, it should be tested it there.
 *
 * Test_Cases in NAME/fct/ can be placed at @ref
 * diagnostics::LEVEL_PROD, @ref diagnostics::LEVEL_DEBUG or @ref
 * diagnostics::LEVEL_AUDIT. A test which is placed at LEVEL_PROD is
 * also placed at the two other levels. A test at LEVEL_DEBUG is also
 * placed LEVEL_AUDIT. A Test_Case in NAME/nonfct/ will be only placed
 * in LEVEL_PROD.
 *
 * Please read the documentation of @ref diagnostics::Level_t for a
 * description of the different build-levels.
 *
 * A test suite iconsists of a definition of test code in terms
 * procedures with the signature void ()(Test_Data &) or classes with a
 * corresponding void operator()(Test_Data &) const. These procedures
 * and and the instances of these classes are placed in the Test_Suite
 * with one of the macros TEST_NORMAL_CASE, TEST_ABNORMAL_CASE,
 * TEST_NONFCT_CASE. 
 *
 * With @ref TEST_SUITE_BEGIN and @ref TEST_SUITE_END the @ref
 * ::diagnostics::unittest::build_test_suite procedure is built.
 *
 * The following is a brief example for a Test_Suite constructed with
 * these macros.
 * @include test_case_cstr_example.cpp
 *
 * @attention There can only one Test_Suite constructed this way per
 * executable.
 *
 * @test diagnostics/unittest/test_case_cstr.t.cpp
 */

#ifndef DIAGNOSTICS__UNITTEST__TEST_CASE_CSTR_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__UNITTEST__TEST_CASE_CSTR_HPP__INCLUDE_GUARD

#include <diagnostics/unittest/testing_namespaces.hpp>
#include <diagnostics/unittest/func_test_case.hpp>
#include <diagnostics/unittest/test_suite.hpp>
#include <diagnostics/unittest/build_test_suite.hpp>

/**
 * @brief Builds a standard @ref diagnostics::unittest::Test_Suite. 
 *
 */
#define TEST_BUILD_TEST_SYSTEM(NAME) \
   ::diagnostics::unittest::Test_Suite* fct_test_suite=     new ::diagnostics::unittest::Test_Suite("fct"); \
   ::diagnostics::unittest::Test_Suite* normal_test_suite=  new ::diagnostics::unittest::Test_Suite("normal"); \
   ::diagnostics::unittest::Test_Suite* abnormal_test_suite=new ::diagnostics::unittest::Test_Suite("abnormal"); \
   ::diagnostics::unittest::Test_Suite* nonfct_test_suite=  new ::diagnostics::unittest::Test_Suite("nonfct"); \
   ::diagnostics::unittest::Test_Suite* impl_test_suite=    new ::diagnostics::unittest::Test_Suite("impl"); \
   ::diagnostics::unittest::Test_Suite* test_suite=         new ::diagnostics::unittest::Test_Suite(NAME); \
   fct_test_suite->add(normal_test_suite); \
   fct_test_suite->add(abnormal_test_suite); \
   test_suite->add(fct_test_suite); \
   test_suite->add(nonfct_test_suite); \
   test_suite->add(impl_test_suite) 

#define TEST_INTERNAL_TO_STRING(A) TEST_INTERNAL_TO_STRING_1(A)
#define TEST_INTERNAL_TO_STRING_1(A) #A

/**
 * @brief Building a Standard @ref diagnostics::unittest::Test_Suite.
 *
 * Used to build a @ref diagnostics::unittest::Test_Suite which contains up the @ref
 * diagnostics::unittest::Test_Suite @a /NAME/fct, @a /NAME/fct/normal, @a
 * /NAME/fct/abnormal, @a /NAME/nonfct and @a /NAME/impl_test_suite.
 * This @ref diagnostics::unittest::Test_Suite will be returned by TEST_SUITE_END -- where
 * NAME the name of the root Test_Suite.
 */
#define TEST_SUITE_BEGIN \
DIAGNOSTICS_NAMESPACE_BEGIN; \
UNITTEST_NAMESPACE_BEGIN; \
::diagnostics::unittest::Test_Suite* build_test_suite() \
{ \
   TEST_BUILD_TEST_SYSTEM(TEST_INTERNAL_TO_STRING(TEST_COMPONENT_NAME)); \
   using namespace :: TEST_COMPONENT_NAMESPACE; \
   using namespace :: TEST_COMPONENT_NAMESPACE :: TEST_NAMESPACE; \
   using namespace :: TEST_COMPONENT_NAMESPACE :: TEST_NAMESPACE :: TEST_COMPONENT_TEST_NAMESPACE

#define TEST_SUITE_END \
   return test_suite; \
} \
UNITTEST_NAMESPACE_END; \
DIAGNOSTICS_NAMESPACE_END


DIAGNOSTICS_NAMESPACE_BEGIN; 
UNITTEST_NAMESPACE_BEGIN;
INTERNAL_NAMESPACE_BEGIN;

/**
 * @name Helper Functions for the adding macros
 *
 * The reason for introducing them is to enforce correct typing... 
 * @{
 */
template<typename FUNC>
inline void fct_test_case_add(FUNC const & func, 
			      ::std::string const & name,
			      int const timeout, 
			      bool const rerunable,
			      Level_t const min_level, 
			      Test_Suite * suite)
{
    suite->add(create_func_test_case<FUNC>(func,name,timeout,rerunable,
					   min_level==LEVEL_PROD,
					   min_level!=LEVEL_AUDIT,
					   true));
}

template<typename FUNC>
inline void nonfct_test_case_add(FUNC const & func, 
				 ::std::string const & name,
				 int const timeout, 
				 bool const rerunable,
				 Test_Suite * suite)
{
    suite->add(create_func_test_case<FUNC>(func,name,timeout,rerunable,
					   true,
					   false,
					   false));
}
// @}
INTERNAL_NAMESPACE_END;
UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

/**
 * @brief Adds a @ref diagnostics::unittest::Test_Case which tests for normal usage 
 *
 * The name of Test_Case is set to #FUNC (a leading '&' is removed),
 * and its timeout is set to 3 seconds.  It is added to the @ref
 * diagnostics::unittest::Test_Suite NAME/fct/normal.
 *
 * @param FUNC a procedure or object with the signature void
 * operator()(@ref diagnostics::unittest::Test_Data & test_data) const
 * @param LEVEL the lowest build @ref diagnostics::Level_t which this
 * @ref diagnostics::unittest::Test_Case is valid for
 */
// braces around FUNC lead to a parse error of gcc-2.95 in the context of templated objects
#define TEST_NORMAL_CASE(FUNC,LEVEL)  \
    ::diagnostics::unittest::internal::fct_test_case_add(FUNC,(#FUNC[0]=='&' ? #FUNC + 1 : #FUNC),3,true,::diagnostics::LEVEL,normal_test_suite)

/**
 * @brief Adds a @ref diagnostics::unittest::Test_Case which tests for normal usage 
 *
 * same parameters as in @ref TEST_NORMAL_CASE but also providing
 * @param RERUNABLE, which is true, if the test case can be run more than once.
 */
#define TEST_NORMAL_CASE1(FUNC,LEVEL,TIMEOUT,RERUNABLE)  \
    ::diagnostics::unittest::internal::fct_test_case_add(FUNC,(#FUNC[0]=='&' ? #FUNC + 1 : #FUNC),\
                                                         TIMEOUT,RERUNABLE,::diagnostics::LEVEL,normal_test_suite)

/**
 * @brief Adds a @ref diagnostics::unittest::Test_Case which tests for normal usage 
 *
 * same parameters as in @ref TEST_NORMAL_CASE2 but also providing
 * @param NAME, which allows to set the name explicitly. 
 */ 
#define TEST_NORMAL_CASE2(FUNC,LEVEL,TIMEOUT,RERUNABLE,NAME)  \
    ::diagnostics::unittest::internal::fct_test_case_add(FUNC,NAME,\
                                                         TIMEOUT,RERUNABLE,::diagnostics::LEVEL,normal_test_suite)



/**
 * @brief Adds a @ref diagnostics::unittest::Test_Case which tests for invalid usage 
 *
 * The name of Test_Case is set to #FUNC (a leading '&' is removed),
 * and its timeout is set to 3 seconds.  It is added to the @ref
 * diagnostics::unittest::Test_Suite NAME/fct/abnormal.
 *
 * @param FUNC a procedure or object with the signature void
 * operator()(@ref diagnostics::unittest::Test_Data & test_data) const
 * @param LEVEL the lowest build @ref diagnostics::Level_t which this @ref
 * diagnostics::unittest::Test_Case is valid for
 */
#define TEST_ABNORMAL_CASE(FUNC,LEVEL) \
  ::diagnostics::unittest::internal::fct_test_case_add(FUNC,(#FUNC[0]=='&' ? #FUNC + 1 : #FUNC),3,true,::diagnostics::LEVEL,abnormal_test_suite)

/**
 * @brief Adds a @ref diagnostics::unittest::Test_Case which test nonfunctional properties 
 *
 * The name of Test_Case is set to #FUNC (a leading '&' is removed),
 * and it is only placed into @ref diagnostics::LEVEL_PROD.  It is added to the
 * @ref diagnostics::unittest::Test_Suite NAME/nonfct.
 *
 * @param FUNC a procedure or object with the signature void
 * operator()(@ref diagnostics::unittest::Test_Data & test_data) const
 * @param TIMEOUT the timeout of the @ref
 * diagnostics::unittest::Test_Case in seconds
 */
#define TEST_NONFCT_CASE(FUNC,TIMEOUT) \
  ::diagnostics::unittest::internal::nonfct_test_case_add((FUNC),(#FUNC[0]=='&' ? #FUNC + 1 : #FUNC),(TIMEOUT),true,nonfct_test_suite)

/**
 * @brief Adds a @ref diagnostics::unittest::Test_Case which test implementation details. 
 *
 * The name of Test_Case is set to #FUNC (a leading '&' is removed),
 * and it is only placed into @ref diagnostics::LEVEL_PROD.  It is added to the
 * @ref diagnostics::unittest::Test_Suite NAME/impl. The timeout is set to 3sec.
 *
 * @param FUNC a procedure or object with the signature void
 * operator()(@ref diagnostics::unittest::Test_Data & test_data) const
 */
#define TEST_IMPL_CASE(FUNC) \
  ::diagnostics::unittest::internal::fct_test_case_add((FUNC),(#FUNC[0]=='&' ? #FUNC + 1 : #FUNC),3,true,::diagnostics::LEVEL_PROD,impl_test_suite)


/**
 * @brief Adds a @ref diagnostics::unittest::Test_Suite
 *
 * @param FUNC a procedure which generates the suite or a directly a
 * suite.
 */
#define TEST_ADD_SUITE(FUNC) test_suite->add(FUNC)
#endif
// vim:ts=4:sw=4
