/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file actions.c
 * \brief Actions related functions
 */

#include <ffgtk.h>

/** global action list */
static GList *psActionList = NULL;

/**
 * \brief Get action list
 * \return global action list
 */
GList *getActions( void ) {
	return psActionList;
}

/**
 * \brief Replace action executive string expression
 * \param pnStr executive string
 * \param pnLocal local number
 * \param pnNumber target number
 * \param pnName target name
 * \return replaced name, you have to free this memory
 */
gchar *replaceNumber( gchar *pnStr, gchar *pnLocal, gchar *pnNumber, gchar *pnName ) {
	struct sPerson *psPerson = findPersonByNumber( pnNumber );
	GRegex *psLocal = g_regex_new( "%LOCALNUMBER%", G_REGEX_DOTALL | G_REGEX_OPTIMIZE, 0, NULL );
	GRegex *psNumber = g_regex_new( "%NUMBER%", G_REGEX_DOTALL | G_REGEX_OPTIMIZE, 0, NULL );
	GRegex *psDisplayName = g_regex_new( "%NAME%", G_REGEX_DOTALL | G_REGEX_OPTIMIZE, 0, NULL );
	GRegex *psFirstName = g_regex_new( "%FIRSTNAME%", G_REGEX_DOTALL | G_REGEX_OPTIMIZE, 0, NULL );
	GRegex *psLastName = g_regex_new( "%LASTNAME%", G_REGEX_DOTALL | G_REGEX_OPTIMIZE, 0, NULL );
	GRegex *psCompany = g_regex_new( "%COMPANY%", G_REGEX_DOTALL | G_REGEX_OPTIMIZE, 0, NULL );
	gchar *pnOut = NULL;

	gchar *pnTmp0 = g_regex_replace_literal( psNumber, pnStr, -1, 0, pnNumber, 0, NULL );
	gchar *pnTmp1 = g_regex_replace_literal( psLocal, pnTmp0, -1, 0, pnLocal, 0, NULL );

	if ( psPerson != NULL ) {
		/* We have person information, replace the regex */
		gchar *pnTmp2 = g_regex_replace_literal( psDisplayName, pnTmp1, -1, 0, psPerson -> pnDisplayName, 0, NULL );
		gchar *pnTmp3 = g_regex_replace_literal( psFirstName, pnTmp2, -1, 0, psPerson -> pnFirstName != NULL ? psPerson -> pnFirstName : "?", 0, NULL );
		gchar *pnTmp4 = g_regex_replace_literal( psCompany, pnTmp3, -1, 0, psPerson -> pnCompany != NULL ? psPerson -> pnCompany : "?", 0, NULL );
		pnOut = g_regex_replace_literal( psLastName, pnTmp4, -1, 0, psPerson -> pnLastName != NULL ? psPerson -> pnLastName : "?", 0, NULL );
		g_free( pnTmp4 );
		g_free( pnTmp3 );
		g_free( pnTmp2 );
	} else {
		/* We have NO person information, replace the regex with "?" */
		gchar *pnTmp2 = g_regex_replace_literal( psDisplayName, pnTmp1, -1, 0, pnName != NULL ? pnName : "???", 0, NULL );
		gchar *pnTmp3 = g_regex_replace_literal( psFirstName, pnTmp2, -1, 0, "?", 0, NULL );
		gchar *pnTmp4 = g_regex_replace_literal( psCompany, pnTmp3, -1, 0, "?", 0, NULL );
		pnOut = g_regex_replace_literal( psLastName, pnTmp4, -1, 0, "?", 0, NULL );
		g_free( pnTmp4 );
		g_free( pnTmp3 );
		g_free( pnTmp2 );
	}

	g_free( pnTmp1 );
	g_free( pnTmp0 );

	g_regex_unref( psLocal );
	g_regex_unref( psNumber );
	g_regex_unref( psDisplayName );
	g_regex_unref( psFirstName );
	g_regex_unref( psLastName );

	return pnOut;
}

/**
 * \brief Get index by number
 * \param pnLocal local number
 * \return index number
 */
int getIndexByNumber( gchar *pnLocal ) {
	const gchar *pnNumber;
	int nIndex;

	for ( nIndex = 0; nIndex < 29; nIndex++ ) {
		pnNumber = routerGetFonNumber( getActiveProfile(), nIndex );
		if ( pnNumber != NULL && strcmp( pnNumber, pnLocal ) == 0 ) {
			return nIndex;
		}
	}

	return 0;
}

/**
 * \brief Execute actions
 * \param nCallType previous call type
 * \param nCurrentType current call type
 * \param pnLocal local number
 * \param pnNumber target number
 * \param pnName target name
 */
void executeActions( gint nCallType, gint nCurrentType, gchar *pnLocal, gchar *pnNumber, gchar *pnName ) {
	GList *psList = getActions();
	struct sAction *psAction = NULL;
	gint nFlags = 0;
	gint nIndex;

	Debug( KERN_DEBUG, "Execute!!\n" );
	if ( psList == NULL ) {
		Debug( KERN_DEBUG, "List null!\n" );
		return;
	}

	if ( nCallType & CALL_TYPE_INCOMING ) {
		switch ( nCurrentType ) {
			case CALL_TYPE_INCOMING:
				/* Ring */
				nFlags = 0x10;
				break;
			case CALL_TYPE_CONNECT:
				/* Connected */
				nFlags = 0x01;
				break;
			case CALL_TYPE_DISCONNECT:
				/* Disconnected */
				if ( nCallType & CALL_TYPE_CONNECT ) {
					nFlags = 0x02;
				} else {
					nFlags = 0x20;
				}
				break;
		}
	} else if ( nCallType & CALL_TYPE_OUTGOING ) {
		switch ( nCurrentType ) {
			case CALL_TYPE_OUTGOING:
				/* DIAL */
				nFlags = 0x40;
				break;
			case CALL_TYPE_CONNECT:
				/* Connected */
				nFlags = 0x04;
				break;
			case CALL_TYPE_DISCONNECT:
				/* Disconnected */
				nFlags = 0x08;
				break;
		}
	}

	nIndex = getIndexByNumber( pnLocal );

	Debug( KERN_DEBUG, "Flags %x, nIndex: %d\n", nFlags, nIndex );

	/* Go through action list and compare flags for current flags, on match execute action */
	while ( psList != NULL ) {
		int nActionFlags;
		psAction = psList -> data;
		if ( psAction != NULL ) {
			nActionFlags = getActionFlags( getActiveProfile(), psAction -> pnName );
			Debug( KERN_DEBUG, "Action: %s, Flags: %x, Check: %d\n", psAction -> pnName, nActionFlags, nActionFlags & nFlags );
			if ( nActionFlags & nFlags ) {
				if ( nActionFlags & ( 1 << ( nIndex + ACTION_TYPE_MAX ) ) ) {
					gchar *pnTmp = replaceNumber( psAction -> pnExec, pnLocal, pnNumber, pnName );
					Debug( KERN_DEBUG, "Want to exec: '%s'\n", pnTmp );
					g_spawn_command_line_async( pnTmp, NULL );
					g_free( pnTmp );
				}
			}
		}

		psList = psList -> next;
	}
}

/**
 * \brief Add action from xmlnode
 * \param psNode xml node structure
 */
static void addAction( xmlnode *psNode ) {
	struct sAction *psAction = NULL;
	xmlnode *psChild = NULL;
	gchar *pnName = NULL;
	gchar *pnDescription = NULL;
	gchar *pnExec = NULL;

	psChild = xmlnode_get_child( psNode, "name" );
	if ( psChild != NULL ) {
		pnName = xmlnode_get_data( psChild );

		psChild = xmlnode_get_child( psNode, "description" );
		if ( psChild != NULL ) {
			pnDescription = xmlnode_get_data( psChild );

			psChild = xmlnode_get_child( psNode, "exec" );
			if ( psChild != NULL ) {
				pnExec = xmlnode_get_data( psChild );

				psAction = g_malloc0( sizeof( struct sAction ) );

				if ( psAction != NULL ) {
					/* Ok, now add information to psAction and prepend it to action list */
					psAction -> pnName = g_strdup( pnName );
					psAction -> pnDescription = g_strdup( pnDescription );
					psAction -> pnExec = g_strdup( pnExec );
					psActionList = g_list_prepend( psActionList, psAction );
				}

				g_free( pnExec );
			}

			g_free( pnDescription );
		}

		g_free( pnName );
	}
}

/**
 * \brief Find action by name
 * \param pnName name to search for
 * \return action structure or NULL
 */
struct sAction *findAction( const gchar *pnName ) {
	GList *psList = NULL;
	struct sAction *psAction = NULL;

	for ( psList = getActions(); psList != NULL && psList -> data != NULL; psList = psList -> next ) {
		psAction = psList -> data;
		if ( !strcmp( psAction -> pnName, pnName ) ) {
			return psAction;
		}
	}

	return NULL;
}

/**
 * \brief Convert action to xml node
 * \param psAction action structure
 * \return action xml node
 */
static xmlnode *actionToXmlnode( struct sAction *psAction ) {
	xmlnode *psNode = NULL;
	xmlnode *psChild = NULL;

	psNode = xmlnode_new( "action" );

	psChild = xmlnode_new_child( psNode, "name" );
	xmlnode_insert_data( psChild, psAction -> pnName, -1 );
	psChild = xmlnode_new_child( psNode, "description" );
	xmlnode_insert_data( psChild, psAction -> pnDescription, -1 );
	psChild = xmlnode_new_child( psNode, "exec" );
	xmlnode_insert_data( psChild, psAction -> pnExec, -1 );

	return psNode;
}

/**
 * \brief Convert actions to xml node
 * \return xml node containing actions
 */
static xmlnode *actionsToXmlnode(void){
	xmlnode *psNode = NULL;
	xmlnode *psChild = NULL;
	GList *psCurrent = NULL;

	psNode = xmlnode_new( "action" );
	xmlnode_set_attrib( psNode, "version", "1.0" );

	for ( psCurrent = getActions(); psCurrent != NULL && psCurrent -> data != NULL; psCurrent = psCurrent -> next ) {
		psChild = actionToXmlnode( psCurrent -> data );
		xmlnode_insert_child( psNode, psChild );
	}

	return psNode;
}

/**
 * \brief Save actions to disk
 */
void saveActions( void ) {
	xmlnode *psNode = NULL;
	gchar *pnData = NULL;

	psNode = actionsToXmlnode();

	pnData = xmlnode_to_formatted_str( psNode, NULL );

	if ( pnData != NULL ) {
		gint nFile = -1;
		gint nResult = 0;
		gchar *pnFile = g_build_filename( getUserDir(), "actions.xml", NULL );

		nFile = open( pnFile, O_RDWR | O_CREAT | O_TRUNC, 0600 );
		if ( nFile > 0 ) {
			nResult = write( nFile, pnData, strlen( pnData ) );
			if ( nResult != strlen( pnData ) ) {
				Debug( KERN_WARNING, "Could not save file %s\n", pnFile );
			}
			close( nFile );
		}
	}

	g_free( pnData );
	xmlnode_free( psNode );
}

/**
 * \brief Remove action by name
 * \param pnName action name we want to remove
 */
void removeAction( const gchar *pnName ) {
	struct sAction *psAction = findAction( pnName );

	if ( psAction != NULL ) {
		psActionList = g_list_remove( psActionList, psAction );
	}
}

/**
 * \brief Create new action with name
 * \param pnName action name
 * \return new action structure
 */
struct sAction *createAction( const gchar *pnName ) {
	struct sAction *psAction = NULL;

	psAction = g_malloc0( sizeof( struct sAction ) );

	if ( psAction != NULL ) {
		psAction -> pnName = g_strdup( pnName );
		psActionList = g_list_prepend( psActionList, psAction );
	} else {
		Debug( KERN_WARNING, "Could not add '%s'\n", pnName );
	}

	return psAction;
}

/**
 * \brief Load actions from disk
 */
void ActionsLoad( void ) {
	xmlnode *psNode = NULL;
	xmlnode *psChild = NULL;
	DIR *psDir = NULL;

	psDir = opendir( getUserDir() );
	if ( psDir == NULL ) {
		g_mkdir( getUserDir(), 0755 );
	} else {
		closedir( psDir );
	}

	psNode = readXmlFromFile( "actions.xml", _( "actions" ) );
	if ( psNode == NULL ) {
		Debug( KERN_DEBUG, "Could not read actions.xml\n" );
		return;
	}

	for ( psChild = xmlnode_get_child( psNode, "action" ); psChild != NULL; psChild = xmlnode_get_next_twin( psChild ) ) {
		addAction( psChild );
	}

	xmlnode_free( psNode );
}
