/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file audio_port.c
 * \brief PortAudio audio plugin
 * Parts are based upon pablio.c:
 *
 * Portable Audio Blocking Input/Output utility.
 *
 * Author: Phil Burk, http://www.softsynth.com
 *
 * This program uses the PortAudio Portable Audio Library.
 * For more information see: http://www.portaudio.com
 * Copyright (c) 1999-2000 Ross Bencina and Phil Burk
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <ffgtk.h>
#include <preferences.h>
#include <portaudio.h>

/* Does not work at the moment */
//#define USE_SPEEX

/** predefined backup values */
static gint nPortChannels = 2;
static gint nPortSampleRate = 8000;
static gint nPortBitsPerSample = 16;

#if defined( __i386__ ) || defined( __i486__ ) || defined( __i586__ ) || defined( __i686__ ) || defined( __x86_64__ )
	#define PaUtil_ReadMemoryBarrier() __asm__ volatile("lfence":::"memory")
	#define PaUtil_WriteMemoryBarrier() __asm__ volatile("sfence":::"memory")
#else
	#define PaUtil_ReadMemoryBarrier()
	#define PaUtil_WriteMemoryBarrier()
#endif

#ifdef USE_SPEEX
#include <speex/speex_echo.h>
SpeexEchoState *psEchoState;
int nSpeexEcho = 1;
#endif

#define BUFFER_SIZE 160

struct sPaUtilRingBuffer {
	long nBufferSize;
	long nWriteIndex;
	long nReadIndex;
	long nBigMask;
	long nSmallMask;
	char *pnBuffer;
};

typedef struct sPaUtilRingBuffer PaUtilRingBuffer;

struct sPortPrivate {
	PaStream *psOutStream;
	PaStream *psInStream;
	PaUtilRingBuffer sOutFifo;
	PaUtilRingBuffer sInFifo;
	int nBytesPerFrame;
	unsigned char nHasOut;
	unsigned char nHasIn;
};

/**
 * \brief Clear buffer
 * \param psBuffer ring buffer structure
 */
void PaUtil_FlushRingBuffer( PaUtilRingBuffer *psBuffer ) {
	psBuffer -> nWriteIndex = psBuffer -> nReadIndex = 0;
}

/**
 * \brief Initialize ring buffer
 * \param psBuffer ring buffer structure
 * \param nBytes buffer size, must be power of 2
 * \param pData allocated buffer
 * \return 0 on success, otherwise error
 */
long PaUtil_InitializeRingBuffer( PaUtilRingBuffer *psBuffer, long nBytes, void *pData ) {
	if ( ( ( nBytes - 1 ) & nBytes ) != 0 ) {
		Debug( KERN_WARNING, "Size not power of 2\n" );
		return -1;
	}

	psBuffer -> nBufferSize = nBytes;
	psBuffer -> pnBuffer = ( char * ) pData;
	PaUtil_FlushRingBuffer( psBuffer );

	psBuffer -> nBigMask = ( nBytes * 2 ) - 1;
	psBuffer -> nSmallMask = ( nBytes ) - 1;

	return 0;
}

/**
 * \brief Return number of bytes available for reading
 * \param psBuffer ring buffer structure
 * \return available size in bytes
 */
long PaUtil_GetRingBufferReadAvailable( PaUtilRingBuffer *psBuffer ) {
	PaUtil_ReadMemoryBarrier();

	return ( ( psBuffer -> nWriteIndex - psBuffer -> nReadIndex ) & psBuffer -> nBigMask );
}

/**
 * \brief Return number of bytes available for writing
 * \param psBuffer ring buffer structure
 * \return available size in bytes
 */
long PaUtil_GetRingBufferWriteAvailable( PaUtilRingBuffer *psBuffer ) {
	return ( psBuffer -> nBufferSize - PaUtil_GetRingBufferReadAvailable( psBuffer ) );
}

/**
 * \brief Get address of region(s) to which we can write data
 * \param psBuffer ring buffer structure
 * \param nBytes length of data
 * \param ppDataPtr1 first data pointer
 * \param pnSizePtr1 first data pointer length
 * \param ppDataPtr2 second data pointer
 * \param pnSizePtr2 second data pointer length
 * \return room available to be written
 */
long PaUtil_GetRingBufferWriteRegions( PaUtilRingBuffer *psBuffer, long nBytes, void **ppDataPtr1, long *pnSizePtr1, void **ppDataPtr2, long *pnSizePtr2 ) {
	long nIndex;
	long nAvailable = PaUtil_GetRingBufferWriteAvailable( psBuffer );

	if ( nBytes > nAvailable ) {
		nBytes = nAvailable;
	}

	/* Check to see if write is not contiguous */
	nIndex = psBuffer -> nWriteIndex & psBuffer -> nSmallMask;

	if ( ( nIndex + nBytes ) > psBuffer -> nBufferSize ) {
		/* Write data in two blocks that wrap the buffer */
		long nFirstHalf = psBuffer -> nBufferSize - nIndex;

		*ppDataPtr1 = &psBuffer -> pnBuffer[ nIndex ];
		*pnSizePtr1 = nFirstHalf;
		*ppDataPtr2 = &psBuffer -> pnBuffer[ 0 ];
		*pnSizePtr2 = nBytes - nFirstHalf;
	} else {
		*ppDataPtr1 = &psBuffer -> pnBuffer[ nIndex ];
		*pnSizePtr1 = nBytes;
		*ppDataPtr2 = NULL;
		*pnSizePtr2 = 0;
	}

	return nBytes;
}

/**
 * \brief Get address of region(s) to which we can read data
 * \param psBuffer ring buffer structure
 * \param nBytes length of data
 * \param ppDataPtr1 first data pointer
 * \param pnSizePtr1 first data pointer length
 * \param ppDataPtr2 second data pointer
 * \param pnSizePtr2 second data pointer length
 * \return room available to be read
 */
long PaUtil_GetRingBufferReadRegions( PaUtilRingBuffer *psBuffer, long nBytes, void **ppDataPtr1, long *pnSizePtr1, void **ppDataPtr2, long *pnSizePtr2 ) {
	long nIndex;
	long nAvailable = PaUtil_GetRingBufferReadAvailable( psBuffer );

	if ( nBytes > nAvailable ) {
		nBytes = nAvailable;
	}

	/* Check to see if write is not contiguous */
	nIndex = psBuffer -> nReadIndex & psBuffer -> nSmallMask;

	if ( ( nIndex + nBytes ) > psBuffer -> nBufferSize ) {
		/* Write data in two blocks that wrap the buffer */
		long nFirstHalf = psBuffer -> nBufferSize - nIndex;

		*ppDataPtr1 = &psBuffer -> pnBuffer[ nIndex ];
		*pnSizePtr1 = nFirstHalf;
		*ppDataPtr2 = &psBuffer -> pnBuffer[ 0 ];
		*pnSizePtr2 = nBytes - nFirstHalf;
	} else {
		*ppDataPtr1 = &psBuffer -> pnBuffer[ nIndex ];
		*pnSizePtr1 = nBytes;
		*ppDataPtr2 = NULL;
		*pnSizePtr2 = 0;
	}

	return nBytes;
}

/**
 * \brief Advance write index
 * \param psBuffer ring buffer structure
 * \param nBytes number of bytes
 * \return new write index
 */
long PaUtil_AdvanceRingBufferWriteIndex( PaUtilRingBuffer *psBuffer, long nBytes ) {
	PaUtil_WriteMemoryBarrier();

	return psBuffer -> nWriteIndex = ( psBuffer -> nWriteIndex + nBytes ) & psBuffer -> nBigMask;
}

/**
 * \brief Advance read index
 * \param psBuffer ring buffer structure
 * \param nBytes number of bytes
 * \return new read index
 */
long PaUtil_AdvanceRingBufferReadIndex( PaUtilRingBuffer *psBuffer, long nBytes ) {
	PaUtil_WriteMemoryBarrier();

	return psBuffer -> nReadIndex = ( psBuffer -> nReadIndex + nBytes ) & psBuffer -> nBigMask;
}

/**
 * \brief Write to ring buffer
 * \param psBuffer ring buffer structure
 * \param pData data that needs to be written
 * \param nBytes length of data
 * \return number of written bytes
 */
long PaUtil_WriteRingBuffer( PaUtilRingBuffer *psBuffer, const void *pData, long nBytes ) {
	long nSize1;
	long nSize2;
	long nWritten;
	void *pData1;
	void *pData2;

	nWritten = PaUtil_GetRingBufferWriteRegions( psBuffer, nBytes, &pData1, &nSize1, &pData2, &nSize2 );
	if ( nSize2 > 0 ) {
		memcpy( pData1, pData, nSize1 );
		pData = ( ( char * ) pData ) + nSize1;
		memcpy( pData2, pData, nSize2 );
	} else {
		memcpy( pData1, pData, nSize1 );
	}

	PaUtil_AdvanceRingBufferWriteIndex( psBuffer, nWritten );

	return nWritten;
}

/**
 * \brief Read from ring buffer
 * \param psBuffer ring buffer structure
 * \param pData data that needs to be written
 * \param nBytes length of data
 * \return number of read bytes
 */
long PaUtil_ReadRingBuffer( PaUtilRingBuffer *psBuffer, void *pData, long nBytes ) {
	long nSize1;
	long nSize2;
	long nRead;
	void *pData1;
	void *pData2;

	nRead = PaUtil_GetRingBufferReadRegions( psBuffer, nBytes, &pData1, &nSize1, &pData2, &nSize2 );
	if ( nSize2 > 0 ) {
		memcpy( pData, pData1, nSize1 );
		pData = ( ( char * ) pData ) + nSize1;
		memcpy( pData, pData2, nSize2 );
	} else {
		memcpy( pData, pData1, nSize1 );
	}

	PaUtil_AdvanceRingBufferReadIndex( psBuffer, nRead );

	return nRead;
}

/**
 * \brief Detect supported audio devices
 * \return 0
 */
static int portAudioDetectDevices( void ) {
	Pa_Initialize();

	return !Pa_GetDeviceCount();
}

/**
 * \brief Initialize audio device
 * \param nChannels number of channels
 * \param nSampleRate sample rate
 * \param nBitsPerSample number of bits per samplerate
 * \return TRUE on success, otherwise error
 */
static int portAudioInit( unsigned char nChannels, unsigned short nSampleRate, unsigned char nBitsPerSample ) {
	Pa_Initialize();

	/* TODO: Check if configuration is valid and usable */
	nPortChannels = nChannels;
	nPortSampleRate = nSampleRate;
	nPortBitsPerSample = nBitsPerSample;

	return 0;
}

/* OK */
static int outputCallback( const void *pInputBuffer, void *pOutputBuffer, unsigned long nFramesPerBuffer, const PaStreamCallbackTimeInfo *psTimeInfo, PaStreamCallbackFlags nStatusFlags, void *pUserData ) {
	struct sPortPrivate *psPrivate = pUserData;
	long nBytes = psPrivate -> nBytesPerFrame * nFramesPerBuffer;
#ifdef USE_SPEEX
	spx_int16_t *pnSpeexPtr = NULL;
#endif

	if ( pOutputBuffer != NULL ) {
		int nIndex;
		int nNumRead = PaUtil_ReadRingBuffer( &psPrivate -> sOutFifo, pOutputBuffer, nBytes );
		for ( nIndex = nNumRead; nIndex < nBytes; nIndex++ ) {
			( ( char * ) pOutputBuffer )[ nIndex ] = 0;
		}

#ifdef USE_SPEEX
		if ( nSpeexEcho ) {
			pnSpeexPtr = ( spx_int16_t * ) pOutputBuffer;

			/* Put frame into playback buffer */
			speex_echo_playback( psEchoState, pnSpeexPtr );
		}
#endif
	}

	return 0;
}

static int inputCallback( const void *pInputBuffer, void *pOutputBuffer, unsigned long nFramesPerBuffer, const PaStreamCallbackTimeInfo *psTimeInfo, PaStreamCallbackFlags nStatusFlags, void *pUserData ) {
	struct sPortPrivate *psPrivate = pUserData;
	long nBytes = psPrivate -> nBytesPerFrame * nFramesPerBuffer;
#ifdef USE_SPEEX
	spx_int16_t *pnSpeexPtr = NULL;
	spx_int16_t anPcm[ 160 ];
#endif

	if ( pInputBuffer != NULL ) {
#ifdef USE_SPEEX
		if ( nSpeexEcho ) {
			pnSpeexPtr = ( ( spx_int16_t * ) pInputBuffer );

			/* Perform echo cancellation */
			speex_echo_capture( psEchoState, pnSpeexPtr, anPcm );
		}
#endif

		int nNumRead = PaUtil_WriteRingBuffer( &psPrivate -> sInFifo, pInputBuffer, nBytes );
		if ( nNumRead != nBytes ) {
			PaUtil_FlushRingBuffer( &psPrivate -> sInFifo );
			PaUtil_WriteRingBuffer( &psPrivate -> sInFifo, pInputBuffer, nBytes );
		}
	}

	return 0;
}

PaError InitFifo( PaUtilRingBuffer *psBuffer, long nFrames, long nBytesPerFrame ) {
	long nBytes = nFrames * nBytesPerFrame;
	char *pnBuffer = ( char * ) malloc( nBytes );

	Debug( KERN_DEBUG, "nBytes: %d\n", nBytes );
	if ( pnBuffer == NULL ) {
		return paInsufficientMemory;
	}

	memset( pnBuffer, 0, nBytes );

	return ( PaError ) PaUtil_InitializeRingBuffer( psBuffer, nBytes, pnBuffer );
}

PaError TermFifo( PaUtilRingBuffer *psBuffer ) {
	if ( psBuffer -> pnBuffer ) {
		free( psBuffer -> pnBuffer );
	}

	psBuffer -> pnBuffer = NULL;

	return paNoError;
}

unsigned long RoundUpToNextPowerOf2( unsigned long nN ) {
	long nBits = 0;

	if ( ( ( nN - 1 ) & nN ) == 0 ) {
		return nN;
	}

	while ( nN > 0 ) {
		nN = nN >> 1;
		nBits++;
	}

	return ( 1 << nBits );
}

/**
 * \brief Open audio device
 * \return private data or NULL on error
 */
static void *portAudioOpen( void ) {
	struct sPortPrivate *psPrivate = malloc( sizeof( struct sPortPrivate ) );
	PaStreamParameters sOutputParameters;
	PaStreamParameters sInputParameters;
	PaError nErr;

	if ( psPrivate == NULL ) {
		return psPrivate;
	}
	memset( psPrivate, 0, sizeof( struct sPortPrivate ) );

	int nIndex;
	int nCount = Pa_GetHostApiCount();
	for ( nIndex = 0; nIndex < nCount; ++nIndex ) {
		const PaHostApiInfo *psHostInfo;

		psHostInfo = Pa_GetHostApiInfo( nIndex );
		if ( !psHostInfo ) {
			continue;
		}

		if ( psHostInfo -> defaultOutputDevice >= 0 ) {
			const PaDeviceInfo *psDevInfo;

			psDevInfo = Pa_GetDeviceInfo( psHostInfo -> defaultOutputDevice );


			if ( psDevInfo -> maxOutputChannels >= 1 ) {
				Debug( KERN_DEBUG, "id: %d\n", psHostInfo -> defaultOutputDevice );
				break;
			}
		}
	}
	Debug( KERN_DEBUG, "default: %d\n", Pa_GetDefaultOutputDevice() );

#ifdef USE_SPEEX
	/* Echo canceller with 100ms tail length */
	psEchoState = speex_echo_state_init( 160, 16000 );
#endif

	sOutputParameters.device = Pa_GetDefaultOutputDevice();
	sOutputParameters.channelCount = nPortChannels;
	sOutputParameters.sampleFormat = paInt16;
	sOutputParameters.hostApiSpecificStreamInfo = NULL;
	//sOutputParameters.suggestedLatency = Pa_GetDeviceInfo( sOutputParameters.device ) -> defaultLowOutputLatency;
	//sOutputParameters.suggestedLatency = ( 1.0 / 50.0 );

	int nNumFrames = RoundUpToNextPowerOf2( BUFFER_SIZE * 20 );
	Debug( KERN_DEBUG, "nNumFrames: %d\n", nNumFrames );
	psPrivate -> nBytesPerFrame = 2;
	nErr = InitFifo( &psPrivate -> sOutFifo, nNumFrames, psPrivate -> nBytesPerFrame );

	nNumFrames = RoundUpToNextPowerOf2( BUFFER_SIZE * 5 );
	sInputParameters.device = Pa_GetDefaultInputDevice();
	sInputParameters.channelCount = nPortChannels;
	sInputParameters.sampleFormat = paInt16;
	sInputParameters.hostApiSpecificStreamInfo = NULL;
	nErr = InitFifo( &psPrivate -> sInFifo, nNumFrames, psPrivate -> nBytesPerFrame );

	nErr = Pa_OpenStream( &psPrivate -> psOutStream, NULL, &sOutputParameters, nPortSampleRate, BUFFER_SIZE, paClipOff, &outputCallback, psPrivate );
	if ( nErr == paNoError ) {
		psPrivate -> nHasOut = 1;
		Debug( KERN_DEBUG, "StartStream\n" );
		Pa_StartStream( psPrivate -> psOutStream );
	} else {
		free( psPrivate );
		psPrivate = NULL;
		goto end;
	}

	nErr = Pa_OpenStream( &psPrivate -> psInStream, &sInputParameters, NULL, nPortSampleRate, BUFFER_SIZE, paClipOff, &inputCallback, psPrivate );
	if ( nErr == paNoError ) {
		psPrivate -> nHasIn = 1;
		Debug( KERN_DEBUG, "StartStream\n" );
		Pa_StartStream( psPrivate -> psInStream );
	} else {
		free( psPrivate );
		psPrivate = NULL;
	}

end:
	Debug( KERN_DEBUG, "done\n" );
	return psPrivate;
}

/**
 * \brief Write audio data
 * \param pPriv private data
 * \param pnData audio data
 * \param nSize size of audio data
 * \return bytes written or error code
 */
static int portAudioWrite( void *pPriv, unsigned char *pnData, unsigned int nSize ) {
	struct sPortPrivate *psPrivate = pPriv;
	//PaError nErr;
	int nWritten = 0;
	int nOffset = 0;

	if ( psPrivate == NULL ) {
		return 0;
	}

again:
	nWritten = PaUtil_WriteRingBuffer( &psPrivate -> sOutFifo, pnData + nOffset, nSize - nOffset );
	if ( nWritten != nSize - nOffset ) {
		int nBytesEmpty = 0;
		nBytesEmpty = PaUtil_GetRingBufferWriteAvailable( &psPrivate -> sOutFifo );

		while ( nBytesEmpty < nSize - nOffset ) {
			Pa_Sleep( 250 );
			nBytesEmpty = PaUtil_GetRingBufferWriteAvailable( &psPrivate -> sOutFifo );
		}
		nOffset = nWritten;
		goto again;
	}

	return nWritten;
}

static int portAudioRead( void *pPriv, unsigned char *pnData, unsigned int nSize ) {
	struct sPortPrivate *psPrivate = pPriv;
	long nTotalBytes = 0;
	long nAvail;
	int nMax = 5000;
	long nRead;
	char *pnPtr = ( char * ) pnData;

	while ( nTotalBytes < nSize && --nMax > 0 ) {
		nAvail = PaUtil_GetRingBufferReadAvailable( &psPrivate -> sInFifo );

		nRead = 0;

		if ( nTotalBytes < nSize && nAvail >= nSize ) {
			nRead = PaUtil_ReadRingBuffer( &psPrivate -> sInFifo, pnPtr, nSize );
			nTotalBytes += nRead;
			pnPtr += nRead;
		} else {
			Pa_Sleep( 100 );
		}
	}

	return nTotalBytes;
}

/**
 * \brief Stop and remove pipeline
 * \param pPriv private data
 * \return error code
 */
int portAudioClose( void *pPriv, gboolean bForce ) {
	struct sPortPrivate *psPrivate = pPriv;
	int nBytesEmpty;

	if ( psPrivate == NULL ) {
		return 0;
	}

	if ( psPrivate -> nHasOut ) {
		int nByteSize = psPrivate -> sOutFifo.nBufferSize;

		if ( nByteSize > 0 ) {
			nBytesEmpty = PaUtil_GetRingBufferWriteAvailable( &psPrivate -> sOutFifo );

			while ( nBytesEmpty < nByteSize ) {
				Pa_Sleep( 10 );
				nBytesEmpty = PaUtil_GetRingBufferWriteAvailable( &psPrivate -> sOutFifo );
			}
		}

		if ( psPrivate -> psOutStream ) {
			Debug( KERN_DEBUG, "Has stream\n" );
			if ( Pa_IsStreamActive( psPrivate -> psOutStream ) ) {
				Debug( KERN_DEBUG, "Stop stream\n" );
				Pa_StopStream( psPrivate -> psOutStream );
			}

			Debug( KERN_DEBUG, "Close stream\n" );
			Pa_CloseStream( psPrivate -> psOutStream );
			psPrivate -> psOutStream = NULL;
			Debug( KERN_DEBUG, "Done stream\n" );
		}

		TermFifo( &psPrivate -> sOutFifo );
	}

	if ( psPrivate -> nHasIn ) {
		if ( psPrivate -> psInStream ) {
			Debug( KERN_DEBUG, "Has stream\n" );
			if ( Pa_IsStreamActive( psPrivate -> psInStream ) ) {
				Debug( KERN_DEBUG, "Stop stream\n" );
				Pa_StopStream( psPrivate -> psInStream );
			}

			Debug( KERN_DEBUG, "Close stream\n" );
			Pa_CloseStream( psPrivate -> psInStream );
			psPrivate -> psInStream = NULL;
			Debug( KERN_DEBUG, "Done stream\n" );
		}

		TermFifo( &psPrivate -> sInFifo );
	}

	free( psPrivate );

	return 0;
}

int portAudioDeinit( void ) {
	return Pa_Terminate();
}

/** audio definition */
struct sAudio sPortAudio = {
	portAudioInit,
	portAudioOpen,
	portAudioWrite,
	portAudioRead,
	portAudioClose,
	portAudioDeinit,
	NULL
};

MODULE_INIT( PLUGIN_TYPE_AUDIO, _( "PortAudio" ), &sPortAudio, NULL, portAudioDetectDevices );
