/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "connection.h"
#include "tool.h"

#include "service_record.h"

using namespace Form;

Entry ServiceRecord::loadEntry( const QString&, PagePolicy::State state )
{
  if ( state != PagePolicy::Modify )
    return Entry();

  Entry entry = Entry::fromLdapEntry( Tool::kolabSettings( Connection::self() ) );

  return entry;
}

bool ServiceRecord::saveEntry( const Entry &entry, PagePolicy::State state, QString &errorMsg )
{
  if ( state != PagePolicy::Modify )
    return false;

  // distribution lists
  if ( !entry.value( "distributionlists" ).isEmpty() ) {
    QString dn = Tool::dnForMailOrAlias( Connection::self(), entry.value( "distributionlists" ) );
    if ( dn.isEmpty() ) {
      errorMsg = QObject::tr( "You have to enter a valid email address of this system!" );
      return false;
    }

    QLdapEntry kolabSettings = Tool::kolabSettings( Connection::self() );

    QStringList distributionLists;
    distributionLists.append( "hostmaster" );
    distributionLists.append( "postmaster" );
    distributionLists.append( "abuse" );
    distributionLists.append( "virusalert" );
    distributionLists.append( "MAILER-DAEMON" );

    for ( int i = 0; i < distributionLists.count(); ++i ) {
      QLdapEntry entry;
      entry.addValue( "objectClass", "top" );
      entry.addValue( "objectClass", "kolabGroupOfNames" );
      entry.addValue( "cn", distributionLists[ i ] );
      entry.addValue( "mail", QString( "%1@%2" ).arg( distributionLists[ i ], kolabSettings.value( "postfix-mydomain" ) ) );
      entry.addValue( "member", dn );

      QString dn = QString( "cn=%1,%2" ).arg( distributionLists[ i ], Connection::self()->baseDn() );
      if ( !Connection::self()->add( dn, entry ) ) {
        errorMsg += QString( "Could not create distribution list '%1': %2" )
                           .arg( entry.value( "cn" ), Connection::self()->errorString() );
      }
    }
  }

  QLdapEntry ldapEntry;

  /**
   * Services
   */
  ldapEntry.setValue( "cyrus-pop3", entry.value( "cyrus-pop3" ) );
  ldapEntry.setValue( "cyrus-pop3s", entry.value( "cyrus-pop3s" ) );
  ldapEntry.setValue( "cyrus-imap", entry.value( "cyrus-imap" ) );
  ldapEntry.setValue( "cyrus-imaps", entry.value( "cyrus-imaps" ) );
  ldapEntry.setValue( "cyrus-sieve", entry.value( "cyrus-sieve" ) );
  ldapEntry.setValue( "proftpd-ftp", entry.value( "proftpd-ftp" ) );
  ldapEntry.setValue( "apache-http", entry.value( "apache-http" ) );
  ldapEntry.setValue( "postfix-enable-virus-scan", entry.value( "postfix-enable-virus-scan" ) );

  /**
   * Quota
   */
  ldapEntry.setValue( "cyrus-quotawarn", entry.value( "cyrus-quotawarn" ) );


  /**
   * Free/Busy
   */
  ldapEntry.setValue( "kolabFreeBusyPast", entry.value( "kolabFreeBusyPast" ) );
  ldapEntry.setValue( "apache-allow-unauthenticated-fb", entry.value( "apache-allow-unauthenticated-fb" ) );

  /**
   * Networks
   */
  const QStringList networks = entry.values( "postfix-mynetworks" );
  for ( int i = 0; i < networks.count(); ++i )
    ldapEntry.addValue( "postfix-mynetworks", networks[ i ] );

  /**
   * Smtp
   */
  ldapEntry.setValue( "postfix-relayhost", entry.value( "postfix-relayhost" ) );
  ldapEntry.setValue( "postfix-relayport", entry.value( "postfix-relayport" ) );

  /**
   * Mail
   */
  ldapEntry.setValue( "postfix-allow-unauthenticated", entry.value( "postfix-allow-unauthenticated" ) );

  /**
   * Mail filter
   */
  ldapEntry.setValue( "kolabfilter-verify-from-header", entry.value( "kolabfilter-verify-from-header" ) );
  ldapEntry.setValue( "kolabfilter-allow-sender-header", entry.value( "kolabfilter-allow-sender-header" ) );
  ldapEntry.setValue( "kolabfilter-reject-forged-from-header", entry.value( "kolabfilter-reject-forged-from-header" ) );

  /**
   * Hosts
   */
  const QStringList hosts = entry.values( "kolabHost" );
  for ( int i = 0; i < hosts.count(); ++i )
    ldapEntry.addValue( "kolabHost", hosts[ i ] );


  /**
   * Modify the data in ldap.
   */
  QString dn = QString( "k=kolab,%1" ).arg( Connection::self()->baseDn() );

  if ( !Connection::self()->modifyAttributes( dn, ldapEntry ) ) {
    errorMsg = Connection::self()->errorString();
    return false;
  }

  return true;
}
