/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "connection.h"
#include "distributionlist_filtermodel.h"

#include "distributionlist_listingmodel.h"

void DistributionListListingModel::doQuery( const QString &userFilter )
{
  QString filter = QString( "(&(cn=*)%1(objectClass=kolabGroupOfNames))" ).arg( userFilter );

  QLdapResponse response = Connection::self()->search( QString(), QLdap::Sub, filter , QStringList(), QLatin1String( "cn" ) );

  QString blacklist;
  blacklist.append( "admin" );
  blacklist.append( "maintainer" );

  QLdapEntry::List entries = response.entries();
  QLdapEntry::List filteredEntries;
  for ( int i = 0; i < entries.count(); ++i ) {
    if ( !blacklist.contains( entries[ i ].value( "cn" ) ) )
      filteredEntries.append( entries[ i ] );
  }

  mData.setEntries( filteredEntries );

  emit layoutChanged();
}

int DistributionListListingModel::columnCount( const QModelIndex &parent ) const
{
  if ( parent.isValid() )
    return 0;

  return 2;
}

int DistributionListListingModel::rowCount( const QModelIndex &parent ) const
{
  if ( parent.isValid() )
    return 0;

  return mData.entries().count();
}

QVariant DistributionListListingModel::data( const QModelIndex &index, int role ) const
{
  if ( !index.isValid() )
    return QVariant();

  const QLdapEntry entry = mData.entries()[ index.row() ];

  if ( role == Qt::DisplayRole ) {
    switch ( index.column() ) {
      case 0:
        return entry.value( "cn" );
        break;
      case 1:
        {
          const QStringList parts = entry.dn().split( "," );
          if ( parts.contains( "cn=internal" ) )
            return tr( "Internal" );
          else
            return tr( "Public" );
        }
        break;
      default:
        return QVariant();
    }
  } else if ( role == HasModifyActionRole ) {
    return true;
  } else if ( role == HasDeleteActionRole ) {
    if ( entry.values( "kolabDeleteflag" ).isEmpty() )
      return true;
    else
      return false;
  } else if ( role == ModifyActionLabelRole ) {
    if ( entry.values( "kolabDeleteflag" ).isEmpty() )
      return tr( "Modify" );
    else
      return tr( "Distribution list deleted, awaiting cleanup..." );
  } else if ( role == DeleteActionLabelRole ) {
    return tr( "Delete" );
  } else if ( role == ModifyActionIdRole ) {
    if ( entry.values( "kolabDeleteflag" ).isEmpty() )
      return mData.entries()[ index.row() ].dn();
    else
      return QString();
  } else if ( role == DeleteActionIdRole ) {
    return mData.entries()[ index.row() ].dn();
  }

  return QVariant();
}

QVariant DistributionListListingModel::headerData( int section, Qt::Orientation, int role ) const
{
  if ( role != Qt::DisplayRole )
    return QVariant();

  switch ( section ) {
    case 0:
      return tr( "Listname" );
      break;
    case 1:
      return tr( "Visibility" );
      break;
    default:
      return QVariant();
  }
}

FilterModel* DistributionListListingModel::filterModel()
{
  return new DistributionListFilterModel( this );
}
