/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     assemble.c                                                     */
/*                                                                          */
/* description:  fe-space independent assemblation routines of matrices     */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"

static const REAL TOO_SMALL = 1.e-15;

/*--------------------------------------------------------------------------*/
/* information about precomputed integrals of basis functions on the        */
/* standard element;                                                        */
/*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/* second order term:                                                       */
/*--------------------------------------------------------------------------*/

const Q11_PSI_PHI *get_q11_psi_phi(const BAS_FCTS *psi, const BAS_FCTS *phi,
				   const QUAD *quad)
{
  FUNCNAME("get_q11_psi_phi");
  static VOID_LIST_ELEMENT  *first = nil;
  VOID_LIST_ELEMENT         *list_el;
  Q11_PSI_PHI               *q11pp;
  const QUAD_FAST           *q_phi, *q_psi;
  int                       dim;
  int                       i, j, k, l, n, iq, all_entries, n_psi, n_phi;
  REAL                      val, grdi, grdj;
  int                       **n_entries = nil,
                            ***kk = nil,
                            ***ll = nil,
                            *k_vec = nil,
                            *l_vec = nil;
  REAL                      ***values = nil, *val_vec = nil;

  if (!psi && !phi)
    return(nil);

  if (!psi)  psi = phi;
  if (!phi)  phi = psi;

  TEST_EXIT(psi->dim == phi->dim, 
	    "Support dimensions for phi and psi do not match!\n");
  dim = phi->dim;

  if (!quad)  quad = get_quadrature(dim, psi->degree+phi->degree-2);

/*--------------------------------------------------------------------------*/
/*  look for an existing entry in the list                                  */
/*--------------------------------------------------------------------------*/

  for (list_el = first; list_el; list_el = list_el->next)
  {
    q11pp = (Q11_PSI_PHI *)list_el->data;
    if (q11pp->psi == psi && q11pp->phi == phi && q11pp->quad == quad)
      return(q11pp);
  }

/*--------------------------------------------------------------------------*/
/*  create a new one                                                        */
/*--------------------------------------------------------------------------*/

  n_psi = psi->n_bas_fcts;
  q_psi = get_quad_fast(psi, quad, INIT_GRD_PHI);
  n_phi = phi->n_bas_fcts;
  q_phi = get_quad_fast(phi, quad, INIT_GRD_PHI);

  list_el = get_void_list_element();
  list_el->data = q11pp = MEM_ALLOC(1, Q11_PSI_PHI);
  list_el->next = first;
  first = list_el;

  n_entries = MAT_ALLOC(n_psi, n_phi, int); 
  values    = MAT_ALLOC(n_psi, n_phi, REAL*);
  kk        = MAT_ALLOC(n_psi, n_phi, int*);
  ll        = MAT_ALLOC(n_psi, n_phi, int*);

  q11pp->psi       = psi;
  q11pp->phi       = phi;
  q11pp->quad      = quad;
  
  q11pp->n_entries = (const int **) n_entries;
  q11pp->values    = (const REAL ***) values;
  q11pp->k         = (const int ***) kk;
  q11pp->l         = (const int ***) ll;

/*--------------------------------------------------------------------------*/
/*  compute first the number of all non zero entries                        */
/*--------------------------------------------------------------------------*/

  all_entries = 0;
  for (i = 0; i < n_psi; i++)
  {
    for (j = 0; j < n_phi; j++)
    {
      for (k = 0; k < dim+1; k++)
      {
	for (l =  0; l < dim+1; l++)
	{
	  for (val = iq = 0; iq < quad->n_points; iq++)
	  {
	    grdi = q_psi->grd_phi[iq][i][k];
	    grdj = q_phi->grd_phi[iq][j][l];

	    val += quad->w[iq]*grdi*grdj;
	  }
	  if (ABS(val) > TOO_SMALL)
	    all_entries++;
	}
      }
    }
  }

/*--------------------------------------------------------------------------*/
/* now, access memory for all information                                   */
/*--------------------------------------------------------------------------*/

  if(all_entries) {
    val_vec = MEM_ALLOC(all_entries, REAL);
    k_vec   = MEM_ALLOC(2*all_entries, int);
  }
  l_vec   = k_vec+all_entries;

/*--------------------------------------------------------------------------*/
/* and now, fill information                                                */
/*--------------------------------------------------------------------------*/

  for (i = 0; i < n_psi; i++)
  {
    for (j = 0; j < n_phi; j++)
    {
      values[i][j] = val_vec; 
      kk[i][j]     = k_vec;
      ll[i][j]     = l_vec;

      for (n = k = 0; k < dim+1; k++)
      {
	for (l =  0; l < dim+1; l++)
	{
	  for (val = iq = 0; iq < quad->n_points; iq++)
	  {
	    grdi = q_psi->grd_phi[iq][i][k];
	    grdj = q_phi->grd_phi[iq][j][l];

	    val += quad->w[iq]*grdi*grdj;
	  }
	  if (ABS(val) > TOO_SMALL) {
	    all_entries--;
	    DEBUG_TEST_EXIT(all_entries >= 0,
			"now more entries found than counted before\n");
	    n++;
	    *val_vec++ = val;
	    *k_vec++ = k;
	    *l_vec++ = l;
	  }
	}
      }
      n_entries[i][j] = n;
    }
  }

  return(q11pp);
}

/*--------------------------------------------------------------------------*/
/* first order term                                                         */
/*--------------------------------------------------------------------------*/

const Q01_PSI_PHI *get_q01_psi_phi(const BAS_FCTS *psi, const BAS_FCTS *phi,
				   const QUAD *quad)
{
  FUNCNAME("get_q01_psi_phi");
  static VOID_LIST_ELEMENT  *first = nil;
  VOID_LIST_ELEMENT         *list_el;
  Q01_PSI_PHI               *q01pp;
  const QUAD_FAST           *q_phi, *q_psi;
  int                       dim;
  int                       i, j, l, n, iq, all_entries, n_psi, n_phi;
  REAL                      val, psii, grdj;
  int                       **n_entries, ***ll, *l_vec = nil;
  REAL                      ***values, *val_vec = nil;

  if (!psi && !phi)
    return(nil);

  if (!psi)  psi = phi;
  if (!phi)  phi = psi;

  TEST_EXIT(psi->dim == phi->dim, 
	    "Support dimensions for phi and psi do not match!\n");
  dim = phi->dim;

  if (!quad)  quad = get_quadrature(dim, psi->degree+phi->degree-1);

/*--------------------------------------------------------------------------*/
/*  look for an existing entry in the list                                  */
/*--------------------------------------------------------------------------*/

  for (list_el = first; list_el; list_el = list_el->next)
  {
    q01pp = (Q01_PSI_PHI *)list_el->data;
    if (q01pp->psi == psi && q01pp->phi == phi && q01pp->quad == quad)
      return(q01pp);
  }

/*--------------------------------------------------------------------------*/
/*  otherwise, create a new one                                             */
/*--------------------------------------------------------------------------*/

  n_psi = psi->n_bas_fcts;
  q_psi = get_quad_fast(psi, quad, INIT_PHI);
  n_phi = phi->n_bas_fcts;
  q_phi = get_quad_fast(phi, quad, INIT_GRD_PHI);

  list_el = get_void_list_element();
  list_el->data = q01pp = MEM_ALLOC(1, Q01_PSI_PHI);
  list_el->next = first;
  first = list_el;

  n_entries = MAT_ALLOC(n_psi, n_phi, int);
  values    = MAT_ALLOC(n_psi, n_phi, REAL*);
  ll        = MAT_ALLOC(n_psi, n_phi, int*);

  q01pp->psi       = psi;
  q01pp->phi       = phi;
  q01pp->quad      = quad;
  q01pp->n_entries = (const int **) n_entries;
  q01pp->values    = (const REAL ***) values;
  q01pp->l         = (const int ***) ll;

/*--------------------------------------------------------------------------*/
/*  compute first the number of all non zero entries                        */
/*--------------------------------------------------------------------------*/

  all_entries = 0;
  for (i = 0; i < n_psi; i++)
  {
    for (j = 0; j < n_phi; j++)
    {
      for (l = 0; l < dim+1; l++)
      {
	for (val = iq = 0; iq < quad->n_points; iq++)
	{
	  psii = q_psi->phi[iq][i];
	  grdj = q_phi->grd_phi[iq][j][l];

	  val += quad->w[iq]*psii*grdj;
	}
	if (ABS(val) > TOO_SMALL)
	  all_entries++;
      }
    }
  }

/*--------------------------------------------------------------------------*/
/* now, access memory for all information                                   */
/*--------------------------------------------------------------------------*/

  if(all_entries) {
    val_vec = MEM_ALLOC(all_entries, REAL);
    l_vec   = MEM_ALLOC(all_entries, int);
  }
/*--------------------------------------------------------------------------*/
/* and now, fill information                                                */
/*--------------------------------------------------------------------------*/

  for (i = 0; i < n_psi; i++)
  {
    for (j = 0; j < n_phi; j++)
    {
      values[i][j] = val_vec;
      ll[i][j]     = l_vec;

      for (n = l = 0; l < dim+1; l++)
      {
	for (val = iq = 0; iq < quad->n_points; iq++)
	{
	  psii = q_psi->phi[iq][i];
	  grdj = q_phi->grd_phi[iq][j][l];

	  val += quad->w[iq]*psii*grdj;
	}
	if (ABS(val) > TOO_SMALL) {
	  all_entries--;
	  DEBUG_TEST_EXIT(all_entries >= 0,
			  "now more entries found than counted before\n");
	  n++;
	  *val_vec++ = val;
	  *l_vec++   = l;
	}
      }
      n_entries[i][j] = n;
    }
  }

  return(q01pp);
}

const Q10_PSI_PHI *get_q10_psi_phi(const BAS_FCTS *psi, const BAS_FCTS *phi,
				   const QUAD *quad)
{
  FUNCNAME("get_q10_psi_phi");
  static VOID_LIST_ELEMENT  *first = nil;
  VOID_LIST_ELEMENT         *list_el;
  Q10_PSI_PHI               *q10pp;
  const QUAD_FAST           *q_phi, *q_psi;
  int                       dim;
  int                       i, j, k, n, iq, all_entries, n_psi, n_phi;
  REAL                      val, phij, grdi;
  int                       **n_entries, ***kk, *k_vec = nil;
  REAL                      ***values, *val_vec = nil;

  if (!psi && !phi)
    return(nil);

  if (!psi)  psi = phi;
  if (!phi)  phi = psi;

  TEST_EXIT(psi->dim == phi->dim,
	    "Support dimensions for phi and psi do not match!\n");
  dim = phi->dim;

  if (!quad)  quad = get_quadrature(dim, psi->degree+phi->degree-1);

/*--------------------------------------------------------------------------*/
/*  look for an existing entry in the list                                  */
/*--------------------------------------------------------------------------*/

  for (list_el = first; list_el; list_el = list_el->next)
  {
    q10pp = (Q10_PSI_PHI *)list_el->data;
    if (q10pp->psi == psi && q10pp->phi == phi && q10pp->quad == quad)
      return(q10pp);
  }

/*--------------------------------------------------------------------------*/
/*  otherwise, create a new one                                             */
/*--------------------------------------------------------------------------*/

  n_psi = psi->n_bas_fcts;
  q_psi = get_quad_fast(psi, quad, INIT_GRD_PHI);
  n_phi = phi->n_bas_fcts;
  q_phi = get_quad_fast(phi, quad, INIT_PHI);

  list_el = get_void_list_element();
  list_el->data = q10pp = MEM_ALLOC(1, Q10_PSI_PHI);
  list_el->next = first;
  first = list_el;

  n_entries = MAT_ALLOC(n_psi, n_phi, int);
  values    = MAT_ALLOC(n_psi, n_phi, REAL*);
  kk        = MAT_ALLOC(n_psi, n_phi, int*);

  q10pp->psi       = psi;
  q10pp->phi       = phi;
  q10pp->quad      = quad;
  q10pp->n_entries = (const int **) n_entries;
  q10pp->values    = (const REAL ***) values;
  q10pp->k         = (const int ***) kk;

/*--------------------------------------------------------------------------*/
/*  compute first the number of all non zero entries                        */
/*--------------------------------------------------------------------------*/

  all_entries = 0;
  for (i = 0; i < n_psi; i++)
  {
    for (j = 0; j < n_phi; j++)
    {
      for (k = 0; k < dim+1; k++)
      {
	for (val = iq = 0; iq < quad->n_points; iq++)
	{
	  grdi = q_psi->grd_phi[iq][i][k];
	  phij = q_phi->phi[iq][j];

	  val += quad->w[iq]*grdi*phij;
	}
	if (ABS(val) > TOO_SMALL)
	  all_entries++;
      }
    }
  }

/*--------------------------------------------------------------------------*/
/* now, access memory for all information                                   */
/*--------------------------------------------------------------------------*/

  if (all_entries) {
    val_vec = MEM_ALLOC(all_entries, REAL);
    k_vec   = MEM_ALLOC(all_entries, int);
  }

/*--------------------------------------------------------------------------*/
/* and now, fill information                                                */
/*--------------------------------------------------------------------------*/

  for (i = 0; i < n_psi; i++)
  {
    for (j = 0; j < n_phi; j++)
    {
      values[i][j] = val_vec;
      kk[i][j]     = k_vec;

      for (n = k = 0; k < dim+1; k++)
      {
	for (val = iq = 0; iq < quad->n_points; iq++)
	{
	  grdi = q_psi->grd_phi[iq][i][k];
	  phij = q_phi->phi[iq][j];

	  val += quad->w[iq]*grdi*phij;
	}
	if (ABS(val) > TOO_SMALL)
	{
	  all_entries--;
	  DEBUG_TEST_EXIT(all_entries >= 0,
			  "now more entries found than counted before\n");
	  n++;
	  *val_vec++ = val;
	  *k_vec++   = k;
	}
      }
      n_entries[i][j] = n;
    }
  }

  return(q10pp);
}

/*--------------------------------------------------------------------------*/
/*  first order term:                                                       */
/*--------------------------------------------------------------------------*/

const Q00_PSI_PHI *get_q00_psi_phi(const BAS_FCTS *psi, const BAS_FCTS *phi,
				   const QUAD *quad)
{
  FUNCNAME("get_q00_psi_phi");
  static VOID_LIST_ELEMENT  *first = nil;
  VOID_LIST_ELEMENT         *list_el;
  Q00_PSI_PHI               *q00pp;
  const QUAD_FAST           *q_phi, *q_psi;
  int                       dim;
  int                       i, j, iq, n_psi, n_phi;
  REAL                      **values, val;

  if (!psi && !phi)
    return(nil);

  if (!psi)  psi = phi;
  if (!phi)  phi = psi;

  TEST_EXIT(psi->dim == phi->dim,
	    "Support dimensions for phi and psi do not match!\n");
  dim = phi->dim;

  if (!quad)  quad = get_quadrature(dim, psi->degree+phi->degree);

/*--------------------------------------------------------------------------*/
/*  look for an existing entry in the list                                  */
/*--------------------------------------------------------------------------*/

  for (list_el = first; list_el; list_el = list_el->next)
  {
    q00pp = (Q00_PSI_PHI *)list_el->data;
    if (q00pp->psi == psi && q00pp->phi == phi && q00pp->quad == quad)
      return(q00pp);
  }

/*--------------------------------------------------------------------------*/
/*  create a new one                                                        */
/*--------------------------------------------------------------------------*/

  n_psi = psi->n_bas_fcts;
  q_psi = get_quad_fast(psi, quad, INIT_PHI);
  n_phi = phi->n_bas_fcts;
  q_phi = get_quad_fast(phi, quad, INIT_PHI);

  list_el = get_void_list_element();
  list_el->data = q00pp = MEM_ALLOC(1, Q00_PSI_PHI);
  list_el->next = first;
  first = list_el;

  values = MAT_ALLOC(n_psi, n_phi, REAL);

  q00pp->psi    = psi;
  q00pp->phi    = phi;
  q00pp->quad   = quad;
  q00pp->values = (const REAL **) values;

  for (i = 0; i < n_psi; i++)
  {
    for (j = 0; j < n_phi; j++)
    {
      for (val = iq = 0; iq < quad->n_points; iq++)
	val += quad->w[iq]*q_psi->phi[iq][i]*q_phi->phi[iq][j];

      if (ABS(val) < TOO_SMALL)
	values[i][j] = 0.0;
      else
	values[i][j] = val;
    }
  }

  return(q00pp);
}

typedef struct fill_info  FILL_INFO;
struct fill_info
{
  const FE_SPACE     *psi_fe;
  const FE_SPACE     *phi_fe;
  const QUAD         *quad[3];

  const Q11_PSI_PHI  *q11_psi_phi;
  const Q01_PSI_PHI  *q01_psi_phi;
  const Q10_PSI_PHI  *q10_psi_phi;
  const Q00_PSI_PHI  *q00_psi_phi;

  const QUAD_FAST    *psi_quad_fast[3];
  const QUAD_FAST    *phi_quad_fast[3];

  int        n_row, n_col;
  REAL       **element_matrix;

  PARAMETRIC *parametric;

  int        (*init_element)(const EL_INFO *, const QUAD *[3], void *);

  void       (*fast_second_order)(const EL_INFO *, const FILL_INFO *);
  void       (*fast_first_order)(const EL_INFO *, const FILL_INFO *);
  void       (*fast_zero_order)(const EL_INFO *, const FILL_INFO *);

  void       (*slow_second_order)(const EL_INFO *, const FILL_INFO *);
  void       (*slow_first_order)(const EL_INFO *, const FILL_INFO *);
  void       (*slow_zero_order)(const EL_INFO *, const FILL_INFO *);

  const REAL (*(*LALt)(const EL_INFO *, const QUAD *, int, void *))[N_LAMBDA];
  int        LALt_symmetric;
  const REAL *(*Lb0)(const EL_INFO *, const QUAD *, int, void *);
  const REAL *(*Lb1)(const EL_INFO *, const QUAD *, int, void *);
  int        Lb0_Lb1_anti_symmetric;
  REAL       (*c)(const EL_INFO *, const QUAD *, int, void *);
  int        c_symmetric;

  void       *user_data;

  FILL_INFO  *next;
};

/*--------------------------------------------------------------------------*/
/*  functions for calculating element stiffness matrices                    */
/*--------------------------------------------------------------------------*/

static void pre_2(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL   (*LALt)[N_LAMBDA];
  const int    **n_entries, *k, *l;
  const REAL   *values;
  int          i, j, m;
  REAL         **mat = info->element_matrix, val;

  LALt = info->LALt(el_info, info->quad[2], 0, info->user_data);
  n_entries = info->q11_psi_phi->n_entries;

  if (info->LALt_symmetric)
  {
    for (i = 0; i < info->n_row; i++)
    {
      k      = info->q11_psi_phi->k[i][i];
      l      = info->q11_psi_phi->l[i][i];
      values = info->q11_psi_phi->values[i][i];
      for (val = m = 0; m < n_entries[i][i]; m++)
	val += values[m]*LALt[k[m]][l[m]];
      mat[i][i] += val;
      for (j = i+1; j < info->n_col; j++)
      {
	k      = info->q11_psi_phi->k[i][j];
	l      = info->q11_psi_phi->l[i][j];
	values = info->q11_psi_phi->values[i][j];
	for (val = m = 0; m < n_entries[i][j]; m++)
	  val += values[m]*LALt[k[m]][l[m]];
	mat[i][j] += val;
	mat[j][i] += val;
      }
    }
  }
  else  /*  A not symmetric or psi != phi        */
  {
    for (i = 0; i < info->n_row; i++)
    {
      for (j = 0; j < info->n_col; j++)
      {
	k      = info->q11_psi_phi->k[i][j];
	l      = info->q11_psi_phi->l[i][j];
	values = info->q11_psi_phi->values[i][j];
	for (val = m = 0; m < n_entries[i][j]; m++)
	  val += values[m]*LALt[k[m]][l[m]];
	mat[i][j] += val;
      }
    }
  }
  return;
}

static void pre_01(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL   *Lb0;
  const int    **n_entries, *l;
  const REAL   *values;
  REAL         val;
  int          i, j, m;
  REAL         **mat = info->element_matrix;

  Lb0 = info->Lb0(el_info, info->quad[1], 0, info->user_data);
  n_entries = info->q01_psi_phi->n_entries;

  for (i = 0; i < info->n_row; i++)
  {
    for (j = 0; j < info->n_col; j++)
    {
      l      = info->q01_psi_phi->l[i][j];
      values = info->q01_psi_phi->values[i][j];
      for (val = m = 0; m < n_entries[i][j]; m++)
	val += values[m]*Lb0[l[m]];
      mat[i][j] += val;
    }
  }
  return;
}

static void pre_10(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL   *Lb1;
  const int    **n_entries, *k;
  const REAL   *values;
  int          i, j, m;
  REAL         **mat = info->element_matrix, val;

  Lb1 = info->Lb1(el_info, info->quad[1], -1, info->user_data);
  n_entries = info->q10_psi_phi->n_entries;

  for (i = 0; i < info->n_row; i++)
  {
    for (j = 0; j < info->n_col; j++)
    {
      k      = info->q10_psi_phi->k[i][j];
      values = info->q10_psi_phi->values[i][j];
      for (val = m = 0; m < n_entries[i][j]; m++)
	val += values[m]*Lb1[k[m]];
      mat[i][j] += val;
    }
  }
  return;
}

static void pre_11(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL   *Lb0, *Lb1;
  const int    **n_entries01, **n_entries10, *k, *l;
  const REAL   *values;
  int          i, j, m;
  REAL         **mat = info->element_matrix, val;

  Lb0 = info->Lb0(el_info, info->quad[1], 0, info->user_data);
  Lb1 = info->Lb1(el_info, info->quad[1], 0, info->user_data);
  n_entries01 = info->q01_psi_phi->n_entries;
  n_entries10 = info->q10_psi_phi->n_entries;

  for (i = 0; i < info->n_row; i++)
  {
    for (j = 0; j < info->n_col; j++)
    {
      l      = info->q01_psi_phi->l[i][j];
      values = info->q01_psi_phi->values[i][j];
      for (val = m = 0; m < n_entries01[i][j]; m++)
	val += values[m]*Lb0[l[m]];

      k      = info->q10_psi_phi->k[i][j];
      values = info->q10_psi_phi->values[i][j];
      for (m = 0; m < n_entries10[i][j]; m++)
	val += values[m]*Lb1[k[m]];
      mat[i][j] += val;
    }
  }
  return;
}

static void pre_0(const EL_INFO *el_info, const FILL_INFO *info)
{
  int        i, j;
  REAL       val, c;
  const REAL **values;
  REAL       **mat = info->element_matrix;
  
  c = info->c(el_info, info->quad[0], 0, info->user_data);
  values = info->q00_psi_phi->values;

  if (info->c_symmetric)
  {
    for (i = 0; i < info->n_row; i++)
    {
      mat[i][i] += c*values[i][i];
      for (j = i+1; j < info->n_col; j++)
      {
	val = c*values[i][j];
	mat[i][j] += val;
	mat[j][i] += val;
      }
    }
  }
  else
  {
    for (i = 0; i < info->n_row; i++)
      for (j = 0; j < info->n_col; j++)
	mat[i][j] += c*values[i][j];
  }
  return;
}

static inline REAL btv(const int n_lambda, const REAL *b, const REAL *v)
{
  int i;
  REAL res = 0.0;

  for(i = 0; i < n_lambda; i++) res += b[i]*v[i];

  return res;
}

static inline REAL utAv(const int n_lambda, const REAL *u, 
			const REAL (*A)[N_LAMBDA], const REAL *v)
{
  int i,j;
  REAL res = 0.0;

  for(i = 0; i < n_lambda; i++)
    for(j = 0; j < n_lambda; j++)
      res += u[i]*A[i][j]*v[j];

  return res;
}

static void quad_2(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL       (*LALt)[N_LAMBDA];
  REAL             val, (*grd_psi)[N_LAMBDA], (*grd_phi)[N_LAMBDA];
  int              iq, i, j, n_lambda = el_info->mesh->dim + 1;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  REAL             **mat = info->element_matrix;

  const REAL (*(*LALt_fct)(const EL_INFO *, const QUAD *, int, void *))[N_LAMBDA];

  LALt_fct = info->LALt;
  quad     = info->quad[2];
  psi_fast = info->psi_quad_fast[2];
  phi_fast = info->phi_quad_fast[2];

  if (info->LALt_symmetric)
  {
    for (iq = 0; iq < quad->n_points; iq++)
    {
      LALt = (*LALt_fct)(el_info, quad, iq, info->user_data);
      grd_psi = psi_fast->grd_phi[iq];
      grd_phi = phi_fast->grd_phi[iq];

      for (i = 0; i < info->n_row; i++)
      {
	mat[i][i] += quad->w[iq]*utAv(n_lambda, grd_psi[i], LALt, grd_phi[i]);

	for (j = i+1; j < info->n_col; j++)
	{
	  val = quad->w[iq]*utAv(n_lambda, grd_psi[i], LALt, grd_phi[j]);
	  mat[i][j] += val;
	  mat[j][i] += val;
	}
      }
    }
  }
  else      /*  non symmetric assembling   */
  {
    for (iq = 0; iq < quad->n_points; iq++)
    {
      LALt = (*LALt_fct)(el_info, quad, iq, info->user_data);
      grd_psi = psi_fast->grd_phi[iq];
      grd_phi = phi_fast->grd_phi[iq];

      for (i = 0; i < info->n_row; i++)
      {
	for (j = 0; j < info->n_col; j++)
	{
	  mat[i][j] += quad->w[iq]*utAv(n_lambda,grd_psi[i], LALt, grd_phi[j]);
	}
      }
    }
  }
  return;
}

static void quad_01(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL       *Lb0;
  REAL             *psi, (*grd_phi)[N_LAMBDA];
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  int              n_lambda = el_info->mesh->dim + 1;
  REAL             **mat = info->element_matrix;

  const REAL *(*Lb0_fct)(const EL_INFO *, const QUAD *, int, void *);

  Lb0_fct  = info->Lb0;
  quad     = info->quad[1];
  psi_fast = info->psi_quad_fast[1];
  phi_fast = info->phi_quad_fast[1];

  for (iq = 0; iq < quad->n_points; iq++)
  {
    Lb0 = (*Lb0_fct)(el_info, quad, iq, info->user_data);

    grd_phi = phi_fast->grd_phi[iq];
    psi     = psi_fast->phi[iq];

    for (i = 0; i < info->n_row; i++)
    {
      for (j = 0; j < info->n_col; j++)
	mat[i][j] += quad->w[iq]*psi[i]*btv(n_lambda,Lb0,grd_phi[j]);
    }
  }
  return;
}

static void quad_10(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL       *Lb1;
  REAL             (*grd_psi)[N_LAMBDA], *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  REAL             **mat = info->element_matrix;
  int              n_lambda = el_info->mesh->dim + 1;

  const REAL *(*Lb1_fct)(const EL_INFO *, const QUAD *, int, void *);

  Lb1_fct  = info->Lb1;
  quad     = info->quad[1];
  psi_fast = info->psi_quad_fast[1];
  phi_fast = info->phi_quad_fast[1];

  for (iq = 0; iq < quad->n_points; iq++)
  {
    Lb1 = (*Lb1_fct)(el_info, quad, iq, info->user_data);

    phi     = phi_fast->phi[iq];
    grd_psi = psi_fast->grd_phi[iq];

    for (i = 0; i < info->n_row; i++)
    {
      for (j = 0; j < info->n_col; j++)
	mat[i][j] += quad->w[iq]*btv(n_lambda,Lb1,grd_psi[i])*phi[j];
    }
  }
  return;
}

static void quad_11(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL       *Lb0, *Lb1;
  REAL             *psi, (*grd_psi)[N_LAMBDA], (*grd_phi)[N_LAMBDA], *phi, val;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  REAL             **mat = info->element_matrix;
  int              n_lambda = el_info->mesh->dim + 1;

  const REAL *(*Lb0_fct)(const EL_INFO *, const QUAD *, int, void *);
  const REAL *(*Lb1_fct)(const EL_INFO *, const QUAD *, int, void *);

  Lb0_fct  = info->Lb0;
  Lb1_fct  = info->Lb1;
  quad     = info->quad[1];
  psi_fast = info->psi_quad_fast[1];
  phi_fast = info->phi_quad_fast[1];

  if (info->Lb0_Lb1_anti_symmetric)
  {
    for (iq = 0; iq < quad->n_points; iq++)
    {
      Lb0 = (*Lb0_fct)(el_info, quad, iq, info->user_data);
      Lb1 = (*Lb1_fct)(el_info, quad, iq, info->user_data);

      grd_phi = phi_fast->grd_phi[iq];
      phi     = phi_fast->phi[iq];
      grd_psi = psi_fast->grd_phi[iq];
      psi     = psi_fast->phi[iq];

      for (i = 0; i < info->n_row; i++)
      {
	for (j = i+1; j < info->n_col; j++)
	{
	  val = psi[i]*btv(n_lambda, Lb0,grd_phi[j]) +
	    btv(n_lambda, Lb1,grd_psi[i])*phi[j];
	  val *= quad->w[iq];
	  mat[i][j] += val;
	  mat[j][i] -= val;
	}
      }
    }
  }
  else
  {
    for (iq = 0; iq < quad->n_points; iq++)
    {
      Lb0 = (*Lb0_fct)(el_info, quad, iq, info->user_data);
      Lb1 = (*Lb1_fct)(el_info, quad, iq, info->user_data);

      grd_phi = phi_fast->grd_phi[iq];
      phi     = phi_fast->phi[iq];
      grd_psi = psi_fast->grd_phi[iq];
      psi     = psi_fast->phi[iq];

      for (i = 0; i < info->n_row; i++)
      {
	for (j = 0; j < info->n_col; j++)
	{
	  val = psi[i]*btv(n_lambda, Lb0,grd_phi[j]) +
	    btv(n_lambda, Lb1,grd_psi[i])*phi[j];
	  mat[i][j] += quad->w[iq]*val;
	}
      }
    }
  }

  return;
}

static void quad_0(const EL_INFO *el_info, const FILL_INFO *info)
{
  REAL             c, val, *psi, *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  REAL             **mat = info->element_matrix;

  REAL       (*c_fct)(const EL_INFO *, const QUAD *, int, void *);

  c_fct         = info->c;
  quad          = info->quad[0];
  psi_fast = info->psi_quad_fast[0];
  phi_fast = info->phi_quad_fast[0];

  if (info->c_symmetric)
  {
    for (iq = 0; iq < quad->n_points; iq++)
    {
      c   = (*c_fct)(el_info, quad, iq, info->user_data);
      psi = psi_fast->phi[iq];
      phi = phi_fast->phi[iq];
      
      for (i = 0; i < info->n_row; i++)
      {
	mat[i][i] += quad->w[iq]*c*psi[i]*phi[i];
	for (j = i+1; j < info->n_col; j++)
	{
	  val = quad->w[iq]*c*psi[i]*phi[j];
	  mat[i][j] += val;
	  mat[j][i] += val;
	}
      }
    }
  }
  else      /*  non symmetric assembling   */
  {
    for (iq = 0; iq < quad->n_points; iq++)
    {
      c   = (*c_fct)(el_info, quad, iq, info->user_data);
      psi = psi_fast->phi[iq];
      phi = phi_fast->phi[iq];
      
      for (i = 0; i < info->n_row; i++)
      {
	for (j = 0; j < info->n_col; j++)
	  mat[i][j] += quad->w[iq]*c*psi[i]*phi[j];
      }
    }
  }
  return;
}

static void quad_2_01(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL       (*LALt)[N_LAMBDA], *Lb0;
  REAL             (*grd_psi)[N_LAMBDA], *psi, (*grd_phi)[N_LAMBDA];
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  REAL             **mat = info->element_matrix;
  int              n_lambda = el_info->mesh->dim + 1;

  const REAL (*(*LALt_fct)(const EL_INFO *, const QUAD *, int, void *))[N_LAMBDA];
  const REAL *(*Lb0_fct)(const EL_INFO *, const QUAD *, int, void *);

  LALt_fct = info->LALt;
  Lb0_fct  = info->Lb0;
  quad     = info->quad[2];
  psi_fast = info->psi_quad_fast[2];
  phi_fast = info->phi_quad_fast[2];

  for (iq = 0; iq < quad->n_points; iq++)
  {
    LALt = LALt_fct(el_info, quad, iq, info->user_data);
    Lb0  = Lb0_fct(el_info, quad, iq, info->user_data);

    grd_psi = psi_fast->grd_phi[iq];
    grd_phi = phi_fast->grd_phi[iq];

    psi     = psi_fast->phi[iq];

    for (i = 0; i < info->n_row; i++)
    {
      for (j = 0; j < info->n_col; j++)
      {
	mat[i][j] += quad->w[iq]*(utAv(n_lambda, grd_psi[i], LALt, grd_phi[j]) 
				  + psi[i]*btv(n_lambda, Lb0, grd_phi[j]));
      }
    }
  }
  return;
}  

static void quad_2_10(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL       (*LALt)[N_LAMBDA], *Lb1;
  REAL             (*grd_psi)[N_LAMBDA], (*grd_phi)[N_LAMBDA], *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  REAL             **mat = info->element_matrix;
  int              n_lambda = el_info->mesh->dim + 1;

  const REAL (*(*LALt_fct)(const EL_INFO *, const QUAD *, int, void *))[N_LAMBDA];
  const REAL *(*Lb1_fct)(const EL_INFO *, const QUAD *, int, void *);

  LALt_fct = info->LALt;
  Lb1_fct  = info->Lb1;
  quad     = info->quad[2];
  psi_fast = info->psi_quad_fast[2];
  phi_fast = info->phi_quad_fast[2];

  for (iq = 0; iq < quad->n_points; iq++)
  {
    LALt = LALt_fct(el_info, quad, iq, info->user_data);
    Lb1  = Lb1_fct(el_info, quad, iq, info->user_data);

    grd_psi = psi_fast->grd_phi[iq];
    grd_phi = phi_fast->grd_phi[iq];

    phi     = phi_fast->phi[iq];

    for (i = 0; i < info->n_row; i++)
    {
      for (j = 0; j < info->n_col; j++)
      {
	mat[i][j] += quad->w[iq]*(utAv(n_lambda, grd_psi[i], LALt, grd_phi[j]) 
				  + btv(n_lambda, Lb1, grd_psi[i])*phi[j]);
      }
    }
  }
  return;
}

static void quad_2_11(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL       (*LALt)[N_LAMBDA], *Lb0, *Lb1;
  REAL             (*grd_psi)[N_LAMBDA], *psi, (*grd_phi)[N_LAMBDA], *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  REAL             **mat = info->element_matrix, val2, val1;
  int              n_lambda = el_info->mesh->dim + 1;

  const REAL (*(*LALt_fct)(const EL_INFO *, const QUAD *, int, void *))[N_LAMBDA];
  const REAL *(*Lb0_fct)(const EL_INFO *, const QUAD *, int, void *);
  const REAL *(*Lb1_fct)(const EL_INFO *, const QUAD *, int, void *);

  LALt_fct = info->LALt;
  Lb0_fct  = info->Lb0;
  Lb1_fct  = info->Lb1;
  quad     = info->quad[2];
  psi_fast = info->psi_quad_fast[2];
  phi_fast = info->phi_quad_fast[2];

  if (info->LALt_symmetric  &&  info->Lb0_Lb1_anti_symmetric)
  {
    for (iq = 0; iq < quad->n_points; iq++)
    {
      LALt = LALt_fct(el_info, quad, iq, info->user_data);
      Lb0  = Lb0_fct(el_info, quad, iq, info->user_data);
      Lb1  = Lb1_fct(el_info, quad, iq, info->user_data);

      grd_phi = phi_fast->grd_phi[iq];
      phi     = phi_fast->phi[iq];
      
      grd_psi = psi_fast->grd_phi[iq];
      psi     = psi_fast->phi[iq];

      for (i = 0; i < info->n_row; i++)
      {
	mat[i][i] += quad->w[iq]*utAv(n_lambda, grd_psi[i], LALt, grd_phi[i]);
	for (j = i+1; j < info->n_col; j++)
	{
	  val2 = quad->w[iq] * utAv(n_lambda, grd_psi[i], LALt, grd_phi[j]);
	  val1 = psi[i]*btv(n_lambda, Lb0,grd_phi[j]) +
	    btv(n_lambda, Lb1,grd_psi[i])*phi[j];
	  val1 *= quad->w[iq];
	  mat[i][j] += val2 + val1;
	  mat[j][i] += val2 - val1;
	}
      }
    }
  }
  else
  {
    for (iq = 0; iq < quad->n_points; iq++)
    {
      LALt = LALt_fct(el_info, quad, iq, info->user_data);
      Lb0  = Lb0_fct(el_info, quad, iq, info->user_data);
      Lb1  = Lb1_fct(el_info, quad, iq, info->user_data);

      grd_phi = phi_fast->grd_phi[iq];
      phi     = phi_fast->phi[iq];
      
      grd_psi = psi_fast->grd_phi[iq];
      psi     = psi_fast->phi[iq];

      for (i = 0; i < info->n_row; i++)
      {
	for (j = 0; j < info->n_col; j++)
	{
	  val2 = utAv(n_lambda, grd_psi[i], LALt, grd_phi[j]);
	  val1 = psi[i]*btv(n_lambda, Lb0,grd_phi[j]) +
	    btv(n_lambda, Lb1,grd_psi[i])*phi[j];
	  mat[i][j] += quad->w[iq]*(val2 + val1);
	}
      }
    }
  }
  return;
}

static void quad_2_0(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL       (*LALt)[N_LAMBDA];
  REAL             c, val, (*grd_psi)[N_LAMBDA];
  REAL             *psi, (*grd_phi)[N_LAMBDA], *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  REAL             **mat = info->element_matrix;
  int              n_lambda = el_info->mesh->dim + 1;

  const REAL (*(*LALt_fct)(const EL_INFO *, const QUAD *, int, void *))[N_LAMBDA];
  REAL       (*c_fct)(const EL_INFO *, const QUAD *, int, void *);

  LALt_fct      = info->LALt;
  c_fct         = info->c;
  quad          = info->quad[2];
  psi_fast = info->psi_quad_fast[2];
  phi_fast = info->phi_quad_fast[2];

  if (info->LALt_symmetric)  /* => psi == phi => c_symmetric  */
  {
    for (iq = 0; iq < quad->n_points; iq++)
    {
      LALt = (*LALt_fct)(el_info, quad, iq, info->user_data);
      c    = (*c_fct)(el_info, quad, iq, info->user_data);

      grd_psi = psi_fast->grd_phi[iq];
      grd_phi = phi_fast->grd_phi[iq];
      
      psi     = psi_fast->phi[iq];
      phi     = phi_fast->phi[iq];
      
      for (i = 0; i < info->n_row; i++)
      {
	val = utAv(n_lambda, grd_psi[i], LALt, grd_phi[i]) + c*psi[i]*phi[i];
	mat[i][i] += quad->w[iq]*val; 
	for (j = i+1; j < info->n_col; j++)
	{
	  val = utAv(n_lambda, grd_psi[i], LALt, grd_phi[j]) + c*psi[i]*phi[j];
	  val *= quad->w[iq];
	  mat[i][j] += val;
	  mat[j][i] += val;
	}
      }
    }
  }
  else      /*  non symmetric assembling   */
  {
    for (iq = 0; iq < quad->n_points; iq++)
    {
      LALt = (*LALt_fct)(el_info, quad, iq, info->user_data);
      c    = (*c_fct)(el_info, quad, iq, info->user_data);

      grd_psi = psi_fast->grd_phi[iq];
      grd_phi = phi_fast->grd_phi[iq];

      psi     = psi_fast->phi[iq];
      phi     = phi_fast->phi[iq];
      
      for (i = 0; i < info->n_row; i++)
      {
	for (j = 0; j < info->n_col; j++)
	{
	  mat[i][j] += quad->w[iq]*(utAv(n_lambda,grd_psi[i], LALt, grd_phi[j])
				    + c*psi[i]*phi[j]);
	}
      }
    }
  }
  return;
}

static void quad_01_0(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL       *Lb0;
  REAL             c, *psi, (*grd_phi)[N_LAMBDA], *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  REAL             **mat = info->element_matrix;
  int              n_lambda = el_info->mesh->dim + 1;

  const REAL *(*Lb0_fct)(const EL_INFO *, const QUAD *, int, void *);
  REAL       (*c_fct)(const EL_INFO *, const QUAD *, int, void *);

  Lb0_fct  = info->Lb0;
  c_fct    = info->c;
  quad     = info->quad[1];
  psi_fast = info->psi_quad_fast[1];
  phi_fast = info->phi_quad_fast[1];

  for (iq = 0; iq < quad->n_points; iq++)
  {
    Lb0  = (*Lb0_fct)(el_info, quad, iq, info->user_data);
    c    = (*c_fct)(el_info, quad, iq, info->user_data);

    grd_phi = phi_fast->grd_phi[iq];
    psi     = psi_fast->phi[iq];
    phi     = phi_fast->phi[iq];

    for (i = 0; i < info->n_row; i++)
    {
      for (j = 0; j < info->n_col; j++)
	mat[i][j] += quad->w[iq]*psi[i]*(btv(n_lambda,Lb0,grd_phi[j]) +
					 c*phi[j]);
    }
  }
  return;
}

static void quad_10_0(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL       *Lb1;
  REAL             c, (*grd_psi)[N_LAMBDA], *psi, *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  REAL             **mat = info->element_matrix;
  int              n_lambda = el_info->mesh->dim + 1;

  const REAL *(*Lb1_fct)(const EL_INFO *, const QUAD *, int, void *);
  REAL       (*c_fct)(const EL_INFO *, const QUAD *, int, void *);

  Lb1_fct  = info->Lb1;
  c_fct    = info->c;
  quad     = info->quad[1];
  psi_fast = info->psi_quad_fast[1];
  phi_fast = info->phi_quad_fast[1];

  for (iq = 0; iq < quad->n_points; iq++)
  {
    Lb1  = (*Lb1_fct)(el_info, quad, iq, info->user_data);
    c    = (*c_fct)(el_info, quad, iq, info->user_data);

    grd_psi = psi_fast->grd_phi[iq];
    psi     = psi_fast->phi[iq];
    phi     = phi_fast->phi[iq];

    for (i = 0; i < info->n_row; i++)
    {
      for (j = 0; j < info->n_col; j++)
	mat[i][j] += quad->w[iq]*(btv(n_lambda,Lb1,grd_psi[i]) +
				  psi[i]*c)*phi[j];
    }
  }
  return;
}

static void quad_11_0(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL       *Lb0, *Lb1;
  REAL             c, (*grd_psi)[N_LAMBDA], *psi, (*grd_phi)[N_LAMBDA], *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  REAL             **mat = info->element_matrix, val0, val1;
  int              n_lambda = el_info->mesh->dim + 1;

  const REAL *(*Lb0_fct)(const EL_INFO *, const QUAD *, int, void *);
  const REAL *(*Lb1_fct)(const EL_INFO *, const QUAD *, int, void *);
  REAL       (*c_fct)(const EL_INFO *, const QUAD *, int, void *);

  Lb0_fct  = info->Lb0;
  Lb1_fct  = info->Lb1;
  c_fct    = info->c;
  quad     = info->quad[1];
  psi_fast = info->psi_quad_fast[1];
  phi_fast = info->phi_quad_fast[1];

  if (info->Lb0_Lb1_anti_symmetric) /* => psi == phi => c symmetric  */
  {
    for (iq = 0; iq < quad->n_points; iq++)
    {
      Lb0  = (*Lb0_fct)(el_info, quad, iq, info->user_data);
      Lb1  = (*Lb1_fct)(el_info, quad, iq, info->user_data);
      c    = (*c_fct)(el_info, quad, iq, info->user_data);

      grd_psi = psi_fast->grd_phi[iq];
      psi     = psi_fast->phi[iq];
      grd_phi = phi_fast->grd_phi[iq];
      phi     = phi_fast->phi[iq];

      for (i = 0; i < info->n_row; i++)
      {
	mat[i][i] += quad->w[iq]*psi[i]*c*phi[i];
	for (j = i+1; j < info->n_col; j++)
	{
	  val1 = psi[i]*btv(n_lambda,Lb0,grd_phi[j]) +
	    btv(n_lambda,Lb1,grd_psi[i])*phi[j];
	  val1 *= quad->w[iq];
	  val0 = quad->w[iq]*psi[i]*c*phi[j];
	  mat[i][j] += val0 + val1;
	  mat[j][i] += val0 - val1;
	}
      }
    }
  }
  else
  {
    for (iq = 0; iq < quad->n_points; iq++)
    {
      Lb0  = (*Lb0_fct)(el_info, quad, iq, info->user_data);
      Lb1  = (*Lb1_fct)(el_info, quad, iq, info->user_data);
      c    = (*c_fct)(el_info, quad, iq, info->user_data);

      grd_psi = psi_fast->grd_phi[iq];
      psi     = psi_fast->phi[iq];
      grd_phi = phi_fast->grd_phi[iq];
      phi     = phi_fast->phi[iq];

      for (i = 0; i < info->n_row; i++)
      {
	for (j = 0; j < info->n_col; j++)
	{
	  val1 = psi[i]*btv(n_lambda,Lb0,grd_phi[j]) +
	    btv(n_lambda,Lb1,grd_psi[i])*phi[j];
	  val0 = psi[i]*c*phi[j];
	  mat[i][j] += quad->w[iq]*(val1 + val0);
	}
      }
    }
  }

  return;
}

static void quad_2_01_0(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL       (*LALt)[N_LAMBDA], *Lb0;
  REAL             c, (*grd_psi)[N_LAMBDA], *psi, (*grd_phi)[N_LAMBDA], *phi, val;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  REAL             **mat = info->element_matrix;
  int              n_lambda = el_info->mesh->dim + 1;

  const REAL (*(*LALt_fct)(const EL_INFO *, const QUAD *, int, void *))[N_LAMBDA];
  const REAL *(*Lb0_fct)(const EL_INFO *, const QUAD *, int, void *);
  REAL       (*c_fct)(const EL_INFO *, const QUAD *, int, void *);

  LALt_fct = info->LALt;
  Lb0_fct  = info->Lb0;
  c_fct    = info->c;
  quad     = info->quad[2];
  psi_fast = info->psi_quad_fast[2];
  phi_fast = info->phi_quad_fast[2];

  for (iq = 0; iq < quad->n_points; iq++)
  {
    LALt = (*LALt_fct)(el_info, quad, iq, info->user_data);
    Lb0  = (*Lb0_fct)(el_info, quad, iq, info->user_data);
    c    = (*c_fct)(el_info, quad, iq, info->user_data);

    grd_psi = psi_fast->grd_phi[iq];
    grd_phi = phi_fast->grd_phi[iq];

    psi     = psi_fast->phi[iq];
    phi     = phi_fast->phi[iq];

    for (i = 0; i < info->n_row; i++)
    {
      for (j = 0; j < info->n_col; j++)
      {
	val  = utAv(n_lambda, grd_psi[i], LALt, grd_phi[j]);
	val += psi[i]*(btv(n_lambda, Lb0, grd_phi[j])+c*phi[j]);
	mat[i][j] += quad->w[iq]*val;
      }
    }
  }
  return;
}

static void quad_2_10_0(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL       (*LALt)[N_LAMBDA], *Lb1;
  REAL             c, (*grd_psi)[N_LAMBDA], *psi, (*grd_phi)[N_LAMBDA], *phi, val;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  REAL             **mat = info->element_matrix;
  int              n_lambda = el_info->mesh->dim + 1;

  const REAL (*(*LALt_fct)(const EL_INFO *, const QUAD *, int, void *))[N_LAMBDA];
  const REAL *(*Lb1_fct)(const EL_INFO *, const QUAD *, int, void *);
  REAL       (*c_fct)(const EL_INFO *, const QUAD *, int, void *);

  LALt_fct = info->LALt;
  Lb1_fct  = info->Lb1;
  c_fct    = info->c;
  quad     = info->quad[2];
  psi_fast = info->psi_quad_fast[2];
  phi_fast = info->phi_quad_fast[2];

  for (iq = 0; iq < quad->n_points; iq++)
  {
    LALt = (*LALt_fct)(el_info, quad, iq, info->user_data);
    Lb1  = (*Lb1_fct)(el_info, quad, iq, info->user_data);
    c    = (*c_fct)(el_info, quad, iq, info->user_data);

    grd_psi = psi_fast->grd_phi[iq];
    grd_phi = phi_fast->grd_phi[iq];

    psi     = psi_fast->phi[iq];
    phi     = phi_fast->phi[iq];

    for (i = 0; i < info->n_row; i++)
    {
      for (j = 0; j < info->n_col; j++)
      {
	val  = utAv(n_lambda, grd_psi[i], LALt, grd_phi[j]);
	val += (btv(n_lambda,Lb1, grd_psi[i])+c*psi[i])*phi[j];
	mat[i][j] += quad->w[iq]*val;
      }
    }
  }
  return;
}

static void quad_2_11_0(const EL_INFO *el_info, const FILL_INFO *info)
{
  const REAL       (*LALt)[N_LAMBDA], *Lb0, *Lb1;
  REAL             c, (*grd_psi)[N_LAMBDA], *psi, (*grd_phi)[N_LAMBDA], *phi;
  int              iq, i, j;
  const QUAD_FAST  *psi_fast, *phi_fast;
  const QUAD       *quad;
  REAL             **mat = info->element_matrix, val20, val1;
  int              n_lambda = el_info->mesh->dim + 1;

  const REAL (*(*LALt_fct)(const EL_INFO *,const QUAD *,int,void *))[N_LAMBDA];
  const REAL *(*Lb0_fct)(const EL_INFO *, const QUAD *, int, void *);
  const REAL *(*Lb1_fct)(const EL_INFO *, const QUAD *, int, void *);
  REAL       (*c_fct)(const EL_INFO *, const QUAD *, int, void *);

  LALt_fct = info->LALt;
  Lb0_fct  = info->Lb0;
  Lb1_fct  = info->Lb1;
  c_fct    = info->c;
  quad     = info->quad[2];
  psi_fast = info->psi_quad_fast[2];
  phi_fast = info->phi_quad_fast[2];

  if (info->LALt_symmetric  &&  info->Lb0_Lb1_anti_symmetric)
  {
    for (iq = 0; iq < quad->n_points; iq++)
    {
      LALt = (*LALt_fct)(el_info, quad, iq, info->user_data);
      Lb0  = (*Lb0_fct)(el_info, quad, iq, info->user_data);
      Lb1  = (*Lb1_fct)(el_info, quad, iq, info->user_data);
      c    = (*c_fct)(el_info, quad, iq, info->user_data);

      grd_psi = psi_fast->grd_phi[iq];
      grd_phi = phi_fast->grd_phi[iq];
      
      psi     = psi_fast->phi[iq];
      phi     = phi_fast->phi[iq];

      for (i = 0; i < info->n_row; i++)
      {
	val20 = utAv(n_lambda, grd_psi[i], LALt, grd_phi[i]) + psi[i]*c*phi[i];
	mat[i][i] += quad->w[iq]*val20; 
	for (j = i+1; j < info->n_col; j++)
	{
	  val20 = utAv(n_lambda, grd_psi[i],
		       LALt, grd_phi[j]) + psi[i]*c*phi[j];
	  val1  = psi[i]*btv(n_lambda, Lb0, grd_phi[j]) +
	    btv(n_lambda, Lb1, grd_psi[i])*phi[j];
	  mat[i][j] += quad->w[iq]*(val20 + val1);
	  mat[j][i] += quad->w[iq]*(val20 - val1);
	}
      }
    }
  }
  else
  {
    for (iq = 0; iq < quad->n_points; iq++)
    {
      LALt = (*LALt_fct)(el_info, quad, iq, info->user_data);
      Lb0  = (*Lb0_fct)(el_info, quad, iq, info->user_data);
      Lb1  = (*Lb1_fct)(el_info, quad, iq, info->user_data);
      c    = (*c_fct)(el_info, quad, iq, info->user_data);

      grd_psi = psi_fast->grd_phi[iq];
      grd_phi = phi_fast->grd_phi[iq];
      
      psi     = psi_fast->phi[iq];
      phi     = phi_fast->phi[iq];

      for (i = 0; i < info->n_row; i++)
      {
	for (j = 0; j < info->n_col; j++)
	{
	  val20 = utAv(n_lambda, grd_psi[i], LALt, grd_phi[j])
	    + psi[i]*c*phi[j];
	  val1  = psi[i]*btv(n_lambda, Lb0, grd_phi[j]) +
	    btv(n_lambda, Lb1, grd_psi[i])*phi[j];
	  mat[i][j] += quad->w[iq]*(val20 + val1);
	}
      }
    }
  }
  return;
}

static const REAL **element_matrix(const EL_INFO *el_info, void *fill_info)
{
  FILL_INFO   *info = (FILL_INFO *)fill_info;
  int         i, j, use_slow = 0;
  REAL        **mat;
  
  mat = info->element_matrix;
  for(i = 0; i < info->n_row; i++) 
    for (j = 0; j < info->n_col; j++)
      mat[i][j] = 0.0;

  /* Use the return value to determine if we are on a parametric element. */
  if (info->init_element)  
    use_slow = (*info->init_element)(el_info, info->quad, info->user_data);

  if (!use_slow && info->fast_second_order)  
      (*info->fast_second_order)(el_info, info);
  else if (info->slow_second_order)  
    (*info->slow_second_order)(el_info, info);

  if (!use_slow && info->fast_first_order)  
    (*info->fast_first_order)(el_info, info);
  else if (info->slow_first_order)  
    (*info->slow_first_order)(el_info, info);

  if (!use_slow && info->fast_zero_order)  
    (*info->fast_zero_order)(el_info, info);
  else if (info->slow_zero_order)  
      (*info->slow_zero_order)(el_info, info);

  return((const REAL **) mat);
}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* Just some defines to make the code below a little bit more readable.
 * also, use |= below instead of += so that the reader knows: it's not
 * magic, it's just a flag value.
 */
#define FLAG_C    0x8
#define FLAG_LB1  0x4
#define FLAG_LB0  0x2
#define FLAG_LALt 0x1

const EL_MATRIX_INFO *fill_matrix_info(const OPERATOR_INFO *operator_info, 
				       EL_MATRIX_INFO *matrix_info)
{
  FUNCNAME("fill_matrix_info");
  static FILL_INFO  *first_fill_info = nil;
  FILL_INFO         *fill_info;
  OPERATOR_INFO     oinfo[1];
  const BAS_FCTS    *psi, *phi;
  PARAMETRIC        *parametric = nil;
  int               dim, not_all_param = false;
  int               psi_deg, phi_deg;
  int               pre2, pre1, pre0, quad2, quad1, quad0;

  *oinfo = *operator_info;

  if (!oinfo->row_fe_space && !oinfo->col_fe_space)
  {
    ERROR("both pointer to row and column FE_SPACEs nil\n");
    ERROR("can not initialize EL_MATRIX_INFO; returning nil\n");
    return(nil);
  }
  else if (!oinfo->row_fe_space)
    oinfo->row_fe_space = oinfo->col_fe_space;
  else if (!oinfo->col_fe_space)
    oinfo->col_fe_space = oinfo->row_fe_space;
  psi = oinfo->row_fe_space->bas_fcts;
  phi = oinfo->col_fe_space->bas_fcts;

  if(phi->dim != psi->dim) {
    ERROR("Support dimensions of phi and psi do not match!\n");
    ERROR("can not initialize EL_MATRIX_INFO; returning nil\n");
    return(nil);
  }

  dim = phi->dim;

  psi_deg = psi->degree;
  phi_deg = phi->degree;

  TEST_EXIT(oinfo->row_fe_space->mesh == oinfo->col_fe_space->mesh,
	    "Mesh must be the same for row and column fe_space!\n");

  parametric = oinfo->row_fe_space->mesh->parametric;

  if (!oinfo->c   &&  !oinfo->Lb0 &&  !oinfo->Lb1  &&  !oinfo->LALt)
  {
    ERROR("no function for 2nd, 1st, and 0 order term;\n");
    ERROR("can not initialize EL_MATRIX_INFO; returning nil\n");
    return(nil);
  }

  if(parametric) {
    if(!oinfo->quad[0] && !oinfo->quad[1] && !oinfo->quad[2]) {
      ERROR("User is responsible for providing at least one quadrature\n");
      ERROR("when using a parametric mesh!\n");
      ERROR("can not initialize EL_MATRIX_INFO; returning nil\n");
      return(nil);
    }
    not_all_param = parametric->not_all;
  }

  if (psi != phi)
  {
    oinfo->LALt_symmetric = 0;
    oinfo->Lb0_Lb1_anti_symmetric = 0;
  }

  if (oinfo->LALt && !oinfo->quad[2])
    oinfo->quad[2] = get_quadrature(dim, psi_deg + phi_deg - 2);
  else if (!oinfo->LALt)
    oinfo->quad[2] = nil;

  if ((oinfo->Lb0 || oinfo->Lb1) && !oinfo->quad[1])
  {
    if ((!oinfo->Lb0_pw_const || !oinfo->Lb1_pw_const) && oinfo->quad[2])
      oinfo->quad[1] = oinfo->quad[2];
    else
      oinfo->quad[1] = get_quadrature(dim, psi_deg + phi_deg - 1);
  }
  else if (!oinfo->Lb0 && !oinfo->Lb1)
    oinfo->quad[1] = nil;

  if (oinfo->c && !oinfo->quad[0])
  {
    if (!oinfo->c_pw_const && oinfo->quad[2])
      oinfo->quad[0] = oinfo->quad[2];
    else if (!oinfo->c_pw_const && oinfo->quad[1])
      oinfo->quad[0] = oinfo->quad[1];
    else
      oinfo->quad[0] = get_quadrature(dim, psi_deg + phi_deg);
  }
  else if (!oinfo->c)
    oinfo->quad[0] = nil;

/*--------------------------------------------------------------------------*/
/*  look for an existing fill_info                                          */
/*--------------------------------------------------------------------------*/

  for (fill_info = first_fill_info; fill_info; fill_info = fill_info->next)
  {
    if (fill_info->psi_fe != oinfo->row_fe_space) continue;
    if (fill_info->phi_fe != oinfo->col_fe_space) continue;
    if (fill_info->quad[2] != oinfo->quad[2]) continue;
    if (fill_info->quad[1] != oinfo->quad[1]) continue;
    if (fill_info->quad[0] != oinfo->quad[0]) continue;
    if (fill_info->parametric != parametric) continue;
    if (fill_info->init_element != oinfo->init_element) continue;
    if (fill_info->LALt != oinfo->LALt) continue;
    if (fill_info->LALt_symmetric != oinfo->LALt_symmetric) continue;
    if (fill_info->Lb0 != oinfo->Lb0) continue;
    if (fill_info->Lb1 != oinfo->Lb1) continue;
    if (fill_info->Lb0_Lb1_anti_symmetric != oinfo->Lb0_Lb1_anti_symmetric)
      continue;
    if (fill_info->c != oinfo->c) continue;
    if (fill_info->user_data != oinfo->user_data) continue;

    break;
  }

  if (!fill_info)
  {
/*--------------------------------------------------------------------------*/
/*  create a new fill_info                                                  */
/*--------------------------------------------------------------------------*/
    const QUAD_FAST **psi_fast, **phi_fast;
    int             n_row = psi->n_bas_fcts, n_col = phi->n_bas_fcts;

    fill_info = MEM_CALLOC(1, FILL_INFO);
    fill_info->next = first_fill_info;
    first_fill_info = fill_info;

    fill_info->psi_fe = oinfo->row_fe_space;
    fill_info->phi_fe = oinfo->col_fe_space;

    fill_info->quad[2] = oinfo->quad[2];
    fill_info->quad[1] = oinfo->quad[1];
    fill_info->quad[0] = oinfo->quad[0];

    fill_info->n_row = n_row;
    fill_info->n_col = n_col;
    fill_info->element_matrix = MAT_ALLOC(n_row, n_col, REAL);

    psi_fast = fill_info->psi_quad_fast;
    phi_fast = fill_info->phi_quad_fast;

    fill_info->parametric = parametric;

    fill_info->init_element = oinfo->init_element;
    fill_info->LALt = oinfo->LALt;
    fill_info->LALt_symmetric = oinfo->LALt_symmetric;
    fill_info->Lb0  = oinfo->Lb0;
    fill_info->Lb1  = oinfo->Lb1;
    fill_info->Lb0_Lb1_anti_symmetric = oinfo->Lb0_Lb1_anti_symmetric;
    fill_info->c    = oinfo->c;
    fill_info->c_symmetric = (phi == psi);

    fill_info->user_data = oinfo->user_data;

    psi_fast[2] = phi_fast[2] = nil;
    pre2 = quad2 = 0;

    if (fill_info->LALt)
    {
      if (oinfo->LALt_pw_const) {
	pre2  |= FLAG_LALt;
	fill_info->q11_psi_phi = get_q11_psi_phi(psi, phi, oinfo->quad[2]);

	if(parametric && !not_all_param) {
	  WARNING("You have selected piecewise constant LALt but seem to\n");
	  WARNING("have a parametric mesh without affine elements!\n");
	}
      }

      if(!oinfo->LALt_pw_const || parametric) {
	quad2 |= FLAG_LALt;
	psi_fast[2] = get_quad_fast(psi, oinfo->quad[2], INIT_GRD_PHI);
	if (psi != phi)
	  phi_fast[2] = get_quad_fast(phi, oinfo->quad[2], INIT_GRD_PHI);
	else
	  phi_fast[2] = psi_fast[2];
      }
    }

    psi_fast[1] = phi_fast[1] = nil;
    pre1 = quad1 = 0;
    if (fill_info->Lb0)
    {
      if (oinfo->Lb0_pw_const) {
	pre1  |= FLAG_LB0;
	fill_info->q01_psi_phi = get_q01_psi_phi(psi, phi, oinfo->quad[1]);

	if(parametric && !not_all_param) {
	  WARNING("You have selected piecewise constant Lb0 but seem to\n");
	  WARNING("have a parametric mesh without affine elements!\n");
	}
      }

      if(!oinfo->Lb0_pw_const || parametric) {
	if (quad2 && oinfo->quad[1] == oinfo->quad[2])
	  quad2 |= FLAG_LB0;
	else
	  quad1 |= FLAG_LB0;

	psi_fast[1] = get_quad_fast(psi, oinfo->quad[1], INIT_PHI);
	phi_fast[1] = get_quad_fast(phi, oinfo->quad[1], INIT_GRD_PHI);
      }
    }

    if (fill_info->Lb1)
    {
      if (oinfo->Lb1_pw_const)
      {
	pre1  |= FLAG_LB1;
	fill_info->q10_psi_phi = get_q10_psi_phi(psi, phi, oinfo->quad[1]);

	if(parametric && !not_all_param) {
	  WARNING("You have selected piecewise constant Lb1 but seem to\n");
	  WARNING("have a parametric mesh without affine elements!\n");
	}
      }

      if(!oinfo->Lb1_pw_const || parametric) {
	if (quad2 && oinfo->quad[1] == oinfo->quad[2])
	  quad2 |= FLAG_LB1;
	else
	  quad1 |= FLAG_LB1;

	psi_fast[1] = get_quad_fast(psi, oinfo->quad[1], INIT_GRD_PHI);
	phi_fast[1] = get_quad_fast(phi, oinfo->quad[1], INIT_PHI);
      }
    }

    psi_fast[0] = phi_fast[0] = nil;
    pre0 = quad0 = 0;

    if (fill_info->c)
    {
      if (oinfo->c_pw_const)
      {
	pre0 |= FLAG_C;
	fill_info->q00_psi_phi = get_q00_psi_phi(psi, phi, oinfo->quad[0]);

	if(parametric && !not_all_param) {
	  WARNING("You have selected piecewise constant c but seem to\n");
	  WARNING("have a parametric mesh without affine elements!\n");
	}
      }
      if(!oinfo->c_pw_const || parametric) {
	if (quad2 && oinfo->quad[0] == oinfo->quad[2])
	  quad2 |= FLAG_C;
	else if (quad1 && oinfo->quad[0] == oinfo->quad[1])
	  quad1 |= FLAG_C;
	else
	  quad0 |= FLAG_C;

	psi_fast[0] = get_quad_fast(psi, oinfo->quad[0], INIT_PHI);
	if (psi != phi)
	  phi_fast[0] = get_quad_fast(phi, oinfo->quad[0], INIT_PHI);
	else
	  phi_fast[0] = psi_fast[0];
      }
    }

    if (pre2)
    {
      fill_info->fast_second_order = pre_2;
      INFO(0,2,"using second order pre_2\n");
    }

    switch (pre1)
    {
    case FLAG_LB0:
      fill_info->fast_first_order = pre_01;
      INFO(0,2,"using first order pre_01\n");
      break;
    case FLAG_LB1:
      fill_info->fast_first_order = pre_10;
      INFO(0,2,"using first order pre_10\n");
      break;
    case FLAG_LB0|FLAG_LB1:
      fill_info->fast_first_order = pre_11;
      INFO(0,2,"using first order pre_11\n");
      break;
    }

    if (pre0 /* == FLAG_C */)
    {
      fill_info->fast_zero_order = pre_0;
      INFO(0,2,"using zero order pre_0\n");
    }

    switch (quad2)
    {
    case FLAG_LALt: 
      fill_info->slow_second_order = quad_2;
      INFO(0,2,"using second order quad_2\n");
      break;
    case FLAG_LALt|FLAG_LB0:
      fill_info->slow_second_order = quad_2_01;
      INFO(0,2,"using second order quad_2_01\n");
      break;
    case FLAG_LALt|FLAG_LB1:
      fill_info->slow_second_order = quad_2_10;
      INFO(0,2,"using second order quad_2_10\n");
      break;
    case FLAG_LALt|FLAG_LB1|FLAG_LB0:
      fill_info->slow_second_order = quad_2_11;
      INFO(0,2,"using second order quad_2_11\n");
      break;
    case FLAG_LALt|FLAG_C:
      fill_info->slow_second_order = quad_2_0;
      INFO(0,2,"using second order quad_2_0\n");
      break;
    case FLAG_LALt|FLAG_LB0|FLAG_C:
      fill_info->slow_second_order = quad_2_01_0;
      INFO(0,2,"using second order quad_2_01_0\n");
      break;
    case FLAG_LALt|FLAG_LB1|FLAG_C:
      fill_info->slow_second_order = quad_2_10_0;
      INFO(0,2,"using second order quad_2_10_0\n");
      break;
    case FLAG_LALt|FLAG_LB1|FLAG_LB0|FLAG_C:
      fill_info->slow_second_order = quad_2_11_0;
      INFO(0,2,"using second order quad2_11_0\n");
      break;
    }

    switch (quad1)
    {
    case FLAG_LB0: 
      fill_info->slow_first_order = quad_01;
      INFO(0,2,"using first order quad_01\n");
      break;
    case FLAG_LB1: 
      fill_info->slow_first_order = quad_10;
      INFO(0,2,"using first order quad_10\n");
      break;
    case FLAG_LB0|FLAG_LB1: 
      fill_info->slow_first_order = quad_11;
      INFO(0,2,"using first order quad_11\n");
      break;
    case FLAG_LB0|FLAG_C: 
      fill_info->slow_first_order = quad_01_0;
      INFO(0,2,"using first order quad_01_0\n");
      break;
    case FLAG_LB1|FLAG_C:
      fill_info->slow_first_order = quad_10_0;
      INFO(0,2,"using first order quad_10_0\n");
      break;
    case FLAG_LB0|FLAG_LB1|FLAG_C: 
      fill_info->slow_first_order = quad_11_0;
      INFO(0,2,"using first order quad_11_0\n");
      break;
    }

    if (quad0 /* == FLAG_C */)
      {
      fill_info->slow_zero_order = quad_0;
      INFO(0,2,"using zero order quad_0\n");
    }
  }

  if (!matrix_info)
    matrix_info = MEM_CALLOC(1, EL_MATRIX_INFO);

  matrix_info->n_row = psi->n_bas_fcts;
  matrix_info->row_admin = oinfo->row_fe_space->admin;
  matrix_info->col_admin = oinfo->col_fe_space->admin;
  matrix_info->get_row_dof = psi->get_dof_indices;

  if (psi != phi)
  {
    matrix_info->n_col = phi->n_bas_fcts;
    matrix_info->get_col_dof = phi->get_dof_indices;
    matrix_info->get_bound = nil;
    matrix_info->fill_flag = oinfo->fill_flag;
  }
  else if (oinfo->use_get_bound)
  {
    matrix_info->get_bound = psi->get_bound;
    matrix_info->fill_flag = oinfo->fill_flag|FILL_BOUND;
  }
  else
  {
    matrix_info->get_bound = nil;
    matrix_info->fill_flag = oinfo->fill_flag;
  }

  matrix_info->factor = 1.0;
  matrix_info->el_matrix_fct = element_matrix;
  matrix_info->fill_info = fill_info;

  return(matrix_info);
}
