/*--------------------------------------------------------------------------*/
/* ALBERTA_UTIL:  tools for messages, memory allocation, parameters, etc.   */
/*                                                                          */
/* file: alberta_util.h                                                     */
/*                                                                          */
/* description: public header file of the ALBERTA_UTIL package              */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.alberta-fem.de                                               */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#ifndef _ALBERTA_UTIL_H_
#define _ALBERTA_UTIL_H_

#include <stdlib.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <float.h>

#ifdef __cplusplus
extern "C" {
#endif

#ifndef true
#define true          1
#define false         0
#endif

#ifndef nil
#define nil           NULL
#endif

#ifndef MAX
#define MAX(a, b)     ((a) > (b) ? (a) : (b))
#define MIN(a, b)     ((a) < (b) ? (a) : (b))
#endif
#ifndef ABS
#define ABS(a)        ((a) >= 0 ? (a) : -(a))
#endif

#ifndef SQR
#define SQR(a)        ((a)*(a))
#endif

#ifndef M_E
#define M_E		2.7182818284590452354
#endif
#ifndef M_LOG2E
#define M_LOG2E		1.4426950408889634074
#endif
#ifndef M_LOG10E
#define M_LOG10E	0.43429448190325182765
#endif
#ifndef M_LN2
#define M_LN2		0.69314718055994530942
#endif
#ifndef M_LN10
#define M_LN10		2.30258509299404568402
#endif
#ifndef M_PI
#define M_PI		3.14159265358979323846
#endif
#ifndef M_PI_2
#define M_PI_2		1.57079632679489661923
#endif
#ifndef M_PI_4
#define M_PI_4		0.78539816339744830962
#endif
#ifndef M_1_PI
#define M_1_PI		0.31830988618379067154
#endif
#ifndef M_2_PI
#define M_2_PI		0.63661977236758134308
#endif
#ifndef M_2_SQRTPI
#define M_2_SQRTPI	1.12837916709551257390
#endif
#ifndef M_SQRT2
#define M_SQRT2		1.41421356237309504880
#endif
#ifndef M_SQRT1_2
#define M_SQRT1_2	0.70710678118654752440
#endif

/*--------------------------------------------------------------------------*/
/*  Definition of basic data types      				    */
/*--------------------------------------------------------------------------*/

#ifndef REAL_DEFINED
#define REAL_DEFINED

#define USE_DOUBLE_AS_REAL  1

#if USE_DOUBLE_AS_REAL

typedef double              REAL;

#define REAL_MANT_DIG     DBL_MANT_DIG     
#define REAL_EPSILON      DBL_EPSILON      
#define REAL_DIG          DBL_DIG          
#define REAL_MIN_EXP      DBL_MIN_EXP      
#define REAL_MIN          DBL_MIN          
#define REAL_MIN_10_EXP   DBL_MIN_10_EXP   
#define REAL_MAX_EXP      DBL_MAX_EXP      
#define REAL_MAX          DBL_MAX          
#define REAL_MAX_10_EXP   DBL_MAX_10_EXP   

#else  /*  USE_DOUBLE_AS_REAL  */

typedef float             REAL;

#define REAL_MANT_DIG     FLT_MANT_DIG     
#define REAL_EPSILON      FLT_EPSILON      
#define REAL_DIG          FLT_DIG          
#define REAL_MIN_EXP      FLT_MIN_EXP      
#define REAL_MIN          FLT_MIN          
#define REAL_MIN_10_EXP   FLT_MIN_10_EXP   
#define REAL_MAX_EXP      FLT_MAX_EXP      
#define REAL_MAX          FLT_MAX          
#define REAL_MAX_10_EXP   FLT_MAX_10_EXP   

#endif  /*  USE_DOUBLE_AS_REAL  */

#ifndef LARGE
#define LARGE         REAL_MAX
#endif

#endif  /*  REAL_DEFINED  */

#ifndef U_CHAR_DEFINED
#define U_CHAR_DEFINED

typedef unsigned char       U_CHAR;
typedef signed char         S_CHAR;
typedef unsigned long       FLAGS;
#endif

#ifndef WORKSPACE_DEFINED
#define WORKSPACE_DEFINED
typedef struct workspace    WORKSPACE;

struct workspace
{
  size_t  size;
  void    *work;
};
#endif

#ifndef VOID_LIST_ELEMENT_DEFINED
#define VOID_LIST_ELEMENT_DEFINED
typedef struct void_list_element VOID_LIST_ELEMENT;

struct void_list_element
{
  void               *data;
  VOID_LIST_ELEMENT  *next;
};
VOID_LIST_ELEMENT *get_void_list_element(void);
void free_void_list_element(VOID_LIST_ELEMENT *);
#endif

/*--------------------------------------------------------------------------*/
/*  functions and macros for messages	                         	    */
/*--------------------------------------------------------------------------*/

#ifdef __GNUC__
# define ALBERTA_UNUSED(proto) proto __attribute__((unused))
# define ALBERTA_DEFUNUSED(proto) ALBERTA_UNUSED(proto); proto
# define ALBERTA_NORETURN(proto) proto __attribute__((noreturn))
# define ALBERTA_DEFNORETURN(proto) ALBERTA_NORETURN(proto); proto
#else
# define ALBERTA_UNUSED(proto) proto
# define ALBERTA_DEFUNUSED(proto) proto
# define ALBERTA_NORETURN(proto) proto
# define ALBERTA_DEFNORETURN(proto) proto
#endif

/*--------------------------------------------------------------------------*/
/* Some of the printing functions are declared to return type int only for  */
/* the purpose of defining the macros below without "if" statements!        */
/*--------------------------------------------------------------------------*/

int  print_msg(const char *format, ...);
int  print_error_msg(const char *format, ...);
int  print_error_msg_exit(const char *format, ...);
int  print_error_funcname(const char *name, const char *file, int line);
void print_warn_msg(const char *format, ...);
void print_warn_funcname(const char *name, const char *file, int line);
int  print_funcname(const char *name);
void print_int_vec(const char *s, const int *vec, int no);
void print_real_vec(const char *s, const REAL *vec, int no);

void change_msg_out(FILE *fp);
void open_msg_file(const char *filename, const char *type);
void change_error_out(FILE *fp);
void open_error_file(const char *filename, const char *type);
const char *generate_filename(const char *, const char *, int);

#define FUNCNAME(nn)    static const char *funcName = nn

#define ERROR          print_error_funcname(funcName, __FILE__, __LINE__),\
                       print_error_msg
#define ERROR_EXIT     print_error_funcname(funcName, __FILE__, __LINE__),\
                       print_error_msg_exit
#define WARNING        print_warn_funcname(funcName, __FILE__, __LINE__),\
                       print_warn_msg

#define TEST(test, ...)						\
  do {								\
    if (!(test)) {						\
      print_error_funcname(funcName, __FILE__, __LINE__);	\
      print_error_msg(__VA_ARGS__);				\
    }								\
  } while (0)
#define TEST_EXIT(test, ...)					\
  do {								\
    if (!(test)) {						\
      print_error_funcname(funcName, __FILE__, __LINE__);	\
      print_error_msg_exit(__VA_ARGS__);			\
    }								\
  } while (0)

/*--------------------------------------------------------------------------*/
/* Some testing macros useful while debugging.                              */
/*--------------------------------------------------------------------------*/

#if ALBERTA_DEBUG==1
#define DEBUG_TEST(test, ...)					\
  do {								\
    if (!(test)) {						\
      print_error_funcname(funcName, __FILE__, __LINE__);	\
      print_error_msg(__VA_ARGS__);				\
    }								\
  } while (0)
#define DEBUG_TEST_EXIT(test, ...)				\
  do {								\
    if (!(test)) {						\
      print_error_funcname(funcName, __FILE__, __LINE__);	\
      print_error_msg_exit(__VA_ARGS__);			\
    }								\
  } while (0)
#else
#define DEBUG_TEST(test, ...)      do { funcName = funcName; } while (0)
#define DEBUG_TEST_EXIT(test, ...) do { funcName = funcName; } while (0)
#endif


#define MSG             print_funcname(funcName), print_msg

#define INFO(info,noinfo, ...)\
  do {							\
    if (msg_info&&(MIN(msg_info,(info))>=(noinfo))) {	\
      print_funcname(funcName); print_msg(__VA_ARGS__);	\
    }							\
  } while (0)

#define PRINT_INFO(info,noinfo, ...)			\
  do {							\
    if (msg_info&&(MIN(msg_info,(info))>=(noinfo))) {	\
      print_msg(__VA_ARGS__);				\
    }							\
  } while (0)

#define PRINT_INT_VEC   print_funcname(funcName), print_int_vec
#define PRINT_REAL_VEC  print_funcname(funcName), print_real_vec

extern void alberta_wait(const char *, int);
#define WAIT            alberta_wait(funcName, msg_wait)
#define WAIT_REALLY     alberta_wait(funcName, 1)

/*--------------------------------------------------------------------------*/
/*  Definition of extern variables      				    */
extern int           msg_wait;                       /* declared in msg.c   */
extern const char    *funcName;                      /* declared in msg.c   */
extern int           msg_info;                       /* declared in msg.c   */

/*--------------------------------------------------------------------------*/
/*  Macros for memory allocation					    */
/*--------------------------------------------------------------------------*/

void *alberta_alloc(size_t, const char *, const char *,int);
void *alberta_realloc(void *, size_t, size_t, const char *, const char *, int);
void *alberta_calloc(size_t, size_t, const char *, const char *,int);
void alberta_free(void *, size_t);
void print_mem_use(void);

void **alberta_matrix(size_t, size_t, size_t, const char *, const char *, int);
void free_alberta_matrix(void **, size_t, size_t, size_t);

WORKSPACE *get_workspace(size_t, const char *, const char *, int);
#define GET_WORKSPACE(s) get_workspace(s,funcName,__FILE__,__LINE__)

WORKSPACE  *realloc_workspace(WORKSPACE *ws, size_t new_size, 
			      const char *fct, const char *file, int line);
#define REALLOC_WORKSPACE(ws,size)\
 (realloc_workspace(ws,size,funcName,__FILE__,__LINE__))

void clear_workspace(WORKSPACE *ws);
#define CLEAR_WORKSPACE(ws) clear_workspace(ws)
void free_workspace(WORKSPACE *ws);
#define FREE_WORKSPACE(ws) free_workspace(ws)

#define MEM_ALLOC(n,type)\
   ((type*)alberta_alloc((n)*sizeof(type),funcName,__FILE__,__LINE__))
#define MEM_REALLOC(ptr,old_n,new_n,type)\
   ((type*)alberta_realloc((void *)ptr,(old_n)*sizeof(type),\
    (new_n)*sizeof(type),funcName,__FILE__,__LINE__))
#define MEM_CALLOC(n,type)\
   ((type*)alberta_calloc((n), sizeof(type),funcName,__FILE__,__LINE__))
#define MEM_FREE(ptr,n,type) alberta_free((void *)ptr,(n)*sizeof(type))

#define MAT_ALLOC(nr,nc,type)\
  ((type**)alberta_matrix((nr),(nc),sizeof(type),funcName,__FILE__, __LINE__))
#define MAT_FREE(ptr,nr,nc,type)\
    free_alberta_matrix((void **)ptr,(nr),(nc),sizeof(type))

#define NAME(struct)\
   ((struct) ? ((struct)->name ? (struct)->name : #struct "->name unknown") :\
   #struct " pointer to nil")

#define GET_STRUCT(struct, from)\
   TEST_EXIT((from)  && (struct = (from)->struct),\
     (from) ? "GET_STRUCT: %s->%s == nil\n" : "GET_STRUCT: %s == nil%s\n",\
      #from, (from) ? #struct : "")

/*--------------------------------------------------------------------------*/
/* functions for handling parameters                                        */
/*--------------------------------------------------------------------------*/

void add_parameter(int p, const char *key, const char *par);
void Add_parameter(int p, const char *key, const char *par,
		   const char *fname, const char *file, int line);
#define ADD_PARAMETER(p,key,param)\
   Add_parameter(p,key,param,funcName, __FILE__, __LINE__)

void init_parameters(int p, const char *filename);
void init_parameters_cpp(int p, const char *fn, const char *flags);

int get_parameter(int flag, const char *key, const char *format, ...);
int init_param_func_name(const char *, const char *, int call_line);

#define GET_PARAMETER  !init_param_func_name(funcName, __FILE__, __LINE__)?\
                       0 : get_parameter
void save_parameters(const char *file, int info);
void print_parameters(void);

/*--------------------------------------------------------------------------*/
/*  generating filenames, general lists                                     */
/*--------------------------------------------------------------------------*/

const char *generate_filename(const char *path, const char *fn, int ntime);
VOID_LIST_ELEMENT *get_void_list_element(void);
void free_void_list_element(VOID_LIST_ELEMENT *list);


/*--------------------------------------------------------------------------*/
/*  library of Orthogonal Error Methods                                     */
/*                                                                          */
/*  most routines are the C-version of OFM-lib by Willy Doerfler            */
/*                                                                          */
/*  author: Kunibert G. Siebert                                             */
/*          Institut fuer Mathematik                                        */
/*          Universitaet Augsburg                                           */
/*          Universitaetsstr. 14                                            */
/*          D-86159 Augsburg, Germany                                       */
/*                                                                          */
/*          http://scicomp.math.uni-augsburg.de/Siebert/                    */
/*                                                                          */
/* (c) by K.G. Siebert (2000-2003)                                          */
/*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*---  data structures for orthogonal error methods  -----------------------*/
/*--------------------------------------------------------------------------*/

typedef struct oem_data OEM_DATA;
struct oem_data
{
  int        (*mat_vec)(void *, int, const REAL *, REAL *);
  void       *mat_vec_data;
  int        (*mat_vec_T)(void *, int, const REAL *, REAL *);
  void       *mat_vec_T_data;
  void       (*left_precon)(void *, int, REAL *);
  void       *left_precon_data;
  void       (*right_precon)(void *, int, REAL *);
  void       *right_precon_data;

  REAL       (*scp)(void *, int, const REAL *, const REAL *);
  void       *scp_data;

  WORKSPACE  *ws;

  REAL       tolerance;
  int        restart;
  int        max_iter;
  int        info;

  REAL       initial_residual;
  REAL       residual;
};

/*--------------------------------------------------------------------------*/
/*--  implemented solvers  -------------------------------------------------*/
/*--------------------------------------------------------------------------*/

int oem_bicgstab(OEM_DATA *, int, const REAL *, REAL *);
int oem_cg(OEM_DATA *, int, const REAL *, REAL *);
int oem_gmres(OEM_DATA *, int, const REAL *, REAL *);
int oem_gmres_k(OEM_DATA *, int, const REAL *, REAL *);
int oem_odir(OEM_DATA *, int, const REAL *, REAL *);
int oem_ores(OEM_DATA *, int, const REAL *, REAL *);
int oem_tfqmr(OEM_DATA *, int, const REAL *, REAL *);

/*--------------------------------------------------------------------------*/
/*  function call:                                                          */
/*    oem_?(oem_data, dim, b, x);                                           */
/*                                                                          */
/*    input:  oem_data:  above described data structure                     */
/*            dim:       dimension of the linear system                     */
/*            b:         vector storing the right hand side of the sytem    */
/*            x:         vector stroing the initial guess                   */
/*                                                                          */
/*    output: x:         solution of the linear system                      */
/*            oem_data->residual:  residual of the equation                 */
/*                                                                          */
/*    return value:      number of iterations                               */
/*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*  Solver for linear saddle point problem                                  */
/*--------------------------------------------------------------------------*/

typedef struct oem_sp_data OEM_SP_DATA;
struct oem_sp_data
{
  int        (*solve_Auf)(void *, int, const REAL *, REAL *);
  void       *solve_Auf_data;  /*--  solve A u = f  in X  ------------------*/
  void       (*B)(void *, REAL, int, const REAL *, int, REAL *);
  void       *B_data;          /*--  compute B p ---------------------------*/
  void       (*Bt)(void *, REAL, int, const REAL *, int, REAL *);
  void       *Bt_data;         /*--  compute B^* u -------------------------*/
  int        (*project)(void *, int, const REAL *, REAL *);
  void       *project_data;    /*--  project B^* u to Y  -------------------*/
  int        (*precon)(void *, int, const REAL *, const REAL *, REAL *);
  void       *precon_data;     /*--  compute C r in Y  ---------------------*/

  WORKSPACE  *ws;              /*--  workspace  ----------------------------*/

  REAL       tolerance;
  int        restart;
  int        max_iter;
  int        info;

  REAL       initial_residual;
  REAL       residual;
};

/*--------------------------------------------------------------------------*/
/*--  implemented solvers  -------------------------------------------------*/
/*--------------------------------------------------------------------------*/

int oem_spcg(OEM_SP_DATA *, int, const REAL *, REAL *, int, const REAL *,
	     REAL*);

/*--------------------------------------------------------------------------*/
/*  library of solvers for nonlinear systems                                */
/*                                                                          */
/*  some routines are the C-version of NGL-lib by Willy Doerfler            */
/*     http://www.mathematik.uni-kl.de/~doerfler                            */
/*                                                                          */
/*  author: Kunibert G. Siebert                                             */
/*          Institut f"ur Angewandte Mathematik                             */
/*          Albert-Ludwigs-Universit"at Freiburg                            */
/*          Hermann-Herder-Str. 10                                          */
/*          79104 Freiburg                                                  */
/*          Germany                                                         */
/*                                                                          */
/* email:  kunibert@mathematik.uni-freiburg.de                              */
/*                                                                          */
/* (c) by K.G. Siebert (2000)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/

typedef struct nls_data NLS_DATA;
struct nls_data
{
  void       (*update)(void *, int, const REAL *, int, REAL *);
  void       *update_data;
  int        (*solve)(void *, int, const REAL *, REAL *);
  void       *solve_data;
  REAL       (*norm)(void *, int, const REAL *);
  void       *norm_data;

  WORKSPACE  *ws;

  REAL       tolerance;
  int        restart;
  int        max_iter;
  int        info;

  REAL       initial_residual;
  REAL       residual;
};

/*--------------------------------------------------------------------------*/
/*  functions:                                                              */
/*--------------------------------------------------------------------------*/
/*    update:  calculate F(x) and DF(x)                                     */
/*    call:                                                                 */
/*             update(update_data, dim, x, ud_DF, Fx)                       */
/*    arguments:                                                            */
/*             update_data: pointer to user data                            */
/*             dim:         dimension of the nonlinear system               */
/*             x:           vector stroring the actual iterate              */
/*             ud_DF:       ud_DF==1: update DF(x)                          */
/*             Fx:          Fx!=nil:  update F(x) and store it in Fx        */
/*--------------------------------------------------------------------------*/
/*    solve:   solve the linearized system DF(x) d = b for the correction   */
/*    call:                                                                 */
/*             solve(solve_data, dim, b, d)                                 */
/*             solve_data:  pointer to user data                            */
/*             dim:         dimension of the nonlinear system               */
/*             x:           vector stroring the actual iterate              */
/*             b:           right hand side of the linearized system        */
/*             d:           initial guess and solution of the system        */
/*                                                                          */
/*             return value is the number of iterations used by an          */
/*             iterative solver                                             */
/*--------------------------------------------------------------------------*/
/*    norm:    compute the norm of the residual                             */
/*             if norm==nil, the Eucledian norm of the coefficient vector   */
/*             in R^dim is used.                                            */
/*    call:                                                                 */
/*             norm(norm_data, dim, d)                                      */
/*             norm_data:   pointer to user data                            */
/*             dim:         dimension of the nonlinear system               */
/*             d:           coefficient vector of the residual              */
/*                                                                          */
/*             return value is the norm of the residual                     */
/*--------------------------------------------------------------------------*/
/*  ws: workspace                                                           */
/*--------------------------------------------------------------------------*/

int nls_newton(NLS_DATA *, int, REAL *);
int nls_newton_ds(NLS_DATA *, int, REAL *);
int nls_newton_fs(NLS_DATA *, int, REAL *);
int nls_newton_br(NLS_DATA *, REAL, int, REAL *);

int nls_banach(NLS_DATA *, int, REAL *);

/*--------------------------------------------------------------------------*/
/*--  inline wrappers to FORTRAN BLAS functions  ---------------------------*/
/*--------------------------------------------------------------------------*/

#include "alberta_util_inlines.h"

#ifdef __cplusplus
}
#endif

#endif  /* !_ALBERTA_UTIL_H_ */
