package Bio::Graphics::Glyph::anchored_arrow;
# package to use for drawing an arrow

use strict;
use base qw(Bio::Graphics::Glyph::arrow);

sub my_description {
    return <<END;
This glyph draws an arrowhead which is anchored at one or both ends
(has a vertical base) or has one or more arrowheads.  The arrowheads
indicate that the feature does not end at the edge of the picture, but
continues.
END
}

sub my_options {
    return {
	tick => [
	    [0..2],
	    0,
	    'Draw a scale with tickmarks on the arrow.',
	    'A value of 0 suppresses the scale.',
	    'A value of 1 draws major ticks only.',
	    'A value of 2 draws major and minor ticks.',],
        relative_coords=> [
	    'boolean',
	    undef,
	    'When drawing the scale, start numbering at position 1 instead of at',
	    'the start of the feature in global (e.g. chromosomal) coordinates.'],
	relative_coords_offset=> [
	    'integer',
	    1,
	    'When drawing a scale with relative_coords set to true, begin numbering',
	    'the scale at this starting value.'],
	no_arrows => [
	    'boolean',
	    undef,
	    "Do not draw an arrow when the glyph is partially offscreen."],
    };
}

sub draw_label {
  my $self = shift;
  my ($gd,$left,$top,$partno,$total_parts) = @_;
  my $label = $self->label or return;
  my $label_align = $self->option('label_align');
  if ($label_align && ($label_align eq 'center' || $label_align eq 'right')) {
      my $x = $self->left + $left;
      my $font = $self->option('labelfont') || $self->font;
      my $middle = $self->left + $left + ($self->right - $self->left) / 2;
      my $label_width = $self->string_width($label,$font);
      if ($label_align eq 'center') {
          my $new_x = $middle - $label_width / 2;
          $x = $new_x if ($new_x > $x);;
      }
      else {
          my $new_x = $left + $self->right - $label_width;
          $x = $new_x if ($new_x > $x);
      }
      $x = $self->panel->left + 1 if $x <= $self->panel->left;
      #detect collision (most likely no bump when want centering label)
      #lay down all features on one line e.g. cyto bands
      return if (!$self->option('bump') && ($label_width + $x) > $self->right);
      $gd->string($font,
                  $x,
                  $self->top + $top,
                  $label,
                  $self->fontcolor);
  }
  else {
      $self->SUPER::draw_label(@_);
  }
}

sub arrowheads {
  my $self = shift;
  my ($ne,$sw,$base_e,$base_w);
  my $feature = $self->feature;
  my $gstart  = $feature->start;
  my $gend    = $feature->end;
  my $pstart  = $self->panel->start;
  my $pend    = $self->panel->end;

  if (!defined $gstart || $gstart <= $pstart) {  # off left end
    $sw = 1;
  }
  if (!defined $gend || $gend >= $pend) { # off right end
    $ne = 1;
  }
  ($sw,$ne) = ($ne,$sw) if $self->panel->{flip};
  return ($sw,$ne,!$sw,!$ne);
}

sub no_trunc {
  !shift->option('no_arrows');
}

1;

__END__

=head1 NAME

Bio::Graphics::Glyph::anchored_arrow - The "anchored_arrow" glyph

=head1 SYNOPSIS

  See L<Bio::Graphics::Panel> and L<Bio::Graphics::Glyph>.

=head1 DESCRIPTION

This glyph draws an arrowhead which is anchored at one or both ends
(has a vertical base) or has one or more arrowheads.  The arrowheads
indicate that the feature does not end at the edge of the picture, but
continues.  For example:

    |-----------------------------|          both ends in picture
 <----------------------|                    left end off picture
         |---------------------------->      right end off picture
 <------------------------------------>      both ends off picture

You can also set the glyph so that the end is just truncated at the
end of the picture.

         |-----------------------------

=head2 OPTIONS

The following options are standard among all Glyphs.  See
L<Bio::Graphics::Glyph> for a full explanation.

  Option      Description                      Default
  ------      -----------                      -------

  -fgcolor      Foreground color	       black

  -outlinecolor	Synonym for -fgcolor

  -bgcolor      Background color               turquoise

  -fillcolor    Synonym for -bgcolor

  -linewidth    Line width                     1

  -height       Height of glyph		       10

  -font         Glyph font		       gdSmallFont

  -connector    Connector type                 0 (false)

  -connector_color
                Connector color                black

  -label        Whether to draw a label	       0 (false)

  -description  Whether to draw a description  0 (false)

  -hilite       Highlight color                undef (no color)

In addition to the standard options, this glyph recognizes the following:

  Option         Description                Default

  -tick          draw a scale               0 (false)

  -relative_coords 
                 use relative coordinates   0 (false)
                 for scale

  -relative_coords_offset 
                 set the relative offset    1 
                 for scale

  -no_arrows     don't draw an arrow when   0 (false)
                 glyph is partly offscreen

The argument for B<-tick> is an integer between 0 and 2 and has the same
interpretation as the B<-tick> option in Bio::Graphics::Glyph::arrow.

If B<-rel_coords> is set to a true value, then the scale drawn on the
glyph will be in relative (1-based) coordinates relative to the beginning
of the glyph.

=head1 BUGS

Please report them.

=head1 SEE ALSO

L<Bio::Graphics::Panel>,
L<Bio::Graphics::Glyph>,
L<Bio::Graphics::Glyph::arrow>,
L<Bio::Graphics::Glyph::cds>,
L<Bio::Graphics::Glyph::crossbox>,
L<Bio::Graphics::Glyph::diamond>,
L<Bio::Graphics::Glyph::dna>,
L<Bio::Graphics::Glyph::dot>,
L<Bio::Graphics::Glyph::ellipse>,
L<Bio::Graphics::Glyph::extending_arrow>,
L<Bio::Graphics::Glyph::generic>,
L<Bio::Graphics::Glyph::graded_segments>,
L<Bio::Graphics::Glyph::heterogeneous_segments>,
L<Bio::Graphics::Glyph::line>,
L<Bio::Graphics::Glyph::pinsertion>,
L<Bio::Graphics::Glyph::primers>,
L<Bio::Graphics::Glyph::rndrect>,
L<Bio::Graphics::Glyph::segments>,
L<Bio::Graphics::Glyph::ruler_arrow>,
L<Bio::Graphics::Glyph::toomany>,
L<Bio::Graphics::Glyph::transcript>,
L<Bio::Graphics::Glyph::transcript2>,
L<Bio::Graphics::Glyph::translation>,
L<Bio::Graphics::Glyph::triangle>,
L<Bio::DB::GFF>,
L<Bio::SeqI>,
L<Bio::SeqFeatureI>,
L<Bio::Das>,
L<GD>

=head1 AUTHOR

Allen Day E<lt>day@cshl.orgE<gt>.

Copyright (c) 2001 Cold Spring Harbor Laboratory

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.  See DISCLAIMER.txt for
disclaimers of warranty.

=cut
