#if !defined (__BRIDGE_HPP)
#define __BRIDGE_HPP

/*
  CoreLinux++ 
  Copyright (C) 1999,2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/   

#if   !defined(__COMMON_HPP)
#include <Common.hpp>
#endif
  

namespace corelinux
{

   /**
   When an abstraction can have one of several possible implementations, 
   the usual way to accommodate them is to use inheritance. An abstract 
   class defines the interface to the abstraction, and concrete subclasses 
   implement it in different ways. But this approach isn't always flexible 
   enough. Inheritance binds an implementation to the abstraction permanently, 
   which makes it difficult to modify, extend, and reuse abstractions and 
   implementations independently. A Bridge decouples an abstraction from 
   its implementation so that the two can vary independently.
   */

   template <class Implementation>
      class Bridge : public CoreLinuxObject
      {
      public:
   
                              /** 
                              Default Constructor requires a Implementation
                              @param Implementation instance
                              */
   
                              Bridge( Implementation aImplementation )
                                 :
                                 CoreLinuxObject(),
                                 theImplementation(aImplementation)
                              {
                                 ;  // do nothing
                              }
   
   
                              /// Virtual Destructor  
   
         virtual              ~Bridge( void )
         {
            ;  // do nothing
         }
   
         //
         // Operator overloads
         //
                  /**
                  Assignment operator overload. This may throw
                  Exception if there is a problem cloning theImplementation.
                  @param  Bridge const reference
                  @return Bridge reference to self
                  @exception Exception - implementation defined
                  */
               
                  Bridge &  operator=( const Bridge & aRef ) throw(Exception)
                  {
                     this->setImplementation( aRef.getImplementation() );
                     return (*this);
                  }
   
                  /**
                  Equality operator overload
                  @param Bridge const reference
                  @return true if equal, false otherwise
                  */
   
                  bool        operator==( const Bridge & ) const
                  {
                     return (this == &aRef);
                  }
   
                  /**
                  Non-equality operator overload
                  @param Bridge const reference
                  @return false if equal, true otherwise
                  */
   
                  bool        operator!=( const Bridge & ) const
                  {
                     return !(*this == aRef);
                  }
   
      protected:
   
                              /** Default Constructor
                              Because a Bridge requires a implementation
                              to work, you can not construct one without it.
                              @param void
                              @exception NEVER_GET_HERE
                              */
                        
                              Bridge( void ) throw (Assertion)
                              {
                                 NEVER_GET_HERE;
                              }
   
                              /** Copy Constructor
                              Because theImplementation is owned by
                              the Bridge, only the assignment operator
                              insures that the derivation is constructed
                              and can be invoked for object management.
                              @param Bridge const reference
                              @exception NEVER_GET_HERE
                              */
   
                              Bridge( const Bridge & ) throw (Assertion)
                              {
                                 NEVER_GET_HERE;
                              }
         //
         // Accessors
         //
                  /**
                  Gets current theImplementation
                  @return Implementation instance
                  */
   
                  Implementation getImplementation( void ) const
                  {
                     return theImplementation;
                  }
   
         //
         // Mutators
         //
   
                  /**
                  Set theImplementation. This in turn calls the pure-virtual
                  method cloneImplementation so that a pointer unique to this
                  object, or one which is referencable can be managed
                  @param Implementation instance
                  @exception Exception - implementation defined
                  */
   
                  void  setImplementation( Implementation aImpl ) throw(Exception)
                  {
                     theImplementation = cloneImplementation(aImpl);
                  }

   
                  /**
                  Pure virtual method to have the derivation contain 
                  theImplementation.by ownership.
                  @param Implementation instance
                  @return Implementation instance
                  @exception Exception - implementation defined
                  */
   
         virtual  Implementation   cloneImplementation
                                          ( 
                                             Implementation
                                          ) throw(Exception) = 0;
   
      private:
   
                  /// Storage for theImplementation object
   
                  Implementation   theImplementation;
   
   
      };

}
          
#endif   // if !defined(__BRIDGE_HPP)

/*
   Common rcs information do not modify
   $Author: prudhomm $
   $Revision: 1.1 $
   $Date: 2000/04/23 20:43:13 $
   $Locker:  $
*/



