
/*
  CoreLinux++ 
  Copyright (C) 1999,2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/

#if   !defined(__COMMON_HPP)
#include <Common.hpp>
#endif

#if   !defined(__EVENTSEMAPHORE_HPP)
#include <EventSemaphore.hpp>
#endif

extern "C"
{
    #include <sys/sem.h>
}

namespace corelinux
{
   //
   // Default constructor not allowed
   //

   EventSemaphore::EventSemaphore( void ) throw(Assertion)
      :
      Semaphore()
   {
      NEVER_GET_HERE;
   }

   //
   // Copy constructor not allowed
   //

   EventSemaphore::EventSemaphore( EventSemaphoreCref ) throw(Assertion)
      :
      Semaphore()
   {
      NEVER_GET_HERE;
   }

   //
   // Default Constructor
   //

   EventSemaphore::EventSemaphore
      ( 
         SemaphoreGroupPtr aGroup,
         SemaphoreIdentifierRef aIdentifier,
         Counter aMaxListener,
         bool aRecursionFlag,
         bool aBalkingFlag
      ) throw ( NullPointerException )
      :
      Semaphore(aGroup,aIdentifier,aRecursionFlag,aBalkingFlag),
      theNumListeners (0), 
      theMaxListeners (aMaxListener)
   {
      setValue( 1 ); // automatically lock upon creation
      Semaphore::setOwnerId();
   }

   //
   // Destructor
   //

   EventSemaphore::~EventSemaphore( void )
   {
      ;  // do nothing
   }

   //
   // Returns true if lock value != 0
   //

   bool  EventSemaphore::isLocked( void )
   {
      return getValue();
   }

   //
   // Indicates owner's commitment to trigger an event
   //

   SemaphoreOperationStatus EventSemaphore::post( void )
      throw(SemaphoreException)
   {
      SemaphoreOperationStatus   aStatus(SUCCESS);

      if ( Thread::getThreadIdentifier() == getOwnerId() )
      {
         if ( theNumListeners == 0 )
         {
            setValue( 1 );  // lock it again
         }
         else
         {
            aStatus = UNAVAILABLE;
         }
      }
      else
      {
         throw SemaphoreException 
            (
               "Not authorized to post an event", 
               LOCATION, 
               Exception::CONTINUABLE
            );
      }
      return aStatus;
   }

   //
   // Call for lock with wait disposition
   //

   SemaphoreOperationStatus EventSemaphore::lockWithWait( void )
      throw(SemaphoreException)
   {
      SemaphoreOperationStatus   aStatus(SUCCESS);
      Guard                      myGuard( Synchronized::access() );

      if ( isLocked() == true ) 
      {
         if (
               ( theMaxListeners >= 0 && theNumListeners < theMaxListeners )
               ||
               theMaxListeners < 0 
             )
         {
            // resource available
            theNumListeners++;
            myGuard.release();
            aStatus = waitZero( 0 );
            if ( aStatus == SUCCESS )
            {
               theNumListeners--;
            }
            else
            {
               ;  // do nothing
            }
         }
         else {
            //
            // theNumListeners >= theMaxListeners >= 0
            //

            myGuard.release();
            if ( isBalkingEnabled() == true )
            {
               aStatus = BALKED;
            }
            else
            {
               aStatus = UNAVAILABLE;
            }
         }
      }
      else 
      {
         //
         // the semaphore is not locked
         //

         myGuard.release();
         aStatus = UNAVAILABLE;
      }
      return aStatus;
   }

   //
   // Call for lock without waiting
   //

   SemaphoreOperationStatus EventSemaphore::lockWithNoWait( void )
      throw( SemaphoreException )
   {
      SemaphoreOperationStatus   aStatus(SUCCESS);
      Guard                      myGuard( Synchronized::access() );

      if ( isLocked() == true ) 
      {
         if (
               ( theMaxListeners >= 0 && theNumListeners < theMaxListeners )
               ||
               theMaxListeners < 0 
             )
         {
            // resource available
            theNumListeners++;
            myGuard.release();
            aStatus = waitZero( IPC_NOWAIT );
            if ( aStatus == SUCCESS )
            {
               theNumListeners--;
            }
            else
            {
               ;  // do nothing
            }
         }
         else {
            //
            // theNumListeners >= theMaxListeners >= 0
            //

            myGuard.release();
            if ( isBalkingEnabled() == true )
            {
               aStatus = BALKED;
            }
            else
            {
               aStatus = UNAVAILABLE;
            }
         }
      }
      else 
      {
         //
         // the semaphore is not locked
         //

         aStatus = UNAVAILABLE;
      }
      return aStatus;
   }

   //
   // Call release for lock
   //

   SemaphoreOperationStatus EventSemaphore::release( void )
      throw( SemaphoreException )
   {
      GUARD;
      SemaphoreOperationStatus   aStatus( SUCCESS );

      //
      // Take the fast way if possible
      //

      aStatus = setLock( IPC_NOWAIT );

      return aStatus;
   }

   //
   // Set number of listeners limit
   //

   void EventSemaphore::setLimit( Counter aLimit )
      throw( SemaphoreException )
   {
      if ( Thread::getThreadIdentifier() == getOwnerId() )
      {
         theMaxListeners = aLimit;
      }
      else
      {
         throw SemaphoreException 
            (
               "Not authorized to set limit", 
               LOCATION, 
               Exception::CONTINUABLE
            );
      }
   }

   //
   // Get number of listeners limit
   //

   Counter EventSemaphore::getLimit( void ) const
   {
      return theMaxListeners;
   }
}

/*
   Common rcs information do not modify
   $Author: dulimart $
   $Revision: 1.4 $
   $Date: 2000/11/15 22:44:14 $
   $Locker:  $
*/


