/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.eviction;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.Region;

/**
 * First-in-first-out algorithm used to evict nodes.
 *
 * @author Daniel Huang - dhuang@jboss.org
 * @author Morten Kvistgaard
 * @version $Revision: 5772 $
 */
public class FIFOAlgorithm extends BaseEvictionAlgorithm
{
   private static final Log log = LogFactory.getLog(FIFOAlgorithm.class);
   private static final boolean trace = log.isTraceEnabled();


   @Override
   protected EvictionQueue setupEvictionQueue(Region region) throws EvictionException
   {
      return new FIFOQueue();
   }

   /**
    * For FIFO, a node should be evicted if the queue size is >= to the configured maxNodes size.
    */
   @Override
   protected boolean shouldEvictNode(NodeEntry ne)
   {
      // check the minimum time to live and see if we should not evict the node.  This check will
      // ensure that, if configured, nodes are kept alive for at least a minimum period of time.
      if (isYoungerThanMinimumTimeToLive(ne)) return false;

      FIFOConfiguration config = (FIFOConfiguration) region.getEvictionPolicyConfig();
      if (trace)
      {
         log.trace("Deciding whether node in queue " + ne.getFqn() + " requires eviction.");
      }

      int size = this.getEvictionQueue().getNumberOfNodes();
      return config.getMaxNodes() != 0 && size > config.getMaxNodes();
   }
}

