/*
Copyright (c) 2007, Dennis M. Sosnoski.
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.
 * Neither the name of JiBX nor the names of its contributors may be used
   to endorse or promote products derived from this software without specific
   prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package org.jibx.runtime.impl;

import java.io.IOException;

/**
 * Interface used to enhance an output stream with a separate method for writing
 * the last chunk of data in a message. This is useful when working with a
 * chunking transport layer, so that the output stream knows which particular
 * write operation is the last one in the sequence.
 *
 * @author Dennis M. Sosnoski
 */
public interface IChunkedOutput
{
    /**
     * Normal write from byte array. This method is included only to make it
     * clear that the {@link #writeFinal(byte[], int, int)} is an alternative to
     * the normal write.
     * 
     * @param b byte array
     * @param off starting offset
     * @param len number of bytes to write
     * @throws IOException on write error
     */
    void write(byte[] b, int off, int len) throws IOException;
    
    /**
     * Write last data chunk from byte array. This method finishes output. It is
     * an error if any further write operations (using either the {@link
     * #write(byte[], int, int)} method defined in this interface, or any of the
     * other <code>java.io.OutputStream</code> methods) are executed after the
     * call to this method.
     *
     * @param b byte array
     * @param off starting offset
     * @param len number of bytes to write
     * @throws IOException on write error
     */
    void writeFinal(byte[] b, int off, int len) throws IOException;
}
