/*
Copyright (c) 2006, Dennis M. Sosnoski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.
 * Neither the name of JiBX nor the names of its contributors may be used
   to endorse or promote products derived from this software without specific
   prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package org.jibx.schema.elements;

import java.io.IOException;

import org.jibx.binding.util.StringArray;
import org.jibx.runtime.BindingDirectory;
import org.jibx.runtime.IBindingFactory;
import org.jibx.runtime.IUnmarshallingContext;
import org.jibx.runtime.JiBXException;
import org.jibx.schema.ISchemaResolver;
import org.jibx.schema.validation.ValidationContext;

/**
 * Base class for elements referencing an external schema. During prevalidation
 * this first reads the referenced schema, so that it'll automatically be
 * included in the prevalidation pass.
 *
 * @author Dennis M. Sosnoski
 */
public abstract class SchemaLocationBase extends AnnotatedBase
{
	/** List of allowed attribute names. */
    public static final StringArray s_allowedAttributes =
        new StringArray(new String[] { "schemaLocation" },
        AnnotatedBase.s_allowedAttributes);
    
    //
    // Instance data
    
    /** 'schemaLocation' attribute value. */
    private String m_location;
    
    /** Referenced schema definition. */
    private SchemaElement m_schema;
    
    /**
     * Constructor.
     * 
     * @param type element type
     */
    protected SchemaLocationBase(int type) {
		super(type);
	}

    /**
     * Load a schema from a resolver.
     *
     * @param vctx validation context
     * @param resolver
     * @return loaded schema
     * @throws JiBXException
     * @throws IOException
     */
    private SchemaElement readSchema(ValidationContext vctx,
        ISchemaResolver resolver) throws JiBXException, IOException {
        IBindingFactory factory =
            BindingDirectory.getFactory(SchemaElement.class);
        IUnmarshallingContext uctx = factory.createUnmarshallingContext();
        uctx.setDocument(resolver.getContent(), resolver.getId(), null);
        uctx.setUserContext(vctx);
        return (SchemaElement)uctx.unmarshalElement();
    }
    
    //
    // Access methods
    
    /**
     * Get 'location' attribute value.
     *
     * @return 'location' value
     */
    public String getLocation() {
        return m_location;
    }
    
    /**
     * Set 'location' attribute value.
     *
     * @param location 'location' value
     */
    public void setLocation(String location) {
        m_location = location;
    }
    
    /**
     * Get referenced schema. This method is only usable after prevalidation.
     *
     * @return schema (<code>null</code> if loading failed)
     */
    public SchemaElement getReferencedSchema() {
        return m_schema;
    }
    
    //
    // Base class overrides

    /* (non-Javadoc)
     * @see org.jibx.schema.elements.AnnotatedBase#prevalidate(org.jibx.schema.validation.ValidationContext)
     */
    public void prevalidate(ValidationContext vctx) {
        if (m_location == null) {
            vctx.addFatal("Missing required 'schemaLocation' value", this);
        } else {
            try {
                
                // find or load referenced schema
                ISchemaResolver resolver =
                    vctx.getCurrentSchema().getResolver().resolve(m_location);
                SchemaElement schema = vctx.getSchema(resolver.getId());
                if (schema == null) {
                    schema = readSchema(vctx, resolver);
                    vctx.setSchema(resolver.getId(), schema);
                    schema.setResolver(resolver);
                }
                m_schema = schema;
                
            } catch (JiBXException e) {
                vctx.addFatal("Error loading schema: " + e.getMessage(), this);
            } catch (IOException e) {
                vctx.addFatal("Error loading schema: " + e.getMessage(), this);
            }
        }
        super.prevalidate(vctx);
    }
}