/* ***************************************************************************
 *
 * Pico Technology USB Device Driver
 *
 *//**
 * \file      TC08Api.h
 * \brief     C API to the USBTC08 driver
 **//*
 *
 * Copyright (c) 2007, Pico Technology.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * The name of Pico Technology may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Version $Id: TC08Api.h,v 1.5 2007/08/02 09:46:43 douglas Exp $
 *
 *************************************************************************** */

#ifdef PREF0
  #undef PREF0
#endif

#ifdef __cplusplus
  #define PREF0 extern "C" 
#else
  #define PREF0
#endif

/* ***************************************************************************
 * DEFINES
 * ************************************************************************* */
 
// Version number of this driver
#define USBTC08_DRIVER_VERSION_MAJOR		1
#define USBTC08_DRIVER_VERSION_MINOR		7
#define USBTC08_DRIVER_VERSION_REVISION		1

// Maximum lengths of strings
#define USBTC08_MAX_DATE_CHARS      9
#define USBTC08_MAX_SERIAL_CHARS   11
#define USBTC08_MAX_INFO_CHARS    256
#define USBTC08_MAX_DATE_CHARS      9
#define USBTC08_MAX_SERIAL_CHARS   11
#define USBTC08_MAX_VERSION_CHARS  25

#define USBTC08_MAX_SAMPLE_BUFFER  600 /* Max readings that usb_tc08_get_temp will ever return */

/* ***************************************************************************
 * ENUMS
 *************************************************************************** */

/// Mains frequency for setting up noise rejection with SetMains. This changes 
/// the sampling time of the ADC in the TC08 to be a whole number of mains 
/// cycles, ensuring that mains-related interference is cancelled out.
enum USBTC08_MAINS_FREQUENCY {
	USBTC08_MAINS_50HZ, ///< Reject 50Hz mains noise. Used in the EU.
	USBTC08_MAINS_60HZ, ///< Reject 60Hz mains noise. Used in the USA.
	USBTC08_MAINS_MAX
};


/// Channel setup modes
enum USBTC08_TCTYPES {
	// Thermocouples
	USBTC08_TCTYPE_B='B',
	USBTC08_TCTYPE_E='E',
	USBTC08_TCTYPE_J='J',
	USBTC08_TCTYPE_K='K',
	USBTC08_TCTYPE_N='N',
	USBTC08_TCTYPE_R='R',
	USBTC08_TCTYPE_S='S',
	USBTC08_TCTYPE_T='T',
	// Other
	USBTC08_TCTYPE_OFF=' ',     ///< Channel is disabled
	USBTC08_TCTYPE_CJC='C',     ///< Used for the CJC only
	USBTC08_TCTYPE_VOLTAGE='X'  ///< Voltage in mV, no CJC correction
};

/// LED colours for SetLED
enum USBTC08_LED_STATES {
	USBTC08_LED_OFF,
	USBTC08_LED_RED,
	USBTC08_LED_GREEN,
	USBTC08_LED_MAX
};


/// Operating states of the unit
enum USBTC08_STATES {
	USBTC08_STATE_DISCONNECTED=-2,  ///< Unit has been unplugged since the TC08Device was created
	USBTC08_STATE_LOCKED=-1,        ///< Unit is open by another process
	USBTC08_STATE_CLOSED=0,         ///< Unit is connected and available for attempts at opening
	USBTC08_STATE_IDLE=1,           ///< Unit is open and not streaming
	USBTC08_STATE_STREAMING=2       ///< Unit is continuously collecting data
};

/// List of input channels
typedef enum enUSBTC08Channels
{
	USBTC08_CHANNEL_CJC = 0,
	USBTC08_CHANNEL_1   = 1,
	USBTC08_CHANNEL_2   = 2,
	USBTC08_CHANNEL_3   = 3,
	USBTC08_CHANNEL_4   = 4,
	USBTC08_CHANNEL_5   = 5,
	USBTC08_CHANNEL_6   = 6,
	USBTC08_CHANNEL_7   = 7,
	USBTC08_CHANNEL_8   = 8,
	USBTC08_CHANNEL_COUNT=9,
	USBTC08_MAX_CHANNELS = USBTC08_CHANNEL_8
} USBTC08_CHANNELS;

typedef enum enUSBTC08Progress
  {
  USBTC08_PROGRESS_FAIL     = -1,
  USBTC08_PROGRESS_PENDING  =  0,
  USBTC08_PROGRESS_COMPLETE =  1
  } USBTC08_PROGRESS;

typedef enum enUSBTC08Error
  {
  USBTC08_ERROR_OK,          
  USBTC08_ERROR_OS_NOT_SUPPORTED,       /* need to use win98 (or later) or win2k (or later) */
  USBTC08_ERROR_NO_CHANNELS_SET,
  USBTC08_ERROR_INVALID_PARAMETER,      /* one or more of the function arguments was invalid */
  USBTC08_ERROR_VARIANT_NOT_SUPPORTED,  /* the hardware version is not supported (download the latest driver) */
  USBTC08_ERROR_INCORRECT_MODE,         /* an incompatible mix of legacy and non-legacy functions was called 
                                         * ... or usb_tc08_get_single was called while streaming was active */
  USBTC08_ERROR_ENUMERATION_INCOMPLETE, /* a second call to usb_tc08_open_unit_async() was made before
                                           asynchronous enumeration had completed */
  USBTC08_ERROR_NOT_RESPONDING,         /* unit is not responding to the driver */
  USBTC08_ERROR_FW_FAIL,                /* unable to download firmware */
  USBTC08_ERROR_CONFIG_FAIL,            /* missing or corrupted eeprom */
  USBTC08_ERROR_NOT_FOUND,              /* cannot find enumerated device */
  USBTC08_ERROR_THREAD_FAIL,            /* a threading function failed */
  USBTC08_ERROR_PIPE_INFO_FAIL,         /* could not get pipes from the device */
  USBTC08_ERROR_NOT_CALIBRATED,         /* no calibration date was found */
  USBTC08_EROOR_PICOPP_TOO_OLD,
  USBTC08_ERROR_COMMUNICATION           /*the PC has lost communication with unit*/
  }	USBTC08_ERROR;

/// Temperature units for converting readings
typedef enum enUSBTC08Units
  {
  USBTC08_UNITS_CENTIGRADE,
  USBTC08_UNITS_FAHRENHEIT,
  USBTC08_UNITS_KELVIN,
  USBTC08_UNITS_RANKINE,
  USBTC08_UNITS_MAX = USBTC08_UNITS_RANKINE, /* Mac */
  USBTC08_MAX_UNITS = USBTC08_UNITS_RANKINE /* Win32 */
  } USBTC08_UNITS;

typedef enum enUBTC08InfoLine
{
  USBTC08LINE_DRIVER_VERSION,
  USBTC08LINE_KERNEL_DRIVER_VERSION,
  USBTC08LINE_HARDWARE_VERSION,
  USBTC08LINE_VARIANT_INFO,
  USBTC08LINE_BATCH_AND_SERIAL,
  USBTC08LINE_CAL_DATE,
} USBTC08_INFO_LINE;


/* ***************************************************************************
 * Structures
 *************************************************************************** */

typedef struct tUSBTC08Info
  {
  short size;
  char DriverVersion[USBTC08_MAX_VERSION_CHARS];
  short PicoppVersion;
  short HardwareVersion;
  short Variant;
  char  szSerial[USBTC08_MAX_SERIAL_CHARS];
  char  szCalDate[USBTC08_MAX_DATE_CHARS];
  } USBTC08_INFO, *LPUSBTC08_INFO;



/* ***************************************************************************
 *     USB TC08 API Calls
 *     These functions are the same as the (non-legacy) Windows API.
 *************************************************************************** */


PREF0 short usb_tc08_open_unit (void);
PREF0 short usb_tc08_open_unit_async (void);
PREF0 short usb_tc08_open_unit_progress (short * handle, short * percent_progress);
PREF0 short usb_tc08_close_unit (short handle);
PREF0 short usb_tc08_stop (short handle);

PREF0 short usb_tc08_set_mains (short handle, short sixty_hertz);
PREF0 long  usb_tc08_get_minimum_interval_ms (short handle);
PREF0 short usb_tc08_set_channel ( 
  short handle,
  short channel,
  char  tc_type);

PREF0 long usb_tc08_run (short handle, long interval_ms);

PREF0 long usb_tc08_get_temp ( 
  short   handle,
  float * temp_buffer,
  long  * times_ms_buffer,
  long    buffer_length,
  short * overflow,
  short   channel,
  short   units,
  short   fill_missing);

PREF0 long usb_tc08_get_temp_deskew ( 
  short   handle,
  float * temp_buffer,
  long  * times_ms_buffer,
  long    buffer_length,
  short * overflow,
  short   channel,
  short   units,
  short   fill_missing);

PREF0 short usb_tc08_get_single (
  short   handle, 
  float * temp,
  short * overflow_flags,
  short   units);

PREF0 short usb_tc08_get_unit_info (short handle, USBTC08_INFO * info);
PREF0 short usb_tc08_get_unit_info2 (
	short			  handle,
	char * 			string,
	short 			string_length,
	short 			line
	);
PREF0 short usb_tc08_get_formatted_info ( 
  short   handle,
  char *  unit_info,
  short   string_length);

PREF0 short usb_tc08_get_last_error (short handle);

