// license:BSD-3-Clause
// copyright-holders:Dan Boris, Fabio Priuli, Mike Saarna, Robert Tuccitto
/***************************************************************************

  a7800.c

  Driver file to handle emulation of the Atari 7800.

  Dan Boris

    2002/05/13 kubecj   added more banks for bankswitching
                        added PAL machine description
                        changed clock to be precise
                        improved cart emulation (in machine/)

    2012/10/25 Robert Tuccitto  NTSC Color Generator utilized for
                color palette with hue shift/start
                based on observation of several
                systems across multiple displays

    2012/11/09 Robert Tuccitto  Fixed 3 degree hue begin point
                miscalculation of color palette

    2012/12/05 Robert Tuccitto  Implemented proper IRE and phase
               value to the palette

    2012/12/14 Robert Tuccitto  Adjusted colorburst/tint/hue of entire
               palette to closer reflect default hardware configuration
               setting of ~180 degrees.  Palette settings now correspond
               documented and calculated settings as follows:

               Contrast = 0.0526 --> 0.05
               Brightness = 0.0 --> 0.00
               Color = 0.2162 --> 0.22
               Phase = 25.714 --> 25.7
               Colorburst/Hue = 180 degrees

    2013/02/27 Robert Tuccitto  Palette rebuild due to misaligned
               data references.  Corrected PAL color sequence order.

    2013/03/19 Robert Tuccitto  Stripped palette to raw video output
               values removing YIQ/YUV infer data.

    2013/04/02 Robert Tuccitto  Corrected rotation values and errors
               including duplicate entries for palette.

    2013/04/07 Robert Tuccitto  Address map locations for the XBOARD
               added.

    2013/05/01 Robert Tuccitto  Red and Blue miscalculated proportions
               fixed.

    2013/08/04 Robert Tuccitto  Green miscalculated proportions fixed.

    2013/08/13 Robert Tuccitto  Normalized contrast and brightness,
               providing a standardize grayscale and adjusted color values.

    2013/09/02 Robert Tuccitto  Stored data for 26.7 & 27.7 phase shifts
               with corrections and label for 25.7 values. Made 26.7
               (medium) default. Phase shifting falls outside the realm of
               video controls and hope to implement a selectable toggle
               hardware option similar to Donkey Kong TKG02/TKG04.

    2013/09/19 Robert Tuccitto  Cleanup of Address Maps, high score maps
               added.

    2013/10/16 Robert Tuccitto  Added Phase Shifts 24.7, 25.2, 26.2, 27.2.
               Phase Shifts 24.7 through 27.7 degrees with 0.5 degree
               increments documented. Phase Shift 26.2 degrees made active.
               Fixed typo under 26.7 7$.

    2013/10/27 Robert Tuccitto  Modernized screen parameters for NTSC & PAL.

    2013/11/03 Robert Tuccitto  Fixed correctly typo under 26.7 7$.

    2013/11/23 Robert Tuccitto  Added NTSC Palette Notes.

    2014/01/02 Robert Tuccitto  Corrected joystick buttons assignment & minor
                                palette notes cleanup.

    2014/01/09 Robert Tuccitto  Positional description for difficulty
                                switches added.

    2014/02/15 Robert Tuccitto  Added more details and clarification
                                regarding the potentiometer.

    2014/03/25 Mike Saarna  Fixed Riot Timer

    2014/04/04 Mike Saarna  Fix to controller button RIOT behavior

    2014/05/06 Mike Saarna/Robert Tuccitto Brought initial Maria cycle counts
               inline from measurements taken with logic analyzer and tests.

    2014/08/25 Fabio Priuli Converted carts to be slot devices and cleaned
               up the driver (removed the pokey, cleaned up rom regions, etc.)

***************************************************************************/

#include "emu.h"
#include "cpu/m6502/m6502.h"
#include "machine/timer.h"
#include "sound/tiaintf.h"
#include "sound/tiasound.h"
#include "machine/mos6530n.h"
#include "video/maria.h"
#include "bus/a7800/a78_carts.h"
#include "screen.h"
#include "softlist.h"
#include "speaker.h"

#define A7800_NTSC_Y1   XTAL(14'318'181)
#define CLK_PAL 1773447


class a7800_state : public driver_device
{
public:
	a7800_state(const machine_config &mconfig, device_type type, const char *tag) :
		driver_device(mconfig, type, tag),
		m_maincpu(*this, "maincpu"),
		m_tia(*this, "tia"),
		m_maria(*this, "maria"),
		m_io_joysticks(*this, "joysticks"),
		m_io_buttons(*this, "buttons"),
		m_io_console_buttons(*this, "console_buttons"),
		m_cart(*this, "cartslot"),
		m_screen(*this, "screen"),
		m_bios(*this, "maincpu")
	{
	}

	void a7800_ntsc(machine_config &config);

protected:
	DECLARE_READ8_MEMBER(bios_or_cart_r);
	DECLARE_READ8_MEMBER(tia_r);
	DECLARE_WRITE8_MEMBER(tia_w);
	DECLARE_PALETTE_INIT(a7800);
	TIMER_DEVICE_CALLBACK_MEMBER(interrupt);
	TIMER_CALLBACK_MEMBER(maria_startdma);
	DECLARE_READ8_MEMBER(riot_joystick_r);
	DECLARE_READ8_MEMBER(riot_console_button_r);
	DECLARE_WRITE8_MEMBER(riot_button_pullup_w);

	virtual void machine_start() override;
	virtual void machine_reset() override;
	void a7800_mem(address_map &map);

	int m_lines;
	int m_ispal;

	int m_ctrl_lock;
	int m_ctrl_reg;
	int m_maria_flag;
	int m_p1_one_button;
	int m_p2_one_button;
	int m_bios_enabled;

private:
	required_device<cpu_device> m_maincpu;
	required_device<tia_device> m_tia;
	required_device<atari_maria_device> m_maria;
	required_ioport m_io_joysticks;
	required_ioport m_io_buttons;
	required_ioport m_io_console_buttons;
	required_device<a78_cart_slot_device> m_cart;
	required_device<screen_device> m_screen;
	required_region_ptr<uint8_t> m_bios;
};

class a7800_ntsc_state : public a7800_state
{
public:
	using a7800_state::a7800_state;
	DECLARE_DRIVER_INIT(a7800_ntsc);
};

class a7800_pal_state : public a7800_state
{
public:
	using a7800_state::a7800_state;
	DECLARE_DRIVER_INIT(a7800_pal);
	void a7800_pal(machine_config &config);

protected:
	DECLARE_PALETTE_INIT(a7800p);
};


/***************************************************************************
 MEMORY HANDLERS
 ***************************************************************************/

// RIOT
READ8_MEMBER(a7800_state::riot_joystick_r)
{
	return m_io_joysticks->read();
}

READ8_MEMBER(a7800_state::riot_console_button_r)
{
	return m_io_console_buttons->read();
}

WRITE8_MEMBER(a7800_state::riot_button_pullup_w)
{
	if(m_maincpu->space(AS_PROGRAM).read_byte(0x283) & 0x04)
		m_p1_one_button = data & 0x04; // pin 6 of the controller port is held high by the riot chip when reading two-button controllers (from schematic)
	if(m_maincpu->space(AS_PROGRAM).read_byte(0x283) & 0x10)
		m_p2_one_button = data & 0x10;
}

READ8_MEMBER(a7800_state::tia_r)
{
	switch (offset & 0x0f)
	{
		case 0x00:
		case 0x01:
		case 0x02:
		case 0x03:
		case 0x04:
		case 0x05:
		case 0x06:
		case 0x07:
			/* Even though the 7800 doesn't use the TIA graphics the collision registers should
			 still return a reasonable value */
			return 0x00;
		case 0x08:
			return ((m_io_buttons->read() & 0x02) << 6);
		case 0x09:
			return ((m_io_buttons->read() & 0x08) << 4);
		case 0x0a:
			return ((m_io_buttons->read() & 0x01) << 7);
		case 0x0b:
			return ((m_io_buttons->read() & 0x04) << 5);
		case 0x0c:
			if (((m_io_buttons->read() & 0x08) ||(m_io_buttons->read() & 0x02)) && m_p1_one_button)
				return 0x00;
			else
				return 0x80;
		case 0x0d:
			if (((m_io_buttons->read() & 0x01) ||(m_io_buttons->read() & 0x04)) && m_p2_one_button)
				return 0x00;
			else
				return 0x80;
		default:
			logerror("undefined TIA read %x\n",offset);

	}
	return 0xff;
}

// TIA
WRITE8_MEMBER(a7800_state::tia_w)
{
	if (offset < 0x20)
	{ //INPTCTRL covers TIA registers 0x00-0x1F until locked
		if (data & 0x01)
		{
			if (m_ctrl_lock && offset == 0x01)
				m_maria_flag = 1;
			else if (!m_ctrl_lock)
				m_maria_flag = 1;
		}
		if (!m_ctrl_lock)
		{
			m_ctrl_lock = data & 0x01;
			m_ctrl_reg = data;
		}
	}
	m_tia->tia_sound_w(space, offset, data);
}


// TIMERS

TIMER_DEVICE_CALLBACK_MEMBER(a7800_state::interrupt)
{
	// DMA Begins 7 cycles after hblank
	machine().scheduler().timer_set(m_maincpu->cycles_to_attotime(7), timer_expired_delegate(FUNC(a7800_state::maria_startdma),this));
	m_maria->interrupt(m_lines);
}

TIMER_CALLBACK_MEMBER(a7800_state::maria_startdma)
{
	m_maria->startdma(m_lines);
}



// ROM
READ8_MEMBER(a7800_state::bios_or_cart_r)
{
	if (!(m_ctrl_reg & 0x04))
		return m_bios[offset];
	else
		return m_cart->read_40xx(space, offset + 0x8000);
}

/***************************************************************************
    ADDRESS MAPS
***************************************************************************/

ADDRESS_MAP_START(a7800_state::a7800_mem)
	AM_RANGE(0x0000, 0x001f) AM_MIRROR(0x300) AM_READWRITE(tia_r, tia_w)
	AM_RANGE(0x0020, 0x003f) AM_MIRROR(0x300) AM_DEVREADWRITE("maria", atari_maria_device, read, write)
	AM_RANGE(0x0040, 0x00ff) AM_RAMBANK("zpmirror") // mirror of 0x2040-0x20ff, for zero page
	AM_RANGE(0x0140, 0x01ff) AM_RAMBANK("spmirror") // mirror of 0x2140-0x21ff, for stack page
	AM_RANGE(0x0280, 0x029f) AM_MIRROR(0x160) AM_DEVICE("riot", mos6532_new_device, io_map)
	AM_RANGE(0x0480, 0x04ff) AM_MIRROR(0x100) AM_DEVICE("riot", mos6532_new_device, ram_map)
	AM_RANGE(0x1800, 0x1fff) AM_RAM AM_SHARE("6116_1")
	AM_RANGE(0x2000, 0x27ff) AM_RAM AM_SHARE("6116_2")
								// According to the official Software Guide, the RAM at 0x2000 is
								// repeatedly mirrored up to 0x3fff, but this is evidently incorrect
								// because the High Score Cartridge maps ROM at 0x3000-0x3fff
								// Hardware tests show that only the page at 0x2700 appears at
								// 0x2800, and only on some hardware (MARIA? motherboard?) revisions,
								// and even then with inconsistent and unreliable results.
	AM_RANGE(0x4000, 0xffff) AM_DEVWRITE("cartslot", a78_cart_slot_device, write_40xx)
	AM_RANGE(0x4000, 0xbfff) AM_DEVREAD("cartslot", a78_cart_slot_device, read_40xx)
	AM_RANGE(0xc000, 0xffff) AM_READ(bios_or_cart_r)    // here also the BIOS can be accessed
ADDRESS_MAP_END


/***************************************************************************
    INPUT PORTS
***************************************************************************/

static INPUT_PORTS_START( a7800 )
	PORT_START("joysticks")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP)    PORT_PLAYER(2) PORT_8WAY
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN)  PORT_PLAYER(2) PORT_8WAY
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT)  PORT_PLAYER(2) PORT_8WAY
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT) PORT_PLAYER(2) PORT_8WAY
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_UP)    PORT_PLAYER(1) PORT_8WAY
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN)  PORT_PLAYER(1) PORT_8WAY
	PORT_BIT(0x40, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT)  PORT_PLAYER(1) PORT_8WAY
	PORT_BIT(0x80, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT) PORT_PLAYER(1) PORT_8WAY

	PORT_START("buttons")
	PORT_BIT(0x01, IP_ACTIVE_HIGH, IPT_BUTTON2)       PORT_PLAYER(2)
	PORT_BIT(0x02, IP_ACTIVE_HIGH, IPT_BUTTON2)       PORT_PLAYER(1)
	PORT_BIT(0x04, IP_ACTIVE_HIGH, IPT_BUTTON1)       PORT_PLAYER(2)
	PORT_BIT(0x08, IP_ACTIVE_HIGH, IPT_BUTTON1)       PORT_PLAYER(1)
	PORT_BIT(0xF0, IP_ACTIVE_LOW, IPT_UNUSED)

	PORT_START("console_buttons")
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_OTHER)  PORT_NAME("Reset")         PORT_CODE(KEYCODE_R)
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_OTHER)  PORT_NAME("Select")        PORT_CODE(KEYCODE_S)
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_OTHER)  PORT_NAME(DEF_STR(Pause))  PORT_CODE(KEYCODE_O)
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_UNUSED)
	PORT_DIPNAME(0x40, 0x40, "Left Difficulty Switch")
	PORT_DIPSETTING(0x40, "A - Right Position" )
	PORT_DIPSETTING(0x00, "B - Left Position" )
	PORT_DIPNAME(0x80, 0x80, "Right Difficulty Switch")
	PORT_DIPSETTING(0x80, "A - Right Position" )
	PORT_DIPSETTING(0x00, "B - Left Position" )
INPUT_PORTS_END

/***************************************************************************
Atari 7800 NTSC Palette Notes:

Palette on a modern flat panel display (LCD, LED, Plasma, etc.) appears
different from a traditional CRT. The most outstanding difference is Hue 1x,
the hue begin point. Hue 1x looks very 'green' (~-60 to -45 degrees -
depending on how poor or well it handles the signal conversion and its
calibration) on a modern flat panel display, as opposed to 'gold' (~-33
degrees) on a CRT.

The potentiometer (pot adjustment) for the 7800 modifies the delay line
regarding colors it will exhibit and is extremely sensitive.  The slightest
turn can have a significant impact.

A system whose potentiometer is not properly calibrated via
'CPS 7800 Diagnostic Test Cartridge' or/and just slightly off from the
desired factory settings may exhibit consequences such as too much blue in
place of green (Pot adjusted slightly too far left) or washed out reddish
tones in place of where most other systems display a darker reddish tones
(Pot adjusted slightly too far right).

This is a result of the phase shifting of lesser degrees (Pot adjusted more
to the left) or phase shifting of greater degrees (Pot adjusted more to the
right).

Turning the pot adjustment to the right, it can be observed that the values
of the higher end of the scale will match the lower end of the scale.
For example, after some turning to the right, the values of Dx, Ex, Fx,
can be set to match 1x, 2x, 3x.

After further turning to the right, now the palette can be brought to make
Ax, Bx, Cx, Dx, Ex, Fx will match 1x, 2x, 3x, 4x, 5x, 6x.

Ultimately though, too much turning to the right results in all colors being
wiped from the scale, excluding the hue begin point 1x (Which remains
unchanged while tweaking the potentiometer either left or right).

Continuously turning the pot adjustment to the left, red and blue become the
most dominant two colors encompassing the palette with only a slight
influence of green at the highest end of the palette (Fx), once turned all
the way leftward.

The degree range for adjustment of the phase shifting on the 7800 appears
to be as low as approximately 15 degrees when tuned all the way left, and
seems to be able to achieve as high as approximately 45 degrees when turned
right before losing all color (Excluding 1x) from the palette scale.

For even a properly calibrated system at power on, the system's phase
shift appears as low as ~23 degrees and after a considerable consistent
runtime ('warm-up'), can be as high as ~28 degrees.

In general, the low end of ~23 degrees lasts for maybe several seconds,
whereas higher values such as ~25-27 degrees is the most dominant during
system run time.  180 degrees colorburst takes place at ~25.7 degrees (A
near exact match of Hue 1x and 15x - To the naked eye they appear to be
the same).

However, consistent system run time causes Hue 15x (F$) to become
stronger/darker gold (More brown then ultimately red-brown); as well
as leans Hue 14x (E$) more brown than green.  Once achieving a phase shift
of 27.7, Hue 14x (E$) and Hue 15x (F$) near-exact match Hue 1x and 2x
respectively.

Therefore, an ideal phase shift while accounting for the reality of
shifting while warming up, as well as maintaining differences between 1x,
2x and 14x, 15x, would likely fall between a 25.7 and 27.7. Phase shifts
26.2 degrees and 26.7 degrees places Hue 15x (F$) between Hue 1x and
Hue 2x, having 26.2 degrees leaning closer to Hue 1x and 26.7 degrees
leaning closer to Hue 2x.

The above notion would also harmonize with what has been documented for
the colors of 1x, 2x, 14x, 15x on the 7800.  1x = Gold, 2x = Orange,
14x (E$) = Orange-Green. 15x (F$) = Light Orange.  Color descriptions are
best measured in the middle of the brightness scale.  It should be
mentioned that Green-Yellow is referenced at Hue 13x (D$), nowhere near
Hue 1x.  A Green-Yellow Hue 1x is how the palette is manipulated and
modified (in part) under a modern flat panel display.

Additionally, the blue to red (And consequently blue to green) ratio
proportions may appear different on a modern flat panel display than a CRT
in some instances for the Atari 7800 system.  Furthermore, you may have
some variation of proportions even within the same display type.

One side effect of this on the console's palette is that some values of
red may appear too pinkish - Too much blue to red.  This is not the same
as a traditional tint-hue control adjustment; rather, can be demonstrated
by changing the blue ratio values via MESS HLSL settings.

Lastly, the Atari 2600 & 5200 NTSC color palettes hold the same hue
structure order and have similar appearance differences that are dependent
upon display type.
***************************************************************************/

/***************************************************************************
    PALETTE - 25.7 PHASE SHIFT
***************************************************************************/

#define NTSC_GREY \
	rgb_t(0x00,0x00,0x00), rgb_t(0x11,0x11,0x11), rgb_t(0x22,0x22,0x22), rgb_t(0x33,0x33,0x33), \
	rgb_t(0x44,0x44,0x44), rgb_t(0x55,0x55,0x55), rgb_t(0x66,0x66,0x66), rgb_t(0x77,0x77,0x77), \
	rgb_t(0x88,0x88,0x88), rgb_t(0x99,0x99,0x99), rgb_t(0xAA,0xAA,0xAA), rgb_t(0xBB,0xBB,0xBB), \
	rgb_t(0xCC,0xCC,0xCC), rgb_t(0xDD,0xDD,0xDD), rgb_t(0xEE,0xEE,0xEE), rgb_t(0xFF,0xFF,0xFF   )

#define NTSC_GOLD \
	rgb_t(0x1A,0x07,0x00), rgb_t(0x2B,0x18,0x00), rgb_t(0x3C,0x29,0x00), rgb_t(0x4D,0x3A,0x00), \
	rgb_t(0x5E,0x4B,0x00), rgb_t(0x6F,0x5C,0x00), rgb_t(0x80,0x6D,0x00), rgb_t(0x91,0x7E,0x09), \
	rgb_t(0xA2,0x8F,0x1A), rgb_t(0xB3,0xA0,0x2B), rgb_t(0xC4,0xB1,0x3C), rgb_t(0xD5,0xC2,0x4D), \
	rgb_t(0xE6,0xD3,0x5E), rgb_t(0xF7,0xE4,0x6F), rgb_t(0xFF,0xF5,0x83), rgb_t(0xFF,0xF7,0x97   )

#define NTSC_ORANGE \
	rgb_t(0x31,0x00,0x00), rgb_t(0x42,0x06,0x00), rgb_t(0x53,0x17,0x00), rgb_t(0x64,0x28,0x00), \
	rgb_t(0x75,0x39,0x00), rgb_t(0x86,0x4A,0x00), rgb_t(0x97,0x5B,0x0A), rgb_t(0xA8,0x6C,0x1B), \
	rgb_t(0xB9,0x7D,0x2C), rgb_t(0xCA,0x8E,0x3D), rgb_t(0xDB,0x9F,0x4E), rgb_t(0xEC,0xB0,0x5F), \
	rgb_t(0xFD,0xC1,0x70), rgb_t(0xFF,0xD2,0x85), rgb_t(0xFF,0xE3,0x9C), rgb_t(0xFF,0xF4,0xB2   )

#define NTSC_RED_ORANGE \
	rgb_t(0x3E,0x00,0x00), rgb_t(0x4F,0x00,0x00), rgb_t(0x60,0x08,0x00), rgb_t(0x71,0x19,0x00), \
	rgb_t(0x82,0x2A,0x0D), rgb_t(0x93,0x3B,0x1E), rgb_t(0xA4,0x4C,0x2F), rgb_t(0xB5,0x5D,0x40), \
	rgb_t(0xC6,0x6E,0x51), rgb_t(0xD7,0x7F,0x62), rgb_t(0xE8,0x90,0x73), rgb_t(0xF9,0xA1,0x83), \
	rgb_t(0xFF,0xB2,0x98), rgb_t(0xFF,0xC3,0xAE), rgb_t(0xFF,0xD4,0xC4), rgb_t(0xFF,0xE5,0xDA   )

#define NTSC_PINK \
	rgb_t(0x3F,0x00,0x03), rgb_t(0x50,0x00,0x0F), rgb_t(0x61,0x00,0x1B), rgb_t(0x72,0x0F,0x2B), \
	rgb_t(0x83,0x20,0x3C), rgb_t(0x94,0x31,0x4D), rgb_t(0xA5,0x42,0x5E), rgb_t(0xB6,0x53,0x6F), \
	rgb_t(0xC7,0x64,0x80), rgb_t(0xD8,0x75,0x91), rgb_t(0xE9,0x86,0xA2), rgb_t(0xFA,0x97,0xB3), \
	rgb_t(0xFF,0xA8,0xC8), rgb_t(0xFF,0xB9,0xDE), rgb_t(0xFF,0xCA,0xEF), rgb_t(0xFF,0xDB,0xF4   )

#define NTSC_PURPLE \
	rgb_t(0x33,0x00,0x35), rgb_t(0x44,0x00,0x41), rgb_t(0x55,0x00,0x4C), rgb_t(0x66,0x0C,0x5C), \
	rgb_t(0x77,0x1D,0x6D), rgb_t(0x88,0x2E,0x7E), rgb_t(0x99,0x3F,0x8F), rgb_t(0xAA,0x50,0xA0), \
	rgb_t(0xBB,0x61,0xB1), rgb_t(0xCC,0x72,0xC2), rgb_t(0xDD,0x83,0xD3), rgb_t(0xEE,0x94,0xE4), \
	rgb_t(0xFF,0xA5,0xE4), rgb_t(0xFF,0xB6,0xE9), rgb_t(0xFF,0xC7,0xEE), rgb_t(0xFF,0xD8,0xF3   )

#define NTSC_PURPLE_BLUE \
	rgb_t(0x1D,0x00,0x5C), rgb_t(0x2E,0x00,0x68), rgb_t(0x40,0x00,0x74), rgb_t(0x51,0x10,0x84), \
	rgb_t(0x62,0x21,0x95), rgb_t(0x73,0x32,0xA6), rgb_t(0x84,0x43,0xB7), rgb_t(0x95,0x54,0xC8), \
	rgb_t(0xA6,0x65,0xD9), rgb_t(0xB7,0x76,0xEA), rgb_t(0xC8,0x87,0xEB), rgb_t(0xD9,0x98,0xEB), \
	rgb_t(0xE9,0xA9,0xEC), rgb_t(0xFB,0xBA,0xEB), rgb_t(0xFF,0xCB,0xEF), rgb_t(0xFF,0xDC,0xF4   )

#define NTSC_BLUE1 \
	rgb_t(0x02,0x00,0x71), rgb_t(0x13,0x00,0x7D), rgb_t(0x24,0x0B,0x8C), rgb_t(0x35,0x1C,0x9D), \
	rgb_t(0x46,0x2D,0xAE), rgb_t(0x57,0x3E,0xBF), rgb_t(0x68,0x4F,0xD0), rgb_t(0x79,0x60,0xE1), \
	rgb_t(0x8A,0x71,0xF2), rgb_t(0x9B,0x82,0xF7), rgb_t(0xAC,0x93,0xF7), rgb_t(0xBD,0xA4,0xF7), \
	rgb_t(0xCE,0xB5,0xF7), rgb_t(0xDF,0xC6,0xF7), rgb_t(0xF0,0xD7,0xF7), rgb_t(0xFF,0xE8,0xF8   )

#define NTSC_BLUE2 \
	rgb_t(0x00,0x00,0x68), rgb_t(0x00,0x0A,0x7C), rgb_t(0x08,0x1B,0x90), rgb_t(0x19,0x2C,0xA1), \
	rgb_t(0x2A,0x3D,0xB2), rgb_t(0x3B,0x4E,0xC3), rgb_t(0x4C,0x5F,0xD4), rgb_t(0x5D,0x70,0xE5), \
	rgb_t(0x6E,0x81,0xF6), rgb_t(0x7F,0x92,0xFF), rgb_t(0x90,0xA3,0xFF), rgb_t(0xA1,0xB4,0xFF), \
	rgb_t(0xB2,0xC5,0xFF), rgb_t(0xC3,0xD6,0xFF), rgb_t(0xD4,0xE7,0xFF), rgb_t(0xE5,0xF8,0xFF   )

#define NTSC_LIGHT_BLUE \
	rgb_t(0x00,0x0A,0x4D), rgb_t(0x00,0x1B,0x63), rgb_t(0x00,0x2C,0x79), rgb_t(0x02,0x3D,0x8F), \
	rgb_t(0x13,0x4E,0xA0), rgb_t(0x24,0x5F,0xB1), rgb_t(0x35,0x70,0xC2), rgb_t(0x46,0x81,0xD3), \
	rgb_t(0x57,0x92,0xE4), rgb_t(0x68,0xA3,0xF5), rgb_t(0x79,0xB4,0xFF), rgb_t(0x8A,0xC5,0xFF), \
	rgb_t(0x9B,0xD6,0xFF), rgb_t(0xAC,0xE7,0xFF), rgb_t(0xBD,0xF8,0xFF), rgb_t(0xCE,0xFF,0xFF   )

#define NTSC_TURQUOISE \
	rgb_t(0x00,0x1A,0x26), rgb_t(0x00,0x2B,0x3C), rgb_t(0x00,0x3C,0x52), rgb_t(0x00,0x4D,0x68), \
	rgb_t(0x06,0x5E,0x7C), rgb_t(0x17,0x6F,0x8D), rgb_t(0x28,0x80,0x9E), rgb_t(0x39,0x91,0xAF), \
	rgb_t(0x4A,0xA2,0xC0), rgb_t(0x5B,0xB3,0xD1), rgb_t(0x6C,0xC4,0xE2), rgb_t(0x7D,0xD5,0xF3), \
	rgb_t(0x8E,0xE6,0xFF), rgb_t(0x9F,0xF7,0xFF), rgb_t(0xB0,0xFF,0xFF), rgb_t(0xC1,0xFF,0xFF   )

#define NTSC_GREEN_BLUE \
	rgb_t(0x00,0x24,0x0B), rgb_t(0x00,0x35,0x10), rgb_t(0x00,0x46,0x22), rgb_t(0x00,0x57,0x38), \
	rgb_t(0x05,0x68,0x4D), rgb_t(0x16,0x79,0x5E), rgb_t(0x27,0x8A,0x6F), rgb_t(0x38,0x9B,0x80), \
	rgb_t(0x49,0xAC,0x91), rgb_t(0x5A,0xBD,0xA2), rgb_t(0x6B,0xCE,0xB3), rgb_t(0x7C,0xDF,0xC4), \
	rgb_t(0x8D,0xF0,0xD5), rgb_t(0x9E,0xFF,0xE5), rgb_t(0xAF,0xFF,0xF1), rgb_t(0xC0,0xFF,0xFD   )

#define NTSC_GREEN \
	rgb_t(0x00,0x27,0x0C), rgb_t(0x00,0x38,0x11), rgb_t(0x00,0x49,0x16), rgb_t(0x00,0x5A,0x1B), \
	rgb_t(0x10,0x6B,0x1B), rgb_t(0x21,0x7C,0x2C), rgb_t(0x32,0x8D,0x3D), rgb_t(0x43,0x9E,0x4E), \
	rgb_t(0x54,0xAF,0x5F), rgb_t(0x65,0xC0,0x70), rgb_t(0x76,0xD1,0x81), rgb_t(0x87,0xE2,0x92), \
	rgb_t(0x98,0xF3,0xA3), rgb_t(0xA9,0xFF,0xB3), rgb_t(0xBA,0xFF,0xBF), rgb_t(0xCB,0xFF,0xCB   )

#define NTSC_YELLOW_GREEN \
	rgb_t(0x00,0x23,0x0A), rgb_t(0x00,0x34,0x10), rgb_t(0x04,0x45,0x13), rgb_t(0x15,0x56,0x13), \
	rgb_t(0x26,0x67,0x13), rgb_t(0x37,0x78,0x13), rgb_t(0x48,0x89,0x14), rgb_t(0x59,0x9A,0x25), \
	rgb_t(0x6A,0xAB,0x36), rgb_t(0x7B,0xBC,0x47), rgb_t(0x8C,0xCD,0x58), rgb_t(0x9D,0xDE,0x69), \
	rgb_t(0xAE,0xEF,0x7A), rgb_t(0xBF,0xFF,0x8B), rgb_t(0xD0,0xFF,0x97), rgb_t(0xE1,0xFF,0xA3   )

#define NTSC_ORANGE_GREEN \
	rgb_t(0x00,0x17,0x07), rgb_t(0x0E,0x28,0x08), rgb_t(0x1F,0x39,0x08), rgb_t(0x30,0x4A,0x08), \
	rgb_t(0x41,0x5B,0x08), rgb_t(0x52,0x6C,0x08), rgb_t(0x63,0x7D,0x08), rgb_t(0x74,0x8E,0x0D), \
	rgb_t(0x85,0x9F,0x1E), rgb_t(0x96,0xB0,0x2F), rgb_t(0xA7,0xC1,0x40), rgb_t(0xB8,0xD2,0x51), \
	rgb_t(0xC9,0xE3,0x62), rgb_t(0xDA,0xF4,0x73), rgb_t(0xEB,0xFF,0x82), rgb_t(0xFC,0xFF,0x8E   )

#define NTSC_LIGHT_ORANGE \
	rgb_t(0x19,0x07,0x00), rgb_t(0x2A,0x18,0x00), rgb_t(0x3B,0x29,0x00), rgb_t(0x4C,0x3A,0x00), \
	rgb_t(0x5D,0x4B,0x00), rgb_t(0x6E,0x5C,0x00), rgb_t(0x7F,0x6D,0x00), rgb_t(0x90,0x7E,0x09), \
	rgb_t(0xA1,0x8F,0x1A), rgb_t(0xB2,0xA0,0x2B), rgb_t(0xC3,0xB1,0x3C), rgb_t(0xD4,0xC2,0x4D), \
	rgb_t(0xE5,0xD3,0x5E), rgb_t(0xF6,0xE4,0x6F), rgb_t(0xFF,0xF5,0x82), rgb_t(0xFF,0xFF,0x96   )

static const rgb_t a7800_palette[256*3] =
{
	NTSC_GREY,
	NTSC_GOLD,
	NTSC_ORANGE,
	NTSC_RED_ORANGE,
	NTSC_PINK,
	NTSC_PURPLE,
	NTSC_PURPLE_BLUE,
	NTSC_BLUE1,
	NTSC_BLUE2,
	NTSC_LIGHT_BLUE,
	NTSC_TURQUOISE,
	NTSC_GREEN_BLUE,
	NTSC_GREEN,
	NTSC_YELLOW_GREEN,
	NTSC_ORANGE_GREEN,
	NTSC_LIGHT_ORANGE
};

static const rgb_t a7800p_palette[256*3] =
{
	NTSC_GREY,
	NTSC_ORANGE_GREEN,
	NTSC_GOLD,
	NTSC_ORANGE,
	NTSC_RED_ORANGE,
	NTSC_PINK,
	NTSC_PURPLE,
	NTSC_PURPLE_BLUE,
	NTSC_BLUE1,
	NTSC_BLUE2,
	NTSC_LIGHT_BLUE,
	NTSC_TURQUOISE,
	NTSC_GREEN_BLUE,
	NTSC_GREEN,
	NTSC_YELLOW_GREEN,
	NTSC_ORANGE_GREEN
};

/***************************************************************************
    PALETTE - 24.7 PHASE SHIFT


define NTSC_GREY
    rgb_t(0x00,0x00,0x00), rgb_t(0x11,0x11,0x11), rgb_t(0x22,0x22,0x22), rgb_t(0x33,0x33,0x33), \
    rgb_t(0x44,0x44,0x44), rgb_t(0x55,0x55,0x55), rgb_t(0x66,0x66,0x66), rgb_t(0x77,0x77,0x77), \
    rgb_t(0x88,0x88,0x88), rgb_t(0x99,0x99,0x99), rgb_t(0xAA,0xAA,0xAA), rgb_t(0xBB,0xBB,0xBB), \
    rgb_t(0xCC,0xCC,0xCC), rgb_t(0xDD,0xDD,0xDD), rgb_t(0xEE,0xEE,0xEE), rgb_t(0xFF,0xFF,0xFF   )

define NTSC_GOLD
    rgb_t(0x1A,0x07,0x00), rgb_t(0x2B,0x18,0x00), rgb_t(0x3C,0x29,0x00), rgb_t(0x4D,0x3A,0x00), \
    rgb_t(0x5E,0x4B,0x00), rgb_t(0x6F,0x5C,0x00), rgb_t(0x80,0x6D,0x00), rgb_t(0x91,0x7E,0x09), \
    rgb_t(0xA2,0x8F,0x1A), rgb_t(0xB3,0xA0,0x2B), rgb_t(0xC4,0xB1,0x3C), rgb_t(0xD5,0xC2,0x4D), \
    rgb_t(0xE6,0xD3,0x5E), rgb_t(0xF7,0xE4,0x6F), rgb_t(0xFF,0xF5,0x83), rgb_t(0xFF,0xF7,0x97   )

define NTSC_ORANGE
    rgb_t(0x30,0x00,0x00), rgb_t(0x41,0x07,0x00), rgb_t(0x52,0x18,0x00), rgb_t(0x63,0x29,0x00), \
    rgb_t(0x74,0x3A,0x00), rgb_t(0x85,0x4B,0x00), rgb_t(0x96,0x5C,0x0A), rgb_t(0xA7,0x6D,0x1B), \
    rgb_t(0xB8,0x7E,0x2C), rgb_t(0xC9,0x8F,0x3D), rgb_t(0xDA,0xA0,0x4E), rgb_t(0xEB,0xB1,0x5F), \
    rgb_t(0xFC,0xC2,0x70), rgb_t(0xFF,0xD3,0x85), rgb_t(0xFF,0xE4,0x9B), rgb_t(0xFF,0xF5,0xB1   )

define NTSC_RED_ORANGE
    rgb_t(0x3D,0x00,0x00), rgb_t(0x4E,0x00,0x00), rgb_t(0x5F,0x09,0x00), rgb_t(0x70,0x1A,0x00), \
    rgb_t(0x81,0x2B,0x09), rgb_t(0x92,0x3C,0x1A), rgb_t(0xA3,0x4D,0x2B), rgb_t(0xB4,0x5E,0x3C), \
    rgb_t(0xC5,0x6F,0x4D), rgb_t(0xD6,0x80,0x5E), rgb_t(0xE7,0x91,0x6F), rgb_t(0xF8,0xA2,0x80), \
    rgb_t(0xFF,0xB3,0x94), rgb_t(0xFF,0xC4,0xAA), rgb_t(0xFF,0xD5,0xC0), rgb_t(0xFF,0xE6,0xD6   )

define NTSC_PINK
    rgb_t(0x3F,0x00,0x00), rgb_t(0x50,0x00,0x09), rgb_t(0x61,0x00,0x15), rgb_t(0x72,0x10,0x26), \
    rgb_t(0x83,0x21,0x37), rgb_t(0x94,0x32,0x48), rgb_t(0xA5,0x43,0x59), rgb_t(0xB6,0x54,0x6A), \
    rgb_t(0xC7,0x65,0x7B), rgb_t(0xD8,0x76,0x8C), rgb_t(0xE9,0x87,0x9D), rgb_t(0xFA,0x98,0xAE), \
    rgb_t(0xFF,0xA9,0xC2), rgb_t(0xFF,0xBA,0xD8), rgb_t(0xFF,0xCB,0xEE), rgb_t(0xFF,0xDC,0xF4   )

define NTSC_PURPLE
    rgb_t(0x36,0x00,0x2E), rgb_t(0x47,0x00,0x3A), rgb_t(0x58,0x00,0x46), rgb_t(0x69,0x0C,0x55), \
    rgb_t(0x7A,0x1D,0x66), rgb_t(0x8B,0x2E,0x77), rgb_t(0x9C,0x3F,0x88), rgb_t(0xAD,0x50,0x99), \
    rgb_t(0xBE,0x61,0xAA), rgb_t(0xCF,0x72,0xBB), rgb_t(0xE0,0x83,0xCC), rgb_t(0xF1,0x94,0xDD), \
    rgb_t(0xFF,0xA5,0xE4), rgb_t(0xFF,0xB6,0xE9), rgb_t(0xFF,0xC7,0xEE), rgb_t(0xFF,0xD8,0xF3   )

define NTSC_PURPLE_BLUE
    rgb_t(0x23,0x00,0x55), rgb_t(0x34,0x00,0x61), rgb_t(0x45,0x00,0x6D), rgb_t(0x56,0x0F,0x7E), \
    rgb_t(0x67,0x20,0x8F), rgb_t(0x78,0x31,0xA0), rgb_t(0x89,0x42,0xB1), rgb_t(0x9A,0x53,0xC2), \
    rgb_t(0xAB,0x64,0xD3), rgb_t(0xBC,0x75,0xE4), rgb_t(0xCD,0x86,0xEA), rgb_t(0xDE,0x97,0xEA), \
    rgb_t(0xEF,0xA8,0xEA), rgb_t(0xFF,0xB9,0xEA), rgb_t(0xFF,0xCA,0xEF), rgb_t(0xFF,0xDB,0xF4   )

define NTSC_BLUE1
    rgb_t(0x09,0x00,0x6E), rgb_t(0x1A,0x00,0x7A), rgb_t(0x2B,0x08,0x88), rgb_t(0x3C,0x19,0x99), \
    rgb_t(0x4D,0x2A,0xAA), rgb_t(0x5E,0x3B,0xBB), rgb_t(0x6F,0x4C,0xCC), rgb_t(0x80,0x5D,0xDD), \
    rgb_t(0x91,0x6E,0xEE), rgb_t(0xA2,0x7F,0xF4), rgb_t(0xB3,0x90,0xF4), rgb_t(0xC4,0xA1,0xF4), \
    rgb_t(0xD5,0xB2,0xF4), rgb_t(0xE6,0xC3,0xF4), rgb_t(0xF7,0xD4,0xF4), rgb_t(0xFF,0xE5,0xF7   )

define NTSC_BLUE2
    rgb_t(0x00,0x00,0x6D), rgb_t(0x00,0x05,0x80), rgb_t(0x10,0x16,0x91), rgb_t(0x21,0x27,0xA2), \
    rgb_t(0x32,0x38,0xB3), rgb_t(0x43,0x49,0xC4), rgb_t(0x54,0x5A,0xD5), rgb_t(0x65,0x6B,0xE6), \
    rgb_t(0x76,0x7C,0xF7), rgb_t(0x87,0x8D,0xFF), rgb_t(0x98,0x9E,0xFF), rgb_t(0xA9,0xAF,0xFF), \
    rgb_t(0xBA,0xC0,0xFF), rgb_t(0xCB,0xD1,0xFF), rgb_t(0xDC,0xE2,0xFF), rgb_t(0xED,0xF3,0xFF   )

define NTSC_LIGHT_BLUE
    rgb_t(0x00,0x05,0x57), rgb_t(0x00,0x16,0x6E), rgb_t(0x00,0x27,0x84), rgb_t(0x09,0x38,0x97), \
    rgb_t(0x1A,0x49,0xA8), rgb_t(0x2B,0x5A,0xB9), rgb_t(0x3C,0x6B,0xCA), rgb_t(0x4D,0x7C,0xDB), \
    rgb_t(0x5E,0x8D,0xEC), rgb_t(0x6F,0x9E,0xFD), rgb_t(0x80,0xAF,0xFF), rgb_t(0x91,0xC0,0xFF), \
    rgb_t(0xA2,0xD1,0xFF), rgb_t(0xB3,0xE2,0xFF), rgb_t(0xC4,0xF3,0xFF), rgb_t(0xD5,0xFF,0xFF   )

define NTSC_TURQUOISE
    rgb_t(0x00,0x15,0x34), rgb_t(0x00,0x26,0x4A), rgb_t(0x00,0x37,0x60), rgb_t(0x00,0x48,0x77), \
    rgb_t(0x0A,0x59,0x8A), rgb_t(0x1B,0x6A,0x9B), rgb_t(0x2C,0x7B,0xAC), rgb_t(0x3D,0x8C,0xBD), \
    rgb_t(0x4E,0x9D,0xCE), rgb_t(0x5F,0xAE,0xDF), rgb_t(0x70,0xBF,0xF0), rgb_t(0x81,0xD0,0xFF), \
    rgb_t(0x92,0xE1,0xFF), rgb_t(0xA3,0xF2,0xFF), rgb_t(0xB4,0xFF,0xFF), rgb_t(0xC5,0xFF,0xFF   )

define NTSC_GREEN_BLUE
    rgb_t(0x00,0x21,0x0A), rgb_t(0x00,0x32,0x1F), rgb_t(0x00,0x43,0x35), rgb_t(0x00,0x54,0x4B), \
    rgb_t(0x04,0x65,0x60), rgb_t(0x15,0x76,0x71), rgb_t(0x26,0x87,0x82), rgb_t(0x37,0x98,0x93), \
    rgb_t(0x48,0xA9,0xA4), rgb_t(0x59,0xBA,0xB5), rgb_t(0x6A,0xCB,0xC6), rgb_t(0x7B,0xDC,0xD7), \
    rgb_t(0x8C,0xED,0xE8), rgb_t(0x9D,0xFE,0xF9), rgb_t(0xAE,0xFF,0xFF), rgb_t(0xBF,0xFF,0xFF   )

define NTSC_GREEN
    rgb_t(0x00,0x27,0x0C), rgb_t(0x00,0x38,0x11), rgb_t(0x00,0x49,0x16), rgb_t(0x00,0x5A,0x1D), \
    rgb_t(0x0A,0x6B,0x30), rgb_t(0x1B,0x7C,0x41), rgb_t(0x2C,0x8D,0x52), rgb_t(0x3D,0x9E,0x63), \
    rgb_t(0x4E,0xAF,0x74), rgb_t(0x5F,0xC0,0x85), rgb_t(0x70,0xD1,0x96), rgb_t(0x81,0xE2,0xA7), \
    rgb_t(0x92,0xF3,0xB8), rgb_t(0xA3,0xFF,0xC8), rgb_t(0xB4,0xFF,0xD3), rgb_t(0xC5,0xFF,0xDF   )

define NTSC_YELLOW_GREEN
    rgb_t(0x00,0x26,0x0B), rgb_t(0x00,0x37,0x10), rgb_t(0x00,0x48,0x16), rgb_t(0x0A,0x59,0x18), \
    rgb_t(0x1B,0x6A,0x18), rgb_t(0x2C,0x7B,0x18), rgb_t(0x3D,0x8C,0x27), rgb_t(0x4E,0x9D,0x38), \
    rgb_t(0x5F,0xAE,0x49), rgb_t(0x70,0xBF,0x5A), rgb_t(0x81,0xD0,0x6B), rgb_t(0x92,0xE1,0x7C), \
    rgb_t(0xA3,0xF2,0x8D), rgb_t(0xB4,0xFF,0x9C), rgb_t(0xC5,0xFF,0xA8), rgb_t(0xD6,0xFF,0xB4   )

define NTSC_ORANGE_GREEN
    rgb_t(0x00,0x1E,0x09), rgb_t(0x00,0x2F,0x0E), rgb_t(0x11,0x40,0x0E), rgb_t(0x22,0x51,0x0E), \
    rgb_t(0x33,0x62,0x0E), rgb_t(0x44,0x73,0x0E), rgb_t(0x55,0x84,0x0E), rgb_t(0x66,0x95,0x17), \
    rgb_t(0x77,0xA6,0x28), rgb_t(0x88,0xB7,0x39), rgb_t(0x99,0xC8,0x4A), rgb_t(0xAA,0xD9,0x5B), \
    rgb_t(0xBB,0xEA,0x6C), rgb_t(0xCC,0xFB,0x7D), rgb_t(0xDD,0xFF,0x8A), rgb_t(0xEE,0xFF,0x96   )

define NTSC_LIGHT_ORANGE
    rgb_t(0x0A,0x11,0x02), rgb_t(0x1B,0x22,0x02), rgb_t(0x2C,0x33,0x02), rgb_t(0x3D,0x44,0x02), \
    rgb_t(0x4E,0x55,0x02), rgb_t(0x5F,0x66,0x02), rgb_t(0x70,0x77,0x02), rgb_t(0x81,0x88,0x09), \
    rgb_t(0x92,0x99,0x1A), rgb_t(0xA3,0xAA,0x2B), rgb_t(0xB4,0xBB,0x3C), rgb_t(0xC5,0xCC,0x4D), \
    rgb_t(0xD6,0xDD,0x5E), rgb_t(0xE7,0xEE,0x6F), rgb_t(0xF8,0xFF,0x80), rgb_t(0xFF,0xFF,0x8F   )
***************************************************************************/


/***************************************************************************
    PALETTE - 25.2 PHASE SHIFT


define NTSC_GREY
    rgb_t(0x00,0x00,0x00), rgb_t(0x11,0x11,0x11), rgb_t(0x22,0x22,0x22), rgb_t(0x33,0x33,0x33), \
    rgb_t(0x44,0x44,0x44), rgb_t(0x55,0x55,0x55), rgb_t(0x66,0x66,0x66), rgb_t(0x77,0x77,0x77), \
    rgb_t(0x88,0x88,0x88), rgb_t(0x99,0x99,0x99), rgb_t(0xAA,0xAA,0xAA), rgb_t(0xBB,0xBB,0xBB), \
    rgb_t(0xCC,0xCC,0xCC), rgb_t(0xDD,0xDD,0xDD), rgb_t(0xEE,0xEE,0xEE), rgb_t(0xFF,0xFF,0xFF   )

define NTSC_GOLD
    rgb_t(0x1A,0x07,0x00), rgb_t(0x2B,0x18,0x00), rgb_t(0x3C,0x29,0x00), rgb_t(0x4D,0x3A,0x00), \
    rgb_t(0x5E,0x4B,0x00), rgb_t(0x6F,0x5C,0x00), rgb_t(0x80,0x6D,0x00), rgb_t(0x91,0x7E,0x09), \
    rgb_t(0xA2,0x8F,0x1A), rgb_t(0xB3,0xA0,0x2B), rgb_t(0xC4,0xB1,0x3C), rgb_t(0xD5,0xC2,0x4D), \
    rgb_t(0xE6,0xD3,0x5E), rgb_t(0xF7,0xE4,0x6F), rgb_t(0xFF,0xF5,0x83), rgb_t(0xFF,0xF7,0x97   )

define NTSC_ORANGE
    rgb_t(0x30,0x00,0x00), rgb_t(0x41,0x07,0x00), rgb_t(0x52,0x18,0x00), rgb_t(0x63,0x29,0x00), \
    rgb_t(0x74,0x3A,0x00), rgb_t(0x85,0x4B,0x00), rgb_t(0x96,0x5C,0x0A), rgb_t(0xA7,0x6D,0x1B), \
    rgb_t(0xB8,0x7E,0x2C), rgb_t(0xC9,0x8F,0x3D), rgb_t(0xDA,0xA0,0x4E), rgb_t(0xEB,0xB1,0x5F), \
    rgb_t(0xFC,0xC2,0x70), rgb_t(0xFF,0xD3,0x85), rgb_t(0xFF,0xE4,0x9B), rgb_t(0xFF,0xF5,0xB1   )

define NTSC_RED_ORANGE
    rgb_t(0x3E,0x00,0x00), rgb_t(0x4F,0x00,0x00), rgb_t(0x60,0x09,0x00), rgb_t(0x71,0x1A,0x00), \
    rgb_t(0x82,0x2B,0x0B), rgb_t(0x93,0x3C,0x1C), rgb_t(0xA4,0x4D,0x2D), rgb_t(0xB5,0x5E,0x3E), \
    rgb_t(0xC6,0x6F,0x4F), rgb_t(0xD7,0x80,0x60), rgb_t(0xE8,0x91,0x71), rgb_t(0xF9,0xA2,0x82), \
    rgb_t(0xFF,0xB3,0x96), rgb_t(0xFF,0xC4,0xAC), rgb_t(0xFF,0xD5,0xC2), rgb_t(0xFF,0xE6,0xD8   )

define NTSC_PINK
    rgb_t(0x3F,0x00,0x00), rgb_t(0x50,0x00,0x0C), rgb_t(0x61,0x00,0x18), rgb_t(0x72,0x0F,0x28), \
    rgb_t(0x83,0x20,0x39), rgb_t(0x94,0x31,0x4A), rgb_t(0xA5,0x42,0x5B), rgb_t(0xB6,0x53,0x6C), \
    rgb_t(0xC7,0x64,0x7D), rgb_t(0xD8,0x75,0x8E), rgb_t(0xE9,0x86,0x9F), rgb_t(0xFA,0x97,0xB0), \
    rgb_t(0xFF,0xA8,0xC5), rgb_t(0xFF,0xB9,0xDB), rgb_t(0xFF,0xCA,0xEF), rgb_t(0xFF,0xDB,0xF4   )

define NTSC_PURPLE
    rgb_t(0x35,0x00,0x31), rgb_t(0x46,0x00,0x3D), rgb_t(0x57,0x00,0x49), rgb_t(0x68,0x0C,0x58), \
    rgb_t(0x79,0x1D,0x69), rgb_t(0x8A,0x2E,0x7A), rgb_t(0x9B,0x3F,0x8B), rgb_t(0xAC,0x50,0x9C), \
    rgb_t(0xBD,0x61,0xAD), rgb_t(0xCE,0x72,0xBE), rgb_t(0xDF,0x83,0xCF), rgb_t(0xF0,0x94,0xE0), \
    rgb_t(0xFF,0xA5,0xE4), rgb_t(0xFF,0xB6,0xE9), rgb_t(0xFF,0xC7,0xEE), rgb_t(0xFF,0xD8,0xF3   )

define NTSC_PURPLE_BLUE
    rgb_t(0x20,0x00,0x59), rgb_t(0x31,0x00,0x65), rgb_t(0x42,0x00,0x71), rgb_t(0x53,0x10,0x82), \
    rgb_t(0x64,0x21,0x93), rgb_t(0x75,0x32,0xA4), rgb_t(0x86,0x43,0xB5), rgb_t(0x97,0x54,0xC6), \
    rgb_t(0xA8,0x65,0xD7), rgb_t(0xB9,0x76,0xE8), rgb_t(0xCA,0x87,0xEB), rgb_t(0xDB,0x98,0xEB), \
    rgb_t(0xEC,0xA9,0xEB), rgb_t(0xFD,0xBA,0xEB), rgb_t(0xFF,0xCB,0xEF), rgb_t(0xFF,0xDC,0xF4   )

define NTSC_BLUE1
    rgb_t(0x05,0x00,0x70), rgb_t(0x16,0x00,0x7C), rgb_t(0x27,0x09,0x8B), rgb_t(0x38,0x1A,0x9C), \
    rgb_t(0x49,0x2B,0xAD), rgb_t(0x5A,0x3C,0xBE), rgb_t(0x6B,0x4D,0xCF), rgb_t(0x7C,0x5E,0xE0), \
    rgb_t(0X8D,0x6F,0xF1), rgb_t(0x9E,0x80,0xF6), rgb_t(0xAF,0x91,0xF6), rgb_t(0xC0,0xA2,0xF6), \
    rgb_t(0xD1,0xB3,0xF6), rgb_t(0xE2,0xC4,0xF6), rgb_t(0xF3,0xD5,0xF6), rgb_t(0xFF,0xE6,0xF7   )

define NTSC_BLUE2
    rgb_t(0x00,0x00,0x6B), rgb_t(0x00,0x08,0x7E), rgb_t(0x0C,0x19,0x91), rgb_t(0x1D,0x2A,0xA2), \
    rgb_t(0x2E,0x3B,0xB3), rgb_t(0x3F,0x4C,0xC4), rgb_t(0x50,0x5D,0xD5), rgb_t(0x61,0x6E,0xE6), \
    rgb_t(0x72,0x7F,0xF7), rgb_t(0x83,0x90,0xFF), rgb_t(0x94,0xA1,0xFF), rgb_t(0xA5,0xB2,0xFF), \
    rgb_t(0xB6,0xC3,0xFF), rgb_t(0xC7,0xD4,0xFF), rgb_t(0xD8,0xE5,0xFF), rgb_t(0xE9,0xF6,0xFF   )

define NTSC_LIGHT_BLUE
    rgb_t(0x00,0x08,0x52), rgb_t(0x00,0x19,0x68), rgb_t(0x00,0x2A,0x7F), rgb_t(0x05,0x3B,0x93), \
    rgb_t(0x16,0x4C,0xA4), rgb_t(0x27,0x5D,0xB5), rgb_t(0x38,0x6E,0xC6), rgb_t(0x49,0x7F,0xD7), \
    rgb_t(0x5A,0x90,0xE8), rgb_t(0x6B,0xA1,0xF9), rgb_t(0x7C,0xB2,0xFF), rgb_t(0x8D,0xC3,0xFF), \
    rgb_t(0x9E,0xD4,0xFF), rgb_t(0xAF,0xE5,0xFF), rgb_t(0xC0,0xF6,0xFF), rgb_t(0xD1,0xFF,0xFF   )

define NTSC_TURQUOISE
    rgb_t(0x00,0x17,0x2D), rgb_t(0x00,0x28,0x43), rgb_t(0x00,0x39,0x59), rgb_t(0x00,0x4A,0x6F), \
    rgb_t(0x08,0x5B,0x83), rgb_t(0x19,0x6C,0x94), rgb_t(0x2A,0x7D,0xA5), rgb_t(0x3B,0x8E,0xB6), \
    rgb_t(0x4C,0x9F,0xC7), rgb_t(0x5D,0xB0,0xD8), rgb_t(0x6E,0xC1,0xE9), rgb_t(0x7F,0xD2,0xFA), \
    rgb_t(0x90,0xE3,0xFF), rgb_t(0xA1,0xF4,0xFF), rgb_t(0xB2,0xFF,0xFF), rgb_t(0xC3,0xFF,0xFF   )

define NTSC_GREEN_BLUE
    rgb_t(0x00,0x23,0x0A), rgb_t(0x00,0x34,0x15), rgb_t(0x00,0x45,0x2B), rgb_t(0x00,0x56,0x41), \
    rgb_t(0x04,0x67,0x56), rgb_t(0x15,0x78,0x67), rgb_t(0x26,0x89,0x78), rgb_t(0x37,0x9A,0x89), \
    rgb_t(0x48,0xAB,0x9A), rgb_t(0x59,0xBC,0xAB), rgb_t(0x6A,0xCD,0xBC), rgb_t(0x7B,0xDE,0xCD), \
    rgb_t(0x8C,0xEF,0xDE), rgb_t(0x9D,0xFF,0xEE), rgb_t(0xAE,0xFF,0xFA), rgb_t(0xBF,0xFF,0xFF   )

define NTSC_GREEN
    rgb_t(0x00,0x27,0x0C), rgb_t(0x00,0x38,0x11), rgb_t(0x00,0x49,0x16), rgb_t(0x00,0x5A,0x1B), \
    rgb_t(0x0D,0x6B,0x25), rgb_t(0x1E,0x7C,0x36), rgb_t(0x2F,0x8D,0x47), rgb_t(0x40,0x9E,0x58), \
    rgb_t(0x51,0xAF,0x69), rgb_t(0x62,0xC0,0x7A), rgb_t(0x73,0xD1,0x8B), rgb_t(0x84,0xE2,0x9C), \
    rgb_t(0x95,0xF3,0xAD), rgb_t(0xA6,0xFF,0xBD), rgb_t(0xB7,0xFF,0xC9), rgb_t(0xC8,0xFF,0xD4   )

define NTSC_YELLOW_GREEN
    rgb_t(0x00,0x24,0x0B), rgb_t(0x00,0x35,0x10), rgb_t(0x00,0x46,0x15), rgb_t(0x10,0x57,0x15), \
    rgb_t(0x21,0x68,0x15), rgb_t(0x32,0x79,0x15), rgb_t(0x43,0x8A,0x1C), rgb_t(0x54,0x9B,0x2D), \
    rgb_t(0x65,0xAC,0x3E), rgb_t(0x76,0xBD,0x4F), rgb_t(0x87,0xCE,0x60), rgb_t(0x98,0xDF,0x71), \
    rgb_t(0xA9,0xF0,0x82), rgb_t(0xBA,0xFF,0x93), rgb_t(0xCB,0xFF,0x9F), rgb_t(0xDC,0xFF,0xAA   )

define NTSC_ORANGE_GREEN
    rgb_t(0x00,0x1B,0x08), rgb_t(0x08,0x2C,0x0B), rgb_t(0x19,0x3D,0x0B), rgb_t(0x2A,0x4E,0x0B), \
    rgb_t(0x3B,0x5F,0x0B), rgb_t(0x4C,0x70,0x0B), rgb_t(0x5D,0x81,0x0B), rgb_t(0x6E,0x92,0x11), \
    rgb_t(0x7F,0xA3,0x22), rgb_t(0x90,0xB4,0x33), rgb_t(0xA1,0xC5,0x44), rgb_t(0xB2,0xD6,0x55), \
    rgb_t(0xC3,0xE7,0x66), rgb_t(0xD4,0xF8,0x77), rgb_t(0xE5,0xFF,0x85), rgb_t(0xF6,0xFF,0x91   )

define NTSC_LIGHT_ORANGE
    rgb_t(0x12,0x0C,0x00), rgb_t(0x23,0x1D,0x00), rgb_t(0x34,0x2E,0x00), rgb_t(0x45,0x3F,0x00), \
    rgb_t(0x56,0x50,0x00), rgb_t(0x67,0x61,0x00), rgb_t(0x78,0x72,0x00), rgb_t(0x89,0x83,0x08), \
    rgb_t(0x9A,0x94,0x19), rgb_t(0xAB,0xA5,0x2A), rgb_t(0xBC,0xB6,0x3B), rgb_t(0xCD,0xC7,0x4C), \
    rgb_t(0xDE,0xD8,0x5D), rgb_t(0xEF,0xE9,0x6E), rgb_t(0xFF,0xFA,0x80), rgb_t(0xFF,0xFF,0x92   )
***************************************************************************/


/***************************************************************************
    PALETTE - 25.7 PHASE SHIFT


define NTSC_GREY
    rgb_t(0x00,0x00,0x00), rgb_t(0x11,0x11,0x11), rgb_t(0x22,0x22,0x22), rgb_t(0x33,0x33,0x33), \
    rgb_t(0x44,0x44,0x44), rgb_t(0x55,0x55,0x55), rgb_t(0x66,0x66,0x66), rgb_t(0x77,0x77,0x77), \
    rgb_t(0x88,0x88,0x88), rgb_t(0x99,0x99,0x99), rgb_t(0xAA,0xAA,0xAA), rgb_t(0xBB,0xBB,0xBB), \
    rgb_t(0xCC,0xCC,0xCC), rgb_t(0xDD,0xDD,0xDD), rgb_t(0xEE,0xEE,0xEE), rgb_t(0xFF,0xFF,0xFF   )

define NTSC_GOLD
    rgb_t(0x1A,0x07,0x00), rgb_t(0x2B,0x18,0x00), rgb_t(0x3C,0x29,0x00), rgb_t(0x4D,0x3A,0x00), \
    rgb_t(0x5E,0x4B,0x00), rgb_t(0x6F,0x5C,0x00), rgb_t(0x80,0x6D,0x00), rgb_t(0x91,0x7E,0x09), \
    rgb_t(0xA2,0x8F,0x1A), rgb_t(0xB3,0xA0,0x2B), rgb_t(0xC4,0xB1,0x3C), rgb_t(0xD5,0xC2,0x4D), \
    rgb_t(0xE6,0xD3,0x5E), rgb_t(0xF7,0xE4,0x6F), rgb_t(0xFF,0xF5,0x83), rgb_t(0xFF,0xF7,0x97   )

define NTSC_ORANGE
    rgb_t(0x31,0x00,0x00), rgb_t(0x42,0x06,0x00), rgb_t(0x53,0x17,0x00), rgb_t(0x64,0x28,0x00), \
    rgb_t(0x75,0x39,0x00), rgb_t(0x86,0X4A,0x00), rgb_t(0x97,0x5B,0x0A), rgb_t(0xA8,0x6C,0x1B), \
    rgb_t(0xB9,0x7D,0x2C), rgb_t(0xCA,0x8E,0x3D), rgb_t(0xDB,0x9F,0x4E), rgb_t(0xEC,0xB0,0x5F), \
    rgb_t(0xFD,0xC1,0x70), rgb_t(0xFF,0xD2,0x85), rgb_t(0xFF,0xE3,0x9C), rgb_t(0xFF,0xF4,0xB2   )

define NTSC_RED_ORANGE
    rgb_t(0x3E,0x00,0x00), rgb_t(0x4F,0x00,0x00), rgb_t(0x60,0x08,0x00), rgb_t(0x71,0x19,0x00), \
    rgb_t(0x82,0x2A,0x0D), rgb_t(0x93,0x3B,0x1E), rgb_t(0xA4,0x4C,0x2F), rgb_t(0xB5,0x5D,0x40), \
    rgb_t(0xC6,0x6E,0x51), rgb_t(0xD7,0x7F,0x62), rgb_t(0xE8,0x90,0x73), rgb_t(0xF9,0xA1,0x83), \
    rgb_t(0xFF,0xB2,0x98), rgb_t(0xFF,0xC3,0xAE), rgb_t(0xFF,0xD4,0xC4), rgb_t(0xFF,0xE5,0xDA   )

define NTSC_PINK
    rgb_t(0x3F,0x00,0x03), rgb_t(0x50,0x00,0x0F), rgb_t(0x61,0x00,0x1B), rgb_t(0x72,0x0F,0x2B), \
    rgb_t(0x83,0x20,0x3C), rgb_t(0x94,0x31,0x4D), rgb_t(0xA5,0x42,0x5E), rgb_t(0xB6,0x53,0x6F), \
    rgb_t(0xC7,0x64,0x80), rgb_t(0xD8,0x75,0x91), rgb_t(0xE9,0x86,0xA2), rgb_t(0xFA,0x97,0xB3), \
    rgb_t(0xFF,0xA8,0xC8), rgb_t(0xFF,0xB9,0xDE), rgb_t(0xFF,0xCA,0xEF), rgb_t(0xFF,0xDB,0xF4   )

define NTSC_PURPLE
    rgb_t(0x33,0x00,0x35), rgb_t(0x44,0x00,0x41), rgb_t(0x55,0x00,0x4C), rgb_t(0x66,0x0C,0x5C), \
    rgb_t(0x77,0x1D,0x6D), rgb_t(0x88,0x2E,0x7E), rgb_t(0x99,0x3F,0x8F), rgb_t(0xAA,0x50,0xA0), \
    rgb_t(0xBB,0x61,0xB1), rgb_t(0xCC,0x72,0xC2), rgb_t(0xDD,0x83,0xD3), rgb_t(0xEE,0x94,0xE4), \
    rgb_t(0xFF,0xA5,0xE4), rgb_t(0xFF,0xB6,0xE9), rgb_t(0xFF,0xC7,0xEE), rgb_t(0xFF,0xD8,0xF3   )

define NTSC_PURPLE_BLUE
    rgb_t(0x1D,0x00,0x5C), rgb_t(0x2E,0x00,0x68), rgb_t(0x40,0x00,0x74), rgb_t(0x51,0x10,0x84), \
    rgb_t(0x62,0x21,0x95), rgb_t(0x73,0x32,0xA6), rgb_t(0x84,0x43,0xB7), rgb_t(0x95,0x54,0xC8), \
    rgb_t(0xA6,0x65,0xD9), rgb_t(0xB7,0x76,0xEA), rgb_t(0xC8,0x87,0xEB), rgb_t(0xD9,0x98,0xEB), \
    rgb_t(0xE9,0xA9,0xEC), rgb_t(0xFB,0xBA,0xEB), rgb_t(0xFF,0xCB,0xEF), rgb_t(0xFF,0xDC,0xF4   )

define NTSC_BLUE1
    rgb_t(0x02,0x00,0x71), rgb_t(0x13,0x00,0x7D), rgb_t(0x24,0x0B,0x8C), rgb_t(0x35,0x1C,0x9D), \
    rgb_t(0x46,0x2D,0xAE), rgb_t(0x57,0x3E,0xBF), rgb_t(0x68,0x4F,0xD0), rgb_t(0x79,0x60,0xE1), \
    rgb_t(0x8A,0x71,0xF2), rgb_t(0x9B,0x82,0xF7), rgb_t(0xAC,0x93,0xF7), rgb_t(0xBD,0xA4,0xF7), \
    rgb_t(0xCE,0xB5,0xF7), rgb_t(0xDF,0xC6,0xF7), rgb_t(0xF0,0xD7,0xF7), rgb_t(0xFF,0xE8,0xF8   )

define NTSC_BLUE2
    rgb_t(0x00,0x00,0x68), rgb_t(0x00,0x0A,0x7C), rgb_t(0x08,0x1B,0x90), rgb_t(0x19,0x2C,0xA1), \
    rgb_t(0x2A,0x3D,0xB2), rgb_t(0x3B,0x4E,0xC3), rgb_t(0x4C,0x5F,0xD4), rgb_t(0x5D,0x70,0xE5), \
    rgb_t(0x6E,0x81,0xF6), rgb_t(0x7F,0x92,0xFF), rgb_t(0x90,0xA3,0xFF), rgb_t(0xA1,0xB4,0xFF), \
    rgb_t(0xB2,0xC5,0xFF), rgb_t(0xC3,0xD6,0xFF), rgb_t(0xD4,0xE7,0xFF), rgb_t(0xE5,0xF8,0xFF   )

define NTSC_LIGHT_BLUE
    rgb_t(0x00,0x0A,0x4D), rgb_t(0x00,0x1B,0x63), rgb_t(0x00,0x2C,0x79), rgb_t(0x02,0x3D,0x8F), \
    rgb_t(0x13,0x4E,0xA0), rgb_t(0x24,0x5F,0xB1), rgb_t(0x35,0x70,0xC2), rgb_t(0x46,0x81,0xD3), \
    rgb_t(0x57,0x92,0xE4), rgb_t(0x68,0xA3,0xF5), rgb_t(0x79,0xB4,0xFF), rgb_t(0x8A,0xC5,0xFF), \
    rgb_t(0x9B,0xD6,0xFF), rgb_t(0xAC,0xE7,0xFF), rgb_t(0xBD,0xF8,0xFF), rgb_t(0xCE,0xFF,0xFF   )

define NTSC_TURQUOISE
    rgb_t(0x00,0x1A,0x26), rgb_t(0x00,0x2B,0x3C), rgb_t(0x00,0x3C,0x52), rgb_t(0x00,0x4D,0x68), \
    rgb_t(0x06,0x5E,0x7C), rgb_t(0x17,0x6F,0x8D), rgb_t(0x28,0x80,0x9E), rgb_t(0x39,0x91,0xAF), \
    rgb_t(0x4A,0xA2,0xC0), rgb_t(0x5B,0xB3,0xD1), rgb_t(0x6C,0xC4,0xE2), rgb_t(0x7D,0xD5,0xF3), \
    rgb_t(0x8E,0xE6,0xFF), rgb_t(0x9F,0xF7,0xFF), rgb_t(0xB0,0xFF,0xFF), rgb_t(0xC1,0xFF,0xFF   )

define NTSC_GREEN_BLUE
    rgb_t(0x00,0x24,0x0B), rgb_t(0x00,0x35,0x10), rgb_t(0x00,0x46,0x22), rgb_t(0x00,0x57,0x38), \
    rgb_t(0x05,0x68,0x4D), rgb_t(0x16,0x79,0x5E), rgb_t(0x27,0x8A,0x6F), rgb_t(0x38,0x9B,0x80), \
    rgb_t(0x49,0xAC,0x91), rgb_t(0x5A,0xBD,0xA2), rgb_t(0x6B,0xCE,0xB3), rgb_t(0x7C,0xDF,0xC4), \
    rgb_t(0x8D,0xF0,0xD5), rgb_t(0x9E,0xFF,0xE5), rgb_t(0xAF,0xFF,0xF1), rgb_t(0xC0,0xFF,0xFD   )

define NTSC_GREEN
    rgb_t(0x00,0x27,0x0C), rgb_t(0x00,0x38,0x11), rgb_t(0x00,0x49,0x16), rgb_t(0x00,0x5A,0x1B), \
    rgb_t(0x10,0x6B,0x1B), rgb_t(0x21,0x7C,0x2C), rgb_t(0x32,0x8D,0x3D), rgb_t(0x43,0x9E,0x4E), \
    rgb_t(0x54,0xAF,0x5F), rgb_t(0x65,0xC0,0x70), rgb_t(0x76,0xD1,0x81), rgb_t(0x87,0xE2,0x92), \
    rgb_t(0x98,0xF3,0xA3), rgb_t(0xA9,0xFF,0xB3), rgb_t(0xBA,0xFF,0xBF), rgb_t(0xCB,0xFF,0xCB   )

define NTSC_YELLOW_GREEN
    rgb_t(0x00,0x23,0x0A), rgb_t(0x00,0x34,0x10), rgb_t(0x04,0x45,0x13), rgb_t(0x15,0x56,0x13), \
    rgb_t(0x26,0x67,0x13), rgb_t(0x37,0x78,0x13), rgb_t(0x48,0x89,0x14), rgb_t(0x59,0x9A,0x25), \
    rgb_t(0x6A,0xAB,0x36), rgb_t(0x7B,0xBC,0x47), rgb_t(0x8C,0xCD,0x58), rgb_t(0x9D,0xDE,0x69), \
    rgb_t(0xAE,0xEF,0x7A), rgb_t(0xBF,0xFF,0x8B), rgb_t(0xD0,0xFF,0x97), rgb_t(0xE1,0xFF,0xA3   )

define NTSC_ORANGE_GREEN
    rgb_t(0x00,0x17,0x07), rgb_t(0x0E,0x28,0x08), rgb_t(0x1F,0x39,0x08), rgb_t(0x30,0x4A,0x08), \
    rgb_t(0x41,0x5B,0x08), rgb_t(0x52,0x6C,0x08), rgb_t(0x63,0x7D,0x08), rgb_t(0x74,0x8E,0x0D), \
    rgb_t(0x85,0x9F,0x1E), rgb_t(0x96,0xB0,0x2F), rgb_t(0xA7,0xC1,0x40), rgb_t(0xB8,0xD2,0x51), \
    rgb_t(0xC9,0xE3,0x62), rgb_t(0xDA,0xF4,0x73), rgb_t(0xEB,0xFF,0x82), rgb_t(0xFC,0xFF,0x8E   )

define NTSC_LIGHT_ORANGE
    rgb_t(0x19,0x07,0x00), rgb_t(0x2A,0x18,0x00), rgb_t(0x3B,0x29,0x00), rgb_t(0x4C,0x3A,0x00), \
    rgb_t(0x5D,0x4B,0x00), rgb_t(0x6E,0x5C,0x00), rgb_t(0x7F,0x6D,0x00), rgb_t(0x90,0x7E,0x09), \
    rgb_t(0xA1,0x8F,0x1A), rgb_t(0xB2,0xA0,0x2B), rgb_t(0xC3,0xB1,0x3C), rgb_t(0xD4,0xC2,0x4D), \
    rgb_t(0xE5,0xD3,0x5E), rgb_t(0xF6,0xE4,0x6F), rgb_t(0xFF,0xF5,0x82), rgb_t(0xFF,0xFF,0x96   )
***************************************************************************/


/***************************************************************************
    PALETTE - 26.2 PHASE SHIFT


#define NTSC_GREY \
    rgb_t(0x00,0x00,0x00), rgb_t(0x11,0x11,0x11), rgb_t(0x22,0x22,0x22), rgb_t(0x33,0x33,0x33), \
    rgb_t(0x44,0x44,0x44), rgb_t(0x55,0x55,0x55), rgb_t(0x66,0x66,0x66), rgb_t(0x77,0x77,0x77), \
    rgb_t(0x88,0x88,0x88), rgb_t(0x99,0x99,0x99), rgb_t(0xAA,0xAA,0xAA), rgb_t(0xBB,0xBB,0xBB), \
    rgb_t(0xCC,0xCC,0xCC), rgb_t(0xDD,0xDD,0xDD), rgb_t(0xEE,0xEE,0xEE), rgb_t(0xFF,0xFF,0xFF   )

#define NTSC_GOLD \
    rgb_t(0x1A,0x07,0x00), rgb_t(0x2B,0x18,0x00), rgb_t(0x3C,0x29,0x00), rgb_t(0x4D,0x3A,0x00), \
    rgb_t(0x5E,0x4B,0x00), rgb_t(0x6F,0x5C,0x00), rgb_t(0x80,0x6D,0x00), rgb_t(0x91,0x7E,0x09), \
    rgb_t(0xA2,0x8F,0x1A), rgb_t(0xB3,0xA0,0x2B), rgb_t(0xC4,0xB1,0x3C), rgb_t(0xD5,0xC2,0x4D), \
    rgb_t(0xE6,0xD3,0x5E), rgb_t(0xF7,0xE4,0x6F), rgb_t(0xFF,0xF5,0x83), rgb_t(0xFF,0xF7,0x97   )

#define NTSC_ORANGE \
    rgb_t(0x31,0x00,0x00), rgb_t(0x42,0x06,0x00), rgb_t(0x53,0x17,0x00), rgb_t(0x64,0x28,0x00), \
    rgb_t(0x75,0x39,0x00), rgb_t(0x86,0X4A,0x00), rgb_t(0x97,0x5B,0x0B), rgb_t(0xA8,0x6C,0x1C), \
    rgb_t(0xB9,0x7D,0x2D), rgb_t(0xCA,0x8E,0x3E), rgb_t(0xDB,0x9F,0x4F), rgb_t(0xEC,0xB0,0x60), \
    rgb_t(0xFD,0xC1,0x71), rgb_t(0xFF,0xD2,0x86), rgb_t(0xFF,0xE3,0x9D), rgb_t(0xFF,0xF4,0xB3   )

#define NTSC_RED_ORANGE \
    rgb_t(0x3E,0x00,0x00), rgb_t(0x4F,0x00,0x00), rgb_t(0x60,0x08,0x00), rgb_t(0x71,0x19,0x00), \
    rgb_t(0x82,0x2A,0x0F), rgb_t(0x93,0x3B,0x20), rgb_t(0xA4,0x4C,0x31), rgb_t(0xB5,0x5D,0x42), \
    rgb_t(0xC6,0x6E,0x53), rgb_t(0xD7,0x7F,0x64), rgb_t(0xE8,0x90,0x75), rgb_t(0xF9,0xA1,0x86), \
    rgb_t(0xFF,0xB2,0x9A), rgb_t(0xFF,0xC3,0xB0), rgb_t(0xFF,0xD4,0xC6), rgb_t(0xFF,0xE5,0xDC   )

#define NTSC_PINK \
    rgb_t(0x3E,0x00,0x06), rgb_t(0x4F,0x00,0x12), rgb_t(0x60,0x00,0x1E), rgb_t(0x71,0x0E,0x2E), \
    rgb_t(0x82,0x1F,0x3F), rgb_t(0x93,0x30,0x50), rgb_t(0xA4,0x41,0x61), rgb_t(0xB5,0x52,0x72), \
    rgb_t(0xC6,0x63,0x83), rgb_t(0xD7,0x74,0x94), rgb_t(0xE8,0x85,0xA5), rgb_t(0xF9,0x96,0xB6), \
    rgb_t(0xFF,0xA7,0xCB), rgb_t(0xFF,0xB8,0xE1), rgb_t(0xFF,0xC9,0xEF), rgb_t(0xFF,0xDA,0xF4   )

#define NTSC_PURPLE \
    rgb_t(0x32,0x00,0x38), rgb_t(0x43,0x00,0x44), rgb_t(0x54,0x00,0x50), rgb_t(0x65,0x0C,0x5F), \
    rgb_t(0x76,0x1D,0x70), rgb_t(0x87,0x2E,0x81), rgb_t(0x98,0x3F,0x92), rgb_t(0xA9,0x50,0xA3), \
    rgb_t(0xBA,0x61,0xB4), rgb_t(0xCB,0x72,0xC5), rgb_t(0xDC,0x83,0xD6), rgb_t(0xED,0x94,0xE4), \
    rgb_t(0xFE,0xA5,0xE4), rgb_t(0xFF,0xB6,0xE9), rgb_t(0xFF,0xC7,0xEE), rgb_t(0xFF,0xD8,0xF3   )

#define NTSC_PURPLE_BLUE \
    rgb_t(0x1B,0x00,0x5F), rgb_t(0x2C,0x00,0x6B), rgb_t(0x3D,0x00,0x77), rgb_t(0x4E,0x11,0x88), \
    rgb_t(0x5F,0x22,0x99), rgb_t(0x70,0x33,0xAA), rgb_t(0x81,0x44,0xBB), rgb_t(0x92,0x55,0xCC), \
    rgb_t(0xA3,0x66,0xDD), rgb_t(0xB4,0x77,0xED), rgb_t(0xC5,0x88,0xED), rgb_t(0xD6,0x99,0xED), \
    rgb_t(0xE7,0xAA,0xED), rgb_t(0xF8,0xBB,0xED), rgb_t(0xFF,0xCC,0xF0), rgb_t(0xFF,0xDD,0xF5   )

#define NTSC_BLUE1 \
    rgb_t(0x00,0x00,0x72), rgb_t(0x10,0x00,0x7E), rgb_t(0x21,0x0D,0x8E), rgb_t(0x32,0x1E,0x9F), \
    rgb_t(0x43,0x2F,0xB0), rgb_t(0x54,0x40,0xC1), rgb_t(0x65,0x51,0xD2), rgb_t(0x76,0x62,0xE3), \
    rgb_t(0x87,0x73,0xF4), rgb_t(0x98,0x84,0xF9), rgb_t(0xA9,0x95,0xF9), rgb_t(0xBA,0xA6,0xF9), \
    rgb_t(0xCB,0xB7,0xF9), rgb_t(0xDC,0xC8,0xF9), rgb_t(0xED,0xD9,0xF9), rgb_t(0xFE,0xEA,0xF9   )

#define NTSC_BLUE2 \
    rgb_t(0x00,0x00,0x65), rgb_t(0x00,0x0C,0x7A), rgb_t(0x05,0x1D,0x8E), rgb_t(0x16,0x2E,0x9F), \
    rgb_t(0x27,0x3F,0xB0), rgb_t(0x38,0x50,0xC1), rgb_t(0x49,0x61,0xD2), rgb_t(0x5A,0x72,0xE3), \
    rgb_t(0x6B,0x83,0xF4), rgb_t(0x7C,0x94,0xFF), rgb_t(0x8D,0xA5,0xFF), rgb_t(0x9E,0xB6,0xFF), \
    rgb_t(0xAF,0xC7,0xFF), rgb_t(0xC0,0xD8,0xFF), rgb_t(0xD1,0xE9,0xFF), rgb_t(0xE2,0xFA,0xFF   )

#define NTSC_LIGHT_BLUE \
    rgb_t(0x00,0x0D,0x48), rgb_t(0x00,0x1E,0x5E), rgb_t(0x00,0x2F,0x74), rgb_t(0x00,0x40,0x8A), \
    rgb_t(0x11,0x51,0x9B), rgb_t(0x22,0x62,0xAC), rgb_t(0x33,0x73,0xBD), rgb_t(0x44,0x84,0xCE), \
    rgb_t(0x55,0x95,0xDF), rgb_t(0x66,0xA6,0xF0), rgb_t(0x77,0xB7,0xFF), rgb_t(0x88,0xC8,0xFF), \
    rgb_t(0x99,0xD9,0xFF), rgb_t(0xAA,0xEA,0xFF), rgb_t(0xBB,0xFB,0xFF), rgb_t(0xCC,0xFF,0xFF   )

#define NTSC_TURQUOISE \
    rgb_t(0x00,0x1C,0x1C), rgb_t(0x00,0x2D,0x32), rgb_t(0x00,0x3E,0x49), rgb_t(0x00,0x4F,0x5F), \
    rgb_t(0x05,0x60,0x73), rgb_t(0x16,0x71,0x84), rgb_t(0x27,0x82,0x95), rgb_t(0x38,0x93,0xA6), \
    rgb_t(0x49,0xA4,0xB7), rgb_t(0x5A,0xB5,0xC8), rgb_t(0x6B,0xC6,0xD9), rgb_t(0x7C,0xD7,0xEA), \
    rgb_t(0x8D,0xE8,0xFB), rgb_t(0x9E,0xF9,0xFF), rgb_t(0xAF,0xFF,0xFF), rgb_t(0xC0,0xFF,0xFF   )

#define NTSC_GREEN_BLUE \
    rgb_t(0x00,0x25,0x0B), rgb_t(0x00,0x36,0x10), rgb_t(0x00,0x47,0x18), rgb_t(0x00,0x58,0x2E), \
    rgb_t(0x07,0x69,0x42), rgb_t(0x18,0x7A,0x53), rgb_t(0x29,0x8B,0x64), rgb_t(0x3A,0x9C,0x75), \
    rgb_t(0x4B,0xAD,0x86), rgb_t(0x5C,0xBE,0x97), rgb_t(0x6D,0xCF,0xA8), rgb_t(0x7E,0xE0,0xB9), \
    rgb_t(0x8F,0xF1,0xCA), rgb_t(0xA0,0xFF,0xDA), rgb_t(0xB1,0xFF,0xE6), rgb_t(0xC2,0xFF,0xF2   )

#define NTSC_GREEN \
    rgb_t(0x00,0x27,0x0C), rgb_t(0x00,0x38,0x11), rgb_t(0x00,0x49,0x16), rgb_t(0x04,0x5A,0x1A), \
    rgb_t(0x15,0x6B,0x1A), rgb_t(0x26,0x7C,0x22), rgb_t(0x37,0x8D,0x33), rgb_t(0x48,0x9E,0x44), \
    rgb_t(0x59,0xAF,0x55), rgb_t(0x6A,0xC0,0x66), rgb_t(0x7B,0xD1,0x77), rgb_t(0x8C,0xE2,0x88), \
    rgb_t(0x9D,0xF3,0x99), rgb_t(0xAE,0xFF,0xA8), rgb_t(0xBF,0xFF,0xB4), rgb_t(0xD0,0xFF,0xC0   )

#define NTSC_YELLOW_GREEN \
    rgb_t(0x00,0x21,0x0A), rgb_t(0x00,0x32,0x0F), rgb_t(0x0A,0x43,0x11), rgb_t(0x1B,0x54,0x11), \
    rgb_t(0x2C,0x65,0x11), rgb_t(0x3D,0x76,0x11), rgb_t(0x4E,0x87,0x11), rgb_t(0x5F,0x98,0x1E), \
    rgb_t(0x70,0xA9,0x2F), rgb_t(0x81,0xBA,0x40), rgb_t(0x92,0xCB,0x51), rgb_t(0xA3,0xDC,0x62), \
    rgb_t(0xB4,0xED,0x73), rgb_t(0xC5,0xFE,0x84), rgb_t(0xD6,0xFF,0x90), rgb_t(0xE7,0xFF,0x9C   )

#define NTSC_ORANGE_GREEN \
    rgb_t(0x05,0x13,0x04), rgb_t(0x16,0x24,0x04), rgb_t(0x27,0x35,0x04), rgb_t(0x38,0x46,0x04), \
    rgb_t(0x49,0x57,0x04), rgb_t(0x5A,0x68,0x04), rgb_t(0x6B,0x79,0x04), rgb_t(0x7C,0x8A,0x09), \
    rgb_t(0x8D,0x9B,0x1A), rgb_t(0x9E,0xAC,0x2B), rgb_t(0xAF,0xBD,0x3C), rgb_t(0xC0,0xCE,0x4D), \
    rgb_t(0xD1,0xDF,0x5E), rgb_t(0xE2,0xF0,0x6F), rgb_t(0xF3,0xFF,0x80), rgb_t(0xFF,0xFF,0x8D   )

#define NTSC_LIGHT_ORANGE \
    rgb_t(0x21,0x02,0x00), rgb_t(0x32,0x13,0x00), rgb_t(0x43,0x24,0x00), rgb_t(0x54,0x35,0x00), \
    rgb_t(0x65,0x46,0x00), rgb_t(0x76,0x57,0x00), rgb_t(0x87,0x68,0x00), rgb_t(0x98,0x79,0x0C), \
    rgb_t(0xA9,0x8A,0x1D), rgb_t(0xBA,0x9B,0x2E), rgb_t(0xCB,0xAC,0x3F), rgb_t(0xDC,0xBD,0x50), \
    rgb_t(0xED,0xCE,0x61), rgb_t(0xFE,0xDF,0x72), rgb_t(0xFF,0xF0,0x87), rgb_t(0xFF,0xFF,0x9D   )
***************************************************************************/


/***************************************************************************
    PALETTE - 26.7 PHASE SHIFT


define NTSC_GREY
    rgb_t(0x00,0x00,0x00), rgb_t(0x11,0x11,0x11), rgb_t(0x22,0x22,0x22), rgb_t(0x33,0x33,0x33), \
    rgb_t(0x44,0x44,0x44), rgb_t(0x55,0x55,0x55), rgb_t(0x66,0x66,0x66), rgb_t(0x77,0x77,0x77), \
    rgb_t(0x88,0x88,0x88), rgb_t(0x99,0x99,0x99), rgb_t(0xAA,0xAA,0xAA), rgb_t(0xBB,0xBB,0xBB), \
    rgb_t(0xCC,0xCC,0xCC), rgb_t(0xDD,0xDD,0xDD), rgb_t(0xEE,0xEE,0xEE), rgb_t(0xFF,0xFF,0xFF   )

define NTSC_GOLD
    rgb_t(0x1A,0x07,0x00), rgb_t(0x2B,0x18,0x00), rgb_t(0x3C,0x29,0x00), rgb_t(0x4D,0x3A,0x00), \
    rgb_t(0x5E,0x4B,0x00), rgb_t(0x6F,0x5C,0x00), rgb_t(0x80,0x6D,0x00), rgb_t(0x91,0x7E,0x09), \
    rgb_t(0xA2,0x8F,0x1A), rgb_t(0xB3,0xA0,0x2B), rgb_t(0xC4,0xB1,0x3C), rgb_t(0xD5,0xC2,0x4D), \
    rgb_t(0xE6,0xD3,0x5E), rgb_t(0xF7,0xE4,0x6F), rgb_t(0xFF,0xF5,0x83), rgb_t(0xFF,0xFF,0x97   )

define NTSC_ORANGE
    rgb_t(0x32,0x00,0x00), rgb_t(0x43,0x06,0x00), rgb_t(0x54,0x17,0x00), rgb_t(0x65,0x28,0x00), \
    rgb_t(0x79,0x39,0x00), rgb_t(0x87,0x4A,0x00), rgb_t(0x98,0x5B,0x0C), rgb_t(0xA9,0x6C,0x1D), \
    rgb_t(0xBA,0x7D,0x2E), rgb_t(0xCB,0x8E,0x3F), rgb_t(0xDC,0x9F,0x50), rgb_t(0xED,0xB0,0x61), \
    rgb_t(0xFE,0xC1,0x72), rgb_t(0xFF,0xD2,0x87), rgb_t(0xFF,0xE3,0x9E), rgb_t(0xFF,0xF4,0xB4   )

define NTSC_RED_ORANGE
    rgb_t(0x3E,0x00,0x00), rgb_t(0x4F,0x00,0x00), rgb_t(0x60,0x07,0x00), rgb_t(0x71,0x18,0x00), \
    rgb_t(0x82,0x29,0x10), rgb_t(0x93,0x3A,0x21), rgb_t(0xA4,0x4B,0x32), rgb_t(0xB5,0x5C,0x43), \
    rgb_t(0xC6,0x6D,0x54), rgb_t(0xD7,0x7E,0x65), rgb_t(0xE8,0x8F,0x76), rgb_t(0xF9,0xA0,0x87), \
    rgb_t(0xFF,0xB1,0x9C), rgb_t(0xFF,0xC2,0xB2), rgb_t(0xFF,0xD3,0xC8), rgb_t(0xFF,0xE4,0xDE   )

define NTSC_PINK
    rgb_t(0x3E,0x00,0x09), rgb_t(0x4F,0x00,0x15), rgb_t(0x60,0x00,0x21), rgb_t(0x71,0x0E,0x31), \
    rgb_t(0x82,0x1F,0x42), rgb_t(0x93,0x30,0x53), rgb_t(0xA4,0x41,0x64), rgb_t(0xB5,0x52,0x75), \
    rgb_t(0xC6,0x63,0x86), rgb_t(0xD7,0x74,0x97), rgb_t(0xE8,0x85,0xA8), rgb_t(0xF9,0x96,0xB9), \
    rgb_t(0xFF,0xA7,0xCE), rgb_t(0xFF,0xB8,0xE4), rgb_t(0xFF,0xC9,0xEF), rgb_t(0xFF,0xDA,0xF4   )

define NTSC_PURPLE
    rgb_t(0x30,0x00,0x3D), rgb_t(0x41,0x00,0x48), rgb_t(0x52,0x00,0x54), rgb_t(0x63,0x0C,0x64), \
    rgb_t(0x74,0x1D,0x75), rgb_t(0x85,0x2E,0x86), rgb_t(0x96,0x3F,0x97), rgb_t(0xA7,0x50,0xA8), \
    rgb_t(0xB8,0x61,0xB9), rgb_t(0xC9,0x72,0xCA), rgb_t(0xDA,0x83,0xDB), rgb_t(0xEB,0x94,0xE5), \
    rgb_t(0xFC,0xA5,0xE5), rgb_t(0xFF,0xB6,0xE9), rgb_t(0xFF,0xC7,0xEE), rgb_t(0xFF,0xD8,0xF3   )

define NTSC_PURPLE_BLUE
    rgb_t(0x18,0x00,0x62), rgb_t(0x29,0x00,0x6E), rgb_t(0x3A,0x01,0x7A), rgb_t(0x4B,0x12,0x8B), \
    rgb_t(0x5C,0x23,0x9C), rgb_t(0x6D,0x34,0xAD), rgb_t(0x7E,0x45,0xBE), rgb_t(0x8F,0x56,0xCF), \
    rgb_t(0xA0,0x67,0xE0), rgb_t(0xB1,0x78,0xEE), rgb_t(0xC2,0x89,0xEE), rgb_t(0xD3,0x9A,0xEE), \
    rgb_t(0xE4,0xAB,0xEE), rgb_t(0xF5,0xBC,0xEE), rgb_t(0xFF,0xCD,0xE0), rgb_t(0xFF,0xDE,0xF5   )

define NTSC_BLUE1
    rgb_t(0x00,0x00,0x72), rgb_t(0x0C,0x00,0x7F), rgb_t(0x1D,0x0E,0x8F), rgb_t(0x2E,0x1F,0xA0), \
    rgb_t(0x3F,0x30,0xB1), rgb_t(0x50,0x41,0xC2), rgb_t(0x61,0x52,0xD3), rgb_t(0x72,0x63,0xE4), \
    rgb_t(0x83,0x74,0xF5), rgb_t(0x94,0x85,0xFA), rgb_t(0xA5,0x96,0xFA), rgb_t(0xB6,0xA7,0xFA), \
    rgb_t(0xC7,0xB8,0xFA), rgb_t(0xD8,0xC9,0xFA), rgb_t(0xE9,0xDA,0xFA), rgb_t(0xFA,0xE8,0xFA   )

define NTSC_BLUE2
    rgb_t(0x00,0x00,0x62), rgb_t(0x00,0x0F,0x77), rgb_t(0x01,0x20,0x8D), rgb_t(0x12,0x31,0x9E), \
    rgb_t(0x23,0x42,0xAF), rgb_t(0x34,0x53,0xC0), rgb_t(0x45,0x64,0xD1), rgb_t(0x56,0x75,0xE2), \
    rgb_t(0x67,0x86,0xF3), rgb_t(0x78,0x97,0xFF), rgb_t(0x89,0xA8,0xFF), rgb_t(0x9A,0xB9,0xFF), \
    rgb_t(0xAB,0xCA,0xFF), rgb_t(0xBC,0xDB,0xFF), rgb_t(0xCD,0xEC,0xFF), rgb_t(0xDE,0xFD,0xFF   )

define NTSC_LIGHT_BLUE
    rgb_t(0x00,0x10,0x42), rgb_t(0x00,0x21,0x58), rgb_t(0x00,0x32,0x6E), rgb_t(0x00,0x43,0x84), \
    rgb_t(0x0E,0x54,0x96), rgb_t(0x1F,0x65,0xA7), rgb_t(0x30,0x76,0xB8), rgb_t(0x41,0x87,0xC9), \
    rgb_t(0x52,0x98,0xDA), rgb_t(0x63,0xA9,0xEB), rgb_t(0x74,0xBA,0xFC), rgb_t(0x85,0xCB,0xFF), \
    rgb_t(0x96,0xDC,0xFF), rgb_t(0xA7,0xED,0xFF), rgb_t(0xB8,0xFE,0xFF), rgb_t(0xC9,0xFF,0xFF   )

define NTSC_TURQUOISE
    rgb_t(0x00,0x1E,0x14), rgb_t(0x00,0x2F,0x2A), rgb_t(0x00,0x40,0x40), rgb_t(0x00,0x51,0x56), \
    rgb_t(0x04,0x62,0x6B), rgb_t(0x15,0x73,0x7C), rgb_t(0x26,0x84,0x8D), rgb_t(0x37,0x95,0x9E), \
    rgb_t(0x48,0xA6,0xAF), rgb_t(0x59,0xB7,0xC0), rgb_t(0x6A,0xC8,0xD1), rgb_t(0x7B,0xD9,0xE2), \
    rgb_t(0x8C,0xEA,0xF3), rgb_t(0x9D,0xFB,0xFF), rgb_t(0xAE,0xFF,0xFF), rgb_t(0xBF,0xFF,0xFF   )

define NTSC_GREEN_BLUE
    rgb_t(0x00,0x26,0x0B), rgb_t(0x00,0x37,0x10), rgb_t(0x00,0x48,0x16), rgb_t(0x00,0x59,0x25), \
    rgb_t(0x08,0x6A,0x38), rgb_t(0x19,0x7B,0x49), rgb_t(0x2A,0x8C,0x5A), rgb_t(0x3B,0x9D,0x6B), \
    rgb_t(0x4C,0xAE,0x7C), rgb_t(0x5D,0xBF,0x8D), rgb_t(0x6E,0xD0,0x9E), rgb_t(0x7F,0xE1,0xAF), \
    rgb_t(0x90,0xF2,0xC0), rgb_t(0xA1,0xFF,0xD0), rgb_t(0xB2,0xFF,0xDC), rgb_t(0xC3,0xFF,0xE8   )

define NTSC_GREEN
    rgb_t(0x00,0x26,0x0B), rgb_t(0x00,0x37,0x10), rgb_t(0x00,0x48,0x16), rgb_t(0x08,0x59,0x18), \
    rgb_t(0x19,0x6A,0x18), rgb_t(0x2A,0x7B,0x18), rgb_t(0x3B,0x8C,0x29), rgb_t(0x4C,0x9D,0x3A), \
    rgb_t(0x5D,0xAE,0x4B), rgb_t(0x6E,0xBF,0x5C), rgb_t(0x7F,0xD0,0x6D), rgb_t(0x90,0xE1,0x7E), \
    rgb_t(0xA1,0xF2,0x8F), rgb_t(0xB2,0xFF,0x9F), rgb_t(0xC3,0xFF,0xAB), rgb_t(0xD4,0xFF,0xB7   )

define NTSC_YELLOW_GREEN
    rgb_t(0x00,0x1E,0x09), rgb_t(0x00,0x2F,0x0E), rgb_t(0x11,0x40,0x0E), rgb_t(0x22,0x51,0x0E), \
    rgb_t(0x33,0x62,0x0E), rgb_t(0x44,0x73,0x0E), rgb_t(0x55,0x84,0x0E), rgb_t(0x66,0x95,0x17), \
    rgb_t(0x77,0xA6,0x28), rgb_t(0x88,0xB7,0x39), rgb_t(0x99,0xC8,0x4A), rgb_t(0xAA,0xD9,0x5B), \
    rgb_t(0xBB,0xEA,0x6C), rgb_t(0xCC,0xFB,0x7D), rgb_t(0xDD,0xFF,0x8A), rgb_t(0xEE,0xFF,0x96   )

define NTSC_ORANGE_GREEN
    rgb_t(0x0D,0x0F,0x01), rgb_t(0x1E,0x20,0x01), rgb_t(0x2F,0x31,0x01), rgb_t(0x40,0x42,0x01), \
    rgb_t(0x51,0x53,0x01), rgb_t(0x62,0x64,0x01), rgb_t(0x73,0x75,0x01), rgb_t(0x84,0x86,0x08), \
    rgb_t(0x95,0x97,0x19), rgb_t(0xA6,0xA8,0x2A), rgb_t(0xB7,0xB9,0x3B), rgb_t(0xC8,0xCA,0x4C), \
    rgb_t(0xD9,0xDB,0x5D), rgb_t(0xEA,0xEC,0x6E), rgb_t(0xFB,0xFD,0x7F), rgb_t(0xFF,0xFF,0x8F   )

define NTSC_LIGHT_ORANGE
    rgb_t(0x28,0x00,0x00), rgb_t(0x39,0x0E,0x00), rgb_t(0x4A,0x1F,0x00), rgb_t(0x5B,0x30,0x00), \
    rgb_t(0x6C,0x41,0x00), rgb_t(0x7D,0x52,0x00), rgb_t(0x8E,0x63,0x00), rgb_t(0x9F,0x74,0x10), \
    rgb_t(0xB0,0x85,0x21), rgb_t(0xC1,0x96,0x32), rgb_t(0xD2,0xA7,0x43), rgb_t(0xE3,0xB8,0x54), \
    rgb_t(0xF4,0xC9,0x65), rgb_t(0xFF,0xDA,0x78), rgb_t(0xFF,0xEB,0x8E), rgb_t(0xFF,0xFC,0xA4   )
***************************************************************************/


/***************************************************************************
    PALETTE - 27.2 PHASE SHIFT


define NTSC_GREY
    rgb_t(0x00,0x00,0x00), rgb_t(0x11,0x11,0x11), rgb_t(0x22,0x22,0x22), rgb_t(0x33,0x33,0x33), \
    rgb_t(0x44,0x44,0x44), rgb_t(0x55,0x55,0x55), rgb_t(0x66,0x66,0x66), rgb_t(0x77,0x77,0x77), \
    rgb_t(0x88,0x88,0x88), rgb_t(0x99,0x99,0x99), rgb_t(0xAA,0xAA,0xAA), rgb_t(0xBB,0xBB,0xBB), \
    rgb_t(0xCC,0xCC,0xCC), rgb_t(0xDD,0xDD,0xDD), rgb_t(0xEE,0xEE,0xEE), rgb_t(0xFF,0xFF,0xFF   )

define NTSC_GOLD
    rgb_t(0x1A,0x07,0x00), rgb_t(0x2B,0x18,0x00), rgb_t(0x3C,0x29,0x00), rgb_t(0x4D,0x3A,0x00), \
    rgb_t(0x5E,0x4B,0x00), rgb_t(0x6F,0x5C,0x00), rgb_t(0x80,0x6D,0x00), rgb_t(0x91,0x7E,0x09), \
    rgb_t(0xA2,0x8F,0x1A), rgb_t(0xB3,0xA0,0x2B), rgb_t(0xC4,0xB1,0x3C), rgb_t(0xD5,0xC2,0x4D), \
    rgb_t(0xE6,0xD3,0x5E), rgb_t(0xF7,0xE4,0x6F), rgb_t(0xFF,0xF5,0x83), rgb_t(0xFF,0xF7,0x97   )

define NTSC_ORANGE
    rgb_t(0x32,0x00,0x00), rgb_t(0x43,0x05,0x00), rgb_t(0x54,0x16,0x00), rgb_t(0x65,0x27,0x00), \
    rgb_t(0x76,0x38,0x00), rgb_t(0x87,0X49,0x00), rgb_t(0x98,0x5A,0x0C), rgb_t(0xA9,0x6B,0x1D), \
    rgb_t(0xBA,0x7C,0x2E), rgb_t(0xCB,0x8D,0x3F), rgb_t(0xDC,0x9E,0x50), rgb_t(0xED,0xAF,0x61), \
    rgb_t(0xFE,0xC0,0x72), rgb_t(0xFF,0xD1,0x88), rgb_t(0xFF,0xE2,0x9E), rgb_t(0xFF,0xF3,0xB4   )

define NTSC_RED_ORANGE
    rgb_t(0x3F,0x00,0x00), rgb_t(0x50,0x00,0x00), rgb_t(0x61,0x07,0x00), rgb_t(0x72,0x18,0x01), \
    rgb_t(0x83,0x29,0x12), rgb_t(0x94,0x3A,0x23), rgb_t(0xA5,0x4B,0x34), rgb_t(0xB6,0x5C,0x45), \
    rgb_t(0xC7,0x6D,0x56), rgb_t(0xD8,0x7E,0x67), rgb_t(0xE9,0x8F,0x78), rgb_t(0xFA,0xA0,0x89), \
    rgb_t(0xFF,0xB1,0x9E), rgb_t(0xFF,0xC2,0xB4), rgb_t(0xFF,0xD3,0xCA), rgb_t(0xFF,0xE4,0xE0   )

define NTSC_PINK
    rgb_t(0x3E,0x00,0x0C), rgb_t(0x4F,0x00,0x18), rgb_t(0x60,0x00,0x24), rgb_t(0x71,0x0E,0x34), \
    rgb_t(0x82,0x1F,0x45), rgb_t(0x93,0x30,0x56), rgb_t(0xA4,0x41,0x67), rgb_t(0xB5,0x52,0x78), \
    rgb_t(0xC6,0x63,0x89), rgb_t(0xD7,0x74,0x9A), rgb_t(0xE8,0x85,0xAB), rgb_t(0xF9,0x96,0xB6), \
    rgb_t(0xFF,0xA7,0xD1), rgb_t(0xFF,0xB8,0xE7), rgb_t(0xFF,0xC9,0xEF), rgb_t(0xFF,0xDA,0xF4   )

define NTSC_PURPLE
    rgb_t(0x2F,0x00,0x3F), rgb_t(0x40,0x00,0x4B), rgb_t(0x51,0x00,0x57), rgb_t(0x62,0x0C,0x66), \
    rgb_t(0x73,0x1D,0x77), rgb_t(0x84,0x2E,0x88), rgb_t(0x95,0x3F,0x99), rgb_t(0xA6,0x50,0xAA), \
    rgb_t(0xB7,0x61,0xBB), rgb_t(0xC8,0x72,0xCC), rgb_t(0xD9,0x83,0xDD), rgb_t(0xEA,0x94,0xE5), \
    rgb_t(0xFB,0xA5,0xE5), rgb_t(0xFF,0xB6,0xE9), rgb_t(0xFF,0xC7,0xEE), rgb_t(0xFF,0xD8,0xF3   )

define NTSC_PURPLE_BLUE
    rgb_t(0x16,0x00,0x64), rgb_t(0x27,0x00,0x70), rgb_t(0x38,0x02,0x7D), rgb_t(0x49,0x13,0x8E), \
    rgb_t(0x5A,0x24,0x9F), rgb_t(0x6B,0x35,0xB0), rgb_t(0x7C,0x46,0xC1), rgb_t(0x8D,0x57,0xD2), \
    rgb_t(0x9E,0x68,0xE3), rgb_t(0xAF,0x79,0xEF), rgb_t(0xC0,0x8A,0xEF), rgb_t(0xD1,0x9D,0xEF), \
    rgb_t(0xE2,0xAC,0xEF), rgb_t(0xF3,0xBD,0xEF), rgb_t(0xFF,0xCE,0xF0), rgb_t(0xFF,0xDF,0xF5   )

define NTSC_BLUE1
    rgb_t(0x00,0x00,0x71), rgb_t(0x09,0x00,0x7F), rgb_t(0x1A,0x10,0x90), rgb_t(0x2B,0x21,0xA1), \
    rgb_t(0x3C,0x32,0xB2), rgb_t(0x4D,0x43,0xC3), rgb_t(0x5E,0x54,0xD4), rgb_t(0x6F,0x65,0xE5), \
    rgb_t(0x80,0x76,0xF6), rgb_t(0x91,0x87,0xFC), rgb_t(0xA2,0x98,0xFC), rgb_t(0xB3,0xA9,0xFC), \
    rgb_t(0xC4,0xBA,0xFC), rgb_t(0xD5,0xCB,0xFC), rgb_t(0xE6,0xDC,0xFC), rgb_t(0xF7,0xED,0xFC   )

define NTSC_BLUE2
    rgb_t(0x00,0x00,0x5E), rgb_t(0x00,0x11,0x74), rgb_t(0x00,0x22,0x8A), rgb_t(0x0F,0x33,0x9C), \
    rgb_t(0x20,0x44,0xAD), rgb_t(0x31,0x55,0xBE), rgb_t(0x42,0x66,0xCF), rgb_t(0x53,0x77,0xE0), \
    rgb_t(0x64,0x88,0xF1), rgb_t(0x75,0x99,0xFF), rgb_t(0x86,0xAA,0xFF), rgb_t(0x97,0xBB,0xFF), \
    rgb_t(0xA8,0xCC,0xFF), rgb_t(0xB9,0xDD,0xFF), rgb_t(0xCA,0xEE,0xFF), rgb_t(0xDB,0xFF,0xFF   )

define NTSC_LIGHT_BLUE
    rgb_t(0x00,0x12,0x3B), rgb_t(0x00,0x23,0x51), rgb_t(0x00,0x34,0x68), rgb_t(0x00,0x45,0x7E), \
    rgb_t(0x0C,0x56,0x90), rgb_t(0x1D,0x67,0xA1), rgb_t(0x2E,0x78,0xB2), rgb_t(0x3F,0x89,0xC3), \
    rgb_t(0x50,0x9A,0xD4), rgb_t(0x61,0xAB,0xE5), rgb_t(0x72,0xBC,0xF6), rgb_t(0x83,0xCD,0xFF), \
    rgb_t(0x94,0xDE,0xFF), rgb_t(0xA5,0xEF,0xFF), rgb_t(0xB6,0xFF,0xFF), rgb_t(0xC7,0xFF,0xFF   )

define NTSC_TURQUOISE
    rgb_t(0x00,0x20,0x0C), rgb_t(0x00,0x31,0x22), rgb_t(0x00,0x42,0x38), rgb_t(0x00,0x53,0x4E), \
    rgb_t(0x04,0x64,0x63), rgb_t(0x15,0x75,0x74), rgb_t(0x26,0x86,0x85), rgb_t(0x37,0x97,0x96), \
    rgb_t(0x48,0xA8,0xA7), rgb_t(0x59,0xB9,0xB8), rgb_t(0x6A,0xCA,0xC9), rgb_t(0x7B,0xDB,0xDA), \
    rgb_t(0x8C,0xEC,0xEB), rgb_t(0x9D,0xFD,0xFC), rgb_t(0xAE,0xFF,0xFF), rgb_t(0xBF,0xFF,0xFF   )

define NTSC_GREEN_BLUE
    rgb_t(0x00,0x27,0x0C), rgb_t(0x00,0x38,0x11), rgb_t(0x00,0x49,0x16), rgb_t(0x00,0x5A,0x1C), \
    rgb_t(0x0B,0x6B,0x2F), rgb_t(0x1C,0x7C,0x40), rgb_t(0x2D,0x8D,0x51), rgb_t(0x3E,0x9E,0x62), \
    rgb_t(0x4F,0xAF,0x73), rgb_t(0x60,0xC0,0x84), rgb_t(0x71,0xD1,0x95), rgb_t(0x82,0xE2,0xA6), \
    rgb_t(0x93,0xF3,0xB7), rgb_t(0xA4,0xFF,0xC6), rgb_t(0xB5,0xFF,0xD2), rgb_t(0xC6,0xFF,0xDE   )

define NTSC_GREEN
    rgb_t(0x00,0x25,0x0B), rgb_t(0x00,0x36,0x10), rgb_t(0x00,0x47,0x15), rgb_t(0x0D,0x58,0x16), \
    rgb_t(0x1E,0x69,0x16), rgb_t(0x2F,0x7A,0x16), rgb_t(0x40,0x8B,0x21), rgb_t(0x51,0x9C,0x32), \
    rgb_t(0x62,0xAD,0x43), rgb_t(0x73,0xBE,0x54), rgb_t(0x84,0xCF,0x65), rgb_t(0x95,0xE0,0x76), \
    rgb_t(0xA6,0xF1,0x87), rgb_t(0xB7,0xFF,0x98), rgb_t(0xC8,0xFF,0xA3), rgb_t(0xD9,0xFF,0xAF   )

define NTSC_YELLOW_GREEN
    rgb_t(0x00,0x1B,0x08), rgb_t(0x06,0x2C,0x0B), rgb_t(0x17,0x3D,0x0B), rgb_t(0x28,0x4E,0x0B), \
    rgb_t(0x39,0x5F,0x0B), rgb_t(0x4A,0x70,0x0B), rgb_t(0x5B,0x81,0x0B), rgb_t(0x6C,0x92,0x12), \
    rgb_t(0x7D,0xA3,0x23), rgb_t(0x8E,0xB4,0x34), rgb_t(0x9F,0xC5,0x45), rgb_t(0xB0,0xD6,0x56), \
    rgb_t(0xC1,0xE7,0x67), rgb_t(0xD2,0xF8,0x78), rgb_t(0xE3,0xFF,0x86), rgb_t(0xF4,0xFF,0x92   )

define NTSC_ORANGE_GREEN
    rgb_t(0x13,0x0B,0x00), rgb_t(0x24,0x1C,0x00), rgb_t(0x35,0x2D,0x00), rgb_t(0x46,0x3E,0x00), \
    rgb_t(0x57,0x4F,0x00), rgb_t(0x68,0x60,0x00), rgb_t(0x79,0x71,0x00), rgb_t(0x8A,0x82,0x08), \
    rgb_t(0x9B,0x93,0x19), rgb_t(0xAC,0xA4,0x2A), rgb_t(0xBD,0xB5,0x3B), rgb_t(0xCE,0xC6,0x4C), \
    rgb_t(0xDF,0xD7,0x5D), rgb_t(0xF0,0xE8,0x6E), rgb_t(0xFF,0xF9,0x7F), rgb_t(0xFF,0xFF,0x92   )

define NTSC_LIGHT_ORANGE
    rgb_t(0x2D,0x00,0x00), rgb_t(0x3E,0x0A,0x00), rgb_t(0x4F,0x1B,0x00), rgb_t(0x60,0x2C,0x00), \
    rgb_t(0x71,0x3D,0x00), rgb_t(0x82,0x4E,0x00), rgb_t(0x93,0x5F,0x05), rgb_t(0xA4,0x70,0x16), \
    rgb_t(0xB5,0x81,0x27), rgb_t(0xC4,0x90,0x37), rgb_t(0xD7,0xA3,0x49), rgb_t(0xE8,0xB4,0x5A), \
    rgb_t(0xF9,0xC5,0x6B), rgb_t(0xFF,0xD6,0x80), rgb_t(0xFF,0xE7,0x96), rgb_t(0xFF,0xF8,0xAC   )
***************************************************************************/


/***************************************************************************
    PALETTE - 27.7 PHASE SHIFT


define NTSC_GREY
    rgb_t(0x00,0x00,0x00), rgb_t(0x11,0x11,0x11), rgb_t(0x22,0x22,0x22), rgb_t(0x33,0x33,0x33), \
    rgb_t(0x44,0x44,0x44), rgb_t(0x55,0x55,0x55), rgb_t(0x66,0x66,0x66), rgb_t(0x77,0x77,0x77), \
    rgb_t(0x88,0x88,0x88), rgb_t(0x99,0x99,0x99), rgb_t(0xAA,0xAA,0xAA), rgb_t(0xBB,0xBB,0xBB), \
    rgb_t(0xCC,0xCC,0xCC), rgb_t(0xDD,0xDD,0xDD), rgb_t(0xEE,0xEE,0xEE), rgb_t(0xFF,0xFF,0xFF   )

define NTSC_GOLD
    rgb_t(0x1A,0x07,0x00), rgb_t(0x2B,0x18,0x00), rgb_t(0x3C,0x29,0x00), rgb_t(0x4D,0x3A,0x00), \
    rgb_t(0x5E,0x4B,0x00), rgb_t(0x6F,0x5C,0x00), rgb_t(0x80,0x6D,0x00), rgb_t(0x91,0x7E,0x09), \
    rgb_t(0xA2,0x8F,0x1A), rgb_t(0xB3,0xA0,0x2B), rgb_t(0xC4,0xB1,0x3C), rgb_t(0xD5,0xC2,0x4D), \
    rgb_t(0xE6,0xD3,0x5E), rgb_t(0xF7,0xE4,0x6F), rgb_t(0xFF,0xF5,0x83), rgb_t(0xFF,0xFF,0x97   )

define NTSC_ORANGE
    rgb_t(0x32,0x00,0x00), rgb_t(0x43,0x05,0x00), rgb_t(0x54,0x16,0x00), rgb_t(0x65,0x27,0x00), \
    rgb_t(0x76,0x38,0x00), rgb_t(0x87,0x49,0x00), rgb_t(0x98,0x5A,0x0C), rgb_t(0xA9,0x6B,0x1D), \
    rgb_t(0xBA,0x7C,0x2E), rgb_t(0xCB,0x8D,0x3F), rgb_t(0xDC,0x9E,0x50), rgb_t(0xED,0xAF,0x61), \
    rgb_t(0xFE,0xC0,0x72), rgb_t(0xFF,0xD1,0x88), rgb_t(0xFF,0xE2,0x9E), rgb_t(0xFF,0xF3,0xB4   )

define NTSC_RED_ORANGE
    rgb_t(0x3F,0x00,0x00), rgb_t(0x50,0x00,0x00), rgb_t(0x61,0x06,0x00), rgb_t(0x72,0x17,0x03), \
    rgb_t(0x83,0x28,0x14), rgb_t(0x94,0x39,0x25), rgb_t(0xA5,0x4A,0x36), rgb_t(0xB6,0x5B,0x47), \
    rgb_t(0xC7,0x6C,0x58), rgb_t(0xD8,0x7D,0x69), rgb_t(0xE9,0x8E,0x7A), rgb_t(0xFA,0x9F,0x8B), \
    rgb_t(0xFF,0xB0,0x9F), rgb_t(0xFF,0xC1,0xB5), rgb_t(0xFF,0xD2,0xCB), rgb_t(0xFF,0xE3,0xE1   )

define NTSC_PINK
    rgb_t(0x3D,0x00,0x10), rgb_t(0x4E,0x00,0x1C), rgb_t(0x5F,0x00,0x27), rgb_t(0x70,0x0D,0x37), \
    rgb_t(0x81,0x1E,0x48), rgb_t(0x92,0x2F,0x59), rgb_t(0xA3,0x40,0x6A), rgb_t(0xB4,0x51,0x7B), \
    rgb_t(0xC5,0x62,0x8C), rgb_t(0xD6,0x73,0x9D), rgb_t(0xE7,0x84,0xAE), rgb_t(0xF8,0x95,0xBF), \
    rgb_t(0xFF,0xA6,0xD3), rgb_t(0xFF,0xB7,0xE9), rgb_t(0xFF,0xC8,0xEE), rgb_t(0xFF,0xD9,0xF4   )

define NTSC_PURPLE
    rgb_t(0x2D,0x00,0x42), rgb_t(0x3E,0x00,0x4E), rgb_t(0x4F,0x00,0x5A), rgb_t(0x60,0x0C,0x6A), \
    rgb_t(0x71,0x1D,0x7B), rgb_t(0x82,0x2E,0x8C), rgb_t(0x93,0x3F,0x9D), rgb_t(0xA4,0x50,0xAE), \
    rgb_t(0xB5,0x61,0xBF), rgb_t(0xC6,0x72,0xD0), rgb_t(0xD7,0x83,0xE1), rgb_t(0xE8,0x94,0xE6), \
    rgb_t(0xF9,0xA5,0xE6), rgb_t(0xFF,0xB6,0xE9), rgb_t(0xFF,0xC7,0xEE), rgb_t(0xFF,0xD8,0xF3   )

define NTSC_PURPLE_BLUE
    rgb_t(0x13,0x00,0x67), rgb_t(0x24,0x00,0x73), rgb_t(0x35,0x03,0x80), rgb_t(0x46,0x14,0x91), \
    rgb_t(0x57,0x25,0xA2), rgb_t(0x68,0x36,0xB3), rgb_t(0x79,0x47,0xC4), rgb_t(0x8A,0x58,0xD5), \
    rgb_t(0x9B,0x69,0xE6), rgb_t(0xAC,0x7A,0xF0), rgb_t(0xBD,0x8B,0xF0), rgb_t(0xCE,0x9C,0xF0), \
    rgb_t(0xDF,0xAD,0xF0), rgb_t(0xF0,0xBE,0xF0), rgb_t(0xFF,0xCF,0xF1), rgb_t(0xFF,0xE0,0xF6   )

define NTSC_BLUE1
    rgb_t(0x00,0x00,0x70), rgb_t(0x05,0x01,0x80), rgb_t(0x16,0x12,0x91), rgb_t(0x27,0x23,0xA2), \
    rgb_t(0x38,0x34,0xB3), rgb_t(0x49,0x45,0xC4), rgb_t(0x5A,0x56,0xD5), rgb_t(0x6B,0x67,0xE6), \
    rgb_t(0x7C,0x78,0xF7), rgb_t(0x8D,0x89,0xFE), rgb_t(0x9E,0x9A,0xFE), rgb_t(0xAF,0xAB,0xFE), \
    rgb_t(0xC0,0xBC,0xFE), rgb_t(0xD1,0xCD,0xFE), rgb_t(0xE2,0xDE,0xFE), rgb_t(0xF3,0xEF,0xFE   )

define NTSC_BLUE2
    rgb_t(0x00,0x03,0x5B), rgb_t(0x00,0x14,0x71), rgb_t(0x00,0x25,0x87), rgb_t(0x0C,0x36,0x9A), \
    rgb_t(0x1D,0x47,0xAB), rgb_t(0x2E,0x58,0xBC), rgb_t(0x3F,0x69,0xCD), rgb_t(0x50,0x7A,0xDE), \
    rgb_t(0x61,0x8B,0xEF), rgb_t(0x72,0x9C,0xFF), rgb_t(0x83,0xAD,0xFF), rgb_t(0x94,0xBE,0xFF), \
    rgb_t(0xA5,0xCF,0xFF), rgb_t(0xB6,0xE0,0xFF), rgb_t(0xC7,0xF1,0xFF), rgb_t(0xD8,0xFF,0xFF   )

define NTSC_LIGHT_BLUE
    rgb_t(0x00,0x15,0x35), rgb_t(0x00,0x26,0x4B), rgb_t(0x00,0x37,0x61), rgb_t(0x00,0x48,0x78), \
    rgb_t(0x0A,0x59,0x8B), rgb_t(0x1B,0x6A,0x9C), rgb_t(0x2C,0x7B,0xAD), rgb_t(0x3D,0x8C,0xBE), \
    rgb_t(0x4E,0x9D,0xCF), rgb_t(0x5F,0xAE,0xE0), rgb_t(0x70,0xBF,0xF1), rgb_t(0x81,0xD0,0xFF), \
    rgb_t(0x92,0xE1,0xFF), rgb_t(0xA3,0xF2,0xFF), rgb_t(0xB4,0xFF,0xFF), rgb_t(0xC5,0xFF,0xFF   )

define NTSC_TURQUOISE
    rgb_t(0x00,0x22,0x0A), rgb_t(0x00,0x33,0x19), rgb_t(0x00,0x44,0x2F), rgb_t(0x00,0x55,0x45), \
    rgb_t(0x04,0x66,0x5A), rgb_t(0x15,0x77,0x6B), rgb_t(0x26,0x88,0x7C), rgb_t(0x37,0x99,0x8D), \
    rgb_t(0x48,0xAA,0x9E), rgb_t(0x59,0xBB,0xAF), rgb_t(0x6A,0xCC,0xC0), rgb_t(0x7B,0xDD,0xD1), \
    rgb_t(0x8C,0xEE,0xE2), rgb_t(0x9D,0xFF,0xF3), rgb_t(0xAE,0xFF,0xFF), rgb_t(0xBF,0xFF,0xFF   )

define NTSC_GREEN_BLUE
    rgb_t(0x00,0x27,0x0C), rgb_t(0x00,0x38,0x11), rgb_t(0x00,0x49,0x16), rgb_t(0x00,0x5A,0x1B), \
    rgb_t(0x0D,0x6B,0x25), rgb_t(0x1E,0x7C,0x36), rgb_t(0x2F,0x8D,0x47), rgb_t(0x40,0x9E,0x58), \
    rgb_t(0x51,0xAF,0x69), rgb_t(0x62,0xC0,0x7A), rgb_t(0x73,0xD1,0x8B), rgb_t(0x84,0xE2,0x9C), \
    rgb_t(0x95,0xF3,0xAD), rgb_t(0xA6,0xFF,0xBD), rgb_t(0xB7,0xFF,0xC9), rgb_t(0xC8,0xFF,0xD4   )

define NTSC_GREEN
    rgb_t(0x00,0x24,0x0B), rgb_t(0x00,0x35,0x10), rgb_t(0x01,0x46,0x15), rgb_t(0x12,0x57,0x15), \
    rgb_t(0x23,0x68,0x15), rgb_t(0x34,0x79,0x15), rgb_t(0x45,0x8A,0x19), rgb_t(0x56,0x9B,0x2A), \
    rgb_t(0x67,0xAC,0x3B), rgb_t(0x78,0xBD,0x4C), rgb_t(0x89,0xCE,0x5D), rgb_t(0x9A,0xDF,0x6E), \
    rgb_t(0xAB,0xF0,0x7F), rgb_t(0xBC,0xFF,0x8F), rgb_t(0xCD,0xFF,0x9B), rgb_t(0xDE,0xFF,0xA7   )

define NTSC_YELLOW_GREEN
    rgb_t(0x00,0x18,0x07), rgb_t(0x00,0x29,0x0C), rgb_t(0x1E,0x3A,0x08), rgb_t(0x2F,0x4B,0x08), \
    rgb_t(0x40,0x5C,0x08), rgb_t(0x51,0x6D,0x08), rgb_t(0x62,0x7E,0x08), rgb_t(0x73,0x8F,0x0D), \
    rgb_t(0x84,0xA0,0x1E), rgb_t(0x95,0xB1,0x2F), rgb_t(0xA6,0xC2,0x40), rgb_t(0xB7,0xD3,0x51), \
    rgb_t(0xC8,0xE4,0x62), rgb_t(0xD9,0xF5,0x73), rgb_t(0xEA,0xFF,0x82), rgb_t(0xFB,0xFF,0x8E   )

define NTSC_ORANGE_GREEN
    rgb_t(0x1B,0x07,0x00), rgb_t(0x2C,0x18,0x00), rgb_t(0x3D,0x29,0x00), rgb_t(0x4E,0x3A,0x00), \
    rgb_t(0x5F,0x4B,0x00), rgb_t(0x70,0x5C,0x00), rgb_t(0x81,0x6D,0x00), rgb_t(0x92,0x7E,0x09), \
    rgb_t(0xA3,0x8F,0x1A), rgb_t(0xB4,0xA0,0x2B), rgb_t(0xC5,0xB1,0x3C), rgb_t(0xD6,0xC2,0x4D), \
    rgb_t(0xE7,0xD3,0x5E), rgb_t(0xF8,0xE4,0x6F), rgb_t(0xFF,0xF5,0x83), rgb_t(0xFF,0xFF,0x97   )

define NTSC_LIGHT_ORANGE
    rgb_t(0x33,0x00,0x00), rgb_t(0x44,0x05,0x00), rgb_t(0x55,0x16,0x00), rgb_t(0x66,0x27,0x00), \
    rgb_t(0x77,0x38,0x00), rgb_t(0x88,0x49,0x00), rgb_t(0x99,0x5A,0x0D), rgb_t(0xAA,0x6B,0x1E), \
    rgb_t(0xBB,0x7C,0x2F), rgb_t(0xCC,0x8D,0x40), rgb_t(0xDD,0x9E,0x51), rgb_t(0xEE,0xAF,0x62), \
    rgb_t(0xFF,0xC0,0x73), rgb_t(0xFF,0xD1,0x89), rgb_t(0xFF,0xE2,0x9F), rgb_t(0xFF,0xF3,0xB5   )
***************************************************************************/

/* Initialise the palette */
PALETTE_INIT_MEMBER(a7800_state, a7800)
{
	palette.set_pen_colors(0, a7800_palette, ARRAY_LENGTH(a7800_palette));
}


PALETTE_INIT_MEMBER(a7800_pal_state,a7800p)
{
	palette.set_pen_colors(0, a7800p_palette, ARRAY_LENGTH(a7800p_palette));
}


/***************************************************************************
    MACHINE DRIVERS
***************************************************************************/

void a7800_state::machine_start()
{
	save_item(NAME(m_p1_one_button));
	save_item(NAME(m_p2_one_button));
	save_item(NAME(m_bios_enabled));
	save_item(NAME(m_ctrl_lock));
	save_item(NAME(m_ctrl_reg));
	save_item(NAME(m_maria_flag));

	// set up RAM mirrors
	uint8_t *ram = reinterpret_cast<uint8_t *>(memshare("6116_2")->ptr());
	membank("zpmirror")->set_base(ram + 0x0040);
	membank("spmirror")->set_base(ram + 0x0140);

	// install additional handlers, if needed
	if (m_cart->exists())
	{
		switch (m_cart->get_cart_type())
		{
			case A78_HSC:
				// ROM+NVRAM accesses for HiScore
				m_maincpu->space(AS_PROGRAM).install_readwrite_handler(0x1000, 0x17ff, read8_delegate(FUNC(a78_cart_slot_device::read_10xx),(a78_cart_slot_device*)m_cart), write8_delegate(FUNC(a78_cart_slot_device::write_10xx),(a78_cart_slot_device*)m_cart));
				m_maincpu->space(AS_PROGRAM).install_readwrite_handler(0x3000, 0x3fff, read8_delegate(FUNC(a78_cart_slot_device::read_30xx),(a78_cart_slot_device*)m_cart), write8_delegate(FUNC(a78_cart_slot_device::write_30xx),(a78_cart_slot_device*)m_cart));
				break;
			case A78_XB_BOARD:
			case A78_TYPE0_POK450:
			case A78_TYPE1_POK450:
			case A78_TYPE6_POK450:
			case A78_TYPEA_POK450:
			case A78_VERSA_POK450:
				// POKEY and RAM regs at 0x400-0x47f
				m_maincpu->space(AS_PROGRAM).install_readwrite_handler(0x0400, 0x047f, read8_delegate(FUNC(a78_cart_slot_device::read_04xx),(a78_cart_slot_device*)m_cart), write8_delegate(FUNC(a78_cart_slot_device::write_04xx),(a78_cart_slot_device*)m_cart));
				break;
			case A78_XM_BOARD:
				// POKEY and RAM and YM regs at 0x400-0x47f
				m_maincpu->space(AS_PROGRAM).install_readwrite_handler(0x0400, 0x047f, read8_delegate(FUNC(a78_cart_slot_device::read_04xx),(a78_cart_slot_device*)m_cart), write8_delegate(FUNC(a78_cart_slot_device::write_04xx),(a78_cart_slot_device*)m_cart));
				// ROM+NVRAM accesses for HiScore
				m_maincpu->space(AS_PROGRAM).install_readwrite_handler(0x1000, 0x17ff, read8_delegate(FUNC(a78_cart_slot_device::read_10xx),(a78_cart_slot_device*)m_cart), write8_delegate(FUNC(a78_cart_slot_device::write_10xx),(a78_cart_slot_device*)m_cart));
				m_maincpu->space(AS_PROGRAM).install_readwrite_handler(0x3000, 0x3fff, read8_delegate(FUNC(a78_cart_slot_device::read_30xx),(a78_cart_slot_device*)m_cart), write8_delegate(FUNC(a78_cart_slot_device::write_30xx),(a78_cart_slot_device*)m_cart));
				break;
		}
	}
}

void a7800_state::machine_reset()
{
	m_ctrl_lock = 0;
	m_ctrl_reg = 0;
	m_maria_flag = 0;
	m_bios_enabled = 0;
}

MACHINE_CONFIG_START(a7800_state::a7800_ntsc)
	/* basic machine hardware */
	MCFG_CPU_ADD("maincpu", M6502, A7800_NTSC_Y1/8) /* 1.79 MHz (switches to 1.19 MHz on TIA or RIOT access) */
	MCFG_CPU_PROGRAM_MAP(a7800_mem)
	MCFG_TIMER_DRIVER_ADD_SCANLINE("scantimer", a7800_state, interrupt, "screen", 0, 1)

	/* video hardware */
	MCFG_SCREEN_ADD("screen", RASTER)
	MCFG_SCREEN_RAW_PARAMS( 7159090, 454, 0, 320, 263, 27, 27 + 192 + 32 )
	MCFG_SCREEN_UPDATE_DEVICE("maria", atari_maria_device, screen_update)
	MCFG_SCREEN_PALETTE("palette")

	MCFG_PALETTE_ADD("palette", ARRAY_LENGTH(a7800_palette) / 3)
	MCFG_PALETTE_INIT_OWNER(a7800_state, a7800)

	MCFG_DEVICE_ADD("maria", ATARI_MARIA, 0)
	MCFG_MARIA_DMACPU("maincpu")

	/* sound hardware */
	MCFG_SPEAKER_STANDARD_MONO("mono")
	MCFG_SOUND_TIA_ADD("tia", 31400)
	MCFG_SOUND_ROUTE(ALL_OUTPUTS, "mono", 1.00)

	/* devices */
	MCFG_DEVICE_ADD("riot", MOS6532_NEW, A7800_NTSC_Y1/8)
	MCFG_MOS6530n_IN_PA_CB(READ8(a7800_state, riot_joystick_r))
	MCFG_MOS6530n_IN_PB_CB(READ8(a7800_state, riot_console_button_r))
	MCFG_MOS6530n_OUT_PB_CB(WRITE8(a7800_state, riot_button_pullup_w))

	MCFG_A78_CARTRIDGE_ADD("cartslot", a7800_cart, nullptr)

	/* software lists */
	MCFG_SOFTWARE_LIST_ADD("cart_list","a7800")
	MCFG_SOFTWARE_LIST_FILTER("cart_list","NTSC")
MACHINE_CONFIG_END


MACHINE_CONFIG_START(a7800_pal_state::a7800_pal)
	a7800_ntsc(config);

	/* basic machine hardware */
	MCFG_CPU_MODIFY("maincpu")
	MCFG_CPU_CLOCK(CLK_PAL)
//  MCFG_TIMER_ADD_SCANLINE("scantimer", a7800_interrupt, "screen", 0, 1)

	MCFG_SCREEN_MODIFY( "screen" )
	MCFG_SCREEN_RAW_PARAMS( 7093788, 454, 0, 320, 313, 35, 35 + 228 + 32 )

	MCFG_PALETTE_MODIFY("palette")
	MCFG_PALETTE_INIT_OWNER(a7800_pal_state, a7800p)

	/* devices */
	MCFG_DEVICE_REMOVE("riot")
	MCFG_DEVICE_ADD("riot", MOS6532_NEW, CLK_PAL)
	MCFG_MOS6530n_IN_PA_CB(READ8(a7800_pal_state, riot_joystick_r))
	MCFG_MOS6530n_IN_PB_CB(READ8(a7800_pal_state, riot_console_button_r))
	MCFG_MOS6530n_OUT_PB_CB(WRITE8(a7800_pal_state, riot_button_pullup_w))

	/* software lists */
	MCFG_DEVICE_REMOVE("cart_list")
	MCFG_SOFTWARE_LIST_ADD("cart_list","a7800")
	MCFG_SOFTWARE_LIST_FILTER("cart_list","PAL")
MACHINE_CONFIG_END


/***************************************************************************
    ROM DEFINITIONS
***************************************************************************/

ROM_START( a7800 )
	ROM_REGION(0x4000, "maincpu", ROMREGION_ERASEFF)
	ROM_SYSTEM_BIOS( 0, "a7800", "Atari 7800" )
	ROMX_LOAD("7800.u7", 0x3000, 0x1000, CRC(5d13730c) SHA1(d9d134bb6b36907c615a594cc7688f7bfcef5b43), ROM_BIOS(1))
	ROM_SYSTEM_BIOS( 1, "a7800pr", "Atari 7800 (prototype with Asteroids)" )
	ROMX_LOAD("c300558-001a.u7", 0x0000, 0x4000, CRC(a0e10edf) SHA1(14584b1eafe9721804782d4b1ac3a4a7313e455f), ROM_BIOS(2))
ROM_END

ROM_START( a7800p )
	ROM_REGION(0x4000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD("7800pal.rom", 0x0000, 0x4000, CRC(d5b61170) SHA1(5a140136a16d1d83e4ff32a19409ca376a8df874))
ROM_END


/***************************************************************************
 DRIVER INIT
 ***************************************************************************/

DRIVER_INIT_MEMBER(a7800_ntsc_state, a7800_ntsc)
{
	m_ispal = false;
	m_lines = 263;
	m_p1_one_button = 1;
	m_p2_one_button = 1;
}


DRIVER_INIT_MEMBER(a7800_pal_state, a7800_pal)
{
	m_ispal = true;
	m_lines = 313;
	m_p1_one_button = 1;
	m_p2_one_button = 1;
}


/***************************************************************************
    GAME DRIVERS
***************************************************************************/

//    YEAR  NAME      PARENT    COMPAT  MACHINE     INPUT  STATE              INIT        COMPANY   FULLNAME             FLAGS
CONS( 1986, a7800,    0,        0,      a7800_ntsc, a7800, a7800_ntsc_state,  a7800_ntsc, "Atari",  "Atari 7800 (NTSC)", 0 )
CONS( 1986, a7800p,   a7800,    0,      a7800_pal,  a7800, a7800_pal_state,   a7800_pal,  "Atari",  "Atari 7800 (PAL)",  0 )
