/***************************** LICENSE START ***********************************

 Copyright 2021 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#ifndef MVQFEATURESHAPEITEM_H
#define MVQFEATURESHAPEITEM_H

#include <QGraphicsItem>
#include <QIcon>
#include <QPainter>
#include <QPixmap>
#include <QPen>

#include "MvQFeatureItem.h"
#include "MvRequest.h"

//---------------------------------
// Marker shapes
//---------------------------------

class MvQFeatureShapeItem : public MvQFeatureItem
{
public:
    MvQFeatureShapeItem(MvQFeatureType* feature, MvQPlotView* view, QGraphicsItem* parent = nullptr);
    void paint(QPainter*, const QStyleOptionGraphicsItem*, QWidget*) override;
    void resize(QRectF rect) override;
    void resizeTo(const MvQFeatureGeometry& geom) override;
    void moveBy(QPointF delta) override;
    void moveTo(const MvQFeatureGeometry& g) override;
    QString describe() const override;

protected:
    MvQFeatureShapeItem(const MvQFeatureShapeItem&);
    void initAtGeoCoord(QPointF co) override;
    void initAtScenePos(const QPointF&) override;
    virtual void updateRect();
    void adjustBRect() override;
    bool getRequestParameters() override;
    double halo() const override;
    virtual void buildShape() {}
    void adjustSizeInReq();

    float width_{30.};
    float height_{30.};
    QBrush brush_;
    QPen pen_;
    QPolygonF shape_;
    QPainterPath shapePath_;
};

class MvQFeatureRectItem : public MvQFeatureShapeItem
{
public:
    MvQFeatureRectItem(MvQFeatureType* feature, MvQPlotView* view);
    MvQFeatureItem* clone() const override;

protected:
    MvQFeatureRectItem(const MvQFeatureRectItem&);
    void buildShape() override;
};


class MvQFeatureTriangleItem : public MvQFeatureShapeItem
{
public:
    MvQFeatureTriangleItem(MvQFeatureType* feature, MvQPlotView* view);
    MvQFeatureItem* clone() const override;

protected:
    MvQFeatureTriangleItem(const MvQFeatureTriangleItem&);
    void buildShape() override;
};

class MvQFeatureDiamondItem : public MvQFeatureShapeItem
{
public:
    MvQFeatureDiamondItem(MvQFeatureType* feature, MvQPlotView* view);
    MvQFeatureItem* clone() const override;

protected:
    MvQFeatureDiamondItem(const MvQFeatureDiamondItem&);
    void buildShape() override;
};

class MvQFeatureStarItem : public MvQFeatureShapeItem
{
public:
    MvQFeatureStarItem(MvQFeatureType* feature, MvQPlotView* view);
    MvQFeatureItem* clone() const override;

protected:
    MvQFeatureStarItem(const MvQFeatureStarItem&);
    void buildShape() override;
};

class MvQFeatureEllipseItem : public MvQFeatureShapeItem
{
public:
    MvQFeatureEllipseItem(MvQFeatureType*, MvQPlotView*);
    MvQFeatureItem* clone() const override;

protected:
    MvQFeatureEllipseItem(const MvQFeatureEllipseItem&);
    void buildShape() override;
};

class MvQFeaturePlacemarkerItem : public MvQFeatureShapeItem
{
public:
    MvQFeaturePlacemarkerItem(MvQFeatureType*, MvQPlotView*);
    MvQFeatureItem* clone() const override;
    void paint(QPainter*, const QStyleOptionGraphicsItem*, QWidget*) override;

protected:
    MvQFeaturePlacemarkerItem(const MvQFeaturePlacemarkerItem&);
    void buildShape() override;
    bool getRequestParameters() override;

    bool fillHole_{true};
    QBrush holeBrush_;
    QPainterPath holePath_;
};


//-------------------------------------------------
// Geo shapes (the edges are straight lines,
// not sampled in lat-lon)
//-------------------------------------------------

class MvQFeatureGeoShapeItem : public MvQFeatureItem
{
public:
    MvQFeatureGeoShapeItem(MvQFeatureType* feature, MvQPlotView* view, QGraphicsItem* parent = nullptr);
    void paint(QPainter*, const QStyleOptionGraphicsItem*, QWidget*) override;
    void resize(QRectF rect) override;
    void resizeTo(const MvQFeatureGeometry& geom) override;
    void moveBy(QPointF delta) override;
    void moveTo(const MvQFeatureGeometry& g) override;
    void reproject() override;
    bool canBeCreatedOutOfView() const override { return false;}
    bool canBePastedOutOfView() const override { return false;}
    QString describe() const override;

protected:
    MvQFeatureGeoShapeItem(const MvQFeatureGeoShapeItem&);
    void initAtGeoCoord(QPointF co) override;
    virtual void updateRect();
    void adjustBRect() override;
    bool getRequestParameters() override;
    double halo() const override;
    virtual void reprojectShape() {}
    virtual void buildShape()=0;
    virtual void adjustShape() {}
    virtual QRectF shapeBoundingRect();

    float width_{100.};
    float height_{100.};
    QBrush brush_;
    QPen pen_;
    QPolygonF shape_;
    QPainterPath shapePath_;
    QList<QPointF> geoCoords_;
};

class MvQFeatureGeoRectItem : public MvQFeatureGeoShapeItem
{
public:
    MvQFeatureGeoRectItem(MvQFeatureType* feature, MvQPlotView* view) :
         MvQFeatureGeoShapeItem(feature, view) {}

    MvQFeatureItem* clone() const override;

protected:
    MvQFeatureGeoRectItem(const MvQFeatureGeoRectItem&);
    void reprojectShape() override;
    void buildShape() override;
    void adjustShape() override;
    void updateGeoCoordsFromShape();
};

#endif // MVQFEATURESHAPEITEM_H

