# Copyright 2017 - Brocade Communications Systems, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License");
#    you may not use this file except in compliance with the License.
#    You may obtain a copy of the License at
#
#        http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS,
#    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#    See the License for the specific language governing permissions and
#    limitations under the License.

import abc

from stevedore import driver

_EXECUTORS = {}


def cleanup():
    global _EXECUTORS
    _EXECUTORS = {}


def get_executor(exec_type):
    global _EXECUTORS

    if not _EXECUTORS.get(exec_type):
        mgr = driver.DriverManager(
            'mistral.executors',
            exec_type,
            invoke_on_load=True
        )

        _EXECUTORS[exec_type] = mgr.driver

    return _EXECUTORS[exec_type]


class Executor(object, metaclass=abc.ABCMeta):
    """Action executor interface."""

    @abc.abstractmethod
    def run_action(self, action, action_ex_id, safe_rerun, exec_ctx,
                   redelivered=False, target=None, async_=True, timeout=None):
        """Runs the given action.

        :param action: Action to run.
            An instance of mistral_lib.actions.Action.
        :param action_ex_id: Corresponding action execution id.
        :param safe_rerun: Tells if given action can be safely rerun.
        :param exec_ctx: A dict of values providing information about
            the current execution.
        :param redelivered: Tells if given action was run before on another
            executor.
        :param target: Target (group of action executors).
        :param async_: If True, run action in asynchronous mode (w/o waiting
            for completion).
        :param timeout: a period of time in seconds after which execution of
            action will be interrupted
        :return: Action result.
        """
        raise NotImplementedError()
