## This file is part of mlpy.
## DWT
   
## This code is written by Davide Albanese, <albanese@fbk.eu>.
## (C) 2009 Fondazione Bruno Kessler - Via Santa Croce 77, 38100 Trento, ITALY.

## This program is free software: you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation, either version 3 of the License, or
## (at your option) any later version.

## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.

## You should have received a copy of the GNU General Public License
## along with this program.  If not, see <http://www.gnu.org/licenses/>.

import uwtcore

__all__ = ['uwt', 'iuwt']


def uwt(x, wf, k, levels=0):
    """    
    Undecimated Wavelet Tranform
    
    :Parameters:
      x : 1d ndarray float (the length is restricted to powers of two)
        data
      wf : string ('d': daubechies, 'h': haar, 'b': bspline)
         wavelet type
      k : integer
        member of the wavelet family
        
        * daubechies : k = 4, 6, ..., 20 with k even
        * haar : the only valid choice of k is k = 2
        * bspline : k = 103, 105, 202, 204, 206, 208, 301, 303, 305 307, 309

     levels : integer
            level of the decomposition (J).
            If levels = 0 this is the value J such that the length of X
            is at least as great as the length of the level J wavelet filter,
            but less than the length of the level J+1 wavelet filter.
            Thus, j <= log_2((n-1)/(l-1)+1), where n is the length of x.

    :Returns:
      X : 2d ndarray float (2J x len(x))
        undecimated wavelet transformed data

        Data::
        
          [[wavelet coefficients W_1]
           [wavelet coefficients W_2]
                         :
           [wavelet coefficients W_J]
           [scaling coefficients V_1]
           [scaling coefficients V_2]
                        :
           [scaling coefficients V_J]]
         
    Example:
    
    >>> import numpy as np
    >>> import mlpy
    >>> x = np.array([1,2,3,4,3,2,1,0])
    >>> mlpy.uwt(x=x, wf='d', k=6, levels=0)
    array([[ 0.0498175 ,  0.22046721,  0.2001825 , -0.47046721, -0.0498175 ,
            -0.22046721, -0.2001825 ,  0.47046721],
           [ 0.28786838,  0.8994525 ,  2.16140162,  3.23241633,  3.71213162,
             3.1005475 ,  1.83859838,  0.76758367]])
    """

    return uwtcore.uwt(x, wf, k)


def iuwt(X, wf, k):
    """
    Inverse Undecimated Wavelet Tranform
    
    :Parameters:
      X : 2d ndarray float
        undecimated wavelet transformed data
      wf : string ('d': daubechies, 'h': haar, 'b': bspline)
         wavelet type
      k : integer
        member of the wavelet family
        
        * daubechies : k = 4, 6, ..., 20 with k even
        * haar : the only valid choice of k is k = 2
        * bspline : k = 103, 105, 202, 204, 206, 208, 301, 303, 305 307, 309
 
    :Returns:
      x : 1d ndarray float
        data

    Example:
    
    >>> import numpy as np
    >>> import mlpy
    >>> X = np.array([[ 0.0498175 ,  0.22046721,  0.2001825 , -0.47046721, -0.0498175,
    ...                -0.22046721, -0.2001825 ,  0.47046721],
    ...               [ 0.28786838,  0.8994525 ,  2.16140162,  3.23241633,  3.71213162,
    ...                 3.1005475 ,  1.83859838,  0.76758367]])
    >>> mlpy.iuwt(X=X, wf='d', k=6)
    array([  1.00000000e+00,   2.00000000e+00,   3.00000000e+00,
             4.00000000e+00,   3.00000000e+00,   2.00000000e+00,
             1.00000000e+00,   2.29246158e-09])
    """

    return uwtcore.iuwt(X, wf, k)
