# libraryview.py
#
#   Copyright (C) 2003 Daniel Burrows <dburrows@debian.org>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# A class that handles the main window of the program.  Currently, it
# modifies and stores configuration data based on the assumption that
# it is the only "view" that is open, meaning that you cannot easily
# create more than one instance without Weird Stuff happening.

import gtk
import gtk.glade
import gtk.gdk
import gobject
import operator
import os
import string
import sys

import config
import filecollection
import filelist
import filestore # For filestore.SaveError
import library
import organize
import libraryeditor
import progress
import status
import tags
import treemodel
import toplevel
import types
import undo

# A list of hooks which are called for each new library view.
new_view_hooks=[]

def add_new_view_hook(f):
    """Add a new function to be called whenever a library view is created.

    The view is passed as an argument to the function.  If any views
    have already been created, the function will immediately be called
    on them."""

    new_view_hooks.append(f)

# Tags that are visible in the tree view.  They are stored as a
# colon-separated list.
#
# adjust some tag visibilities with organization changes?
config.add_option('LibraryView', 'VisibleColumns',
                  ['tracknumber', 'genre'],
                  lambda x:
                      type(x)==types.ListType and
                      reduce(lambda a,b:a and b,
                             map(lambda y:tags.known_tags.has_key(y),
                                 x)))

# The current organization of the view.
config.add_option('LibraryView', 'Organization',
                  'ArtistAlbum',
                  lambda x:
                      type(x)==types.StringType and
                      organize.CANNED_ORGANIZATIONS.has_key(x))

# The last few (5?) libraries that were loaded.
config.add_option('LibraryView', 'PastLibraries',
                  [],
                  lambda x:
                      type(x)==types.ListType and
                      reduce(operator.__and__,
                             map(lambda y:(isinstance(y, tuple) and
                                           len(y) == 1 and
                                           isinstance(y[0], basestring)) or
                                 libraryeditor.valid_library(y), x),
                             True))

# Column generators compartmentalize the generation of columns.  They
# contain information about how to create a column in the model.  The
# get_type() method on a column generator returns the GType which is
# appropriate for the column, and the get_value() method returns a
# value of the appropriate type.

# A class to generate a column from a string tag.  If several entries
# are available, the first one is arbitrarily chosen.  (a Tag object
# (as below) is stored)
class StringTagColumn:
    def __init__(self, tag):
        self.tag=tag

    def get_title(self):
        return self.tag.title

    def get_tag(self):
        return self.tag.tag

    def get_type(self):
        return gobject.TYPE_STRING

    def get_value(self, f):
        vals=f.get_tag(self.tag.tag)
        if len(vals)>0:
            return vals[0]
        else:
            return None

    def set_value(self, f, val):
        f.set_tag(self.tag.tag, val)

class MusicLibraryView:
    def __init__(self, store, glade_location):
        toplevel.add(self)

        self.store=store
        self.current_organization=organize.CANNED_ORGANIZATIONS[config.get_option('LibraryView', 'Organization')]

        self.glade_location=glade_location
        xml=gtk.glade.XML(glade_location, root='main_window')

        # Connect up the signals.

        # This (ab)uses the undo-manager to manage other things that need
        # to be updated every time a file's state changes.
        self.undo_manager=undo.UndoManager(self.__cleanup_after_change,
                                           self.__cleanup_after_change,
                                           self.__cleanup_after_change)

        xml.signal_autoconnect({'close' : self.close,
                                'quit_program' : self.quit_program,
                                'on_row_activated' : self.handle_row_activated,
                                'on_open_directory1_activate' : self.handle_open_directory,
                                'on_edit_libraries_activate' : self.handle_edit_libraries,
                                'on_create_library_activate' : self.handle_create_library,
                                'on_revert_changes1_activate' : self.handle_revert_changes,
                                'on_save_changes' : self.handle_save_changes,
                                'on_undo1_activate' : self.undo_manager.undo,
                                'on_redo1_activate' : self.undo_manager.redo,
                                'on_list_button_press' : self.handle_button_press})

        # Extract widgets from the tree
        self.music_list=xml.get_widget('music_list')
        self.status=status.Status(xml.get_widget('statusbar1'))
        self.status_progress=xml.get_widget('status_progress')
        self.status_progress.hide()
        self.toolbar=xml.get_widget('toolbar')
        self.main_menu=xml.get_widget('main_menu')
        self.file_menu=xml.get_widget('file_menu').get_submenu()
        self.edit_menu=xml.get_widget('edit_menu').get_submenu()
        self.view_menu=xml.get_widget('view_menu').get_submenu()
        self.file_open_directory_item=xml.get_widget('open_directory_item')
        self.file_open_library_item=xml.get_widget('open_library_menu')
        self.file_open_recent_item=xml.get_widget('open_recent_menu')
        self.file_save_item=xml.get_widget('save_changes_item')
        self.file_revert_item=xml.get_widget('discard_changes_item')
        self.undo_item=xml.get_widget('undo_item')
        self.redo_item=xml.get_widget('redo_item')
        self.organizeview_item=xml.get_widget('organize_view')
        self.showhide_item=xml.get_widget('show/hide_columns')
        self.main_widget=xml.get_widget('main_window')

        # FIXME: the text on toolbar buttons doesn't match the text in
        # the menu for space reasons, but it's ugly to have this
        # discrepancy.
        self.open_button=xml.get_widget('toolbar_open')
        self.save_button=xml.get_widget('toolbar_save')
        self.revert_button=xml.get_widget('toolbar_revert')
        self.undo_button=xml.get_widget('toolbar_undo')
        self.redo_button=xml.get_widget('toolbar_redo')

        self.update_undo_sensitivity()

        # Make sure the recent items start out hidden/insensitive.
        self.file_open_recent_item.hide()
        self.file_open_recent_item.set_sensitive(0)

        # Set up the context menu.
        self.context_menu=gtk.Menu()
        self.tooltips=gtk.Tooltips()

        self.library_name=None
        self.new_library_name=None

        self.past_libraries=[]
        pl=config.get_option('LibraryView', 'PastLibraries')
        # This line is needed so that the list of past libraries
        # doesn't flip every time it's loaded:
        pl.reverse()
        for x in pl:
            self.add_past_library(x)

        config.add_listener('General', 'DefinedLibraries',
                            self.update_library_menu)
        self.update_library_menu()

        self.selection=self.music_list.get_selection()
        # Allow multiple selection
        self.selection.set_mode(gtk.SELECTION_MULTIPLE)

        self.model=None
        self.library=None

        self.building_tree=False
        self.restart_build_tree=False
        self.destroyed=False

        self.toolbar_customized=False

        self.update_organizations()

        self.update_known_tags()

        selection=self.music_list.get_selection()
        # Allow multiple selection
        selection.set_mode(gtk.SELECTION_MULTIPLE)

        # Call hooks
        for f in new_view_hooks:
            f(self)

        # set_column_drag_function isn't wrapped -- use this once it
        # is:
        #
        #def handledrop(view, col, prev, next):
        #    return prev <> None
        #
        #self.music_list.set_column_drag_function(handledrop)

        for x in [self.file_save_item, self.save_button,
                  self.file_revert_item, self.revert_button]:
            x.set_sensitive(False)
        self.status.set_message('You have not opened a library')

        config.add_listener('General', 'DefinedLibraries',
                            self.handle_library_edited)

        libraryeditor.library_edits.add_listener(self.handle_library_name_changes)

    def handle_library_name_changes(self, changes):
        for old_name,new_name in changes:
            if old_name == self.new_library_name:
                self.new_library_name = new_name

    def handle_button_press(self, treeview, event):
        if event.type == gtk.gdk.BUTTON_PRESS and event.button == 3 and len(self.context_menu.get_children())>0:
            loc=treeview.get_path_at_pos(int(event.x), int(event.y))
            if not loc:
                return False

            path,col,x,y=loc

            treeview.grab_focus()
            treeview.set_cursor(path, col, False)

            model=treeview.get_model()
            iter=model.get_iter(path)
            obj=model.get_value(iter, treemodel.COLUMN_PYOBJ)

            # Disgusting, but pygtk provides no way to pass data at
            # popup time :-(
            self.context_menu.popup_data=obj,col,path
            self.context_menu.popup(None, None, None, event.button, event.time)

            return True

        return False

    def handle_row_activated(self, treeview, path, column):
        if not path:
            return False

        treeview.grab_focus()
        treeview.set_cursor(path, column, False)

        model=treeview.get_model()
        iter=model.get_iter(path)
        obj=model.get_value(iter, treemodel.COLUMN_PYOBJ)

        self.do_edit_cell(obj, column, path)

        return True

    def add_menubar_item(self, item):
        """Adds the given menu item to the menu bar."""
        return self.main_menu.append(item)

    def add_edit_menu_item(self, item):
        """Adds the given menu item to the end of the edit menu."""
        return self.edit_menu.append(item)

    def add_toolbar_item(self, text, tooltip_text, tooltip_private_text, icon, callback, *args):
        """Adds a new item to the toolbar in the main window.

        The arguments to this function are identical to the arguments
        to gtk.Toolbar.append_item()."""

        if self.toolbar_customized == False:
            self.toolbar.insert(gtk.SeparatorToolItem(), -1)
            self.toolbar_customized=True

        rval=gtk.ToolButton(icon, text)
        self.toolbar.insert(rval, -1)
        rval.connect('clicked', callback, *args)
        self.tooltips.set_tip(rval, tooltip_text, tooltip_private_text)
        return rval

    def add_context_menu_item(self, title, tooltip, icon, callback):
        """Adds a new item to the music list's popup context menu.
        The callback is called with three arguments: the object
        representing the row that was clicked on, an object (a
        GtkCellRenderer) representing the column was clicked on, and a
        tree path representing the tree item that was clicked on."""

        if icon:
            item=gtk.ImageMenuItem(title)
            im=gtk.Image()
            im.set_from_stock(icon, gtk.ICON_SIZE_MENU)
            im.show()
            item.set_image(im)
        else:
            item=gtk.MenuItem(title)

        self.tooltips.set_tip(item, tooltip, '')

        item.connect('activate',
                     lambda item:apply(callback,self.context_menu.popup_data))

        item.show()

        self.context_menu.append(item)

    # Maybe this should be done by inheriting from a widget and using
    # standard GTK signal connection?
    def add_selection_listener(self, f):
        """Registers f as a callback for changes to the selection.

        Whenever the selection changes, f will be called with the view
        as a parameter."""

        self.selection.connect('changed', lambda *args:f(self))

    def purge_empty(self):
        """Removes any empty groups from the tree.  This should be
        done after changing tags. (it is not done automatically
        because it is expensive, so it should only be called once if
        you update lots of files)"""

        self.toplevel.purge_empty()

    def get_selected_objects(self):
        """Return a list of the UI objects which are currently selected."""


        if self.model==None:
            return []
        else:
            rval=[]
            self.selection.selected_foreach(lambda model,path,iter:rval.append(self.model.get_value(iter, treemodel.COLUMN_PYOBJ)))

            return rval

    def get_selected_files(self):
        """Return a list of the files which are currently selected.

        This is a list of file objects from the library.  They are not
        guaranteed to be unique."""

        rval=[]
        for obj in self.get_selected_objects():
            obj.add_underlying_files(rval)

        return rval

    def create_library_menu(self):
        menu=gtk.Menu()

        defined_libraries=config.get_option('General', 'DefinedLibraries')

        keys=defined_libraries.keys()
        if len(keys)>0:
            keys.sort()
            count=0
            for name in keys:
                count+=1
                w=gtk.MenuItem('_%d. %s'%(count, name))
                w.connect('activate',
                          lambda w, name: self.open_library((name,)),
                          name)
                w.show()
                menu.append(w)

            w=gtk.SeparatorMenuItem()
            w.show()
            menu.append(w)

        w=gtk.MenuItem('_Create Library From View...')
        self.tooltips.set_tip(w, 'Create a library containing the files that are visible in this window.')
        w.connect('activate', self.handle_create_library)
        w.show()
        menu.append(w)

        return menu

    def update_library_menu(self, *args):
        """Regenerates the 'Libraries' menu using the current
        configuration setting."""

        menu=self.create_library_menu()

        self.file_open_library_item.set_submenu(menu)

    # Adds the given library to the stored set, and the File menu
    # if appropriate.
    def add_past_library(self, lib):
        # Backwards compatibility.
        if isinstance(lib, basestring):
            lib=[lib]

        if not isinstance(lib, tuple):
            # Support non-list mutable sequences
            for i in range(0, len(lib)):
                lib[i]=os.path.realpath(lib[i])

        def libname(lib):
            if isinstance(lib, tuple):
                assert(len(lib)==1)
                return '"%s"'%lib[0]

            else:
                rval=','.join(lib)
                # semi-arbitrary limit
                if len(rval)>30:
                    rval=','.join(map(lambda x:'.../%s'%os.path.basename(x),
                                      lib))

                    if len(rval)>30:
                        rval='%s,...'%lib[0]

                        if len(rval)>30:
                            rval='../%s,...'%os.path.basename(lib[0])

                return rval

        menu=gtk.Menu()

        self.past_libraries=([lib]+filter(lambda x:x<>lib,
                                          self.past_libraries))[:6]

        self.file_open_recent_item.set_sensitive(1)
        self.file_open_recent_item.show()

        count=0
        for x in self.past_libraries:
            count+=1
            w=gtk.MenuItem('_%d. %s'%(count, libname(x)))
            w.connect('activate', lambda w, l:self.open_library(l), x)
            w.show()
            menu.append(w)

        self.file_open_recent_item.set_submenu(menu)

        config.set_option('LibraryView', 'PastLibraries', self.past_libraries)

    # Should these be here?  Would deriving the GUI window from
    # UndoManager be better?
    def open_undo_group(self):
        self.undo_manager.open_undo_group()

    def close_undo_group(self):
        self.undo_manager.close_undo_group()

    def __cleanup_after_change(self):
        self.update_undo_sensitivity()
        self.update_saverevert_sensitivity()

        self.toplevel.purge_empty()

    # Sometimes does slightly more than necessary, but safer than
    # trying to be overly clever.
    def update_undo_sensitivity(self):
        self.undo_item.set_sensitive(self.undo_manager.has_undos())
        self.undo_button.set_sensitive(self.undo_manager.has_undos())

        self.redo_item.set_sensitive(self.undo_manager.has_redos())
        self.redo_button.set_sensitive(self.undo_manager.has_redos())

    def update_saverevert_sensitivity(self):
        sensitive=self.library.modified_count()>0

        self.file_save_item.set_sensitive(sensitive)
        self.save_button.set_sensitive(sensitive)

        self.file_revert_item.set_sensitive(sensitive)
        self.revert_button.set_sensitive(sensitive)

    # TODO: check for modifications and pop up a dialog as appropriate
    # For now, just quit the program when a window is closed (need to
    # be cleverer, maybe even kill file->quit)
    def close(self, *args):
        self.destroyed=True
        toplevel.remove(self)

    # main_quit is just broken and doesn't behave.
    def quit_program(self,*args):
        sys.exit(0)

    def column_toggled(self, menu_item, tag):
        nowvisible=menu_item.get_active()

        currcols=config.get_option('LibraryView',
                                   'VisibleColumns')

        if nowvisible:
            if not tag in currcols:
                currcols.append(tag)
        else:
            currcols.remove(tag)

        config.set_option('LibraryView', 'VisibleColumns',
                          currcols)

        self.guicolumns[tag].set_visible(menu_item.get_active())

    # Handles a change to the set of organizations.  Used to construct
    # the "organizations" menu up front, and to select the appropriate
    # initial value (from "config").
    def update_organizations(self):
        # Can I sort them in a better way?
        organizations=organize.CANNED_ORGANIZATIONS.keys()
        organizations.sort(lambda a,b:cmp(organize.CANNED_ORGANIZATIONS[a][2],
                                          organize.CANNED_ORGANIZATIONS[b][2]))

        menu=gtk.Menu()

        curr_organization=config.get_option('LibraryView', 'Organization')

        # taking this from the pygtk demo -- not sure how it works in
        # general :-/
        group=None

        for o in organizations:
            title=organize.CANNED_ORGANIZATIONS[o][2]

            menuitem=gtk.RadioMenuItem(group, title)

            if o == curr_organization:
                menuitem.set_active(o == curr_organization)

            group=menuitem
            menuitem.show()

            menu.add(menuitem)

            menuitem.connect('activate', self.choose_canned_organization, o)

        self.organizeview_item.set_submenu(menu)


    # Handles a change to the set of known tags
    def update_known_tags(self):
        self.extracolumns=map(lambda x:StringTagColumn(x),
                              tags.known_tags.values())
        self.extracolumns.sort(lambda a,b:cmp(a.get_title(),b.get_title()))
        self.guicolumns={}
        self.tagmenuitems={}

        # Create a menu for toggling column visibility.
        menu=gtk.Menu()

        renderer=gtk.CellRendererText()
        renderer.connect('edited', self.handle_edit, None)
        renderer.set_property('editable', False)
        col=gtk.TreeViewColumn('Title',
                               renderer,
                               cell_background=treemodel.COLUMN_BACKGROUND,
                               text=treemodel.COLUMN_LABEL)
        self.music_list.append_column(col)

        initial_visible_columns=config.get_option('LibraryView', 'VisibleColumns')

        # Add the remaining columns
        for n in range(0, len(self.extracolumns)):
            col=self.extracolumns[n]

            # Set up a menu item for this.
            is_visible=(col.get_tag() in initial_visible_columns)
            menuitem=gtk.CheckMenuItem(col.get_title())
            menuitem.set_active(is_visible)
            menuitem.show()
            menuitem.connect('activate', self.column_toggled, col.get_tag())
            menu.add(menuitem)

            src=n+treemodel.COLUMN_FIRST_NONRESERVED
            renderer=gtk.CellRendererText()
            renderer.connect('edited', self.handle_edit, col)
            renderer.connect('editing_canceled', self.handle_edit_cancel, col)
            renderer.set_property('editable', False)
            gcol=gtk.TreeViewColumn(col.get_title(),
                                    renderer,
                                    text=src,
                                    cell_background=treemodel.COLUMN_BACKGROUND)
            gcol.set_visible(is_visible)
            gcol.set_reorderable(True)
            self.guicolumns[col.get_tag()]=gcol
            self.music_list.append_column(gcol)

        self.showhide_item.set_submenu(menu)

    def set_organization(self, organization):
        if organization <> self.current_organization:
            self.current_organization=organization
            self.build_tree()

    # signal handler:
    def choose_canned_organization(self, widget, name):
        # GTK+ bug: "activate" is emitted for a widget when it's
        # either activated OR deactivated.
        if widget.get_active():
            # FIXME: show an error dialog if the name doesn't exist.
            self.set_organization(organize.CANNED_ORGANIZATIONS[name])
            config.set_option('LibraryView', 'Organization', name)

    def handle_edit(self, cell, path_string, new_text, column):
        """Handle an edit of a field.  'column' is a column generator
        or None for the label column."""
        cell.set_property('editable', 0)
        iter=self.model.get_iter(path_string)
        rowobj=self.model.get_value(iter, treemodel.COLUMN_PYOBJ)

        # Handle the label specially
        if column==None:
            if new_text <> '' and not rowobj.parent.validate_child_label(rowobj, new_text):
                return
            rowobj.parent.set_child_label(rowobj, new_text)
        else:
            tag=tags.known_tags[column.get_tag()]
            # hm, should '' be handled specially? ew.
            if new_text <> '' and not tag.validate(rowobj, new_text):
                return
            column.set_value(rowobj, new_text)

        # find the root and purge empty groups from it
        #
        # why not just use toplevel?
        root=rowobj
        while root.parent:
            root=root.parent

        root.purge_empty()

        self.model.sort_column_changed()

    def handle_edit_cancel(self, cell):
        """Handle a cancelled edit of a cell, mainly by making it un-editable again."""
        cell.set_property('editable', 0)

    def do_edit_cell(self, row, col, path):
        col.get_cell_renderers()[0].set_property('editable', 1)
        self.music_list.set_cursor(path, col, True)

    # Handle the "open directory" menu function.
    def handle_open_directory(self, *args):
        def on_ok(widget):
            fn=filesel.get_filename()
            filesel.destroy()

            if not os.path.isdir(fn):
                msgdlg=gtk.MessageDialog(None, 0, gtk.MESSAGE_WARNING,
                                         gtk.BUTTONS_OK, 'You must pick a directory' )
                msgdlg.show()
                msgdlg.connect('response', lambda *args:self.handle_open_directory())
                msgdlg.connect('response', lambda *args:msgdlg.destroy())
            else:
                self.open_library(fn)

        filesel=gtk.FileSelection("Choose a directory")
        filesel.ok_button.connect('clicked', on_ok)
        filesel.cancel_button.connect('clicked', lambda *args:filesel.destroy())
        filesel.show()

    # Handle the "edit libraries" menu function.
    #
    # Right now you can only edit the "defined" libraries; there's no
    # way to get at an "implicitly defined" library.  That would be
    # nice but a pain to code.
    def handle_edit_libraries(self, *args):
        libraryeditor.show_library_editor(self.glade_location)

    # Handle the "create library" menu function.
    def handle_create_library(self, *args):
        if self.library == None:
            dirs=[]
        else:
            dirs=self.library.dirs
        libraryeditor.show_library_editor(self.glade_location,
                                          'New Library',
                                          dirs)

    # Handle the "revert changes" menu function.
    def handle_revert_changes(self, widget):
        self.status.push()
        try:
            self.library.revert(progress.ProgressUpdater("Discarding changes",
                                                         self.show_progress))
            self.toplevel.purge_empty()
        finally:
            self.status_progress.hide()
            self.status.pop()
            self.__cleanup_after_change()

    # Handle the "save changes" menu function.
    def handle_save_changes(self, widget):
        self.status.push()
        try:
            try:
                self.library.commit(progress.ProgressUpdater("Saving changes",
                                                             self.show_progress))
            except filestore.SaveError, e:
                m=gtk.MessageDialog(type=gtk.MESSAGE_ERROR,
                                    buttons=gtk.BUTTONS_OK,
                                    message_format=e.strerror)

                m.connect('response', lambda *args:m.destroy())

                m.show()
        finally:
            self.status_progress.hide()
            self.status.pop()
            self.__cleanup_after_change()

    # Set the progress display to the given amount, with the given
    # percentage and message.
    #
    # Assumes it needs to call gtk_poll()
    def show_progress(self, message, percent):
        if self.destroyed:
            return
        self.status_progress.show()

        if percent <> None:
            self.status.set_message('%s: %d%%'%(message, int(percent*100)))
            self.status_progress.set_fraction(percent)
        else:
            self.status.set_message('%s'%message)
            self.status_progress.pulse()

        while(gtk.events_pending()):
            gtk.main_iteration_do(False)

    # makegroup takes a model argument and returns the base group.
    def build_tree(self, callback=None):
        if not self.library:
            return

        # pygtk is disgustingly broken; this "restart_build_tree"
        # business is an attempt to fake what would happen if it
        # correctly passed exceptions through gtk.mainiter().
        if self.building_tree:
            self.restart_build_tree=True
            return

        self.status.push()

        try:
            success=False
            self.building_tree=True
            while not success and not self.destroyed:
                self.restart_build_tree=False

                # Needed here because self isn't bound in the argument list:
                if callback==None:
                    callback=progress.ProgressUpdater("Building music tree",
                                                      self.show_progress)

                self.model=apply(gtk.TreeStore,[gobject.TYPE_PYOBJECT,
                                                gobject.TYPE_STRING,
                                                gobject.TYPE_STRING]+map(lambda x:x.get_type(),
                                                                         self.extracolumns))

                self.toplevel=self.current_organization[0](self.model,
                                                           self,
                                                           self.extracolumns)

                cur=0
                max=len(self.library.files)
                for file in self.library.files:
                    callback(cur, max)

                    if self.restart_build_tree or self.destroyed:
                        break

                    cur+=1
                    self.toplevel.add_file(file)

                if self.restart_build_tree or self.destroyed:
                    continue

                callback(max, max)

                if self.restart_build_tree or self.destroyed:
                    continue
                success=True

                # I'd like to have these up above in order to provide more
                # visual feedback while building the tree.  Unfortunately,
                # adding items to a tree with a sorting function is
                # hideously expensive, so I have to do this here (IMO this
                # is a bug in the TreeStore)
                #
                # eg: with 660 items, if I set the sorter before building
                # the tree, it is called 26000 times; if I set it here, it
                # is called 2300 times.  For 96 items it is called 3975
                # and 445 times, respectively.
                self.model.set_sort_func(0, self.current_organization[1])
                self.model.set_sort_column_id(0, gtk.SORT_ASCENDING)
                self.music_list.set_model(self.model)
                # Changing the organization can change the optimal
                # column width:
                self.music_list.columns_autosize()

                self.status_progress.hide()

            self.building_tree=False
        finally:
            if not self.destroyed:
                self.status.pop()

    def __do_add_undo(self, file, olddict):
        # Only add an undo item if it actually changed.  This avoids
        # adding undo items for, eg, changes to the "modified" flag.
        if olddict <> file.comments:
            self.undo_manager.add_undo(undo.TagUndo(file, olddict, file.comments))

    def handle_library_edited(self, section, name, old_val, new_val):
        """When a user-defined library is edited, this routine will
        regenerate the main tree model as appropriate."""

        if self.library_name == None:
            return

        assert(old_val.has_key(self.library_name))

        if self.new_library_name == None:
            # FIXME: test this case, make it robust.
            self.model=None
            self.music_list.set_model(self.model)
            self.library_name = None
            self.new_library_name = None
            self.library = None
        elif old_val[self.library_name] <> new_val[self.new_library_name]:
            self.open_library((self.new_library_name,))
        else:
            self.library_name=self.new_library_name

    def open_library(self, lib):
        """Open the given list of directories as a library.  If lib is
        a tuple of one element, it is the name of a defined library to
        open."""

        if isinstance(lib, tuple):
            assert(len(lib)==1 and isinstance(lib[0], basestring))

            key=lib[0]

            defined_libraries=config.get_option('General', 'DefinedLibraries')

            if not defined_libraries.has_key(key):
                m=gtk.MessageDialog(type=gtk.MESSAGE_ERROR,
                                    buttons=gtk.BUTTONS_OK,
                                    message_format='The library "%s" doesn\'t seem to exist!'%key)

                m.connect('response', lambda *args:m.destroy())

                m.show()
                return

            self.add_past_library(lib)

            lib=defined_libraries[key]
            self.library_name=key
            self.new_library_name=key
        elif isinstance(lib, basestring):
            # Backwards compatibility:
            lib=[lib]
            self.add_past_library(lib)
            self.library_name=None
            self.new_library_name=None
        else:
            self.add_past_library(lib)
            self.library_name=None
            self.new_library_name=None

        self.status.push()

        try:
            # TODO: do I need to recover sensitivity if an exception occurs?
            for x in [self.file_open_library_item, self.file_open_directory_item,
                      self.file_open_recent_item, self.file_save_item,
                      self.file_revert_item, self.open_button,
                      self.save_button, self.revert_button]:
                x.set_sensitive(False)

            self.library=library.MusicLibrary(self.store, [], [])
            try:
                self.library.add_dirs(lib,
                                      map(lambda loc:progress.ProgressUpdater("Indexing music files in %s"%loc, self.show_progress),
                                          lib))
            except filestore.LoadError,e:
                m=gtk.MessageDialog(type=gtk.MESSAGE_ERROR,
                                    buttons=gtk.BUTTONS_OK,
                                    message_format=e.strerror)

                m.connect('response', lambda *args:m.destroy())

                m.show()
            except filestore.NotDirectoryError,e:
                m=gtk.MessageDialog(type=gtk.MESSAGE_ERROR,
                                    buttons=gtk.BUTTONS_OK,
                                    message_format=e.strerror)

                m.connect('response', lambda *args:m.destroy())

                m.show()

            if self.destroyed:
                return

            self.store.cache.flush()

            self.build_tree()

            if self.destroyed:
                return

            for file in self.library.files:
                file.add_listener(self.__do_add_undo)

        finally:
            if not self.destroyed:
                for x in [self.file_open_library_item, self.file_open_directory_item,
                          self.file_open_recent_item, self.open_button]:
                    x.set_sensitive(True)
                self.status.pop()

        self.status.set_message('Editing %s'%(','.join(lib)))
