# `asyncro` [![NPM](https://img.shields.io/npm/v/asyncro.svg?style=flat)](https://www.npmjs.org/package/asyncro) [![travis-ci](https://travis-ci.org/developit/asyncro.svg?branch=master)](https://travis-ci.org/developit/asyncro)

The same `map()`, `reduce()` & `filter()` you know and love, but with async iterator functions!

Do `fetch()` networking in loops, resolve Promises, anything async goes. Performance-friendly _by default_.

**Here's what it looks like:**

<img src="https://i.imgur.com/GcykVyN.png" width="404" alt="Asyncro Example">

* * *

## What's in the Box

<img src="https://i.imgur.com/yiiq6Gx.png" width="275" alt="Asyncro Example 2">

* * *

## Installation

```sh
npm install --save asyncro
```

## Import and Usage Example

```js
import { map } from 'asyncro';

async function example() {
	return await map(
		['foo', 'bar', 'baz'],
		async name => fetch('./'+name)
	)
}
```

## API

<!-- Generated by documentation.js. Update this documentation by updating the source code. -->

### reduce

Invoke an async reducer function on each item in the given Array,
where the reducer transforms an accumulator value based on each item iterated over.
**Note:** because `reduce()` is order-sensitive, iteration is sequential.

> This is an asynchronous version of `Array.prototype.reduce()`

**Parameters**

-   `array` **[Array](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array)** The Array to reduce
-   `reducer` **[Function](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/function)** Async function, gets passed `(accumulator, value, index, array)` and returns a new value for `accumulator`
-   `accumulator` **\[any]** Optional initial accumulator value

**Examples**

```javascript
await reduce(
	['/foo', '/bar', '/baz'],
	async (accumulator, value) => {
		accumulator[v] = await fetch(value);
		return accumulator;
	},
	{}
);
```

Returns **any** final `accumulator` value

### map

Invoke an async transform function on each item in the given Array **in parallel**,
returning the resulting Array of mapped/transformed items.

> This is an asynchronous, parallelized version of `Array.prototype.map()`.

**Parameters**

-   `array` **[Array](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array)** The Array to map over
-   `mapper` **[Function](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/function)** Async function, gets passed `(value, index, array)`, returns the new value.

**Examples**

```javascript
await map(
	['foo', 'baz'],
	async v => await fetch(v)
)
```

Returns **[Array](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array)** resulting mapped/transformed values.

### filter

Invoke an async filter function on each item in the given Array **in parallel**,
returning an Array of values for which the filter function returned a truthy value.

> This is an asynchronous, parallelized version of `Array.prototype.filter()`.

**Parameters**

-   `array` **[Array](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array)** The Array to filter
-   `filterer` **[Function](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/function)** Async function. Gets passed `(value, index, array)`, returns true to keep the value in the resulting filtered Array.

**Examples**

```javascript
await filter(
	['foo', 'baz'],
	async v => (await fetch(v)).ok
)
```

Returns **[Array](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array)** resulting filtered values

### find

Invoke an async function on each item in the given Array **in parallel**,
 returning the first element predicate returns truthy for.

> This is an asynchronous, parallelized version of `Array.prototype.find()`.

**Parameters**

-   `array` **[Array](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array)** The Array to find
-   `predicate` **[Function](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/function)** Async function. Gets passed `(value, index, array)`, returns true to be the find result.

**Examples**

```javascript
await find(
	['foo', 'baz', 'root'],
	async v => (await fetch(v)).name === 'baz'
)
```

Returns **any** resulting find value

### every

Checks if predicate returns truthy for **all** elements of collection **in parallel**.

> This is an asynchronous, parallelized version of `Array.prototype.every()`.

**Parameters**

-   `array` **[Array](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array)** The Array to iterate over.
-   `predicate` **[Function](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/function)** Async function. Gets passed `(value, index, array)`, The function invoked per iteration.

**Examples**

```javascript
await every(
	[2, 3],
	async v => (await fetch(v)).ok
)
```

Returns **[Boolean](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Boolean)** Returns true if **all** element passes the predicate check, else false.

### some

Checks if predicate returns truthy for **any** element of collection **in parallel**.

> This is an asynchronous, parallelized version of `Array.prototype.some()`.

**Parameters**

-   `array` **[Array](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array)** The Array to iterate over.
-   `filterer` **[Function](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/function)** Async function. Gets passed `(value, index, array)`, The function invoked per iteration.

**Examples**

```javascript
await some(
	['foo', 'baz'],
	async v => (await fetch(v)).ok
)
```

Returns **[Boolean](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Boolean)** Returns true if **any** element passes the predicate check, else false.

### parallel

Invoke all async functions in an Array or Object **in parallel**, returning the result.

**Parameters**

-   `list` **([Array](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array)&lt;[Function](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/function)> | [Object](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Object)&lt;[Function](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/function)>)** Array/Object with values that are async functions to invoke.

**Examples**

```javascript
await parallel([
	async () => await fetch('foo'),
	async () => await fetch('baz')
])
```

Returns **([Array](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array) \| [Object](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Object))** same structure as `list` input, but with values now resolved.

### series

Invoke all async functions in an Array or Object **sequentially**, returning the result.

**Parameters**

-   `list` **([Array](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array)&lt;[Function](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/function)> | [Object](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Object)&lt;[Function](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Statements/function)>)** Array/Object with values that are async functions to invoke.

**Examples**

```javascript
await series([
	async () => await fetch('foo'),
	async () => await fetch('baz')
])
```

Returns **([Array](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array) \| [Object](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Object))** same structure as `list` input, but with values now resolved.

## License

[MIT](https://oss.ninja/mit/developit)
