/**
* @license Apache-2.0
*
* Copyright (c) 2020 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

/**
* Compute the sum of double-precision floating-point strided array elements, ignoring `NaN` values and using an improved Kahan–Babuška algorithm.
*
* @module @stdlib/blas/ext/base/dnannsumkbn
*
* @example
* var Float64Array = require( '@stdlib/array/float64' );
* var dnannsumkbn = require( '@stdlib/blas/ext/base/dnannsumkbn' );
*
* var x = new Float64Array( [ 1.0, -2.0, NaN, 2.0 ] );
* var out = new Float64Array( 2 );
*
* var v = dnannsumkbn( x.length, x, 1, out, 1 );
* // returns <Float64Array>[ 1.0, 3 ]
*
* @example
* var Float64Array = require( '@stdlib/array/float64' );
* var floor = require( '@stdlib/math/base/special/floor' );
* var dnannsumkbn = require( '@stdlib/blas/ext/base/dnannsumkbn' );
*
* var x = new Float64Array( [ 2.0, 1.0, 2.0, -2.0, -2.0, 2.0, 3.0, 4.0, NaN, NaN ] );
* var out = new Float64Array( 2 );
*
* var N = floor( x.length / 2 );
*
* var v = dnannsumkbn.ndarray( N, x, 2, 1, out, 1, 0 );
* // returns <Float64Array>[ 5.0, 4 ]
*/

// MODULES //

var join = require( 'path' ).join;
var tryRequire = require( '@stdlib/utils/try-require' );
var dnannsumkbn = require( './main.js' );


// MAIN //

var tmp = tryRequire( join( __dirname, './native.js' ) );
if ( !(tmp instanceof Error) ) {
	dnannsumkbn = tmp;
}


// EXPORTS //

module.exports = dnannsumkbn;
