/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/arcsine/cdf' );
import Arcsine = require( './../../../../../base/dists/arcsine/ctor' );
import entropy = require( './../../../../../base/dists/arcsine/entropy' );
import kurtosis = require( './../../../../../base/dists/arcsine/kurtosis' );
import logcdf = require( './../../../../../base/dists/arcsine/logcdf' );
import logpdf = require( './../../../../../base/dists/arcsine/logpdf' );
import mean = require( './../../../../../base/dists/arcsine/mean' );
import median = require( './../../../../../base/dists/arcsine/median' );
import mode = require( './../../../../../base/dists/arcsine/mode' );
import pdf = require( './../../../../../base/dists/arcsine/pdf' );
import quantile = require( './../../../../../base/dists/arcsine/quantile' );
import skewness = require( './../../../../../base/dists/arcsine/skewness' );
import stdev = require( './../../../../../base/dists/arcsine/stdev' );
import variance = require( './../../../../../base/dists/arcsine/variance' );

/**
* Interface describing the `arcsine` namespace.
*/
interface Namespace {
	/**
	* Arcsine distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 5.0, 0.0, 4.0 );
	* // returns 1.0
	*
	* var mycdf = ns.cdf.factory( 0.0, 10.0 );
	* y = mycdf( 0.5 );
	* // returns ~0.144
	*
	* y = mycdf( 8.0 );
	* // returns ~0.705
	*/
	cdf: typeof cdf;

	/**
	* Arcsine Distribution.
	*/
	Arcsine: typeof Arcsine;

	/**
	* Returns the differential entropy of an arcsine distribution.
	*
	* ## Notes
	*
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns entropy
	*
	* @example
	* var v = ns.entropy( 0.0, 1.0 );
	* // returns ~-0.242
	*
	* @example
	* var v = ns.entropy( 4.0, 12.0 );
	* // returns ~1.838
	*
	* @example
	* var v = ns.entropy( -4.0, 4.0 );
	* // returns ~1.838
	*
	* @example
	* var v = ns.entropy( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN, 2.0 );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of an arcsine distribution.
	*
	* ## Notes
	*
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns excess kurtosis
	*
	* @example
	* var v = ns.kurtosis( 0.0, 1.0 );
	* // returns -1.5
	*
	* @example
	* var v = ns.kurtosis( 4.0, 12.0 );
	* // returns -1.5
	*
	* @example
	* var v = ns.kurtosis( -4.0, 4.0 );
	* // returns -1.5
	*
	* @example
	* var v = ns.kurtosis( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN, 2.0 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Arcsine distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 5.0, 0.0, 4.0 );
	* // returns 0.0
	*
	* var mylogcdf = ns.logcdf.factory( 0.0, 10.0 );
	* y = mylogcdf( 0.5 );
	* // returns ~-1.938
	*
	* y = mylogcdf( 8.0 );
	* // returns ~-0.35
	*/
	logcdf: typeof logcdf;

	/**
	* Arcsine distribution logarithm of probability density function (PDF).
	*
	* @param x - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 3.0, 2.0, 6.0 );
	* // returns ~-1.694
	*
	* var mylogpdf = ns.logpdf.factory( 6.0, 7.0 );
	* y = mylogpdf( 7.0 );
	* // returns Infinity
	*
	* y = mylogpdf( 5.0 );
	* // returns -Infinity
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value of an arcsine distribution.
	*
	* ## Notes
	*
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 0.0, 1.0 );
	* // returns 0.5
	*
	* @example
	* var v = ns.mean( 4.0, 12.0 );
	* // returns 8.0
	*
	* @example
	* var v = ns.mean( -4.0, 4.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mean( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN, 2.0 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median of an arcsine distribution.
	*
	* ## Notes
	*
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns median
	*
	* @example
	* var v = ns.median( 0.0, 1.0 );
	* // returns 0.5
	*
	* @example
	* var v = ns.median( 4.0, 12.0 );
	* // returns 8.0
	*
	* @example
	* var v = ns.median( -4.0, 4.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.median( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.median( NaN, 2.0 );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Returns the mode of an arcsine distribution.
	*
	* ## Notes
	*
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns mode
	*
	* @example
	* var v = ns.mode( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mode( 4.0, 12.0 );
	* // returns 4.0
	*
	* @example
	* var v = ns.mode( -4.0, 4.0 );
	* // returns -4.0
	*
	* @example
	* var v = ns.mode( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN, 2.0 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Arcsine distribution probability density function (PDF).
	*
	* @param x - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 3.0, 2.0, 6.0 );
	* // returns ~0.184
	*
	* var mypdf = ns.pdf.factory( 6.0, 7.0 );
	* y = mypdf( 7.0 );
	* // returns Infinity
	*
	* y = mypdf( 5.0 );
	* // returns 0.0
	*/
	pdf: typeof pdf;

	/**
	* Arcsine distribution quantile function.
	*
	* @param x - input value
	* @param a - minimum support
	* @param b - maximum support
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.5, 0.0, 10.0 );
	* // returns ~5.0
	*
	* var myQuantile = ns.quantile.factory( 0.0, 4.0 );
	* y = myQuantile( 0.8 );
	* // returns ~3.618
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of an arcsine distribution.
	*
	* ## Notes
	*
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.skewness( 4.0, 12.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.skewness( -4.0, 4.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.skewness( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN, 2.0 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of an arcsine distribution.
	*
	* ## Notes
	*
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 0.0, 1.0 );
	* // returns ~0.354
	*
	* @example
	* var v = ns.stdev( 4.0, 12.0 );
	* // returns ~2.828
	*
	* @example
	* var v = ns.stdev( -4.0, 4.0 );
	* // returns ~2.828
	*
	* @example
	* var v = ns.stdev( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN, 2.0 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of an arcsine distribution.
	*
	* ## Notes
	*
	* -   If provided `a >= b`, the function returns `NaN`.
	*
	* @param a - minimum support
	* @param b - maximum support
	* @returns variance
	*
	* @example
	* var v = ns.variance( 0.0, 1.0 );
	* // returns ~0.125
	*
	* @example
	* var v = ns.variance( 4.0, 12.0 );
	* // returns 8.0
	*
	* @example
	* var v = ns.variance( -4.0, 4.0 );
	* // returns 8.0
	*
	* @example
	* var v = ns.variance( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN, 2.0 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Arcsine distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
