/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isnan = require( '@stdlib/math/base/assert/is-nan' );


// MAIN //

/**
* Evaluates the probability density function (PDF) for a triangular distribution with lower limit `a` and upper limit `b` and mode `c` at a value `x`.
*
* @param {number} x - input value
* @param {number} a - lower limit
* @param {number} b - upper limit
* @param {number} c - mode
* @returns {number} evaluated PDF
*
* @example
* var y = pdf( 0.5, -1.0, 1.0, 0.0 );
* // returns 0.5
*
* @example
* var y = pdf( 0.5, -1.0, 1.0, 0.5 );
* // returns 1.0
*
* @example
* var y = pdf( -10.0, -20.0, 0.0, -2.0 );
* // returns ~0.056
*
* @example
* var y = pdf( -2.0, -1.0, 1.0, 0.0 );
* // returns 0.0
*
* @example
* var y = pdf( NaN, 0.0, 1.0, 0.5 );
* // returns NaN
*
* @example
* var y = pdf( 0.0, NaN, 1.0, 0.5 );
* // returns NaN
*
* @example
* var y = pdf( 0.0, 0.0, NaN, 0.5 );
* // returns NaN
*
* @example
* var y = pdf( 2.0, 1.0, 0.0, NaN );
* // returns NaN
*
* @example
* var y = pdf( 2.0, 1.0, 0.0, 1.5 );
* // returns NaN
*/
function pdf( x, a, b, c ) {
	if (
		isnan( x ) ||
		isnan( a ) ||
		isnan( b ) ||
		isnan( c ) ||
		a > c ||
		c > b
	) {
		return NaN;
	}
	if ( x < a ) {
		return 0.0;
	}
	// Case: x >= a
	if ( x < c ) {
		return ( 2.0 * ( x - a ) ) / ( ( b - a ) * ( c - a ) );
	}
	if ( x === c ) {
		return 2.0 / ( b - a );
	}
	// Case: x > c
	if ( x <= b ) {
		return ( 2.0 * ( b - x ) ) / ( ( b - a ) * ( b - c ) );
	}
	// Case: x > b
	return 0.0;
}


// EXPORTS //

module.exports = pdf;
