#include <sstream>
#include <fstream>
#include <iostream>
#include <cstring>
#include "ossimBmvUtil.h"
#include <ossim/base/ossimFilename.h>
#include "ossimArrayHolder.h"
#include <ossim/base/ossimEndian.h>
#include <ossim/imaging/ossimFilterResampler.h>

static ossimFilterResampler filterResampler;
ossimBmvUtil::ossimBmvUtil(ossim_uint32 tileSize,
                           ossim_uint32 level0TilesWide,
                           ossim_uint32 level0TilesHigh)
      :theTileSize(tileSize),
       theLevel0TilesWide(level0TilesWide),
       theLevel0TilesHigh(level0TilesHigh)
{
}


void ossimBmvUtil::getNumberOfTilesGivenLevel(ossim_uint32 level,
                                              ossim_uint32& tw,
                                              ossim_uint32& th)const
{
   ossim_uint32 multiplier = (ossim_uint32) std::pow((double)2.0, (double)level);
   tw = theLevel0TilesWide*multiplier;
   th = theLevel0TilesHigh*multiplier;
}

void ossimBmvUtil::getWidthHeightGivenLevel(ossim_uint32 level,
                                            ossim_uint32& w,
                                            ossim_uint32 h)const
{
   ossim_uint32 multiplier = (ossim_uint32) std::pow((double)2.0, (double)level);
   w = theLevel0TilesWide*theTileSize*multiplier;
   h = theLevel0TilesHigh*theTileSize*multiplier;
}

ossim_float64 ossimBmvUtil::getSpacing(ossim_uint32 level)const
{
   ossim_float64 multiplier = (ossim_uint32) std::pow((double)2.0, (double)level);
   
   return 180.0/(theLevel0TilesHigh*theTileSize*multiplier);
}

void ossimBmvUtil::getTileRowCol(ossim_uint32 level,
                                 const ossimGpt& gpt,
                                 ossim_uint32& row,
                                 ossim_uint32& col)const
{
   ossim_float64 multiplier = (ossim_uint32) std::pow((double)2.0, (double)level);

   row = (ossim_uint32)(multiplier*theLevel0TilesHigh*((gpt.latd() +90.0)/180.0));
   col = (ossim_uint32)(multiplier*theLevel0TilesHigh*((gpt.lond() +180.0)/180.0));
}

ossim_uint64 ossimBmvUtil::getTotalNumberOfTiles(ossim_uint32 maxLevel)const
{
   ossim_uint64 totalNumberOfTiles = 0;
   ossim_uint32 currentLevel = 0;
   ossim_uint32 tw,th;
   while(currentLevel <= maxLevel)
   {
      getNumberOfTilesGivenLevel(currentLevel,tw,th);

      totalNumberOfTiles += (tw*th);
   }

   return totalNumberOfTiles;
}

ossimIrect ossimBmvUtil::getBounds(ossim_uint32 level)const
{
   ossim_uint32 w = theLevel0TilesWide*theTileSize*(ossim_uint32)std::pow(2.0, (double)level);
   ossim_uint32 h = theLevel0TilesHigh*theTileSize*(ossim_uint32)std::pow(2.0, (double)level);
   
   ossimIrect result = ossimIrect(0,
                                  0,
                                  w - 1,
                                  h - 1);
   
   return result;
}

ossimIrect ossimBmvUtil::getBounds(ossim_uint32 level,
                                   const ossimGpt& ul,
                                   const ossimGpt& lr)const
{
   ossim_float64 spacing = getSpacing(level);
   ossim_float64 minLat = lr.latd() - 90;
   ossim_float64 minLon = ul.lond() + 180;
   ossim_float64 maxLat = ul.latd() - 90;
   ossim_float64 maxLon = lr.lond() + 180;

   if(minLat > 0.0 ) minLat = 0.0;
   if(minLat < 180.0 ) minLat = -180.0;
   if(maxLat > 0.0 ) minLat = 0.0;
   if(maxLat < 180.0 ) minLat = -180.0;

   if(minLon < 0.0 )   minLon = 0.0;
   if(minLon > 360.0 ) minLon = 360;
   if(maxLon < 0.0 )   maxLon = 0.0;
   if(maxLon > 360.0 ) maxLon = 360;
   
   

   
   ossimIpt u(ossim::round<int>(minLon/spacing),
               ossim::round<int>(-maxLat/spacing));
   ossimIpt l(ossim::round<int>(maxLon/spacing),
              ossim::round<int>(-minLat/spacing));

    ossimIrect temp(u, l);
    temp = temp.clipToRect(getBounds(level));
    temp.stretchToTileBoundary(ossimIpt(theTileSize,
                                        theTileSize));
               
   return temp;
}

ossim_uint32 ossimBmvUtil::findClosestLevel(ossim_float64 gsd)const
{
   ossim_uint32 level = 0;
   ossim_float64 temp = getSpacing(level);

   while(temp > gsd)
   {
      ++level;
      temp = getSpacing(level);
   }

   if(level != 0)
   {
      --level;
   }
   
   return level;
}

ossim_uint32 ossimBmvUtil::getLevel0TilesWide()const
{
   return theLevel0TilesWide;
}

ossim_uint32 ossimBmvUtil::getLevel0TilesHigh()const
{
   return theLevel0TilesHigh;
}

ossim_uint32 ossimBmvUtil::getTileSize()const
{
   return theTileSize;
}

ossimFilename ossimBmvUtil::getFilename(ossim_uint32 level,
                                        ossim_uint32 row,
                                        ossim_uint32 col)const
{
   ostringstream out;

   
   out << std::setw(4) << std::setfill('0') << row
       << std::setw(0) << "_" << std::setw(4) << std::setfill('0') << col
       << std::ends;

   return ossimFilename(out.str());
}

void ossimBmvUtil::readTile(ossim_uint16* destination,
                            const ossimFilename& baseLandDir,
                            ossim_uint32 level,
                            ossim_uint32 row,
                            ossim_uint32 col)const
{
   
   ossimFilename tempFile = baseLandDir;
   tempFile = tempFile.dirCat(ossimFilename(ossimString::toString(level)).dirCat(getFilename(level, row, col) + ".5551"));
   if(tempFile.exists())
   {
      std::ifstream in(tempFile.c_str(), ios::in|ios::binary);
      ossimEndian endian;

      if(in)
      {
         in.read((char*)(destination), theTileSize*theTileSize*2);
         if(endian.getSystemEndianType() != OSSIM_LITTLE_ENDIAN)
         {
            endian.swap((ossim_uint16*)destination, theTileSize*theTileSize);
         }
      }
      
      return;
   }
   else if(level==0)
   {
      memset((char*)(destination),
             0,theTileSize*theTileSize*2);
      return;
   }

   else
   {
      ossimArrayHolder<ossim_uint16, 256*256> tempBuffer;
      
      readTile((ossim_uint16*)tempBuffer, baseLandDir, level-1, row>>1, col>>1);
      ossimArrayHolder<ossim_uint8, 128*128*3> subBuffer8bit;
      ossim_uint32 srcOffset = (row&1)*(128)*256+(col&1)*(128);
      
      ossim_uint16* srcPtr  = ((ossim_uint16*)tempBuffer) + srcOffset;
      ossim_uint32 y = 0;
      ossim_uint32 x = 0;
      // ossim_uint32 upperBounds = theTileSize*theTileSize;
      ossim_uint32 idx = 0;
      ossim_uint8* rbuf = subBuffer8bit;
      ossim_uint8* gbuf = rbuf+128*128;
      ossim_uint8* bbuf = gbuf+(128*128);
      for(y=0;y<128;++y)
      {
         for(x = 0; x < 128; ++x)
         {
            *rbuf = ((srcPtr[x]>>8)&0xf8);
            *gbuf = ((srcPtr[x]>>3)&0xf8);
            *bbuf = ((srcPtr[x]<<2)&0xf8);
            
            ++rbuf;
            ++gbuf;
            ++bbuf;
         }
         srcPtr += theTileSize;
      }

      rbuf = subBuffer8bit;
      gbuf = rbuf + (128*128);
      bbuf = gbuf+(128*128);

      idx = 0;
      ossim_uint16* destPtr = destination;
      // ossim_uint16 tempColor;
      ossimArrayHolder<ossim_int32, 256> mapBuffer;
      
      for(x = 0; x < 256; ++x)
      {
         mapBuffer[x] = x/2;
      }
      
      // ossim_int32 mappedx;
      // ossim_int32 mappedy;

      ossim_int32 ulx;
      ossim_int32 uly;
      ossim_int32 urx;
      ossim_int32 ury;
      ossim_int32 lrx;
      ossim_int32 lry;
      ossim_int32 llx;
      ossim_int32 lly;
      ossim_uint8 testLocation;
      ossim_uint8 resultr = 0;
      ossim_uint8 resultg = 0;
      ossim_uint8 resultb = 0;
      ossim_int32 offset1;
      ossim_int32 offset2;
      ossim_int32 offset3;
      ossim_int32 offset4;
      for(y = 0; y < 256; ++y)
      {
         uly = mapBuffer[y];
         if(uly == 127) uly--;
         for(x = 0; x < 256; ++x)
         {
            testLocation = ((y&1) << 1) | (x&1);
            
            ulx = mapBuffer[x];
            if(ulx == 127)
            {
               ulx--;
            }
            
            offset1 = uly*128 + ulx;
            switch(testLocation)
            {
               case 0: // upper left pixel
               {
                  resultr = *(rbuf + offset1);
                  resultg = *(gbuf + offset1);
                  resultb = *(bbuf + offset1);
                  
                  break;
               }
               case 1: // upper right pixel
               {
                  urx = ulx + 1;
                  ury = uly;
                  offset2 = ury*128 + urx;
                  resultr = (ossim_uint8)(((ossim_uint32)*(rbuf + offset1)  + (ossim_uint32)*(rbuf + offset2))*(.5));
                  resultg = (ossim_uint8)(((ossim_uint32)*(gbuf + offset1)  + (ossim_uint32)*(gbuf + offset2))*(.5));
                   resultb = (ossim_uint8)(((ossim_uint32)*(bbuf + offset1)  + (ossim_uint32)*(bbuf + offset2))*(.5));
                   break;
               }
               case 2: // lower left pixel
               {
                  llx = ulx;
                  lly = uly + 1;
                  
                  offset2 = lly*128 + llx;
                  resultr = (ossim_uint8)(((ossim_uint32)*(rbuf + offset1)  + (ossim_uint32)*(rbuf + offset2))*(.5));
                  resultg = (ossim_uint8)(((ossim_uint32)*(gbuf + offset1)  + (ossim_uint32)*(gbuf + offset2))*(.5));
                  resultb = (ossim_uint8)(((ossim_uint32)*(bbuf + offset1)  + (ossim_uint32)*(bbuf + offset2))*(.5));
                  break;
               }
               case 3: // lower right pixel
               {
                  urx = ulx + 1;
                  ury = uly;
                  lrx = urx;
                  lry = uly + 1;
                  llx = ulx;
                  lly = lry;
                  offset2 = ury*128 + urx;
                  offset3 = lry*128 + lrx;
                  offset4 = lly*128 + llx;
                  resultr = (ossim_uint8)(((ossim_uint32)*(rbuf + offset1) +
                                           (ossim_uint32)*(rbuf + offset2) +
                                           (ossim_uint32)*(rbuf + offset3) +
                                           (ossim_uint32)*(rbuf + offset4))*(.25));
                  resultg = (ossim_uint8)(((ossim_uint32)*(gbuf + offset1) +
                                           (ossim_uint32)*(gbuf + offset2) +
                                           (ossim_uint32)*(gbuf + offset3) +
                                           (ossim_uint32)*(gbuf + offset4))*(.25));
                  resultb = (ossim_uint8)(((ossim_uint32)*(bbuf + offset1) +
                                           (ossim_uint32)*(bbuf + offset2) +
                                           (ossim_uint32)*(bbuf + offset3) +
                                           (ossim_uint32)*(bbuf + offset4))*(.25));
                  
                  break;
               }
             }
                  
             *(destPtr+x) = (((resultr&0xf8)<<8)|((resultg&0xf8)<<3)|((resultb&0xf8)>>2)|1);
         }
         destPtr+=(theTileSize);
      }
   }
}
