///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/Vector3PropertyUI.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(Vector3PropertyUI, FloatPropertyUI)

/******************************************************************************
* Constructor for a Qt property.
******************************************************************************/
Vector3PropertyUI::Vector3PropertyUI(PropertiesEditor* parentEditor, const char* propertyName, size_t vectorComponent, const QString& labelText, ParameterUnit* parameterUnit) 
	: FloatPropertyUI(parentEditor, propertyName, labelText, parameterUnit), component(vectorComponent)
{
	OVITO_ASSERT_MSG(vectorComponent >= 0 && vectorComponent < 3, "Vector3PropertyUI constructor", "The vector component must be in the range 0-2.");
}

/******************************************************************************
* Constructor for a PropertyField property.
******************************************************************************/
Vector3PropertyUI::Vector3PropertyUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& propField, size_t vectorComponent) 
	: FloatPropertyUI(parentEditor, propField), component(vectorComponent)
{
	OVITO_ASSERT_MSG(vectorComponent >= 0 && vectorComponent < 3, "Vector3PropertyUI constructor", "The vector component must be in the range 0-2.");

	switch(component) {
		case 0: label()->setText(propField.displayName() + " (X):"); break;
		case 1: label()->setText(propField.displayName() + " (Y):"); break;
		case 2: label()->setText(propField.displayName() + " (Z):"); break;
	}
}

/******************************************************************************
* Takes the value entered by the user and stores it in the parameter object
* this parameter UI is bound to.
******************************************************************************/
void Vector3PropertyUI::updatePropertyValue()
{
	if(editObject() && spinner()) {
		if(propertyName()) {
			QVariant currentValue = editObject()->property(propertyName());
			if(currentValue.canConvert<Vector3>()) {
				Vector3 val = currentValue.value<Vector3>();
				val[component] = spinner()->floatValue();
				currentValue.setValue(val);
			}
			else if(currentValue.canConvert<Point3>()) {
				Point3 val = currentValue.value<Point3>();
				val[component] = spinner()->floatValue();
				currentValue.setValue(val);
			}
			if(!editObject()->setProperty(propertyName(), currentValue)) {
				OVITO_ASSERT_MSG(false, "Vector3PropertyUI::updatePropertyValue()", QString("The value of property %1 of object class %2 could not be set.").arg(QString(propertyName()), editObject()->metaObject()->className()).toLocal8Bit().constData());
			}
		}
		else if(propertyField()) {
			QVariant currentValue = editObject()->getPropertyFieldValue(*propertyField());
			if(currentValue.canConvert<Vector3>()) {
				Vector3 val = currentValue.value<Vector3>();
				val[component] = spinner()->floatValue();
				currentValue.setValue(val);
			}
			else if(currentValue.canConvert<Point3>()) {
				Point3 val = currentValue.value<Point3>();
				val[component] = spinner()->floatValue();
				currentValue.setValue(val);
			}
			editObject()->setPropertyFieldValue(*propertyField(), currentValue);						
		}
	}
}

/******************************************************************************
* This method updates the displayed value of the parameter UI.
******************************************************************************/
void Vector3PropertyUI::updateUI()
{
	if(editObject() && spinner() && !spinner()->isDragging()) {
		QVariant val;
		if(propertyName()) {
			val = editObject()->property(propertyName());
			OVITO_ASSERT_MSG(val.isValid() && (val.canConvert<Vector3>() || val.canConvert<Point3>()), "Vector3PropertyUI::updateUI()", QString("The object class %1 does not define a property with the name %2 that can be cast to Vector3/Point3 type.").arg(editObject()->metaObject()->className(), QString(propertyName())).toLocal8Bit().constData());
			if(!val.isValid() || !(val.canConvert<Vector3>() || val.canConvert<Point3>())) {
				throw Exception(tr("The object class %1 does not define a property with the name %2 that can be cast to Vector3/Point3 type.").arg(editObject()->metaObject()->className(), QString(propertyName())));
			}
		}
		else if(propertyField()) {
			val = editObject()->getPropertyFieldValue(*propertyField());
			OVITO_ASSERT(val.isValid() && (val.canConvert<Vector3>() || val.canConvert<Point3>()));
		}
		else return;
		
		if(val.canConvert<Vector3>())
			spinner()->setFloatValue(val.value<Vector3>()[component]);
		else if(val.canConvert<Point3>())
			spinner()->setFloatValue(val.value<Point3>()[component]);
	}
}

};

