// Copyright (C) 2016 The Qt Company Ltd.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR GPL-3.0-only WITH Qt-GPL-exception-1.0
#include <QTest>
#include <QtCore/QSize>
#include <QtCore/QRectF>
#include <QtGui/QTransform>
#include <QtGui/QGuiApplication>
#include <QtGui/QPainterPath>
#include <QtGui/QScreen>
#include <QtWidgets/QStyle>
#include <QtWidgets/QStyleFactory>

#include "tst_qgraphicsview.h"

Q_DECLARE_METATYPE(ExpectedValueDescription)
Q_DECLARE_METATYPE(QList<int>)
Q_DECLARE_METATYPE(QList<QRectF>)
Q_DECLARE_METATYPE(QPainterPath)
Q_DECLARE_METATYPE(Qt::ScrollBarPolicy)
Q_DECLARE_METATYPE(ScrollBarCount)

static void _scrollBarRanges_addTestData(const QByteArray &style, bool styled)
{
    const int viewWidth = 250;
    const int viewHeight = 100;

    static constexpr struct Data {
        const char *name;
        QRectF sceneRect;
        ScrollBarCount sceneRectOffsetFactors;
        int scale;
        Qt::ScrollBarPolicy hbarpolicy, vbarpolicy;
        ExpectedValueDescription hmin, hmax, vmin, vmax;
    } data [] = {
        {
            "1",
            QRectF(0, 0, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(),
            ExpectedValueDescription(),
            ExpectedValueDescription(),
            ExpectedValueDescription(),
        },
        {
            "2",
            QRectF(0, 0, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(),
            ExpectedValueDescription(50, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 1, 1),
        },
        {
            "3",
            QRectF(0, 0, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(),
            ExpectedValueDescription(50, 1, 1),
            ExpectedValueDescription(0, 0),
            ExpectedValueDescription(100, 1, 1),
        },
        {
            "4",
            QRectF(-100, -100, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(),
            ExpectedValueDescription(),
            ExpectedValueDescription(),
            ExpectedValueDescription(),
        },
        {
            "5",
            QRectF(-100, -100, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(-100),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-100),
            ExpectedValueDescription(-100, 1, 1),
        },
        {
            "6",
            QRectF(-100, -100, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(-100),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-100),
            ExpectedValueDescription(0, 1, 1),
        },
        {
            "7",
            QRectF(0, 0, viewWidth + 1, viewHeight + 1),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(),
            ExpectedValueDescription(1, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(1, 1, 1),
        },
        {
            "8",
            QRectF(0, 0, viewWidth + 51, viewHeight + 1),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(),
            ExpectedValueDescription(51, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(1, 1, 1),
        },
        {
            "9",
            QRectF(0, 0, viewWidth + 51, viewHeight + 101),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(),
            ExpectedValueDescription(51, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(101, 1, 1),
        },
        {
            "10",
            QRectF(-101, -101, viewWidth + 1, viewHeight + 1),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-100, 1, 1),
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-100, 1, 1),
        },
        {
            "11",
            QRectF(-101, -101, viewWidth + 51, viewHeight + 1),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-100, 1, 1),
        },
        {
            "12",
            QRectF(-101, -101, viewWidth + 51, viewHeight + 101),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-101),
            ExpectedValueDescription(0, 1, 1),
        },
        {
            "13",
            QRectF(0, 0, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 1, 1),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 2, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 2, 1),
        },
        {
            "14",
            QRectF(0, 0, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 1, 1),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(),
            ExpectedValueDescription(50, 2, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 2, 1),
        },
        {
            "15",
            QRectF(0, 0, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 1, 1),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(),
            ExpectedValueDescription(50, 2, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(100, 2, 1),
        },
        {
            "16",
            QRectF(-100, -100, viewWidth, viewHeight),
            ScrollBarCount(-1, -1, 1, 1),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-100, 1, 1),
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-100, 1, 1),
        },
        {
            "17",
            QRectF(-100, -100, viewWidth + 50, viewHeight),
            ScrollBarCount(-1, -1, 1, 1),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-100, 1, 1),
        },
        {
            "18",
            QRectF(-100, -100, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(-1, -1, 1, 1),
            1,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(0, 1, 1),
        },
        {
            "1 x2",
            QRectF(0, 0, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(),
            ExpectedValueDescription(viewWidth, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(viewHeight, 1, 1),
        },
        {
            "2 x2",
            QRectF(0, 0, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(),
            ExpectedValueDescription(viewWidth + 100, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(viewHeight, 1, 1),
        },
        {
            "3 x2",
            QRectF(0, 0, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(),
            ExpectedValueDescription(viewWidth + 100, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(viewHeight + 200, 1, 1),
        },
        {
            "4 x2",
            QRectF(-100, -100, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewWidth - 200, 1, 1),
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewHeight - 200, 1, 1),
        },
        {
            "5 x2",
            QRectF(-100, -100, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewWidth - 100, 1, 1),
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewHeight - 200, 1, 1),
        },
        {
            "6 x2",
            QRectF(-100, -100, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAsNeeded,
            Qt::ScrollBarAsNeeded,
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewWidth - 100, 1, 1),
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewHeight, 1, 1),
        },
        {
            "1 No ScrollBars",
            QRectF(0, 0, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(),
            ExpectedValueDescription(),
            ExpectedValueDescription(),
            ExpectedValueDescription(),
        },
        {
            "2 No ScrollBars",
            QRectF(0, 0, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(),
            ExpectedValueDescription(50),
            ExpectedValueDescription(),
            ExpectedValueDescription(),
        },
        {
            "3 No ScrollBars",
            QRectF(0, 0, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(),
            ExpectedValueDescription(50),
            ExpectedValueDescription(),
            ExpectedValueDescription(100),
        },
        {
            "4 No ScrollBars",
            QRectF(-100, -100, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(),
            ExpectedValueDescription(),
            ExpectedValueDescription(),
            ExpectedValueDescription(),
        },
        {
            "5 No ScrollBars",
            QRectF(-100, -100, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(-100),
            ExpectedValueDescription(-50),
            ExpectedValueDescription(),
            ExpectedValueDescription(),
        },
        {
            "6 No ScrollBars",
            QRectF(-100, -100, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(-100),
            ExpectedValueDescription(-50),
            ExpectedValueDescription(-100),
            ExpectedValueDescription(),
        },
        {
            "7 No ScrollBars",
            QRectF(0, 0, viewWidth + 1, viewHeight + 1),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(),
            ExpectedValueDescription(1),
            ExpectedValueDescription(),
            ExpectedValueDescription(1),
        },
        {
            "8 No ScrollBars",
            QRectF(0, 0, viewWidth + 51, viewHeight + 1),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(),
            ExpectedValueDescription(51),
            ExpectedValueDescription(),
            ExpectedValueDescription(1),
        },
        {
            "9 No ScrollBars",
            QRectF(0, 0, viewWidth + 51, viewHeight + 101),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(),
            ExpectedValueDescription(51),
            ExpectedValueDescription(),
            ExpectedValueDescription(101),
        },
        {
            "10 No ScrollBars",
            QRectF(-101, -101, viewWidth + 1, viewHeight + 1),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-100),
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-100),
        },
        {
            "11 No ScrollBars",
            QRectF(-101, -101, viewWidth + 51, viewHeight + 1),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-50),
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-100),
        },
        {
            "12 No ScrollBars",
            QRectF(-101, -101, viewWidth + 51, viewHeight + 101),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-50),
            ExpectedValueDescription(-101),
            ExpectedValueDescription(),
        },
        {
            "13 No ScrollBars",
            QRectF(0, 0, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 1, 1),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 1),
        },
        {
            "14 No ScrollBars",
            QRectF(0, 0, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 1, 1),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(),
            ExpectedValueDescription(50, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 1),
        },
        {
            "15 No ScrollBars",
            QRectF(0, 0, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 1, 1),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(),
            ExpectedValueDescription(50, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(100, 1),
        },
        {
            "16 No ScrollBars",
            QRectF(-100, -100, viewWidth, viewHeight),
            ScrollBarCount(-1, -1, 1, 1),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-100),
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-100),
        },
        {
            "17 No ScrollBars",
            QRectF(-100, -100, viewWidth + 50, viewHeight),
            ScrollBarCount(-1, -1, 1, 1),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-50),
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-100),
        },
        {
            "18 No ScrollBars",
            QRectF(-100, -100, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(-1, -1, 1, 1),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-50),
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(),
        },
        {
            "1 x2 No ScrollBars",
            QRectF(0, 0, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(),
            ExpectedValueDescription(viewWidth),
            ExpectedValueDescription(),
            ExpectedValueDescription(viewHeight),
        },
        {
            "2 x2 No ScrollBars",
            QRectF(0, 0, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(),
            ExpectedValueDescription(viewWidth + 100),
            ExpectedValueDescription(),
            ExpectedValueDescription(viewHeight),
        },
        {
            "3 x2 No ScrollBars",
            QRectF(0, 0, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(),
            ExpectedValueDescription(viewWidth + 100),
            ExpectedValueDescription(),
            ExpectedValueDescription(viewHeight + 200),
        },
        {
            "4 x2 No ScrollBars",
            QRectF(-100, -100, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewWidth - 200),
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewHeight - 200),
        },
        {
            "5 x2 No ScrollBars",
            QRectF(-100, -100, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewWidth - 100),
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewHeight - 200),
        },
        {
            "6 x2 No ScrollBars",
            QRectF(-100, -100, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOff,
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewWidth - 100),
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewHeight),
        },
        {
            "1 Always ScrollBars",
            QRectF(0, 0, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 1, 1),
        },
        {
            "2 Always ScrollBars",
            QRectF(0, 0, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(50, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 1, 1),
        },
        {
            "3 Always ScrollBars",
            QRectF(0, 0, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(50, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(100, 1, 1),
        },
        {
            "4 Always ScrollBars",
            QRectF(-100, -100, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-100),
            ExpectedValueDescription(-100, 1, 1),
            ExpectedValueDescription(-100),
            ExpectedValueDescription(-100, 1, 1),
        },
        {
            "5 Always ScrollBars",
            QRectF(-100, -100, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-100),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-100),
            ExpectedValueDescription(-100, 1, 1),
        },
        {
            "6 Always ScrollBars",
            QRectF(-100, -100, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-100),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-100),
            ExpectedValueDescription(0, 1, 1),
        },
        {
            "7 Always ScrollBars",
            QRectF(0, 0, viewWidth + 1, viewHeight + 1),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(1, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(1, 1, 1),
        },
        {
            "8 Always ScrollBars",
            QRectF(0, 0, viewWidth + 51, viewHeight + 1),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(51, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(1, 1, 1),
        },
        {
            "9 Always ScrollBars",
            QRectF(0, 0, viewWidth + 51, viewHeight + 101),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(51, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(101, 1, 1),
        },
        {
            "10 Always ScrollBars",
            QRectF(-101, -101, viewWidth + 1, viewHeight + 1),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-100, 1, 1),
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-100, 1, 1),
        },
        {
            "11 Always ScrollBars",
            QRectF(-101, -101, viewWidth + 51, viewHeight + 1),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-100, 1, 1),
        },
        {
            "12 Always ScrollBars",
            QRectF(-101, -101, viewWidth + 51, viewHeight + 101),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-101),
            ExpectedValueDescription(0, 1, 1),
        },
        {
            "13 Always ScrollBars",
            QRectF(0, 0, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 1, 1),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 2, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 2, 1),
        },
        {
            "14 Always ScrollBars",
            QRectF(0, 0, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 1, 1),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(50, 2, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 2, 1),
        },
        {
            "15 Always ScrollBars",
            QRectF(0, 0, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 1, 1),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(50, 2, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(100, 2, 1),
        },
        {
            "16 Always ScrollBars",
            QRectF(-100, -100, viewWidth, viewHeight),
            ScrollBarCount(-1, -1, 1, 1),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-100, 1, 1),
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-100, 1, 1),
        },
        {
            "17 Always ScrollBars",
            QRectF(-100, -100, viewWidth + 50, viewHeight),
            ScrollBarCount(-1, -1, 1, 1),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-100, 1, 1),
        },
        {
            "18 Always ScrollBars",
            QRectF(-100, -100, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(-1, -1, 1, 1),
            1,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(0, 1, 1),
        },
        {
            "1 x2 Always ScrollBars",
            QRectF(0, 0, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(viewWidth, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(viewHeight, 1, 1),
        },
        {
            "2 x2 Always ScrollBars",
            QRectF(0, 0, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(viewWidth + 100, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(viewHeight, 1, 1),
        },
        {
            "3 x2 Always ScrollBars",
            QRectF(0, 0, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(viewWidth + 100, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(viewHeight + 200, 1, 1),
        },
        {
            "4 x2 Always ScrollBars",
            QRectF(-100, -100, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewWidth - 200, 1, 1),
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewHeight - 200, 1, 1),
        },
        {
            "5 x2 Always ScrollBars",
            QRectF(-100, -100, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewWidth - 100, 1, 1),
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewHeight - 200, 1, 1),
        },
        {
            "6 x2 Always ScrollBars",
            QRectF(-100, -100, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOn,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewWidth - 100, 1, 1),
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewHeight, 1, 1),
        },
        {
            "1 Vertical Only",
            QRectF(0, 0, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(),
        },
        {
            "2 Vertical Only",
            QRectF(0, 0, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(50, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(),
        },
        {
            "3 Vertical Only",
            QRectF(0, 0, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(50, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(100),
        },
        {
            "4 Vertical Only",
            QRectF(-100, -100, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-100),
            ExpectedValueDescription(-100, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(),
        },
        {
            "5 Vertical Only",
            QRectF(-100, -100, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-100),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(),
        },
        {
            "6 Vertical Only",
            QRectF(-100, -100, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-100),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-100),
            ExpectedValueDescription(),
        },
        {
            "7 Vertical Only",
            QRectF(0, 0, viewWidth + 1, viewHeight + 1),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(1, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(1),
        },
        {
            "8 Vertical Only",
            QRectF(0, 0, viewWidth + 51, viewHeight + 1),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(51, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(1),
        },
        {
            "9 Vertical Only",
            QRectF(0, 0, viewWidth + 51, viewHeight + 101),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(51, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(101),
        },
        {
            "10 Vertical Only",
            QRectF(-101, -101, viewWidth + 1, viewHeight +1),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-100, 1, 1),
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-100),
        },
        {
            "11 Vertical Only",
            QRectF(-101, -101, viewWidth + 51, viewHeight + 1),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-100),
        },
        {
            "12 Vertical Only",
            QRectF(-101, -101, viewWidth + 51, viewHeight + 101),
            ScrollBarCount(0, 0, 0, 0),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-101),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-101),
            ExpectedValueDescription(),
        },
        {
            "13 Vertical Only",
            QRectF(0, 0, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 1, 1),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 2, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 1),
        },
        {
            "14 Vertical Only",
            QRectF(0, 0, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 1, 1),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(50, 2, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(0, 1),
        },
        {
            "15 Vertical Only",
            QRectF(0, 0, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 1, 1),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(50, 2, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(100, 1),
        },
        {
            "16 Vertical Only",
            QRectF(-100, -100, viewWidth, viewHeight),
            ScrollBarCount(-1, -1, 1, 1),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-100, 1, 1),
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-100),
        },
        {
            "17 Vertical Only",
            QRectF(-100, -100, viewWidth + 50, viewHeight),
            ScrollBarCount(-1, -1, 1, 1),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-100),
        },
        {
            "18 Vertical Only",
            QRectF(-100, -100, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(-1, -1, 1, 1),
            1,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(-50, 1, 1),
            ExpectedValueDescription(-100, -1),
            ExpectedValueDescription(),
        },
        {
            "1 x2 Vertical Only",
            QRectF(0, 0, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(viewWidth, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(viewHeight),
        },
        {
            "2 x2 Vertical Only",
            QRectF(0, 0, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(viewWidth + 100, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(viewHeight),
        },
        {
            "3 x2 Vertical Only",
            QRectF(0, 0, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(),
            ExpectedValueDescription(viewWidth + 100, 1, 1),
            ExpectedValueDescription(),
            ExpectedValueDescription(viewHeight + 200),
        },
        {
            "4 x2 Vertical Only",
            QRectF(-100, -100, viewWidth, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewWidth - 200, 1, 1),
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewHeight - 200),
        },
        {
            "5 x2 Vertical Only",
            QRectF(-100, -100, viewWidth + 50, viewHeight),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewWidth - 100, 1, 1),
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewHeight - 200),
        },
        {
            "6 x2 Vertical Only",
            QRectF(-100, -100, viewWidth + 50, viewHeight + 100),
            ScrollBarCount(0, 0, 0, 0),
            2,
            Qt::ScrollBarAlwaysOff,
            Qt::ScrollBarAlwaysOn,
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewWidth - 100, 1, 1),
            ExpectedValueDescription(-200),
            ExpectedValueDescription(viewHeight),
        },
    };

    const QSize viewSize(viewWidth, viewHeight);

    for (const Data &e : data) {
        QTest::addRow("%s%s, %s", style.data(), styled ? ", Styled" : "", e.name)
            << style << viewSize
            << e.sceneRect
            << e.sceneRectOffsetFactors
            << QTransform().scale(e.scale, e.scale)
            << e.hbarpolicy
            << e.vbarpolicy
            << e.hmin << e.hmax << e.vmin << e.vmax
            << styled;
    }
}

void _scrollBarRanges_data()
{
    QTest::addColumn<QByteArray>("style");
    QTest::addColumn<QSize>("viewportSize");
    QTest::addColumn<QRectF>("sceneRect");
    QTest::addColumn<ScrollBarCount>("sceneRectOffsetFactors");
    QTest::addColumn<QTransform>("transform");
    QTest::addColumn<Qt::ScrollBarPolicy>("hbarpolicy");
    QTest::addColumn<Qt::ScrollBarPolicy>("vbarpolicy");
    QTest::addColumn<ExpectedValueDescription>("hmin");
    QTest::addColumn<ExpectedValueDescription>("hmax");
    QTest::addColumn<ExpectedValueDescription>("vmin");
    QTest::addColumn<ExpectedValueDescription>("vmax");
    QTest::addColumn<bool>("useStyledPanel");

    const auto styles = QStyleFactory::keys();
    for (const QString &style : styles) {
        _scrollBarRanges_addTestData(style.toLatin1(), false);
        _scrollBarRanges_addTestData(style.toLatin1(), true);
    }

    const QScreen *screen = QGuiApplication::primaryScreen();
    if (screen && qFuzzyCompare((double)screen->logicalDotsPerInchX(), 96.0)) {
        _scrollBarRanges_addTestData("motif", false);
        _scrollBarRanges_addTestData("motif", true);
    }
}
