/*
 * Copyright (C) 2007, 2008 Apple Inc.  All rights reserved.
 * Copyright (C) 2008, 2009 Anthony Ricaud <rik@webkit.org>
 * Copyright (C) 2011 Google Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 * 3.  Neither the name of Apple Computer, Inc. ("Apple") nor the names of
 *     its contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE AND ITS CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE OR ITS CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 * @unrestricted
 */
Network.HARWriter = class {
  /**
   * @param {!Common.OutputStream} stream
   * @param {!Array.<!SDK.NetworkRequest>} requests
   * @param {!Common.Progress} progress
   */
  write(stream, requests, progress) {
    this._stream = stream;
    this._harLog = (new SDK.HARLog(requests)).build();
    this._pendingRequests = 1;  // Guard against completing resource transfer before all requests are made.
    var entries = this._harLog.entries;
    for (var i = 0; i < entries.length; ++i) {
      var content = requests[i].content;
      if (typeof content === 'undefined' && requests[i].finished) {
        ++this._pendingRequests;
        requests[i].requestContent().then(this._onContentAvailable.bind(this, entries[i], requests[i]));
      } else if (content !== null) {
        this._setEntryContent(entries[i], requests[i]);
      }
    }
    var compositeProgress = new Common.CompositeProgress(progress);
    this._writeProgress = compositeProgress.createSubProgress();
    if (--this._pendingRequests) {
      this._requestsProgress = compositeProgress.createSubProgress();
      this._requestsProgress.setTitle(Common.UIString('Collecting content…'));
      this._requestsProgress.setTotalWork(this._pendingRequests);
    } else {
      this._beginWrite();
    }
  }

  /**
   * @param {!Object} entry
   * @param {!SDK.NetworkRequest} request
   */
  _setEntryContent(entry, request) {
    if (request.content !== null)
      entry.response.content.text = request.content;
    if (request.contentEncoded)
      entry.response.content.encoding = 'base64';
  }

  /**
   * @param {!Object} entry
   * @param {!SDK.NetworkRequest} request
   * @param {?string} content
   */
  _onContentAvailable(entry, request, content) {
    this._setEntryContent(entry, request);
    if (this._requestsProgress)
      this._requestsProgress.worked();
    if (!--this._pendingRequests) {
      this._requestsProgress.done();
      this._beginWrite();
    }
  }

  _beginWrite() {
    const jsonIndent = 2;
    this._text = JSON.stringify({log: this._harLog}, null, jsonIndent);
    this._writeProgress.setTitle(Common.UIString('Writing file…'));
    this._writeProgress.setTotalWork(this._text.length);
    this._bytesWritten = 0;
    this._writeNextChunk(this._stream);
  }

  /**
   * @param {!Common.OutputStream} stream
   * @param {string=} error
   */
  _writeNextChunk(stream, error) {
    if (this._bytesWritten >= this._text.length || error) {
      stream.close();
      this._writeProgress.done();
      return;
    }
    const chunkSize = 100000;
    var text = this._text.substring(this._bytesWritten, this._bytesWritten + chunkSize);
    this._bytesWritten += text.length;
    stream.write(text, this._writeNextChunk.bind(this));
    this._writeProgress.setWorked(this._bytesWritten);
  }
};
