// Copyright (C) 2013 Google Inc. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//    * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//    * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//    * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#ifndef TextRunIterator_h
#define TextRunIterator_h

#include "platform/text/TextRun.h"
#include "wtf/Allocator.h"

namespace blink {

class TextRunIterator {
  DISALLOW_NEW();

 public:
  TextRunIterator() : m_textRun(0), m_offset(0), m_length(0) {}

  TextRunIterator(const TextRun* textRun, unsigned offset)
      : m_textRun(textRun), m_offset(offset), m_length(m_textRun->length()) {}

  TextRunIterator(const TextRunIterator& other)
      : m_textRun(other.m_textRun),
        m_offset(other.m_offset),
        m_length(m_textRun->length()) {}

  unsigned offset() const { return m_offset; }
  void increment() { m_offset++; }
  bool atEnd() const { return m_offset >= m_length; }
  UChar current() const { return (*m_textRun)[m_offset]; }
  WTF::Unicode::CharDirection direction() const {
    return atEnd() ? WTF::Unicode::OtherNeutral
                   : WTF::Unicode::direction(current());
  }
  bool atParagraphSeparator() const { return current() == '\n'; }

  bool operator==(const TextRunIterator& other) {
    return m_offset == other.m_offset && m_textRun == other.m_textRun;
  }

  bool operator!=(const TextRunIterator& other) { return !operator==(other); }

 private:
  const TextRun* m_textRun;
  unsigned m_offset;
  unsigned m_length;
};

}  // namespace blink

#endif  // TextRunIterator_h
