// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef WebGestureEvent_h
#define WebGestureEvent_h

#include "WebGestureDevice.h"
#include "WebInputEvent.h"

namespace blink {

// See WebInputEvent.h for details why this pack is here.
#pragma pack(push, 4)

// WebGestureEvent ---------------------------------------------------------

class WebGestureEvent : public WebInputEvent {
 public:
  enum ScrollUnits {
    PrecisePixels = 0,  // generated by high precision devices.
    Pixels,             // large pixel jump duration; should animate to delta.
    Page                // page (visible viewport) based scrolling.
  };

  enum InertialPhaseState {
    UnknownMomentumPhase = 0,  // No phase information.
    NonMomentumPhase,          // Regular scrolling phase.
    MomentumPhase,             // Momentum phase.
  };

  int x;
  int y;
  int globalX;
  int globalY;
  WebGestureDevice sourceDevice;

  // If the WebGestureEvent has sourceDevice=WebGestureDeviceTouchscreen, this
  // field contains the unique identifier for the touch event that released
  // this event at TouchDispositionGestureFilter. If the WebGestureEvents was
  // not released through a touch event (e.g. timer-released gesture events or
  // gesture events with sourceDevice!=WebGestureDeviceTouchscreen), the field
  // contains 0. See crbug.com/618738.
  uint32_t uniqueTouchEventId;

  // This field exists to allow BrowserPlugin to mark GestureScroll events as
  // 'resent' to handle the case where an event is not consumed when first
  // encountered; it should be handled differently by the plugin when it is
  // sent for thesecond time. No code within Blink touches this, other than to
  // plumb it through event conversions.
  int resendingPluginId;

  union {
    // Tap information must be set for GestureTap, GestureTapUnconfirmed,
    // and GestureDoubleTap events.
    struct {
      int tapCount;
      float width;
      float height;
    } tap;

    struct {
      float width;
      float height;
    } tapDown;

    struct {
      float width;
      float height;
    } showPress;

    struct {
      float width;
      float height;
    } longPress;

    struct {
      float firstFingerWidth;
      float firstFingerHeight;
    } twoFingerTap;

    struct {
      // Initial motion that triggered the scroll.
      // May be redundant with deltaX/deltaY in the first scrollUpdate.
      float deltaXHint;
      float deltaYHint;
      // Default initialized to ScrollUnits::PrecisePixels.
      ScrollUnits deltaHintUnits;
      // If true, this event will skip hit testing to find a scroll
      // target and instead just scroll the viewport.
      bool targetViewport;
      // The state of inertial phase scrolling. OSX has unique phases for normal
      // and momentum scroll events. Should always be UnknownMomentumPhase for
      // touch based input as it generates GestureFlingStart instead.
      InertialPhaseState inertialPhase;
      // True if this event was synthesized in order to force a hit test;
      // avoiding scroll latching behavior until crbug.com/526463 is fully
      // implemented.
      bool synthetic;

      // number of pointers down.
      int pointerCount;
    } scrollBegin;

    struct {
      float deltaX;
      float deltaY;
      float velocityX;
      float velocityY;
      // Whether any previous GestureScrollUpdate in the current scroll
      // sequence was suppressed (e.g., the causal touchmove was
      // preventDefault'ed). This bit is particularly useful for
      // determining whether the observed scroll update sequence captures
      // the entirety of the generative motion.
      bool previousUpdateInSequencePrevented;
      bool preventPropagation;
      InertialPhaseState inertialPhase;
      // Default initialized to ScrollUnits::PrecisePixels.
      ScrollUnits deltaUnits;
    } scrollUpdate;

    struct {
      // The original delta units the scrollBegin and scrollUpdates
      // were sent as.
      ScrollUnits deltaUnits;
      // The state of inertial phase scrolling. OSX has unique phases for normal
      // and momentum scroll events. Should always be UnknownMomentumPhase for
      // touch based input as it generates GestureFlingStart instead.
      InertialPhaseState inertialPhase;
      // True if this event was synthesized in order to generate the proper
      // GSB/GSU/GSE matching sequences. This is a temporary so that a future
      // GSB will generate a hit test so latching behavior is avoided
      // until crbug.com/526463 is fully implemented.
      bool synthetic;
    } scrollEnd;

    struct {
      float velocityX;
      float velocityY;
      // If true, this event will skip hit testing to find a scroll
      // target and instead just scroll the viewport.
      bool targetViewport;
    } flingStart;

    struct {
      // If set to true, don't treat flingCancel
      // as a part of fling boost events sequence.
      bool preventBoosting;
    } flingCancel;

    struct {
      bool zoomDisabled;
      float scale;
    } pinchUpdate;
  } data;

  WebGestureEvent()
      : WebInputEvent(sizeof(WebGestureEvent)),
        x(0),
        y(0),
        globalX(0),
        globalY(0),
        sourceDevice(WebGestureDeviceUninitialized),
        resendingPluginId(-1) {}
};

#pragma pack(pop)

}  // namespace blink

#endif  // WebGestureEvent_h
