% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/transcriptToX.R
\name{transcriptToProtein}
\alias{transcriptToProtein}
\title{Map transcript-relative coordinates to amino acid residues of the
encoded protein}
\usage{
transcriptToProtein(
  x,
  db,
  id = "name",
  proteins = NA,
  exons = NA,
  transcripts = NA
)
}
\arguments{
\item{x}{\code{IRanges} with the coordinates within the transcript. Coordinates
are counted from the start of the transcript (including the 5' UTR). The
Ensembl IDs of the corresponding transcripts have to be provided either
as \code{names} of the \code{IRanges}, or in one of its metadata columns.}

\item{db}{\code{EnsDb} object.}

\item{id}{\code{character(1)} specifying where the transcript identifier can be
found. Has to be either \code{"name"} or one of \code{colnames(mcols(prng))}.}

\item{proteins}{\code{DFrame} object generated by \code{\link[=proteins]{proteins()}}.}

\item{exons}{\code{CompressedGRangesList} object generated by \code{\link[=exonsBy]{exonsBy()}} where
by = 'tx'.}

\item{transcripts}{\code{GRanges} object generated by \code{\link[=transcripts]{transcripts()}}.}
}
\value{
\code{IRanges} with the same length (and order) than the input \code{IRanges}
\code{x}. Each element in \code{IRanges} provides the coordinates within the
protein sequence, names being the (Ensembl) IDs of the protein. The
original transcript ID and the transcript-relative coordinates are provided
as metadata columns. Metadata columns \code{"cds_ok"} indicates whether the
length of the transcript's CDS matches the length of the encoded protein.
\code{IRanges} with a start coordinate of \code{-1} is returned for transcript
coordinates that can not be mapped to protein-relative coordinates
(either no transcript was found for the provided ID, the transcript
does not encode a protein or the provided coordinates are not within
the coding region of the transcript).
}
\description{
\code{transcriptToProtein} maps within-transcript coordinates to the corresponding
coordinates within the encoded protein sequence. The provided coordinates
have to be within the coding region of the transcript (excluding the stop
codon) but are supposed to be relative to the first nucleotide of the
transcript (which includes the 5' UTR). Positions relative to the CDS of a
transcript (e.g. /PKP2 c.1643delg/) have to be first converted to
transcript-relative coordinates. This can be done with the
\code{\link[=cdsToTranscript]{cdsToTranscript()}} function.
}
\details{
Transcript-relative coordinates are mapped to the amino acid residues they
encode. As an example, positions within the transcript that correspond to
nucleotides 1 to 3 in the CDS are mapped to the first position in the
protein sequence (see examples for more details).
}
\examples{

library(EnsDb.Hsapiens.v86)
## Restrict all further queries to chromosome x to speed up the examples
edbx <- filter(EnsDb.Hsapiens.v86, filter = ~ seq_name == "X")

## Define an IRanges with the positions of the first 2 nucleotides of the
## coding region for the transcript ENST00000381578
txpos <- IRanges(start = 692, width = 2, names = "ENST00000381578")

## Map these to the corresponding residues in the protein sequence
## The protein-relative coordinates are returned as an `IRanges` object,
## with the original, transcript-relative coordinates provided in metadata
## columns tx_start and tx_end
transcriptToProtein(txpos, edbx)

## We can also map multiple ranges. Note that for any of the 3 nucleotides
## encoding the same amino acid the position of this residue in the
## protein sequence is returned. To illustrate this we map below each of the
## first 4 nucleotides of the CDS to the corresponding position within the
## protein.
txpos <- IRanges(start = c(692, 693, 694, 695),
    width = rep(1, 4), names = rep("ENST00000381578", 4))
transcriptToProtein(txpos, edbx)

## If the mapping fails, an IRanges with negative start position is returned.
## Mapping can fail (as below) because the ID is not known.
transcriptToProtein(IRanges(1, 1, names = "unknown"), edbx)

## Or because the provided coordinates are not within the CDS
transcriptToProtein(IRanges(1, 1, names = "ENST00000381578"), edbx)

## Meanwhile, this function can be called in parallel processes if you preload
## the protein, exons and transcripts database.

proteins <- proteins(edbx)
exons <- exonsBy(edbx)
transcripts <- transcripts(edbx)

txpos <- IRanges(start = c(692, 693, 694, 695),
    width = rep(1, 4), 
    names = c(rep("ENST00000381578", 2), rep("ENST00000486554", 2)), 
    info='test')

transcriptToProtein(txpos,edbx,proteins = proteins,exons = exons,transcripts = transcripts)
}
\seealso{
\code{\link[=cdsToTranscript]{cdsToTranscript()}} and \code{\link[=transcriptToCds]{transcriptToCds()}} for conversion between
CDS- and transcript-relative coordinates.

Other coordinate mapping functions: 
\code{\link{cdsToTranscript}()},
\code{\link{genomeToProtein}()},
\code{\link{genomeToTranscript}()},
\code{\link{proteinToGenome}()},
\code{\link{proteinToTranscript}()},
\code{\link{transcriptToCds}()},
\code{\link{transcriptToGenome}()}
}
\author{
Johannes Rainer
}
\concept{coordinate mapping functions}
